# -*- coding: utf-8 -*-
"""
Combined Sorting Application Module

This module provides the CombinedSortingApp class for sorting Excel sheet data
by passage numbers (지문번호) in the AutoQM application.

The app supports three sorting methods:
1. Ascending order by passage number
2. Descending order by passage number
3. Sort all sheets based on the order in the queue sheet (대기열)
"""

import tkinter as tk
from tkinter import messagebox
import pandas as pd
from openpyxl import load_workbook
from openpyxl.styles import Alignment
from openpyxl.utils import get_column_letter
import customtkinter
from customtkinter import CTkFrame, CTkRadioButton

from modules.font_handler import font_handler


class CombinedSortingApp(customtkinter.CTkToplevel):
    """
    A popup window for sorting Excel sheets by passage number.

    Provides three sorting options:
    1. Sort current sheet in ascending order by passage number
    2. Sort current sheet in descending order by passage number
    3. Sort all sheets based on the queue sheet (대기열) order

    Parameters
    ----------
    main_frame : MainFrame
        Reference to the main application frame
    sheet_selector : widget
        Reference to the sheet selector widget
    load_excel_func : callable
        Function to load Excel data into treeview
    update_sheet_func : callable
        Function to update the sheet selector
    excel_file : str
        Path to the Excel file
    sheet_name : str
        Name of the current sheet
    """

    # Class constant for Excel column widths
    CUSTOM_COLUMN_WIDTHS = [10, 80, 80, 20, 60]

    def __init__(self, main_frame, sheet_selector, load_excel_func, update_sheet_func, excel_file, sheet_name):
        super().__init__()

        self.main_frame = main_frame  # Reference to MainFrame
        self.sheet_selector = sheet_selector  # Reference to the sheet selector widget
        sheet_name = self.sheet_selector.get()
        self.load_excel_into_treeview = load_excel_func  # Function to load Excel into treeview
        self.update_sheet_selector = update_sheet_func  # Function to update the sheet selector

        self.title('지문번호 정렬 툴')
        self.geometry('400x200')
        self.minsize(400, 200)

        self.default_font = font_handler.default_font

        # Store excel_file and sheet_name as instance variables
        self.excel_file = excel_file
        self.sheet_name = sheet_name

        # Create a frame for the sorting options
        self.file_frame = CTkFrame(self, fg_color="transparent")
        self.file_frame.pack(pady=10, padx=20, fill='both', expand=True)

        # Variable to store the selected sorting method
        self.sorting_method = tk.IntVar(value=1)  # Default selection is the first radio button

        # Radio buttons for sorting methods
        sorting_radio_1 = CTkRadioButton(
            self.file_frame,
            text='지문번호를 기준으로 오름차순 정렬합니다.',
            font=(self.default_font, 13),
            variable=self.sorting_method,
            value=1,
            text_color="black",
            hover_color="#BB6C25",
            fg_color="#DDA15C"
        )
        sorting_radio_1.pack(side='top', padx=5, pady=(5, 5), fill='x', expand=False)

        sorting_radio_2 = CTkRadioButton(
            self.file_frame,
            text='지문번호를 기준으로 내림차순 정렬합니다.',
            font=(self.default_font, 13),
            variable=self.sorting_method,
            value=2,
            text_color="black",
            hover_color="#BB6C25",
            fg_color="#DDA15C"
        )
        sorting_radio_2.pack(side='top', padx=5, pady=(5, 5), fill='x', expand=False)

        sorting_radio_3 = CTkRadioButton(
            self.file_frame,
            text='대기열의 지문번호 순서에 따라 모든 시트를 정렬합니다.',
            font=(self.default_font, 13),
            variable=self.sorting_method,
            value=3,
            text_color="black",
            hover_color="#BB6C25",
            fg_color="#DDA15C"
        )
        sorting_radio_3.pack(side='top', padx=5, pady=(5, 5), fill='x', expand=False)

        # Frame for the execute buttons
        self.button_frame = CTkFrame(self.file_frame, fg_color="transparent")
        self.button_frame.pack(side='top', padx=5, pady=(10, 5), fill='x', expand=False)

        self.execute_button1 = customtkinter.CTkButton(
            self.button_frame,
            text="현재 유형만 실행",
            font=(self.default_font, 13, 'bold'),
            command=self.execute_sorting_current,
            width=160,
            height=40,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black"
        )
        self.execute_button1.pack(side='left', padx=5, pady=(10, 5), fill='x', expand=False)

        self.execute_button2 = customtkinter.CTkButton(
            self.button_frame,
            text="전체 유형 실행",
            font=(self.default_font, 13, 'bold'),
            command=self.execute_sorting_all,
            width=160,
            height=40,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black"
        )
        self.execute_button2.pack(side='right', padx=5, pady=(10, 5), fill='x', expand=False)

        # Update buttons based on selected sorting method
        self.sorting_method.trace('w', self.update_buttons)

        self.bind("<Escape>", self.on_popup_close)

    def update_buttons(self, *args):
        """Disable '현재 유형만 실행' button when '대기열' sorting is selected."""
        if self.sorting_method.get() == 3:
            self.execute_button1.configure(state='disabled')
        else:
            self.execute_button1.configure(state='normal')

    def on_popup_close(self, event=None):
        """Close the popup window."""
        self.destroy()

    def execute_sorting_current(self):
        """Execute sorting for current sheet only."""
        if self.sorting_method.get() in [1, 2]:
            self.sort_current_sheet()
        else:
            messagebox.showinfo("알림", "'대기열' 정렬은 현재 유형만 실행할 수 없습니다.")

    def execute_sorting_all(self):
        """Execute sorting for all sheets."""
        if self.sorting_method.get() in [1, 2]:
            self.sort_all_sheets()
        elif self.sorting_method.get() == 3:
            self.sort_all_sheets_by_queue()
        messagebox.showinfo("알림", "실행이 완료되었습니다.")
        self.destroy()

    def sort_current_sheet(self):
        """Sort the current sheet in ascending or descending order by passage number."""
        # Load current sheet
        sheet_name = self.sheet_selector.get()
        df = pd.read_excel(self.excel_file, sheet_name=sheet_name, header=None)

        if df.empty or 0 not in df.columns:
            print(f"Sheet '{sheet_name}' is empty or has incorrect data.")
            return

        # Sort DataFrame
        ascending = self.sorting_method.get() == 1
        sorted_df = df.sort_values(by=0, ascending=ascending)

        # Save sorted DataFrame
        with pd.ExcelWriter(self.excel_file, engine='openpyxl', mode='a', if_sheet_exists='replace') as writer:
            sorted_df.to_excel(writer, sheet_name=sheet_name, index=False, header=False)

        # Adjust Excel formatting
        self.adjust_excel_formatting(sheet_name, self.CUSTOM_COLUMN_WIDTHS)

        # Reload the sheet into the treeview
        self.load_excel_into_treeview(self.excel_file, sheet_name)

        messagebox.showinfo("알림", "실행이 완료되었습니다.")
        self.destroy()

    def sort_all_sheets(self):
        """Sort all sheets in ascending or descending order by passage number."""
        workbook = load_workbook(self.excel_file)
        sheet_names = workbook.sheetnames

        for sheet_name in sheet_names:
            df = pd.read_excel(self.excel_file, sheet_name=sheet_name, header=None)
            if df.empty or 0 not in df.columns:
                print(f"Skipping sheet '{sheet_name}' due to empty or incorrect data.")
                continue

            ascending = self.sorting_method.get() == 1
            sorted_df = df.sort_values(by=0, ascending=ascending)

            with pd.ExcelWriter(self.excel_file, engine='openpyxl', mode='a', if_sheet_exists='replace') as writer:
                sorted_df.to_excel(writer, sheet_name=sheet_name, index=False, header=False)

            self.adjust_excel_formatting(sheet_name, self.CUSTOM_COLUMN_WIDTHS)

        # Reload the current sheet into the treeview
        current_sheet = self.sheet_selector.get()
        self.load_excel_into_treeview(self.excel_file, current_sheet)

    def sort_all_sheets_by_queue(self):
        """Sort all sheets based on the order defined in the queue sheet (대기열)."""
        workbook = load_workbook(self.excel_file)
        sheet_names = workbook.sheetnames

        queue_sheet_name = '대기열'
        if queue_sheet_name not in sheet_names:
            messagebox.showerror("오류", f"Sheet '{queue_sheet_name}' not found.")
            return

        df_queue = pd.read_excel(self.excel_file, sheet_name=queue_sheet_name, header=None)
        queue_order = df_queue.iloc[:, 0].tolist()
        queue_order_mapping = {value: index for index, value in enumerate(queue_order)}

        for sheet_name in sheet_names:
            if sheet_name == queue_sheet_name:
                continue

            df = pd.read_excel(self.excel_file, sheet_name=sheet_name, header=None)
            if df.empty or 0 not in df.columns:
                print(f"Skipping sheet '{sheet_name}' due to empty or incorrect data.")
                continue

            df_in_queue = df[df[0].isin(queue_order_mapping.keys())].copy()
            df_not_in_queue = df[~df[0].isin(queue_order_mapping.keys())].copy()

            df_in_queue['sort_key'] = df_in_queue[0].map(queue_order_mapping)
            df_not_in_queue = df_not_in_queue.sort_values(by=0)
            max_sort_key = df_in_queue['sort_key'].max() if not df_in_queue.empty else -1
            df_not_in_queue['sort_key'] = range(max_sort_key + 1, max_sort_key + 1 + len(df_not_in_queue))

            df_combined = pd.concat([df_in_queue, df_not_in_queue], ignore_index=True)
            df_combined = df_combined.sort_values(by='sort_key').drop(columns=['sort_key'])

            with pd.ExcelWriter(self.excel_file, engine='openpyxl', mode='a', if_sheet_exists='replace') as writer:
                df_combined.to_excel(writer, sheet_name=sheet_name, index=False, header=False)

            self.adjust_excel_formatting(sheet_name, self.CUSTOM_COLUMN_WIDTHS)

        # Reload the current sheet into the treeview
        current_sheet = self.sheet_selector.get()
        self.load_excel_into_treeview(self.excel_file, current_sheet)

    def adjust_excel_formatting(self, sheet_name, custom_column_widths):
        """
        Adjust Excel formatting for the specified sheet.

        Parameters
        ----------
        sheet_name : str
            Name of the sheet to format
        custom_column_widths : list
            List of column widths to apply
        """
        workbook = load_workbook(self.excel_file)
        sheet = workbook[sheet_name]
        for col_num, column_width in enumerate(custom_column_widths, start=1):
            sheet.column_dimensions[get_column_letter(col_num)].width = column_width
        for row in sheet.iter_rows():
            for cell in row:
                cell.alignment = Alignment(wrapText=True)
        workbook.save(self.excel_file)
