# modules/converter.py
# excel to word file
from .editing_functions import FUNCTION_MAP
import re
import random
from collections import defaultdict
from pathlib import Path
from tkinter import messagebox, Toplevel
import inspect
from docx import Document
from docx.shared import Inches, Pt, Cm, RGBColor
from docx.image.exceptions import UnrecognizedImageError
from docx.enum.text import WD_PARAGRAPH_ALIGNMENT, WD_LINE_SPACING, WD_BREAK
from docx.oxml import OxmlElement, parse_xml
from docx.oxml.ns import qn, nsdecls
from docx.enum.style import WD_STYLE_TYPE
from docx.enum.text import WD_ALIGN_PARAGRAPH
from docx.enum.table import WD_TABLE_ALIGNMENT
from docx.enum.table import WD_ALIGN_VERTICAL
from docx.enum.text import WD_COLOR_INDEX
from docx.enum.section import WD_SECTION

from openpyxl import load_workbook
import shutil
import os.path
import tempfile
import unicodedata
import requests
from io import BytesIO
import spacy
import sys
import zlib
import base64

# Import customtkinter components
from customtkinter import CTkLabel, CTkFrame, CTkButton

# spaCy model will be passed from autoQM.py
nlp = None




########################################################
################### 편집 함수  ###################  
########################################################


# Function to parse the 'question_part' data and extract table information
def parse_data(question_part):
    lines = question_part.strip().split('\n')[2:]  # Skip header and separator lines
    data = []
    for line in lines:
        columns = line.split('|')[1:-1]  # Split by '|' and remove first and last empty strings
        data.append([col.strip() for col in columns])  # Strip spaces and add to data list
    return data

# Function to insert data into a table in the document
def insert_data_into_table(document, data):
    max_cols = max(len(row) for row in data)
    table = document.add_table(rows=len(data) + 1, cols=max_cols)
    
    column_widths = [Inches(1), Inches(1), Inches(3), Inches(3)][:max_cols]  # Adjust based on the number of columns
    
    for row in table.rows:
        for idx, width in enumerate(column_widths):
            row.cells[idx].width = width

    # Add header row and format cells
    #for i, header in enumerate(["번호", "단어", "뜻", "예문"]):
    for i, header in enumerate(["영어", "뜻", "동의어", "반의어"]):
        cell = table.cell(0, i)
        cell.text = header
        set_cell_border(
            cell,
            top={"sz": 5, "val": "single", "color": "auto"},
            bottom={"sz": 5, "val": "single", "color": "auto"},
            left={"sz": 5, "val": "single", "color": "auto"},
            right={"sz": 5, "val": "single", "color": "auto"}
        )
    
    # Insert data rows and format cells
    for row_idx, row_data in enumerate(data, start=1):
        for col_idx, cell_data in enumerate(row_data):
            cell = table.cell(row_idx, col_idx)
            cell.text = cell_data
            set_cell_border(
                cell,
                top={"sz": 5, "val": "single", "color": "auto"},
                bottom={"sz": 5, "val": "single", "color": "auto"},
                left={"sz": 5, "val": "single", "color": "auto"},
                right={"sz": 5, "val": "single", "color": "auto"}
            )
    
    # Set table cell margins after formatting cells if needed
    set_table_cell_margins(table, top=3, bottom=3, left=5, right=5)



# Function to insert data into a table in the document
def insert_data_into_table_영영정의(document, data):
    max_cols = max(len(row) for row in data)
    table = document.add_table(rows=len(data) + 1, cols=max_cols)
    
    column_widths = [Inches(1), Inches(1), Inches(6)][:max_cols]  # Adjust based on the number of columns
    
    for row in table.rows:
        for idx, width in enumerate(column_widths):
            row.cells[idx].width = width

    # Add header row and format cells
    #for i, header in enumerate(["번호", "단어", "뜻", "예문"]):
    for i, header in enumerate(["영어", "뜻", "영영정의"]):
        cell = table.cell(0, i)
        cell.text = header
        set_cell_border(
            cell,
            top={"sz": 5, "val": "single", "color": "auto"},
            bottom={"sz": 5, "val": "single", "color": "auto"},
            left={"sz": 5, "val": "single", "color": "auto"},
            right={"sz": 5, "val": "single", "color": "auto"}
        )
    
    # Insert data rows and format cells
    for row_idx, row_data in enumerate(data, start=1):
        for col_idx, cell_data in enumerate(row_data):
            cell = table.cell(row_idx, col_idx)
            cell.text = cell_data
            set_cell_border(
                cell,
                top={"sz": 5, "val": "single", "color": "auto"},
                bottom={"sz": 5, "val": "single", "color": "auto"},
                left={"sz": 5, "val": "single", "color": "auto"},
                right={"sz": 5, "val": "single", "color": "auto"}
            )
    
    # Set table cell margins after formatting cells if needed
    set_table_cell_margins(table, top=3, bottom=3, left=5, right=5)


def set_cell_border(cell, **kwargs):
    tc = cell._tc
    tcPr = tc.get_or_add_tcPr()

    for edge in ('top', 'bottom', 'left', 'right', 'insideH', 'insideV'):
        edge_data = kwargs.get(edge)
        if edge_data:
            tag = 'w:{}'.format(edge)
            element = OxmlElement(tag)
            element.set(qn('w:val'), edge_data.get('val', 'single'))
            element.set(qn('w:sz'), str(edge_data.get('sz', 5)))
            element.set(qn('w:space'), str(edge_data.get('space', 0)))
            element.set(qn('w:color'), edge_data.get('color', 'auto'))
            tcPr.append(element)


def set_table_cell_margins(table, top=0, bottom=0, left=0, right=0):
    # Convert points to dxa (1 point = 20 dxa)
    top_dxa = str(int(top * 20))
    bottom_dxa = str(int(bottom * 20))
    left_dxa = str(int(left * 20))
    right_dxa = str(int(right * 20))

    for row in table.rows:
        for cell in row.cells:
            tcPr = cell._element.get_or_add_tcPr()
            tcMar = OxmlElement('w:tcMar')

            # Set top margin
            top_mar = OxmlElement('w:top')
            top_mar.set(qn('w:w'), top_dxa)
            top_mar.set(qn('w:type'), "dxa")
            tcMar.append(top_mar)

            # Set bottom margin
            bottom_mar = OxmlElement('w:bottom')
            bottom_mar.set(qn('w:w'), bottom_dxa)
            bottom_mar.set(qn('w:type'), "dxa")
            tcMar.append(bottom_mar)

            # Set left margin
            left_mar = OxmlElement('w:left')
            left_mar.set(qn('w:w'), left_dxa)
            left_mar.set(qn('w:type'), "dxa")
            tcMar.append(left_mar)

            # Set right margin
            right_mar = OxmlElement('w:right')
            right_mar.set(qn('w:w'), right_dxa)
            right_mar.set(qn('w:type'), "dxa")
            tcMar.append(right_mar)

            tcPr.append(tcMar)








def add_bold_underline_to_brackets(cell, paragraph_texts):
    """
    - <...>  => Bold + Underline
    - [[...]] => Bold + Underline
    - [...] => Bold + Underline
         If { ... } appears inside [ ... ], then the { ... } text (without braces)
         will be Bold + Italic + Underline.
    - { ... } => Italic
         If [ ... ] appears inside { ... }, then the [ ... ] text (without brackets)
         will be Bold + Italic + Underline.
    """
    # Main pattern to capture the four bracket types.
    pattern = re.compile(
        r"\<([^>]*)\>"       # group(1): <...>
        r"|\[\[([^\]]*)\]\]" # group(2): [[...]]
        r"|\[([^\]]*)\]"     # group(3): [...]
        r"|\{([^}]*)\}"      # group(4): {...}
    )
    
    # Inner pattern for detecting [ ... ] inside { ... }.
    inner_square_pattern = re.compile(r"\[([^\]]+)\]")
    # New inner pattern for detecting { ... } inside [ ... ].
    inner_curly_pattern = re.compile(r"\{([^}]+)\}")

    for idx, paragraph_text in enumerate(paragraph_texts):
        # If it's the first paragraph, clear the existing one.
        if idx == 0:
            para = cell.paragraphs[0]
            para.clear()
        else:
            para = cell.add_paragraph()
        
        # Choose paragraph style.
        if paragraph_text.startswith('*'):
            para.style = 'Normal_right'
        else:
            para.style = 'Normal_modified'

        last_idx = 0
        for match in pattern.finditer(paragraph_text):
            start, end = match.span()
            
            # Add text before the current match as normal (unformatted).
            if start > last_idx:
                para.add_run(paragraph_text[last_idx:start])
            
            # Check which group matched.
            group1 = match.group(1)  # <...>
            group2 = match.group(2)  # [[...]]
            group3 = match.group(3)  # [...]
            group4 = match.group(4)  # {...}
            
            if group1 is not None:
                # <...> => Bold + Underline.
                run = para.add_run(group1)
                run.bold = True
                run.underline = True
            elif group2 is not None:
                # [[...]] => Bold + Underline.
                run = para.add_run(group2)
                run.bold = True
                run.underline = True
            elif group3 is not None:
                # [...] => Bold + Underline.
                # Check for nested { ... } inside the square brackets.
                last_inner_idx = 0
                found_inner = False
                for inner_match in inner_curly_pattern.finditer(group3):
                    found_inner = True
                    inner_start, inner_end = inner_match.span()
                    # Add text before { ... } as Bold + Underline.
                    if inner_start > last_inner_idx:
                        run = para.add_run(group3[last_inner_idx:inner_start])
                        run.bold = True
                        run.underline = True
                    # Add the text inside { ... } (without braces) as Bold + Italic + Underline.
                    run = para.add_run(inner_match.group(1))
                    run.bold = True
                    run.italic = True
                    run.underline = True
                    last_inner_idx = inner_end
                if found_inner:
                    # Add any remaining text after the last { ... } as Bold + Underline.
                    if last_inner_idx < len(group3):
                        run = para.add_run(group3[last_inner_idx:])
                        run.bold = True
                        run.underline = True
                else:
                    # No nested { ... } found: add the whole text.
                    run = para.add_run(group3)
                    run.bold = True
                    run.underline = True
            elif group4 is not None:
                # {...} => Italic.
                # Check for nested [ ... ] inside the curly braces.
                last_inner_idx = 0
                found_inner = False
                for inner_match in inner_square_pattern.finditer(group4):
                    found_inner = True
                    inner_start, inner_end = inner_match.span()
                    # Add text before [ ... ] as Italic.
                    if inner_start > last_inner_idx:
                        run = para.add_run(group4[last_inner_idx:inner_start])
                        run.italic = True
                    # Add the text inside [ ... ] (without brackets) as Bold + Italic + Underline.
                    run = para.add_run(inner_match.group(1))
                    run.bold = True
                    run.italic = True
                    run.underline = True
                    last_inner_idx = inner_end
                if found_inner:
                    # Add any remaining text after the last [ ... ] as Italic.
                    if last_inner_idx < len(group4):
                        run = para.add_run(group4[last_inner_idx:])
                        run.italic = True
                else:
                    # No nested [ ... ] found: add the whole text as Italic.
                    run = para.add_run(group4)
                    run.italic = True
            last_idx = end
        
        # Add any remaining text after the last match.
        if last_idx < len(paragraph_text):
            para.add_run(paragraph_text[last_idx:])







def remove_brackets(cell, paragraph_texts):
    # Define the pattern to match text within < > or [ ]
    pattern = re.compile(r"\<([^>]*)\>|\[([^\]]*)\]")

    # Iterate over each paragraph text
    for paragraph_text in paragraph_texts:
        # If it's the first paragraph, use the existing one; otherwise, create a new one
        if paragraph_texts.index(paragraph_text) == 0:
            para = cell.paragraphs[0]
            para.clear()  # Clear the existing paragraph to remove any default text
            para.style = 'Normal_modified'  # Apply the 'Normal_modified' style
        else:
            para = cell.add_paragraph(style='Normal_modified')

        last_idx = 0  # Start index of the last added run
        for match in pattern.finditer(paragraph_text):
            # Add the text before the match
            start, end = match.span()
            para.add_run(paragraph_text[last_idx:start])
            # Extract the text without the brackets for bold and underline
            text_within_brackets = match.group(1) or match.group(2)
            # Add the extracted text in bold and underline
            run = para.add_run(text_within_brackets)
            run.bold = False
            run.underline = False
            # Update the last index
            last_idx = end

        # Add any remaining text after the last match
        para.add_run(paragraph_text[last_idx:])





class ExcelToWordConverter:
    def __init__(self, base_path, app_logger, global_text_input_to_extract,
                 selected_font_for_title, selected_font_for_num, selected_font_for_text,
                 selected_font_for_exp, size_heading, size_num, size_text, size_exp,
                 color_heading, color_num, color_text, color_exp, line_text, line_exp,
                 tagremoval_flag, applyoption_var, book_cover_path=None, text_school_grade=None, text_exam_type=None, text_week=None, two_column=False, border_flag=True, passage_number_numeric=False, separate_answer_key=False, base_url=None, nlp=None, jimun_analysis_interpretation_right=None, jimun_analysis_font=None, jimun_analysis_font_size=None, jimun_analysis_font_color=None, jimun_analysis_line_spacing=None, jimun_analysis_sentence_split=None, hanjul_haesuk_font=None, hanjul_haesuk_font_size=None, hanjul_haesuk_font_color=None, hanjul_haesuk_line_spacing=None, hanjul_interpretation_include_answer=None, hanjul_writing_include_answer=None, hanjul_writing_korean_only=None, hanjul_writing_word_scramble=None, hanjul_writing_word_form_change=None, naeyongilchi_question_type=None, new_page_per_section=None, page_break_after_passages=None, user_token=None, margin_top=None, margin_bottom=None, margin_left=None, margin_right=None, col_spacing=None, enable_mirror_margins=False):
        self.base_path = base_path
        self.app_logger = app_logger
        self.global_text_input_to_extract = global_text_input_to_extract
        self.nlp = nlp  # Store spaCy model
        self.selected_font_for_title = selected_font_for_title
        self.selected_font_for_num = selected_font_for_num
        self.selected_font_for_text = selected_font_for_text
        self.selected_font_for_exp = selected_font_for_exp
        self.size_heading = size_heading
        self.size_num = size_num
        self.size_text = size_text
        self.size_exp = size_exp
        self.color_heading = color_heading
        self.color_num = color_num
        self.color_text = color_text
        self.color_exp = color_exp
        self.line_text = line_text
        self.line_exp = line_exp
        self.tagremoval_flag = tagremoval_flag
        self.applyoption_var = applyoption_var

        # 교재제작 옵션 - '지문분석' formatting
        self.jimun_analysis_font = jimun_analysis_font or "나눔고딕"
        # Use float() to handle decimal values like "8.5" and strip whitespace
        try:
            self.jimun_analysis_font_size = float(str(jimun_analysis_font_size).strip()) if jimun_analysis_font_size else 8.0
        except (ValueError, AttributeError):
            self.jimun_analysis_font_size = 8.0
        self.jimun_analysis_font_color = jimun_analysis_font_color or "#000000"
        # Parse line spacing defensively
        try:
            self.jimun_analysis_line_spacing = float(str(jimun_analysis_line_spacing).strip()) if jimun_analysis_line_spacing else 1.2
        except (ValueError, AttributeError):
            self.jimun_analysis_line_spacing = 1.2

        # 지문분석 옵션 - 지문을 문장별로 분리 (0=off, 1=on)
        self.jimun_analysis_sentence_split = 0 if jimun_analysis_sentence_split is None else int(jimun_analysis_sentence_split)
        # 지문분석 옵션 - 해석을 지문 우측에 두기 (0=off, 1=on)
        self.jimun_analysis_interpretation_right = 0 if jimun_analysis_interpretation_right is None else int(jimun_analysis_interpretation_right)

        # 교재제작 옵션 - '한글해석' formatting
        self.hanjul_haesuk_font = hanjul_haesuk_font or "나눔고딕"
        try:
            self.hanjul_haesuk_font_size = float(str(hanjul_haesuk_font_size).strip()) if hanjul_haesuk_font_size else 6.0
        except (ValueError, AttributeError):
            self.hanjul_haesuk_font_size = 6.0
        self.hanjul_haesuk_font_color = hanjul_haesuk_font_color or "#000000"
        try:
            self.hanjul_haesuk_line_spacing = float(str(hanjul_haesuk_line_spacing).strip()) if hanjul_haesuk_line_spacing else 1.0
        except (ValueError, AttributeError):
            self.hanjul_haesuk_line_spacing = 1.0

        # 교재제작 옵션 - Include answer flags (default to 1 = include)
        self.hanjul_interpretation_include_answer = 1 if hanjul_interpretation_include_answer is None else int(hanjul_interpretation_include_answer)
        self.hanjul_writing_include_answer = 1 if hanjul_writing_include_answer is None else int(hanjul_writing_include_answer)

        # 교재제작 옵션 - 한줄영작연습 hint mode flags
        self.hanjul_writing_korean_only = 0 if hanjul_writing_korean_only is None else int(hanjul_writing_korean_only)
        self.hanjul_writing_word_scramble = 1 if hanjul_writing_word_scramble is None else int(hanjul_writing_word_scramble)
        self.hanjul_writing_word_form_change = 0 if hanjul_writing_word_form_change is None else int(hanjul_writing_word_form_change)

        # 문제 옵션 - 내용일치 question type (0=T/F 형식, 1=5지선다)
        self.naeyongilchi_question_type = 0 if naeyongilchi_question_type is None else int(naeyongilchi_question_type)

        # 내지 서식 옵션 - 유형마다 새 페이지 (1=새 페이지 삽입, 0=삽입 안함)
        # Default is 1 to maintain existing behavior (page break between sections)
        self.new_page_per_section = 1 if new_page_per_section is None else int(new_page_per_section)

        # 내지 서식 옵션 - 페이지당 문제 수 (0=disabled)
        try:
            self.page_break_after_passages = int(page_break_after_passages)
        except (TypeError, ValueError):
            self.page_break_after_passages = 0
        if self.page_break_after_passages < 1:
            self.page_break_after_passages = 0

        # Margin settings (in cm) - safe float conversion with fallback defaults
        def safe_float(val, default):
            try:
                return float(val) if val is not None else default
            except (TypeError, ValueError):
                return default

        self.margin_top = safe_float(margin_top, 1.5)
        self.margin_bottom = safe_float(margin_bottom, 1.6)
        self.margin_left = safe_float(margin_left, 1.9)
        self.margin_right = safe_float(margin_right, 1.9)

        # Column spacing (in cm) - safe float conversion with fallback default
        self.col_spacing = safe_float(col_spacing, 1.27)

        # Mirror margins setting (coerce to boolean)
        self.enable_mirror_margins = bool(enable_mirror_margins) if enable_mirror_margins else False

        # Set book cover path with fallback logic
        if book_cover_path:
            self.book_cover_path = book_cover_path
        else:
            # Try to find default_bookcover01.docx (may have numeric prefix)
            found_path = self._find_book_cover_by_basename("default_bookcover01.docx")
            if found_path:
                self.book_cover_path = found_path
            else:
                # Fallback to hardcoded path (even if it doesn't exist)
                self.book_cover_path = self.base_path / "assets" / "bookcovers" / "default_bookcover01.docx"

        self.close_export_popup = False

        # Track warnings during export
        self.export_warnings = []  # List of tuples: (sheet_type, passage_id, reason)

        self.text_school_grade = text_school_grade
        self.text_exam_type = text_exam_type
        self.text_week = text_week
        self.two_column = two_column
        self.border_flag = border_flag
        self.passage_number_numeric = passage_number_numeric
        self.separate_answer_key = separate_answer_key
        self.base_url = base_url
        self.user_token = user_token
        self._plantuml_cache = {}


        print(f"self.border_flag: {self.border_flag}")


    def _find_book_cover_by_basename(self, target_basename):
        """
        Find a book cover file by its basename, ignoring numeric prefixes.
        Only matches exact filename with or without numeric prefix (e.g., ###_filename.docx).

        Args:
            target_basename: The basename to search for (e.g., "default_bookcover01.docx")

        Returns:
            Path object if found, None otherwise
        """
        bookcovers_dir = self.base_path / "assets" / "bookcovers"
        if not bookcovers_dir.exists():
            return None

        # Search for exact match with or without numeric prefix
        for file_path in bookcovers_dir.glob("*.docx"):
            filename = file_path.name

            # Check exact match (e.g., "default_bookcover01.docx")
            if filename == target_basename:
                return file_path

            # Check with numeric prefix (e.g., "001_default_bookcover01.docx")
            if "_" in filename:
                parts = filename.split("_", 1)
                if len(parts) == 2 and parts[0].isdigit() and parts[1] == target_basename:
                    return file_path

        return None

    def _apply_mirror_margins(self, section):
        """Add <w:mirrorMargins/> element to section if enabled.

        Mirror margins cause odd/even pages to have swapped left/right margins,
        which is useful for book-style printing where pages are bound on one side.

        Args:
            section: A python-docx Section object
        """
        if not self.enable_mirror_margins:
            return

        from docx.oxml.ns import qn
        from docx.oxml import OxmlElement

        sectPr = section._sectPr
        # Check if mirrorMargins already exists
        existing = sectPr.find(qn('w:mirrorMargins'))
        if existing is None:
            # Create empty element: <w:mirrorMargins/>
            mirror_elem = OxmlElement('w:mirrorMargins')
            sectPr.append(mirror_elem)

    def _add_page_number_footer(self, section):
        """Add page numbers at center bottom of the page (footer).

        Creates a footer with centered page number using Word field codes.

        Args:
            section: A python-docx Section object
        """
        from docx.oxml.ns import qn
        from docx.oxml import OxmlElement
        from docx.enum.text import WD_PARAGRAPH_ALIGNMENT

        # Get or create the footer
        footer = section.footer
        footer.is_linked_to_previous = False

        # Add a new paragraph for page number (footer always has at least one paragraph)
        footer_para = footer.paragraphs[0]
        footer_para.clear()  # Clear any existing content
        footer_para.alignment = WD_PARAGRAPH_ALIGNMENT.CENTER

        # Create PAGE field: <w:fldSimple w:instr=" PAGE "/>
        run = footer_para.add_run()
        fldSimple = OxmlElement('w:fldSimple')
        fldSimple.set(qn('w:instr'), ' PAGE ')
        run._r.append(fldSimple)

        # Set font size for page number
        run.font.size = Pt(10)

    # PlantUML Functions
    def cm_to_twips(self, cm):
        """Convert centimeters to twips (1 cm ≈ 567 twips)"""
        return int(cm * 567)

    def encode_plantuml(self, plantuml_text):
        """
        Encode PlantUML text for web API using deflate + custom base64 encoding
        """
        plantuml_alphabet = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz-_'

        # Compress the text
        compressed = zlib.compress(plantuml_text.encode('utf-8'))[2:-4]  # Remove zlib header/footer

        # Encode to PlantUML's custom base64
        encoded = ''
        for i in range(0, len(compressed), 3):
            if i + 2 < len(compressed):
                b1, b2, b3 = compressed[i], compressed[i + 1], compressed[i + 2]
                encoded += plantuml_alphabet[(b1 >> 2) & 0x3F]
                encoded += plantuml_alphabet[((b1 & 0x3) << 4) | ((b2 >> 4) & 0xF)]
                encoded += plantuml_alphabet[((b2 & 0xF) << 2) | ((b3 >> 6) & 0x3)]
                encoded += plantuml_alphabet[b3 & 0x3F]
            elif i + 1 < len(compressed):
                b1, b2 = compressed[i], compressed[i + 1]
                encoded += plantuml_alphabet[(b1 >> 2) & 0x3F]
                encoded += plantuml_alphabet[((b1 & 0x3) << 4) | ((b2 >> 4) & 0xF)]
                encoded += plantuml_alphabet[((b2 & 0xF) << 2)]
            else:
                b1 = compressed[i]
                encoded += plantuml_alphabet[(b1 >> 2) & 0x3F]
                encoded += plantuml_alphabet[(b1 & 0x3) << 4]

        return encoded

    def generate_diagram_from_plantuml(self, plantuml_code):
        """
        Generate diagram image from PlantUML code.

        Flow:
        1. Check in-memory cache (desktop-side)
        2. Try private Flask API (if user_token and base_url configured)
        3. Fall back to public PlantUML API (original behavior)

        Returns BytesIO object containing the PNG image, or None if all attempts fail.
        """
        import time

        # Encode diagram
        encoded = self.encode_plantuml(plantuml_code)

        # Check in-memory cache first
        cached_image = self._plantuml_cache.get(encoded)
        if cached_image is not None:
            print("PlantUML diagram served from desktop cache")
            return BytesIO(cached_image)

        # Try private Flask API first (if configured) with retry for transient 404s
        if self.user_token and self.base_url:
            api_url = f"{self.base_url}/api/plantuml/png/{encoded}"
            headers = {
                "Authorization": f"Bearer {self.user_token}",
                "User-Agent": "AutoQM/2.4.4"
            }

            max_private_retries = 2  # Retry once on 404 (total 2 attempts)
            for attempt in range(max_private_retries):
                try:
                    print(f"Attempting PlantUML generation via private API: {encoded[:20]}...")
                    response = requests.get(api_url, headers=headers, timeout=15)

                    if response.status_code == 200:
                        # Success - cache and return
                        self._plantuml_cache[encoded] = response.content
                        print("PlantUML diagram generated via private API")
                        return BytesIO(response.content)
                    elif response.status_code == 404 and attempt < max_private_retries - 1:
                        # 404 might be transient routing issue - retry once
                        self.app_logger.warning(
                            f"Private PlantUML API returned 404, retrying in 1s... (attempt {attempt + 1})"
                        )
                        time.sleep(1)
                        continue
                    else:
                        # Other error or final 404 - fall back to public
                        self.app_logger.warning(
                            f"Private PlantUML API returned {response.status_code}, "
                            f"falling back to public API"
                        )
                        break

                except Exception as e:
                    # Private API error - log and fall back to public
                    self.app_logger.warning(
                        f"Private PlantUML API error: {e}, falling back to public API"
                    )
                    break

        # Fall back to public PlantUML API (original logic)
        max_retries = 3

        # Add delay before first public API attempt to avoid rate limiting
        time.sleep(0.5)

        request_headers = {"User-Agent": "AutoQM/2.4.4"}

        for attempt in range(max_retries):
            try:
                # Build public API URL
                api_url = f"https://www.plantuml.com/plantuml/png/{encoded}"

                # Fetch the image with timeout
                response = requests.get(api_url, timeout=15, headers=request_headers)
                response.raise_for_status()

                # Success - cache and return
                self._plantuml_cache[encoded] = response.content
                self.app_logger.info("PlantUML diagram generated via public API")
                return BytesIO(response.content)

            except Exception as e:
                if attempt < max_retries - 1:
                    wait_time = (attempt + 1) * 3  # 3, 6, 9 seconds
                    self.app_logger.warning(
                        f"Public API attempt {attempt + 1} failed: {e}. "
                        f"Retrying in {wait_time}s..."
                    )
                    time.sleep(wait_time)
                else:
                    self.app_logger.error(
                        f"Failed to generate PlantUML diagram after {max_retries} "
                        f"public API attempts: {e}"
                    )
                    return None

    # Helper Functions
    def remove_last_page_break(self, doc):
        
        for para in reversed(doc.paragraphs):
            # Look for any run that is exactly a page break
            if any('w:br' in run._element.xml and 'type="page"' in run._element.xml
                for run in para.runs):
                # Remove that paragraph element
                p = para._element
                p.getparent().remove(p)
                break


    def apply_remove_brackets_to_paragraph(self, paragraph, text_list):
        """Remove brackets but maintain text within them, with border support"""
        pattern = re.compile(r"\<([^>]*)\>|\[([^\]]*)\]")
        
        # Process all paragraphs
        paragraphs_to_create = []
        for text in text_list:
            is_right_aligned = text.startswith('*')
            style = 'Normal_right' if is_right_aligned else 'Normal_modified_box'
            paragraphs_to_create.append((text, style, is_right_aligned))
        
        # Create paragraphs with appropriate borders
        for idx, (text, style, is_right_aligned) in enumerate(paragraphs_to_create):
            # Create new paragraph with appropriate style
            new_para = paragraph._parent.add_paragraph(style=style)
            
            # Apply borders based on position
            if idx == 0:
                # First paragraph - all borders
                self.add_paragraph_border(new_para, top=True, bottom=(len(paragraphs_to_create) == 1), 
                                        left=True, right=True)
            elif idx == len(paragraphs_to_create) - 1:
                # Last paragraph - only left, right, and bottom
                self.add_paragraph_border(new_para, top=False, bottom=True, left=True, right=True)
            else:
                # Middle paragraphs - only left and right
                self.add_paragraph_border(new_para, top=False, bottom=False, left=True, right=True)
            
            # Apply padding
            new_para.paragraph_format.left_indent = Pt(5)
            new_para.paragraph_format.right_indent = Pt(5)
            new_para.paragraph_format.space_before = Pt(0)
            new_para.paragraph_format.space_after = Pt(0)
            
            # Process text removing brackets
            text_to_process = text
            last_idx = 0
            
            for match in pattern.finditer(text_to_process):
                start, end = match.span()
                new_para.add_run(text_to_process[last_idx:start])
                text_within_brackets = match.group(1) or match.group(2)
                if text_within_brackets:
                    new_para.add_run(text_within_brackets)
                last_idx = end
            
            new_para.add_run(text_to_process[last_idx:])
        
        # Remove the original paragraph
        paragraph._element.getparent().remove(paragraph._element)



    def apply_italic_to_paragraph(self, paragraph, text_list):
        """Apply italic formatting to text within curly braces {...} with borders"""
        # Pattern to match only curly braces
        inner_curly_pattern = re.compile(r"\{([^}]+)\}")

        # Process all paragraphs and determine which need borders
        paragraphs_to_create = []
        for text in text_list:
            is_right_aligned = text.startswith('*')
            style = 'Normal_right' if is_right_aligned else 'Normal_modified_box'
            paragraphs_to_create.append((text, style, is_right_aligned))

        # Create paragraphs with appropriate borders
        for idx, (text, style, is_right_aligned) in enumerate(paragraphs_to_create):
            # Create new paragraph with appropriate style
            new_para = paragraph._parent.add_paragraph(style=style)

            # Apply borders based on position
            if idx == 0:
                # First paragraph - all borders
                self.add_paragraph_border(new_para, top=True, bottom=(len(paragraphs_to_create) == 1),
                                        left=True, right=True)
            elif idx == len(paragraphs_to_create) - 1:
                # Last paragraph - only left, right, and bottom
                self.add_paragraph_border(new_para, top=False, bottom=True, left=True, right=True)
            else:
                # Middle paragraphs - only left and right
                self.add_paragraph_border(new_para, top=False, bottom=False, left=True, right=True)

            # Apply padding
            new_para.paragraph_format.left_indent = Pt(5)
            new_para.paragraph_format.right_indent = Pt(5)
            new_para.paragraph_format.space_before = Pt(0)
            new_para.paragraph_format.space_after = Pt(0)

            # Process text with curly brace formatting only
            text_to_process = text
            last_idx = 0

            for match in inner_curly_pattern.finditer(text_to_process):
                start, end = match.span()

                # Add text before the match (plain text)
                if start > last_idx:
                    new_para.add_run(text_to_process[last_idx:start])

                # Add text inside curly braces as italic (without the braces)
                run = new_para.add_run(match.group(1))
                run.italic = True

                last_idx = end

            # Add any remaining text (plain text)
            if last_idx < len(text_to_process):
                new_para.add_run(text_to_process[last_idx:])

        # Remove the original paragraph that was passed in
        paragraph._element.getparent().remove(paragraph._element)


    def apply_bold_underline_to_paragraph(self, paragraph, text_list):
        """Apply bold and underline formatting to brackets in a paragraph"""
        pattern = re.compile(
            r"\<([^>]*)\>"       # group(1): <...>
            r"|\[\[([^\]]*)\]\]" # group(2): [[...]]
            r"|\[([^\]]*)\]"     # group(3): [...]
            r"|\{([^}]*)\}"      # group(4): {...}
            r"|(\(\d+\)❮[^❯]*❯)" # group(5): (number)❮ ... ❯ for vocabulary questions
        )
        
        inner_square_pattern = re.compile(r"\[([^\]]+)\]")
        inner_curly_pattern = re.compile(r"\{([^}]+)\}")
        
        # Process all paragraphs and determine which need borders
        paragraphs_to_create = []
        for text in text_list:
            is_right_aligned = text.startswith('*')
            style = 'Normal_right' if is_right_aligned else 'Normal_modified_box'
            paragraphs_to_create.append((text, style, is_right_aligned))
        
        # Create paragraphs with appropriate borders
        for idx, (text, style, is_right_aligned) in enumerate(paragraphs_to_create):
            # Create new paragraph with appropriate style
            new_para = paragraph._parent.add_paragraph(style=style)
            
            # Apply borders based on position
            if idx == 0:
                # First paragraph - all borders
                self.add_paragraph_border(new_para, top=True, bottom=(len(paragraphs_to_create) == 1), 
                                        left=True, right=True)
            elif idx == len(paragraphs_to_create) - 1:
                # Last paragraph - only left, right, and bottom
                self.add_paragraph_border(new_para, top=False, bottom=True, left=True, right=True)
            else:
                # Middle paragraphs - only left and right
                self.add_paragraph_border(new_para, top=False, bottom=False, left=True, right=True)
            
            # Apply padding
            new_para.paragraph_format.left_indent = Pt(5)
            new_para.paragraph_format.right_indent = Pt(5)
            new_para.paragraph_format.space_before = Pt(0)
            new_para.paragraph_format.space_after = Pt(0)
            
            # Determine which Korean words need to be formatted in this text
            words_to_format = []
            if '없는' in text:
                words_to_format.append('없는')
            if '않는' in text:
                words_to_format.append('않는')
            if '않은' in text:
                words_to_format.append('않은')
            if '틀린' in text:
                words_to_format.append('틀린')
            if '2개' in text:
                words_to_format.append('2개')
            if '3개' in text:
                words_to_format.append('3개')
            
            # Process text with formatting
            text_to_process = text
            last_idx = 0
            
            for match in pattern.finditer(text_to_process):
                start, end = match.span()
                
                # Add text before the match with Korean word formatting
                if start > last_idx:
                    text_segment = text_to_process[last_idx:start]
                    self.add_formatted_text(new_para, text_segment, words_to_format)
                
                group1 = match.group(1)  # <...>
                group2 = match.group(2)  # [[...]]
                group3 = match.group(3)  # [...]
                group4 = match.group(4)  # {...}
                group5 = match.group(5)  # (number)❮ ... ❯
                
                if group1 is not None:
                    # Apply bold to <보기> including angle brackets (no underline)
                    if group1 == '보기':
                        run = new_para.add_run('<' + group1 + '>')
                        run.bold = True
                    else:
                        run = new_para.add_run(group1)
                        run.bold = True
                        run.underline = True
                elif group2 is not None:
                    run = new_para.add_run(group2)
                    run.bold = True
                    run.underline = True
                elif group3 is not None:
                    # Check if it's a single letter A-E (multiple choice option)
                    if len(group3) == 1 and group3 in 'ABCDE':
                        # Leave [A], [B], [C], [D], [E] as-is without formatting
                        new_para.add_run('[' + group3 + ']')
                    # NEW: Check if it contains a forward slash
                    elif '/' in group3:
                        # Apply bold to the entire [text / text] including brackets
                        run = new_para.add_run('[' + group3 + ']')
                        run.bold = True
                    # Apply bold to [보기] including brackets
                    elif group3 == '보기':
                        run = new_para.add_run('[' + group3 + ']')
                        run.bold = True
                    else:
                        # Handle nested curly braces for other [...] patterns
                        self.process_nested_formatting(new_para, group3, inner_curly_pattern,
                                                    bold=True, underline=True)
                elif group4 is not None:
                    # Handle italic
                    run = new_para.add_run(group4)
                    run.italic = True
                elif group5 is not None:
                    # Handle vocabulary questions - bold only
                    run = new_para.add_run(group5)
                    run.bold = True
                
                last_idx = end
            
            # Add any remaining text with Korean word formatting
            if last_idx < len(text_to_process):
                text_segment = text_to_process[last_idx:]
                self.add_formatted_text(new_para, text_segment, words_to_format)
        
        # Remove the original paragraph that was passed in
        paragraph._element.getparent().remove(paragraph._element)


    def split_text_at_circled_numbers(self, text):
        """
        Split text at circled numbers, inserting newlines before each circled number (except the first one).

        Circled number ranges:
        - ① to ⑳ (1-20): chr(9312) to chr(9331)
        - ㉑ to ㉟ (21-35): chr(12881) to chr(12895)
        - ㊱ to ㊿ (36-50): chr(12977) to chr(12991)
        """
        # Build regex pattern for all circled numbers (① to ⑳, ㉑ to ㉟, ㊱ to ㊿)
        circled_chars = []
        # ① to ⑳ (1-20)
        circled_chars.extend([chr(9311 + n) for n in range(1, 21)])
        # ㉑ to ㉟ (21-35)
        circled_chars.extend([chr(12860 + n) for n in range(21, 36)])
        # ㊱ to ㊿ (36-50)
        circled_chars.extend([chr(12941 + n) for n in range(36, 51)])

        # Create regex pattern that matches any circled number
        pattern = '(' + '|'.join(re.escape(c) for c in circled_chars) + ')'

        # Split text, keeping the circled numbers as separate elements
        parts = re.split(pattern, text)

        result = []
        for i, part in enumerate(parts):
            if not part:
                continue
            # If this is a circled number and it's not at the beginning, prepend newline
            if part in circled_chars:
                if result:  # Not the first element
                    result.append('\n')
                result.append(part)
            else:
                result.append(part)

        return ''.join(result)


    def apply_jimun_analysis_formatting(self, paragraph, text_list):
        """Apply custom jimun_analysis formatting to brackets in a paragraph"""
        pattern = re.compile(
            r"\<([^>]*)\>"       # group(1): <...>
            r"|\[\[([^\]]*)\]\]" # group(2): [[...]]
            r"|\[([^\]]*)\]"     # group(3): [...]
            r"|\{([^}]*)\}"      # group(4): {...}
            r"|(\(\d+\)❮[^❯]*❯)" # group(5): (number)❮ ... ❯ for vocabulary questions
        )

        inner_square_pattern = re.compile(r"\[([^\]]+)\]")
        inner_curly_pattern = re.compile(r"\{([^}]+)\}")

        # Process all paragraphs and determine which need borders
        paragraphs_to_create = []
        for text in text_list:
            is_right_aligned = text.startswith('*')
            style = 'Normal_right' if is_right_aligned else 'Normal_modified_box'
            paragraphs_to_create.append((text, style, is_right_aligned))

        # Create paragraphs with appropriate borders
        for idx, (text, style, is_right_aligned) in enumerate(paragraphs_to_create):
            # Create new paragraph with appropriate style
            new_para = paragraph._parent.add_paragraph(style=style)

            # Apply borders based on position
            if idx == 0:
                # First paragraph - all borders
                self.add_paragraph_border(new_para, top=True, bottom=(len(paragraphs_to_create) == 1),
                                        left=True, right=True)
            elif idx == len(paragraphs_to_create) - 1:
                # Last paragraph - only left, right, and bottom
                self.add_paragraph_border(new_para, top=False, bottom=True, left=True, right=True)
            else:
                # Middle paragraphs - only left and right
                self.add_paragraph_border(new_para, top=False, bottom=False, left=True, right=True)

            # Apply padding
            new_para.paragraph_format.left_indent = Pt(5)
            new_para.paragraph_format.right_indent = Pt(5)
            new_para.paragraph_format.space_before = Pt(0)
            new_para.paragraph_format.space_after = Pt(0)

            # Apply custom jimun_analysis line spacing
            new_para.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
            new_para.paragraph_format.line_spacing = self.jimun_analysis_line_spacing

            # Determine which Korean words need to be formatted in this text
            words_to_format = []
            if '없는' in text:
                words_to_format.append('없는')
            if '않는' in text:
                words_to_format.append('않는')
            if '않은' in text:
                words_to_format.append('않은')
            if '틀린' in text:
                words_to_format.append('틀린')
            if '2개' in text:
                words_to_format.append('2개')
            if '3개' in text:
                words_to_format.append('3개')

            # Process text with formatting
            text_to_process = text
            last_idx = 0

            # Convert hex color to RGB
            rgb_color = self.hex_to_rgb(self.jimun_analysis_font_color)

            for match in pattern.finditer(text_to_process):
                start, end = match.span()

                # Add text before the match with Korean word formatting
                if start > last_idx:
                    text_segment = text_to_process[last_idx:start]
                    self.add_jimun_analysis_formatted_text(new_para, text_segment, words_to_format, rgb_color)

                group1 = match.group(1)  # <...>
                group2 = match.group(2)  # [[...]]
                group3 = match.group(3)  # [...]
                group4 = match.group(4)  # {...}
                group5 = match.group(5)  # (number)❮ ... ❯

                if group1 is not None:
                    # Apply bold to <보기> including angle brackets (no underline)
                    if group1 == '보기':
                        run = new_para.add_run('<' + group1 + '>')
                        run.bold = True
                    else:
                        run = new_para.add_run(group1)
                        run.bold = True
                        run.underline = True
                    # Apply custom font and color
                    run.font.name = self.jimun_analysis_font
                    run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                    run.font.size = Pt(self.jimun_analysis_font_size)
                    run.font.color.rgb = RGBColor(*rgb_color)
                elif group2 is not None:
                    run = new_para.add_run(group2)
                    run.bold = True
                    run.underline = True
                    # Apply custom font and color
                    run.font.name = self.jimun_analysis_font
                    run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                    run.font.size = Pt(self.jimun_analysis_font_size)
                    run.font.color.rgb = RGBColor(*rgb_color)
                elif group3 is not None:
                    # Check if it's a single letter A-E (multiple choice option)
                    if len(group3) == 1 and group3 in 'ABCDE':
                        # Leave [A], [B], [C], [D], [E] as-is without formatting
                        run = new_para.add_run('[' + group3 + ']')
                        # Apply custom font and color
                        run.font.name = self.jimun_analysis_font
                        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                        run.font.size = Pt(self.jimun_analysis_font_size)
                        run.font.color.rgb = RGBColor(*rgb_color)
                    # NEW: Check if it contains a forward slash
                    elif '/' in group3:
                        # Apply bold to the entire [text / text] including brackets
                        run = new_para.add_run('[' + group3 + ']')
                        run.bold = True
                        # Apply custom font and color
                        run.font.name = self.jimun_analysis_font
                        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                        run.font.size = Pt(self.jimun_analysis_font_size)
                        run.font.color.rgb = RGBColor(*rgb_color)
                    # Apply bold to [보기] including brackets
                    elif group3 == '보기':
                        run = new_para.add_run('[' + group3 + ']')
                        run.bold = True
                        # Apply custom font and color
                        run.font.name = self.jimun_analysis_font
                        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                        run.font.size = Pt(self.jimun_analysis_font_size)
                        run.font.color.rgb = RGBColor(*rgb_color)
                    else:
                        # Handle nested curly braces for other [...] patterns
                        self.process_jimun_analysis_nested_formatting(new_para, group3, inner_curly_pattern,
                                                    bold=True, underline=True, rgb_color=rgb_color)
                elif group4 is not None:
                    # Handle italic
                    run = new_para.add_run(group4)
                    run.italic = True
                    # Apply custom font and color
                    run.font.name = self.jimun_analysis_font
                    run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                    run.font.size = Pt(self.jimun_analysis_font_size)
                    run.font.color.rgb = RGBColor(*rgb_color)
                elif group5 is not None:
                    # Handle vocabulary questions - bold only
                    run = new_para.add_run(group5)
                    run.bold = True
                    # Apply custom font and color
                    run.font.name = self.jimun_analysis_font
                    run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                    run.font.size = Pt(self.jimun_analysis_font_size)
                    run.font.color.rgb = RGBColor(*rgb_color)

                last_idx = end

            # Add any remaining text with Korean word formatting
            if last_idx < len(text_to_process):
                text_segment = text_to_process[last_idx:]
                self.add_jimun_analysis_formatted_text(new_para, text_segment, words_to_format, rgb_color)

        # Remove the original paragraph that was passed in
        paragraph._element.getparent().remove(paragraph._element)


    def add_jimun_analysis_formatted_text(self, paragraph, text, words_to_format, rgb_color):
        """Helper method to add text with jimun_analysis formatting"""
        # Normalize and strip invisibles coming from Excel
        t = unicodedata.normalize("NFC", text)
        t = re.sub(r'[\u200B-\u200D\uFEFF\u00A0]', ' ', t)  # ZW*, BOM, NBSP -> space

        words = [unicodedata.normalize("NFC", w) for w in words_to_format if w]
        if not words:
            run = paragraph.add_run(t)
            # Apply custom font and color
            run.font.name = self.jimun_analysis_font
            run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
            run.font.size = Pt(self.jimun_analysis_font_size)
            run.font.color.rgb = RGBColor(*rgb_color)
            return

        # Longest-first to avoid partial-overlap issues (e.g., '않는' vs '않은')
        words = sorted(set(words), key=len, reverse=True)
        pat = re.compile("|".join(re.escape(w) for w in words))

        pos = 0
        for m in pat.finditer(t):
            if m.start() > pos:
                run = paragraph.add_run(t[pos:m.start()])
                # Apply custom font and color
                run.font.name = self.jimun_analysis_font
                run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                run.font.size = Pt(self.jimun_analysis_font_size)
                run.font.color.rgb = RGBColor(*rgb_color)
            run = paragraph.add_run(m.group(0))
            run.bold = True
            run.underline = True
            # Apply custom font and color
            run.font.name = self.jimun_analysis_font
            run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
            run.font.size = Pt(self.jimun_analysis_font_size)
            run.font.color.rgb = RGBColor(*rgb_color)
            pos = m.end()
        if pos < len(t):
            run = paragraph.add_run(t[pos:])
            # Apply custom font and color
            run.font.name = self.jimun_analysis_font
            run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
            run.font.size = Pt(self.jimun_analysis_font_size)
            run.font.color.rgb = RGBColor(*rgb_color)


    def process_jimun_analysis_nested_formatting(self, paragraph, text, inner_pattern, bold=False, underline=False, rgb_color=None):
        """Process nested formatting patterns with jimun_analysis formatting"""
        last_inner_idx = 0
        found_inner = False

        for inner_match in inner_pattern.finditer(text):
            found_inner = True
            inner_start, inner_end = inner_match.span()

            if inner_start > last_inner_idx:
                run = paragraph.add_run(text[last_inner_idx:inner_start])
                run.bold = bold
                run.underline = underline
                # Apply custom font and color
                run.font.name = self.jimun_analysis_font
                run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                run.font.size = Pt(self.jimun_analysis_font_size)
                run.font.color.rgb = RGBColor(*rgb_color)

            run = paragraph.add_run(inner_match.group(1))
            run.bold = True
            run.italic = True
            run.underline = True
            # Apply custom font and color
            run.font.name = self.jimun_analysis_font
            run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
            run.font.size = Pt(self.jimun_analysis_font_size)
            run.font.color.rgb = RGBColor(*rgb_color)

            last_inner_idx = inner_end

        if found_inner:
            if last_inner_idx < len(text):
                run = paragraph.add_run(text[last_inner_idx:])
                run.bold = bold
                run.underline = underline
                # Apply custom font and color
                run.font.name = self.jimun_analysis_font
                run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                run.font.size = Pt(self.jimun_analysis_font_size)
                run.font.color.rgb = RGBColor(*rgb_color)
        else:
            run = paragraph.add_run(text)
            run.bold = bold
            run.underline = underline
            # Apply custom font and color
            run.font.name = self.jimun_analysis_font
            run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
            run.font.size = Pt(self.jimun_analysis_font_size)
            run.font.color.rgb = RGBColor(*rgb_color)


    def _tokenize_keep_brackets(self, text):
        """
        Tokenize text, keeping bracketed phrases [...] and <...> as single tokens.
        Filters out punctuation. Returns list of tokens.
        """
        bracket_pattern = r'(\[[^\]]+\]|<[^>]+>)'
        parts = re.split(bracket_pattern, text)

        tokens = []
        for part in parts:
            part = part.strip()
            if not part:
                continue
            # If it's a bracketed phrase, keep as-is
            if part.startswith('[') or part.startswith('<'):
                tokens.append(part)
            else:
                # Tokenize the rest with spaCy (or fallback to split)
                if self.nlp is not None:
                    doc = self.nlp(part)
                    # Filter out punctuation tokens and numbers
                    tokens.extend([token.text for token in doc if not token.is_space and not token.is_punct and not token.like_num])
                else:
                    # Fallback: split and filter common punctuation and numbers
                    words = part.split()
                    tokens.extend([w.strip('.,!?;:') for w in words if w.strip('.,!?;:') and not w.strip('.,!?;:').isdigit()])

        return tokens


    def _lemmatize_tokens(self, sentence):
        """
        Tokenize, remove articles/pronouns, lemmatize, keep brackets in-place.
        Filters punctuation. Returns list of tokens.
        """
        # Tokenize with brackets preserved (punctuation already filtered)
        tokens = self._tokenize_keep_brackets(sentence)

        # Fallback if spaCy not loaded
        if self.nlp is None:
            # Simple fallback: lowercase non-bracket tokens
            return [
                t if (t.startswith('[') or t.startswith('<')) else t.lower()
                for t in tokens
            ]

        # Process each token
        processed_tokens = []
        for token_text in tokens:
            # Keep bracket tokens as-is
            if token_text.startswith('[') or token_text.startswith('<'):
                processed_tokens.append(token_text)
                continue

            # Process with spaCy
            doc = self.nlp(token_text)
            for token in doc:
                # Skip punctuation (already filtered, but double-check)
                if token.is_punct:
                    continue
                # Skip articles
                if token.pos_ == "DET" and token.lemma_.lower() in ["a", "an", "the"]:
                    continue
                # Skip pronouns
                if token.pos_ == "PRON":
                    continue
                # Preserve proper nouns (original case)
                if token.pos_ == "PROPN":
                    processed_tokens.append(token.text)
                # Lemmatize everything else (lowercase)
                else:
                    processed_tokens.append(token.lemma_.lower())

        return processed_tokens


    def add_paragraph_with_formatting(self, doc, text, style='Normal_modified', margin=False):
        """Add a paragraph with bold/underline/italic formatting and Korean word formatting"""

        para = doc.add_paragraph(style=style)

        # Add margin if requested
        if margin:
            para.paragraph_format.space_before = Pt(6)
            para.paragraph_format.space_after = Pt(6)
        
        # Check if text starts with numbered symbols ①②③④⑤
        skip_korean_formatting = False
        if text and (text[0] in '①②③④⑤' or text.startswith(('(1)', '(2)', '(3)', '(4)', '(5)'))):
            skip_korean_formatting = True
        
        # Determine which Korean words need to be formatted (only if not skipping)
        words_to_format = []
        if not skip_korean_formatting:
            if '없는' in text:
                words_to_format.append('없는')
            if '않는' in text:
                words_to_format.append('않는')
            if '않은' in text:
                words_to_format.append('않은')
            if '틀린' in text:
                words_to_format.append('틀린')
            if '다른' in text:
                words_to_format.append('다른')
            if '어색한' in text:
                words_to_format.append('어색한')
            if '잘못된' in text:
                words_to_format.append('잘못된')
            if '2개' in text:
                words_to_format.append('2개')
            if '두 개' in text:
                words_to_format.append('두 개')
            if '3개' in text:
                words_to_format.append('3개')
            if '세 개' in text:
                words_to_format.append('세 개')
        
        # Pattern to match all formatting types
        pattern = re.compile(
            r"\<([^>]*)\>"       # group(1): <...> for bold+underline
            r"|\[\[([^\]]*)\]\]" # group(2): [[...]] for bold+underline
            r"|\[([^\]]*)\]"     # group(3): [...] for bold+underline (with exceptions)
            r"|\{([^}]*)\}"      # group(4): {...} for italic
            r"|(\(\d+\)❮[^❯]*❯)" # group(5): (number)❮...❯ for bold only
        )
        
        # Pattern for nested formatting
        inner_curly_pattern = re.compile(r"\{([^}]+)\}")
        
        last_idx = 0
        
        for match in pattern.finditer(text):
            start, end = match.span()
            
            # Add text before the match with Korean word formatting
            if start > last_idx:
                text_segment = text[last_idx:start]
                self.add_formatted_text(para, text_segment, words_to_format)
            
            group1 = match.group(1)  # <...>
            group2 = match.group(2)  # [[...]]
            group3 = match.group(3)  # [...]
            group4 = match.group(4)  # {...}
            group5 = match.group(5)  # (number)❮...❯
            
            if group1 is not None:
                # <...> - bold and underline
                # Apply bold to <보기> including angle brackets (no underline)
                if group1 == '보기':
                    run = para.add_run('<' + group1 + '>')
                    run.bold = True
                else:
                    run = para.add_run(group1)
                    run.bold = True
                    run.underline = True
            elif group2 is not None:
                # [[...]] - bold and underline
                run = para.add_run(group2)
                run.bold = True
                run.underline = True
            elif group3 is not None:
                # [...] - with special cases
                # Check if it's a single letter A-E (multiple choice option)
                if len(group3) == 1 and group3 in 'ABCDE':
                    # Leave [A], [B], [C], [D], [E] as-is without formatting
                    para.add_run('[' + group3 + ']')
                # Check if it contains a forward slash
                elif '/' in group3:
                    # Apply bold to the entire [text / text] including brackets
                    run = para.add_run('[' + group3 + ']')
                    run.bold = True
                # Apply bold to [보기] including brackets
                elif group3 == '보기':
                    run = para.add_run('[' + group3 + ']')
                    run.bold = True
                else:
                    # Handle nested curly braces for other [...] patterns
                    # Apply bold and underline to content
                    self.process_nested_formatting(para, group3, inner_curly_pattern,
                                                bold=True, underline=True)
            elif group4 is not None:
                # {...} - italic
                run = para.add_run(group4)
                run.italic = True
            elif group5 is not None:
                # (number)❮...❯ - bold only
                run = para.add_run(group5)
                run.bold = True
            
            last_idx = end
        
        # Add any remaining text with Korean word formatting
        if last_idx < len(text):
            text_segment = text[last_idx:]
            self.add_formatted_text(para, text_segment, words_to_format)
        
        return para




    def add_paragraph_with_formatting동형문제(self, doc, text, style='Normal_modified', margin=False, is_first_paragraph=False):
        """Add a paragraph with bold/underline/italic formatting and Korean word formatting"""

        # Replace '괄호 한' with '밑줄 친'
        text = text.replace('괄호 한', '밑줄 친')

        para = doc.add_paragraph(style=style)

        # Add margin if requested
        if margin:
            para.paragraph_format.space_before = Pt(6)
            para.paragraph_format.space_after = Pt(6)

        # Check if text starts with numbered symbols ①②③④⑤
        skip_korean_formatting = False
        if text and text[0] in '①②③④⑤':
            skip_korean_formatting = True

        # Determine which Korean words need to be formatted (only in first paragraph and if not skipping)
        words_to_format = []
        if not skip_korean_formatting and is_first_paragraph:
            if '없는' in text:
                words_to_format.append('없는')
            if '않는' in text:
                words_to_format.append('않는')
            if '않은' in text:
                words_to_format.append('않은')
            if '틀린' in text:
                words_to_format.append('틀린')
            if '다른' in text:
                words_to_format.append('다른')
            if '어색한' in text:
                words_to_format.append('어색한')
            if '잘못된' in text:
                words_to_format.append('잘못된')
            if '2개' in text:
                words_to_format.append('2개')
            if '두 개' in text:
                words_to_format.append('두 개')
            if '3개' in text:
                words_to_format.append('3개')
            if '세 개' in text:
                words_to_format.append('세 개')
            if '모두' in text:
                words_to_format.append('모두')
        
        # Pattern to match italic formatting only
        pattern = re.compile(r"\{([^}]*)\}")  # {...} for italic

        last_idx = 0

        for match in pattern.finditer(text):
            start, end = match.span()

            # Add text before the match with Korean word formatting
            if start > last_idx:
                text_segment = text[last_idx:start]
                self.add_formatted_text(para, text_segment, words_to_format)

            # {...} - italic
            content = match.group(1)
            run = para.add_run(content)
            run.italic = True

            
            last_idx = end
        
        # Add any remaining text with Korean word formatting
        if last_idx < len(text):
            text_segment = text[last_idx:]
            self.add_formatted_text(para, text_segment, words_to_format)
        
        return para





    def add_paragraph_without_formatting(self, doc, text, style='Normal_modified', margin=False):
        """Add a paragraph without bold/underline formatting - for 동형문제 non-first paragraphs.

        Only italic formatting for {...} is applied.
        Bold/underline markers like <>, [], [[]] are preserved as plain text.

        Exception: [...] gets bold/underline when preceded by:
        - Circled numbers: ①②③...
        - Circled letters: ⓐⓑⓒ...
        - Parenthesized letters: (a), (b), (c)... or (A), (B), (C)...
        - Parenthesized Korean consonants: (ㄱ), (ㄴ), (ㄷ)...
        - Parenthesized Korean syllables: (가), (나), (다)...
        """

        para = doc.add_paragraph(style=style)

        # Add margin if requested
        if margin:
            para.paragraph_format.space_before = Pt(6)
            para.paragraph_format.space_after = Pt(6)

        # Define prefix patterns for bold/underline brackets
        # Circled numbers (①-⑳) and circled letters (ⓐ-ⓩ)
        circled_chars = '①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳ⓐⓑⓒⓓⓔⓕⓖⓗⓘⓙⓚⓛⓜⓝⓞⓟⓠⓡⓢⓣⓤⓥⓦⓧⓨⓩⒶⒷⒸⒹⒺⒻⒼⒽⒾⒿⓀⓁⓂⓃⓄⓅⓆⓇⓈⓉⓊⓋⓌⓍⓎⓏ'
        # Korean consonants (ㄱ-ㅎ) and syllables (가나다라마바사아자차카타파하)
        korean_chars = 'ㄱㄴㄷㄹㅁㅂㅅㅇㅈㅊㅋㅌㅍㅎ가나다라마바사아자차카타파하'

        # Pattern to match:
        # 1. Circled prefix + [...] for bold+underline
        # 2. Parenthesized prefix + [...] for bold+underline
        # 3. {...} for italic only
        pattern = re.compile(
            rf"([{circled_chars}])\[([^\]]*)\]"           # group(1,2): circled + content
            rf"|(\([a-zA-Z{korean_chars}]\))\[([^\]]*)\]" # group(3,4): paren prefix + content
            r"|\{([^}]*)\}"                               # group(5): {...} for italic
        )

        # Pattern for nested formatting (curly braces inside brackets)
        inner_curly_pattern = re.compile(r"\{([^}]+)\}")

        last_idx = 0

        for match in pattern.finditer(text):
            start, end = match.span()

            # Add text before the match (plain text)
            if start > last_idx:
                text_segment = text[last_idx:start]
                para.add_run(text_segment)

            circled_prefix = match.group(1)   # Circled number/letter prefix
            circled_content = match.group(2)  # Content after circled prefix
            paren_prefix = match.group(3)     # Parenthesized prefix
            paren_content = match.group(4)    # Content after paren prefix
            curly_content = match.group(5)    # {...} content

            if circled_prefix is not None:
                # Circled prefix + [...] - add prefix as plain, content as bold+underline
                para.add_run(circled_prefix)
                self.process_nested_formatting(para, circled_content, inner_curly_pattern,
                                              bold=True, underline=True)
            elif paren_prefix is not None:
                # Parenthesized prefix + [...] - add prefix as plain, content as bold+underline
                para.add_run(paren_prefix)
                self.process_nested_formatting(para, paren_content, inner_curly_pattern,
                                              bold=True, underline=True)
            elif curly_content is not None:
                # {...} - italic only (brackets removed)
                run = para.add_run(curly_content)
                run.italic = True

            last_idx = end

        # Add any remaining text (plain text)
        if last_idx < len(text):
            text_segment = text[last_idx:]
            para.add_run(text_segment)

        return para


    def add_paragraph_for_교재제작(self, doc, text, style='Normal_modified'):
        """Add a paragraph with italic formatting for {...}"""

        para = doc.add_paragraph(style=style)

        # Pattern to match only curly braces
        pattern = re.compile(r"\{([^}]*)\}")

        last_idx = 0

        for match in pattern.finditer(text):
            start, end = match.span()

            # Add text before the match (plain text)
            if start > last_idx:
                text_segment = text[last_idx:start]
                para.add_run(text_segment)

            # {...} - italic only (brackets removed)
            run = para.add_run(match.group(1))
            run.italic = True

            last_idx = end

        # Add any remaining text (plain text)
        if last_idx < len(text):
            text_segment = text[last_idx:]
            para.add_run(text_segment)

        return para


    def add_paragraph_with_formatting_old(self, doc, text, style='Normal_modified'):
        """Add a paragraph with bold/underline/italic formatting and Korean word formatting"""
        
        para = doc.add_paragraph(style=style)
        
        # Determine which Korean words need to be formatted
        words_to_format = []
        if '없는' in text:
            words_to_format.append('없는')
        if '않는' in text:
            words_to_format.append('않는')
        if '않은' in text:
            words_to_format.append('않은')
        if '틀린' in text:
            words_to_format.append('틀린')
        if '다른' in text:
            words_to_format.append('다른')
        if '어색한' in text:
            words_to_format.append('어색한')
        if '2개' in text:
            words_to_format.append('2개')
        if '두 개' in text:
            words_to_format.append('두 개')
        if '3개' in text:
            words_to_format.append('3개')
        if '세 개' in text:
            words_to_format.append('세 개')
        
        # Pattern to match all formatting types
        pattern = re.compile(
            r"\<([^>]*)\>"       # group(1): <...> for bold+underline
            r"|\[\[([^\]]*)\]\]" # group(2): [[...]] for bold+underline
            r"|\[([^\]]*)\]"     # group(3): [...] for bold+underline (with exceptions)
            r"|\{([^}]*)\}"      # group(4): {...} for italic
            r"|(\(\d+\)❮[^❯]*❯)" # group(5): (number)❮...❯ for bold only
        )
        
        # Pattern for nested formatting
        inner_curly_pattern = re.compile(r"\{([^}]+)\}")
        
        last_idx = 0
        
        for match in pattern.finditer(text):
            start, end = match.span()
            
            # Add text before the match with Korean word formatting
            if start > last_idx:
                text_segment = text[last_idx:start]
                self.add_formatted_text(para, text_segment, words_to_format)
            
            group1 = match.group(1)  # <...>
            group2 = match.group(2)  # [[...]]
            group3 = match.group(3)  # [...]
            group4 = match.group(4)  # {...}
            group5 = match.group(5)  # (number)❮...❯
            
            if group1 is not None:
                # <...> - bold and underline
                # Apply bold to <보기> including angle brackets (no underline)
                if group1 == '보기':
                    run = para.add_run('<' + group1 + '>')
                    run.bold = True
                else:
                    run = para.add_run(group1)
                    run.bold = True
                    run.underline = True
            elif group2 is not None:
                # [[...]] - bold and underline
                run = para.add_run(group2)
                run.bold = True
                run.underline = True
            elif group3 is not None:
                # [...] - with special cases
                # Check if it's a single letter A-E (multiple choice option)
                if len(group3) == 1 and group3 in 'ABCDE':
                    # Leave [A], [B], [C], [D], [E] as-is without formatting
                    para.add_run('[' + group3 + ']')
                # Check if it contains a forward slash
                elif '/' in group3:
                    # Apply bold to the entire [text / text] including brackets
                    run = para.add_run('[' + group3 + ']')
                    run.bold = True
                # Apply bold to [보기] including brackets
                elif group3 == '보기':
                    run = para.add_run('[' + group3 + ']')
                    run.bold = True
                else:
                    # Handle nested curly braces for other [...] patterns
                    # Apply bold and underline to content
                    self.process_nested_formatting(para, group3, inner_curly_pattern,
                                                bold=True, underline=True)
            elif group4 is not None:
                # {...} - italic
                run = para.add_run(group4)
                run.italic = True
            elif group5 is not None:
                # (number)❮...❯ - bold only
                run = para.add_run(group5)
                run.bold = True
            
            last_idx = end
        
        # Add any remaining text with Korean word formatting
        if last_idx < len(text):
            text_segment = text[last_idx:]
            self.add_formatted_text(para, text_segment, words_to_format)
        
        return para





    def create_bordered_container(self, doc):
        """Create a single-cell table that acts as a bordered container for multiple paragraphs"""
        table = doc.add_table(rows=1, cols=1)
        cell = table.cell(0, 0)
        
        # Set border for the cell
        self.set_cell_border(
            cell,
            top={"sz": 10, "val": "single", "color": "000000"},
            bottom={"sz": 10, "val": "single", "color": "000000"},
            left={"sz": 10, "val": "single", "color": "000000"},
            right={"sz": 10, "val": "single", "color": "000000"}
        )
        
        # Set cell margins
        self.set_cell_margins(cell, top=5, bottom=5, left=5, right=5)

        # Remove default paragraph
        cell.paragraphs[0].clear()

        return cell


    def create_two_column_passage_table(self, doc, left_ratio=7, right_ratio=3):
        """
        Create a two-column table for passage display with interpretation on the right.

        Args:
            doc: Document object
            left_ratio: Ratio for left column (default 7)
            right_ratio: Ratio for right column (default 3)

        Returns:
            tuple: (left_cell, right_cell)
        """
        table = doc.add_table(rows=1, cols=2)

        # Disable autofit to allow explicit width control
        table.autofit = False

        # Calculate column widths based on the *actual* usable page width
        # (page_width - left_margin - right_margin) so it always fills the document width.
        total_ratio = left_ratio + right_ratio
        try:
            section = doc.sections[-1]
            total_width = section.page_width - section.left_margin - section.right_margin
        except Exception:
            # Fallback for safety; should not happen in normal usage
            total_width = Cm(16)

        # Keep exact sum == total_width (avoid rounding drift)
        left_width = int(total_width * left_ratio / total_ratio)
        right_width = int(total_width) - left_width

        left_cell = table.cell(0, 0)
        right_cell = table.cell(0, 1)

        # Set column widths on both columns AND cells for reliability
        table.columns[0].width = left_width
        table.columns[1].width = right_width
        left_cell.width = left_width
        right_cell.width = right_width

        # Set borders for both cells
        for cell in [left_cell, right_cell]:
            self.set_cell_border(
                cell,
                top={"sz": 10, "val": "single", "color": "000000"},
                bottom={"sz": 10, "val": "single", "color": "000000"},
                left={"sz": 10, "val": "single", "color": "000000"},
                right={"sz": 10, "val": "single", "color": "000000"}
            )
            # Set cell margins
            self.set_cell_margins(cell, top=5, bottom=5, left=5, right=5)
            # Clear default paragraph
            cell.paragraphs[0].clear()

        return left_cell, right_cell


    def add_circled_numbers_to_sentences(self, text):
        """
        Add circled numbers to the beginning of each sentence.
        Used for 한글해석 when displayed alongside 지문분석.

        Args:
            text: The text to process

        Returns:
            Text with circled numbers prepended to each sentence
        """
        def get_circled_number(n):
            """Convert number to circled Unicode character"""
            if 1 <= n <= 20:
                return chr(9311 + n)  # ① to ⑳
            elif 21 <= n <= 35:
                return chr(12860 + n)  # ㉑ to ㉟
            elif 36 <= n <= 50:
                return chr(12941 + n)  # ㊱ to ㊿
            else:
                return f"({n})"  # Fallback for numbers > 50

        # Pre-processing replacements to handle abbreviations
        replacements = {
            r'\b(Mr|Mrs|Ms|Mt|Dr|Prof|St|Ave|Blvd|Capt|Col|Gen|Lt|Pvt|Jr|Sr|Rev|Etc|Inc|Ltd|Co|esp|Fig|Op|No|Vol|Rd|Tel|Gov|Esq|Maj|Sgt|Cpl|Adm|Mdm|Fr|Bros|Dept|Assn|Univ|Inst|Corp|Rep|Sen|Ed|Fwd|Mfg|Jan|Feb|Mar|Apr|Aug|Sept|Oct|Nov|Dec|CEO|CFO|CIO|vs)\.\s+': r'\1• ',
            r'(\d+|sq|cu)\s+(in|ft|oz|lb|ct|yr|wk|mo)\.\s+': r'\1 \2• ',
            r'(?<=\.[A-Za-z])\.\s+': '• ',
        }
        replacement_patterns = [
            (". ", ".\n"), (".) ", ".)\n"), ("? ", "?\n"), ("?) ", "?)\n"), ("! ", "!\n"), ("!) ", "!)\n"), #기본
            ('." ', '."\n'), ('?" ', '?"\n'), ('!" ', '!"\n'), #큰따옴표
            (".' ", ".'\n"), ("?' ", "?'\n"), ("!' ", "!'\n"), #작은따옴표
            ('.’ ', ".’\n"), ('?’ ', "?’\n"), ('!’ ', '!’\n'), #tick
            (".” ", ".”\n"), ("?” ", "?”\n"), ("!” ", "!”\n") #double tick
        ]

        # Normalize line endings
        normalized_text = text.replace('\r\n', '\n').replace('\r', '\n')

        parts = re.split(r'(\n+)', normalized_text)
        processed_parts = []
        sentence_idx = 1

        for part in parts:
            if re.fullmatch(r'\n+', part):
                processed_parts.append(part)
                continue

            # Apply abbreviation protection
            protected_part = part
            for pattern, repl in replacements.items():
                protected_part = re.sub(pattern, repl, protected_part)

            # Split into sentences
            for old, new in replacement_patterns:
                protected_part = protected_part.replace(old, new)

            sentences = protected_part.split('\n')
            numbered_sentences = []
            for sentence in sentences:
                sentence = sentence.strip()
                if sentence:
                    # Restore the • back to .
                    sentence = sentence.replace('•', '.')
                    numbered_sentences.append(f"{get_circled_number(sentence_idx)}{sentence}")
                    sentence_idx += 1

            processed_parts.append('\n'.join(numbered_sentences))

        return ''.join(processed_parts)


    def apply_jimun_analysis_formatting_to_cell(self, cell, text_list):
        """
        Apply jimun_analysis formatting to text in a table cell.
        Similar to apply_jimun_analysis_formatting but for cells.

        Args:
            cell: Table cell to add content to
            text_list: List of text lines to add
        """
        pattern = re.compile(
            r"\<([^>]*)\>"       # group(1): <...>
            r"|\[\[([^\]]*)\]\]" # group(2): [[...]]
            r"|\[([^\]]*)\]"     # group(3): [...]
            r"|\{([^}]*)\}"      # group(4): {...}
        )

        # Convert hex color to RGB
        rgb_color = self.hex_to_rgb(self.jimun_analysis_font_color)

        for idx, text in enumerate(text_list):
            if idx == 0:
                para = cell.paragraphs[0]
            else:
                para = cell.add_paragraph()

            # Apply line spacing
            para.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
            para.paragraph_format.line_spacing = self.jimun_analysis_line_spacing
            para.paragraph_format.space_before = Pt(0)
            para.paragraph_format.space_after = Pt(0)

            # Process text with formatting
            last_idx = 0
            for match in pattern.finditer(text):
                start, end = match.span()

                # Add text before the match
                if start > last_idx:
                    run = para.add_run(text[last_idx:start])
                    run.font.name = self.jimun_analysis_font
                    run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                    run.font.size = Pt(self.jimun_analysis_font_size)
                    run.font.color.rgb = RGBColor(*rgb_color)

                # Add formatted text based on bracket type
                if match.group(1) or match.group(3):  # <...> or [...]
                    text_within = match.group(1) or match.group(3)
                    if text_within:
                        run = para.add_run(text_within)
                        run.bold = True
                        run.font.name = self.jimun_analysis_font
                        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                        run.font.size = Pt(self.jimun_analysis_font_size)
                        run.font.color.rgb = RGBColor(*rgb_color)
                elif match.group(2):  # [[...]]
                    text_within = match.group(2)
                    if text_within:
                        run = para.add_run(text_within)
                        run.underline = True
                        run.font.name = self.jimun_analysis_font
                        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                        run.font.size = Pt(self.jimun_analysis_font_size)
                        run.font.color.rgb = RGBColor(*rgb_color)
                elif match.group(4):  # {...}
                    text_within = match.group(4)
                    if text_within:
                        run = para.add_run(text_within)
                        run.underline = True
                        run.font.name = self.jimun_analysis_font
                        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                        run.font.size = Pt(self.jimun_analysis_font_size)
                        run.font.color.rgb = RGBColor(*rgb_color)

                last_idx = end

            # Add remaining text
            if last_idx < len(text):
                run = para.add_run(text[last_idx:])
                run.font.name = self.jimun_analysis_font
                run._element.rPr.rFonts.set(qn('w:eastAsia'), self.jimun_analysis_font)
                run.font.size = Pt(self.jimun_analysis_font_size)
                run.font.color.rgb = RGBColor(*rgb_color)


    def apply_hanjul_haesuk_formatting_to_cell(self, cell, text_list):
        """
        Apply hanjul_haesuk formatting to text in a table cell (right column).
        Uses hanjul_haesuk-specific font, size, color, and line spacing settings.

        Args:
            cell: Table cell to add content to
            text_list: List of text lines to add
        """
        pattern = re.compile(
            r"\<([^>]*)\>"       # group(1): <...>
            r"|\[\[([^\]]*)\]\]" # group(2): [[...]]
            r"|\[([^\]]*)\]"     # group(3): [...]
            r"|\{([^}]*)\}"      # group(4): {...}
        )

        # Convert hex color to RGB
        rgb_color = self.hex_to_rgb(self.hanjul_haesuk_font_color)

        for idx, text in enumerate(text_list):
            if idx == 0:
                para = cell.paragraphs[0]
            else:
                para = cell.add_paragraph()

            # Apply line spacing
            para.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
            para.paragraph_format.line_spacing = self.hanjul_haesuk_line_spacing
            para.paragraph_format.space_before = Pt(0)
            para.paragraph_format.space_after = Pt(0)

            # Process text with formatting
            last_idx = 0
            for match in pattern.finditer(text):
                start, end = match.span()

                # Add text before the match
                if start > last_idx:
                    run = para.add_run(text[last_idx:start])
                    run.font.name = self.hanjul_haesuk_font
                    run._element.rPr.rFonts.set(qn('w:eastAsia'), self.hanjul_haesuk_font)
                    run.font.size = Pt(self.hanjul_haesuk_font_size)
                    run.font.color.rgb = RGBColor(*rgb_color)

                # Add formatted text based on bracket type
                if match.group(1) or match.group(3):  # <...> or [...]
                    text_within = match.group(1) or match.group(3)
                    if text_within:
                        run = para.add_run(text_within)
                        run.bold = True
                        run.font.name = self.hanjul_haesuk_font
                        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.hanjul_haesuk_font)
                        run.font.size = Pt(self.hanjul_haesuk_font_size)
                        run.font.color.rgb = RGBColor(*rgb_color)
                elif match.group(2):  # [[...]]
                    text_within = match.group(2)
                    if text_within:
                        run = para.add_run(text_within)
                        run.underline = True
                        run.font.name = self.hanjul_haesuk_font
                        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.hanjul_haesuk_font)
                        run.font.size = Pt(self.hanjul_haesuk_font_size)
                        run.font.color.rgb = RGBColor(*rgb_color)
                elif match.group(4):  # {...}
                    text_within = match.group(4)
                    if text_within:
                        run = para.add_run(text_within)
                        run.underline = True
                        run.font.name = self.hanjul_haesuk_font
                        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.hanjul_haesuk_font)
                        run.font.size = Pt(self.hanjul_haesuk_font_size)
                        run.font.color.rgb = RGBColor(*rgb_color)

                last_idx = end

            # Add remaining text
            if last_idx < len(text):
                run = para.add_run(text[last_idx:])
                run.font.name = self.hanjul_haesuk_font
                run._element.rPr.rFonts.set(qn('w:eastAsia'), self.hanjul_haesuk_font)
                run.font.size = Pt(self.hanjul_haesuk_font_size)
                run.font.color.rgb = RGBColor(*rgb_color)


    def add_formatted_text_to_cell(self, cell, text, tab_stops_inches=None):
        """Add text to a cell with curly brace italic formatting

        Args:
            cell: The table cell to add text to
            text: The text to add (may contain {italic} formatting)
            tab_stops_inches: Optional list of tab stop positions in inches (e.g., [2.0, 4.0])
        """
        # Clear existing content
        cell.text = ""
        paragraph = cell.paragraphs[0]

        # Set tab stops if provided
        if tab_stops_inches:
            tab_stops = paragraph.paragraph_format.tab_stops
            for position in tab_stops_inches:
                tab_stops.add_tab_stop(Inches(position))

        # Pattern to match curly braces
        pattern = re.compile(r"\{([^}]*)\}")

        last_idx = 0
        for match in pattern.finditer(text):
            start, end = match.span()

            # Add text before the match
            if start > last_idx:
                paragraph.add_run(text[last_idx:start])

            # Add the text inside braces as italic (without the braces)
            text_inside_braces = match.group(1)
            run = paragraph.add_run(text_inside_braces)
            run.italic = True

            last_idx = end

        # Add any remaining text
        if last_idx < len(text):
            paragraph.add_run(text[last_idx:])

            

    def process_nested_formatting(self, paragraph, text, inner_pattern, bold=False, underline=False):
        """Process nested formatting patterns"""
        last_inner_idx = 0
        found_inner = False
        
        for inner_match in inner_pattern.finditer(text):
            found_inner = True
            inner_start, inner_end = inner_match.span()
            
            if inner_start > last_inner_idx:
                run = paragraph.add_run(text[last_inner_idx:inner_start])
                run.bold = bold
                run.underline = underline
            
            run = paragraph.add_run(inner_match.group(1))
            run.bold = True
            run.italic = True
            run.underline = True
            
            last_inner_idx = inner_end
        
        if found_inner:
            if last_inner_idx < len(text):
                run = paragraph.add_run(text[last_inner_idx:])
                run.bold = bold
                run.underline = underline
        else:
            run = paragraph.add_run(text)
            run.bold = bold
            run.underline = underline


    def add_paragraph_border(self, paragraph, top=True, bottom=True, left=True, right=True):
        """Add specific borders to a paragraph"""
        pPr = paragraph._p.get_or_add_pPr()
        pBdr = OxmlElement('w:pBdr')
        
        border_attrs = {
            'w:val': 'single',
            'w:sz': '5',
            'w:space': '6',
            'w:color': '000000'
        }
        
        borders_to_add = []
        if top:
            borders_to_add.append('top')
        if bottom:
            borders_to_add.append('bottom')
        if left:
            borders_to_add.append('left')
        if right:
            borders_to_add.append('right')
        
        for border_side in borders_to_add:
            border_element = OxmlElement(f'w:{border_side}')
            for attr, value in border_attrs.items():
                border_element.set(qn(attr), value)
            pBdr.append(border_element)
        
        pPr.append(pBdr)

    def set_underline_color(self, run, color='000000'):
        """Set underline with a specific color independent of font color"""
        run.underline = True
        rPr = run._r.get_or_add_rPr()
        # Create underline element with color
        u = OxmlElement('w:u')
        u.set(qn('w:val'), 'single')
        u.set(qn('w:color'), color)  # Underline color
        # Remove any existing underline element
        for child in list(rPr):
            if child.tag == qn('w:u'):
                rPr.remove(child)
        # Add the new underline with color
        rPr.append(u)


    def create_bordered_paragraph(self, doc, style='Normal_modified_box'):
        """Create an empty paragraph that will be replaced by apply_bold_underline_to_paragraph"""
        paragraph = doc.add_paragraph(style=style)
        # Don't add border here - it will be handled in apply_bold_underline_to_paragraph
        return paragraph


    def create_borderless_paragraph(self, doc, style='Normal_modified'):
        paragraph = doc.add_paragraph(style=style)
        return paragraph



    def add_formatted_text(self, paragraph, text, words_to_format):
        # Normalize and strip invisibles coming from Excel
        t = unicodedata.normalize("NFC", text)
        t = re.sub(r'[\u200B-\u200D\uFEFF\u00A0]', ' ', t)  # ZW*, BOM, NBSP -> space

        words = [unicodedata.normalize("NFC", w) for w in words_to_format if w]
        if not words:
            paragraph.add_run(t)
            return

        # Longest-first to avoid partial-overlap issues (e.g., '않는' vs '않은')
        words = sorted(set(words), key=len, reverse=True)
        pat = re.compile("|".join(re.escape(w) for w in words))

        pos = 0
        for m in pat.finditer(t):
            if m.start() > pos:
                paragraph.add_run(t[pos:m.start()])
            run = paragraph.add_run(m.group(0))
            run.bold = True
            run.underline = True
            pos = m.end()
        if pos < len(t):
            paragraph.add_run(t[pos:])




    def old_add_formatted_text(self, paragraph, text, words_to_format):
        
        remaining_text = text
        for word in words_to_format:
            if word in remaining_text:
                parts = remaining_text.split(word, 1)
                if parts[0]:
                    paragraph.add_run(parts[0])
                run = paragraph.add_run(word)
                run.bold = True
                run.underline = True
                remaining_text = parts[1]
            else:
                paragraph.add_run(remaining_text)
                remaining_text = ''
        if remaining_text:
            paragraph.add_run(remaining_text)    
        
    def 테이블없는본문_괄호에볼드밑줄더하기(self, paragraph, text):
        remaining_text = text
        
        # Process double brackets first
        while '[[' in remaining_text and ']]' in remaining_text:
            before_brackets, after_brackets = remaining_text.split('[[', 1)
            inside_brackets, remaining_text = after_brackets.split(']]', 1)
            
            # Add the text before the brackets normally
            if before_brackets:
                paragraph.add_run(before_brackets)
            
            # Add the text inside double brackets with bold and underline
            run = paragraph.add_run(inside_brackets)
            run.bold = True
            run.underline = True
        
        # Then process single brackets
        while '[' in remaining_text and ']' in remaining_text:
            before_brackets, after_brackets = remaining_text.split('[', 1)
            inside_brackets, remaining_text = after_brackets.split(']', 1)
            
            # Add the text before the brackets normally
            if before_brackets:
                paragraph.add_run(before_brackets)
            
            # Add the text inside single brackets with bold only
            run = paragraph.add_run(inside_brackets)
            run.bold = True
            run.underline = True

        
        # Finally, add any remaining text
        if remaining_text:
            paragraph.add_run(remaining_text)

    def 테이블없는본문_영작단어수에볼드밑줄더하기(self, paragraph, text):
        # Define the regex pattern to match one or two digits followed by '단어'
        pattern = re.compile(r'\d{1,2}단어')
        
        # Initialize the starting index
        last_end = 0
        
        # Iterate over all matches in the text
        for match in pattern.finditer(text):
            start, end = match.span()
            
            # Add the text before the matched pattern normally
            if start > last_end:
                normal_text = text[last_end:start]
                if normal_text:
                    paragraph.add_run(normal_text)
            
            # Add the matched text with bold and underline formatting
            matched_text = text[start:end]
            if matched_text:
                run = paragraph.add_run(matched_text)
                run.bold = True
                run.underline = True
            
            # Update the last_end to the end of the current match
            last_end = end
        
        # Add any remaining text after the last match
        if last_end < len(text):
            remaining_text = text[last_end:]
            if remaining_text:
                paragraph.add_run(remaining_text)


    def hex_to_rgb(self, hex_color):
        hex_color = hex_color.lstrip('#')  # Remove '#' at the start if present
        lv = len(hex_color)
        return tuple(int(hex_color[i:i + lv // 3], 16) for i in range(0, lv, lv // 3))

    def set_cell_paragraph_style(self, cell, style_name):
        for paragraph in cell.paragraphs:
            paragraph.style = style_name

    def set_cell_font_hdr(self, cell, font_name, font_size):
        font_size += 1
        for paragraph in cell.paragraphs:
            for run in paragraph.runs:
                run.font.name = font_name
                run._element.rPr.rFonts.set(qn('w:eastAsia'), font_name)  # Korean font setting
                run.font.size = Pt(font_size)
                run.font.bold = True
            paragraph.alignment = WD_PARAGRAPH_ALIGNMENT.CENTER
            paragraph.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
            paragraph.paragraph_format.line_spacing = 1.1

    def set_cell_border(self, cell, **kwargs):
        tc = cell._tc
        tcPr = tc.get_or_add_tcPr()

        for edge in ('top', 'bottom', 'left', 'right', 'insideH', 'insideV'):
            edge_data = kwargs.get(edge)
            if edge_data:
                tag = f'w:{edge}'
                element = OxmlElement(tag)
                element.set(qn('w:val'), edge_data.get('val', 'single'))
                element.set(qn('w:sz'), str(edge_data.get('sz', 5)))
                element.set(qn('w:space'), str(edge_data.get('space', 0)))
                element.set(qn('w:color'), edge_data.get('color', 'auto'))
                tcPr.append(element)

    def set_table_cell_margins(self, table, top=0, bottom=0, left=0, right=0):
        for row in table.rows:
            for cell in row.cells:
                self.set_cell_margins(cell, top, bottom, left, right)

    def set_cell_margins(self, cell, top=0, bottom=0, left=0, right=0):
        top_dxa = str(int(top * 20))
        bottom_dxa = str(int(bottom * 20))
        left_dxa = str(int(left * 20))
        right_dxa = str(int(right * 20))

        tcPr = cell._element.get_or_add_tcPr()
        tcMar = OxmlElement('w:tcMar')

        for mar, value in [('top', top_dxa), ('bottom', bottom_dxa),
                          ('left', left_dxa), ('right', right_dxa)]:
            mar_element = OxmlElement(f'w:{mar}')
            mar_element.set(qn('w:w'), value)
            mar_element.set(qn('w:type'), "dxa")
            tcMar.append(mar_element)

        tcPr.append(tcMar)

    def count_words(self, text):
        return len(text.split())

    def create_vocabulary_table_4col(self, doc, vocabulary_text):
        """
        Create a 4-column vocabulary table with checkboxes.
        Format: ☐ English | Korean | ☐ English | Korean
        Each row contains 2 vocabulary entries.
        """
        import math

        # Parse vocabulary entries
        lines = [line.strip() for line in vocabulary_text.split('\n') if line.strip()]
        vocab_entries = []

        for line in lines:
            # Split by colon, handling both "English: Korean" and "English : Korean"
            parts = re.split(r'\s*:\s*', line, maxsplit=1)
            if len(parts) == 2:
                english, korean = parts
                vocab_entries.append((english.strip(), korean.strip()))

        if not vocab_entries:
            return

        # Calculate number of rows needed (2 entries per row)
        num_rows = math.ceil(len(vocab_entries) / 2)

        # Create table with 4 columns
        table = doc.add_table(rows=num_rows, cols=4)

        # Fill table
        for row_idx in range(num_rows):
            row = table.rows[row_idx]

            # First entry (columns 0 and 1)
            entry_idx_1 = row_idx * 2
            if entry_idx_1 < len(vocab_entries):
                english1, korean1 = vocab_entries[entry_idx_1]
                row.cells[0].text = f"☐ {english1}"
                row.cells[1].text = korean1

            # Second entry (columns 2 and 3)
            entry_idx_2 = row_idx * 2 + 1
            if entry_idx_2 < len(vocab_entries):
                english2, korean2 = vocab_entries[entry_idx_2]
                row.cells[2].text = f"☐ {english2}"
                row.cells[3].text = korean2

        # Apply styling to all cells
        for row in table.rows:
            for cell in row.cells:
                # Set cell borders
                self.set_cell_border(
                    cell,
                    top={"sz": 5, "val": "single", "color": "auto"},
                    bottom={"sz": 5, "val": "single", "color": "auto"},
                    left={"sz": 5, "val": "single", "color": "auto"},
                    right={"sz": 5, "val": "single", "color": "auto"}
                )

                # Apply Normal_modified style and smaller font size to paragraphs
                for paragraph in cell.paragraphs:
                    paragraph.style = 'Normal_modified'
                    # Set font size to one smaller than size_text
                    for run in paragraph.runs:
                        run.font.size = Pt(self.size_text - 1)

        # Set cell margins
        self.set_table_cell_margins(table, top=3, bottom=3, left=5, right=5)

    def create_vocabulary_table_2col(self, doc, vocabulary_text):
        """
        Create a 2-column vocabulary table with checkboxes.
        Format: ☐ English | Korean
        Each row contains 1 vocabulary entry.
        Used when two_column layout is enabled.
        """
        # Parse vocabulary entries
        lines = [line.strip() for line in vocabulary_text.split('\n') if line.strip()]
        vocab_entries = []

        for line in lines:
            # Split by colon, handling both "English: Korean" and "English : Korean"
            parts = re.split(r'\s*:\s*', line, maxsplit=1)
            if len(parts) == 2:
                english, korean = parts
                vocab_entries.append((english.strip(), korean.strip()))

        if not vocab_entries:
            return

        # Create table with 2 columns (one entry per row)
        table = doc.add_table(rows=len(vocab_entries), cols=2)

        # Fill table
        for row_idx, (english, korean) in enumerate(vocab_entries):
            row = table.rows[row_idx]
            row.cells[0].text = f"☐ {english}"
            row.cells[1].text = korean

        # Apply styling to all cells
        for row in table.rows:
            for cell in row.cells:
                # Set cell borders
                self.set_cell_border(
                    cell,
                    top={"sz": 5, "val": "single", "color": "auto"},
                    bottom={"sz": 5, "val": "single", "color": "auto"},
                    left={"sz": 5, "val": "single", "color": "auto"},
                    right={"sz": 5, "val": "single", "color": "auto"}
                )

                # Apply Normal_modified style and smaller font size to paragraphs
                for paragraph in cell.paragraphs:
                    paragraph.style = 'Normal_modified'
                    # Set font size to one smaller than size_text
                    for run in paragraph.runs:
                        run.font.size = Pt(self.size_text - 1)

        # Set cell margins
        self.set_table_cell_margins(table, top=3, bottom=3, left=5, right=5)

    def add_markdown_content(self, doc, markdown_text):
        """
        Convert markdown-formatted text to Word document formatting.
        Handles: headers (#, ##, ###), bold (**text**), lists (-), quotes (>), code blocks (```).
        """
        lines = markdown_text.split('\n')
        i = 0
        in_code_block = False
        code_block_lines = []

        while i < len(lines):
            line = lines[i]
            stripped = line.strip()

            # Handle code blocks (```)
            if stripped.startswith('```'):
                if not in_code_block:
                    # Start of code block
                    in_code_block = True
                    code_block_lines = []
                    i += 1
                    continue
                else:
                    # End of code block
                    if code_block_lines:
                        code_text = '\n'.join(code_block_lines)

                        # Check if this is a PlantUML diagram
                        if code_text.strip().startswith('@startuml'):
                            # Generate PlantUML diagram
                            diagram_image = self.generate_diagram_from_plantuml(code_text)
                            if diagram_image:
                                try:
                                    diagram_image.seek(0)
                                    doc.add_picture(diagram_image, width=Inches(2))
                                    # Add a small spacing after the diagram
                                    doc.add_paragraph()
                                except UnrecognizedImageError:
                                    self.app_logger.error(
                                        "PlantUML private API returned unreadable image bytes; falling back to code block"
                                    )
                                    bordered_para = self.create_bordered_paragraph(doc, style='Normal_code_block')
                                    run = bordered_para.add_run(code_text)
                                    run.font.name = 'Courier New'
                                    run.font.size = Pt(self.size_text - 1)
                                except Exception as img_error:
                                    self.app_logger.error(
                                        f"Failed to insert PlantUML diagram: {img_error}"
                                    )
                                    bordered_para = self.create_bordered_paragraph(doc, style='Normal_code_block')
                                    run = bordered_para.add_run(code_text)
                                    run.font.name = 'Courier New'
                                    run.font.size = Pt(self.size_text - 1)
                            else:
                                # If diagram generation failed, fall back to code block
                                bordered_para = self.create_bordered_paragraph(doc, style='Normal_code_block')
                                run = bordered_para.add_run(code_text)
                                run.font.name = 'Courier New'
                                run.font.size = Pt(self.size_text - 1)
                        else:
                            # Regular code block - create bordered paragraph with left alignment
                            bordered_para = self.create_bordered_paragraph(doc, style='Normal_code_block')
                            run = bordered_para.add_run(code_text)
                            # Use monospace font for code
                            run.font.name = 'Courier New'
                            run.font.size = Pt(self.size_text - 1)
                    in_code_block = False
                    code_block_lines = []
                    i += 1
                    continue

            # If inside code block, collect lines
            if in_code_block:
                code_block_lines.append(line)
                i += 1
                continue

            # Handle horizontal rules (---) - convert to blank line
            if stripped == '---':
                doc.add_paragraph()  # Add empty paragraph for spacing
                i += 1
                continue

            # Handle headers
            if stripped.startswith('### '):
                # H3 - Bold, size_text
                text = stripped[4:]
                # Remove ** markers since entire header is bold
                text = text.replace('**', '')
                para = doc.add_paragraph(style='Normal_modified')
                run = para.add_run(text)
                run.bold = True
                run.font.size = Pt(self.size_text)
                i += 1
                continue
            elif stripped.startswith('## '):
                # H2 - Bold, size_text + 1
                text = stripped[3:]
                # Remove ** markers since entire header is bold
                text = text.replace('**', '')
                para = doc.add_paragraph(style='Normal_modified')
                run = para.add_run(text)
                run.bold = True
                run.font.size = Pt(self.size_text + 1)
                i += 1
                continue
            elif stripped.startswith('# '):
                # H1 - Bold, size_text + 2
                text = stripped[2:]
                # Remove ** markers since entire header is bold
                text = text.replace('**', '')
                para = doc.add_paragraph(style='Normal_modified')
                run = para.add_run(text)
                run.bold = True
                run.font.size = Pt(self.size_text + 2)
                i += 1
                continue

            # Handle blockquotes
            if stripped.startswith('> '):
                text = stripped[2:]
                para = doc.add_paragraph(style='Normal_modified')
                # Add indentation
                para.paragraph_format.left_indent = Inches(0.5)
                # Parse inline bold formatting
                self._add_text_with_bold_markdown(para, text)
                # Make entire quote italic
                for run in para.runs:
                    run.italic = True
                i += 1
                continue

            # Handle lists
            if stripped.startswith('- '):
                text = stripped[2:]
                para = doc.add_paragraph(style='Normal_modified')
                # Add bullet prefix
                para.add_run('• ')
                # Parse inline bold formatting
                self._add_text_with_bold_markdown(para, text)
                i += 1
                continue

            # Handle regular text (with inline bold formatting)
            if stripped:  # Only add non-empty lines
                para = doc.add_paragraph(style='Normal_modified')
                self._add_text_with_bold_markdown(para, stripped)

            i += 1

    def _add_text_with_bold_markdown(self, para, text):
        """
        Helper function to add text to paragraph with **bold** inline formatting.
        """
        # Pattern to match **text**
        pattern = re.compile(r'\*\*([^*]+)\*\*')
        last_idx = 0

        for match in pattern.finditer(text):
            start, end = match.span()

            # Add text before the match (plain text)
            if start > last_idx:
                text_segment = text[last_idx:start]
                para.add_run(text_segment)

            # Add bold text (without the ** markers)
            run = para.add_run(match.group(1))
            run.bold = True

            last_idx = end

        # Add any remaining text (plain text)
        if last_idx < len(text):
            text_segment = text[last_idx:]
            para.add_run(text_segment)

    #def add_page_break(self, doc):
    #    doc.add_paragraph(style='Normal_modified').add_run().add_break(WD_BREAK.PAGE)

    def add_page_break(self, doc):
        """Add a page break, removing any trailing empty paragraphs first"""
        # Remove trailing empty paragraph if it exists
        if doc.paragraphs and not doc.paragraphs[-1].text.strip():
            p = doc.paragraphs[-1]._element
            p.getparent().remove(p)
        
        # Now add the page break
        doc.add_page_break()

    def add_section_break(self, doc):
        
        """Add a section break instead of a page break"""
        # Remove trailing empty paragraph if it exists
        if doc.paragraphs and not doc.paragraphs[-1].text.strip():
            p = doc.paragraphs[-1]._element
            p.getparent().remove(p)
        
        # Add a new section with continuous break
        doc.add_section(WD_SECTION.NEW_PAGE)
        
        # Copy margins from previous section
        if len(doc.sections) > 1:
            prev_section = doc.sections[-2]
            new_section = doc.sections[-1]
            new_section.top_margin = prev_section.top_margin
            new_section.bottom_margin = prev_section.bottom_margin
            new_section.left_margin = prev_section.left_margin
            new_section.right_margin = prev_section.right_margin

    def is_section_title(self, para_text):
        pattern = r"^\d+\.\s(어휘|내용|밑줄의미|함축의미|빈칸추론|추론|순서|순서2단계|삽입|주제|제목|요지|연결어|어법|빈칸암기|동반의어|영영정의|요약|영작|동형문제|커스텀|실전모의고사|기출문제세트).*"
        return bool(re.match(pattern, para_text))

    def is_answer_and_explanation(self, para_text):
        return para_text.strip() == "정답 및 해설"

    def fast_answer(self, para_text):
        return para_text.strip() == "빠른 정답 찾기"


    def is_answer_heading(self, para_text):
        return para_text.endswith("정답")

    def define_styles(self, doc):

        # Define 'Heading 1' style
        heading1_style = doc.styles['Heading 1']
        heading1_style.font.name = self.selected_font_for_title
        heading1_style._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_title)
        heading1_style.font.size = Pt(self.size_heading)
        heading1_style.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
        heading1_style.paragraph_format.line_spacing = 1.2
        heading1_style.paragraph_format.space_before = Pt(0)
        heading1_style.paragraph_format.space_after = Pt(0)
        heading1_style.paragraph_format.alignment = WD_PARAGRAPH_ALIGNMENT.JUSTIFY

        rgb_color_heading = self.hex_to_rgb(self.color_heading)
        heading1_style.font.color.rgb = RGBColor(*rgb_color_heading)
        heading1_style.font.bold = True

        # Define 'Heading_ans' style
        heading1ans_style = doc.styles.add_style('Heading_ans', WD_STYLE_TYPE.PARAGRAPH)
        heading1ans_style.base_style = doc.styles['Heading 2']
        heading1ans_style.font.name = self.selected_font_for_title
        heading1ans_style._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_title)
        heading1ans_style.font.size = Pt(self.size_heading)
        heading1ans_style.paragraph_format.space_before = Pt(0)
        heading1ans_style.paragraph_format.space_after = Pt(0)
        heading1ans_style.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
        heading1ans_style.paragraph_format.line_spacing = 1.2
        heading1ans_style.paragraph_format.alignment = WD_PARAGRAPH_ALIGNMENT.JUSTIFY
        heading1ans_style.font.color.rgb = RGBColor(*rgb_color_heading)
        heading1ans_style.font.bold = True

        # Define 'Heading_num' style
        heading_num_style = doc.styles.add_style('Heading_num', WD_STYLE_TYPE.PARAGRAPH)
        heading_num_style.base_style = doc.styles['Heading 2']
        heading_num_style.font.name = self.selected_font_for_num
        heading_num_style._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_num)
        heading_num_style.font.size = Pt(self.size_num)
        heading_num_style.paragraph_format.space_before = Pt(0)
        heading_num_style.paragraph_format.space_after = Pt(0)
        heading_num_style.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
        heading_num_style.paragraph_format.line_spacing = 1.2
        rgb_color_num = self.hex_to_rgb(self.color_num)
        heading_num_style.font.color.rgb = RGBColor(*rgb_color_num)
        heading_num_style.font.bold = True

        # Define 'Normal' style
        normal_style = doc.styles['Normal']
        normal_style.font.name = '나눔고딕'
        normal_style._element.rPr.rFonts.set(qn('w:eastAsia'), '나눔고딕')
        normal_style.font.size = Pt(8)
        normal_style.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
        normal_style.paragraph_format.line_spacing = 1.2
        normal_style.paragraph_format.space_before = Pt(0)
        normal_style.paragraph_format.space_after = Pt(0)
        normal_style.paragraph_format.alignment = WD_PARAGRAPH_ALIGNMENT.JUSTIFY

        # Define 'Normal_modified' style
        normal_style_modified = doc.styles.add_style('Normal_modified', WD_STYLE_TYPE.PARAGRAPH)
        normal_style_modified.base_style = doc.styles['Normal']
        normal_style_modified.font.name = self.selected_font_for_text
        normal_style_modified._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_text)
        normal_style_modified.font.size = Pt(self.size_text)
        normal_style_modified.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
        normal_style_modified.paragraph_format.line_spacing = self.line_text
        normal_style_modified.paragraph_format.space_before = Pt(0)
        normal_style_modified.paragraph_format.space_after = Pt(0)
        normal_style_modified.paragraph_format.alignment = WD_PARAGRAPH_ALIGNMENT.JUSTIFY
        rgb_color_text = self.hex_to_rgb(self.color_text)
        normal_style_modified.font.color.rgb = RGBColor(*rgb_color_text)


        # Define 'Normal_modified_box' style (new - identical to Normal_modified)
        normal_style_modified_box = doc.styles.add_style('Normal_modified_box', WD_STYLE_TYPE.PARAGRAPH)
        normal_style_modified_box.base_style = doc.styles['Normal']
        normal_style_modified_box.font.name = self.selected_font_for_text
        normal_style_modified_box._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_text)
        normal_style_modified_box.font.size = Pt(self.size_text)
        normal_style_modified_box.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
        normal_style_modified_box.paragraph_format.line_spacing = self.line_text
        normal_style_modified_box.paragraph_format.space_before = Pt(0)
        normal_style_modified_box.paragraph_format.space_after = Pt(0)
        normal_style_modified_box.paragraph_format.alignment = WD_PARAGRAPH_ALIGNMENT.JUSTIFY
        normal_style_modified_box.font.color.rgb = RGBColor(*rgb_color_text)

        # Define 'Normal_code_block' style (for code blocks with left alignment)
        normal_style_code_block = doc.styles.add_style('Normal_code_block', WD_STYLE_TYPE.PARAGRAPH)
        normal_style_code_block.base_style = doc.styles['Normal']
        normal_style_code_block.font.name = self.selected_font_for_text
        normal_style_code_block._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_text)
        normal_style_code_block.font.size = Pt(self.size_text)
        normal_style_code_block.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
        normal_style_code_block.paragraph_format.line_spacing = self.line_text
        normal_style_code_block.paragraph_format.space_before = Pt(0)
        normal_style_code_block.paragraph_format.space_after = Pt(0)
        normal_style_code_block.paragraph_format.alignment = WD_PARAGRAPH_ALIGNMENT.LEFT
        normal_style_code_block.font.color.rgb = RGBColor(*rgb_color_text)

        # Define 'Normal_left' style
        normal_style_left = doc.styles.add_style('Normal_left', WD_STYLE_TYPE.PARAGRAPH)
        normal_style_left.base_style = doc.styles['Normal']
        normal_style_left.font.name = self.selected_font_for_text
        normal_style_left._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_text)
        normal_style_left.font.size = Pt(self.size_text)
        normal_style_left.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
        normal_style_left.paragraph_format.line_spacing = self.line_text
        normal_style_left.paragraph_format.space_before = Pt(0)
        normal_style_left.paragraph_format.space_after = Pt(0)
        normal_style_left.paragraph_format.alignment = WD_PARAGRAPH_ALIGNMENT.LEFT
        normal_style_left.font.color.rgb = RGBColor(*rgb_color_text)

        # Define 'Normal_right' style (주석어휘)
        normal_style_right = doc.styles.add_style('Normal_right', WD_STYLE_TYPE.PARAGRAPH)
        normal_style_right.base_style = doc.styles['Normal']
        normal_style_right.font.name = self.selected_font_for_text
        normal_style_right._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_text)
        normal_style_right.font.size = Pt(self.size_text - 1)
        normal_style_right.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
        normal_style_right.paragraph_format.line_spacing = self.line_text
        normal_style_right.paragraph_format.space_before = Pt(0)
        normal_style_right.paragraph_format.space_after = Pt(0)
        normal_style_right.paragraph_format.alignment = WD_PARAGRAPH_ALIGNMENT.RIGHT
        normal_style_right.font.color.rgb = RGBColor(*rgb_color_text)

        # Define 'Normal_exp' style
        normal_style_explanation = doc.styles.add_style('Normal_exp', WD_STYLE_TYPE.PARAGRAPH)
        normal_style_explanation.base_style = doc.styles['Normal']
        normal_style_explanation.font.name = self.selected_font_for_exp
        normal_style_explanation._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_exp)
        normal_style_explanation.font.size = Pt(self.size_exp)
        normal_style_explanation.paragraph_format.line_spacing_rule = WD_LINE_SPACING.MULTIPLE
        normal_style_explanation.paragraph_format.line_spacing = self.line_exp
        normal_style_explanation.paragraph_format.space_before = Pt(0)
        normal_style_explanation.paragraph_format.space_after = Pt(0)
        normal_style_explanation.paragraph_format.alignment = WD_PARAGRAPH_ALIGNMENT.LEFT
        rgb_color_exp = self.hex_to_rgb(self.color_exp)
        normal_style_explanation.font.color.rgb = RGBColor(*rgb_color_exp)



        
    def handle_empty_export(self):
        error_dialog = Toplevel()
        error_dialog.title("오류")
        error_dialog.geometry("500x150")
        error_dialog.grab_set()
        error_dialog.focus_set()
        error_dialog.grid_rowconfigure(0, weight=1)
        error_dialog.grid_columnconfigure(0, weight=1)

        CTkLabel(
            error_dialog, 
            fg_color="white", 
            width=300, 
            height=120, 
            text="-추출할 지문이 없습니다!\n'추출할 지문번호 입력'창에 추출할 지문번호가 올바르게 입력되어 있는지 확인하세요.", 
            text_color="black"
        ).grid(row=0, column=0, sticky="nsew")

        button_frame = CTkFrame(error_dialog, fg_color="white", width=300, height=30, corner_radius=0)
        button_frame.grid(row=1, column=0, sticky="swe")
        button_frame.grid_columnconfigure(0, weight=1)

        def on_ok():
            self.close_export_popup = True
            error_dialog.grab_release()
            error_dialog.destroy()

        ok_button = CTkButton(
            button_frame, 
            border_width=2, 
            border_color="gray", 
            text="OK", 
            text_color="black", 
            width=70, 
            height=30, 
            command=on_ok, 
            fg_color="#FEF9E0", 
            hover_color="#DDA15C"
        )
        ok_button.grid(row=0, column=0, padx=10, pady=10)







    def should_apply_two_column(self, row_key):

        """Check if two-column format should be applied to this row_key"""
        if not self.two_column:

            return False

        # Exclude specific row_keys from two-column format
        if row_key == "동반의어" or row_key == "영영정의":

            return False


        return True

    def should_create_answer_sections(self, section_titles):
        """
        Check if we should create '정답 및 해설' and '빠른 정답 찾기' sections.
        Returns False if ALL sections are types that don't need answer sections.

        Sheet types without answer sections:
        - 동반의어 / 동반의어_Normal (but NOT 동반의어문제1/2)
        - 영영정의 / 영영정의_Normal (but NOT 영영정의문제)
        - 커스텀 / 커스텀_Normal
        - 요약문 / 요약문_Normal
        - 본문정리
        - 교재 types: 지문분석, 한글해석, 직독직해, 단어정리, 한줄해석연습, 한줄영작연습
        """
        if not section_titles:
            return False

        # Types that DON'T need answer sections (only if they DON'T contain "문제")
        no_answer_types = ["동반의어", "영영정의", "커스텀", "요약문", "본문정리",
                          "지문분석", "한글해석", "직독직해", "단어정리", "한줄해석연습", "한줄영작연습", "한줄영작연습 (1)", "한줄영작연습 (2)", "한줄영작연습 (3)"]

        # Check if ALL sections are from the "no answer section" list
        for title in section_titles:
            # If title contains "문제", it's a question type that needs answers
            if "문제" in title:
                return True

            # If title doesn't contain "문제", check if it's in the exclusion list
            title_needs_exclusion = any(x in title for x in no_answer_types)
            if not title_needs_exclusion:
                # Found a type that needs answers
                return True

        # All sections are from the exclusion list (no "문제" types found)
        return False

    def should_skip_section_in_answers(self, section_title):
        """
        Check if a single section should be skipped in answer sections.
        Returns True if the section doesn't need answers/explanations.

        Args:
            section_title: The section title to check (e.g., "► 한줄영작연습 (1)")

        Returns:
            True if section should be excluded from answer sections, False otherwise
        """
        # Types that DON'T need answer sections (only if they DON'T contain "문제")
        no_answer_types = ["동반의어", "영영정의", "커스텀", "요약문", "본문정리",
                          "지문분석", "한글해석", "직독직해", "단어정리",
                          "한줄해석연습", "한줄영작연습"]

        # If section contains "문제", it's a question type that needs answers
        if "문제" in section_title:
            return False

        # Check if section title matches any no-answer type (substring matching)
        for no_answer_type in no_answer_types:
            if no_answer_type in section_title:
                return True

        return False

    def excel_to_word(self, excel_file, sheet_name, output_doc, last_state, rows_order):
        """
        Converts specified sheets from an Excel file to a formatted Word document.

        Parameters:
            excel_file (str): Path to the Excel file.
            sheet_name (str): Name of the sheet to process.
            output_doc (str): Path to save the generated Word document.
            last_state (dict): Dictionary indicating which sheets to process.
            rows_order (list): List indicating the order of processing sheets.
        """
        

        # Don't override if book_cover_path is already set
        if not hasattr(self, 'book_cover_path') or not self.book_cover_path:
            # Try to find default_bookcover01.docx (may have numeric prefix)
            found_path = self._find_book_cover_by_basename("default_bookcover01.docx")
            if found_path:
                self.book_cover_path = found_path
            else:
                # Fallback to hardcoded path (even if it doesn't exist)
                self.book_cover_path = self.base_path / "assets" / "bookcovers" / "default_bookcover01.docx"

        # Log the current book cover path for debugging
        print("=========================")
        print("excel_to_word called")
        print(f"Using book cover: {self.book_cover_path}")
        print("=========================")

        # -------------------------------------------------------------------------
        # 1) Load the workbook and check for sheets
        # -------------------------------------------------------------------------
        try:
            wb = load_workbook(filename=excel_file, read_only=True)

        except Exception as e:
            self.app_logger.error(f"Failed to load workbook '{excel_file}': {e}")
            return None

        try:
            waiting_sheet = wb["대기열"]
        except KeyError:
            self.app_logger.error(f"Sheet '대기열' does not exist in the workbook.")
            return None

        # -------------------------------------------------------------------------
        # 2) Instead of creating a brand-new Document, load your cover doc
        # -------------------------------------------------------------------------
        #self.app_logger.info(f"Cover doc path: {self.book_cover_path}")
        # Merge with cover if option is set
        

        p = Path(self.book_cover_path)
        




        delete_empty_paragraph = False
        default_bookcover = False

        if "default_bookcover" in p.name.lower():
            default_bookcover = True

        print(f"default_bookcover: {default_bookcover}")
        print(f"self.two_column: {self.two_column}")


        # Check for "no cover" in filename (handles numeric prefix like "000_no cover.docx")
        is_no_cover = "no cover" in p.name.lower()

        if self.applyoption_var.get() == "1" and not is_no_cover:
            print("옵션 적용!")
            doc = Document(self.book_cover_path)

            # ADD THIS: Check if the document already ends with a page break
            def document_ends_with_page_break(doc):
                """Check if the last element in the document is a page break"""
                if not doc.paragraphs:
                    return False
                    
                # Check the last few paragraphs (sometimes page breaks create empty paragraphs)
                for para in reversed(doc.paragraphs[-3:]):  # Check last 3 paragraphs
                    for run in para.runs:
                        # Check if the run contains a page break
                        if run._element.xml:
                            if 'w:br' in run._element.xml and 'type="page"' in run._element.xml:
                                return True
                return False
            
            # Only add page break if document doesn't already end with one
            if not document_ends_with_page_break(doc):
                doc.add_page_break()
                print("Page break added after bookcover")

            else:
                print("Bookcover already ends with page break, skipping")
                delete_empty_paragraph = True

        elif self.applyoption_var.get() == "1" and is_no_cover:
            print("옵션 적용, 커버 없음")
            doc = Document()

            # Set A4 page size when not using a cover
            section = doc.sections[0]
            section.page_width = Cm(21)  # A4 width
            section.page_height = Cm(29.7)  # A4 height

            # Remove the initial empty paragraph that comes with Document()
            if doc.paragraphs:
                p_elem = doc.paragraphs[0]._element
                p_elem.getparent().remove(p_elem)

            # Add page numbers at center bottom (footer)
            self._add_page_number_footer(section)

        else:
            print("옵션 비적용")
            doc = Document()
            # Set A4 page size when not using a cover
            section = doc.sections[0]
            section.page_width = Cm(21)  # A4 width
            section.page_height = Cm(29.7)  # A4 height




        # -------------------------------------------------------------------------
        # 3) Define styles and set margins on this cover doc
        # -------------------------------------------------------------------------

        self.define_styles(doc)
        section = doc.sections[0]
        section.top_margin = Cm(self.margin_top)
        section.bottom_margin = Cm(self.margin_bottom)
        section.left_margin = Cm(self.margin_left)
        section.right_margin = Cm(self.margin_right)
        self._apply_mirror_margins(section)




        # Check if any row_keys that will actually be processed need two-column formatting
        rows_to_process = []
        for row_key in rows_order:
            normal_key = f"{row_key}_Normal"
            hard_key = f"{row_key}_Hard"
            process_normal = last_state.get(normal_key, False)
            process_hard = last_state.get(hard_key, False)
            process_mock_test = last_state.get(row_key, False)
            process_past_test = last_state.get(row_key, False)
            
            if process_normal or process_hard or process_mock_test or process_past_test:
                rows_to_process.append(row_key)



        # Initialize with single-column mode
        self.current_column_mode = False  # Start with single-column
        processed_any_row = False




        if self.applyoption_var.get() == "1" and not is_no_cover and delete_empty_paragraph:  # Only if using a cover doc
            
            # Remove trailing empty paragraphs
            while doc.paragraphs and not doc.paragraphs[-1].text.strip():
                p = doc.paragraphs[-1]._element
                p.getparent().remove(p)
                if not doc.paragraphs:  # Safety check to avoid removing all paragraphs
                    break

        sections_dict = defaultdict(list)
        section_number = 1




        

        # -------------------------------------------------------------------------
        # 4) Process each row_key (sheet) in the specified order
        # -------------------------------------------------------------------------



        # Add enumerate to get index for look-ahead
        processed_rows_count = 0
        skipped_rows = []
        
        for idx, row_key in enumerate(rows_order):
            
            normal_key = f"{row_key}_Normal"
            hard_key = f"{row_key}_Hard"
            process_normal = last_state.get(normal_key, False)
            process_hard = last_state.get(hard_key, False)
            process_mock_test = last_state.get(row_key, False)
            process_past_test = last_state.get(row_key, False)
            

            
            

            
            # Skip if this row won't be processed
            if not (process_normal or process_hard or process_mock_test or process_past_test):
                
                skipped_rows.append(row_key)
                continue
            
            processed_rows_count += 1
                
            # Check if this row_key needs different column formatting
            self.row_needs_two_column = self.should_apply_two_column(row_key)
            
            
            
            # Track if we add a section break due to column change
            section_break_added_for_column_change = False
            
            # If column mode changes, add a new section
            if self.row_needs_two_column != self.current_column_mode:
            
                
                if processed_any_row:
            
                    doc.add_section(WD_SECTION.NEW_PAGE)
                    new_sec = doc.sections[-1]
                    section_break_added_for_column_change = True
                else:
                    
                    if default_bookcover:
                    
                        new_sec = doc.sections[-1]
                    else:
                        new_sec = doc.sections[-1]
                




                sectPr = new_sec._sectPr
                                
                # First, remove any existing column settings
                for cols in sectPr.xpath('./w:cols'):
                    sectPr.remove(cols)

                if self.row_needs_two_column:
                    # Switch to two-column
                    cols = OxmlElement('w:cols')
                    cols.set(qn('w:num'), '2')
                    cols.set(qn('w:sep'), '1')
                    cols.set(qn('w:space'), str(self.cm_to_twips(self.col_spacing)))
                    # Append instead of insert to ensure proper ordering
                    sectPr.append(cols)
                else:
                    # Ensure single-column
                    one_col = OxmlElement('w:cols')
                    one_col.set(qn('w:num'), '1')
                    sectPr.append(one_col)



                # Update margins
                new_sec.top_margin = Cm(self.margin_top)
                new_sec.bottom_margin = Cm(self.margin_bottom)
                new_sec.left_margin = Cm(self.margin_left)
                new_sec.right_margin = Cm(self.margin_right)
                self._apply_mirror_margins(new_sec)

                self.current_column_mode = self.row_needs_two_column
                


            processed_any_row = True
            


            # Look ahead BEFORE processing sheets
            will_next_need_column_change = False
            if idx + 1 < len(rows_order):
                next_row_key = rows_order[idx + 1]
                
                
                # Check if next row will be processed
                next_normal_key = f"{next_row_key}_Normal"
                next_hard_key = f"{next_row_key}_Hard"
                next_will_process = (last_state.get(next_normal_key, False) or 
                                last_state.get(next_hard_key, False) or
                                last_state.get(next_row_key, False))
                
                
                
                if next_will_process:
                    next_needs_two_column = self.should_apply_two_column(next_row_key)
                
                    if next_needs_two_column != self.current_column_mode:
                        will_next_need_column_change = True
                
            #else:
                #print(f"   🔮 No next row to look ahead to (last row)")
            
            # Track if any sheet was processed for this row_key
            any_sheet_processed = False
            


            # Continue with existing processing logic...
            normal_key = f"{row_key}_Normal"
            hard_key = f"{row_key}_Hard"
            process_normal = last_state.get(normal_key, False)
            process_hard = last_state.get(hard_key, False)
            process_mock_test = last_state.get(row_key, False)
            process_past_test = last_state.get(row_key, False)


            # --------------------------------------------------------------
            # A) Handle Normal/Hard sheets
            # --------------------------------------------------------------
            for sheet_idx, (sheet_suffix, process) in enumerate([('Normal', process_normal), ('Hard', process_hard)]):

                if process:
                    any_sheet_processed = True


                    # Determine the sheet type
                    if "동형문제" in row_key or "실전모의고사" in row_key or "기출문제세트" in row_key:
                        sheet_type = f"{row_key}"
                    else:
                        sheet_type = f"{row_key}_{sheet_suffix}"


                    if sheet_type in wb.sheetnames:
                        extracting_sheet = wb[sheet_type]

                        # Fetch text from global_text_input_to_extract
                        text_content = self.global_text_input_to_extract.get("1.0", "end-1c")
                        lines = [line.strip() for line in text_content.split("\n") if line.strip()]

                        추출할지문번호 = []
                        for line in lines:
                            if line:
                                stripped_passage_id = line.strip()
                                if stripped_passage_id not in 추출할지문번호:
                                    추출할지문번호.append(stripped_passage_id)
                                else:
                                    self.app_logger.info(f"●[오류] 지문번호 중복: {stripped_passage_id}\n"
                                                        f"첫 번째 지문만 추출됩니다.")

                        # Build a mapping from passage ID -> row data FIRST
                        passage_id_to_row_data = {}
                        for row in extracting_sheet.iter_rows(min_row=1, values_only=True):
                            if row[0] is not None:
                                passage_id = row[0].strip()
                                passage_id_to_row_data[passage_id] = row[:5]  # first 5 columns

                        # Pre-validate passages to check if any will actually be processed
                        # This allows us to skip the section title if all passages are invalid
                        valid_passage_ids = []
                        for passage_id in 추출할지문번호:
                            if passage_id not in passage_id_to_row_data:
                                # Will be skipped - passage not found
                                continue
                            row_data = passage_id_to_row_data[passage_id]
                            row_data = tuple("" if cell is None else cell for cell in row_data)
                            if len(row_data) < 5:
                                row_data += ("",) * (5 - len(row_data))
                            title, passage, result, answer_part, explanation_part = row_data[:5]
                            if title:
                                title = title.strip()
                                # Check skip conditions
                                if title in 추출할지문번호 and result in ("연결어 없음", "문제 미출제 (문장 수 부족)", None, ""):
                                    continue  # Will be skipped
                                if result and (result.startswith("Error") or result.startswith("재출제 중...")):
                                    continue  # Will be skipped
                                # This passage is valid
                                valid_passage_ids.append(passage_id)

                        # Skip this section entirely if no valid passages
                        if not valid_passage_ids:
                            self.app_logger.info(f"[스킵] '{sheet_type}' 섹션에 유효한 지문이 없어 섹션을 건너뜁니다.")
                            # Still log warnings for skipped passages
                            for passage_id in 추출할지문번호:
                                if passage_id not in passage_id_to_row_data:
                                    self.export_warnings.append((sheet_type, passage_id, "지문이 해당 유형에 존재하지 않음"))
                                else:
                                    row_data = passage_id_to_row_data[passage_id]
                                    row_data = tuple("" if cell is None else cell for cell in row_data)
                                    if len(row_data) < 5:
                                        row_data += ("",) * (5 - len(row_data))
                                    title, passage, result, answer_part, explanation_part = row_data[:5]
                                    if title:
                                        title = title.strip()
                                        if title in 추출할지문번호 and result in ("연결어 없음", "문제 미출제 (문장 수 부족)", None, ""):
                                            reason = result if result else "문제 미출제"
                                            self.export_warnings.append((sheet_type, title, f"문제 미출제: {reason}"))
                                        elif result and result.startswith("Error"):
                                            self.export_warnings.append((sheet_type, title, f"출제 오류: {result}"))
                                        elif result and result.startswith("재출제 중..."):
                                            self.export_warnings.append((sheet_type, title, f"출제 오류: {result}"))
                            continue  # Skip to next sheet_type

                        total_valid_passages = len(valid_passage_ids)
                        passages_on_page = 0
                        # Now we know there are valid passages - write the section title
                        section_title = f"{section_number}. {sheet_type}"

                        # Optionally remove difficulty tag
                        if self.tagremoval_flag == 1:
                            modified_sheet_type = sheet_type.split('_')[0].strip()
                            section_title = f"{section_number}. {modified_sheet_type}"


                        para = doc.add_paragraph(style='Heading 1')
                        run = para.add_run(section_title)
                        # Explicitly set font on run to fix "no cover" issue
                        run.font.name = self.selected_font_for_title
                        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_title)

                        # Extract the rows in the order they appear in 추출할지문번호
                        processed_count = 0
                        skipped_count = 0
                        question_number = 1  # Initialize question counter for numeric numbering
                        for passage_id in 추출할지문번호:
                            if passage_id not in passage_id_to_row_data:
                                error_msg = f"'{sheet_type}' 유형에 '{passage_id}' 지문이 없음"
                                self.app_logger.error(f"[에러★] {error_msg}. (추출할 유형과 지문번호를 다시 확인해보세요)")
                                # Track warning
                                self.export_warnings.append((sheet_type, passage_id, "지문이 해당 유형에 존재하지 않음"))
                                skipped_count += 1
                                continue

                            row_data = passage_id_to_row_data[passage_id]
                            row_data = tuple("" if cell is None else cell for cell in row_data)
                            if len(row_data) < 5:
                                row_data += ("",) * (5 - len(row_data))
                            title, passage, result, answer_part, explanation_part = row_data[:5]

                            if title:
                                title = title.strip()
                                원래title = title  # Always save original for skip conditions and logging

                                # Apply numeric numbering if enabled
                                if self.passage_number_numeric:
                                    title = f"{question_number}."  # Create numeric title
                                    해설지로보낼title = f"{title} {원래title}"  # Combined for 해설지
                                else:
                                    해설지로보낼title = title  # Use original title

                                # Skip conditions (use original passage_id for checking)
                                if 원래title in 추출할지문번호 and result in ("연결어 없음", "문제 미출제 (문장 수 부족)", None, ""):
                                    reason = result if result else "문제 미출제"
                                    self.app_logger.info(f"{sheet_type} - {원래title}: 문제가 없거나 조건에 맞지 않아 추출하지 않습니다.")
                                    # Track warning
                                    self.export_warnings.append((sheet_type, 원래title, f"문제 미출제: {reason}"))
                                    skipped_count += 1
                                    continue
                                if result and result.startswith("Error"):
                                    self.app_logger.info(f"{sheet_type} - {원래title}: 문제 에러로 추출하지 않습니다. "
                                                        f"재출제하세요.")
                                    # Track warning
                                    self.export_warnings.append((sheet_type, 원래title, f"출제 오류: {result}"))
                                    skipped_count += 1
                                    continue
                                elif result and result.startswith("재출제 중..."):
                                    self.app_logger.info(f"{sheet_type} - {원래title}: 문제 에러로 추출하지 않습니다. "
                                                        f"재출제하세요.")
                                    # Track warning
                                    self.export_warnings.append((sheet_type, 원래title, f"출제 오류: {result}"))
                                    skipped_count += 1
                                    continue                                

                                ########### two_column 일때 편집 변경 필요한 부분들 (가령 선지 ③을 다음줄로 보내기)
                                if '동반의어문제' in sheet_type and self.row_needs_two_column:
                                    
                                    result = result.replace("③", "\n③")



                                # Determine function name
                                function_name = sheet_type.replace('_Normal', '_편집')\
                                                        .replace('_Hard', '_편집')\
                                                        .replace("내용일치(한)", "내용일치한")\
                                                        .replace("무관한 문장", "무관한문장")


                                if function_name in FUNCTION_MAP:
                                    # Title as a heading (passage_number_numeric=0: original passage_id, 1: numeric numbering)
                                    para_title = doc.add_paragraph(style='Heading_num')
                                    run = para_title.add_run(str(title))
                                    # Explicitly set font on run to fix "no cover" issue
                                    run.font.name = self.selected_font_for_num
                                    run._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_num)

                                    processed = self.process_question_part(
                                        doc, function_name, title, passage, result,
                                        answer_part, explanation_part
                                    )

                                    if processed:
                                        (title, passage, question_line, question_part,
                                        answer_part, explanation_part, cell, table) = processed

                                        # Additional formatting logic (동반의어, 영영정의, etc.) ...
                                        # EXACT same code as your original, just referencing `doc` for paragraphs
                                        # For brevity, not re-pasting all. Example snippet:
                                        if '동반의어_편집' in function_name:
                                            data = parse_data(question_part)
                                            insert_data_into_table(doc, data)
                                            cell = None
                                            answer_part = " "
                                            explanation_part = " "

                                        elif '영영정의_편집' in function_name:
                                            data = parse_data(question_part)
                                            insert_data_into_table_영영정의(doc, data)
                                            cell = None
                                            answer_part = " "
                                            explanation_part = " "



                                        elif '커스텀' in function_name or '요약문' in function_name:
                                            for paragraph in question_part.split('\n'):
                                                self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')
                                            cell = None
                                            answer_part = " "
                                            explanation_part = " "

                                        elif '어휘1단계' in function_name or '어휘2단계' in function_name or '어법1단계' in function_name:
                                            
                                            if self.border_flag:
                                                # Create initial paragraph (will be replaced)
                                                temp_para = doc.add_paragraph()
                                                paragraphs = question_part.split('\n')
                                                
                                                # Use the modified apply_bold_underline_to_paragraph function
                                                self.apply_bold_underline_to_paragraph(temp_para, paragraphs)
                                                
                                                # Set cell and table to None since we're not using tables
                                                cell = None
                                                table = None
                                            else:
                                                # Add question without borders but with margins
                                                paragraphs = question_part.split('\n')
                                                for idx, paragraph_text in enumerate(paragraphs):
                                                    para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')
                                                    if idx == 0:
                                                        para.paragraph_format.space_before = Pt(1)
                                                    if idx == len(paragraphs) - 1:
                                                        para.paragraph_format.space_after = Pt(6)

                                                # Set cell and table to None since we're not using tables
                                                cell = None
                                                table = None
                                                
  


                                        elif '어휘3단계' in function_name or '어법2단계' in function_name:

                                            if self.border_flag:
                                                # Create bordered paragraph for passage
                                                bordered_para = self.create_bordered_paragraph(doc)
                                                paragraphs = passage.split('\n')
                                                
                                                # Apply the add_bold_underline_to_brackets logic to the paragraph
                                                self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)
                                                
                                                # Add question part without border
                                                question_line_part = doc.add_paragraph(style='Normal_modified')
                                                words_to_format = ["없는", "않는", "않은", "틀린", "2개", "3개", "모두"]
                                                self.add_formatted_text(question_line_part, question_part, words_to_format)                                            

                                                
                                                cell = None
                                                table = None
                                            else:
                                                # Add passage without borders but with margins
                                                paragraphs = passage.split('\n')
                                                for idx, paragraph_text in enumerate(paragraphs):
                                                    para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')
                                                    if idx == 0:
                                                        para.paragraph_format.space_before = Pt(1)
                                                    if idx == len(paragraphs) - 1:
                                                        para.paragraph_format.space_after = Pt(6)

                                                # Add question part without border
                                                question_line_part = doc.add_paragraph(style='Normal_modified')
                                                words_to_format = ["없는", "않는", "않은", "틀린", "2개", "3개", "모두"]
                                                self.add_formatted_text(question_line_part, question_part, words_to_format)

                                                cell = None
                                                table = None




                                        elif '밑줄의미' in function_name or '함축의미' in function_name:
                                            #문제를 요약문과 선지로 쪼개기
                                            split_point = question_part.find("①")
                                            문제라인파트 = question_part[:split_point].strip()
                                            문제나머지파트 = question_part[split_point:].strip()

                                            if self.border_flag:
                                                #문제라인 추가
                                                if '밑줄의미' in function_name:
                                                    question_line_part = doc.add_paragraph(style='Normal_modified')
                                                    words_to_format = ["않은"]
                                                    self.add_formatted_text(question_line_part, 문제라인파트, words_to_format)
                                                else:
                                                    question_line_part = doc.add_paragraph(style='Normal_modified')
                                                    self.테이블없는본문_괄호에볼드밑줄더하기(question_line_part, 문제라인파트)

                                                #지문 추가 - Create bordered paragraph instead of table
                                                bordered_para = self.create_bordered_paragraph(doc)
                                                paragraphs = passage.split('\n')
                                                self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)

                                                #문제 나머지파트 추가
                                                for paragraph in 문제나머지파트.split('\n'):
                                                    self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')

                                                cell = None
                                                table = None
                                            else:
                                                #문제라인 추가
                                                if '밑줄의미' in function_name:
                                                    question_line_part = doc.add_paragraph(style='Normal_modified')
                                                    words_to_format = ["않은"]
                                                    self.add_formatted_text(question_line_part, 문제라인파트, words_to_format)
                                                else:
                                                    question_line_part = doc.add_paragraph(style='Normal_modified')
                                                    self.테이블없는본문_괄호에볼드밑줄더하기(question_line_part, 문제라인파트)

                                                # Add passage without borders but with margins
                                                paragraphs = passage.split('\n')
                                                for idx, paragraph_text in enumerate(paragraphs):
                                                    para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')
                                                    if idx == 0:
                                                        para.paragraph_format.space_before = Pt(1)
                                                    if idx == len(paragraphs) - 1:
                                                        para.paragraph_format.space_after = Pt(6)

                                                #문제 나머지파트 추가
                                                for paragraph in 문제나머지파트.split('\n'):
                                                    self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')

                                                cell = None
                                                table = None


                                        elif '무관한문장' in function_name:


                                            #문제를 요약문과 선지로 쪼개기
                                            split_point = question_part.find("윗글에서")
                                            if split_point == -1:
                                                문제파트 = ""
                                                문제라인파트 = question_part.strip()
                                            else:
                                                문제파트 = question_part[:split_point].strip()
                                                문제라인파트 = question_part[split_point:].strip()



                                            if self.border_flag:
                                                #문제파트를 먼저
                                                # Create bordered paragraph instead of table

                                                if "Normal" in sheet_type:
                                                    bordered_para = self.create_bordered_paragraph(doc)
                                                    paragraphs = 문제파트.split('\n')
                                                    self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)
                                                else:
                                                    문제파트 = 문제파트.replace("[", "").replace("]", "")
                                                    bordered_para = self.create_bordered_paragraph(doc)
                                                    paragraphs = 문제파트.split('\n')
                                                    self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)


                                                #문제라인파트를 border 없이 추가.
                                                question_line_part = doc.add_paragraph(style='Normal_modified')
                                                self.add_formatted_text(question_line_part, 문제라인파트, ["없는"])


                                                cell = None
                                                table = None
                                            else:
                                                # Add 문제파트 without borders but with margins
                                                if "Normal" in sheet_type:
                                                    paragraphs = 문제파트.split('\n')
                                                else:
                                                    문제파트 = 문제파트.replace("[", "").replace("]", "")
                                                    paragraphs = 문제파트.split('\n')

                                                for idx, paragraph_text in enumerate(paragraphs):
                                                    para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')
                                                    if idx == 0:
                                                        para.paragraph_format.space_before = Pt(1)
                                                    if idx == len(paragraphs) - 1:
                                                        para.paragraph_format.space_after = Pt(6)

                                                #문제라인파트를 border 없이 추가.
                                                question_line_part = doc.add_paragraph(style='Normal_modified')
                                                self.add_formatted_text(question_line_part, 문제라인파트, ["없는"])


                                                cell = None
                                                table = None




                                        elif '빈칸추론' in function_name:
                                            # Create bordered paragraph instead of table


                                            if self.border_flag:
                                                bordered_para = self.create_bordered_paragraph(doc)
                                                paragraphs = passage.split('\n')
                                                self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)


                                                # Apply remove_brackets logic to paragraph
                                                pattern = re.compile(r"\<([^>]*)\>|\[([^\]]*)\]")

                                                for idx, paragraph_text in enumerate(paragraphs):
                                                    if idx > 0:
                                                        bordered_para.add_run('\n')

                                                    last_idx = 0
                                                    for match in pattern.finditer(paragraph_text):
                                                        start, end = match.span()
                                                        bordered_para.add_run(paragraph_text[last_idx:start])
                                                        text_within_brackets = match.group(1) or match.group(2)
                                                        if text_within_brackets:
                                                            bordered_para.add_run(text_within_brackets)
                                                        last_idx = end

                                                    bordered_para.add_run(paragraph_text[last_idx:])

                                                for paragraph in question_part.split('\n'):
                                                    self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')

                                                cell = None
                                                table = None


                                            else:

                                                # Add passage without borders but with margins
                                                paragraphs = passage.split('\n')
                                                for idx, paragraph_text in enumerate(paragraphs):
                                                    para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')

                                                    # Add margin before first paragraph
                                                    if idx == 0: 
                                                        para.paragraph_format.space_before = Pt(1) #set to 1.

                                                    # Add margin after last paragraph
                                                    if idx == len(paragraphs) - 1:
                                                        para.paragraph_format.space_after = Pt(6)

                                                for paragraph in question_part.split('\n'):
                                                    self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')


                                        elif any(keyword in function_name for keyword in ['내용일치', '내용일치(한)', '추론', '주제_편집', '제목', '요지', '어휘종합', '내용종합']):
                                            if self.border_flag:
                                                # Create bordered paragraph
                                                bordered_para = self.create_bordered_paragraph(doc)
                                                paragraphs = passage.split('\n')
                                                self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)
                                                # Add question part without border
                                                if '내용일치' in function_name:
                                                    # For 내용일치(한), add question without formatting
                                                    if self.naeyongilchi_question_type == 0:
                                                        pass
                                                    elif self.naeyongilchi_question_type == 1:
                                                        question_part = question_part.replace("(1)", "①")
                                                        question_part = question_part.replace("(2)", "②")
                                                        question_part = question_part.replace("(3)", "③")
                                                        question_part = question_part.replace("(4)", "④")
                                                        question_part = question_part.replace("(5)", "⑤")
                                                        question_part = question_part.replace(" ❮ T / F ❯", "")

                                                        # Count number of 'T' in answer_part
                                                        true_count = answer_part.count('T')
                                                        if true_count == 1:
                                                            question_line = f"다음 중 윗글의 내용과 일치하는 것을 고르시오"
                                                        elif true_count == 4:
                                                            question_line = f"다음 중 윗글의 내용과 일치하지 않는 것을 고르시오"
                                                        else:
                                                            question_line = f"다음 중 윗글의 내용과 일치하는 것을 {true_count}개 고르시오"

                                                        self.add_paragraph_with_formatting(doc, question_line, style='Normal_left')

                                                    for paragraph in question_part.split('\n'):
                                                        if paragraph.strip():  # Skip empty paragraphs
                                                            doc.add_paragraph(paragraph, style='Normal_modified')

                                                else:
                                                    # For other types, apply formatting
                                                    for paragraph in question_part.split('\n'):
                                                        self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')

                                                cell = None
                                                table = None
                                            else:
                                                # Add passage without borders but with margins
                                                paragraphs = passage.split('\n')
                                                for idx, paragraph_text in enumerate(paragraphs):
                                                    para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')
                                                    if idx == 0:
                                                        para.paragraph_format.space_before = Pt(1)
                                                    if idx == len(paragraphs) - 1:
                                                        para.paragraph_format.space_after = Pt(6)

                                                # Add question part without border
                                                if '내용일치' in function_name:
                                                    # For 내용일치(한), add question without formatting
                                                    if self.naeyongilchi_question_type == 0:
                                                        pass
                                                    elif self.naeyongilchi_question_type == 1:
                                                        question_part = question_part.replace("(1)", "①")
                                                        question_part = question_part.replace("(2)", "②")
                                                        question_part = question_part.replace("(3)", "③")
                                                        question_part = question_part.replace("(4)", "④")
                                                        question_part = question_part.replace("(5)", "⑤")
                                                        question_part = question_part.replace(" ❮ T / F ❯", "")

                                                        # Count number of 'T' in answer_part
                                                        true_count = answer_part.count('T')
                                                        if true_count == 1:
                                                            question_line = f"다음 중 윗글의 내용과 일치하는 것을 고르시오"
                                                        elif true_count == 4:
                                                            question_line = f"다음 중 윗글의 내용과 일치하지 않는 것을 고르시오"
                                                        else:
                                                            question_line = f"다음 중 윗글의 내용과 일치하는 것을 {true_count}개 고르시오"

                                                        self.add_paragraph_with_formatting(doc, question_line, style='Normal_left')

                                                    for paragraph in question_part.split('\n'):
                                                        if paragraph.strip():  # Skip empty paragraphs
                                                            doc.add_paragraph(paragraph, style='Normal_modified')
                                                else:
                                                    # For other types, apply formatting
                                                    for paragraph in question_part.split('\n'):
                                                        self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')

                                                cell = None
                                                table = None


                                        elif any(keyword in function_name for keyword in ['순서', '삽입']):
                                            # Create bordered paragraph
                                            bordered_para = self.create_bordered_paragraph(doc)
                                            paragraphs = passage.split('\n')
                                            self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)
                                            # Add question part without border

                                            for paragraph in question_part.split('\n'):
                                                self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')

                                            cell = None
                                            table = None

                                            

                                        elif '요약_편집' in function_name:
                                            question_part = question_part.replace("윗글의 내용을 한 문장으로 요약하고자 한다. 빈칸 (A), (B)에 들어갈 말로 가장 적절한 것은?\n", "")

                                            #문제 라인 입력
                                            question_line = "다음 글의 내용을 한 문장으로 요약하고자 한다. 빈칸 (A), (B)에 들어갈 말로 가장 적절한 것은?"
                                            para_question_line = doc.add_paragraph(question_line, style='Normal_modified')
                                            para_question_line.alignment = WD_ALIGN_PARAGRAPH.LEFT


                                            if self.border_flag:
                                                #지문 - Create bordered paragraph instead of table
                                                bordered_para = self.create_bordered_paragraph(doc)
                                                paragraphs = passage.split('\n')
                                                self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)

                                            else:
                                                # Add passage without borders but with margins
                                                paragraphs = passage.split('\n')
                                                for idx, paragraph_text in enumerate(paragraphs):
                                                    para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')

                                                    # Add margin before first paragraph
                                                    if idx == 0: 
                                                        para.paragraph_format.space_before = Pt(1) #set to 1.

                                                    # Add margin after last paragraph
                                                    if idx == len(paragraphs) - 1:
                                                        para.paragraph_format.space_after = Pt(6)


                                            #요약문 기호 입력
                                            요약문기호 = "⇩"
                                            para = doc.add_paragraph(요약문기호, style='Normal_modified')
                                            para.alignment = WD_ALIGN_PARAGRAPH.CENTER

                                            #문제를 요약문과 선지로 쪼개기
                                            split_point = question_part.find("(A) -")
                                            요약문파트 = question_part[:split_point].strip()
                                            선지파트 = question_part[split_point:].strip()
                                            
                                            #요약문 파트 - Create bordered paragraph with special remove_brackets formatting
                                            bordered_para2 = self.create_bordered_paragraph(doc)
                                            요약문파트_paragraphs = 요약문파트.split('\n')
                                            
                                            # Use a modified version of apply_bold_underline_to_paragraph that removes brackets
                                            self.apply_remove_brackets_to_paragraph(bordered_para2, 요약문파트_paragraphs)

                                            # 선지파트 탭 적용 후 넣기
                                            선지파트 = 선지파트.replace("(A) - (B)", "          (A)\t\t       (B)")
                                            선지파트 = 선지파트.replace(" -", "\t...\t")

                                            for paragraph_text in 선지파트.split('\n'):
                                                paragraph = doc.add_paragraph(paragraph_text, style='Normal_modified')
                                                tab_stops = paragraph.paragraph_format.tab_stops
                                                for tab_pos in range(1, 2):
                                                    tab_stops.add_tab_stop(Inches(1.1 * tab_pos))
                                            
                                            cell = None
                                            table = None
                                        


                                        elif '영영정의문제' in function_name:
                                            # Create bordered paragraph instead of table
                                            bordered_para = self.create_bordered_paragraph(doc)
                                            paragraphs = passage.split('\n')
                                            self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)


                                            pattern = re.compile(r"(\<[^>]*\>|\[[^\]]*\])")
                                            
                                            for idx, paragraph_text in enumerate(paragraphs):
                                                if idx > 0:
                                                    bordered_para.add_run('\n')
                                                
                                                last_idx = 0
                                                for match in pattern.finditer(paragraph_text):
                                                    start, end = match.span()
                                                    bordered_para.add_run(paragraph_text[last_idx:start])
                                                    run = bordered_para.add_run(match.group())
                                                    run.bold = True
                                                    last_idx = end
                                                
                                                bordered_para.add_run(paragraph_text[last_idx:])
                                            
                                            for paragraph in question_part.split('\n'):
                                                self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')
                                            
                                            cell = None
                                            table = None




                                        elif '영작1' in function_name:
                                            # Find and capture the "문제파트"
                                            문제파트 = ""
                                            match = re.search(r"다음의.*?\)", question_part)
                                            if match:
                                                문제파트 = match.group(0)

                                            question_part = question_part.replace(문제파트, "").strip()

                                            if '[보기]' in question_part:
                                                지문파트, 보기파트 = question_part.split('[보기]', 1)
                                                지문파트 = 지문파트.strip()
                                                # Check if 보기 has numbered format
                                                보기_content = 보기파트.strip()
                                                if re.match(r'^\(\d+\)', 보기_content):
                                                    보기파트 = '[보기]\n' + 보기_content
                                                else:
                                                    보기파트 = '[보기] ' + 보기_content
                                            else:
                                                지문파트 = question_part.strip()
                                                보기파트 = ""

                                            # 문제라인
                                            question_line_part = doc.add_paragraph(style='Normal_modified')
                                            self.테이블없는본문_영작단어수에볼드밑줄더하기(question_line_part, 문제파트)

                                            # 지문파트 - Create bordered paragraph instead of table
                                            bordered_para = self.create_bordered_paragraph(doc)
                                            paragraphs = 지문파트.split('\n')
                                            self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)

                                            # 보기파트
                                            # Check if 보기파트 has numbered items and add corresponding numbered underlines
                                            numbered_items = re.findall(r'^\(\d+\)', 보기파트, re.MULTILINE)
                                            if numbered_items:
                                                # Multi-item case: add numbered underlines
                                                underlines = []
                                                for i in range(1, len(numbered_items) + 1):
                                                    if self.row_needs_two_column:
                                                        underlines.append(f"→ ({i})_____________________________________________\n")
                                                    else:
                                                        underlines.append(f"→ ({i})____________________________________________________________________________________________")
                                                보기파트 = 보기파트 + "\n\n" + "\n".join(underlines)
                                            else:
                                                # Single-item case: add single underline
                                                if self.row_needs_two_column:
                                                    보기파트 = 보기파트 + "\n\n→ ________________________________________________\n"
                                                else:
                                                    보기파트 = 보기파트 + "\n\n→ ____________________________________________________________________________________________"



                                            for paragraph in 보기파트.split('\n'):
                                                self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')
                                            
                                            cell = None
                                            table = None

                                        elif '영작2' in function_name:
                                            # Find and capture the "문제파트"
                                            문제파트 = ""
                                            match = re.search(r"\[보기\]에.*?\)", question_part)
                                            if match:
                                                문제파트 = match.group(0)

                                            question_part = question_part.replace(문제파트, "").strip()

                                            if '[보기]' in question_part:
                                                지문파트, 보기파트 = question_part.split('[보기]', 1)
                                                지문파트 = 지문파트.strip()
                                                # Check if 보기 has numbered format
                                                보기_content = 보기파트.strip()
                                                if re.match(r'^\(\d+\)', 보기_content):
                                                    보기파트 = '[보기]\n' + 보기_content
                                                else:
                                                    보기파트 = '[보기] ' + 보기_content
                                            else:
                                                지문파트 = question_part.strip()
                                                보기파트 = ""

                                            # 문제라인
                                            question_line_part = doc.add_paragraph(style='Normal_modified')
                                            self.테이블없는본문_영작단어수에볼드밑줄더하기(question_line_part, 문제파트)


                                            
                                            # 지문파트 - Create bordered paragraph instead of table
                                            bordered_para = self.create_bordered_paragraph(doc)
                                            paragraphs = 지문파트.split('\n')
                                            self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)

                                            # 보기파트
                                            # Check if 보기파트 has numbered items and add corresponding numbered underlines
                                            numbered_items = re.findall(r'^\(\d+\)', 보기파트, re.MULTILINE)
                                            if numbered_items:
                                                # Multi-item case: add numbered underlines
                                                underlines = []
                                                for i in range(1, len(numbered_items) + 1):
                                                    if self.row_needs_two_column:
                                                        underlines.append(f"→ ({i})_____________________________________________\n")
                                                    else:
                                                        underlines.append(f"→ ({i})____________________________________________________________________________________________")
                                                보기파트 = 보기파트 + "\n\n" + "\n".join(underlines)
                                            else:
                                                # Single-item case: add single underline
                                                if self.row_needs_two_column:
                                                    보기파트 = 보기파트 + "\n\n→ ________________________________________________\n"
                                                else:
                                                    보기파트 = 보기파트 + "\n\n→ ____________________________________________________________________________________________"


                                            for paragraph in 보기파트.split('\n'):
                                                self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')
                                            
                                            cell = None
                                            table = None




                                        elif '주제영작' in function_name:

                                            # Extract 지문파트 (everything before '윗글의')
                                            지문파트 = ""
                                            지문_match = re.search(r"(.*?)윗글의", question_part, re.DOTALL)
                                            if 지문_match:
                                                지문파트 = 지문_match.group(1).strip()

                                            # Find and capture the "문제파트"
                                            문제파트 = ""
                                            match = re.search(r"윗글의.*?\)", question_part)
                                            if match:
                                                문제파트 = match.group(0)

                                            # Extract 보기파트 (everything after 문제파트)
                                            보기파트 = question_part.replace(지문파트, "").replace(문제파트, "").strip()


                                            # 지문파트 - Create bordered paragraph instead of table
                                            bordered_para = self.create_bordered_paragraph(doc)
                                            paragraphs = 지문파트.split('\n')
                                            self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)



                                            # 문제라인
                                            question_line_part = doc.add_paragraph(style='Normal_modified')
                                            self.테이블없는본문_영작단어수에볼드밑줄더하기(question_line_part, 문제파트)


                                            # 보기파트
                                            # Check if 보기파트 has numbered items and add corresponding numbered underlines
                                            numbered_items = re.findall(r'^\(\d+\)', 보기파트, re.MULTILINE)
                                            if numbered_items:
                                                # Multi-item case: add numbered underlines
                                                underlines = []
                                                for i in range(1, len(numbered_items) + 1):
                                                    if self.row_needs_two_column:
                                                        underlines.append(f"→ ({i})_____________________________________________")
                                                    else:
                                                        underlines.append(f"→ ({i})____________________________________________________________________________________________")
                                                보기파트 = 보기파트 + "\n\n" + "\n".join(underlines)
                                            else:
                                                # Single-item case: add single underline
                                                if self.row_needs_two_column:
                                                    보기파트 = 보기파트 + "\n\n→ ________________________________________________"
                                                else:
                                                    보기파트 = 보기파트 + "\n\n→ ____________________________________________________________________________________________"

                                            for paragraph in 보기파트.split('\n'):
                                                self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')


                                            cell = None
                                            table = None


                                        elif '동형문제' in function_name:
                                            #지문 추가 - Create bordered paragraph instead of table
                                            bordered_para = self.create_bordered_paragraph(doc)
                                            paragraphs = passage.split('\n')
                                            self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)

                                            question_part = question_part.replace("괄호 []한", "밑줄 친").replace("괄호 [[]]한", "밑줄 친").replace("괄호 한", "밑줄 친").replace("괄호한", "밑줄 친")

                                            #문제 추가
                                            for idx, paragraph in enumerate(question_part.split('\n')):
                                                self.add_paragraph_with_formatting동형문제(doc, paragraph, style='Normal_modified', is_first_paragraph=(idx == 0))

                                            cell = None
                                            table = None


                                        elif '연결어' in function_name:
                                            # Extract the modified passage from question_part
                                            # The editing function has already split and modified it
                                            split_point = question_part.find("(1) ")
                                            modified_passage = question_part[:split_point].strip() if split_point != -1 else ""
                                            questions_part = question_part[split_point:].strip() if split_point != -1 else question_part

                                            if self.border_flag:
                                                # Create bordered paragraph for modified passage
                                                bordered_para = self.create_bordered_paragraph(doc)
                                                paragraphs = modified_passage.split('\n')

                                                # Add borders (all borders since this is a single paragraph containing all lines)
                                                self.add_paragraph_border(bordered_para, top=True, bottom=True, left=True, right=True)

                                                # Apply padding
                                                bordered_para.paragraph_format.left_indent = Pt(5)
                                                bordered_para.paragraph_format.right_indent = Pt(5)
                                                bordered_para.paragraph_format.space_before = Pt(0)
                                                bordered_para.paragraph_format.space_after = Pt(0)

                                                # Apply formatting (handles <>, [], {} brackets)
                                                pattern = re.compile(r"\<([^>]*)\>|\[([^\]]*)\]|\{([^}]*)\}")
                                                for idx, paragraph_text in enumerate(paragraphs):
                                                    if idx > 0:
                                                        bordered_para.add_run('\n')

                                                    last_idx = 0
                                                    for match in pattern.finditer(paragraph_text):
                                                        start, end = match.span()
                                                        bordered_para.add_run(paragraph_text[last_idx:start])

                                                        if match.group(1) or match.group(2):  # Angle or square brackets
                                                            text_within_brackets = match.group(1) or match.group(2)
                                                            if text_within_brackets:
                                                                run = bordered_para.add_run(text_within_brackets)
                                                                run.bold = True
                                                                run.underline = True
                                                        elif match.group(3):  # Curly braces
                                                            text_inside_braces = match.group(3)
                                                            run = bordered_para.add_run(text_inside_braces)
                                                            run.italic = True

                                                        last_idx = end

                                                    bordered_para.add_run(paragraph_text[last_idx:])

                                                # Add questions part
                                                sections = [s.strip() for s in questions_part.strip().split('\n(') if s]
                                                for i, section in enumerate(sections):
                                                    if i > 0:
                                                        section = f'({section}'

                                                    lines = section.split('\n')
                                                    question = lines[0]
                                                    options = lines[1:]

                                                    doc.add_paragraph(question, style='Normal_modified')

                                                    if self.row_needs_two_column and len(options) == 4:
                                                        row1 = doc.add_paragraph(style='Normal_modified')
                                                        row1.add_run(options[0])
                                                        row1.add_run('\t')
                                                        row1.add_run(options[1])
                                                        ts1 = row1.paragraph_format.tab_stops
                                                        ts1.add_tab_stop(Inches(1.5))

                                                        row2 = doc.add_paragraph(style='Normal_modified')
                                                        row2.add_run(options[2])
                                                        row2.add_run('\t')
                                                        row2.add_run(options[3])
                                                        ts2 = row2.paragraph_format.tab_stops
                                                        ts2.add_tab_stop(Inches(1.5))
                                                    else:
                                                        options_paragraph = doc.add_paragraph(style='Normal_modified')
                                                        for j, option in enumerate(options):
                                                            if j > 0:
                                                                options_paragraph.add_run('\t')
                                                            options_paragraph.add_run(option)
                                                        tab_stops = options_paragraph.paragraph_format.tab_stops
                                                        for tab_pos in range(1, len(options)):
                                                            tab_stops.add_tab_stop(Inches(1.5 * tab_pos))

                                                    if i < len(sections) - 1:
                                                        doc.add_paragraph(style='Normal_modified')

                                                cell = None
                                                table = None

                                            else:
                                                # Add modified passage without borders but with margins
                                                paragraphs = modified_passage.split('\n')
                                                for idx, paragraph_text in enumerate(paragraphs):
                                                    para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')
                                                    if idx == 0:
                                                        para.paragraph_format.space_before = Pt(1)
                                                    if idx == len(paragraphs) - 1:
                                                        para.paragraph_format.space_after = Pt(6)

                                                # Add questions part
                                                sections = [s.strip() for s in questions_part.strip().split('\n(') if s]
                                                for i, section in enumerate(sections):
                                                    if i > 0:
                                                        section = f'({section}'

                                                    lines = section.split('\n')
                                                    question = lines[0]
                                                    options = lines[1:]

                                                    doc.add_paragraph(question, style='Normal_modified')

                                                    if self.row_needs_two_column and len(options) == 4:
                                                        row1 = doc.add_paragraph(style='Normal_modified')
                                                        row1.add_run(options[0])
                                                        row1.add_run('\t')
                                                        row1.add_run(options[1])
                                                        ts1 = row1.paragraph_format.tab_stops
                                                        ts1.add_tab_stop(Inches(1.5))

                                                        row2 = doc.add_paragraph(style='Normal_modified')
                                                        row2.add_run(options[2])
                                                        row2.add_run('\t')
                                                        row2.add_run(options[3])
                                                        ts2 = row2.paragraph_format.tab_stops
                                                        ts2.add_tab_stop(Inches(1.5))
                                                    else:
                                                        options_paragraph = doc.add_paragraph(style='Normal_modified')
                                                        for j, option in enumerate(options):
                                                            if j > 0:
                                                                options_paragraph.add_run('\t')
                                                            options_paragraph.add_run(option)
                                                        tab_stops = options_paragraph.paragraph_format.tab_stops
                                                        for tab_pos in range(1, len(options)):
                                                            tab_stops.add_tab_stop(Inches(1.5 * tab_pos))

                                                    if i < len(sections) - 1:
                                                        doc.add_paragraph(style='Normal_modified')

                                                cell = None
                                                table = None


                                        elif '빈칸암기' in function_name:
                                            # question_part contains the processed result from editing function
                                            # Split result by new lines to handle hard enters as separate paragraphs
                                            lines = question_part.split('\n')

                                            # Check if this is Hard difficulty (no first letter hint)
                                            is_hard = 'Hard' in sheet_type

                                            if self.border_flag:
                                                # Process each line as a separate bordered paragraph
                                                for idx, line_content in enumerate(lines):
                                                    bordered_para = self.create_bordered_paragraph(doc)

                                                    # Add borders based on position
                                                    if idx == 0:
                                                        # First line - all borders
                                                        self.add_paragraph_border(bordered_para, top=True, bottom=(len(lines) == 1),
                                                                                left=True, right=True)
                                                    elif idx == len(lines) - 1:
                                                        # Last line - only left, right, and bottom
                                                        self.add_paragraph_border(bordered_para, top=False, bottom=True, left=True, right=True)
                                                    else:
                                                        # Middle lines - only left and right
                                                        self.add_paragraph_border(bordered_para, top=False, bottom=False, left=True, right=True)

                                                    # Apply padding
                                                    bordered_para.paragraph_format.left_indent = Pt(5)
                                                    bordered_para.paragraph_format.right_indent = Pt(5)
                                                    bordered_para.paragraph_format.space_before = Pt(0)
                                                    bordered_para.paragraph_format.space_after = Pt(0)

                                                    # Combined pattern for brackets and curly braces
                                                    combined_pattern = re.compile(r'(\[)(.*?)(])|(\{)([^}]*)(\})')
                                                    last_idx = 0

                                                    for match in combined_pattern.finditer(line_content):
                                                        start, end = match.span()
                                                        bordered_para.add_run(line_content[last_idx:start])

                                                        if match.group(1) == '[':  # Bracket match
                                                            word = match.group(2)

                                                            if is_hard:
                                                                # Hard difficulty: hide entire word with white font + white bg + black underline
                                                                if word:
                                                                    word_run = bordered_para.add_run(word)
                                                                    word_run.font.color.rgb = RGBColor(0xFF, 0xFF, 0xFF)
                                                                    shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                                                    word_run._r.get_or_add_rPr().append(shd)
                                                                    self.set_underline_color(word_run, '000000')  # Black underline

                                                                run = bordered_para.add_run("    ")  # 4 spaces after word
                                                                run.font.color.rgb = RGBColor(0xFF, 0xFF, 0xFF)
                                                                shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                                                run._r.get_or_add_rPr().append(shd)
                                                                self.set_underline_color(run, '000000')  # Black underline
                                                            else:
                                                                # Normal difficulty: show first letter as hint
                                                                if word:
                                                                    # First letter visible (black)
                                                                    first_letter_run = bordered_para.add_run(word[0])
                                                                    first_letter_run.font.color.rgb = RGBColor(0x00, 0x00, 0x00)
                                                                    shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                                                    first_letter_run._r.get_or_add_rPr().append(shd)
                                                                    self.set_underline_color(first_letter_run, '000000')  # Black underline

                                                                    # Rest of word hidden (white)
                                                                    if len(word) > 1:
                                                                        rest_of_word_run = bordered_para.add_run(word[1:])
                                                                        rest_of_word_run.font.color.rgb = RGBColor(0xFF, 0xFF, 0xFF)
                                                                        shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                                                        rest_of_word_run._r.get_or_add_rPr().append(shd)
                                                                        self.set_underline_color(rest_of_word_run, '000000')  # Black underline

                                                                run = bordered_para.add_run("    ")  # 4 spaces after word
                                                                run.font.color.rgb = RGBColor(0xFF, 0xFF, 0xFF)
                                                                shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                                                run._r.get_or_add_rPr().append(shd)
                                                                self.set_underline_color(run, '000000')  # Black underline

                                                        elif match.group(4) is not None:  # Curly brace match
                                                            text_inside_braces = match.group(5)
                                                            run = bordered_para.add_run(text_inside_braces)
                                                            run.italic = True

                                                        last_idx = end

                                                    bordered_para.add_run(line_content[last_idx:])

                                                # Append explanation_part under the bordered paragraphs
                                                if explanation_part and explanation_part.strip():
                                                    explanation_lines = explanation_part.split('\n')
                                                    for line in explanation_lines:
                                                        self.add_paragraph_with_formatting(doc, line, style='Normal_modified')

                                                cell = None
                                                table = None

                                            else:
                                                # Add lines without borders but with margins
                                                for idx, line_content in enumerate(lines):
                                                    # Combined pattern for brackets and curly braces
                                                    combined_pattern = re.compile(r'(\[)(.*?)(])|(\{)([^}]*)(\})')
                                                    para = doc.add_paragraph(style='Normal_modified')

                                                    if idx == 0:
                                                        para.paragraph_format.space_before = Pt(1)
                                                    if idx == len(lines) - 1:
                                                        para.paragraph_format.space_after = Pt(6)

                                                    last_idx = 0
                                                    for match in combined_pattern.finditer(line_content):
                                                        start, end = match.span()
                                                        para.add_run(line_content[last_idx:start])

                                                        if match.group(1) == '[':  # Bracket match
                                                            word = match.group(2)

                                                            if is_hard:
                                                                # Hard difficulty: hide entire word with white font + white bg + black underline
                                                                if word:
                                                                    word_run = para.add_run(word)
                                                                    word_run.font.color.rgb = RGBColor(0xFF, 0xFF, 0xFF)
                                                                    shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                                                    word_run._r.get_or_add_rPr().append(shd)
                                                                    self.set_underline_color(word_run, '000000')  # Black underline

                                                                run = para.add_run("    ")  # 4 spaces after word
                                                                run.font.color.rgb = RGBColor(0xFF, 0xFF, 0xFF)
                                                                shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                                                run._r.get_or_add_rPr().append(shd)
                                                                self.set_underline_color(run, '000000')  # Black underline
                                                            else:
                                                                # Normal difficulty: show first letter as hint
                                                                if word:
                                                                    # First letter visible (black)
                                                                    first_letter_run = para.add_run(word[0])
                                                                    first_letter_run.font.color.rgb = RGBColor(0x00, 0x00, 0x00)
                                                                    shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                                                    first_letter_run._r.get_or_add_rPr().append(shd)
                                                                    self.set_underline_color(first_letter_run, '000000')  # Black underline

                                                                    # Rest of word hidden (white)
                                                                    if len(word) > 1:
                                                                        rest_of_word_run = para.add_run(word[1:])
                                                                        rest_of_word_run.font.color.rgb = RGBColor(0xFF, 0xFF, 0xFF)
                                                                        shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                                                        rest_of_word_run._r.get_or_add_rPr().append(shd)
                                                                        self.set_underline_color(rest_of_word_run, '000000')  # Black underline

                                                                run = para.add_run("    ")  # 4 spaces after word
                                                                run.font.color.rgb = RGBColor(0xFF, 0xFF, 0xFF)
                                                                shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                                                run._r.get_or_add_rPr().append(shd)
                                                                self.set_underline_color(run, '000000')  # Black underline

                                                        elif match.group(4) is not None:  # Curly brace match
                                                            text_inside_braces = match.group(5)
                                                            run = para.add_run(text_inside_braces)
                                                            run.italic = True

                                                        last_idx = end

                                                    para.add_run(line_content[last_idx:])

                                                # Append explanation_part
                                                if explanation_part and explanation_part.strip():
                                                    explanation_lines = explanation_part.split('\n')
                                                    for line in explanation_lines:
                                                        self.add_paragraph_with_formatting(doc, line, style='Normal_modified')

                                                cell = None
                                                table = None

                                        #Note: 연결어 and 빈칸암기 passage insertion now handled above with border_flag logic.
                                        #The editing functions in editing_functions.py still handle text transformation.

                                        # If there's a table cell to style...
                                        if cell is not None:
                                            self.set_cell_border(
                                                cell,
                                                top={"sz": 8, "val": "single", "color": "auto"},
                                                bottom={"sz": 8, "val": "single", "color": "auto"},
                                                left={"sz": 8, "val": "single", "color": "auto"},
                                                right={"sz": 8, "val": "single", "color": "auto"}
                                            )
                                            if table is not None:
                                                self.set_table_cell_margins(
                                                    table, top=5, bottom=5, left=7, right=7
                                                )

                                        doc.add_paragraph(style='Normal_modified')  # blank line

                                        # Check if answer_part is empty
                                        if not answer_part and "영작" not in sheet_type:
                                            self.show_error_and_log(
                                                f"오류: '{sheet_type}' 유형의 '{원래title}' 지문에서 "
                                                f"문제 오류가 발견되었습니다.\n(정답이 비어 있음)"
                                            )
                                            return

                                        # Transform answer_part and explanation_part for 5지선다 format
                                        if self.naeyongilchi_question_type == 1 and '내용일치' in sheet_type:
                                            # Transform answer_part: extract numbers with 'T' and 'F'
                                            true_numbers = []
                                            false_numbers = []
                                            for line in answer_part.split('\n'):
                                                line = line.strip()
                                                if line:
                                                    # Extract number from format like "(1) T" or "(1) F"
                                                    match_t = re.match(r'\((\d+)\)\s*T', line)
                                                    match_f = re.match(r'\((\d+)\)\s*F', line)
                                                    if match_t:
                                                        true_numbers.append(int(match_t.group(1)))
                                                    elif match_f:
                                                        false_numbers.append(int(match_f.group(1)))

                                            # Convert to circled numbers
                                            circle_map = {'1': '①', '2': '②', '3': '③', '4': '④', '5': '⑤'}

                                            # If there are 4 T's and 1 F, use the F as answer (내용 불일치 문제)
                                            if len(true_numbers) == 4 and len(false_numbers) == 1:
                                                answer_part = circle_map[str(false_numbers[0])]
                                            else:
                                                # Otherwise, use T's as answers
                                                answer_part = ''.join([circle_map[str(n)] for n in true_numbers if str(n) in circle_map])

                                            # Transform explanation_part: replace (1), (2), etc. with circled numbers
                                            for num in range(1, 6):
                                                explanation_part = explanation_part.replace(f"({num})", circle_map[str(num)])

                                        sections_dict[section_title].append(
                                            (해설지로보낼title, answer_part, explanation_part)
                                        )
                                        processed_count += 1

                                        # Increment question number for numeric mode
                                        if self.passage_number_numeric:
                                            question_number += 1

                                        if self.page_break_after_passages:
                                            passages_on_page += 1
                                            if (passages_on_page >= self.page_break_after_passages
                                                    and processed_count < total_valid_passages):
                                                self.add_page_break(doc)
                                                passages_on_page = 0
                                else:
                                    self.app_logger.error(
                                        f"Function {function_name} not found in FUNCTION_MAP."
                                    )
                                    skipped_count += 1
                                    continue

                        section_number += 1
                        #self.add_page_break(doc)
                        



                        # Add section break between Normal and Hard sheets
                        # Check if there's another sheet to process for this row_key
                        has_more_sheets = False
                        if sheet_idx == 0:  # Currently processing Normal
                            # Check if Hard will be processed
                            if process_hard:
                                has_more_sheets = True
                        
                        # Also check for other types (동형문제, mock test, past test)
                        if not has_more_sheets:
                            if "동형문제" in row_key and last_state.get(row_key, False):
                                has_more_sheets = True
                            elif "실전모의고사" in row_key and last_state.get(row_key, False):
                                has_more_sheets = True
                            elif "기출문제세트" in row_key and last_state.get(row_key, False):
                                has_more_sheets = True
                        
                        if has_more_sheets and self.new_page_per_section:

                            self.add_section_break(doc)




            # --------------------------------------------------------------
            # B) Handle 동형문제
            # --------------------------------------------------------------
            if "동형문제" in row_key:

                process_동형문제 = last_state.get(row_key, False)

                if process_동형문제:
                    any_sheet_processed = True

                    sheet_type = row_key  # e.g., "동형문제 1회"

                    # Initialize counters for 동형문제 section
                    processed_count = 0
                    skipped_count = 0
                    question_number = 1  # Initialize question counter for numeric numbering


                    if sheet_type in wb.sheetnames:
                        extracting_sheet = wb[sheet_type]
                        
                        # Fetch text from global_text_input_to_extract
                        text_content = self.global_text_input_to_extract.get("1.0", "end-1c")
                        lines = [line.strip() for line in text_content.split("\n") if line.strip()]
                        
                        추출할지문번호 = []
                        for line in lines:
                            if line:
                                stripped_passage_id = line.strip()
                                if stripped_passage_id not in 추출할지문번호:
                                    추출할지문번호.append(stripped_passage_id)
                                else:
                                    self.app_logger.info(f"●[오류] 지문번호 중복: {stripped_passage_id}\n"
                                                    f"첫 번째 지문만 추출됩니다.")
                        
                        section_title = f"{section_number}. {sheet_type}"
                        para = doc.add_paragraph(style='Heading 1')
                        run = para.add_run(section_title)
                        # Explicitly set font on run to fix "no cover" issue
                        run.font.name = self.selected_font_for_title
                        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_title)

                        # Build a mapping from passage ID -> row data
                        passage_id_to_row_data = {}
                        for row in extracting_sheet.iter_rows(min_row=1, values_only=True):
                            if row[0] is not None:
                                passage_id = row[0].strip()
                                passage_id_to_row_data[passage_id] = row[:5]  # first 5 columns

                        valid_passage_ids = []
                        for passage_id in 추출할지문번호:
                            if passage_id not in passage_id_to_row_data:
                                continue
                            row_data = passage_id_to_row_data[passage_id]
                            row_data = tuple("" if cell is None else cell for cell in row_data)
                            if len(row_data) < 5:
                                row_data += ("",) * (5 - len(row_data))
                            title, passage, result, answer_part, explanation_part = row_data[:5]
                            if title:
                                title = title.strip()
                                if title in 추출할지문번호 and result in ("연결어 없음", "문제 미출제 (문장 수 부족)", None, ""):
                                    continue
                                if result and result.startswith("Error"):
                                    continue
                                valid_passage_ids.append(passage_id)

                        total_valid_passages = len(valid_passage_ids)
                        passages_on_page = 0
                        # Extract the rows in the order they appear in 추출할지문번호
                        for passage_id in 추출할지문번호:
                            if passage_id not in passage_id_to_row_data:
                                error_msg = f"'{sheet_type}' 유형에 '{passage_id}' 지문이 없음"
                                self.app_logger.error(f"[에러★] {error_msg}. (추출할 유형과 지문번호를 다시 확인해보세요)")
                                # Track warning
                                self.export_warnings.append((sheet_type, passage_id, "지문이 해당 유형에 존재하지 않음"))
                                continue
                            
                            row_data = passage_id_to_row_data[passage_id]
                            row_data = tuple("" if cell is None else cell for cell in row_data)
                            if len(row_data) < 5:
                                row_data += ("",) * (5 - len(row_data))
                            title, passage, result, answer_part, explanation_part = row_data[:5]
                            
                            if title:
                                title = title.strip()
                                원래title = title  # Always save original for skip conditions and logging

                                # Apply numeric numbering if enabled
                                if self.passage_number_numeric:
                                    title = f"{question_number}."  # Create numeric title
                                    해설지로보낼title = f"{title} {원래title}"  # Combined for 해설지
                                else:
                                    해설지로보낼title = title  # Use original title

                                # Skip conditions (use original passage_id for checking)
                                if 원래title in 추출할지문번호 and result in ("연결어 없음", "문제 미출제 (문장 수 부족)", None, ""):
                                    reason = result if result else "문제 미출제"
                                    self.app_logger.info(f"{sheet_type} - {원래title}: 문제가 없거나 조건에 맞지 않아 추출하지 않습니다.")
                                    # Track warning
                                    self.export_warnings.append((sheet_type, 원래title, f"문제 미출제: {reason}"))
                                    continue
                                if result and result.startswith("Error"):
                                    self.app_logger.info(f"{sheet_type} - {원래title}: 문제 에러로 추출하지 않습니다. "
                                                    f"재출제하세요.")
                                    # Track warning
                                    self.export_warnings.append((sheet_type, 원래title, f"출제 오류: {result}"))
                                    continue

                                # For 동형문제, use a specific function name
                                function_name = "동형문제_편집"
                                
                                
                            
                                
                                if function_name in FUNCTION_MAP:
                                    # Title as a heading
                                    para_title = doc.add_paragraph(style='Heading_num')
                                    run = para_title.add_run(str(title))
                                    # Explicitly set font on run to fix "no cover" issue
                                    run.font.name = self.selected_font_for_num
                                    run._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_num)
                                    
                                    processed = self.process_question_part(
                                        doc, function_name, title, passage, result,
                                        answer_part, explanation_part
                                    )
                                    
                                    if processed:
                                        (title, passage, question_line, question_part,
                                        answer_part, explanation_part, cell, table) = processed
                                        
                                        # Handle 동형문제 formatting
                                        # 지문 추가 - Create bordered paragraph instead of table
                                        if passage:
                                            paragraphs = passage.split('\n')
                                            bordered_para = self.create_bordered_paragraph(doc)
                                            self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)
                                        
                                        # 문제 추가
                                        print("동형문제 문제 추가")
                                        for idx, paragraph in enumerate(question_part.split('\n')):
                                            self.add_paragraph_with_formatting동형문제(doc, paragraph, style='Normal_modified', is_first_paragraph=(idx == 0))

                                        
                                        doc.add_paragraph(style='Normal_modified')  # blank line
                                        
                                        # Check if answer_part is empty
                                        if not answer_part:
                                            self.show_error_and_log(
                                                f"오류: '{sheet_type}' 유형의 '{원래title}' 지문에서 "
                                                f"문제 오류가 발견되었습니다.\n(정답이 비어 있음)"
                                            )
                                            return

                                        # Transform answer_part and explanation_part for 5지선다 format
                                        if self.naeyongilchi_question_type == 1 and '내용일치' in sheet_type:
                                            # Transform answer_part: extract numbers with 'T' and 'F'
                                            true_numbers = []
                                            false_numbers = []
                                            for line in answer_part.split('\n'):
                                                line = line.strip()
                                                if line:
                                                    # Extract number from format like "(1) T" or "(1) F"
                                                    match_t = re.match(r'\((\d+)\)\s*T', line)
                                                    match_f = re.match(r'\((\d+)\)\s*F', line)
                                                    if match_t:
                                                        true_numbers.append(int(match_t.group(1)))
                                                    elif match_f:
                                                        false_numbers.append(int(match_f.group(1)))

                                            # Convert to circled numbers
                                            circle_map = {'1': '①', '2': '②', '3': '③', '4': '④', '5': '⑤'}

                                            # If there are 4 T's and 1 F, use the F as answer (내용 불일치 문제)
                                            if len(true_numbers) == 4 and len(false_numbers) == 1:
                                                answer_part = circle_map[str(false_numbers[0])]
                                            else:
                                                # Otherwise, use T's as answers
                                                answer_part = ''.join([circle_map[str(n)] for n in true_numbers if str(n) in circle_map])

                                            # Transform explanation_part: replace (1), (2), etc. with circled numbers
                                            for num in range(1, 6):
                                                explanation_part = explanation_part.replace(f"({num})", circle_map[str(num)])

                                        sections_dict[section_title].append(
                                            (해설지로보낼title, answer_part, explanation_part)
                                        )
                                        processed_count += 1

                                        # Increment question number for numeric mode
                                        if self.passage_number_numeric:
                                            question_number += 1

                                        if self.page_break_after_passages:
                                            passages_on_page += 1
                                            if (passages_on_page >= self.page_break_after_passages
                                                    and processed_count < total_valid_passages):
                                                self.add_page_break(doc)
                                                passages_on_page = 0
                                else:
                                    self.app_logger.error(
                                        f"Function {function_name} not found in FUNCTION_MAP."
                                    )
                                    skipped_count += 1
                                    continue

                        section_number += 1
                        #self.add_page_break(doc)
                        


            # --------------------------------------------------------------
            # C) Handle 교재 Types - Combined into ONE "본문정리" section
            # --------------------------------------------------------------
            # Check if ANY of the 6 교재 types should be processed
            교재_types = ["지문분석", "한글해석", "직독직해", "단어정리", "한줄해석연습", "한줄영작연습"]
            should_process_교재 = any(last_state.get(type_name, False) for type_name in 교재_types)

            if row_key in 교재_types and should_process_교재:
                # Only process once for all 6 types (when we hit the first one in rows_order)
                # Check if we've already processed 본문정리
                already_processed_교재 = any(t for t in 교재_types if t in rows_order[:idx] and last_state.get(t, False))

                if not already_processed_교재:
                    any_sheet_processed = True

                    

                    # Check which types are enabled
                    enabled_types = [t for t in 교재_types if last_state.get(t, False)]
                    


                    # Track if we need single-column for passage tables (jimun_analysis_interpretation_right mode)
                    needs_single_column_for_passage_table = (self.jimun_analysis_interpretation_right and
                                                             self.two_column and self.current_column_mode)

                    # If single-column needed AND there's already content, switch BEFORE the section title
                    if needs_single_column_for_passage_table and len(doc.paragraphs) > 0:
                        doc.add_section(WD_SECTION.NEW_PAGE)
                        new_sec = doc.sections[-1]
                        sectPr = new_sec._sectPr
                        for cols in sectPr.xpath('./w:cols'):
                            sectPr.remove(cols)
                        one_col = OxmlElement('w:cols')
                        one_col.set(qn('w:num'), '1')
                        sectPr.append(one_col)
                        new_sec.top_margin = Cm(self.margin_top)
                        new_sec.bottom_margin = Cm(self.margin_bottom)
                        new_sec.left_margin = Cm(self.margin_left)
                        new_sec.right_margin = Cm(self.margin_right)
                        self._apply_mirror_margins(new_sec)
                    elif needs_single_column_for_passage_table:
                        # No content yet, just modify the first section to single-column
                        section = doc.sections[0]
                        sectPr = section._sectPr
                        for cols in sectPr.xpath('./w:cols'):
                            sectPr.remove(cols)
                        one_col = OxmlElement('w:cols')
                        one_col.set(qn('w:num'), '1')
                        sectPr.append(one_col)

                    # Create section title
                    section_title = f"{section_number}. 본문정리"
                    para = doc.add_paragraph(style='Heading 1')
                    run = para.add_run(section_title)
                    # Explicitly set font on run to fix "no cover" issue
                    run.font.name = self.selected_font_for_title
                    run._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_title)

                    # Get passage IDs to extract
                    text_content = self.global_text_input_to_extract.get("1.0", "end-1c")
                    lines = [line.strip() for line in text_content.split("\n") if line.strip()]

                    추출할지문번호 = []
                    for line in lines:
                        if line:
                            stripped_passage_id = line.strip()
                            if stripped_passage_id not in 추출할지문번호:
                                추출할지문번호.append(stripped_passage_id)
                            else:
                                self.app_logger.info(f"●[오류] 지문번호 중복: {stripped_passage_id}\n"
                                                    f"첫 번째 지문만 추출됩니다.")



                    # Build mappings for all enabled sheets
                    type_to_passage_data = {}
                    for type_name in enabled_types:
                        if type_name in wb.sheetnames:
                            sheet = wb[type_name]
                            passage_id_to_row_data = {}
                            for row in sheet.iter_rows(min_row=1, values_only=True):
                                if row[0] is not None:
                                    passage_id = row[0].strip()
                                    passage_id_to_row_data[passage_id] = row[:5]
                            type_to_passage_data[type_name] = passage_id_to_row_data

                    # Process each passage_id
                    for passage_idx, passage_id in enumerate(추출할지문번호):
                        

                        # Get passage data from the first available type
                        passage_data = None
                        passage_text = None
                        for type_name in enabled_types:
                            if type_name in type_to_passage_data:
                                if passage_id in type_to_passage_data[type_name]:
                                    row_data = type_to_passage_data[type_name][passage_id]
                                    row_data = tuple("" if cell is None else cell for cell in row_data)
                                    if len(row_data) < 5:
                                        row_data += ("",) * (5 - len(row_data))
                                    title, passage, result, answer_part, explanation_part = row_data[:5]
                                    passage_text = passage
                                    break

                        if not passage_text:
                            self.app_logger.error(f"[에러★] '{passage_id}' 지문을 찾을 수 없습니다.")
                            continue

                        # For subsequent passages (after first), switch to single-column before passage ID
                        if passage_idx > 0 and needs_single_column_for_passage_table:
                            doc.add_section(WD_SECTION.NEW_PAGE)
                            new_sec = doc.sections[-1]
                            sectPr = new_sec._sectPr
                            for cols in sectPr.xpath('./w:cols'):
                                sectPr.remove(cols)
                            one_col = OxmlElement('w:cols')
                            one_col.set(qn('w:num'), '1')
                            sectPr.append(one_col)
                            new_sec.top_margin = Cm(self.margin_top)
                            new_sec.bottom_margin = Cm(self.margin_bottom)
                            new_sec.left_margin = Cm(self.margin_left)
                            new_sec.right_margin = Cm(self.margin_right)
                            self._apply_mirror_margins(new_sec)

                        # Add passage ID as heading
                        para_title = doc.add_paragraph(style='Heading_num')
                        run = para_title.add_run(str(passage_id))
                        # Explicitly set font on run to fix "no cover" issue
                        run.font.name = self.selected_font_for_num
                        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_num)

                        # Determine passage to display (will be shown before 지문분석 if enabled)
                        # If 지문분석 is enabled, extract passage from its data
                        passage_to_display = passage_text
                        if "지문분석" in enabled_types and "지문분석" in type_to_passage_data:
                            if passage_id in type_to_passage_data["지문분석"]:
                                지문분석_row_data = type_to_passage_data["지문분석"][passage_id]
                                지문분석_row_data = tuple("" if cell is None else cell for cell in 지문분석_row_data)
                                if len(지문분석_row_data) >= 3:
                                    지문분석_result = 지문분석_row_data[2]  # result column
                                    if 지문분석_result and "# 1. 전체 지문의 내용 구조 정리" in 지문분석_result:
                                        # Extract passage (everything before the first markdown heading)
                                        passage_to_display = 지문분석_result.split("# 1. 전체 지문의 내용 구조 정리")[0].strip()

                        # Note: Passage will be displayed right before 지문분석 section (if enabled)
                        # or at the beginning if 지문분석 is not enabled

                        # Process each subsection in order from rows_order
                        # Extract 교재 types from rows_order in the user's specified order
                        교재_in_order = [t for t in rows_order if t in 교재_types and t in enabled_types]

                        # Determine page break logic:
                        # - 지문분석 and 한글해석: no page break
                        # - Others: page break
                        subsections_order = []
                        for i, subsection_name in enumerate(교재_in_order):
                            # Apply page break for all except 지문분석 and 한글해석
                            needs_page_break = subsection_name not in ["지문분석", "한글해석"]
                            subsections_order.append((subsection_name, needs_page_break))

                        passage_displayed = False  # Track if passage has been shown
                        한글해석_displayed_in_table = False  # Track if 한글해석 was displayed alongside passage

                        # [FIX] Display passage ALWAYS at the top, before processing any subsections
                        if passage_to_display:
                            # Apply sentence splitting at circled numbers if enabled
                            processed_passage = passage_to_display
                            if self.jimun_analysis_sentence_split:
                                processed_passage = self.split_text_at_circled_numbers(passage_to_display)
                            passage_paragraphs = processed_passage.split('\n')

                            # Check if we should use two-column layout with interpretation on the right
                            if self.jimun_analysis_interpretation_right:
                                # Create two-column table (7:3 ratio)
                                left_cell, right_cell = self.create_two_column_passage_table(doc)

                                # Put passage in left cell
                                self.apply_jimun_analysis_formatting_to_cell(left_cell, passage_paragraphs)

                                # Check if 한글해석 is in enabled_types and has data
                                if "한글해석" in enabled_types and "한글해석" in type_to_passage_data:
                                    if passage_id in type_to_passage_data["한글해석"]:
                                        한글해석_row_data = type_to_passage_data["한글해석"][passage_id]
                                        한글해석_row_data = tuple("" if c is None else c for c in 한글해석_row_data)
                                        if len(한글해석_row_data) >= 3:
                                            한글해석_result = 한글해석_row_data[2]  # result column
                                            if 한글해석_result and not 한글해석_result.startswith("Error"):
                                                # Apply circled numbers if 지문분석 is also enabled
                                                if "지문분석" in enabled_types:
                                                    한글해석_result = self.add_circled_numbers_to_sentences(한글해석_result)
                                                # Put 한글해석 content in right cell (use hanjul_haesuk formatting)
                                                한글해석_paragraphs = 한글해석_result.split('\n')
                                                self.apply_hanjul_haesuk_formatting_to_cell(right_cell, 한글해석_paragraphs)
                                                한글해석_displayed_in_table = True
                                # If 한글해석 not available, right cell remains empty

                                # Switch back to two-column for subsequent sections (지문분석, etc.)
                                if needs_single_column_for_passage_table:
                                    doc.add_section(WD_SECTION.NEW_PAGE)
                                    new_sec = doc.sections[-1]
                                    sectPr = new_sec._sectPr
                                    for cols in sectPr.xpath('./w:cols'):
                                        sectPr.remove(cols)
                                    cols = OxmlElement('w:cols')
                                    cols.set(qn('w:num'), '2')
                                    cols.set(qn('w:sep'), '1')
                                    cols.set(qn('w:space'), str(self.cm_to_twips(self.col_spacing)))
                                    sectPr.append(cols)
                                    new_sec.top_margin = Cm(self.margin_top)
                                    new_sec.bottom_margin = Cm(self.margin_bottom)
                                    new_sec.left_margin = Cm(self.margin_left)
                                    new_sec.right_margin = Cm(self.margin_right)
                                    self._apply_mirror_margins(new_sec)
                            else:
                                # Use the existing bordered paragraph approach
                                bordered_para = self.create_bordered_paragraph(doc)
                                self.apply_jimun_analysis_formatting(bordered_para, passage_paragraphs)

                            passage_displayed = True

                        for subsection_name, needs_page_break in subsections_order:
                            # Skip 한글해석 if it was already displayed in the two-column table
                            if subsection_name == "한글해석" and 한글해석_displayed_in_table:
                                continue
                            if subsection_name not in enabled_types:
                                continue

                            if subsection_name not in type_to_passage_data:
                                continue

                            if passage_id not in type_to_passage_data[subsection_name]:
                                print(f"    !!!  '{subsection_name}' data not found for {passage_id}")
                                continue

                            # Get question data
                            row_data = type_to_passage_data[subsection_name][passage_id]
                            row_data = tuple("" if cell is None else cell for cell in row_data)
                            if len(row_data) < 5:
                                row_data += ("",) * (5 - len(row_data))
                            title, passage, result, answer_part, explanation_part = row_data[:5]

                            # Skip if error or empty
                            if result and result.startswith("Error"):
                                print(f"    !!!  '{subsection_name}' has error, skipping")
                                continue
                            if not result or result.strip() == "":
                                print(f"    !!!  '{subsection_name}' is empty, skipping")
                                continue

                            # Process through editing function
                            function_name = f"{subsection_name}_편집"
                            if function_name in FUNCTION_MAP:
                                processed = self.process_question_part(
                                    doc, function_name, title, passage, result,
                                    answer_part, explanation_part
                                )

                                if processed:
                                    (title, passage, question_line, question_part,
                                    answer_part, explanation_part, cell, table) = processed

                                    # Add subsection title (► 지문분석, etc.) with bold and larger font
                                    # Skip main title for 한줄영작연습 (will use numbered titles instead)
                                    if subsection_name != "한줄영작연습":
                                        subsection_para = doc.add_paragraph(style='Heading_num')
                                        run = subsection_para.add_run(f"► {subsection_name}")
                                        run.bold = True
                                        run.font.size = Pt(self.size_text + 1)
                                        # Explicitly set font on run to fix "no cover" issue
                                        run.font.name = self.selected_font_for_num
                                        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_num)

                                    # Add question content based on type
                                    if subsection_name == "한글해석":
                                        # For 한글해석, use bordered paragraph
                                        # If both 지문분석 and 한글해석 are being processed, add numbered sentences
                                        if "지문분석" in enabled_types:
                                            def get_circled_number(n):
                                                """Convert number to circled Unicode character"""
                                                if 1 <= n <= 20:
                                                    return chr(9311 + n)  # ① to ⑳
                                                elif 21 <= n <= 35:
                                                    return chr(12860 + n)  # ㉑ to ㉟
                                                elif 36 <= n <= 50:
                                                    return chr(12941 + n)  # ㊱ to ㊿
                                                else:
                                                    return f"({n})"  # Fallback for numbers > 50


                                            # Pre-processing replacements to handle abbreviations
                                            replacements = {
                                                # e.g. Dr. Bell → Dr• Bell   (won't split after Dr•)
                                                # Note: Removed measurement units (in/ft/oz/lb/ct/yr/wk/mo) from general list - handled separately below
                                                r'\b(Mr|Mrs|Ms|Mt|Dr|Prof|St|Ave|Blvd|Capt|Col|Gen|Lt|Pvt|Jr|Sr|Rev|Etc|Inc|Ltd|Co|esp|Fig|Op|No|Vol|Rd|Tel|Gov|Esq|Maj|Sgt|Cpl|Adm|Mdm|Fr|Bros|Dept|Assn|Univ|Inst|Corp|Rep|Sen|Ed|Fwd|Mfg|Jan|Feb|Mar|Apr|Aug|Sept|Oct|Nov|Dec|CEO|CFO|CIO|vs)\.\s+': r'\1• ',
                                                # Measurement units - only when preceded by digit or other abbreviation
                                                r'(\d+|sq|cu)\s+(in|ft|oz|lb|ct|yr|wk|mo)\.\s+': r'\1 \2• ',
                                                # e.g. U.S. Army → U.S• Army  (second dot becomes •)
                                                r'(?<=\.[A-Za-z])\.\s+': '• ',
                                            }
                                            replacement_patterns = [
                                                (". ", ".\n"), (".) ", ".)\n"), ("? ", "?\n"), ("?) ", "?)\n"), ("! ", "!\n"), ("!) ", "!)\n"), #기본
                                                ('." ', '."\n'), ('?" ', '?"\n'), ('!" ', '!"\n'), #큰따옴표
                                                (".' ", ".'\n"), ("?' ", "?'\n"), ("!' ", "!'\n"), #작은따옴표
                                                ('.’ ', ".’\n"), ('?’ ', "?’\n"), ('!’ ', '!’\n'), #tick
                                                (".” ", ".”\n"), ("?” ", "?”\n"), ("!” ", "!”\n") #double tick
                                            ]

                                            # Normalize line endings to avoid losing paragraph breaks on Windows inputs
                                            normalized_question_part = question_part.replace('\r\n', '\n').replace('\r', '\n')

                                            parts = re.split(r'(\n+)', normalized_question_part)
                                            processed_parts = []
                                            sentence_idx = 1

                                            for part in parts:
                                                if re.fullmatch(r'\n+', part):
                                                    processed_parts.append(part)
                                                    continue

                                                if not part.strip():
                                                    processed_parts.append(part)
                                                    continue

                                                processed_segment = part
                                                for pattern, replacement in replacements.items():
                                                    processed_segment = re.sub(pattern, replacement, processed_segment)

                                                for old, new in replacement_patterns:
                                                    processed_segment = processed_segment.replace(old, new)

                                                processed_segment = processed_segment.replace("•", ".")
                                                sentences = [s.strip() for s in processed_segment.split('\n') if s.strip()]

                                                numbered_segment_sentences = []
                                                for sentence in sentences:
                                                    numbered_segment_sentences.append(f"{get_circled_number(sentence_idx)}{sentence}")
                                                    sentence_idx += 1

                                                processed_parts.append(' '.join(numbered_segment_sentences))

                                            question_part = ''.join(processed_parts)

                                        
                                        bordered_q_para = self.create_bordered_paragraph(doc)
                                        q_paragraphs = question_part.split('\n')
                                        self.apply_italic_to_paragraph(bordered_q_para, q_paragraphs)

                                    elif subsection_name == "단어정리":
                                        # For 단어정리, use 2-column table when two_column is enabled, otherwise 4-column
                                        if self.two_column:
                                            self.create_vocabulary_table_2col(doc, question_part)
                                        else:
                                            self.create_vocabulary_table_4col(doc, question_part)
                                    elif subsection_name == "지문분석":
                                        # For 지문분석, extract markdown part (without passage)
                                        markdown_content = question_part
                                        if "# 1. 전체 지문의 내용 구조 정리" in question_part:
                                            # Extract only the markdown part (from the first heading onwards)
                                            parts = question_part.split("# 1. 전체 지문의 내용 구조 정리", 1)
                                            markdown_content = "# 1. 전체 지문의 내용 구조 정리" + parts[1]
                                        self.add_markdown_content(doc, markdown_content)
                                    elif subsection_name == "직독직해":
                                        # For 직독직해, process each line as-is
                                        for line in question_part.split('\n'):
                                            self.add_paragraph_for_교재제작(doc, line, style='Normal_modified')


                                    elif subsection_name == "한줄해석연습":
                                        lines = question_part.split('\n')
                                        in_english_section = False
                                        in_korean_section = False
                                        skip_korean_section = False  # Flag to skip the entire Korean answer section

                                        for line in lines:
                                            stripped = line.strip()

                                            # Check if we're entering the Korean section
                                            if stripped == "[해석]":
                                                # Only add page break and [정답] if include_answer is enabled
                                                if self.hanjul_interpretation_include_answer:
                                                    # Add page break before Korean section
                                                    self.add_page_break(doc)
                                                    in_english_section = False
                                                    in_korean_section = True
                                                    skip_korean_section = False
                                                    # Replace header text and add with normal style
                                                    self.add_paragraph_for_교재제작(doc, "[한줄해석연습 정답]", style='Normal_modified')
                                                else:
                                                    # Skip the entire Korean section (including the [해석] header)
                                                    in_english_section = False
                                                    in_korean_section = False
                                                    skip_korean_section = True
                                                continue  # Skip writing the [해석] line itself
                                            elif stripped == "[영어]":
                                                # Entering English section - reset skip flag
                                                skip_korean_section = False
                                                in_english_section = True
                                                in_korean_section = False
                                                # Replace header text
                                                self.add_paragraph_for_교재제작(doc, "다음 주어진 문장의  한글 해석을 쓰세요.", style='Normal_modified')
                                                continue

                                            # If we're skipping the Korean section, skip all lines until next section marker
                                            if skip_korean_section:
                                                continue

                                            # Check if this is a numbered sentence in the English section
                                            if in_english_section and stripped and stripped[0].isdigit() and '.' in stripped[:3]:
                                                # This is a numbered English sentence
                                                self.add_paragraph_for_교재제작(doc, line, style='Normal_modified')
                                                # Add arrow line
                                                self.add_paragraph_for_교재제작(doc, '→', style='Normal_modified')
                                                # Add blank line
                                                self.add_paragraph_for_교재제작(doc, '', style='Normal_modified')
                                            elif in_korean_section:
                                                # Korean section - use smaller font size
                                                para = self.add_paragraph_for_교재제작(doc, line, style='Normal_modified')
                                                # Reduce font size by 1 for all runs in this paragraph
                                                for run in para.runs:
                                                    if run.font.size:
                                                        run.font.size = Pt(self.size_text - 1)
                                            else:
                                                # All other lines (blank lines, etc.)
                                                self.add_paragraph_for_교재제작(doc, line, style='Normal_modified')
                                    

                                    elif subsection_name == "한줄영작연습":
                                        # Phase 1: Data Collection (BEFORE Skip Logic)
                                        lines = question_part.split('\n')
                                        english_sentences = []
                                        korean_sentences = []
                                        in_english_section = False
                                        in_korean_section = False

                                        # Collect all sentences (ignore include_answer flag during collection)
                                        for line in lines:
                                            stripped = line.strip()

                                            if stripped == "[영어]":
                                                in_english_section = True
                                                in_korean_section = False
                                                continue
                                            elif stripped == "[해석]":
                                                in_english_section = False
                                                in_korean_section = True
                                                continue

                                            # Collect ALL sentences
                                            if in_english_section and stripped:
                                                english_sentences.append(stripped)
                                            elif in_korean_section and stripped:
                                                korean_sentences.append(stripped)

                                        # Guard against mismatched lengths
                                        if len(korean_sentences) != len(english_sentences):
                                            self.app_logger.warning(
                                                f"한줄영작연습: Korean sentences ({len(korean_sentences)}) != "
                                                f"English sentences ({len(english_sentences)}). Padding shorter list."
                                            )
                                            max_len = max(len(korean_sentences), len(english_sentences))
                                            korean_sentences.extend([''] * (max_len - len(korean_sentences)))
                                            english_sentences.extend([''] * (max_len - len(english_sentences)))

                                        # Phase 2: Render Question Sections (Hint Modes)
                                        # Order: word_scramble → word_form_change → korean_only
                                        modes_to_render = []
                                        if self.hanjul_writing_word_scramble:
                                            modes_to_render.append('word_scramble')
                                        if self.hanjul_writing_word_form_change:
                                            modes_to_render.append('word_form_change')
                                        if self.hanjul_writing_korean_only:
                                            modes_to_render.append('korean_only')

                                        for idx, mode in enumerate(modes_to_render):
                                            if idx > 0:
                                                self.add_page_break(doc)

                                            # Add section title with numbered suffix (using heading style)
                                            sub_section_number = idx + 1
                                            section_title = f"► 한줄영작연습 ({sub_section_number})"
                                            subsection_para = doc.add_paragraph(style='Heading_num')
                                            run = subsection_para.add_run(section_title)
                                            run.bold = True
                                            run.font.size = Pt(self.size_text + 1)
                                            # Explicitly set font on run to fix "no cover" issue
                                            run.font.name = self.selected_font_for_num
                                            run._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_num)

                                            if mode == 'word_scramble':
                                                self.add_paragraph_for_교재제작(doc, "다음 주어진 단어를 올바른 순서로 배열하여 영어로 영작하세요.", style='Normal_modified')
                                                for korean_sent, english_sent in zip(korean_sentences, english_sentences):
                                                    self.add_paragraph_for_교재제작(doc, korean_sent, style='Normal_modified')
                                                    # Tokenize, shuffle, format as (word1 / word2 / ...)
                                                    tokens = self._tokenize_keep_brackets(english_sent)
                                                    random.shuffle(tokens)
                                                    # Strip brackets and lowercase (except proper nouns and "I")
                                                    processed_tokens = []
                                                    for token in tokens:
                                                        stripped = token.strip('[]<>')
                                                        # Check if this is a proper noun or "I"
                                                        if self.nlp is not None:
                                                            doc_token = self.nlp(token.strip('[]<>'))
                                                            # Keep original case for proper nouns and "I"
                                                            if doc_token and (doc_token[0].pos_ == "PROPN" or stripped == "I"):
                                                                processed_tokens.append(stripped)
                                                            else:
                                                                processed_tokens.append(stripped.lower())
                                                        else:
                                                            # Fallback: only preserve "I"
                                                            if stripped == "I":
                                                                processed_tokens.append(stripped)
                                                            else:
                                                                processed_tokens.append(stripped.lower())
                                                    hint = f"({' / '.join(processed_tokens)})"
                                                    self.add_paragraph_for_교재제작(doc, hint, style='Normal_modified')
                                                    # Add arrow line for handwriting space
                                                    self.add_paragraph_for_교재제작(doc, '→', style='Normal_modified')
                                                    # Add blank line (double if >20 words)
                                                    word_count = len(english_sent.split())

                                                    if self.two_column:
                                                        self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')                                                        
                                                        self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')                                                        
                                                        self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')                                                        

                                                        if word_count > 15:
                                                            self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')
                                                            self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')

                                                        if word_count > 25:
                                                            self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')
                                                            self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')

                                                        if word_count > 35:
                                                            self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')
                                                            self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')
                                                            
                                                    else:
                                                        self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')
                                                        if word_count > 20:
                                                            self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')
                                                        if word_count > 40:
                                                            self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')
                                                            self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')



                                            elif mode == 'word_form_change':
                                                self.add_paragraph_for_교재제작(doc, "다음 주어진 단어를 활용하여 영어로 영작하세요. (어형변화 가능, 단어추가 가능)", style='Normal_modified')
                                                for korean_sent, english_sent in zip(korean_sentences, english_sentences):
                                                    # Calculate word count from English answer (excluding sentence number)
                                                    # Remove leading number pattern like "1. ", "2. ", etc.
                                                    english_text_only = re.sub(r'^\d+\.\s*', '', english_sent)
                                                    word_count = len(english_text_only.split())
                                                    # Add Korean sentence with word count
                                                    korean_with_count = f"{korean_sent} (총 {word_count}단어)"
                                                    self.add_paragraph_for_교재제작(doc, korean_with_count, style='Normal_modified')
                                                    # Lemmatize tokens (keeping brackets in place)
                                                    lemmatized_tokens = self._lemmatize_tokens(english_sent)
                                                    random.shuffle(lemmatized_tokens)
                                                    # Strip brackets and join
                                                    stripped_tokens = [t.strip('[]<>') for t in lemmatized_tokens]
                                                    hint = f"({' / '.join(stripped_tokens)})"
                                                    self.add_paragraph_for_교재제작(doc, hint, style='Normal_modified')
                                                    # Add arrow line for handwriting space
                                                    self.add_paragraph_for_교재제작(doc, '→', style='Normal_modified')
                                                    # Add blank line (double if >20 words)
                                                    self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')
                                                    if word_count > 20:
                                                        self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')

                                            elif mode == 'korean_only':
                                                self.add_paragraph_for_교재제작(doc, "다음 주어진 문장을 영어로 영작하세요.", style='Normal_modified')
                                                for korean_sent, english_sent in zip(korean_sentences, english_sentences):
                                                    self.add_paragraph_for_교재제작(doc, korean_sent, style='Normal_modified')
                                                    # Add arrow line for handwriting space
                                                    self.add_paragraph_for_교재제작(doc, '→', style='Normal_modified')
                                                    # Add blank line (double if >20 words)
                                                    word_count = len(english_sent.split())
                                                    self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')
                                                    if word_count > 20:
                                                        self.add_paragraph_for_교재제작(doc, "", style='Normal_modified')

                                        # Phase 3: Render "[정답]" Section (Once, at End)
                                        if self.hanjul_writing_include_answer and modes_to_render:
                                            self.add_page_break(doc)
                                            self.add_paragraph_for_교재제작(doc, "[한줄영작연습 정답]", style='Normal_modified')
                                            for english_sent in english_sentences:
                                                self.add_paragraph_for_교재제작(doc, english_sent, style='Normal_modified')

                                    # Add blank line or page break
                                    if needs_page_break:
                                        self.add_page_break(doc)
                                    else:
                                        doc.add_paragraph(style='Normal_modified')  # blank line

                        # Add page break after processing this passage, unless it's the last one
                        if passage_idx < len(추출할지문번호) - 1:
                            self.add_page_break(doc)

                    # Add to sections_dict
                    for passage_id in 추출할지문번호:
                        sections_dict[section_title].append(
                            (passage_id, " ", " ")
                        )

                    section_number += 1




            # --------------------------------------------------------------
            # D) Handle Mock Tests (실전모의고사)
            # --------------------------------------------------------------

            elif "실전모의고사" in row_key and process_mock_test:
                
                any_sheet_processed = True
                self.handle_mock_test_문제지(wb, row_key, doc, section_number, sections_dict)
                section_number += 1
                

            # --------------------------------------------------------------
            # E) Handle Past Test (기출문제세트)
            # --------------------------------------------------------------
            elif "기출문제세트" in row_key and process_past_test:

                any_sheet_processed = True
                self.handle_past_test_문제지(wb, row_key, doc, section_number, sections_dict)
                section_number += 1
                



            # Add section break after processing all sheets for this row_key
            # Add section break after processing all sheets for this row_key
            if any_sheet_processed:
                
                # Check if this is the last row that will be processed
                is_last_processed_row = True
                for future_idx in range(idx + 1, len(rows_order)):
                    future_row_key = rows_order[future_idx]
                    future_normal_key = f"{future_row_key}_Normal"
                    future_hard_key = f"{future_row_key}_Hard"
                    if (last_state.get(future_normal_key, False) or 
                        last_state.get(future_hard_key, False) or
                        last_state.get(future_row_key, False)):
                        is_last_processed_row = False
                        break
                
                # Always add section break between different sections unless:
                # 1. The next row will trigger a column change (will_next_need_column_change)
                # 2. This is the last row and we're in two-column mode
                
                if is_last_processed_row and self.current_column_mode:
                    pass
                elif will_next_need_column_change:
                    pass
                elif not self.new_page_per_section:
                    pass  # Skip section break if "유형마다 새 페이지" is unchecked
                else:
                    self.add_section_break(doc)




        # -------------------------------------------------------------------------
        # 5) Add "정답 및 해설" Section
        # -------------------------------------------------------------------------
        # Check if we need to create answer sections
        section_titles = list(sections_dict.keys())
        should_create_answers = self.should_create_answer_sections(section_titles)

        if not should_create_answers:
            print("정답 및 해설 및 빠른 정답 찾기 섹션 생략 (처리된 모든 유형이 정답/해설이 필요 없는 유형)")

            # Remove trailing empty content that creates blank pages
            # Step 1: Remove trailing empty paragraphs
            while doc.paragraphs and not doc.paragraphs[-1].text.strip():
                p = doc.paragraphs[-1]._element
                p.getparent().remove(p)

            # Step 2: Remove the last section break if one was added after final content
            # This prevents blank page at end when no answer sections follow
            if len(doc.sections) > 1:
                # Get the body element
                body = doc._element.body
                # Find and remove any sectPr in the last paragraph's pPr
                for para in reversed(list(body)):
                    if para.tag == qn('w:p'):
                        pPr = para.find(qn('w:pPr'))
                        if pPr is not None:
                            sectPr = pPr.find(qn('w:sectPr'))
                            if sectPr is not None:
                                # This paragraph has a section break - remove it
                                pPr.remove(sectPr)
                                # If pPr is now empty, remove it
                                if len(pPr) == 0:
                                    para.remove(pPr)
                                break  # Found and removed the last section break

        # Check if we should create separate answer file
        if self.separate_answer_key and should_create_answers:
            # Save main document (questions only) without answer sections
            try:
                doc.save(output_doc)
            except Exception as e:
                self.app_logger.error(f"Failed to save main document '{output_doc}': {e}")
                return None

            # Process text replacement for main document
            self.process_text_replacement(output_doc)

            # Create separate answer document
            answer_doc_path = self._create_answer_document(output_doc, sections_dict, should_create_answers)

            # Log completion for both files
            output_doc_normalized = unicodedata.normalize('NFC', output_doc)
            answer_doc_normalized = unicodedata.normalize('NFC', answer_doc_path)
            self.app_logger.info(
                f"-------------------\n"
                f"편집본 추출 완료 (정답지 분리)\n"
                f"-------------------\n"
                f"문제 파일: {output_doc_normalized}\n"
                f"정답지: {answer_doc_normalized}"
            )
            self.close_export_popup = True
            return self.export_warnings

        if should_create_answers:
            if self.current_column_mode:
                # Use ODD_PAGE section break to ensure it starts on odd page
                doc.add_section(WD_SECTION.ODD_PAGE)
                new_sec = doc.sections[-1]
                sectPr = new_sec._sectPr

                # Remove any existing multi-column definition
                for cols in sectPr.xpath('./w:cols'):
                    sectPr.remove(cols)

                # Set it back to one column
                one_col = OxmlElement('w:cols')
                one_col.set(qn('w:num'), '1')
                sectPr.append(one_col)
            else:
                # Even if already in single-column mode, add ODD_PAGE section break
                doc.add_section(WD_SECTION.ODD_PAGE)
                new_sec = doc.sections[-1]

            # Update margins for the new section
            new_sec.top_margin = Cm(self.margin_top)
            new_sec.bottom_margin = Cm(self.margin_bottom)
            new_sec.left_margin = Cm(self.margin_left)
            new_sec.right_margin = Cm(self.margin_right)
            self._apply_mirror_margins(new_sec)


            print("해설지 시작")
            doc.add_paragraph("\n\n\n\n\n")  # spacing before section
            paragraph = doc.add_paragraph("정답 및 해설", style='Heading 1')
            paragraph.alignment = WD_PARAGRAPH_ALIGNMENT.CENTER





            if paragraph.runs:
                paragraph.runs[0].font.size = Pt(20)
            else:
                run = paragraph.add_run("정답 및 해설")
                run.font.size = Pt(20)

            doc.add_page_break()


            # Identify section titles
            try:
                section_titles = list(sections_dict.keys())
            except IndexError:
                messagebox.showerror("오류", "●추출할 지문이 없습니다.")
                self.app_logger.info("●[오류] 추출할 지문이 없습니다.")
                return None
            except Exception as e:
                self.app_logger.info(f"An unexpected error occurred: {e}")
                return None



            # Decide what the last_section_title is
            try:
                # Filter out sections that don't need answer sections using helper method
                filtered_section_titles = [
                    t for t in section_titles
                    if not self.should_skip_section_in_answers(t)
                ]

                if filtered_section_titles:
                    last_section_title = filtered_section_titles[-1]
                else:
                    last_section_title = section_titles[-1]
            except IndexError as e:

                def nothing_to_export():
                    error_dialog = Toplevel()
                    error_dialog.title("오류")
                    error_dialog.geometry("500x150")
                    error_dialog.grab_set()
                    error_dialog.focus_set()
                    error_dialog.grid_rowconfigure(0, weight=1)
                    error_dialog.grid_columnconfigure(0, weight=1)

                    CTkLabel(
                        error_dialog,
                        fg_color="white",
                        width=300,
                        height=120,
                        text="-추출할 지문이 없습니다.\n'추출할 지문번호 입력'창에 지문번호가 올바르게 입력되어 있는지 확인하세요!",
                        text_color="black"
                    ).grid(row=0, column=0, sticky="nsew")

                    button_frame = CTkFrame(error_dialog, fg_color="white", width=300, height=30, corner_radius=0)
                    button_frame.grid(row=1, column=0, sticky="swe")
                    button_frame.grid_columnconfigure(0, weight=1)

                    def on_ok():
                        self.close_export_popup = True
                        error_dialog.grab_release()
                        error_dialog.destroy()

                    ok_button = CTkButton(
                        button_frame,
                        border_width=2,
                        border_color="gray",
                        text="OK",
                        text_color="black",
                        width=70,
                        height=30,
                        command=on_ok,
                        fg_color="#FEF9E0",
                        hover_color="#DDA15C"
                    )
                    ok_button.grid(row=0, column=0, padx=10, pady=10)

                self.close_export_popup = False
                nothing_to_export()
                return None
            except Exception as e:
                self.app_logger.info(f"An unexpected error occurred: {e}")
                return None

            # Make the 해설지
            self.process_sections_dict_해설지(doc, sections_dict, last_section_title)


        # -------------------------------------------------------------------------
        # 6) "빠른 정답 찾기" Section
        # -------------------------------------------------------------------------
        if should_create_answers:
            # Add ODD_PAGE section break to ensure it starts on odd page
            doc.add_section(WD_SECTION.ODD_PAGE)
            new_sec = doc.sections[-1]

            # Ensure single column mode (in case it wasn't already)
            sectPr = new_sec._sectPr
            for cols in sectPr.xpath('./w:cols'):
                sectPr.remove(cols)
            one_col = OxmlElement('w:cols')
            one_col.set(qn('w:num'), '1')
            sectPr.append(one_col)

            # Update margins
            new_sec.top_margin = Cm(self.margin_top)
            new_sec.bottom_margin = Cm(self.margin_bottom)
            new_sec.left_margin = Cm(self.margin_left)
            new_sec.right_margin = Cm(self.margin_right)
            self._apply_mirror_margins(new_sec)

            #self.app_logger.info("빠른 정답 찾기 시작")
            paragraph = doc.add_paragraph("빠른 정답 찾기", style='Heading 1')
            paragraph.alignment = WD_PARAGRAPH_ALIGNMENT.CENTER
            if paragraph.runs:
                paragraph.runs[0].font.size = Pt(20)
            else:
                run = paragraph.add_run("빠른 정답 찾기")
                run.font.size = Pt(20)



            # Same logic to get titles
            try:
                section_titles = list(sections_dict.keys())
            except IndexError:
                messagebox.showerror("오류", "●추출할 지문이 없습니다.")
                self.app_logger.info("●[오류] 추출할 지문이 없습니다.")
                return
            except Exception as e:
                self.app_logger.info(f"An unexpected error occurred: {e}")
                return

            try:
                filtered_section_titles = [
                    t for t in section_titles
                    if "동반의어_Normal" not in t
                    and "영영정의_Normal" not in t
                    and "커스텀" not in t
                    and "요약문" not in t
                    and "본문정리" not in t
                    and t not in ["동반의어", "영영정의"]
                ]
                if filtered_section_titles:
                    last_section_title = filtered_section_titles[-1]
                else:
                    last_section_title = section_titles[-1]
            except IndexError:
                def nothing_to_export():
                    error_dialog = Toplevel()
                    error_dialog.title("오류")
                    error_dialog.geometry("500x150")
                    error_dialog.grab_set()
                    error_dialog.focus_set()
                    error_dialog.grid_rowconfigure(0, weight=1)
                    error_dialog.grid_columnconfigure(0, weight=1)

                    CTkLabel(
                        error_dialog,
                        fg_color="white",
                        width=300,
                        height=120,
                        text="-추출할 지문이 없습니다.\n'추출할 지문번호 입력'창에 지문번호가 올바르게 입력되어 있는지 확인해주세요.",
                        text_color="black"
                    ).grid(row=0, column=0, sticky="nsew")

                    button_frame = CTkFrame(error_dialog, fg_color="white", width=300, height=30, corner_radius=0)
                    button_frame.grid(row=1, column=0, sticky="swe")
                    button_frame.grid_columnconfigure(0, weight=1)

                    def on_ok():
                        self.close_export_popup = True
                        error_dialog.grab_release()
                        error_dialog.destroy()

                    ok_button = CTkButton(
                        button_frame,
                        border_width=2,
                        border_color="gray",
                        text="OK",
                        text_color="black",
                        width=70,
                        height=30,
                        command=on_ok,
                        fg_color="#FEF9E0",
                        hover_color="#DDA15C"
                    )
                    ok_button.grid(row=0, column=0, padx=10, pady=10)

                self.close_export_popup = False
                nothing_to_export()
                return None
            except Exception as e:
                self.app_logger.info(f"An unexpected error occurred: {e}")
                return None

            self.빠른정답찾기용(doc, sections_dict, last_section_title)

        # -------------------------------------------------------------------------
        # 7) Save the final document (no merging needed)
        # -------------------------------------------------------------------------
        try:
            doc.save(output_doc)
        except Exception as e:
            self.app_logger.error(f"Failed to save Word document '{output_doc}': {e}")
            return None

        # Optional: If you do cover-page text replacements:
        self.process_text_replacement(output_doc)

        output_doc = unicodedata.normalize('NFC', output_doc)

        self.app_logger.info(
            f"-------------------\n편집본 추출 완료\n-------------------\n편집본 경로: {output_doc}"
        )
        self.close_export_popup = True

        # Return export warnings
        return self.export_warnings













    def process_question_part(self, doc, function_name, title, passage, result, answer_part, explanation_part):




        func = FUNCTION_MAP.get(function_name)
        if not func:
            self.app_logger.error(f"Function {function_name} not found in FUNCTION_MAP.")
            # Return default values to prevent further processing
            return title, passage, "", "", answer_part, explanation_part, None, None

        sig = inspect.signature(func)
        params = sig.parameters

        # Assemble the base args
        base_args = [title, passage, result, answer_part, explanation_part]

        # If the function expects the Document:
        if 'doc' in params:
            base_args += [doc]

        # Always pass the logger:
        base_args += [self.app_logger]

        # Now, if this is one of the two special functions, inject the boolean:
        if '연결어' in function_name:
            base_args += [self.row_needs_two_column]

        try:
            processed = func(*base_args)

        except Exception as e:
            self.app_logger.error(f"Error executing {function_name}: {e}")
            # Return default values in case of error
            return title, passage, "", "", answer_part, explanation_part, None, None

        # Initialize default return values
        question_line = ""
        question_part = ""
        cell = None
        table = None



        # Determine the structure of the returned tuple
        if processed and isinstance(processed, tuple):

            if len(processed) >= 6:
                # Expecting at least: title, passage, question_line, question_part, answer_part, explanation_part
                title, passage, question_line, question_part, answer_part, explanation_part = processed[:6]
            elif len(processed) >= 5:
                # Some functions might not return question_line
                title, passage, question_part, answer_part, explanation_part = processed[:5]
                question_line = ""
            else:
                self.app_logger.error(f"Unexpected return format from {function_name}.")
                return title, passage, "", "", answer_part, explanation_part, None, None

            # Check if cell and table are returned
            #if len(processed) >= 8:
            #    # Functions that return cell and table as well
            #    cell, table = processed[6], processed[7]

        else:
            self.app_logger.error(f"Unexpected return format from {function_name}.")
            return title, passage, "", "", answer_part, explanation_part, None, None

        
        return title, passage, question_line, question_part, answer_part, explanation_part, cell, table




    def show_error_and_log(self, message):
        messagebox.showerror("오류", message)
        self.app_logger.error(message)

    def add_answer_and_explanation(self, doc):
        #self.app_logger.info("해설지 시작")
        doc.add_paragraph("\n\n\n\n\n")
        paragraph = doc.add_paragraph("정답 및 해설", style='Heading 1')
        paragraph.alignment = WD_PARAGRAPH_ALIGNMENT.CENTER

        if paragraph.runs:
            paragraph.runs[0].font.size = Pt(20)
        else:
            run = paragraph.add_run("정답 및 해설")
            run.font.size = Pt(20)

        doc.add_page_break()





    def handle_regular_section_해설지(self, doc, section_title, title_answer_explanations, column_configurations, sections_dict):

        # Helper function to extract number and original title when numeric mode is enabled
        def extract_title_parts(title):
            """Extract numeric part and original title based on passage_number_numeric setting."""
            if self.passage_number_numeric:
                title_str = str(title)
                match = re.match(r'^(\d+\.)\s*(.*)', title_str)
                if match:
                    number_part = match.group(1)  # "1."
                    rest_of_title = match.group(2).strip()  # "original_passage_id"
                else:
                    # Fallback: use full title if pattern doesn't match
                    number_part = title_str
                    rest_of_title = ""
                return number_part, rest_of_title
            else:
                # Non-numeric mode: use title as-is, no rest
                return str(title), ""

        if "동반의어_Normal" in section_title or "영영정의_Normal" in section_title or "커스텀_Normal" in section_title or "요약문_Normal" in section_title:
            return
        if section_title == "동반의어" or section_title == "영영정의" or section_title == "커스텀" or section_title == "요약문":
            return

        answer_section_title = f"{section_title}_정답"
        doc.add_paragraph(answer_section_title, style='Heading_ans')

        빈칸암기flag = False
        cleaned_section_title = re.sub(r'^\d+\.\s+', '', section_title)

        if "빈칸암기" in cleaned_section_title:
            빈칸암기flag = True
        else:
            base_section_title = cleaned_section_title.split('_')[0]
            # Also strip space-separated suffixes like "1회", "2회", etc.
            base_section_title = re.sub(r'\s+\d+회$', '', base_section_title).strip()

        


        if not 빈칸암기flag:

            if any(x in base_section_title for x in ["순서", "삽입", "연결어"]):
                # Check if ANY item has [전문해석]
                has_full_translation = any("[전문해석]" in exp for _, _, exp in title_answer_explanations)


            # Special handling for 어휘1단계 - create two-column layout
            if any(x in base_section_title for x in ["어휘1단계", "어휘2단계", "어법1단계", "동반의어문제1", "동반의어문제2", "영영정의문제"]):


                # Check if ANY item has [전문해석]
                has_full_translation = any("[전문해석]" in exp for _, _, exp in title_answer_explanations)

                if not has_full_translation:
                    # Create 4-column table ONCE (번호, 정답, 번호, 정답)
                    total_items = len(title_answer_explanations)
                    items_per_column = (total_items + 1) // 2  # Round up division

                    # Create a table with 4 columns
                    table = doc.add_table(rows=items_per_column + 1, cols=4)

                    # Set headers for both column sets
                    headers = ['번호', '정답', '번호', '정답']
                    for idx, header in enumerate(headers):
                        table.cell(0, idx).text = header

                    # Apply styles to header
                    for cell in table.rows[0].cells:
                        self.set_cell_font_hdr(cell, self.selected_font_for_exp, self.size_exp)
                        self.set_cell_border(
                            cell,
                            top={"sz": 8, "val": "single", "color": "auto"},
                            bottom={"sz": 8, "val": "single", "color": "auto"},
                            left={"sz": 8, "val": "single", "color": "auto"},
                            right={"sz": 8, "val": "single", "color": "auto"}
                        )
                        self.set_cell_margins(cell, top=1, bottom=1, left=1, right=1)

                    # Fill data in two columns
                    for i in range(items_per_column):
                        # Left column data
                        if i < len(title_answer_explanations):
                            title, answer_part, _ = title_answer_explanations[i]
                            number_part, _ = extract_title_parts(title)
                            table.cell(i + 1, 0).text = number_part
                            table.cell(i + 1, 1).text = answer_part

                        # Right column data
                        right_index = i + items_per_column
                        if right_index < len(title_answer_explanations):
                            title, answer_part, _ = title_answer_explanations[right_index]
                            number_part, _ = extract_title_parts(title)
                            table.cell(i + 1, 2).text = number_part
                            table.cell(i + 1, 3).text = answer_part

                    # Apply styles to all data cells
                    for row_idx in range(1, items_per_column + 1):
                        for col_idx in range(4):
                            cell = table.rows[row_idx].cells[col_idx]
                            self.set_cell_paragraph_style(cell, 'Normal_exp')
                            self.set_cell_border(
                                cell,
                                top={"sz": 8, "val": "single", "color": "auto"},
                                bottom={"sz": 8, "val": "single", "color": "auto"},
                                left={"sz": 8, "val": "single", "color": "auto"},
                                right={"sz": 8, "val": "single", "color": "auto"}
                            )
                            self.set_cell_margins(cell, top=5, bottom=5, left=5, right=5)

                    # Set column widths
                    for row in table.rows:
                        row.cells[0].width = Inches(1)  # 번호 column (left)
                        row.cells[1].width = Inches(2.5)  # 정답 column (left)
                        row.cells[2].width = Inches(1)  # 번호 column (right)
                        row.cells[3].width = Inches(2.5)  # 정답 column (right)

                else:  # has_full_translation is True
                    # 어휘1, 어휘2, 어법1에 대해서 [전문해석] 존재할 경우 3칼럼으로 해설지 만들기.
                    if any(x in base_section_title for x in ["어휘1단계", "어휘2단계", "어법1단계"]):
                        if "어휘1단계" in base_section_title or "어법1단계" in base_section_title:
                            col_count, headers, widths = column_configurations["어휘1전문해석있을때"]
                        elif "어휘2단계" in base_section_title:
                            col_count, headers, widths = column_configurations["어휘2전문해석있을때"]


                        table = doc.add_table(rows=1, cols=col_count)

                        # Set header row
                        for idx, header in enumerate(headers):
                            table.cell(0, idx).text = header

                        # Apply styles to header
                        for cell in table.rows[0].cells:
                            self.set_cell_font_hdr(cell, self.selected_font_for_exp, self.size_exp)
                            self.set_cell_border(
                                cell,
                                top={"sz": 8, "val": "single", "color": "auto"},
                                bottom={"sz": 8, "val": "single", "color": "auto"},
                                left={"sz": 8, "val": "single", "color": "auto"},
                                right={"sz": 8, "val": "single", "color": "auto"}
                            )
                            self.set_cell_margins(cell, top=1, bottom=1, left=1, right=1)

                        # Add data rows
                        for title, answer_part, explanation_part in title_answer_explanations:
                            cells = table.add_row().cells

                            # Extract title parts for numeric mode
                            number_part, rest_of_title = extract_title_parts(title)

                            cells[0].text = number_part
                            cells[1].text = answer_part
                            if col_count == 3:
                                explanation_part = explanation_part.replace("[전문해석]", "\n[전문해석]").replace("[지문어휘]", "\n[지문어휘]")

                                # Split into two parts: translation and vocabulary
                                if "[지문어휘]" in explanation_part:
                                    translation_block, _, vocab_block = explanation_part.partition("[지문어휘]")

                                    # Format vocabulary into columns
                                    #어휘1단계, 어법1단계는 3 columns. But 어휘2단계는 2 columns.
                                    lines = [line.strip() for line in vocab_block.strip().splitlines() if line.strip()]

                                    # Determine number of columns based on section type
                                    if "어휘2단계" in base_section_title:
                                        cols = 2
                                        tab_stops = [2.5]  # Single tab stop for 2 columns
                                    else:  # 어휘1단계 or 어법1단계
                                        cols = 3
                                        tab_stops = [1.5, 3]  # Two tab stops for 3 columns

                                    rows = []
                                    for i in range(0, len(lines), cols):
                                        chunk = lines[i:i + cols]
                                        if len(chunk) < cols:
                                            chunk.extend([''] * (cols - len(chunk)))  # pad the last row
                                        rows.append('\t'.join(chunk))

                                    formatted_vocab = '\n'.join(rows)

                                    # Combine translation + formatted vocabulary
                                    formatted_explanation_part = translation_block.strip() + "\n\n[지문어휘]\n" + formatted_vocab
                                else:
                                    # No vocabulary section, just use as-is
                                    formatted_explanation_part = explanation_part

                                # Append original title to explanation in parentheses (for numeric mode)
                                if rest_of_title:
                                    formatted_explanation_part = f"{formatted_explanation_part}\n({rest_of_title})"

                                # Set tab stops based on section type
                                if "어휘2단계" in base_section_title:
                                    tab_stops = [2.5]  # 2-column layout
                                else:
                                    tab_stops = [1.5, 3]  # 3-column layout
                                self.add_formatted_text_to_cell(cells[2], formatted_explanation_part, tab_stops_inches=tab_stops)

                            # Apply styles to content cells
                            for cell in cells:
                                self.set_cell_paragraph_style(cell, 'Normal_exp')
                                self.set_cell_border(
                                    cell,
                                    top={"sz": 8, "val": "single", "color": "auto"},
                                    bottom={"sz": 8, "val": "single", "color": "auto"},
                                    left={"sz": 8, "val": "single", "color": "auto"},
                                    right={"sz": 8, "val": "single", "color": "auto"}
                                )
                                self.set_cell_margins(cell, top=5, bottom=5, left=5, right=5)

                        # Set the widths for each column
                        for row in table.rows:
                            for idx, width in enumerate(widths):
                                row.cells[idx].width = width

            elif any(x in base_section_title for x in ["순서", "삽입", "연결어"]) and has_full_translation:
                if "순서" in base_section_title or "삽입" in base_section_title :
                    col_count, headers, widths = column_configurations["순서삽입전문해석있을때"]
                elif "연결어" in base_section_title:
                    col_count, headers, widths = column_configurations["연결어전문해석있을때"]

                

                table = doc.add_table(rows=1, cols=col_count)

                # Set header row
                for idx, header in enumerate(headers):
                    table.cell(0, idx).text = header

                # Apply styles to header
                for cell in table.rows[0].cells:
                    self.set_cell_font_hdr(cell, self.selected_font_for_exp, self.size_exp)
                    self.set_cell_border(
                        cell,
                        top={"sz": 8, "val": "single", "color": "auto"},
                        bottom={"sz": 8, "val": "single", "color": "auto"},
                        left={"sz": 8, "val": "single", "color": "auto"},
                        right={"sz": 8, "val": "single", "color": "auto"}
                    )
                    self.set_cell_margins(cell, top=1, bottom=1, left=1, right=1)

                for title, answer_part, explanation_part in title_answer_explanations:
                    cells = table.add_row().cells

                    # Extract title parts for numeric mode
                    number_part, rest_of_title = extract_title_parts(title)

                    cells[0].text = number_part
                    cells[1].text = answer_part
                    if col_count == 3:
                        explanation_part = explanation_part.replace("[전문해석]", "\n[전문해석]").replace("[지문어휘]", "\n[지문어휘]")

                        # Split into two parts: translation and vocabulary
                        if "[지문어휘]" in explanation_part:
                            translation_block, _, vocab_block = explanation_part.partition("[지문어휘]")

                            if "순서" in base_section_title or "삽입" in base_section_title:
                                # Format vocabulary into 3 columns
                                lines = [line.strip() for line in vocab_block.strip().splitlines() if line.strip()]
                                cols = 3
                                rows = []
                                for i in range(0, len(lines), cols):
                                    chunk = lines[i:i + cols]
                                    if len(chunk) < cols:
                                        chunk.extend([''] * (cols - len(chunk)))  # pad the last row
                                    rows.append('\t'.join(chunk))
                                formatted_vocab = '\n'.join(rows)
                            elif "연결어" in base_section_title:
                                # Format vocabulary into 2 columns
                                lines = [line.strip() for line in vocab_block.strip().splitlines() if line.strip()]
                                cols = 2
                                rows = []
                                for i in range(0, len(lines), cols):
                                    chunk = lines[i:i + cols]
                                    if len(chunk) < cols:
                                        chunk.extend([''] * (cols - len(chunk)))  # pad the last row
                                    rows.append('\t'.join(chunk))
                                formatted_vocab = '\n'.join(rows)
                            else:
                                # Fallback: use vocabulary as-is
                                formatted_vocab = vocab_block.strip()
                            # Combine translation + formatted vocabulary
                            formatted_explanation_part = translation_block.strip() + "\n\n[지문어휘]\n" + formatted_vocab

                        else:
                            # No vocabulary section, just use as-is
                            formatted_explanation_part = explanation_part

                        # Append original title to explanation in parentheses (for numeric mode)
                        if rest_of_title:
                            formatted_explanation_part = f"{formatted_explanation_part}\n({rest_of_title})"

                        # Set tab stops for 3-column vocabulary layout (adjust values as needed)
                        self.add_formatted_text_to_cell(cells[2], formatted_explanation_part, tab_stops_inches=[1.5, 3])

                    # Apply styles to content cells
                    for cell in cells:
                        self.set_cell_paragraph_style(cell, 'Normal_exp')
                        self.set_cell_border(
                            cell,
                            top={"sz": 8, "val": "single", "color": "auto"},
                            bottom={"sz": 8, "val": "single", "color": "auto"},
                            left={"sz": 8, "val": "single", "color": "auto"},
                            right={"sz": 8, "val": "single", "color": "auto"}
                        )
                        self.set_cell_margins(cell, top=5, bottom=5, left=5, right=5)

                # Set the widths for each column
                for row in table.rows:
                    for idx, width in enumerate(widths):
                        row.cells[idx].width = width



            else: #나머지 다른 유형들

                if base_section_title in column_configurations:
                    col_count, headers, widths = column_configurations[base_section_title]
                    #print(f"Configuration found: {base_section_title} -> {col_count}, {headers}, {widths}")  # Debug

                    table = doc.add_table(rows=1, cols=col_count)

                    # Set header row
                    for idx, header in enumerate(headers):
                        table.cell(0, idx).text = header

                    # Apply styles to header
                    for cell in table.rows[0].cells:
                        self.set_cell_font_hdr(cell, self.selected_font_for_exp, self.size_exp)
                        self.set_cell_border(
                            cell,
                            top={"sz": 8, "val": "single", "color": "auto"},
                            bottom={"sz": 8, "val": "single", "color": "auto"},
                            left={"sz": 8, "val": "single", "color": "auto"},
                            right={"sz": 8, "val": "single", "color": "auto"}
                        )
                        self.set_cell_margins(cell, top=1, bottom=1, left=1, right=1)

                    # Add data rows
                    for title, answer_part, explanation_part in title_answer_explanations:
                        cells = table.add_row().cells

                        # Extract title parts for numeric mode
                        number_part, rest_of_title = extract_title_parts(title)

                        if "영작" in base_section_title:
                            cells[0].text = number_part
                            
                            if explanation_part == "" or explanation_part == None:
                                self.app_logger.error(f"{base_section_title} - {title}: 해설 부분이 비어 있습니다. 해당 문제를 재출제하세요.")
                                messagebox.showerror("Error", f"{base_section_title} - {title}: 해설 부분이 비어 있습니다. 해당 문제를 재출제하세요.")
                                continue

                            # Format [지문어휘] section into 3 columns
                            explanation_part = explanation_part.replace("[지문어휘]", "\n[지문어휘]")

                            # Split into two parts: translation and vocabulary
                            if "[지문어휘]" in explanation_part:
                                translation_block, _, vocab_block = explanation_part.partition("[지문어휘]")

                                # Format vocabulary into 3 columns
                                lines = [line.strip() for line in vocab_block.strip().splitlines() if line.strip()]
                                cols = 3
                                rows = []
                                for i in range(0, len(lines), cols):
                                    chunk = lines[i:i + cols]
                                    if len(chunk) < cols:
                                        chunk.extend([''] * (cols - len(chunk)))  # pad the last row
                                    rows.append('\t'.join(chunk))

                                formatted_vocab = '\n'.join(rows)

                                # Combine translation + formatted vocabulary
                                formatted_explanation_part = translation_block.strip() + "\n\n[지문어휘]\n" + formatted_vocab
                            else:
                                # No vocabulary section, just use as-is
                                formatted_explanation_part = explanation_part

                            # Append original title to explanation in parentheses (for numeric mode)
                            if rest_of_title:
                                formatted_explanation_part = f"{formatted_explanation_part}\n({rest_of_title})"

                            # Set tab stops for 3-column vocabulary layout
                            self.add_formatted_text_to_cell(cells[1], formatted_explanation_part, tab_stops_inches=[1.5, 3])






                        else:
                            cells[0].text = number_part
                            cells[1].text = answer_part
                            if col_count == 3:
                                if (
                                    not any(k in cleaned_section_title for k in ["어휘1", "어법1", "어휘2", "순서", "삽입", "빈칸암기"]) 
                                    and not explanation_part
                                ):
                                    self.show_error_and_log(
                                        f"오류: '{cleaned_section_title}' 유형의 '{title}' 지문에서 문제 오류가 발견되었습니다.\n"
                                        f"(오류내용: '해설'칸이 비어 있음)\n해당 문제를 삭제하거나 재출제하세요.\n\n"
                                        f"(재출제 방법: 해당 유형, 해당 지문번호의 '문제' 칸을 더블클릭하여 내용을 지운 후 다시 'Make Questions' → '출제 시작'을 클릭하세요.)"
                                    )
                                    return
                                else:
                                    #cells[2].text = explanation_part if explanation_part else ""
                                    if explanation_part:
                                        #전문해석, 지문어휘 처리
                                        explanation_part = explanation_part.replace("[지문어휘]", "\n[지문어휘]")

                                        # Split into two parts: translation and vocabulary
                                        if "[지문어휘]" in explanation_part:
                                            translation_block, _, vocab_block = explanation_part.partition("[지문어휘]")


                                            # Format vocabulary into 3 columns
                                            lines = [line.strip() for line in vocab_block.strip().splitlines() if line.strip()]
                                            cols = 3
                                            rows = []
                                            for i in range(0, len(lines), cols):
                                                chunk = lines[i:i + cols]
                                                if len(chunk) < cols:
                                                    chunk.extend([''] * (cols - len(chunk)))  # pad the last row
                                                rows.append('\t'.join(chunk))

                                            formatted_vocab = '\n'.join(rows)

                                            # Combine translation + formatted vocabulary
                                            formatted_explanation_part = translation_block.strip() + "\n\n[지문어휘]\n" + formatted_vocab
                                        else:
                                            # No vocabulary section, just use as-is
                                            formatted_explanation_part = explanation_part

                                        # Append original title to explanation in parentheses (for numeric mode)
                                        if rest_of_title:
                                            formatted_explanation_part = f"{formatted_explanation_part}\n({rest_of_title})"

                                        # Set tab stops for 3-column vocabulary layout (adjust values as needed)
                                        self.add_formatted_text_to_cell(cells[2], formatted_explanation_part, tab_stops_inches=[1.5, 3])
                                    else:
                                        cells[2].text = ""



                        # Apply styles to content cells
                        for cell in cells:
                            self.set_cell_paragraph_style(cell, 'Normal_exp')
                            self.set_cell_border(
                                cell,
                                top={"sz": 8, "val": "single", "color": "auto"},
                                bottom={"sz": 8, "val": "single", "color": "auto"},
                                left={"sz": 8, "val": "single", "color": "auto"},
                                right={"sz": 8, "val": "single", "color": "auto"}
                            )
                            self.set_cell_margins(cell, top=5, bottom=5, left=5, right=5)

                        # Set column widths
                        #for idx, width in enumerate(widths):
                        #    table.rows[-1].cells[idx].width = width

                        # Set the widths for each column
                        for row in table.rows:
                            for idx, width in enumerate(widths):
                                row.cells[idx].width = width


                else:
                    
                    print(f"No column configuration found for section: {base_section_title}")  # Debug
                    


        else: ###빈칸암기 편집용


            for title, answer_part, explanation_part in title_answer_explanations:
                # Extract title parts for numeric mode
                number_part, rest_of_title = extract_title_parts(title)

                para_title = doc.add_paragraph(style='Heading_num')
                run = para_title.add_run(str(number_part))
                # Explicitly set font on run to fix "no cover" issue
                run.font.name = self.selected_font_for_num
                run._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_num)

                # Use FUNCTION_MAP to retrieve the function
                function_name = '빈칸암기_정답_편집'  # Ensure this key exists in FUNCTION_MAP

                # Append original title to explanation if in numeric mode → 취소
                #if rest_of_title:
                #    explanation_part = f"{explanation_part}\n({rest_of_title})" if explanation_part else f"({rest_of_title})"

                # Process the question part
                processed = self.process_question_part(
                    doc, function_name, number_part, answer_part, answer_part, answer_part, explanation_part
                )

                if processed:
                    (title_ret, passage_ret, question_line, question_part, answer_part_ret, explanation_part_ret, cell, table) = processed

                    # Split result by new lines to handle hard enters as separate paragraphs
                    lines = question_part.split('\n')

                    if self.border_flag:
                        # Process each line as a separate bordered paragraph
                        for idx, line_content in enumerate(lines):
                            bordered_para = self.create_bordered_paragraph(doc)

                            # Add borders based on position
                            if idx == 0:
                                # First line - all borders
                                self.add_paragraph_border(bordered_para, top=True, bottom=(len(lines) == 1),
                                                        left=True, right=True)
                            elif idx == len(lines) - 1:
                                # Last line - only left, right, and bottom
                                self.add_paragraph_border(bordered_para, top=False, bottom=True, left=True, right=True)
                            else:
                                # Middle lines - only left and right
                                self.add_paragraph_border(bordered_para, top=False, bottom=False, left=True, right=True)

                            # Apply padding
                            bordered_para.paragraph_format.left_indent = Pt(5)
                            bordered_para.paragraph_format.right_indent = Pt(5)
                            bordered_para.paragraph_format.space_before = Pt(0)
                            bordered_para.paragraph_format.space_after = Pt(0)

                            # Combined pattern for brackets and curly braces
                            combined_pattern = re.compile(r'(\[)(.*?)(])|(\{)([^}]*)(\})')
                            last_idx = 0

                            for match in combined_pattern.finditer(line_content):
                                start, end = match.span()
                                bordered_para.add_run(line_content[last_idx:start])

                                if match.group(1) == '[':  # Bracket match
                                    # Answer sheet: show all letters visible with black font and black underline
                                    word = match.group(2)
                                    if word:
                                        # Show entire word in black with underline
                                        word_run = bordered_para.add_run(word)
                                        word_run.font.color.rgb = RGBColor(0x00, 0x00, 0x00)
                                        shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                        word_run._r.get_or_add_rPr().append(shd)
                                        self.set_underline_color(word_run, '000000')  # Black underline

                                    run = bordered_para.add_run("    ")  # 4 spaces after word
                                    run.font.color.rgb = RGBColor(0x00, 0x00, 0x00)
                                    shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                    run._r.get_or_add_rPr().append(shd)
                                    self.set_underline_color(run, '000000')  # Black underline

                                elif match.group(4) is not None:  # Curly brace match
                                    text_inside_braces = match.group(5)
                                    run = bordered_para.add_run(text_inside_braces)
                                    run.italic = True

                                last_idx = end

                            bordered_para.add_run(line_content[last_idx:])

                    else:
                        # Add lines without borders but with margins
                        for idx, line_content in enumerate(lines):
                            # Combined pattern for brackets and curly braces
                            combined_pattern = re.compile(r'(\[)(.*?)(])|(\{)([^}]*)(\})')
                            para = doc.add_paragraph(style='Normal_modified')

                            if idx == 0:
                                para.paragraph_format.space_before = Pt(1)
                            if idx == len(lines) - 1:
                                para.paragraph_format.space_after = Pt(6)

                            last_idx = 0
                            for match in combined_pattern.finditer(line_content):
                                start, end = match.span()
                                para.add_run(line_content[last_idx:start])

                                if match.group(1) == '[':  # Bracket match
                                    # Answer sheet: show all letters visible with black font and black underline
                                    word = match.group(2)
                                    if word:
                                        # Show entire word in black with underline
                                        word_run = para.add_run(word)
                                        word_run.font.color.rgb = RGBColor(0x00, 0x00, 0x00)
                                        shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                        word_run._r.get_or_add_rPr().append(shd)
                                        self.set_underline_color(word_run, '000000')  # Black underline

                                    run = para.add_run("    ")  # 4 spaces after word
                                    run.font.color.rgb = RGBColor(0x00, 0x00, 0x00)
                                    shd = parse_xml(r'<w:shd {} w:fill="FFFFFF"/>'.format(nsdecls('w')))
                                    run._r.get_or_add_rPr().append(shd)
                                    self.set_underline_color(run, '000000')  # Black underline

                                elif match.group(4) is not None:  # Curly brace match
                                    text_inside_braces = match.group(5)
                                    run = para.add_run(text_inside_braces)
                                    run.italic = True

                                last_idx = end

                            para.add_run(line_content[last_idx:])

                    # Append explanation_part under the bordered paragraphs
                    if explanation_part and explanation_part.strip():
                        explanation_lines = explanation_part.split('\n')
                        for line in explanation_lines:
                            self.add_paragraph_with_formatting(doc, line, style='Normal_modified')

                    # Add a blank line after
                    doc.add_paragraph(style='Normal_modified')





    def handle_mock_test_문제지(self, wb, row_key, doc, section_number, sections_dict):
        """
        Handles the extraction and formatting of mock test sheets.

        Parameters:
            wb (Workbook): The loaded Excel workbook.
            row_key (str): The key identifying the mock test sheet. (실전모의 몇 회인지 표시)
            doc (Document): The Word document being generated.
            section_number (int): The current section number.
            sections_dict (defaultdict): Dictionary to store section data.
        """
        if row_key not in wb.sheetnames:
            self.app_logger.error(f"Sheet {row_key} does not exist in the workbook.")
            return

        extracting_sheet = wb[row_key]
        section_title = f"{section_number}. {row_key}"
        para = doc.add_paragraph(style='Heading 1')
        run = para.add_run(section_title)
        # Explicitly set font on run to fix "no cover" issue
        run.font.name = self.selected_font_for_title
        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_title)

        passage_id_to_row_data = {}
        duplicate_passages = []
        row_count = 0
        for row in extracting_sheet.iter_rows(min_row=1, values_only=True):
            if row[0]:
                row_count += 1
                passage_id = row[0].strip()
                if passage_id in passage_id_to_row_data:
                    # Duplicate detected!
                    duplicate_passages.append(passage_id)
                    self.export_warnings.append((row_key, passage_id, "중복 지문 ID (엑셀에서 같은 지문번호가 2번 이상 나타남, 마지막 것만 처리됨)"))
                passage_id_to_row_data[passage_id] = row[:5]  # Later one overwrites earlier

        ordered_passages = list(passage_id_to_row_data.items())
        valid_passage_ids = []
        for passage_id, row_data in ordered_passages:
            row_data = tuple("" if cell is None else cell for cell in row_data)
            if len(row_data) < 5:
                row_data += ("",) * (5 - len(row_data))
            passageid, passage, result, answer_part, explanation_part = row_data[:5]

            if not passageid:
                continue

            try:
                유형 = passageid.split(" // ")[1].strip()
                원래title = passageid.split(" // ")[0].strip()
            except IndexError:
                continue

            if result in ["문제 미출제 (문장 수 부족)", "", None] or (result and result.startswith("Error")):
                continue

            유형 = 유형 + "_실모"
            function_name = 유형.replace('_Normal', '_편집').replace('_Hard', '_편집').replace("내용일치(한)", "내용일치한")
            if function_name not in FUNCTION_MAP:
                continue

            valid_passage_ids.append(passage_id)

        total_valid_passages = len(valid_passage_ids)
        passages_on_page = 0
        processed_passages_for_breaks = 0

        question_number = 1
        processed_count = 0
        skipped_count = 0

        for passage_id, row_data in ordered_passages:
            # Ensure row_data has at least 5 elements
            row_data = tuple("" if cell is None else cell for cell in row_data)
            if len(row_data) < 5:
                row_data += ("",) * (5 - len(row_data))
            passageid, passage, result, answer_part, explanation_part = row_data[:5]

            if not passageid:
                skipped_count += 1
                continue

            try:
                유형 = passageid.split(" // ")[1].strip()
                원래title = passageid.split(" // ")[0].strip()
            except IndexError:
                self.app_logger.error(f"Invalid passage_id format: {passageid}")
                # Track warning
                self.export_warnings.append((row_key, passageid, "지문 ID 형식 오류 (' // ' 구분자 없음)"))
                skipped_count += 1
                continue

            title = f"{question_number}."
            해설지로보낼title = f"{title} {원래title}"

            if result in ["문제 미출제 (문장 수 부족)", "", None] or (result and result.startswith("Error")):
                reason = result if result else "문제 미출제"
                self.app_logger.info(f"{passageid}: 문제 조건에 맞지 않아 추출하지 않습니다.")
                # Track warning
                if result and result.startswith("Error"):
                    self.export_warnings.append((row_key, 원래title, f"출제 오류: {result}"))
                else:
                    self.export_warnings.append((row_key, 원래title, f"문제 미출제: {reason}"))
                skipped_count += 1
                continue

            유형 = 유형 + "_실모"

            function_name = 유형.replace('_Normal', '_편집').replace('_Hard', '_편집').replace("내용일치(한)", "내용일치한")

            if function_name not in FUNCTION_MAP:
                self.app_logger.error(f"Function {function_name} not found in FUNCTION_MAP.")
                # Track warning
                self.export_warnings.append((row_key, 원래title, f"처리 함수 없음: {function_name}"))
                skipped_count += 1
                continue

            # Process the question part
            processed = self.process_question_part(
                doc, function_name, title, passage, result, answer_part, explanation_part
            )

            if processed:
                title, passage, question_line, question_part, answer_part, explanation_part, cell, table = processed

                # Check if this is 동형문제 with multiple sub-questions
                is_multi_sub = False
                sub_questions = []
                sub_answers = []
                sub_explanations = []

                if "동형문제" in 유형:
                    # Try to split question_part, answer_part, and explanation_part
                    # Uses Korean prefixes: 문제1), 정답1), 해설1), etc.
                    sub_questions = self.split_sub_questions(question_part, prefix="문제") if question_part else []
                    sub_answers = self.split_sub_questions(answer_part, prefix="정답") if answer_part else []
                    sub_explanations = self.split_sub_questions(explanation_part, prefix="해설") if explanation_part else []

                    # Check if we found multiple sub-questions
                    # CRITICAL: Only use sub_questions count - answers/explanations may have false positive markers
                    if len(sub_questions) > 1:
                        is_multi_sub = True

                # Prepare the title
                base_title = title + " " + question_line
                base_title = base_title.strip()

                # If multi-sub, show range in title and reconstruct question_part
                if is_multi_sub:
                    # CRITICAL: Use ONLY sub_questions length - it's the authoritative source
                    # Answers and explanations may contain false positive markers like "$3.59"
                    num_subs = len(sub_questions)
                    display_title = f"[{question_number}-{question_number + num_subs - 1}] 다음 글을 읽고 물음에 답하시오.".strip()

                    # Reconstruct question_part with actual question numbers and newlines
                    reconstructed_questions = []
                    for sub_idx in range(num_subs):
                        sub_q_marker, sub_q_text = sub_questions[sub_idx] if sub_idx < len(sub_questions) else ("", "")
                        actual_q_num = question_number + sub_idx
                        reconstructed_line = f"{actual_q_num}. {sub_q_text}"
                        reconstructed_questions.append(reconstructed_line)

                    # Join with newlines between questions
                    question_part = "\n\n".join(reconstructed_questions)
                else:
                    display_title = base_title

                # Create a paragraph for the title
                para_title = doc.add_paragraph(style='Normal_left')



                if "함축의미" in 유형:
                    self.테이블없는본문_괄호에볼드밑줄더하기(para_title, display_title)

                elif "영작2" in 유형:
                    self.테이블없는본문_영작단어수에볼드밑줄더하기(para_title, display_title)

                else:
                    # Determine which words need to be formatted
                    words_to_format = []
                    if '없는' in display_title:
                        words_to_format.append('없는')
                    if '않는' in display_title:
                        words_to_format.append('않는')
                    if '않은' in display_title:
                        words_to_format.append('않은')
                    if '틀린' in display_title:
                        words_to_format.append('틀린')
                    if '2개' in display_title:
                        words_to_format.append('2개')
                    if '3개' in display_title:
                        words_to_format.append('3개')
                    if '모두' in display_title:
                        words_to_format.append('모두')


                    # Add the title with formatting
                    self.add_formatted_text(para_title, display_title, words_to_format)

                


                #지문
                # Create the table for the passage
                
                if "영작1" in 유형:
                    # Find and capture the "문제파트"
                    문제파트 = ""
                    match = re.search(r"다음의.*?\)", question_part)
                    if match:
                        문제파트 = match.group(0)

                    question_part = question_part.replace(문제파트, "").strip()

                    if '[보기]' in question_part:
                        지문파트, 보기파트 = question_part.split('[보기]', 1)
                        지문파트 = 지문파트.strip()
                        # Check if 보기 has numbered format
                        보기_content = 보기파트.strip()
                        if re.match(r'^\(\d+\)', 보기_content):
                            보기파트 = '[보기]\n' + 보기_content
                        else:
                            보기파트 = '[보기] ' + 보기_content
                    else:
                        지문파트 = question_part.strip()
                        보기파트 = ""

                    # 지문파트 - Create bordered or borderless paragraph based on border_flag
                    if self.border_flag:
                        bordered_para = self.create_bordered_paragraph(doc)
                        paragraphs = 지문파트.split('\n')
                        self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)
                    else:
                        paragraphs = 지문파트.split('\n')
                        for idx, paragraph_text in enumerate(paragraphs):
                            para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')
                            if idx == 0:
                                para.paragraph_format.space_before = Pt(1)
                            if idx == len(paragraphs) - 1:
                                para.paragraph_format.space_after = Pt(6)

                    # 보기파트
                    # Check if 보기파트 has numbered items and add corresponding numbered underlines
                    numbered_items = re.findall(r'^\(\d+\)', 보기파트, re.MULTILINE)
                    if numbered_items:
                        # Multi-item case: add numbered underlines
                        underlines = []
                        for i in range(1, len(numbered_items) + 1):
                            if self.row_needs_two_column:
                                underlines.append(f"→ ({i})_____________________________________________")
                            else:
                                underlines.append(f"→ ({i})____________________________________________________________________________________________")
                        보기파트 = 보기파트 + "\n\n" + "\n".join(underlines)
                    else:
                        # Single-item case: add single underline
                        if self.row_needs_two_column:
                            보기파트 = 보기파트 + "\n\n→ ________________________________________________"
                        else:
                            보기파트 = 보기파트 + "\n\n→ ____________________________________________________________________________________________"


                    for paragraph in 보기파트.split('\n'):
                        self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')
                    
                    cell = None
                    table = None                    


                elif "영작2" in 유형:
                    # Find and capture the "문제파트"
                    문제파트 = ""
                    match = re.search(r"\[보기\]에.*?\)", question_part)
                    if match:
                        문제파트 = match.group(0)

                    question_part = question_part.replace(문제파트, "").strip()

                    if '[보기]' in question_part:
                        지문파트, 보기파트 = question_part.split('[보기]', 1)
                        지문파트 = 지문파트.strip()
                        # Check if 보기 has numbered format
                        보기_content = 보기파트.strip()
                        if re.match(r'^\(\d+\)', 보기_content):
                            보기파트 = '[보기]\n' + 보기_content
                        else:
                            보기파트 = '[보기] ' + 보기_content
                    else:
                        지문파트 = question_part.strip()
                        보기파트 = ""

                    # 지문파트 - Create bordered or borderless paragraph based on border_flag
                    if self.border_flag:
                        bordered_para = self.create_bordered_paragraph(doc)
                        paragraphs = 지문파트.split('\n')
                        self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)
                    else:
                        paragraphs = 지문파트.split('\n')
                        for idx, paragraph_text in enumerate(paragraphs):
                            para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')
                            if idx == 0:
                                para.paragraph_format.space_before = Pt(1)
                            if idx == len(paragraphs) - 1:
                                para.paragraph_format.space_after = Pt(6)

                    # 보기파트
                    # Check if 보기파트 has numbered items and add corresponding numbered underlines
                    numbered_items = re.findall(r'^\(\d+\)', 보기파트, re.MULTILINE)
                    if numbered_items:
                        # Multi-item case: add numbered underlines
                        underlines = []
                        for i in range(1, len(numbered_items) + 1):
                            if self.row_needs_two_column:
                                underlines.append(f"→ ({i})_____________________________________________")
                            else:
                                underlines.append(f"→ ({i})____________________________________________________________________________________________")
                        보기파트 = 보기파트 + "\n\n" + "\n".join(underlines)
                    else:
                        # Single-item case: add single underline
                        if self.row_needs_two_column:
                            보기파트 = 보기파트 + "\n\n→ ________________________________________________"
                        else:
                            보기파트 = 보기파트 + "\n\n→ ____________________________________________________________________________________________"


                    for paragraph in 보기파트.split('\n'):
                        self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')
                    
                    cell = None
                    table = None




                elif "주제영작" in 유형:

                    # Extract 지문파트 (everything before '윗글의')
                    지문파트 = ""
                    지문_match = re.search(r"(.*?)윗글의", question_part, re.DOTALL)
                    if 지문_match:
                        지문파트 = 지문_match.group(1).strip()

                    # Find and capture the "문제파트"
                    문제파트 = ""
                    match = re.search(r"윗글의.*?\)", question_part)
                    if match:
                        문제파트 = match.group(0)

                    # Extract 보기파트 (everything after 문제파트)
                    보기파트 = question_part.replace(지문파트, "").replace(문제파트, "").strip()


                    # 지문파트 - Create bordered or borderless paragraph based on border_flag
                    if self.border_flag:
                        bordered_para = self.create_bordered_paragraph(doc)
                        paragraphs = 지문파트.split('\n')
                        self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)
                    else:
                        paragraphs = 지문파트.split('\n')
                        for idx, paragraph_text in enumerate(paragraphs):
                            para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')
                            if idx == 0:
                                para.paragraph_format.space_before = Pt(1)
                            if idx == len(paragraphs) - 1:
                                para.paragraph_format.space_after = Pt(6)


                    # 보기파트
                    # Check if 보기파트 has numbered items and add corresponding numbered underlines
                    numbered_items = re.findall(r'^\(\d+\)', 보기파트, re.MULTILINE)
                    if numbered_items:
                        # Multi-item case: add numbered underlines
                        underlines = []
                        for i in range(1, len(numbered_items) + 1):
                            if self.row_needs_two_column:
                                underlines.append(f"→ ({i})_____________________________________________")
                            else:
                                underlines.append(f"→ ({i})____________________________________________________________________________________________")
                        보기파트 = 보기파트 + "\n\n" + "\n".join(underlines)
                    else:
                        # Single-item case: add single underline
                        if self.row_needs_two_column:
                            보기파트 = 보기파트 + "\n\n→ ________________________________________________"
                        else:
                            보기파트 = 보기파트 + "\n\n→ ____________________________________________________________________________________________"

                    for paragraph in 보기파트.split('\n'):
                        self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')
                    
                    cell = None
                    table = None


                elif "요약" in 유형:
                    question_part = question_part.replace("윗글의 내용을 한 문장으로 요약하고자 한다. 빈칸 (A), (B)에 들어갈 말로 가장 적절한 것은?\n", "")

                    #지문 - Create bordered or borderless paragraph based on border_flag
                    if self.border_flag:
                        bordered_para = self.create_bordered_paragraph(doc)
                        paragraphs = passage.split('\n')
                        self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)
                    else:
                        paragraphs = passage.split('\n')
                        for idx, paragraph_text in enumerate(paragraphs):
                            para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')
                            if idx == 0:
                                para.paragraph_format.space_before = Pt(1)
                            if idx == len(paragraphs) - 1:
                                para.paragraph_format.space_after = Pt(6)

                    #요약문 기호 입력
                    요약문기호 = "⇩"
                    para = doc.add_paragraph(요약문기호, style='Normal_modified')
                    para.alignment = WD_ALIGN_PARAGRAPH.CENTER

                    #문제를 요약문과 선지로 쪼개기
                    split_point = question_part.find("(A) -")
                    요약문파트 = question_part[:split_point].strip()
                    선지파트 = question_part[split_point:].strip()

                    #요약문 파트 - Create bordered or borderless paragraph based on border_flag
                    if self.border_flag:
                        bordered_para2 = self.create_bordered_paragraph(doc)
                        요약문파트_paragraphs = 요약문파트.split('\n')

                        # Use the new method that handles borders properly
                        self.apply_remove_brackets_to_paragraph(bordered_para2, 요약문파트_paragraphs)
                    else:
                        요약문파트_paragraphs = 요약문파트.split('\n')
                        for idx, paragraph_text in enumerate(요약문파트_paragraphs):
                            para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')
                            if idx == 0:
                                para.paragraph_format.space_before = Pt(1)
                            if idx == len(요약문파트_paragraphs) - 1:
                                para.paragraph_format.space_after = Pt(6)

                    # 선지파트 탭 적용 후 넣기
                    선지파트 = 선지파트.replace("(A) - (B)", "          (A)\t\t       (B)")
                    선지파트 = 선지파트.replace(" -", "\t...\t")

                    for paragraph_text in 선지파트.split('\n'):
                        paragraph = doc.add_paragraph(paragraph_text, style='Normal_modified')
                        tab_stops = paragraph.paragraph_format.tab_stops
                        for tab_pos in range(1, 2):
                            tab_stops.add_tab_stop(Inches(1.1 * tab_pos))

                    cell = None
                    table = None



                else:
                    # Create bordered or borderless paragraph based on border_flag

                    if "무관한문장_Hard" in 유형:
                        passage = passage.replace("[", "").replace("]", "")


                    if self.border_flag:
                        bordered_para = self.create_bordered_paragraph(doc)
                        paragraphs = passage.split('\n')
                        self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)
                    else:
                        if passage:
                            paragraphs = passage.split('\n')
                            for idx, paragraph_text in enumerate(paragraphs):
                                para = self.add_paragraph_with_formatting(doc, paragraph_text, style='Normal_modified')
                                if idx == 0:
                                    para.paragraph_format.space_before = Pt(1)
                                if idx == len(paragraphs) - 1:
                                    para.paragraph_format.space_after = Pt(6)

                    #문제 (어법2단계 처럼 passage에 문제가 있는 경우 이 부분은 비어있음. 그래서 한줄이 더 들어감. 그래서 question_part 비어있으면 패스)
                    if question_part.strip() != "":
                        if "동형문제" in function_name:
                            # 동형문제: 첫 번째 단락만 formatting, 나머지는 without formatting
                            # sub_questions가 있으면 각 sub_question의 첫 단락만 formatting
                            # Note: question_part is already reconstructed with proper numbering (3., 4., etc.)
                            #       at lines 5966-5974 using "\n\n" as separator
                            if is_multi_sub:
                                # Multiple sub_questions - split by "\n\n" and format first paragraph of EACH
                                sub_question_blocks = question_part.split('\n\n')
                                for block_idx, block in enumerate(sub_question_blocks):
                                    if block.strip():
                                        paragraphs = block.split('\n')
                                        for idx, paragraph in enumerate(paragraphs):
                                            if paragraph.strip():
                                                if idx == 0:
                                                    self.add_paragraph_with_formatting동형문제(doc, paragraph, style='Normal_modified', is_first_paragraph=True)
                                                else:
                                                    self.add_paragraph_without_formatting(doc, paragraph, style='Normal_modified')
                                        # Add newline between sub-questions (not after the last one)
                                        if block_idx < len(sub_question_blocks) - 1:
                                            doc.add_paragraph(style='Normal_modified')
                            else:
                                # Single question - format only first paragraph
                                for idx, paragraph in enumerate(question_part.split('\n')):
                                    if paragraph.strip():
                                        if idx == 0:
                                            self.add_paragraph_with_formatting동형문제(doc, paragraph, style='Normal_modified', is_first_paragraph=True)
                                        else:
                                            self.add_paragraph_without_formatting(doc, paragraph, style='Normal_modified')
                        else:
                            for paragraph in question_part.split('\n'):
                                self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')

                    cell = None
                    table = None




                # Apply styles
                if cell is not None:
                    self.set_cell_border(
                        cell,
                        top={"sz": 8, "val": "single", "color": "auto"},
                        bottom={"sz": 8, "val": "single", "color": "auto"},
                        left={"sz": 8, "val": "single", "color": "auto"},
                        right={"sz": 8, "val": "single", "color": "auto"}
                    )
                    if table is not None:
                        self.set_table_cell_margins(table, top=5, bottom=5, left=7, right=7)
                doc.add_paragraph(style='Normal_modified')




                # Check if answer_part is empty
                if not answer_part and "영작" not in 유형:
                    self.show_error_and_log(
                        f"오류: '{row_key}' 유형의 '{title}' 지문에서 문제 오류가 발견되었습니다.\n"
                        f"(오류내용: '정답'칸이 비어 있음)\n해당 문제를 삭제하거나 재출제하세요."
                    )
                    return

                # Add to sections_dict - use the already computed is_multi_sub flag
                if is_multi_sub:
                    # CRITICAL: Use ONLY sub_questions length - it's the authoritative source
                    # Answers and explanations may contain false positive markers like "$3.59"
                    num_subs = len(sub_questions)

                    # Process each sub-question
                    for sub_idx in range(num_subs):
                        # Extract sub-question parts (use empty string if index out of range)
                        sub_q_marker, sub_q = sub_questions[sub_idx] if sub_idx < len(sub_questions) else ("", "")
                        sub_a_marker, sub_a = sub_answers[sub_idx] if sub_idx < len(sub_answers) else ("", "")
                        sub_e_marker, sub_e = sub_explanations[sub_idx] if sub_idx < len(sub_explanations) else ("", "")

                        # Create title for this sub-question
                        sub_title = f"{question_number}. {원래title}"

                        # Add to sections_dict
                        sections_dict[section_title].append((sub_title, sub_a, sub_e))
                        question_number += 1
                        processed_count += 1
                else:
                    # Single question, process normally
                    sections_dict[section_title].append((해설지로보낼title, answer_part, explanation_part))
                    question_number += 1
                    processed_count += 1

                if self.page_break_after_passages:
                    passages_on_page += 1
                    processed_passages_for_breaks += 1
                    if (passages_on_page >= self.page_break_after_passages
                            and processed_passages_for_breaks < total_valid_passages):
                        self.add_page_break(doc)
                        passages_on_page = 0


    def handle_mock_test_section_해설지(self, doc, section_title, title_answer_explanations, column_configurations):
        """
        Handles the formatting and insertion of mock test sections into the Word document.

        Parameters:
            doc (Document): The Word document being generated.
            section_title (str): The title of the mock test section.
            title_answer_explanations (list): List of tuples containing title, answer_part, and explanation_part.
            column_configurations (dict): Dictionary containing column configurations.
        """
        answer_section_title = f"{section_title}_정답"
        doc.add_paragraph(answer_section_title, style='Heading_ans')

        # Fetch column configurations for mock tests
        if '실전모의고사' not in column_configurations:
            self.app_logger.error(f"No column configuration found for '실전모의고사'.")
            return

        col_count, headers, widths = column_configurations['실전모의고사']
        table = doc.add_table(rows=1, cols=col_count)

        # Set header row
        for idx, header in enumerate(headers):
            table.cell(0, idx).text = header

        # Apply styles to header
        for cell in table.rows[0].cells:
            self.set_cell_font_hdr(cell, self.selected_font_for_exp, self.size_exp)
            self.set_cell_border(
                cell,
                top={"sz": 8, "val": "single", "color": "auto"},
                bottom={"sz": 8, "val": "single", "color": "auto"},
                left={"sz": 8, "val": "single", "color": "auto"},
                right={"sz": 8, "val": "single", "color": "auto"}
            )
            self.set_cell_margins(cell, top=1, bottom=1, left=1, right=1)

        # Add data rows
        for title, answer_part, explanation_part in title_answer_explanations:
            cells = table.add_row().cells

            # Extract number and rest of title
            # e.g., "1. 25-03-고1-18 (1)" -> number="1.", rest="25-03-고1-18 (1)"
            title_str = str(title)
            match = re.match(r'^(\d+\.)\s*(.*)', title_str)
            if match:
                number_part = match.group(1)  # "1."
                rest_of_title = match.group(2).strip()  # "25-03-고1-18 (1)"
            else:
                # Fallback: use full title if pattern doesn't match
                number_part = title_str
                rest_of_title = ""

            cells[0].text = number_part
            cells[1].text = answer_part

            # Add rest of title to explanation in parentheses
            if explanation_part:
                if rest_of_title:
                    modified_explanation = f"{explanation_part}\n(지문: {rest_of_title})"
                else:
                    modified_explanation = explanation_part
                self.add_formatted_text_to_cell(cells[2], modified_explanation)
            else:
                if rest_of_title:
                    cells[2].text = f"(지문: {rest_of_title})"
                else:
                    cells[2].text = ""

            # Apply styles to content cells
            for cell in cells:
                self.set_cell_paragraph_style(cell, 'Normal_exp')
                self.set_cell_border(
                    cell,
                    top={"sz": 8, "val": "single", "color": "auto"},
                    bottom={"sz": 8, "val": "single", "color": "auto"},
                    left={"sz": 8, "val": "single", "color": "auto"},
                    right={"sz": 8, "val": "single", "color": "auto"}
                )
                self.set_cell_margins(cell, top=5, bottom=5, left=5, right=5)

            # Set column widths
            #for idx, width in enumerate(widths):
            #    table.rows[-1].cells[idx].width = width
            # Set the widths for each column
            for row in table.rows:
                for idx, width in enumerate(widths):
                    row.cells[idx].width = width





    def split_sub_questions(self, text, prefix="문제"):
        """
        Splits text by Korean sub-question markers like '문제1)', '문제2)', etc.

        Args:
            text: The text to split
            prefix: The Korean prefix to look for. Options:
                - "문제" for questions (문제1), 문제2), etc.)
                - "정답" for answers (정답1), 정답2), etc.)
                - "해설" for explanations (해설1), 해설2), etc.)

        Returns a list of (marker, content) tuples.
        If no markers are found, returns empty list (treat as single section).

        E.g. with prefix="문제":
            "문제1) First question\n문제2) Second question"
        becomes:
            [("문제1)", "First question"), ("문제2)", "Second question")]
        """
        # Pattern: prefix + number + )
        # e.g., 문제1), 정답2), 해설3)
        pattern = rf"({re.escape(prefix)}\d+\))"

        # Check if the pattern exists in text
        if not re.search(pattern, text):
            # No markers found - return empty list (caller will treat as single section)
            return []

        # Split while keeping the delimiter
        split_text = re.split(pattern, text)

        # e.g. split_text might look like ["", "문제1)", " ...content1.. ", "문제2)", " ...content2..." ]
        results = []
        i = 1
        while i < len(split_text):
            marker = split_text[i].strip()
            content = split_text[i+1].strip() if i+1 < len(split_text) else ""
            results.append((marker, content))
            i += 2

        return results


    def split_multi_questions(self, text):
        """
        Splits text by '문제1)', '문제2)', etc.
        Returns a list of (question_number, question_text).
        E.g.:
            문제1) 윗글의 제목으로 가장 적절한 것은?...
            문제2) 밑줄 친 (a)~(e) 중에서...
        becomes:
            [
            ("문제1)", "윗글의 제목으로 가장 적절한 것은?..."),
            ("문제2)", "밑줄 친 (a)~(e) 중에서...")
            ]
        """
        # This pattern captures '문제1)' or '문제2)' or up to 문제9) etc.
        # If you expect more digits, you can adjust the pattern
        pattern = r"(문제\d+\))"

        # Split while keeping the delimiter
        split_text = re.split(pattern, text)

        # e.g. split_text might look like ["", "문제1)", " ...question1.. ", "문제2)", " ...question2..." ]
        # so we pair them up
        results = []
        # first element might be empty. We'll pair them as (문제X), rest_of_text
        i = 1
        while i < len(split_text):
            question_number = split_text[i].strip()
            question_text = split_text[i+1].strip() if i+1 < len(split_text) else ""
            results.append((question_number, question_text))
            i += 2

        return results


    def split_answers(self, answer_part):
        """
        Splits two answers, e.g. "①④" -> ["①", "④"].
        This is purely an example; if you have
        more answers or a different format, adjust accordingly.
        """
        # If answers are always two circled numbers stuck together,
        # you can just split them individually.
        # e.g. "①④" -> ["①", "④"]
        # If you might have ①④⑤, adapt your logic accordingly.
        return list(answer_part)


    def split_explanations(self, explanation_part):
        """
        Splits explanations by '문제1)', '문제2)', etc.
        Returns a dict: { "문제1)": "explanation1...", "문제2)": "explanation2..." }
        or a list of tuples like the questions.
        """
        # Reuse the same logic from split_multi_questions, if you want consistent parsing:
        items = self.split_multi_questions(explanation_part)
        # items looks like [("문제1)", "..."), ("문제2)", "...")]
        # Turn it into a dict for convenient access.
        exp_dict = {k: v for (k, v) in items}
        return exp_dict



    def handle_past_test_문제지(self, wb, row_key, doc, section_number, sections_dict):
        """
        기출문제세트 편집
        Handles the extraction and formatting of mock test sheets.

        Parameters:
            wb (Workbook): The loaded Excel workbook.
            row_key (str): The key identifying the mock test sheet. (실전모의 몇 회인지 표시)
            doc (Document): The Word document being generated.
            section_number (int): The current section number.
            sections_dict (defaultdict): Dictionary to store section data.
        """
        if row_key not in wb.sheetnames:
            self.app_logger.error(f"Sheet {row_key} does not exist in the workbook.")
            return

        extracting_sheet = wb[row_key]
        section_title = f"{section_number}. {row_key}"
        para = doc.add_paragraph(style='Heading 1')
        run = para.add_run(section_title)
        # Explicitly set font on run to fix "no cover" issue
        run.font.name = self.selected_font_for_title
        run._element.rPr.rFonts.set(qn('w:eastAsia'), self.selected_font_for_title)

        passage_id_to_row_data = {}
        duplicate_passages = []
        row_count = 0
        for row in extracting_sheet.iter_rows(min_row=1, values_only=True):
            if row[0]:
                row_count += 1
                passage_id = row[0].strip()
                if passage_id in passage_id_to_row_data:
                    # Duplicate detected!
                    duplicate_passages.append(passage_id)
                    self.export_warnings.append((row_key, passage_id, "중복 지문 ID (엑셀에서 같은 지문번호가 2번 이상 나타남, 마지막 것만 처리됨)"))
                passage_id_to_row_data[passage_id] = row[:5]  # Later one overwrites earlier

        ordered_passages = list(passage_id_to_row_data.items())
        valid_passage_ids = []
        for passage_id, row_data in ordered_passages:
            row_data = tuple("" if cell is None else cell for cell in row_data)
            if len(row_data) < 5:
                row_data += ("",) * (5 - len(row_data))
            passageid, passage, result, answer_part, explanation_part = row_data[:5]

            if not passageid:
                continue

            try:
                유형 = passageid.split(" // ")[1].strip()
                원래title = passageid.split(" // ")[0].strip()
            except IndexError:
                continue

            if result in ["문제 미출제 (문장 수 부족)", "", None] or (result and result.startswith("Error")):
                continue

            유형 = 유형 + "_기출"
            function_name = 유형.replace('_Normal', '_편집').replace('_Hard', '_편집').replace("내용일치(한)", "내용일치한").replace("무관한 문장", "무관한문장")
            if function_name not in FUNCTION_MAP:
                continue

            valid_passage_ids.append(passage_id)

        total_valid_passages = len(valid_passage_ids)
        passages_on_page = 0
        processed_passages_for_breaks = 0

        question_number = 1
        processed_count = 0
        skipped_count = 0

        for passage_id, row_data in ordered_passages:
            # Ensure row_data has at least 5 elements
            row_data = tuple("" if cell is None else cell for cell in row_data)
            if len(row_data) < 5:
                row_data += ("",) * (5 - len(row_data))
            passageid, passage, result, answer_part, explanation_part = row_data[:5]

            if not passageid:
                skipped_count += 1
                continue

            try:
                유형 = passageid.split(" // ")[1].strip()
                원래title = passageid.split(" // ")[0].strip()

            except IndexError:
                self.app_logger.error(f"Invalid passage_id format: {passageid}")
                # Track warning
                self.export_warnings.append((row_key, passageid, "지문 ID 형식 오류 (' // ' 구분자 없음)"))
                skipped_count += 1
                continue

            title = f"{question_number}."
            #해설지로보낼title = f"{title} {원래title}"

            해설지로보낼title = f"{question_number}"

            if result in ["문제 미출제 (문장 수 부족)", "", None] or (result and result.startswith("Error")):
                reason = result if result else "문제 미출제"
                self.app_logger.info(f"{passageid}: 문제 조건에 맞지 않아 추출하지 않습니다.")
                # Track warning
                if result and result.startswith("Error"):
                    self.export_warnings.append((row_key, 원래title, f"출제 오류: {result}"))
                else:
                    self.export_warnings.append((row_key, 원래title, f"문제 미출제: {reason}"))
                skipped_count += 1
                continue

            유형 = 유형 + "_기출"


            function_name = 유형.replace('_Normal', '_편집').replace('_Hard', '_편집').replace("내용일치(한)", "내용일치한").replace("무관한 문장", "무관한문장")

            if function_name not in FUNCTION_MAP:
                self.app_logger.error(f"Function {function_name} not found in FUNCTION_MAP.")
                # Track warning
                self.export_warnings.append((row_key, 원래title, f"처리 함수 없음: {function_name}"))
                skipped_count += 1
                continue


            # Process the question part
            processed = self.process_question_part(
                doc, function_name, title, passage, result, answer_part, explanation_part
            )

            if processed:
                title, passage, question_line, question_part, answer_part, explanation_part, cell, table = processed



                #해설지에 출처 첨부
                # 1. Reorder: the third part should come first.
                parts = 원래title.split("-")  # ["24", "11", "고3", "25"]
                reordered = [parts[2], parts[0], parts[1], parts[3]]  # ["고3", "24", "11", "25"]
                reordered_title = "-".join(reordered)  # "고3-24-11-25"

                # 2. Convert hyphens and append "번"
                split_reordered = reordered_title.split("-")  
                # split_reordered = ["고3", "24", "11", "25"]
                출처 = f"{split_reordered[0]} {split_reordered[1]}년 {split_reordered[2]}월 {split_reordered[3]}번"
                # 출처 = "고3 24년 11월 25번"

                # 3. If 출처 contains both "고3" and "11월"...
                if "고3" in 출처 and "11월" in 출처:
                    # (a) Remove "고3"
                    출처 = 출처.replace("고3", "")
                    # (b) Replace "11월" with "수능"
                    출처 = 출처.replace("11월", "수능")
                    # (c) Find a pattern like "24년", remove "년" and increment the number 24 -> 25
                    pattern = r'(\d+)년'  # Captures the number before "년"
                    match = re.search(pattern, 출처)
                    if match:
                        year_number = int(match.group(1))  # e.g. 24
                        incremented_year_number = year_number + 1  # e.g. 25
                        # Replacing "24년" with just "25" (i.e., removing "년")
                        출처 = re.sub(pattern, str(incremented_year_number), 출처)
                    
                    # Trim any extra spaces left after removing "고3"
                    출처 = 출처.strip()

                explanation_part += f"\n[출처] {출처}"



                # -----------------------------------------------------
                #  Check for "장문독해1"
                # -----------------------------------------------------
                if "장문독해1" in 유형:
                    """
                    1) We want each sub-question to have its own question_number
                    (19, 20, etc. instead of 1, 2).
                    2) We'll display the `question_line` first (or not, depending on your design).
                    3) We'll display the passage once in a table.
                    4) Then for each sub-question (문제1), (문제2), we increment question_number.
                    """
                                    
                    question_text = "[" + str(question_number) + "~" + str(question_number + 1) + "] " + question_line
                    paragraph = doc.add_paragraph('', style='Normal_modified')
                    run = paragraph.add_run(question_text)
                    run.bold = True

                    # (★) -- Put the passage in bordered paragraph instead of table
                    bordered_para = self.create_bordered_paragraph(doc)
                    paragraphs = passage.split('\n')
                    self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)

                    # (★) -- Add a blank line after the passage
                    # 3) Split the multi-question text
                    question_list = self.split_multi_questions(result)
                    # e.g. [("문제1)", "..."), ("문제2)", "...")]

                    # 4) Split answers and explanations
                    answers = self.split_answers(answer_part)  # e.g. ["①","④"]

                    
                    ################ 전체 해설 다 넣기 전 임시용 ################
                    explanation_part = explanation_part.replace("[해설]", "[해설] 문제2) [해설]")
                    
                    explanations_dict = self.split_explanations(explanation_part)
                    
                    # (★) -- Instead of sub_q_number = 1, start from current question_number
                    sub_q_number = question_number  

                    # 5) For each sub-question...
                    for idx, (mq_key, mq_text) in enumerate(question_list):
                        # Typically "mq_key" is "문제1)", and "mq_text" is the content after that.
                        
                        # We want to do something like:
                        # 19. 윗글의 제목으로 가장 적절한 것은?
                        #  (choices)
                        # 20. 밑줄 친 (a)~(e)...
                        
                        # The code below: figure out the first line vs. the rest
                        lines = mq_text.split('\n', 1)
                        if len(lines) > 1:
                            first_line = lines[0].strip()
                            rest = lines[1]
                        else:
                            first_line = lines[0].strip()
                            rest = ""


                        # (★) -- Now produce the question line: e.g. "19. 윗글의 제목으로..."
                        question_label = f"{sub_q_number}. {first_line}"
                        #doc.add_paragraph(question_label, style='Normal_modified')

                        문제라인 = doc.add_paragraph(style='Normal_left')

                        words_to_format = []
                        if '없는' in question_label:
                            words_to_format.append('없는')
                        if '않는' in question_label:
                            words_to_format.append('않는')
                        if '않은' in question_label:
                            words_to_format.append('않은')
                        if '다른' in question_label:
                            words_to_format.append('다른')
                        if '틀린' in question_label:
                            words_to_format.append('틀린')                    
                        if '2개' in question_label:
                            words_to_format.append('2개')
                        if '3개' in question_label:
                            words_to_format.append('3개')


                        # Add the title with formatting
                        self.add_formatted_text(문제라인, question_label, words_to_format)

                        if rest:
                            for line in rest.split('\n'):
                                if "① (a) ② (b) ③ (c) ④ (d) ⑤ (e)" in line:
                                    line = line.replace("① (a) ② (b) ③ (c) ④ (d) ⑤ (e)", "① (a)\t② (b)\t③ (c)\t④ (d)\t⑤ (e)")
                                    
                                doc.add_paragraph(line, style='Normal_modified')

                        # After writing the sub-question...
                        if idx < len(question_list) - 1:
                            # Only add a blank line if this is NOT the last sub-question
                            doc.add_paragraph("")

                        # pick the matching answer and explanation
                        try:
                            answer_for_this = answers[sub_q_number - question_number]  
                            # Explanation: If sub_q_number == question_number, index=0
                            # If sub_q_number == question_number+1, index=1, etc.
                        except IndexError:
                            answer_for_this = "N/A"

                        explanation_for_this = explanations_dict.get(mq_key, "")
                        explanation_for_this = explanation_for_this.replace(mq_key, "").strip()




                        # Save in sections_dict so you can do 해설 later
                        if mq_key == "문제1)":
                            passageid해설지용 = passageid.replace("41~42", "41")
                        elif mq_key == "문제2)":
                            passageid해설지용 = passageid.replace("41~42", "42")
                        passageid해설지용 = passageid해설지용.split(" // ")[0].strip()


                        #sub_title_for_해설 = f"{sub_q_number}. {passageid해설지용}"
                        sub_title_for_해설 = f"{sub_q_number}"

                        sections_dict[section_title].append(
                            (sub_title_for_해설, answer_for_this, explanation_for_this)
                        )

                        # (★) -- increment question_number for the next sub-question
                        sub_q_number += 1

                    # (★) -- After finishing ALL sub-questions for this 장문독해1,
                    #         update our main question_number for the next top-level question
                    question_number = sub_q_number

                    # Add an extra paragraph if you like for spacing
                    doc.add_paragraph(style='Normal_modified')


                elif "장문독해2" in 유형:
                    """
                    장문독해2 block (3 sub-questions).
                    Essentially the same loop, but we *expect* 3 questions:
                    문젯글에 '문제1)', '문제2)', '문제3)' 
                    """
                    #doc.add_paragraph(question_line, style='Normal_modified')
                    question_text = "[" + str(question_number) + "~" + str(question_number + 2) + "] " + question_line
                    paragraph = doc.add_paragraph('', style='Normal_modified')
                    run = paragraph.add_run(question_text)
                    run.bold = True

                    for paragraph in passage.split('\n'):
                        if "(A)" in paragraph or "(B)" in paragraph or "(C)" in paragraph or "(D)" in paragraph:
                            para = self.add_paragraph_with_formatting(doc, paragraph, style='Normal_modified')
                            para.alignment = WD_ALIGN_PARAGRAPH.CENTER
                        elif paragraph.startswith('*'):
                            passage_style = 'Normal_right'
                            doc.add_paragraph(paragraph, style=passage_style)
                        else:
                            # Create bordered paragraph instead of table
                            bordered_para = self.create_bordered_paragraph(doc)
                            # Split the single paragraph into lines if needed
                            para_lines = paragraph.split('\n') if '\n' in paragraph else [paragraph]
                            self.apply_bold_underline_to_paragraph(bordered_para, para_lines)



                    """
                    for paragraph in passage.split('\n'):
                        if "(A)" in paragraph or "(B)" in paragraph or "(C)" in paragraph or "(D)" in paragraph:                        
                            para = doc.add_paragraph(paragraph, style='Normal_modified')
                            para.alignment = WD_ALIGN_PARAGRAPH.CENTER


                        elif paragraph.startswith('*'):
                            passage_style = 'Normal_right'
                            doc.add_paragraph(paragraph, style=passage_style)

                        else:
                            table = doc.add_table(rows=1, cols=1)
                            cell = table.cell(0, 0)
                            cell.text = ""
                            paragraphs = paragraph.split('\n')
                            add_bold_underline_to_brackets(cell, paragraphs)
                        
                            self.set_cell_border(
                                cell,
                                top={"sz": 8, "val": "single", "color": "auto"},
                                bottom={"sz": 8, "val": "single", "color": "auto"},
                                left={"sz": 8, "val": "single", "color": "auto"},
                                right={"sz": 8, "val": "single", "color": "auto"}
                            )
                            self.set_table_cell_margins(table, top=5, bottom=5, left=7, right=7)
                    """

                    question_list = self.split_multi_questions(result)
                    
                    # (Optional) Confirm we got exactly 3 sub-questions:
                    # if len(question_list) != 3:
                    #     self.app_logger.error(f"장문독해2 should have 3 questions, found {len(question_list)}.")
                    #     continue  # or handle how you like

                    answers = self.split_answers(answer_part)

                    ################ 전체 해설 다 넣기 전 임시용 ################
                    explanation_part = explanation_part.replace("[해석]", "[해석] 문제3) [해석]")
                    explanations_dict = self.split_explanations(explanation_part)
                    sub_q_number = question_number

                    for idx, (mq_key, mq_text) in enumerate(question_list):
                        lines = mq_text.split('\n', 1)
                        if len(lines) > 1:
                            first_line = lines[0].strip()
                            rest = lines[1]
                        else:
                            first_line = lines[0].strip()
                            rest = ""

                        question_label = f"{sub_q_number}. {first_line}"
                        #doc.add_paragraph(question_label, style='Normal_modified')

                        문제라인 = doc.add_paragraph(style='Normal_left')

                        words_to_format = []
                        if '없는' in question_label:
                            words_to_format.append('없는')
                        if '않는' in question_label:
                            words_to_format.append('않는')
                        if '않은' in question_label:
                            words_to_format.append('않은')
                        if '다른' in question_label:
                            words_to_format.append('다른')
                        if '틀린' in question_label:
                            words_to_format.append('틀린')                    
                        if '2개' in question_label:
                            words_to_format.append('2개')
                        if '3개' in question_label:
                            words_to_format.append('3개')

                        # Add the title with formatting
                        self.add_formatted_text(문제라인, question_label, words_to_format)

                        if rest:
                            for line in rest.split('\n'):
                                if "① (a) ② (b) ③ (c) ④ (d) ⑤ (e)" in line:
                                    line = line.replace("① (a) ② (b) ③ (c) ④ (d) ⑤ (e)", "① (a)\t② (b)\t③ (c)\t④ (d)\t⑤ (e)")

                                doc.add_paragraph(line, style='Normal_modified')
                        
                        # Insert a blank paragraph if we want spacing between each sub-question
                        if idx < len(question_list) - 1:
                            doc.add_paragraph("")
                        
                        try:
                            answer_for_this = answers[sub_q_number - question_number]
                        except IndexError:
                            answer_for_this = "N/A"

                        explanation_for_this = explanations_dict.get(mq_key, "")
                        explanation_for_this = explanation_for_this.replace(mq_key, "").strip()


                        # Save in sections_dict so you can do 해설 later
                        if mq_key == "문제1)":
                            passageid해설지용 = passageid.replace("43~45", "43")
                        elif mq_key == "문제2)":
                            passageid해설지용 = passageid.replace("43~45", "44")
                        elif mq_key == "문제3)":
                            passageid해설지용 = passageid.replace("43~45", "45")

                        passageid해설지용 = passageid해설지용.split(" // ")[0].strip()                        
                        #sub_title_for_해설 = f"{sub_q_number}. {passageid해설지용}"
                        sub_title_for_해설 = f"{sub_q_number}"

                        sections_dict[section_title].append(
                            (sub_title_for_해설, answer_for_this, explanation_for_this)
                        )

                        sub_q_number += 1

                    question_number = sub_q_number
                    doc.add_paragraph(style='Normal_modified')








                else:
                    # -----------------------------------------------------
                    # Default processing for normal cases (no "장문독해1" or "장문독해2")
                    # -----------------------------------------------------


                    title = title + " " + question_line  #출처제외
                    #title = title + " " + question_line + f" [{출처}]" #출처포함
                    title = title.strip()
                    # Create a paragraph for the title
                    para_title = doc.add_paragraph(style='Normal_left')



                    if "함축의미" in 유형:
                        self.테이블없는본문_괄호에볼드밑줄더하기(para_title, title)

                    elif "영작2" in 유형:
                        self.테이블없는본문_영작단어수에볼드밑줄더하기(para_title, title)
                    
                    else:
                        # Determine which words need to be formatted
                        words_to_format = []
                        if '없는' in title:
                            words_to_format.append('없는')
                        if '않는' in title:
                            words_to_format.append('않는')
                        if '않은' in title:
                            words_to_format.append('않은')
                        if '틀린' in title:
                            words_to_format.append('틀린')                    
                        if '2개' in title:
                            words_to_format.append('2개')
                        if '3개' in title:
                            words_to_format.append('3개')


                        # Add the title with formatting
                        self.add_formatted_text(para_title, title, words_to_format)

                                        

                    if "요약" in 유형:
                        question_part = question_part.replace("윗글의 내용을 한 문장으로 요약하고자 한다. 빈칸 (A), (B)에 들어갈 말로 가장 적절한 것은?\n", "")

                        #지문 - Create bordered paragraph instead of table
                        bordered_para = self.create_bordered_paragraph(doc)
                        paragraphs = passage.split('\n')
                        self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)

                        #요약문 기호 입력
                        요약문기호 = "⇩"
                        para = doc.add_paragraph(요약문기호, style='Normal_modified')
                        para.alignment = WD_ALIGN_PARAGRAPH.CENTER

                        #문제를 요약문과 선지로 쪼개기
                        split_point = question_part.find("(A) -")
                        요약문파트 = question_part[:split_point].strip()
                        선지파트 = question_part[split_point:].strip()

                        #요약문 파트 - Create bordered paragraph with special remove_brackets formatting
                        bordered_para2 = self.create_bordered_paragraph(doc)
                        요약문파트_paragraphs = 요약문파트.split('\n')
                        
                        # Use the new method that handles borders properly
                        self.apply_remove_brackets_to_paragraph(bordered_para2, 요약문파트_paragraphs)

                        # 선지파트 탭 적용 후 넣기
                        선지파트 = 선지파트.replace("(A) - (B)", "          (A)\t\t       (B)")
                        선지파트 = 선지파트.replace(" -", "\t...\t")

                        for paragraph_text in 선지파트.split('\n'):
                            paragraph = doc.add_paragraph(paragraph_text, style='Normal_modified')
                            tab_stops = paragraph.paragraph_format.tab_stops
                            for tab_pos in range(1, 2):
                                tab_stops.add_tab_stop(Inches(1.1 * tab_pos))
                        
                        cell = None
                        table = None



                    elif "무관한문장" in 유형 or "무관한 문장" in 유형:
                        #박스에 넣을 지문 없는 경우
                        for paragraph in question_part.split('\n'):
                            if paragraph.startswith('*'):
                                passage_style = 'Normal_right'
                            else:
                                passage_style = 'Normal_modified'
                            doc.add_paragraph(paragraph, style=passage_style)

                    

                    elif "도표" in 유형 or "안내문" in 유형:

                        # 1) Build the image URL to fetch from Flask
                        #    Example: "http://localhost:5000/get_image/24-11-고3-25.jpeg"
                        #    or whatever base URL & port your Flask server uses
                        영어파일명 = 원래title.replace("고", "")
                        image_url = f"{self.base_url}/get_image/{영어파일명}.png"


                        # 2) Attempt to fetch via HTTP
                        image_paragraph = doc.add_paragraph()
                        image_paragraph.alignment = WD_ALIGN_PARAGRAPH.LEFT  # Center-align the paragraph

                        try:
                            resp = requests.get(image_url)
                            if resp.status_code == 200:
                                run = image_paragraph.add_run()
                                run.add_picture(BytesIO(resp.content), width=Inches(3))
                            else:
                                self.app_logger.warning(
                                    f"Image fetch failed for {image_url}, status={resp.status_code}"
                                )
                        except Exception as e:
                            self.app_logger.error(f"Exception fetching image from {image_url}: {e}")

                        # 3) Process the question_part paragraphs as before
                        for paragraph in question_part.split('\n'):
                            if paragraph.startswith('*'):
                                passage_style = 'Normal_right'
                            else:
                                passage_style = 'Normal_modified'
                            doc.add_paragraph(paragraph, style=passage_style)





                    else:
                        #박스에 넣을 지문 - Create bordered paragraph instead of table
                        bordered_para = self.create_bordered_paragraph(doc)
                        paragraphs = passage.split('\n')
                        self.apply_bold_underline_to_paragraph(bordered_para, paragraphs)

                        #나머지 (선지들)
                        for paragraph in question_part.split('\n'):
                            if paragraph.startswith('*'):
                                passage_style = 'Normal_right'
                            else:
                                passage_style = 'Normal_modified'
                            doc.add_paragraph(paragraph, style=passage_style)
                        
                        cell = None
                        table = None




                    # Apply styles
                    if cell is not None:
                        self.set_cell_border(
                            cell,
                            top={"sz": 8, "val": "single", "color": "auto"},
                            bottom={"sz": 8, "val": "single", "color": "auto"},
                            left={"sz": 8, "val": "single", "color": "auto"},
                            right={"sz": 8, "val": "single", "color": "auto"}
                        )
                        if table is not None:
                            self.set_table_cell_margins(table, top=5, bottom=5, left=7, right=7)
                    doc.add_paragraph(style='Normal_modified')




                    # Check if answer_part is empty
                    if not answer_part and "영작" not in 유형:
                        self.show_error_and_log(
                            f"오류: '{row_key}' 유형의 '{title}' 지문에서 문제 오류가 발견되었습니다.\n"
                            f"(오류내용: '정답'칸이 비어 있음)\n해당 문제를 삭제하거나 재출제하세요."
                        )
                        return

                    sections_dict[section_title].append((해설지로보낼title, answer_part, explanation_part))

                    question_number += 1
                    processed_count += 1

                    if self.page_break_after_passages:
                        passages_on_page += 1
                        processed_passages_for_breaks += 1
                        if (passages_on_page >= self.page_break_after_passages
                                and processed_passages_for_breaks < total_valid_passages):
                            self.add_page_break(doc)
                            passages_on_page = 0





    def handle_past_test_section_해설지(self, doc, section_title, title_answer_explanations, column_configurations):
        """
        Handles the formatting and insertion of mock test sections into the Word document.

        Parameters:
            doc (Document): The Word document being generated.
            section_title (str): The title of the mock test section.
            title_answer_explanations (list): List of tuples containing title, answer_part, and explanation_part.
            column_configurations (dict): Dictionary containing column configurations.
        """
        answer_section_title = f"{section_title}_정답"
        doc.add_paragraph(answer_section_title, style='Heading_ans')

        # Fetch column configurations for mock tests
        if '기출문제세트' not in column_configurations:
            self.app_logger.error(f"No column configuration found for '기출문제세트'.")
            return

        col_count, headers, widths = column_configurations['기출문제세트']
        #print(f"col_count: {col_count}, headers: {headers}, widths: {widths}")
        table = doc.add_table(rows=1, cols=col_count)

        # Set header row
        for idx, header in enumerate(headers):
            table.cell(0, idx).text = header

        # Apply styles to header
        for cell in table.rows[0].cells:
            self.set_cell_font_hdr(cell, self.selected_font_for_exp, self.size_exp)
            self.set_cell_border(
                cell,
                top={"sz": 8, "val": "single", "color": "auto"},
                bottom={"sz": 8, "val": "single", "color": "auto"},
                left={"sz": 8, "val": "single", "color": "auto"},
                right={"sz": 8, "val": "single", "color": "auto"}
            )
            self.set_cell_margins(cell, top=1, bottom=1, left=1, right=1)

        # Add data rows
        for title, answer_part, explanation_part in title_answer_explanations:
            cells = table.add_row().cells
            cells[0].text = str(title)
            cells[1].text = answer_part

            if explanation_part:
                self.add_formatted_text_to_cell(cells[2], explanation_part)
            else:
                cells[2].text = ""

            # Apply styles to content cells
            for cell in cells:
                self.set_cell_paragraph_style(cell, 'Normal_exp')
                self.set_cell_border(
                    cell,
                    top={"sz": 8, "val": "single", "color": "auto"},
                    bottom={"sz": 8, "val": "single", "color": "auto"},
                    left={"sz": 8, "val": "single", "color": "auto"},
                    right={"sz": 8, "val": "single", "color": "auto"}
                )
                self.set_cell_margins(cell, top=5, bottom=5, left=5, right=5)

            # Set column widths
            #for idx, width in enumerate(widths):
            #    table.rows[-1].cells[idx].width = width

            # Set the widths for each column
            for row in table.rows:
                for idx, width in enumerate(widths):
                    row.cells[idx].width = width




    def get_column_configurations(self):

        return {
            '어휘1단계': (2, ('번호', '정답'), (Inches(1), Inches(2))),
            '어휘2단계': (2, ('번호', '정답'), (Inches(1), Inches(2))),
            '어휘1전문해석있을때': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.5), Inches(5.5))),
            '어휘2전문해석있을때': (3, ('번호', '정답', '해설'), (Inches(1), Inches(2), Inches(4))),
            '어휘3단계': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '내용일치': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.5), Inches(5.5))),
            '내용일치(한)': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.5), Inches(5.5))),
            '밑줄의미': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '함축의미': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '무관한문장': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '빈칸추론': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '추론': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '추론불가': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '순서': (2, ('번호', '정답'), (Inches(1), Inches(1))),
            '삽입': (2, ('번호', '정답'), (Inches(1), Inches(1))),
            '순서삽입전문해석있을때': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.5), Inches(5.5))),
            '주제': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '제목': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '요지': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '연결어_Normal': (2, ('번호', '정답'), (Inches(1), Inches(2))),
            '연결어_Hard': (3, ('번호', '정답', '해설'), (Inches(1), Inches(2), Inches(2))),
            '연결어전문해석있을때': (3, ('번호', '정답', '해설'), (Inches(1), Inches(2), Inches(4))),
            '연결어': (3, ('번호', '정답', '해설'), (Inches(1), Inches(2), Inches(2))),
            '요약': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '어법1단계': (2, ('번호', '정답'), (Inches(1), Inches(2))),
            '어법2단계': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '빈칸암기': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '어휘종합': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.7), Inches(5.3))),
            '내용종합': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.7), Inches(5.3))),
            '영작1': (2, ('번호', '정답'), (Inches(1), Inches(6))),
            '영작2': (2, ('번호', '정답'), (Inches(1), Inches(6))),
            '주제영작': (2, ('번호', '정답'), (Inches(1), Inches(6))),            
            '동형문제': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '동반의어문제1': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.5), Inches(1.5))),
            '동반의어문제2': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.5), Inches(1.5))),
            '영영정의문제': (3, ('번호', '정답', '해설'), (Inches(1), Inches(1.2), Inches(1.2))),
            '실전모의고사': (3, ('번호', '정답', '해설'), (Inches(1), Inches(0.3), Inches(5.7))),
            '기출문제세트': (3, ('번호', '정답', '해설'), (Inches(0.3), Inches(0.3), Inches(6.7)))            
        }



    def process_sections_dict_해설지(self, doc, sections_dict, last_section_title):
        #print("process_sections_dict_해설지 called")
        #print(f"Initial doc sections count: {len(doc.sections)}")
        #print(f"self.two_column setting: {self.two_column}")

        """
        Processes all sections in the sections_dict by adding them to the Word document.
        Handles the final section differently if needed.

        Args:
            doc (Document): The Word document being generated.
            sections_dict (dict): A dictionary mapping section titles to lists of (title, answer_part, explanation_part) tuples.
            last_section_title (str): The title of the last section for special handling.
        """

        try:
            section_titles = list(sections_dict.keys())  # Get all the section titles
            #self.app_logger.info("Section Titles: %s", section_titles)
        except IndexError:
            # Handle the error or pass if there's nothing specific to do
            messagebox.showerror("오류", "●추출할 지문이 없습니다. '추출할 지문번호 입력' 창에 지문번호가 올바르게 입력되어 있는지 확인하세요.")
            self.app_logger.info("●[오류] 추출할 지문이 없습니다.")
            return
        except Exception as e:
            # It's good practice to catch other potential errors
            self.app_logger.info(f"An unexpected error occurred: {e}")
            return

        try:
            # Filter out sections that don't need answer sections using helper method
            filtered_section_titles = [
                title for title in section_titles
                if not self.should_skip_section_in_answers(title)
            ]
            if filtered_section_titles:
                last_section_title = filtered_section_titles[-1]  # Get the last section title
            else:
                last_section_title = section_titles[-1]  # Fallback to the last section title
        except IndexError:
            # Handle the error or notify the user
            def nothing_to_export():
                error_dialog = Toplevel()
                error_dialog.title("오류")
                error_dialog.geometry("500x150")
                error_dialog.grab_set()  # Make the dialog modal
                error_dialog.focus_set()  # Make the dialog modal
                error_dialog.grid_rowconfigure(0, weight=1)
                error_dialog.grid_columnconfigure(0, weight=1)

                CTkLabel(
                    error_dialog, 
                    fg_color="white", 
                    width=300, 
                    height=120, 
                    text="-추출할 지문이 없습니다-\n'추출할 지문번호 입력'창에 추출할 지문번호가 올바르게 입력되어 있는지 확인하세요.", 
                    text_color="black"
                ).grid(row=0, column=0, sticky="nsew")

                # Frame for buttons
                button_frame = CTkFrame(error_dialog, fg_color="white", width=300, height=30, corner_radius=0)
                button_frame.grid(row=1, column=0, sticky="swe")
                button_frame.grid_columnconfigure(0, weight=1)

                # "OK" button action
                def on_ok():
                    self.close_export_popup = True
                    error_dialog.grab_release()
                    error_dialog.destroy()

                # Create "OK" button
                ok_button = CTkButton(
                    button_frame, 
                    border_width=2, 
                    border_color="gray", 
                    text="OK", 
                    text_color="black", 
                    width=70, 
                    height=30, 
                    command=on_ok, 
                    fg_color="#FEF9E0", 
                    hover_color="#DDA15C"
                )
                ok_button.grid(row=0, column=0, padx=10, pady=10)

            nothing_to_export()
            return
        except Exception as e:
            # It's good practice to catch other potential errors
            self.app_logger.info(f"An unexpected error occurred: {e}")
            return

        # Define column configurations if not already defined
        column_configurations = self.get_column_configurations()

        #print("sections_dict")
        #print(sections_dict)

        # Track if we're currently in a two-column section
        currently_in_two_column = False
        
        # Track the previous row_key
        previous_row_key = None

        for section_title, title_answer_explanations in sections_dict.items():
            #print(f"\n--- Processing section: {section_title} ---")

            # Skip sections that don't need answer sections
            if self.should_skip_section_in_answers(section_title):
                continue

            # Extract the base row_key from section_title
            # section_title is like "1. 빈칸암기_Normal" or "5. 동형문제 1회"
            parts = section_title.split('. ', 1)
            if len(parts) > 1:
                base_section_title = parts[1]
                # Remove _Normal or _Hard suffix to get the row_key
                row_key = base_section_title.split('_')[0]
                # Also strip space-separated suffixes like "1회", "2회", etc.
                row_key = re.sub(r'\s+\d+회$', '', row_key).strip()
            else:
                base_section_title = section_title  # Assign base_section_title for use later ??????????
                row_key = section_title.split('_')[0]
                # Also strip space-separated suffixes like "1회", "2회", etc.
                row_key = re.sub(r'\s+\d+회$', '', row_key).strip()

            #print(f"Extracted row_key: {row_key}")
            #print(f"Base section title: {base_section_title}")
            #print(f"Previous row_key: {previous_row_key}")
            #print(f"Currently in two-column: {currently_in_two_column}")

            # ONLY apply two-column for 빈칸암기 in 해설지
            section_needs_two_column = (row_key == "빈칸암기" and self.two_column)
            #print(f"section_needs_two_column: {section_needs_two_column} (checking if row_key == '빈칸암기' and self.two_column)")
            
            # Check if we need to switch column format
            if section_needs_two_column and not currently_in_two_column:
                #print("Switching from single-column to two-column for 빈칸암기...")
                # Add a new two-column section
                doc.add_section(WD_SECTION.NEW_PAGE)
                new_sec = doc.sections[-1]
                #print(f"New section added. Total sections now: {len(doc.sections)}")
                
                sectPr = new_sec._sectPr
                
                # Remove any existing cols element first
                for cols in sectPr.xpath('./w:cols'):
                    #print("Removing existing cols element")
                    sectPr.remove(cols)
                
                # Add new cols element
                cols = OxmlElement('w:cols')
                cols.set(qn('w:num'), '2')
                cols.set(qn('w:sep'), '1')
                cols.set(qn('w:space'), str(self.cm_to_twips(self.col_spacing)))
                sectPr.insert(0, cols)
                #print("Two-column format applied to new section")

                # Set margins for the new section
                new_sec.top_margin = Cm(self.margin_top)
                new_sec.bottom_margin = Cm(self.margin_bottom)
                new_sec.left_margin = Cm(self.margin_left)
                new_sec.right_margin = Cm(self.margin_right)
                self._apply_mirror_margins(new_sec)
                print("Margins set for new section")

                currently_in_two_column = True

                # Verify the section was created with two columns
                cols_elements = sectPr.xpath('./w:cols')
                if cols_elements:
                    num_cols = cols_elements[0].get(qn('w:num'))
                    #print(f"Verification: Section has {num_cols} columns")
                else:
                    print("WARNING: No cols element found after adding!")
                    
            elif not section_needs_two_column and currently_in_two_column:
                #print("Switching from two-column back to single-column...")
                # Add a new single-column section
                doc.add_section(WD_SECTION.NEW_PAGE)
                new_sec = doc.sections[-1]
                #print(f"New section added for single column. Total sections now: {len(doc.sections)}")
                
                sectPr = new_sec._sectPr
                
                # Ensure single column
                for cols in sectPr.xpath('./w:cols'):
                    sectPr.remove(cols)
                
                one_col = OxmlElement('w:cols')
                one_col.set(qn('w:num'), '1')
                sectPr.insert(0, one_col)
                #print("Single-column format applied to new section")

                # Set margins
                new_sec.top_margin = Cm(self.margin_top)
                new_sec.bottom_margin = Cm(self.margin_bottom)
                new_sec.left_margin = Cm(self.margin_left)
                new_sec.right_margin = Cm(self.margin_right)
                self._apply_mirror_margins(new_sec)

                currently_in_two_column = False
                #print("Switched back to single-column")
            #else:
                #print(f"No column format change needed (currently_in_two_column: {currently_in_two_column}, section_needs_two_column: {section_needs_two_column})")

            # Continue with the rest of the section processing...
            unwanted_sections = ["동반의어_Normal", "영영정의_Normal", "요약문", "커스텀"]

            if any(unwanted_section in section_title for unwanted_section in unwanted_sections):
                print("Skipping unwanted section")
                continue
            elif base_section_title == "동반의어" or base_section_title == "영영정의":
                print("Skipping 동반의어/영영정의 section")
                continue

            #print(f"Processing content for section: {section_title}")
            
            if "실전모의고사" in section_title:
                self.handle_mock_test_section_해설지(doc, section_title, title_answer_explanations, column_configurations)
            elif "기출문제세트" in section_title:
                self.handle_past_test_section_해설지(doc, section_title, title_answer_explanations, column_configurations)
            else:
                self.handle_regular_section_해설지(doc, section_title, title_answer_explanations, column_configurations, sections_dict)

            # Add page break between sections if "유형마다 새 페이지" is checked
            if self.new_page_per_section:
                self.add_page_break(doc)

            # Update previous_row_key
            previous_row_key = row_key
            
        #print(f"\nFinal doc sections count: {len(doc.sections)}")
        #print(f"Currently in two-column at end: {currently_in_two_column}")

        


    def 빠른정답찾기용(self, doc, sections_dict, last_section_title):

        try:
            section_titles = list(sections_dict.keys())  # Get all the section titles
            #self.app_logger.info("Section Titles: %s", section_titles)
        except IndexError:
            # Handle the error or pass if there's nothing specific to do
            messagebox.showerror("오류", "●추출할 지문이 없습니다. '추출할 지문번호 입력' 창에 지문번호가 올바르게 입력되어 있는지 확인하세요.")
            self.app_logger.info("●[오류] 추출할 지문이 없습니다.")
            return
        except Exception as e:
            # It's good practice to catch other potential errors
            self.app_logger.info(f"An unexpected error occurred: {e}")
            return

        try:
            # Filter out sections that don't need answer sections using helper method
            filtered_section_titles = [
                title for title in section_titles
                if not self.should_skip_section_in_answers(title)
            ]
            if filtered_section_titles:
                last_section_title = filtered_section_titles[-1]  # Get the last section title
            else:
                last_section_title = section_titles[-1]  # Fallback to the last section title
        except IndexError:
            # Handle the error or notify the user
            def nothing_to_export():
                error_dialog = Toplevel()
                error_dialog.title("오류")
                error_dialog.geometry("500x150")
                error_dialog.grab_set()  # Make the dialog modal
                error_dialog.focus_set()  # Make the dialog modal
                error_dialog.grid_rowconfigure(0, weight=1)
                error_dialog.grid_columnconfigure(0, weight=1)

                CTkLabel(
                    error_dialog, 
                    fg_color="white", 
                    width=300, 
                    height=120, 
                    text="-추출할 지문이 없습니다.\n'추출할 지문번호 입력'창에 추출할 지문번호가 올바르게 입력되어 있는지 확인하세요.", 
                    text_color="black"
                ).grid(row=0, column=0, sticky="nsew")

                # Frame for buttons
                button_frame = CTkFrame(error_dialog, fg_color="white", width=300, height=30, corner_radius=0)
                button_frame.grid(row=1, column=0, sticky="swe")
                button_frame.grid_columnconfigure(0, weight=1)

                # "OK" button action
                def on_ok():
                    self.close_export_popup = True
                    error_dialog.grab_release()
                    error_dialog.destroy()

                # Create "OK" button
                ok_button = CTkButton(
                    button_frame, 
                    border_width=2, 
                    border_color="gray", 
                    text="OK", 
                    text_color="black", 
                    width=70, 
                    height=30, 
                    command=on_ok, 
                    fg_color="#FEF9E0", 
                    hover_color="#DDA15C"
                )
                ok_button.grid(row=0, column=0, padx=10, pady=10)

            nothing_to_export()
            return
        except Exception as e:
            # It's good practice to catch other potential errors
            self.app_logger.info(f"An unexpected error occurred: {e}")
            return




        for section_title, title_answer_explanations in sections_dict.items():

            parts = section_title.split('. ', 1)
            if len(parts) > 1:
                base_section_title = parts[1]
            else:
                base_section_title = section_title  # Fallback if the format isn't as expected

            # Skip sections that don't need answer sections
            if self.should_skip_section_in_answers(section_title):
                continue

            self.빠른정답찾기(doc, section_title, title_answer_explanations)

            #if section_title != last_section_title:
                #print("Adding page break...")
                #self.add_page_break(doc)






    def 빠른정답찾기(self, doc, section_title, title_answer_explanations):
        #print(f"빠른정답찾기 section_title: {section_title}")

        # 1) Special case: "빈칸암기" => skip table creation
        if "빈칸암기" in section_title:
            print("Detected '빈칸암기' in section_title -- skipping table creation.")
            answer_section_title = f"{section_title}_정답"
            doc.add_paragraph(answer_section_title, style='Heading_ans')
            para_title = doc.add_paragraph(style='Normal_modified')
            para_title.add_run("'정답 및 해설' 참조")
            return

        # Otherwise, create a heading
        answer_section_title = f"{section_title}_정답"
        doc.add_paragraph(answer_section_title, style='Heading_ans')

        # Decide table columns (default: 10, center alignment)
        col_count = 10
        horizontal_alignment = WD_ALIGN_PARAGRAPH.CENTER
        vertical_alignment = WD_ALIGN_VERTICAL.TOP
        use_explanation_for_second_row = False

        # Extract base section title (remove number prefix like "1. ")
        parts = section_title.split('. ', 1)
        base_section_title = parts[1] if len(parts) > 1 else section_title

        # Check if base title (ignoring _Normal/_Hard suffix and space-separated suffixes) matches 영작 sections
        base_without_suffix = base_section_title.split('_')[0].strip()
        # Also strip space-separated suffixes like "1회", "2회", etc.
        base_without_suffix = re.sub(r'\s+\d+회$', '', base_without_suffix).strip()

        # "영작1", "영작2", "주제영작" => 5 columns, left-align, second row uses explanation_part
        if base_without_suffix in ["영작1", "영작2", "주제영작"]:
            #print(f"Detected '{base_without_suffix}' in section_title -> 5 columns, left-align, second row = explanation_part.")
            col_count = 5
            horizontal_alignment = WD_ALIGN_PARAGRAPH.LEFT
            use_explanation_for_second_row = True

        # "영영정의문제" => 5 columns, left-align, second row = answer_part (default)
        elif "영영정의문제" in section_title:
            #print("Detected '영영정의문제' in section_title -> 5 columns, left-align.")
            col_count = 5
            horizontal_alignment = WD_ALIGN_PARAGRAPH.LEFT

        # ----------------------------
        # LINE-BY-LINE MODIFICATION
        # ----------------------------
        def modify_answer_part_line_by_line(ans: str, symbols):
            """
            Splits `ans` into lines, for each line:
            - find the first occurrence of any symbol in `symbols`
            - keep only the substring up to (and including) that symbol
            Then join the lines back.
            """
            lines = ans.splitlines()
            new_lines = []
            for line in lines:
                # Debug: show original line
                #print(f"  [modify_line] original: '{line}'")
                truncated_line = line
                found_index = None

                # For each symbol (e.g. ⓐ, ⓑ), find its index if it exists
                for sym in symbols:
                    idx = line.find(sym)
                    if idx != -1:
                        # If we find a symbol, note the earliest index
                        # (We only keep the earliest occurrence among all symbols)
                        if found_index is None or idx < found_index:
                            found_index = idx
                if found_index is not None:
                    # Keep everything up to and including the symbol
                    symbol = line[found_index : found_index+1]  # e.g., "ⓑ"
                    # But we also need the full length of that symbol (some are 1 char, some are 2, e.g. ③ might be 1 char in Unicode).
                    # If each symbol is 1 Unicode character, we can do:
                    truncated_line = line[: found_index + 1 + 0]  
                    # However, if the symbol is "③" (which is also 1 codepoint), then the length is 1.
                    # For safety, if your symbols are definitely 1 codepoint each, the above is okay.

                    #print(f"    Found symbol at index {found_index}, new_line = '{truncated_line}'")
                else:
                    print(f"    No symbol found in this line; keeping entire line: '{line}'")
                    
                    
                new_lines.append(truncated_line)
            # Reconstruct multi-line string
            return "\n".join(new_lines)

        # For "어휘1단계", "어휘2단계", "어법1단계" => keep up to ⓐ or ⓑ
        def modify_answer_part_voca_grammar(ans: str) -> str:
            #print(f"[modify_answer_part_voca_grammar] multiline input:\n{ans}")
            return modify_answer_part_line_by_line(ans, ["ⓐ", "ⓑ"])

        # For "연결어" => keep up to ①, ②, ③, ④, or ⑤
        def modify_answer_part_connectives(ans: str) -> str:
            #print(f"[modify_answer_part_connectives] multiline input:\n{ans}")
            return modify_answer_part_line_by_line(ans, ["①","②","③","④","⑤"])

        # Combined approach
        def combined_modify_answer_part(ans: str) -> str:
            return ans  # default pass-through

        # If 어휘1단계, 어휘2단계, 어법1단계 => apply the voca/grammar logic
        if any(x in section_title for x in ["어휘1단계", "어휘2단계", "어법1단계"]):
            #print("Detected one of 어휘1단계 / 어휘2단계 / 어법1단계 in section_title.")
            old_func = combined_modify_answer_part
            def combined_modify_answer_part(ans: str) -> str:
                ans = old_func(ans)
                ans = modify_answer_part_voca_grammar(ans)
                return ans

        # If 연결어 => apply connectives logic
        if "연결어" in section_title:
            #print("Detected '연결어' in section_title.")
            old_func = combined_modify_answer_part
            def combined_modify_answer_part(ans: str) -> str:
                ans = old_func(ans)
                ans = modify_answer_part_connectives(ans)
                return ans

        # Finally, use `combined_modify_answer_part` in the table
        modify_answer_part = combined_modify_answer_part

        # Helper function for "영작" sections to extract answer before "[전문해석]"
        def process_영작_explanation(text: str, section_type: str = None) -> str:
            """
            For 영작 sections:
            1. Extract only the part before "[전문해석]" if it exists
            2. Remove "정답: " prefix
            3. For "주제영작", remove Korean translations in parentheses
            """
            # First, extract text before "[전문해석]"
            if "[전문해석]" in text:
                text = text.split("[전문해석]")[0]

            # Remove "정답: " prefix
            if text.strip().startswith("정답:"):
                text = text.strip()[3:].strip()  # Remove "정답:" (3 chars) and strip whitespace

            # For "주제영작", remove Korean text in parentheses
            if section_type == "주제영작":
                # Remove text from '(' to ')' including the parentheses
                # Using regex to handle multi-line parenthetical content
                text = re.sub(r'\s*\([^)]*\)\s*', '', text, flags=re.DOTALL)

            return text.strip()

        # Column widths
        widths10 = [Inches(0.7)] * 10
        widths5 = [Inches(1.4)] * 5

        chunk_size = col_count  # either 5 or 10
        total_items = len(title_answer_explanations)

        # Helper to shade rows
        def set_cell_shading(cell, fill_color):
            tc_pr = cell._tc.get_or_add_tcPr()
            shd = OxmlElement('w:shd')
            shd.set(qn('w:val'), 'clear')
            shd.set(qn('w:color'), "auto")
            shd.set(qn('w:fill'), fill_color)
            tc_pr.append(shd)

        # Build tables in chunks
        for i in range(0, total_items, chunk_size):
            chunk = title_answer_explanations[i : i + chunk_size]

            table = doc.add_table(rows=2, cols=col_count)

            # Row 0 => titles
            for col_idx, (title, answer_part, explanation_part) in enumerate(chunk):
                # For mock tests (실전모의) or when passage_number_numeric is enabled, extract just the number
                title_str = str(title)
                if "실전모의" in section_title or self.passage_number_numeric:
                    match = re.match(r'^(\d+\.)', title_str)
                    if match:
                        title_str = match.group(1)  # Use just "1.", "2.", etc.

                table.cell(0, col_idx).text = title_str

            # Row 1 => answer_part (or explanation_part for "영작")
            for col_idx, (title, answer_part, explanation_part) in enumerate(chunk):
                if use_explanation_for_second_row:
                    # For "영작" sections, process the explanation_part
                    text_line = process_영작_explanation(str(explanation_part), base_without_suffix)
                else:
                    text_line = modify_answer_part(answer_part)

                table.cell(1, col_idx).text = text_line

            # Style, alignment, shading
            for row_idx in range(2):
                for col_idx in range(col_count):
                    cell = table.cell(row_idx, col_idx)
                    self.set_cell_paragraph_style(cell, 'Normal_exp')
                    self.set_cell_border(
                        cell,
                        top={"sz": 8, "val": "single", "color": "auto"},
                        bottom={"sz": 8, "val": "single", "color": "auto"},
                        left={"sz": 8, "val": "single", "color": "auto"},
                        right={"sz": 8, "val": "single", "color": "auto"}
                    )
                    self.set_cell_margins(cell, top=5, bottom=5, left=5, right=5)

                    cell.vertical_alignment = vertical_alignment
                    for paragraph in cell.paragraphs:
                        paragraph.alignment = horizontal_alignment

                    # shade row 0 => gray + bold
                    if row_idx == 0:
                        set_cell_shading(cell, "D9D9D9")
                        for paragraph in cell.paragraphs:
                            for run in paragraph.runs:
                                run.bold = True

            # Set column widths
            if col_count == 10:
                for row in table.rows:
                    for col_idx, width in enumerate(widths10):
                        row.cells[col_idx].width = width
            else:
                for row in table.rows:
                    for col_idx, width in enumerate(widths5):
                        row.cells[col_idx].width = width

            # Optional page break
            # self.add_page_break(doc)
            


    def show_error_and_log(self, message):
        messagebox.showerror("오류", message)
        self.app_logger.error(message)

    def _create_answer_document(self, main_output_path, sections_dict, should_create_answers):
        """
        Create a separate answer document with book cover and answer sections.

        Args:
            main_output_path (str): Path of main document (for deriving answer path)
            sections_dict (dict): Dictionary of sections from question processing
            should_create_answers (bool): Whether answer sections should be created

        Returns:
            str: Path to created answer document
        """
        if not should_create_answers:
            return None

        # Calculate answer document path
        p_main = Path(main_output_path)
        answer_doc_path = str(p_main.parent / f"{p_main.stem}_정답지{p_main.suffix}")

        # Guard against empty/None text_week
        original_text_week = (self.text_week or "").strip()
        modified_text_week = f"{original_text_week} 정답지".strip()

        # Save original for restoration
        saved_text_week = self.text_week

        try:
            # Temporarily modify text_week for answer document
            self.text_week = modified_text_week

            # Create document from book cover (same logic as main document)
            p = Path(self.book_cover_path)
            default_bookcover = "default_bookcover" in p.name.lower()
            is_no_cover = "no cover" in p.name.lower()
            delete_empty_paragraph = False

            if self.applyoption_var.get() == "1" and not is_no_cover:
                answer_doc = Document(self.book_cover_path)

                # Delete content after first page (keep only first page of bookcover)
                first_page_break_found = False
                paragraphs_to_delete = []

                for para in answer_doc.paragraphs:
                    if first_page_break_found:
                        # Mark for deletion - everything after first page break
                        paragraphs_to_delete.append(para)
                    else:
                        # Check if this paragraph contains a page break
                        for run in para.runs:
                            if run._element.xml:
                                if 'w:br' in run._element.xml and 'type="page"' in run._element.xml:
                                    first_page_break_found = True
                                    # Also delete this paragraph (contains the page break)
                                    paragraphs_to_delete.append(para)
                                    break

                # Delete all marked paragraphs
                for para in paragraphs_to_delete:
                    p_elem = para._element
                    p_elem.getparent().remove(p_elem)

                # Check if document ends with page break
                def document_ends_with_page_break(doc):
                    if not doc.paragraphs:
                        return False
                    for para in reversed(doc.paragraphs[-3:]):
                        for run in para.runs:
                            if run._element.xml:
                                if 'w:br' in run._element.xml and 'type="page"' in run._element.xml:
                                    return True
                    return False

                if not document_ends_with_page_break(answer_doc):
                    answer_doc.add_page_break()
                else:
                    delete_empty_paragraph = True

            elif self.applyoption_var.get() == "1" and is_no_cover:
                answer_doc = Document()
                section = answer_doc.sections[0]
                section.page_width = Cm(21)
                section.page_height = Cm(29.7)

                # Remove the initial empty paragraph that comes with Document()
                if answer_doc.paragraphs:
                    p_elem = answer_doc.paragraphs[0]._element
                    p_elem.getparent().remove(p_elem)

                # Add page numbers at center bottom (footer)
                self._add_page_number_footer(section)

            else:
                answer_doc = Document()
                section = answer_doc.sections[0]
                section.page_width = Cm(21)
                section.page_height = Cm(29.7)

            # Define styles and set margins
            self.define_styles(answer_doc)
            section = answer_doc.sections[0]
            section.top_margin = Cm(self.margin_top)
            section.bottom_margin = Cm(self.margin_bottom)
            section.left_margin = Cm(self.margin_left)
            section.right_margin = Cm(self.margin_right)
            self._apply_mirror_margins(section)

            # Remove trailing empty paragraphs if needed
            if delete_empty_paragraph and self.applyoption_var.get() == "1" and not is_no_cover:
                while answer_doc.paragraphs and not answer_doc.paragraphs[-1].text.strip():
                    p_elem = answer_doc.paragraphs[-1]._element
                    p_elem.getparent().remove(p_elem)
                    if not answer_doc.paragraphs:
                        break

            # Add NEW_PAGE section break before answer sections
            answer_doc.add_section(WD_SECTION.NEW_PAGE)
            new_sec = answer_doc.sections[-1]

            # Ensure single column mode
            sectPr = new_sec._sectPr
            for cols in sectPr.xpath('./w:cols'):
                sectPr.remove(cols)
            one_col = OxmlElement('w:cols')
            one_col.set(qn('w:num'), '1')
            sectPr.append(one_col)

            # Update margins
            new_sec.top_margin = Cm(self.margin_top)
            new_sec.bottom_margin = Cm(self.margin_bottom)
            new_sec.left_margin = Cm(self.margin_left)
            new_sec.right_margin = Cm(self.margin_right)
            self._apply_mirror_margins(new_sec)

            # Add "정답 및 해설" section
            print("정답지 파일 - 해설지 시작")
            """
            answer_doc.add_paragraph("\n\n\n\n\n")
            paragraph = answer_doc.add_paragraph("정답 및 해설", style='Heading 1')
            paragraph.alignment = WD_PARAGRAPH_ALIGNMENT.CENTER
            if paragraph.runs:
                paragraph.runs[0].font.size = Pt(20)
            else:
                run = paragraph.add_run("정답 및 해설")
                run.font.size = Pt(20)
            
            answer_doc.add_page_break()
            """
            
            # Get section titles
            section_titles = list(sections_dict.keys())
            # Filter out sections that don't need answer sections using helper method
            filtered_section_titles = [
                t for t in section_titles
                if not self.should_skip_section_in_answers(t)
            ]

            if filtered_section_titles:
                last_section_title = filtered_section_titles[-1]
            else:
                last_section_title = section_titles[-1] if section_titles else None

            # Add answer explanation section
            if last_section_title:
                self.process_sections_dict_해설지(answer_doc, sections_dict, last_section_title)

            # Add ODD_PAGE section break before quick answer section
            answer_doc.add_section(WD_SECTION.ODD_PAGE)
            new_sec = answer_doc.sections[-1]

            # Ensure single column mode
            sectPr = new_sec._sectPr
            for cols in sectPr.xpath('./w:cols'):
                sectPr.remove(cols)
            one_col = OxmlElement('w:cols')
            one_col.set(qn('w:num'), '1')
            sectPr.append(one_col)

            # Update margins
            new_sec.top_margin = Cm(self.margin_top)
            new_sec.bottom_margin = Cm(self.margin_bottom)
            new_sec.left_margin = Cm(self.margin_left)
            new_sec.right_margin = Cm(self.margin_right)
            self._apply_mirror_margins(new_sec)

            # Add "빠른 정답 찾기" section
            paragraph = answer_doc.add_paragraph("빠른 정답 찾기", style='Heading 1')
            paragraph.alignment = WD_PARAGRAPH_ALIGNMENT.CENTER
            if paragraph.runs:
                paragraph.runs[0].font.size = Pt(20)
            else:
                run = paragraph.add_run("빠른 정답 찾기")
                run.font.size = Pt(20)

            # Add quick answer section
            if last_section_title:
                self.빠른정답찾기용(answer_doc, sections_dict, last_section_title)

            # Save answer document
            answer_doc.save(answer_doc_path)

            # Process text replacement with modified text_week
            self.process_text_replacement(answer_doc_path)

            return answer_doc_path

        finally:
            # Restore original text_week
            self.text_week = saved_text_week

    def process_text_replacement(self, file_path):

        # Replace text in the document
        if self.text_school_grade or self.text_exam_type or self.text_week:
            doc = Document(file_path)
            for paragraph in doc.paragraphs:
                if paragraph.style.name == "School" and "2025 학교이름 ㅇ학년 ㅇ학기" in paragraph.text and self.text_school_grade:
                    paragraph.text = paragraph.text.replace("2025 학교이름 ㅇ학년 ㅇ학기", self.text_school_grade)
                elif paragraph.style.name == "Examtype" and "ㅇㅇ고사 내신대비" in paragraph.text and self.text_exam_type:
                    paragraph.text = paragraph.text.replace("ㅇㅇ고사 내신대비", self.text_exam_type)
                elif paragraph.style.name == "Week" and "ㅇ주차 숙제" in paragraph.text and self.text_week:
                    paragraph.text = paragraph.text.replace("ㅇ주차 숙제", self.text_week)
            doc.save(file_path)
