from pathlib import Path
import json
import os
import sys
from typing import Optional, Type
import tkinter as tk
from tkinter import messagebox

import customtkinter
from customtkinter import (
    CTkButton,
    CTkCheckBox,
    CTkEntry,
    CTkFrame,
    CTkLabel,
    CTkRadioButton,
    CTkScrollableFrame,
    CTkToplevel,
)

from modules import make_questions_options as question_option_windows
from modules.copyquestion import CopyQuestionPopup


QUESTION_TYPE_CLASS_MAP = {
    "어휘1단계": "QuestionOptions어휘1단계",
    "어휘2단계": "QuestionOptions어휘2단계",
    "어휘3단계": "QuestionOptions어휘3단계",
    "내용일치": "QuestionOptions내용일치",
    "내용일치(한)": "QuestionOptions내용일치한",
    "밑줄의미": "QuestionOptions밑줄의미",
    "함축의미": "QuestionOptions함축의미",
    "무관한문장": "QuestionOptions무관한문장",
    "빈칸추론": "QuestionOptions빈칸추론",
    "추론": "QuestionOptions추론",
    "추론불가": "QuestionOptions추론불가",
    "순서": "QuestionOptions순서",
    "삽입": "QuestionOptions삽입",
    "주제": "QuestionOptions주제",
    "제목": "QuestionOptions제목",
    "요지": "QuestionOptions요지",
    "연결어": "QuestionOptions연결어",
    "요약": "QuestionOptions요약",
    "어법1단계": "QuestionOptions어법1단계",
    "어법2단계": "QuestionOptions어법2단계",
    "영작1": "QuestionOptions영작1",
    "영작2": "QuestionOptions영작2",
    "어휘종합": "QuestionOptions어휘종합",
    "내용종합": "QuestionOptions내용종합",
    "주제영작": "QuestionOptions주제영작",
    "단어정리": "QuestionOptions단어정리",
}


class OptionsDashboard(CTkToplevel):
    """
    Unified dashboard showing all question type options in a single glance.
    Mirrors the overview layout from the web Settings page.
    """

    @staticmethod
    def _storage_key(question_type: str) -> str:
        """Return the canonical key used in option JSON for a question type."""
        return question_type.replace("(", "").replace(")", "")

    @staticmethod
    def _get_option_value(data: dict, storage_key: str, question_type: str, suffix: str, default=None):
        """Read option value using storage key, falling back to legacy keys with parentheses."""
        return data.get(f"{storage_key}{suffix}", data.get(f"{question_type}{suffix}", default))

    TYPES_WITH_COUNTS = {"빈칸추론", "내용일치", "내용일치(한)", "추론", "추론불가"}
    TYPES_WITH_MODE = {"빈칸추론", "내용일치", "내용일치(한)", "추론", "추론불가", "요지", "주제영작"}
    TYPES_WITH_BLANK_MODE = {"영작1", "영작2"}
    COUNT_DEFAULTS = {
        "빈칸추론": {"normal_min": 1, "normal_max": 4, "hard_min": 5, "hard_max": 10},
        "내용일치": {"normal_min": 10, "normal_max": 20, "hard_min": 15, "hard_max": 25},
        "내용일치(한)": {"normal_min": 10, "normal_max": 20, "hard_min": 15, "hard_max": 25},
        "추론": {"normal_min": 10, "normal_max": 15, "hard_min": 17, "hard_max": 22},
        "추론불가": {"normal_min": 10, "normal_max": 15, "hard_min": 17, "hard_max": 22},
    }

    def __init__(self, parent, default_font="Arial", main_frame_ref=None, manual_mocktest_popup=None):
        super().__init__(parent)
        self.title("문제 옵션 관리")
        self.geometry("700x720")
        self.minsize(680, 680)

        self.is_options_dashboard = True

        self.parent = parent
        self.main_frame_ref = main_frame_ref
        self.manual_mocktest_popup = manual_mocktest_popup
        self.default_font = default_font
        self.app_data_dir = Path.home() / ".my_application_data"
        self.make_questions_options_file_path = os.path.join(
            self.app_data_dir, "make_questions_options.json"
        )

        self.question_types = [
            "어휘1단계",
            "어휘2단계",
            "어휘3단계",
            "내용일치",
            "내용일치(한)",
            "밑줄의미",
            "함축의미",
            "무관한문장",
            "빈칸추론",
            "추론",
            "추론불가",
            "순서",
            "삽입",
            "주제",
            "제목",
            "요지",
            "연결어",
            "요약",
            "어법1단계",
            "어법2단계",
            "영작1",
            "영작2",
            "어휘종합",
            "내용종합",
            "주제영작",
            "동형문제",
            "단어정리",
        ]

        self.selected_types = set()
        self.row_widgets = {}

        self.all_options_data = {}
        self.options_state = {}
        self.baseline_state = {}

        self._load_all_options()
        self._initialise_state()
        self._build_ui()
        self._setup_key_bindings()

    # ------------------------------------------------------------------ #
    # Key bindings
    # ------------------------------------------------------------------ #
    def _setup_key_bindings(self):
        """Setup keyboard shortcuts for the dashboard"""
        # ESC to cancel
        self.bind("<Escape>", lambda e: self.on_save_and_close())

        # Command+Enter (Mac) or Control+Enter (Windows/Linux) to save and close
        if sys.platform == 'darwin':
            self.bind("<Command-Return>", lambda e: self.on_save_and_close())
        else:
            self.bind("<Control-Return>", lambda e: self.on_save_and_close())

    # ------------------------------------------------------------------ #
    # Data handling
    # ------------------------------------------------------------------ #
    def _load_all_options(self):
        if os.path.exists(self.make_questions_options_file_path):
            try:
                with open(self.make_questions_options_file_path, "r", encoding="utf-8") as f:
                    self.all_options_data = json.load(f)
            except Exception:
                self.all_options_data = {}
        else:
            self.all_options_data = {}

    def _initialise_state(self):
        for question_type in self.question_types:
            baseline = self.get_baseline_values(question_type)
            self.baseline_state[question_type] = baseline
            self.options_state[question_type] = self.get_current_values(question_type, baseline)

    def get_baseline_values(self, question_type):
        baseline = {
            "paraphrase_enabled": False,
            "difficulty_level": 2,
            "length_level": 2,
            "content_change_mode": 0,
        }

        if question_type in self.TYPES_WITH_COUNTS:
            defaults = self.COUNT_DEFAULTS.get(
                question_type,
                {"normal_min": 1, "normal_max": 5, "hard_min": 0, "hard_max": 3},
            )
            # 내용일치 and 내용일치(한) have mode=0 (랜덤), others have mode=1
            default_mode = 0 if question_type in {"내용일치", "내용일치(한)"} else 1
            baseline.update(
                {
                    "normal_min_val": defaults["normal_min"],
                    "normal_max_val": defaults["normal_max"],
                    "hard_min_val": defaults["hard_min"],
                    "hard_max_val": defaults["hard_max"],
                    "mode": default_mode,
                }
            )
        elif question_type in {"요지"}:
            baseline["mode"] = 1

        if question_type == "주제영작":
            baseline["mode"] = 1
            baseline["number_of_words"] = 10

        if question_type in self.TYPES_WITH_BLANK_MODE:
            baseline["blank_mode"] = 0
            baseline["questions_per_passage"] = 1  # Default: 1개

        if question_type == "단어정리":
            baseline["vocab_level"] = 1  # Default: 고1

        if question_type == "어법2단계":
            baseline["mode"] = 1  # Default: 1개 (backward compatible)

        return baseline

    def get_current_values(self, question_type, baseline):
        values = baseline.copy()
        data = self.all_options_data
        storage_key = self._storage_key(question_type)

        values["paraphrase_enabled"] = bool(
            self._get_option_value(data, storage_key, question_type, "paraphrase_enabled", baseline.get("paraphrase_enabled", False))
        )
        values["difficulty_level"] = self._int_or_default(
            self._get_option_value(data, storage_key, question_type, "difficulty_level"), baseline.get("difficulty_level", 2)
        )
        values["length_level"] = self._int_or_default(
            self._get_option_value(data, storage_key, question_type, "length_level"), baseline.get("length_level", 2)
        )
        values["content_change_mode"] = self._int_or_default(
            self._get_option_value(data, storage_key, question_type, "content_change_mode"), baseline.get("content_change_mode", 0)
        )
        if not values["paraphrase_enabled"]:
            values["difficulty_level"] = baseline.get("difficulty_level", 2)
            values["length_level"] = baseline.get("length_level", 2)
            values["content_change_mode"] = baseline.get("content_change_mode", 0)

        if question_type in self.TYPES_WITH_COUNTS:
            values["normal_min_val"] = self._int_or_default(
                self._get_option_value(data, storage_key, question_type, "normal_min_val"), baseline.get("normal_min_val", 1)
            )
            values["normal_max_val"] = self._int_or_default(
                self._get_option_value(data, storage_key, question_type, "normal_max_val"), baseline.get("normal_max_val", 5)
            )
            values["hard_min_val"] = self._int_or_default(
                self._get_option_value(data, storage_key, question_type, "hard_min_val"), baseline.get("hard_min_val", 0)
            )
            values["hard_max_val"] = self._int_or_default(
                self._get_option_value(data, storage_key, question_type, "hard_max_val"), baseline.get("hard_max_val", 3)
            )
            values["mode"] = self._int_or_default(
                self._get_option_value(data, storage_key, question_type, "mode"), baseline.get("mode", 1)
            )
        elif question_type in {"요지", "주제영작"}:
            values["mode"] = self._int_or_default(
                self._get_option_value(data, storage_key, question_type, "mode"), baseline.get("mode", 1)
            )

        if question_type in self.TYPES_WITH_BLANK_MODE:
            values["blank_mode"] = self._int_or_default(
                self._get_option_value(data, storage_key, question_type, "_blank_mode"), baseline.get("blank_mode", 0)
            )
            values["questions_per_passage"] = self._int_or_default(
                self._get_option_value(data, storage_key, question_type, "_questions_per_passage"), baseline.get("questions_per_passage", 1)
            )

        if question_type == "주제영작":
            values["number_of_words"] = self._int_or_default(
                self._get_option_value(data, storage_key, question_type, "number_of_words"), baseline.get("number_of_words", 10)
            )

        if question_type == "단어정리":
            values["vocab_level"] = self._int_or_default(
                self._get_option_value(data, storage_key, question_type, "vocab_level"),
                baseline.get("vocab_level", 1)
            )

        if question_type == "어법2단계":
            values["mode"] = self._int_or_default(
                self._get_option_value(data, storage_key, question_type, "mode"),
                baseline.get("mode", 1)  # Default: 1개
            )

        return values

    def _int_or_default(self, value, default):
        try:
            if value is None or value == "":
                return default
            return int(value)
        except (TypeError, ValueError):
            return default

    # ------------------------------------------------------------------ #
    # UI construction
    # ------------------------------------------------------------------ #
    def _build_ui(self):
        main_container = CTkFrame(self, fg_color="transparent")
        main_container.pack(fill="both", expand=True, padx=16, pady=16)
        main_container.grid_rowconfigure(0, weight=1)
        main_container.grid_columnconfigure(0, weight=1)

        table_wrapper = CTkFrame(main_container, fg_color="#FFFFFF", corner_radius=12)
        table_wrapper.grid(row=0, column=0, sticky="nsew", pady=(0, 12))
        table_wrapper.grid_rowconfigure(1, weight=1)
        table_wrapper.grid_columnconfigure(0, weight=1)

        header_frame = CTkFrame(table_wrapper, fg_color="#E5ECE1", corner_radius=0)
        header_frame.grid(row=0, column=0, sticky="ew")
        self._configure_table_columns(header_frame)

        self.select_all_var = tk.IntVar(value=0)
        self.select_all_checkbox = CTkCheckBox(
            header_frame,
            text="",
            variable=self.select_all_var,
            command=self.toggle_select_all,
            width=18,
        )
        self.select_all_checkbox.grid(row=0, column=0, padx=(16, 8), pady=12)

        CTkLabel(
            header_frame,
            text="유형",
            font=(self.default_font, 10, "bold"),
            text_color="#2D3A28",
        ).grid(row=0, column=1, sticky="w", padx=6)

        CTkLabel(
            header_frame,
            text="상태",
            font=(self.default_font, 10, "bold"),
            text_color="#2D3A28",
        ).grid(row=0, column=2, sticky="w", padx=6)

        CTkLabel(
            header_frame,
            text="주요 설정",
            font=(self.default_font, 10, "bold"),
            text_color="#2D3A28",
        ).grid(row=0, column=3, sticky="w", padx=6)

        CTkLabel(
            header_frame,
            text="작업",
            font=(self.default_font, 10, "bold"),
            text_color="#2D3A28",
        ).grid(row=0, column=4, sticky="e", padx=(6, 12))

        self.table_body = CTkScrollableFrame(
            table_wrapper,
            fg_color="#FFFFFF",
            corner_radius=0,
        )
        self.table_body.grid(row=1, column=0, sticky="nsew")
        self.table_body.grid_columnconfigure(0, weight=1)

        self.reset_selected_btn = None
        self._render_rows()

        footer = CTkFrame(main_container, fg_color="transparent")
        footer.grid(row=1, column=0, sticky="ew", pady=(0, 0))

        self.reset_selected_btn = CTkButton(
            footer,
            text="선택한 유형 기본값 복원",
            command=self.reset_selected_to_defaults,
            fg_color="#FFFFFF",
            hover_color="#E4E7E2",
            text_color="#1F2D1A",
            font=(self.default_font, 11),
            width=130,
            state="disabled",
        )
        self.reset_selected_btn.grid(row=0, column=0, sticky="w", padx=(0, 8))

        self._update_reset_selected_state()

        CTkButton(
            footer,
            text="전체 기본값 복원",
            command=self.reset_all_to_defaults,
            fg_color="#FF6B6B",
            hover_color="#FF5252",
            text_color="white",
            font=(self.default_font, 11, "bold"),
            width=120,
        ).grid(row=0, column=1, sticky="w")

        CTkButton(
            footer,
            text="지문변형 일괄 적용",
            command=self.open_bulk_paraphrase_settings,
            fg_color="#4A90E2",
            hover_color="#3A7BC8",
            text_color="white",
            font=(self.default_font, 11, "bold"),
            width=130,
        ).grid(row=0, column=2, sticky="w", padx=(8, 0))

        # Spacer column (column 3) expands to push right buttons to the right
        footer.grid_columnconfigure(3, weight=1)

        CTkButton(
            footer,
            text="닫기",
            command=self.on_save_and_close,
            fg_color="#DDA15C",
            hover_color="#BB873F",
            text_color="black",
            font=(self.default_font, 11, "bold"),
            width=100,
        ).grid(row=0, column=4)

    def _configure_table_columns(self, frame):
        frame.grid_columnconfigure(0, minsize=28)
        frame.grid_columnconfigure(1, minsize=100)
        frame.grid_columnconfigure(2, minsize=80)
        frame.grid_columnconfigure(3, weight=1, minsize=250)
        frame.grid_columnconfigure(4, minsize=100)

    def _render_rows(self):
        for child in self.table_body.winfo_children():
            child.destroy()
        self.row_widgets.clear()

        for index, question_type in enumerate(self.question_types):
            row = CTkFrame(self.table_body, fg_color="#FFFFFF", border_color="#EEF2ED", border_width=1)
            row.grid(row=index, column=0, sticky="ew", padx=(0, 0), pady=(0, 6))
            self._configure_table_columns(row)

            select_var = tk.IntVar(value=1 if question_type in self.selected_types else 0)
            checkbox = CTkCheckBox(
                row,
                text="",
                variable=select_var,
                command=lambda qt=question_type: self.handle_row_selection(qt),
                width=18,
            )
            checkbox.grid(row=0, column=0, padx=(16, 8), pady=10, sticky="w")

            type_label = CTkLabel(
                row,
                text=question_type,
                font=(self.default_font, 11, "bold"),
                text_color="#1F2D1A",
            )
            type_label.grid(row=0, column=1, sticky="w")

            status_label = CTkLabel(
                row,
                text="",
                font=(self.default_font, 11, "bold"),
                text_color="#2F855A",
                fg_color="#E5F4E3",
                corner_radius=999,
                padx=8,
                pady=3,
            )
            status_label.grid(row=0, column=2, sticky="w", padx=6)

            summary_label = CTkLabel(
                row,
                text="",
                justify="left",
                anchor="w",
                font=(self.default_font, 11),
                text_color="#394338",
            )
            summary_label.grid(row=0, column=3, sticky="ew", padx=6, pady=2)

            edit_button = CTkButton(
                row,
                text="옵션 수정",
                command=lambda qt=question_type: self.open_edit_dialog(qt),
                width=85,
                fg_color="#2F855A",
                hover_color="#25684A",
                text_color="white",
                font=(self.default_font, 11, "bold"),
            )
            edit_button.grid(row=0, column=4, sticky="e", padx=(8, 12))

            self.row_widgets[question_type] = {
                "frame": row,
                "select_var": select_var,
                "checkbox": checkbox,
                "status_label": status_label,
                "summary_label": summary_label,
                "edit_button": edit_button,
            }

            self.refresh_row(question_type)

        self._update_select_all_checkbox()
        self._update_reset_selected_state()

    # ------------------------------------------------------------------ #
    # Row interactions
    # ------------------------------------------------------------------ #
    def open_edit_dialog(self, question_type):
        if question_type == "동형문제":
            popup = self._create_copy_question_popup()
            if popup is None:
                return
            self._make_popup_modal(popup)
            self.wait_window(popup)
            self._reload_state_for(question_type)
            return

        popup_cls = self._get_popup_class(question_type)
        if popup_cls is None:
            messagebox.showinfo("알림", f"'{question_type}' 옵션 창은 아직 준비 중입니다.", parent=self)
            return

        popup = self._create_option_popup(popup_cls, question_type)
        if popup is None:
            return

        self._make_popup_modal(popup)
        self.wait_window(popup)
        self._reload_state_for(question_type)

    def _get_popup_class(self, question_type: str) -> Optional[Type[CTkToplevel]]:
        class_name = QUESTION_TYPE_CLASS_MAP.get(question_type)
        if not class_name:
            return None
        popup_cls = getattr(question_option_windows, class_name, None)
        return popup_cls if isinstance(popup_cls, type) else None

    def _create_option_popup(self, popup_cls: Type[CTkToplevel], question_type: str) -> Optional[CTkToplevel]:
        kwargs = {"parent": self, "default_font": self.default_font}
        if self.main_frame_ref is not None:
            kwargs["main_frame"] = self.main_frame_ref
        try:
            return popup_cls(**kwargs)
        except TypeError:
            kwargs.pop("main_frame", None)
            try:
                return popup_cls(**kwargs)
            except Exception as exc:
                messagebox.showerror(
                    "옵션 창 오류",
                    f"'{question_type}' 옵션 창을 여는 중 오류가 발생했습니다.\n\n{exc}",
                    parent=self,
                )
                return None
        except Exception as exc:
            messagebox.showerror(
                "옵션 창 오류",
                f"'{question_type}' 옵션 창을 여는 중 오류가 발생했습니다.\n\n{exc}",
                parent=self,
            )
            return None

    def _create_copy_question_popup(self) -> Optional[CTkToplevel]:
        if self.main_frame_ref is None:
            messagebox.showinfo("알림", "메인 창이 준비되지 않아 '동형문제' 옵션을 열 수 없습니다.", parent=self)
            return None
        try:
            return CopyQuestionPopup(
                parent=self,
                default_font=self.default_font,
                owner_app=self.main_frame_ref,
                dashboard_ref=self  # Pass dashboard reference for real-time updates
            )
        except Exception as exc:
            messagebox.showerror("옵션 창 오류", f"'동형문제' 옵션 창을 여는 중 오류가 발생했습니다.\n\n{exc}", parent=self)
            return None

    def _make_popup_modal(self, popup: CTkToplevel) -> None:
        try:
            popup.transient(self)
        except Exception:
            pass
        try:
            popup.grab_set()
        except Exception:
            pass
        try:
            popup.focus_set()
        except Exception:
            pass

    def _reload_state_for(self, question_type: str) -> None:
        self._load_all_options()
        baseline = self.baseline_state.get(question_type)
        if baseline is None:
            baseline = self.get_baseline_values(question_type)
            self.baseline_state[question_type] = baseline
        self.options_state[question_type] = self.get_current_values(question_type, baseline)
        self.refresh_row(question_type)

    def refresh_row(self, question_type):
        row = self.row_widgets.get(question_type)
        if not row:
            return

        # Safety check: verify widgets still exist before configuring
        try:
            # Test if the widget still exists by checking if it has a valid master
            if not row["summary_label"].winfo_exists():
                return
        except tk.TclError:
            # Widget has been destroyed
            return

        summary_lines = self.compute_summary_lines(question_type)
        summary_text = "\n".join(f"• {line}" for line in summary_lines)

        try:
            row["summary_label"].configure(text=summary_text)
        except tk.TclError:
            # Widget was destroyed between the check and configure
            return

        is_modified = self.is_type_modified(question_type)
        try:
            if is_modified:
                row["status_label"].configure(
                    text="수정됨",
                    fg_color="#FEEBC8",
                    text_color="#9C4221",
                )
            else:
                row["status_label"].configure(
                    text="기본값",
                    fg_color="#E5F4E3",
                    text_color="#2F855A",
                )
        except tk.TclError:
            # Widget was destroyed during configuration
            return

    def handle_row_selection(self, question_type):
        row = self.row_widgets.get(question_type)
        if not row:
            return
        selected = bool(row["select_var"].get())
        self._set_row_selection(question_type, selected, update_var=False)

    def toggle_select_all(self):
        should_select = bool(self.select_all_var.get())
        for question_type in self.question_types:
            self._set_row_selection(question_type, should_select, update_var=True)

    def _set_row_selection(self, question_type, selected, update_var=True):
        row = self.row_widgets.get(question_type)
        if not row:
            return
        if update_var:
            row["select_var"].set(1 if selected else 0)

        if selected:
            self.selected_types.add(question_type)
        else:
            self.selected_types.discard(question_type)

        self._update_select_all_checkbox()
        self._update_reset_selected_state()

    def _update_select_all_checkbox(self):
        total = len(self.question_types)
        selected = len(self.selected_types)
        self.select_all_var.set(1 if selected == total and total > 0 else 0)

    def _update_reset_selected_state(self):
        if not getattr(self, "reset_selected_btn", None):
            return
        state = "normal" if self.selected_types else "disabled"
        self.reset_selected_btn.configure(state=state)

    # ------------------------------------------------------------------ #
    # Summary helpers
    # ------------------------------------------------------------------ #
    def compute_summary_lines(self, question_type):
        values = self.options_state.get(question_type, {})
        lines = []

        if "paraphrase_enabled" in values:
            lines.append(f"지문 변형: {'사용' if values['paraphrase_enabled'] else '미사용'}")

        if values.get("paraphrase_enabled", False):
            lines.append(f"난이도: {self._difficulty_label(values.get('difficulty_level', 2))}")
            lines.append(f"길이: {self._length_label(values.get('length_level', 2))}")
            lines.append(f"내용 변경: {self._content_change_label(values.get('content_change_mode', 0))}")

        if question_type in self.TYPES_WITH_COUNTS:
            lines.append(
                f"Normal 선지 길이: {values.get('normal_min_val', '-')} ~ {values.get('normal_max_val', '-')} 단어"
            )
            lines.append(
                f"Hard 선지 길이: {values.get('hard_min_val', '-')} ~ {values.get('hard_max_val', '-')} 단어"
            )

        if question_type in {"추론", "추론불가"}:
            lines.append(f"정답 개수: {'2개' if values.get('mode', 1) == 2 else '1개'}")
        elif question_type == "빈칸추론":
            lines.append(
                f"정답 재진술: {'원문 그대로' if values.get('mode', 1) == 2 else '재진술'}"
            )
        elif question_type == "요지":
            lines.append(
                f"선지 언어: {'영어' if values.get('mode', 1) == 2 else '한글'}"
            )
        elif question_type in {"내용일치", "내용일치(한)"}:
            mode = values.get('mode', 0)
            mode_text = {
                0: "랜덤(1~4개)",
                1: "1개",
                2: "2개",
                3: "3개",
                4: "4개"
            }.get(mode, "랜덤(1~4개)")
            lines.append(f"정답(일치) 개수: {mode_text}")
        elif question_type == "주제영작":
            lines.append(
                f"주제 유형: {'문장' if values.get('mode', 1) == 2 else '명사구'}"
            )
            lines.append(f"주제(문) 길이: {values.get('number_of_words', 10)} 단어")
        elif question_type == "어법2단계":
            mode = values.get('mode', 1)  # Default: 1개
            mode_text = {
                1: "1개",
                2: "2개",
                3: "3개",
                4: "4개(맞는 것 고르기)",
                0: "랜덤(2~4개)"
            }.get(mode, "1개")
            lines.append(f"정답(틀린 것) 개수: {mode_text}")

        if question_type in self.TYPES_WITH_BLANK_MODE:
            lines.append(
                f"한글 뜻 없이 빈칸 출제: {'예' if values.get('blank_mode', 0) == 1 else '아니오'}"
            )
            lines.append(f"한 지문당 영작 문제 수: {values.get('questions_per_passage', 1)}개")

        if question_type == "단어정리":
            vocab_labels = {0: "중3", 1: "고1", 2: "고2", 3: "고3"}
            lines.append(f"어휘 수준: {vocab_labels.get(values.get('vocab_level', 1), '고1')}")

        if not lines:
            lines.append("기본 옵션 사용 중")

        return lines

    def _difficulty_label(self, value):
        return {1: "쉽게", 2: "같은 난이도", 3: "어렵게"}.get(value, "같은 난이도")

    def _length_label(self, value):
        return {1: "짧게", 2: "비슷하게", 3: "길게"}.get(value, "비슷하게")

    def _content_change_label(self, value):
        return {0: "내용 변화 없음", 1: "세부 사항 변경", 2: "요지 변경", 3: "간접연계"}.get(value, "내용 변화 없음")

    def is_type_modified(self, question_type):
        baseline = self.baseline_state.get(question_type, {})
        current = self.options_state.get(question_type, {})
        for key, base in baseline.items():
            if current.get(key) != base:
                return True
        return False

    # ------------------------------------------------------------------ #
    # Actions
    # ------------------------------------------------------------------ #
    def reset_selected_to_defaults(self):
        if not self.selected_types:
            messagebox.showinfo("알림", "기본값으로 되돌릴 유형을 먼저 선택하세요.", parent=self)
            return

        if not messagebox.askyesno(
            "기본값 복원 확인",
            f"선택한 {len(self.selected_types)}개 유형을 기본값으로 되돌리시겠습니까?",
            parent=self,
        ):
            return

        for question_type in list(self.selected_types):
            self.reset_type_to_defaults(question_type)
            self._set_row_selection(question_type, False, update_var=True)

        # Immediately save changes to file
        self.save_all_options()
        messagebox.showinfo("완료", "선택한 유형을 기본값으로 되돌렸습니다.", parent=self)

    def reset_type_to_defaults(self, question_type):
        baseline = self.baseline_state.get(question_type, {}).copy()
        self.options_state[question_type] = baseline
        self.refresh_row(question_type)

    def reset_all_to_defaults(self):
        if not messagebox.askyesno(
            "전체 기본값 복원",
            "모든 문제 유형의 옵션을 기본값으로 복원하시겠습니까?\n\n이 작업은 되돌릴 수 없습니다.",
            parent=self,
        ):
            return

        for question_type in self.question_types:
            self.reset_type_to_defaults(question_type)
        self.selected_types.clear()
        self._update_select_all_checkbox()
        self._update_reset_selected_state()

        # Immediately save changes to file
        self.save_all_options()
        messagebox.showinfo("완료", "모든 옵션이 기본값으로 복원되었습니다.", parent=self)

    def save_all_options(self):
        if not os.path.exists(self.app_data_dir):
            os.makedirs(self.app_data_dir, exist_ok=True)

        existing_data = {}
        if os.path.exists(self.make_questions_options_file_path):
            try:
                with open(self.make_questions_options_file_path, "r", encoding="utf-8") as f:
                    existing_data = json.load(f)
            except Exception as exc:
                print(f"[OptionsDashboard] 기존 옵션 파일 읽기 실패: {exc}")
                existing_data = {}

        serialized = {}
        for question_type in self.question_types:
            storage_key = self._storage_key(question_type)
            values = self.options_state.get(question_type, {})
            serialized[f"{storage_key}paraphrase_enabled"] = bool(values.get("paraphrase_enabled", False))
            serialized[f"{storage_key}difficulty_level"] = int(values.get("difficulty_level", 2))
            serialized[f"{storage_key}length_level"] = int(values.get("length_level", 2))
            # Save content_change_mode (None when paraphrase disabled, actual value when enabled)
            if values.get("paraphrase_enabled", False):
                serialized[f"{storage_key}content_change_mode"] = int(values.get("content_change_mode", 0))
            else:
                serialized[f"{storage_key}content_change_mode"] = None

            if question_type in self.TYPES_WITH_COUNTS:
                defaults = self.COUNT_DEFAULTS.get(
                    question_type,
                    {"normal_min": 1, "normal_max": 5, "hard_min": 0, "hard_max": 3},
                )
                serialized[f"{storage_key}normal_min_val"] = int(
                    values.get("normal_min_val", defaults["normal_min"])
                )
                serialized[f"{storage_key}normal_max_val"] = int(
                    values.get("normal_max_val", defaults["normal_max"])
                )
                serialized[f"{storage_key}hard_min_val"] = int(
                    values.get("hard_min_val", defaults["hard_min"])
                )
                serialized[f"{storage_key}hard_max_val"] = int(
                    values.get("hard_max_val", defaults["hard_max"])
                )
                serialized[f"{storage_key}mode"] = int(values.get("mode", 1))
            elif question_type in {"요지", "주제영작"}:
                serialized[f"{storage_key}mode"] = int(values.get("mode", 1))

            if question_type in self.TYPES_WITH_BLANK_MODE:
                serialized[f"{storage_key}_blank_mode"] = int(values.get("blank_mode", 0))
                serialized[f"{storage_key}_questions_per_passage"] = int(values.get("questions_per_passage", 1))

            if question_type == "주제영작":
                serialized[f"{storage_key}number_of_words"] = int(values.get("number_of_words", 10))

            if question_type == "단어정리":
                serialized[f"{storage_key}vocab_level"] = int(values.get("vocab_level", 1))

            if question_type == "어법2단계":
                serialized[f"{storage_key}mode"] = int(values.get("mode", 1))  # Default: 1개

        # Preserve metadata (migration flags, etc.) that are stored alongside options.
        for key, value in existing_data.items():
            if key.startswith("_"):
                serialized.setdefault(key, value)

        with open(self.make_questions_options_file_path, "w", encoding="utf-8") as f:
            json.dump(serialized, f, ensure_ascii=False, indent=4)

        self.all_options_data = serialized

        # Update option buttons in main frame
        if self.main_frame_ref and hasattr(self.main_frame_ref, "update_option_button_color"):
            for question_type in self.question_types:
                try:
                    self.main_frame_ref.update_option_button_color(question_type)
                except tk.TclError:
                    continue
                except Exception as exc:
                    print(f"Failed to refresh option button for {question_type}: {exc}")

        # Update option buttons in manual mocktest popup if it exists
        if self.manual_mocktest_popup and hasattr(self.manual_mocktest_popup, "update_all_option_button_colors"):
            try:
                self.manual_mocktest_popup.update_all_option_button_colors()
                print("[INFO] Updated manual mocktest popup option buttons from dashboard")
            except tk.TclError:
                pass
            except Exception as exc:
                print(f"Failed to refresh manual mocktest popup buttons: {exc}")

    def on_save_and_close(self):
        self.save_all_options()
        self.destroy()

    def on_cancel(self):
        self.destroy()

    def open_bulk_paraphrase_settings(self):
        """Open popup to apply paraphrase settings to selected types."""
        if not self.selected_types:
            messagebox.showinfo("알림", "지문변형 옵션을 적용할 유형을 먼저 선택하세요.", parent=self)
            return

        popup = BulkParaphrasePopup(parent=self, default_font=self.default_font, dashboard=self)
        self._make_popup_modal(popup)
        self.wait_window(popup)


class BulkParaphrasePopup(CTkToplevel):
    """Popup window for bulk applying paraphrase settings to multiple question types."""

    def __init__(self, parent, default_font="Arial", dashboard=None):
        super().__init__(parent)
        self.title("지문변형 일괄 적용")
        self.geometry("650x280")
        self.minsize(650, 280)

        self.parent = parent
        self.dashboard = dashboard
        self.default_font = default_font

        # Paraphrase transformation variables
        self.지문변형_var = tk.BooleanVar(value=False)
        self.지문난이도_var = tk.IntVar(value=2)
        self.지문길이_var = tk.IntVar(value=2)
        self.content_change_var = tk.IntVar(value=0)

        self._build_ui()
        self.toggle_difficulty_options()

    def _build_ui(self):
        # Main frame
        main_frame = CTkFrame(self, fg_color="transparent")
        main_frame.pack(pady=10, padx=20, fill='both', expand=True)

        # Info label
        info_label = CTkLabel(
            main_frame,
            text=f"선택한 {len(self.dashboard.selected_types)}개 유형에 아래 설정을 적용합니다.",
            font=(self.default_font, 12, "bold"),
            text_color="#2F855A",
        )
        info_label.pack(anchor="w", pady=(0, 10))

        # Passage transformation frame
        transform_frame = CTkFrame(main_frame, fg_color="transparent")
        transform_frame.pack(side='top', fill='x', expand=False, pady=(10, 0))

        # Checkbox
        self.transform_checkbox = CTkCheckBox(
            transform_frame,
            text="지문을 변형하여 출제",
            variable=self.지문변형_var,
            command=self.toggle_difficulty_options,
            font=(self.default_font, 13),
            text_color="black",
            hover_color="#BB6C25",
            fg_color="#DDA15C",
            border_color="gray"
        )
        self.transform_checkbox.pack(side='top', anchor='w', pady=(0, 5))

        # Difficulty and length options container
        self.difficulty_container = CTkFrame(transform_frame, fg_color="transparent")
        self.difficulty_container.grid_columnconfigure(0, weight=0)
        self.difficulty_container.grid_columnconfigure(1, weight=1)

        # Difficulty label and radio buttons
        self.difficulty_label = CTkLabel(
            self.difficulty_container,
            text="지문 난이도:",
            font=(self.default_font, 13, 'bold'),
            text_color="black"
        )
        self.difficulty_label.grid(row=0, column=0, padx=(0, 10), pady=5, sticky="w")

        difficulty_radio_frame = CTkFrame(self.difficulty_container, fg_color="transparent")
        difficulty_radio_frame.grid(row=0, column=1, sticky="w")

        self.radio_easier = CTkRadioButton(
            difficulty_radio_frame, text='더 쉽게', font=(self.default_font, 13),
            variable=self.지문난이도_var, value=1, text_color="black",
            hover_color="#BB6C25", fg_color="#DDA15C", border_color="gray"
        )
        self.radio_easier.pack(side='left', padx=(0, 10))

        self.radio_same = CTkRadioButton(
            difficulty_radio_frame, text='같은 난이도', font=(self.default_font, 13),
            variable=self.지문난이도_var, value=2, text_color="black",
            hover_color="#BB6C25", fg_color="#DDA15C", border_color="gray"
        )
        self.radio_same.pack(side='left', padx=(0, 10))

        self.radio_harder = CTkRadioButton(
            difficulty_radio_frame, text='더 높게', font=(self.default_font, 13),
            variable=self.지문난이도_var, value=3, text_color="black",
            hover_color="#BB6C25", fg_color="#DDA15C", border_color="gray"
        )
        self.radio_harder.pack(side='left')

        # Length label and radio buttons
        self.length_label = CTkLabel(
            self.difficulty_container,
            text="지문 길이:",
            font=(self.default_font, 13, 'bold'),
            text_color="black"
        )
        self.length_label.grid(row=1, column=0, padx=(0, 10), pady=5, sticky="w")

        length_radio_frame = CTkFrame(self.difficulty_container, fg_color="transparent")
        length_radio_frame.grid(row=1, column=1, sticky="w")

        self.radio_shorter = CTkRadioButton(
            length_radio_frame, text='더 짧게', font=(self.default_font, 13),
            variable=self.지문길이_var, value=1, text_color="black",
            hover_color="#BB6C25", fg_color="#DDA15C", border_color="gray"
        )
        self.radio_shorter.pack(side='left', padx=(0, 10))

        self.radio_similar = CTkRadioButton(
            length_radio_frame, text='비슷하게', font=(self.default_font, 13),
            variable=self.지문길이_var, value=2, text_color="black",
            hover_color="#BB6C25", fg_color="#DDA15C", border_color="gray"
        )
        self.radio_similar.pack(side='left', padx=(0, 10))

        self.radio_longer = CTkRadioButton(
            length_radio_frame, text='더 길게', font=(self.default_font, 13),
            variable=self.지문길이_var, value=3, text_color="black",
            hover_color="#BB6C25", fg_color="#DDA15C", border_color="gray"
        )
        self.radio_longer.pack(side='left')

        # Content change mode label and radio buttons
        self.content_change_label = CTkLabel(
            self.difficulty_container,
            text="내용 변경 방식:",
            font=(self.default_font, 13, 'bold'),
            text_color="black"
        )
        self.content_change_label.grid(row=2, column=0, padx=(0, 10), pady=5, sticky="w")

        content_change_radio_frame = CTkFrame(self.difficulty_container, fg_color="transparent")
        content_change_radio_frame.grid(row=2, column=1, sticky="w")

        self.radio_content_none = CTkRadioButton(
            content_change_radio_frame, text='내용 변화 없음', font=(self.default_font, 13),
            variable=self.content_change_var, value=0, text_color="black",
            hover_color="#BB6C25", fg_color="#DDA15C", border_color="gray"
        )
        self.radio_content_none.pack(side='left', padx=(0, 10))

        self.radio_content_detail = CTkRadioButton(
            content_change_radio_frame, text='세부 사항 변경', font=(self.default_font, 13),
            variable=self.content_change_var, value=1, text_color="black",
            hover_color="#BB6C25", fg_color="#DDA15C", border_color="gray"
        )
        self.radio_content_detail.pack(side='left', padx=(0, 10))

        self.radio_content_main = CTkRadioButton(
            content_change_radio_frame, text='요지 변경', font=(self.default_font, 13),
            variable=self.content_change_var, value=2, text_color="black",
            hover_color="#BB6C25", fg_color="#DDA15C", border_color="gray"
        )
        self.radio_content_main.pack(side='left', padx=(0, 10))

        self.radio_content_external = CTkRadioButton(
            content_change_radio_frame, text='간접연계', font=(self.default_font, 13),
            variable=self.content_change_var, value=3, text_color="black",
            hover_color="#BB6C25", fg_color="#DDA15C", border_color="gray"
        )
        self.radio_content_external.pack(side='left')

        # Pack difficulty container
        self.difficulty_container.pack(side='top', fill='x', expand=False, padx=(20, 0))

        # Action buttons at the bottom
        button_frame = CTkFrame(main_frame, fg_color="transparent")
        button_frame.pack(side='bottom', fill='x', pady=(15, 0))

        CTkButton(
            button_frame,
            text="취소",
            command=self.destroy,
            fg_color="#F0F2ED",
            hover_color="#E0E4DA",
            text_color="#1F2D1A",
            font=(self.default_font, 12),
            width=100,
        ).pack(side='right', padx=(10, 0))

        CTkButton(
            button_frame,
            text="확인",
            command=self.apply_settings,
            fg_color="#DDA15C",
            hover_color="#BB873F",
            text_color="black",
            font=(self.default_font, 12, "bold"),
            width=100,
        ).pack(side='right')

    def toggle_difficulty_options(self):
        """Enable/disable radio buttons based on checkbox state."""
        is_enabled = self.지문변형_var.get()
        state = "normal" if is_enabled else "disabled"
        text_color = "black" if is_enabled else "gray"
        border_color = "gray" if is_enabled else "#d0d0d0"

        # Update labels
        self.difficulty_label.configure(text_color=text_color)
        self.length_label.configure(text_color=text_color)
        self.content_change_label.configure(text_color=text_color)

        # Update radio buttons
        self.radio_easier.configure(state=state, text_color=text_color, border_color=border_color)
        self.radio_same.configure(state=state, text_color=text_color, border_color=border_color)
        self.radio_harder.configure(state=state, text_color=text_color, border_color=border_color)
        self.radio_shorter.configure(state=state, text_color=text_color, border_color=border_color)
        self.radio_similar.configure(state=state, text_color=text_color, border_color=border_color)
        self.radio_longer.configure(state=state, text_color=text_color, border_color=border_color)
        self.radio_content_none.configure(state=state, text_color=text_color, border_color=border_color)
        self.radio_content_detail.configure(state=state, text_color=text_color, border_color=border_color)
        self.radio_content_main.configure(state=state, text_color=text_color, border_color=border_color)
        self.radio_content_external.configure(state=state, text_color=text_color, border_color=border_color)

    def apply_settings(self):
        """Apply paraphrase settings to all selected question types."""
        paraphrase_enabled = self.지문변형_var.get()
        difficulty_level = self.지문난이도_var.get()
        length_level = self.지문길이_var.get()
        content_change_mode = self.content_change_var.get()

        # Apply to all selected types
        for question_type in self.dashboard.selected_types:
            state = self.dashboard.options_state.get(question_type, {})
            state["paraphrase_enabled"] = paraphrase_enabled
            state["difficulty_level"] = difficulty_level
            state["length_level"] = length_level
            state["content_change_mode"] = content_change_mode
            self.dashboard.options_state[question_type] = state
            self.dashboard.refresh_row(question_type)

        # Immediately save changes to file
        self.dashboard.save_all_options()

        messagebox.showinfo(
            "완료",
            f"선택한 {len(self.dashboard.selected_types)}개 유형에 지문변형 설정을 적용했습니다.",
            parent=self
        )
        self.destroy()
