# modules/email_inquiry_dialog.py

import logging
from tkinter import messagebox
from customtkinter import CTkToplevel, CTkFrame, CTkLabel, CTkEntry, CTkTextbox, CTkButton

app_logger = logging.getLogger("app")


class EmailInquiryDialog(CTkToplevel):
    """Dialog for collecting inquiry information before sending email."""

    def __init__(self, parent, username, excel_file, mailer, default_font="Arial"):
        super().__init__(parent)
        self.title("문의하기")
        self.geometry("600x650")
        self.resizable(False, False)
        self.configure(fg_color="white")

        self.username = username
        self.excel_file = excel_file
        self.mailer = mailer
        self.default_font = default_font
        self.email_sent = False  # Track if email was sent successfully
        self.sending_in_progress = False  # Prevent duplicate sends

        # Make dialog modal
        self.transient(parent)
        self.grab_set()

        # Main frame
        main_frame = CTkFrame(self, fg_color="#eff3f0", corner_radius=10)
        main_frame.pack(fill="both", expand=True, padx=20, pady=20)

        # ID field (read-only)
        """
        id_label = CTkLabel(main_frame, text="ID:", font=(self.default_font, 13, "bold"),
                           text_color="black", anchor="w")
        id_label.pack(fill="x", padx=10, pady=(10, 2))

        self.id_entry = CTkEntry(main_frame, font=(self.default_font, 13),
                                 fg_color="white", text_color="gray", height=35)
        self.id_entry.insert(0, self.username)
        self.id_entry.configure(state="disabled")
        self.id_entry.pack(fill="x", padx=10, pady=(0, 10))
        """
        # FILE field (read-only)
        file_label = CTkLabel(main_frame, text="FILE:", font=(self.default_font, 13, "bold"),
                             text_color="black", anchor="w")
        file_label.pack(fill="x", padx=10, pady=(10, 2))

        # Truncate long file path for display
        display_path = self.excel_file
        if len(display_path) > 60:
            display_path = "..." + display_path[-57:]

        self.file_entry = CTkEntry(main_frame, font=(self.default_font, 13),
                                   fg_color="white", text_color="gray", height=35)
        self.file_entry.insert(0, display_path)
        self.file_entry.configure(state="disabled")
        self.file_entry.pack(fill="x", padx=10, pady=(0, 10))
        
        
        # 문의 내용 (inquiry content)
        inquiry_label = CTkLabel(main_frame, text="문의 내용:", font=(self.default_font, 13, "bold"),
                                text_color="black", anchor="w")
        inquiry_label.pack(fill="x", padx=10, pady=(10, 2))

        self.inquiry_text = CTkTextbox(main_frame, font=(self.default_font, 13),
                                       fg_color="white", text_color="black",
                                       height=150, wrap="word")
        self.inquiry_text.pack(fill="both", expand=True, padx=10, pady=(0, 10))

        # Add placeholder
        self.placeholder_text = "문의사항 혹은 오류 내용을 적어주세요"
        self.inquiry_text.insert("1.0", self.placeholder_text)
        self.inquiry_text.configure(text_color="gray")

        # Placeholder handlers
        def on_inquiry_focus_in(event):
            if self.inquiry_text.get("1.0", "end-1c").strip() == self.placeholder_text:
                self.inquiry_text.delete("1.0", "end")
                self.inquiry_text.configure(text_color="black")

        def on_inquiry_focus_out(event):
            if not self.inquiry_text.get("1.0", "end-1c").strip():
                self.inquiry_text.insert("1.0", self.placeholder_text)
                self.inquiry_text.configure(text_color="gray")

        self.inquiry_text.bind("<FocusIn>", on_inquiry_focus_in)
        self.inquiry_text.bind("<FocusOut>", on_inquiry_focus_out)

        # 이메일 주소 (email address)
        email_label = CTkLabel(main_frame, text="이메일 주소:", font=(self.default_font, 13, "bold"),
                              text_color="black", anchor="w")
        email_label.pack(fill="x", padx=10, pady=(10, 2))

        self.email_entry = CTkEntry(main_frame, font=(self.default_font, 13),
                                    placeholder_text="example@email.com",
                                    fg_color="white", text_color="black", height=35)
        self.email_entry.pack(fill="x", padx=10, pady=(0, 10))

        # 휴대폰 번호 (phone number)
        phone_label = CTkLabel(main_frame, text="휴대폰 번호:", font=(self.default_font, 13, "bold"),
                              text_color="black", anchor="w")
        phone_label.pack(fill="x", padx=10, pady=(10, 2))

        self.phone_entry = CTkEntry(main_frame, font=(self.default_font, 13),
                                    placeholder_text="010-1234-5678",
                                    fg_color="white", text_color="black", height=35)
        self.phone_entry.pack(fill="x", padx=10, pady=(0, 10))

        # Button frame
        button_frame = CTkFrame(main_frame, fg_color="transparent")
        button_frame.pack(fill="x", padx=10, pady=(10, 10))
        button_frame.grid_columnconfigure(0, weight=1)
        button_frame.grid_columnconfigure(1, weight=1)

        # 취소 button
        cancel_button = CTkButton(button_frame, text="취소",
                                  command=self.on_cancel,
                                  font=(self.default_font, 13, "bold"),
                                  fg_color="white", hover_color="#DDA15C",
                                  text_color="black", border_width=2,
                                  border_color="#DDA15C",
                                  width=120, height=40)
        cancel_button.grid(row=0, column=0, padx=(0, 5))

        # 보내기 button
        send_button = CTkButton(button_frame, text="보내기",
                               command=self.on_send,
                               font=(self.default_font, 13, "bold"),
                               fg_color="#FEF9E0", hover_color="#DDA15C",
                               text_color="black",
                               width=120, height=40)
        send_button.grid(row=0, column=1, padx=(5, 0))

        # Bind Escape key
        self.bind("<Escape>", lambda e: self.on_cancel())
        # Note: Removed <Return> binding to prevent accidental send when typing in textbox

        # Center the dialog
        self.update_idletasks()
        x = (self.winfo_screenwidth() // 2) - (self.winfo_width() // 2)
        y = (self.winfo_screenheight() // 2) - (self.winfo_height() // 2)
        self.geometry(f"+{x}+{y}")

    def on_cancel(self):
        """Close dialog without sending."""
        self.email_sent = False
        self.destroy()

    def on_send(self):
        """Validate and send email."""
        # Prevent duplicate sends
        if self.sending_in_progress:
            return

        # Get values
        inquiry_content = self.inquiry_text.get("1.0", "end-1c").strip()
        email = self.email_entry.get().strip()
        phone = self.phone_entry.get().strip()

        # Check if inquiry is placeholder or empty
        if inquiry_content == self.placeholder_text or not inquiry_content:
            inquiry_content = ""

        # Validate all fields are filled
        if not inquiry_content or not email or not phone:
            messagebox.showwarning(
                "입력 필요",
                "문의 내용, 이메일 주소, 휴대폰 번호를 모두 입력해주세요."
            )
            return

        # Create email body with inquiry information
        body = f"""문의 정보:

ID: {self.username}
이메일: {email}
휴대폰: {phone}
파일: {self.excel_file}

문의 내용:
{inquiry_content}
"""

        # Send email
        try:
            self.sending_in_progress = True  # Lock to prevent duplicate sends
            # Update mailer's excel_file to current file before sending
            self.mailer.excel_file = self.excel_file
            self.mailer.send_backup_email(body_text=body)
            messagebox.showinfo("전송 완료", "문의 내용이 성공적으로 전송되었습니다. 확인 후 빠른 시일 내에 답변드리겠습니다.")
            self.email_sent = True
            self.destroy()
        except Exception as e:
            self.sending_in_progress = False  # Unlock on error to allow retry
            app_logger.error(f"메일 전송 실패: {e}")
            messagebox.showerror("전송 실패", f"메일 전송 중 오류가 발생했습니다:\n{e}")
