#modules/load_autoqm_questions.py
import customtkinter
from tkinter import messagebox, Menu, filedialog
import tkinter as tk
import re
import requests
import sys
from io import BytesIO
from openpyxl import load_workbook, Workbook
from tkinter import ttk
from tkinter.filedialog import asksaveasfilename
from pathlib import Path
import os
import unicodedata
from modules.excel_cache import get_excel_file_cached



class MultiDragDropTreeview(ttk.Treeview):
    """
    Enhanced Treeview with drag & drop reordering featuring:
    - Drag handle column (≡) for initiating drag
    - Orange drop indicator line showing insertion point
    - Autoscroll when dragging near top/bottom edges
    """
    AUTOSCROLL_MARGIN = 24  # Pixels from edge to trigger autoscroll
    AUTOSCROLL_SPEED = 2    # Scroll units per tick
    AUTOSCROLL_INTERVAL = 20  # Milliseconds between scroll ticks

    def __init__(self, master, **kwargs):
        super().__init__(master, **kwargs)

        self._drag_start_item = None
        self._is_dragging = False
        self._did_reorder = False

        self._clicked_row = None
        self._initial_selection = ()
        self._drag_item_iid = None  # The item being dragged

        # Drop indicator
        self._drop_indicator = None
        self._drop_target_index = None

        # Autoscroll
        self._autoscroll_job = None
        self._autoscroll_dir = 0  # -1 = up, 0 = stop, +1 = down

        # Handle column detection
        self._handle_column_id = "handle"  # Column ID for the drag handle
        self._drag_from_handle = False  # Only drag when clicking handle

        self.bind("<Button-1>", self.on_button_press, add=True)
        self.bind("<B1-Motion>", self.on_mouse_drag, add=True)
        self.bind("<ButtonRelease-1>", self.on_button_release, add=True)

    def on_button_press(self, event):
        self._did_reorder = False
        self._is_dragging = False
        self._drag_from_handle = False

        self._clicked_row = self.identify_row(event.y)
        self._initial_selection = self.selection()

        # Check if click is on handle column
        col = self.identify_column(event.x)
        region = self.identify_region(event.x, event.y)

        # Handle column is #1 (first column after tree column)
        if col == "#1" and region == "cell" and self._clicked_row:
            self._drag_from_handle = True
            self._is_dragging = True
            self._drag_item_iid = self._clicked_row

            # Select the clicked row if not already selected
            if self._clicked_row not in self._initial_selection:
                self.selection_set(self._clicked_row)

            # Configure cursor for dragging
            try:
                self.configure(cursor="sb_v_double_arrow")
            except Exception:
                pass

            return "break"  # Prevent default selection behavior

    def on_mouse_drag(self, event):
        if not self._is_dragging or not self._drag_from_handle:
            return

        # Update autoscroll based on mouse position
        self._update_autoscroll(event.y)

        # Find row under mouse
        row_under_mouse = self.identify_row(event.y)

        # Calculate drop position (before or after the row)
        if row_under_mouse:
            bbox = self.bbox(row_under_mouse)
            if bbox:
                row_y = bbox[1]
                row_height = bbox[3]
                mid_y = row_y + row_height // 2

                if event.y < mid_y:
                    # Insert before this row
                    self._drop_target_index = self.index(row_under_mouse)
                    self._show_drop_indicator(row_under_mouse, before=True)
                else:
                    # Insert after this row
                    self._drop_target_index = self.index(row_under_mouse) + 1
                    self._show_drop_indicator(row_under_mouse, before=False)
        else:
            # Mouse is outside rows - check if above first or below last
            all_items = self.get_children()
            if all_items:
                first_item = all_items[0]
                last_item = all_items[-1]

                first_bbox = self.bbox(first_item)
                last_bbox = self.bbox(last_item)

                if first_bbox and event.y < first_bbox[1]:
                    # Above first row
                    self._drop_target_index = 0
                    self._show_drop_indicator(first_item, before=True)
                elif last_bbox and event.y > last_bbox[1] + last_bbox[3]:
                    # Below last row
                    self._drop_target_index = len(all_items)
                    self._show_drop_indicator(last_item, before=False)
                else:
                    self._hide_drop_indicator()

    def on_button_release(self, event):
        try:
            self._stop_autoscroll()

            if self._is_dragging and self._drag_from_handle and self._drag_item_iid:
                # Perform the reorder
                selected_items = self.selection()
                if selected_items and self._drop_target_index is not None:
                    self._perform_reorder(selected_items, self._drop_target_index)
        finally:
            # Clean up
            self._hide_drop_indicator()
            self._is_dragging = False
            self._drag_from_handle = False
            self._clicked_row = None
            self._initial_selection = ()
            self._drag_item_iid = None
            self._drop_target_index = None

            # Reset cursor
            try:
                self.configure(cursor="")
            except Exception:
                pass

    def _perform_reorder(self, selected_items, target_index):
        """Move selected items to the target index position."""
        if not selected_items:
            return

        # Get current indices and sort selected items by index
        items_with_indices = [(item, self.index(item)) for item in selected_items]
        items_with_indices.sort(key=lambda x: x[1])

        first_selected_idx = items_with_indices[0][1]

        # Calculate adjusted target index
        # Count how many selected items are before the target
        items_before_target = sum(1 for _, idx in items_with_indices if idx < target_index)

        # If we're moving down, target index needs adjustment
        if target_index > first_selected_idx:
            adjusted_target = target_index - items_before_target
        else:
            adjusted_target = target_index

        # Move items (from last to first to avoid index shifting issues)
        for i, (item, _) in enumerate(items_with_indices):
            self.move(item, "", adjusted_target + i)

    def _show_drop_indicator(self, item_iid, before=True):
        """Show orange drop indicator line above or below the item."""
        bbox = self.bbox(item_iid)
        if not bbox:
            return

        x, y, width, height = bbox

        # Position indicator
        if before:
            indicator_y = y
        else:
            indicator_y = y + height

        # Create or update indicator
        if self._drop_indicator is None:
            self._drop_indicator = tk.Frame(
                self,
                bg="#BB6C25",  # Orange color
                height=3
            )

        # Calculate full width of the treeview
        tree_width = self.winfo_width()

        self._drop_indicator.place(
            x=0,
            y=indicator_y,
            width=tree_width,
            height=3
        )
        self._drop_indicator.lift()

    def _hide_drop_indicator(self):
        """Hide the drop indicator line."""
        if self._drop_indicator is not None:
            self._drop_indicator.place_forget()

    def _update_autoscroll(self, mouse_y):
        """Start/stop autoscroll based on mouse position relative to treeview edges."""
        tree_height = self.winfo_height()

        if mouse_y < self.AUTOSCROLL_MARGIN:
            # Near top - scroll up
            if self._autoscroll_dir != -1:
                self._start_autoscroll(-1)
        elif mouse_y > tree_height - self.AUTOSCROLL_MARGIN:
            # Near bottom - scroll down
            if self._autoscroll_dir != 1:
                self._start_autoscroll(1)
        else:
            # Not near edges - stop scrolling
            if self._autoscroll_dir != 0:
                self._stop_autoscroll()

    def _start_autoscroll(self, direction):
        """Start autoscrolling in the given direction (-1 = up, +1 = down)."""
        self._autoscroll_dir = direction
        if self._autoscroll_job is None:
            self._do_autoscroll()

    def _stop_autoscroll(self):
        """Stop autoscrolling."""
        self._autoscroll_dir = 0
        if self._autoscroll_job is not None:
            self.after_cancel(self._autoscroll_job)
            self._autoscroll_job = None

    def _do_autoscroll(self):
        """Perform one autoscroll step and schedule the next."""
        if self._autoscroll_dir == 0:
            self._autoscroll_job = None
            return

        # Scroll the treeview
        self.yview_scroll(self._autoscroll_dir * self.AUTOSCROLL_SPEED, "units")

        # Schedule next scroll
        self._autoscroll_job = self.after(self.AUTOSCROLL_INTERVAL, self._do_autoscroll)
        
        
        
class MultiDragDropTreeview3(ttk.Treeview):
    """
    A simple Treeview that supports multi-selection drag-and-drop reordering
    of items. This will be used for the right-side tree in this example.
    """
    def __init__(self, master, **kwargs):
        super().__init__(master, **kwargs)
        self._drag_start_item = None   # item ID of the anchor item for the drag
        self._selected_items = []
        self._is_dragging = False

        self.bind("<Button-1>", self.on_button_press)
        self.bind("<B1-Motion>", self.on_mouse_drag)
        self.bind("<ButtonRelease-1>", self.on_button_release)

    def on_button_press(self, event):
        """
        Record the item(s) selected at the time of mouse press.
        """
        # Identify which row was clicked
        row_id = self.identify_row(event.y)
        if not row_id:
            return

        # Standard Treeview selection handling
        # If you want multi-select with SHIFT/CTRL, ensure selectmode="extended" is set.
        self._selected_items = self.selection()
        if not self._selected_items:
            # If nothing was selected before, select the clicked row
            self.selection_set(row_id)
            self._selected_items = (row_id,)

        # We'll consider the first selected item as the "anchor"
        self._drag_start_item = self._selected_items[0]
        self._is_dragging = True

    def on_mouse_drag(self, event):
        """
        When the mouse moves, check if we need to reorder the tree items.
        """

        if not self._is_dragging or not self._drag_start_item:
            return

        # Identify the row currently under the mouse
        row_under_mouse = self.identify_row(event.y)
        if not row_under_mouse or row_under_mouse == self._drag_start_item:
            return

        # Move each selected item. We do a naive approach here, moving them one by one.
        # Because a Treeview is hierarchical, we must specify parent="", index, etc.
        # We'll move them relative to row_under_mouse's index in the same parent.
        parent_id = self.parent(row_under_mouse)  # Usually "" for a flat list
        target_index = self.index(row_under_mouse)

        # Important: If we move multiple items at once, 
        # we typically remove them from the tree in reverse order or so to avoid reindexing issues.
        # But for simplicity, we just do them in the order of self._selected_items.
        for item_id in self._selected_items:
            self.move(item_id, parent_id, target_index)
        
        # Update anchor to the newly moved location
        self._drag_start_item = row_under_mouse

    def on_button_release(self, event):
        """
        Finish the drag-and-drop operation.
        """
        self._is_dragging = False
        self._drag_start_item = None
        self._selected_items = []

class ToolTip(object):
    """A simple tooltip class for tkinter widgets."""
    def __init__(self, widget, default_font):
        self.widget = widget
        self.tipwindow = None
        self.text = ""
        self.default_font = default_font

    def showtip(self, text, x, y):
        """Display text in tooltip window at (x, y) screen coordinates."""
        self.text = text
        if self.tipwindow or not self.text:
            return
        self.tipwindow = tw = tk.Toplevel(self.widget)
        tw.wm_overrideredirect(True)
        tw.wm_geometry("+%d+%d" % (x, y))

        if sys.platform.startswith('darwin'):
            # Mac OS
            fontsize = 18
        elif sys.platform.startswith('win'):
            # Windows
            fontsize = 13
        else:
            fontsize = 12  # fallback

        label = tk.Label(tw, text=self.text, justify="left",
                         background="#ffffe0", relief="solid", borderwidth=1, fg="black",
                         font=(self.default_font, fontsize, "normal"), wraplength=650)
        label.pack(ipadx=1, ipady=1)

    def hidetip(self):
        tw = self.tipwindow
        self.tipwindow = None
        if tw:
            tw.destroy()




# Define this as a standalone function outside of any class
def show_duplicate_dialog(parent, duplicate_tracking, default_font):
    """Show a modal dialog with duplicate information and wait until it's closed"""
    dialog = customtkinter.CTkToplevel(parent)
    dialog.title("중복 알림")
    dialog.geometry("500x500")
    dialog.resizable(True, True)
    dialog.transient(parent)  # Make dialog transient to parent
    
    # Make it modal
    dialog.grab_set()
    
    # Create UI elements
    # Title label
    label = customtkinter.CTkLabel(
        dialog,
        text="다음 항목들은 이미 존재하여 추가되지 않았습니다:",
        font=(default_font, 14, "bold"),
        text_color=("black", "white")
    )
    label.pack(pady=(20, 10), padx=20)
    
    # Frame for textbox
    frame = customtkinter.CTkFrame(dialog)
    frame.pack(fill="both", expand=True, padx=20, pady=(0, 10))
    
    # Scrollable textbox
    textbox = customtkinter.CTkTextbox(
        frame,
        font=(default_font, 12),
        wrap="word",
        text_color="black",
        height=300
    )
    textbox.pack(fill="both", expand=True, padx=5, pady=5)
    
    # Insert duplicate information
    for sheet_name, dupes in duplicate_tracking.items():
        if dupes:
            unique_dupes = sorted(set(dupes))
            textbox.insert("end", f"\n• {sheet_name}:\n")
            textbox.insert("end", f"  {', '.join(unique_dupes)}\n")
    
    # Make textbox read-only
    textbox.configure(state="disabled")
    
    # Create a flag to track dialog closure
    dialog_closed = tk.BooleanVar(value=False)
    
    # Function to handle dialog close
    def on_close():
        dialog_closed.set(True)
        dialog.grab_release()
        dialog.destroy()
    
    # OK button
    ok_button = customtkinter.CTkButton(
        dialog,
        text="확인",
        font=(default_font, 13, "bold"),
        fg_color="#FEF9E0",
        hover_color="#DDA15C",
        text_color="black",
        command=on_close
    )
    ok_button.pack(pady=(0, 20))
    
    # Set protocol for window close button
    dialog.protocol("WM_DELETE_WINDOW", on_close)
    
    # Bind escape key
    dialog.bind("<Escape>", lambda e: on_close())
    
    # Center the dialog
    dialog.update_idletasks()
    width = dialog.winfo_width()
    height = dialog.winfo_height()
    x = parent.winfo_rootx() + (parent.winfo_width() // 2) - (width // 2)
    y = parent.winfo_rooty() + (parent.winfo_height() // 2) - (height // 2)
    dialog.geometry(f"+{x}+{y}")
    
    # Make it topmost initially to ensure visibility
    dialog.attributes('-topmost', True)
    dialog.after(500, lambda: dialog.attributes('-topmost', False))
    
    # Wait for the dialog to be closed before returning
    parent.wait_variable(dialog_closed)

class LoadQuestions_변형문제(customtkinter.CTkToplevel):
    def __init__(
        self, parent,
        default_font="Arial",
        excel_file=None,
        treeview=None,
        row_to_treeview_id_map=None,
        base_url=None,
        user_token=None,
    ):
        super().__init__(parent)
        self.title("모의고사 변형문제 불러오기")
        self.geometry("800x600")
        self.minsize(width=800, height=600)        
        self.bind("<Escape>", self.on_close)

        self.parent = parent
        self.default_font = default_font
        self.excel_file = excel_file         # local Excel for "existing_set" usage
        print(f"self.excel_file: {self.excel_file}")
        if self.excel_file:
            self.file_name = os.path.basename(self.excel_file)
        else:
            self.file_name = "없음"

        self.base_url = base_url             # Flask server URL
        self.current_fetched_filename = None # track which file we fetched
        self.all_file_entries = []           # will store server data

        self.item_data_map = {}
        self.file_saved = None
        self.user_token = user_token

        # GUI layout
        main_frame = customtkinter.CTkFrame(self, fg_color="transparent")
        main_frame.pack(fill="both", expand=True, padx=10, pady=10)

        # ======================================================
        # =============== TOP LABEL FRAME  ======================
        # ======================================================
        top_label_frame = customtkinter.CTkFrame(main_frame, fg_color="transparent")
        top_label_frame.pack(fill="x", pady=(0))

        # ========== 1) label1_frame (fixed width = 200) ==========
        label1_frame = customtkinter.CTkFrame(top_label_frame, fg_color="transparent")
        label1_frame.pack(side="left", fill="y", padx=5, pady=0)
        label1_frame.pack_propagate(False)
        label1_frame.configure(width=200, height=30)

        label_font = (self.default_font, 15, "bold")
        label1 = customtkinter.CTkLabel(
            label1_frame,
            text="1. 가져올 모의고사 선택",
            fg_color="transparent",
            font=label_font,
            text_color=("black", "white"),
            anchor="w"
        )
        label1.pack(side="left", fill="x", expand=True, padx=(10, 5), pady=(0, 5))


        # ========== 2) Container for label2_frame & label3_frame ==========
        label_2_3_container = customtkinter.CTkFrame(top_label_frame, fg_color="transparent")
        label_2_3_container.pack(side="left", fill="both", expand=True)

        # label2_frame
        label2_frame = customtkinter.CTkFrame(label_2_3_container, fg_color="transparent")
        label2_frame.pack(side="left", fill="both", expand=True, padx=5, pady=0)
        label2_frame.pack_propagate(False)
        label2_frame.configure(height=30)

        label2 = customtkinter.CTkLabel(
            label2_frame,
            text="2. 가져올 문제 선택",
            fg_color="transparent",
            font=label_font,
            text_color=("black", "white"),
            anchor="w"
        )
        label2.pack(side="left", fill="x", expand=True, padx=(10, 5), pady=(0, 5))

        # label3_frame
        label3_frame = customtkinter.CTkFrame(label_2_3_container, fg_color="transparent")
        label3_frame.pack(side="left", fill="both", expand=True, padx=5, pady=0)
        label3_frame.pack_propagate(False)
        label3_frame.configure(height=30)

        self.label3 = customtkinter.CTkLabel(
            label3_frame,
            text="3. 선택된 문제 (총 0개)",
            font=label_font,
            text_color=("black", "white"),
            anchor="w"
        )
        self.label3.pack(side="left", fill="x", expand=True, padx=(10, 5), pady=(0, 5))


        self.전체삭제_button = customtkinter.CTkButton(
            master=label3_frame,
            text="전체삭제",
            font=(self.default_font, 10, 'bold'),
            width=60,      # Adjust width as desired
            height=20,     # Adjust height as desired
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=self.clear_right_tree
        )
        self.전체삭제_button.pack(side="bottom", anchor="center", expand=True)




        # ======================================================
        # =============== BOTTOM FRAME (3 columns) ============= 
        # ======================================================
        bottom_frame = customtkinter.CTkFrame(main_frame, fg_color="transparent")
        bottom_frame.pack(fill="both", expand=True)

        # ========== 1) col1 (fixed width = 200) ==========
        col1 = customtkinter.CTkFrame(bottom_frame, fg_color="transparent")
        col1.pack(side="left", fill="y", padx=5, pady=0)
        col1.pack_propagate(False)
        col1.configure(width=200, height=550)

        # top_controls_1 only
        top_controls_1 = customtkinter.CTkFrame(col1, fg_color="transparent")
        top_controls_1.pack(fill="x", pady=(0, 5))


        grade_label_left = customtkinter.CTkLabel(
            top_controls_1,
            text="학년:",
            text_color=("black", "white"),
            font=(self.default_font, 12)
        )
        grade_label_left.grid(row=1, column=0, sticky="e", padx=5)
        self.grade_combo_left = customtkinter.CTkOptionMenu(
            top_controls_1,
            values=["1학년", "2학년", "3학년"],
            font=(self.default_font, 12),
            fg_color="white",
            text_color="black",
            button_color="#DDA15C",
            button_hover_color="#BB6C25",
            command=self.on_grade_changed_left
        )
        self.grade_combo_left.set("1학년")
        self.grade_combo_left.grid(row=1, column=1, sticky="w", pady=5)

        year_label_left = customtkinter.CTkLabel(
            top_controls_1,
            text="시행년도:",
            text_color=("black", "white"),
            font=(self.default_font, 12)
        )
        year_label_left.grid(row=2, column=0, sticky="e", padx=5, pady=5)
        self.year_combo_left = customtkinter.CTkOptionMenu(
            top_controls_1,
            values=["Loading..."],
            font=(self.default_font, 12),
            fg_color="white",
            text_color="black",
            button_color="#DDA15C",
            button_hover_color="#BB6C25",
            command=self.on_year_changed_left
        )
        self.year_combo_left.grid(row=2, column=1, sticky="w")

        month_label_left = customtkinter.CTkLabel(
            top_controls_1,
            text="시행월:",
            text_color=("black", "white"),
            font=(self.default_font, 12)
        )
        month_label_left.grid(row=3, column=0, sticky="e", padx=5, pady=5)
        self.month_combo_left = customtkinter.CTkOptionMenu(
            top_controls_1,
            values=[""],
            font=(self.default_font, 12),
            fg_color="white",
            text_color="black",
            button_color="#DDA15C",
            button_hover_color="#BB6C25",
            command=self.on_month_changed_left
            
        )
        self.month_combo_left.grid(row=3, column=1, sticky="w", pady=5)

        button_frame1 = customtkinter.CTkFrame(col1, fg_color="#2B3D2D")
        button_frame1.pack(side="bottom", fill='x', padx=0, pady=0)
        fetch_button = customtkinter.CTkButton(
            button_frame1,
            text="불러오기",
            font=(self.default_font, 13, 'bold'),
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=self.on_fetch_button_click
        )
        fetch_button.pack(side="bottom", pady=10)

        # ========== 2) container for col2 & col3 ==========
        # Container for left_tree and right_tree
        col_2_3_container = customtkinter.CTkFrame(bottom_frame, fg_color="transparent")
        col_2_3_container.pack(side="left", fill="both", expand=True, padx=0, pady=0)

        # Left tree
        col2 = customtkinter.CTkFrame(col_2_3_container, fg_color="transparent")
        col2.pack(side="left", fill="both", expand=True, padx=5, pady=0)
        self.left_list_frame = customtkinter.CTkFrame(col2, fg_color="transparent")
        self.left_list_frame.pack(fill="both", expand=True)

        # 1) Configure your left_tree with two columns:
        self.left_tree = ttk.Treeview(
            self.left_list_frame,
            columns=("colA", "sourceFile"),  # 2 columns
            show="headings",
            selectmode="extended"
        )
        self.left_tree.heading("colA", text="문항")
        self.left_tree.column("colA", width=200, anchor="w")
        # We do not show heading for the second column
        # or we hide it by setting width=0
        self.left_tree.heading("sourceFile", text="sourceFile")
        self.left_tree.column("sourceFile", width=0, stretch=False)

        self.left_tree.pack(fill="both", expand=True)



        self.tooltip_var = tk.BooleanVar(value=True)
        self.tooltip_checkbox = customtkinter.CTkCheckBox(
            col2,
            text="지문 미리보기",
            variable=self.tooltip_var,
            font=(self.default_font, 12),
            text_color=("black", "white"),
            fg_color="#CA7900", hover_color="#DDA15C",
            width=20, height=20, checkbox_width=20, checkbox_height=20,
            onvalue=True,
            offvalue=False,
            command=self.update_all_tooltips
        )
        self.tooltip_checkbox.pack(side="top", pady=(2,5), fill="x", expand=False)



        copy_buttons_frame = customtkinter.CTkFrame(col2, fg_color="#2B3D2D")
        copy_buttons_frame.pack(pady=0, fill="x")
        self.copy_button = customtkinter.CTkButton(
            copy_buttons_frame,
            width=90,
            text="선택 항목 추가",
            font=(self.default_font, 13, 'bold'),
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=self.copy_selected_left_items_to_right
        )
        self.copy_button.pack(side="left", fill="x", expand=True, padx=(10, 5), pady=10)

        self.copy_all_button = customtkinter.CTkButton(
            copy_buttons_frame,
            width=90,
            text="전체 추가",
            font=(self.default_font, 13, 'bold'),
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=self.copy_all_left_items_to_right
        )
        self.copy_all_button.pack(side="right", fill="x", expand=True, padx=(5, 10), pady=10)

        # Right tree
        col3 = customtkinter.CTkFrame(col_2_3_container, fg_color="transparent")
        col3.pack(side="left", fill="both", expand=True, padx=5, pady=0)
        self.right_list_frame = customtkinter.CTkFrame(col3, fg_color="transparent")
        self.right_list_frame.pack(fill="both", expand=True)

        self.right_tree = MultiDragDropTreeview(
            self.right_list_frame,
            columns=("handle", "colA", "sourceFile"),
            show="headings",
            selectmode="extended"
        )
        # Handle column (≡) for drag initiation - first column
        self.right_tree.heading("handle", text="≡")
        self.right_tree.column("handle", width=24, minwidth=24, stretch=False, anchor="center")

        self.right_tree.heading("colA", text="문항")
        self.right_tree.column("colA", width=180, anchor="w")

        self.right_tree.heading("sourceFile", text="sourceFile")
        self.right_tree.column("sourceFile", width=0, stretch=False)

        self.right_tree.pack(fill="both", expand=True)


        # Create and bind right-click menu
        self.create_right_click_menu()
        self.right_tree.bind("<Button-3>", self.show_right_click_menu)  # Windows/Linux right-click
        self.right_tree.bind("<Button-2>", self.show_right_click_menu)  # Mac right-click

        # Bind delete and backspace keys to the right tree
        self.right_tree.bind('<Delete>', self.delete_selected_items)
        self.right_tree.bind('<BackSpace>', self.delete_selected_items)
        
        




        
        # "새로운 세트" or "기존 세트"
        self.set_generation_var = tk.StringVar(value="new")
        
        new_set_radio = customtkinter.CTkRadioButton(
            col3,
            text="새로운 파일로 저장",
            variable=self.set_generation_var,
            value="new",
            radiobutton_height=18,
            radiobutton_width=18,
            font=(self.default_font, 12),
            text_color=("black", "white"),
            fg_color="#CA7900", hover_color="#DDA15C",
        )
        new_set_radio.pack(anchor="w", pady=(5, 0))

        
        existing_set_radio = customtkinter.CTkRadioButton(
            col3,
            text="현재 파일에 추가",
            variable=self.set_generation_var,
            value="existing",
            radiobutton_height=18,
            radiobutton_width=18,
            font=(self.default_font, 12),
            text_color=("black", "white"),
            fg_color="#CA7900", hover_color="#DDA15C",
        )
        existing_set_radio.pack(anchor="w", pady=(5, 5))
        


        
        
        # The textbox
        self.existing_file_path = customtkinter.CTkTextbox(
            col3,
            font=(self.default_font, 10),
            wrap="word",
            text_color=("black", "white"),
            width=200,
            height=38,
            fg_color="transparent",
            border_width=0,
            border_spacing=0,
            corner_radius=0
        )
        self.existing_file_path.pack(fill="x", expand=False, padx=(0, 0), pady=(0))
        
        
        # Insert text (your existing code):
        self.existing_file_path.insert("1.0", f"※현재 파일: {self.file_name}")
        
        # Make it read-only (existing code)
        self.existing_file_path.configure(state="disabled")
        
        # Optionally call the adjust method once, right after inserting:
        self.adjust_textbox_height()



        """
        # Create a frame to hold the radio buttons
        radio_frame = customtkinter.CTkFrame(col3, fg_color="transparent")
        radio_frame.pack(fill="x", padx=0, pady=5)  # adjust as desired

        new_set_radio = customtkinter.CTkRadioButton(
            radio_frame,
            text="새 파일로 저장",
            variable=self.set_generation_var,
            value="new",
            radiobutton_height=18,
            radiobutton_width=18,
            font=(self.default_font, 12),
            text_color="black",
            fg_color="#CA7900",
            hover_color="#DDA15C",
        )
        new_set_radio.pack(side="left", padx=10)

        existing_set_radio = customtkinter.CTkRadioButton(
            radio_frame,
            text="현재 파일에 추가",
            variable=self.set_generation_var,
            value="existing",
            radiobutton_height=18,
            radiobutton_width=18,
            font=(self.default_font, 12),
            text_color="black",
            fg_color="#CA7900",
            hover_color="#DDA15C",
        )
        existing_set_radio.pack(side="left", padx=10)
        """




        button_frame3 = customtkinter.CTkFrame(col3, fg_color="#2B3D2D")
        button_frame3.pack(side="bottom", fill='x', padx=0, pady=0)
        self.generate_set_button = customtkinter.CTkButton(
            button_frame3,
            text="autoQM으로 가져오기",
            font=(self.default_font, 13, 'bold'),
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=self.on_generate_set_button_click
        )
        self.generate_set_button.pack(pady=10)

        self.load_year_options_from_server()


        # Setup tooltips
        self.enable_tooltips_for_trees()




    # -------------------- Methods --------------------
    def update_right_tree_count(self):
        """Update label3 to show current item count in right tree."""
        count = len(self.right_tree.get_children())
        self.label3.configure(text=f"3. 선택된 문제 (총 {count}개)")

    def delete_selected_items(self, event=None):
        """Delete selected items from the right tree and then select the next item."""
        selected_items = self.right_tree.selection()
        if not selected_items:
            return

        # 1) Sort selected items by ascending index so we can find the 'earliest' index
        selected_items_sorted = sorted(selected_items, key=lambda iid: self.right_tree.index(iid))
        first_deleted_index = self.right_tree.index(selected_items_sorted[0])

        # 2) Delete them
        for item_id in selected_items:
            self.right_tree.delete(item_id)

        # 3) Get the remaining items in the tree
        remaining_items = self.right_tree.get_children()
        if not remaining_items:
            return  # Tree is now empty; nothing to select

        # 4) Attempt to select the item at the same index we just removed
        #    If first_deleted_index is out of range (e.g., we deleted the bottom item),
        #    select the last item in the tree.
        if first_deleted_index >= len(remaining_items):
            first_deleted_index = len(remaining_items) - 1

        next_item_to_select = remaining_items[first_deleted_index]
        self.right_tree.selection_set(next_item_to_select)
        self.right_tree.focus(next_item_to_select)

        self.update_right_tree_count()

    def adjust_textbox_height(self):
        """Resize the `existing_file_path` height to match its content."""
        # Temporarily enable editing so we can read content
        self.existing_file_path.configure(state="normal")
        text_content = self.existing_file_path.get("1.0", "end-1c")
        self.existing_file_path.configure(state="disabled")

        # Number of lines
        lines = text_content.count("\n") + 1
        # Estimate line height in CTkTextbox. Adjust as needed.
        line_height_px = 18

        # You may want a max-height to prevent it from growing too large
        new_height = lines * line_height_px

        # Set the new height
        self.existing_file_path.configure(height=new_height)


    # ----------------------------------------------------------------------
            
    # Combo handling
    # 1) Simplify load_year_options_from_server to JUST fetch + store
    #    Then call update_year_options (which also sets months).
    def load_year_options_from_server(self):
        try:
            api_url = f"{self.base_url}/list_excel_files_autoqm"
            response = requests.get(api_url)
            if response.status_code == 200:
                data = response.json()
                self.all_file_entries = data.get("files", [])
            else:
                self.all_file_entries = []
        except Exception as e:
            print("Error fetching year options:", e)
            self.all_file_entries = []

        # Now that self.all_file_entries is set, update year options
        self.update_year_options()

    # ----------------------------------------------------------------------
    # 2) Whenever the grade changes, re-calculate year options (desc order),
    #    pick first year, then re-calc month options (asc order).
    def on_grade_changed_left(self, val):
        #print("on_grade_changed_left called")
        self.on_fetch_button_click()
        self.update_year_options()

    # ----------------------------------------------------------------------
    # 3) If the user picks a different year, only update months.
    def on_year_changed_left(self, val):
        #print("on_year_changed_left called")
        self.on_fetch_button_click()
        self.update_month_options()

    def on_month_changed_left(self, val):
        #print("on_month_changed_left called")
        self.on_fetch_button_click()


    # ----------------------------------------------------------------------
    # 4) Build the list of years from all_file_entries filtered by grade
    #    Sort them descending, pick the first, then call update_month_options
    def update_year_options(self):
        selected_grade = self.grade_combo_left.get()

        # Filter all_file_entries by the selected grade
        matched_entries = [
            e for e in self.all_file_entries if e["grade"] == selected_grade
        ]

        # Gather all years for this grade
        years = sorted({ e["year"] for e in matched_entries }, reverse=True)

        if not years:
            years = [""]  # fallback if none matched

        # Remember the user’s currently selected year
        old_selection = self.year_combo_left.get()

        # Update the year combo’s possible values
        self.year_combo_left.configure(values=years)

        # If the old selection is still valid, keep it;
        # otherwise, fallback to the first or empty
        if old_selection in years:
            self.year_combo_left.set(old_selection)
        else:
            if years:
                self.year_combo_left.set(years[0])
            else:
                self.year_combo_left.set("")

        # Finally, update months based on the new/maintained year
        self.update_month_options()


    # ----------------------------------------------------------------------
    # 5) Build the list of months for the current grade+year
    #    Sort them ascending, e.g. 3, 5, 10, etc., then suffix them with "월"
    def update_month_options(self):
        selected_year = self.year_combo_left.get()
        selected_grade = self.grade_combo_left.get()

        # Among all_file_entries, match grade & year
        matched_entries = [
            e for e in self.all_file_entries
            if e["year"] == selected_year and e["grade"] == selected_grade
        ]

        months = sorted({ e["month"] for e in matched_entries }, key=lambda x: int(x))  # e.g. ["3", "5", "10"]
        if not months:
            months = [""]  # fallback

        # Convert e.g. "3" to "3월"
        months_display = [f"{m}월" for m in months if m.isdigit()]

        # Remember the user’s currently selected month (like "3월")
        old_selection = self.month_combo_left.get()

        # Update the month combo’s possible values
        self.month_combo_left.configure(values=months_display)

        # If the old selection is still valid, keep it;
        # otherwise, fallback to the first or empty
        if old_selection in months_display:
            self.month_combo_left.set(old_selection)
        else:
            if months_display:
                self.month_combo_left.set(months_display[0])
            else:
                self.month_combo_left.set("")

    # ----------------------------------------------------------------------
    # Fetching data
    def on_fetch_button_click(self):
        #print("on_fetch_button_click called")
        for item in self.left_tree.get_children():
            self.left_tree.delete(item)

        self.item_data_map.clear()


        selected_year = self.year_combo_left.get()   # "2024"
        selected_month = self.month_combo_left.get() # "10월"
        selected_grade = self.grade_combo_left.get() # "1학년"

        month_str = selected_month.replace("월","").strip()
        grade_map = {"1학년": "g1", "2학년": "g2", "3학년": "g3"}
        grade_code = grade_map.get(selected_grade, "g1")
        yy_2digits = selected_year[-2:]

        filename = f"{yy_2digits}-{month_str}-{grade_code}.xlsx"
        self.fetch_data_and_add_to_left_tree(filename, "대기열")


    def fetch_data_and_add_to_left_tree(self, filename, sheet_name):
        try:
            api_url = f"{self.base_url}/get_excel_file_autoqm/{filename}"

            # Add Authorization header and query params if user is authenticated
            headers = {}
            params = {}
            if hasattr(self.parent, 'user_token') and self.parent.user_token:
                headers['Authorization'] = f'Bearer {self.parent.user_token}'
                if hasattr(self.parent, 'username'):
                    params['username'] = self.parent.username
                if hasattr(self.parent, 'user_role') and self.parent.user_role:
                    params['user_role'] = ', '.join(self.parent.user_role) if isinstance(self.parent.user_role, list) else str(self.parent.user_role)

            file_bytes, status_code = get_excel_file_cached(api_url, headers=headers, params=params)
            if status_code != 200 or file_bytes is None:
                messagebox.showerror("오류", f"서버에서 {filename} 파일을 찾을 수 없습니다.")
                return

            wb = load_workbook(file_bytes, read_only=True, data_only=True)
            if sheet_name not in wb.sheetnames:
                messagebox.showerror("오류", f"{filename}에 '{sheet_name}' 시트가 없습니다.")
                return

            sheet = wb[sheet_name]
            count = 0
            for row in sheet.iter_rows(values_only=True, min_row=1, max_col=2):
                colA_value = row[0]
                colB_value = row[1] if len(row) > 1 else None
                if colA_value:
                    # Insert item_id + sourceFile
                    self.left_tree.insert("", "end", values=(colA_value, filename))
                    # Keep your tooltip logic
                    self.item_data_map[str(colA_value)] = str(colB_value) if colB_value else ""
                    count += 1

            if count == 0:
                messagebox.showinfo("정보", "대기열 시트에 데이터가 없습니다.")

            # Track which file was *most recently* fetched
            self.current_fetched_filename = filename

            self.update_all_tooltips()

        except Exception as e:
            print("Error fetching data:", e)
            messagebox.showerror("오류", "데이터를 불러오는 중 오류가 발생했습니다.")



    # ----------------------------------------------------------------------
    # Copy selected/all from left -> right
    def copy_selected_left_items_to_right(self):
        selected_iids = self.left_tree.selection()
        if not selected_iids:
            messagebox.showinfo("정보", "위 목록에서 선택된 항목이 없습니다.")
            return

        # Gather what's in right_tree to avoid duplicates
        # Column order: (handle, colA, sourceFile) - colA is at index 1
        existing_right_items = {
            self.right_tree.item(iid, "values")[1]
            for iid in self.right_tree.get_children()
        }
        duplicates = []

        for iid in selected_iids:
            left_vals = self.left_tree.item(iid, "values")  # (colA_value, sourceFile)
            item_id, source_filename = left_vals[0], left_vals[1]

            if item_id in existing_right_items:
                duplicates.append(item_id)
            else:
                # Insert with handle column value "≡" first
                self.right_tree.insert("", "end", values=("≡", item_id, source_filename))
                existing_right_items.add(item_id)

        if duplicates:
            messagebox.showwarning("중복 알림", f"{', '.join(duplicates)}는 이미 존재하여 추가되지 않았습니다.")

        self.update_right_tree_count()

    def copy_all_left_items_to_right(self):
        left_iids = self.left_tree.get_children()
        if not left_iids:
            return

        # Column order: (handle, colA, sourceFile) - colA is at index 1
        right_item_ids = {
            self.right_tree.item(iid, "values")[1]
            for iid in self.right_tree.get_children()
        }
        duplicates = []
        for iid in left_iids:
            left_vals = self.left_tree.item(iid, "values")  # Get all values
            row_val = left_vals[0]
            source_file = left_vals[1]  # Get the source file
            if row_val in right_item_ids:
                duplicates.append(row_val)
            else:
                # Insert with handle column value "≡" first
                self.right_tree.insert("", "end", values=("≡", row_val, source_file))
                right_item_ids.add(row_val)

        if duplicates:
            messagebox.showwarning("중복 경고", f"이미 존재하는 항목: {', '.join(duplicates)}")

        self.update_right_tree_count()

    # ----------------------------------------------------------------------
    # tooltip



    def update_all_tooltips(self):
        show_tooltip = self.tooltip_var.get()
        self.show_tooltip_enabled = show_tooltip
        if not show_tooltip and getattr(self, 'left_tooltip_object', None):
            self.left_tooltip_object.hidetip()
        if not show_tooltip and getattr(self, 'right_tooltip_object', None):
            self.right_tooltip_object.hidetip()

    def enable_tooltips_for_trees(self):
        # Left tree tooltip
        self.left_tooltip_object = ToolTip(self.left_tree, self.default_font)
        self.left_tree.bind("<Motion>", self.on_tree_motion_left)
        self.left_tree.bind("<Leave>", lambda e: self.left_tooltip_object.hidetip())

        # Right tree tooltip
        self.right_tooltip_object = ToolTip(self.right_tree, self.default_font)
        self.right_tree.bind("<Motion>", self.on_tree_motion_right)
        self.right_tree.bind("<Leave>", lambda e: self.right_tooltip_object.hidetip())



    def on_tree_motion_left(self, event):
        if not getattr(self, 'show_tooltip_enabled', True):
            self.left_tooltip_object.hidetip()
            return

        row_id = self.left_tree.identify_row(event.y)
        if row_id:
            vals = self.left_tree.item(row_id, "values")
            if not vals:
                self.left_tooltip_object.hidetip()
                return
            item_id = vals[0]  # the unique ID
            if item_id in self.item_data_map:
                B_value = self.item_data_map[item_id]
                tooltip = (
                    f"{B_value}"
                )
                x = self.left_tree.winfo_rootx() + event.x + 20
                y = self.left_tree.winfo_rooty() + event.y + 20
                self.left_tooltip_object.hidetip()
                self.left_tooltip_object.showtip(tooltip, x, y)
            else:
                self.left_tooltip_object.hidetip()
        else:
            self.left_tooltip_object.hidetip()

    def on_tree_motion_right(self, event):
        if not getattr(self, 'show_tooltip_enabled', True):
            self.right_tooltip_object.hidetip()
            return

        row_id = self.right_tree.identify_row(event.y)
        if row_id:
            vals = self.right_tree.item(row_id, "values")
            if not vals or len(vals) < 2:
                self.right_tooltip_object.hidetip()
                return
            # Column order: (handle, colA, sourceFile) - colA is at index 1
            item_id = vals[1]
            if item_id in self.item_data_map:
                B_value = self.item_data_map[item_id]
                tooltip = (
                    f"{B_value}"
                )
                x = self.right_tree.winfo_rootx() + event.x + 20
                y = self.right_tree.winfo_rooty() + event.y + 20
                self.right_tooltip_object.hidetip()
                self.right_tooltip_object.showtip(tooltip, x, y)
            else:
                self.right_tooltip_object.hidetip()
        else:
            self.right_tooltip_object.hidetip()






    # ----------------------------------------------------------------------
    # Generate set logic
    def on_generate_set_button_click(self):
        right_items = self.right_tree.get_children()
        if not right_items:
            messagebox.showerror("오류", "위 '3. 선택된 문제'에 항목이 없습니다.")
            return

        # Build the EXACT user order as a list of (item_id, source_file) tuples
        # Column order: (handle, colA, sourceFile) - colA is at index 1, sourceFile at index 2
        ordered_items = []  # List of (item_id, source_file) in exact user order
        unique_source_files = set()
        for iid in right_items:
            vals = self.right_tree.item(iid, "values")
            item_id, source_file = vals[1], vals[2]
            ordered_items.append((item_id, source_file))
            unique_source_files.add(source_file)

        # Credit deduction for non-premium members (only after basic validation)
        user_role = self.parent.fetch_user_role_custom(self.user_token)
        is_allowed = self.parent.check_role_with_flask(user_role)

        if not is_allowed:
            # Non-premium member - second-stage confirmation
            result = self.parent.show_credit_confirmation_dialog(200)
            if result != 'confirm':
                return  # User cancelled or clicked upgrade

            # Check sufficient credits
            try:
                current_credits = int(self.parent.credits)
            except (ValueError, TypeError, AttributeError):
                current_credits = 0

            if current_credits < 200:
                self.parent.show_insufficient_credits_dialog(current_credits)
                return  # Abort due to insufficient credits

            # Deduct credits
            response = self.parent.deduct_credit(
                self.user_token,
                amount=200,
                task='load_modified_questions'
            )

            # Handle response carefully
            if not response or not isinstance(response, dict) or 'credits' not in response:
                messagebox.showerror("오류", "포인트 차감 중 오류가 발생했습니다.")
                return  # Abort

            if not response.get('success'):
                current_credits = response.get('credits', 0)
                self.parent.show_insufficient_credits_dialog(current_credits)
                return  # Abort

            # Update credits display
            self.parent.update_credits(response['credits'])

        # Decide if user wants new or existing
        create_new_set = (self.set_generation_var.get() == "new")

        if create_new_set:
            # Create a brand-new workbook or let user name a file
            save_path = asksaveasfilename(
                defaultextension=".xlsx",
                filetypes=[("Excel files","*.xlsx"), ("All files","*.*")],
                title="새로운 세트 저장"
            )
            if not save_path:
                return  # user canceled

            # Start an empty workbook
            from openpyxl import Workbook
            final_wb = Workbook()
            # remove default "Sheet" if present
            if "Sheet" in final_wb.sheetnames:
                del final_wb["Sheet"]

            # Download all source files and write items in EXACT user order
            self.download_and_write_in_user_order(
                ordered_items=ordered_items,
                unique_source_files=unique_source_files,
                target_wb=final_wb
            )

            final_wb.save(save_path)
            messagebox.showinfo("완료", f"새 파일이 저장되었습니다: {save_path}")

            # If you want to load into parent or log something:
            self.parent.load_file_from_server(save_path)
            self.parent.app_logger.info("문제 불러오기를 완료했습니다.")
            self.destroy()


        else:
            # 3B) If “현재 파일에 추가” (existing)
            print("existing_set")
            # existing_set

            if not self.excel_file:
                messagebox.showinfo("Info", "현재 사용중인 파일이 없습니다. 문제를 불러오기 전에 파일을 먼저 저장해야 합니다. 'OK'를 클릭하면 파일을 저장합니다.")                        

                self.new_새파일저장묻기()

                if self.file_saved == True:
                    self.excel_file = unicodedata.normalize('NFC', self.excel_file)
                    self.parent.app_logger.info(f"★[알림] 저장 성공\n(파일 경로: {self.excel_file})\n삭제/덮어쓰기 주의!\n★Make Questions를 클릭하여 출제를 시작하세요.\n")
                    self.parent.filepath_label.configure(text=f"현재 파일 경로: {self.excel_file}")
                    # Add the file to recent files list
                    self.parent.add_to_recent_files(self.excel_file)
                    self.parent.load_file_from_server(self.excel_file)
                else:
                    messagebox.showerror("오류", "파일이 저장되지 않았습니다. 현재 파일을 먼저 저장해야 합니다.")
                    return

            # Load existing file
            from openpyxl import load_workbook
            final_wb = load_workbook(self.excel_file)

            # Download all source files and write items in EXACT user order
            self.download_and_write_in_user_order(
                ordered_items=ordered_items,
                unique_source_files=unique_source_files,
                target_wb=final_wb
            )

            final_wb.save(self.excel_file)
            # Refresh parent
            self.parent.load_excel_into_treeview(self.excel_file, "대기열")
            self.parent.update_sheet_selector_realtime(self.excel_file, "대기열")
            self.parent.app_logger.info("문제 불러오기를 완료했습니다.")
            self.destroy()


    def delete_unwanted_rows_in_local_file(self, file_path, item_ids_to_keep):
        wb_local = load_workbook(file_path)
        for sheet_name in wb_local.sheetnames:
            ws = wb_local[sheet_name]
            for row_idx in range(ws.max_row, 0, -1):
                val = ws.cell(row=row_idx, column=1).value
                if val not in item_ids_to_keep:
                    ws.delete_rows(row_idx, 1)
        wb_local.save(file_path)


    """
    def append_sheets_to_main_excel(self, source_file, target_file):
        source_wb = load_workbook(source_file)
        target_wb = load_workbook(target_file)
        for sheet_name in source_wb.sheetnames:
            source_ws = source_wb[sheet_name]
            if sheet_name in target_wb.sheetnames:
                target_ws = target_wb[sheet_name]
                for row in source_ws.iter_rows(values_only=True):
                    target_ws.append(row)
            else:
                target_ws = target_wb.create_sheet(sheet_name)
                for r_idx, row in enumerate(source_ws.iter_rows(values_only=True), start=1):
                    for c_idx, cell_val in enumerate(row, start=1):
                        target_ws.cell(row=r_idx, column=c_idx, value=cell_val)
        target_wb.save(target_file)
        """ 

    def append_sheets_to_main_excel(self, source_file, target_file):
        source_wb = load_workbook(source_file)
        target_wb = load_workbook(target_file)

        duplicates = []

        # ---------------------------------------------------
        # 1) Handle "대기열" sheet (Check duplicates in col A)
        # ---------------------------------------------------
        if "대기열" in source_wb.sheetnames:
            source_ws = source_wb["대기열"]

            # If "대기열" doesn't exist in the target, create it and copy everything.
            if "대기열" not in target_wb.sheetnames:
                target_ws = target_wb.create_sheet("대기열")
                # No duplicates check needed (brand-new sheet)
                for r_idx, row in enumerate(source_ws.iter_rows(values_only=True), start=1):
                    for c_idx, cell_val in enumerate(row, start=1):
                        target_ws.cell(row=r_idx, column=c_idx, value=cell_val)
            else:
                # "대기열" exists in target → gather existing col A into a set
                target_ws = target_wb["대기열"]
                existing_colA = set()

                for row in target_ws.iter_rows(values_only=True, min_row=1, max_col=1):
                    if row[0] is not None:
                        existing_colA.add(str(row[0]))

                # Check each row in the source for duplicates based on col A
                for row in source_ws.iter_rows(values_only=True):
                    if not row:
                        continue
                    colA_val = row[0]
                    if colA_val is None or colA_val == "":
                        continue

                    colA_str = str(colA_val)
                    if colA_str not in existing_colA:
                        # Not a duplicate → append & update the set
                        target_ws.append(row)
                        existing_colA.add(colA_str)
                    else:
                        # It's a duplicate → collect in our duplicates list
                        duplicates.append(colA_str)
        else:
            print("'대기열' sheet not found in source, skipping duplicate check.")

        # ---------------------------------------------------
        # 2) Copy (or append) the OTHER sheets (no duplicate check)
        # ---------------------------------------------------
        for sheet_name in source_wb.sheetnames:
            if sheet_name == "대기열":
                # Already handled above; skip.
                continue

            source_ws = source_wb[sheet_name]
            if sheet_name not in target_wb.sheetnames:
                # Create the sheet in target (full copy)
                target_ws = target_wb.create_sheet(sheet_name)
                for r_idx, row in enumerate(source_ws.iter_rows(values_only=True), start=1):
                    for c_idx, cell_val in enumerate(row, start=1):
                        target_ws.cell(row=r_idx, column=c_idx, value=cell_val)
            else:
                # Sheet exists in target → do some "append" strategy
                target_ws = target_wb[sheet_name]
                for row in source_ws.iter_rows(values_only=True):
                    target_ws.append(row)

        # ---------------------------------------------------
        # 3) If duplicates were found, show one warning message
        # ---------------------------------------------------
        if duplicates:
            # Optionally convert to set to remove repeated duplicates
            unique_duplicates = list(set(duplicates))
            messagebox.showwarning(
                "중복 알림",
                f"{', '.join(unique_duplicates)}는 이미 존재하여 추가되지 않았습니다."
            )

        # Finally, save the changes
        target_wb.save(target_file)


    def update_sheet_selector(self, excel_file):
        wb = load_workbook(filename=excel_file, read_only=True)
        sheet_names = wb.sheetnames
        self.parent.sheet_selector.configure(values=sheet_names)






    def new_새파일저장묻기(self, event=None):
        
        if self.excel_file is None:
            file_path = filedialog.asksaveasfilename(defaultextension=".xlsx", filetypes=[("Excel files", "*.xlsx"), ("All files", "*.*")])                
            
            if not file_path:  # If the user cancels the save dialog
                self.parent.app_logger.info(f"저장이 취소되었습니다.\n")
                return  # Do not proceed further

            # Create a new Excel workbook and add a default sheet
            self.file_saved = True
            wb = Workbook()
            ws = wb.active
            ws.title = "대기열"
            # Save the new workbook to the chosen file path
            wb.save(filename=file_path)
            self.excel_file = file_path  # Update the global excel_file variable
            

            sheet_name = "대기열"  
            print("프린트 self.save_treeview_to_excel:" + str(sheet_name))
            self.parent.save_treeview_to_excel(self.excel_file, sheet_name)
            self.parent.update_sheet_selector(self.excel_file)




    ######## 우클릭 메뉴
    def create_right_click_menu(self):
        """Create right-click context menu for the right tree."""
        self.right_click_menu = Menu(self, tearoff=0)
        self.right_click_menu.add_command(label="위로 이동", command=self.move_items_up)
        self.right_click_menu.add_command(label="아래로 이동", command=self.move_items_down)

    def show_right_click_menu(self, event):
        """Show the context menu on right-click."""
        if self.right_tree.selection():  # Only show menu if items are selected
            self.right_click_menu.tk_popup(event.x_root, event.y_root)

    def move_items_up(self):
        """Move selected items up one position."""
        selected_items = self.right_tree.selection()
        if not selected_items:
            return

        for item in selected_items:
            idx = self.right_tree.index(item)
            if idx > 0:  # Can't move up if already at top
                self.right_tree.move(item, "", idx - 1)

    def move_items_down(self):
        """Move selected items down one position."""
        selected_items = self.right_tree.selection()
        if not selected_items:
            return

        # Process items in reverse order when moving down
        for item in reversed(selected_items):
            idx = self.right_tree.index(item)
            last_idx = len(self.right_tree.get_children()) - 1
            if idx < last_idx:  # Can't move down if already at bottom
                self.right_tree.move(item, "", idx + 1)


    def on_close(self, event=None):
        self.destroy()

    def clear_right_tree(self):
        """Delete all items from self.right_tree."""
        for item_id in self.right_tree.get_children():
            self.right_tree.delete(item_id)
        self.update_right_tree_count()

    def get_filename_from_item_id(item_id: str) -> str:
        """
        Given an item ID like '23-03-고2-18', return the
        corresponding file name, e.g. '23-03-g2.xlsx'.
        We assume the item_id is always of the form YY-MM-고X-XX,
        where X is 1,2,3, etc.
        """
        # Example: item_id = "23-03-고2-18"
        # Split by '-': ["23", "03", "고2", "18"]
        parts = item_id.split("-")
        if len(parts) < 3:
            return None
        
        # parts[0] = "23"
        # parts[1] = "03"
        # parts[2] = "고2"

        # Replace '고' -> 'g'
        grade_part = parts[2].replace("고", "g")
        
        # Construct the filename
        filename = f"{parts[0]}-{parts[1]}-{grade_part}.xlsx"  
        # e.g. "23-03-g2.xlsx"
        return filename

    def download_and_write_in_user_order(self, ordered_items, unique_source_files, target_wb):
        """
        Download all source files, then write items to target_wb in EXACT user order.
        This preserves the interleaved order when items come from multiple source files.

        Args:
            ordered_items: List of (item_id, source_file) tuples in exact user order
            unique_source_files: Set of unique source file names
            target_wb: Target openpyxl Workbook to write to
        """
        # Step 1: Download all source files and build lookup tables
        # Structure: {source_file -> {sheet_name -> {item_id -> row_tuple}}}
        file_data_cache = {}

        # Add Authorization header and query params if user is authenticated
        headers = {}
        params = {}
        if hasattr(self.parent, 'user_token') and self.parent.user_token:
            headers['Authorization'] = f'Bearer {self.parent.user_token}'
            if hasattr(self.parent, 'username'):
                params['username'] = self.parent.username
            if hasattr(self.parent, 'user_role') and self.parent.user_role:
                params['user_role'] = ', '.join(self.parent.user_role) if isinstance(self.parent.user_role, list) else str(self.parent.user_role)

        for source_file in unique_source_files:
            download_url = f"{self.base_url}/get_excel_file_autoqm/{source_file}"
            file_bytes, status_code = get_excel_file_cached(download_url, headers=headers, params=params)

            if status_code != 200 or file_bytes is None:
                messagebox.showerror("오류", f"{source_file} 다운로드 실패.")
                continue

            # Log usage
            if self.user_token:
                usage_url = "https://frontierenglish.net/wp-json/myapp/v1/store-file-usage"
                log_headers = {
                    "Authorization": f"Bearer {self.user_token}",
                    "Content-Type": "application/json"
                }
                data = {
                    "filename": source_file,
                    "type": "변형문제",
                    "price": "14,000",
                }
                try:
                    usage_resp = requests.post(usage_url, headers=log_headers, json=data)
                    usage_resp.raise_for_status()
                except requests.RequestException:
                    pass

            # Load the workbook and build lookup table
            source_wb = load_workbook(file_bytes)
            file_data_cache[source_file] = {}

            for sheet_name in source_wb.sheetnames:
                ws = source_wb[sheet_name]
                file_data_cache[source_file][sheet_name] = {}
                for row in ws.iter_rows(values_only=True):
                    if row and row[0] is not None:
                        item_id = str(row[0])
                        file_data_cache[source_file][sheet_name][item_id] = row

        # Step 2: Collect existing item IDs from target workbook for duplicate checking
        existing_items_by_sheet = {}
        for sheet_name in target_wb.sheetnames:
            ws = target_wb[sheet_name]
            existing_items_by_sheet[sheet_name] = set()
            for row_idx in range(1, ws.max_row + 1):
                val = ws.cell(row=row_idx, column=1).value
                if val is not None:
                    existing_items_by_sheet[sheet_name].add(str(val))

        # Step 3: Write items in EXACT user order
        duplicate_tracking = {}
        items_written_by_sheet = {}  # Track next write row for each sheet

        for item_id, source_file in ordered_items:
            if source_file not in file_data_cache:
                continue

            # Write this item's data to each sheet that has it
            for sheet_name, sheet_data in file_data_cache[source_file].items():
                if item_id not in sheet_data:
                    continue

                row_data = sheet_data[item_id]

                # Create sheet if it doesn't exist
                if sheet_name not in target_wb.sheetnames:
                    target_wb.create_sheet(sheet_name)
                    existing_items_by_sheet[sheet_name] = set()
                    items_written_by_sheet[sheet_name] = 1
                elif sheet_name not in items_written_by_sheet:
                    # Find the next empty row
                    ws = target_wb[sheet_name]
                    next_row = 1
                    while ws.cell(row=next_row, column=1).value is not None:
                        next_row += 1
                    items_written_by_sheet[sheet_name] = next_row

                # Check for duplicates
                if item_id in existing_items_by_sheet[sheet_name]:
                    if sheet_name not in duplicate_tracking:
                        duplicate_tracking[sheet_name] = []
                    duplicate_tracking[sheet_name].append(item_id)
                    continue

                # Write the row
                target_ws = target_wb[sheet_name]
                next_row = items_written_by_sheet[sheet_name]
                for col_idx, cell_val in enumerate(row_data, start=1):
                    target_ws.cell(row=next_row, column=col_idx, value=cell_val)

                existing_items_by_sheet[sheet_name].add(item_id)
                items_written_by_sheet[sheet_name] = next_row + 1

        # Show duplicate warning if any
        if duplicate_tracking:
            show_duplicate_dialog(self, duplicate_tracking, self.default_font)

    def download_and_append_source_file(self, source_file, item_ids_to_keep, target_wb):
        """
        1) Download `source_file` from server
        2) Log usage
        3) Delete rows not in item_ids_to_keep
        4) Append to target_wb with duplicate-check in '대기열'
        """
        download_url = f"{self.base_url}/get_excel_file_autoqm/{source_file}"

        # Add Authorization header and query params if user is authenticated
        headers = {}
        params = {}
        if hasattr(self.parent, 'user_token') and self.parent.user_token:
            headers['Authorization'] = f'Bearer {self.parent.user_token}'
            if hasattr(self.parent, 'username'):
                params['username'] = self.parent.username
            if hasattr(self.parent, 'user_role') and self.parent.user_role:
                params['user_role'] = ', '.join(self.parent.user_role) if isinstance(self.parent.user_role, list) else str(self.parent.user_role)

        file_bytes, status_code = get_excel_file_cached(download_url, headers=headers, params=params)
        if status_code != 200 or file_bytes is None:
            messagebox.showerror("오류", f"{source_file} 다운로드 실패.")
            return

        # 2) usage log
        if self.user_token:
            usage_url = "https://frontierenglish.net/wp-json/myapp/v1/store-file-usage"
            log_headers = {
                "Authorization": f"Bearer {self.user_token}",
                "Content-Type": "application/json"
            }
            data = {
                "filename": source_file,
                "type": "변형문제",
                "price": "14,000",
            }
            try:
                usage_resp = requests.post(usage_url, headers=log_headers, json=data)
                usage_resp.raise_for_status()
            except requests.RequestException as e:
                pass

        # 3) Load the downloaded file
        temp_file = file_bytes
        source_wb = load_workbook(temp_file)

        # Build a map of item_id -> row_data for each sheet (to preserve user's order)
        # item_ids_to_keep is now an ordered list, not a set
        sheet_row_data = {}  # sheet_name -> {item_id -> row_tuple}
        for sheet_name in source_wb.sheetnames:
            ws = source_wb[sheet_name]
            sheet_row_data[sheet_name] = {}
            for row in ws.iter_rows(values_only=True):
                if row and row[0] is not None:
                    item_id = str(row[0])
                    if item_id in item_ids_to_keep or row[0] in item_ids_to_keep:
                        # Store with string key for consistent lookup
                        sheet_row_data[sheet_name][item_id] = row

        # 4) Append to target workbook with 대기열 duplicate-check, preserving order
        self.append_sheets_with_daegiyeol_dupe_check_ordered(source_wb, target_wb, item_ids_to_keep, sheet_row_data)


    def append_sheets_with_daegiyeol_dupe_check(self, source_wb, target_wb):
        duplicate_tracking = {}  # Will track duplicates per sheet
        
        # ---------------------------------------------------------
        # Process each sheet independently with its own duplicate checking
        # ---------------------------------------------------------
        for sheet_name in source_wb.sheetnames:
            source_ws = source_wb[sheet_name]
            sheet_duplicates = []
            
            if sheet_name not in target_wb.sheetnames:
                # Create a new sheet in target if it doesn't exist
                target_ws = target_wb.create_sheet(sheet_name)
                next_row = 1
                existing_colA = set()  # Empty since this is a new sheet
            else:
                # Sheet exists - collect existing column A values
                target_ws = target_wb[sheet_name]
                existing_colA = set()
                
                # Find the first empty row and collect existing values
                next_row = 1
                while target_ws.cell(row=next_row, column=1).value is not None:
                    col_a_value = target_ws.cell(row=next_row, column=1).value
                    existing_colA.add(str(col_a_value))
                    next_row += 1
            
            # Now process rows from source
            source_row_count = 0
            accepted_row_count = 0
            
            for row in source_ws.iter_rows(values_only=True):
                source_row_count += 1
                if not row:
                    continue
                    
                colA_val = row[0]
                if colA_val is None or colA_val == "":
                    continue

                colA_str = str(colA_val)
                
                if colA_str not in existing_colA:
                    # Not a duplicate in this sheet - add it
                    for col_idx, cell_val in enumerate(row, start=1):
                        target_ws.cell(row=next_row, column=col_idx, value=cell_val)
                    next_row += 1
                    existing_colA.add(colA_str)
                    accepted_row_count += 1
                else:
                    # It's a duplicate in this sheet
                    sheet_duplicates.append(colA_str)
            
            # Track duplicates per sheet
            if sheet_duplicates:
                duplicate_tracking[sheet_name] = sheet_duplicates
        
        # ---------------------------------------------------------
        # Show appropriate warnings about duplicates
        # ---------------------------------------------------------
        if duplicate_tracking:
            # Create a global dialog instance that isn't tied to self
            # and wait for it to be closed before continuing
            show_duplicate_dialog(self, duplicate_tracking, self.default_font)


    def append_sheets_with_daegiyeol_dupe_check_ordered(self, source_wb, target_wb, ordered_item_ids, sheet_row_data):
        """
        Append rows to target workbook in the order specified by ordered_item_ids.
        This preserves the user's drag-and-drop order from the right tree.
        """
        duplicate_tracking = {}  # Will track duplicates per sheet

        # ---------------------------------------------------------
        # Process each sheet independently with its own duplicate checking
        # ---------------------------------------------------------
        for sheet_name in source_wb.sheetnames:
            sheet_duplicates = []
            row_data_map = sheet_row_data.get(sheet_name, {})

            if not row_data_map:
                continue  # No rows to add for this sheet

            if sheet_name not in target_wb.sheetnames:
                # Create a new sheet in target if it doesn't exist
                target_ws = target_wb.create_sheet(sheet_name)
                next_row = 1
                existing_colA = set()  # Empty since this is a new sheet
            else:
                # Sheet exists - collect existing column A values
                target_ws = target_wb[sheet_name]
                existing_colA = set()

                # Find the first empty row and collect existing values
                next_row = 1
                while target_ws.cell(row=next_row, column=1).value is not None:
                    col_a_value = target_ws.cell(row=next_row, column=1).value
                    existing_colA.add(str(col_a_value))
                    next_row += 1

            # Now process rows in the ORDER specified by ordered_item_ids
            for item_id in ordered_item_ids:
                item_id_str = str(item_id)

                if item_id_str not in row_data_map:
                    continue  # This item doesn't exist in this sheet

                row = row_data_map[item_id_str]

                if item_id_str not in existing_colA:
                    # Not a duplicate in this sheet - add it
                    for col_idx, cell_val in enumerate(row, start=1):
                        target_ws.cell(row=next_row, column=col_idx, value=cell_val)
                    next_row += 1
                    existing_colA.add(item_id_str)
                else:
                    # It's a duplicate in this sheet
                    sheet_duplicates.append(item_id_str)

            # Track duplicates per sheet
            if sheet_duplicates:
                duplicate_tracking[sheet_name] = sheet_duplicates

        # ---------------------------------------------------------
        # Show appropriate warnings about duplicates
        # ---------------------------------------------------------
        if duplicate_tracking:
            show_duplicate_dialog(self, duplicate_tracking, self.default_font)
