#modules/load_questions3.py

import customtkinter
from tkinter import messagebox, Menu, filedialog
import tkinter as tk
import re
import requests
import sys
from io import BytesIO
from openpyxl import load_workbook, Workbook
from tkinter import ttk
import unicodedata
from tkinter.filedialog import asksaveasfilename
from modules.excel_cache import get_excel_file_cached



class MultiDragDropTreeview(ttk.Treeview):
    """
    Enhanced Treeview with drag & drop reordering featuring:
    - Drag handle column (≡) for initiating drag
    - Orange drop indicator line showing insertion point
    - Autoscroll when dragging near top/bottom edges
    """
    AUTOSCROLL_MARGIN = 24  # Pixels from edge to trigger autoscroll
    AUTOSCROLL_SPEED = 2    # Scroll units per tick
    AUTOSCROLL_INTERVAL = 20  # Milliseconds between scroll ticks

    def __init__(self, master, **kwargs):
        super().__init__(master, **kwargs)

        self._drag_start_item = None
        self._is_dragging = False
        self._did_reorder = False

        self._clicked_row = None
        self._initial_selection = ()
        self._drag_item_iid = None  # The item being dragged

        # Drop indicator
        self._drop_indicator = None
        self._drop_target_index = None

        # Autoscroll
        self._autoscroll_job = None
        self._autoscroll_dir = 0  # -1 = up, 0 = stop, +1 = down

        # Handle column detection
        self._handle_column_id = "handle"  # Column ID for the drag handle
        self._drag_from_handle = False  # Only drag when clicking handle

        self.bind("<Button-1>", self.on_button_press, add=True)
        self.bind("<B1-Motion>", self.on_mouse_drag, add=True)
        self.bind("<ButtonRelease-1>", self.on_button_release, add=True)

    def on_button_press(self, event):
        self._did_reorder = False
        self._is_dragging = False
        self._drag_from_handle = False

        self._clicked_row = self.identify_row(event.y)
        self._initial_selection = self.selection()

        # Check if click is on handle column
        col = self.identify_column(event.x)
        region = self.identify_region(event.x, event.y)

        # Handle column is #1 (first column after tree column)
        if col == "#1" and region == "cell" and self._clicked_row:
            self._drag_from_handle = True
            self._is_dragging = True
            self._drag_item_iid = self._clicked_row

            # Select the clicked row if not already selected
            if self._clicked_row not in self._initial_selection:
                self.selection_set(self._clicked_row)

            # Configure cursor for dragging
            try:
                self.configure(cursor="sb_v_double_arrow")
            except Exception:
                pass

            return "break"  # Prevent default selection behavior

    def on_mouse_drag(self, event):
        if not self._is_dragging or not self._drag_from_handle:
            return

        # Update autoscroll based on mouse position
        self._update_autoscroll(event.y)

        # Find row under mouse
        row_under_mouse = self.identify_row(event.y)

        # Calculate drop position (before or after the row)
        if row_under_mouse:
            bbox = self.bbox(row_under_mouse)
            if bbox:
                row_y = bbox[1]
                row_height = bbox[3]
                mid_y = row_y + row_height // 2

                if event.y < mid_y:
                    # Insert before this row
                    self._drop_target_index = self.index(row_under_mouse)
                    self._show_drop_indicator(row_under_mouse, before=True)
                else:
                    # Insert after this row
                    self._drop_target_index = self.index(row_under_mouse) + 1
                    self._show_drop_indicator(row_under_mouse, before=False)
        else:
            # Mouse is outside rows - check if above first or below last
            all_items = self.get_children()
            if all_items:
                first_item = all_items[0]
                last_item = all_items[-1]

                first_bbox = self.bbox(first_item)
                last_bbox = self.bbox(last_item)

                if first_bbox and event.y < first_bbox[1]:
                    # Above first row
                    self._drop_target_index = 0
                    self._show_drop_indicator(first_item, before=True)
                elif last_bbox and event.y > last_bbox[1] + last_bbox[3]:
                    # Below last row
                    self._drop_target_index = len(all_items)
                    self._show_drop_indicator(last_item, before=False)
                else:
                    self._hide_drop_indicator()

    def on_button_release(self, event):
        try:
            self._stop_autoscroll()

            if self._is_dragging and self._drag_from_handle and self._drag_item_iid:
                # Perform the reorder
                selected_items = self.selection()
                if selected_items and self._drop_target_index is not None:
                    self._perform_reorder(selected_items, self._drop_target_index)
        finally:
            # Clean up
            self._hide_drop_indicator()
            self._is_dragging = False
            self._drag_from_handle = False
            self._clicked_row = None
            self._initial_selection = ()
            self._drag_item_iid = None
            self._drop_target_index = None

            # Reset cursor
            try:
                self.configure(cursor="")
            except Exception:
                pass

    def _perform_reorder(self, selected_items, target_index):
        """Move selected items to the target index position."""
        if not selected_items:
            return

        # Get current indices and sort selected items by index
        items_with_indices = [(item, self.index(item)) for item in selected_items]
        items_with_indices.sort(key=lambda x: x[1])

        first_selected_idx = items_with_indices[0][1]

        # Calculate adjusted target index
        # Count how many selected items are before the target
        items_before_target = sum(1 for _, idx in items_with_indices if idx < target_index)

        # If we're moving down, target index needs adjustment
        if target_index > first_selected_idx:
            adjusted_target = target_index - items_before_target
        else:
            adjusted_target = target_index

        # Move items (from last to first to avoid index shifting issues)
        for i, (item, _) in enumerate(items_with_indices):
            self.move(item, "", adjusted_target + i)

    def _show_drop_indicator(self, item_iid, before=True):
        """Show orange drop indicator line above or below the item."""
        bbox = self.bbox(item_iid)
        if not bbox:
            return

        x, y, width, height = bbox

        # Position indicator
        if before:
            indicator_y = y
        else:
            indicator_y = y + height

        # Create or update indicator
        if self._drop_indicator is None:
            self._drop_indicator = tk.Frame(
                self,
                bg="#BB6C25",  # Orange color
                height=3
            )

        # Calculate full width of the treeview
        tree_width = self.winfo_width()

        self._drop_indicator.place(
            x=0,
            y=indicator_y,
            width=tree_width,
            height=3
        )
        self._drop_indicator.lift()

    def _hide_drop_indicator(self):
        """Hide the drop indicator line."""
        if self._drop_indicator is not None:
            self._drop_indicator.place_forget()

    def _update_autoscroll(self, mouse_y):
        """Start/stop autoscroll based on mouse position relative to treeview edges."""
        tree_height = self.winfo_height()

        if mouse_y < self.AUTOSCROLL_MARGIN:
            # Near top - scroll up
            if self._autoscroll_dir != -1:
                self._start_autoscroll(-1)
        elif mouse_y > tree_height - self.AUTOSCROLL_MARGIN:
            # Near bottom - scroll down
            if self._autoscroll_dir != 1:
                self._start_autoscroll(1)
        else:
            # Not near edges - stop scrolling
            if self._autoscroll_dir != 0:
                self._stop_autoscroll()

    def _start_autoscroll(self, direction):
        """Start autoscrolling in the given direction (-1 = up, +1 = down)."""
        self._autoscroll_dir = direction
        if self._autoscroll_job is None:
            self._do_autoscroll()

    def _stop_autoscroll(self):
        """Stop autoscrolling."""
        self._autoscroll_dir = 0
        if self._autoscroll_job is not None:
            self.after_cancel(self._autoscroll_job)
            self._autoscroll_job = None

    def _do_autoscroll(self):
        """Perform one autoscroll step and schedule the next."""
        if self._autoscroll_dir == 0:
            self._autoscroll_job = None
            return

        # Scroll the treeview
        self.yview_scroll(self._autoscroll_dir * self.AUTOSCROLL_SPEED, "units")

        # Schedule next scroll
        self._autoscroll_job = self.after(self.AUTOSCROLL_INTERVAL, self._do_autoscroll)


class ToolTip(object):
    """A simple tooltip class for tkinter widgets."""
    def __init__(self, widget, default_font):
        self.widget = widget
        self.tipwindow = None
        self.text = ""
        self.default_font = default_font

    def showtip(self, text, x, y):
        """Display text in tooltip window at (x, y) screen coordinates."""
        self.text = text
        if self.tipwindow or not self.text:
            return
        self.tipwindow = tw = tk.Toplevel(self.widget)
        tw.wm_overrideredirect(True)
        tw.wm_geometry("+%d+%d" % (x, y))

        if sys.platform.startswith('darwin'):
            # Mac OS
            fontsize = 18
        elif sys.platform.startswith('win'):
            # Windows
            fontsize = 13
        else:
            fontsize = 12  # fallback

        label = tk.Label(tw, text=self.text, justify="left",
                         background="#ffffe0", relief="solid", borderwidth=1, fg="black",
                         font=(self.default_font, fontsize, "normal"), wraplength=650)
        label.pack(ipadx=1, ipady=1)

    def hidetip(self):
        tw = self.tipwindow
        self.tipwindow = None
        if tw:
            tw.destroy()

class DragDropDialogTK(customtkinter.CTkToplevel):
    def __init__(
        self, parent,
        default_font="Arial",
        excel_file=None,
        treeview=None,
        row_to_treeview_id_map=None,
        base_url=None,
        user_token=None,
        #save_treeview_to_excel=None,
        #sheet_selector=None,
        #load_excel_into_treeview=None
    ):
        super().__init__(parent)
        self.title("모의고사 원본문제 불러오기")
        self.geometry("800x600")
        self.minsize(width=800, height=600)        
        self.bind("<Escape>", self.on_close)

        customtkinter.set_appearance_mode("System")
        customtkinter.set_default_color_theme("blue")

        # Store parameters
        self.parent = parent
        self.default_font = default_font
        self.excel_file = excel_file
        self.treeview = treeview
        self.row_to_treeview_id_map = row_to_treeview_id_map
        self.base_url = base_url
        self.user_token = user_token
        #self.save_treeview_to_excel = save_treeview_to_excel
        #self.sheet_selector = sheet_selector
        #self.parent.load_excel_into_treeview = load_excel_into_treeview
        self.file_saved = None
        
        self.left_anchor = None #shift+up/down 선택용
        self.right_anchor = None

        
        self.reference_items = set()
        self.item_data_map = {}
        self.original_order = {}  # to store an integer rank for each item for default ordering

        label_font = (self.default_font, 15, "bold")

        # ================= Main frame =================
        main_frame = customtkinter.CTkFrame(self, fg_color="transparent")
        main_frame.pack(fill="both", expand=True, padx=10, pady=10)

        # ======================================================
        # =============== TOP LABEL FRAME  ======================
        # ======================================================
        top_label_frame = customtkinter.CTkFrame(main_frame, fg_color="transparent")
        top_label_frame.pack(fill="x", pady=(0))

        # ========== 1) label1_frame (fixed width = 200) ==========
        label1_frame = customtkinter.CTkFrame(top_label_frame, fg_color="transparent")
        label1_frame.pack(side="left", fill="y", padx=5, pady=0)
        label1_frame.pack_propagate(False)
        label1_frame.configure(width=200, height=30)

        label1 = customtkinter.CTkLabel(
            label1_frame,
            text="1. 가져올 모의고사 선택",
            fg_color="transparent",
            font=label_font,
            text_color=("black", "white"),
            anchor="w"
        )
        label1.pack(side="left", fill="x", expand=True, padx=(10, 5), pady=0)

        # ========== 2) Container for label2_frame & label3_frame ==========
        label_2_3_container = customtkinter.CTkFrame(top_label_frame, fg_color="transparent")
        label_2_3_container.pack(side="left", fill="both", expand=True)

        # label2_frame
        label2_frame = customtkinter.CTkFrame(label_2_3_container, fg_color="transparent")
        label2_frame.pack(side="left", fill="both", expand=True, padx=5, pady=0)
        label2_frame.pack_propagate(False)
        label2_frame.configure(height=30)

        label2 = customtkinter.CTkLabel(
            label2_frame,
            text="2. 가져올 문제 선택",
            fg_color="transparent",
            font=label_font,
            text_color=("black", "white"),
            anchor="w"
        )
        label2.pack(side="left", fill="x", expand=True, padx=(10, 5), pady=0)

        # label3_frame
        label3_frame = customtkinter.CTkFrame(label_2_3_container, fg_color="transparent")
        label3_frame.pack(side="left", fill="both", expand=True, padx=5, pady=0)
        label3_frame.pack_propagate(False)
        label3_frame.configure(height=30)

        self.label3 = customtkinter.CTkLabel(
            label3_frame,
            text="3. 선택된 문제 (총 0개)",
            font=label_font,
            text_color=("black", "white"),
            anchor="w"
        )
        self.label3.pack(side="left", fill="x", expand=True, padx=(10, 5), pady=0)

        # ======================================================
        # =============== BOTTOM FRAME (3 columns) ============= 
        # ======================================================
        bottom_frame = customtkinter.CTkFrame(main_frame, fg_color="transparent")
        bottom_frame.pack(fill="both", expand=True)

        # ========== 1) col1 (fixed width = 200) ==========
        col1 = customtkinter.CTkFrame(bottom_frame, fg_color="transparent")
        col1.pack(side="left", fill="y", padx=5, pady=0)
        col1.pack_propagate(False)
        col1.configure(width=200, height=550)

        # --- Controls in col1 ---
        top_controls_1 = customtkinter.CTkFrame(col1, fg_color="transparent")
        top_controls_1.pack(fill="x", pady=(15, 5))

        self.left_radio_var = tk.BooleanVar(value=True)

        self.left_radio_button = customtkinter.CTkRadioButton(
            top_controls_1,
            text="모의고사 통째로 불러오기",
            variable=self.left_radio_var,
            value=True,
            radiobutton_height=18,
            radiobutton_width=18,
            font=(self.default_font, 13),
            text_color=("black", "white"),
            fg_color="#CA7900", hover_color="#DDA15C",
            command=self.on_radio_toggle
        )
        self.left_radio_button.grid(row=0, column=0, columnspan=2, sticky="w", pady=(0, 5))
        grade_label_left = customtkinter.CTkLabel(
            top_controls_1,
            text="학년:",
            text_color=("black", "white"),
            font=(self.default_font, 12)
        )
        grade_label_left.grid(row=1, column=0, sticky="e", padx=5)

        self.grade_combo_left = customtkinter.CTkOptionMenu(
            top_controls_1,
            values=["1학년", "2학년", "3학년"],
            font=(self.default_font, 12),
            fg_color="white",
            text_color="black",
            button_color="#DDA15C",
            button_hover_color="#BB6C25",
            command=lambda val: self.on_grade_changed_left(val)
        )
        self.grade_combo_left.set("1학년")
        self.grade_combo_left.grid(row=1, column=1, sticky="w", pady=5)

        year_label_left = customtkinter.CTkLabel(
            top_controls_1,
            text="시행년도:",
            text_color=("black", "white"),
            font=(self.default_font, 12)
        )
        year_label_left.grid(row=2, column=0, sticky="e", padx=5, pady=5)

        self.year_combo_left = customtkinter.CTkOptionMenu(
            top_controls_1,
            values=["Loading..."],
            font=(self.default_font, 12),
            fg_color="white",
            text_color="black",
            button_color="#DDA15C",
            button_hover_color="#BB6C25",
            command=lambda val: self.on_year_changed_left(val)
        )
        self.year_combo_left.grid(row=2, column=1, sticky="w")

        month_label_left = customtkinter.CTkLabel(
            top_controls_1,
            text="시행월:",
            text_color=("black", "white"),
            font=(self.default_font, 12)
        )
        month_label_left.grid(row=3, column=0, sticky="e", padx=5, pady=5)

        self.month_combo_left = customtkinter.CTkOptionMenu(
            top_controls_1,
            values=[""],
            font=(self.default_font, 12),
            fg_color="white",
            text_color="black",
            button_color="#DDA15C",
            button_hover_color="#BB6C25",
            command=lambda val: self.on_month_changed_left(val)
        )
        self.month_combo_left.grid(row=3, column=1, sticky="w", pady=5)

        top_controls_2 = customtkinter.CTkFrame(col1, fg_color="transparent")
        top_controls_2.pack(fill="x", pady=(15, 5))

        self.right_radio_button = customtkinter.CTkRadioButton(
            top_controls_2,
            text="문제 유형별로 불러오기",
            variable=self.left_radio_var,
            value=False,
            radiobutton_height=18,
            radiobutton_width=18,
            font=(self.default_font, 13),
            text_color=("black", "white"),
            fg_color="#CA7900", hover_color="#DDA15C",
            command=self.on_radio_toggle
        )
        self.right_radio_button.grid(row=0, column=0, columnspan=2, sticky="w", pady=5)

        grade_label_right = customtkinter.CTkLabel(
            top_controls_2,
            text="학년:",
            text_color=("black", "white"),
            font=(self.default_font, 12)
        )
        grade_label_right.grid(row=1, column=0, sticky="e", padx=5, pady=5)

        self.grade_combo_right = customtkinter.CTkOptionMenu(
            top_controls_2,
            values=["1학년", "2학년", "3학년"],
            font=(self.default_font, 12),
            fg_color="white",
            text_color="black",
            button_color="#DDA15C",
            button_hover_color="#BB6C25",
            command=lambda value: self.on_optionmenu_change()
        )
        self.grade_combo_right.set("1학년")
        self.grade_combo_right.grid(row=1, column=1, sticky="w", pady=5)

        year_label_right = customtkinter.CTkLabel(
            top_controls_2,
            text="시행년도:",
            text_color=("black", "white"),
            font=(self.default_font, 12)
        )
        year_label_right.grid(row=2, column=0, sticky="e", padx=5, pady=5)

        self.year_combo_right = customtkinter.CTkOptionMenu(
            top_controls_2,
            values=["Loading..."],
            font=(self.default_font, 12),
            fg_color="white",
            text_color="black",
            button_color="#DDA15C",
            button_hover_color="#BB6C25",
            command=lambda value: self.on_optionmenu_change()
        )
        self.year_combo_right.grid(row=2, column=1, sticky="w", pady=5)

        type_label = customtkinter.CTkLabel(
            top_controls_2,
            text="유형:",
            text_color=("black", "white"),
            font=(self.default_font, 12)
        )
        type_label.grid(row=3, column=0, sticky="e", padx=5, pady=5)

        self.type_combo_right = customtkinter.CTkOptionMenu(
            top_controls_2,
            values=[
                "목적", "심경", "주장", "함축의미", "요지", "주제", "제목", "도표", "일치", "안내문",
                "어법", "어휘", "빈칸추론", "무관한 문장", "순서", "삽입",
                "요약", "장문독해1", "장문독해2"
            ],
            font=(self.default_font, 12),
            fg_color="white",
            text_color="black",
            button_color="#DDA15C",
            button_hover_color="#BB6C25",
            command=lambda value: self.on_optionmenu_change()
        )
        self.type_combo_right.set("목적")
        self.type_combo_right.grid(row=3, column=1, sticky="w", pady=5)

        button_frame1 = customtkinter.CTkFrame(col1, fg_color="#2B3D2D")
        button_frame1.pack(side="bottom", fill='x', padx=0, pady=0)

        fetch_button = customtkinter.CTkButton(
            button_frame1,
            text="불러오기",
            font=(self.default_font, 13, 'bold'),
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=self.on_fetch_button_click
        )
        fetch_button.pack(side="bottom", pady=10)

        # ========== 2) container for col2 & col3 ==========
        col_2_3_container = customtkinter.CTkFrame(bottom_frame, fg_color="transparent")
        col_2_3_container.pack(side="left", fill="both", expand=True, padx=0, pady=0)

        # col2: left treeview
        col2 = customtkinter.CTkFrame(col_2_3_container, fg_color="transparent")
        col2.pack(side="left", fill="both", expand=True, padx=5, pady=0)
        col2.pack_propagate(False)
        col2.configure(height=550)

        self.left_list_frame = customtkinter.CTkFrame(col2, fg_color="transparent")
        self.left_list_frame.pack(fill="both", expand=True, padx=0, pady=(0))


        arrow_button_frame1 = customtkinter.CTkFrame(self.left_list_frame, fg_color="transparent")
        arrow_button_frame1.pack(side="top", fill="x", padx=(0, 5), pady=(0))

        # Up/Down buttons to reorder the LEFT tree by 2nd column (정답률)
        self.up_button_left = customtkinter.CTkButton(
            arrow_button_frame1,
            text="⬆︎",
            font=(self.default_font, 10, 'bold'),
            width=20,
            height=20,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=lambda: self.reorder_tree(self.left_tree, "col3", ascending=True)
        )
        self.up_button_left.pack(side="right", padx=0, pady=0)

        self.down_button_left = customtkinter.CTkButton(
            arrow_button_frame1,
            text="⬇︎",
            font=(self.default_font, 10, 'bold'),
            width=20,
            height=20,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=lambda: self.reorder_tree(self.left_tree, "col3", ascending=False)
        )
        self.down_button_left.pack(side="right", padx=0, pady=0)

        self.정답률정렬1_label = customtkinter.CTkLabel(
            arrow_button_frame1,
            text="정답률순",
            font=(self.default_font, 10, 'bold'),
            text_color=("black", "white"),
        )
        self.정답률정렬1_label.pack(side="right", padx=(0, 3), pady=0)


        style = ttk.Style()
        style.theme_use("clam")


        # Create a frame to hold the tree & scrollbar
        left_tree_container = customtkinter.CTkFrame(self.left_list_frame, fg_color="transparent")
        left_tree_container.pack(fill="both", expand=True)

        # Create the scrollbar
        left_tree_scrollbar = ttk.Scrollbar(left_tree_container, orient="vertical")

        # Create the tree (same as before, except parent is left_tree_container)
        self.left_tree = ttk.Treeview(
            left_tree_container,
            columns=("col1", "col2", "col3"),
            show="headings",
            selectmode="extended",
            yscrollcommand=left_tree_scrollbar.set  # link the tree to scrollbar
        )
        self.left_tree.heading("col1", text="문제번호")
        self.left_tree.heading("col2", text="유형")
        self.left_tree.heading("col3", text="정답률")
        self.left_tree.column("col1", width=100, anchor="w")
        self.left_tree.column("col2", width=50, anchor="w")
        self.left_tree.column("col3", width=50, anchor="center")

        self.left_tree.bind('<Configure>', self.update_column_widths)

        # Pack the tree on the left, scrollbar on the right
        self.left_tree.pack(side="left", fill="both", expand=True)
        left_tree_scrollbar.pack(side="right", fill="y")

        # Configure the scrollbar’s command
        left_tree_scrollbar.config(command=self.left_tree.yview)




        self.tooltip_var = tk.BooleanVar(value=True)
        self.tooltip_checkbox = customtkinter.CTkCheckBox(
            col2,
            text="지문 미리보기",
            variable=self.tooltip_var,
            font=(self.default_font, 12),
            text_color=("black", "white"),
            fg_color="#CA7900", hover_color="#DDA15C",
            width=20, height=20, checkbox_width=20, checkbox_height=20,
            onvalue=True,
            offvalue=False,
            command=self.update_all_tooltips
        )
        self.tooltip_checkbox.pack(side="top", pady=(2,5), fill="x", expand=False)

        copy_buttons_frame = customtkinter.CTkFrame(col2, fg_color="#2B3D2D")
        copy_buttons_frame.pack(pady=0, fill="x")

        self.copy_button = customtkinter.CTkButton(
            copy_buttons_frame,
            width=90,
            text="선택 항목 추가",
            font=(self.default_font, 13, 'bold'),
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=self.copy_selected_left_items_to_right
        )
        self.copy_button.pack(side="left", fill="x", expand=True, padx=(10, 5), pady=10)

        self.copy_all_button = customtkinter.CTkButton(
            copy_buttons_frame,
            width=90,
            text="전체 추가",
            font=(self.default_font, 13, 'bold'),
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=self.copy_all_left_items_to_right
        )
        self.copy_all_button.pack(side="right", fill="x", expand=True, padx=(5, 10), pady=10)

        # col3: right treeview (MultiDragDropTreeview)
        col3 = customtkinter.CTkFrame(col_2_3_container, fg_color="transparent")
        col3.pack(side="left", fill="both", expand=True, padx=5, pady=0)
        col3.pack_propagate(False)
        col3.configure(height=550)

        self.right_list_frame = customtkinter.CTkFrame(col3, fg_color="transparent")
        self.right_list_frame.pack(fill="both", expand=True, padx=0, pady=0)

        arrow_button_frame2 = customtkinter.CTkFrame(self.right_list_frame, fg_color="transparent")
        arrow_button_frame2.pack(side="top", fill="x", padx=(0, 5), pady=(0))



        self.번호정렬_label = customtkinter.CTkLabel(
            arrow_button_frame2,
            text="번호순",
            font=(self.default_font, 10, 'bold'),
            text_color=("black", "white"),
        )
        self.번호정렬_label.pack(side="left", padx=(5, 3), pady=0)


        self.번호down_button_left = customtkinter.CTkButton(
            arrow_button_frame2,
            text="⬇︎",
            font=(self.default_font, 10, 'bold'),
            width=20,
            height=20,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=lambda: self.reorder_tree(self.right_tree, "col1", ascending=False)
        )
        self.번호down_button_left.pack(side="left", padx=0, pady=0)

        
        self.번호up_button_left = customtkinter.CTkButton(
            arrow_button_frame2,
            text="⬆︎",
            font=(self.default_font, 10, 'bold'),
            width=20,
            height=20,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=lambda: self.reorder_tree(self.right_tree, "col1", ascending=True)
        )
        self.번호up_button_left.pack(side="left", padx=0, pady=0)


        # Up/Down buttons to reorder the LEFT tree by 2nd column (정답률)
        self.정답률up_button_left = customtkinter.CTkButton(
            arrow_button_frame2,
            text="⬆︎",
            font=(self.default_font, 10, 'bold'),
            width=20,
            height=20,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=lambda: self.reorder_tree(self.right_tree, "col3", ascending=True)
        )
        self.정답률up_button_left.pack(side="right", padx=0, pady=0)

        self.정답률down_button_left = customtkinter.CTkButton(
            arrow_button_frame2,
            text="⬇︎",
            font=(self.default_font, 10, 'bold'),
            width=20,
            height=20,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=lambda: self.reorder_tree(self.right_tree, "col3", ascending=False)
        )
        self.정답률down_button_left.pack(side="right", padx=(0), pady=0)


        self.정답률정렬2_label = customtkinter.CTkLabel(
            arrow_button_frame2,
            text="정답률순",
            font=(self.default_font, 10, 'bold'),
            text_color=("black", "white"),
        )
        self.정답률정렬2_label.pack(side="right", padx=(0, 3), pady=0)



        # Create a center frame that will expand between left and right items
        center_frame = customtkinter.CTkFrame(arrow_button_frame2, fg_color="transparent")
        center_frame.pack(side="left", expand=True, fill="x")

        self.전체삭제_button = customtkinter.CTkButton(
            master=center_frame,
            text="전체삭제",
            font=(self.default_font, 10, 'bold'),
            width=60,      # Adjust width as desired
            height=20,     # Adjust height as desired
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=self.clear_right_tree
        )
        self.전체삭제_button.pack(side="top", anchor="center", expand=True)



        right_tree_container = customtkinter.CTkFrame(self.right_list_frame, fg_color="transparent")
        right_tree_container.pack(fill="both", expand=True)

        right_tree_scrollbar = ttk.Scrollbar(right_tree_container, orient="vertical")

        self.right_tree = MultiDragDropTreeview(
            right_tree_container,
            columns=("handle", "col1", "col2", "col3"),
            show="headings",
            selectmode="extended",
        )
        self.right_tree.configure(yscrollcommand=right_tree_scrollbar.set)

        self.right_tree.heading("handle", text="≡")
        self.right_tree.column("handle", width=24, minwidth=24, stretch=False, anchor="center")
        self.right_tree.heading("col1", text="문제번호")
        self.right_tree.heading("col2", text="유형")
        self.right_tree.heading("col3", text="정답률")
        self.right_tree.column("col1", width=150, anchor="w")
        self.right_tree.column("col2", width=50, anchor="w")
        self.right_tree.column("col3", width=50, anchor="center")

        self.right_tree.bind('<Configure>', self.update_column_widths)

        self.right_tree.pack(side="left", fill="both", expand=True)
        right_tree_scrollbar.pack(side="right", fill="y")

        right_tree_scrollbar.config(command=self.right_tree.yview)



        # Create and bind right-click menu
        self.create_right_click_menu()
        self.right_tree.bind("<Button-3>", self.show_right_click_menu)  # Windows/Linux right-click
        self.right_tree.bind("<Button-2>", self.show_right_click_menu)  # Mac right-click


        # Bind delete and backspace keys to the right tree
        self.right_tree.bind('<Delete>', self.delete_selected_items)
        self.right_tree.bind('<BackSpace>', self.delete_selected_items)



        self.set_generation_var = tk.StringVar(value="new")
        new_set_radio = customtkinter.CTkRadioButton(
            col3,
            text="새로운 세트 생성",
            variable=self.set_generation_var,
            value="new",
            radiobutton_height=18,
            radiobutton_width=18,
            font=(self.default_font, 12),
            text_color=("black", "white"),
            fg_color="#CA7900", hover_color="#DDA15C",
            command=self.on_set_generation_toggle
        )
        new_set_radio.pack(anchor="w", pady=(5, 0))

        existing_set_frame = customtkinter.CTkFrame(col3, fg_color="transparent")
        existing_set_frame.pack(fill="x", pady=0)
        
        existing_set_radio = customtkinter.CTkRadioButton(
            existing_set_frame,
            text="기존 세트에 추가",
            variable=self.set_generation_var,
            value="existing",
            radiobutton_height=18,
            radiobutton_width=18,
            font=(self.default_font, 12),
            text_color=("black", "white"),
            fg_color="#CA7900", hover_color="#DDA15C",
            command=self.on_set_generation_toggle
        )
        existing_set_radio.pack(side="left")

        self.existing_set_combo = customtkinter.CTkOptionMenu(
            existing_set_frame,
            values=["기존 세트 없음"],
            font=(self.default_font, 12),
            fg_color="white",
            text_color="black",
            button_color="#DDA15C",
            button_hover_color="#BB6C25",
            command=self.기존세트에추가체크
        )
        self.existing_set_combo.pack(side="left", fill="x", expand=True, padx=5, pady=5)

        button_frame3 = customtkinter.CTkFrame(col3, fg_color="#2B3D2D")
        button_frame3.pack(side="bottom", fill='x', padx=0, pady=0)

        self.generate_set_button = customtkinter.CTkButton(
            button_frame3,
            text="autoQM으로 가져오기",
            font=(self.default_font, 13, 'bold'),
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            command=self.on_generate_set_button_click
        )
        self.generate_set_button.pack(pady=10)

        # ========== Initialize UI (placeholders) ==========
        self.load_year_options_from_server()
        self.load_year_options_for_right()
        self.on_radio_toggle()
        self.on_set_generation_toggle()

        existing_sets = self.get_existing_sets()
        if existing_sets:
            self.existing_set_combo.configure(values=existing_sets)
            self.existing_set_combo.set(existing_sets[0])
        else:
            self.existing_set_combo.configure(values=["기존 세트 없음"])
            self.existing_set_combo.set("기존 세트 없음")

        # Setup tooltips
        self.enable_tooltips_for_trees()

        self.left_tree.bind("<Control-a>", self.select_all_left_tree)
        self.left_tree.bind("<Command-a>", self.select_all_left_tree)

        self.right_tree.bind("<Control-a>", self.select_all_right_tree)
        self.right_tree.bind("<Command-a>", self.select_all_right_tree)




        self.left_tree.bind("<Shift-Up>", self.shift_select_up_left)
        self.left_tree.bind("<Shift-Down>", self.shift_select_down_left)

        
        self.right_tree.bind("<Shift-Up>", self.shift_select_up_right)
        self.right_tree.bind("<Shift-Down>", self.shift_select_down_right)




    # -------------------- Methods --------------------

    def update_right_tree_count(self):
        """Update label3 to show current item count in right tree."""
        count = len(self.right_tree.get_children())
        self.label3.configure(text=f"3. 선택된 문제 (총 {count}개)")

    def shift_select_up_left(self, event):

        current_selection = self.left_tree.selection()
        if current_selection:
            index = self.left_tree.index(current_selection[0])
            if index > 0:
                prev_item = self.left_tree.get_children()[index - 1]
                self.left_tree.selection_add(prev_item)
                self.left_tree.see(prev_item)
        return "break"  # Prevents default handling

    def shift_select_down_left(self, event):
        current_selection = self.left_tree.selection()
        if current_selection:
            index = self.left_tree.index(current_selection[-1])
            children = self.left_tree.get_children()
            if index < len(children) - 1:
                next_item = children[index + 1]
                self.left_tree.selection_add(next_item)
                self.left_tree.see(next_item)
        return "break"  # Prevents default handling


    def shift_select_up_right(self, event):

        current_selection = self.right_tree.selection()
        if current_selection:
            index = self.right_tree.index(current_selection[0])
            if index > 0:
                prev_item = self.right_tree.get_children()[index - 1]
                self.right_tree.selection_add(prev_item)
                self.right_tree.see(prev_item)
        return "break"  # Prevents default handling

    def shift_select_down_right(self, event):
        current_selection = self.right_tree.selection()
        if current_selection:
            index = self.right_tree.index(current_selection[-1])
            children = self.right_tree.get_children()
            if index < len(children) - 1:
                next_item = children[index + 1]
                self.right_tree.selection_add(next_item)
                self.right_tree.see(next_item)
        return "break"  # Prevents default handling









    def delete_selected_items(self, event=None):
        """Delete selected items from the right tree, then select the next item."""
        selected_items = self.right_tree.selection()
        if not selected_items:
            return
        
        # 1) Get the index of the first selected item
        #    We'll use the lowest index, so we know where to move the selection afterward.
        indices = [self.right_tree.index(iid) for iid in selected_items]
        if not indices:
            return
        lowest_index = min(indices)

        # 2) Delete the items
        for item_id in selected_items:
            self.right_tree.delete(item_id)

        # 3) Figure out which item to select next.  
        #    We'll try to select the item at 'lowest_index' (the "next" item after deletion).
        #    But if that index is out of range (e.g., deleted the last item), 
        #    then we pick the last item in the tree.
        all_items = self.right_tree.get_children()
        if not all_items:  # If the tree is now empty, nothing to select
            return
        if lowest_index >= len(all_items):
            lowest_index = len(all_items) - 1  # Adjust if we deleted the last item

        # 4) Select the new item
        item_to_select = all_items[lowest_index]
        self.right_tree.selection_set(item_to_select)
        self.right_tree.focus(item_to_select)  # Optional: move focus, too

        self.update_right_tree_count()

    def 기존세트에추가체크(self, event=None):
        self.set_generation_var.set("existing")

    def update_column_widths(self, event=None):
        total_width = self.left_tree.winfo_width()
        col1_width = int(total_width * (2/4))
        col2_width = int(total_width * (1/4))
        col3_width = int(total_width * (1/4))
        self.left_tree.column("col1", width=col1_width)
        self.left_tree.column("col2", width=col2_width)
        self.left_tree.column("col3", width=col3_width)
        self.right_tree.column("col1", width=col1_width)
        self.right_tree.column("col2", width=col2_width)
        self.right_tree.column("col3", width=col3_width)


    def select_all_left_tree(self, event=None):
        self.left_tree.selection_remove(self.left_tree.selection())
        self.left_tree.selection_set(self.left_tree.get_children())
        return "break"

    def select_all_right_tree(self, event=None):
        self.right_tree.selection_remove(self.right_tree.selection())
        self.right_tree.selection_set(self.right_tree.get_children())
        return "break"
    

    def reorder_tree(self, tree, col, ascending=True):
        """
        Reorder the Treeview by the chosen column.
        If col == "col3", empty cells always go to the bottom
        whether ascending or descending.
        """

        items = tree.get_children("")
        valid_rows = []  # rows that have a valid value in col3
        empty_rows = []  # rows whose col3 is empty

        for iid in items:
            row_values = tree.item(iid, "values")
            if not row_values:
                continue

            col1, col2, col3 = row_values[0], row_values[1], row_values[2]

            # We only treat col3 specially
            if col == "col3":
                # If col3 is empty or just whitespace
                if not col3.strip():
                    empty_rows.append(iid)
                else:
                    try:
                        # remove '%' if it exists and convert to float
                        val = float(col3.replace('%', ''))
                    except ValueError:
                        val = 0.0
                    valid_rows.append((iid, val))
            else:
                # Fall back to sorting by col1 in this example
                valid_rows.append((iid, col1))

        # Sort only valid rows
        # (For col1 you might need a different float/string logic)
        valid_rows.sort(key=lambda x: x[1], reverse=not ascending)

        # Merge valid rows + empty rows (empty always at the bottom)
        sorted_items = valid_rows + [(iid, None) for iid in empty_rows]

        # Re-insert them into the Treeview
        for index, (iid, _) in enumerate(sorted_items):
            tree.move(iid, "", index)

    def copy_selected_left_items_to_right(self):
        """
        Copies selected rows from the left tree to the right tree,
        skipping duplicates.
        """
        selected_iids = self.left_tree.selection()
        if not selected_iids:
            messagebox.showinfo("정보", "위 목록에서 선택된 항목이 없습니다.")
            return

        # Gather the existing item IDs in the right tree
        right_item_ids = set()
        for iid in self.right_tree.get_children():
            val = self.right_tree.item(iid, "values")
            if val:
                right_item_ids.add(val[1])  # val[0] is handle, val[1] is the unique ID

        duplicates = []
        for iid in selected_iids:
            row_vals = self.left_tree.item(iid, "values")
            item_id = row_vals[0]
            if item_id in right_item_ids:
                duplicates.append(item_id)
            else:
                self.right_tree.insert("", "end", values=("≡",) + tuple(row_vals))
                right_item_ids.add(item_id)

        if duplicates:
            messagebox.showwarning("중복 알림", f"이미 존재하여 추가되지 않은 항목: {', '.join(duplicates)}")

        self.update_right_tree_count()

    def copy_all_left_items_to_right(self):
        """
        Copies all rows from the left tree to the right tree,
        skipping duplicates.
        """
        left_iids = self.left_tree.get_children()
        if not left_iids:
            return

        right_item_ids = set()
        for iid in self.right_tree.get_children():
            val = self.right_tree.item(iid, "values")
            if val:
                right_item_ids.add(val[1])  # val[0] is handle, val[1] is the unique ID

        duplicates = []
        for iid in left_iids:
            row_vals = self.left_tree.item(iid, "values")
            item_id = row_vals[0]
            if item_id in right_item_ids:
                duplicates.append(item_id)
            else:
                self.right_tree.insert("", "end", values=("≡",) + tuple(row_vals))
                right_item_ids.add(item_id)

        if duplicates:
            messagebox.showwarning("중복 알림", f"이미 존재하여 추가되지 않은 항목: {', '.join(duplicates)}")

        self.update_right_tree_count()

    def on_set_generation_toggle(self):
        # Enable/disable combo depending on radio
        if self.set_generation_var.get() == "new":
            # creating new set
            if not self.get_existing_sets():
                self.existing_set_combo.configure(values=["기존 세트 없음"])
                self.existing_set_combo.set("기존 세트 없음")
        else:
            # existing sets scenario
            existing_sets = self.get_existing_sets()
            if existing_sets:
                self.existing_set_combo.configure(values=existing_sets)
                self.existing_set_combo.set(existing_sets[0])
            else:
                self.existing_set_combo.configure(values=["기존 세트 없음"])
                self.existing_set_combo.set("기존 세트 없음")

    def on_radio_toggle(self):
        left_checked = self.left_radio_var.get()
        # Enable/disable left combos
        if left_checked:
            self.year_combo_left.configure(state="normal")
            self.month_combo_left.configure(state="normal")
            self.grade_combo_left.configure(state="normal")
            self.year_combo_right.configure(state="disabled")
            self.type_combo_right.configure(state="disabled")
            self.grade_combo_right.configure(state="disabled")
        else:
            self.year_combo_left.configure(state="disabled")
            self.month_combo_left.configure(state="disabled")
            self.grade_combo_left.configure(state="disabled")
            self.year_combo_right.configure(state="normal")
            self.type_combo_right.configure(state="normal")
            self.grade_combo_right.configure(state="normal")


    def load_year_options_for_right(self):
        try:
            api_url = f"{self.base_url}/list_excel_files"
            response = requests.get(api_url)
            if response.status_code == 200:
                data = response.json()
                years = data.get("years", [])
                if not years:
                    years = ["2024", "2023"]
            else:
                years = ["2024", "2023"]
        except Exception:
            years = ["2024", "2023"]

        self.year_combo_right.configure(values=years)
        if years:
            self.year_combo_right.set(years[0])

    def load_year_options_from_server(self):
        try:
            api_url = f"{self.base_url}/list_excel_files"
            response = requests.get(api_url)
            if response.status_code == 200:
                data = response.json()
                years = data.get("years", [])
                if not years:
                    years = ["2024", "2023"]
            else:
                years = ["2024", "2023"]
        except Exception:
            years = ["2024", "2023"]

        self.year_combo_left.configure(values=years)
        if years:
            self.year_combo_left.set(years[0])
            self.update_month_options(self.year_combo_left.get(), self.get_current_grade_left())

    def on_year_changed_left(self, new_year):
        self.update_month_options(new_year, self.get_current_grade_left())
        self.on_fetch_button_click()

    def on_grade_changed_left(self, new_grade):
        self.update_month_options(self.year_combo_left.get(), self.get_current_grade_left())
        self.on_fetch_button_click()

    def on_month_changed_left(self, new_grade):
        self.on_fetch_button_click()
        
    def on_optionmenu_change(self):
        self.on_fetch_button_click()




    def get_current_grade_left(self):
        grade_map = {"1학년": "고1", "2학년": "고2", "3학년": "고3"}
        return grade_map.get(self.grade_combo_left.get(), "고1")

    def get_current_grade_right(self):
        grade_map = {"1학년": "고1", "2학년": "고2", "3학년": "고3"}
        return grade_map.get(self.grade_combo_right.get(), "고1")

    def update_month_options(self, year_str, grade_str):
        if not year_str:
            self.month_combo_left.configure(values=[""])
            self.month_combo_left.set("")
            return

        try:
            api_url = f"{self.base_url}/get_excel_file/{year_str}.xlsx"

            # Add Authorization header and query params if user is authenticated
            headers = {}
            params = {}
            if hasattr(self.parent, 'user_token') and self.parent.user_token:
                headers['Authorization'] = f'Bearer {self.parent.user_token}'
                if hasattr(self.parent, 'username'):
                    params['username'] = self.parent.username
                if hasattr(self.parent, 'user_role') and self.parent.user_role:
                    params['user_role'] = ', '.join(self.parent.user_role) if isinstance(self.parent.user_role, list) else str(self.parent.user_role)

            file_bytes, status_code = get_excel_file_cached(api_url, headers=headers, params=params)
            if status_code != 200 or file_bytes is None:
                self.month_combo_left.configure(values=[""])
                self.month_combo_left.set("")
                return

            wb = load_workbook(file_bytes, read_only=True, data_only=True)
            sheet_name = "대기열"
            if sheet_name not in wb.sheetnames:
                self.month_combo_left.configure(values=[""])
                self.month_combo_left.set("")
                return
            sheet = wb[sheet_name]

            year_two_digits = year_str[-2:]
            matched_months = set()

            for row in sheet.iter_rows(values_only=True, min_row=1, max_col=1):
                cell_value = row[0]
                if cell_value and isinstance(cell_value, str):
                    parts = cell_value.split('-')
                    if len(parts) >= 3:
                        if parts[0] == year_two_digits and parts[2] == grade_str:
                            mm = parts[1]
                            if mm.isdigit() and len(mm) == 2:
                                matched_months.add(mm)

            sorted_months = sorted(matched_months, key=lambda x: int(x))
            month_values = [f"{int(m)}월" for m in sorted_months]

            if not month_values:
                month_values = [""]

            self.month_combo_left.configure(values=month_values)
            self.month_combo_left.set(month_values[0] if month_values else "")
        except Exception:
            self.month_combo_left.configure(values=[""])
            self.month_combo_left.set("")

    def on_fetch_button_click(self):
        # Clear the left_tree first
        for item in self.left_tree.get_children():
            self.left_tree.delete(item)
        self.reference_items.clear()
        self.item_data_map.clear()
        self.original_order.clear()

        if self.left_radio_var.get():
            # Left scenario (모의고사 통째로)
            selected_year = self.year_combo_left.get()
            selected_month = self.month_combo_left.get()
            selected_grade = self.grade_combo_left.get()
            grade_map = {"1학년": "고1", "2학년": "고2", "3학년": "고3"}
            year_str = selected_year[-2:]
            month_str = selected_month.replace("월","").strip()
            if month_str.isdigit():
                month_str = f"{int(month_str):02d}"
            else:
                messagebox.showerror("오류", "유효한 월을 선택하세요.")
                return
            grade_str = grade_map.get(selected_grade, "고1")
            prefix = f"{year_str}-{month_str}-{grade_str}"
            self.fetch_data_and_add_to_left_tree(selected_year, "대기열", prefix, filter_type=None)
        else:
            # Right scenario (문제 유형별)
            selected_year = self.year_combo_right.get()
            selected_grade = self.grade_combo_right.get()
            selected_type = self.type_combo_right.get()
            grade_map = {"1학년": "고1", "2학년": "고2", "3학년": "고3"}
            year_str = selected_year[-2:]
            grade_str = grade_map.get(selected_grade, "고1")
            self.fetch_data_and_add_to_left_tree(selected_year, "대기열", None,
                                                filter_type=selected_type,
                                                year_str=year_str,
                                                grade_str=grade_str)

    def fetch_data_and_add_to_left_tree(self, excel_file, sheet_name, prefix, filter_type=None, year_str=None, grade_str=None):
        try:
            #print("[DEBUG] Starting fetch_data_and_add_to_left_tree")
            #print(f"[DEBUG] excel_file={excel_file}, sheet_name={sheet_name}, prefix={prefix}, filter_type={filter_type}, year_str={year_str}, grade_str={grade_str}")
            
            api_url = f"{self.base_url}/get_excel_file/{excel_file}.xlsx"
            #print(f"[DEBUG] Fetching data from URL: {api_url}")

            # Add Authorization header and query params if user is authenticated
            headers = {}
            params = {}
            if hasattr(self.parent, 'user_token') and self.parent.user_token:
                headers['Authorization'] = f'Bearer {self.parent.user_token}'
                if hasattr(self.parent, 'username'):
                    params['username'] = self.parent.username
                if hasattr(self.parent, 'user_role') and self.parent.user_role:
                    params['user_role'] = ', '.join(self.parent.user_role) if isinstance(self.parent.user_role, list) else str(self.parent.user_role)

            file_bytes, status_code = get_excel_file_cached(api_url, headers=headers, params=params)
            if status_code != 200 or file_bytes is None:
                messagebox.showerror("오류", "서버에 접근할 수 없습니다.")
                return

            wb = load_workbook(file_bytes, read_only=True, data_only=True)
            if sheet_name not in wb.sheetnames:
                messagebox.showerror("오류", "요청한 시트가 존재하지 않습니다.")
                return
            sheet = wb[sheet_name]

            matched_rows = 0
            duplicates = []
            existing_values = set()
            order_counter = 1

            pattern = None
            if prefix is None and year_str and grade_str:
                # e.g. r"^23-\d{2}-고2-\d+$"
                #pattern = f"^{year_str}-\\d{{2}}-{grade_str}-\\d+$"
                pattern = f"^{year_str}-\\d{{2}}-{grade_str}-\\d+(~\\d+)?$"

                #print(f"[DEBUG] Using regex pattern: {pattern}")

            #print("[DEBUG] Beginning iteration over rows...")
            for idx, row in enumerate(sheet.iter_rows(values_only=True, min_row=1, max_col=14), start=1):
                # A bit of debugging info for each row
                #print(f"[DEBUG] Row {idx}: {row}")

                if not row:
                    #print(f"[DEBUG] Row {idx} is empty or None. Skipping.")
                    continue

                row_values = [(str(v) if v is not None else "") for v in row]
                A_value = row_values[0]
                C_value = row_values[2]
                D_value = row_values[3]

                H_value = row_values[7]  # 정답률
                I_value = row_values[8]
                J_value = row_values[9]
                K_value = row_values[10]
                L_value = row_values[11]
                M_value = row_values[12]
                N_value = row_values[13]  # 유형

                # Debug print for key cells
                #print(f"[DEBUG] A_value={A_value}, N_value={N_value}")

                # If prefix is provided, we only check if A_value starts with prefix.
                if prefix is not None:
                    if A_value.startswith(prefix):
                        #print(f"[DEBUG] A_value starts with prefix '{prefix}'")
                        if A_value in existing_values:
                            #print(f"[DEBUG] Found duplicate: {A_value}")
                            duplicates.append(A_value)
                            continue
                        #print(f"[DEBUG] Adding {A_value} to existing_values and tree.")
                        existing_values.add(A_value)
                        self.add_item_to_left_tree(A_value, C_value, D_value,
                                                H_value, I_value, J_value,
                                                K_value, L_value, M_value,
                                                N_value, order_counter)
                        order_counter += 1
                        matched_rows += 1
                    #else:
                        #print(f"[DEBUG] A_value '{A_value}' does not start with prefix '{prefix}'. Skipping.")

                else:
                    # If prefix is None, we check the pattern and filter_type
                    if pattern and re.match(pattern, A_value) and (N_value == filter_type):
                        #print(f"[DEBUG] A_value '{A_value}' matches pattern and filter_type.")
                        if A_value in existing_values:
                            duplicates.append(A_value)
                            continue
                        #print(f"[DEBUG] Adding {A_value} to existing_values and tree.")
                        existing_values.add(A_value)
                        self.add_item_to_left_tree(A_value, C_value, D_value,
                                                H_value, I_value, J_value,
                                                K_value, L_value, M_value,
                                                N_value, order_counter)
                        order_counter += 1
                        matched_rows += 1
                    #else:
                        # Debug: if pattern or N_value didn't match
                        #print(f"[DEBUG] Row {idx} did not match pattern/filter conditions. Skipping.")

            self.reference_items = existing_values.copy()
            #print(f"[DEBUG] Finished iterating rows. matched_rows={matched_rows}, duplicates={duplicates}")
            #print(f"[DEBUG] reference_items after processing: {self.reference_items}")

            # Warn if duplicates were found
            if duplicates:
                messagebox.showwarning("중복 알림", f"{', '.join(duplicates)}는 이미 존재하여 추가되지 않았습니다.")

            # If no rows matched and no duplicates, show an error
            if matched_rows == 0 and not duplicates:
                messagebox.showerror("오류", "서버에 데이터가 없습니다.")
                return

            # Finally, update tooltips
            self.update_all_tooltips()
            #print("[DEBUG] Completed fetch_data_and_add_to_left_tree successfully.")

        except Exception as e:
            messagebox.showerror("오류", "데이터를 불러오는 중 오류가 발생했습니다.")

    def add_item_to_left_tree(self, A_value, C_value, D_value, H_value, I_value, J_value, K_value, L_value, M_value, N_value, order_counter):
        # Convert G_value to text like "85%"
        try:
            gval_text = f"{float(H_value)*100:.0f}%"
        except:
            gval_text = str(H_value)

        # Insert into left tree: col1= A_value, col2= gval_text
        self.left_tree.insert("", "end", values=(A_value, N_value, gval_text))

        # Store additional data for tooltip
        self.item_data_map[A_value] = (C_value, D_value, H_value, I_value, J_value, K_value, L_value, M_value, N_value)
        self.original_order[A_value] = order_counter

    def update_all_tooltips(self):
        show_tooltip = self.tooltip_var.get()
        self.show_tooltip_enabled = show_tooltip
        if not show_tooltip and getattr(self, 'left_tooltip_object', None):
            self.left_tooltip_object.hidetip()
        if not show_tooltip and getattr(self, 'right_tooltip_object', None):
            self.right_tooltip_object.hidetip()

    def enable_tooltips_for_trees(self):
        # Left tree tooltip
        self.left_tooltip_object = ToolTip(self.left_tree, self.default_font)
        self.left_tree.bind("<Motion>", self.on_tree_motion_left)
        self.left_tree.bind("<Leave>", lambda e: self.left_tooltip_object.hidetip())

        # Right tree tooltip
        self.right_tooltip_object = ToolTip(self.right_tree, self.default_font)
        self.right_tree.bind("<Motion>", self.on_tree_motion_right)
        self.right_tree.bind("<Leave>", lambda e: self.right_tooltip_object.hidetip())

    def on_tree_motion_left(self, event):
        if not getattr(self, 'show_tooltip_enabled', True):
            self.left_tooltip_object.hidetip()
            return

        row_id = self.left_tree.identify_row(event.y)
        if row_id:
            vals = self.left_tree.item(row_id, "values")
            if not vals:
                self.left_tooltip_object.hidetip()
                return
            item_id = vals[0]  # the unique ID
            if item_id in self.item_data_map:
                C_value, D_value, H_value, I_value, J_value, K_value, L_value, M_value, N_value = self.item_data_map[item_id]
                선지선택비율 = f"①: {I_value} ②: {J_value} ③: {K_value} ④: {L_value} ⑤: {M_value}" 
                try:
                    H_value_as_text = f"{float(H_value)*100:.0f}%"
                except:
                    H_value_as_text = str(H_value)
                tooltip = (
                    f"{C_value}\n\n정답: {D_value}\n정답률: {H_value_as_text}\n선지 선택 비율: {선지선택비율}"
                )
                x = self.left_tree.winfo_rootx() + event.x + 20
                y = self.left_tree.winfo_rooty() + event.y + 20
                self.left_tooltip_object.hidetip()
                self.left_tooltip_object.showtip(tooltip, x, y)
            else:
                self.left_tooltip_object.hidetip()
        else:
            self.left_tooltip_object.hidetip()

    def on_tree_motion_right(self, event):
        if not getattr(self, 'show_tooltip_enabled', True):
            self.right_tooltip_object.hidetip()
            return

        row_id = self.right_tree.identify_row(event.y)
        if row_id:
            vals = self.right_tree.item(row_id, "values")
            if not vals:
                self.right_tooltip_object.hidetip()
                return
            item_id = vals[1]  # vals[0] is handle, vals[1] is the item_id
            if item_id in self.item_data_map:
                #C_value, D_value, G_value, M_value = self.item_data_map[item_id]
                C_value, D_value, H_value, I_value, J_value, K_value, L_value, M_value, N_value = self.item_data_map[item_id]
                선지선택비율 = f"①: {I_value} ②: {J_value} ③: {K_value} ④: {L_value} ⑤: {M_value}" 

                try:
                    H_value_as_text = f"{float(H_value)*100:.0f}%"
                except:
                    H_value_as_text = str(H_value)
                tooltip = (
                    f"{C_value}\n\n정답: {D_value}\n정답률: {H_value_as_text}\n선지 선택 비율: {선지선택비율}"
                )
                x = self.right_tree.winfo_rootx() + event.x + 20
                y = self.right_tree.winfo_rooty() + event.y + 20
                self.right_tooltip_object.hidetip()
                self.right_tooltip_object.showtip(tooltip, x, y)
            else:
                self.right_tooltip_object.hidetip()
        else:
            self.right_tooltip_object.hidetip()









    def on_generate_set_button_click(self):
        """
        An example:
        - If we're creating a new set, we manually write data into the new sheet
            so the first row is truly row 1.
        - If we're updating an existing set, we still use .append().

        We also still check for duplicates in the local sheet.
        """
        # 1) Gather items from right_tree
        right_items = self.right_tree.get_children()
        if not right_items:
            messagebox.showerror("오류", "위 '3. 선택된 문제'에 항목이 없습니다.")
            return

        create_new_set = (self.set_generation_var.get() == "new")
        selected_sheet = None

        # 2) If appending to an existing set
        if not create_new_set:
            selected_sheet = self.existing_set_combo.get()
            if not self.existing_set_radio_is_enabled() or selected_sheet == "기존 세트 없음":
                messagebox.showwarning("경고", "기존 세트가 없습니다. '새로운 세트 생성' 옵션을 선택하세요.")
                return


        # 3) Group items by year => fetch each year's .xlsx from the server
        collected_data = {}  # {item_id: [A, B, C, D, E, ...], ...}
        ordered_item_ids = []  # Preserve user's order from right_tree

        for iid in right_items:
            row_vals = self.right_tree.item(iid, "values")
            item_id = row_vals[1]  # row_vals[0] is handle, row_vals[1] is the item_id

            parts = item_id.split("-")
            if len(parts) < 4:
                continue
            
            year_2digits = parts[0]
            full_year = f"20{year_2digits}"
            excel_filename = f"{full_year}.xlsx"
            api_url = f"{self.base_url}/get_excel_file/{excel_filename}"

            # Add Authorization header and query params if user is authenticated
            headers = {}
            params = {}
            if hasattr(self.parent, 'user_token') and self.parent.user_token:
                headers['Authorization'] = f'Bearer {self.parent.user_token}'
                if hasattr(self.parent, 'username'):
                    params['username'] = self.parent.username
                if hasattr(self.parent, 'user_role') and self.parent.user_role:
                    params['user_role'] = ', '.join(self.parent.user_role) if isinstance(self.parent.user_role, list) else str(self.parent.user_role)

            file_bytes, status_code = get_excel_file_cached(api_url, headers=headers, params=params)
            if status_code != 200 or file_bytes is None:
                messagebox.showwarning("오류", f"{excel_filename} 파일을 서버에서 불러올 수 없습니다.")
                continue

            try:
                wb_remote = load_workbook(file_bytes, read_only=True, data_only=True)
            except Exception as e:
                messagebox.showwarning("오류", f"{excel_filename} 파일을 읽는 중 오류가 발생했습니다.")
                continue

            sheet_name_remote = "대기열"
            if sheet_name_remote not in wb_remote.sheetnames:
                messagebox.showwarning("오류", f"{excel_filename}에 '대기열' 시트가 없습니다.")
                continue
            
            remote_sheet = wb_remote[sheet_name_remote]
            row_data_map = {}
            for row in remote_sheet.iter_rows(values_only=True, min_row=1):
                if not row:
                    continue
                row_vals_str = [str(v) if v is not None else "" for v in row]
                while len(row_vals_str) < 8:
                    row_vals_str.append("")
                A_val = row_vals_str[0].strip()
                row_data_map[A_val] = row_vals_str

            if item_id in row_data_map:
                collected_data[item_id] = row_data_map[item_id]
                ordered_item_ids.append(item_id)  # Preserve order
            else:
                messagebox.showwarning("정보", f"{excel_filename}에서 {item_id} 데이터를 찾지 못했습니다.")

        if not collected_data:
            messagebox.showinfo("정보", "서버에서 가져올 데이터가 없습니다.")
            return

        # Credit deduction for non-premium members (only after successful data collection)
        user_role = self.parent.fetch_user_role_custom(self.user_token)
        is_allowed = self.parent.check_role_with_flask(user_role)

        if not is_allowed:
            # Non-premium member - second-stage confirmation
            result = self.parent.show_credit_confirmation_dialog(40)
            if result != 'confirm':
                return  # User cancelled or clicked upgrade

            # Check sufficient credits
            try:
                current_credits = int(self.parent.credits)
            except (ValueError, TypeError, AttributeError):
                current_credits = 0

            if current_credits < 40:
                self.parent.show_insufficient_credits_dialog(current_credits)
                return  # Abort due to insufficient credits

            # Deduct credits
            response = self.parent.deduct_credit(
                self.user_token,
                amount=40,
                task='load_original_questions'
            )

            # Handle response carefully
            if not response or not isinstance(response, dict) or 'credits' not in response:
                messagebox.showerror("오류", "포인트 차감 중 오류가 발생했습니다.")
                return  # Abort

            if not response.get('success'):
                current_credits = response.get('credits', 0)
                self.parent.show_insufficient_credits_dialog(current_credits)
                return  # Abort

            # Update credits display
            self.parent.update_credits(response['credits'])

        # 4) Open local Excel

        if not self.excel_file:
            messagebox.showinfo("Info", "현재 사용중인 파일이 없습니다. 문제를 불러오기 전에 파일을 먼저 저장해야 합니다. '확인'을 누르면 파일을 저장합니다.")                        

            self.new_새파일저장묻기()

            if self.file_saved == True:
                self.excel_file = unicodedata.normalize('NFC', self.excel_file)
                self.parent.app_logger.info(f"★[알림] 저장 성공\n(파일 경로: {self.excel_file})\n삭제/덮어쓰기 주의!\n★Make Questions를 클릭하여 출제를 시작하세요.\n")
                self.parent.filepath_label.configure(text=f"현재 파일 경로: {self.excel_file}")
                # Add the file to recent files list
                self.parent.add_to_recent_files(self.excel_file)
                self.parent.load_file_from_server(self.excel_file)
            else:
                messagebox.showerror("오류", "파일이 저장되지 않았습니다. 현재 파일을 먼저 저장해야 합니다.")
                return



        try:
            wb_local = load_workbook(self.excel_file)
        except Exception:
            messagebox.showwarning("오류", f"{self.excel_file} 파일을 열 수 없습니다.")
            return

        # 5) Decide whether to create a new sheet or use existing
        if create_new_set:
            base_name = "기출문제세트"
            counter = 1
            while f"{base_name}{counter}" in wb_local.sheetnames:
                counter += 1
            new_sheet_name = f"{base_name}{counter}"

            ws_local = wb_local.create_sheet(new_sheet_name)
            target_sheet_name = new_sheet_name
        else:
            ws_local = wb_local[selected_sheet]
            target_sheet_name = selected_sheet

        # 6) Build a set of existing IDs to detect duplicates
        existing_ids = set()
        for row in ws_local.iter_rows(values_only=True, min_row=1):
            if not row:
                continue
            first_cell = str(row[0]).strip() if row[0] else ""
            if first_cell:
                existing_ids.add(first_cell)

        duplicates = []
        inserted_count = 0

        # 7) If new sheet => MANUALLY write data starting from row 1
        if create_new_set:
            row_index = 1

            # Write each item to the sheet in the ORDER from right_tree
            for item_id in ordered_item_ids:
                row_vals = collected_data[item_id]
                if item_id in existing_ids:
                    duplicates.append(item_id)
                    continue

                # 1) First, write all columns as-is (just like before)
                col_index = 1
                for cell_value in row_vals:
                    ws_local.cell(row=row_index, column=col_index, value=cell_value)
                    col_index += 1

                # 2) Modify column A by appending column N data (as in your original code)
                col_n_value = row_vals[13] if len(row_vals) > 13 else ""  # Column N is index 13
                modified_col_a = f"{item_id} // {col_n_value}_Normal"
                ws_local.cell(row=row_index, column=1, value=modified_col_a)

                # 3) Build the combined text for column E:

                col_e_original = row_vals[4] if len(row_vals) > 4 else ""
                col_f_original = row_vals[5] if len(row_vals) > 5 else ""
                col_g_original = row_vals[6] if len(row_vals) > 6 else ""
                col_h_original = row_vals[7] if len(row_vals) > 7 else ""
                col_i_original = row_vals[8] if len(row_vals) > 8 else ""
                col_j_original = row_vals[9] if len(row_vals) > 9 else ""
                col_k_original = row_vals[10] if len(row_vals) > 10 else ""
                col_l_original = row_vals[11] if len(row_vals) > 11 else ""
                col_m_original = row_vals[12] if len(row_vals) > 12 else ""

                #print(f"col_g_original: {col_g_original}")

                # Process column G: split by "\n", strip, and join with " / "
                g_parts = [part.strip() for part in col_g_original.split("\n") if part.strip()]
                g_joined = " / ".join(g_parts)

                if col_e_original != "":
                    col_e_original = f"[해설] {col_e_original}\n"

                combined_e_value = (
                    f"{col_e_original}"
                    f"[해석] {col_f_original}\n" 
                    f"[어휘] {g_joined}\n"
                    f"[정답률] {col_h_original}\n"
                    f"[선지 선택 비율] ①: {col_i_original} "
                    f"②: {col_j_original} "
                    f"③: {col_k_original} "
                    f"④: {col_l_original} "
                    f"⑤: {col_m_original}"
                )

                # 4) Overwrite column E with the combined value
                ws_local.cell(row=row_index, column=5, value=combined_e_value)

                # 5) Finish up
                row_index += 1
                inserted_count += 1
                existing_ids.add(item_id)

        else:
            # Existing set => modify the row_vals before appending in ORDER from right_tree
            for item_id in ordered_item_ids:
                row_vals = collected_data[item_id]
                if item_id in existing_ids:
                    duplicates.append(item_id)
                    continue

                # 1) Create a copy of row_vals so we can modify it
                modified_row_vals = list(row_vals)

                # 2) Modify column A by appending column N data (as in your original code)
                col_n_value = row_vals[13] if len(row_vals) > 13 else ""  # Column N is index 13
                modified_row_vals[0] = f"{item_id} // {col_n_value}_Normal"

                # 3) Build the combined text for column E:
                col_e_original = row_vals[4] if len(row_vals) > 4 else ""
                col_f_original = row_vals[5] if len(row_vals) > 5 else ""
                col_g_original = row_vals[6] if len(row_vals) > 6 else ""
                col_h_original = row_vals[7] if len(row_vals) > 7 else ""
                col_i_original = row_vals[8] if len(row_vals) > 8 else ""
                col_j_original = row_vals[9] if len(row_vals) > 9 else ""
                col_k_original = row_vals[10] if len(row_vals) > 10 else ""
                col_l_original = row_vals[11] if len(row_vals) > 11 else ""
                col_m_original = row_vals[12] if len(row_vals) > 12 else ""

                # Process column F: split by "\n", strip, and join with " / "
                g_parts = [part.strip() for part in col_g_original.split("\n") if part.strip()]
                g_joined = " / ".join(g_parts)

                if col_e_original != "":
                    col_e_original = f"[해설] {col_e_original}\n"

                combined_e_value = (
                    f"{col_e_original}"
                    f"[해석] {col_f_original}\n" 
                    f"[어휘] {g_joined}\n"
                    f"[정답률] {col_h_original}\n"
                    f"[선지 선택 비율] ①: {col_i_original} "
                    f"②: {col_j_original} "
                    f"③: {col_k_original} "
                    f"④: {col_l_original} "
                    f"⑤: {col_m_original}"
                )

                # 6) Overwrite column E (index 4) with this new multiline text
                modified_row_vals[4] = combined_e_value

                # 7) Append the modified row to the existing sheet
                ws_local.append(modified_row_vals)
                inserted_count += 1
                existing_ids.add(item_id)

                
        # 8) Duplicate Warning
        if duplicates:
            messagebox.showwarning(
                "중복 알림", 
                f"이미 존재하는 데이터가 있어 추가되지 않았습니다: {', '.join(duplicates)}"
            )

        # 9) Save
        wb_local.save(self.excel_file)


        # 10) Usage log
        if self.user_token:
            usage_url = "https://frontierenglish.net/wp-json/myapp/v1/store-file-usage"
            headers = {
                "Authorization": f"Bearer {self.user_token}",
                "Content-Type": "application/json"
            }
            data = {
                "filename": excel_filename,
                "type": "기출문제(원본)",
                "price": "2,800",
            }
            try:
                resp = requests.post(usage_url, headers=headers, json=data)
                resp.raise_for_status()
                print(f"Usage log saved: {resp.json()}")
            except requests.RequestException as e:
                print(f"Failed to save usage log: {e}")
        else:
            print("No token, skipping usage log.")

            

        # 11) Update UI
        self.update_sheet_selector(self.excel_file)
        self.parent.sheet_selector.set(target_sheet_name)
        self.parent.load_excel_into_treeview(self.excel_file, target_sheet_name)
        self.parent.app_logger.info("문제 불러오기를 완료했습니다.")
        self.destroy()
        """
        if inserted_count > 0:
            messagebox.showinfo("정보", f"데이터를 성공적으로 가져왔습니다. 추가된 항목: {inserted_count}개")

        # 11) Clear right_tree
        for iid in right_items:
            self.right_tree.delete(iid)
        """
        
    def get_existing_sets(self):
        try:
            wb = load_workbook(self.excel_file, read_only=True)
            sheets = wb.sheetnames
            existing_sets = [s for s in sheets if s.startswith("기출문제세트")]
            return existing_sets
        except Exception:
            return []

    def existing_set_radio_is_enabled(self):
        return True

    def update_sheet_selector(self, excel_file):
        wb = load_workbook(filename=excel_file, read_only=True)
        sheet_names = wb.sheetnames
        self.parent.sheet_selector.configure(values=sheet_names)

    def clear_right_tree(self):
        """Delete all items from self.right_tree."""
        for item_id in self.right_tree.get_children():
            self.right_tree.delete(item_id)
        self.update_right_tree_count()

    def new_새파일저장묻기(self, event=None):
        
        if self.excel_file is None:
            file_path = filedialog.asksaveasfilename(defaultextension=".xlsx", filetypes=[("Excel files", "*.xlsx"), ("All files", "*.*")])                
            
            if not file_path:  # If the user cancels the save dialog
                self.parent.app_logger.info(f"저장이 취소되었습니다.\n")
                return  # Do not proceed further

            # Create a new Excel workbook and add a default sheet
            self.file_saved = True
            wb = Workbook()
            ws = wb.active
            ws.title = "대기열"
            # Save the new workbook to the chosen file path
            wb.save(filename=file_path)
            self.excel_file = file_path  # Update the global excel_file variable
            

            sheet_name = "대기열"  
            self.parent.save_treeview_to_excel(self.excel_file, sheet_name)
            




    ######## 우클릭 메뉴
    def create_right_click_menu(self):
        """Create right-click context menu for the right tree."""
        self.right_click_menu = Menu(self, tearoff=0)
        self.right_click_menu.add_command(label="위로 이동", command=self.move_items_up)
        self.right_click_menu.add_command(label="아래로 이동", command=self.move_items_down)

    def show_right_click_menu(self, event):
        """Show the context menu on right-click."""
        if self.right_tree.selection():  # Only show menu if items are selected
            self.right_click_menu.tk_popup(event.x_root, event.y_root)

    def move_items_up(self):
        """Move selected items up one position."""
        selected_items = self.right_tree.selection()
        if not selected_items:
            return

        for item in selected_items:
            idx = self.right_tree.index(item)
            if idx > 0:  # Can't move up if already at top
                self.right_tree.move(item, "", idx - 1)

    def move_items_down(self):
        """Move selected items down one position."""
        selected_items = self.right_tree.selection()
        if not selected_items:
            return

        # Process items in reverse order when moving down
        for item in reversed(selected_items):
            idx = self.right_tree.index(item)
            last_idx = len(self.right_tree.get_children()) - 1
            if idx < last_idx:  # Can't move down if already at bottom
                self.right_tree.move(item, "", idx + 1)

    def on_close(self, event=None):
        self.destroy()