# -*- coding: utf-8 -*-
"""
LoginWindowApp Module

This module contains the LoginWindowApp class, which provides the login interface
for the AutoQM application. It was extracted from the main autoQM.py file to
improve code organization and maintainability.

The LoginWindowApp class handles:
- User authentication via WordPress JWT API
- Credential storage and retrieval (encrypted with Fernet)
- Auto-login functionality
- User role fetching
- Credit balance retrieval
- UI rendering for the login screen

Dependencies:
- customtkinter for modern UI components
- PIL for image handling
- requests for API communication
- cryptography.fernet for credential encryption
- webbrowser for opening external links
"""

import sys
import os
import json
import webbrowser
import uuid
import threading
import time
from pathlib import Path
from tkinter import messagebox

import customtkinter
from customtkinter import CTkFrame, CTkLabel, CTkEntry, CTkButton, CTkCheckBox, CTkProgressBar, StringVar, CTkImage, CTkToplevel
from PIL import Image
import requests
import cryptography.fernet
from cryptography.fernet import Fernet

from modules.font_handler import font_handler
from modules.api_utils import get_with_retry, NetworkError, APIError


class LoginWindowApp(CTkFrame):
    """
    Login window frame for AutoQM application.

    This class provides a login interface that authenticates users against a WordPress
    backend using JWT tokens. It supports credential persistence (encrypted), auto-login,
    and fetches user credits and roles after successful authentication.

    Parameters
    ----------
    master : CTk
        The parent window (MainApp instance)
    base_url : str
        The base URL for the Flask backend API
    currentappver : str
        Current application version string (e.g., "v2.4.4")
    credentials_file : Path
        Path to the encrypted credentials file
    cipher_suite : Fernet
        Fernet cipher suite for encrypting/decrypting credentials
    MainFrame : class
        Reference to the MainFrame class for frame switching

    Attributes
    ----------
    login_entry : CTkEntry
        Entry widget for username input
    pw_entry : CTkEntry
        Entry widget for password input
    remember_me_checkbox : CTkCheckBox
        Checkbox for "remember me" functionality
    auto_login_checkbox : CTkCheckBox
        Checkbox for auto-login functionality

    Global Variables Modified
    -------------------------
    user_token : str
        JWT authentication token (set upon successful login)
    credits : int
        User's credit balance (set after fetching from server)
    is_logged_in : bool
        Login status flag (set to True upon successful login)

    Notes
    -----
    - Credentials are encrypted using Fernet symmetric encryption
    - Auto-login is triggered 100ms after initialization if credentials exist
    - The class communicates with the parent MainApp via:
        - self.master.frames[MainFrame] - to access MainFrame instance
        - self.master.show_frame() - to switch between frames
        - self.master.load_notice_data_mainframe() - to load notice data
        - self.master.manual_window() - to show manual dialog
        - self.master.안내메시지 - to access the manual message text
    """

    def __init__(self, master, base_url, currentappver, credentials_file, cipher_suite, MainFrame):
        super().__init__(master)
        self.master = master
        self.base_url = base_url
        self.currentappver = currentappver
        self.credentials_file = credentials_file
        self.cipher_suite = cipher_suite
        self.MainFrame = MainFrame

        # Configure frame
        self.configure(fg_color="#FFFFFF")
        self.columnconfigure(0, weight=1)
        self.rowconfigure(2, weight=1)
        self.rowconfigure(0, minsize=150)

        # Set up font
        font_handler.set_default_font()
        self.default_font = font_handler.default_font

        # Determine base path for assets
        if getattr(sys, 'frozen', False):
            # If the application is bundled, the _MEIPASS attribute contains the path to the temporary directory
            self.base_path = Path(sys._MEIPASS)
        else:
            # If running as a regular script, use the directory of the script file as the base path
            self.base_path = Path(__file__).parent.parent

        # Draw UI elements
        self.draw_login_elements()

        # Load saved credentials
        credentials = self.load_credentials()

        if credentials:
            # Set the login entry and password entry with the credentials
            self.login_entry.insert(0, credentials['username'])
            self.pw_entry.insert(0, credentials['password'])
            # Determine the checkbox state based on 'auto_login' and 'remember' flags
            if 'remember' in credentials and credentials['remember']:
                self.remember_me_checkbox.select()
            else:
                self.remember_me_checkbox.deselect()

            if 'auto_login' in credentials and credentials['auto_login']:
                self.auto_login_checkbox.select()  # Check the auto_login_checkbox if auto_login is True
            else:
                self.auto_login_checkbox.deselect()  # Uncheck the auto_login_checkbox if auto_login is not True
        else:
            # Check for social login credentials if no regular credentials exist
            social_creds = self.load_social_credentials()
            if social_creds and social_creds.get('auto_login'):
                # For social login with auto_login, both checkboxes should be checked
                self.remember_me_checkbox.select()
                self.auto_login_checkbox.select()

        # Delay auto-login check to ensure UI is ready
        self.master.after(100, self.auto_login_check)

        # Bind Enter key to login
        self.bind('<Return>', self.실행_perform_login)

        # Set up exit handler
        self.master.protocol("WM_DELETE_WINDOW", self.on_exit)

    @staticmethod
    def fetch_credits(token, show_error=True):
        """
        Fetch user credits from WordPress API with automatic retry.

        This function uses automatic retry logic (3 attempts with exponential backoff)
        to handle transient network errors and server issues.

        Parameters
        ----------
        token : str
            JWT authentication token
        show_error : bool, optional
            Whether to show error dialog on failure (default: True)

        Returns
        -------
        int or None
            User's credit balance, or None if fetch fails after all retries
        """
        credits_url = 'https://frontierenglish.net/wp-json/myapp/v1/credits'
        headers = {'Authorization': f'Bearer {token}'}

        try:
            # Use retry mechanism: 3 attempts with exponential backoff (1s, 2s, 4s)
            response = get_with_retry(
                url=credits_url,
                headers=headers,
                timeout=10,
                max_retries=3,
                backoff_factor=2.0
            )

            # Parse response
            credits_info = response.json()
            credits = credits_info.get('credits', 0)
            return credits

        except (NetworkError, APIError) as e:
            # Network or API errors after all retries exhausted
            if show_error:
                messagebox.showerror(
                    "포인트 조회 실패",
                    "포인트 정보를 가져올 수 없습니다.\n네트워크 연결을 확인하고 다시 시도해주세요."
                )
            return None

        except (ValueError, requests.exceptions.JSONDecodeError) as e:
            # JSON parsing error
            if show_error:
                messagebox.showerror(
                    "포인트 조회 실패",
                    "서버 응답 형식이 올바르지 않습니다.\n관리자에게 문의하세요."
                )
            return None

        except Exception as e:
            # Unexpected errors
            if show_error:
                messagebox.showerror(
                    "포인트 조회 실패",
                    f"포인트 정보를 가져올 수 없습니다.\n오류: {str(e)}"
                )
            return None

    def draw_login_elements(self):
        """Draw all UI elements for the login screen."""
        # Top frame
        self.login_upperframe = CTkFrame(master=self, fg_color="white", width=1100, height=150, corner_radius=0)
        self.login_upperframe.grid(row=0, column=0, sticky="nsew")
        self.login_upperframe.rowconfigure(0, weight=0)
        self.login_upperframe.columnconfigure(0, weight=1)
        self.login_upperframe.columnconfigure(1, weight=1)

        self.login_upperframeleft = CTkFrame(master=self.login_upperframe, fg_color="#2B3D2D", width=550, height=150, corner_radius=0)
        self.login_upperframeleft.grid(row=0, column=0, sticky="nsw")
        self.login_upperframeleft.rowconfigure(0, weight=1)
        self.login_upperframeleft.grid_propagate(False)
        self.login_upperframeright = CTkFrame(master=self.login_upperframe, fg_color="white", width=550, height=150, corner_radius=0)
        self.login_upperframeright.grid(row=0, column=1, sticky="nse")
        self.login_upperframeright.rowconfigure(0, weight=1)
        self.login_upperframeright.grid_propagate(False)

        titlelabel = CTkLabel(self.login_upperframeleft, text="Automatic Question Maker", font=(self.default_font, 30, "bold"), text_color="white", fg_color="transparent")
        titlelabel.grid(row=0, column=0, sticky="sw", padx=(50, 5), pady=9)
        titlelabel2 = CTkLabel(self.login_upperframeleft, text=self.currentappver, font=(self.default_font, 15), text_color="white", fg_color="transparent")
        titlelabel2.grid(row=0, column=1, sticky="se", padx=(15, 5), pady=7)

        logo_image = CTkImage(light_image=Image.open(self.base_path / "assets" / "images" / "frontierlogo1.png"), size=(100, 100))
        logo_label = CTkButton(self.login_upperframeright, image=logo_image, width=130, height=130, text="", fg_color="transparent", hover_color="#DDA15C", command=self.open_website)
        logo_label.pack(side="right", padx=20, pady=(10, 0))

        # Middle frame - progress bar
        self.login_middleframe = CTkFrame(master=self, fg_color="white", width=1100, height=4)
        self.login_middleframe.grid(row=1, column=0, sticky="ew")

        login_progress_bar = CTkProgressBar(self.login_middleframe, orientation="horizontal", bg_color='#2B3D2D', height=4, progress_color='#BF7503', determinate_speed=0.2, width=1100)
        login_progress_bar.grid(row=0, column=0, sticky="ew")
        login_progress_bar.configure(mode="determinate")
        login_progress_bar.set(0)
        login_progress_bar.start()

        # Bottom frame
        self.login_bottomframe = CTkFrame(master=self, fg_color="transparent", width=1100)
        self.login_bottomframe.grid(row=2, column=0, sticky="nsew")
        self.login_bottomframe.rowconfigure(0, weight=1)

        # Bottom left frame - introduction
        self.login_bottomleft_frame = CTkFrame(self.login_bottomframe, width=550, fg_color="#2B3D2D", corner_radius=0)
        self.login_bottomleft_frame.grid(row=0, column=0, sticky="ns")

        introducing_label = CTkLabel(self.login_bottomleft_frame, text="Introducing the\nUltimate Question Generator:", font=('Times New Roman', 28), text_color="white", fg_color="transparent", justify="left")
        introducing_label.place(x=50, y=100)
        introducing_label2 = CTkLabel(self.login_bottomleft_frame, text="With its sophisticated question formulation, versatile adaptability, and user-friendly interface, autoQM stands as the epitome of streamlined question generation.\n\nThis cutting-edge tool is poised to elevate the quality and efficiency of creating questions for quizzes, assessments, and educational content.", font=('Times New Roman', 16), text_color="white", fg_color="transparent", wraplength=450, justify="left")
        introducing_label2.place(x=50, y=200)

        contact_label = CTkLabel(self.login_bottomleft_frame, text="Contact: emong111@naver.com", font=('Times New Roman', 14), text_color="white", fg_color="transparent", justify="left")
        contact_label.place(x=340, y=500)

        # Bottom right frame - login form
        self.login_bottomright_frame = CTkFrame(self.login_bottomframe, width=550, fg_color="white", corner_radius=0)
        self.login_bottomright_frame.grid(row=0, column=1, sticky="ns")

        loginlabel = CTkLabel(self.login_bottomright_frame, text="login", font=('Times New Roman', 24), text_color="#CA7900", fg_color="transparent")
        loginlabel.place(x=56, y=108)

        loginentryframe = CTkFrame(self.login_bottomright_frame, width=438, height=80, corner_radius=10, fg_color="transparent")
        loginentryframe.place(x=56, y=147)

        login_entry_label = CTkLabel(loginentryframe, text="ID:", font=('Times New Roman', 13, "bold"), text_color="black", fg_color="transparent")
        login_entry_label.place(x=10, y=5)

        self.login_entry = CTkEntry(loginentryframe, text_color="black", fg_color="#FAF3E8", width=430, height=40, font=('Times New Roman', 17), border_width=0, corner_radius=10)
        self.login_entry.place(x=10, y=30)

        pwentryframe = CTkFrame(self.login_bottomright_frame, width=438, height=80, corner_radius=10, fg_color="transparent")
        pwentryframe.place(x=56, y=216)
        pw_entry_label = CTkLabel(pwentryframe, text="PASSWORD:", font=('Times New Roman', 13, "bold"), text_color="black", fg_color="transparent")
        pw_entry_label.place(x=10, y=5)
        self.pw_entry = CTkEntry(pwentryframe, text_color="black", fg_color="#FAF3E8", width=430, height=40, font=('Times New Roman', 17), border_width=0, corner_radius=10, show="*")
        self.pw_entry.place(x=10, y=30)

        # Variables for checkbox states
        self.remember_me_var = StringVar(value="0")
        # Create "remember me" checkbox
        self.remember_me_checkbox = CTkCheckBox(self.login_bottomright_frame, text="remember me", variable=self.remember_me_var, font=('Times New Roman', 12), text_color="black", border_width=2, checkbox_width=18, checkbox_height=18, fg_color="#CA7900", hover_color="#CA7900", onvalue="1", offvalue="0")
        self.remember_me_checkbox.place(x=315, y=290)

        self.auto_login_var = StringVar(value="0")
        # Create "auto login" checkbox
        self.auto_login_checkbox = CTkCheckBox(self.login_bottomright_frame, text="auto login", variable=self.auto_login_var, command=self.on_auto_login_check, font=('Times New Roman', 12), text_color="black", border_width=2, checkbox_width=18, checkbox_height=18, fg_color="#CA7900", hover_color="#CA7900", onvalue="1", offvalue="0")
        self.auto_login_checkbox.place(x=420, y=290)

        login_button = CTkButton(self.login_bottomright_frame, text="login", command=self.실행_perform_login, width=430, height=50, font=('Times New Roman', 15, "bold"), border_width=0, corner_radius=10, text_color="white", fg_color="#DDA15C", hover_color="#CA7900")
        login_button.place(x=66, y=340)

        singup_button = CTkButton(self.login_bottomright_frame, text="Sign up", command=self.open_signup_website, width=55, height=20, font=('Times New Roman', 12, "bold"), border_width=0, corner_radius=10, text_color="#CA7900", fg_color="transparent", hover=False)
        singup_button.place(x=56, y=398)

        forget_pw_button = CTkButton(self.login_bottomright_frame, text="Forgot ID/Password?", command=self.open_forgot_website, width=80, height=20, font=('Times New Roman', 12, "bold"), border_width=0, corner_radius=10, text_color="#CA7900", fg_color="transparent", hover=False)
        forget_pw_button.place(x=370, y=398)

        # Social login divider
        divider_frame = CTkFrame(self.login_bottomright_frame, fg_color="transparent", width=430, height=30)
        divider_frame.place(x=66, y=430)

        divider_left = CTkFrame(divider_frame, fg_color="#E5E7EB", width=130, height=1)
        divider_left.place(x=0, y=15)

        divider_text = CTkLabel(divider_frame, text="또는 소셜 로그인", font=(self.default_font, 11), text_color="#6B7280", fg_color="transparent")
        divider_text.place(x=215, y=15, anchor='center')

        divider_right = CTkFrame(divider_frame, fg_color="#E5E7EB", width=130, height=1)
        divider_right.place(x=300, y=15)

        # Social login buttons frame
        social_buttons_frame = CTkFrame(self.login_bottomright_frame, fg_color="transparent", width=430, height=45)
        social_buttons_frame.place(x=66, y=465)

        # Google button
        google_button = CTkButton(
            social_buttons_frame,
            text="Google",
            command=lambda: self.social_login('google'),
            width=135,
            height=40,
            font=(self.default_font, 12),
            border_width=1,
            corner_radius=6,
            text_color="#374151",
            fg_color="#FFFFFF",
            hover_color="#F3F4F6",
            border_color="#D1D5DB"
        )
        google_button.place(x=0, y=0)

        # Naver button
        naver_button = CTkButton(
            social_buttons_frame,
            text="네이버",
            command=lambda: self.social_login('naver'),
            width=135,
            height=40,
            font=(self.default_font, 12),
            border_width=0,
            corner_radius=6,
            text_color="#FFFFFF",
            fg_color="#03C75A",
            hover_color="#02B350"
        )
        naver_button.place(x=147, y=0)

        # Kakao button
        kakao_button = CTkButton(
            social_buttons_frame,
            text="카카오",
            command=lambda: self.social_login('kakao'),
            width=135,
            height=40,
            font=(self.default_font, 12),
            border_width=0,
            corner_radius=6,
            text_color="#000000",
            fg_color="#FEE500",
            hover_color="#FFEB3B"
        )
        kakao_button.place(x=294, y=0)

    def 실행_perform_login(self, event=None):
        """
        Execute login process with current form values.

        This is triggered by clicking the login button or pressing Enter.

        Parameters
        ----------
        event : Event, optional
            Tkinter event (when triggered by Enter key)
        """
        username = self.login_entry.get()
        password = self.pw_entry.get()
        remember = self.remember_me_var.get() == '1'
        auto_login_str = self.auto_login_var.get()

        if username == "":
            messagebox.showerror("Login Failed", "아이디를 입력하세요.")
            return
        elif password == "":
            messagebox.showerror("Login Failed", "비밀번호를 입력하세요.")
        else:
            auto_login = auto_login_str == '1'
            self.perform_login(username, password, remember, auto_login)

    def offline_perform_login(self, username, password, remember, auto_login=False):
        """
        Perform offline login (for testing purposes).

        This method sets login state without contacting the server.

        Parameters
        ----------
        username : str
            Username
        password : str
            Password
        remember : bool
            Whether to remember credentials
        auto_login : bool, optional
            Whether auto-login is enabled
        """
        global user_token, credits, is_logged_in

        is_logged_in = True
        credits = 99999
        user_role = "test"
        user_token = "test_token"

        self.master.show_frame(self.MainFrame)

        # Set logged in state and credentials on MainFrame instance
        if self.master and hasattr(self.master, 'frames'):
            main_frame_instance = self.master.frames.get(self.MainFrame)
            if main_frame_instance:
                main_frame_instance.is_logged_in = True
                main_frame_instance.user_token = user_token
                main_frame_instance.credits = credits
                print(f"DEBUG test_login: Set main_frame_instance.is_logged_in = True, credits = {credits}")
                main_frame_instance.update_run_button_state()

    def perform_login(self, username, password, remember, auto_login=False):
        """
        Perform login via WordPress JWT API.

        This method authenticates the user, fetches credits and role information,
        updates the MainFrame with user data, and switches to the main application view.

        Parameters
        ----------
        username : str
            WordPress username
        password : str
            WordPress password
        remember : bool
            Whether to save credentials for future sessions
        auto_login : bool, optional
            Whether to enable auto-login for future sessions

        Global Variables Modified
        -------------------------
        user_token : str
            Set to the JWT token returned by WordPress
        credits : int
            Set to the user's credit balance
        is_logged_in : bool
            Set to True upon successful login
        """
        global user_token, credits, is_logged_in

        # Additional logic to clear previous credentials if logging in with a new account
        existing_credentials = self.load_credentials()
        if existing_credentials and (username != existing_credentials.get('username') or not remember):
            # Clear existing credentials if usernames don't match or remember is unchecked
            self.save_credentials(username, password, False)

        login_response = self.login_to_wordpress(username, password)

        if remember:
            # Call save_credentials only once with all the necessary flags
            self.save_credentials(username, password, remember, auto_login)
            # Clear social credentials when using regular login with auto_login
            if auto_login:
                self.clear_social_credentials()

        is_logged_in = True

        if login_response and 'token' in login_response:
            user_token = login_response['token']

            # Log successful login to server
            try:
                log_url = f'{self.base_url}/log_login'

                # Make sure to send credits after fetching them
                fetched_credits = LoginWindowApp.fetch_credits(user_token, show_error=False)
                if fetched_credits is not None:
                    credits = fetched_credits  # Update the global credits variable
                    log_data = {'username': username, 'credits': credits, 'app_version': self.currentappver}
                    requests.post(log_url, json=log_data)
                else:
                    messagebox.showerror("Login Error", "로그인은 성공했으나 포인트 정보를 가져오는 데 실패했습니다.\n네트워크 연결을 확인하고 다시 시도해주세요.")
                    # Still log the login even if credits fetch failed
                    log_data = {'username': username, 'credits': 'unknown'}
                    requests.post(log_url, json=log_data)
            except Exception as e:
                print(f"Error logging login: {str(e)}")

            # Fetch custom message after successful login (for specific users)
            block_message = self.get_block_message(username)
            if block_message:
                messagebox.showinfo("이용 불가 알림", block_message)
                return

            is_logged_in = True

            # Fetch user role
            user_role = self.fetch_user_role_custom(user_token)

            if self.master and hasattr(self.master, 'frames'):
                main_frame_instance = self.master.frames.get(self.MainFrame)
                if main_frame_instance:
                    # Set logged in state and credentials on MainFrame instance
                    main_frame_instance.is_logged_in = True
                    main_frame_instance.user_token = user_token
                    main_frame_instance.credits = credits
                    #print(f"DEBUG perform_login: Set main_frame_instance.is_logged_in = True, user_token set, credits = {credits}")
                    main_frame_instance.update_credits(credits)
                    main_frame_instance.set_username(username)
                    main_frame_instance.set_user_role(user_role)
                    # Also update button state after login
                    main_frame_instance.update_run_button_state()
                    self.role_korean = "일반회원"
                    if 'subscriber' in user_role or 'um_custom_role_1' in user_role:
                        self.role_korean = "정회원"
                    main_frame_instance.role_label.configure(text=f"{self.role_korean}")

                    # Token monitoring disabled - token validity period is sufficient
                    # main_frame_instance.initialize_token_manager()

            self.master.show_frame(self.MainFrame)

            data = self.master.load_notice_data_mainframe()
            dont_show_again = data.get("dont_show_again", False)
            if not dont_show_again:
                self.master.manual_window(self.master.안내메시지)

        else:
            return  # Early return if login failed

    def on_auto_login_check(self):
        """
        Handle auto-login checkbox state change.

        Automatically checks "Remember Me" when "Auto Login" is checked.
        """
        if self.auto_login_var.get() == "1":
            self.remember_me_checkbox.select()
        else:
            self.remember_me_checkbox.deselect()

    def auto_login_check(self):
        """
        Check if auto-login should be performed based on saved credentials.

        This is called 100ms after initialization to allow UI to be ready.
        Checks both regular credentials and social login credentials.
        """
        # First, check for social login credentials
        social_creds = self.load_social_credentials()
        if social_creds and social_creds.get('auto_login'):
            token = social_creds.get('token')
            if token:
                validation_result = self.validate_token(token)
                if validation_result is True:
                    print("[Social Login] Auto-login with saved token...")
                    self.complete_social_login_from_saved(token)
                    return
                elif validation_result is False:
                    # Token is definitely expired/invalid, clear it
                    print("[Social Login] Saved token is invalid or expired, clearing...")
                    self.clear_social_credentials()
                else:
                    # Network error - try to use the token anyway
                    # (it might still work, and user can re-login if it fails)
                    print("[Social Login] Network error during validation, attempting auto-login anyway...")
                    self.complete_social_login_from_saved(token)
                    return

        # Fall back to regular credentials
        credentials = self.load_credentials()
        if credentials and 'auto_login' in credentials and credentials['auto_login']:
            # If auto_login is true in the saved credentials, perform login automatically
            self.perform_login(credentials['username'], credentials['password'], True, credentials['auto_login'])

    def on_exit(self):
        """
        Handle window close event.

        Saves credentials if "remember me" is checked, then exits the application.
        """
        # Fetch current input values
        username = self.login_entry.get()
        password = self.pw_entry.get()
        remember = self.remember_me_var.get() == '1'
        auto_login = self.auto_login_var.get() == '1'

        self.save_credentials(username, password, remember, auto_login)
        sys.exit()

    def save_credentials(self, username, password, remember, auto_login=False):
        """
        Save user credentials to encrypted file.

        Parameters
        ----------
        username : str
            Username to save
        password : str
            Password to save (will be encrypted)
        remember : bool
            Whether to save credentials
        auto_login : bool, optional
            Whether auto-login is enabled
        """
        if remember:
            data = {'username': username, 'password': password, 'remember': remember, 'auto_login': auto_login}
            encrypted_data = self.cipher_suite.encrypt(json.dumps(data).encode('utf-8'))
            with open(self.credentials_file, 'wb') as file:
                file.write(encrypted_data)
        else:
            if os.path.exists(self.credentials_file):
                os.remove(self.credentials_file)

    def load_credentials(self):
        """
        Load saved credentials from encrypted file.

        Returns
        -------
        dict or None
            Dictionary containing 'username', 'password', 'remember', and 'auto_login' keys,
            or None if no credentials exist or decryption fails
        """
        if os.path.exists(self.credentials_file):
            try:
                with open(self.credentials_file, 'rb') as file:
                    encrypted_data = file.read()
                    decrypted_data = self.cipher_suite.decrypt(encrypted_data)
                    data = json.loads(decrypted_data.decode('utf-8'))
                    return data
            except (cryptography.fernet.InvalidToken, json.JSONDecodeError) as e:
                # Handle the error by removing the corrupted file and returning None
                print(f"Error decrypting credentials: {e}")
                os.remove(self.credentials_file)
            except Exception as e:
                print(f"Unexpected error: {e}")
        return None

    def login_to_wordpress(self, username, password):
        """
        Authenticate user via WordPress JWT endpoint.

        Parameters
        ----------
        username : str
            WordPress username
        password : str
            WordPress password

        Returns
        -------
        dict or None
            JSON response containing 'token' key if successful, None if failed
        """
        login_url = 'https://frontierenglish.net/wp-json/jwt-auth/v1/token'
        data = {
            'username': username,
            'password': password
        }
        try:
            response = requests.post(login_url, data=data)
            response.raise_for_status()  # Raise an error for bad responses
            return response.json()  # Return the token if login is successful
        except requests.RequestException as e:
            messagebox.showerror("Login Failed", "서버에 문제가 있거나 아이디 혹은 비밀번호를 확인할 수 없습니다. 관리자에게 문의하세요.")
            return None

    def get_block_message(self, username):
        """
        Fetch block message for specific user (if any).

        Parameters
        ----------
        username : str
            Username to check

        Returns
        -------
        str or None
            Block message if user is blocked, None otherwise
        """
        block_message_url = f'{self.base_url}/block_message'
        try:
            response = requests.post(block_message_url, json={'username': username})
            response.raise_for_status()
            return response.json().get('block_messages')
        except requests.RequestException:
            return None

    def fetch_user_role(self, token):
        """
        Fetch user role from WordPress API.

        Parameters
        ----------
        token : str
            JWT authentication token

        Returns
        -------
        list or None
            List of user roles, or None if fetch fails
        """
        url = "https://frontierenglish.net/wp-json/wp/v2/users/me"
        headers = {"Authorization": f"Bearer {token}"}

        try:
            response = requests.get(url, headers=headers)
            response.raise_for_status()
            user_data = response.json()

            # 'roles' is typically an array. For example: ["administrator"] or ["subscriber", "customer"].
            roles = user_data.get("roles", [])

            if roles:
                return roles  # or roles[0] if you only care about the first role
            else:
                return None
        except requests.RequestException as e:
            messagebox.showerror("Fetch User Role Failed", "사용자의 권한 정보를 가져올 수 없습니다.")
            return None

    def fetch_user_role_custom(self, token):
        """
        Fetch user role from custom WordPress endpoint.

        Parameters
        ----------
        token : str
            JWT authentication token

        Returns
        -------
        list
            List of user roles, or empty list if fetch fails
        """
        url = 'https://frontierenglish.net/wp-json/myapp/v1/role'
        headers = {'Authorization': f'Bearer {token}'}
        try:
            response = requests.get(url, headers=headers)
            response.raise_for_status()
            data = response.json()
            return data.get('roles', [])
        except requests.RequestException:
            messagebox.showerror("Fetch Failed", "사용자의 권한 정보를 가져올 수 없습니다.")
            return []

    def open_website(self):
        """Open main website in browser."""
        webbrowser.open("https://frontierenglish.net")

    def open_signup_website(self):
        """Open signup page in browser."""
        webbrowser.open("https://frontierenglish.net/register/")

    def open_forgot_website(self):
        """Open password reset page in browser."""
        webbrowser.open("https://frontierenglish.net/password-reset/")

    # ========== Social Login Methods ==========

    def social_login(self, provider):
        """
        Initiate social login flow for the specified provider.

        Opens browser for OAuth authentication and starts polling for completion.

        Parameters
        ----------
        provider : str
            Social login provider: 'google', 'naver', or 'kakao'
        """
        # Prevent multiple simultaneous login attempts
        if hasattr(self, 'polling_active') and self.polling_active:
            print("[Social Login] Login already in progress, ignoring...")
            return

        # Generate unique session_id for this login attempt
        session_id = str(uuid.uuid4())
        self.social_login_session_id = session_id

        # WordPress social login endpoint with desktop flag
        wp_url = 'https://frontierenglish.net'
        login_url = f'{wp_url}/wp-json/autoqm-social/v1/login/{provider}?session_id={session_id}&desktop=1'

        # Open browser for OAuth flow
        webbrowser.open(login_url)

        # Show waiting dialog and start polling
        self.show_social_waiting_dialog(provider)

    def show_social_waiting_dialog(self, provider):
        """
        Show a dialog while waiting for social login to complete.

        Parameters
        ----------
        provider : str
            Social login provider name for display
        """
        provider_names = {
            'google': 'Google',
            'naver': '네이버',
            'kakao': '카카오'
        }
        provider_display = provider_names.get(provider, provider)

        # Create waiting dialog
        self.waiting_dialog = CTkToplevel(self.master)
        self.waiting_dialog.title("소셜 로그인")
        self.waiting_dialog.geometry("350x180")
        self.waiting_dialog.resizable(False, False)
        self.waiting_dialog.transient(self.master)
        self.waiting_dialog.grab_set()

        # Center the dialog
        self.waiting_dialog.update_idletasks()
        x = (self.waiting_dialog.winfo_screenwidth() - 350) // 2
        y = (self.waiting_dialog.winfo_screenheight() - 180) // 2
        self.waiting_dialog.geometry(f"350x180+{x}+{y}")

        # Content
        CTkLabel(
            self.waiting_dialog,
            text=f"{provider_display}로 로그인 중...",
            font=(self.default_font, 16, "bold"),
            text_color="#333333"
        ).pack(pady=(30, 10))

        CTkLabel(
            self.waiting_dialog,
            text="브라우저에서 로그인을 완료해주세요.",
            font=(self.default_font, 12),
            text_color="#666666"
        ).pack(pady=5)

        # Progress bar
        self.social_progress = CTkProgressBar(self.waiting_dialog, width=280, height=6)
        self.social_progress.pack(pady=15)
        self.social_progress.configure(mode="indeterminate")
        self.social_progress.start()

        # Cancel button
        CTkButton(
            self.waiting_dialog,
            text="취소",
            command=self.cancel_social_login,
            width=100,
            height=32,
            fg_color="#E5E7EB",
            hover_color="#D1D5DB",
            text_color="#374151",
            font=(self.default_font, 12)
        ).pack(pady=10)

        # Flag to control polling
        self.polling_active = True

        # Start polling in background thread
        self.poll_thread = threading.Thread(target=self.poll_auth_status, daemon=True)
        self.poll_thread.start()

        # Handle dialog close
        self.waiting_dialog.protocol("WM_DELETE_WINDOW", self.cancel_social_login)

    def poll_auth_status(self):
        """
        Poll WordPress for authentication status.

        Runs in a background thread, checking every 2 seconds for up to 5 minutes.
        """
        wp_url = 'https://frontierenglish.net'
        status_url = f'{wp_url}/wp-json/autoqm-social/v1/auth-status'
        session_id = self.social_login_session_id

        max_attempts = 150  # 5 minutes at 2-second intervals
        attempt = 0

        while self.polling_active and attempt < max_attempts:
            try:
                response = requests.get(
                    status_url,
                    params={'session_id': session_id},
                    timeout=10
                )

                print(f"[Social Login] Poll attempt {attempt + 1}: status_code={response.status_code}")

                if response.status_code == 200:
                    data = response.json()
                    # Don't print full response to avoid exposing token in logs
                    print(f"[Social Login] Response status: {data.get('status')}")

                    if data.get('status') == 'complete':
                        # Login successful - got token
                        token = data.get('token')
                        if token:
                            print("[Social Login] Token received, completing login...")
                            # Schedule UI update on main thread
                            self.master.after(0, lambda t=token: self.complete_social_login(t))
                            return

                    elif data.get('status') == 'error':
                        # Login failed
                        error_msg = data.get('message', '로그인에 실패했습니다.')
                        self.master.after(0, lambda m=error_msg: self.social_login_error(m))
                        return

                    elif data.get('status') == 'needs_registration':
                        # New user needs to register
                        reg_url = data.get('registration_url', '')
                        print(f"[Social Login] Registration needed: {reg_url}")
                        # Open registration URL in browser if not already there
                        if reg_url:
                            webbrowser.open(reg_url)
                        # Continue polling - user will complete registration

                    # status == 'pending' or other - continue polling

            except requests.RequestException as e:
                print(f"[Social Login] Polling error: {e}")
                # Continue polling on network errors

            time.sleep(2)
            attempt += 1

        # Timeout reached
        if self.polling_active:
            self.master.after(0, lambda: self.social_login_error("로그인 시간이 초과되었습니다. 다시 시도해주세요."))

    def cancel_social_login(self):
        """Cancel the social login process."""
        self.polling_active = False
        if hasattr(self, 'waiting_dialog') and self.waiting_dialog:
            self.waiting_dialog.destroy()
            self.waiting_dialog = None

    def social_login_error(self, message):
        """
        Handle social login error.

        Parameters
        ----------
        message : str
            Error message to display
        """
        self.polling_active = False
        if hasattr(self, 'waiting_dialog') and self.waiting_dialog:
            self.waiting_dialog.destroy()
            self.waiting_dialog = None

        messagebox.showerror("로그인 실패", message)

    def complete_social_login(self, token):
        """
        Complete the social login process with the received token.

        Parameters
        ----------
        token : str
            JWT token from WordPress
        """
        global user_token, credits, is_logged_in

        self.polling_active = False
        if hasattr(self, 'waiting_dialog') and self.waiting_dialog:
            self.waiting_dialog.destroy()
            self.waiting_dialog = None

        # Store the token
        user_token = token
        is_logged_in = True

        # Fetch credits
        fetched_credits = LoginWindowApp.fetch_credits(token, show_error=False)
        if fetched_credits is not None:
            credits = fetched_credits
        else:
            credits = 0
            messagebox.showwarning("알림", "로그인은 성공했으나 포인트 정보를 가져오는 데 실패했습니다.")

        # Fetch user role
        user_role = self.fetch_user_role_custom(token)

        # Get username from WordPress
        username = self.fetch_username_from_token(token)

        # Check for block message (same as regular login)
        if username:
            block_message = self.get_block_message(username)
            if block_message:
                messagebox.showinfo("이용 불가 알림", block_message)
                return

        # Save social login credentials if auto_login is checked
        if self.auto_login_var.get() == '1':
            self.save_social_credentials(token, username)
            # Clear regular credentials to avoid conflict
            if os.path.exists(self.credentials_file):
                os.remove(self.credentials_file)

        # Log login to server
        try:
            log_url = f'{self.base_url}/log_login'
            log_data = {
                'username': username or 'social_user',
                'credits': credits,
                'app_version': self.currentappver,
                'login_type': 'social'
            }
            requests.post(log_url, json=log_data, timeout=5)
        except Exception as e:
            print(f"Error logging social login: {e}")

        # Update MainFrame
        if self.master and hasattr(self.master, 'frames'):
            main_frame_instance = self.master.frames.get(self.MainFrame)
            if main_frame_instance:
                main_frame_instance.is_logged_in = True
                main_frame_instance.user_token = user_token
                main_frame_instance.credits = credits
                main_frame_instance.update_credits(credits)
                if username:
                    main_frame_instance.set_username(username)
                main_frame_instance.set_user_role(user_role)
                main_frame_instance.update_run_button_state()

                role_korean = "일반회원"
                if 'subscriber' in user_role or 'um_custom_role_1' in user_role:
                    role_korean = "정회원"
                main_frame_instance.role_label.configure(text=f"{role_korean}")

        # Switch to main frame
        self.master.show_frame(self.MainFrame)

        # Show manual if needed
        data = self.master.load_notice_data_mainframe()
        dont_show_again = data.get("dont_show_again", False)
        if not dont_show_again:
            self.master.manual_window(self.master.안내메시지)

    def fetch_username_from_token(self, token):
        """
        Fetch username from WordPress using the token.

        Parameters
        ----------
        token : str
            JWT authentication token

        Returns
        -------
        str or None
            Username if successful, None otherwise
        """
        url = "https://frontierenglish.net/wp-json/wp/v2/users/me"
        headers = {"Authorization": f"Bearer {token}"}

        try:
            response = requests.get(url, headers=headers, timeout=10)
            response.raise_for_status()
            user_data = response.json()
            return user_data.get("slug") or user_data.get("name") or f"user-{user_data.get('id', 'unknown')}"
        except requests.RequestException:
            return None

    # ========== Social Login Auto-Login Methods ==========

    def save_social_credentials(self, token, username=None):
        """
        Save social login credentials (JWT token) to encrypted file.

        Parameters
        ----------
        token : str
            JWT token to save
        username : str, optional
            Username for display purposes
        """
        social_creds_file = self.credentials_file.parent / 'social_credentials.pkl'
        data = {
            'token': token,
            'username': username,
            'auto_login': True,
            'saved_at': time.time()
        }
        try:
            encrypted_data = self.cipher_suite.encrypt(json.dumps(data).encode('utf-8'))
            with open(social_creds_file, 'wb') as file:
                file.write(encrypted_data)
            print(f"[Social Login] Credentials saved for auto-login")
        except Exception as e:
            print(f"[Social Login] Failed to save credentials: {e}")

    def load_social_credentials(self):
        """
        Load saved social login credentials from encrypted file.

        Returns
        -------
        dict or None
            Dictionary containing 'token', 'username', 'auto_login' keys,
            or None if no credentials exist or decryption fails
        """
        social_creds_file = self.credentials_file.parent / 'social_credentials.pkl'
        if social_creds_file.exists():
            try:
                with open(social_creds_file, 'rb') as file:
                    encrypted_data = file.read()
                    decrypted_data = self.cipher_suite.decrypt(encrypted_data)
                    data = json.loads(decrypted_data.decode('utf-8'))
                    return data
            except (cryptography.fernet.InvalidToken, json.JSONDecodeError) as e:
                print(f"[Social Login] Error decrypting credentials: {e}")
                social_creds_file.unlink(missing_ok=True)
            except Exception as e:
                print(f"[Social Login] Unexpected error loading credentials: {e}")
        return None

    def clear_social_credentials(self):
        """Clear saved social login credentials."""
        social_creds_file = self.credentials_file.parent / 'social_credentials.pkl'
        if social_creds_file.exists():
            social_creds_file.unlink(missing_ok=True)
            print("[Social Login] Credentials cleared")

    def validate_token(self, token):
        """
        Validate if a JWT token is still valid by calling WordPress API.

        Parameters
        ----------
        token : str
            JWT token to validate

        Returns
        -------
        bool or None
            True if token is valid, False if token is invalid/expired,
            None if network error (can't determine)
        """
        url = "https://frontierenglish.net/wp-json/wp/v2/users/me"
        headers = {"Authorization": f"Bearer {token}"}

        try:
            response = requests.get(url, headers=headers, timeout=10)
            if response.status_code == 200:
                return True
            elif response.status_code in (401, 403):
                # Token is definitely invalid/expired
                return False
            else:
                # Other error - can't determine
                return None
        except requests.RequestException as e:
            print(f"[Social Login] Network error during token validation: {e}")
            # Network error - can't determine if token is valid
            return None

    def complete_social_login_from_saved(self, token):
        """
        Complete social login using a saved token (for auto-login).

        Parameters
        ----------
        token : str
            Saved JWT token
        """
        global user_token, credits, is_logged_in

        # Get username first for block check
        username = self.fetch_username_from_token(token)

        # Check for block message
        if username:
            block_message = self.get_block_message(username)
            if block_message:
                messagebox.showinfo("이용 불가 알림", block_message)
                self.clear_social_credentials()  # Clear so they can't auto-login again
                return

        user_token = token
        is_logged_in = True

        # Fetch credits
        fetched_credits = LoginWindowApp.fetch_credits(token, show_error=False)
        if fetched_credits is not None:
            credits = fetched_credits
        else:
            credits = 0

        # Fetch user role
        user_role = self.fetch_user_role_custom(token)

        # Log login to server
        try:
            log_url = f'{self.base_url}/log_login'
            log_data = {
                'username': username or 'social_user',
                'credits': credits,
                'app_version': self.currentappver,
                'login_type': 'social_auto'
            }
            requests.post(log_url, json=log_data, timeout=5)
        except Exception as e:
            print(f"[Social Login] Error logging auto-login: {e}")

        # Update MainFrame
        if self.master and hasattr(self.master, 'frames'):
            main_frame_instance = self.master.frames.get(self.MainFrame)
            if main_frame_instance:
                main_frame_instance.is_logged_in = True
                main_frame_instance.user_token = user_token
                main_frame_instance.credits = credits
                main_frame_instance.update_credits(credits)
                if username:
                    main_frame_instance.set_username(username)
                main_frame_instance.set_user_role(user_role)
                main_frame_instance.update_run_button_state()

                role_korean = "일반회원"
                if 'subscriber' in user_role or 'um_custom_role_1' in user_role:
                    role_korean = "정회원"
                main_frame_instance.role_label.configure(text=f"{role_korean}")

        # Switch to main frame
        self.master.show_frame(self.MainFrame)

        # Show manual if needed
        data = self.master.load_notice_data_mainframe()
        dont_show_again = data.get("dont_show_again", False)
        if not dont_show_again:
            self.master.manual_window(self.master.안내메시지)
