"""
Price Table Module
Displays the price table for different question types in a dialog window.
"""

import requests
from tkinter import messagebox
from customtkinter import CTkToplevel, CTkFrame, CTkLabel, CTkButton, CTkScrollableFrame


# Hard-coded group definitions from autoQM.py
문제rows = ["어휘1단계", "어휘2단계", "어휘3단계", "내용일치(한)", "내용일치", "밑줄의미", "함축의미", "무관한문장", "빈칸추론", "추론", "추론불가", "순서", "삽입", "주제", "제목", "요지", "연결어", "요약", "어법1단계", "어법2단계", "빈칸암기", "어휘종합", "내용종합", "영작1", "영작2", "주제영작", "동형문제"]
자료rows = ["동반의어", "동반의어문제1", "동반의어문제2", "영영정의", "영영정의문제", "요약문", "커스텀", "음성파일"]
교재rows = ["지문분석", "한글해석", "직독직해", "단어정리", "한줄해석연습", "한줄영작연습"]


def show_price_table(parent, base_url, default_font):
    """
    Display the price table in a new dialog window.

    Args:
        parent: Parent window (MainFrame instance)
        base_url: Flask server base URL
        default_font: Font to use for labels
    """
    try:
        # Fetch price data from Flask endpoint
        response = requests.get(f'{base_url}/get_price_list', timeout=10)
        response.raise_for_status()
        data = response.json()

        base_credit_map = data.get('base_credit_map', {})
        additional_info = data.get('additional_info', [])

        # Create dialog window
        dialog = CTkToplevel(parent)
        dialog.title("유형별 가격표")
        dialog.geometry("500x750")
        dialog.grab_set()  # Make the dialog modal
        dialog.focus_set()

        # Create main container with scrollbar
        main_frame = CTkFrame(dialog, fg_color="white")
        main_frame.pack(fill="both", expand=True, padx=10, pady=10)

        # Title label
        title_label = CTkLabel(
            main_frame,
            text="유형별 가격표 (포인트)",
            font=(default_font, 18, 'bold'),
            text_color="black"
        )
        title_label.pack(pady=(10, 0))

        notice_label = CTkLabel(
            main_frame,
            text="가격은 상황에 따라 변동될 수 있습니다.",
            font=(default_font, 11),
            text_color="black"
        )
        notice_label.pack(pady=0)


        # Create scrollable frame for table
        scrollable_frame = CTkScrollableFrame(
            main_frame,
            fg_color="white",
            height=400
        )
        scrollable_frame.pack(fill="both", expand=True, padx=10, pady=5)

        # Create table header
        header_frame = CTkFrame(scrollable_frame, fg_color="#FEF9E0")
        header_frame.pack(fill="x", pady=(0, 2))

        CTkLabel(
            header_frame,
            text="유형",
            font=(default_font, 13, 'bold'),
            text_color="black",
            width=250
        ).pack(side="left", padx=10, pady=5)

        CTkLabel(
            header_frame,
            text="Normal",
            font=(default_font, 13, 'bold'),
            text_color="black",
            width=80
        ).pack(side="left", padx=10, pady=5)

        CTkLabel(
            header_frame,
            text="Hard",
            font=(default_font, 13, 'bold'),
            text_color="black",
            width=80
        ).pack(side="left", padx=10, pady=5)

        # Define group colors
        group_colors = {
            "문제": "#E3F2FD",  # Light blue for question types
            "자료": "#E8F5E9",  # Light green for material types
            "교재": "#F3E5F5",  # Light purple for textbook types
        }

        # Helper function to determine group
        def get_group(type_name):
            if type_name in 문제rows:
                return "문제"
            elif type_name in 자료rows:
                return "자료"
            elif type_name in 교재rows:
                return "교재"
            else:
                return None

        # Helper function to extract prices for a type
        def get_prices_for_type(type_name):
            normal_key = f"{type_name}_Normal"
            hard_key = f"{type_name}_Hard"

            # Use '-' only when key doesn't exist, keep 0 as 0
            normal_price = base_credit_map[normal_key] if normal_key in base_credit_map else '-'
            hard_price = base_credit_map[hard_key] if hard_key in base_credit_map else '-'

            return normal_price, hard_price

        # Create table rows in exact order: 문제rows → 자료rows → 교재rows
        all_rows = 문제rows + 자료rows + 교재rows

        for type_name in all_rows:
            # Determine background color based on group
            group = get_group(type_name)
            if group is None:
                continue  # Skip types not in any group

            bg_color = group_colors.get(group, "#FFFFFF")
            normal_price, hard_price = get_prices_for_type(type_name)

            row_frame = CTkFrame(scrollable_frame, fg_color=bg_color)
            row_frame.pack(fill="x", pady=1)

            CTkLabel(
                row_frame,
                text=type_name,
                font=(default_font, 12),
                text_color="black",
                width=250,
                anchor="center",
                fg_color=bg_color
            ).pack(side="left", padx=10, pady=2)

            CTkLabel(
                row_frame,
                text=str(normal_price),
                font=(default_font, 12),
                text_color="black",
                width=80,
                fg_color=bg_color
            ).pack(side="left", padx=10, pady=2)

            CTkLabel(
                row_frame,
                text=str(hard_price),
                font=(default_font, 12),
                text_color="black",
                width=80,
                fg_color=bg_color
            ).pack(side="left", padx=10, pady=2)

        # Add separator
        separator = CTkFrame(main_frame, fg_color="gray", height=2)
        separator.pack(fill="x", padx=10, pady=10)

        # Create additional information section
        info_frame = CTkFrame(main_frame, fg_color="white")
        info_frame.pack(fill="x", padx=10, pady=0)

        info_title = CTkLabel(
            info_frame,
            text="추가 정보",
            font=(default_font, 14, 'bold'),
            text_color="black",
            anchor="w"
        )
        info_title.pack(fill="x", padx=5, pady=(5, 5))

        for info in additional_info:
            info_label = CTkLabel(
                info_frame,
                text=f"• {info}",
                font=(default_font, 11),
                text_color="black",
                anchor="w",
                wraplength=450,
                justify="left"
            )
            info_label.pack(fill="x", padx=5, pady=1)

        # Close button
        button_frame = CTkFrame(main_frame, fg_color="white")
        button_frame.pack(fill="x", pady=10)

        close_button = CTkButton(
            button_frame,
            text="닫기",
            text_color="black",
            width=100,
            height=30,
            command=dialog.destroy,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            border_width=2,
            border_color="gray"
        )
        close_button.pack(pady=5)

        # Bind Escape key to close dialog
        dialog.bind("<Escape>", lambda event: dialog.destroy())

    except requests.exceptions.RequestException as e:
        messagebox.showerror(
            "오류",
            f"가격 정보를 불러오는데 실패했습니다.\n서버에 연결할 수 없습니다.\n\n에러: {str(e)}"
        )
    except Exception as e:
        messagebox.showerror(
            "오류",
            f"가격표를 표시하는 중 오류가 발생했습니다.\n\n에러: {str(e)}"
        )
