# -*- coding: utf-8 -*-
"""
Result Processing Functions for Flask API Responses

These functions parse and format question generation results from the Flask backend.
Each function corresponds to a specific question type (e.g., 어휘, 어법, 내용일치).

Original location: autoQM.py lines 15886-16810
Extracted: 2025-01-21
"""

import re
import os
import json
import logging
from pathlib import Path
from modules.nlp_utils import scramble_sentences

# Initialize logger
app_logger = logging.getLogger("app")


######################## result가공 functions starts here ##########################

def result가공_어휘12단계(result):
    print("어휘1&2가공 시작")
    question_part = ""
    answer_part = ""
    explanation_part = ""
    marker = "[전문해석]"

    question_part, sep, rest = result.partition(marker)

    if sep:
        explanation_part = (sep + rest).strip()
        question_part = question_part.strip()
    else:
        explanation_part = ""
        question_part = result.strip()

    return question_part, answer_part, explanation_part


def result가공_어법1단계(result):
    print("어법1가공 시작")
    result_modified = result.replace('❮ To / For ❯', 'To').replace('❮ to / for ❯', 'to').replace('❮ to the / to be the ❯', 'to the').replace('❮ to / toing ❯', 'to').replace('❮ to an / aning ❯', 'to an')

    question_part = ""
    answer_part = ""
    explanation_part = ""
    marker = "[전문해석]"

    question_part, sep, rest = result.partition(marker)

    if sep:
        explanation_part = (sep + rest).strip()
        question_part = question_part.strip()
    else:
        explanation_part = ""
        question_part = result.strip()

    return question_part, answer_part, explanation_part



def result가공_어법2단계(result, sheet_type):
    print("어법2가공 시작")

    # Check for error cases more robustly (strip whitespace, check prefix)
    if result.strip().startswith("Error") or "Answer:" not in result:
        question_content = "Error" if result.strip().startswith("Error") else result
        answer_content = ""
        explanation_content = ""
        return question_content, answer_content, explanation_content

    else:
        if "_Normal" in sheet_type:
            question_part, rest = result.split('Answer:', 1)
            question_part = question_part.strip()
            print(f"[DEBUG 어법2가공] rest after split on Answer: '{rest[:200]}...'")

            # Extract all circled numbers from answer line (before Original form:)
            answer_line_match = re.search(r'\s*(.*?)(?=\nOriginal form:|Original form:)', rest, re.DOTALL)
            answer_line = answer_line_match.group(1).strip() if answer_line_match else ""
            answer_numbers = re.findall(r'[①②③④⑤]', answer_line)
            print(f"[DEBUG 어법2가공] answer_line: '{answer_line}', answer_numbers: {answer_numbers}")

            # Extract original form section first to check for multi-line format
            original_form_match = re.search(r'Original form:(.*?)$', rest, re.DOTALL)
            original_form_full = original_form_match.group(1).strip() if original_form_match else ""

            # Count circled numbers in the Original form section
            original_form_numbers = re.findall(r'[①②③④⑤]', original_form_full)
            print(f"[DEBUG 어법2가공] original_form_numbers in Original form: {original_form_numbers}")

            # Use multi-answer parsing if:
            # 1. Multiple answers in answer line, OR
            # 2. Single answer but multiple numbered items in Original form (e.g., answer ⑤ but ①②③④ in Original form)
            use_multi_parsing = len(answer_numbers) > 1 or (len(answer_numbers) == 1 and len(original_form_numbers) > 1)
            print(f"[DEBUG 어법2가공] use_multi_parsing: {use_multi_parsing}")

            if use_multi_parsing:
                print(f"[DEBUG 어법2가공] Multi-answer mode")
                # Multi-answer mode: use Hard-style parsing (multi-line Original form)
                answer_part = ''.join(answer_numbers)
                print(f"[DEBUG 어법2가공] answer_part: {answer_part}")
                # Split the original form section into corrections and additional content
                translation_match = re.search(r'\n\n(\[전문해석\].*)', original_form_full, re.DOTALL)
                if translation_match:
                    print(f"[DEBUG 어법2가공] translation_match: {translation_match}")
                    original_form = original_form_full[:translation_match.start()].strip()
                    additional_content = translation_match.group(1).strip()
                    print(f"[DEBUG 어법2가공] additional_content: {additional_content}")
                else:
                    print(f"[DEBUG 어법2가공] no translation_match")
                    original_form = original_form_full
                    additional_content = ""
                    print(f"[DEBUG 어법2가공] additional_content: {additional_content}")
                # Process explanation part (each line: "② original_form")
                explanation_parts = []
                print(f"[DEBUG 어법2가공] original_form: {original_form}")
                for line in original_form.split('\n'):
                    if line.strip():
                        parts = line.split(None, 1)
                        print(f"[DEBUG 어법2가공] parts: {parts}")
                        if len(parts) >= 2:
                            number = parts[0].strip()
                            original = parts[1].strip()
                            print(f"[DEBUG 어법2가공] number: {number}, original: {original}")

                            # Find the corresponding bracketed text in the question part
                            bracket_match = re.search(rf'{re.escape(number)}\[(.*?)\]', question_part)
                            bracketed_text = bracket_match.group(1) if bracket_match else ""
                            print(f"[DEBUG 어법2가공] bracketed_text: {bracketed_text}")
                            explanation_parts.append(f"{number} {bracketed_text} > {original}")
                            print(f"[DEBUG 어법2가공] explanation_parts: {explanation_parts}")
                explanation_part = '\n'.join(explanation_parts)
                print(f"[DEBUG 어법2가공] explanation_part: {explanation_part}")
                # Append additional content with two newlines
                if additional_content:
                    explanation_part += f"\n\n{additional_content}"
                    print(f"[DEBUG 어법2가공] explanation_part with additional_content: {explanation_part}")
            else:
                # Single-answer mode (mode = 1): use original inline parsing
                answer_part = answer_numbers[0] if answer_numbers else ""
                print(f"[DEBUG 어법2가공] Single-answer mode, answer_part: {answer_part}")

                # Extracting everything after "Original form:" (including translations)
                # Handle both old format "Original form: are given" and new format "Original form:\n② are given"
                # Also handle Windows line endings (\r\n)
                original_form_section = re.search(r"Original form:[\s](.*)", rest, re.DOTALL)
                print(f"[DEBUG 어법2가공] original_form_section match: {original_form_section}")
                original_form_text = original_form_section.group(1).strip() if original_form_section else ""
                print(f"[DEBUG 어법2가공] original_form_text: '{original_form_text}'")

                # Split to get the correction and additional content
                lines = original_form_text.split('\n', 1)
                print(f"[DEBUG 어법2가공] lines after split: {lines}")
                original_expression = lines[0].strip()
                print(f"[DEBUG 어법2가공] original_expression before sub: '{original_expression}'")

                # Strip leading circled number if present (new format: "② are given" -> "are given")
                original_expression = re.sub(r'^[①②③④⑤]\s*', '', original_expression)
                print(f"[DEBUG 어법2가공] original_expression after sub: '{original_expression}'")

                additional_content = lines[1].strip() if len(lines) > 1 else ""

                # Finding the phrase within the brackets of the answer number
                answer_phrase_match = re.search(rf'{answer_part}\[(.*?)\]', question_part)
                answer_phrase = answer_phrase_match.group(1) if answer_phrase_match else ""
                print(f"[DEBUG 어법2가공] answer_phrase: '{answer_phrase}'")

                # Combining with two newlines above additional content
                # Include the circled number prefix (e.g., "③ is affected > are affected")
                explanation_part = f"{answer_part} {answer_phrase} > {original_expression}\n\n{additional_content}"
                print(f"[DEBUG 어법2가공] Final explanation_part: '{explanation_part}'")

        elif "_Hard" in sheet_type:
            # Extract question part
            question_part, rest = result.split('Answer:', 1)
            question_part = question_part.strip()

            # Extract answer part (all numbers after "Answer: ")
            answer_match = re.search(r'\s*(.*?)(?=\nOriginal form:|$)', rest, re.DOTALL)
            answer_part = answer_match.group(1).strip() if answer_match else ""

            # Extract original form section (everything after "Original form:")
            original_form_match = re.search(r'Original form:(.*?)$', rest, re.DOTALL)
            original_form_full = original_form_match.group(1).strip() if original_form_match else ""

            # Split the original form section into corrections and additional content
            # Find where [전문해석] or similar sections start
            translation_match = re.search(r'\n\n(\[전문해석\].*)', original_form_full, re.DOTALL)
            if translation_match:
                original_form = original_form_full[:translation_match.start()].strip()
                additional_content = translation_match.group(1).strip()
            else:
                original_form = original_form_full
                additional_content = ""

            # Process explanation part
            explanation_parts = []
            for line in original_form.split('\n'):
                if line.strip():
                    number, original = line.split(None, 1)
                    number = number.strip()
                    original = original.strip()
                    
                    # Find the corresponding bracketed text in the question part
                    bracket_match = re.search(rf'{number}\[(.*?)\]', question_part)
                    bracketed_text = bracket_match.group(1) if bracket_match else ""
                    
                    explanation_parts.append(f"{number} {bracketed_text} > {original}")

            explanation_part = '\n'.join(explanation_parts)
            
            # Append additional content with two newlines
            if additional_content:
                explanation_part += f"\n\n{additional_content}"

        answer_part = answer_part.replace(" ", "")

        return question_part, answer_part, explanation_part



def result가공_어휘3단계(result):
    if result == "Error":
        question_content = "Error"
        answer_content = ""
        explanation_content = ""
        return question_content, answer_content, explanation_content
    else:
        step_1_start = result.find("Step 1:") + len("Step 1:")
        step_2_start = result.find("Step 2:")
        answer_start = result.find("Answer:", step_2_start) + len("Answer:")
        explanation_start = result.find("Explanation:", answer_start) + len("Explanation:")

        question_content = result[step_1_start:step_2_start].strip()

        answer_section = result[answer_start:explanation_start].strip()
        answer_numbers = re.findall(r'(①|②|③|④|⑤|⑥|⑦|⑧|⑨|⑩|⑪|⑫|⑬|⑭|⑮|⑯|⑰|⑱|⑲|⑳)', answer_section)
        answer_part = ''.join(answer_numbers)

        explanation_content = result[explanation_start:].strip()
        explanation_content = answer_section + "\n" + explanation_content

        return question_content, answer_part, explanation_content
    
    
    
def result가공_내용일치(result): #내용일치 가공
    #print("result가공_내용일치 called")
    # Check for error cases more robustly (strip whitespace, check prefix, check required delimiter)
    if result.strip().startswith("Error") or "Answers:" not in result:
        question_content = "Error" if result.strip().startswith("Error") else result
        answer_content = ""
        explanation_content = ""
        return question_content, answer_content, explanation_content

    else:
        questions_part, answers_part = result.split('Answers:', 1)
        questions_part = questions_part.split('Questions:', 1)[1].strip()
    
        # 'explanation_part' will contain the text following 'Answers:'
        explanation_part = answers_part.strip()
        answer_part_only = answers_part.split('[해석]')[0].strip()



        # Process 'answers_part' to create 'answer_sheet'
        answers_lines = answer_part_only.split('\n')
        answer_sheet = []
        for line in answers_lines:
            if line.startswith('('):  # Assuming each answer starts with '('
                # Extract the number and the True/False value
                number = line.split(')')[0] + ')'  # Get the parenthesized number
                truth_value = 'T' if 'True.' in line else 'F'
                answer_sheet.append(f"{number} {truth_value}")
        
        # Join 'answer_sheet' list into a string
        answer_sheet = '\n'.join(answer_sheet)
        answer_sheet = answer_sheet.strip()

        ##explanation_part = explanation_part.replace("[해석]", "\n[해석]").replace("[어휘]", "\n[어휘]").replace("[전문해석]", "\n[전문해석]").replace("[지문어휘]", "\n[지문어휘]")

        #print(f"\n\nexplanation_part:\n{explanation_part}")

        return questions_part, answer_sheet, explanation_part



def result가공_내용일치_실모(result): #내용일치 가공
    # Check for error cases more robustly (strip whitespace, check prefix, check required delimiter)
    if result.strip().startswith("Error") or "Answers:" not in result:
        question_content = "Error" if result.strip().startswith("Error") else result
        answer_content = ""
        explanation_content = ""
        return question_content, answer_content, explanation_content

    else:
        questions_part, answers_part = result.split('Answers:', 1)
        questions_part = questions_part.split('Questions:', 1)[1].strip()
        questions_part = questions_part.replace("❮ T / F ❯", "")

        # 'explanation_part' will contain the text following 'Answers:'
        explanation_part = answers_part.strip()
        answer_part_only = answers_part.split('[해석]')[0].strip()


        # Process 'answers_part' to create 'answer_sheet'
        answers_lines = answer_part_only.split('\n')
        answer_sheet = []
        for line in answers_lines:
            if line.startswith('('):  # Assuming each answer starts with '('
                # Extract the number and the True/False value
                number = line.split(')')[0] + ')'  # Get the parenthesized number
                truth_value = 'T' if 'True.' in line else 'F'
                answer_sheet.append(f"{number} {truth_value}")
        
        # Join 'answer_sheet' list into a string
        answer_sheet = '\n'.join(answer_sheet)
        answer_sheet = answer_sheet.strip()

        #explanation_part = explanation_part.replace("[해석]", "\n[해석]").replace("[어휘]", "\n[어휘]").replace("[전문해석]", "\n[전문해석]").replace("[지문어휘]", "\n[지문어휘]")


        return questions_part, answer_sheet, explanation_part




def result가공_옵션anal말고exp(result): #밑줄의미 가공
    #print("result가공_옵션anal말고exp" + result + "\n")
    print("result가공_옵션anal말고exp 가공시작")
    if result == "Error":
        question_content = "Error"
        answer_content = ""
        explanation_content = ""
        return question_content, answer_content, explanation_content

    else:
        ################
        # Splitting at 'Answer:' to separate question_part and the rest
        if 'Answer:' in result:
            question_part, rest = [part.strip() for part in result.split('Answer:', 1)]
        else:
            # Handle the case where 'Answer:' is not found
            question_part = result
            rest = ""

        question_part = question_part.strip()

        # Using regular expression to find the circled number for answer_part
        answer_match = re.search(r'Answer:\s*(①|②|③|④|⑤)', result)
        answer_part = answer_match.group(1) if answer_match else ""

        # Splitting at 'Explanation:' to get explanation_part
        explanation_part = rest.split('Explanation:', 1)[1].strip() if 'Explanation:' in rest else ""

        #explanation_part = explanation_part.replace("[해석]", "\n[해석]").replace("[어휘]", "\n[어휘]").replace("[전문해석]", "\n[전문해석]").replace("[지문어휘]", "\n[지문어휘]")

        ################
        #print(question_part)
        #print(answer_part)
        #print(explanation_part)
        return question_part, answer_part, explanation_part






def result가공_빈칸추론(result): #빈칸추론 가공
    print("result가공_빈칸추론")
    if result == "Error":
        question_content = "Error"
        answer_content = ""
        explanation_content = ""
        return question_content, answer_content, explanation_content

    else:
        ################
        if 'Answer:' in result:
            question_part, rest = [part.strip() for part in result.split('Answer:', 1)]
        else:
            return "error", "", ""

        answer_match = re.search(r'Answer:\s*(①|②|③|④|⑤)', result)
        answer_part = answer_match.group(1) if answer_match else ""

        # Assuming 'result' contains your text
        original_expression_match = re.search(r"<(Original expression: .*?)>", result)
        # Extracting the matched part of the text, including "Original expression: "
        original_expression = original_expression_match.group(1) if original_expression_match else ""

        # Assuming 'result' contains the text provided in your example
        explanation_match = re.search(r"1\. Option Analysis:(.*)", result, re.DOTALL)
        # Extracting the matched part of the text
        explanation_section = explanation_match.group(1).strip() if explanation_match else ""

        explanation_parts = []
        if original_expression:
            explanation_parts.append(original_expression)
        if explanation_section:
            explanation_parts.append(explanation_section)

        explanation_part = "\n".join(explanation_parts)
        if not explanation_part and rest:
            explanation_part = rest.strip()

        ##explanation_part = explanation_part.replace("[해석]", "\n[해석]").replace("[어휘]", "\n[어휘]").replace("[전문해석]", "\n[전문해석]").replace("[지문어휘]", "\n[지문어휘]")

        ################
        return question_part, answer_part, explanation_part




def result가공_지문추가(passage, result): #'Answer' 전까지 문제추출 + Answer 정답 + 1. Option Analysis /// 추론, 추론불가, 삽입
    print("result가공_지문추가 시작")
    if result == "Error":
        question_content = "Error"
        answer_content = ""
        explanation_content = ""
        return question_content, answer_content, explanation_content

    else:
        ################
        # Extract 'question_part'
        question_part_match = re.search(r"(.*?)Answer:", result, re.DOTALL)  # Non-greedy to match FIRST "Answer:"
        question_part = question_part_match.group(1).strip() if question_part_match else ""
        # 지문추가
        question_part = passage + "\n" + question_part

        # Extract 'answer_part'
        answer_part_match = re.search(r"Answer: (\s*①|②|③|④|⑤)", result)
        answer_part = answer_part_match.group(1).strip() if answer_part_match else ""

        # Extract 'explanation_part'
        explanation_part_match = re.search(r"1\. Option Analysis:(.*)", result, re.DOTALL)
        explanation_part = explanation_part_match.group(1).strip() if explanation_part_match else ""
        ################

        #explanation_part = explanation_part.replace("[해석]", "\n[해석]").replace("[어휘]", "\n[어휘]").replace("[전문해석]", "\n[전문해석]").replace("[지문어휘]", "\n[지문어휘]")


        return question_part, answer_part, explanation_part




def result가공_3단분리(result): #가져온거에서 지문/문제/정답해설 다 쪼개넣기 
    print("result가공_3단분리 시작")
    if result == "Error":
        question_content = "Error"
        answer_content = ""
        explanation_content = ""
        return question_content, answer_content, explanation_content

    else:
        ################
        # Extract 'question_part'
        question_part_match = re.search(r"(.*?)Answer:", result, re.DOTALL)  # Non-greedy to match FIRST "Answer:"
        question_part = question_part_match.group(1).strip() if question_part_match else ""

        # Extract 'answer_part'
        answer_part_match = re.search(r"Answer: (\s*①|②|③|④|⑤)", result)
        answer_part = answer_part_match.group(1).strip() if answer_part_match else ""

        # Extract 'explanation_part'
        explanation_part_match = re.search(r"1\. Option Analysis:(.*)", result, re.DOTALL)
        explanation_part = explanation_part_match.group(1).strip() if explanation_part_match else ""

        ################

        #explanation_part = explanation_part.replace("[해석]", "\n[해석]").replace("[어휘]", "\n[어휘]").replace("[전문해석]", "\n[전문해석]").replace("[지문어휘]", "\n[지문어휘]")

        return question_part, answer_part, explanation_part




def result가공_문제만(result): #주제제목요지 등
    print("result가공_지문없이 문제만 시작")
    if result == "Error":
        question_content = "Error"
        answer_content = ""
        explanation_content = ""
        return question_content, answer_content, explanation_content

    else:
        ################
        # Extract 'question_part'
        question_part_match = re.search(r"(.*?)Answer:", result, re.DOTALL)  # Non-greedy to match FIRST "Answer:"
        question_part = question_part_match.group(1).strip() if question_part_match else ""

        # Extract 'answer_part'
        #answer_part_match = re.search(r"Answer: (\s*①|②|③|④|⑤)", result) #정답 1개만 추출
        answer_part_match = re.search(r"Answer:\s*([①②③④⑤]+)", result) #정답 2개 이상도 추출
        answer_part = answer_part_match.group(1).strip() if answer_part_match else ""

        # Extract 'explanation_part'
        explanation_part_match = re.search(r"1\. Option Analysis:(.*)", result, re.DOTALL)
        explanation_part = explanation_part_match.group(1).strip() if explanation_part_match else ""
        ################

        #print(f"explanation_part:\n{explanation_part}")

        #explanation_part = explanation_part.replace("[해석]", "\n[해석]").replace("[어휘]", "\n[어휘]").replace("[전문해석]", "\n[전문해석]").replace("[지문어휘]", "\n[지문어휘]")

        return question_part, answer_part, explanation_part



def result가공_순서(result): #순서
    print("result가공_순서 시작 (순서)")
    if result == "Error":
        app_logger.info("순서 문제 출제 중 오류가 발생했습니다.")
        question_content = "Error"
        answer_content = ""
        explanation_content = ""
        # Return in order: question, answer, explanation (for columns 3, 4, 5)
        return [question_content, answer_content, explanation_content]
    elif "문제 미출제" in result:
        app_logger.info("문장 수 부족으로 순서 문제가 생성되지 않았습니다. (포인트 차감 없음)")
        question_content = "문제 미출제 (문장 수 부족)"
        answer_content = ""
        explanation_content = ""
        # Return in order: question, answer, explanation (for columns 3, 4, 5)
        return [question_content, answer_content, explanation_content]


    else:
        marker = "[전문해석]"
        question_block, sep, explanation_block = result.partition(marker)

        if sep:
            explanation_part = (sep + explanation_block).strip()
            main_part = question_block.strip()
        else:
            explanation_part = ""
            main_part = result.strip()

        answer_header_pattern = re.compile(r"Answer\s*[:：]\s*(.+)", re.IGNORECASE | re.DOTALL)
        answer_header_match = answer_header_pattern.search(main_part)

        if answer_header_match:
            question_part = main_part[:answer_header_match.start()].strip()
            raw_answer_section = answer_header_match.group(1).strip()
        else:
            question_part = main_part.strip()
            raw_answer_section = ""

        answer_part = ""
        if raw_answer_section:
            first_line = ""
            for line in raw_answer_section.splitlines():
                stripped = line.strip()
                if stripped:
                    first_line = stripped
                    break

            answer_token_pattern = re.compile(
                r"(①|②|③|④|⑤|[1-5]|"
                r"\(\w\)\s*-\s*\(\w\)\s*-\s*\(\w\)\s*(?:-\s*\(\w\))?)"
            )
            token_match = answer_token_pattern.search(first_line)
            answer_part = token_match.group(1).strip() if token_match else first_line

        if not explanation_part:
            explanation_match = re.search(r"1\.\s*Option Analysis:(.*)", main_part, re.DOTALL)
            explanation_part = explanation_match.group(1).strip() if explanation_match else ""

        return [question_part, answer_part, explanation_part]




def result가공_삽입(result):
    print("result가공_삽입 시작")
    if result == "Error":
        question_content = "Error"
        answer_content = ""
        explanation_content = ""
        # Return in order: question, answer, explanation (for columns 3, 4, 5)
        return [question_content, answer_content, explanation_content]


    else:
        marker = "[전문해석]"
        question_block, sep, explanation_block = result.partition(marker)

        if sep:
            explanation_part = (sep + explanation_block).strip()
            main_part = question_block.strip()
        else:
            explanation_part = ""
            main_part = result.strip()

        question_part_match = re.search(r"(.*?)Answer:", main_part, re.DOTALL)  # Non-greedy to match FIRST "Answer:"
        question_part = question_part_match.group(1).strip() if question_part_match else ""

        answer_part_regex = r"Answer:\s*(①|②|③|④|⑤|\(\w\) - \(\w\) - \(\w\) - \(\w\)|\(\w\) - \(\w\) - \(\w\))"
        answer_part_match = re.search(answer_part_regex, main_part)
        answer_part = answer_part_match.group(1).strip() if answer_part_match else ""

        if not explanation_part:
            explanation_match = re.search(r"1\. Option Analysis:(.*)", main_part, re.DOTALL)
            explanation_part = explanation_match.group(1).strip() if explanation_match else ""

        return [question_part, answer_part, explanation_part]

def result가공_연결어(result): 
    print("result가공_연결어 시작")
    if result == "Error":
        question_content = "Error"
        answer_content = ""
        explanation_content = ""
        return question_content, answer_content, explanation_content
    elif result == "연결어 없음":
        question_content = "연결어 없음"
        answer_content = ""
        explanation_content = ""
        
        return question_content, answer_content, explanation_content


    else:
        ################
        # Extract 'question_part'
        question_part_match = re.search(r"(.*?)Answer:", result, re.DOTALL)  # Non-greedy to match FIRST "Answer:"
        question_part = question_part_match.group(1).strip() if question_part_match else ""

        # Extract 'answer_part'
        answer_part_match = re.search(r"Answer:\s*(.+?)\s*Explanation:", result, re.DOTALL)
        # Extracting the matched content
        answer_part = answer_part_match.group(1).strip() if answer_part_match else ""

        # Extract 'explanation_part'
        explanation_part_match = re.search(r"Explanation:(.*)", result, re.DOTALL)
        explanation_part = explanation_part_match.group(1).strip() if explanation_part_match else ""
        ################
        return question_part, answer_part, explanation_part



def result가공_영작1(result):
    print("result가공_영작1 시작")
    print(f"result: {result}")
    count = result.count('[[')
    if count == 1:

        # 1. Fetch the data from "해석: " to the end of that line and put it into 'translation'.
        translation_match = re.search(r'해석: (.*?)(?=\n|$)', result)
        translation = translation_match.group(1).strip() if translation_match else ''

        # 2. Fetch the data between [[ ]] and put it into 'answer' (the original English text).
        answer_match = re.search(r'\[\[(.*?)\]\]', result, re.DOTALL)
        answer = answer_match.group(1) if answer_match else ''

        # 3. Extract everything after the blank line following "해석:" (includes [전문해석] and [지문어휘])
        additional_content_match = re.search(r'해석:.*?\n\n(.*)', result, re.DOTALL)
        additional_content = additional_content_match.group(1).strip() if additional_content_match else ''

        # 4. Replace the content between [[ ]] with 'translation', keeping the [[ and ]]
        result = re.sub(r'\[\[.*?\]\]', f'[[{translation}]]', result, flags=re.DOTALL)

        # 5. Remove everything from '해석:' onwards
        result = re.sub(r'\n*해석:.*', '', result, flags=re.DOTALL).strip()

        # Check if blank mode is enabled (checkbox checked)
        blank_mode_enabled = False
        app_data_dir = Path.home() / ".my_application_data"
        make_questions_options_file_path = os.path.join(app_data_dir, "make_questions_options.json")
        if os.path.exists(make_questions_options_file_path):
            try:
                with open(make_questions_options_file_path, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    blank_mode_enabled = data.get("영작1_blank_mode", 0) == 1
            except:
                pass

        if blank_mode_enabled:
            # Replace [[ ]] with underscores
            result = re.sub(r'\[\[.*?\]\]', '___________________', result)
            question_line = "다음의 빈칸에 들어갈 말을 영작하고자 한다. [보기]에 주어진 단어를 올바르게 배열하시오. (어형변화 불가, 단어추가 불가)"
        else:
            question_line = "다음의 밑줄 친 부분을 영작하고자 한다. [보기]에 주어진 단어를 올바르게 배열하시오. (어형변화 불가, 단어추가 불가)"

        question = "[보기] " + scramble_sentences(answer)
        result = question_line + "\n" + result + "\n\n" + question

        # 6. Build explanation_part: English answer + two newlines + additional content
        explanation_part = f"정답: {answer}\n\n{additional_content}"

        if not blank_mode_enabled:
            result = result.replace('.]].', ']].')

    elif count > 1:

        print(f"result: {result}")
        # Numbers (1), (2), etc. are already present in the result
        # No need to add them again
        print(f"수정후 result: {result}")




        # 1. Fetch translations from "해석: " section - can be multiple lines with (1), (2), etc.
        translation_dict = {}
        translation_section_match = re.search(r'해석:\s*\n((?:\(\d+\)[^\n]*\n?)+)', result, re.DOTALL)
        if translation_section_match:
            print(f"DEBUG: Found translation section: {translation_section_match.group(1)}")
            translation_lines = translation_section_match.group(1).strip().split('\n')
            for line in translation_lines:
                # Match (1) translation, (2) translation, etc.
                match = re.match(r'\((\d+)\)\s*(.*)', line.strip())
                if match:
                    num = match.group(1)
                    translation = match.group(2).strip()
                    translation_dict[num] = translation
                    print(f"DEBUG: Added translation {num}: {translation}")
        else:
            print(f"DEBUG: No translation section match found!")

        print(f"DEBUG: Final translation_dict: {translation_dict}")

        # 2. Fetch all answers between [[ ]] with their numbers
        answer_dict = {}
        for match in re.finditer(r'\((\d+)\)\[\[(.*?)\]\]', result, re.DOTALL):
            num = match.group(1)
            answer = match.group(2).strip()
            answer_dict[num] = answer
            print(f"DEBUG: Found answer {num}: {answer[:50]}...")

        print(f"DEBUG: Final answer_dict keys: {list(answer_dict.keys())}")

        # 3. Extract everything after the blank line following "해석:" (includes [전문해석] and [지문어휘])
        additional_content_match = re.search(r'해석:.*?\n\n(.*)', result, re.DOTALL)
        additional_content = additional_content_match.group(1).strip() if additional_content_match else ''

        # 4. Replace the content between [[ ]] with translations, keeping the [[ ]] and numbers
        for num in sorted(translation_dict.keys()):
            if num in translation_dict:
                pattern = rf'\({num}\)\[\[.*?\]\]'
                replacement = f'({num})[[{translation_dict[num]}]]'
                print(f"DEBUG: Replacing pattern for {num}: {pattern}")
                print(f"DEBUG: Replacement text: {replacement}")
                old_result = result
                result = re.sub(pattern, replacement, result, count=1, flags=re.DOTALL)
                if result != old_result:
                    print(f"DEBUG: Replacement successful for {num}")
                else:
                    print(f"DEBUG: WARNING - No replacement made for {num}!")

        # 5. Remove everything from '해석:' onwards
        result = re.sub(r'\n*해석:.*', '', result, flags=re.DOTALL).strip()

        # Check if blank mode is enabled (checkbox checked)
        blank_mode_enabled = False
        app_data_dir = Path.home() / ".my_application_data"
        make_questions_options_file_path = os.path.join(app_data_dir, "make_questions_options.json")
        if os.path.exists(make_questions_options_file_path):
            try:
                with open(make_questions_options_file_path, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    blank_mode_enabled = data.get("영작1_blank_mode", 0) == 1
            except:
                pass

        if blank_mode_enabled:
            # Replace [[ ]] with underscores, keeping the numbers
            result = re.sub(r'\((\d+)\)\[\[.*?\]\]', r'(\1)___________________', result, flags=re.DOTALL)
            question_line = "다음의 빈칸에 들어갈 말을 영작하고자 한다. [보기]에 주어진 단어를 올바르게 배열하시오. (어형변화 불가, 단어추가 불가)"
        else:
            question_line = "다음의 밑줄 친 부분을 영작하고자 한다. [보기]에 주어진 단어를 올바르게 배열하시오. (어형변화 불가, 단어추가 불가)"

        # Build question with multiple numbered items
        question_parts = []
        for num in sorted(answer_dict.keys()):
            scrambled = scramble_sentences(answer_dict[num])
            question_parts.append(f"({num}) {scrambled}")

        question = "[보기] \n" + "\n".join(question_parts)
        result = question_line + "\n" + result + "\n\n" + question

        # 6. Build explanation_part with multiple numbered answers
        explanation_lines = []
        for num in sorted(answer_dict.keys()):
            explanation_lines.append(f"({num}) {answer_dict[num]}")

        explanation_part = f"정답: \n" + "\n".join(explanation_lines) + f"\n\n{additional_content}"

        if not blank_mode_enabled:
            result = result.replace('.]].', ']].')




    print("result가공_영작1 완료")
    print(f"가공후 result: {result}")
    print(f"가공후 explanation_part: {explanation_part}")

    return result, "해설참조", explanation_part
    


def result가공_영작2(result):
    print("result가공_영작2 시작")
    print(f"result: {result}")
    count = result.count('[[')

    if count == 1:
        # Single-item case
        # 1. Fetch the data from "해석: " to the end of that line and put it into 'translation'.
        translation_match = re.search(r'해석: (.*?)(?=\n|$)', result)
        translation = translation_match.group(1).strip() if translation_match else ''

        # 2. Fetch the data between [[ ]] and put it into 'answer' (the original English text).
        answer_match = re.search(r'\[\[(.*?)\]\]', result, re.DOTALL)
        answer = answer_match.group(1) if answer_match else ''

        # 3. Extract [보기] content (handles both same-line and next-line format)
        보기_match = re.search(r'\[보기\][ \t]*\n?(.*?)(?=\n\n|$)', result, re.DOTALL)
        보기 = 보기_match.group(1).strip() if 보기_match else ''
        print(f"DEBUG 영작2 single-item: 보기 = {보기}")

        # 4. Extract everything after the blank line following "[보기]" (includes [전문해석] and [지문어휘])
        additional_content_match = re.search(r'\[보기\].*?\n\n(.*)', result, re.DOTALL)
        additional_content = additional_content_match.group(1).strip() if additional_content_match else ''

        # 5. Replace the content between [[ ]] with 'translation', keeping the [[ and ]]
        result = re.sub(r'\[\[.*?\]\]', f'[[{translation}]]', result, flags=re.DOTALL)

        # 6. Remove everything from '해석:' onwards
        result = re.sub(r'\n*해석:.*', '', result, flags=re.DOTALL).strip()

        # Check if blank mode is enabled (checkbox checked)
        blank_mode_enabled = False
        app_data_dir = Path.home() / ".my_application_data"
        make_questions_options_file_path = os.path.join(app_data_dir, "make_questions_options.json")
        if os.path.exists(make_questions_options_file_path):
            try:
                with open(make_questions_options_file_path, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    blank_mode_enabled = data.get("영작2_blank_mode", 0) == 1
            except:
                pass

        if blank_mode_enabled:
            # Replace [[ ]] with underscores
            result = re.sub(r'\[\[.*?\]\]', '___________________', result)

        number_of_words = len(answer.split())

        if blank_mode_enabled:
            question_line = f"[보기]에 주어진 단어를 활용하여 빈칸에 들어갈 말을 {number_of_words}단어로 영작하시오. (어형변화 가능, 단어추가 가능)\n"
        else:
            question_line = f"[보기]에 주어진 단어를 활용하여 밑줄 친 부분을 {number_of_words}단어로 영작하시오. (어형변화 가능, 단어추가 가능)\n"

        # Add [보기] line to result
        result = question_line + result + "\n[보기] " + 보기

        # 7. Build explanation_part with answer + additional content
        explanation_part = f"정답: {answer}\n\n{additional_content}"

        if not blank_mode_enabled:
            result = result.replace('.]].', ']].')
        result = re.sub(r'\n+', '\n', result)

    elif count > 1:
        # Multi-item case
        print(f"result: {result}")
        # Numbers (1), (2), etc. are already present in the result
        print(f"수정후 result: {result}")

        # 1. Fetch translations from "해석: " section - can be multiple lines with (1), (2), etc.
        translation_dict = {}
        translation_section_match = re.search(r'해석:\s*\n((?:\(\d+\)[^\n]*\n?)+)', result, re.DOTALL)
        if translation_section_match:
            print(f"DEBUG: Found translation section: {translation_section_match.group(1)}")
            translation_lines = translation_section_match.group(1).strip().split('\n')
            for line in translation_lines:
                match = re.match(r'\((\d+)\)\s*(.*)', line.strip())
                if match:
                    num = match.group(1)
                    translation = match.group(2).strip()
                    translation_dict[num] = translation
                    print(f"DEBUG: Added translation {num}: {translation}")
        else:
            print(f"DEBUG: No translation section match found!")

        print(f"DEBUG: Final translation_dict: {translation_dict}")

        # 2. Fetch all answers between [[ ]] with their numbers
        answer_dict = {}
        for match in re.finditer(r'\((\d+)\)\[\[(.*?)\]\]', result, re.DOTALL):
            num = match.group(1)
            answer = match.group(2).strip()
            answer_dict[num] = answer
            print(f"DEBUG: Found answer {num}: {answer[:50]}...")

        print(f"DEBUG: Final answer_dict keys: {list(answer_dict.keys())}")

        # 3. Extract [보기] content (single shared 보기 for all items, handles multi-line format)
        보기_match = re.search(r'\[보기\][ \t]*\n?(.*?)(?=\n\n|$)', result, re.DOTALL)
        보기 = 보기_match.group(1).strip() if 보기_match else ''
        print(f"DEBUG 영작2 multi-item: 보기 = {보기}")

        # 4. Extract everything after the blank line following "[보기]" (includes [전문해석] and [지문어휘])
        additional_content_match = re.search(r'\[보기\].*?\n\n(.*)', result, re.DOTALL)
        additional_content = additional_content_match.group(1).strip() if additional_content_match else ''

        # 5. Replace the content between [[ ]] with translations, keeping the [[ ]] and numbers
        for num in sorted(translation_dict.keys()):
            if num in translation_dict:
                pattern = rf'\({num}\)\[\[.*?\]\]'
                replacement = f'({num})[[{translation_dict[num]}]]'
                print(f"DEBUG: Replacing pattern for {num}: {pattern}")
                print(f"DEBUG: Replacement text: {replacement}")
                old_result = result
                result = re.sub(pattern, replacement, result, count=1, flags=re.DOTALL)
                if result != old_result:
                    print(f"DEBUG: Replacement successful for {num}")
                else:
                    print(f"DEBUG: WARNING - No replacement made for {num}!")

        # 6. Remove everything from '해석:' onwards
        result = re.sub(r'\n*해석:.*', '', result, flags=re.DOTALL).strip()

        # Check if blank mode is enabled (checkbox checked)
        blank_mode_enabled = False
        app_data_dir = Path.home() / ".my_application_data"
        make_questions_options_file_path = os.path.join(app_data_dir, "make_questions_options.json")
        if os.path.exists(make_questions_options_file_path):
            try:
                with open(make_questions_options_file_path, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    blank_mode_enabled = data.get("영작2_blank_mode", 0) == 1
            except:
                pass

        if blank_mode_enabled:
            # Replace [[ ]] with underscores, keeping the numbers
            result = re.sub(r'\((\d+)\)\[\[.*?\]\]', r'(\1)___________________', result, flags=re.DOTALL)

        # Calculate total word count from all answers
        total_words = sum(len(answer.split()) for answer in answer_dict.values())

        if blank_mode_enabled:
            question_line = f"[보기]에 주어진 단어를 활용하여 빈칸에 들어갈 말을 영작하시오. (어형변화 가능, 단어추가 가능)\n"
        else:
            question_line = f"[보기]에 주어진 단어를 활용하여 밑줄 친 부분을 영작하시오. (어형변화 가능, 단어추가 가능)\n"

        # Add [보기] line to result (with newline for multi-item format)
        # Check if 보기 contains numbered items
        if re.search(r'^\(\d+\)', 보기.strip(), re.MULTILINE):
            result = question_line + result + "\n[보기]\n" + 보기
        else:
            result = question_line + result + "\n[보기] " + 보기

        # 7. Build explanation_part with multiple numbered answers
        explanation_lines = []
        for num in sorted(answer_dict.keys()):
            explanation_lines.append(f"({num}) {answer_dict[num]}")

        explanation_part = f"정답: \n" + "\n".join(explanation_lines) + f"\n\n{additional_content}"

        if not blank_mode_enabled:
            result = result.replace('.]].', ']].')
        result = re.sub(r'\n+', '\n', result)

    else:
        # count == 0: No [[ ]] brackets found (error case or plain text)
        # Return the result as-is with empty explanation
        explanation_part = ""

    print("result가공_영작2 완료")
    print(f"가공후 result: {result}")
    print(f"가공후 explanation_part: {explanation_part}")

    return result, "해설참조", explanation_part
    


def result가공_주제영작(result, sheet_type):

    # Load 주제영작_mode from JSON to determine question_line
    주제영작_mode = 1  # Default to mode 1 (noun phrase)
    app_data_dir = Path.home() / ".my_application_data"
    make_questions_options_file_path = os.path.join(app_data_dir, "make_questions_options.json")
    if os.path.exists(make_questions_options_file_path):
        try:
            with open(make_questions_options_file_path, 'r', encoding='utf-8') as f:
                data = json.load(f)
                주제영작_mode = data.get("주제영작mode", 1)
        except:
            pass

    # Extract the passage, which is before '주제: '
    passage_match = re.search(r'(.*?)주제: ', result, re.DOTALL)
    passage = passage_match.group(1).strip() if passage_match else ''

    # Extract the English topic after '주제: '
    topic_eng_match = re.search(r'주제: (.*?)(?=\n|$)', result)
    topic_eng = topic_eng_match.group(1).strip() if topic_eng_match else ''


    # Extract the Korean topic after '해석: '
    topic_kor_match = re.search(r'해석: (.*?)(?=\n|$)', result)
    topic_kor = topic_kor_match.group(1).strip() if topic_kor_match else ''


    # Extract everything after the blank line following "해석:" (includes [전문해석] and [지문어휘])
    additional_content_match = re.search(r'해석:.*?\n\n(.*)', result, re.DOTALL)
    additional_content = additional_content_match.group(1).strip() if additional_content_match else ''

    if "_Hard" in sheet_type:
        # Extract 보기
        보기_match = re.search(r'\[보기\] (.*)', result)  # CORRECT
        보기 = 보기_match.group(1) if 보기_match else ''
        number_of_words = len(보기.split())

    # Set the question line based on mode
    if 주제영작_mode == 2 and "_Normal" in sheet_type:
        question_line = "윗글의 주제문을 하나의 문장으로 영작하고자 한다. [보기]에 주어진 단어를 올바르게 배열하시오. (어형변화 불가, 단어추가 불가)"
        
    elif 주제영작_mode == 2 and "_Hard" in sheet_type:
        question_line = f"윗글의 주제문을 하나의 문장으로 영작하고자 한다. [보기]에 주어진 단어를 활용하여 주제문을 영작하시오. (어형변화 가능, 단어추가 가능)"

    elif 주제영작_mode == 1 and "_Normal" in sheet_type:
        question_line = "윗글의 주제를 영작하고자 한다. [보기]에 주어진 단어를 올바르게 배열하시오. (어형변화 불가, 단어추가 불가)"

    elif 주제영작_mode == 1 and "_Hard" in sheet_type:
        question_line = f"윗글의 주제를 영작하고자 한다. [보기]에 주어진 단어를 활용하여 주제를 영작하시오. (어형변화 가능, 단어추가 가능)"

    
    if "_Normal" in sheet_type:
        # Create the question with scrambled topic_eng
        question = "[보기] " + scramble_sentences(topic_eng)
    elif "_Hard" in sheet_type:
        question = "[보기] " + 보기


    # Combine question_line, result, and question
    result = passage + "\n" + question_line + "\n" + question


    # Set explanation_part with topic_kor in parentheses + two newlines + additional content
    explanation_part = "정답: " + topic_eng + "\n" + "(" + topic_kor + ")" + "\n\n" + additional_content

    return result, "해설참조", explanation_part

            

def result가공_동형문제(result):

    # Check if all required section headers exist
    required_sections = ['[Passage]', '[Question]', '[Answer]', '[Explanation]']
    missing_sections = [section for section in required_sections if section not in result]

    if missing_sections:

        return "Error: Missing sections", "", ""

    # Regular expression pattern to capture content after each section header
    pattern = r'\[Passage\](.*?)\[Question\](.*?)\[Answer\](.*?)\[Explanation\](.*)'

    # Using regex to match and extract each part
    match = re.search(pattern, result, re.DOTALL)
    if match:
        passage = match.group(1).strip()
        question_part = match.group(2).strip()
        answer_part = match.group(3).strip()
        explanation_part = match.group(4).strip()

        question_part = "\n[Question]\n" + question_part
        # Print or use the variables as needed


        question_part = passage + "\n" + question_part

        return question_part, answer_part, explanation_part
    else:

        return "Error: Pattern matching failed", "", ""
    



def result가공_어휘종합(passage, result):
    if result == "Error":
        question_content = "Error"
        answer_content = ""
        explanation_content = ""
        return question_content, answer_content, explanation_content
    else:
        # Extract 'question_part'
        question_part_match = re.search(r"(.*?)Answers:", result, re.DOTALL)  # Non-greedy
        question_part = question_part_match.group(1).strip() if question_part_match else ""

        # Extract 'explanation_part'
        explanation_part_match = re.search(r"Answers:(.*)", result, re.DOTALL)
        explanation_part = explanation_part_match.group(1).strip() if explanation_part_match else ""

        # Extract 'answer_part'
        # This regex matches a parenthesized number followed by a space and a circled number
        answer_part_matches = re.findall(r"\(\d+\) [①②③④]", explanation_part)

        # Join all matches to form the answer_part string
        answer_part = "\n".join(answer_part_matches)


        # Modified part to extract phrases, determine their lengths, and replace accordingly

        questions_and_phrases = re.findall(r"\(\d+\).*? \"(.*?)\"(?:.*?\"(.*?)\")?.*?", question_part, flags=re.IGNORECASE)
        for question, phrases in enumerate(questions_and_phrases, start=1):
            # Adjust here for one or two phrases
            if phrases[1]:  # If there are two phrases
                short_phrase, long_phrase = sorted(phrases, key=len)
            else:  # If there is only one phrase
                short_phrase = long_phrase = phrases[0]
                
            # Normalize the long_phrase to ignore punctuation and case
            normalized_long_phrase = re.sub(r"[,.!?]", r"[,.!?]?", re.escape(long_phrase), flags=re.IGNORECASE)

            def replace_short_in_long(match):
                long_text = match.group(0)
                escaped_short_phrase = re.escape(short_phrase)
                short_pattern = r"\b(" + escaped_short_phrase + r")\b"
                replaced_text = re.sub(short_pattern, f"({question})[\\1]", long_text, count=1, flags=re.IGNORECASE)
                return replaced_text

            # Use the normalized pattern to match the long_phrase in the passage
            pattern = r"\b(" + normalized_long_phrase + r")\b"
            if not re.search(pattern, passage, flags=re.DOTALL | re.IGNORECASE):
                # Logic for redefining long_phrase only applies if there were originally two phrases
                if phrases[1]:
                    words_around_short = r"(\w*\s)?\b" + re.escape(short_phrase) + r"\b(\s\w*)?"
                    match = re.search(words_around_short, long_phrase, flags=re.IGNORECASE)
                    if match:
                        long_phrase = match.group(0)
                        normalized_long_phrase = re.sub(r"[,.!?]", r"[,.!?]?", re.escape(long_phrase), flags=re.IGNORECASE)
                        pattern = r"\b(" + normalized_long_phrase + r")\b"

            passage = re.sub(pattern, replace_short_in_long, passage, count=1, flags=re.DOTALL | re.IGNORECASE)

        question_part = passage + "\n" + question_part
        return question_part, answer_part, explanation_part
    


def result가공_내용종합(passage, result):
    if result == "Error":
        question_content = "Error"
        answer_content = ""
        explanation_content = ""
        return question_content, answer_content, explanation_content
    else:
        # Extract 'question_part'
        question_part_match = re.search(r"(.*?)Answers:", result, re.DOTALL)  # Non-greedy
        question_part = question_part_match.group(1).strip() if question_part_match else ""

        # Extract 'explanation_part'
        explanation_part_match = re.search(r"Answers:(.*)", result, re.DOTALL)
        explanation_part = explanation_part_match.group(1).strip() if explanation_part_match else ""

        # Extract 'answer_part'
        # This regex matches a parenthesized number followed by a space and a circled number
        answer_part_matches = re.findall(r"\(\d+\) [①②③④]", explanation_part)

        # Join all matches to form the answer_part string
        answer_part = "\n".join(answer_part_matches)


        # Modified part to extract phrases, determine their lengths, and replace accordingly
        return question_part, answer_part, explanation_part
    

def result가공_동반의어문제(result):
    # Check for error cases more robustly (strip whitespace, check prefix, check required delimiters)
    if result.strip().startswith("Error") or "Answers:" not in result or "Explanations:" not in result:
        question_content = "Error" if result.strip().startswith("Error") else result
        return question_content, "", ""

    # Split the data into parts at 'Answers:' and 'Explanations:'
    split_result = result.split('Answers:')
    question_part = split_result[0].strip()

    answer_and_explanation = split_result[1].split('Explanations:')
    answer_part = answer_and_explanation[0].strip()
    explanation_part = answer_and_explanation[1].strip()

    # Clean up 'Answers:' and 'Explanations:' labels from the final strings
    answer_part = answer_part.replace('Answers:', '').strip()
    explanation_part = explanation_part.replace('Explanations:', '').strip()

    return question_part, answer_part, explanation_part


def result가공_영영정의문제(result):
    # Check for error cases more robustly (strip whitespace, check prefix, check required delimiters)
    if result.strip().startswith("Error") or "Answers:" not in result or "Explanations:" not in result:
        question_content = "Error" if result.strip().startswith("Error") else result
        return question_content, "", ""

    split_result = result.split('Answers:')
    question_part = split_result[0].strip()

    answer_and_explanation = split_result[1].split('Explanations:')
    answer_part = answer_and_explanation[0].strip()
    explanation_part = answer_and_explanation[1].strip()

    # Clean up 'Answers:' and 'Explanations:' labels from the final strings
    answer_part = answer_part.replace('Answers:', '').strip()
    explanation_part = explanation_part.replace('Explanations:', '').strip()

    return question_part, answer_part, explanation_part




def result가공_커스텀(result):
    
    if result == "Error":
        question_content = "Error"
        answer_content = ""
        explanation_content = ""
        return question_content, answer_content, explanation_content
    else:
        # Attempt to extract 'question_part'
        question_part_match = re.search(r"^(.*?)Answer:", result, re.DOTALL)
        if question_part_match:
            question_part = question_part_match.group(1).strip()
        else:
            # If there is no 'Answer:', put the entire 'result' into 'question_part'
            question_part = result.strip()

        # Extract 'answer_part'
        answer_part_match = re.search(r"Answer:(.*?)Explanation:", result, re.DOTALL)
        answer_part = answer_part_match.group(1).strip() if answer_part_match else ""

        # Extract 'explanation_part'
        explanation_part_match = re.search(r"Explanation:(.*)", result, re.DOTALL)
        explanation_part = explanation_part_match.group(1).strip() if explanation_part_match else ""

        return question_part, answer_part, explanation_part



######################## result가공 functions ends here ##########################
