# -*- coding: utf-8 -*-
"""
Resource management utilities for AutoQM application.
Provides context managers for proper cleanup of Excel/Word resources.
"""

import gc
import logging
from contextlib import contextmanager
from typing import Optional
from pathlib import Path
from openpyxl import load_workbook, Workbook
from docx import Document


logger = logging.getLogger(__name__)


@contextmanager
def excel_workbook(filepath: str, read_only: bool = False, data_only: bool = False):
    """
    Context manager for safely opening and closing Excel workbooks.

    Args:
        filepath: Path to Excel file
        read_only: Open in read-only mode
        data_only: Read only cell values, not formulas

    Yields:
        Workbook object

    Example:
        with excel_workbook('data.xlsx') as wb:
            ws = wb.active
            # ... work with workbook
        # Automatically closed and cleaned up
    """
    wb = None
    try:
        wb = load_workbook(filename=filepath, read_only=read_only, data_only=data_only)
        yield wb
    except Exception as e:
        logger.error(f"Error working with Excel file {filepath}: {str(e)}")
        raise
    finally:
        if wb is not None:
            try:
                wb.close()
            except Exception as e:
                logger.warning(f"Error closing workbook {filepath}: {str(e)}")
        # Force garbage collection to free memory
        gc.collect()


@contextmanager
def excel_workbook_write(filepath: str, create_if_missing: bool = False):
    """
    Context manager for safely writing to Excel workbooks.
    Automatically saves changes on successful completion.

    Args:
        filepath: Path to Excel file
        create_if_missing: Create new workbook if file doesn't exist

    Yields:
        Workbook object

    Example:
        with excel_workbook_write('data.xlsx') as wb:
            ws = wb.active
            ws['A1'] = 'New value'
        # Automatically saved and closed
    """
    wb = None
    file_path = Path(filepath)

    try:
        if file_path.exists():
            wb = load_workbook(filename=filepath)
        elif create_if_missing:
            wb = Workbook()
        else:
            raise FileNotFoundError(f"Excel file not found: {filepath}")

        yield wb

        # Save changes if successful
        wb.save(filepath)

    except Exception as e:
        logger.error(f"Error writing to Excel file {filepath}: {str(e)}")
        raise
    finally:
        if wb is not None:
            try:
                wb.close()
            except Exception as e:
                logger.warning(f"Error closing workbook {filepath}: {str(e)}")
        gc.collect()


@contextmanager
def word_document(filepath: Optional[str] = None):
    """
    Context manager for safely working with Word documents.

    Args:
        filepath: Path to existing Word file, or None to create new document

    Yields:
        Document object

    Example:
        with word_document('output.docx') as doc:
            doc.add_paragraph('Hello World')
        # Automatically cleaned up
    """
    doc = None
    try:
        if filepath:
            doc = Document(filepath)
        else:
            doc = Document()

        yield doc

    except Exception as e:
        logger.error(f"Error working with Word document {filepath}: {str(e)}")
        raise
    finally:
        # Document objects don't have explicit close method
        # But we can trigger garbage collection
        doc = None
        gc.collect()


@contextmanager
def word_document_write(filepath: str):
    """
    Context manager for creating and saving Word documents.
    Automatically saves on successful completion.

    Args:
        filepath: Path where document will be saved

    Yields:
        Document object

    Example:
        with word_document_write('output.docx') as doc:
            doc.add_paragraph('Hello World')
        # Automatically saved
    """
    doc = None
    try:
        doc = Document()
        yield doc

        # Save document if successful
        doc.save(filepath)
        logger.info(f"Word document saved: {filepath}")

    except Exception as e:
        logger.error(f"Error creating Word document {filepath}: {str(e)}")
        raise
    finally:
        doc = None
        gc.collect()


class MemoryMonitor:
    """Monitor and log memory usage for debugging memory leaks."""

    def __init__(self, threshold_mb: float = 500.0):
        """
        Initialize memory monitor.

        Args:
            threshold_mb: Log warning if memory usage exceeds this threshold (in MB)
        """
        self.threshold_mb = threshold_mb
        self.enabled = False

        try:
            import psutil
            self.psutil = psutil
            self.enabled = True
        except ImportError:
            logger.warning("psutil not available, memory monitoring disabled")

    def get_memory_usage(self) -> Optional[float]:
        """Get current memory usage in MB."""
        if not self.enabled:
            return None

        try:
            import os
            process = self.psutil.Process(os.getpid())
            mem_info = process.memory_info()
            return mem_info.rss / 1024 / 1024  # Convert to MB
        except Exception as e:
            logger.error(f"Error getting memory usage: {str(e)}")
            return None

    def check_and_log(self, context: str = ""):
        """Check memory usage and log if above threshold."""
        if not self.enabled:
            return

        usage_mb = self.get_memory_usage()
        if usage_mb is None:
            return

        if usage_mb > self.threshold_mb:
            logger.warning(f"High memory usage ({usage_mb:.1f} MB) at: {context}")
            # Suggest garbage collection
            collected = gc.collect()
            logger.info(f"Garbage collection freed {collected} objects")

            # Log again after GC
            new_usage = self.get_memory_usage()
            if new_usage:
                logger.info(f"Memory after GC: {new_usage:.1f} MB")

    @contextmanager
    def monitor(self, context: str = ""):
        """
        Context manager for monitoring memory usage during operations.

        Example:
            monitor = MemoryMonitor()
            with monitor.monitor("Processing 100 questions"):
                # ... do work
            # Logs memory usage before and after
        """
        if self.enabled:
            before = self.get_memory_usage()
            logger.debug(f"Memory before {context}: {before:.1f} MB")

        try:
            yield
        finally:
            if self.enabled:
                after = self.get_memory_usage()
                logger.debug(f"Memory after {context}: {after:.1f} MB")

                if before and after:
                    delta = after - before
                    if abs(delta) > 10:  # Log if change > 10MB
                        logger.info(f"Memory change for {context}: {delta:+.1f} MB")

                self.check_and_log(context)


def force_cleanup():
    """Force garbage collection and cleanup of unreferenced objects."""
    collected = gc.collect()
    logger.debug(f"Forced garbage collection: {collected} objects freed")
    return collected


# Global memory monitor instance
memory_monitor = MemoryMonitor(threshold_mb=500.0)
