"""Cross-platform helpers for preventing and releasing system sleep."""

import ctypes
import subprocess
import sys


def prevent_sleep():
    """Prevent the host OS from sleeping; returns a process handle on macOS."""
    if sys.platform.startswith('darwin'):
        return subprocess.Popen(['caffeinate'])
    if sys.platform.startswith('win'):
        ctypes.windll.kernel32.SetThreadExecutionState(0x80000002)
    return None


def allow_sleep(caffeinate_process):
    """Release any sleep-prevention previously requested."""
    if caffeinate_process:
        try:
            caffeinate_process.terminate()
            # Wait up to 2 seconds for graceful termination
            caffeinate_process.wait(timeout=2)
        except subprocess.TimeoutExpired:
            # Force kill if still running after timeout
            caffeinate_process.kill()
            caffeinate_process.wait()  # Clean up zombie process
        except Exception:
            # If process already dead or other error, ignore
            pass
    if sys.platform.startswith('win'):
        ctypes.windll.kernel32.SetThreadExecutionState(0x80000000)


__all__ = ["prevent_sleep", "allow_sleep"]
