"""
Token Manager Module for AutoQM Desktop Application

Handles automatic JWT token refresh to prevent session expiration during long operations.

Features:
- Periodic background token refresh (every 30 minutes)
- Pre-expiration refresh (refresh 5 minutes before expiry)
- Thread-safe token updates
- Graceful shutdown on app exit

Usage:
    token_manager = TokenManager(
        refresh_callback=lambda old_token: new_token,
        update_callback=lambda new_token: save_token(new_token),
        interval_minutes=30
    )
    token_manager.start()
    # ... app runs ...
    token_manager.stop()  # On app exit
"""

import threading
import time
import logging
import jwt
from typing import Callable, Optional
from datetime import datetime, timedelta

logger = logging.getLogger(__name__)


class TokenManager:
    """
    Manages automatic JWT token refresh for desktop application.

    Runs a background thread that periodically refreshes tokens before they expire.
    """

    def __init__(
        self,
        refresh_callback: Callable[[str], Optional[str]],
        update_callback: Callable[[str], None],
        current_token: str,
        interval_minutes: int = 30,
        min_validity_minutes: int = 5
    ):
        """
        Initialize TokenManager.

        Args:
            refresh_callback: Function that takes old token and returns new token
                             Should call WordPress refresh endpoint
            update_callback: Function that saves the new token to instance variable
            current_token: Initial JWT token
            interval_minutes: How often to check/refresh token (default: 30 minutes)
            min_validity_minutes: Refresh if token expires in less than this (default: 5 minutes)
        """
        self.refresh_callback = refresh_callback
        self.update_callback = update_callback
        self.current_token = current_token
        self.interval_seconds = interval_minutes * 60
        self.min_validity_seconds = min_validity_minutes * 60

        self._stop_event = threading.Event()
        self._refresh_thread: Optional[threading.Thread] = None
        self._lock = threading.Lock()
        self._running = False

    def start(self):
        """Start the background token refresh thread."""
        if self._running:
            logger.warning("TokenManager already running")
            return

        self._stop_event.clear()
        self._refresh_thread = threading.Thread(
            target=self._refresh_loop,
            daemon=True,
            name="TokenRefreshThread"
        )
        self._refresh_thread.start()
        self._running = True
        logger.info(f"TokenManager started (refresh interval: {self.interval_seconds}s)")

    def stop(self):
        """Stop the background token refresh thread."""
        if not self._running:
            return

        logger.info("Stopping TokenManager...")
        self._stop_event.set()

        if self._refresh_thread:
            self._refresh_thread.join(timeout=5.0)

        self._running = False
        logger.info("TokenManager stopped")

    def update_token(self, new_token: str):
        """
        Update the current token (thread-safe).

        Args:
            new_token: New JWT token to store
        """
        with self._lock:
            self.current_token = new_token

    def get_token(self) -> str:
        """
        Get the current token (thread-safe).

        Returns:
            Current JWT token
        """
        with self._lock:
            return self.current_token

    def _get_token_expiry(self, token: str) -> Optional[datetime]:
        """
        Extract expiration time from JWT token.

        Args:
            token: JWT token string

        Returns:
            Expiration datetime or None if invalid
        """
        try:
            decoded = jwt.decode(
                token,
                algorithms=['HS256'],
                options={'verify_signature': False}
            )
            exp_timestamp = decoded.get('exp')
            if exp_timestamp:
                return datetime.fromtimestamp(exp_timestamp)
            return None
        except Exception as e:
            logger.error(f"Failed to decode token: {e}")
            return None

    def _should_refresh(self, token: str) -> bool:
        """
        Check if token should be refreshed.

        Args:
            token: JWT token to check

        Returns:
            True if token expires soon or is invalid
        """
        expiry = self._get_token_expiry(token)
        if not expiry:
            logger.warning("Cannot determine token expiry - refreshing to be safe")
            return True

        time_until_expiry = (expiry - datetime.now()).total_seconds()

        if time_until_expiry <= 0:
            logger.warning("Token already expired!")
            return True

        if time_until_expiry < self.min_validity_seconds:
            logger.info(f"Token expires in {time_until_expiry:.1f}s - refreshing")
            return True

        logger.debug(f"Token valid for {time_until_expiry:.1f}s - no refresh needed")
        return False

    def _refresh_loop(self):
        """
        Background thread that periodically checks and refreshes tokens.
        """
        logger.info("Token refresh loop started")

        while not self._stop_event.is_set():
            try:
                # Get current token (thread-safe)
                current_token = self.get_token()

                # Check if refresh is needed
                if self._should_refresh(current_token):
                    logger.info("Refreshing token...")

                    # Call refresh callback (should make API call to WordPress)
                    new_token = self.refresh_callback(current_token)

                    if new_token:
                        # Update stored token (thread-safe)
                        self.update_token(new_token)

                        # Notify parent class to update its token
                        self.update_callback(new_token)

                        logger.info("✓ Token refreshed successfully")
                    else:
                        logger.error("✗ Token refresh failed - keeping old token")

            except Exception as e:
                logger.error(f"Error in token refresh loop: {e}", exc_info=True)

            # Wait for next check (or until stop signal)
            self._stop_event.wait(self.interval_seconds)

        logger.info("Token refresh loop exited")

    def force_refresh(self) -> bool:
        """
        Force an immediate token refresh (for manual refresh).

        Returns:
            True if refresh succeeded, False otherwise
        """
        try:
            current_token = self.get_token()
            new_token = self.refresh_callback(current_token)

            if new_token:
                self.update_token(new_token)
                self.update_callback(new_token)
                logger.info("Manual token refresh successful")
                return True
            else:
                logger.error("Manual token refresh failed")
                return False

        except Exception as e:
            logger.error(f"Error during manual refresh: {e}")
            return False
