# -*- coding: utf-8 -*-
"""
API utilities for AutoQM application.
Provides retry logic, timeout handling, and health checks.
"""

import requests
import logging
import time
from typing import Optional, Dict, Any, Callable
from functools import wraps
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry


logger = logging.getLogger(__name__)


class APIError(Exception):
    """Custom exception for API errors."""
    pass


class NetworkError(Exception):
    """Custom exception for network errors."""
    pass


class RateLimitError(Exception):
    """Custom exception for rate limit errors."""
    pass


class BatchSizeAdjustmentRequired(Exception):
    """Raised when the backend enforces a smaller batch size mid-run."""

    def __init__(self, new_size: int):
        self.new_size = max(1, int(new_size))
        super().__init__(f"Batch size must be adjusted to {self.new_size}")


def create_retry_session(
    retries: int = 3,
    backoff_factor: float = 1.0,
    status_forcelist: tuple = (408, 429, 500, 502, 503, 504),
    timeout: int = 30
) -> requests.Session:
    """
    Create a requests session with automatic retry logic.

    Args:
        retries: Maximum number of retries
        backoff_factor: Factor for exponential backoff (sleep = backoff_factor * (2 ** retry_count)
        status_forcelist: HTTP status codes to retry on
        timeout: Default timeout in seconds

    Returns:
        Configured requests.Session object
    """
    session = requests.Session()

    retry_strategy = Retry(
        total=retries,
        backoff_factor=backoff_factor,
        status_forcelist=status_forcelist,
        allowed_methods=["GET", "POST", "PUT", "DELETE", "HEAD", "OPTIONS"]
    )

    adapter = HTTPAdapter(max_retries=retry_strategy)
    session.mount("http://", adapter)
    session.mount("https://", adapter)

    # Set default timeout
    session.timeout = timeout

    return session


def retry_on_exception(
    max_attempts: int = 3,
    delay: float = 1.0,
    backoff: float = 2.0,
    exceptions: tuple = (requests.RequestException, NetworkError)
):
    """
    Decorator for retrying functions with exponential backoff.

    Args:
        max_attempts: Maximum number of attempts
        delay: Initial delay between retries in seconds
        backoff: Multiplier for delay after each retry
        exceptions: Tuple of exceptions to catch and retry

    Example:
        @retry_on_exception(max_attempts=3, delay=1.0, backoff=2.0)
        def my_api_call():
            return requests.get(url)
    """
    def decorator(func: Callable) -> Callable:
        @wraps(func)
        def wrapper(*args, **kwargs):
            attempt = 1
            current_delay = delay

            while attempt <= max_attempts:
                try:
                    return func(*args, **kwargs)
                except exceptions as e:
                    if attempt == max_attempts:
                        logger.error(f"Function {func.__name__} failed after {max_attempts} attempts: {str(e)}")
                        raise

                    logger.warning(
                        f"Attempt {attempt}/{max_attempts} failed for {func.__name__}: {str(e)}. "
                        f"Retrying in {current_delay:.1f}s..."
                    )

                    time.sleep(current_delay)
                    current_delay *= backoff
                    attempt += 1

        return wrapper
    return decorator


def make_api_request(
    method: str,
    url: str,
    headers: Optional[Dict[str, str]] = None,
    json_data: Optional[Dict[str, Any]] = None,
    params: Optional[Dict[str, Any]] = None,
    timeout: int = 30,
    max_retries: int = 3,
    backoff_factor: float = 1.0
) -> requests.Response:
    """
    Make an API request with automatic retry logic and timeout.

    Args:
        method: HTTP method (GET, POST, etc.)
        url: Request URL
        headers: Request headers
        json_data: JSON data for POST/PUT requests
        params: Query parameters
        timeout: Request timeout in seconds
        max_retries: Maximum number of retries
        backoff_factor: Factor for exponential backoff

    Returns:
        Response object

    Raises:
        NetworkError: For network-related errors
        RateLimitError: For rate limit errors (429)
        APIError: For other API errors
    """
    session = create_retry_session(
        retries=max_retries,
        backoff_factor=backoff_factor,
        timeout=timeout
    )

    try:
        response = session.request(
            method=method,
            url=url,
            headers=headers,
            json=json_data,
            params=params,
            timeout=timeout
        )

        # Check for rate limiting
        if response.status_code == 429:
            retry_after = response.headers.get('Retry-After', '60')
            raise RateLimitError(f"Rate limit exceeded. Retry after {retry_after} seconds.")

        # Check for batch size adjustment (HTTP 400 with max_batch_size field)
        if response.status_code == 400:
            try:
                error_data = response.json()
                if 'max_batch_size' in error_data:
                    new_batch_size = error_data['max_batch_size']
                    logger.warning(f"Server enforces batch size limit: {new_batch_size}")
                    raise BatchSizeAdjustmentRequired(new_batch_size)
            except (ValueError, requests.exceptions.JSONDecodeError):
                # If JSON parsing fails, fall through to generic error handling
                pass

        # Check for other errors
        if not response.ok:
            error_msg = f"API request failed with status {response.status_code}: {response.text}"
            logger.error(error_msg)
            raise APIError(error_msg)

        return response

    except requests.Timeout as e:
        error_msg = f"Request timed out after {timeout} seconds: {url}"
        logger.error(error_msg)
        raise NetworkError(error_msg) from e

    except requests.ConnectionError as e:
        error_msg = f"Connection error for {url}: {str(e)}"
        logger.error(error_msg)
        raise NetworkError(error_msg) from e

    except requests.RequestException as e:
        error_msg = f"Request failed for {url}: {str(e)}"
        logger.error(error_msg)
        raise NetworkError(error_msg) from e

    finally:
        session.close()


def check_server_health(base_url: str, timeout: int = 5) -> bool:
    """
    Check if the Flask backend is running and healthy.

    Args:
        base_url: Base URL of the Flask server
        timeout: Timeout in seconds

    Returns:
        True if server is healthy, False otherwise
    """
    health_url = f"{base_url}/health"

    try:
        response = requests.get(health_url, timeout=timeout)
        return response.ok
    except requests.RequestException:
        return False


def wait_for_server(base_url: str, max_wait: int = 30, check_interval: int = 2) -> bool:
    """
    Wait for the Flask backend to become available.

    Args:
        base_url: Base URL of the Flask server
        max_wait: Maximum time to wait in seconds
        check_interval: Time between checks in seconds

    Returns:
        True if server became available, False if timeout
    """
    elapsed = 0

    while elapsed < max_wait:
        if check_server_health(base_url):
            logger.info(f"Server at {base_url} is healthy")
            return True

        logger.info(f"Waiting for server at {base_url}... ({elapsed}s/{max_wait}s)")
        time.sleep(check_interval)
        elapsed += check_interval

    logger.error(f"Server at {base_url} did not become available within {max_wait}s")
    return False


# Convenience functions for common HTTP methods
def get_with_retry(url: str, headers: Optional[Dict] = None, timeout: int = 30, **kwargs) -> requests.Response:
    """GET request with automatic retry."""
    return make_api_request("GET", url, headers=headers, timeout=timeout, **kwargs)


def post_with_retry(url: str, json_data: Optional[Dict] = None, headers: Optional[Dict] = None,
                    timeout: int = 30, **kwargs) -> requests.Response:
    """POST request with automatic retry."""
    return make_api_request("POST", url, headers=headers, json_data=json_data, timeout=timeout, **kwargs)


def post_with_json_retry(
    url: str,
    json_data: Optional[Dict] = None,
    headers: Optional[Dict] = None,
    timeout: int = 30,
    max_retries: int = 3,
    backoff_factor: float = 2.0
) -> Dict[str, Any]:
    """
    POST request with automatic retry AND JSON parsing with retry.

    This function handles cases where:
    1. Network fails (connection errors, timeouts)
    2. Server returns error status codes
    3. Server returns HTTP 200 but empty/invalid JSON (e.g., Flask reload during request)

    Args:
        url: Request URL
        json_data: JSON payload
        headers: Request headers
        timeout: Request timeout in seconds
        max_retries: Maximum number of retries
        backoff_factor: Exponential backoff multiplier

    Returns:
        Parsed JSON response as dictionary

    Raises:
        APIError: After all retries exhausted
    """
    attempt = 1
    current_delay = 1.0

    while attempt <= max_retries:
        try:
            # Make the request
            response = make_api_request(
                method="POST",
                url=url,
                headers=headers,
                json_data=json_data,
                timeout=timeout,
                max_retries=1,  # Don't retry at this level, we'll handle it here
                backoff_factor=backoff_factor
            )

            # Try to parse JSON
            try:
                # Force UTF-8 encoding to prevent Korean character corruption
                response.encoding = 'utf-8'
                return response.json()
            except (ValueError, requests.exceptions.JSONDecodeError) as json_error:
                # Empty or invalid JSON - likely server restarted
                error_msg = (
                    f"Failed to parse JSON response (attempt {attempt}/{max_retries}): {str(json_error)}\n"
                    f"Response status: {response.status_code}\n"
                    f"Response body: {response.text[:200]}"
                )
                logger.warning(error_msg)

                if attempt == max_retries:
                    raise APIError(f"Failed to parse JSON after {max_retries} attempts: {str(json_error)}")

                # Retry with exponential backoff
                logger.info(f"Retrying in {current_delay:.1f}s... (Flask may have reloaded)")
                time.sleep(current_delay)
                current_delay *= backoff_factor
                attempt += 1

        except BatchSizeAdjustmentRequired:
            # Don't retry - let it propagate to caller for batch size adjustment
            raise

        except (NetworkError, APIError, RateLimitError) as e:
            # Network or API errors
            if attempt == max_retries:
                logger.error(f"Request failed after {max_retries} attempts: {str(e)}")
                raise

            logger.warning(
                f"Request attempt {attempt}/{max_retries} failed: {str(e)}. "
                f"Retrying in {current_delay:.1f}s..."
            )
            time.sleep(current_delay)
            current_delay *= backoff_factor
            attempt += 1

    # Should never reach here
    raise APIError(f"Request failed after {max_retries} attempts")
