# -*- coding: utf-8 -*-
"""
Auto-save functionality for AutoQM application.
Provides automatic periodic saving to prevent data loss.
"""

import threading
import time
import logging
from typing import Optional, Callable
from pathlib import Path
from datetime import datetime


logger = logging.getLogger(__name__)


class AutoSaver:
    """
    Manages automatic periodic saving of Excel files.

    Example usage:
        auto_saver = AutoSaver(
            save_callback=lambda: save_excel_file(excel_file),
            interval_minutes=5
        )
        auto_saver.start()
        # ... work continues
        auto_saver.stop()  # Stop auto-save when done
    """

    def __init__(
        self,
        save_callback: Callable[[], bool],
        interval_minutes: int = 5,
        check_unsaved_changes: Optional[Callable[[], bool]] = None
    ):
        """
        Initialize auto-saver.

        Args:
            save_callback: Function to call for saving. Should return True on success.
            interval_minutes: How often to auto-save (in minutes)
            check_unsaved_changes: Optional function to check if there are unsaved changes
        """
        self.save_callback = save_callback
        self.interval_seconds = interval_minutes * 60
        self.check_unsaved_changes = check_unsaved_changes

        self._thread: Optional[threading.Thread] = None
        self._stop_event = threading.Event()
        self._running = False
        self._last_save_time: Optional[datetime] = None

        logger.info(f"AutoSaver initialized with {interval_minutes} minute interval")

    def _auto_save_loop(self):
        """Internal loop that runs in background thread."""
        logger.info("Auto-save loop started")

        while not self._stop_event.is_set():
            # Wait for the interval or until stopped
            if self._stop_event.wait(timeout=self.interval_seconds):
                break  # Stop event was set

            # Check if there are unsaved changes (if callback provided)
            if self.check_unsaved_changes and not self.check_unsaved_changes():
                logger.debug("No unsaved changes, skipping auto-save")
                continue

            # Perform auto-save
            try:
                logger.info("Auto-save triggered")
                success = self.save_callback()

                if success:
                    self._last_save_time = datetime.now()
                    logger.info(f"Auto-save successful at {self._last_save_time}")
                else:
                    logger.warning("Auto-save callback returned False")

            except Exception as e:
                logger.error(f"Auto-save failed: {str(e)}", exc_info=True)

    def start(self):
        """Start auto-save in background thread."""
        if self._running:
            logger.warning("Auto-save already running")
            return

        self._stop_event.clear()
        self._running = True

        self._thread = threading.Thread(
            target=self._auto_save_loop,
            name="AutoSaveThread",
            daemon=True  # Daemon thread won't prevent program exit
        )
        self._thread.start()

        logger.info("Auto-save started")

    def stop(self, wait: bool = True):
        """
        Stop auto-save thread.

        Args:
            wait: If True, wait for thread to finish
        """
        if not self._running:
            return

        logger.info("Stopping auto-save...")
        self._stop_event.set()

        if wait and self._thread and self._thread.is_alive():
            self._thread.join(timeout=5.0)

        self._running = False
        logger.info("Auto-save stopped")

    def trigger_save_now(self):
        """Manually trigger an immediate save (doesn't reset the timer)."""
        if not self._running:
            logger.warning("Auto-save not running, cannot trigger manual save")
            return False

        try:
            logger.info("Manual save triggered")
            success = self.save_callback()

            if success:
                self._last_save_time = datetime.now()
                logger.info(f"Manual save successful at {self._last_save_time}")
                return True
            else:
                logger.warning("Manual save callback returned False")
                return False

        except Exception as e:
            logger.error(f"Manual save failed: {str(e)}", exc_info=True)
            return False

    def get_last_save_time(self) -> Optional[datetime]:
        """Get the timestamp of the last successful auto-save."""
        return self._last_save_time

    def is_running(self) -> bool:
        """Check if auto-save is currently running."""
        return self._running

    def get_time_since_last_save(self) -> Optional[float]:
        """Get seconds since last save, or None if never saved."""
        if self._last_save_time is None:
            return None

        delta = datetime.now() - self._last_save_time
        return delta.total_seconds()


class AutoSaveManager:
    """
    Manages multiple auto-savers for different resources.
    Singleton pattern to ensure only one manager exists.
    """

    _instance = None
    _lock = threading.Lock()

    def __new__(cls):
        if cls._instance is None:
            with cls._lock:
                if cls._instance is None:
                    cls._instance = super().__new__(cls)
                    cls._instance._initialized = False
        return cls._instance

    def __init__(self):
        if self._initialized:
            return

        self._auto_savers = {}
        self._initialized = True
        logger.info("AutoSaveManager initialized")

    def register(
        self,
        name: str,
        save_callback: Callable[[], bool],
        interval_minutes: int = 5,
        check_unsaved_changes: Optional[Callable[[], bool]] = None,
        auto_start: bool = True
    ) -> AutoSaver:
        """
        Register a new auto-saver.

        Args:
            name: Unique identifier for this auto-saver
            save_callback: Function to call for saving
            interval_minutes: How often to auto-save
            check_unsaved_changes: Optional function to check if there are unsaved changes
            auto_start: If True, start immediately

        Returns:
            AutoSaver instance
        """
        if name in self._auto_savers:
            logger.warning(f"Auto-saver '{name}' already registered, stopping old one")
            self._auto_savers[name].stop()

        auto_saver = AutoSaver(
            save_callback=save_callback,
            interval_minutes=interval_minutes,
            check_unsaved_changes=check_unsaved_changes
        )

        self._auto_savers[name] = auto_saver

        if auto_start:
            auto_saver.start()

        logger.info(f"Registered auto-saver: {name}")
        return auto_saver

    def unregister(self, name: str):
        """Stop and remove an auto-saver."""
        if name in self._auto_savers:
            self._auto_savers[name].stop()
            del self._auto_savers[name]
            logger.info(f"Unregistered auto-saver: {name}")

    def get(self, name: str) -> Optional[AutoSaver]:
        """Get an auto-saver by name."""
        return self._auto_savers.get(name)

    def stop_all(self):
        """Stop all registered auto-savers."""
        logger.info("Stopping all auto-savers...")
        for name, auto_saver in list(self._auto_savers.items()):
            auto_saver.stop()
        logger.info("All auto-savers stopped")

    def get_status(self) -> dict:
        """Get status of all auto-savers."""
        status = {}
        for name, auto_saver in self._auto_savers.items():
            status[name] = {
                'running': auto_saver.is_running(),
                'last_save': auto_saver.get_last_save_time(),
                'seconds_since_save': auto_saver.get_time_since_last_save()
            }
        return status


# Global singleton instance
def get_auto_save_manager() -> AutoSaveManager:
    """Get the global AutoSaveManager instance."""
    return AutoSaveManager()
