#modules/copyquestion.py

import tkinter as tk
from tkinter import messagebox
import customtkinter as CTk
import json
from pathlib import Path
import sys
from .copymanager import CopyManager
from .functions import ToolTip유형설명


class CategorySelectionDialog(CTk.CTkToplevel):
    """Modal dialog for selecting or creating a category when saving a preset."""

    def __init__(self, parent, presets_data, font_family, font_size, initial_category=None):
        super().__init__(parent)

        self.title("카테고리 선택")
        self.geometry("400x400")
        self.minsize(width=400, height=40)
        self.transient(parent)
        self.grab_set()
        try:
            self.focus_force()
        except:
            pass

        self.presets_data = presets_data
        self.font_family = font_family
        self.font_size = font_size
        self.selected_category = None

        # Colors
        COLORS = {
            'bg_window': '#F7F8FA',
            'bg_card': '#FFFFFF',
            'text_primary': '#2C3E50',
            'border_light': '#E0E0E0',
            'accent': '#DDA15C',
            'accent_hover': '#BB6C25',
            'text_inverse': '#FFFFFF',
            'bg_transparent': 'transparent',
        }

        self.configure(fg_color=COLORS['bg_window'])

        # Main container
        main_frame = CTk.CTkFrame(self, fg_color=COLORS['bg_transparent'])
        main_frame.pack(fill='both', expand=True, padx=20, pady=20)

        # Header
        header_label = CTk.CTkLabel(
            main_frame,
            text="저장할 카테고리를 선택하세요",
            font=(font_family, font_size + 2, 'bold'),
            text_color=COLORS['text_primary']
        )
        header_label.pack(pady=(0, 15))

        # Category selection frame
        category_frame = CTk.CTkFrame(
            main_frame,
            fg_color=COLORS['bg_card'],
            border_width=1,
            border_color=COLORS['border_light'],
            corner_radius=8
        )
        category_frame.pack(fill='both', expand=True, pady=(0, 15))

        # Get all categories
        categories = list(self.presets_data.get("categories", {}).keys())

        # Determine initial selection: use provided initial_category if valid, else "미분류"
        if initial_category and initial_category in categories:
            default_category = initial_category
        else:
            default_category = "미분류" if "미분류" in categories else (categories[0] if categories else "미분류")

        # Category radio buttons
        self.category_var = tk.StringVar(value=default_category)
        # Sort with "미분류" first
        categories.sort(key=lambda x: (x != "미분류", x))

        # Scrollable frame for categories
        scroll_frame = CTk.CTkScrollableFrame(
            category_frame,
            fg_color=COLORS['bg_transparent']
        )
        scroll_frame.pack(fill='both', expand=True, padx=10, pady=10)

        for category in categories:
            radio = CTk.CTkRadioButton(
                scroll_frame,
                text=category,
                variable=self.category_var,
                value=category,
                font=(font_family, font_size),
                text_color=COLORS['text_primary'],
                fg_color=COLORS['accent'],
                hover_color=COLORS['accent_hover'],
                border_color=COLORS['border_light']
            )
            radio.pack(anchor='w', pady=5, padx=10)

        # New category section
        new_category_frame = CTk.CTkFrame(main_frame, fg_color=COLORS['bg_transparent'])
        new_category_frame.pack(fill='x', pady=(0, 15))

        new_category_label = CTk.CTkLabel(
            new_category_frame,
            text="새 카테고리:",
            font=(font_family, font_size),
            text_color=COLORS['text_primary']
        )
        new_category_label.pack(side='left', padx=(0, 10))

        self.new_category_entry = CTk.CTkEntry(
            new_category_frame,
            placeholder_text="새 카테고리 이름 입력",
            font=(font_family, font_size)
        )
        self.new_category_entry.pack(side='left', fill='x', expand=True)

        # Buttons
        button_frame = CTk.CTkFrame(main_frame, fg_color=COLORS['bg_transparent'])
        button_frame.pack(fill='x')

        cancel_button = CTk.CTkButton(
            button_frame,
            text="취소",
            width=100,
            height=32,
            fg_color=COLORS['bg_card'],
            hover_color='#F0F0F0',
            text_color=COLORS['text_primary'],
            border_width=1,
            border_color=COLORS['border_light'],
            font=(font_family, font_size),
            command=self.cancel,
            corner_radius=6
        )
        cancel_button.pack(side='left', padx=(0, 10))

        confirm_button = CTk.CTkButton(
            button_frame,
            text="확인",
            width=100,
            height=32,
            fg_color=COLORS['accent'],
            hover_color=COLORS['accent_hover'],
            text_color=COLORS['text_inverse'],
            font=(font_family, font_size),
            command=self.confirm,
            corner_radius=6
        )
        confirm_button.pack(side='left')

        # Keyboard shortcuts
        self.bind("<Return>", lambda e: self.confirm())
        self.bind("<Escape>", lambda e: self.cancel())
        self.protocol("WM_DELETE_WINDOW", self.cancel)

    def confirm(self):
        """Confirm category selection."""
        # Check if user entered a new category
        new_category = self.new_category_entry.get().strip()
        if new_category:
            self.selected_category = new_category
        else:
            self.selected_category = self.category_var.get()

        self.destroy()

    def cancel(self):
        """Cancel the dialog."""
        self.selected_category = None
        self.destroy()


class StyledInputDialog(CTk.CTkToplevel):
    """Uniform input dialog matching PresetManagerDialog button styles."""

    def __init__(self, parent, title, prompt, colors, prefill="", default_font=None):
        super().__init__(parent)
        self.title(title)
        self.geometry("360x150")
        self.minsize(320, 140)
        self.transient(parent)
        self.grab_set()
        self.focus_force()

        self._value = None
        self.colors = colors
        self.default_font = default_font or "Arial"
        font_family = self.default_font[0] if isinstance(self.default_font, tuple) else self.default_font
        font_size = 12

        container = CTk.CTkFrame(self, fg_color=colors['bg_window'])
        container.pack(fill='both', expand=True, padx=14, pady=14)
        container.grid_columnconfigure(0, weight=1)

        CTk.CTkLabel(
            container,
            text=prompt,
            font=(font_family, font_size),
            text_color=colors['text_primary'],
            anchor='w',
            justify='left'
        ).grid(row=0, column=0, sticky='w', pady=(0, 10))

        self.entry = CTk.CTkEntry(
            container,
            font=(font_family, font_size),
            fg_color=colors['bg_card'],
            border_color=colors['border_light'],
            text_color="black"
        )
        self.entry.grid(row=1, column=0, sticky='ew', pady=(0, 12))
        if prefill:
            self.entry.insert(0, prefill)
            self.entry.select_range(0, tk.END)
        self.entry.focus_set()

        btns = CTk.CTkFrame(container, fg_color=colors['bg_transparent'])
        btns.grid(row=2, column=0, sticky='e')

        cancel_btn = CTk.CTkButton(
            btns,
            text="취소",
            width=90,
            height=30,
            fg_color=colors['bg_card'],
            hover_color="#F0F0F0",
            text_color=colors['text_primary'],
            border_width=1,
            border_color=colors['border_light'],
            command=self._on_cancel,
            corner_radius=6,
            font=(font_family, font_size)
        )
        cancel_btn.pack(side='left', padx=(0, 8))

        ok_btn = CTk.CTkButton(
            btns,
            text="확인",
            width=90,
            height=30,
            fg_color=colors['accent'],
            hover_color=colors['accent_hover'],
            text_color=colors['text_inverse'],
            command=self._on_ok,
            corner_radius=6,
            font=(font_family, font_size)
        )
        ok_btn.pack(side='left')

        self.bind("<Return>", lambda e: self._on_ok())
        self.bind("<KP_Enter>", lambda e: self._on_ok())
        self.bind("<Escape>", lambda e: self._on_cancel())
        self.protocol("WM_DELETE_WINDOW", self._on_cancel)

    def _on_ok(self):
        self._value = self.entry.get().strip()
        self.destroy()

    def _on_cancel(self):
        self._value = None
        self.destroy()

    def get_input(self):
        self.wait_window()
        return self._value


class ImportChoiceDialog(CTk.CTkToplevel):
    """Dialog for choosing import mode: overwrite, merge, or cancel."""

    def __init__(self, parent, colors, default_font=None):
        super().__init__(parent)
        self.title("가져오기 방식 선택")
        self.geometry("420x180")
        self.minsize(400, 160)
        self.transient(parent)
        self.grab_set()
        self.focus_force()
        self.configure(fg_color="white")

        self._choice = None  # "overwrite", "merge", or None (cancel)
        self.colors = colors
        self.default_font = default_font or "Arial"
        font_family = self.default_font[0] if isinstance(self.default_font, tuple) else self.default_font
        font_size = 12

        container = CTk.CTkFrame(self, fg_color="white")
        container.pack(fill='both', expand=True, padx=20, pady=20)

        # Message
        message = (
            "기존 프리셋을 어떻게 처리하시겠습니까?\n\n"
            "• 덮어쓰기: 기존 프리셋 삭제 후 새 파일로 교체\n"
            "• 합치기: 기존 프리셋 유지하며 새 프리셋 추가"
        )
        CTk.CTkLabel(
            container,
            text=message,
            font=(font_family, font_size),
            text_color=colors['text_primary'],
            anchor='w',
            justify='left'
        ).pack(fill='x')

        # Buttons frame - pack at bottom
        btns = CTk.CTkFrame(container, fg_color='transparent')
        btns.pack(fill='x', side='bottom')

        # Cancel button (left side)
        CTk.CTkButton(
            btns,
            text="취소",
            width=100,
            height=32,
            fg_color=colors['bg_card'],
            hover_color="#F0F0F0",
            text_color=colors['text_primary'],
            border_width=1,
            border_color=colors['border_light'],
            command=self._on_cancel,
            corner_radius=6,
            font=(font_family, font_size)
        ).pack(side='left')

        # Merge button (right side)
        CTk.CTkButton(
            btns,
            text="합치기",
            width=100,
            height=32,
            fg_color=colors['accent'],
            hover_color=colors['accent_hover'],
            text_color=colors['text_inverse'],
            command=self._on_merge,
            corner_radius=6,
            font=(font_family, font_size, 'bold')
        ).pack(side='right')

        # Overwrite button (right side, before merge)
        CTk.CTkButton(
            btns,
            text="덮어쓰기",
            width=100,
            height=32,
            fg_color="#D9534F",
            hover_color="#C9302C",
            text_color="white",
            command=self._on_overwrite,
            corner_radius=6,
            font=(font_family, font_size, 'bold')
        ).pack(side='right', padx=(0, 10))

        self.bind("<Escape>", lambda e: self._on_cancel())
        self.protocol("WM_DELETE_WINDOW", self._on_cancel)

    def _on_overwrite(self):
        self._choice = "overwrite"
        self.destroy()

    def _on_merge(self):
        self._choice = "merge"
        self.destroy()

    def _on_cancel(self):
        self._choice = None
        self.destroy()

    def get_choice(self):
        self.wait_window()
        return self._choice


class TransformOptionsPopup(CTk.CTkToplevel):
    """Modal window to select passage transformation options."""

    def __init__(
        self,
        parent,
        transform_var,
        difficulty_var,
        length_var,
        content_change_var,
        font_family,
        font_size,
        on_confirm,
        on_cancel,
    ):
        super().__init__(parent)

        # === Design Token System ===
        SPACING = {
            'xs': 4,   # Radio button internal spacing
            'sm': 8,   # Between radio buttons
            'md': 12,  # Label-to-content gap
            'lg': 16,  # Section separation
            'xl': 20,  # Header top margin
            'xxl': 24  # Outer container padding
        }
        CARD_GAP = SPACING['sm']

        COLORS = {
            # Text
            'text_primary': '#2C3E50',
            'text_inverse': '#FFFFFF',
            'text_disabled': '#A0A6AD',

            # Backgrounds
            'bg_window': '#F7F8FA',
            'bg_card': '#FFFFFF',
            'bg_transparent': 'transparent',

            # Borders
            'border_light': '#E0E0E0',
            'border_medium': '#D0D0D0',
            'border_disabled': '#D8D8D8',

            # Interactive
            'accent': '#DDA15C',
            'accent_hover': '#BB6C25',
            'secondary': '#FFFFFF',
            'secondary_hover': '#F0F0F0'
        }
        
        # === Window Configuration ===
        BASE_WIDTH = 600
        BASE_HEIGHT = 400
        
        self.title("지문 변형 옵션")
        self.geometry(f"{BASE_WIDTH}x{BASE_HEIGHT}")
        self.minsize(width=600, height=400)
        self.resizable(True, True)
        self.transient(parent)
        self.grab_set()
        self.focus_force()
        self.configure(fg_color=COLORS['bg_window'])
        
        # === Instance Variables ===
        self.transform_var = transform_var
        self.difficulty_var = difficulty_var
        self.length_var = length_var
        self.content_change_var = content_change_var
        self.initial_transform = transform_var.get()
        self.initial_difficulty = difficulty_var.get()
        self.initial_length = length_var.get()
        self.initial_content_change = content_change_var.get()
        self.on_confirm = on_confirm
        self.on_cancel = on_cancel
        self._option_controls = []
        self._section_labels = []
        self._colors = COLORS
        
        # === Font Sizing ===
        header_font_size = max(font_size + 2, 14)
        body_font_size = max(font_size, 12)
        
        # === HEADER SECTION ===
        header_label = CTk.CTkLabel(
            self,
            text="지문 변형 옵션을 선택하세요.",
            font=(font_family, header_font_size, 'bold'),
            text_color=COLORS['text_primary'],
            anchor="center"
        )
        header_label.pack(pady=(SPACING['xl'], SPACING['lg']))

        # === TRANSFORM TOGGLE ===
        toggle_frame = CTk.CTkFrame(
            self,
            fg_color=COLORS['bg_transparent'],
            border_width=0
        )
        toggle_frame.pack(padx=SPACING['xxl'], pady=(0, SPACING['md']), fill='x')
        self.transform_checkbox = CTk.CTkCheckBox(
            toggle_frame,
            text="지문을 변형하여 출제",
            font=(font_family, body_font_size, 'bold'),
            variable=self.transform_var,
            text_color=COLORS['text_primary'],
            fg_color=COLORS['accent'],
            hover_color=COLORS['accent_hover'],
            border_color=COLORS['border_medium'],
            border_width=1,
            command=self.update_option_states,
        )
        self.transform_checkbox.pack(anchor="w")
        
        # === OPTIONS CONTAINER ===
        options_frame = CTk.CTkFrame(
            self,
            fg_color=COLORS['bg_transparent'],
            border_width=0
        )
        options_frame.pack(
            pady=(0, SPACING['lg']),
            padx=SPACING['xxl'],
            fill='both',
            expand=True
        )
        options_frame.grid_columnconfigure(0, weight=0, minsize=100)
        options_frame.grid_columnconfigure(1, weight=1)
        
        # === DIFFICULTY CARD ===
        difficulty_card = CTk.CTkFrame(
            options_frame,
            fg_color=COLORS['bg_card'],
            corner_radius=8,
            border_width=1,
            border_color=COLORS['border_light']
        )
        difficulty_card.grid(
            row=0, column=0, columnspan=2,
            sticky="ew",
            pady=(0, CARD_GAP),
            padx=0
        )
        difficulty_card.grid_columnconfigure(1, weight=1)
        
        # Difficulty label
        difficulty_label = CTk.CTkLabel(
            difficulty_card,
            text="지문 난이도:",
            font=(font_family, body_font_size, 'bold'),
            text_color=COLORS['text_primary'],
            anchor="w"
        )
        difficulty_label.grid(
            row=0, column=0,
            padx=SPACING['md'],
            pady=SPACING['sm'],
            sticky="w"
        )
        self._section_labels.append(difficulty_label)
        
        # Difficulty radio group
        difficulty_group = CTk.CTkFrame(difficulty_card, fg_color=COLORS['bg_transparent'])
        difficulty_group.grid(
            row=0, column=1,
            padx=(0, SPACING['md']),
            pady=SPACING['sm'],
            sticky="ew"
        )
        
        difficulty_options = [
            ('더 쉽게', 1),
            ('같은 난이도', 2),
            ('더 높게', 3)
        ]
        
        for text, value in difficulty_options:
            radio = CTk.CTkRadioButton(
                difficulty_group,
                text=text,
                font=(font_family, body_font_size),
                variable=self.difficulty_var,
                value=value,
                text_color=COLORS['text_primary'],
                hover_color=COLORS['accent_hover'],
                fg_color=COLORS['accent'],
                border_color=COLORS['border_medium']
            )
            radio.pack(side='left', padx=SPACING['xs'])
            self._option_controls.append(radio)

        # === LENGTH CARD ===
        length_card = CTk.CTkFrame(
            options_frame,
            fg_color=COLORS['bg_card'],
            corner_radius=8,
            border_width=1,
            border_color=COLORS['border_light']
        )
        length_card.grid(
            row=1, column=0, columnspan=2,
            sticky="ew",
            pady=(0, CARD_GAP),
            padx=0
        )
        length_card.grid_columnconfigure(1, weight=1)
        
        # Length label
        length_label = CTk.CTkLabel(
            length_card,
            text="지문 길이:",
            font=(font_family, body_font_size, 'bold'),
            text_color=COLORS['text_primary'],
            anchor="w"
        )
        length_label.grid(
            row=0, column=0,
            padx=SPACING['md'],
            pady=SPACING['sm'],
            sticky="w"
        )
        self._section_labels.append(length_label)
        
        # Length radio group
        length_group = CTk.CTkFrame(length_card, fg_color=COLORS['bg_transparent'])
        length_group.grid(
            row=0, column=1,
            padx=(0, SPACING['md']),
            pady=SPACING['sm'],
            sticky="ew"
        )
        
        length_options = [
            ('더 짧게', 1),
            ('비슷하게', 2),
            ('더 길게', 3)
        ]
        
        for text, value in length_options:
            radio = CTk.CTkRadioButton(
                length_group,
                text=text,
                font=(font_family, body_font_size),
                variable=self.length_var,
                value=value,
                text_color=COLORS['text_primary'],
                hover_color=COLORS['accent_hover'],
                fg_color=COLORS['accent'],
                border_color=COLORS['border_medium']
            )
            radio.pack(side='left', padx=SPACING['xs'])
            self._option_controls.append(radio)

        # === CONTENT CHANGE CARD ===
        content_change_card = CTk.CTkFrame(
            options_frame,
            fg_color=COLORS['bg_card'],
            corner_radius=8,
            border_width=1,
            border_color=COLORS['border_light']
        )
        content_change_card.grid(
            row=2, column=0, columnspan=2,
            sticky="ew",
            pady=0,
            padx=0
        )
        content_change_card.grid_columnconfigure(1, weight=1)

        # Content change label
        content_change_label = CTk.CTkLabel(
            content_change_card,
            text="내용 변경 방식:",
            font=(font_family, body_font_size, 'bold'),
            text_color=COLORS['text_primary'],
            anchor="w"
        )
        content_change_label.grid(
            row=0, column=0,
            padx=SPACING['md'],
            pady=SPACING['sm'],
            sticky="w"
        )
        self._section_labels.append(content_change_label)

        # Content change radio group
        content_change_group = CTk.CTkFrame(content_change_card, fg_color=COLORS['bg_transparent'])
        content_change_group.grid(
            row=0, column=1,
            padx=(0, SPACING['md']),
            pady=SPACING['sm'],
            sticky="ew"
        )

        content_change_options = [
            ('내용 변화 없음', 0, '글의 요지 및 세부사항을 그대로 유지합니다.'),
            ('세부 사항 변경', 1, '글의 요지는 유지하되, 세부사항에 변화를 줍니다.'),
            ('요지 변경', 2, '글의 요지 자체를 바꿉니다'),
            ('간접연계', 3, 'EBS 간접연계처럼 소재만 연계하여 새로운 글을 씁니다.')
        ]

        for text, value, tooltip_text in content_change_options:
            radio = CTk.CTkRadioButton(
                content_change_group,
                text=text,
                font=(font_family, body_font_size),
                variable=self.content_change_var,
                value=value,
                text_color=COLORS['text_primary'],
                hover_color=COLORS['accent_hover'],
                fg_color=COLORS['accent'],
                border_color=COLORS['border_medium']
            )
            radio.pack(side='left', padx=SPACING['xs'])
            ToolTip유형설명(radio, tooltip_text, font_family)
            self._option_controls.append(radio)

        # === BUTTON SECTION ===
        button_frame = CTk.CTkFrame(self, fg_color=COLORS['bg_transparent'])
        button_frame.pack(pady=(SPACING['md'], SPACING['xl']), fill='x')
        
        button_container = CTk.CTkFrame(button_frame, fg_color=COLORS['bg_transparent'])
        button_container.pack(anchor='center')
        
        BUTTON_WIDTH = 100
        BUTTON_HEIGHT = 36
        
        
        # (cancel)
        cancel_button = CTk.CTkButton(
            button_container,
            text="취소",
            width=BUTTON_WIDTH,
            height=BUTTON_HEIGHT,
            fg_color=COLORS['secondary'],
            hover_color=COLORS['secondary_hover'],
            text_color=COLORS['text_primary'],
            border_width=1,
            border_color=COLORS['border_medium'],
            font=(font_family, body_font_size, 'bold'),
            command=self.handle_cancel,
            corner_radius=6,
            cursor="hand2"
        )
        cancel_button.pack(side=tk.LEFT, padx=SPACING['sm'])

        #(confirm)
        confirm_button = CTk.CTkButton(
            button_container,
            text="확인",
            width=BUTTON_WIDTH,
            height=BUTTON_HEIGHT,
            fg_color=COLORS['accent'],
            hover_color=COLORS['accent_hover'],
            text_color=COLORS['text_inverse'],
            font=(font_family, body_font_size, 'bold'),
            command=self.handle_confirm,
            corner_radius=6,
            cursor="hand2"
        )
        confirm_button.pack(side=tk.LEFT, padx=SPACING['sm'])


        # === KEYBOARD SHORTCUTS ===
        self.bind("<Return>", lambda e: self.handle_confirm())
        self.bind("<Escape>", lambda e: self.handle_cancel())
        
        self.protocol("WM_DELETE_WINDOW", self.handle_cancel)
        self.update_option_states()

    def update_option_states(self):
        """Enable or disable option controls based on transform toggle."""
        is_enabled = bool(self.transform_var.get())
        state = tk.NORMAL if is_enabled else tk.DISABLED
        text_color = self._colors['text_primary'] if is_enabled else self._colors['text_disabled']
        border_color = self._colors['border_medium'] if is_enabled else self._colors['border_disabled']
        fg_color = self._colors['accent'] if is_enabled else self._colors['border_disabled']

        for label in self._section_labels:
            label.configure(text_color=text_color)

        for control in self._option_controls:
            try:
                control.configure(
                    state=state,
                    text_color=text_color,
                    border_color=border_color,
                    fg_color=fg_color
                )
            except tk.TclError:
                try:
                    control.configure(state=state, text_color=text_color, fg_color=fg_color)
                except tk.TclError:
                    control.configure(state=state)

    def handle_confirm(self):
        try:
            self.grab_release()
        except tk.TclError:
            pass
        if self.on_confirm:
            self.on_confirm()
        self.destroy()

    def handle_cancel(self):
        self.transform_var.set(self.initial_transform)
        self.update_option_states()
        self.difficulty_var.set(self.initial_difficulty)
        self.length_var.set(self.initial_length)
        self.content_change_var.set(self.initial_content_change)
        try:
            self.grab_release()
        except tk.TclError:
            pass
        if self.on_cancel:
            self.on_cancel()
        self.destroy()

class PresetTooltip:
    """Simple tooltip widget for showing preset question preview."""

    def __init__(self, widget, text, font_family, font_size):
        self.widget = widget
        self.text = text
        self.font_family = font_family
        self.font_size = font_size
        self.tooltip_window = None

        # Bind events
        self.widget.bind("<Enter>", self.show_tooltip)
        self.widget.bind("<Leave>", self.hide_tooltip)

    def show_tooltip(self, event=None):
        """Show tooltip after a short delay."""
        if self.tooltip_window or not self.text:
            return

        # Create tooltip window
        x = self.widget.winfo_rootx() + 20
        y = self.widget.winfo_rooty() + 20

        self.tooltip_window = CTk.CTkToplevel(self.widget)
        self.tooltip_window.wm_overrideredirect(True)
        self.tooltip_window.wm_geometry(f"+{x}+{y}")

        # Create frame
        frame = CTk.CTkFrame(
            self.tooltip_window,
            fg_color="#2C3E50",
            corner_radius=6,
            border_width=1,
            border_color="#DDA15C"
        )
        frame.pack(fill='both', expand=True)

        # Add label with text
        label = CTk.CTkLabel(
            frame,
            text=self.text,
            font=(self.font_family, self.font_size - 1),
            text_color="white",
            justify="left",
            wraplength=300
        )
        label.pack(padx=8, pady=6)

    def hide_tooltip(self, event=None):
        """Hide tooltip."""
        if self.tooltip_window:
            self.tooltip_window.destroy()
            self.tooltip_window = None

class PresetManagerDialog(CTk.CTkToplevel):
    """Dialog for managing preset categories with drag-and-drop support.

    Can be launched either from CopyQuestionPopup (provider supplied) or standalone
    from the main menu (falls back to internal file-backed provider).
    """

    def __init__(
        self,
        parent,
        presets_data=None,
        default_font=None,
        on_close_callback=None,
        provider=None,
        transient_parent=None,
    ):
        super().__init__(parent)

        # Window behaviour
        self.title("동형문제 프리셋 관리")
        self.geometry("850x600")
        self.minsize(width=850, height=600)
        # Use provided transient parent for menu launch, otherwise fallback to parent
        self.transient(transient_parent if transient_parent else parent)
        self.grab_set()
        self.focus_force()

        # Provider handles persistence/mutations. Default to self (file-backed mode).
        self.provider = provider if provider else self

        # Ensure storage paths when running standalone
        self._init_storage()

        # Initial presets data (dict shared with provider when passed in)
        if presets_data is None:
            self.presets_data = self._load_presets()
        else:
            self.presets_data = presets_data

        self.default_font = default_font or "Arial"
        self.font_family = self.default_font[0] if isinstance(self.default_font, tuple) else self.default_font
        self.font_size = 12
        self.on_close_callback = on_close_callback

        self.selected_category = None
        self.selected_preset = None
        self.selected_presets = set()  # Track multiple selected presets
        self.last_clicked_preset = None  # Track last clicked item for shift-click selection
        self.preset_order = []  # Track preset display order

        # Build default provider for standalone usage
        self._default_provider = self._build_default_provider()
        if provider is None:
            self.provider = self._default_provider
        else:
            self.provider = provider
            if hasattr(self.provider, "presets"):
                self.presets_data = self.provider.presets

        # Drag-and-drop state (presets)
        self.dragging_preset = None
        self.drag_source_category = None
        self.drag_ghost_frame = None
        self.drop_indicator = None  # Visual indicator for drop position
        self.preset_checkboxes = {}  # Track checkbox widgets
        self.preset_tooltips = {}  # Track tooltip widgets
        self.preset_item_frames = {}  # Track preset item frames for drop indicator

        # Drag-and-drop state (categories)
        self.dragging_category = None
        self.category_ghost_frame = None
        self.category_drop_indicator = None
        self.category_item_frames = {}  # Track category item frames for drop indicator
        self.drag_candidate_category = None
        self.drag_start_x = None
        self.drag_start_y = None
        self.category_drop_target = None
        self.category_drop_before = None

        # Colors
        self.COLORS = {
            'bg_window': '#F7F8FA',
            'bg_card': '#FFFFFF',
            'text_primary': '#2C3E50',
            'border_light': '#E0E0E0',
            'accent': '#DDA15C',
            'accent_hover': '#BB6C25',
            'selected': '#FEF9E0',
            'hover_category': '#E8F5E9',
            'bg_transparent': 'transparent',
            'text_inverse': '#FFFFFF'
        }

        self.configure(fg_color=self.COLORS['bg_window'])

        # Main container
        main_frame = CTk.CTkFrame(self, fg_color='transparent')
        main_frame.pack(fill='both', expand=True, padx=20, pady=20)
        main_frame.grid_columnconfigure(0, weight=1)
        main_frame.grid_columnconfigure(1, weight=2)
        main_frame.grid_rowconfigure(1, weight=1)

        # Header
        header_label = CTk.CTkLabel(
            main_frame,
            text="동형문제 프리셋 카테고리 관리",
            font=(self.font_family, self.font_size + 4, 'bold'),
            text_color=self.COLORS['text_primary']
        )
        header_label.grid(row=0, column=0, columnspan=2, pady=(0, 20))

        # Left Panel - Categories
        category_frame = CTk.CTkFrame(main_frame, fg_color=self.COLORS['bg_card'], border_width=1, border_color=self.COLORS['border_light'])
        category_frame.grid(row=1, column=0, sticky='nsew', padx=(0, 10))

        category_header = CTk.CTkLabel(
            category_frame,
            text="카테고리",
            font=(self.font_family, self.font_size, 'bold'),
            text_color=self.COLORS['text_primary']
        )
        category_header.pack(pady=10)

        # Category list (scrollable)
        self.category_list_frame = CTk.CTkScrollableFrame(
            category_frame,
            fg_color='transparent',
            height=300
        )
        self.category_list_frame.pack(fill='both', expand=True, padx=10, pady=(0, 10))

        # Category buttons
        category_btn_frame = CTk.CTkFrame(category_frame, fg_color='transparent')
        category_btn_frame.pack(fill='x', padx=10, pady=10)

        CTk.CTkButton(
            category_btn_frame,
            text="추가",
            command=self.add_category,
            width=50,
            height=28,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color='white',
            font=(self.font_family, self.font_size - 1)
        ).pack(side='left', padx=2)

        self.rename_btn = CTk.CTkButton(
            category_btn_frame,
            text="이름변경",
            command=self.rename_category,
            width=70,
            height=28,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color='white',
            font=(self.font_family, self.font_size - 1),
            state='disabled'
        )
        self.rename_btn.pack(side='left', padx=2)

        self.delete_btn = CTk.CTkButton(
            category_btn_frame,
            text="삭제",
            command=self.delete_category,
            width=50,
            height=28,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color='white',
            font=(self.font_family, self.font_size - 1),
            state='disabled'
        )
        self.delete_btn.pack(side='left', padx=2)

        CTk.CTkButton(
            category_btn_frame,
            text="정렬▲",
            command=self.sort_categories_ascending,
            width=55,
            height=28,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color='white',
            font=(self.font_family, self.font_size - 1)
        ).pack(side='left', padx=2)

        CTk.CTkButton(
            category_btn_frame,
            text="정렬▼",
            command=self.sort_categories_descending,
            width=55,
            height=28,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color='white',
            font=(self.font_family, self.font_size - 1)
        ).pack(side='left', padx=2)

        # Right Panel - Presets
        preset_frame = CTk.CTkFrame(main_frame, fg_color=self.COLORS['bg_card'], border_width=1, border_color=self.COLORS['border_light'])
        preset_frame.grid(row=1, column=1, sticky='nsew')

        self.preset_header = CTk.CTkLabel(
            preset_frame,
            text="프리셋 (카테고리를 선택하세요)",
            font=(self.font_family, self.font_size, 'bold'),
            text_color=self.COLORS['text_primary']
        )
        self.preset_header.pack(pady=10)

        # Selection control bar
        selection_bar = CTk.CTkFrame(preset_frame, fg_color='transparent')
        selection_bar.pack(fill='x', padx=10, pady=(0, 5))

        CTk.CTkButton(
            selection_bar,
            text="추가",
            command=self.add_preset,
            width=50,
            height=24,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color='white',
            font=(self.font_family, self.font_size - 2)
        ).pack(side='left', padx=2)

        self.select_toggle_btn = CTk.CTkButton(
            selection_bar,
            text="전체 선택",
            command=self.toggle_select_all_presets,
            width=80,
            height=24,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color='white',
            font=(self.font_family, self.font_size - 2)
        )
        self.select_toggle_btn.pack(side='left', padx=2)

        # Sorting buttons
        CTk.CTkButton(
            selection_bar,
            text="정렬 ▲",
            command=self.sort_presets_ascending,
            width=60,
            height=24,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color='white',
            font=(self.font_family, self.font_size - 2)
        ).pack(side='left', padx=2)

        CTk.CTkButton(
            selection_bar,
            text="정렬 ▼",
            command=self.sort_presets_descending,
            width=60,
            height=24,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color='white',
            font=(self.font_family, self.font_size - 2)
        ).pack(side='left', padx=2)

        self.selection_count_label = CTk.CTkLabel(
            selection_bar,
            text="선택: 0개",
            font=(self.font_family, self.font_size - 2),
            text_color=self.COLORS['text_primary']
        )
        self.selection_count_label.pack(side='right', padx=5)

        # Preset list (scrollable)
        self.preset_list_frame = CTk.CTkScrollableFrame(
            preset_frame,
            fg_color='transparent',
            height=300
        )
        self.preset_list_frame.pack(fill='both', expand=True, padx=10, pady=(0, 10))

        # Bottom buttons
        bottom_frame = CTk.CTkFrame(main_frame, fg_color='transparent')
        bottom_frame.grid(row=2, column=0, columnspan=2, pady=(20, 0))

        CTk.CTkButton(
            bottom_frame,
            text="Import",
            command=self.import_presets,
            width=80,
            height=32,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color='white',
            font=(self.font_family, self.font_size, 'bold')
        ).pack(side='left', padx=5)

        CTk.CTkButton(
            bottom_frame,
            text="Export",
            command=self.export_presets,
            width=80,
            height=32,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color='white',
            font=(self.font_family, self.font_size, 'bold')
        ).pack(side='left', padx=5)

        CTk.CTkButton(
            bottom_frame,
            text="닫기",
            command=self.close_dialog,
            width=100,
            height=32,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color='white',
            font=(self.font_family, self.font_size, 'bold')
        ).pack(side='left', padx=5)

        # Initialize UI
        self.refresh_category_list()

        # Restore last selected category, or fall back to "미분류"
        categories = self.presets_data.get("categories", {})
        last_selected = self.presets_data.get("last_selected_category")
        if last_selected and last_selected in categories:
            self.select_category(last_selected)
        elif "미분류" in categories:
            self.select_category("미분류")

        self.protocol("WM_DELETE_WINDOW", self.close_dialog)
        self.bind("<Escape>", lambda e: self.close_dialog())

    def refresh_category_list(self):
        """Refresh the category list display."""
        # COMPLETE drag state cleanup
        self.category_item_frames.clear()
        self._clear_category_drop_indicator()
        self.dragging_category = None
        self.drag_candidate_category = None
        self.drag_start_x = None
        self.drag_start_y = None
        if hasattr(self, 'category_ghost_frame') and self.category_ghost_frame:
            try:
                self.category_ghost_frame.destroy()
            except:
                pass
            self.category_ghost_frame = None

        # Clear existing category items
        for widget in self.category_list_frame.winfo_children():
            widget.destroy()

        categories = self.presets_data.get("categories", {})

        # Get custom order or fall back to alphabetical
        custom_order = self.get_category_order()

        if custom_order:
            # Validate: keep only existing categories
            valid_order = [c for c in custom_order if c in categories]

            # Add new categories not in order (after 미분류 if present)
            new_cats = sorted([c for c in categories if c not in valid_order])

            if "미분류" in valid_order:
                # Insert new categories after 미분류
                category_names = valid_order[:1] + new_cats + valid_order[1:]
            else:
                category_names = valid_order + new_cats

            # Ensure 미분류 is first if it exists
            if "미분류" in category_names and category_names[0] != "미분류":
                category_names = ["미분류"] + [c for c in category_names if c != "미분류"]

            # Update if changed
            if category_names != custom_order:
                self.set_category_order(category_names)
        else:
            # No custom order - use alphabetical with 미분류 first
            category_names = sorted(categories.keys(), key=lambda x: (x != "미분류", x))

        # Create UI items
        for category_name in category_names:
            self.create_category_item(category_name)

        # Auto-select "미분류" if it's the only category
        if len(category_names) == 1 and category_names[0] == "미분류":
            self.select_category("미분류")

    def create_category_item(self, category_name):
        """Create a category item widget."""
        item_frame = CTk.CTkFrame(
            self.category_list_frame,
            fg_color='transparent',
            cursor='hand2'
        )
        item_frame.pack(fill='x', pady=2)

        # Count presets (excluding _order metadata key)
        category_data = self.presets_data["categories"].get(category_name, {})
        preset_count = sum(1 for k in category_data if k != "_order")

        label = CTk.CTkLabel(
            item_frame,
            text=f"📁 {category_name} ({preset_count})",
            font=(self.font_family, self.font_size),
            anchor='w',
            text_color='black'
        )
        label.pack(fill='x', padx=5, pady=5)

        # Bind drag events to both label and frame
        for widget in [label, item_frame]:
            widget.bind("<Button-1>", lambda e, cat=category_name: self.on_category_mouse_down(e, cat))
            widget.bind("<B1-Motion>", self.on_category_mouse_motion)
            widget.bind("<ButtonRelease-1>", self.on_category_mouse_up)

        # Store reference for highlighting and drag operations
        item_frame.category_name = category_name
        item_frame.label = label
        self.category_item_frames[category_name] = item_frame

    def select_category(self, category_name):
        """Handle category selection."""
        self.selected_category = category_name
        self.selected_preset = None

        # Save the selected category to presets file for persistence
        self.presets_data["last_selected_category"] = category_name
        self.provider.save_presets_to_file()

        # Update button states
        if category_name == "미분류":
            self.rename_btn.configure(state='disabled')
            self.delete_btn.configure(state='disabled')
        else:
            self.rename_btn.configure(state='normal')
            self.delete_btn.configure(state='normal')

        # Highlight selected category
        for widget in self.category_list_frame.winfo_children():
            if hasattr(widget, 'category_name'):
                if widget.category_name == category_name:
                    widget.configure(fg_color='#FEF9E0')
                else:
                    widget.configure(fg_color='transparent')

        # Update preset header
        self.preset_header.configure(text=f"프리셋 - {category_name}")

        # Refresh preset list
        self.refresh_preset_list()

    def get_preset_order(self, category):
        """Get the custom order for presets in a category, or None if not set."""
        category_data = self.presets_data["categories"].get(category, {})
        return category_data.get("_order", None)

    def get_current_preset_count(self):
        """Return number of presets in the selected category (excluding metadata)."""
        if not self.selected_category:
            return 0
        presets = self.presets_data["categories"].get(self.selected_category, {})
        return sum(1 for k in presets if k != "_order")

    def set_preset_order(self, category, order_list):
        """Set the custom order for presets in a category."""
        if category in self.presets_data["categories"]:
            self.presets_data["categories"][category]["_order"] = order_list
            self.provider.save_presets_to_file()

    def get_category_order(self):
        """Get saved category order, ensuring 미분류 is first."""
        order = self.presets_data.get("_category_order")
        if not order:
            return None

        # Auto-fix if 미분류 isn't first
        if "미분류" in order and order[0] != "미분류":
            order = ["미분류"] + [c for c in order if c != "미분류"]
            self.set_category_order(order)  # Saves automatically

        return order

    def set_category_order(self, order_list):
        """Save category order to presets and persist to file."""
        if order_list:
            self.presets_data["_category_order"] = order_list
            self.provider.save_presets_to_file()

    def sort_categories_ascending(self):
        """Sort categories A→Z (미분류 stays first)."""
        categories = self.presets_data.get("categories", {})
        if not categories:
            return

        uncategorized = ["미분류"] if "미분류" in categories else []
        others = sorted([c for c in categories.keys() if c != "미분류"])
        new_order = uncategorized + others
        self.set_category_order(new_order)
        self.refresh_category_list()
        if self.selected_category:
            self.select_category(self.selected_category if self.selected_category in new_order else new_order[0])

    def sort_categories_descending(self):
        """Sort categories Z→A (미분류 stays first)."""
        categories = self.presets_data.get("categories", {})
        if not categories:
            return

        uncategorized = ["미분류"] if "미분류" in categories else []
        others = sorted([c for c in categories.keys() if c != "미분류"], reverse=True)
        new_order = uncategorized + others
        self.set_category_order(new_order)
        self.refresh_category_list()
        if self.selected_category:
            self.select_category(self.selected_category if self.selected_category in new_order else new_order[0])

    def refresh_preset_list(self):
        """Refresh the preset list for selected category."""
        # Clear existing preset items
        for widget in self.preset_list_frame.winfo_children():
            widget.destroy()

        # Clear selection tracking
        self.selected_presets.clear()
        self.preset_checkboxes.clear()
        self.preset_tooltips.clear()
        self.preset_item_frames.clear()
        self.preset_order = []
        self.last_clicked_preset = None
        self.update_selection_count()

        if not self.selected_category:
            return

        presets = self.presets_data["categories"].get(self.selected_category, {})

        # Check if there are any actual presets (excluding _order metadata)
        preset_count = sum(1 for k in presets if k != "_order")
        if preset_count == 0:
            CTk.CTkLabel(
                self.preset_list_frame,
                text="(프리셋이 없습니다)",
                font=(self.font_family, self.font_size),
                text_color='#999999'
            ).pack(pady=20)
            # Reset scroll position
            self.reset_preset_scroll()
            return

        # Get custom order or use alphabetical sort
        custom_order = self.get_preset_order(self.selected_category)

        if custom_order:
            # Filter out any presets in _order that no longer exist
            # and add any new presets not in _order
            valid_order = [name for name in custom_order if name in presets and name != "_order"]
            new_presets = [name for name in sorted(presets.keys()) if name not in valid_order and name != "_order"]
            preset_names = valid_order + new_presets

            # Update _order if it changed
            if preset_names != custom_order:
                self.set_preset_order(self.selected_category, preset_names)
        else:
            # No custom order, use alphabetical and exclude _order key
            preset_names = sorted([k for k in presets.keys() if k != "_order"])

        # Track order and create items
        for preset_name in preset_names:
            self.preset_order.append(preset_name)
            preset_data = presets[preset_name]
            self.create_preset_item(preset_name, preset_data)

        # Reset scroll position to top after loading items
        self.reset_preset_scroll()

    def reset_preset_scroll(self):
        """Reset the preset list scroll position to the top."""
        try:
            # CTkScrollableFrame has an internal canvas that handles scrolling
            if hasattr(self.preset_list_frame, '_parent_canvas'):
                self.preset_list_frame._parent_canvas.yview_moveto(0.0)
        except:
            pass

    def extract_question_from_preset(self, preset_data):
        """Extract the [Question] section from preset data for tooltip preview."""
        if not preset_data or "문제" not in preset_data:
            return None

        question_text = preset_data.get("문제", "").strip()
        if not question_text:
            return None

        # Limit preview length
        max_length = 200
        if len(question_text) > max_length:
            question_text = question_text[:max_length] + "..."

        return question_text

    def create_preset_item(self, preset_name, preset_data):
        """Create a preset item widget with checkbox, tooltip, and drag support."""
        item_frame = CTk.CTkFrame(
            self.preset_list_frame,
            fg_color='#F5F5F5',
            corner_radius=4
        )
        item_frame.pack(fill='x', pady=2, padx=5)

        # Checkbox for selection (with shift-click support)
        checkbox_var = tk.BooleanVar(value=False)
        checkbox = CTk.CTkCheckBox(
            item_frame,
            text="",
            variable=checkbox_var,
            width=20,
            fg_color='#DDA15C',
            hover_color='#BB6C25'
        )
        checkbox.pack(side='left', padx=(8, 0), pady=8)

        # Bind checkbox click with shift-click support
        checkbox.bind("<Button-1>", lambda e, p=preset_name: self.on_checkbox_click(e, p))

        # Label with drag cursor - truncate long names to prevent pushing buttons off-screen
        max_display_chars = 20
        if len(preset_name) > max_display_chars:
            display_name = preset_name[:max_display_chars] + "..."
        else:
            display_name = preset_name

        label = CTk.CTkLabel(
            item_frame,
            text=f"📄 {display_name}",
            font=(self.font_family, self.font_size),
            anchor='w',
            cursor='hand2',
            text_color='black'
        )
        label.pack(side='left', fill='x', expand=True, padx=(5, 10), pady=8)

        # Button container on the right
        button_frame = CTk.CTkFrame(
            item_frame,
            fg_color='transparent'
        )
        button_frame.pack(side='right', padx=(5, 8), pady=8)

        # Preview icon/button
        preview_btn = CTk.CTkButton(
            button_frame,
            text="👁️",
            width=30,
            height=24,
            fg_color='transparent',
            hover_color='#DDA15C',
            text_color='#666666',
            font=(self.font_family, self.font_size),
            command=lambda: None,
            cursor='hand2',
            corner_radius=4
        )
        preview_btn.pack(side='left', padx=2)

        # Edit button
        edit_btn = CTk.CTkButton(
            button_frame,
            text="✏️",
            width=30,
            height=24,
            fg_color='transparent',
            hover_color='#DDA15C',
            text_color='#666666',
            font=(self.font_family, self.font_size),
            command=lambda p=preset_name: self.edit_preset_content(p),
            cursor='hand2',
            corner_radius=4
        )
        edit_btn.pack(side='left', padx=2)

        # Delete button
        delete_btn = CTk.CTkButton(
            button_frame,
            text="🗑️",
            width=30,
            height=24,
            fg_color='transparent',
            hover_color='#DC3545',
            text_color='#666666',
            font=(self.font_family, self.font_size),
            command=lambda p=preset_name: self.delete_preset_from_dialog(p),
            cursor='hand2',
            corner_radius=4
        )
        delete_btn.pack(side='left', padx=2)

        # Add tooltip with question preview on the preview icon
        question_text = self.extract_question_from_preset(preset_data)
        if question_text:
            tooltip = PresetTooltip(preview_btn, question_text, self.font_family, self.font_size)
            self.preset_tooltips[preset_name] = tooltip

        # Store references
        item_frame.preset_name = preset_name
        item_frame.checkbox_var = checkbox_var
        item_frame.label = label
        self.preset_checkboxes[preset_name] = checkbox_var
        self.preset_item_frames[preset_name] = item_frame  # Store for drop indicator

        # Bind drag events to label only (not checkbox)
        label.bind("<Button-1>", lambda e, p=preset_name: self.start_drag(e, p))
        label.bind("<B1-Motion>", self.on_drag_motion)
        label.bind("<ButtonRelease-1>", self.end_drag)

    def on_checkbox_click(self, event, preset_name):
        """Handle checkbox click with shift-click support for range selection."""
        # Check if shift key is held
        if event.state & 0x0001:  # Shift key mask
            if self.last_clicked_preset and self.last_clicked_preset in self.preset_order:
                # Get indices for range selection
                try:
                    start_idx = self.preset_order.index(self.last_clicked_preset)
                    end_idx = self.preset_order.index(preset_name)

                    # Ensure start is before end
                    if start_idx > end_idx:
                        start_idx, end_idx = end_idx, start_idx

                    # Select all items in range
                    for i in range(start_idx, end_idx + 1):
                        range_preset = self.preset_order[i]
                        self.preset_checkboxes[range_preset].set(True)
                        self.selected_presets.add(range_preset)

                    self.update_selection_count()
                    return  # Prevent default checkbox toggle
                except (ValueError, IndexError):
                    pass

        # Normal click - let checkbox toggle normally, then update selection
        # Need to schedule after the checkbox state updates
        self.after(10, lambda: self.on_preset_checkbox_changed(preset_name))
        self.last_clicked_preset = preset_name

    def on_preset_checkbox_changed(self, preset_name):
        """Handle checkbox state change."""
        if self.preset_checkboxes[preset_name].get():
            self.selected_presets.add(preset_name)
        else:
            self.selected_presets.discard(preset_name)
        self.update_selection_count()

    def select_all_presets(self):
        """Select all presets in the current category."""
        for preset_name, checkbox_var in self.preset_checkboxes.items():
            checkbox_var.set(True)
            self.selected_presets.add(preset_name)
        self.update_selection_count()

    def deselect_all_presets(self):
        """Deselect all presets."""
        for preset_name, checkbox_var in self.preset_checkboxes.items():
            checkbox_var.set(False)
        self.selected_presets.clear()
        self.update_selection_count()

    def toggle_select_all_presets(self):
        """Toggle selection based on current state."""
        total = self.get_current_preset_count()
        if total > 0 and len(self.selected_presets) == total:
            self.deselect_all_presets()
        else:
            self.select_all_presets()

    def update_selection_count(self):
        """Update the selection count label."""
        count = len(self.selected_presets)
        if hasattr(self, 'selection_count_label'):
            self.selection_count_label.configure(text=f"선택: {count}개")
        self.update_select_toggle_button()

    def update_select_toggle_button(self):
        """Update the select/deselect all toggle button label."""
        if not hasattr(self, "select_toggle_btn"):
            return
        total = self.get_current_preset_count()
        if total > 0 and len(self.selected_presets) == total:
            self.select_toggle_btn.configure(text="전체 해제")
        else:
            self.select_toggle_btn.configure(text="전체 선택")

    def _clear_drop_indicator(self):
        """Remove the drop indicator widget if it exists."""
        if self.drop_indicator is not None:
            try:
                self.drop_indicator.place_forget()
                self.drop_indicator.destroy()
            except Exception:
                pass
            self.drop_indicator = None

    def _show_drop_indicator(self, frame, before=True):
        """Display the drop indicator above or below the given frame."""
        if frame is None:
            return

        try:
            frame.update_idletasks()
            x = frame.winfo_x()
            y = frame.winfo_y()
            width = frame.winfo_width()
            height = frame.winfo_height()
        except Exception:
            return

        if width <= 0:
            width = frame.winfo_reqwidth()
        if width <= 0:
            width = 200

        indicator_height = 3

        if before:
            y = max(0, y)
        else:
            y += height

        if self.drop_indicator is None or not self.drop_indicator.winfo_exists():
            try:
                self.drop_indicator = CTk.CTkFrame(
                    self.preset_list_frame,
                    fg_color="#BB6C25",
                    height=indicator_height,
                    width=width,
                    corner_radius=2,
                )
            except Exception:
                return
        else:
            try:
                self.drop_indicator.configure(height=indicator_height, width=width)
            except Exception:
                pass

        try:
            self.drop_indicator.place(x=x, y=y)
            self.drop_indicator.lift()
        except Exception:
            pass

    def sort_presets_ascending(self):
        """Sort presets in ascending (A-Z) order."""
        if not self.selected_category:
            return

        presets = self.presets_data["categories"].get(self.selected_category, {})
        preset_names = sorted([k for k in presets.keys() if k != "_order"])

        # Save new order
        self.set_preset_order(self.selected_category, preset_names)

        # Refresh display
        self.refresh_preset_list()

    def sort_presets_descending(self):
        """Sort presets in descending (Z-A) order."""
        if not self.selected_category:
            return

        presets = self.presets_data["categories"].get(self.selected_category, {})
        preset_names = sorted([k for k in presets.keys() if k != "_order"], reverse=True)

        # Save new order
        self.set_preset_order(self.selected_category, preset_names)

        # Refresh display
        self.refresh_preset_list()

    def reorder_presets(self, dragged_presets, target_preset, insert_before=True):
        """Reorder presets within the same category.

        Args:
            dragged_presets: List of preset names being dragged
            target_preset: Preset name that defines the drop position
            insert_before: Whether the dragged presets should be inserted before the target
        """
        if not self.selected_category:
            return False

        # Get current order
        current_order = self.get_preset_order(self.selected_category)
        if not current_order:
            # No custom order yet, get current display order
            presets = self.presets_data["categories"].get(self.selected_category, {})
            current_order = sorted([k for k in presets.keys() if k != "_order"])

        index_lookup = {name: idx for idx, name in enumerate(current_order)}

        # Remove dragged items from current order
        new_order = [p for p in current_order if p not in dragged_presets]

        # Find target index
        if target_preset is None:
            target_idx = len(new_order)
        else:
            try:
                target_idx = new_order.index(target_preset)
            except ValueError:
                original_idx = index_lookup.get(target_preset)
                if original_idx is None:
                    target_idx = len(new_order)
                else:
                    num_dragged_before = sum(
                        1
                        for preset in dragged_presets
                        if index_lookup.get(preset, float("inf")) < original_idx
                    )
                    target_idx = max(0, original_idx - num_dragged_before)

            if not insert_before:
                target_idx += 1

        target_idx = max(0, min(target_idx, len(new_order)))

        # Insert dragged items before target
        for i, preset in enumerate(dragged_presets):
            insert_at = min(target_idx + i, len(new_order))
            new_order.insert(insert_at, preset)

        # Save new order
        self.set_preset_order(self.selected_category, new_order)
        return True

    def _sort_presets_by_display_order(self, preset_names):
        """Return a list ordered the same as the current preset list UI."""
        if not preset_names:
            return []

        order_lookup = {name: idx for idx, name in enumerate(self.preset_order)}
        sortable = [
            (order_lookup.get(name, float("inf")), fallback_idx, name)
            for fallback_idx, name in enumerate(preset_names)
        ]
        sortable.sort()
        return [name for _, _, name in sortable]

    def start_drag(self, event, preset_name):
        """Start dragging preset(s). If preset is not selected, drag only that one."""
        self.selected_preset = preset_name
        self.drag_source_category = self.selected_category

        # Determine what to drag
        if preset_name in self.selected_presets:
            # Drag all selected presets
            self.dragging_preset = list(self.selected_presets)
        else:
            # Drag only this preset
            self.dragging_preset = [preset_name]

        self.dragging_preset = self._sort_presets_by_display_order(self.dragging_preset)

        # Create ghost frame for visual feedback
        self.drag_ghost_frame = CTk.CTkFrame(
            self,
            fg_color='#DDA15C',
            corner_radius=4,
            border_width=2,
            border_color='#BB6C25'
        )

        # Show count if dragging multiple
        if len(self.dragging_preset) > 1:
            ghost_text = f"📄 {len(self.dragging_preset)}개 항목"
        else:
            ghost_text = f"📄 {self.dragging_preset[0]}"

        ghost_label = CTk.CTkLabel(
            self.drag_ghost_frame,
            text=ghost_text,
            font=(self.font_family, self.font_size),
            text_color='white'
        )
        ghost_label.pack(padx=10, pady=5)

        # Position ghost at cursor
        self.drag_ghost_frame.place(x=event.x_root - self.winfo_rootx(),
                                     y=event.y_root - self.winfo_rooty())
        self.drag_ghost_frame.lift()

    def on_drag_motion(self, event):
        """Handle drag motion - update ghost position and show drop indicator."""
        if not self.dragging_preset or not self.drag_ghost_frame:
            return

        # Update ghost position
        try:
            x = event.x_root - self.winfo_rootx()
            y = event.y_root - self.winfo_rooty()
            self.drag_ghost_frame.place(x=x, y=y)
        except:
            pass

        # Check if over preset list (for reordering) or category list (for moving)
        self.update_drop_indicator(event)
        # Highlight category under cursor (for moving to different category)
        self.highlight_drop_target(event)

    def update_drop_indicator(self, event):
        """Update drop indicator position when dragging over preset list."""
        try:
            preset_x = event.x_root - self.preset_list_frame.winfo_rootx()
            preset_y = event.y_root - self.preset_list_frame.winfo_rooty()

            # Check if cursor is over preset list
            if (0 <= preset_x <= self.preset_list_frame.winfo_width() and
                0 <= preset_y <= self.preset_list_frame.winfo_height()):

                # Find which preset item is under cursor
                target_frame = None
                target_preset = None

                for preset_name in self.preset_order:
                    frame = self.preset_item_frames.get(preset_name)
                    if frame is None:
                        continue

                    try:
                        frame_y = frame.winfo_y()
                        frame_height = frame.winfo_height()

                        if frame_y <= preset_y <= frame_y + frame_height:
                            target_frame = frame
                            target_preset = preset_name
                            break
                    except:
                        continue

                if target_frame and target_preset:
                    # Determine if cursor is in top or bottom half of frame
                    frame_y = target_frame.winfo_y()
                    frame_height = target_frame.winfo_height() or 1
                    y_within_frame = preset_y - frame_y
                    before = y_within_frame < (frame_height / 2)

                    # Show indicator
                    self._show_drop_indicator(target_frame, before=before)
                    return

                # Check if below all items (drop at end)
                if self.preset_order:
                    last_preset = self.preset_order[-1]
                    last_frame = self.preset_item_frames.get(last_preset)
                    if last_frame:
                        try:
                            last_bottom = last_frame.winfo_y() + last_frame.winfo_height()
                            if preset_y > last_bottom:
                                self._show_drop_indicator(last_frame, before=False)
                                return
                        except:
                            pass

            # Not over preset list, clear indicator
            self._clear_drop_indicator()
        except:
            self._clear_drop_indicator()

    def highlight_drop_target(self, event):
        """Highlight the category that the cursor is over."""
        # Get cursor position relative to category list frame
        try:
            cat_x = event.x_root - self.category_list_frame.winfo_rootx()
            cat_y = event.y_root - self.category_list_frame.winfo_rooty()

            # Check if cursor is over category list
            if (0 <= cat_x <= self.category_list_frame.winfo_width() and
                0 <= cat_y <= self.category_list_frame.winfo_height()):

                # Find which category item is under cursor
                for widget in self.category_list_frame.winfo_children():
                    if hasattr(widget, 'category_name'):
                        widget_y = widget.winfo_y()
                        widget_height = widget.winfo_height()

                        if widget_y <= cat_y <= widget_y + widget_height:
                            # Highlight this category
                            widget.configure(fg_color='#E8F5E9')
                        elif widget.category_name != self.selected_category:
                            # Remove highlight
                            widget.configure(fg_color='transparent')
        except:
            pass

    def end_drag(self, event):
        """End dragging - perform move to another category or reorder within same category."""
        if not self.dragging_preset:
            return

        # Destroy ghost frame and clear drop indicator
        if self.drag_ghost_frame:
            self.drag_ghost_frame.destroy()
            self.drag_ghost_frame = None
        self._clear_drop_indicator()

        # First check if dropping on preset list (for reordering within category)
        target_preset = None
        target_insert_before = True
        try:
            preset_x = event.x_root - self.preset_list_frame.winfo_rootx()
            preset_y = event.y_root - self.preset_list_frame.winfo_rooty()

            if (0 <= preset_x <= self.preset_list_frame.winfo_width() and
                0 <= preset_y <= self.preset_list_frame.winfo_height()):

                # Find which preset item is under cursor
                last_seen_widget = None
                for widget in self.preset_list_frame.winfo_children():
                    if hasattr(widget, 'preset_name'):
                        widget_y = widget.winfo_y()
                        widget_height = widget.winfo_height()
                        if widget_height <= 0:
                            widget_height = widget.winfo_reqheight()
                        if widget_height <= 0:
                            widget_height = 1

                        if widget_y <= preset_y <= widget_y + widget_height:
                            # Found the target preset – decide before/after based on cursor position
                            target_preset = widget.preset_name
                            y_within_frame = preset_y - widget_y
                            target_insert_before = y_within_frame < (widget_height / 2)
                            break
                        elif preset_y > widget_y + widget_height:
                            last_seen_widget = widget

                if target_preset is None:
                    if last_seen_widget is not None:
                        target_preset = last_seen_widget.preset_name
                        target_insert_before = False
                    elif self.preset_order:
                        first_preset = self.preset_order[0]
                        first_frame = self.preset_item_frames.get(first_preset)
                        if first_frame and preset_y < first_frame.winfo_y():
                            target_preset = first_preset
                            target_insert_before = True

                # If target_preset is set, perform reordering
                if target_preset:
                    if target_preset in self.dragging_preset:
                        # Dropping back onto the dragged selection – nothing to do
                        self.dragging_preset = None
                        self.drag_source_category = None
                        return

                    if self.reorder_presets(
                        self.dragging_preset,
                        target_preset,
                        insert_before=target_insert_before
                    ):
                        self.refresh_preset_list()

                    # Reset drag state and return early
                    self.dragging_preset = None
                    self.drag_source_category = None
                    return
        except:
            pass

        # If not dropping on preset list, check category list (for moving to another category)
        target_category = None
        try:
            cat_x = event.x_root - self.category_list_frame.winfo_rootx()
            cat_y = event.y_root - self.category_list_frame.winfo_rooty()

            if (0 <= cat_x <= self.category_list_frame.winfo_width() and
                0 <= cat_y <= self.category_list_frame.winfo_height()):

                for widget in self.category_list_frame.winfo_children():
                    if hasattr(widget, 'category_name'):
                        widget_y = widget.winfo_y()
                        widget_height = widget.winfo_height()

                        if widget_y <= cat_y <= widget_y + widget_height:
                            target_category = widget.category_name
                            break
        except:
            pass

        # Perform move if valid drop target
        if target_category and target_category != self.drag_source_category:
            # Move all dragged presets
            success_count = 0
            for preset_name in self.dragging_preset:
                if self.provider.move_preset_to_category(
                    preset_name,
                    self.drag_source_category,
                    target_category
                ):
                    success_count += 1

            if success_count > 0:
                # Refresh both category list (to update counts) and preset list
                self.refresh_category_list()
                self.select_category(self.selected_category)

        # Reset drag state and restore category highlighting
        self.dragging_preset = None
        self.drag_source_category = None

        # Restore normal highlighting
        for widget in self.category_list_frame.winfo_children():
            if hasattr(widget, 'category_name'):
                if widget.category_name == self.selected_category:
                    widget.configure(fg_color='#FEF9E0')
                else:
                    widget.configure(fg_color='transparent')

    # Category drag-and-drop handlers

    def on_category_mouse_down(self, event, category_name):
        """Handle mouse down - always select, prepare for potential drag."""
        # ALWAYS select on click (preserves existing behavior)
        self.select_category(category_name)

        # Store drag state (but don't create ghost yet)
        self.drag_start_x = event.x_root
        self.drag_start_y = event.y_root
        self.drag_candidate_category = category_name
        self.dragging_category = None  # Not dragging yet

    def on_category_mouse_motion(self, event):
        """Create ghost only if moved beyond threshold (5 pixels)."""
        if not hasattr(self, 'drag_candidate_category') or self.drag_candidate_category is None:
            return

        # Check if already dragging
        if self.dragging_category:
            self.update_category_drag_motion(event)
            return

        # Check threshold
        dx = abs(event.x_root - self.drag_start_x)
        dy = abs(event.y_root - self.drag_start_y)

        if dx > 5 or dy > 5:
            # Start actual drag
            category_name = self.drag_candidate_category

            # Don't allow dragging "미분류"
            if category_name == "미분류":
                return

            self.dragging_category = category_name
            self.create_category_ghost(event)

    def create_category_ghost(self, event):
        """Create visual ghost frame for dragging category."""
        # Create ghost frame for visual feedback
        self.category_ghost_frame = CTk.CTkFrame(
            self,
            fg_color='#DDA15C',
            corner_radius=4,
            border_width=2,
            border_color='#BB6C25'
        )

        ghost_text = f"📁 {self.dragging_category}"

        ghost_label = CTk.CTkLabel(
            self.category_ghost_frame,
            text=ghost_text,
            font=(self.font_family, self.font_size),
            text_color='white'
        )
        ghost_label.pack(padx=10, pady=5)

        # Position ghost at cursor
        x = event.x_root - self.winfo_rootx()
        y = event.y_root - self.winfo_rooty()
        self.category_ghost_frame.place(x=x, y=y)
        self.category_ghost_frame.lift()

    def update_category_drag_motion(self, event):
        """Update ghost position and drop indicator during drag."""
        # Update ghost frame position
        if self.category_ghost_frame:
            try:
                x = event.x_root - self.winfo_rootx()
                y = event.y_root - self.winfo_rooty()
                self.category_ghost_frame.place(x=x, y=y)
            except:
                pass

        # Update drop indicator
        self.update_category_drop_indicator(event)

    def update_category_drop_indicator(self, event):
        """Show drop indicator and record drop target state."""
        try:
            cat_x = event.x_root - self.category_list_frame.winfo_rootx()
            cat_y = event.y_root - self.category_list_frame.winfo_rooty()

            # Check if cursor is over category list
            if (0 <= cat_x <= self.category_list_frame.winfo_width() and
                0 <= cat_y <= self.category_list_frame.winfo_height()):

                # Find target category under cursor
                for cat_name, frame in self.category_item_frames.items():
                    try:
                        frame_y = frame.winfo_y()
                        frame_height = frame.winfo_height()

                        if frame_y <= cat_y <= frame_y + frame_height:
                            # Determine top/bottom half
                            y_within_frame = cat_y - frame_y
                            before = y_within_frame < (frame_height / 2)

                            # RECORD DROP TARGET STATE
                            self.category_drop_target = cat_name
                            self.category_drop_before = before

                            # Show indicator
                            self._show_category_drop_indicator(frame, before=before)
                            return
                    except:
                        continue

            # Cursor left category list - CLEAR state
            self.category_drop_target = None
            self.category_drop_before = None
            self._clear_category_drop_indicator()
        except:
            self.category_drop_target = None
            self.category_drop_before = None
            self._clear_category_drop_indicator()

    def _show_category_drop_indicator(self, frame, before=True):
        """Display the drop indicator above or below the given frame."""
        if frame is None:
            return

        try:
            frame.update_idletasks()
            x = frame.winfo_x()
            y = frame.winfo_y()
            width = frame.winfo_width()
            height = frame.winfo_height()
        except Exception:
            return

        if width <= 0:
            width = frame.winfo_reqwidth()
        if width <= 0:
            width = 200

        indicator_height = 3

        if before:
            y = max(0, y)
        else:
            y += height

        if self.category_drop_indicator is None or not self.category_drop_indicator.winfo_exists():
            try:
                self.category_drop_indicator = CTk.CTkFrame(
                    self.category_list_frame,
                    fg_color="#BB6C25",
                    height=indicator_height,
                    width=width,
                    corner_radius=2,
                )
            except Exception:
                return
        else:
            try:
                self.category_drop_indicator.configure(height=indicator_height, width=width)
            except Exception:
                pass

        try:
            self.category_drop_indicator.place(x=x, y=y)
            self.category_drop_indicator.lift()
        except Exception:
            pass

    def _clear_category_drop_indicator(self):
        """Remove the drop indicator widget if it exists."""
        if self.category_drop_indicator is not None:
            try:
                self.category_drop_indicator.place_forget()
                self.category_drop_indicator.destroy()
            except Exception:
                pass
            self.category_drop_indicator = None

    def on_category_mouse_up(self, event):
        """Handle mouse up - complete drag or just finish click."""
        if self.dragging_category:
            self.complete_category_drag(event)

        # Clean up drag state
        self.drag_candidate_category = None
        self.drag_start_x = None
        self.drag_start_y = None

    def complete_category_drag(self, event):
        """Complete category reordering."""
        if not self.dragging_category:
            return

        # Destroy ghost and clear indicator
        if self.category_ghost_frame:
            try:
                self.category_ghost_frame.destroy()
            except:
                pass
            self.category_ghost_frame = None
        self._clear_category_drop_indicator()

        # Check if we have a valid drop target (set by update_category_drop_indicator)
        if self.category_drop_target:
            self.reorder_categories(
                self.dragging_category,
                self.category_drop_target,
                insert_before=self.category_drop_before
            )

        # Reset drag state
        self.dragging_category = None
        self.category_drop_target = None
        self.category_drop_before = None

    def reorder_categories(self, dragged_cat, target_cat, insert_before):
        """Reorder categories via drag-and-drop."""
        # Get current order
        current_order = self.get_category_order()
        if not current_order:
            categories = list(self.presets_data["categories"].keys())
            current_order = sorted(categories, key=lambda x: (x != "미분류", x))

        # Remove dragged category
        new_order = [c for c in current_order if c != dragged_cat]

        # Find insert position
        try:
            target_idx = new_order.index(target_cat)
        except ValueError:
            return False

        if not insert_before:
            target_idx += 1

        # Insert at new position
        new_order.insert(target_idx, dragged_cat)

        # Enforce 미분류 first
        if "미분류" in new_order:
            new_order = ["미분류"] + [c for c in new_order if c != "미분류"]

        # Save and refresh
        self.set_category_order(new_order)  # Saves automatically
        self.refresh_category_list()
        return True

    def _prefill_input_dialog(self, dialog, initial_text):
        """Prefill CTkInputDialog entry once the widget is available."""
        if not initial_text:
            return

        def try_prefill():
            entry_widget = getattr(dialog, "_entry", None)
            if entry_widget is None:
                for child in dialog.winfo_children():
                    if isinstance(child, CTk.CTkEntry):
                        entry_widget = child
                        break
            if entry_widget is None:
                dialog.after(20, try_prefill)
                return

            entry_widget.delete(0, tk.END)
            entry_widget.insert(0, initial_text)
            entry_widget.select_range(0, tk.END)
            entry_widget.focus_set()

        dialog.after(20, try_prefill)

    def add_category(self):
        """Add a new category."""
        dialog = StyledInputDialog(
            parent=self,
            title="카테고리 추가",
            prompt="새 카테고리 이름을 입력하세요:",
            colors=self.COLORS,
            default_font=self.default_font
        )
        category_name = dialog.get_input()

        if category_name:
            if self.provider.create_category(category_name):
                self.refresh_category_list()

    def rename_category(self):
        """Rename the selected category."""
        if not self.selected_category or self.selected_category == "미분류":
            return

        dialog = StyledInputDialog(
            parent=self,
            title="카테고리 이름 변경",
            prompt=f"'{self.selected_category}' 카테고리의 새 이름을 입력하세요:",
            colors=self.COLORS,
            prefill=self.selected_category,
            default_font=self.default_font
        )
        new_name = dialog.get_input()

        if new_name:
            if self.provider.rename_category(self.selected_category, new_name):
                self.selected_category = new_name
                self.refresh_category_list()
                self.select_category(new_name)

    def delete_category(self):
        """Delete the selected category."""
        if not self.selected_category or self.selected_category == "미분류":
            return

        if self.provider.delete_category(self.selected_category):
            self.selected_category = None
            self.refresh_category_list()
            self.refresh_preset_list()

    def rename_preset(self, preset_name):
        """Show input dialog to rename a preset."""
        if not self.selected_category:
            return

        # Show input dialog with current name pre-filled
        dialog = CTk.CTkInputDialog(
            text=f"'{preset_name}' 프리셋의 새 이름을 입력하세요:",
            title="프리셋 이름 변경"
        )

        self._prefill_input_dialog(dialog, preset_name)

        new_name = dialog.get_input()

        if new_name and new_name.strip():
            if self.provider.rename_preset_in_category(
                preset_name,
                new_name.strip(),
                self.selected_category
            ):
                # Refresh to show new name
                self.refresh_category_list()  # Update counts
                self.select_category(self.selected_category)  # Refresh list

    def delete_preset_from_dialog(self, preset_name):
        """Delete a preset after confirmation."""
        if not self.selected_category:
            return

        # Confirm deletion
        confirm = messagebox.askyesno(
            "Confirm Delete",
            f"프리셋 '{preset_name}'을(를) 삭제하시겠습니까?"
        )

        if confirm:
            # Delete from category
            if preset_name in self.presets_data["categories"][self.selected_category]:
                del self.presets_data["categories"][self.selected_category][preset_name]

                # Remove from _order if it exists
                custom_order = self.get_preset_order(self.selected_category)
                if custom_order and preset_name in custom_order:
                    custom_order.remove(preset_name)
                    self.set_preset_order(self.selected_category, custom_order)
                else:
                    self.provider.save_presets_to_file()

                # Remove from selection if it was selected
                self.selected_presets.discard(preset_name)

                # Refresh UI
                self.refresh_category_list()  # Update counts
                self.select_category(self.selected_category)  # Refresh list

                messagebox.showinfo("Deleted", f"프리셋 '{preset_name}'이(가) 삭제되었습니다.")

    def close_dialog(self):
        """Close the dialog and notify parent."""
        try:
            self.grab_release()
        except:
            pass
        if self.on_close_callback:
            self.on_close_callback()
        self.destroy()

    def export_presets(self):
        """Export copyquestion_presets.json to a user-selected folder."""
        from tkinter import filedialog

        # Open folder selection dialog
        folder_path = filedialog.askdirectory(
            title="프리셋 파일을 내보낼 폴더 선택",
            parent=self
        )

        if not folder_path:
            return  # User cancelled

        try:
            import shutil
            destination = Path(folder_path) / "copyquestion_presets.json"

            # Check if file already exists at destination
            if destination.exists():
                if not messagebox.askyesno(
                    "확인",
                    f"'{destination}'에 이미 파일이 존재합니다.\n덮어쓰시겠습니까?",
                    parent=self
                ):
                    return

            # Copy the presets file
            shutil.copy2(self.presets_file, destination)
            messagebox.showinfo(
                "완료",
                f"프리셋 파일이 내보내기되었습니다:\n{destination}\n\n"
                "다른 컴퓨터에서 Import를 클릭 후 이 파일을 선택하면 프리셋을 불러올 수 있습니다.",
                parent=self
            )

        except Exception as e:
            messagebox.showerror("오류", f"내보내기 중 오류가 발생했습니다:\n{e}", parent=self)

    def import_presets(self):
        """Import copyquestion_presets.json from a user-selected file."""
        from tkinter import filedialog
        import shutil
        from datetime import datetime

        # Open file selection dialog - filter to JSON files
        file_path = filedialog.askopenfilename(
            title="가져올 프리셋 파일 선택 (copyquestion_presets.json)",
            parent=self,
            filetypes=[("JSON 파일", "*.json"), ("모든 파일", "*.*")]
        )

        if not file_path:
            return  # User cancelled

        # Verify it's the correct filename
        if Path(file_path).name != "copyquestion_presets.json":
            messagebox.showerror(
                "오류",
                "copyquestion_presets.json 파일만 가져올 수 있습니다.",
                parent=self
            )
            return

        # Load and validate the imported file
        try:
            with open(file_path, "r", encoding="utf-8") as f:
                imported_data = json.load(f)
        except json.JSONDecodeError:
            messagebox.showerror("오류", "잘못된 JSON 형식입니다.", parent=self)
            return
        except Exception as e:
            messagebox.showerror("오류", f"파일을 읽는 중 오류가 발생했습니다:\n{e}", parent=self)
            return

        # Validate structure - must have 'categories' key
        if "categories" not in imported_data:
            messagebox.showerror("오류", "유효하지 않은 프리셋 파일 형식입니다.", parent=self)
            return

        # Ask user: overwrite or merge using custom dialog
        dialog = ImportChoiceDialog(
            parent=self,
            colors=self.COLORS,
            default_font=self.default_font
        )
        choice = dialog.get_choice()

        if choice is None:  # Cancel
            return

        # Create backup of current presets (only after user confirms)
        backup_dir = self.app_data_dir / "backups"
        backup_dir.mkdir(exist_ok=True)
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        backup_file = backup_dir / f"copyquestion_presets_backup_{timestamp}.json"

        try:
            shutil.copy2(self.presets_file, backup_file)
        except Exception as e:
            messagebox.showerror("오류", f"백업 생성 중 오류가 발생했습니다:\n{e}", parent=self)
            return

        # Clean up old backups (older than 30 days)
        self._cleanup_old_backups(backup_dir, days=30)

        try:
            if choice == "overwrite":
                self.presets_data = imported_data
            else:  # merge
                self._merge_presets(imported_data)

            # Save to file
            self._save_presets_to_file()

            # Refresh UI
            self.refresh_category_list()

            # Select first category if current selection is gone
            categories = self.presets_data.get("categories", {})
            if self.selected_category not in categories:
                if "미분류" in categories:
                    self.select_category("미분류")
                elif categories:
                    self.select_category(list(categories.keys())[0])
                else:
                    self.selected_category = None
                    self.refresh_preset_list()
            else:
                self.refresh_preset_list()

            mode_text = "덮어쓰기" if choice == "overwrite" else "합치기"
            messagebox.showinfo(
                "완료",
                f"프리셋 가져오기가 완료되었습니다. ({mode_text})\n"
                f"백업 파일: {backup_file.name}",
                parent=self
            )

        except Exception as e:
            messagebox.showerror("오류", f"가져오기 중 오류가 발생했습니다:\n{e}", parent=self)

    def _merge_presets(self, imported_data):
        """Merge imported presets with existing presets."""
        imported_categories = imported_data.get("categories", {})
        existing_categories = self.presets_data.get("categories", {})

        for category_name, category_presets in imported_categories.items():
            if category_name not in existing_categories:
                # New category - add it entirely
                existing_categories[category_name] = category_presets
            else:
                # Existing category - merge presets
                for preset_name, preset_data in category_presets.items():
                    if preset_name == "_order":
                        # Merge order lists
                        existing_order = existing_categories[category_name].get("_order", [])
                        imported_order = preset_data
                        for item in imported_order:
                            if item not in existing_order:
                                existing_order.append(item)
                        existing_categories[category_name]["_order"] = existing_order
                    elif preset_name not in existing_categories[category_name]:
                        # New preset - add it
                        existing_categories[category_name][preset_name] = preset_data
                    # If preset already exists, keep existing one (don't overwrite)

        # Merge category order
        existing_cat_order = self.presets_data.get("_category_order", [])
        imported_cat_order = imported_data.get("_category_order", [])
        for cat in imported_cat_order:
            if cat not in existing_cat_order and cat in existing_categories:
                existing_cat_order.append(cat)
        if existing_cat_order:
            self.presets_data["_category_order"] = existing_cat_order

    def _cleanup_old_backups(self, backup_dir, days=30):
        """Delete backup files older than specified days."""
        from datetime import datetime, timedelta
        import os

        try:
            cutoff_date = datetime.now() - timedelta(days=days)
            deleted_count = 0

            for backup_file in backup_dir.glob("copyquestion_presets_backup_*.json"):
                # Get file modification time
                file_mtime = datetime.fromtimestamp(os.path.getmtime(backup_file))
                if file_mtime < cutoff_date:
                    try:
                        backup_file.unlink()
                        deleted_count += 1
                    except Exception:
                        pass  # Ignore individual file deletion errors

            if deleted_count > 0:
                print(f"Cleaned up {deleted_count} old backup file(s) older than {days} days")
        except Exception as e:
            print(f"Error during backup cleanup: {e}")

    def edit_preset_content(self, preset_name):
        """Open full edit dialog for a preset."""
        if not self.selected_category:
            return

        # Get preset data
        preset_data = self.presets_data["categories"][self.selected_category].get(preset_name)
        if not preset_data:
            messagebox.showerror("오류", f"프리셋 '{preset_name}'을(를) 찾을 수 없습니다.")
            return

        EditPresetDialog(
            parent=self,
            provider=self.provider,
            category=self.selected_category,
            preset_name=preset_name,
            preset_data=preset_data,
            font_family=self.font_family,
            font_size=self.font_size,
            on_save_callback=self.on_preset_edited
        )

    def add_preset(self):
        """Add a new preset in the current category using the full editor."""
        if not self.selected_category:
            messagebox.showwarning("경고", "카테고리를 먼저 선택하세요.")
            return

        empty_data = {"지문": "", "문제": "", "정답": "", "해설": ""}
        EditPresetDialog(
            parent=self,
            provider=self.provider,
            category=self.selected_category,
            preset_name="",
            preset_data=empty_data,
            font_family=self.font_family,
            font_size=self.font_size,
            on_save_callback=self.on_preset_edited
        )

    def on_preset_edited(self):
        """Refresh lists after preset edit."""
        # Reload provider presets if it exposes load_presets
        if hasattr(self.provider, "load_presets"):
            try:
                self.provider.load_presets()
                if hasattr(self.provider, "presets"):
                    self.presets_data = self.provider.presets
            except Exception:
                pass

        self.refresh_category_list()
        if self.selected_category:
            self.select_category(self.selected_category)

    # ---------------------------
    # Default data provider for standalone launch
    # ---------------------------
    def _build_default_provider(self):
        """Create a lightweight provider object that mirrors the API expected by the dialog."""
        from types import SimpleNamespace
        return SimpleNamespace(
            presets=self.presets_data,
            save_presets_to_file=self._save_presets_to_file,
            load_presets=self._load_presets,
            move_preset_to_category=self._move_preset_to_category,
            create_category=self._create_category,
            rename_category=self._rename_category,
            delete_category=self._delete_category,
            rename_preset_in_category=self._rename_preset_in_category,
        )

    def _init_storage(self):
        """Prepare app data directory and preset file path."""
        self.app_data_dir = Path.home() / ".my_application_data"
        self.app_data_dir.mkdir(exist_ok=True)
        self.presets_file = self.app_data_dir / "copyquestion_presets.json"

    def _default_presets(self):
        """Return default preset structure."""
        return {"categories": {"미분류": {}}, "last_selected_category": "미분류"}

    def _save_presets_to_file(self):
        """Persist presets_data to disk (standalone provider path)."""
        if not hasattr(self, "presets_file"):
            return
        try:
            with open(self.presets_file, "w", encoding="utf-8") as file:
                json.dump(self.presets_data, file, ensure_ascii=False, indent=4)
        except Exception as e:
            print(f"Error saving presets: {e}")

    def _load_presets(self):
        """Load presets from disk; migrate old flat format if needed."""
        if not hasattr(self, "presets_file"):
            self._init_storage()

        if self.presets_file.exists():
            try:
                with open(self.presets_file, "r", encoding="utf-8") as file:
                    loaded = json.load(file)
            except Exception as e:
                print(f"Error loading presets: {e}")
                loaded = self._default_presets()
        else:
            loaded = self._default_presets()

        # Migrate flat structure -> categories
        if "categories" not in loaded:
            loaded = {"categories": {"미분류": loaded}, "last_selected_category": "미분류"}
            self.presets_data = loaded
            self._save_presets_to_file()
        else:
            self.presets_data = loaded

        # Keep default provider in sync
        if hasattr(self, "_default_provider"):
            self._default_provider.presets = self.presets_data

        return self.presets_data

    # CRUD helpers (used when provider is the default data provider)
    def _rename_preset_in_category(self, old_name, new_name, category):
        if category not in self.presets_data.get("categories", {}):
            return False
        category_data = self.presets_data["categories"][category]
        if old_name not in category_data:
            return False
        if new_name in category_data:
            messagebox.showwarning("경고", f"'{new_name}' 프리셋이 이미 존재합니다.")
            return False

        category_data[new_name] = category_data.pop(old_name)
        order = category_data.get("_order")
        if order and old_name in order:
            order[order.index(old_name)] = new_name
        self._save_presets_to_file()
        return True

    def _move_preset_to_category(self, preset_name, from_category, to_category):
        if from_category not in self.presets_data.get("categories", {}):
            return False
        if to_category not in self.presets_data.get("categories", {}):
            return False
        from_data = self.presets_data["categories"][from_category]
        to_data = self.presets_data["categories"][to_category]
        if preset_name not in from_data:
            return False
        if preset_name in to_data:
            messagebox.showwarning("경고", f"'{preset_name}' 프리셋이 대상 카테고리에 이미 존재합니다.")
            return False

        to_data[preset_name] = from_data[preset_name]
        del from_data[preset_name]

        from_order = from_data.get("_order")
        if from_order and preset_name in from_order:
            from_order.remove(preset_name)
        self._save_presets_to_file()
        return True

    def _create_category(self, category_name):
        if not category_name or not category_name.strip():
            messagebox.showwarning("Warning", "카테고리 이름을 입력해주세요.")
            return False
        category_name = category_name.strip()
        if category_name in self.presets_data.get("categories", {}):
            messagebox.showwarning("Warning", f"카테고리 '{category_name}'이(가) 이미 존재합니다.")
            return False
        if "categories" not in self.presets_data:
            self.presets_data["categories"] = {}
        self.presets_data["categories"][category_name] = {}

        order = list(self.presets_data.get("_category_order", []))
        if "미분류" in self.presets_data["categories"] and "미분류" not in order:
            order.insert(0, "미분류")
        if category_name not in order:
            if "미분류" in order:
                order.insert(1, category_name)
            else:
                order.append(category_name)
        self.set_category_order(order)
        return True

    def _rename_category(self, old_name, new_name):
        if old_name == "미분류":
            messagebox.showwarning("Warning", "'미분류' 카테고리는 이름을 변경할 수 없습니다.")
            return False
        if not new_name or not new_name.strip():
            messagebox.showwarning("Warning", "새 카테고리 이름을 입력해주세요.")
            return False
        new_name = new_name.strip()
        if old_name not in self.presets_data.get("categories", {}):
            return False
        if new_name in self.presets_data["categories"]:
            messagebox.showwarning("Warning", f"카테고리 '{new_name}'이(가) 이미 존재합니다.")
            return False
        self.presets_data["categories"][new_name] = self.presets_data["categories"].pop(old_name)

        order = self.presets_data.get("_category_order")
        if order and old_name in order:
            order[order.index(old_name)] = new_name
        self._save_presets_to_file()
        return True

    def _delete_category(self, category_name):
        if category_name == "미분류":
            messagebox.showwarning("Warning", "'미분류' 카테고리는 삭제할 수 없습니다.")
            return False
        if category_name not in self.presets_data.get("categories", {}):
            return False
        # Count presets (exclude _order metadata)
        presets_in_category = self.presets_data["categories"][category_name]
        preset_count = sum(1 for k in presets_in_category if k != "_order")

        if preset_count > 0:
            confirm = messagebox.askyesno(
                "Confirm Delete",
                f"카테고리 '{category_name}'에 {preset_count}개의 프리셋이 있습니다.\n"
                f"카테고리를 삭제하면 모든 프리셋이 '미분류'로 이동됩니다.\n계속하시겠습니까?"
            )
            if not confirm:
                return False

            # Ensure '미분류' exists
            if "미분류" not in self.presets_data["categories"]:
                self.presets_data["categories"]["미분류"] = {}

            # Move presets (skip _order metadata)
            for preset_name, preset_data in presets_in_category.items():
                if preset_name != "_order":
                    self.presets_data["categories"]["미분류"][preset_name] = preset_data

        # Delete category
        del self.presets_data["categories"][category_name]

        # Update order
        order = list(self.presets_data.get("_category_order", []))
        if category_name in order:
            order.remove(category_name)

        # Ensure '미분류' stays first if present
        if "미분류" in self.presets_data.get("categories", {}) and "미분류" not in order:
            order.insert(0, "미분류")

        self.presets_data["_category_order"] = order
        self._save_presets_to_file()
        return True

class HelpDialog(CTk.CTkToplevel):
    """Dialog for displaying preset usage help information."""

    def __init__(self, parent, font_family, font_size):
        super().__init__(parent)

        self.parent_dialog = parent  # Store reference to EditPresetDialog
        self.title("동형문제 프리셋 도움말")
        self.geometry("700x520")
        self.minsize(width=700, height=520)
        self.transient(parent)
        self.grab_set()
        self.focus_force()

        self.COLORS = {
            'bg_window': '#F7F8FA',
            'bg_card': '#FFFFFF',
            'text_primary': '#2C3E50',
            'text_secondary': '#7F8C8D',
            'border_light': '#E0E0E0',
            'accent': '#DDA15C',
            'accent_hover': '#BB6C25',
            'text_inverse': '#FFFFFF',
        }

        self.configure(fg_color=self.COLORS['bg_window'])

        main_frame = CTk.CTkFrame(self, fg_color='transparent')
        main_frame.pack(fill='both', expand=True, padx=20, pady=20)

        # Header
        header = CTk.CTkLabel(
            main_frame,
            text="동형문제 프리셋 사용 방법",
            font=(font_family, font_size + 4, 'bold'),
            text_color=self.COLORS['text_primary']
        )
        header.pack(pady=(0, 15))

        # Content frame with scrollable text
        content_frame = CTk.CTkFrame(
            main_frame,
            fg_color=self.COLORS['bg_card'],
            border_width=1,
            border_color=self.COLORS['border_light'],
            corner_radius=8
        )
        content_frame.pack(fill='both', expand=True, pady=(0, 15))

        # Scrollable text widget
        text_widget = CTk.CTkTextbox(
            content_frame,
            font=(font_family, 13),
            text_color=self.COLORS['text_primary'],
            fg_color=self.COLORS['bg_card'],
            border_width=0,
            wrap='word',
            activate_scrollbars=True
        )
        text_widget.pack(fill='both', expand=True, padx=15, pady=15)

        # Help content
        help_text = """이곳에 문제의 샘플을 입력해두면 이와 동일한 유형으로 문제를 제작합니다.
'지문', '문제', '정답', '해설'을 각각 적어주세요. ('해설'은 비워두어도 됩니다)
좀 더 정교한 지시가 필요하다면 "문제"칸 가장 하단에 다음과 같은 방식으로 지시사항을 추가하셔도 됩니다.
(Instruction: Use Korean for the options. But use English for proper nouns. Do not include this instruction in your result.)

주의사항 #1
 '지문' 창에 밑줄이 필요한 경우 밑줄 대신 대괄호 [ ]를 사용하여 표시한 뒤, '문제' 창에서도 '밑줄' 대신 '괄호'란 말을 사용하세요."
ex) 윗글의 밑줄 친 부분을… → 윗글의 괄호 한 부분을…

주의사항 #2
하나의 지문에 여러 문제를 출제할 경우, 문제 번호는 반드시 "문제1)", "문제2)", ... 를 붙여야 합니다.
ex)
문제1) 윗글의 제목으로 가장 적절한 것은?
문제2) 윗글의 내용과 일치하지 않는 것은?

'정답'과 '해설'란에도 다음과 같이 각각 구분해서 입력해주세요.
ex) 
정답1) ①
정답2) ③

ex)
해설1) ~~~
해설2) ~~~"""

        text_widget.insert("1.0", help_text)
        text_widget.configure(state='disabled')  # Make read-only

        # Button frame
        button_frame = CTk.CTkFrame(main_frame, fg_color='transparent')
        button_frame.pack()

        # Sample button
        sample_btn = CTk.CTkButton(
            button_frame,
            text="샘플 사용해보기",
            width=120,
            height=36,
            fg_color=self.COLORS['bg_card'],
            hover_color='#F0F0F0',
            text_color=self.COLORS['text_primary'],
            border_width=1,
            border_color=self.COLORS['border_light'],
            font=(font_family, 12, 'bold'),
            command=self.load_sample,
            corner_radius=6
        )
        sample_btn.pack(side='left', padx=(0, 10))

        # Close button
        close_btn = CTk.CTkButton(
            button_frame,
            text="확인",
            width=100,
            height=36,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color=self.COLORS['text_inverse'],
            font=(font_family, 12, 'bold'),
            command=self.destroy,
            corner_radius=6
        )
        close_btn.pack(side='left')

        self.bind("<Escape>", lambda e: self.destroy())
        self.protocol("WM_DELETE_WINDOW", self.destroy)

    def load_sample(self):
        """Load sample preset data into the EditPresetDialog."""
        sample_passage = """(A)
Linguists have studied how children acquire their mother tongue. They are also interested in how a second or foreign language is learned. With language acquisition, there are three major camps: nativists, behaviorists, and interactionists. Nativists believe language is a biological instinct in humans. Grammar and syntax are inbred, not learned. This biology of the brain allows children to learn their own language. In other words, humans have an innate ability to acquire language, and language is most easily acquired during a critical period in early childhood. On the other hand, behaviorists believe a child's language is shaped by their parents' speech. How parents speak becomes the foundation of the child's language. Human beings are born empty and infants are considered to be blank slates. This theory proposes that language is a learned behavior, acquired through operant conditioning, imitation and practice. Recently, however, interactionists are gaining a strong foothold. They claim the interaction between biological and social aspects of language acquisition is important. Interactionist theory asserts that language acquisition has both biological and social components."""

        sample_question = """아래 (B)는 위 (A)를 바탕으로 내용을 작성한 것이다. (B)의 내용 중 (A)의 내용과 일치하지 않는 것은?

(B)
Theories of Language Acquisition

Nativist theory
- Language acquisition is ①[biologically] programmed
- Noam Chomsky posited that people are born with language
acquisition device (LAD), a brain system that allows children to quickly learn language
- Language: easily acquired during ②[critical period]
- LAD: most active during these early years of development

Behaviorist theory
- Language is a learned behavior, acquired through operant conditioning, ③[limitation], and practice
- Humans are born as ④[scholars or intellects]

Interactionist theory
- Language acquisition has ⑤[biological and social] components
- Interaction between two theories above brings the optimal result"""

        sample_answer = "④"

        sample_explanation = "(A)에서는 인간은 태어날 때 blank slates(빈 서판)라고 함. '학자/지성인'은 정 반대 의미."

        # Clear existing content
        self.parent_dialog.text_edits[0].delete("1.0", "end")
        self.parent_dialog.text_edits[1].delete("1.0", "end")
        self.parent_dialog.text_edits[2].delete("1.0", "end")
        self.parent_dialog.text_edits[3].delete("1.0", "end")

        # Insert sample data
        self.parent_dialog.text_edits[0].insert("1.0", sample_passage)
        self.parent_dialog.text_edits[1].insert("1.0", sample_question)
        self.parent_dialog.text_edits[2].insert("1.0", sample_answer)
        self.parent_dialog.text_edits[3].insert("1.0", sample_explanation)

        # Close the help dialog
        self.destroy()


class EditPresetDialog(CTk.CTkToplevel):
    """Dialog for editing all fields of a preset (name + 4 content sections)."""

    def __init__(self, parent, provider, category, preset_name, preset_data, font_family, font_size, on_save_callback=None):
        super().__init__(parent)

        title_name = preset_name if preset_name else "새 프리셋"
        self.title(f"프리셋 편집: {title_name}")
        self.geometry("1250x720")
        self.minsize(width=1250, height=680)
        self.resizable(True, True)
        self.transient(parent)
        self.grab_set()
        self.focus_force()

        self.provider = provider
        self.category = category
        self.original_preset_name = preset_name
        self.preset_data = preset_data
        self.font_family = font_family
        self.on_save_callback = on_save_callback

        # Define the application data directory (needed for load_font_size)
        self.app_data_dir = Path.home() / ".my_application_data"
        self.app_data_dir.mkdir(exist_ok=True)

        # Load the last used font size (shared with CopyQuestionPopup)
        self.font_size = self.load_font_size()

        self.COLORS = {
            'bg_window': '#F7F8FA',
            'bg_card': '#FFFFFF',
            'text_primary': '#2C3E50',
            'text_inverse': '#FFFFFF',
            'border_light': '#E0E0E0',
            'border_medium': '#D0D0D0',
            'accent': '#DDA15C',
            'accent_hover': '#BB6C25',
            'bg_transparent': 'transparent'
        }

        self.configure(fg_color=self.COLORS['bg_window'])

        if sys.platform.startswith('darwin'):
            self.copy_manager = CopyManager(self)
        else:
            self.copy_manager = None

        self._build_ui()
        self._populate_fields()

        # Keyboard shortcuts
        self.bind("<Escape>", lambda e: self.cancel())
        self.bind("<Command-Return>", self._on_modifier_enter)  # macOS
        self.bind("<Control-Return>", self._on_modifier_enter)  # Windows/Linux
        self.bind("<Command-KP_Enter>", self._on_modifier_enter)  # macOS numpad
        self.bind("<Control-KP_Enter>", self._on_modifier_enter)  # Windows/Linux numpad
        # Font size shortcuts
        self.bind("<Command-equal>", lambda e: self.increase_font_size())  # macOS Cmd+=
        self.bind("<Command-plus>", lambda e: self.increase_font_size())  # macOS Cmd++
        self.bind("<Command-minus>", lambda e: self.decrease_font_size())  # macOS Cmd+-
        self.bind("<Control-equal>", lambda e: self.increase_font_size())  # Windows/Linux Ctrl+=
        self.bind("<Control-plus>", lambda e: self.increase_font_size())  # Windows/Linux Ctrl++
        self.bind("<Control-minus>", lambda e: self.decrease_font_size())  # Windows/Linux Ctrl+-
        self.protocol("WM_DELETE_WINDOW", self.cancel)

    def _build_ui(self):
        main_container = CTk.CTkFrame(self, fg_color=self.COLORS['bg_transparent'])
        main_container.pack(fill='both', expand=True, padx=20, pady=20)
        main_container.grid_rowconfigure(1, weight=1)
        main_container.grid_columnconfigure(0, weight=1)

        name_frame = CTk.CTkFrame(main_container, fg_color=self.COLORS['bg_transparent'])
        name_frame.grid(row=0, column=0, sticky='ew', pady=(0, 15))

        CTk.CTkLabel(
            name_frame,
            text="프리셋 이름:",
            font=(self.font_family, 12, 'bold'),
            text_color=self.COLORS['text_primary']
        ).pack(side='left', padx=(0, 10))

        self.name_entry = CTk.CTkEntry(
            name_frame,
            font=(self.font_family, 12),
            placeholder_text="프리셋 이름을 입력하세요",
            border_color=self.COLORS['border_light'],
            fg_color=self.COLORS['bg_card']
        )
        self.name_entry.pack(side='left', fill='x', expand=True, padx=(0, 10))

        CTk.CTkButton(
            name_frame,
            text="도움말",
            width=80,
            height=30,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color=self.COLORS['text_inverse'],
            font=(self.font_family, 12),
            command=self.show_help,
            corner_radius=6
        ).pack(side='left')

        content = CTk.CTkFrame(main_container, fg_color=self.COLORS['bg_transparent'])
        content.grid(row=1, column=0, sticky='nsew')
        content.grid_rowconfigure(1, weight=1)
        for col in range(4):
            content.grid_columnconfigure(col, weight=10 if col != 2 else 1)

        headers = ["지문", "문제", "정답", "해설"]
        self.text_edits = []
        for col, header_text in enumerate(headers):
            CTk.CTkLabel(
                content,
                text=header_text,
                font=(self.font_family, 12, 'bold'),
                text_color=self.COLORS['text_primary'],
                anchor='w'
            ).grid(row=0, column=col, padx=5, pady=(0, 5), sticky='w')

            textbox = CTk.CTkTextbox(
                content,
                border_width=1,
                corner_radius=6,
                text_color=self.COLORS['text_primary'],
                fg_color=self.COLORS['bg_card'],
                border_color=self.COLORS['border_light'],
                font=(self.font_family, self.font_size),
                activate_scrollbars=True,
                undo=True,
                wrap='word'
            )
            textbox.grid(row=1, column=col, padx=5, pady=(0, 10), sticky='nsew')
            self.text_edits.append(textbox)

            if self.copy_manager:
                textbox.bind("<KeyPress>", self.copy_manager.handle_keypress)

        btn_frame = CTk.CTkFrame(main_container, fg_color=self.COLORS['bg_transparent'])
        btn_frame.grid(row=2, column=0, pady=(15, 0), sticky='ew')

        # Font size buttons on the left
        CTk.CTkButton(
            btn_frame,
            text="Aa+",
            fg_color="white",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.font_family, 14, 'bold'),
            command=self.increase_font_size,
            width=40,
            height=28
        ).pack(side='left', padx=(0, 5))

        CTk.CTkButton(
            btn_frame,
            text="Aa-",
            fg_color="white",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.font_family, 11, 'bold'),
            command=self.decrease_font_size,
            width=40,
            height=28
        ).pack(side='left')

        # Action buttons on the right
        CTk.CTkButton(
            btn_frame,
            text="저장",
            width=100,
            height=36,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color=self.COLORS['text_inverse'],
            font=(self.font_family, 12, 'bold'),
            command=self.save_changes,
            corner_radius=6
        ).pack(side='right')

        CTk.CTkButton(
            btn_frame,
            text="취소",
            width=100,
            height=36,
            fg_color=self.COLORS['bg_card'],
            hover_color='#F0F0F0',
            text_color=self.COLORS['text_primary'],
            border_width=1,
            border_color=self.COLORS['border_medium'],
            font=(self.font_family, 12),
            command=self.cancel,
            corner_radius=6
        ).pack(side='right', padx=(0, 10))

    def _populate_fields(self):
        self.name_entry.insert(0, self.original_preset_name or "")
        if self.original_preset_name:
            self.name_entry.select_range(0, tk.END)

        self.text_edits[0].insert("1.0", self.preset_data.get("지문", ""))
        self.text_edits[1].insert("1.0", self.preset_data.get("문제", ""))
        self.text_edits[2].insert("1.0", self.preset_data.get("정답", ""))
        self.text_edits[3].insert("1.0", self.preset_data.get("해설", ""))

    def _on_modifier_enter(self, event):
        """Handle Ctrl/Cmd+Enter with explicit modifier key verification.

        Some Windows systems (especially with Korean IME) may incorrectly
        trigger Control-Return when only Enter is pressed. This method
        verifies that the modifier key is actually held.
        """
        # Check if Control key is actually held (bitmask 0x0004)
        # On Mac, Command key uses different detection but the binding itself is reliable
        if sys.platform.startswith('darwin'):
            # macOS: Trust the Command-Return binding
            self.save_changes()
        else:
            # Windows/Linux: Verify Control key is actually pressed
            # event.state & 0x0004 checks if Control is held
            if event.state & 0x0004:
                self.save_changes()
            else:
                # Control key not actually held - ignore (likely IME issue)
                print(f"[DEBUG] Enter pressed without Ctrl modifier (state={event.state})")
                return "break"
        return "break"

    def save_changes(self):
        print("save_changes called")
        new_name = self.name_entry.get().strip()
        if not new_name:
            messagebox.showwarning("경고", "프리셋 이름을 입력하세요.")
            return

        passage = self.text_edits[0].get("1.0", "end-1c").strip()
        question = self.text_edits[1].get("1.0", "end-1c").strip()
        answer = self.text_edits[2].get("1.0", "end-1c").strip()
        explanation = self.text_edits[3].get("1.0", "end-1c").strip()


        # Validation
        if not passage or not question or not answer:
            # Auto-fill answer template if question has multiple questions
            if "문제1)" in question and "문제2)" in question:
                question_count = 0
                for i in range(1, 20):
                    if f"문제{i})" in question:
                        question_count = i
                    else:
                        break
                if question_count >= 2:
                    answer_template = "\n".join([f"정답{i})" for i in range(1, question_count + 1)])
                    self.text_edits[2].delete("1.0", "end")
                    self.text_edits[2].insert("1.0", answer_template)

            messagebox.showwarning("경고", "'지문', '문제', '정답'란을 모두 채워주세요.")
            return

        elif "문제1)" in question and "문제2)" in question and ("정답1)" not in answer or "정답2)" not in answer):
            # Count how many questions exist (문제1), 문제2), 문제3), etc.)
            question_count = 0
            for i in range(1, 20):  # Check up to 20 questions
                if f"문제{i})" in question:
                    question_count = i
                else:
                    break

            # Generate and fill answer template
            if question_count >= 2:
                answer_template = "\n".join([f"정답{i})" for i in range(1, question_count + 1)])
                self.text_edits[2].delete("1.0", "end")
                self.text_edits[2].insert("1.0", answer_template)

            messagebox.showwarning("경고", "복수 개의 문제가 발견되었습니다. 정답도 '정답'란에 '정답1)' '정답2)' 형식으로 구분해서 입력해주세요.")
            return

        elif "문제1)" in question and "문제2)" in question and "정답1)" in answer and "정답2)" in answer and explanation != "" and ("해설1)" not in explanation or "해설2)" not in explanation):
            messagebox.showwarning("경고", "복수 개의 문제가 발견되었습니다. 해설도 '해설'란에 '해설1)' '해설2)' 형식으로 구분해서 입력해주세요.")
            return

        # Auto-fill explanation if empty
        if explanation == "":
            if "문제1)" in question and "문제2)" in question:
                explanation = "(Write your own explanations in Korean. Write an explanation for each answer with this format:\n해설1) explanation for answer 1 in Korean\n해설2) explanation for answer 2 in Korean\n...)"
            else:
                explanation = "(Write your own explanation in Korean.)"
            

        if not hasattr(self.provider, "presets"):
            messagebox.showerror("오류", "프리셋 데이터를 찾을 수 없습니다.")
            return

        # Ensure category exists
        if "categories" not in self.provider.presets:
            self.provider.presets["categories"] = {}
        category_data = self.provider.presets["categories"].setdefault(self.category, {})

        is_new = not self.original_preset_name

        if not is_new:
            if new_name != self.original_preset_name:
                if new_name in category_data:
                    messagebox.showwarning("경고", f"'{new_name}' 프리셋이 이미 존재합니다.")
                    return
                existing_data = category_data[self.original_preset_name].copy()
                del category_data[self.original_preset_name]
                category_data[new_name] = existing_data
                order = category_data.get("_order")
                if order and self.original_preset_name in order:
                    order[order.index(self.original_preset_name)] = new_name
        else:
            if new_name in category_data:
                messagebox.showwarning("경고", f"'{new_name}' 프리셋이 이미 존재합니다.")
                return
            category_data[new_name] = {}
            # add to order list for new preset
            order = category_data.get("_order")
            if order is None:
                order = []
                category_data["_order"] = order
            if new_name not in order:
                order.append(new_name)

        category_data[new_name].update({
            "지문": passage,
            "문제": question,
            "정답": answer,
            "해설": explanation
        })

        if hasattr(self.provider, "save_presets_to_file"):
            self.provider.save_presets_to_file()
        if hasattr(self.provider, "load_presets"):
            self.provider.load_presets()

        if self.on_save_callback:
            self.on_save_callback()

        #messagebox.showinfo("성공", f"프리셋 '{new_name}'이(가) 저장되었습니다.")
        self.destroy()

    def show_help(self):
        """Show help dialog with preset usage instructions."""
        HelpDialog(self, self.font_family, self.font_size)

    def load_font_size(self):
        """
        Load the last used font size (shared with CopyQuestionPopup).
        """
        font_size_file = self.app_data_dir / 'copyquestion_font_size.json'
        if font_size_file.exists():
            try:
                with open(font_size_file, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    return data.get('font_size', 12)  # Default to 12 if not found
            except (json.JSONDecodeError, OSError):
                return 12
        return 12  # Default font size if file doesn't exist

    def save_font_size(self, font_size):
        """
        Save the font size (shared with CopyQuestionPopup).
        """
        font_size_file = self.app_data_dir / 'copyquestion_font_size.json'
        try:
            with open(font_size_file, 'w', encoding='utf-8') as f:
                json.dump({'font_size': font_size}, f)
        except OSError as e:
            print(f"Failed to save font size: {e}")

    def increase_font_size(self):
        """Increase font size for all text edits."""
        self.font_size += 1
        for text_edit in self.text_edits:
            text_edit.configure(font=(self.font_family, self.font_size))
        self.save_font_size(self.font_size)

    def decrease_font_size(self):
        """Decrease font size for all text edits (minimum 1)."""
        if self.font_size > 1:
            self.font_size -= 1
            for text_edit in self.text_edits:
                text_edit.configure(font=(self.font_family, self.font_size))
            self.save_font_size(self.font_size)

    def cancel(self):
        self.destroy()


class CopyQuestionHelpDialog(CTk.CTkToplevel):
    """Dialog for displaying preset usage help information for CopyQuestionPopup."""

    def __init__(self, parent, font_family, font_size):
        super().__init__(parent)

        self.parent_dialog = parent  # Store reference to CopyQuestionPopup
        self.title("동형문제 프리셋 도움말")
        self.geometry("700x520")
        self.minsize(width=700, height=520)
        self.transient(parent)
        self.grab_set()
        self.focus_force()

        self.COLORS = {
            'bg_window': '#F7F8FA',
            'bg_card': '#FFFFFF',
            'text_primary': '#2C3E50',
            'text_secondary': '#7F8C8D',
            'border_light': '#E0E0E0',
            'accent': '#DDA15C',
            'accent_hover': '#BB6C25',
            'text_inverse': '#FFFFFF',
        }

        self.configure(fg_color=self.COLORS['bg_window'])

        main_frame = CTk.CTkFrame(self, fg_color='transparent')
        main_frame.pack(fill='both', expand=True, padx=20, pady=20)

        # Header
        header = CTk.CTkLabel(
            main_frame,
            text="동형문제 프리셋 사용 방법",
            font=(font_family, font_size + 4, 'bold'),
            text_color=self.COLORS['text_primary']
        )
        header.pack(pady=(0, 15))

        # Content frame with scrollable text
        content_frame = CTk.CTkFrame(
            main_frame,
            fg_color=self.COLORS['bg_card'],
            border_width=1,
            border_color=self.COLORS['border_light'],
            corner_radius=8
        )
        content_frame.pack(fill='both', expand=True, pady=(0, 15))

        # Scrollable text widget
        text_widget = CTk.CTkTextbox(
            content_frame,
            font=(font_family, 13),
            text_color=self.COLORS['text_primary'],
            fg_color=self.COLORS['bg_card'],
            border_width=0,
            wrap='word',
            activate_scrollbars=True
        )
        text_widget.pack(fill='both', expand=True, padx=15, pady=15)

        # Help content
        help_text = """이곳에 문제의 샘플을 입력해두면 이와 동일한 유형으로 문제를 제작합니다.
'지문', '문제', '정답', '해설'을 각각 적어주세요. ('해설'은 비워두어도 됩니다)
좀 더 정교한 지시가 필요하다면 "문제"칸 가장 하단에 다음과 같은 방식으로 지시사항을 추가하셔도 됩니다.
(Instruction: Use Korean for the options. But use English for proper nouns. Do not include this instruction in your result.)

주의사항 #1
 '지문' 창에 밑줄이 필요한 경우 밑줄 대신 대괄호 [ ]를 사용하여 표시한 뒤, '문제' 창에서도 '밑줄' 대신 '괄호'란 말을 사용하세요."
ex) 윗글의 밑줄 친 부분을… → 윗글의 괄호 한 부분을…

주의사항 #2
하나의 지문에 여러 문제를 출제할 경우, 문제 번호는 반드시 "문제1)", "문제2)", ... 를 붙여야 합니다.
ex)
문제1) 윗글의 제목으로 가장 적절한 것은?
문제2) 윗글의 내용과 일치하지 않는 것은?

'정답'과 '해설'란에도 다음과 같이 각각 구분해서 입력해주세요.
ex) 
정답1) ①
정답2) ③

ex)
해설1) ~~~
해설2) ~~~"""

        text_widget.insert("1.0", help_text)
        text_widget.configure(state='disabled')  # Make read-only

        # Button frame
        button_frame = CTk.CTkFrame(main_frame, fg_color='transparent')
        button_frame.pack()

        # Sample button
        sample_btn = CTk.CTkButton(
            button_frame,
            text="샘플 사용해보기",
            width=120,
            height=36,
            fg_color=self.COLORS['bg_card'],
            hover_color='#F0F0F0',
            text_color=self.COLORS['text_primary'],
            border_width=1,
            border_color=self.COLORS['border_light'],
            font=(font_family, 12, 'bold'),
            command=self.load_sample,
            corner_radius=6
        )
        sample_btn.pack(side='left', padx=(0, 10))

        # Close button
        close_btn = CTk.CTkButton(
            button_frame,
            text="확인",
            width=100,
            height=36,
            fg_color=self.COLORS['accent'],
            hover_color=self.COLORS['accent_hover'],
            text_color=self.COLORS['text_inverse'],
            font=(font_family, 12, 'bold'),
            command=self.destroy,
            corner_radius=6
        )
        close_btn.pack(side='left')

        self.bind("<Escape>", lambda e: self.destroy())
        self.protocol("WM_DELETE_WINDOW", self.destroy)

    def load_sample(self):
        """Load sample preset data into the CopyQuestionPopup."""
        sample_passage = """(A)
Linguists have studied how children acquire their mother tongue. They are also interested in how a second or foreign language is learned. With language acquisition, there are three major camps: nativists, behaviorists, and interactionists. Nativists believe language is a biological instinct in humans. Grammar and syntax are inbred, not learned. This biology of the brain allows children to learn their own language. In other words, humans have an innate ability to acquire language, and language is most easily acquired during a critical period in early childhood. On the other hand, behaviorists believe a child's language is shaped by their parents' speech. How parents speak becomes the foundation of the child's language. Human beings are born empty and infants are considered to be blank slates. This theory proposes that language is a learned behavior, acquired through operant conditioning, imitation and practice. Recently, however, interactionists are gaining a strong foothold. They claim the interaction between biological and social aspects of language acquisition is important. Interactionist theory asserts that language acquisition has both biological and social components."""

        sample_question = """아래 (B)는 위 (A)를 바탕으로 내용을 작성한 것이다. (B)의 내용 중 (A)의 내용과 일치하지 않는 것은?

(B)
Theories of Language Acquisition

Nativist theory
- Language acquisition is ①[biologically] programmed
- Noam Chomsky posited that people are born with language
acquisition device (LAD), a brain system that allows children to quickly learn language
- Language: easily acquired during ②[critical period]
- LAD: most active during these early years of development

Behaviorist theory
- Language is a learned behavior, acquired through operant conditioning, ③[limitation], and practice
- Humans are born as ④[scholars or intellects]

Interactionist theory
- Language acquisition has ⑤[biological and social] components
- Interaction between two theories above brings the optimal result"""

        sample_answer = "④"

        sample_explanation = "(A)에서는 인간은 태어날 때 blank slates(빈 서판)라고 함. '학자/지성인'은 정 반대 의미."

        # Clear existing content
        self.parent_dialog.text_edits[0].delete("1.0", "end")
        self.parent_dialog.text_edits[1].delete("1.0", "end")
        self.parent_dialog.text_edits[2].delete("1.0", "end")
        self.parent_dialog.text_edits[3].delete("1.0", "end")

        # Insert sample data
        self.parent_dialog.text_edits[0].insert("1.0", sample_passage)
        self.parent_dialog.text_edits[1].insert("1.0", sample_question)
        self.parent_dialog.text_edits[2].insert("1.0", sample_answer)
        self.parent_dialog.text_edits[3].insert("1.0", sample_explanation)

        # Close the help dialog
        self.destroy()


class CopyQuestionPopup(CTk.CTkToplevel):
    def __init__(self, parent, default_font="Arial", owner_app=None, is_manual_mocktest=False, row_idx=None, manual_mocktest_popup=None, dashboard_ref=None):
        super().__init__(parent)
        self.title("동형문제 복사기")

        # Store manual mocktest related parameters
        self.is_manual_mocktest = is_manual_mocktest
        #print("is_manual_mocktest:", self.is_manual_mocktest)

        self.row_idx = row_idx
        self.manual_mocktest_popup = manual_mocktest_popup
        self.owner_app = owner_app  # Parent application reference for UI updates
        self.dashboard_ref = dashboard_ref  # Dashboard reference for real-time status updates
        self.유형 = "동형문제"  # Question type for override detection

        # Passage transformation variables
        self.지문변형_var = tk.BooleanVar(value=False)
        self.지문난이도_var = tk.IntVar(value=2)  # Default: 같은 난이도
        self.지문길이_var = tk.IntVar(value=2)    # Default: 비슷하게
        self.content_change_var = tk.IntVar(value=0)  # Default: 내용 변화 없음

        self.geometry("1250x720")
        self.minsize(width=1250, height=680)
        #self.grab_set()
        #self.focus_set()

        # Define the application data directory (needed for load_font_size)
        self.app_data_dir = Path.home() / ".my_application_data"
        self.app_data_dir.mkdir(exist_ok=True)  # Create the directory if it does not exist

        # Load the last used font size
        self.default_font_size = self.load_font_size()
        self.default_font = (default_font, self.default_font_size)  # Adjust as needed

        # Ensure the popup window itself is resizable
        self.grid_rowconfigure(0, weight=1)
        self.grid_columnconfigure(0, weight=1)

        # Create the main content frame
        content_frame = CTk.CTkFrame(self, fg_color="transparent", border_width=0, corner_radius=0)
        content_frame.grid(row=0, column=0, sticky="nsew", padx=7, pady=0)
        content_frame.grid_rowconfigure(1, weight=1)
        content_frame.grid_columnconfigure(0, weight=1)

        # Create the label frame
        label_frame = CTk.CTkFrame(content_frame, fg_color="transparent", border_width=0, corner_radius=0)
        label_frame.grid(row=0, column=0, sticky="ew", pady=(0, 5))

        # Create the text frame
        text_frame = CTk.CTkFrame(content_frame, fg_color="transparent", border_width=0, corner_radius=0)
        text_frame.grid(row=1, column=0, sticky="nsew")
        text_frame.grid_rowconfigure(0, weight=1)

        # Configure column ratios for label_frame and text_frame
        for frame in (label_frame, text_frame):
            frame.grid_columnconfigure(0, weight=10)  # 지문
            frame.grid_columnconfigure(1, weight=10)  # 문제
            frame.grid_columnconfigure(2, weight=2)   # 정답
            frame.grid_columnconfigure(3, weight=10)  # 해설

        # Calculate base width for textboxes
        base_width = 100  # Adjust this value as needed

        # Create labels
        labels = ["지문", "문제", "정답", "해설"]
        ratios = [10, 10, 2, 10]

        for i, (label_text, ratio) in enumerate(zip(labels, ratios)):
            label = CTk.CTkLabel(
                label_frame, 
                text=label_text, 
                font=(self.default_font[0], 13, 'bold'), 
                height=20, 
                fg_color="transparent",
                width=base_width * ratio  # Set width based on ratio
            )
            label.grid(row=0, column=i, sticky="ew", padx=3)

        # Initialize self.text_edits
        self.text_edits = []

        # Create text_edits for 지문, 문제, 정답, 해설 (self.text_edits[0] to self.text_edits[3])
        for i, width_ratio in enumerate([10, 10, 2, 10]):
            text_edit = CTk.CTkTextbox(
                master=text_frame,
                border_width=1,
                corner_radius=0,
                text_color=("black", "#DBE1EC"),
                fg_color=("white", "#353638"),
                font=(self.default_font[0], self.default_font_size),
                activate_scrollbars=True,
                undo=True,
                wrap="word",
                width=base_width * width_ratio  # Set minimum width based on ratio
            )
            text_edit.grid(row=0, column=i, sticky="nsew", padx=3, pady=(0, 10))
            # Removed direct insertion of values
            # text_edit.insert("0.0", values[i])  # Insert corresponding value
            self.text_edits.append(text_edit)

        # $출제창안내문(sheet_name)
        self.출제창안내문 = CTk.CTkLabel(
            self, 
            text="이곳에 문제의 샘플을 입력해두면 이와 동일한 유형으로 문제를 제작합니다. '지문', '문제', '정답', '해설'을 각각 적어주세요. ('해설'은 비워두어도 됩니다)\n[주의사항] 지문에 밑줄이 필요한 경우 밑줄 대신 대괄호 [ ]를 사용하고 문제에서도 '밑줄' 대신 '괄호'란 말을 사용하세요.", 
            font=(self.default_font[0], 12), 
            height=40, 
            text_color="black", 
            fg_color="white"
        )
        self.출제창안내문.grid(row=1, column=0, pady=1, padx=10, sticky="ew")

        # Create a frame for buttons and preset controls, placed at the bottom
        button_preset_frame = CTk.CTkFrame(
            self,
            fg_color="#2B3D2D",
            border_width=0,
            corner_radius=3,
            height=35
        )
        button_preset_frame.grid(row=2, column=0, sticky="ew", padx=10, pady=10)
        button_preset_frame.grid_propagate(False)  # Prevent frame from shrinking to fit its contents
        button_preset_frame.pack_propagate(False)  # Pack is used for children; disable its propagation

        # Preset Controls Frame (Left Side)
        preset_controls_frame = CTk.CTkFrame(button_preset_frame, fg_color="transparent")
        preset_controls_frame.pack(side=tk.LEFT, padx=(0, 20), pady=0)



        # Manage Presets Button
        manage_preset_button = CTk.CTkButton(
            preset_controls_frame,
            text="관리",
            command=self.open_preset_manager,
            width=40,
            height=25,
            fg_color="white",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.default_font[0], 12, 'bold')
        )
        manage_preset_button.pack(side=tk.LEFT, padx=(10, 5))


        # Category Combobox Label
        CTk.CTkLabel(preset_controls_frame, text="카테고리:", font=(self.default_font[0], 12), text_color="white").pack(side=tk.LEFT, padx=(10, 5))

        # Category Combobox
        self.category_combobox = CTk.CTkComboBox(
            preset_controls_frame,
            font=(self.default_font[0], 12),
            command=self.on_category_selected,
            state='readonly',
            width=120
        )
        self.category_combobox.pack(side=tk.LEFT, padx=5)
        self.selected_save_category = "미분류"

        # Preset Combobox Label
        CTk.CTkLabel(preset_controls_frame, text="불러오기:", font=(self.default_font[0], 12), text_color="white").pack(side=tk.LEFT, padx=(10, 5))

        # Preset Combobox
        self.preset_combobox = CTk.CTkComboBox(
            preset_controls_frame,
            font=(self.default_font[0], 12),
            command=self.on_preset_selected,
            state='readonly',
            width=150
        )
        self.preset_combobox.pack(side=tk.LEFT, padx=5)
        self.preset_combobox.set("불러올 프리셋 선택")

        # Delete Preset Button - Consistent styling as per user request
        delete_preset_button = CTk.CTkButton(
            preset_controls_frame,
            text="삭제",
            command=self.delete_preset,
            width=40,
            height=25,
            fg_color="white",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.default_font[0], 12, 'bold')
        )
        delete_preset_button.pack(side=tk.LEFT, padx=(5, 0))

        # Up Arrow Button - Select previous preset
        prev_preset_button = CTk.CTkButton(
            preset_controls_frame,
            text="▲",
            command=self.select_previous_preset,
            width=25,
            height=25,
            fg_color="white",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.default_font[0], 12, 'bold')
        )
        prev_preset_button.pack(side=tk.LEFT, padx=(5, 0))

        # Down Arrow Button - Select next preset
        next_preset_button = CTk.CTkButton(
            preset_controls_frame,
            text="▼",
            command=self.select_next_preset,
            width=25,
            height=25,
            fg_color="white",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.default_font[0], 12, 'bold')
        )
        next_preset_button.pack(side=tk.LEFT, padx=(5, 30))

        # Preset Name Entry
        self.preset_name_entry = CTk.CTkEntry(
            preset_controls_frame,
            width=120,
            placeholder_text="저장할 프리셋 이름"
        )
        self.preset_name_entry.pack(side=tk.LEFT, padx=5)

        # Save Preset Button - Consistent styling as per user request
        save_preset_button = CTk.CTkButton(
            preset_controls_frame,
            text="저장",
            command=self.save_preset,
            width=40,
            height=25,
            fg_color="white",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.default_font[0], 12, 'bold')
        )
        save_preset_button.pack(side=tk.LEFT, padx=5)




        # Transform options dialog state
        self.transform_dialog = None

        # Right side frame to hold checkbox and button(s) close together
        right_side_frame = CTk.CTkFrame(button_preset_frame, fg_color="transparent")
        right_side_frame.pack(side=tk.RIGHT, padx=(0, 10), pady=0)

        # Font size buttons
        increase_font_button = CTk.CTkButton(
            right_side_frame,
            text="Aa+",
            fg_color="white",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.default_font[0], 14, 'bold'),
            command=self.increase_font_size,
            width=40,
            height=25
        )
        increase_font_button.pack(side=tk.LEFT, padx=(0, 5), pady=0)

        decrease_font_button = CTk.CTkButton(
            right_side_frame,
            text="Aa-",
            fg_color="white",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.default_font[0], 11, 'bold'),
            command=self.decrease_font_size,
            width=40,
            height=25
        )
        decrease_font_button.pack(side=tk.LEFT, padx=(0, 10), pady=0)

        # Transform options button (opens 지문 변형 옵션 dialog)
        self._transform_button_default_style = {
            "fg_color": "white",
            "hover_color": "#DDA15C",
            "text_color": "black",
        }
        self._transform_button_active_style = {
            "fg_color": "#DDA15C",
            "hover_color": "#BB873F",
            "text_color": "black",
        }
        self.transform_options_button = CTk.CTkButton(
            right_side_frame,
            text="옵션",
            command=self.open_transform_options,
            fg_color=self._transform_button_default_style["fg_color"],
            hover_color=self._transform_button_default_style["hover_color"],
            text_color=self._transform_button_default_style["text_color"],
            font=(self.default_font[0], 12, 'bold'),
            width=50,
            height=25
        )
        self.transform_options_button.pack(side=tk.LEFT, padx=(0, 10))
        self.지문변형_var.trace_add("write", self._on_transform_var_changed)
        self.update_transform_button_style()

        # Conditional buttons
        if self.is_manual_mocktest:
            # "모든 동형문제에 적용" Button
            apply_all_button = CTk.CTkButton(
                right_side_frame,
                text="동형문제 전체 적용",
                fg_color="#FEF9E0",
                hover_color="#DDA15C",
                text_color='black',
                font=(self.default_font[0], 12, 'bold'),
                command=self.apply_to_all,
                width=110,
                height=30
            )
            apply_all_button.pack(side=tk.LEFT, padx=5)


            # "현재 동형문제에만 적용" Button
            apply_current_button = CTk.CTkButton(
                right_side_frame,
                text="현재 문제에만 적용",
                fg_color="#FEF9E0",
                hover_color="#DDA15C",
                text_color='black',
                font=(self.default_font[0], 12, 'bold'),
                command=self.apply_to_current,
                width=110,
                height=30
            )
            apply_current_button.pack(side=tk.LEFT, padx=(5, 0))



        else:
            # Help button
            help_button = CTk.CTkButton(
                right_side_frame,
                text="도움말",
                fg_color="white",
                hover_color="#DDA15C",
                text_color='black',
                font=(self.default_font[0], 12, 'bold'),
                command=self.show_help,
                width=60,
                height=25
            )
            help_button.pack(side=tk.LEFT, padx=(0, 5))

            # Original "저장 후 닫기" Button for regular use
            save_close_button = CTk.CTkButton(
                right_side_frame,
                text="저장",
                fg_color="#FEF9E0",
                hover_color="#DDA15C",
                text_color='black',
                font=(self.default_font[0], 12, 'bold'),
                command=self.save_and_close,
                width=60,
                height=25
            )
            save_close_button.pack(side=tk.LEFT)





        # Preset File Path within app_data_dir
        self.presets_file = self.app_data_dir / "copyquestion_presets.json"

        # Last Copy Question Content File Path
        self.저장된동형문제샘플 = self.app_data_dir / "last_copy_question.txt"
        self.make_questions_options_file = self.app_data_dir / "make_questions_options.json"

        # Load presets and last copy question content
        self.load_presets()

        # Check if this is for manual mocktest and if there's existing content for this row
        if self.is_manual_mocktest and self.manual_mocktest_popup and self.row_idx:
            if hasattr(self.manual_mocktest_popup, 'copy_question_contents'):
                if self.row_idx in self.manual_mocktest_popup.copy_question_contents:
                    # Load the row-specific content
                    self.load_row_specific_content(self.manual_mocktest_popup.copy_question_contents[self.row_idx])
                else:
                    # Load the default last copy question
                    self.load_last_copy_question()
            else:
                # Load the default last copy question
                self.load_last_copy_question()
        else:
            # Normal behavior - load the last copy question
            self.load_last_copy_question()

        # Load saved transformation settings AFTER loading sample content
        # This ensures the authoritative settings from make_questions_options.json
        # override any settings saved in the sample file
        self.load_transform_settings()
        self.update_transform_button_style()

        # Auto-select matching preset if content matches an existing preset
        self.select_matching_preset()


        if sys.platform.startswith('darwin'):            
            # Initialize the CopyManager with the current window
            self.copy_manager = CopyManager(self)
            
            self.text_edits[0].bind("<KeyPress>", self.copy_manager.handle_keypress)
            self.text_edits[1].bind("<KeyPress>", self.copy_manager.handle_keypress)
            self.text_edits[2].bind("<KeyPress>", self.copy_manager.handle_keypress)
            self.text_edits[3].bind("<KeyPress>", self.copy_manager.handle_keypress)

        self.bind("<Escape>", self.on_popup_close)

        # Font size keyboard shortcuts (Command+ / Command- for Mac, Control+ / Control- for Windows)
        self.bind("<Command-plus>", lambda e: self.increase_font_size())
        self.bind("<Command-equal>", lambda e: self.increase_font_size())  # + requires Shift, so also bind =
        self.bind("<Command-minus>", lambda e: self.decrease_font_size())
        self.bind("<Control-plus>", lambda e: self.increase_font_size())
        self.bind("<Control-equal>", lambda e: self.increase_font_size())  # + requires Shift, so also bind =
        self.bind("<Control-minus>", lambda e: self.decrease_font_size())

        # Preset navigation keyboard shortcuts (Command/Control + Shift + Up/Down)
        self.bind("<Command-Shift-Up>", lambda e: self.select_previous_preset())
        self.bind("<Command-Shift-Down>", lambda e: self.select_next_preset())
        self.bind("<Control-Shift-Up>", lambda e: self.select_previous_preset())
        self.bind("<Control-Shift-Down>", lambda e: self.select_next_preset())

        # "현재 문제에만 적용" keyboard shortcut (Command+Enter for Mac, Control+Enter for Windows)
        if self.is_manual_mocktest:
            self.bind("<Command-Return>", lambda e: self.apply_to_current())
            self.bind("<Control-Return>", lambda e: self.apply_to_current())
            self.bind("<Command-KP_Enter>", lambda e: self.apply_to_current())  # Numpad Enter
            self.bind("<Control-KP_Enter>", lambda e: self.apply_to_current())  # Numpad Enter

    def on_popup_close(self, event=None):
        # CRITICAL: Call option override handler BEFORE destroying the dialog
        if hasattr(self, '_option_override_manager'):
            manager = getattr(self, '_option_override_manager')
            if manager and hasattr(manager, '_handle_option_popup_destroy'):
                manager._handle_option_popup_destroy(self)
        elif hasattr(self, 'manual_mocktest_popup'):
            # Fallback: update button colors in manual mocktest popup if it exists
            if self.manual_mocktest_popup and hasattr(self.manual_mocktest_popup, 'update_all_option_button_colors'):
                self.manual_mocktest_popup.update_all_option_button_colors()

        self.destroy()

    def open_transform_options(self):
        """Open the transform options dialog if not already open."""
        if self.transform_dialog and self.transform_dialog.winfo_exists():
            self.transform_dialog.focus_force()
            return

        font_family = self.default_font[0] if isinstance(self.default_font, tuple) else "Arial"
        font_size = 12  # Fixed font size for dialog, not affected by text_edit font size

        self.transform_dialog = TransformOptionsPopup(
            parent=self,
            transform_var=self.지문변형_var,
            difficulty_var=self.지문난이도_var,
            length_var=self.지문길이_var,
            content_change_var=self.content_change_var,
            font_family=font_family,
            font_size=font_size,
            on_confirm=self.on_transform_popup_confirmed,
            on_cancel=self.on_transform_popup_cancelled
        )

    def update_transform_button_style(self, *_):
        """Visually indicate whether transform options are active."""
        if not hasattr(self, "transform_options_button"):
            return
        if self.지문변형_var.get():
            self.transform_options_button.configure(
                fg_color="#DDA15C",
                hover_color="#BB6C25",
                text_color="white"
            )
        else:
            self.transform_options_button.configure(
                fg_color="#FEF9E0",
                hover_color="#DDA15C",
                text_color="black"
            )

    def load_transform_settings(self):
        """Load saved transform options from make_questions_options.json."""
        try:
            if self.make_questions_options_file.exists():
                with open(self.make_questions_options_file, 'r', encoding='utf-8') as f:
                    data = json.load(f)

                # Load transformation settings with defaults if not present
                paraphrase_enabled = data.get("동형문제paraphrase_enabled", False)
                difficulty_level = data.get("동형문제difficulty_level", 2)
                length_level = data.get("동형문제length_level", 2)
                content_change_value = data.get("동형문제content_change_mode", 0)

                # Update the variables with loaded values
                self.지문변형_var.set(paraphrase_enabled)
                self.지문난이도_var.set(difficulty_level)
                self.지문길이_var.set(length_level)
                # Handle None case - convert to 0 to avoid TclError
                final_content_change = content_change_value if content_change_value is not None else 0
                self.content_change_var.set(final_content_change)

                self.notify_option_button()
                self.update_transform_button_style()
        except (json.JSONDecodeError, OSError) as e:
            print(f"Failed to load 동형문제 transform settings: {e}")
            # Keep default values if loading fails

    def persist_transform_settings(self):
        """Persist current transform options to make_questions_options.json without clobbering other entries."""
        try:
            if self.make_questions_options_file.exists():
                with open(self.make_questions_options_file, 'r', encoding='utf-8') as f:
                    data = json.load(f)
            else:
                data = {}
        except (json.JSONDecodeError, OSError):
            data = {}

        paraphrase_enabled = self.지문변형_var.get()
        difficulty_level = self.지문난이도_var.get()
        length_level = self.지문길이_var.get()
        content_change_mode = self.content_change_var.get() if self.지문변형_var.get() else None

        data["동형문제paraphrase_enabled"] = paraphrase_enabled
        data["동형문제difficulty_level"] = difficulty_level
        data["동형문제length_level"] = length_level
        data["동형문제content_change_mode"] = content_change_mode

        try:
            with open(self.make_questions_options_file, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=4)
        except OSError as e:
            print(f"Failed to persist 동형문제 transform settings: {e}")

        # Update button colors in manual mocktest popup if it exists
        if hasattr(self, 'manual_mocktest_popup'):
            if self.manual_mocktest_popup and hasattr(self.manual_mocktest_popup, 'update_all_option_button_colors'):
                self.manual_mocktest_popup.update_all_option_button_colors()

    def notify_option_button(self):
        """Ask parent app (if provided) to refresh 동형문제 option button highlight."""
        if self.owner_app and hasattr(self.owner_app, "update_option_button_color"):
            try:
                self.owner_app.update_option_button_color("동형문제")
            except Exception as e:
                print(f"Failed to update 동형문제 option button color: {e}")

        # Also notify dashboard to refresh status in real-time
        if self.dashboard_ref and hasattr(self.dashboard_ref, "_reload_state_for"):
            try:
                self.dashboard_ref._reload_state_for("동형문제")
            except Exception as e:
                print(f"Failed to refresh dashboard for 동형문제: {e}")

    def _on_transform_var_changed(self, *args):
        """Update local transform button styling when toggle changes."""
        self.update_transform_button_style()

    def update_transform_button_style(self):
        """Reflect transform toggle state on the 옵션 button."""
        if not hasattr(self, "transform_options_button") or self.transform_options_button is None:
            return

        is_enabled = bool(self.지문변형_var.get())
        style = self._transform_button_active_style if is_enabled else self._transform_button_default_style
        try:
            self.transform_options_button.configure(
                fg_color=style["fg_color"],
                hover_color=style["hover_color"],
                text_color=style["text_color"],
            )
        except tk.TclError:
            pass

    def on_transform_popup_confirmed(self):
        """Callback when the difficulty popup is confirmed."""
        self.transform_dialog = None
        self.focus_force()
        self.persist_transform_settings()
        self.notify_option_button()
        self.update_transform_button_style()

    def on_transform_popup_cancelled(self):
        """Callback when the difficulty popup is cancelled."""
        self.transform_dialog = None
        self.focus_force()
        self.notify_option_button()
        self.update_transform_button_style()

    def load_font_size(self):
        """
        Load the last used font size for CopyQuestionPopup (separate from 문제검토창).
        """
        font_size_file = self.app_data_dir / 'copyquestion_font_size.json'
        if font_size_file.exists():
            try:
                with open(font_size_file, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    return data.get('font_size', 12)  # Default to 12 if not found
            except (json.JSONDecodeError, OSError):
                return 12
        return 12  # Default font size if file doesn't exist

    def save_font_size(self, font_size):
        """
        Save the font size for CopyQuestionPopup (separate from 문제검토창).
        """
        font_size_file = self.app_data_dir / 'copyquestion_font_size.json'
        try:
            with open(font_size_file, 'w', encoding='utf-8') as f:
                json.dump({'font_size': font_size}, f)
        except OSError as e:
            print(f"Failed to save font size: {e}")

    def increase_font_size(self):
        """Increase font size for all text edits."""
        self.default_font_size += 1
        for text_edit in self.text_edits:
            text_edit.configure(font=(self.default_font[0], self.default_font_size))
        self.save_font_size(self.default_font_size)

    def decrease_font_size(self):
        """Decrease font size for all text edits (minimum 1)."""
        if self.default_font_size > 1:
            self.default_font_size -= 1
            for text_edit in self.text_edits:
                text_edit.configure(font=(self.default_font[0], self.default_font_size))
            self.save_font_size(self.default_font_size)

    def _get_ordered_preset_names(self, category):
        """Return presets for a category honoring custom order."""
        if not category:
            return []

        categories = self.presets.get("categories", {})
        category_data = categories.get(category, {})
        if not isinstance(category_data, dict):
            return []

        ordered_names = []
        order = category_data.get("_order")
        if isinstance(order, list):
            for name in order:
                preset_data = category_data.get(name)
                if isinstance(preset_data, dict):
                    ordered_names.append(name)

        for name, preset_data in category_data.items():
            if name == "_order":
                continue
            if not isinstance(preset_data, dict):
                continue
            if name not in ordered_names:
                ordered_names.append(name)

        return ordered_names

    def _get_ordered_categories(self):
        """Return categories honoring custom order with 미분류 first."""
        categories = self.presets.get("categories", {})
        if not categories:
            return []

        # Get custom order from presets
        custom_order = self.presets.get("_category_order")

        if custom_order:
            # Validate: keep only existing categories
            valid_order = [c for c in custom_order if c in categories]

            # Add new categories not in order (after 미분류 if present)
            new_cats = sorted([c for c in categories if c not in valid_order])

            if "미분류" in valid_order:
                # Insert new categories after 미분류
                category_names = valid_order[:1] + new_cats + valid_order[1:]
            else:
                category_names = valid_order + new_cats

            # Ensure 미분류 is first if it exists
            if "미분류" in category_names and category_names[0] != "미분류":
                category_names = ["미분류"] + [c for c in category_names if c != "미분류"]

            return category_names
        else:
            # No custom order - use alphabetical with 미분류 first
            category_list = list(categories.keys())
            category_list.sort(key=lambda x: (x != "미분류", x))
            return category_list

    def _apply_category_selection(self, selected_category, preserve_preset_selection=False):
        """Update selection state and preset combobox for a category."""
        placeholder = "불러올 프리셋을 선택하세요"
        self.selected_save_category = selected_category

        if not hasattr(self, "preset_combobox"):
            return

        if not selected_category:
            self.preset_combobox.configure(values=[])
            self.preset_combobox.set("")
            return

        preset_names = self._get_ordered_preset_names(selected_category)
        current_value = self.preset_combobox.get()
        self.preset_combobox.configure(values=preset_names)

        if preserve_preset_selection and current_value in preset_names:
            self.preset_combobox.set(current_value)
        elif preset_names:
            self.preset_combobox.set(placeholder)
        else:
            self.preset_combobox.set("")

    def find_matching_preset(self):
        """
        Find a preset that matches the first three text fields (지문, 문제, 정답).
        해설 is excluded from matching criteria.
        Returns (category, preset_name) if found, otherwise (None, None).
        """
        # Get current content from text fields (excluding 해설)
        current_passage = self.text_edits[0].get("1.0", "end-1c").strip()
        current_question = self.text_edits[1].get("1.0", "end-1c").strip()
        current_answer = self.text_edits[2].get("1.0", "end-1c").strip()

        # Skip matching if all fields are empty or contain only placeholders
        if not current_passage and not current_question and not current_answer:
            return None, None

        categories = self.presets.get("categories", {})
        for category, presets in categories.items():
            if not isinstance(presets, dict):
                continue
            for preset_name, preset_data in presets.items():
                # Skip metadata entries like "_order"
                if preset_name.startswith("_") or not isinstance(preset_data, dict):
                    continue

                # Compare only 지문, 문제, 정답 (exclude 해설)
                preset_passage = preset_data.get("지문", "").strip()
                preset_question = preset_data.get("문제", "").strip()
                preset_answer = preset_data.get("정답", "").strip()

                if (current_passage == preset_passage and
                    current_question == preset_question and
                    current_answer == preset_answer):
                    return category, preset_name

        return None, None

    def select_matching_preset(self):
        """
        Find a matching preset and update the category/preset comboboxes if found.
        """
        category, preset_name = self.find_matching_preset()
        if category and preset_name:
            # Update category combobox
            if hasattr(self, "category_combobox"):
                self.category_combobox.set(category)
                # Update the preset combobox for this category
                self._apply_category_selection(category, preserve_preset_selection=False)
            # Update preset combobox
            if hasattr(self, "preset_combobox"):
                self.preset_combobox.set(preset_name)

    def migrate_presets_to_categories(self, old_presets):
        """
        Migrate old flat preset structure to new categorized structure.
        Old format: {preset_name: {data}}
        New format: {"categories": {"미분류": {preset_name: {data}}}}
        """
        if not old_presets:
            return {"categories": {"미분류": {}}}

        # Check if already migrated
        if "categories" in old_presets:
            return old_presets

        # Create backup
        backup_file = self.presets_file.with_suffix('.json.backup')
        try:
            with open(backup_file, 'w', encoding='utf-8') as file:
                json.dump(old_presets, file, ensure_ascii=False, indent=4)
        except Exception as e:
            print(f"Warning: Could not create backup file: {e}")

        # Migrate to new structure
        migrated = {
            "categories": {
                "미분류": old_presets.copy()
            }
        }

        return migrated

    def load_presets(self):
        """
        Load presets from the JSON file located in app_data_dir.
        Handles both old flat structure and new categorized structure.
        """
        if self.presets_file.exists():
            try:
                with open(self.presets_file, 'r', encoding='utf-8') as file:
                    loaded_data = json.load(file)

                # Migrate if needed
                if "categories" not in loaded_data:
                    self.presets = self.migrate_presets_to_categories(loaded_data)
                    # Save migrated structure
                    self.save_presets_to_file()
                else:
                    self.presets = loaded_data

            except json.JSONDecodeError:
                messagebox.showerror("Error", "프리셋 파일을 불러오는 중 오류가 발생했습니다.")
                self.presets = {"categories": {"미분류": {}}}
        else:
            self.presets = {"categories": {"미분류": {}}}

        # Update category combobox with custom order
        categories = self._get_ordered_categories()

        # Restore last selected category from saved preferences
        last_selected = self.presets.get("last_selected_category")
        save_needed = False

        if categories:
            # Use last selected category if it exists, otherwise fall back to defaults
            if last_selected and last_selected in categories:
                current_category = last_selected
            else:
                current_category = "미분류" if "미분류" in categories else categories[0]
                # Save the fallback category if we didn't have a saved selection
                if not last_selected or last_selected not in categories:
                    save_needed = True
        else:
            current_category = None

        if hasattr(self, 'category_combobox'):
            self.category_combobox.configure(values=categories)
            if current_category:
                self.category_combobox.set(current_category)
            else:
                self.category_combobox.set("")

        self._apply_category_selection(current_category, preserve_preset_selection=True)

        # Save the initial category selection if needed
        if save_needed and current_category:
            self.presets["last_selected_category"] = current_category
            self.save_presets_to_file()

    def on_category_selected(self, selected_category):
        """Handle category selection for saving presets."""
        self._apply_category_selection(selected_category, preserve_preset_selection=False)

        # Save the selected category to presets file for persistence
        self.presets["last_selected_category"] = selected_category
        self.save_presets_to_file()

    def open_preset_manager(self):
        """Open the preset manager dialog."""
        if hasattr(self, 'preset_manager_dialog') and self.preset_manager_dialog and self.preset_manager_dialog.winfo_exists():
            self.preset_manager_dialog.focus_force()
            return

        self.preset_manager_dialog = PresetManagerDialog(
            parent=self,
            presets_data=self.presets,
            default_font=self.default_font,
            on_close_callback=self.on_preset_manager_closed,
            provider=self
        )

    def on_preset_manager_closed(self):
        """Callback when preset manager dialog is closed."""
        self.preset_manager_dialog = None
        # Reload presets to reflect any changes
        self.load_presets()

    def create_category(self, category_name):
        """
        Create a new category.
        Returns True if successful, False if category already exists or name is invalid.
        """
        if not category_name or not category_name.strip():
            messagebox.showwarning("Warning", "카테고리 이름을 입력해주세요.")
            return False

        category_name = category_name.strip()

        if category_name in self.presets.get("categories", {}):
            messagebox.showwarning("Warning", f"카테고리 '{category_name}'이(가) 이미 존재합니다.")
            return False

        # Create new empty category
        if "categories" not in self.presets:
            self.presets["categories"] = {}
        self.presets["categories"][category_name] = {}

        # Get or create order
        order = list(self.presets.get("_category_order", []))

        # GUARD: Only inject 미분류 if it actually exists as a category
        if "미분류" in self.presets["categories"] and "미분류" not in order:
            order.insert(0, "미분류")

        # Add new category
        if category_name not in order:
            # Insert after 미분류 if it exists in order, otherwise append
            if "미분류" in order:
                order.insert(1, category_name)
            else:
                order.append(category_name)

        self.set_category_order(order)  # Saves automatically
        return True

    def delete_category(self, category_name):
        """
        Delete a category. Presets in the category are moved to '미분류'.
        Cannot delete '미분류' category.
        Returns True if successful, False otherwise.
        """
        if category_name == "미분류":
            messagebox.showwarning("Warning", "'미분류' 카테고리는 삭제할 수 없습니다.")
            return False

        if category_name not in self.presets.get("categories", {}):
            messagebox.showwarning("Warning", f"카테고리 '{category_name}'을(를) 찾을 수 없습니다.")
            return False

        # Check if category has presets (excluding _order metadata)
        presets_in_category = self.presets["categories"][category_name]
        preset_count = sum(1 for k in presets_in_category if k != "_order")

        if preset_count > 0:
            confirm = messagebox.askyesno(
                "Confirm Delete",
                f"카테고리 '{category_name}'에 {preset_count}개의 프리셋이 있습니다.\n"
                f"카테고리를 삭제하면 모든 프리셋이 '미분류'로 이동됩니다.\n계속하시겠습니까?"
            )
            if not confirm:
                return False

            # Move presets to '미분류' (excluding _order)
            if "미분류" not in self.presets["categories"]:
                self.presets["categories"]["미분류"] = {}

            for preset_name, preset_data in presets_in_category.items():
                if preset_name != "_order":  # Skip _order metadata
                    self.presets["categories"]["미분류"][preset_name] = preset_data

        # Delete category
        del self.presets["categories"][category_name]

        # Update order
        order = list(self.presets.get("_category_order", []))
        if category_name in order:
            order.remove(category_name)

        # Ensure 미분류 still at index 0 if it exists
        if "미분류" in self.presets["categories"] and "미분류" not in order:
            order.insert(0, "미분류")

        self.set_category_order(order)  # Saves automatically
        return True

    def rename_category(self, old_name, new_name):
        """
        Rename a category.
        Cannot rename '미분류' category.
        Returns True if successful, False otherwise.
        """
        if old_name == "미분류":
            messagebox.showwarning("Warning", "'미분류' 카테고리는 이름을 변경할 수 없습니다.")
            return False

        if not new_name or not new_name.strip():
            messagebox.showwarning("Warning", "새 카테고리 이름을 입력해주세요.")
            return False

        new_name = new_name.strip()

        if old_name not in self.presets.get("categories", {}):
            messagebox.showwarning("Warning", f"카테고리 '{old_name}'을(를) 찾을 수 없습니다.")
            return False

        if new_name in self.presets["categories"]:
            messagebox.showwarning("Warning", f"카테고리 '{new_name}'이(가) 이미 존재합니다.")
            return False

        # Rename category
        self.presets["categories"][new_name] = self.presets["categories"].pop(old_name)

        # Update in order
        order = list(self.presets.get("_category_order", []))
        if old_name in order:
            idx = order.index(old_name)
            order[idx] = new_name

        self.set_category_order(order)  # Saves automatically
        return True

    def move_preset_to_category(self, preset_name, from_category, to_category):
        """
        Move a preset from one category to another.
        Returns True if successful, False otherwise.
        """
        if from_category not in self.presets.get("categories", {}):
            messagebox.showwarning("Warning", f"카테고리 '{from_category}'을(를) 찾을 수 없습니다.")
            return False

        if to_category not in self.presets.get("categories", {}):
            messagebox.showwarning("Warning", f"카테고리 '{to_category}'을(를) 찾을 수 없습니다.")
            return False

        if preset_name not in self.presets["categories"][from_category]:
            messagebox.showwarning("Warning", f"프리셋 '{preset_name}'을(를) 찾을 수 없습니다.")
            return False

        # Check if preset with same name exists in target category
        if preset_name in self.presets["categories"][to_category]:
            confirm = messagebox.askyesno(
                "Confirm Move",
                f"카테고리 '{to_category}'에 이미 '{preset_name}' 프리셋이 존재합니다.\n덮어쓰시겠습니까?"
            )
            if not confirm:
                return False

        # Move preset
        preset_data = self.presets["categories"][from_category].pop(preset_name)
        self.presets["categories"][to_category][preset_name] = preset_data

        # Update _order in source category (remove)
        from_order = self.presets["categories"][from_category].get("_order", [])
        if preset_name in from_order:
            from_order.remove(preset_name)
            self.presets["categories"][from_category]["_order"] = from_order

        # Update _order in target category (add to end)
        to_order = self.presets["categories"][to_category].get("_order", [])
        if to_order and preset_name not in to_order:
            to_order.append(preset_name)
            self.presets["categories"][to_category]["_order"] = to_order

        self.save_presets_to_file()
        return True

    def rename_preset_in_category(self, old_name, new_name, category):
        """
        Rename a preset within a category.
        Returns True if successful, False otherwise.
        """
        if not new_name or not new_name.strip():
            messagebox.showwarning("Warning", "프리셋 이름을 입력해주세요.")
            return False

        new_name = new_name.strip()

        # Check if category exists
        if category not in self.presets.get("categories", {}):
            messagebox.showwarning("Warning", f"카테고리 '{category}'을(를) 찾을 수 없습니다.")
            return False

        # Check if old preset exists
        if old_name not in self.presets["categories"][category]:
            messagebox.showwarning("Warning", f"프리셋 '{old_name}'을(를) 찾을 수 없습니다.")
            return False

        # Check if new name already exists (and it's not the same name)
        if new_name in self.presets["categories"][category] and new_name != old_name:
            messagebox.showwarning("Warning", f"프리셋 '{new_name}'이(가) 이미 존재합니다.")
            return False

        # If same name, no change needed
        if new_name == old_name:
            return True

        # Rename: copy data to new key, delete old key
        preset_data = self.presets["categories"][category].pop(old_name)
        self.presets["categories"][category][new_name] = preset_data

        # Update name in _order if it exists
        order = self.presets["categories"][category].get("_order", [])
        if old_name in order:
            idx = order.index(old_name)
            order[idx] = new_name
            self.presets["categories"][category]["_order"] = order

        # Save to file
        self.save_presets_to_file()

        # Reload in main window to update combobox
        self.load_presets()

        return True

    def load_last_copy_question(self):
        """
        Load the last saved copy_question_content into the text_edit widgets.
        """
        if self.저장된동형문제샘플.exists():
            try:
                with open(self.저장된동형문제샘플, 'r', encoding='utf-8') as file:
                    content = file.read()
                
                # Split the content into sections based on the predefined format
                sections = content.split("\n")
                # Initialize variables to store each section
                passage = ""
                question = ""
                answer = ""
                explanation = ""
                transform_enabled = False
                difficulty_level = 2
                length_level = 2
                content_change_mode = 0

                current_section = None
                for line in sections:
                    if line.startswith("[Passage]"):
                        current_section = "passage"
                        passage += line.replace("[Passage]", "").strip() + "\n"
                    elif line.startswith("[Question]"):
                        current_section = "question"
                        question += line.replace("[Question]", "").strip() + "\n"
                    elif line.startswith("[Answer]"):
                        current_section = "answer"
                        answer += line.replace("[Answer]", "").strip() + "\n"
                    elif line.startswith("[Explanation]"):
                        current_section = "explanation"
                        explanation += line.replace("[Explanation]", "").strip() + "\n"
                    elif line.startswith("[Transform]"):
                        current_section = "transform"
                    elif current_section == "transform":
                        if line.startswith("enabled="):
                            transform_enabled = line.replace("enabled=", "").strip().lower() == "true"
                        elif line.startswith("difficulty="):
                            difficulty_level = int(line.replace("difficulty=", "").strip())
                        elif line.startswith("length="):
                            length_level = int(line.replace("length=", "").strip())
                        elif line.startswith("content_change="):
                            content_change_mode = int(line.replace("content_change=", "").strip())
                    else:
                        if current_section == "passage":
                            passage += line + "\n"
                        elif current_section == "question":
                            question += line + "\n"
                        elif current_section == "answer":
                            answer += line + "\n"
                        elif current_section == "explanation":
                            explanation += line + "\n"

                # Remove the last newline character
                passage = passage.strip()
                question = question.strip()
                answer = answer.strip()
                explanation = explanation.strip()
                if explanation == "":
                    if "문제1)" in question and "문제2)" in question:
                        explanation = "(Write your own explanations in Korean. Write an explanation for each answer with this format:\n해설1) explanation for answer 1 in Korean\n해설2) explanation for answer 2 in Korean)\n...)"
                    else:
                        explanation = "(Write your own explanation in Korean.)"
                    

                # Set transform settings (temporarily from saved sample)
                # NOTE: These will be overridden by load_transform_settings() called after this
                self.지문변형_var.set(transform_enabled)
                self.지문난이도_var.set(difficulty_level)
                self.지문길이_var.set(length_level)
                self.content_change_var.set(content_change_mode)
                # Don't persist here - let load_transform_settings() handle authoritative settings

                # Insert the loaded content into the text_edit widgets
                loaded_values = [passage, question, answer, explanation]
                for i, value in enumerate(loaded_values):
                    self.text_edits[i].delete("1.0", "end")
                    self.text_edits[i].insert("1.0", value)
                
            except Exception as e:
                messagebox.showerror("Error", f"Failed to load last copy question content: {e}")
        else:
            # If the file doesn't exist, initialize with empty strings or default values
            default_values = ["", "", "", ""]
            for i, value in enumerate(default_values):
                self.text_edits[i].delete("1.0", "end")
                self.text_edits[i].insert("1.0", value)

    def save_preset(self):
        """
        Save the current text fields as a new preset to the selected category.
        """
        preset_name = self.preset_name_entry.get().strip()
        if not preset_name:
            messagebox.showwarning("Warning", "프리셋 이름을 입력해주세요.")
            return

        # Open category selection dialog
        font_family = self.default_font[0] if isinstance(self.default_font, tuple) else "Arial"
        font_size = 12  # Fixed font size for dialog

        # Get current category from the combobox to use as initial selection
        current_category = None
        if hasattr(self, "category_combobox"):
            current_category = self.category_combobox.get().strip()

        category_dialog = CategorySelectionDialog(
            parent=self,
            presets_data=self.presets,
            font_family=font_family,
            font_size=font_size,
            initial_category=current_category
        )

        # Wait for the dialog to close
        self.wait_window(category_dialog)

        # Get the selected category
        target_category = category_dialog.selected_category
        if not target_category:
            # User cancelled
            return

        # Check if preset already exists in this category
        if target_category in self.presets.get("categories", {}):
            if preset_name in self.presets["categories"][target_category]:
                overwrite = messagebox.askyesno("Overwrite", f"카테고리 '{target_category}'에 프리셋 '{preset_name}'이(가) 이미 존재합니다.\n덮어쓰시겠습니까?")
                if not overwrite:
                    return

        # Collect current data
        passage = self.text_edits[0].get("1.0", "end-1c").strip()    # 지문
        question = self.text_edits[1].get("1.0", "end-1c").strip()   # 문제
        answer = self.text_edits[2].get("1.0", "end-1c").strip()     # 정답
        explanation = self.text_edits[3].get("1.0", "end-1c").strip() # 해설
        transform_enabled = self.지문변형_var.get()
        difficulty_level = self.지문난이도_var.get()
        length_level = self.지문길이_var.get()

        # Ensure categories structure exists
        if "categories" not in self.presets:
            self.presets["categories"] = {}
        if target_category not in self.presets["categories"]:
            self.presets["categories"][target_category] = {}

        # Check if this is a new preset (not overwriting)
        is_new_preset = preset_name not in self.presets["categories"][target_category]

        # Save to selected category
        self.presets["categories"][target_category][preset_name] = {
            "지문": passage,
            "문제": question,
            "정답": answer,
            "해설": explanation,
            "transform_enabled": transform_enabled,
            "difficulty_level": difficulty_level,
            "length_level": length_level
        }

        # If it's a new preset, add to end of _order
        if is_new_preset:
            order = self.presets["categories"][target_category].get("_order", [])
            if order is not None and preset_name not in order:
                order.append(preset_name)
                self.presets["categories"][target_category]["_order"] = order

        self.save_presets_to_file()

        # Reload presets to update UI
        self.load_presets()

        # Ensure comboboxes reflect the saved preset
        if hasattr(self, "category_combobox") and target_category:
            self.category_combobox.set(target_category)
        if hasattr(self, "preset_combobox"):
            self.preset_combobox.set(preset_name)

        self.preset_name_entry.delete(0, tk.END)
        messagebox.showinfo("Success", f"프리셋 '{preset_name}'이(가) '{target_category}' 카테고리에 저장되었습니다.")

    def delete_preset(self):
        """
        Delete the selected preset from the currently selected category.
        """
        if not hasattr(self, "preset_combobox"):
            return

        preset_name = self.preset_combobox.get().strip()
        placeholder = "불러올 프리셋을 선택하세요"
        if not preset_name or preset_name == placeholder:
            messagebox.showwarning("Warning", "삭제할 프리셋을 선택해주세요.")
            return

        category = self.selected_save_category
        if not category and hasattr(self, "category_combobox"):
            category = self.category_combobox.get().strip()

        if not category:
            messagebox.showwarning("Warning", "카테고리를 먼저 선택해주세요.")
            return

        categories = self.presets.get("categories", {})
        category_data = categories.get(category)
        if not isinstance(category_data, dict):
            messagebox.showwarning("Warning", f"카테고리 '{category}'을(를) 찾을 수 없습니다.")
            return

        if preset_name not in category_data:
            messagebox.showwarning("Warning", f"프리셋 '{preset_name}'을(를) 찾을 수 없습니다.")
            return

        confirm = messagebox.askyesno(
            "Confirm Delete",
            f"카테고리 '{category}'에서 프리셋 '{preset_name}'을(를) 삭제하시겠습니까?"
        )
        if confirm:
            del category_data[preset_name]

            # Remove from _order if it exists
            order = category_data.get("_order", [])
            if isinstance(order, list) and preset_name in order:
                order.remove(preset_name)
                category_data["_order"] = order

            self.save_presets_to_file()

            # Reload presets to update UI
            self.load_presets()
            if hasattr(self, "category_combobox"):
                self.category_combobox.set(category)
            messagebox.showinfo("Deleted", f"프리셋 '{preset_name}'이(가) 삭제되었습니다.")

    def on_preset_selected(self, preset_display_name):
        """
        Load the selected preset into the text fields.
        """
        placeholder = "불러올 프리셋을 선택하세요"
        if not preset_display_name or preset_display_name == placeholder:
            return

        preset_name = preset_display_name.strip()
        category = self.selected_save_category
        if not category and hasattr(self, "category_combobox"):
            category = self.category_combobox.get().strip()

        # Backwards compatibility for prefix format "[category] preset_name"
        # Only treat as category prefix if the extracted category actually exists
        if preset_name.startswith("[") and "]" in preset_name:
            try:
                end_bracket = preset_name.index("]")
                potential_category = preset_name[1:end_bracket]
                potential_preset_name = preset_name[end_bracket + 2:].strip()
                # Only use this parsing if the extracted category actually exists
                categories = self.presets.get("categories", {})
                if potential_category in categories and potential_preset_name:
                    category = potential_category
                    preset_name = potential_preset_name
                # Otherwise, keep the original preset_name as-is (it's just a name with brackets)
            except (ValueError, IndexError):
                pass  # Keep original preset_name

        if not category:
            messagebox.showwarning("Warning", "카테고리를 먼저 선택해주세요.")
            return

        categories = self.presets.get("categories", {})
        category_data = categories.get(category)
        if not isinstance(category_data, dict):
            messagebox.showwarning("Warning", f"카테고리 '{category}'을(를) 찾을 수 없습니다.")
            return

        preset = category_data.get(preset_name)
        if not isinstance(preset, dict):
            messagebox.showwarning("Warning", f"프리셋 '{preset_name}'을(를) 찾을 수 없습니다.")
            return

        # Load preset data into text fields
        self.text_edits[0].delete("1.0", "end")
        self.text_edits[0].insert("1.0", preset["지문"])
        self.text_edits[1].delete("1.0", "end")
        self.text_edits[1].insert("1.0", preset["문제"])
        self.text_edits[2].delete("1.0", "end")
        self.text_edits[2].insert("1.0", preset["정답"])
        self.text_edits[3].delete("1.0", "end")
        self.text_edits[3].insert("1.0", preset["해설"])

        # Load transform settings (with defaults for old presets)
        transform_enabled = preset.get("transform_enabled", False)
        difficulty_level = preset.get("difficulty_level", 2)
        length_level = preset.get("length_level", 2)
        self.지문변형_var.set(transform_enabled)
        self.지문난이도_var.set(difficulty_level)
        self.지문길이_var.set(length_level)

    def select_previous_preset(self):
        """Select the previous preset in the list."""
        self._navigate_preset(-1)

    def select_next_preset(self):
        """Select the next preset in the list."""
        self._navigate_preset(1)

    def _navigate_preset(self, direction):
        """
        Navigate to the previous (-1) or next (+1) preset.
        Wraps around at the beginning/end of the list.
        """
        category = self.selected_save_category
        if not category and hasattr(self, "category_combobox"):
            category = self.category_combobox.get().strip()

        if not category:
            return

        preset_names = self._get_ordered_preset_names(category)
        if not preset_names:
            return

        current_preset = self.preset_combobox.get().strip()
        placeholder = "불러올 프리셋을 선택하세요"

        # If no preset is selected or placeholder is shown, select the first/last based on direction
        if not current_preset or current_preset == placeholder or current_preset not in preset_names:
            if direction > 0:
                new_index = 0  # Start from first
            else:
                new_index = len(preset_names) - 1  # Start from last
        else:
            current_index = preset_names.index(current_preset)
            new_index = (current_index + direction) % len(preset_names)

        # Select and load the new preset
        new_preset = preset_names[new_index]
        self.preset_combobox.set(new_preset)
        self.on_preset_selected(new_preset)

    def save_presets_to_file(self):
        """
        Save the current presets to the JSON file located in app_data_dir.
        """
        try:
            with open(self.presets_file, 'w', encoding='utf-8') as file:
                json.dump(self.presets, file, ensure_ascii=False, indent=4)
        except Exception as e:
            messagebox.showerror("Error", f"프리셋을 저장하는 중 오류가 발생했습니다: {e}")

    def save_and_close(self):
        """
        Collect data from text fields, concatenate it, save, and close the popup.
        """
        # Reuse shared formatter to ensure transform settings persist
        copy_question_content = self.get_copy_question_content()
        if copy_question_content is None:
            return

        # Save the concatenated content to the last_copy_question_file
        try:
            with open(self.저장된동형문제샘플, 'w', encoding='utf-8') as file:
                file.write(copy_question_content)
        except Exception as e:
            messagebox.showerror("Error", f"Failed to save copy question content: {e}")
            return

        # Optionally, you can also save the presets or perform other actions here

        # Close the popup
        self.persist_transform_settings()
        self.notify_option_button()
        self.update_transform_button_style()

        # CRITICAL: Call option override handler BEFORE destroying the dialog
        if hasattr(self, '_option_override_manager'):
            manager = getattr(self, '_option_override_manager')
            if manager and hasattr(manager, '_handle_option_popup_destroy'):
                manager._handle_option_popup_destroy(self)
        elif hasattr(self, 'manual_mocktest_popup'):
            # Fallback: update button colors in manual mocktest popup if it exists
            if self.manual_mocktest_popup and hasattr(self.manual_mocktest_popup, 'update_all_option_button_colors'):
                self.manual_mocktest_popup.update_all_option_button_colors()

        self.destroy()








    def get_copy_question_content(self):
        """Extract and format the copy question content"""
        passage = self.text_edits[0].get("1.0", "end-1c").strip()
        question = self.text_edits[1].get("1.0", "end-1c").strip()
        answer = self.text_edits[2].get("1.0", "end-1c").strip()
        explanation = self.text_edits[3].get("1.0", "end-1c").strip()


        print(f"passage: {passage}")
        print(f"question: {question}")
        print(f"answer: {answer}")
        print(f"explanation: {explanation}")

            
        # Validation
        if not passage or not question or not answer:
            # Auto-fill answer template if question has multiple questions
            if "문제1)" in question and "문제2)" in question:
                question_count = 0
                for i in range(1, 20):
                    if f"문제{i})" in question:
                        question_count = i
                    else:
                        break
                if question_count >= 2:
                    answer_template = "\n".join([f"정답{i})" for i in range(1, question_count + 1)])
                    self.text_edits[2].delete("1.0", "end")
                    self.text_edits[2].insert("1.0", answer_template)

            messagebox.showwarning("경고", "'지문', '문제', '정답'란을 모두 채워주세요.")
            return None

        elif "문제1)" in question and "문제2)" in question and ("정답1)" not in answer or "정답2)" not in answer):
            # Count how many questions exist (문제1), 문제2), 문제3), etc.)
            question_count = 0
            for i in range(1, 20):  # Check up to 20 questions
                if f"문제{i})" in question:
                    question_count = i
                else:
                    break

            # Generate and fill answer template
            if question_count >= 2:
                answer_template = "\n".join([f"정답{i})" for i in range(1, question_count + 1)])
                self.text_edits[2].delete("1.0", "end")
                self.text_edits[2].insert("1.0", answer_template)

            messagebox.showwarning("경고", "복수 개의 문제가 발견되었습니다. 정답도 '정답'란에 '정답1)' '정답2)' 형식으로 구분해서 입력해주세요.")
            return

        elif "문제1)" in question and "문제2)" in question and "정답1)" in answer and "정답2)" in answer and explanation != "" and ("해설1)" not in explanation or "해설2)" not in explanation):
            messagebox.showwarning("경고", "복수 개의 문제가 발견되었습니다. 해설도 '해설'란에 '해설1)' '해설2)' 형식으로 구분해서 입력해주세요.")
            return

        if explanation == "":
            if "문제1)" in question and "문제2)" in question:
                explanation = "(Write your own explanations in Korean. Write an explanation for each answer with this format:\n해설1) explanation for answer 1 in Korean\n해설2) explanation for answer 2 in Korean\n...)"
            else:
                explanation = "(Write your own explanation in Korean.)"

        # Get transform settings
        transform_enabled = self.지문변형_var.get()
        difficulty_level = self.지문난이도_var.get()
        length_level = self.지문길이_var.get()
        content_change_mode = self.content_change_var.get() if transform_enabled else 0

        return (
            "[Passage]\n" + passage +
            "\n[Question]\n" + question +
            "\n[Answer]\n" + answer +
            "\n[Explanation]\n" + explanation +
            "\n[Transform]\n" +
            f"enabled={transform_enabled}\n" +
            f"difficulty={difficulty_level}\n" +
            f"length={length_level}\n" +
            f"content_change={content_change_mode}"
        )

    def apply_to_current(self):
        """Apply copy question content only to the current row"""
        content = self.get_copy_question_content()
        if content is None:
            return
            
        # Store content for the current row
        if self.manual_mocktest_popup:
            if not hasattr(self.manual_mocktest_popup, 'copy_question_contents'):
                self.manual_mocktest_popup.copy_question_contents = {}
            self.manual_mocktest_popup.copy_question_contents[self.row_idx] = content
            # Update the option tooltip to show the saved content
            if hasattr(self.manual_mocktest_popup, 'update_option_tooltip'):
                self.manual_mocktest_popup.update_option_tooltip(self.row_idx, content)

        # Save to file for persistence
        try:
            with open(self.저장된동형문제샘플, 'w', encoding='utf-8') as file:
                file.write(content)
        except Exception as e:
            messagebox.showerror("Error", f"Failed to save copy question content: {e}")
            return
            
        self.persist_transform_settings()
        self.notify_option_button()

        # CRITICAL: Call option override handler BEFORE destroying the dialog
        if hasattr(self, '_option_override_manager'):
            manager = getattr(self, '_option_override_manager')
            if manager and hasattr(manager, '_handle_option_popup_destroy'):
                manager._handle_option_popup_destroy(self)
        elif hasattr(self, 'manual_mocktest_popup'):
            # Fallback: update button colors in manual mocktest popup if it exists
            if self.manual_mocktest_popup and hasattr(self.manual_mocktest_popup, 'update_all_option_button_colors'):
                self.manual_mocktest_popup.update_all_option_button_colors()

        self.destroy()

    def show_help(self):
        """Show help dialog with preset usage instructions."""
        CopyQuestionHelpDialog(self, self.default_font[0], self.default_font_size)

    def apply_to_all(self):
        """Apply copy question content to all 동형문제 rows"""
        content = self.get_copy_question_content()
        if content is None:
            return
            
        # Show warning message
        result = messagebox.askyesno(
            "확인", 
            "실전모의고사의 모든 동형문제를 현재 유형으로 출제합니다. 개별 문제마다 별도로 선택한 유형이 있을 경우 현재 유형으로 덮어씌워집니다."
        )
        
        if not result:
            return
            
        # Apply to all 동형문제 rows
        if self.manual_mocktest_popup:
            if not hasattr(self.manual_mocktest_popup, 'copy_question_contents'):
                self.manual_mocktest_popup.copy_question_contents = {}
                
            # Find all rows with 동형문제 selected
            for row_data in self.manual_mocktest_popup.row_widgets:
                if row_data['problem_var'].get() == "동형문제":
                    row_idx = row_data['row_idx']
                    self.manual_mocktest_popup.copy_question_contents[row_idx] = content
                    # Update the option tooltip to show the saved content
                    if hasattr(self.manual_mocktest_popup, 'update_option_tooltip'):
                        self.manual_mocktest_popup.update_option_tooltip(row_idx, content)

        # Save to file for persistence
        try:
            with open(self.저장된동형문제샘플, 'w', encoding='utf-8') as file:
                file.write(content)
        except Exception as e:
            messagebox.showerror("Error", f"Failed to save copy question content: {e}")
            return
            
        self.persist_transform_settings()
        self.notify_option_button()

        # CRITICAL: Call option override handler BEFORE destroying the dialog
        if hasattr(self, '_option_override_manager'):
            manager = getattr(self, '_option_override_manager')
            if manager and hasattr(manager, '_handle_option_popup_destroy'):
                manager._handle_option_popup_destroy(self)
        elif hasattr(self, 'manual_mocktest_popup'):
            # Fallback: update button colors in manual mocktest popup if it exists
            if self.manual_mocktest_popup and hasattr(self.manual_mocktest_popup, 'update_all_option_button_colors'):
                self.manual_mocktest_popup.update_all_option_button_colors()

        self.destroy()


    def load_row_specific_content(self, content):
        """Load the specific content for this row into the text_edit widgets."""
        try:
            # Split the content into sections based on the predefined format
            sections = content.split("\n")
            # Initialize variables to store each section
            passage = ""
            question = ""
            answer = ""
            explanation = ""
            transform_enabled = False
            difficulty_level = 2
            length_level = 2
            content_change_mode = 0

            current_section = None
            for line in sections:
                if line.startswith("[Passage]"):
                    current_section = "passage"
                    passage += line.replace("[Passage]", "").strip() + "\n"
                elif line.startswith("[Question]"):
                    current_section = "question"
                    question += line.replace("[Question]", "").strip() + "\n"
                elif line.startswith("[Answer]"):
                    current_section = "answer"
                    answer += line.replace("[Answer]", "").strip() + "\n"
                elif line.startswith("[Explanation]"):
                    current_section = "explanation"
                    explanation += line.replace("[Explanation]", "").strip() + "\n"
                elif line.startswith("[Transform]"):
                    current_section = "transform"
                elif current_section == "transform":
                    if line.startswith("enabled="):
                        transform_enabled = line.replace("enabled=", "").strip().lower() == "true"
                    elif line.startswith("difficulty="):
                        difficulty_level = int(line.replace("difficulty=", "").strip())
                    elif line.startswith("length="):
                        length_level = int(line.replace("length=", "").strip())
                    elif line.startswith("content_change="):
                        content_change_mode = int(line.replace("content_change=", "").strip())
                else:
                    if current_section == "passage":
                        passage += line + "\n"
                    elif current_section == "question":
                        question += line + "\n"
                    elif current_section == "answer":
                        answer += line + "\n"
                    elif current_section == "explanation":
                        explanation += line + "\n"

            # Remove the last newline character
            passage = passage.strip()
            question = question.strip()
            answer = answer.strip()
            explanation = explanation.strip()

            # Set transform settings (temporarily from saved content)
            # NOTE: These will be overridden by load_transform_settings() called after this
            self.지문변형_var.set(transform_enabled)
            self.지문난이도_var.set(difficulty_level)
            self.지문길이_var.set(length_level)
            self.content_change_var.set(content_change_mode)
            # Don't persist here - let load_transform_settings() handle authoritative settings

            # Insert the loaded content into the text_edit widgets
            loaded_values = [passage, question, answer, explanation]
            for i, value in enumerate(loaded_values):
                self.text_edits[i].delete("1.0", "end")
                self.text_edits[i].insert("1.0", value)
                
        except Exception as e:
            print(f"Error loading row-specific content: {e}")
            # Fall back to loading the default content
            self.load_last_copy_question()

# Test Block to Run the Popup Independently
if __name__ == "__main__":
    import sys

    # Initialize CustomTkinter appearance and theme
    CTk.set_appearance_mode("System")  # Options: "System" (default), "Dark", "Light"
    CTk.set_default_color_theme("blue")  # Options: "blue" (default), "green", "dark-blue"

    # Define dummy functions for testing
    def load_window_size():
        """
        Dummy function to return window size.
        """
        return (1250, 680)  # Width x Height

    def load_font_size():
        """
        Dummy function to return font size.
        """
        return 12  # Default font size

    # Initialize the main Tkinter root window
    root = CTk.CTk()
    root.withdraw()  # Hide the root window

    # Instantiate the CopyQuestionPopup
    popup = CopyQuestionPopup(
        parent=root,
    )

    # Start the popup's event loop
    popup.mainloop()
