#modules/editing_functions.py
import re
import random

from docx.shared import Inches, Pt, RGBColor
from docx.oxml import OxmlElement, parse_xml
from docx.oxml.ns import qn, nsdecls
from docx.enum.text import WD_LINE_SPACING, WD_PARAGRAPH_ALIGNMENT, WD_BREAK





################## 서식 관련 함수 ###################
def create_bordered_paragraph(doc, style='Normal_modified'):
    """Create an empty paragraph with border and return it for further formatting"""
    paragraph = doc.add_paragraph(style=style)
    
    # Add paragraph border
    pPr = paragraph._p.get_or_add_pPr()
    pBdr = OxmlElement('w:pBdr')
    
    border_attrs = {
        'w:val': 'single',
        'w:sz': '5',
        'w:space': '6',
        'w:color': '000000'
    }
    
    for border_side in ['top', 'left', 'bottom', 'right']:
        border_element = OxmlElement(f'w:{border_side}')
        for attr, value in border_attrs.items():
            border_element.set(qn(attr), value)
        pBdr.append(border_element)
    
    pPr.append(pBdr)
    
    # Add padding
    paragraph_format = paragraph.paragraph_format
    paragraph_format.left_indent = Pt(5)
    paragraph_format.right_indent = Pt(5)
    paragraph_format.space_before = Pt(0)
    paragraph_format.space_after = Pt(0)
    
    return paragraph

def add_paragraph_with_formatting(doc, text, style='Normal_modified'):
    """Add a paragraph with curly brace italic formatting"""
    para = doc.add_paragraph(style=style)
    
    # Pattern to match curly braces
    pattern = re.compile(r"\{([^}]*)\}")
    
    last_idx = 0
    for match in pattern.finditer(text):
        start, end = match.span()
        
        # Add text before the match
        if start > last_idx:
            para.add_run(text[last_idx:start])
        
        # Add the text inside braces as italic (without the braces)
        text_inside_braces = match.group(1)
        run = para.add_run(text_inside_braces)
        run.italic = True
        
        last_idx = end
    
    # Add any remaining text
    if last_idx < len(text):
        para.add_run(text[last_idx:])
    
    return para


def set_cell_border(cell, **kwargs):
    tc = cell._tc
    tcPr = tc.get_or_add_tcPr()

    for edge in ('top', 'bottom', 'left', 'right', 'insideH', 'insideV'):
        edge_data = kwargs.get(edge)
        if edge_data:
            tag = 'w:{}'.format(edge)
            element = OxmlElement(tag)
            element.set(qn('w:val'), edge_data.get('val', 'single'))
            element.set(qn('w:sz'), str(edge_data.get('sz', 12)))
            element.set(qn('w:space'), str(edge_data.get('space', 0)))
            element.set(qn('w:color'), edge_data.get('color', 'auto'))
            tcPr.append(element)


def set_table_cell_margins(table, top=0, bottom=0, left=0, right=0):
    # Convert points to dxa (1 point = 20 dxa)
    top_dxa = str(int(top * 20))
    bottom_dxa = str(int(bottom * 20))
    left_dxa = str(int(left * 20))
    right_dxa = str(int(right * 20))

    for row in table.rows:
        for cell in row.cells:
            tcPr = cell._element.get_or_add_tcPr()
            tcMar = OxmlElement('w:tcMar')

            # Set top margin
            top_mar = OxmlElement('w:top')
            top_mar.set(qn('w:w'), top_dxa)
            top_mar.set(qn('w:type'), "dxa")
            tcMar.append(top_mar)

            # Set bottom margin
            bottom_mar = OxmlElement('w:bottom')
            bottom_mar.set(qn('w:w'), bottom_dxa)
            bottom_mar.set(qn('w:type'), "dxa")
            tcMar.append(bottom_mar)

            # Set left margin
            left_mar = OxmlElement('w:left')
            left_mar.set(qn('w:w'), left_dxa)
            left_mar.set(qn('w:type'), "dxa")
            tcMar.append(left_mar)

            # Set right margin
            right_mar = OxmlElement('w:right')
            right_mar.set(qn('w:w'), right_dxa)
            right_mar.set(qn('w:type'), "dxa")
            tcMar.append(right_mar)

            tcPr.append(tcMar)




########################################################
################### 편집 함수  ###################  
########################################################




def add_bold_underline_to_brackets(cell, paragraph_texts):
    # Define the pattern to match text within < > or [ ]
    pattern = re.compile(r"\<([^>]*)\>|\[([^\]]*)\]")

    # Iterate over each paragraph text
    for paragraph_text in paragraph_texts:
        # If it's the first paragraph, use the existing one; otherwise, create a new one
        if paragraph_texts.index(paragraph_text) == 0:
            para = cell.paragraphs[0]
            para.clear()  # Clear the existing paragraph to remove any default text
            para.style = 'Normal_modified'  # Apply the 'Normal_modified' style
        else:
            para = cell.add_paragraph(style='Normal_modified')

        last_idx = 0  # Start index of the last added run
        for match in pattern.finditer(paragraph_text):
            # Add the text before the match
            start, end = match.span()
            para.add_run(paragraph_text[last_idx:start])
            # Extract the text without the brackets for bold and underline
            text_within_brackets = match.group(1) or match.group(2)
            # Add the extracted text in bold and underline
            run = para.add_run(text_within_brackets)
            run.bold = True
            run.underline = True
            # Update the last index
            last_idx = end

        # Add any remaining text after the last match
        para.add_run(paragraph_text[last_idx:])




def 어휘1단계_편집(title, passage, result, answer_part, explanation_part, doc, app_logger):
    # Initialize an empty answer sheet
    answer_sheet = ""
    # Find all occurrences of the patterns within brackets
    brackets = re.findall(r"❮[^❯]*❯", result)
    modified_result = result

    for idx, bracket in enumerate(brackets, start=1):
        # Randomly decide to swap
        if "■" in bracket and "□" in bracket and random.random() > 0.5:
            swapped_bracket = re.sub(r"■(.*?)\/ □(.*?)❯", r"□\2 / ■\1❯", bracket)
            swapped = True
        else:
            swapped_bracket = bracket
            swapped = False

        # Determine the correct choice before swapping
        correct_choice_before_swap = "ⓐ" if "■" in swapped_bracket.split('/')[0] else "ⓑ"

        # Replace ■ and □ with ⓐ and ⓑ respectively, ensuring ⓐ comes before ⓑ
        # Ensure the left word symbol is always ⓐ and the right word symbol is always ⓑ
        if swapped:
            modified_bracket = re.sub(r"□(.*?)\/ ■(.*?)❯", r"ⓐ\1 / ⓑ\2❯", swapped_bracket)
        else:
            modified_bracket = re.sub(r"■(.*?)\/ □(.*?)❯", r"ⓐ\1 / ⓑ\2❯", swapped_bracket)

        # The correct choice remains the same as before swapping and symbol replacement
        correct_choice = correct_choice_before_swap

        words = re.search(r"ⓐ(.*?)\s*/\s*ⓑ(.*?)❯", modified_bracket)

        if words:
            word_a = words.group(1).strip()
            word_b = words.group(2).strip()
            # Proceed with your logic as before
        else:
            # Handle the case where the pattern is not found
            app_logger.warning(f"Pattern not found in bracket: {modified_bracket}")
            continue  # Skip this iteration or handle appropriately


        # Choose the word next to the correct choice
        correct_word = word_a if correct_choice == "ⓐ" else word_b

        # Add numbers and the correct choice to the answer sheet (without the word)
        answer_sheet += f"{idx}. {correct_choice}\n"
        #answer_sheet += f"{idx}. {correct_choice} {correct_word}\n"

        # Replace the original bracket with the modified one in the result text
        numbered_bracket = f"({idx}){modified_bracket}"
        modified_result = modified_result.replace(bracket, numbered_bracket, 1)
        modified_result = modified_result.strip()
        #explanation_part = ""

    answer_sheet = answer_sheet.rstrip('\n')

    # Use app_logger instead of print to avoid cp949 encoding errors on Windows
    """
    try:
        app_logger.info("==========================")
        app_logger.info("어휘1단계 편집 return")
        app_logger.info(f"title: {title}")
        app_logger.info(f"passage: {passage}")
        app_logger.info(f"modified_result: {modified_result}")
        app_logger.info(f"answer_sheet: {answer_sheet}")
        app_logger.info(f"explanation_part: {explanation_part}")
        app_logger.info("==========================")
    except UnicodeEncodeError:
        # Silently skip logging if encoding fails (contains unsupported characters)
        pass
    """

    return title, passage, modified_result, answer_sheet, explanation_part



def 어휘2단계_편집(title, passage, result, answer_part, explanation_part, doc, app_logger):
    # Split the result into the passage and the original words parts
    try:
        passage_part, original_words_part = result.split("Original Words:")
    except Exception as e:
        #print(f"어휘 2단계 Error: {e} - 어휘2단계 문제가 형식에 맞지 않습니다.")
        app_logger.error(f"어휘 2단계 Error: {e} - 어휘2단계 문제가 형식에 맞지 않습니다.")
        return None, None, None, None, None
        

    passage_part = passage_part.strip()
    original_words_part = original_words_part.strip()
    

    # Add a space after each closing parenthesis if it's not followed by a space already
    original_words_part = re.sub(r'\)(?=[^\s])', ') ', original_words_part)
    

    # Clean up and create a dictionary of original words with their corresponding numbers
    original_words_dict = {}
    for word in original_words_part.split(" ("):
        if word:
            
            try:
                number, original_word = word.strip(")").split(") ")
                number = number.replace('(', '')  # Remove the left parenthesis
                original_words_dict[int(number)] = original_word
            except ValueError as e:
                app_logger.error(f"Error processing word '{word}': {e}")
                continue

    # Find all occurrences of the patterns within brackets in the passage part
    brackets = re.findall(r"\(\d+\)❮[^❯]*❯", passage_part)
    modified_passage = passage_part
    answer_sheet = ""
    
    for idx, bracket in enumerate(brackets, start=1):
        number = int(re.search(r"\((\d+)\)", bracket).group(1))
        # Randomly decide to swap
        if "■" in bracket and "□" in bracket and random.random() > 0.5:
            swapped_bracket = re.sub(r"■(.*?)\/ □(.*?)❯", r"□\2 / ■\1❯", bracket)
            swapped = True
        else:
            swapped_bracket = bracket
            swapped = False

        # Determine the correct choice before swapping
        correct_choice_before_swap = "ⓐ" if "■" in swapped_bracket.split('/')[0] else "ⓑ"
        
        # Replace ■ and □ with ⓐ and ⓑ respectively, ensuring ⓐ comes before ⓑ
        # Ensure the left word symbol is always ⓐ and the right word symbol is always ⓑ
        if swapped:
            #modified_bracket = re.sub(r"□(.*?)\/ ■(.*?)❯", r"ⓐ\1 / ⓑ\2❯", swapped_bracket)
            modified_bracket = re.sub(r"□(.*?)\/ ■(.*?)❯", r"ⓐ\1/ ⓑ\2❯", swapped_bracket)
        else:
            #modified_bracket = re.sub(r"■(.*?)\/ □(.*?)❯", r"ⓐ\1 / ⓑ\2❯", swapped_bracket)
            modified_bracket = re.sub(r"■(.*?)\/ □(.*?)❯", r"ⓐ\1/ ⓑ\2❯", swapped_bracket)
        # The correct choice remains the same as before swapping
        correct_choice = correct_choice_before_swap



        words = re.search(r"ⓐ(.*?)\s*/\s*ⓑ(.*?)❯", modified_bracket)

        if words:
            word_a = words.group(1).strip()
            word_b = words.group(2).strip()
            # Proceed with your logic as before
        else:
            # Handle the case where the pattern is not found
            app_logger.warning(f"Pattern not found in bracket: {modified_bracket}")
            continue  # Skip this iteration or handle appropriately

        """
        # Extract words next to ⓐ and ⓑ
        words = re.search(r"ⓐ(.*?) / ⓑ(.*?)❯", modified_bracket)
        word_a = words.group(1).strip()
        word_b = words.group(2).strip()
        """
        
        # Choose the word next to the correct choice
        correct_word = word_a if correct_choice == "ⓐ" else word_b

        # Fetch the original word for the correct choice
        original_word = original_words_dict.get(number, "")

        # Add numbers and the correct choice with its word to the answer sheet
        answer_sheet += f"{idx}. {correct_choice} {correct_word} (={original_word})\n"        

        # Replace the original bracket with the modified one in the passage text
        modified_bracket_with_index = re.sub(r"\(\d+\)", f"({idx})", modified_bracket)
        modified_passage = modified_passage.replace(bracket, modified_bracket_with_index, 1)
        modified_passage = modified_passage.strip()
    answer_sheet = answer_sheet.rstrip('\n')


    # 문제 정답 숫자 맞는지 오류점검
    #max_passage_number = max([int(re.search(r"\((\d+)\)", bracket).group(1)) for bracket in brackets])
    #max_original_words_number = max(original_words_dict.keys())
    if brackets:  # Check if brackets is not empty
        max_passage_number = max([int(re.search(r"\((\d+)\)", bracket).group(1)) for bracket in brackets])
        max_original_words_number = max(original_words_dict.keys())

        # Log an error if the numbers do not match
        if max_passage_number != max_original_words_number:
            app_logger.info(f"{title}: Error! The last number in the passage ({max_passage_number}) does not match the last number in the original words ({max_original_words_number}).")
    else:
        # Handle the case when brackets is empty - possibly log a warning or set default values
        app_logger.warning(f"{title}: Warning! No brackets found in the passage.")
        # Optionally set default values for max_passage_number and max_original_words_number if needed for further processing
        max_passage_number = 0
        max_original_words_number = 0


    # Log an error if the numbers do not match
    if max_passage_number != max_original_words_number:
        app_logger.info(f"{title}: Error! The last number in the passage ({max_passage_number}) does not match the last number in the original words ({max_original_words_number}).")



    return title, passage, modified_passage, answer_sheet, explanation_part



def 어휘3단계_편집(title, passage, result, answer_part, explanation_part, app_logger):
    split_point = result.find("밑줄 친 부분 중")
    passage = result[:split_point].strip()
    modified_result = result[split_point:].strip()

    return title, passage, modified_result, answer_part, explanation_part



def 내용일치_편집(title, passage, result, answer_part, explanation_part, app_logger):
    # Split the result into 'Questions' and 'Answers' parts


    return title, passage, result, answer_part, explanation_part



def 내용일치한_편집(title, passage, result, answer_part, explanation_part, app_logger):
    # Split the result into 'Questions' and 'Answers' parts


    return title, passage, result, answer_part, explanation_part


def 밑줄의미_편집(title, passage, result, answer_part, explanation_part, app_logger):
    # Split the result into 'Questions' and 'Answers' parts
    # Splitting the text into two parts as requested
    split_point = result.find("Which of the")
    passage = result[:split_point].strip()
    modified_result = result[split_point:].strip()

    modified_result = modified_result.replace("Which of the following understanding for ⓐ-ⓔ is NOT correct?\n", "밑줄 친 ⓐ-ⓔ에 대한 설명으로 적절하지 않은 것을 고르시오.\n")

    modified_result = modified_result.replace("Which of the following interpretations of ⓐ-ⓔ is NOT correct?\n", "밑줄 친 ⓐ-ⓔ에 대한 설명으로 적절하지 않은 것을 고르시오.\n")


    return title, passage, modified_result, answer_part, explanation_part



def 함축의미_편집(title, passage, result, answer_part, explanation_part, app_logger):
    # Split the result into 'Questions' and 'Answers' parts
    # Splitting the text into two parts as requested
    split_point = result.find("Which of the")
    passage = result[:split_point].strip()
    modified_result = result[split_point:].strip()

    #Fetch the data between [[ ]] and put it into a variable 'answer'.
    underline_match = re.search(r'\[\[(.*?)\]\]', passage)
    underline_part = underline_match.group(1) if underline_match else ''

    modified_result = modified_result.replace("Which of the following is the most appropriate meaning of the bracketed expression?", f"밑줄 친 [[{underline_part}]]가 다음 글에서 의미하는 바로 가장 적절한 것은?")
    

    return title, passage, modified_result, answer_part, explanation_part



def 무관한문장_편집(title, passage, result, answer_part, explanation_part, app_logger):
    # Split the result into 'Questions' and 'Answers' parts
    # Splitting the text into two parts as requested
    
    

    return title, passage, result, answer_part, explanation_part



def 추론_편집(title, passage, result, answer_part, explanation_part, app_logger):
    return title, passage, result, answer_part, explanation_part

def 빈칸추론_편집(title, passage, result, answer_part, explanation_part, app_logger):
    # Split the result into 'Questions' and 'Answers' parts
    # Splitting the text into two parts as requested
    split_point = result.find("Which of the")
    passage = result[:split_point].strip()
    modified_result = result[split_point:].strip()

    return title, passage, modified_result, answer_part, explanation_part



def 추론불가_편집(title, passage, result, answer_part, explanation_part, app_logger):
    return title, passage, result, answer_part, explanation_part

def 순서_편집(title, passage, result, answer_part, explanation_part, app_logger):
    # Splitting the text
    split_index = result.index("(A) ")  # Find the index where (A) starts
    passage = result[:split_index].strip()  # Everything before (A)
    result = result[split_index:].strip()  # Everything from (A) onward
    return title, passage, result, answer_part, explanation_part


def 삽입_편집(title, passage, result, answer_part, explanation_part, app_logger):
    # Splitting the data
    split_point = result.find('❯') + 1  # Find the index where the first part ends
    passage = result[:split_point].strip('❮❯')  # Extract and clean the first part
    result = result[split_point:].strip()  # Extract the remaining part
    return title, passage, result, answer_part, explanation_part


def 주제_편집(title, passage, result, answer_part, explanation_part, app_logger):
    return title, passage, result, answer_part, explanation_part
def 제목_편집(title, passage, result, answer_part, explanation_part, app_logger):
    return title, passage, result, answer_part, explanation_part
def 요지_편집(title, passage, result, answer_part, explanation_part, app_logger):
    return title, passage, result, answer_part, explanation_part


"""
Border and Margin Configuration Guide:

MARGINS (Cell Padding):
- Values are in twips: 1 inch = 1440 twips, 1 cm = 567 twips
- Example: 144 twips = 0.1 inch, 288 twips = 0.2 inch

BORDER STYLES (val):
- 'single' - single line
- 'double' - double line
- 'dotted' - dotted line
- 'dashed' - dashed line
- 'thick' - thick line
- 'thickThinSmallGap' - thick outside, thin inside
- 'thinThickSmallGap' - thin outside, thick inside
- 'none' - no border

BORDER WIDTH (sz):
- Values in eighths of a point
- 8 = 1pt, 12 = 1.5pt, 16 = 2pt, 24 = 3pt, 32 = 4pt

BORDER COLOR:
- Hex color without # (e.g., '000000' for black, 'FF0000' for red)
"""

def set_underline_color(run, color='000000'):
    """Set underline with a specific color independent of font color"""
    from docx.oxml import OxmlElement
    from docx.oxml.ns import qn

    run.underline = True
    rPr = run._r.get_or_add_rPr()
    # Create underline element with color
    u = OxmlElement('w:u')
    u.set(qn('w:val'), 'single')
    u.set(qn('w:color'), color)  # Underline color
    # Remove any existing underline element
    for child in list(rPr):
        if child.tag == qn('w:u'):
            rPr.remove(child)
    # Add the new underline with color
    rPr.append(u)


def apply_paragraph_formatting(para, style='Normal_modified'):
    """Apply the exact same formatting as create_bordered_paragraph to an existing paragraph"""
    from docx.oxml import OxmlElement
    from docx.oxml.ns import qn
    from docx.shared import Pt

    # Apply the style
    try:
        para.style = style
    except:
        pass  # Style might not exist

    # Add padding (exact same as create_bordered_paragraph)
    paragraph_format = para.paragraph_format
    paragraph_format.left_indent = Pt(5)
    paragraph_format.right_indent = Pt(5)
    paragraph_format.space_before = Pt(0)
    paragraph_format.space_after = Pt(0)

    return para


def 빈칸암기_편집(title, passage, result, answer_part, explanation_part, doc, app_logger):
    if explanation_part is None or explanation_part.strip() == "":
        explanation_part = " "
    answer_part = result

    # Simply return the result as question_part
    # The converter.py will handle the actual document insertion with border_flag logic
    question_part = result

    return title, passage, question_part, answer_part, explanation_part



def 빈칸암기_정답_편집(title, passage, result, answer_part, explanation_part, doc, app_logger):
    # Return the result as question_part for converter.py to handle border logic
    # This matches the new pattern used by 빈칸암기_편집
    if explanation_part is None or explanation_part.strip() == "":
        explanation_part = " "

    answer_part = result
    question_part = result  # The passage with blank markers [word]

    return title, passage, question_part, answer_part, explanation_part


def 빈칸암기_편집old(title, passage, result, answer_part, explanation_part, doc, app_logger):
    if explanation_part is None or explanation_part.strip() == "":
        explanation_part = " "
    answer_part = result

    # Split result by new lines to handle hard enters as separate paragraphs
    lines = result.split('\n')

    # Create bordered paragraph instead of table
    bordered_para = create_bordered_paragraph(doc)
    
    for line_number, line_content in enumerate(lines):
        if line_number > 0:
            bordered_para.add_run('\n')
        
        # Combined pattern for brackets and curly braces
        combined_pattern = re.compile(r'(\[)(.*?)(])|(\{)([^}]*)(\})')
        last_idx = 0

        for match in combined_pattern.finditer(line_content):
            start, end = match.span()
            bordered_para.add_run(line_content[last_idx:start])

            if match.group(1) == '[':  # This is a bracket match
                run = bordered_para.add_run(" ")
                run.font.color.rgb = RGBColor(0xD8, 0xD8, 0xD8)
                shd = parse_xml(r'<w:shd {} w:fill="D8D8D8"/>'.format(nsdecls('w')))
                run._r.get_or_add_rPr().append(shd)

                word = match.group(2)
                if word:
                    first_letter_run = bordered_para.add_run(word[0])
                    first_letter_run.font.color.rgb = RGBColor(0x00, 0x00, 0x00)
                    shd = parse_xml(r'<w:shd {} w:fill="D8D8D8"/>'.format(nsdecls('w')))
                    first_letter_run._r.get_or_add_rPr().append(shd)

                    if len(word) > 1:
                        rest_of_word_run = bordered_para.add_run(word[1:])
                        rest_of_word_run.font.color.rgb = RGBColor(0xD8, 0xD8, 0xD8)
                        shd = parse_xml(r'<w:shd {} w:fill="D8D8D8"/>'.format(nsdecls('w')))
                        rest_of_word_run._r.get_or_add_rPr().append(shd)

                run = bordered_para.add_run("   ")
                run.font.color.rgb = RGBColor(0xD8, 0xD8, 0xD8)
                shd = parse_xml(r'<w:shd {} w:fill="D8D8D8"/>'.format(nsdecls('w')))
                run._r.get_or_add_rPr().append(shd)
                
            elif match.group(4) is not None:  # This is a curly brace match
                # Add italicized text without braces
                text_inside_braces = match.group(5)
                run = bordered_para.add_run(text_inside_braces)
                run.italic = True

            last_idx = end

        bordered_para.add_run(line_content[last_idx:])

    # Append explanation_part under the bordered paragraph, handling hard enters
    if explanation_part.strip():
        explanation_lines = explanation_part.split('\n')
        for line in explanation_lines:
            # Use the helper function to handle curly braces
            add_paragraph_with_formatting(doc, line)

    return title, passage, result, answer_part, explanation_part



def 빈칸암기_정답_편집old(title, passage, result, answer_part, explanation_part, doc, app_logger):
    # Split result by new lines to handle hard enters as separate paragraphs
    lines = result.split('\n')

    # Create bordered paragraph instead of table
    bordered_para = create_bordered_paragraph(doc)
    
    for line_number, line_content in enumerate(lines):
        if line_number > 0:
            bordered_para.add_run('\n')
        
        # Combined pattern for brackets and curly braces
        combined_pattern = re.compile(r'(\[)(.*?)(])|(\{)([^}]*)(\})')
        last_idx = 0

        for match in combined_pattern.finditer(line_content):
            start, end = match.span()
            bordered_para.add_run(line_content[last_idx:start])

            if match.group(1) == '[':  # This is a bracket match
                run = bordered_para.add_run(" ")
                run.font.color.rgb = RGBColor(0x00, 0x00, 0x00)
                shd = parse_xml(r'<w:shd {} w:fill="D8D8D8"/>'.format(nsdecls('w'), "D8D8D8"))
                run._r.get_or_add_rPr().append(shd)

                word = match.group(2)
                if word:
                    first_letter_run = bordered_para.add_run(word[0])
                    first_letter_run.font.color.rgb = RGBColor(0x00, 0x00, 0x00)
                    shd = parse_xml(r'<w:shd {} w:fill="D8D8D8"/>'.format(nsdecls('w'), "D8D8D8"))
                    first_letter_run._r.get_or_add_rPr().append(shd)

                    if len(word) > 1:
                        rest_of_word_run = bordered_para.add_run(word[1:])
                        rest_of_word_run.font.color.rgb = RGBColor(0x00, 0x00, 0x00)
                        shd = parse_xml(r'<w:shd {} w:fill="D8D8D8"/>'.format(nsdecls('w'), "D8D8D8"))
                        rest_of_word_run._r.get_or_add_rPr().append(shd)

                run = bordered_para.add_run("   ")
                run.font.color.rgb = RGBColor(0x00, 0x00, 0x00)
                shd = parse_xml(r'<w:shd {} w:fill="D8D8D8"/>'.format(nsdecls('w'), "D8D8D8"))
                run._r.get_or_add_rPr().append(shd)
                
            elif match.group(4) is not None:  # This is a curly brace match
                # Add italicized text without braces
                text_inside_braces = match.group(5)
                run = bordered_para.add_run(text_inside_braces)
                run.italic = True

            last_idx = end

        bordered_para.add_run(line_content[last_idx:])

    # Append explanation_part under the bordered paragraph, handling hard enters
    if explanation_part and explanation_part.strip():
        explanation_lines = explanation_part.split('\n')
        for line in explanation_lines:
            # Use the helper function to handle curly braces
            add_paragraph_with_formatting(doc, line)

    # Add a blank line (blank paragraph) after the bordered paragraph
    doc.add_paragraph(style='Normal_modified')

    return title, "", "", result, ""



def 연결어_편집(title, passage, result, answer_part, explanation_part, doc, app_logger, two_column=False):

    # Process explanation_part for None
    if explanation_part is None:
        explanation_part = ""

    # Splitting the text
    split_point = result.find("(1) ")
    passage, result = result[:split_point], result[split_point:]
    passage = passage.strip()

    # Removing phrases in brackets and replacing with ________
    modified_passage = re.sub(r"\[.*?\]", "______", passage)

    # Combine modified_passage and result for question_part
    # The converter.py will handle the actual document insertion with border_flag logic
    question_part = modified_passage + "\n" + result

    return title, passage, question_part, answer_part, explanation_part


def 어휘종합_편집(title, passage, result, answer_part, explanation_part, app_logger):
    # Splitting the text
    split_point = result.find("(1) ")
    passage, result = result[:split_point], result[split_point:]
    passage = passage.rstrip('\n')

    # Make sure each section starts with a newline and is properly enclosed in parentheses
    sections = result.strip().split('\n(')
    processed_result = '\n\n('.join(sections).strip()
    # Add the initial parenthesis back to the first section
    processed_result = processed_result.rstrip('\n')


    return title, passage, processed_result, answer_part, explanation_part



def 내용종합_편집(title, passage, result, answer_part, explanation_part, app_logger):
    # Make sure each section starts with a newline and is properly enclosed in parentheses
    sections = result.strip().split('\n(')
    processed_result = '\n\n('.join(sections).strip()
    # Add the initial parenthesis back to the first section
    processed_result = processed_result.rstrip('\n')

    return title, passage, processed_result, answer_part, explanation_part


def 요약_편집(title, passage, result, answer_part, explanation_part, app_logger):
    return title, passage, result, answer_part, explanation_part

def 영작1_편집(title, passage, result, answer_part, explanation_part, app_logger):
    
    return title, passage, result, answer_part, explanation_part

def 영작2_편집(title, passage, result, answer_part, explanation_part, app_logger):
    return title, passage, result, answer_part, explanation_part

def 주제영작_편집(title, passage, result, answer_part, explanation_part, app_logger):
    return title, passage, result, answer_part, explanation_part


def 동형문제_편집(title, passage, result, answer_part, explanation_part, app_logger):
    print("동형문제_편집 called")
    
    split_point = result.find("[Question]")
    passage = result[:split_point].strip()
    question_part = result[split_point:].strip()

    question_part = question_part.replace("[Question]", "").strip()

    
    return title, passage, question_part, answer_part, explanation_part


def 동반의어_편집(title, passage, result, answer_part, explanation_part, app_logger):
    return title, passage, result, answer_part, explanation_part

def 영영정의_편집(title, passage, result, answer_part, explanation_part, app_logger):
    return title, passage, result, answer_part, explanation_part


def 동반의어문제1_편집(title, passage, result, answer_part, explanation_part, doc, app_logger):
    # Split the result into individual lines
    lines = result.split('\n')
    
    for line in lines:
        # Create a new paragraph for each line
        paragraph = doc.add_paragraph(style='Normal_modified')
        
        # Split the line by '아닌'
        parts = line.split('아닌')
        
        # Add each part and '아닌' (if not the last part)
        for i, part in enumerate(parts):
            paragraph.add_run(part)
            if i < len(parts) - 1:  # If not the last part
                run = paragraph.add_run('아닌')
                run.bold = True
                run.underline = True
        
        # Set custom tab stops for the paragraph
        tab_stops = paragraph.paragraph_format.tab_stops
        
        # Set 4 tab stops (for 4 options)
        for tab_pos in range(1, 5):
            tab_stops.add_tab_stop(Inches(1.5 * tab_pos))

    return title, passage, result, answer_part, explanation_part


def 동반의어문제2_편집(title, passage, result, answer_part, explanation_part, doc, app_logger):
    # Split the result into individual lines
    lines = result.split('\n')
    
    for line in lines:
        # Create a new paragraph for each line
        paragraph = doc.add_paragraph(style='Normal_modified')
        
        # Split the line by '아닌'
        parts = line.split('아닌')
        
        # Add each part and '아닌' (if not the last part)
        for i, part in enumerate(parts):
            paragraph.add_run(part)
            if i < len(parts) - 1:  # If not the last part
                run = paragraph.add_run('아닌')
                run.bold = True
                run.underline = True
        
        # Set custom tab stops for the paragraph
        tab_stops = paragraph.paragraph_format.tab_stops
        
        # Set 4 tab stops (for 4 options)
        for tab_pos in range(1, 5):
            tab_stops.add_tab_stop(Inches(1.5 * tab_pos))

    return title, passage, result, answer_part, explanation_part


def 영영정의문제_편집(title, passage, result, answer_part, explanation_part, doc, app_logger):

    # Split the result at "아래 주어진"
    parts = result.split("아래 주어진", 1)
    
    if len(parts) == 2:
        passage = parts[0].strip()
        question_part = "아래 주어진 " + parts[1].strip()
    else:
        # If "아래 주어진" is not found, keep passage as is and put everything in question_part
        print("아래 주어진~ 으로 시작하는 문장이 없습니다.")
        question_part = result.strip()

    return title, passage, question_part, answer_part, explanation_part


def 요약문_편집(title, passage, result, answer_part, explanation_part, app_logger):
    
    return title, passage, result, answer_part, explanation_part


def 커스텀_편집(title, passage, result, answer_part, explanation_part, app_logger):
    return title, passage, result, answer_part, explanation_part


def 어법1단계_편집(title, passage, result, answer_part, explanation_part, doc, app_logger):
    
    # Initialize an empty answer sheet
    answer_sheet = ""
    # Find all occurrences of the patterns within brackets
    result = result.replace("❮", "❮ ■").replace("/", "/ □")
    
    brackets = re.findall(r"❮[^❯]*❯", result)
    modified_result = result
    
    for idx, bracket in enumerate(brackets, start=1):
        # Randomly decide to swap
        if "■" in bracket and "□" in bracket and random.random() > 0.5:
            swapped_bracket = re.sub(r"■(.*?)\/ □(.*?)❯", r"□\2 / ■\1❯", bracket)
            swapped = True
        else:
            swapped_bracket = bracket
            swapped = False

        # Determine the correct choice before swapping
        correct_choice_before_swap = "ⓐ" if "■" in swapped_bracket.split('/')[0] else "ⓑ"
        
        # Replace ■ and □ with ⓐ and ⓑ respectively, ensuring ⓐ comes before ⓑ
        # Ensure the left word symbol is always ⓐ and the right word symbol is always ⓑ
        if swapped:
            modified_bracket = re.sub(r"□(.*?)\/ ■(.*?)❯", r"ⓐ\1 / ⓑ\2❯", swapped_bracket)
        else:
            modified_bracket = re.sub(r"■(.*?)\/ □(.*?)❯", r"ⓐ\1 / ⓑ\2❯", swapped_bracket)
        
        # The correct choice remains the same as before swapping and symbol replacement
        correct_choice = correct_choice_before_swap
        
        words = re.search(r"ⓐ(.*?)\s*/\s*ⓑ(.*?)❯", modified_bracket)

        if words:
            word_a = words.group(1).strip()
            word_b = words.group(2).strip()
            # Proceed with your logic as before
        else:
            # Handle the case where the pattern is not found
            app_logger.warning(f"Pattern not found in bracket: {modified_bracket}")
            continue  # Skip this iteration or handle appropriately
        
        # Choose the word next to the correct choice
        correct_word = word_a if correct_choice == "ⓐ" else word_b
        
        # Add numbers and the correct choice with its word to the answer sheet
        #answer_sheet += f"{idx}. {correct_choice} {correct_word}\n"
        answer_sheet += f"{idx}. {correct_choice}\n"

        # Replace the original bracket with the modified one in the result text
        numbered_bracket = f"({idx}){modified_bracket}"
        modified_result = modified_result.replace(bracket, numbered_bracket, 1)
        modified_result = modified_result.strip()
        #explanation_part = ""
    answer_sheet = answer_sheet.rstrip('\n')


    return title, passage, modified_result, answer_sheet, explanation_part





def 밑줄친_편집_실모(title, passage, question_part, answer_part, explanation_part, app_logger):
    # Remove the line that starts with '밑줄 친'
    question_part_lines = question_part.split('\n')
    question_part = '\n'.join(line for line in question_part_lines if not line.startswith('밑줄 친'))
    question_part = question_part.strip()

    return title, passage, question_part, answer_part, explanation_part



def 어법2단계_편집(title, passage, result, answer_part, explanation_part, app_logger):
    if "밑줄 친 부분 중 어법상 틀린 것을 고르시오." in result and len(answer_part) > 1:
        result = result.replace("밑줄 친 부분 중 어법상 틀린 것을 고르시오.", "")
        result = result + "밑줄 친 부분 중 어법상 틀린 것을 모두 고르시오."

    split_point = result.find("밑줄 친 부분 중")
    passage = result[:split_point].strip()
    modified_result = result[split_point:].strip()

    return title, passage, modified_result, answer_part, explanation_part


def 동형문제_편집_실모(title, passage, result, answer_part, explanation_part, app_logger):
    
    split_point = result.find("[Question]")
    passage = result[:split_point].strip()
    question_part = result[split_point:].strip()

    question_part = question_part.replace("[Question]", "").strip()

    
    return title, passage, question_part, answer_part, explanation_part



def 지문분석_편집(title, passage, result, answer_part, explanation_part, app_logger):
    
    question_part = result.strip()
    
    return title, passage, question_part, answer_part, explanation_part



def 한글해석_편집(title, passage, result, answer_part, explanation_part, app_logger):
    
    question_part = result.strip()
    
    return title, passage, question_part, answer_part, explanation_part


def 직독직해_편집(title, passage, result, answer_part, explanation_part, app_logger):

    question_part = result.strip()
    
    return title, passage, question_part, answer_part, explanation_part

def 단어정리_편집(title, passage, result, answer_part, explanation_part, app_logger):
    
    question_part = result.strip()
    
    return title, passage, question_part, answer_part, explanation_part 
 
def 한줄해석연습_편집(title, passage, result, answer_part, explanation_part, app_logger):
    
    question_part = result.strip()
    
    return title, passage, question_part, answer_part, explanation_part

def 한줄영작연습_편집(title, passage, result, answer_part, explanation_part, app_logger):
    
    question_part = result.strip()
    
    return title, passage, question_part, answer_part, explanation_part







############################ 실모 #############################33



def 어휘3단계_편집_실모(title, passage, result, answer_part, explanation_part, app_logger):
    split_point = result.find("밑줄 친 부분 중")
    passage = result[:split_point].strip()
    modified_result = result[split_point:].strip()

    question_line = "다음 글의 밑줄 친 부분 중, 문맥상 낱말의 쓰임이 적절하지 않은 것은?"

    modified_result = modified_result.replace("밑줄 친 부분 중 문맥상 적절하지 않은 것을 고르시오.", "")

    return title, passage, question_line, modified_result, answer_part, explanation_part



def 어법2단계_편집_실모(title, passage, result, answer_part, explanation_part, app_logger):
    split_point = result.find("밑줄 친 부분 중")
    passage = result[:split_point].strip()
    modified_result = result[split_point:].strip()

    modified_result = modified_result.replace("밑줄 친 부분 중 어법상 틀린 것을 고르시오.", "").replace("밑줄 친 부분 중 어법상 틀린 것을 모두 고르시오.", "")
    modified_result = modified_result.strip() 

    if "모두 고르시오" in result:
        question_line = "다음 글의 밑줄 친 부분 중, 어법상 적절하지 않은 것을 모두 고르시오."
    else:
        question_line = "다음 글의 밑줄 친 부분 중, 어법상 적절하지 않은 것은?"


    return title, passage, question_line, modified_result, answer_part, explanation_part


def 내용일치_편집_실모(title, passage, question_part, answer_part, explanation_part, app_logger):
    # Replace numbered markers and remove "❮ T / F ❯" from question_part
    question_part = question_part.replace("(1)", "①").replace("(2)", "②").replace("(3)", "③").replace("(4)", "④").replace("(5)", "⑤").replace("❮ T / F ❯", "")

    # Count the number of 'T's in 'answer_part'
    correct_count = answer_part.count('T')
    
    # Prepend the appropriate line to 'question_part'
    if correct_count == 1:
        question_line = "다음 글의 내용과 일치하는 것을 고르시오."
    elif correct_count == 2:
        question_line = "다음 글의 내용과 일치하는 것을 2개 고르시오."
    elif correct_count == 3:
        question_line = "다음 글의 내용과 일치하지 않는 것을 2개 고르시오."
    elif correct_count == 4:
        question_line = "다음 글의 내용과 일치하지 않는 것을 고르시오."

    # Fetch the parenthesized numbers corresponding to 'T's or 'F's
    answer_lines = answer_part.split('\n')
    selected_numbers = []

    if correct_count == 1:
        selected_numbers = [re.search(r'\((\d+)\)', line).group(1) for line in answer_lines if 'T' in line]
    elif correct_count == 2:
        selected_numbers = [re.search(r'\((\d+)\)', line).group(1) for line in answer_lines if 'T' in line]
    elif correct_count == 3:
        selected_numbers = [re.search(r'\((\d+)\)', line).group(1) for line in answer_lines if 'F' in line]
    elif correct_count == 4:
        selected_numbers = [re.search(r'\((\d+)\)', line).group(1) for line in answer_lines if 'F' in line]

    # Convert the parenthesized numbers to circled numbers (①②③④⑤)
    circled_numbers = ['①', '②', '③', '④', '⑤']
    selected_circled_numbers = [circled_numbers[int(num) - 1] for num in selected_numbers]

    # Update the 'answer_part'
    answer_part = ''.join(selected_circled_numbers)

    # Replace numbered markers in explanation_part
    explanation_part = explanation_part.replace("(1)", "①").replace("(2)", "②").replace("(3)", "③").replace("(4)", "④").replace("(5)", "⑤")
    
    return title, passage, question_line, question_part, answer_part, explanation_part




def 밑줄의미_편집_실모(title, passage, question_part, answer_part, explanation_part, app_logger):
    # Split the result into 'Questions' and 'Answers' parts
    # Splitting the text into two parts as requested
    split_point = question_part.find("Which of the")
    passage = question_part[:split_point].strip()
    modified_result = question_part[split_point:].strip()

    modified_result = modified_result.replace("Which of the following understanding for ⓐ-ⓔ is NOT correct?\n", "")
    modified_result = modified_result.replace("Which of the following interpretations of ⓐ-ⓔ is NOT correct?\n", "")

    question_line = "밑줄 친 ⓐ-ⓔ에 대한 설명으로 적절하지 않은 것을 고르시오."


    return title, passage, question_line, modified_result, answer_part, explanation_part




def 함축의미_편집_실모(title, passage, result, answer_part, explanation_part, app_logger):
    # Split the result into 'Questions' and 'Answers' parts
    # Splitting the text into two parts as requested
    split_point = result.find("Which of the")
    passage = result[:split_point].strip()
    modified_result = result[split_point:].strip()

    #Fetch the data between [[ ]] and put it into a variable 'answer'.
    underline_match = re.search(r'\[\[(.*?)\]\]', passage)
    underline_part = underline_match.group(1) if underline_match else ''

    modified_result = modified_result.replace("Which of the following is the most appropriate meaning of the bracketed expression?\n", "")

    question_line = f"밑줄 친 [[{underline_part}]]가 다음 글에서 의미하는 바로 가장 적절한 것은?"


    return title, passage, question_line, modified_result, answer_part, explanation_part




def 무관한문장_편집_실모(title, passage, question_part, answer_part, explanation_part, app_logger):
    # Split the result into 'Questions' and 'Answers' parts
    # Splitting the text into two parts as requested
    split_point = question_part.find("윗글에서")
    passage = question_part[:split_point].strip()
    modified_result = question_part[split_point:].strip()
    
    question_line = "다음 글에서 전체 흐름과 관계 없는 문장은?"

    modified_result = modified_result.replace("윗글에서 전체 흐름과 관계 없는 문장은?", "")


    return title, passage, question_line, modified_result, answer_part, explanation_part



def 빈칸추론_편집_실모(title, passage, question_part, answer_part, explanation_part, app_logger):

    # Splitting the text into two parts as requested
    split_point = question_part.find("Which of the")
    passage = question_part[:split_point].strip()
    passage = passage.replace("[", "").replace("]", "")

    modified_result = question_part[split_point:].strip() #이게 Which 부터 시작하는 부분


    # Replace numbered markers and remove "❮ T / F ❯" from question_part
    question_line = "다음 빈칸에 들어갈 말로 가장 적절한 것을 고르시오."    

    # Remove the line that starts with 'Which of the following'
    question_part_lines = modified_result.split('\n')
    modified_result = '\n'.join(line for line in question_part_lines if not line.startswith('Which of the following'))
    modified_result = modified_result.strip()

    return title, passage, question_line, modified_result, answer_part, explanation_part



def 순서_편집_실모(title, passage, result, answer_part, explanation_part, app_logger):
    # Splitting the text
    split_index = result.index("(A) ")  # Find the index where (A) starts
    passage = result[:split_index].strip()  # Everything before (A)
    result = result[split_index:].strip()  # Everything from (A) onward

    question_line = "주어진 글 다음에 이어질 글의 순서로 가장 적절한 것을 고르시오."

    return title, passage, question_line, result, answer_part, explanation_part


def 삽입_편집_실모(title, passage, result, answer_part, explanation_part, app_logger):
    # Splitting the data
    split_point = result.find('❯') + 1  # Find the index where the first part ends
    passage = result[:split_point].strip('❮❯')  # Extract and clean the first part
    result = result[split_point:].strip()  # Extract the remaining part

    question_line = "글의 흐름으로 보아, 주어진 문장이 들어가기에 가장 적절한 곳을 고르시오."

    return title, passage, question_line, result, answer_part, explanation_part


def 주제_편집_실모(title, passage, question_part, answer_part, explanation_part, app_logger):

    # Remove the line that starts with 'Which of the following'
    question_part_lines = question_part.split('\n')
    question_part = '\n'.join(line for line in question_part_lines if not line.startswith('What is the most'))
    question_part = question_part.strip()

    question_line = "다음 글의 주제로 가장 적절한 것은?"

    return title, passage, question_line, question_part, answer_part, explanation_part


def 제목_편집_실모(title, passage, question_part, answer_part, explanation_part, app_logger):

    # Remove the line that starts with 'Which of the following'
    question_part_lines = question_part.split('\n')
    question_part = '\n'.join(line for line in question_part_lines if not line.startswith('What is the most'))
    question_part = question_part.strip()

    question_line = "다음 글의 제목으로 가장 적절한 것은?"

    return title, passage, question_line, question_part, answer_part, explanation_part




def 요지_편집_실모(title, passage, question_part, answer_part, explanation_part, app_logger):
    # Remove the line that starts with 'Which of the following'
    question_part_lines = question_part.split('\n')
    question_part = '\n'.join(line for line in question_part_lines if not line.startswith('윗글의 요지로'))
    question_part = question_part.strip()

    question_line = "다음 글의 요지로 가장 적절한 것은?"

    return title, passage, question_line, question_part, answer_part, explanation_part






def 추론_편집_실모(title, passage, question_part, answer_part, explanation_part, app_logger):
    # Replace numbered markers and remove "❮ T / F ❯" from question_part
    
    if "Choose Two" in question_part:
        number_of_answers = "2개 "
    elif "Choose Three" in question_part:
        number_of_answers = "3개 "
    else:
        number_of_answers = ""

    question_line = f"다음 글을 바탕으로 추론할 수 있는 것을 {number_of_answers}고르시오.\n"    

    # Remove the line that starts with 'Which of the following'
    question_part_lines = question_part.split('\n')
    question_part = '\n'.join(line for line in question_part_lines if not line.startswith('Which of the following'))
    question_part = question_part.strip()

    return title, passage, question_line, question_part, answer_part, explanation_part




def 추론불가_편집_실모(title, passage, question_part, answer_part, explanation_part, app_logger):
    # Replace numbered markers and remove "❮ T / F ❯" from question_part
    if "Choose Two" in question_part:
        number_of_answers = "2개 "
    elif "Choose Three" in question_part:
        number_of_answers = "3개 "
    else:
        number_of_answers = ""

    question_line = f"다음 글을 바탕으로 추론할 수 없는 것을 {number_of_answers}고르시오.\n"    


    # Remove the line that starts with 'Which of the following'
    question_part_lines = question_part.split('\n')
    question_part = '\n'.join(line for line in question_part_lines if not line.startswith('Which of the following'))
    question_part = question_part.strip()

    return title, passage, question_line, question_part, answer_part, explanation_part


def 요약_편집_실모(title, passage, result, answer_part, explanation_part, app_logger):
    question_line = "다음 글의 내용을 한 문장으로 요약하고자 한다. 빈칸 (A), (B)에 들어갈 말로 가장 적절한 것은?"
    question_part = result
    
    return title, passage, question_line, question_part, answer_part, explanation_part




def 영작1_편집_실모(title, passage, result, answer_part, explanation_part, app_logger):
    
    question_line = ""
    # Use regex to match the entire problem part before the actual content
    match = re.search(r"다음의 밑줄 친.*?\)", result)
    if match:
        question_line = match.group(0)  # Capture the entire "문제파트"

    # Remove the "문제파트" from the original text
    #result = result.replace(question_line, "").strip()

    return title, passage, question_line, result, answer_part, explanation_part

def 영작2_편집_실모(title, passage, result, answer_part, explanation_part, app_logger):
    # Find and capture the "문제파트"
    question_line = ""
    # Use regex to match the entire problem part before the actual content
    match = re.search(r"\[보기\]에.*?\)", result)
    if match:
        question_line = match.group(0)  # Capture the entire "문제파트"

    # Remove the "문제파트" from the original text
    #result = result.replace(question_line, "").strip()


    return title, passage, question_line, result, answer_part, explanation_part




def 주제영작_편집_실모(title, passage, result, answer_part, explanation_part, app_logger):
    
    question_line = ""
    # Use regex to match the entire problem part before the actual content
    match = re.search(r"윗글의.*?\)", result)
    if match:
        question_line = match.group(0)  # Capture the entire "문제파트"

    question_line = question_line.replace("윗글의", "다음 글의")

    # Remove the "문제파트" from the original text
    #result = result.replace(question_line, "").strip()

    return title, passage, question_line, result, answer_part, explanation_part






def 함축의미_편집_기출(title, passage, question_part, answer_part, explanation_part, app_logger):

    # Splitting the text into two parts as requested
    split_point = question_part.find("① ")
    passage = question_part[:split_point].strip()
    modified_question_part = question_part[split_point:].strip()

    #Fetch the data between [ ] and put it into a variable 'answer'.
    underline_match = re.search(r'\[(.*?)\]', passage)
    underline_part = underline_match.group(1) if underline_match else ''

    question_line = f"밑줄 친 [{underline_part}]가 다음 글에서 의미하는 바로 가장 적절한 것은?"


    passage_lines = passage.split('\n')
    modified_passage = '\n'.join(line for line in passage_lines if not line.startswith('밑줄 친'))
    modified_passage = modified_passage.strip()

    return title, modified_passage, question_line, modified_question_part, answer_part, explanation_part



def 다음글_편집_기출(title, passage, question_part, answer_part, explanation_part, app_logger):

    # Splitting the text into two parts as requested
    split_point = question_part.find("① ")
    passage = question_part[:split_point].strip()
    passage = passage.replace("[", "").replace("]", "")

    modified_question_part = question_part[split_point:].strip() #이게 '① '부터 시작하는 부분
    
    passage_lines = passage.split('\n')
    question_line = '\n'.join(line for line in passage_lines if line.startswith('다음 글'))
    question_line  = question_line.strip()

    modified_passage = '\n'.join(line for line in passage_lines if not line.startswith('다음 글'))
    modified_passage = modified_passage.strip()

    return title, modified_passage, question_line, modified_question_part, answer_part, explanation_part



def 일치안내문_편집_기출(title, passage, question_part, answer_part, explanation_part, app_logger):


    # Splitting the text into two parts as requested
    split_point = question_part.find("① ")
    passage = question_part[:split_point].strip()
    passage = passage.replace("[", "").replace("]", "")

    modified_question_part = question_part[split_point:].strip() #이게 '① '부터 시작하는 부분(선택지)
    
    passage_lines = passage.split('\n')
    question_line = '\n'.join(line for line in passage_lines if '다음' in line) #이게 발문
    question_line  = question_line.strip()

    modified_passage = '\n'.join(line for line in passage_lines if '다음' not in line) #이게 지문 부분
    modified_passage = modified_passage.strip()

    return title, modified_passage, question_line, modified_question_part, answer_part, explanation_part




def 어법어휘_편집_기출(title, passage, result, answer_part, explanation_part, app_logger):
    
    passage_lines = result.split('\n')
    question_line = '\n'.join(line for line in passage_lines if line.startswith('다음 글의'))
    question_line  = question_line.strip()

    modified_question_part = '\n'.join(line for line in passage_lines if not line.startswith('다음 글의'))
    modified_question_part = modified_question_part.strip()

    return title, modified_question_part, question_line, "", answer_part, explanation_part


def 빈칸추론_편집_기출(title, passage, question_part, answer_part, explanation_part, app_logger):
    # Splitting the text into two parts as requested
    split_point = question_part.find("① ")
    passage = question_part[:split_point].strip()
    passage = passage.replace("[", "").replace("]", "")

    modified_question_part = question_part[split_point:].strip() #이게 '① '부터 시작하는 부분

    
    passage_lines = passage.split('\n')
    question_line = '\n'.join(line for line in passage_lines if line.startswith('다음 빈칸에'))
    question_line  = question_line.strip()

    modified_passage = '\n'.join(line for line in passage_lines if not line.startswith('다음 빈칸에'))
    modified_passage = modified_passage.strip()

    return title, modified_passage, question_line, modified_question_part, answer_part, explanation_part


def 연결어_편집_기출(title, passage, question_part, answer_part, explanation_part, app_logger):
    # Splitting the text into two parts as requested
    split_point = question_part.find("(A) - (B)")
    passage = question_part[:split_point].strip()
    passage = passage.replace("[", "").replace("]", "")

    modified_question_part = question_part[split_point:].strip() #이게 '(A) - (B)'부터 시작하는 부분

    
    passage_lines = passage.split('\n')
    # Modified to include both lines starting with '다음 글의' and '빈칸 '
    question_line = '\n'.join(line for line in passage_lines if line.startswith('다음 글의') or line.startswith('빈칸 '))
    question_line = question_line.strip()

    # Modified to exclude both types of lines from the passage
    modified_passage = '\n'.join(line for line in passage_lines if not line.startswith('다음 글의') and not line.startswith('빈칸 '))
    modified_passage = modified_passage.strip()

    return title, modified_passage, question_line, modified_question_part, answer_part, explanation_part



def 무관한문장_편집_기출(title, passage, result, answer_part, explanation_part, app_logger):
    passage_lines = result.split('\n')
    question_line = '\n'.join(line for line in passage_lines if line.startswith('다음'))
    question_line  = question_line.strip()

    modified_question_part = '\n'.join(line for line in passage_lines if not line.startswith('다음'))
    modified_question_part = modified_question_part.strip()

    return title, "", question_line, modified_question_part, answer_part, explanation_part




def 순서_편집_기출(title, passage, result, answer_part, explanation_part, app_logger):
    # Splitting the text
    split_index = result.index("(A) ")  # Find the index where (A) starts
    passage = result[:split_index].strip()  # Everything before (A)
    result = result[split_index:].strip()  # Everything from (A) onward

    passage = passage.replace("주어진 글 다음에 이어질 글의 순서로 가장 적절한 것을 고르시오.", "").strip()
    question_line = "주어진 글 다음에 이어질 글의 순서로 가장 적절한 것을 고르시오."

    return title, passage, question_line, result, answer_part, explanation_part


def 삽입_편집_기출(title, passage, result, answer_part, explanation_part, app_logger):
    # Splitting the data
    split_point = result.find(']]') + 2  # Find the index where the first part ends
    passage = result[:split_point].strip()  # Extract and clean the first part
    passage = passage.replace('[[', '').replace(']]', '')

    result = result[split_point:].strip()  # Extract the remaining part

    passage = passage.replace("글의 흐름으로 보아, 주어진 문장이 들어가기에 가장 적절한 곳을 고르시오.", "").strip()
    question_line = "글의 흐름으로 보아, 주어진 문장이 들어가기에 가장 적절한 곳을 고르시오."

    return title, passage, question_line, result, answer_part, explanation_part


def 요약_편집_기출(title, passage, result, answer_part, explanation_part, app_logger):
    print("요약_편집_기출 called")

    for paragraph in result.split('\n'):
        if paragraph.startswith('*'):
            주석어휘 = paragraph
            
            passage = passage + "\n" + 주석어휘
            


    def get_question_part(result):
        # First, split into paragraphs
        paragraphs = result.split('\n')
        question_text = ''
        
        # Find the last paragraph containing "(A)" and collect everything after it
        start_index = -1
        for i, paragraph in enumerate(paragraphs):
            if '(A)__' in paragraph:  # Look specifically for "(A)__" instead of just "(A)"
                start_index = i
        
        if start_index != -1:
            question_text = '\n'.join(paragraphs[start_index:])
        
        return question_text.strip()




    question_part = get_question_part(result)
    question_part = question_part.replace("다음 글의 내용을 한 문장으로 요약하고자 한다. 빈칸 (A), (B)에 들어갈 말로 가장 적절한 것은?", "").strip()

    question_line = "다음 글의 내용을 한 문장으로 요약하고자 한다. 빈칸 (A), (B)에 들어갈 말로 가장 적절한 것은?"


    return title, passage, question_line, question_part, answer_part, explanation_part

def 장문독해1_편집_기출(title, passage, result, answer_part, explanation_part, app_logger):
    split_index = result.index("문제1") 
    passage = result[:split_index].strip()
    question_part = result[split_index:].strip() 

    passage = passage.replace("다음 글을 읽고, 물음에 답하시오.", "").strip()
    question_line = "다음 글을 읽고, 물음에 답하시오."


    return title, passage, question_line, question_part, answer_part, explanation_part

def 장문독해2_편집_기출(title, passage, result, answer_part, explanation_part, app_logger):
    split_index = result.index("문제1") 
    passage = result[:split_index].strip()
    question_part = result[split_index:].strip() 

    passage = passage.replace("다음 글을 읽고, 물음에 답하시오.", "").strip()
    question_line = "다음 글을 읽고, 물음에 답하시오."


    return title, passage, question_line, question_part, answer_part, explanation_part
    




FUNCTION_MAP = {
    '어휘1단계_편집': 어휘1단계_편집,
    '어휘2단계_편집': 어휘2단계_편집,
    '어휘3단계_편집': 어휘3단계_편집,
    '내용일치_편집': 내용일치_편집,
    '내용일치한_편집': 내용일치한_편집,
    '밑줄의미_편집': 밑줄의미_편집,
    '함축의미_편집': 함축의미_편집,
    '무관한문장_편집': 무관한문장_편집,
    '추론_편집': 추론_편집,
    '빈칸추론_편집': 빈칸추론_편집,
    '추론불가_편집': 추론불가_편집,
    '순서_편집': 순서_편집,
    '삽입_편집': 삽입_편집,
    '주제_편집': 주제_편집,
    '제목_편집': 제목_편집,
    '요지_편집': 요지_편집,
    '빈칸암기_편집': 빈칸암기_편집,
    '빈칸암기_정답_편집': 빈칸암기_정답_편집,
    '연결어_편집': 연결어_편집,
    '어휘종합_편집': 어휘종합_편집,
    '내용종합_편집': 내용종합_편집,
    '요약_편집': 요약_편집,
    '영작1_편집': 영작1_편집,
    '영작2_편집': 영작2_편집,
    '주제영작_편집': 주제영작_편집,
    '동형문제_편집': 동형문제_편집,
    '동반의어_편집': 동반의어_편집,
    '영영정의_편집': 영영정의_편집,
    '동반의어문제1_편집': 동반의어문제1_편집,
    '동반의어문제2_편집': 동반의어문제2_편집,
    '영영정의문제_편집': 영영정의문제_편집,
    '요약문_편집': 요약문_편집,
    '커스텀_편집': 커스텀_편집,
    '어법1단계_편집': 어법1단계_편집,
    '어법2단계_편집': 어법2단계_편집,
    '지문분석_편집': 지문분석_편집,
    '한글해석_편집': 한글해석_편집,
    '직독직해_편집': 직독직해_편집,
    '단어정리_편집': 단어정리_편집,
    '한줄해석연습_편집': 한줄해석연습_편집,
    '한줄영작연습_편집': 한줄영작연습_편집,



    '어휘3단계_편집_실모': 어휘3단계_편집_실모,
    '내용일치_편집_실모': 내용일치_편집_실모,
    '내용일치한_편집_실모': 내용일치_편집_실모,
    '밑줄의미_편집_실모': 밑줄의미_편집_실모,
    '함축의미_편집_실모': 함축의미_편집_실모,
    '무관한문장_편집_실모': 무관한문장_편집_실모,
    '빈칸추론_편집_실모': 빈칸추론_편집_실모,
    '추론_편집_실모': 추론_편집_실모,
    '추론불가_편집_실모': 추론불가_편집_실모,    
    '순서_편집_실모': 순서_편집_실모,
    '삽입_편집_실모': 삽입_편집_실모,
    '주제_편집_실모': 주제_편집_실모,
    '제목_편집_실모': 제목_편집_실모,    
    '요지_편집_실모': 요지_편집_실모,
    '어법2단계_편집_실모': 어법2단계_편집_실모,    
    '요약_편집_실모': 요약_편집_실모,
    '영작1_편집_실모': 영작1_편집_실모,
    '영작2_편집_실모': 영작2_편집_실모,
    '주제영작_편집_실모': 주제영작_편집_실모,    
    '동형문제_편집_실모': 동형문제_편집,



    '목적_편집_기출': 다음글_편집_기출,
    '심경_편집_기출': 다음글_편집_기출,
    '주장_편집_기출': 다음글_편집_기출,
    '함축의미_편집_기출': 함축의미_편집_기출,
    '요지_편집_기출': 다음글_편집_기출,
    '주제_편집_기출': 다음글_편집_기출,
    '제목_편집_기출': 다음글_편집_기출,
    '도표_편집_기출': 무관한문장_편집_기출,
    '일치_편집_기출': 일치안내문_편집_기출,
    '안내문_편집_기출': 일치안내문_편집_기출,
    '어법_편집_기출': 어법어휘_편집_기출,
    '어휘_편집_기출': 어법어휘_편집_기출,
    '빈칸추론_편집_기출': 빈칸추론_편집_기출,
    '연결어_편집_기출': 연결어_편집_기출,
    '무관한문장_편집_기출': 무관한문장_편집_기출,   
    '순서_편집_기출': 순서_편집_기출,
    '삽입_편집_기출': 삽입_편집_기출,
    '요약_편집_기출': 요약_편집_기출,    
    '장문독해1_편집_기출': 장문독해1_편집_기출,
    '장문독해2_편집_기출': 장문독해2_편집_기출,    

}


