# -*- coding: utf-8 -*-
"""
EditorApp - Text Editor Window for AutoQM
Extracted from autoQM.py NewWindowApp class
"""

# Standard library imports
import tkinter as tk
from tkinter import messagebox, Toplevel, Text, Button, Scrollbar, END, ttk, BooleanVar, Canvas, Entry, font, IntVar, PhotoImage
import os
import logging
import sys
from pathlib import Path
import time
import re
import random
import math
import unicodedata

# Third-party imports
from openpyxl import load_workbook
from openpyxl.utils import get_column_letter
import customtkinter
from customtkinter import *

import en_core_web_sm  # Import the model directly
#print(en_core_web_sm.__file__)
# Load the model
nlp = en_core_web_sm.load()

# Local imports
from modules.font_handler import font_handler
from modules.functions import CopyManager, ToolTip
from modules.macrofunctions import setup_macro_bindings

# Get logger
app_logger = logging.getLogger("app")

# Determine base_path
if getattr(sys, 'frozen', False):
    base_path = Path(sys._MEIPASS)
else:
    base_path = Path(__file__).parent.parent



class UndoManager1:
    """Undo/Redo manager for treeview operations"""

    def __init__(self, on_change_callback=None):
        self.undo_stack = []
        self.redo_stack = []
        self.on_change_callback = on_change_callback  # Store the callback

    def save_state(self, treeview):
        # Save the current state of the treeview
        rows = [(item, treeview.item(item)['values']) for item in treeview.get_children()]
        self.undo_stack.append(rows)
        self.redo_stack.clear()  # Clear the redo stack on new action

    def undo(self, treeview):
        if self.undo_stack:
            current_state = [(item, treeview.item(item)['values']) for item in treeview.get_children()]
            previous_state = self.undo_stack.pop()
            self.redo_stack.append(current_state)
            self._restore_state(treeview, previous_state)

            if self.on_change_callback:
                self.on_change_callback()

    def redo(self, treeview):
        if self.redo_stack:
            next_state = self.redo_stack.pop()
            current_state = [(item, treeview.item(item)['values']) for item in treeview.get_children()]
            self.undo_stack.append(current_state)
            self._restore_state(treeview, next_state)

            if self.on_change_callback:
                self.on_change_callback()

    def _restore_state(self, treeview, state):
        # Clear the treeview
        for item in treeview.get_children():
            treeview.delete(item)
        # Restore the saved state
        for item_id, values in state:
            treeview.insert('', 'end', iid=item_id, values=values)


class EditorApp:

    def __init__(self, main_frame, main_treeview, main_new_새파일저장묻기, main_update_sheet_selector, main_load_excel_into_treeview, main_save_treeview_to_excel, current_excel_file=None, get_excel_file_func=None, get_file_saved_func=None, username=None):
        self.main_frame = main_frame  # Store the reference to MainFrame
        self.tk_master = main_frame.master  # Reference to the Tkinter master/parent window
        self.current_excel_file = current_excel_file  # Store the current excel file path
        self.username = username  # Store the username for validation checks

        #self.master = master
        self.main_treeview = main_treeview  # Reference to the main window's treeview
        self.EditorApp_새파일저장묻기 = main_new_새파일저장묻기  # Function reference
        self.update_sheet_selector = main_update_sheet_selector
        self.load_excel_into_treeview = main_load_excel_into_treeview
        self.save_treeview_to_excel = main_save_treeview_to_excel

        # Helper functions to get current values from autoQM globals
        self.get_excel_file = get_excel_file_func if get_excel_file_func else lambda: __import__('autoQM').excel_file
        self.get_file_saved = get_file_saved_func if get_file_saved_func else lambda: __import__('autoQM').file_saved
        self.undo_manager = UndoManager1(self.on_treeview_change)
        self.last_click_time = 0

        self.window = CTkToplevel(self.tk_master)  # Create a Toplevel window as a popup
        self.window.title("지문 편집기")
        self.window.geometry("1100x700")
        self.window.minsize(1100, 700)
        self.window.configure(fg_color="#FFFFFF")
        self.window.resizable(True, True)
        #self.window.attributes("-topmost", True)  # This makes the window stay on top
        self.window.grab_set()
        self.window.focus_set()

        font_handler.set_default_font()
        self.default_font = font_handler.default_font
        self.setul_편집기_default_value()

        
        #### setup basic
        self.setup_key_bindings()
        self.setup_ui()
        self.create_window_menu()
        self.tree_view_tooltip = ToolTip(self.tree_view)
        
        # Bind tree view events
        self.tree_view.bind("<Motion>", self.on_treeview_motion)
        self.tree_view.bind("<Leave>", self.on_treeview_leave)


        ### 한글삭제 체크박스 기억하게 하기.....관련해서 디폴트값 미리 선택             
        self.한글옵션_radio_state = 0
        ### 영어삭제 체크박스 기억하게 하기.....관련해서 디폴트값 미리 선택             
        self.start_with_case_states = [1, 1]
        ### 숫자/문자 체크박스 기억하게 하기.....관련해서 디폴트값 미리 선택
        self.checkbox_states = [1] + [0]*12  # Assuming you want the first checkbox selected by default and others not
        self.radio_button_state = 0 # 시작하는 단락 전체 or 숫자만
        ### 단어n개이상이하 체크박스 기억하게 하기.....관련해서 디폴트값 미리 선택             
        self.remembered_number = 10  # Default value or last remembered value
        self.whatnumber_state = 2

        #self.edit_box.bind('<Command-v>', self.on_paste)
        #self.edit_box.bind('<Control-v>', self.on_paste)

        #복붙
        if sys.platform.startswith('darwin'):            
            # Initialize the CopyManager with the current window
            self.copy_manager = CopyManager(self.window, lambda: self.current_excel_file)
            # Bind the keypress events to the CopyManager's handle_keypress method
            self.window.bind("<KeyPress>", self.copy_manager.handle_keypress)

       
        self.edit_box.bind("<Button-2>", self.show_context_menu2)
        self.edit_box.bind("<Button-3>", self.show_context_menu2)

        self.tree_view.bind("<Button-2>", self.on_right_click)  # For Windows use <Button-3>, for MacOS use <Button-2>
        self.tree_view.bind("<Button-3>", self.on_right_click)  # For Windows use <Button-3>, for MacOS use <Button-2>






    def setup_ui(self):
       
        self.drawing_elements_for_New()

        # Create a tooltip for tree_view
        self.tree_view_tooltip = ToolTip(self.tree_view)

        # Bind the motion event to show tooltips on tree_view
        self.tree_view.bind("<Motion>", self.on_treeview_motion)


        # Context menu for tree_view
        self.context_menu = tk.Menu(self.window, tearoff=0)
        self.context_menu.add_command(label="행 삽입(위)", command=self.insert_row_above)
        self.context_menu.add_command(label="행 삽입(아래)", command=self.insert_row_below)
        self.context_menu.add_command(label="행 삭제", command=self.delete_row)
        self.context_menu.add_separator()
        self.context_menu.add_command(label="실행취소", command=lambda: self.undo_manager.undo(self.tree_view))
        self.context_menu.add_command(label="재실행", command=lambda: self.undo_manager.redo(self.tree_view))

        #self.text용 메뉴
        self.context_menu2 = tk.Menu(self.window, tearoff=0)
        self.context_menu2.add_command(label="복사", command=self.copy_text)
        self.context_menu2.add_command(label="붙여넣기", command=self.paste_text)
        self.context_menu2.add_command(label="잘라내기", command=self.cut_text)
        self.context_menu2.add_separator()
        self.context_menu2.add_command(label="실행취소", command=self.undo_text)
        self.context_menu2.add_command(label="재실행", command=self.redo_text)




        # Check the operating system to determine the correct modifier key
        if sys.platform.startswith('darwin'):
            # Mac OS
            modifier_key = 'Command'
        elif sys.platform.startswith('win'):
            # Windows
            modifier_key = 'Control'
        else:
            # Linux or other; adjust as needed
            modifier_key = 'Control'
            
        self.window.bind(f'<{modifier_key}-Return>', self.append_content_to_main_treeview)
        # Bind 'Escape' to cancel_edit as before
        self.window.bind("<Escape>", self.on_popup_close)
        self.window.protocol("WM_DELETE_WINDOW", self.on_popup_close)

        self.find_textbox.bind('<Tab>', self.move_to_replace)




    def move_to_replace(self, event):
        self.replace_textbox.focus_set()

        self.편집기_radio_var.set(4)
        return "break"  # Prevents the default Tab behavior



    def on_paste(self, event):
        # Get clipboard content
        clipboard_content = self.window.clipboard_get()

        # Modify the clipboard content as needed
        #modified_content = clipboard_content.replace('', '\n')
        modified_content = (clipboard_content
            .replace("\u2028", "\n")
            .replace("\u2029", "\n")
            .replace("\f", "")
            .replace("\u2013", "-")
            .replace("\u200B", "")   # zero-width space
            .replace("\u200C", "")   # zero-width non-joiner
            .replace("\u200D", "")   # zero-width joiner
            .replace("\uFEFF", "")   # byte order mark
        )



        # Clear the existing clipboard content and set the modified content
        self.window.clipboard_clear()
        self.window.clipboard_append(modified_content)

        # Insert the modified content into the CTkTextbox
        event.widget.insert(tk.INSERT, modified_content)

        # Now perform the paste operation
        event.widget.event_generate("<<Paste>>")

        # Return 'break' to prevent the default paste action
        return 'break'


    #################### 마우스 복붙
    def show_context_menu2(self, event):
        self.context_menu2.tk_popup(event.x_root, event.y_root)


    def copy_text(self):
        self.window.clipboard_clear()
        try:
            selected_text = self.edit_box.selection_get()
            self.window.clipboard_append(selected_text)
        except tk.TclError:
            pass  # No text selected

    def cut_text(self):
        self.window.clipboard_clear()
        try:
            selected_text = self.edit_box.selection_get()
            self.window.clipboard_append(selected_text)
            self.edit_box.delete(tk.SEL_FIRST, tk.SEL_LAST)
        except tk.TclError:
            pass  # No text selected

    def paste_text(self):
        self.edit_box.focus_set()
        try:
            clipboard_content = self.window.clipboard_get()
            modified_content = clipboard_content.replace("\u2028", "\n").replace("\u2029", "\n").replace("\f", "").replace("\u2013", "-")


            #print(modified_content)
            
            # Check if there's a selection
            try:
                selection_start = self.edit_box.index("sel.first")
                selection_end = self.edit_box.index("sel.last")
                has_selection = True
            except tk.TclError:
                has_selection = False

            if has_selection:
                # If there's a selection, replace it with the pasted content
                self.edit_box.delete(selection_start, selection_end)
                self.edit_box.insert(selection_start, modified_content)
                # Move the cursor to the end of the inserted text
                self.edit_box.mark_set(tk.INSERT, f"{selection_start}+{len(modified_content)}c")
            else:
                # If there's no selection, insert at the current cursor position
                current_position = self.edit_box.index(tk.INSERT)
                self.edit_box.insert(current_position, modified_content)
                # Move the cursor to the end of the inserted text
                self.edit_box.mark_set(tk.INSERT, f"{current_position}+{len(modified_content)}c")
            
        except tk.TclError:
            pass  # Clipboard is empty or contains non-text data
        
    def undo_text(self):
        try:
            self.edit_box.edit_undo()
        except tk.TclError:
            pass  # Nothing to undo

    def redo_text(self):
        try:
            self.edit_box.edit_redo()
        except tk.TclError:
            pass  # Nothing to redo


    #########################



    def setul_편집기_default_value(self):

        self.remembered_number = None
        # Initialize variables for checkboxes
        self.start_with_number1 = IntVar()
        self.start_with_number2 = IntVar()
        self.start_with_number3 = IntVar()
        self.start_with_number4 = IntVar()
        self.start_with_number5 = IntVar()
        self.start_with_number6 = IntVar()
        self.start_with_number7 = IntVar()
        self.start_with_number8 = IntVar()
        self.start_with_number9 = IntVar()
        self.start_with_number10 = IntVar()
        self.start_with_number11 = IntVar()
        self.start_with_number12 = IntVar()
        self.start_with_number13 = IntVar()
        self.start_with_number1.set(1)

        
        self.number_radio_var = IntVar()
        self.number_radio_var.set(0)
        

        self.한글옵션_radio_var = IntVar()
        self.한글옵션_radio_var.set(0)


        # 특정 문자열 var
        self.specific_string_delete_start_var = IntVar()
        self.specific_string_delete_contain_var = IntVar()
        # Default = start is checked, contain is unchecked
        self.specific_string_delete_start_var.set(1)
        self.specific_string_delete_contain_var.set(0)



        # Initialize variables for checkboxes
        self.start_with_uppercase_var = IntVar()
        self.start_with_lowercase_var = IntVar()
        self.start_with_uppercase_var.set(1)
        self.start_with_lowercase_var.set(1)

        self.unwanted_prefixes = ["다음 글", "주어진 글", "밑줄", "다음 빈칸", "글의 흐름", "윗 글", "윗글", "다음 밑줄", "→"]
        self.unwanted_phrases = ["내용과 일치하는 것은", "내용과 일치하지 않는 것은"]

        self.제목인식 = ""


    def create_window_menu(self):
        # Create a new menu bar for this window
        window_menu_bar = tk.Menu(self.window)


        # Edit menu integration
        if self.window.tk.call('tk', 'windowingsystem') == 'aqua':  # Checks if the OS is macOS
            modifier = "Cmd"
        else:
            modifier = "Ctrl"
        
        edit_menu = tk.Menu(window_menu_bar, tearoff=0)
        window_menu_bar.add_cascade(label="편집", menu=edit_menu)
        edit_menu.add_command(
            label=f"실행취소({modifier}+z)",
            command=lambda: self.undo_manager.undo(self.tree_view)
        )
        edit_menu.add_command(
            label=f"재실행({modifier}+Shift+z)",
            command=lambda: self.undo_manager.redo(self.tree_view)
        )

        # Set the newly created menu_bar as the menu for this window
        self.window.config(menu=window_menu_bar)

    #def window_close(self):
        # Close the window or perform other cleanup actions
        
        #self.window.destroy()




    ########################### 실행취소 undo manager 
    def on_treeview_change(self, event=None):
        for index, item in enumerate(self.tree_view.get_children()):
            # Check if item has error tag
            current_tags = self.tree_view.item(item, 'tags')
            has_error = 'error' in current_tags if current_tags else False

            # Set row tag and preserve error tag if needed
            if has_error:
                if index % 2 == 0:
                    self.tree_view.item(item, tags=('evenrow', 'error'))
                else:
                    self.tree_view.item(item, tags=('oddrow', 'error'))
            else:
                if index % 2 == 0:
                    self.tree_view.item(item, tags=('evenrow',))
                else:
                    self.tree_view.item(item, tags=('oddrow',))

        # After inserting rows into the Treeview, configure tag colors
        self.tree_view.tag_configure('evenrow', background='#F5F5F5')
        self.tree_view.tag_configure('oddrow', background='#FAFAFA')
        self.tree_view.tag_configure('error', foreground='red')



    def insert_row_above(self):
        self.undo_manager.save_state(self.tree_view)

        selected_items = self.tree_view.selection()
        if selected_items:
            # Insert a new row above the first selected item with placeholder values
            self.tree_view.insert("", self.tree_view.index(selected_items[0]), text="New Row", values=("","","",""))
        else:
            # Insert at the end if no row is selected, with placeholder values
            self.tree_view.insert("", tk.END, text="New Row", values=("","","",""))

        self.on_treeview_change()
        self.validate_and_highlight_errors()



    def insert_row_below(self):
        self.undo_manager.save_state(self.tree_view)

        selected_items = self.tree_view.selection()
        if selected_items:
            # Get the index of the selected item and add 1 to insert below
            index = self.tree_view.index(selected_items[0]) + 1
            # Insert a new row below the selected item with placeholder values
            self.tree_view.insert("", index, text="New Row", values=("","","",""))
        else:
            # Insert at the end if no row is selected, with placeholder values
            self.tree_view.insert("", tk.END, text="New Row", values=("","","",""))

        self.on_treeview_change()
        self.validate_and_highlight_errors()


    def delete_row(self):
        self.undo_manager.save_state(self.tree_view)

        selected_items = self.tree_view.selection()
        all_items = self.tree_view.get_children()

        # Check if we're about to delete all rows
        if set(selected_items) == set(all_items):
            # Keep one empty row
            self.tree_view.delete(*selected_items[:-1])
            last_item = selected_items[-1]
            self.tree_view.item(last_item, values=('',) * len(self.tree_view['columns']))
        else:
            for item in selected_items:
                self.tree_view.delete(item)

        # If all rows were deleted, add an empty row
        if not self.tree_view.get_children():
            self.tree_view.insert('', 'end', values=('',) * len(self.tree_view['columns']))

        self.on_treeview_change()
        self.validate_and_highlight_errors()

    def setup_key_bindings(self):
        # Modifier keys
        modifier = "Command" if self.window.tk.call('tk', 'windowingsystem') == 'aqua' else "Control"

        # Define event handlers
        def undo_event(event=None):
            self.undo_manager.undo(self.tree_view)

        def redo_event(event=None):
            self.undo_manager.redo(self.tree_view)

        # Binding Undo and Redo
        self.window.bind(f"<{modifier}-z>", undo_event)
        self.window.bind(f"<{modifier}-Shift-Z>", redo_event)  # Use Shift modifier for Redo





    def on_treeview_motion(self, event):

        item_id = self.tree_view.identify_row(event.y)
        if item_id:
            column = self.tree_view.identify_column(event.x)
            if column.startswith('#'):  # Ensure the column string starts with '#'
                column_no_str = column.replace('#', '')
                if column_no_str.isdigit():  # Ensure the resulting string is a digit before converting
                    column_no = int(column_no_str)
                    # Proceed with the rest of your code
                    values = self.tree_view.item(item_id, 'values')
                    if 0 <= column_no - 1 < len(values):
                        cell_value = values[column_no - 1]
                        # Get the bounding box for the item and column
                        bbox = self.tree_view.bbox(item_id, column)
                        # Check if bbox is not empty
                        if bbox:
                            x, y, _, _ = bbox
                            # Convert to root window coordinates
                            abs_x, abs_y = self.tree_view.winfo_pointerxy()
                            # Hide the existing tooltip (if any) to reset it
                            self.tree_view_tooltip.hide_tip()
                            # Show the tooltip with new content at the updated position
                            self.tree_view_tooltip.show_tip(str(cell_value), abs_x, abs_y)
                        else:
                            # Hide the tooltip if bbox could not be determined
                            self.tree_view_tooltip.hide_tip()
                else:
                    pass
            else:
                pass
        else:
            # Hide the tooltip if the cursor is not over a cell
            self.tree_view_tooltip.hide_tip()


    def on_treeview_leave(self, event):
        self.tree_view_tooltip.hide_tip()


    def on_right_click(self, event):
        # Attempt to identify the Treeview item row that was clicked
        row_id = self.tree_view.identify_row(event.y)
        if row_id:
            # Check if the clicked row is part of the current selection
            if row_id in self.tree_view.selection():
                # Show the context menu without changing the selection if the clicked row is already selected
                self.context_menu.tk_popup(event.x_root, event.y_root)
            else:
                # Optional: You might still want to show the context menu even if clicking outside the selection
                # but avoid changing the current selection.
                # Remove or comment out the next line if you don't want to show the context menu in this case.
                self.context_menu.tk_popup(event.x_root, event.y_root)
        else:
            # Clear selection if empty space is clicked - this line could be removed if you don't want to clear the selection
            #self.treeview.selection_remove(self.treeview.selection())
            pass  # Do nothing, or optionally close the context menu if it's open



    ############ 편집창 ############

    def edit_cell(self, event):




        # Hide the tooltip right at the beginning of the handler
        self.tree_view_tooltip.hide_tip()

        current_click_time = time.time()  # Access the current time
        if current_click_time - self.last_click_time < 0.3:
            return "break"
        self.last_click_time = current_click_time

        selection = self.tree_view.selection()
        if not selection:  # Check if the selection is empty
            return "break"  # Exit the function if nothing is selected


        row_id = self.tree_view.selection()[0]
        column = self.tree_view.identify_column(event.x)
        column_no = int(column.replace('#', ''))  # Convert column ID to index
        values = self.tree_view.item(row_id, 'values')

        # Check if column_no is within the bounds of values
        if column_no - 1 < len(values):
            current_value = values[column_no - 1]
            # Proceed with the rest of the function...
        else:
            # Possibly log an error or handle the case where the column index is out of bounds
            return



        def save_edit(event=None):

            new_value = text_edit.get("1.0", "end-1c")
            if new_value != current_value:
                self.undo_manager.save_state(self.tree_view)  # Save the entire treeview state before the edit
                self.tree_view.set(row_id, column=column, value=new_value)
                self.on_treeview_change()
                self.validate_and_highlight_errors()  # Re-validate after editing
            popup.destroy()



        # Function to cancel editing and close the popup
        def cancel_edit(event=None):  # Allow for calling without event argument
            popup.destroy()


        # Create a popup window for editing
        popup = tk.Toplevel(self.window)
        popup.title("Edit Cell")
        popup.geometry("600x400")  # Make the popup larger
        popup.grab_set()
        popup.focus_set()
        

        # Create a Scrollbar and a Text widget in the popup for editing
        text_scroll = tk.Scrollbar(popup)
        text_scroll.pack(side=tk.RIGHT, fill=tk.Y)

        text_edit = tk.Text(popup, width=50, height=10, font=(self.default_font, 16), yscrollcommand=text_scroll.set, undo=True, wrap=tk.WORD)  # Adjust width and height as needed
        text_edit.pack(pady=10, padx=10, fill="both", expand=True)
        text_edit.insert("1.0", current_value)  # Pre-fill the Text widget with the current cell value

        text_scroll.config(command=text_edit.yview)


        # Create a frame for buttons
        button_frame = tk.Frame(popup)
        button_frame.pack(pady=10, fill="x")

        # Create Save and Cancel buttons in the button frame
        # Check the operating system to determine the correct modifier key
        if sys.platform.startswith('darwin'):
            # Mac OS
            modifier_key = 'Command'
        elif sys.platform.startswith('win'):
            # Windows
            modifier_key = 'Control'
        else:
            # Linux or other; adjust as needed
            modifier_key = 'Control'

            
        # Create Save button in the button frame
        save_button = CTkButton(button_frame, text=f"Save", fg_color="#FEF9E0", hover_color="#DDA15C", text_color='black', font=(self.default_font, 14, 'bold'), command=save_edit, width=127, height=30)
        save_button.pack(side=tk.LEFT, padx=10, expand=True)

        # Create Cancel button in the button frame
        cancel_button = CTkButton(button_frame, text="Cancel", fg_color="#FEF9E0", hover_color="#DDA15C", text_color='black', font=(self.default_font, 14, 'bold'), command=cancel_edit, width=127, height=30)
        cancel_button.pack(side=tk.RIGHT, padx=10, expand=True)



        # Set focus on the Text widget for easy editing
        text_edit.focus_set()



        # Modify the bind calls inside the edit_cell function accordingly
        #popup.bind(f'<{modifier_key}-Return>', lambda event=None: save_edit(event) or "break")
        text_edit.bind(f'<{modifier_key}-Return>', lambda event=None: save_edit(event) or "break")

        #popup.bind(f'<{modifier_key}-Return>', save_edit)
        # Bind 'Escape' to cancel_edit as before
        popup.bind("<Escape>", cancel_edit)


        """
def custom_key_handler(event):
    # Check if the Command (on Mac) or Control (elsewhere) modifier is held down
    modifier_held = (event.state & 0x0004) if sys.platform == "darwin" else (event.state & 0x0004)

    if modifier_held and event.keysym == 'Return':  # Check for Command+Enter or Control+Enter
        save_edit(event)
        return "break"  # Prevent further processing

# Bind this custom handler to the text_edit widget for all KeyPress events
text_edit.bind('<KeyPress>', custom_key_handler, add='+')
        
        """




    def drawing_elements_for_New(self):


        self.window.grid_rowconfigure(0, weight=1)  # Make the window fill the height
        self.window.grid_columnconfigure(0, weight=1)  # Make the window fill the width
        self.window.focus_set()  # Set the focus to the window
        
        self.canvasframe = CTkFrame(master=self.window, fg_color="transparent", width=1100)
        self.canvasframe.grid(row=0, column=0, sticky="nsew")  # Place the canvasframe in the window
        self.canvasframe.grid_rowconfigure(0, weight=1)  # To make the frames fill the height
        self.canvasframe.grid_columnconfigure(0, weight=1)  # To make the left frame fill the width

        self.canvasframe.grid_columnconfigure(2, minsize=210)  # To make the left frame fill the width

        """
        # 폰트별
        if any(substring in self.default_font for substring in ("나눔 고딕", "나눔고딕", "NanumGothic", "Nanum Gothic")):
            self.canvasframe.grid_columnconfigure(2, minsize=210)  # To make the left frame fill the width
        else:
            self.canvasframe.grid_columnconfigure(2, minsize=200)  # To make the left frame fill the width
        """


        self.left_frame = CTkFrame(self.canvasframe, width=620, fg_color="transparent", corner_radius=0)
        self.left_frame.grid(row=0, column=0, sticky="nsew")
        self.left_frame.grid_rowconfigure(1, weight=1)  # To make the frames fill the height
        self.left_frame.grid_columnconfigure(0, weight=1)  # To make the left frame fill the width

        self.left_frame_top = CTkFrame(self.left_frame, width=620, height=30, fg_color="transparent", corner_radius=0)
        self.left_frame_top.grid(row=0, column=0, sticky="ew")
        self.left_frame_top.grid_columnconfigure(0, weight=1)  # Give weight to the column

        self.left_frame_bottom = CTkFrame(self.left_frame, width=620, fg_color="transparent", corner_radius=0)
        self.left_frame_bottom.grid(row=1, column=0, sticky="nsew")
        self.left_frame_bottom.grid_rowconfigure(0, weight=1)
        self.left_frame_bottom.grid_columnconfigure(0, weight=1)

        self.left_frame_title = CTkLabel(self.left_frame_top, font=(self.default_font, 12, "bold"), text="편집창", fg_color="transparent", text_color="black")
        self.left_frame_title.grid(row=0, column=0, sticky="nsew", padx=(0, 0), pady=(0, 0))  # Center the label

        self.edit_box = CTkTextbox(self.left_frame_bottom, text_color="black", fg_color=('#FEF9E0'), corner_radius=0, font=(self.default_font, 11), wrap="word", undo=True)
        self.edit_box.grid(row=0, column=0, sticky="nsew", padx=0, pady=0)

        placeholder_text = "Write here."
        self.edit_box.insert("0.0", placeholder_text)
        self.edit_box.configure(text_color='grey', font=(self.default_font, 25))

        # Bind focus in and focus out events
        self.edit_box.bind('<FocusIn>', self.on_entry_click)
        self.edit_box.bind('<FocusOut>', self.on_entry_focusout)








        ######### 센터 프레임 (대기열 미리보기)

        self.center_frame = CTkFrame(self.canvasframe, width=210, fg_color="transparent", corner_radius=0)
        self.center_frame.grid(row=0, column=1, sticky="ns")
        self.center_frame.grid_rowconfigure(1, weight=1)  # To make the frames fill the height
        self.center_frame.grid_columnconfigure(0, weight=1)  # To make the left frame fill the width


        self.center_frame_top = CTkFrame(self.center_frame, width=210, height=30, fg_color="transparent")
        self.center_frame_top.grid(row=0, column=0, sticky="nsew")
        self.center_frame_top.grid_columnconfigure(0, weight=1)  # Give weight to the column

        self.center_frame_bottom = CTkFrame(self.center_frame, width=210, fg_color="transparent", border_width=1, border_color="black")
        self.center_frame_bottom.grid(row=1, column=0, sticky="nsew")
        self.center_frame_bottom.grid_rowconfigure(0, weight=1)
        self.center_frame_bottom.grid_columnconfigure(0, weight=1)

        self.center_frame_title = CTkLabel(self.center_frame_top, font=(self.default_font, 12, "bold"), text="대기열 미리보기", text_color="black", fg_color="transparent")
        self.center_frame_title.grid(row=0, column=0, sticky="nsew", padx=(0, 0), pady=(0, 0))  # Center the label

        ### New 창 트리뷰
        style = ttk.Style(self.window)
        style.theme_use('clam')  # Experiment with 'alt', 'default', 'classic' as well
        style.layout("Treeview", [('self.treeview.treearea', {'sticky': 'nswe'})])  # Remove borders
        # Configure the Treeview colors (optional)
        style.configure("Treeview", background="#FFFFFF", borderwidth=1)
        style.configure("self.treeview.Heading", relief="flat", borderwidth=2, bordercolor="#FF0000")

    
        
        # Then, create the Treeview inside this frame
        self.tree_view = ttk.Treeview(self.center_frame_bottom)
        self.tree_view.grid(row=0, column=0, sticky="nsew")
        #self.tree_view.pack(fill='both', expand=True)  # Make the Treeview fill the frame
        
        # Define columns
        self.tree_view['columns'] = ('1', '2')
        self.tree_view['show'] = 'headings'  # Hide the first column
        
        # Setup column heading and width
        self.tree_view.heading('1', text='지문번호')
        self.tree_view.heading('2', text='지문')
        
        self.tree_view.column('1', width=105)
        self.tree_view.column('2', width=105)
        

        # Bind events for the Treeview
        self.tree_view.bind("<Double-1>", self.edit_cell)
        self.tree_view.bind("<Button-1>", lambda event: self.tree_view_tooltip.hide_tip())
        self.tree_view.bind('<Motion>', self.on_treeview_motion)
        self.tree_view.bind("<Button-2>", self.on_right_click)  # Use "<Button-2>" for macOS
        self.tree_view.bind("<Button-3>", self.on_right_click)  # Use "<Button-2>" for macOS
        self.tree_view.bind("<Leave>", self.on_treeview_leave)




        ################ 우측 프레임 도구


        ##################### 탭뷰1
        self.tabview = customtkinter.CTkTabview(master=self.canvasframe, width=230, corner_radius=10, border_width=0, text_color="white", fg_color="#2B3D2D", segmented_button_fg_color="#FEF9E0", segmented_button_selected_color="#CA7900", segmented_button_selected_hover_color="#DDA15C", segmented_button_unselected_color="#B4B4B4", segmented_button_unselected_hover_color="#E3E3E3")
        self.tabview.grid(row=0, column=2, sticky="ns", padx=0, pady=0)
        #self.tabview.grid_propagate (False)

        self.tabview._segmented_button.configure(font=(self.default_font, 11, 'bold'))
        self.tabview.add("출제용 편집기")  # add tab at the end
        self.tabview.add("고급 편집기")  # add tab at the end
        self.tabview.set("출제용 편집기")  # set currently visible tab

        self.tabview.tab("출제용 편집기").grid_rowconfigure(0, weight=1)
        self.tabview.tab("출제용 편집기").grid_columnconfigure(0, weight=1)

        self.tabview.tab("고급 편집기").grid_rowconfigure(0, weight=1)
        self.tabview.tab("고급 편집기").grid_columnconfigure(0, weight=1)


    
        self.right_frame = CTkFrame(self.tabview.tab("출제용 편집기"), width=230, fg_color="#2B3D2D", corner_radius=0)
        self.right_frame.grid(row=0, column=0, sticky="nsew", padx=0, pady=0)
        #self.right_frame.grid_propagate(False)
        self.right_frame.grid_rowconfigure(2, weight=1)
        self.right_frame.grid_columnconfigure(0, weight=1)
        

        self.right_frame_top = CTkFrame(self.right_frame, width=220, fg_color="transparent")
        self.right_frame_top.grid(row=0, column=0, sticky="ew", padx=0, pady=0)
        self.right_frame_top.grid_columnconfigure(0, weight=1)
        self.right_frame_top.grid_columnconfigure(1, weight=1)

        self.right_frame_middle = CTkFrame(self.right_frame, width=220, height=300, fg_color="transparent")
        self.right_frame_middle.grid(row=1, column=0, sticky="ew")

        self.right_frame_bottom = CTkFrame(self.right_frame, width=220, fg_color="#2B3D2D", corner_radius=7)
        self.right_frame_bottom.grid(row=2, column=0, pady=(10, 0), sticky="sew")
        self.right_frame_bottom.grid_columnconfigure(0, weight=1)


        분류실행_button = CTkButton(self.right_frame_top, text="분류 실행", width=80, height=30, font=(self.default_font, 12, "bold"), command=self.process_and_update_tree_view, text_color="black", fg_color="#FEF9E0", hover_color="#DDA15C") 
        분류실행_button.grid(row=0, column=0, padx=(15, 0), pady=(5, 5), sticky="w")
        대기열로_button = CTkButton(self.right_frame_top, text="저장", width=80, height=30, font=(self.default_font, 12, "bold"), command=self.append_content_to_main_treeview, text_color="black", fg_color="#FEF9E0", hover_color="#DDA15C")
        대기열로_button.grid(row=0, column=1, padx=(0, 15), pady=(5, 5), sticky="e")

        ######
        지문분류방식_label = CTkLabel(self.right_frame_middle, text="◼ 지문 분류 방식", font=(self.default_font, 15, "bold"), text_color="#FEF9E0", fg_color="transparent")
        지문분류방식_label.grid(row=0, column=0, padx=(10, 0), pady=3, sticky="w")
        self.radio_value = tk.IntVar(value=1)  # Set default selection to the first radio button
        분류1_radio = CTkRadioButton(self.right_frame_middle, text="지문 통째로 사용하기", font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C", radiobutton_width=20, radiobutton_height=20, variable=self.radio_value, value=1)
        분류1_radio.grid(row=1, column=0, padx=(30, 0), pady=2, sticky="w")
        분류2_radio = CTkRadioButton(self.right_frame_middle, text="단락 별로 나누기", font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C", radiobutton_width=20, radiobutton_height=20, variable=self.radio_value, value=2)
        분류2_radio.grid(row=2, column=0, padx=(30, 0), pady=2, sticky="w")
        분류3_radio = CTkRadioButton(self.right_frame_middle, text="기준 대로 나누기", font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C", radiobutton_width=20, radiobutton_height=20, variable=self.radio_value, value=3)
        분류3_radio.grid(row=3, column=0, padx=(30, 0), pady=2, sticky="w")

        분류3_entry_frame = CTkFrame(self.right_frame_middle, fg_color="transparent")
        분류3_entry_frame.grid(row=4, column=0, padx=0, pady=2, sticky="ew")
        분류3_entry_label0 = CTkLabel(분류3_entry_frame, text="( 1 단락", font=(self.default_font, 12, "bold"), text_color="#FEF9E0")
        분류3_entry_label0.grid(row=0, column=0, padx=(30, 0), pady=2, sticky="w")
        self.분류3_entry = CTkEntry(분류3_entry_frame, width=40, height=18, text_color="black", fg_color="white")
        self.분류3_entry.grid(row=0, column=1, padx=(5), pady=2, sticky="w")
        self.분류3_entry.insert(0, '150')
        분류3_entry_label = CTkLabel(분류3_entry_frame, text="단어 이하로 )", font=(self.default_font, 12), text_color="#FEF9E0")
        분류3_entry_label.grid(row=0, column=2, padx=(0, 0), pady=2, sticky="w")

        ######
        지문번호규칙_label = CTkLabel(self.right_frame_middle, text="◼ 지문 번호 인식 규칙", font=(self.default_font, 15, "bold"), text_color="#FEF9E0", fg_color="transparent")
        지문번호규칙_label.grid(row=5, column=0, padx=(10, 0), pady=(10, 3), sticky="w")


    
        지문번호규칙_frame = CTkFrame(self.right_frame_middle, fg_color="transparent", width=210)
        지문번호규칙_frame.grid(row=6, column=0, padx=0, pady=0, sticky="w")
        지문번호규칙_frame.grid_columnconfigure(0, weight=1)
        지문번호규칙_frame.grid_columnconfigure(1, weight=1)



        def buttonon():
            self.checkbox_1.configure(state="normal", fg_color="#DDA15C", border_color="#FEF9E0")
            self.checkbox_2.configure(state="normal", fg_color="#DDA15C", border_color="#FEF9E0")
            self.checkbox_3.configure(state="normal", fg_color="#DDA15C", border_color="#FEF9E0")
            self.지문번호규칙_entry.configure(state="normal", fg_color="white")
            self.지문번호수동지정_entry.configure(state="disabled", fg_color="gray")

        def buttonoff():
            self.checkbox_1.configure(state="disabled", fg_color="gray", border_color="gray")
            self.checkbox_2.configure(state="disabled", fg_color="gray", border_color="gray")
            self.checkbox_3.configure(state="disabled", fg_color="gray", border_color="gray")
            self.지문번호규칙_entry.configure(state="disabled", fg_color="gray")
            self.지문번호수동지정_entry.configure(state="normal", fg_color="white", text_color="gray")


        self.지문번호규칙radio_value = tk.IntVar(value=1)  # Set default selection to the first radio button
        self.지문번호규칙자동_radio = CTkRadioButton(지문번호규칙_frame, text="자동 인식", font=(self.default_font, 12), width=80, text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C", radiobutton_width=20, radiobutton_height=20, variable=self.지문번호규칙radio_value, value=1, command=buttonon)
        self.지문번호규칙자동_radio.grid(row=0, column=0, padx=(10, 0), pady=2, sticky="w")

        self.지문번호규칙수동_radio = CTkRadioButton(지문번호규칙_frame, text="수동 인식", font=(self.default_font, 12), width=70, text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C", radiobutton_width=20, radiobutton_height=20, variable=self.지문번호규칙radio_value, value=2, command=buttonoff)
        self.지문번호규칙수동_radio.grid(row=0, column=1, padx=(10, 0), pady=2, sticky="w")


        지문번호규칙_frame_left = CTkFrame(지문번호규칙_frame, fg_color="transparent", width=105)
        지문번호규칙_frame_left.grid(row=1, column=0, padx=0, pady=0, sticky="w")
        지문번호규칙_frame_right = CTkFrame(지문번호규칙_frame, fg_color="transparent", width=80)
        지문번호규칙_frame_right.grid(row=1, column=1, padx=0, pady=0, sticky="w")





        지문번호규칙_entry_frame = CTkFrame(지문번호규칙_frame_left, fg_color="transparent")
        지문번호규칙_entry_frame.grid(row=0, column=0, padx=0, pady=2, sticky="ew")
        self.지문번호규칙_entry = CTkEntry(지문번호규칙_entry_frame, width=30, height=18, text_color="black", fg_color="white")
        self.지문번호규칙_entry.grid(row=0, column=0, padx=(15, 5), pady=0, sticky="w")
        self.지문번호규칙_entry.insert(0, '10')
        지문번호규칙_entry_label = CTkLabel(지문번호규칙_entry_frame, text="단어 미만임", font=(self.default_font, 11), text_color="#FEF9E0")
        지문번호규칙_entry_label.grid(row=0, column=1, padx=(0, 0), pady=0, sticky="w")



        self.checkbox_1 = CTkCheckBox(지문번호규칙_frame_left, text=".로 끝나지 않음", width=20, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 11), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.checkbox_1.grid(row=1, column=0, padx=(15, 0), pady=2, sticky='w')

        self.checkbox_2 = CTkCheckBox(지문번호규칙_frame_left, text="!로 끝나지 않음", width=20, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 11), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.checkbox_2.grid(row=2, column=0, padx=(15, 0), pady=2, sticky='w')

        self.checkbox_3 = CTkCheckBox(지문번호규칙_frame_left, text="?로 끝나지 않음", width=20, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 11), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.checkbox_3.grid(row=3, column=0, padx=(15, 0), pady=2, sticky='w')

        self.checkbox_1.select()
        self.checkbox_2.select()
        self.checkbox_3.select()



        """
        self.지문번호수동지정 = CTkCheckBox(지문번호규칙_frame, text="지문번호 수동 지정", width=20, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.지문번호수동지정.grid(row=1, column=1, padx=(30, 0), pady=2, sticky='w')
        """        
       

        self.지문번호수동지정_entry = CTkTextbox(지문번호규칙_frame_right, font=(self.default_font, 11), text_color="black", fg_color="white", width=65, height=100)
        self.지문번호수동지정_entry.grid(row=0, column=0, padx=(15, 10), pady=(5, 0), sticky="w")

        placeholder_text_수동 = "대소문자 구분, 여러개 입력시 엔터로 구분"
        self.지문번호수동지정_entry.insert("0.0", placeholder_text_수동)
        self.지문번호수동지정_entry.configure(text_color='black', font=(self.default_font, 11))
        self.지문번호수동지정_entry.configure(state="disabled", fg_color="gray")

        def on_entry_click2(event):
            if self.지문번호수동지정_entry.get("1.0", "end-1c").strip() == placeholder_text_수동:
                self.지문번호수동지정_entry.delete("1.0", "end")
                self.지문번호수동지정_entry.configure(text_color=('black'), font=(self.default_font, 11))  # Change text color back to normal

        def on_entry_focusout2(event):
            if not self.지문번호수동지정_entry.get("1.0", "end-1c").strip():
                self.지문번호수동지정_entry.insert("1.0", placeholder_text_수동)
                self.지문번호수동지정_entry.configure(text_color='grey', font=(self.default_font, 11))  # Change text color back to normal

        self.지문번호수동지정_entry.bind('<FocusIn>', on_entry_click2)
        self.지문번호수동지정_entry.bind('<FocusOut>', on_entry_focusout2)

        
        ######
        한글삭제옵션_label = CTkLabel(self.right_frame_bottom, text="◼ 한글 삭제 옵션", font=(self.default_font, 15, "bold"), text_color="#FEF9E0", fg_color="transparent")
        한글삭제옵션_label.grid(row=0, column=0, padx=(10, 0), pady=(5, 3), sticky="w")


        self.한글삭제checkbox = CTkCheckBox(self.right_frame_bottom, text="한글 삭제", width=20, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.한글삭제checkbox.grid(row=1, column=0, padx=(25, 0), pady=2, sticky='w')
        self.한글삭제checkbox.deselect()

        삭제예외_label = CTkLabel(self.right_frame_bottom, text="다음 문자로 시작하면 삭제 예외", font=(self.default_font, 12), text_color="#FEF9E0", fg_color="transparent")
        삭제예외_label.grid(row=2, column=0, padx=(25, 0), pady=1, sticky="w")

        self.삭제예외_entry = CTkTextbox(self.right_frame_bottom, font=(self.default_font, 11), text_color="black", fg_color="white", width=145, height=50)
        self.삭제예외_entry.grid(row=3, column=0, padx=(15, 15), pady=1, sticky="ew")
        
        
        placeholder_text_a = "여러개 입력시 엔터로 구분"
        self.삭제예외_entry.insert("0.0", placeholder_text_a)
        self.삭제예외_entry.configure(text_color='grey', font=(self.default_font, 11))
        # Event handlers for the Text widget
        def on_text_click(event):
            if self.삭제예외_entry.get("1.0", "end-1c").strip() == placeholder_text_a:
                self.삭제예외_entry.delete("1.0", "end")
                self.삭제예외_entry.configure(text_color='black', font=(self.default_font, 11))

        def on_text_focusout(event):
            if not self.삭제예외_entry.get("1.0", "end-1c").strip():
                self.삭제예외_entry.insert("1.0", placeholder_text_a)
                self.삭제예외_entry.configure(text_color='grey', font=(self.default_font, 11))

        self.삭제예외_entry.bind('<FocusIn>', on_text_click)
        self.삭제예외_entry.bind('<FocusOut>', on_text_focusout)
        

        #####
        추가삭제옵션_label = CTkLabel(self.right_frame_bottom, text="◼ 추가 삭제 옵션", font=(self.default_font, 15, "bold"), text_color="#FEF9E0", fg_color="transparent")
        추가삭제옵션_label.grid(row=4, column=0, padx=(10, 0), pady=(10, 0), sticky="w")


        추가삭제옵션_label2 = CTkLabel(self.right_frame_bottom, text="다음 문자로 시작하면 삭제", font=(self.default_font, 12), text_color="#FEF9E0", fg_color="transparent")
        추가삭제옵션_label2.grid(row=5, column=0, padx=(25, 0), pady=0, sticky="w")

     
        self.추가삭제옵션_entry = CTkTextbox(self.right_frame_bottom, font=(self.default_font, 11), text_color="black", fg_color="white", width=145, height=50)
        self.추가삭제옵션_entry.grid(row=6, column=0, padx=(15, 15), pady=(1, 10), sticky="ew")
        #self.추가삭제옵션_entry.insert("0.0", "[\n{\n*") 
       




        """
        image_image_1 = PhotoImage(file=self.relative_to_assets("image_1.png"))
        self.image_1 = self.canvas.create_image(1941.0, 4657.0, image=image_image_1)

        image_image_2 = PhotoImage(file=self.relative_to_assets("image_2.png"))
        self.image_2 = self.canvas.create_image(1044.0, 131.0, image=image_image_2)

        image_image_3 = PhotoImage(file=self.relative_to_assets("image_3.png"))
        self.image_3 = self.canvas.create_image(1043.0, 271.0, image=image_image_3)

        image_image_4 = PhotoImage(file=self.relative_to_assets("image_4.png"))
        self.image_4 = self.canvas.create_image(1043.0, 431.0, image=image_image_4)

        image_image_5 = PhotoImage(file=self.relative_to_assets("image_5.png"))
        self.image_5 = self.canvas.create_image(1054.0, 484.0, image=image_image_5)
        """

        ##################### 탭뷰2

        self.right_frame2 = CTkFrame(self.tabview.tab("고급 편집기"), width=230, fg_color="transparent", corner_radius=0)
        self.right_frame2.grid(row=0, column=0, sticky="nsew", padx=0, pady=0)
        #self.right_frame2.grid_propagate(False)
        #self.right_frame2.grid_rowconfigure(13, weight=1)
        self.right_frame2.grid_columnconfigure(0, weight=1)
        
 


     

    

        ######
        self.right_frame_01 = CTkFrame(self.right_frame2, width=220, fg_color="transparent")
        self.right_frame_01.grid(row=0, column=0, sticky="ew", padx=0, pady=0)
        self.right_frame_01.grid_columnconfigure(0, weight=1)       

        삭제기_label = CTkLabel(self.right_frame_01, text="◼ 삭제기", font=(self.default_font, 15, "bold"), text_color="#FEF9E0", fg_color="transparent")
        삭제기_label.grid(row=0, column=0, padx=(12, 0), pady=3, sticky='w')
        삭제기_button = CTkButton(self.right_frame_01, text="삭제 실행", width=60, height=25, font=(self.default_font, 12, "bold"), text_color="black", border_color="#FEF9E0", hover_color="#DDA15C", fg_color="#FEF9E0", command=self.pre_execute_delete)
        삭제기_button.grid(row=0, column=1, padx=(12, 12), pady=3, sticky='w')



        self.right_frame_02 = CTkFrame(self.right_frame2, width=220, fg_color="transparent")
        self.right_frame_02.grid(row=1, column=0, sticky="ew", padx=0, pady=0)
        self.right_frame_02.grid_columnconfigure(0, weight=1)   
        
        self.check01 = CTkCheckBox(self.right_frame_02, text="공백 2칸", width=20, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.check01.grid(row=0, column=0, padx=(15, 0), pady=3, sticky='w')
        self.check02 = CTkCheckBox(self.right_frame_02, text="빈 줄(enter)", width=20, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.check02.grid(row=1, column=0, padx=(15, 0), pady=3, sticky='w')
        self.check02_1 = CTkCheckBox(self.right_frame_02, text="단락 앞/뒤 공백", width=20, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.check02_1.grid(row=2, column=0, padx=(15, 0), pady=3, sticky='w')




        self.right_frame_03 = CTkFrame(self.right_frame2, width=220, fg_color="transparent")
        self.right_frame_03.grid(row=2, column=0, sticky="ew", padx=0, pady=0)
        self.right_frame_03.grid_columnconfigure(0, weight=1)

        
        self.check03 = CTkCheckBox(self.right_frame_03, text="한글 문장", width=20, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.check03.grid(row=0, column=0, padx=(15, 0), pady=3, sticky='w')
        self.check03_button = CTkButton(self.right_frame_03, text="옵션", width=35, height=20, font=(self.default_font, 11), text_color="black", border_color="#FEF9E0", hover_color="#DDA15C", fg_color="#EFF3F0", command=self.create_options_popup_한글삭제)
        self.check03_button.grid(row=0, column=1, padx=(12), pady=3, sticky='w')        


        self.check04 = CTkCheckBox(self.right_frame_03, text="영어 문장", width=20, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.check04.grid(row=1, column=0, padx=(15, 0), pady=3, sticky='w')
        self.check04_button = CTkButton(self.right_frame_03, text="옵션", width=35, height=20, font=(self.default_font, 11), text_color="black", border_color="#FEF9E0", hover_color="#DDA15C", fg_color="#EFF3F0", command=self.create_options_popup_영어삭제)
        self.check04_button.grid(row=1, column=1, padx=(12), pady=3, sticky='w')
    
        self.check05 = CTkCheckBox(self.right_frame_03, text="숫자/문자", width=20, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.check05.grid(row=2, column=0, padx=(15, 0), pady=3, sticky='w')
        self.check05_button = CTkButton(self.right_frame_03, text="옵션", width=35, height=20, font=(self.default_font, 11), text_color="black", border_color="#FEF9E0", hover_color="#DDA15C", fg_color="#EFF3F0", command=self.create_options_popup_숫자)
        self.check05_button.grid(row=2, column=1, padx=(12), pady=3, sticky='w')




        self.right_frame_04 = CTkFrame(self.right_frame2, width=220, fg_color="transparent")
        self.right_frame_04.grid(row=3, column=0, sticky="ew", padx=0, pady=0)
        self.right_frame_04.grid_columnconfigure(0, weight=1)

        self.check06 = CTkCheckBox(self.right_frame_04, text="특정 문자열", width=20, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.check06.grid(row=0, column=0, padx=(15, 0), pady=3, sticky='w')
        self.check06_button = CTkButton(self.right_frame_04, text="옵션", width=35, height=20, font=(self.default_font, 11), text_color="black", border_color="#FEF9E0", hover_color="#DDA15C", fg_color="#EFF3F0", command=self.create_options_popup_특정문자열)
        self.check06_button.grid(row=0, column=1, padx=(12), pady=3, sticky='w')


        self.check06_textbox = CTkTextbox(self.right_frame_04, font=(self.default_font, 11), text_color="black", fg_color="white", width=145, height=40)
        self.check06_textbox.grid(row=1, column=0, columnspan=2, padx=(35, 12), pady=5, sticky='ew')
        self.check06_textbox.bind('<KeyRelease>', self.on_textbox_change)




        placeholder_text = "여러 항목은 엔터로 구분"
        self.check06_textbox.insert("0.0", placeholder_text)
        self.check06_textbox.configure(text_color='grey', font=(self.default_font, 11))

        def on_entry_click(event):
            if self.check06_textbox.get("1.0", "end-1c").strip() == placeholder_text:
                self.check06_textbox.delete("1.0", "end")
                self.check06_textbox.configure(text_color=('black'), font=(self.default_font, 11))  # Change text color back to normal
        def on_entry_focusout(event):
            if not self.check06_textbox.get("1.0", "end-1c").strip():
                self.check06_textbox.insert("1.0", placeholder_text)
                self.check06_textbox.configure(text_color='grey', font=(self.default_font, 11))  # Change text color back to normal
        self.check06_textbox.bind('<FocusIn>', on_entry_click)
        self.check06_textbox.bind('<FocusOut>', on_entry_focusout)


        self.right_frame_05 = CTkFrame(self.right_frame2, width=220, fg_color="transparent")
        self.right_frame_05.grid(row=4, column=0, sticky="ew", padx=0, pady=0)
        self.right_frame_05.grid_columnconfigure(0, weight=1)

        self.check07 = CTkCheckBox(self.right_frame_05, text="단어 n개 이상/이하", width=100, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.check07.grid(row=0, column=0, padx=(15, 0), pady=3, sticky='w')
        self.check07_button = CTkButton(self.right_frame_05, text="옵션", width=35, height=20, font=(self.default_font, 11), text_color="black", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#EFF3F0", command=self.create_options_popup_단어이상이하)
        self.check07_button.grid(row=0, column=1, padx=(12), pady=3, sticky='w')

        self.check08 = CTkCheckBox(self.right_frame_05, text="수능 독해 발문", width=100, height=20, checkbox_width=20, checkbox_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C")
        self.check08.grid(row=1, column=0, padx=(15, 0), pady=3, sticky='w')
        self.check08_button = CTkButton(self.right_frame_05, text="옵션", width=35, height=20, font=(self.default_font, 11), text_color="black", border_color="#FEF9E0", hover_color="#DDA15C", fg_color="#EFF3F0", command=self.create_options_popup_발문)
        self.check08_button.grid(row=1, column=1, padx=(12), pady=3, sticky='w')



        self.right_frame_06 = CTkFrame(self.right_frame2, width=220, fg_color="transparent")
        self.right_frame_06.grid(row=5, column=0, sticky="ew", padx=0, pady=3)
        self.right_frame_06.grid_columnconfigure(0, weight=1)
        self.right_frame_06.grid_columnconfigure(1, weight=4)



        self.delete_exception_label = CTkLabel(self.right_frame_06, text="삭제\n 예외:", text_color="#FEF9E0", fg_color="transparent", font=(self.default_font, 12))
        self.delete_exception_label.grid(row=0, column=0, padx=(5, 0), pady=0, sticky='w')
        self.delete_exception_textbox = CTkTextbox(self.right_frame_06, font=(self.default_font, 11), text_color="black", fg_color="white", width=130, height=40)
        self.delete_exception_textbox.grid(row=0, column=1, padx=(5, 12), pady=0, sticky='ew')

        placeholder_text2 = "이 문자로 시작시 삭제 예외\n(여러 항목은 엔터로 구분)"
        self.delete_exception_textbox.insert("0.0", placeholder_text2)
        self.delete_exception_textbox.configure(text_color='grey', font=(self.default_font, 11))

        def on_entry_click(event):
            if self.delete_exception_textbox.get("1.0", "end-1c").strip() == placeholder_text2:
                self.delete_exception_textbox.delete("1.0", "end")
                self.delete_exception_textbox.configure(text_color=('black'), font=(self.default_font, 11))  # Change text color back to normal
        def on_entry_focusout(event):
            if not self.delete_exception_textbox.get("1.0", "end-1c").strip():
                self.delete_exception_textbox.insert("1.0", placeholder_text2)
                self.delete_exception_textbox.configure(text_color='grey', font=(self.default_font, 11))  # Change text color back to normal
        self.delete_exception_textbox.bind('<FocusIn>', on_entry_click)
        self.delete_exception_textbox.bind('<FocusOut>', on_entry_focusout)





        ######
        
        self.right_frame_편집기1 = CTkFrame(self.right_frame2, width=220, fg_color="transparent", corner_radius=7)
        self.right_frame_편집기1.grid(row=6, column=0, pady=(15, 0), sticky="ew")
        self.right_frame_편집기1.grid_columnconfigure(0, weight=1)
        편집기_label = CTkLabel(self.right_frame_편집기1, text="◼ 편집기", font=(self.default_font, 15, "bold"), text_color="#FEF9E0", fg_color="transparent")
        편집기_label.grid(row=0, column=0, padx=(10, 0), pady=(5, 4), sticky="w")
        편집기_button = CTkButton(self.right_frame_편집기1, text="편집 실행", width=60, height=25, font=(self.default_font, 12, "bold"), text_color="black", border_color="#FEF9E0", hover_color="#DDA15C", fg_color="#FEF9E0", command=self.pre_execute_edit)
        편집기_button.grid(row=0, column=1, padx=(12, 12), pady=3, sticky='w')


        self.right_frame_편집기2 = CTkFrame(self.right_frame2, width=220, fg_color="transparent", corner_radius=7)
        self.right_frame_편집기2.grid(row=7, column=0, pady=(2), sticky="ew")
        self.right_frame_편집기2.grid_columnconfigure(0, weight=1)
        self.편집기_radio_var = IntVar(value=0)
        self.edit_check1 = CTkRadioButton(self.right_frame_편집기2, text="문장 쪼개기", width=50, height=20, radiobutton_width=20, radiobutton_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C", variable=self.편집기_radio_var, value=1)
        self.edit_check1.grid(row=0, column=0, padx=(15, 0), pady=0, sticky='w')



        self.right_frame_편집기3 = CTkFrame(self.right_frame2, width=220, fg_color="transparent", corner_radius=7)
        self.right_frame_편집기3.grid(row=8, column=0, pady=(2), sticky="ew")
        self.right_frame_편집기3.grid_columnconfigure(0, weight=1)
        self.edit_check2 = CTkRadioButton(self.right_frame_편집기3, text="문장 합치기", width=50, height=20, radiobutton_width=20, radiobutton_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C", variable=self.편집기_radio_var, value=2)
        self.edit_check2.grid(row=0, column=0, padx=(15, 0), pady=0, sticky='w')
        self.edit_check2_옵션 = CTkButton(self.right_frame_편집기3, text="옵션", width=35, height=20, font=(self.default_font, 11), text_color="black", border_color="#FEF9E0", hover_color="#DDA15C", fg_color="#EFF3F0", command=self.create_options_popup_문장합치기)
        self.edit_check2_옵션.grid(row=0, column=1, padx=(12), pady=0, sticky='w')



        self.right_frame_편집기4 = CTkFrame(self.right_frame2, width=220, fg_color="transparent", corner_radius=7)
        self.right_frame_편집기4.grid(row=9, column=0, pady=(2), sticky="ew")
        self.right_frame_편집기4.grid_columnconfigure(0, weight=1)
        self.edit_check3 = CTkRadioButton(self.right_frame_편집기4, text="정답 채우기", width=50, height=20, radiobutton_width=20, radiobutton_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C", variable=self.편집기_radio_var, value=3)
        #self.edit_check3 = CTkRadioButton(self.right_frame_편집기4, text="정답 채우기", width=50, height=20, radiobutton_width=20, radiobutton_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C", variable=self.편집기_radio_var, value=3, command=self.create_options_popup_정답채우기)
        self.edit_check3.grid(row=0, column=0, padx=(15, 0), pady=0, sticky='w')
        self.edit_check3_옵션 = CTkButton(self.right_frame_편집기4, text="옵션", width=35, height=20, font=(self.default_font, 11), text_color="black", border_color="#FEF9E0", hover_color="#DDA15C", fg_color="#EFF3F0", command=self.create_options_popup_정답채우기)
        self.edit_check3_옵션.grid(row=0, column=1, padx=(12), pady=0, sticky='w')





        ################
        self.right_frame_편집기6 = CTkFrame(self.right_frame2, width=220, fg_color="transparent", corner_radius=7)
        self.right_frame_편집기6.grid(row=10, column=0, pady=2, sticky="ew")
        self.right_frame_편집기6.grid_columnconfigure(0, weight=1)
        self.edit_check5 = CTkRadioButton(self.right_frame_편집기6, text="스크램블", width=50, height=20, radiobutton_width=20, radiobutton_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C", variable=self.편집기_radio_var, value=5)
        self.edit_check5.grid(row=0, column=0, padx=(15, 0), pady=0, sticky='w')



        ################
        def 한영교차준비():
            교차준비문구 = """(A)\n\n(B)\n\n"""
            self.edit_box.configure(text_color=('black'), font=(self.default_font, 12))  # Change text color back to normal
            
            data = self.edit_box.get("1.0", "end-1c")
            data = data.replace("Write here.", "")
            self.edit_box.delete("1.0", "end")
            
            data = 교차준비문구 + data
            self.edit_box.insert("1.0", data)
            


        self.right_frame_편집기7 = CTkFrame(self.right_frame2, width=220, fg_color="transparent", corner_radius=7)
        self.right_frame_편집기7.grid(row=11, column=0, pady=2, sticky="ew")
        self.right_frame_편집기7.grid_columnconfigure(0, weight=1)
        self.edit_check6 = CTkRadioButton(self.right_frame_편집기7, text="한영교차", width=50, height=20, radiobutton_width=20, radiobutton_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C", variable=self.편집기_radio_var, value=6, command=한영교차준비)
        self.edit_check6.grid(row=0, column=0, padx=(15, 0), pady=0, sticky='w')
        
        ##################

        self.right_frame_편집기8 = CTkFrame(self.right_frame2, width=220, fg_color="transparent", corner_radius=7)
        self.right_frame_편집기8.grid(row=12, column=0, pady=2, sticky="ew")
        self.right_frame_편집기8.grid_columnconfigure(0, weight=1)
        self.edit_check7 = CTkRadioButton(self.right_frame_편집기8, text="단어 뜻 분리", width=50, height=20, radiobutton_width=20, radiobutton_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C", variable=self.편집기_radio_var, value=7)
        self.edit_check7.grid(row=0, column=0, padx=(15, 0), pady=0, sticky='w')


        #################

        self.right_frame_편집기5 = CTkFrame(self.right_frame2, width=220, fg_color="transparent", corner_radius=7)
        self.right_frame_편집기5.grid(row=13, column=0, pady=(0), sticky="ew")
        self.right_frame_편집기5.grid_columnconfigure(0, weight=1)
        self.edit_check4 = CTkRadioButton(self.right_frame_편집기5, text="찾아 바꾸기", width=50, height=20, radiobutton_width=20, radiobutton_height=20, font=(self.default_font, 12), text_color="#FEF9E0", border_color="#FEF9E0", hover_color="#BB6C25", fg_color="#DDA15C", variable=self.편집기_radio_var, value=4)
        self.edit_check4.grid(row=0, column=0, padx=(15, 0), pady=2, sticky='w')


        self.right_frame_찾아바꾸기 = CTkFrame(self.right_frame2, width=220, fg_color="transparent", corner_radius=0)
        self.right_frame_찾아바꾸기.grid(row=14, column=0, pady=4, sticky="new")
        self.right_frame_찾아바꾸기.grid_columnconfigure(0, weight=1)
        self.right_frame_찾아바꾸기.grid_columnconfigure(2, weight=1)

        def 찾아바꾸기자동선택(event=None):
            # If there's any text in the textbox, select the checkbox
            if self.check06_textbox.get("1.0", "end-1c").strip():
                self.edit_check4.select()
            else:
                self.edit_check4.deselect()

        self.find_textbox = CTkTextbox(self.right_frame_찾아바꾸기, font=(None, 10), text_color="black", fg_color="white", width=60, height=40)
        self.find_textbox.grid(row=0, column=0, padx=5, pady=0, sticky='ew')
        self.find_textbox.bind('<KeyRelease>', 찾아바꾸기자동선택)

        arrow_label = CTkLabel(self.right_frame_찾아바꾸기, text="→", font=(self.default_font, 15, "bold"), text_color="#FEF9E0", fg_color="transparent")
        arrow_label.grid(row=0, column=1, padx=0, pady=0)
        self.replace_textbox = CTkTextbox(self.right_frame_찾아바꾸기, font=(None, 10), text_color="black", fg_color="white", width=60, height=40)
        self.replace_textbox.grid(row=0, column=2, padx=5, pady=0, sticky='ew')




    def execute_sorting(sefl):
        messagebox.showinfo("알림", "준비중입니다.")

    def pre_execute_delete(self):
        if self.check01.get() == 0 and self.check02.get() == 0 and self.check02_1.get() == 0 and self.check02.get() == 0 and self.check03.get() == 0 and self.check04.get() == 0 and self.check05.get() == 0 and self.check06.get() == 0 and self.check07.get() == 0 and self.check08.get() == 0:
            messagebox.showerror("오류", "삭제할 유형을 선택하세요.")            
        elif self.edit_box.get("1.0", "end-1c").strip() == "Write here." or self.edit_box.get("1.0", "end-1c").strip() == "":
            messagebox.showerror("오류", "지문을 입력하세요")
        else:
            self.execute_delete()

    def execute_delete(self):
        # Fetch the current content of the edit_box, excluding the automatic newline at the end
        original_text = self.edit_box.get("1.0", "end-1c")

        # 탭 삭제 → 공백 2칸 삭제
        if self.check01.get() == 1:
            original_text = original_text.replace("  ", " ")  # Remove all tab characters

        # 엔터 삭제
        if self.check02.get() == 1:
            original_text = re.sub(r'\n{2,}', '\n', original_text)  # Replace two or more consecutive "\n" with a single "\n"


        # Fetch and split the exception content from the textbox
        exception_phrases = self.delete_exception_textbox.get("1.0", "end-1c").split('\n')
        exception_phrases = [phrase.strip() for phrase in exception_phrases if phrase.strip()]

        # Split the original text into paragraphs
        paragraphs = original_text.split('\n')
        processed_paragraphs = []
        # Identify excepted paragraphs and prepare for processing others


        for paragraph in paragraphs:
            modified_paragraph = paragraph  # Initialize with original paragraph content


            # Check for exception phrases at the start of the paragraph
            if any(paragraph.startswith(phrase) for phrase in exception_phrases):
                processed_paragraphs.append(paragraph)
                continue  # Skip further processing for this paragraph
            
            # 단락 앞뒤 공백 제거
            if self.check02_1.get() == 1:
                modified_paragraph = modified_paragraph.strip()


            # 한글 삭제
            if self.check03.get() == 1:
                # Case when we need to check if the paragraph starts with a Korean letter
                if self.한글옵션_radio_var.get() == 0:
                    if re.match(r"^\s*[\uAC00-\uD7A3]", modified_paragraph):
                        continue  # Skip this paragraph since it starts with a Korean letter

                # Case when we need to check if the paragraph contains any Korean letter
                elif self.한글옵션_radio_var.get() == 1:
                    if re.search(r"[\uAC00-\uD7A3]", modified_paragraph):
                        continue  # Skip this paragraph since it contains a Korean letter


            # 영어로 시작
            if self.check04.get() == 1:
                if (self.start_with_uppercase_var.get() == 1 and re.match(r"^\s*[A-Z]", modified_paragraph)) or \
                (self.start_with_lowercase_var.get() == 1 and re.match(r"^\s*[a-z]", modified_paragraph)):
                    continue  # Skip this paragraph


            # 숫자/문자
            if self.check05.get() == 1:
                # Define the patterns for each type of number based on checkboxes
                patterns = {
                    1: "1234567890",  # Standard numbers
                    2: "①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳㉑㉒㉓㉔㉕㉖㉗㉘㉙㉚㉛㉜㉝㉞㉟㊱㊲㊳㊴㊵",  # Enclosed numbers
                    3: "❶❷❸❹❺❻❼❽❾❿⓫⓬⓭⓮⓯⓰⓱⓲⓳⓴",
                    4: "\\(1\\)|\\(2\\)|\\(3\\)|\\(4\\)|\\(5\\)|\\(6\\)|\\(7\\)|\\(8\\)|\\(9\\)|\\(10\\)|\\(11\\)|\\(12\\)|\\(13\\)|\\(14\\)|\\(15\\)|\\(16\\)|\\(17\\)|\\(18\\)|\\(19\\)|\\(20\\)|\\( 1 \\)|\\( 2 \\)|\\( 3 \\)|\\( 4 \\)|\\( 5 \\)|\\( 6 \\)|\\( 7 \\)|\\( 8 \\)|\\( 9 \\)|\\( 10 \\)|\\( 11 \\)|\\( 12 \\)|\\( 13 \\)|\\( 14 \\)|\\( 15 \\)|\\( 16 \\)|\\( 17 \\)|\\( 18 \\)|\\( 19 \\)|\\( 20 \\)|⑴|⑵|⑶|⑷|⑸|⑹|⑺|⑻|⑼|⑽|⑾|⑿|⒀|⒁|⒂|⒃|⒄|⒅|⒆|⒇",
                    5: "\\(①\\)|\\(②\\)|\\(③\\)|\\(④\\)|\\(⑤\\)|\\(⑥\\)|\\(⑦\\)|\\(⑧\\)|\\(⑨\\)|\\(⑩\\)|\\(⑪\\)|\\(⑫\\)|\\(⑬\\)|\\(⑭\\)|\\(⑮\\)|\\(⑯\\)|\\(⑰\\)|\\(⑱\\)|\\(⑲\\)|\\(⑳\\)|\\( ① \\)|\\( ② \\)|\\( ③ \\)|\\( ④ \\)|\\( ⑤ \\)|\\( ⑥ \\)|\\( ⑦ \\)|\\( ⑧ \\)|\\( ⑨ \\)|\\( ⑩ \\)|\\( ⑪ \\)|\\( ⑫ \\)|\\( ⑬ \\)|\\( ⑭ \\)|\\( ⑮ \\)|\\( ⑯ \\)|\\( ⑰ \\)|\\( ⑱ \\)|\\( ⑲ \\)|\\( ⑳ \\)",
                    6: "ⒶⒷⒸⒹⒺⒻⒼⒽⒾⒿⓀⓁⓂⓃⓄⓅⓆⓇⓈⓉⓊⓋⓌⓍⓎⓏ",
                    7: "ⓐⓑⓒⓓⓔⓕⓖⓗⓘⓙⓚⓛⓜⓝⓞⓟⓠⓡⓢⓣⓤⓥⓦⓧⓨⓩ",
                    8: "\\(A\\)|\\(B\\)|\\(C\\)|\\(D\\)|\\(E\\)|\\(F\\)|\\(G\\)|\\(H\\)|\\(I\\)|\\(J\\)|\\(K\\)|\\(L\\)|\\(M\\)|\\(N\\)|\\(O\\)|\\(P\\)|\\(Q\\)|\\(R\\)|\\(S\\)|\\(T\\)|\\(U\\)|\\(V\\)|\\(W\\)|\\(X\\)|\\(Y\\)|\\(Z\\)|\\( A \\)|\\( B \\)|\\( C \\)|\\( D \\)|\\( E \\)|\\( F \\)|\\( G \\)|\\( H \\)|\\( I \\)|\\( J \\)|\\( K \\)|\\( L \\)|\\( M \\)|\\( N \\)|\\( O \\)|\\( P \\)|\\( Q \\)|\\( R \\)|\\( S \\)|\\( T \\)|\\( U \\)|\\( V \\)|\\( W \\)|\\( X \\)|\\( Y \\)|\\( Z \\)",
                    9: "\\(a\\)|\\(b\\)|\\(c\\)|\\(d\\)|\\(e\\)|\\(f\\)|\\(g\\)|\\(h\\)|\\(i\\)|\\(j\\)|\\(k\\)|\\(l\\)|\\(m\\)|\\(n\\)|\\(o\\)|\\(p\\)|\\(q\\)|\\(r\\)|\\(s\\)|\\(t\\)|\\(u\\)|\\(v\\)|\\(w\\)|\\(x\\)|\\(y\\)|\\(z\\)|\\( a \\)|\\( b \\)|\\( c \\)|\\( d \\)|\\( e \\)|\\( f \\)|\\( g \\)|\\( h \\)|\\( i \\)|\\( j \\)|\\( k \\)|\\( l \\)|\\( m \\)|\\( n \\)|\\( o \\)|\\( p \\)|\\( q \\)|\\( r \\)|\\( s \\)|\\( t \\)|\\( u \\)|\\( v \\)|\\( w \\)|\\( x \\)|\\( y \\)|\\( z \\)|⒜⒝⒞⒟⒠⒡⒢⒣⒤⒥⒦⒧⒨⒩⒪⒫⒬⒭⒮⒯⒰⒱⒲⒳⒴⒵",
                    10: "㉠㉡㉢㉣㉤㉥㉦㉧㉨㉩㉪㉫㉬㉭",
                    11: "㉮㉯㉰㉱㉲㉳㉴㉵㉶㉷㉸㉹㉺㉻",
                    12: "\\(ㄱ\\)|\\(ㄴ\\)|\\(ㄷ\\)|\\(ㄹ\\)|\\(ㅁ\\)|\\(ㅂ\\)|\\(ㅅ\\)|\\(ㅇ\\)|\\(ㅈ\\)|\\(ㅊ\\)|\\(ㅋ\\)|\\(ㅌ\\)|\\(ㅍ\\)|\\(ㅎ\\)|\\( ㄱ \\)|\\( ㄴ \\)|\\( ㄷ \\)|\\( ㄹ \\)|\\( ㅁ \\)|\\( ㅂ \\)|\\( ㅅ \\)|\\( ㅇ \\)|\\( ㅈ \\)|\\( ㅊ \\)|\\( ㅋ \\)|\\( ㅌ \\)|\\( ㅍ \\)|\\( ㅎ \\)|㈀㈁㈂㈃㈄㈅㈆㈇㈈㈉㈊㈋㈌㈍",
                    13: "\\(가\\)|\\(나\\)|\\(다\\)|\\(라\\)|\\(마\\)|\\(바\\)|\\(사\\)|\\(아\\)|\\(자\\)|\\(차\\)|\\(카\\)|\\(타\\)|\\(파\\)|\\(하\\)|\\( 가 \\)|\\( 나 \\)|\\( 다 \\)|\\( 라 \\)|\\( 마 \\)|\\( 바 \\)|\\( 사 \\)|\\( 아 \\)|\\( 자 \\)|\\( 차 \\)|\\( 카 \\)|\\( 타 \\)|\\( 파 \\)|\\( 하 \\)|㈎㈏㈐㈑㈒㈓㈔㈕㈖㈗㈘㈙㈚㈛"
                }

                pattern = ""
                sequence_patterns = []  # For patterns that should be matched as a whole sequence

                # Concatenate all selected patterns
                if self.start_with_number1.get() == 1:
                    pattern += patterns[1]
                if self.start_with_number2.get() == 1:
                    pattern += patterns[2]
                if self.start_with_number3.get() == 1:
                    pattern += patterns[3]
                if self.start_with_number4.get() == 1:
                    sequence_patterns.append(patterns[4])
                if self.start_with_number5.get() == 1:
                    sequence_patterns.append(patterns[5])
                if self.start_with_number6.get() == 1:
                    pattern += patterns[6]
                if self.start_with_number7.get() == 1:
                    pattern += patterns[7]
                if self.start_with_number8.get() == 1:
                    sequence_patterns.append(patterns[8])
                if self.start_with_number9.get() == 1:
                    sequence_patterns.append(patterns[9])
                if self.start_with_number10.get() == 1:
                    pattern += patterns[10]
                if self.start_with_number11.get() == 1:
                    pattern += patterns[11]
                if self.start_with_number12.get() == 1:
                    sequence_patterns.append(patterns[12])
                if self.start_with_number13.get() == 1:
                    sequence_patterns.append(patterns[13])


                # Ensure global flags are correctly placed at the start of the pattern
                if self.number_radio_var.get() == 0:
                    
                    # Option "all paragraphs that start with the chosen number"
                    regex_pattern = f"(?m)^\\s*([{pattern}]).*$(\n)?"
                elif self.number_radio_var.get() == 1:
                    if sequence_patterns:
                        # If there are sequence patterns, join them with an alternation operator
                        sequence_pattern = '|'.join(sequence_patterns)
                        if pattern:  # If there are also simple character patterns
                            regex_pattern = f"[{pattern}]|{sequence_pattern}"
                        else:  # If only sequence patterns
                            regex_pattern = sequence_pattern
                    else:  # If only simple character patterns
                        regex_pattern = f"[{pattern}]"
                
                modified_paragraph = re.sub(regex_pattern, '', modified_paragraph)
                modified_paragraph = re.sub(r'\s{2,}', ' ', modified_paragraph) #공백 2개 이상을 공백 1개로 줄이기


            # 특정 문자열
            if self.check06.get() == 1:
                # Fetch and split the content from the textbox
                specific_strings = self.check06_textbox.get("1.0", "end-1c").split('\n')
                specific_strings = [s.strip() for s in specific_strings if s.strip()]  # Remove empty strings

                # Compile a single regex pattern to match any of the specific strings at the start of a paragraph
                # No need to re.escape here since we're matching the start and not forming a character class
                pattern = r'(?mi)^(' + '|'.join(re.escape(s) for s in specific_strings) + ')'
                compiled_pattern = re.compile(pattern)

                contain_pattern = r'(' + '|'.join(re.escape(s) for s in specific_strings) + ')'
                compiled_contain_pattern = re.compile(contain_pattern, re.IGNORECASE | re.MULTILINE)


                if self.specific_string_delete_start_var.get() == 1:
                    # Do "start with" deletion logic
                    if compiled_pattern.match(modified_paragraph):
                        continue

                if self.specific_string_delete_contain_var.get() == 1:
                    # Do "contains" deletion logic
                    if compiled_contain_pattern.search(modified_paragraph):
                        continue


        
            # 단어 이상/이하
            if self.check07.get() == 1 and self.remembered_number is not None:
                word_count = len(modified_paragraph.split())
                if (self.whatnumber_radio_var.get() == 1 and word_count >= self.remembered_number) or \
                (self.whatnumber_radio_var.get() == 2 and word_count < self.remembered_number):
                    continue  # Skip this paragraph
                
                    #messagebox.showinfo("오류", "'~단어 이상/이하' 옵션에서 기준 단어수를 입력하세요.")
                    
            #수능독해발문 
            if self.check08.get() == 1:
                # Combine both sets of patterns for prefixes and phrases
                combined_patterns = self.unwanted_prefixes + self.unwanted_phrases
                combined_pattern = r'(?mi)^\s*(' + '|'.join(re.escape(p) for p in combined_patterns) + ')'
                # Check if the paragraph matches any combined pattern
                if re.match(combined_pattern, paragraph):
                    continue  # Skip this paragraph

            # Add the processed (or unmodified, if excepted) paragraph
            processed_paragraphs.append(modified_paragraph)
            
        # Join the processed paragraphs into final text
        final_text = '\n'.join(processed_paragraphs)
        
        # Clear the current content of the edit_box
        self.edit_box.delete("1.0", "end")
        # Insert the modified text back into the edit_box
        self.edit_box.insert("1.0", final_text)




    def pre_execute_edit(self):        
        if self.편집기_radio_var.get() == 0:
            messagebox.showerror("오류", "편집할 유형을 선택하세요.")
        elif self.edit_box.get("1.0", "end-1c").strip() == "Write here." or self.edit_box.get("1.0", "end-1c").strip() == "":  
            messagebox.showerror("오류", "지문을 입력하세요")
        else:
            self.execute_edit()


    def execute_edit(self):
        # Fetch the current content of the edit_box, excluding the automatic newline at the end
        original_text = self.edit_box.get("1.0", "end-1c")
        final_text = ""

        # Pre-processing replacements
        replacements = {
            # e.g. Dr. Bell → Dr• Bell   (won't split after Dr•)
            # Note: Removed measurement units (in/ft/oz/lb/ct/yr/wk/mo) from general list - handled separately below
            r'\b(Mr|Mrs|Ms|Mt|Dr|Prof|St|Ave|Blvd|Capt|Col|Gen|Lt|Pvt|Jr|Sr|Rev|Etc|Inc|Ltd|Co|esp|Fig|Op|No|Vol|Rd|Tel|Gov|Esq|Maj|Sgt|Cpl|Adm|Mdm|Fr|Bros|Dept|Assn|Univ|Inst|Corp|Rep|Sen|Ed|Fwd|Mfg|Jan|Feb|Mar|Apr|Aug|Sept|Oct|Nov|Dec|CEO|CFO|CIO|vs)\.\s+': r'\1• ',
            # Measurement units - only when preceded by digit or other abbreviation
            r'(\d+|sq|cu)\s+(in|ft|oz|lb|ct|yr|wk|mo)\.\s+': r'\1 \2• ',
            # e.g. U.S. Army → U.S• Army  (second dot becomes •)
            r'(?<=\.[A-Za-z])\.\s+': '• ',
        }
        for pattern, replacement in replacements.items():
            original_text = re.sub(pattern, replacement, original_text)

        processed_text = original_text  # Starting with original_text for further modifications

        if self.편집기_radio_var.get() == 1:
            #문장 쪼개기 (using spaCy for accurate sentence tokenization)
            # Split into sentences using spaCy
            #spacy_doc = nlp(processed_text)
            #sentences = [sent.text.strip() for sent in spacy_doc.sents]

            # Join sentences with newlines
            #processed_text = '\n'.join(sentences)

            #processed_text = processed_text.replace("•", ".")  # Restore original periods for special cases

            #final_text = processed_text

            #문장 쪼개기
            # Normalize multiple spaces to single space (CRITICAL FIX for double spaces after periods)
            processed_text = re.sub(r"([.!?][\"'']?)\s{2,}", r'\1 ', processed_text)

            replacement_patterns = [
                (". ", ".\n"), (".) ", ".)\n"), ("? ", "?\n"), ("?) ", "?)\n"), ("! ", "!\n"), ("!) ", "!)\n"), #기본
                ('." ', '."\n'), ('?" ', '?"\n'), ('!" ', '!"\n'), #큰따옴표
                (".' ", ".'\n"), ("?' ", "?'\n"), ("!' ", "!'\n"), #작은따옴표
                ('.’ ', ".’\n"), ('?’ ', "?’\n"), ('!’ ', '!’\n'), #tick
                (".” ", ".”\n"), ("?” ", "?”\n"), ("!” ", "!”\n") #double tick
            ]
            for old, new in replacement_patterns:
                processed_text = processed_text.replace(old, new)
            
            processed_text = processed_text.replace("•", ".")  # Restore original periods for special cases

            final_text = processed_text


        elif self.편집기_radio_var.get() == 2:
            #문장 합치기
            # Normalize multiple SPACES to single space (but preserve newlines!)
            # Changed from \s to [ ] to only match spaces, not newlines
            processed_text = re.sub(r"([.!?][\"'']?) {2,}", r'\1 ', processed_text)

            # Fetch titles or phrases that should trigger a new paragraph
            title_triggers = self.제목인식.split('\n')
            title_triggers = [title.strip() for title in title_triggers if title.strip()]

            # If no title triggers set, just join all lines with single space
            if not title_triggers:
                lines = processed_text.split('\n')
                lines = [line.strip() for line in lines if line.strip()]
                processed_text = ' '.join(lines)
                processed_text = re.sub(r' {2,}', ' ', processed_text).strip()
                processed_text = processed_text.replace("•", ".")
                final_text = processed_text
            else:
                # Split by newlines
                lines = processed_text.split('\n')

                # Group lines by passage
                passages = []
                current_passage_title = None
                current_passage_lines = []

                for line in lines:
                    line_stripped = line.strip()

                    # Skip empty lines
                    if not line_stripped:
                        continue

                    # Check if this line is a title (starts with trigger)
                    is_title = any(line_stripped.startswith(trigger) for trigger in title_triggers)

                    if is_title:
                        # Save previous passage if exists
                        if current_passage_title is not None:
                            # Join all lines of the passage with single space
                            passage_text = ' '.join(current_passage_lines)
                            # Collapse multiple spaces
                            passage_text = re.sub(r' {2,}', ' ', passage_text).strip()
                            passages.append(f"{current_passage_title}\n{passage_text}")

                        # Start new passage
                        current_passage_title = line_stripped
                        current_passage_lines = []
                    else:
                        # Add non-empty lines to current passage
                        current_passage_lines.append(line_stripped)

                # Don't forget the last passage
                if current_passage_title is not None:
                    passage_text = ' '.join(current_passage_lines)
                    passage_text = re.sub(r' {2,}', ' ', passage_text).strip()
                    passages.append(f"{current_passage_title}\n{passage_text}")

                # Join passages with space + newline between them
                processed_text = ' \n'.join(passages)
                processed_text = processed_text.replace("•", ".")  # Restore original periods for special cases

                final_text = processed_text

        
        elif self.편집기_radio_var.get() == 3:
            #정답 채우기

            ########################################
            original_text = self.edit_box.get("1.0", "end-1c")
            # 탭 삭제
            if self.check01.get() == 1:
                original_text = original_text.replace("  ", " ")  # Remove all tab characters
            # 엔터 삭제
            if self.check02.get() == 1:
                original_text = re.sub(r'\n{2,}', '\n', original_text)  # Replace two or more consecutive "\n" with a single "\n"
            # Fetch and split the exception content from the textbox
            exception_phrases = self.delete_exception_textbox.get("1.0", "end-1c").split('\n')
            exception_phrases = [phrase.strip() for phrase in exception_phrases if phrase.strip()]

            # Split the original text into paragraphs
            paragraphs = original_text.split('\n')
            processed_paragraphs = []
            # Identify excepted paragraphs and prepare for processing others
            for paragraph in paragraphs:
                modified_paragraph = paragraph  # Initialize with original paragraph content
                # Check for exception phrases at the start of the paragraph
                if any(paragraph.startswith(phrase) for phrase in exception_phrases):
                    processed_paragraphs.append(paragraph)
                    continue  # Skip further processing for this paragraph
                

                modified_paragraph = modified_paragraph.strip()


                
                #수능독해발문 지우기
                # Combine both sets of patterns for prefixes and phrases
                #combined_patterns = self.unwanted_prefixes + self.unwanted_phrases
                #combined_pattern = fr'(?mi)^\s*(' + '|'.join(re.escape(p) for p in combined_patterns) + ')'
                # Check if the paragraph matches any combined pattern
                #if re.match(combined_pattern, paragraph):
                #    continue  # Skip this paragraph
                

                processed_paragraphs.append(modified_paragraph)


            # Join the processed paragraphs into final text
            final_text = '\n'.join(processed_paragraphs)
            original_text = final_text
            ################### 발문삭제 끝 ###################






            #options_pattern = re.compile(r'(?m)^\n?(①|②|③|④|⑤|1|2|3|4|5)\s*(.*?)(ㅁ?)\s*$')
            options_pattern = re.compile(r'(?m)^\n?(①|②|③|④|⑤|1|2|3|4|5)\s*(.*?)(ㅁ?)\s*(?=\n|$)')

            # Pattern to find correct answer when 'ㅁ' is not used
            answer_key_pattern = re.compile(r'정답.*?([①②③④⑤12345])', re.DOTALL)
            # Define a new pattern that matches from the start of a paragraph to the '정답' indicator 
            정답라인지우기pattern = re.compile(r'^.*정답.*\n*[①②③④⑤12345].*$', re.MULTILINE)

            original_text = original_text.replace('­', '-')
            original_text = original_text.replace('[3점]', '')

            if re.search(r'\( ?①', original_text): # 삽입문제 유형 확인
                normalized_text = re.sub(r'\n{2,}', '\n', original_text)  # Clean up extra new lines
            elif re.search(r'→|>|->', original_text):  # 어법어휘 유형확인
                normalized_text = re.sub(r'\n{2,}', '\n', original_text)  # Clean up extra new lines    
            
            else: # 삽입 제외한 나머지는 선지 12345 를 각 줄로 쪼갬.
                # Normalize the text to handle different option formats
                normalized_text = re.sub(r'(①|②|③|④|⑤)', r'\n\1', original_text)
                normalized_text = re.sub(r'\n{2,}', '\n', normalized_text)  # Clean up extra new lines
                # Pattern to match options with potential correct answer marker 'ㅁ'


            # Mapping from regular numbers to circled numbers
            number_mapping = {'1': '①', '2': '②', '3': '③', '4': '④', '5': '⑤'}

            processed_sections = []



                                    

            title_triggers = self.제목인식.split('\n')

            # User-defined triggers (treat as literals)
            user_triggers = [title.strip() for title in title_triggers if title.strip()]
            escaped_user_triggers = [re.escape(t) for t in user_triggers]

            # Built-in triggers
            literal_triggers = ["[문제]"]
            pattern_triggers = [r"\d{2}-\d{2}-고\d-\d{2}"]

            # Process them separately
            escaped_literals = [re.escape(t) for t in literal_triggers]
            all_triggers = escaped_user_triggers + escaped_literals + pattern_triggers


            # Fix: Move (?m) to the very beginning, outside the capturing group
            title_pattern = r'(?m)^(?=(' + '|'.join(all_triggers) + r').*)'
            title_pattern여러문제처리 = r"(?m)(^►?(?:" + "|".join(all_triggers) + r").*?\n)"

            sections = re.split(title_pattern, normalized_text)



            # Post-process sections to remove empty strings or isolated triggers that are not part of a full title
            cleaned_sections = []
            for section in sections:
                # Remove leading/trailing whitespace
                
                stripped_section = section.strip()
                stripped_section = "►" + stripped_section

                #print("\nstripped_section")
                #print(stripped_section)
                

                # Check if the section is non-empty and not just a title trigger
                if stripped_section and not any(stripped_section == trigger.replace('\\', '') for trigger in title_triggers):
                    # If it's actual content (not just a title), add to cleaned_sections
                    cleaned_sections.append(stripped_section)






            for i, section in enumerate(cleaned_sections):

                title_search = re.search(title_pattern여러문제처리, section, re.DOTALL)     
                #print(f"여기 타이틀서치: {title_search}")


                ################ 제목 없이 한번에 여러 지문 입력되어 있는지 검사 ###############
                if not title_search and len(section) > 20: ### ['►', '►1-', '►1-1\nThomas Ediso...  이거 하나하나가 다 섹션으로 인식되어서.. 일단 글자수 20으로 집어넣어서 본문만 잡아내게 함.
                    conditions_met = 0
                    
                    if "___" in normalized_text:
                        conditions_met += 1
                    
                    if re.search(r'\( ?①', normalized_text):
                        conditions_met += 1
                    
                    if "→" in normalized_text or ">" in normalized_text or "->" in normalized_text:
                        conditions_met += 1
                    
                    if all(x in normalized_text for x in ["(A)", "(B)", "(C)"]) and "(D)" not in normalized_text:
                        conditions_met += 1
                    
                    if all(x in normalized_text for x in ["(A)", "(B)", "(C)", "(D)"]):
                        conditions_met += 1
                    
                    if conditions_met >= 2:
                        messagebox.showwarning("경고", '제목이 없는 채로 2개 이상의 문제가 인식되었습니다. 문제들이 구분될 수 있도록 각 문제마다 위에 "[문제]"(따옴표 제외)라고 적어주세요.')
                        return                
                ###########################################################################


                #print(f"section!!!!!:\n{section}")


                ####################### 장문독해 처리 ##########################################################
                if "다음 글을 읽고, 물음에 답하시오.\n(A)" in section:

                    def remove_after_question2(text):
                        # Find the index where '문제2)' starts
                        index = text.find('문제2)')

                        # Return the text up to that index
                        if index != -1:
                            return text[:index]
                        return text

                    section = remove_after_question2(section)
                    section = section.replace("문제1) ", "").replace("주어진 글 (A)에 이어질 내용을 순서에 맞게 배열한 것으로 가장 적절한 것은?\n", "").replace("다음 글을 읽고, 물음에 답하시오.\n", "")


                elif "다음 글을 읽고, 물음에 답하시오." in section:

                    def remove_between_questions(text):
                        # Find the starting and ending indices
                        start_idx = text.find('문제1)')
                        end_idx = text.find('문제2)')

                        if start_idx != -1 and end_idx != -1:
                            # Combine the text before start_idx and after end_idx
                            return text[:start_idx] + text[end_idx:]
                        return text

                    section = remove_between_questions(section)
                    section = section.replace("문제2) ", "").replace("다음 글을 읽고, 물음에 답하시오.\n", "").replace("다음 글을 읽고, 물음에 답하시오\n", "")

                ####################### 장문독해 처리 ##########################################################










                #################################################################################
                if "요약" in section or "Summary" in section:


                    if title_search:
                        # Step 1: Extract first paragraph, title, and remainder of the section
                        paragraphs = section.strip().split('\n', 1)
                        문제title = paragraphs[0]
                        section = paragraphs[1]
                        



                    def remove_A_lines(text):
                        # Split the text into lines
                        lines = text.split('\n')
                        
                        # Keep only lines that don't contain '(A)'
                        filtered_lines = [line for line in lines if '(A)' not in line]
                        
                        # Join the remaining lines back together
                        result = '\n'.join(filtered_lines)
                        
                        return result

                    section = remove_A_lines(section)


                    # Final reformatting and concatenation
                    section = section.strip() + "\n"
                    section = section.replace("►", "")
                    if title_search:
                        문제title = 문제title.replace("►", "")
                        section = 문제title + "\n" + section

                    processed_sections.append(section)


                ############ 빈칸 두개 있는 연결어 ###########################################################

                elif "빈칸 (A), (B)" in section:

                    if title_search:
                        # Step 1: Extract first paragraph (the title), and remainder
                        paragraphs = section.strip().split('\n', 1)
                        문제title = paragraphs[0]
                        section = paragraphs[1]

                    # 1) Remove any line containing "글의"
                    lines = section.split('\n')
                    lines = [line for line in lines if "글의" not in line and "윗글의" not in line and "빈칸" not in line]
                    # Rejoin for further processing
                    text = "\n".join(lines)
                    
                    # 2) Find the "정답 N" line
                    #    We'll look for something like "정답 1" or "정답: 3"
                    answer_num_match = re.search(r'정답\s*[:=]?\s*([1-5])', text)
                    if not answer_num_match:
                        # If no answer found, just store the text and move on
                        processed_sections.append(text)
                        return

                    correct_number = answer_num_match.group(1)  # e.g. '1'
                    
                    # 3) Locate the matching line that starts with ①..⑤
                    circle_map = {'1': '①', '2': '②', '3': '③', '4': '④', '5': '⑤'}
                    circle_mark = circle_map.get(correct_number, '')  # e.g. '①'
                    
                    # Regex to find lines like "① However - Thus"
                    choice_pattern = re.compile(r'^([①②③④⑤])\s+(.*)')
                    
                    chosen_line = None
                    for line in text.split('\n'):
                        m = choice_pattern.match(line.strip())
                        if m:
                            # e.g. m.group(1) = '①', m.group(2) = 'However - Thus'
                            if m.group(1) == circle_mark:
                                chosen_line = m.group(2)  # e.g. "However - Thus"
                                break
                    
                    if not chosen_line:
                        # Could not find the correct choice line
                        processed_sections.append(text)
                        return

                    # chosen_line => something like "However - Thus"
                    chosen_answers = [s.strip() for s in re.split(r'\s*-\s*', chosen_line)]
                    # Expecting exactly 2 items for (A) and (B)
                    # If you had (C) or (D) as well, you’d expand accordingly.
                    if len(chosen_answers) != 2:
                        processed_sections.append(text)
                        return

                    ansA, ansB = chosen_answers  # e.g. "However", "Thus"

                    # 4) Fill the two blanks in the passage with ansA and ansB.
                    #    The text has (A)___ or (A)_____ or so on. 
                    #    We'll do a pattern like r"\(A\)\s*_+" to capture (A) plus underscores
                    text = re.sub(r'\(A\)\s*_{3,}', ansA, text)   # (A) + 3+ underscores
                    text = re.sub(r'\(B\)\s*_{3,}', ansB, text)   # (B) + 3+ underscores

                    # 5) Remove any line containing "(A) - (B)" if it appears
                    #    Also remove lines that start with '①', '②', etc. or "정답"
                    lines = text.split('\n')
                    filtered = []
                    for line in lines:
                        stripped = line.strip()
                        if stripped.startswith('(A) - (B)'):
                            continue
                        if re.match(r'^[①②③④⑤]', stripped):
                            continue
                        if stripped.startswith('정답'):
                            continue
                        filtered.append(line)
                    
                    text = "\n".join(filtered)
                    
                    # 6) If you want to remove leftover "(A)" or "(B)" tags themselves (in case they appear),
                    #    do a simple re.sub as well:
                    text = re.sub(r'\([AB]\)', '', text)

                    # Tidy up extra slashes, brackets, etc. (if they appear)
                    # (Not strictly necessary unless your text actually contains them)
                    text = text.replace('[', '').replace(']', '').replace('/', '')
                    
                    # Clean up multiple spaces
                    text = re.sub(r' {2,}', ' ', text)
                    # Clean up blank lines
                    text = re.sub(r'\n{2,}', '\n', text).strip()

                    # Reattach title if there was one
                    section = text.strip() + "\n"
                    section = section.replace("►", "")
                    if title_search:
                        문제title = 문제title.replace("►", "")
                        section = 문제title + "\n" + section

                    # Finally add to processed_sections
                    processed_sections.append(section)




                
           

                ################## 빈칸추론 ##############################
                elif "___" in section or "      " in section.strip():
                    # 제목 없이 문제가 2개 이상 입력되어 있는지 점검
                    if not title_search and section.count(" ___") > 1 and section.count("①") > 1:
                        messagebox.showwarning("경고", '빈칸추론 문제가 2개 이상 인식되었습니다. 문제가 구분될 수 있도록 각 문제마다 위에 "[문제]"(따옴표 제외)라고 적어주세요.')
                        return
                    elif not title_search and section.count("    ") > 1 and section.count("①") > 1:
                        messagebox.showwarning("경고", '빈칸추론 문제가 2개 이상 인식되었습니다. 문제가 구분될 수 있도록 각 문제마다 위에 "[문제]"(따옴표 제외)라고 적어주세요.')
                        return

                    correct_option = None
                    answer_found = False

                    # 1) First see if there's a 'ㅁ' marked correct option
                    options = options_pattern.findall(section)
                    for number, text, marker in options:
                        if marker == 'ㅁ':
                            correct_option = text.strip()
                            answer_found = True
                            break  # Correct option found with 'ㅁ'

                    # 2) If no 'ㅁ', look for '정답'
                    if not correct_option:
                        answer_key_match = answer_key_pattern.search(section)
                        if answer_key_match:
                            correct_answer_number = answer_key_match.group(1)
                            # Convert '1' -> '①', '2' -> '②', etc.
                            correct_answer_number = number_mapping.get(correct_answer_number, correct_answer_number)
                            answer_found = True

                            # Remove lines containing '정답'
                            section = re.sub(정답라인지우기pattern, '', section)

                            # Find the matching option text
                            for number, text, _ in options:
                                mapped_number = number_mapping.get(number, number)
                                if mapped_number == correct_answer_number:
                                    correct_option = text.strip()
                                    break

                    # 3) If we found a correct option, remove all option lines from the text
                    if correct_option:
                        section = re.sub(options_pattern, '', section)

                        # --- VOWEL LOGIC FOR a(n) + underscores ---
                        # We only handle underscores here. If you have 4 spaces, do your existing approach as well.
                        # Let's unify everything: if "___" is present, do the advanced replacement with a(n).

                        if "___" in section:
                            # We'll use a regex that captures "a(n) " (optionally) + 3 or more underscores.
                            # Then, in a callback, we decide whether to produce "an X" vs "a X" vs just "X".
                            def replace_underscore(m):
                                # m.group(1) is 'a(n) ' if present, else None
                                optional_a_n = m.group(1)
                                # correct_option is what we want to fill in
                                if optional_a_n is not None:
                                    # Check if correct_option starts with a vowel
                                    first_char = correct_option[0].lower() if correct_option else ''
                                    if first_char in 'aeiou':
                                        return "an " + correct_option
                                    else:
                                        return "a " + correct_option
                                else:
                                    # No "a(n) " prefix, just return the correct_option
                                    return correct_option

                            # Pattern: group(1) is the optional a(n) + whitespace
                            # group(2) are the underscores
                            pattern = re.compile(r'(a\(n\)\s+)?(_{3,})')
                            # Replace the first occurrence. If you want to replace *all* underscores, remove count=1
                            section = pattern.sub(replace_underscore, section, count=1)

                        elif "    " in section:
                            # Pattern that looks for optional "a(n) " followed by 4+ spaces
                            space_pattern = re.compile(r'(a\(n\)\s+)?(\s{4,})')
                            
                            def replace_spaces(m):
                                # m.group(1) is 'a(n) ' if present, else None
                                optional_a_n = m.group(1)
                                
                                if optional_a_n is not None:
                                    # Check if correct_option starts with a vowel
                                    first_char = correct_option[0].lower() if correct_option else ''
                                    if first_char in 'aeiou':
                                        return "an " + correct_option
                                    else:
                                        return "a " + correct_option
                                else:
                                    # No "a(n) " prefix, just return correct_option with a space before it
                                    return " " + correct_option
                            
                            # Replace the first occurrence
                            section = space_pattern.sub(replace_spaces, section, count=1)

                    # 4) If we never found an answer, show a warning at the last section
                    if (i == len(sections) - 1) and not answer_found:
                        section += "\n\n정답이 발견되지 않았습니다. 각 문제마다 지문 아래에 다음과 같이 정답을 적어주세요\n정답: 3\n\n"

                    # Finish up
                    section = section.strip() + "\n"
                    section = section.replace("►", "")
                    processed_sections.append(section)


                                        
                    
                #########  삽입 기존코드
                elif re.search(r'\( ?①', section):

                    if not title_search and (section.count("(①)") > 1 or section.count("( ① )") > 1):
                        messagebox.showwarning("경고", '삽입 문제가 2개 이상 인식되었습니다. 문제가 구분될 수 있도록 각 문제마다 위에 "[문제]"(따옴표 제외)라고 적어주세요.')
                        return

                    section = section.replace("글의 흐름으로 보아, 주어진 문장이 들어가기에 가장 적절한 곳을 고르시오.\n", "").replace("글의 흐름으로 보아, 주어진 문장이 들어가기에 가장 적절한 곳을 고르시오\n", "").replace("글의 흐름으로 보아, 주어진 문장이 들어가기에 가장 적절한 곳은?\n", "")

                    if '[[' in section:
                        section = section.replace("[[", "").replace("]]", "")



                    if title_search:
                        # Extract first paragraph
                        paragraphs = section.strip().split('\n', 2)

                        삽입title = paragraphs[0]
                        first_paragraph = paragraphs[1]
                        section = paragraphs[2]  # Remainder of the section after removing the first paragraph

                    else:
                        # Extract first paragraph
                        paragraphs = section.strip().split('\n', 1)

                        #삽입title = paragraphs[0]
                        first_paragraph = paragraphs[0]
                        section = paragraphs[1]  # Remainder of the section after removing the first paragraph


                    # Find the correct option by searching for the circle number before 'ㅁ'
                    answer_number = None
                    match = re.search(r'\(\s*(①|②|③|④|⑤)\s*ㅁ\s*\)', section)

                    if match:
                        answer_number = match.group(1)  # This will correctly capture just the number

                        # Construct the answer pattern with regex to handle optional spaces
                        answer_pattern = re.compile(r'\(\s*' + re.escape(answer_number) + r'\s*ㅁ\s*\)')
                        # Use re.sub to replace the pattern with the first paragraph
                        section = answer_pattern.sub(first_paragraph, section)

                        # Remove all other circle number markers
                        section = re.sub(r'\( ?① ?\) |\( ?② ?\) |\( ?③ ?\) |\( ?④ ?\) |\( ?⑤ ?\)', '', section)


                    else:
                        answer_number = answer_key_pattern.search(section)
                        if answer_number:
                            answer_number = answer_number.group(1)
                            answer_number = number_mapping.get(answer_number, answer_number)  # Map to circled if needed
                            # Remove lines containing '정답' indication
                            section = re.sub(정답라인지우기pattern, '', section)

                            # Construct the answer pattern with regex to handle optional spaces
                            answer_pattern = re.compile(r'\(\s*' + re.escape(answer_number) + r'\s*\)')
                            # Use re.sub to replace the pattern with the first paragraph
                            section = answer_pattern.sub(first_paragraph, section)

                            # Remove all other circle number markers
                            section = re.sub(r'\( ?① ?\) |\( ?② ?\) |\( ?③ ?\) |\( ?④ ?\) |\( ?⑤ ?\)', '', section)
                        else:
                            section += "\n\n정답이 발견되지 않았습니다. 각 문제마다 지문 아래에 다음과 같이 정답을 적어주세요\n정답: 3\n\n"


                    section = section.strip() + "\n"
                    section = section.replace("  ", " ")

                    section = section.replace("►", "")
                    if title_search:
                        삽입title = 삽입title.replace("►", "")
                        section = 삽입title + "\n" + section

                    processed_sections.append(section)
                    


                                    



                #########  어법어휘
                # Grammar/Vocabulary questions with inline corrections
                # Format: "...text ②[what] there is..." with "정답: 2" and "what > that" below
                # Expected output: "...text that there is..." (replace word + remove markers)
                # Key: The words around the answer marker are PART OF THE PASSAGE, not separate choices!
                #elif "→" in section or ">" in section or "->" in section:
                #elif ("→" in section or ">" in section or "->" in section) and ("①[" in section or "① [" in section or "(a)[" in section or "(a) [" in section or "⒜[" in section or "⒜ [" in section):
                elif ("→" in section or ">" in section or "->" in section) and (
                    "①[" in section or
                    "① [" in section or
                    "(a)[" in section or
                    "(a) [" in section or
                    "⒜[" in section or
                    "⒜ [" in section or
                    "①" in section or
                    "⒜" in section or
                    "(a)" in section
                ):
                    if not title_search and section.count(">") > 1:
                        messagebox.showwarning("경고", '어법/어휘 문제가 2개 이상 인식되었습니다. 문제가 구분될 수 있도록 각 문제마다 위에 "[문제]"(따옴표 제외)라고 적어주세요.')
                        return

                    section = section.replace("(a)", "①")
                    section = section.replace("(b)", "②")
                    section = section.replace("(c)", "③")
                    section = section.replace("(d)", "④")
                    section = section.replace("(e)", "⑤")
                    section = section.replace("⒜", "①")
                    section = section.replace("⒝", "②")
                    section = section.replace("⒞", "③")
                    section = section.replace("⒟", "④")
                    section = section.replace("⒠", "⑤")
                    section = section.replace("[", "")
                    section = section.replace("]", "")

                    def remove_marked_paragraphs(text, markers):
                        # Split the text into paragraphs
                        paragraphs = text.split('\n')
                        
                        # Filter out paragraphs that contain any of the specified markers
                        markers_set = set(markers)  # Convert list to set for faster checking
                        new_paragraphs = [paragraph for paragraph in paragraphs if not any(marker in paragraph for marker in markers_set)]
                        
                        # Rejoin the remaining paragraphs into a single text block
                        return '\n'.join(new_paragraphs)



                    if title_search:
                        # Step 1: Extract first paragraph, title, and remainder of the section
                        paragraphs = section.strip().split('\n', 1)
                        문제title = paragraphs[0]
                        section = paragraphs[1]
                        

                    # Step 2: Find the correct option by searching for the circle number before 'ㅁ'
                    answer_number = None
                    match = re.search(r'(①|②|③|④|⑤)\s*ㅁ', section)  # No parentheses needed here

                    if match:
                        answer_number = match.group(1)  # This captures just the number before 'ㅁ'

                        # Construct regex to capture the correct answer and the next 1-3 words
                        # Changed {2} to {0,2} to allow 1-3 words instead of exactly 3
                        answer_pattern = re.compile(r'{}\s+([\w\'\-"\u2014!]+(?:\s+[\w\'\-"\u2014!]+){{0,2}})'.format(re.escape(answer_number)))

                        section_for_replacement_match = answer_pattern.search(section)

                        if section_for_replacement_match:
                            section_for_replacement = f"{answer_number}ㅁ {section_for_replacement_match.group(1)}"

                            # Identify the expressions around '→', '>', or '->'
                            expression_match = re.search(r'^(.*?)(→|>|->)(.*)$', section, re.MULTILINE)

                            if expression_match:
                                before_expression = expression_match.group(1).strip()
                                after_expression = expression_match.group(3).strip()

                                # Replace 'before_expression' with 'after_expression' in 'section_for_replacement'
                                if before_expression and after_expression:
                                    section_for_replacement = re.sub(r'\b{}\b'.format(re.escape(before_expression)), after_expression, section_for_replacement)
                                else:
                                    section += "\n\n수정할 표현이 발견되지 않았습니다. 어떤 표현을 어떻게 고쳐야 하는지 문재 아래에 다음과 같이 적어주세요.\ninclude -> exclude\n\n"
                            else:
                                section += "\n\n수정할 표현이 발견되지 않았습니다. 어떤 표현을 어떻게 고쳐야 하는지 문재 아래에 다음과 같이 적어주세요.\ninclude -> exclude\n\n"

                            # Replace the original matched section with the modified 'section_for_replacement'
                            section = answer_pattern.sub(section_for_replacement, section)


                    else:
                        answer_number = answer_key_pattern.search(section)

                        if answer_number:
                            answer_number = answer_number.group(1)
                            answer_number = number_mapping.get(answer_number, answer_number)  # Map to circled if needed

                            # Remove lines containing '정답' indication
                            section = re.sub(정답라인지우기pattern, '', section)

                            # Construct regex to capture the correct answer and the next 1-3 words
                            # Changed {2} to {0,2} to allow 1-3 words instead of exactly 3
                            #answer_pattern = re.compile(r'{}\s*(\w+\s+\w+\s+\w+)'.format(re.escape(answer_number))) #이걸 쓰면 - 같은 문장부호가 인식 안됨
                            #answer_pattern = re.compile(r'{}\s+([\w\'\-"\u2014!]+(?:\s+[\w\'\-"\u2014!]+){{2}})'.format(re.escape(answer_number))) ## 이거 쓰려면 ④ 뒤에 공백 한칸 필요.
                            answer_pattern = re.compile(r'{}\s*([\w\'\-"\u2014!]+(?:\s+[\w\'\-"\u2014!]+){{0,2}})'.format(re.escape(answer_number)))

                            section_for_replacement_match = answer_pattern.search(section)

                            if section_for_replacement_match:
                                section_for_replacement = f"{answer_number} {section_for_replacement_match.group(1)}"

                                # Identify the expressions around '→', '>', or '->'
                                expression_match = re.search(r'^(.*?)(→|>|->)(.*)$', section, re.MULTILINE)

                                if expression_match:
                                    before_expression = expression_match.group(1).strip()
                                    after_expression = expression_match.group(3).strip()

                                    if after_expression == None or after_expression == "":
                                        after_expression = " "

                                    # Replace 'before_expression' with 'after_expression' in 'section_for_replacement'
                                    if before_expression and after_expression:
                                        section_for_replacement = re.sub(r'\b{}\b'.format(re.escape(before_expression)), after_expression, section_for_replacement)
                                    else:
                                        section += "\n\n수정할 표현이 발견되지 않았습니다. 어떤 표현을 어떻게 고쳐야 하는지 문재 아래에 다음과 같이 적어주세요.\ninclude -> exclude\n\n"

                                else:
                                    section += "\n\n수정할 표현이 발견되지 않았습니다. 어떤 표현을 어떻게 고쳐야 하는지 문재 아래에 다음과 같이 적어주세요.\ninclude -> exclude\n\n"

                                # Replace the original matched section with the modified 'section_for_replacement'
                                section = answer_pattern.sub(section_for_replacement, section)

                                # Remove the answer explanation line (e.g., "insignificant > large")
                                section = remove_marked_paragraphs(section, ['→', '>', '->'])

                        else:
                            section += "\n\n정답이 발견되지 않았습니다. 각 문제마다 지문 아래에 다음과 같이 정답을 적어주세요\n정답: 3\n\n"



                    section = remove_marked_paragraphs(section, ['→', '>', '->'])

                    # Define a pattern that matches any of the specified characters
                    pattern = r'[①②③④⑤ㅁ]'
                    # Use re.sub() to replace any character that matches the pattern with an empty string
                    section = re.sub(pattern, '', section) #선지랑 ㅁ 제거하기
                    #section = re.sub(r'\s{2,}', ' ', section) #공백 2개 이상을 공백 1개로 줄이기
                    # Collapse runs of spaces/tabs but keep \n intact
                    section = re.sub(r'[ \t]{2,}', ' ', section)

                    # Final reformatting and concatenation
                    section = section.strip() + "\n"
                    section = section.replace("►", "")
                    if title_search:
                        문제title = 문제title.replace("►", "")
                        section = 문제title + "\n" + section

                    processed_sections.append(section)


                        




                #################################################################################
                elif "관계 없는 문장" in section:

                    if title_search:
                        # Step 1: Extract first paragraph, title, and remainder of the section
                        paragraphs = section.strip().split('\n', 1)
                        문제title = paragraphs[0]
                        section = paragraphs[1]


                    # Step 2: Find the correct option by searching for the circle number before 'ㅁ'


                    def remove_answer_text(section: str, answer_number: str) -> str:
                        # Define the order of markers
                        markers = ['①','②','③','④','⑤']

                        # Find index of the answer_number
                        try:
                            idx = markers.index(answer_number)
                        except ValueError:
                            # If answer_number is not one of the known markers, just return section unchanged
                            return section

                        # Find the next marker if it exists
                        if idx < len(markers) - 1:
                            next_marker = markers[idx + 1]
                        else:
                            next_marker = None

                        # If we have a next marker, remove from answer_number up to the next marker
                        # If there's no next marker (answer_number is '⑤'), remove from answer_number to the end.

                        if next_marker:
                            # Regex to remove everything between 'answer_number' and the next marker
                            pattern = rf'{re.escape(answer_number)}.*?(?={re.escape(next_marker)})'
                        else:
                            # Regex to remove from 'answer_number' to the end
                            pattern = rf'{re.escape(answer_number)}.*'

                        # Use DOTALL so '.' matches newlines
                        result = re.sub(pattern, '', section, flags=re.DOTALL)
                        return result


                    answer_number = None
                    match = re.search(r'(①|②|③|④|⑤)\s*ㅁ', section)  # No parentheses needed here

                    if match:
                        answer_number = match.group(1)  # This captures just the number before 'ㅁ'



                    else:
                        answer_number = answer_key_pattern.search(section)

                        if answer_number:
                            answer_number = answer_number.group(1)
                            answer_number = number_mapping.get(answer_number, answer_number)  # Map to circled if needed

                            # Remove lines containing '정답' indication
                            section = re.sub(정답라인지우기pattern, '', section)



                        else:
                            section += "\n\n정답이 발견되지 않았습니다. 각 문제마다 지문 아래에 다음과 같이 정답을 적어주세요\n정답: 3\n\n"


                    section = remove_answer_text(section, answer_number)





                    # Define a pattern that matches any of the specified characters
                    pattern = r'[①②③④⑤ㅁ]'
                    # Use re.sub() to replace any character that matches the pattern with an empty string
                    section = re.sub(pattern, '', section) #선지랑 ㅁ 제거하기
                    section = re.sub(r'[ \t]{2,}', ' ', section) #공백 2개 이상을 공백 1개로 줄이기

                    # Join sentences back into a single paragraph
                    lines = section.split('\n')

                    # Separate title line, paragraph lines, and footnote lines
                    title_line = ""
                    paragraph_lines = []
                    footnote_lines = []

                    for line in lines:
                        line = line.strip()
                        if not line:
                            continue

                        # Line containing "관계 없는 문장" is the title/question
                        if "관계 없는 문장" in line:
                            title_line = line
                        # Lines starting with * are footnotes
                        elif line.startswith('*'):
                            footnote_lines.append(line)
                        # Everything else is part of the paragraph
                        else:
                            paragraph_lines.append(line)

                    # Join paragraph lines into a single paragraph
                    paragraph = ' '.join(paragraph_lines)

                    # Reconstruct the section
                    section_parts = []
                    if title_line:
                        section_parts.append(title_line)
                    if paragraph:
                        section_parts.append(paragraph)
                    section_parts.extend(footnote_lines)

                    section = '\n'.join(section_parts)

                    # Final reformatting and concatenation
                    section = section.strip() + "\n"
                    section = section.replace("►", "")
                    if title_search:
                        문제title = 문제title.replace("►", "")
                        section = 문제title + "\n" + section

                    processed_sections.append(section)


                        


                #################################################################################
                elif "밑줄 친 부분이 가리키는 대상이" in section or "가리키는 대상이 나머지" in section:


                    if title_search:
                        # Step 1: Extract first paragraph, title, and remainder of the section
                        paragraphs = section.strip().split('\n', 1)
                        문제title = paragraphs[0]
                        section = paragraphs[1]
                        


                    # Step 2: Find the correct option by searching for the circle number before 'ㅁ'


                    answer_number = answer_key_pattern.search(section)
                    
                    if answer_number:
                        answer_number = answer_number.group(1)
                        answer_number = number_mapping.get(answer_number, answer_number)  # Map to circled if needed
                        
                        # Remove lines containing '정답' indication
                        section = re.sub(정답라인지우기pattern, '', section)
        


                    else:
                        section += "\n\n정답이 발견되지 않았습니다. 각 문제마다 지문 아래에 다음과 같이 정답을 적어주세요\n정답: 3\n\n"


                    # Define a pattern that matches any of the specified characters
                    pattern = r'[①②③④⑤ㅁ]'
                    # Use re.sub() to replace any character that matches the pattern with an empty string
                    section = re.sub(pattern, '', section) #선지랑 ㅁ 제거하기
                    section = re.sub(r'[ \t]{2,}', ' ', section) #공백 2개 이상을 공백 1개로 줄이기

                    # Final reformatting and concatenation
                    section = section.strip() + "\n"
                    section = section.replace("►", "")
                    if title_search:
                        문제title = 문제title.replace("►", "")
                        section = 문제title + "\n" + section


                    #print("section:")
                    #print(section)
                    processed_sections.append(section)


                        


                #################################################################################
                elif "각 네모 안에서" in section:

                    if title_search:
                        # Step 1: Extract first paragraph, title, and remainder of the section
                        paragraphs = section.strip().split('\n', 1)
                        문제title = paragraphs[0]
                        section = paragraphs[1]



                    # 1) Remove the line containing "각 네모 안에서"
                    lines = section.split('\n')
                    lines = [line for line in lines if "각 네모 안에서" not in line]
                    
                    # Rejoin for further processing
                    text = "\n".join(lines)
                    
                    # 2) Read the "정답 N" line to figure out which choice is correct
                    #    We'll look for something like "정답 2" or "정답: 3"
                    answer_num_match = re.search(r'정답\s*[:=]?\s*([1-5])', text)
                    if not answer_num_match:
                        # If we didn't find any 정답 line, just return the text as-is
                        # or raise an error, depending on your needs
                        return text
                    
                    correct_number = answer_num_match.group(1)  # e.g. '2'
                    
                    # 3) We now find the matching line that starts with '①'..'⑤'
                    #    We'll map '1' -> '①', '2' -> '②', etc.
                    circle_map = {'1':'①', '2':'②', '3':'③', '4':'④', '5':'⑤'}
                    circle_mark = circle_map.get(correct_number, '')  # e.g. '②'
                    
                    # Regex to find lines like "① Adopt - them - where"
                    choice_pattern = re.compile(r'^([①②③④⑤])\s+(.*)')
                    
                    chosen_line = None
                    for line in text.split('\n'):
                        m = choice_pattern.match(line.strip())
                        if m:
                            # e.g. m.group(1) = '②', m.group(2) = 'Adopt - themselves - where'
                            if m.group(1) == circle_mark:
                                chosen_line = m.group(2)  # e.g. "Adopt - themselves - where"
                                break
                    
                    if not chosen_line:
                        # Could not find the correct choice line
                        return text

                    # chosen_line => something like "Adopt - themselves - where"
                    # Split by '-' or ' - '
                    #chosen_answers = [s.strip() for s in re.split(r'\s*-\s*', chosen_line)]
                    chosen_answers = re.split(r'\s-\s', chosen_line)

                                        
                    # Expecting exactly 3 items for (A), (B), (C)
                    # If you have 4 items for (D), expand as needed.
                    if len(chosen_answers) != 3:
                        return text  # or raise an error


                    ansA, ansB, ansC = chosen_answers  # e.g. "Adopt", "themselves", "where"

                    # 4) Replace (A)[Adopt / Adopting], (B)[them / themselves], (C)[where / which] with the chosen answers
                    #    We'll do a small pattern for each bracket:

                    # a. Replace the entire "(A)[...]" block with ansA
                    #    We expect something like "(A)[Adopt / Adopting]"
                    #    So a pattern: r'\(A\)\[[^\]]+\]' and we replace with ansA
                    #    The "[^\]]+" means "all text until the next ']'"
                    text = re.sub(r'\(A\)\[[^\]]+\]', ansA, text)
                    text = re.sub(r'\(B\)\[[^\]]+\]', ansB, text)
                    text = re.sub(r'\(C\)\[[^\]]+\]', ansC, text)

                    # 5) Remove the line containing "(A) - (B) - (C)"
                    #    Or any line that looks like that
                    #    Also remove lines starting with "① ", "② ", etc.
                    lines = text.split('\n')
                    filtered = []
                    for line in lines:
                        stripped = line.strip()
                        if stripped.startswith('(A) - (B) - (C)'):
                            continue
                        # Also skip lines that start with '①', '②', etc. or "정답"
                        if re.match(r'^[①②③④⑤]', stripped):
                            continue
                        if stripped.startswith('정답'):
                            continue
                        
                        filtered.append(line)
                    
                    text = "\n".join(filtered)
                    
                    # 6) Finally, remove leftover '/', '[', ']', or the label text
                    #    But be careful not to remove square brackets if you actually need them.
                    #    If you want to remove *all* slashes or leftover brackets in the text:
                    text = text.replace('[', '').replace(']', '').replace('/', '')
                    
                    # Also remove leftover (A), (B), (C), etc. if they might still remain somewhere:
                    # For example, just remove raw "(A)", "(B)", "(C)" if desired:
                    text = re.sub(r'\([A-Z]\)', '', text)
                    
                    # Tidy up multiple spaces
                    text = re.sub(r' {2,}', ' ', text)
                    # Tidy up blank lines
                    text = re.sub(r'\n{2,}', '\n', text).strip()


                    section = text
                    # Final reformatting and concatenation
                    section = section.strip() + "\n"
                    section = section.replace("►", "")
                    if title_search:
                        문제title = 문제title.replace("►", "")
                        section = 문제title + "\n" + section


                    processed_sections.append(section)






                #########  순서 ##############
                #elif "(A)-(C)-(B)" in section or "(A) - (C) - (B)" in section or "A-C-B" in section or "A - C - B" in section or "(A) ― (C) ― (B)" in section:
                elif all(x in section for x in ["(A)", "(B)", "(C)"]) and "(D)" not in section:

                    if not title_search and section.count("(A)") > 6:
                        messagebox.showwarning("경고", '순서 문제가 2개 이상 인식되었습니다. 문제가 구분될 수 있도록 각 문제마다 위에 "[문제]"(따옴표 제외)라고 적어주세요.')
                        return


                    section = section.replace("주어진 글 다음에 이어질 글의 순서로 가장 적절한 것을 고르시오.", "").replace("주어진 글 다음에 이어질 글의 순서로 가장 적절한 것을 고르시오", "")

                    # Initialize title as empty string
                    title = ""

                    # Check for the special character 'ㅁ' and extract the answer if present
                    special_char_match = re.search(r'①.*ㅁ|②.*ㅁ|③.*ㅁ|④.*ㅁ|⑤.*ㅁ', section)
                    if special_char_match:
                        answer_char = special_char_match.group()[0]  # Extracts '①', '②', '③', '④', or '⑤'
                    else:
                        # If 'ㅁ' is not found, use regex to find the correct answer indicated after [정답/모범답안]
                        answer_key_pattern = re.compile(r'정답.*?([①②③④⑤12345])', re.DOTALL)
                        match = answer_key_pattern.search(section)
                        answer_char = match.group(1) if match else None

                    # Pattern to match options with potential correct answer marker 'ㅁ'
                    options_pattern = re.compile(r'(?m)^\n?(①|②|③|④|⑤|1|2|3|4|5)\s*(.*?)(ㅁ?)\s*$')
                    정답라인지우기pattern = re.compile(r'^.*정답.*\n*[①②③④⑤12345].*$', re.MULTILINE)

                    # Mapping from answer number to its sequence
                    answer_mapping = {
                        '①': 'A-C-B',
                        '②': 'B-A-C',
                        '③': 'B-C-A',
                        '④': 'C-A-B',
                        '⑤': 'C-B-A',
                        '1': 'A-C-B',
                        '2': 'B-A-C',
                        '3': 'B-C-A',
                        '4': 'C-A-B',
                        '5': 'C-B-A',
                    }

                    # Extract the sequence for the correct answer
                    sequence = answer_mapping.get(answer_char, '')

                    # Split the sequence into individual parts
                    sequence_order = sequence.split('-')




                    ############ 주석어휘 capture ##############
                    # Regular expression pattern to find a paragraph starting with '*'
                    pattern = re.compile(r'^\*([^\n]*)', re.MULTILINE)
                    # Search for the pattern in the section
                    match = pattern.search(section)
                    if match:
                        asterisk_paragraph = match.group(0)
                    else:
                        asterisk_paragraph = ""
                    ##########################################

                    # Normalize the text to handle different option formats
                    normalized_text = re.sub(r'(①|②|③|④|⑤)', r'\n\1', section)
                    normalized_text = re.sub(r'\n{2,}', '\n', normalized_text)  # Clean up extra new lines
                    # Remove all option lines
                    section = re.sub(options_pattern, '', normalized_text)
                    # Remove lines containing '정답' indication
                    section = re.sub(정답라인지우기pattern, '', section)

                    # Extract parts of the section
                    parts = re.findall(r'\([A-C]\) .*', section)

                    # Modified patterns to work with or without a title
                    first_section_pattern = r'(.*?)(?=\(A\))'  # This will capture everything before (A)
                    title_pattern_ = r'((' + '|'.join(title_triggers) + r').*?\n)'  # This pattern remains the same


                    preceding_passage_search = re.search(first_section_pattern, section, re.DOTALL)
                    if preceding_passage_search:
                        preceding_passage = preceding_passage_search.group(1).strip()
                    else:
                        preceding_passage = ''  # Or handle the missing preceding passage case as needed
                    
                    title_search = re.search(title_pattern_, section, re.DOTALL)                    
                    if title_search:
                        title = title_search.group(1).strip()
                        # Remove the title from the preceding passage to avoid duplication
                        preceding_passage = preceding_passage.replace(title, '', 1).strip()
                    else:
                        title = ""  # Set to empty string if no title is found

                    # Rearrange according to the sequence
                    rearranged_passage = preceding_passage + " "  # Changed to space instead of newline
                    for part in sequence_order:
                        for p in parts:
                            if p.startswith(f'({part})'):
                                rearranged_passage += p + ' '  # Add space instead of newline

                    rearranged_passage = re.sub(r'\([ABC]\) ', '', rearranged_passage)  # Remove labels like "(A) "
                    rearranged_passage = re.sub(r'\s{2,}', ' ', rearranged_passage)  # Replace two or more consecutive spaces with a single space
                    rearranged_passage = rearranged_passage.strip()  # Remove leading/trailing whitespace

                    # Construct the final section, with or without title
                    if title:
                        section = title + "\n" + rearranged_passage + "\n"
                    else:
                        section = rearranged_passage + "\n"
                    
                    section = section.replace("►", "")
                    section = section + asterisk_paragraph + "\n"
                    section = section.replace("\n\n*", "\n*")


                    processed_sections.append(section)






                ######### 43-45번 장문독해 ##############

                elif all(x in section for x in ["(A)", "(B)", "(C)", "(D)"]):
                    title = ""

                    # Check for the special character 'ㅁ' and extract the answer if present
                    special_char_match = re.search(r'①.*ㅁ|②.*ㅁ|③.*ㅁ|④.*ㅁ|⑤.*ㅁ', section)
                    if special_char_match:
                        answer_char = special_char_match.group()[0]  # Extracts '①', '②', '③', '④', or '⑤'
                    else:
                        # If 'ㅁ' is not found, use regex to find the correct answer indicated after [정답/모범답안]
                        answer_key_pattern = re.compile(r'정답.*?([①②③④⑤12345])', re.DOTALL)
                        match = answer_key_pattern.search(section)
                        answer_char = match.group(1) if match else None

                    # Pattern to match options with potential correct answer marker 'ㅁ'
                    options_pattern = re.compile(r'(?m)^\n?(①|②|③|④|⑤|1|2|3|4|5)\s*(.*?)(ㅁ?)\s*$')
                    정답라인지우기pattern = re.compile(r'^.*정답.*\n*[①②③④⑤12345].*$', re.MULTILINE)

                    # Mapping from answer number to its sequence
                    answer_mapping = {
                        '①': 'B-D-C',
                        '②': 'C-B-D',
                        '③': 'C-D-B',
                        '④': 'D-B-C',
                        '⑤': 'D-C-B',
                        '1': 'B-D-C',
                        '2': 'C-B-D',
                        '3': 'C-D-B',
                        '4': 'D-B-C',
                        '5': 'D-C-B',
                    }

                    # Extract the sequence for the correct answer
                    sequence = answer_mapping.get(answer_char, '')
                    sequence_order = sequence.split('-')


                    ############ 주석어휘 capture ##############
                    # Regular expression pattern to find a paragraph starting with '*'
                    pattern = re.compile(r'^\*([^\n]*)', re.MULTILINE)
                    # Search for the pattern in the section
                    match = pattern.search(section)
                    if match:
                        asterisk_paragraph = match.group(0)
                        #print("주석어휘:", asterisk_paragraph)  
                    else:
                        asterisk_paragraph = ""
                    ##########################################33

                    # Normalize the text to handle different option formats
                    normalized_text = re.sub(r'(①|②|③|④|⑤)', r'\n\1', section)
                    normalized_text = re.sub(r'\n{2,}', '\n', normalized_text)  # Clean up extra new lines

                    # Remove all option lines
                    section = re.sub(options_pattern, '', normalized_text)

                    # Remove lines containing '정답' indication
                    section = re.sub(정답라인지우기pattern, '', section)

                    # Extract parts of the section
                    parts_pattern = re.compile(r'\(([A-D])\)\s*(.*?)\s*(?=\([A-D]\)|$)', re.DOTALL)
                    parts = parts_pattern.findall(section)
                    for label, content in parts:
                        pass

                    first_section_pattern = r'(' + '|'.join(title_triggers) + r').*?\n(.*?)(?=\(A\))'  ### A앞의 preceding passage 캡쳐
                    title_pattern_ = r'((' + '|'.join(title_triggers) + r').*?\n)' ### 이게 제목만 캡쳐

                    #print("섹션")
                    #print(section)

                    preceding_passage_search = re.search(first_section_pattern, section, re.DOTALL)
                    if preceding_passage_search:
                        preceding_passage = preceding_passage_search.group(2).strip()  # (1)은 title trigger만 캡쳐됨
                    else:
                        preceding_passage = ''  # Or handle the missing preceding passage case as needed
                    
                    #title_search = re.search(title_pattern_, section, re.DOTALL)                    
                    if title_search:
                        title = title_search.group(1).strip()
                        title = "►" + title
                        #print("\ntitle")
                        #print(title)
                    else:
                        preceding_passage = ''  # Or handle the missing preceding passage case as needed
                        title = ""

                    # Rearrange according to the sequence
                    rearranged_passage = preceding_passage + "\n"
                    part_dict = {part: content for part, content in parts}

                    # Add the first part (A)
                    rearranged_passage += f"(A) {part_dict['A'].strip()}\n"

                    # Add the remaining parts according to the sequence
                    for part in sequence_order:
                        rearranged_passage += f'({part}) {part_dict[part].strip()}\n'


                    # Remove the markers (a), (b), (c), (d), and (e)
                    rearranged_passage = re.sub(r'\([a-e]\)', '', rearranged_passage)

                    rearranged_passage = re.sub(r'\([A-D]\) ', '', rearranged_passage)  # Remove labels like "(A) "

                    rearranged_passage = re.sub(r'\n{2,}', '\n', rearranged_passage)  # Replace two or more consecutive "\n" with a single "\n"

                    rearranged_passage = re.sub(r' {2,}', ' ', rearranged_passage)  # Replace two or more consecutive spaces with a single space

                    section = title + "\n" + rearranged_passage.strip() + "\n"

                    section = section.replace("►", "")
                    section = section + "\n" + asterisk_paragraph + "\n"
                    section = section.replace("\n\n*", "\n*")

                    processed_sections.append(section)

                elif "일치" in section: #도표 일치 안내문
                    if title_search:
                        # Step 1: Extract first paragraph, title, and remainder of the section
                        paragraphs = section.strip().split('\n', 1)
                        문제title = paragraphs[0]
                        section = paragraphs[1]



                    passage_lines = section.split('\n')
                    modified_section = '\n'.join(line for line in passage_lines if '일치' not in line) #이게 지문 부분
                    section = modified_section.strip()


                    section = section.strip() + "\n"
                    section = section.replace("►", "")
                    if title_search:
                        문제title = 문제title.replace("►", "")
                        section = 문제title + "\n" + section


                    processed_sections.append(section)


 

                else: #그 외 그냥 지문
                    #print("-------------------")
                    if title_search and len(section) > 20:
                        section = section.replace("►", "")
                        section = section + "\n"
                        processed_sections.append(section)




                #################### 공통
                processed_text = ''.join(processed_sections).strip()

            # Strip each paragraph (line)
            stripped_paragraphs = [line.strip() for line in processed_text.split('\n')]
            # Join the stripped lines back together, separated by newline characters
            final_text = '\n'.join(stripped_paragraphs)

            #print(f"*********************final_text:\n{final_text}\n***********************")
            ## 주석어휘처리
            final_text = final_text.replace("*", "\n*")
            final_text = final_text.replace("*\n*", "**")
            final_text = final_text.replace("**\n*", "***")
            final_text = final_text.replace("\n\n*", "\n*")
            final_text = final_text.replace("\n**", " **")
        
            





            #수능독해발문 지우기
            paragraphs = final_text.split('\n')
            processed_paragraphs = []

            for paragraph in paragraphs:
                # Combine both sets of patterns for prefixes and phrases
                combined_patterns = self.unwanted_prefixes + self.unwanted_phrases
                combined_pattern = r'(?mi)^\s*(' + '|'.join(re.escape(p) for p in combined_patterns) + ')'
                # Check if the paragraph matches any combined pattern
                if re.match(combined_pattern, paragraph):
                    continue  # Skip this paragraph

                processed_paragraphs.append(paragraph)
            final_text = '\n'.join(processed_paragraphs)
            
            #print(f"*********************수능독해발문 지우기 후 final_text:\n{final_text}\n***********************")


            # 어법어휘 > 라인 지우기          
            paragraphs = final_text.split("\n")
            filtered_paragraphs = []
            for paragraph in paragraphs:
                if ">" not in paragraph:
                    filtered_paragraphs.append(paragraph)
            final_text = "\n".join(filtered_paragraphs)
            
            #print(f"*********************어법어휘 > 라인 지우기 후 final_text:\n{final_text}\n***********************")


            # 주석어휘와 정답 지우기
            paragraphs = final_text.split("\n")
            # 2. Filter out paragraphs that start with "정답" or "*"
            filtered_paragraphs = []
            for paragraph in paragraphs:
                # Strip leading/trailing whitespace for consistent checks
                stripped = paragraph.strip()              
                # If it doesn't start with "정답" or "*", we keep it
                if not (stripped.startswith("정답") or stripped.startswith("*") or stripped.startswith("①") or stripped.startswith("②") or stripped.startswith("③") or stripped.startswith("④") or stripped.startswith("⑤")):
                    filtered_paragraphs.append(paragraph)

            # 3. Re-join the remaining paragraphs with double newlines
            final_text = "\n".join(filtered_paragraphs)

            # [ ] 지우기            
            final_text = final_text.replace("[", "").replace("]", "")
            
            # • 을 .로 돌려놓기
            final_text = final_text.replace("•", ".")  # Restore original periods for special cases

            #print(f"*********************최종 final_text:\n{final_text}\n***********************")


        ################ 정답채우기 끝 ################33



        if final_text and final_text != "":
            #################### 최종 (processed_text) ############
            # Clear the current content of the edit_box
            self.edit_box.delete("1.0", "end")
            # Insert the modified text back into the edit_box
            self.edit_box.insert("1.0", final_text)




        ######### 찾아바꾸기
        
        if self.편집기_radio_var.get() == 4:
            #find_text = self.find_textbox.get("1.0", "end-1c").strip()
            find_text = self.find_textbox.get("1.0", "end-1c")
            replace_text = self.replace_textbox.get("1.0", "end-1c")
            
            if not find_text:
                messagebox.showwarning("경고", "찾을 텍스트를 왼쪽 텍스트창에 입력하세요.")
                return

            # Convert string literal '\n' to actual newline characters
            find_text = find_text.replace('\\n', '\n').replace('\\t', '\t')
            replace_text = replace_text.replace('\\n', '\n').replace('\\t', '\t')

            content = self.edit_box.get("1.0", "end-1c")

            # Use re.DOTALL flag to make '.' match newlines as well
            new_content, count = re.subn(re.escape(find_text), replace_text.replace('\\', '\\\\'), content, flags=re.DOTALL)
            
            if count > 0:
                self.edit_box.delete("1.0", "end")
                self.edit_box.insert("1.0", new_content)

                #messagebox.showinfo("완료", f"총 {count}개의 항목이 교체되었습니다.")
            else:
                messagebox.showinfo("알림", "일치하는 텍스트를 찾지 못했습니다.")



        # 스크램블

        elif self.편집기_radio_var.get() == 5:
            def _tokenize_keep_brackets(text: str):
                """
                Split text into tokens, but treat anything inside [...] as one token.
                Example:  'obsessing over their [shape and weight]'
                        → ['obsessing', 'over', 'their', '[shape and weight]']
                """
                # ①  find [...] chunks intact, ② all other non‑space sequences
                return re.findall(r'\[[^\]]+\]|[^\s\[\]]+', text)

            def jumble_sentence(sentence):
                # strip commas / periods so they don’t become tokens
                cleaned = sentence.replace(',', '').replace('.', '')

                tokens = _tokenize_keep_brackets(cleaned)
                if not tokens:
                    return ''

                # Build an NLP doc *without* the square brackets so spaCy tags aren’t harmed
                doc = nlp(re.sub(r'[\[\]]', '', cleaned))

                # If the very first word isn’t a proper noun, down‑case its initial
                first_word = doc[0]
                if first_word.pos_ != 'PROPN' and first_word.text[0].isupper():
                    tokens[0] = first_word.text[0].lower() + first_word.text[1:]

                # Shuffle the list in‑place
                random.shuffle(tokens)

                # Remove the brackets for display, e.g. '[shape and weight]' → 'shape and weight'
                tokens = [t[1:-1] if t.startswith('[') and t.endswith(']') else t for t in tokens]

                # Join with the requested “ / ” separator
                return ' / '.join(tokens)

            def scramble_sentences(text):
                # Split the text into paragraphs by newline
                paragraphs = text.strip().split('\n')
                scrambled_paragraphs = []
                
                for paragraph in paragraphs:
                    scrambled = jumble_sentence(paragraph)
                    scrambled_paragraphs.append(scrambled)
                    
                return '\n'.join(scrambled_paragraphs)

            # Get the text from the editor
            text_to_scramble = self.edit_box.get("1.0", "end-1c")
            
            # Scramble the text
            scrambled_output = scramble_sentences(text_to_scramble)

            # Replace the original text with the scrambled text
            self.edit_box.delete("1.0", "end")
            self.edit_box.insert("1.0", scrambled_output)


            """
        elif self.편집기_radio_var.get() == 5:
            def jumble_sentence(sentence):
                # Remove commas and periods
                cleaned_sentence = sentence.replace(',', '').replace('.', '')
                
                # Split the sentence into words without changing their case
                words = cleaned_sentence.split()
                
                if not words:
                    return ''  # Return empty string if there are no words
                
                # Identify the original first word
                original_first_word = words[0]
                
                # Convert the first character of the original first word to lowercase if it's uppercase
                if original_first_word[0].isupper():
                    words[0] = original_first_word[0].lower() + original_first_word[1:]
                
                # Shuffle the words randomly
                random.shuffle(words)
                
                # Join the words with slashes and spaces
                return ' / '.join(words)

            def scramble_sentences(text):
                # Split the text into paragraphs by newline
                paragraphs = text.strip().split('\n')
                scrambled_paragraphs = []
                
                for paragraph in paragraphs:
                    scrambled = jumble_sentence(paragraph)
                    scrambled_paragraphs.append(scrambled)
                    
                return '\n'.join(scrambled_paragraphs)

            # Get the text from the editor
            text_to_scramble = self.edit_box.get("1.0", "end-1c")
            
            # Scramble the text
            scrambled_output = scramble_sentences(text_to_scramble)

            # Replace the original text with the scrambled text
            self.edit_box.delete("1.0", "end")
            self.edit_box.insert("1.0", scrambled_output)
        """




        # 한영교차
        elif self.편집기_radio_var.get() == 6:
            def process_bilingual_text(text):
                # Check if both '(A)' and '(B)' are in the text
                if '(A)' not in text or '(B)' not in text:
                    return 교차준비문구

                try:
                    # Split the sections
                    korean_part = text.split('(A)')[1].split('(B)')[0].strip()
                    english_part = text.split('(B)')[1].strip()
                    
                    # Split each part into paragraphs
                    korean_paragraphs = korean_part.split('\n')
                    english_paragraphs = english_part.split('\n')
                    
                    # Check if the number of paragraphs matches
                    if len(korean_paragraphs) != len(english_paragraphs):
                        return 에러문구
                    
                    # Interleave Korean and English paragraphs with numbering
                    interleaved_text = []
                    for index, (korean, english) in enumerate(zip(korean_paragraphs, english_paragraphs), start=1):
                        interleaved_text.append(f"{index}. {korean}")
                        interleaved_text.append(english + "\n")
                    
                    return '\n'.join(interleaved_text)
                except IndexError:
                    # This will catch any issues with splitting the text
                    return 교차준비문구
            
            교차준비문구 = """교차할 문장들을 아래와 같이 (A)와 (B)로 구분한 뒤 "편집 실행"을 클릭하세요. \n\n(A)\n한국어 문장1\n한국어 문장2\n한국어 문장3\n\n(B)\n영어 문장1\n영어 문장2\n영어 문장3 \n\n(한글/영어 순서 반대도 가능)\n----------------------------------"""
            에러문구 = "Error: The number of Korean and English sentences does not match."
            교차할텍스트 = self.edit_box.get("1.0", "end-1c")
            result = process_bilingual_text(교차할텍스트)

            if result == 에러문구:
                result = 에러문구 + "\n\n" + 교차할텍스트
            elif result == 교차준비문구:
                result = 교차준비문구 + "\n\n" + 교차할텍스트

            self.edit_box.delete("1.0", "end")
            self.edit_box.insert("1.0", result)


        # 단어 뜻 분리
        elif self.편집기_radio_var.get() == 7:
            data = self.edit_box.get("1.0", "end-1c")
            lines = data.split('\n')
            
            self.edit_box.delete("1.0", "end")
            
            # compile once
            pattern = re.compile(
                r'([A-Za-z][A-Za-z~\.…\'’\s-]*)\s+'                 # English
                r'([\u3130-\u318F\u4E00-\u9FFF가-힣A-Za-z0-9~\[\],.;()·ㆍ…\'’\s-]+?)'
                r'(?=\s+[A-Za-z][A-Za-z~\.…\'’-]*\s+[\u3130-\u318F\u4E00-\u9FFF가-힣]|$)',
                flags=re.UNICODE
            )

            # then process every line
            for line in lines:
                if not line.strip():          # blank line → keep blank
                    self.edit_box.insert('end', '\n')
                    continue

                pairs = pattern.findall(line) # run on each individual line
                if pairs:
                    for eng, kor in pairs:
                        self.edit_box.insert('end', f'{eng.strip()}\t{kor.strip()}\n')
                else:
                    self.edit_box.insert('end', line + '\n')





    def on_textbox_change(self, event=None):
        # If there's any text in the textbox, select the checkbox
        if self.check06_textbox.get("1.0", "end-1c").strip():
            self.check06.select()
        else:
            self.check06.deselect()





    def create_options_popup_한글삭제(self):
        # Create the popup window as a top-level window
        self.options_popup = CTkToplevel(self.window)
        self.options_popup.title("Options")
        self.options_popup.geometry("300x160")  # Adjust the window size as needed
        self.options_popup.resizable(False, False)
        self.options_popup.grab_set()
        self.options_popup.focus_set()
        self.options_popup.grid_rowconfigure(0, weight=1)
        self.options_popup.grid_columnconfigure(0, weight=1)


        option_frame = CTkFrame(self.options_popup, fg_color="#EFF3F0", width=300, height=100)
        option_frame.grid(row=0, column=0, sticky="nsew", padx=0, pady=0)
        option_frame.grid_rowconfigure(2, weight=1)
        option_frame.grid_columnconfigure(0, weight=1)

        button_frame = CTkFrame(self.options_popup, fg_color="#2B3D2D", width=300, height=60, corner_radius=0)
        button_frame.grid(row=1, column=0, sticky="ew", padx=0, pady=0)
        button_frame.grid_rowconfigure(0, weight=1)
        button_frame.grid_columnconfigure(0, weight=1)


        # Create checkboxes
        label_option = CTkLabel(option_frame, text="한글 삭제 옵션", font=(self.default_font, 15, "bold"), text_color="black", fg_color="transparent")
        label_option.grid(row=0, column=0, padx=10, pady=5, sticky='ewn')

        #self.한글옵션_radio_var = IntVar(value=0)

        checkbox_koreanoption1 = CTkRadioButton(option_frame, text="한글로 시작하면 삭제", variable=self.한글옵션_radio_var, value=0, font=(self.default_font, 12), text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C")
        checkbox_koreanoption1.grid(row=1, column=0, padx=10, pady=5, sticky='ewn')
        checkbox_koreanoption2 = CTkRadioButton(option_frame, text="한글이 포함되면 삭제", variable=self.한글옵션_radio_var, value=1, font=(self.default_font, 12), text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C")
        checkbox_koreanoption2.grid(row=2, column=0, padx=10, pady=5, sticky='ewn')
        #checkbox_koreanoption1.select()
    
        # Restore radio button state
        self.한글옵션_radio_var.set(self.한글옵션_radio_state)


        def submit(event=None):
            self.check03.select()

            # Save states of checkboxes/radio
            self.한글옵션_radio_state = self.한글옵션_radio_var.get()

            self.options_popup.destroy()

        def close_popup(event=None):
            # Save states of checkboxes/radio
            self.한글옵션_radio_state = self.한글옵션_radio_var.get()

            self.options_popup.destroy()


        ok_button = CTkButton(button_frame, width=70, height=25, text="확인", fg_color="#FEF9E0", hover_color="#DDA15C", text_color='black', font=(self.default_font, 13, 'bold'), corner_radius=5, command=submit)
        ok_button.grid(row=0, column=0, padx=20, pady=10)

        # Check the operating system to determine the correct modifier key
        if sys.platform.startswith('darwin'):
            # Mac OS
            modifier_key = 'Command'
        elif sys.platform.startswith('win'):
            # Windows
            modifier_key = 'Control'
        else:
            # Linux or other; adjust as needed
            modifier_key = 'Control'
            
        self.options_popup.bind(f'<{modifier_key}-Return>', submit)
        # Bind 'Escape' to cancel_edit as before
        self.options_popup.bind("<Escape>", close_popup)






    def create_options_popup_영어삭제(self):
        # Create the popup window as a top-level window
        self.options_popup = CTkToplevel(self.window)
        self.options_popup.title("Options")
        self.options_popup.geometry("300x160")  # Adjust the window size as needed
        self.options_popup.resizable(False, False)
        self.options_popup.grab_set()
        self.options_popup.focus_set()
        self.options_popup.grid_rowconfigure(0, weight=1)
        self.options_popup.grid_columnconfigure(0, weight=1)


        option_frame = CTkFrame(self.options_popup, fg_color="#EFF3F0", width=300, height=100)
        option_frame.grid(row=0, column=0, sticky="nsew", padx=0, pady=0)
        option_frame.grid_rowconfigure(2, weight=1)
        option_frame.grid_columnconfigure(0, weight=1)

        button_frame = CTkFrame(self.options_popup, fg_color="#2B3D2D", width=300, height=60, corner_radius=0)
        button_frame.grid(row=1, column=0, sticky="ew", padx=0, pady=0)
        button_frame.grid_rowconfigure(0, weight=1)
        button_frame.grid_columnconfigure(0, weight=1)



        # Create checkboxes
        label_option = CTkLabel(option_frame, text="대소문자 옵션", font=(self.default_font, 15, "bold"), text_color="black", fg_color="transparent")
        label_option.grid(row=0, column=0, padx=10, pady=5, sticky='ewn')
        checkbox_uppercase = CTkCheckBox(option_frame, text="영어 대문자로 시작하면 삭제", font=(self.default_font, 12), variable=self.start_with_uppercase_var, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C")
        checkbox_uppercase.grid(row=1, column=0, padx=10, pady=5, sticky='ewn')
        checkbox_lowercase = CTkCheckBox(option_frame, text="영어 소문자로 시작하면 삭제", font=(self.default_font, 12), variable=self.start_with_lowercase_var, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C")
        checkbox_lowercase.grid(row=2, column=0, padx=10, pady=5, sticky='ewn')

        ####### 기억하게 하기
        self.start_with_uppercase_var.set(self.start_with_case_states[0])
        self.start_with_lowercase_var.set(self.start_with_case_states[1])

        def submit(event=None):
            self.check04.select()

            self.start_with_case_states = [
                self.start_with_uppercase_var.get(), self.start_with_lowercase_var.get(),
            ]
            self.options_popup.destroy()

        def close_popup(event=None):
            self.start_with_case_states = [
                self.start_with_uppercase_var.get(), self.start_with_lowercase_var.get(),
            ]
            self.options_popup.destroy()

        ok_button = CTkButton(button_frame, width=70, height=25, text="확인", fg_color="#FEF9E0", hover_color="#DDA15C", text_color='black', font=(self.default_font, 13, 'bold'), corner_radius=5, command=submit)
        ok_button.grid(row=0, column=0, padx=20, pady=10)

        # Check the operating system to determine the correct modifier key
        if sys.platform.startswith('darwin'):
            # Mac OS
            modifier_key = 'Command'
        elif sys.platform.startswith('win'):
            # Windows
            modifier_key = 'Control'
        else:
            # Linux or other; adjust as needed
            modifier_key = 'Control'
            
        self.options_popup.bind(f'<{modifier_key}-Return>', submit)
        # Bind 'Escape' to cancel_edit as before
        self.options_popup.bind("<Escape>", close_popup)





    def create_options_popup_숫자(self):
        # Create the popup window as a top-level window
        self.options_popup2 = CTkToplevel(self.window)
        self.options_popup2.title("Options")
        self.options_popup2.geometry("600x550")  # Adjust the window size as needed
        self.options_popup2.resizable(False, True)
        self.options_popup2.focus_set()
        self.options_popup2.grab_set()

        self.options_popup2.grid_rowconfigure(0, weight=1)
        self.options_popup2.grid_columnconfigure(0, weight=1)


        option_frame = CTkFrame(self.options_popup2, fg_color="#EFF3F0", width=500)
        option_frame.grid(row=0, column=0, sticky="nsew", padx=0, pady=0)
        option_frame.grid_rowconfigure(0, weight=1)
        option_frame.grid_columnconfigure(0, weight=1)
        option_frame.grid_columnconfigure(1, weight=1)

        option_frame_left = CTkFrame(option_frame, fg_color="#EFF3F0", width=350)
        option_frame_left.grid(row=0, column=0, sticky="nsew", padx=0, pady=0)
        option_frame_left.grid_rowconfigure(13, weight=1)
        option_frame_left.grid_columnconfigure(0, weight=1)

        option_frame_right = CTkFrame(option_frame, fg_color="#EFF3F0", width=250)
        option_frame_right.grid(row=0, column=1, sticky="nsew", padx=0, pady=0)
        option_frame_right.grid_rowconfigure(2, weight=1)
        option_frame_right.grid_columnconfigure(0, weight=1)

        button_frame = CTkFrame(self.options_popup2, fg_color="#2B3D2D", width=300, height=60, corner_radius=0)
        button_frame.grid(row=1, column=0, sticky="sew", padx=0, pady=0)
        button_frame.grid_rowconfigure(0, weight=1)
        button_frame.grid_columnconfigure(0, weight=1)

        # Create checkboxes
        label_option = CTkLabel(option_frame_left, text="숫자/문자 종류", font=(self.default_font, 15, "bold"), text_color="black", fg_color="transparent")
        label_option.grid(row=0, column=0, padx=20, pady=5, sticky='ewn')
        numbercheck1 = CTkCheckBox(option_frame_left, text="1234567890", font=(self.default_font, 12), variable=self.start_with_number1, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C", border_width=2)
        numbercheck1.grid(row=1, column=0, padx=20, pady=5, sticky='ewn')
        numbercheck2 = CTkCheckBox(option_frame_left, text="①②③④⑤ ... ⑳", font=(self.default_font, 12), variable=self.start_with_number2, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C", border_width=2)
        numbercheck2.grid(row=2, column=0, padx=20, pady=5, sticky='ewn')
        numbercheck3 = CTkCheckBox(option_frame_left, text="❶❷❸❹❺ ... ⓴", font=(self.default_font, 12), variable=self.start_with_number3, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C", border_width=2)
        numbercheck3.grid(row=3, column=0, padx=20, pady=5, sticky='ewn')
        numbercheck4 = CTkCheckBox(option_frame_left, text="(1)(2)(3) ... (20) & ( 1 ), ( 2 ) ...", font=(self.default_font, 12), variable=self.start_with_number4, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C", border_width=2)
        numbercheck4.grid(row=4, column=0, padx=20, pady=5, sticky='ewn')
        numbercheck5 = CTkCheckBox(option_frame_left, text="(①)(②)(③) ... (⑳) & ( ① ), ( ② ) ...", font=(self.default_font, 12), variable=self.start_with_number5, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C", border_width=2)
        numbercheck5.grid(row=5, column=0, padx=20, pady=5, sticky='ewn')
        numbercheck6 = CTkCheckBox(option_frame_left, text="ⒶⒷⒸⒹⒺⒻⒼⒽ ... Ⓩ", font=(self.default_font, 12), variable=self.start_with_number6, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C", border_width=2)
        numbercheck6.grid(row=6, column=0, padx=20, pady=5, sticky='ewn')
        numbercheck7 = CTkCheckBox(option_frame_left, text="ⓐⓑⓒⓓⓔⓕⓖⓗ ... ⓩ", font=(self.default_font, 12), variable=self.start_with_number7, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C", border_width=2)
        numbercheck7.grid(row=7, column=0, padx=20, pady=5, sticky='ewn')
        numbercheck8 = CTkCheckBox(option_frame_left, text="(A)(B)(C) ... (Z) & ( A ), ( B ) ...", font=(self.default_font, 12), variable=self.start_with_number8, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C", border_width=2)
        numbercheck8.grid(row=8, column=0, padx=20, pady=5, sticky='ewn')
        numbercheck9 = CTkCheckBox(option_frame_left, text="(a)(b)(c) ... (z) & ( a ), ( b ) ...", font=(self.default_font, 12), variable=self.start_with_number9, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C", border_width=2)
        numbercheck9.grid(row=9, column=0, padx=20, pady=5, sticky='ewn')
        numbercheck10 = CTkCheckBox(option_frame_left, text="㉠㉡㉢㉣㉤㉥㉦㉧㉨㉩㉪㉫㉬㉭", font=(self.default_font, 12), variable=self.start_with_number10, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C", border_width=2)
        numbercheck10.grid(row=10, column=0, padx=20, pady=5, sticky='ewn')
        numbercheck11 = CTkCheckBox(option_frame_left, text="㉮㉯㉰㉱㉲㉳㉴㉵㉶㉷㉸㉹㉺㉻", font=(self.default_font, 12), variable=self.start_with_number11, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C", border_width=2)
        numbercheck11.grid(row=11, column=0, padx=20, pady=5, sticky='ewn')
        numbercheck12 = CTkCheckBox(option_frame_left, text="(ㄱ)(ㄴ)(ㄷ) ... (ㅎ)) & ( ㄱ ), ( ㄴ ) ...", font=(self.default_font, 12), variable=self.start_with_number12, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C", border_width=2)
        numbercheck12.grid(row=12, column=0, padx=20, pady=5, sticky='ewn')
        numbercheck13 = CTkCheckBox(option_frame_left, text="(가)(나)(다) ... (하) & ( 가 ), ( 나 ) ...", font=(self.default_font, 12), variable=self.start_with_number13, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C", border_width=2)
        numbercheck13.grid(row=13, column=0, padx=20, pady=5, sticky='ewn')




        #numbercheck1.select()
        #self.number_radio_var = IntVar(value=0)

        label_option = CTkLabel(option_frame_right, text="삭제 방법", font=(self.default_font, 15, "bold"), text_color="black", fg_color="transparent")
        label_option.grid(row=0, column=0, padx=10, pady=5, sticky='ewn')
        numbercheck1 = CTkRadioButton(option_frame_right, text="해당 숫자/문자로 시작하는 단락 통째로 삭제", font=(self.default_font, 12), variable=self.number_radio_var, value=0, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C")
        numbercheck1.grid(row=1, column=0, padx=10, pady=5, sticky='ewn')
        numbercheck2 = CTkRadioButton(option_frame_right, text="해당 숫자/문자만 삭제", font=(self.default_font, 12), variable=self.number_radio_var, value=1, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C")
        numbercheck2.grid(row=2, column=0, padx=10, pady=5, sticky='ewn')

        ############# 기억하게 하기 코드 ####################
        # Checkbox creations are the same, just make sure to set their state
        self.start_with_number1.set(self.checkbox_states[0])
        self.start_with_number2.set(self.checkbox_states[1])
        self.start_with_number3.set(self.checkbox_states[2])
        self.start_with_number4.set(self.checkbox_states[3])
        self.start_with_number5.set(self.checkbox_states[4])
        self.start_with_number6.set(self.checkbox_states[5])
        self.start_with_number7.set(self.checkbox_states[6])
        self.start_with_number8.set(self.checkbox_states[7])
        self.start_with_number9.set(self.checkbox_states[8])
        self.start_with_number10.set(self.checkbox_states[9])
        self.start_with_number11.set(self.checkbox_states[10])
        self.start_with_number12.set(self.checkbox_states[11])
        self.start_with_number13.set(self.checkbox_states[12])

        # Restore radio button state
        self.number_radio_var.set(self.radio_button_state)
        ####################################################
        
        def submit(event=None):
            self.check05.select()
            # Save states of checkboxes/radio
            self.checkbox_states = [
                self.start_with_number1.get(), self.start_with_number2.get(),
                self.start_with_number3.get(), self.start_with_number4.get(),
                self.start_with_number5.get(), self.start_with_number6.get(),
                self.start_with_number7.get(), self.start_with_number8.get(),
                self.start_with_number9.get(), self.start_with_number10.get(),
                self.start_with_number11.get(), self.start_with_number12.get(),
                self.start_with_number13.get()
            ]
            self.radio_button_state = self.number_radio_var.get()

            self.options_popup2.destroy()

        def close_popup(event=None):
            # Save states of checkboxes/radio
            self.checkbox_states = [
                self.start_with_number1.get(), self.start_with_number2.get(),
                self.start_with_number3.get(), self.start_with_number4.get(),
                self.start_with_number5.get(), self.start_with_number6.get(),
                self.start_with_number7.get(), self.start_with_number8.get(),
                self.start_with_number9.get(), self.start_with_number10.get(),
                self.start_with_number11.get(), self.start_with_number12.get(),
                self.start_with_number13.get()
            ]
            self.radio_button_state = self.number_radio_var.get()
            
            self.options_popup2.destroy()

        ok_button = CTkButton(button_frame, width=70, height=25, text="확인", fg_color="#FEF9E0", hover_color="#DDA15C", text_color='black', font=(self.default_font, 13, 'bold'), corner_radius=5, command=submit)
        ok_button.grid(row=0, column=0, padx=20, pady=10)

        # Check the operating system to determine the correct modifier key
        if sys.platform.startswith('darwin'):
            # Mac OS
            modifier_key = 'Command'
        elif sys.platform.startswith('win'):
            # Windows
            modifier_key = 'Control'
        else:
            # Linux or other; adjust as needed
            modifier_key = 'Control'
            
        self.options_popup2.bind(f'<{modifier_key}-Return>', submit)
        # Bind 'Escape' to cancel_edit as before
        self.options_popup2.bind("<Escape>", close_popup)




    def create_options_popup_특정문자열(self):
        # Create the popup window as a top-level window
        self.options_popup = CTkToplevel(self.window)
        self.options_popup.title("특정 문자열 옵션")
        self.options_popup.geometry("300x160")  # Adjust window size as needed
        self.options_popup.resizable(False, False)
        self.options_popup.grab_set()
        self.options_popup.focus_set()
        self.options_popup.grid_rowconfigure(1, weight=1)
        self.options_popup.grid_columnconfigure(0, weight=1)

        # ---- option_frame ----
        option_frame = CTkFrame(self.options_popup, fg_color="#EFF3F0") #EFF3F0
        option_frame.grid(row=0, column=0, sticky="nsew", padx=0, pady=0)
        option_frame.grid_rowconfigure(0, weight=1)
        option_frame.grid_rowconfigure(1, weight=1)
        option_frame.grid_columnconfigure(0, weight=1)

        label_option = CTkLabel(
            option_frame,
            text="특정 문자열 삭제 옵션",
            font=(self.default_font, 15, "bold"),
            text_color="black",
            fg_color="transparent"
        )
        label_option.grid(row=0, column=0, padx=10, pady=(10, 5), sticky='n')

        # Two checkboxes
        self.checkbox_start = CTkCheckBox(
            option_frame,
            text="특정 문자열로 시작하면 삭제",
            variable=self.specific_string_delete_start_var,
            font=(self.default_font, 12),
            text_color="black",
            border_color="gray",
            hover_color="#BB6C25",
            fg_color="#DDA15C"
        )
        self.checkbox_start.grid(row=1, column=0, padx=15, pady=2, sticky='w')

        self.checkbox_contain = CTkCheckBox(
            option_frame,
            text="특정 문자열을 포함하면 삭제",
            variable=self.specific_string_delete_contain_var,
            font=(self.default_font, 12),
            text_color="black",
            border_color="gray",
            hover_color="#BB6C25",
            fg_color="#DDA15C"
        )
        self.checkbox_contain.grid(row=2, column=0, padx=15, pady=(2, 15), sticky='w')

        # ---- button_frame ----
        button_frame = CTkFrame(self.options_popup, fg_color="#2B3D2D", corner_radius=0)
        button_frame.grid(row=1, column=0, sticky="ew", padx=0, pady=0)
        button_frame.grid_rowconfigure(0, weight=1)
        button_frame.grid_columnconfigure(0, weight=1)

        def submit(event=None):
            # If either of these is checked, it means we want to use "특정 문자열" deletion.
            # So we ensure our main checkbox is selected:
            self.check06.select()
            # Close the popup
            self.options_popup.destroy()

        def close_popup(event=None):
            self.options_popup.destroy()

        ok_button = CTkButton(
            button_frame,
            width=70,
            height=25,
            text="확인",
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.default_font, 13, 'bold'),
            corner_radius=5,
            command=submit
        )
        ok_button.grid(row=0, column=0, padx=20, pady=10)

        # Optionally bind Command/Control+Return and Escape
        if sys.platform.startswith('darwin'):
            modifier_key = 'Command'
        elif sys.platform.startswith('win'):
            modifier_key = 'Control'
        else:
            modifier_key = 'Control'

        self.options_popup.bind(f'<{modifier_key}-Return>', submit)
        self.options_popup.bind("<Escape>", close_popup)




    def create_options_popup_단어이상이하(self):
        # Create the popup window as a top-level window
        self.options_popup = CTkToplevel(self.window)
        self.options_popup.title("Options")
        self.options_popup.geometry("300x200")  # Adjust the window size as needed
        self.options_popup.resizable(False, False)
        self.options_popup.grab_set()
        self.options_popup.focus_set()
        self.options_popup.grid_rowconfigure(2, weight=1)
        self.options_popup.grid_columnconfigure(0, weight=1)


        title_frame = CTkFrame(self.options_popup, fg_color="#EFF3F0", width=300, height=30)
        title_frame.grid(row=0, column=0, sticky="ewn", padx=0, pady=0)
        title_frame.grid_rowconfigure(0, weight=1)
        title_frame.grid_columnconfigure(0, weight=1)        
        label_option = CTkLabel(title_frame, text="삭제 방법", font=(self.default_font, 15, "bold"), text_color="black", fg_color="transparent")
        label_option.grid(row=0, column=0, padx=10, pady=5, sticky='ewn')


        number_frame = CTkFrame(self.options_popup, fg_color="#EFF3F0", width=300, height=50)
        number_frame.grid(row=1, column=0, sticky="ewn", padx=0, pady=0)
        number_frame.grid_rowconfigure(0, weight=1)
        number_frame.grid_columnconfigure(1, weight=1)

        def select_all(event=None):
            """Select all text in number_entry widget."""
            self.number_entry.select_range(0, 'end')
            # Optionally, position cursor at the end
            self.number_entry.icursor('end')       
        label_option2 = CTkLabel(number_frame, text="기준 단어수: ", font=(self.default_font, 12), text_color="black", fg_color="transparent")
        label_option2.grid(row=0, column=0, padx=(25, 10), pady=5, sticky='w')
        self.number_entry = CTkEntry(number_frame, justify=RIGHT, width=60)
        self.number_entry.insert(0, str(self.remembered_number))  # Restore the remembered value
        self.number_entry.grid(row=0, column=1, padx=(10, 25), pady=5, sticky="w")
        self.number_entry.focus()
        self.number_entry.bind("<FocusIn>", select_all)





        option_frame = CTkFrame(self.options_popup, fg_color="#EFF3F0", width=300, height=100)
        option_frame.grid(row=2, column=0, sticky="nsew", padx=0, pady=0)
        option_frame.grid_rowconfigure(0, weight=1)
        option_frame.grid_rowconfigure(1, weight=1)        
        option_frame.grid_columnconfigure(0, weight=1)
        

        self.whatnumber_radio_var = IntVar(value=2)
        number이상이하check1 = CTkRadioButton(option_frame, text="기준 단어수 '이상'인 단락 모두 삭제", font=(self.default_font, 12), variable=self.whatnumber_radio_var, value=1, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C")
        number이상이하check1.grid(row=0, column=0, padx=(25, 10), pady=3, sticky='ew')
        number이상이하check2 = CTkRadioButton(option_frame, text="기준 단어수 '미만'인 단락 모두 삭제", font=(self.default_font, 12), variable=self.whatnumber_radio_var, value=2, text_color="black", border_color="gray", hover_color="#BB6C25", fg_color="#DDA15C")
        number이상이하check2.grid(row=1, column=0, padx=(25, 10), pady=3, sticky='ew')

        button_frame = CTkFrame(self.options_popup, fg_color="#2B3D2D", width=300, height=60, corner_radius=0)
        button_frame.grid(row=3, column=0, sticky="ews", padx=0, pady=(10, 0))
        button_frame.grid_rowconfigure(0, weight=1)
        button_frame.grid_columnconfigure(0, weight=1)


        # Restore radio button state
        self.whatnumber_radio_var.set(self.whatnumber_state)


        def submit(event=None):
            try:
                self.remembered_number = int(self.number_entry.get())
            except ValueError:
                # Handle the case where the entry does not contain a valid integer
                self.remembered_number = None  # Or set a default value or handle the error as needed

            self.whatnumber_state = self.whatnumber_radio_var.get()
            self.options_popup.destroy()

        def close_popup(event=None):
            try:
                self.remembered_number = int(self.number_entry.get())
            except ValueError:
                # Handle the case where the entry does not contain a valid integer
                self.remembered_number = None  # Or set a default value or handle the error as needed

            self.whatnumber_state = self.whatnumber_radio_var.get()
            self.options_popup.destroy()

        ok_button = CTkButton(button_frame, width=70, height=25, text="확인", fg_color="#FEF9E0", hover_color="#DDA15C", text_color='black', font=(self.default_font, 13, 'bold'), corner_radius=5, command=submit)
        ok_button.grid(row=0, column=0, padx=20, pady=10)

        # Check the operating system to determine the correct modifier key
        if sys.platform.startswith('darwin'):
            # Mac OS
            modifier_key = 'Command'
        elif sys.platform.startswith('win'):
            # Windows
            modifier_key = 'Control'
        else:
            # Linux or other; adjust as needed
            modifier_key = 'Control'
            
        self.options_popup.bind(f'<{modifier_key}-Return>', submit)
        # Bind 'Escape' to cancel_edit as before
        self.options_popup.bind("<Escape>", close_popup)







    def create_options_popup_발문(self):
        # Create the popup window as a top-level window
        self.options_popup = CTkToplevel(self.window)
        self.options_popup.title("Options")
        self.options_popup.geometry("500x350")  # Adjust the window size as needed
        self.options_popup.resizable(False, False)
        self.options_popup.grab_set()
        self.options_popup.focus_set()
        self.options_popup.grid_rowconfigure(1, weight=1)
        self.options_popup.grid_columnconfigure(0, weight=1)


        title_frame = CTkFrame(self.options_popup, fg_color="#EFF3F0", width=500, height=30)
        title_frame.grid(row=0, column=0, sticky="ewn", padx=0, pady=0)
        title_frame.grid_rowconfigure(0, weight=1)
        title_frame.grid_columnconfigure(0, weight=1)        
        label_option = CTkLabel(title_frame, text="삭제할 발문 편집", font=(self.default_font, 15, "bold"), text_color="black", fg_color="transparent")
        label_option.grid(row=0, column=0, padx=10, pady=5, sticky='ewn')


        option_frame = CTkFrame(self.options_popup, fg_color="#EFF3F0", width=500, height=310)
        option_frame.grid(row=1, column=0, sticky="nsew", padx=0, pady=0)
        option_frame.grid_rowconfigure(0, weight=1)
        option_frame.grid_columnconfigure(0, weight=1)
        option_frame.grid_columnconfigure(1, weight=1)


        # Combine list items into a single string, each item on a new line
        combined_text_prefixes = "\n".join(self.unwanted_prefixes)
        combined_text_phrases = "\n".join(self.unwanted_phrases)

        발문text1 = CTkTextbox(option_frame, wrap="word", font=(self.default_font, 13), fg_color="#FFFFF8", border_width=1, corner_radius=10)  # Adjust font and size as needed
        발문text1.grid(row=0, column=0, sticky="nsew", padx=20, pady=(20, 10))
        발문text1.delete("1.0", END)  # Clear the textbox
        발문text1.insert("1.0", combined_text_prefixes)

        발문text2 = CTkTextbox(option_frame, wrap="word", font=(self.default_font, 13), fg_color="#FFFFF8", border_width=1, corner_radius=10)  # Adjust font and size as needed
        발문text2.grid(row=0, column=1, sticky="nsew", padx=20, pady=(20, 10))
        발문text2.delete("1.0", END)  # Clear the textbox
        발문text2.insert("1.0", combined_text_phrases)
        

        발문text1_label = CTkLabel(option_frame, height=30, text="위 어구로 시작하는 단락 삭제", text_color="black", font=(self.default_font, 13))
        발문text1_label.grid(row=1, column=0, sticky="ew", padx=(10, 5), pady=(0, 10))

        발문text2_label = CTkLabel(option_frame, height=30, text="위 어구를 포함하는 단락 삭제", text_color="black", font=(self.default_font, 13))
        발문text2_label.grid(row=1, column=1, sticky="ew", padx=(5, 10), pady=(0, 10))


        button_frame = CTkFrame(self.options_popup, fg_color="#2B3D2D", width=300, height=60, corner_radius=0)
        button_frame.grid(row=2, column=0, sticky="ews", padx=0, pady=0)
        button_frame.grid_rowconfigure(0, weight=1)
        button_frame.grid_columnconfigure(0, weight=1)
        button_frame.grid_columnconfigure(1, weight=1)



        def submit(event=None):
            self.check08.select()

            # Extract content from the textboxes
            extracted_prefixes_text = 발문text1.get("1.0", "end-1c")  # Fetch content
            extracted_phrases_text = 발문text2.get("1.0", "end-1c")  # Fetch content

            # Split the content into lists
            extracted_unwanted_prefixes = extracted_prefixes_text.split('\n')
            extracted_unwanted_phrases = extracted_phrases_text.split('\n')

            # Optionally, filter out any empty strings if blank lines were entered
            self.unwanted_prefixes = [item for item in extracted_unwanted_prefixes if item.strip()]
            self.unwanted_phrases = [item for item in extracted_unwanted_phrases if item.strip()]

            self.options_popup.destroy()

        def restore_default(event=None):
            self.unwanted_prefixes = ["다음 글", "주어진 글", "밑줄", "다음 빈칸", "글의 흐름", "윗 글", "윗글", "다음 밑줄", "→"]
            self.unwanted_phrases = ["내용과 일치하는 것은", "내용과 일치하지 않는 것은"]            
            combined_text_prefixes = "\n".join(self.unwanted_prefixes)
            combined_text_phrases = "\n".join(self.unwanted_phrases)
            발문text1.delete("1.0", END)  # Clear the textbox
            발문text1.insert("1.0", combined_text_prefixes)
            발문text2.delete("1.0", END)  # Clear the textbox
            발문text2.insert("1.0", combined_text_phrases)

        def close_popup(event=None):
            self.options_popup.destroy()
        default_button = CTkButton(button_frame, width=70, height=25, text="기본값", fg_color="#FEF9E0", hover_color="#DDA15C", text_color='black', font=(self.default_font, 13, 'bold'), corner_radius=5, command=restore_default)
        default_button.grid(row=0, column=0, padx=20, pady=10)

        ok_button = CTkButton(button_frame, width=70, height=25, text="확인", fg_color="#FEF9E0", hover_color="#DDA15C", text_color='black', font=(self.default_font, 13, 'bold'), corner_radius=5, command=submit)
        ok_button.grid(row=0, column=1, padx=20, pady=10)

        # Check the operating system to determine the correct modifier key
        if sys.platform.startswith('darwin'):
            # Mac OS
            modifier_key = 'Command'
        elif sys.platform.startswith('win'):
            # Windows
            modifier_key = 'Control'
        else:
            # Linux or other; adjust as needed
            modifier_key = 'Control'
            
        self.options_popup.bind(f'<{modifier_key}-Return>', submit)
        # Bind 'Escape' to cancel_edit as before
        self.options_popup.bind("<Escape>", close_popup)


    def create_options_popup_문장합치기(self):
        # Create the popup window as a top-level window
        self.options_popup = CTkToplevel(self.window)
        self.options_popup.title("Options")
        self.options_popup.geometry("500x350")  # Adjust the window size as needed
        self.options_popup.resizable(False, False)
        self.options_popup.grab_set()
        self.options_popup.focus_set()
        self.options_popup.grid_rowconfigure(1, weight=1)
        self.options_popup.grid_columnconfigure(0, weight=1)


        title_frame = CTkFrame(self.options_popup, fg_color="#EFF3F0", width=500, height=30)
        title_frame.grid(row=0, column=0, sticky="ewn", padx=0, pady=0)
        title_frame.grid_rowconfigure(0, weight=1)
        title_frame.grid_columnconfigure(0, weight=1)        
        label_option = CTkLabel(title_frame, text="제목으로 인식할 문자열", font=(self.default_font, 15, "bold"), text_color="black", fg_color="transparent")
        label_option.grid(row=0, column=0, padx=10, pady=5, sticky='ewn')


        option_frame = CTkFrame(self.options_popup, fg_color="#EFF3F0", width=500, height=310)
        option_frame.grid(row=1, column=0, sticky="nsew", padx=0, pady=0)
        option_frame.grid_rowconfigure(0, weight=1)
        option_frame.grid_columnconfigure(0, weight=1)


        self.제목text = CTkTextbox(option_frame, wrap="word", font=(self.default_font, 13), text_color="black", fg_color="#FFFFF8", border_width=1, corner_radius=10)  # Adjust font and size as needed
        self.제목text.grid(row=0, column=0, sticky="nsew", padx=20, pady=5)

        self.제목text.delete("1.0", "end")
        self.제목text.insert("1.0", self.제목인식)
        self.제목text.focus_set()

        제목text_label = CTkLabel(option_frame, height=30, text="위 문자열이 나올 때마다 하나의 문제로 인식합니다.\n(여러 항목인 경우 엔터로 구분)", text_color="black", font=(self.default_font, 13))
        제목text_label.grid(row=1, column=0, sticky="ew", padx=5, pady=(5, 10))


        button_frame = CTkFrame(self.options_popup, fg_color="#2B3D2D", width=300, height=60, corner_radius=0)
        button_frame.grid(row=2, column=0, sticky="ews", padx=0, pady=0)
        button_frame.grid_rowconfigure(0, weight=1)
        button_frame.grid_columnconfigure(0, weight=1)
        


        def submit(event=None):
            self.edit_check2.select()
            # Extract content from the textboxes
            self.제목인식 = self.제목text.get("1.0", "end-1c")  # Fetch content
           
            self.options_popup.destroy()


        def close_popup(event=None):
            self.options_popup.destroy()
        #default_button = CTkButton(button_frame, width=70, height=25, text="기본값", fg_color="#FEF9E0", hover_color="#DDA15C", text_color='black', font=(self.default_font, 13, 'bold'), corner_radius=5, command=restore_default)
        #default_button.grid(row=0, column=0, padx=20, pady=10)

        ok_button = CTkButton(button_frame, width=70, height=25, text="확인", fg_color="#FEF9E0", hover_color="#DDA15C", text_color='black', font=(self.default_font, 13, 'bold'), corner_radius=5, command=submit)
        ok_button.grid(row=0, column=0, padx=20, pady=10)

        # Check the operating system to determine the correct modifier key
        if sys.platform.startswith('darwin'):
            # Mac OS
            modifier_key = 'Command'
        elif sys.platform.startswith('win'):
            # Windows
            modifier_key = 'Control'
        else:
            # Linux or other; adjust as needed
            modifier_key = 'Control'
            
        self.options_popup.bind(f'<{modifier_key}-Return>', submit)
        # Bind 'Escape' to cancel_edit as before
        self.options_popup.bind("<Escape>", close_popup)

        

    def create_options_popup_정답채우기(self):
        
        # Create the popup window as a top-level window
        self.options_popup = CTkToplevel(self.window)
        self.options_popup.title("Options")
        self.options_popup.geometry("500x450")  # Adjust the window size as needed
        self.options_popup.resizable(False, False)
        self.options_popup.grab_set()
        self.options_popup.focus_set()
        self.options_popup.grid_rowconfigure(1, weight=1)
        self.options_popup.grid_columnconfigure(0, weight=1)


        title_frame = CTkFrame(self.options_popup, fg_color="#EFF3F0", width=500, height=30)
        title_frame.grid(row=0, column=0, sticky="ewn", padx=0, pady=0)
        title_frame.grid_rowconfigure(0, weight=1)
        title_frame.grid_columnconfigure(0, weight=1)        
        label_option = CTkLabel(title_frame, text="제목으로 인식할 문자열", font=(self.default_font, 15, "bold"), text_color="black", fg_color="transparent")
        label_option.grid(row=0, column=0, padx=10, pady=5, sticky='ewn')


        option_frame = CTkFrame(self.options_popup, fg_color="#EFF3F0", width=500, height=310)
        option_frame.grid(row=1, column=0, sticky="nsew", padx=0, pady=0)
        option_frame.grid_rowconfigure(0, weight=1)
        option_frame.grid_columnconfigure(0, weight=1)


        self.제목text = CTkTextbox(option_frame, wrap="word", font=(self.default_font, 13), text_color="black", fg_color="#FFFFF8", border_width=1, corner_radius=10)  # Adjust font and size as needed
        self.제목text.grid(row=0, column=0, sticky="nsew", padx=20, pady=5)

        self.제목text.delete("1.0", "end")
        self.제목text.insert("1.0", self.제목인식)
        self.제목text.focus_set()


        제목text_label = CTkLabel(option_frame, height=30, text="위 문자열로 시작할 때마다 하나의 문제로 인식합니다.\n(여러 항목은 엔터로 구분)", text_color="black", font=(self.default_font, 13))
        제목text_label.grid(row=1, column=0, sticky="ew", padx=5, pady=(5, 10))

        설명내용 = """[사용방법] 
사용법: 문제 하단에 정답을 아래와 같은 형식으로 적어주세요.
ex) 정답 3

"어법/어휘" 유형의 경우 지문의 밑줄친 단어에 대괄호를 씌운 후 (ex: ①[increased])
정답 아래에 "기존단어 > 바꿀단어" 형태로 적어주세요.
ex) 정답 3
cold > hot

주의사항 : 빈칸추론의 경우 지문에 "___" (언더바 3개 이상) 혹은\n "    " (공백 4칸 이상)이 있어야 빈칸으로 인식합니다. (따옴표는 제외)
"""
        설명 = CTkLabel(option_frame, width=500, text=설명내용, font=(self.default_font, 12), text_color="black", fg_color="#EFF3F0", justify=LEFT)  # Adjust font and size as needed
        설명.grid(row=2, column=0, sticky="ew", padx=20, pady=5)





        button_frame = CTkFrame(self.options_popup, fg_color="#2B3D2D", width=300, height=60, corner_radius=0)
        button_frame.grid(row=2, column=0, sticky="ews", padx=0, pady=0)
        button_frame.grid_rowconfigure(0, weight=1)
        button_frame.grid_columnconfigure(0, weight=1)
        


        def submit(event=None):
            self.edit_check3.select()
            # Extract content from the textboxes
            self.제목인식 = self.제목text.get("1.0", "end-1c")  # Fetch content
           
            self.options_popup.destroy()


        def close_popup(event=None):
            self.options_popup.destroy()
        #default_button = CTkButton(button_frame, width=70, height=25, text="기본값", fg_color="#FEF9E0", hover_color="#DDA15C", text_color='black', font=(self.default_font, 13, 'bold'), corner_radius=5, command=restore_default)
        #default_button.grid(row=0, column=0, padx=20, pady=10)

        ok_button = CTkButton(button_frame, width=70, height=25, text="확인", fg_color="#FEF9E0", hover_color="#DDA15C", text_color='black', font=(self.default_font, 13, 'bold'), corner_radius=5, command=submit)
        ok_button.grid(row=0, column=0, padx=20, pady=10)

        # Check the operating system to determine the correct modifier key
        if sys.platform.startswith('darwin'):
            # Mac OS
            modifier_key = 'Command'
        elif sys.platform.startswith('win'):
            # Windows
            modifier_key = 'Control'
        else:
            # Linux or other; adjust as needed
            modifier_key = 'Control'
            
        self.options_popup.bind(f'<{modifier_key}-Return>', submit)
        # Bind 'Escape' to cancel_edit as before
        self.options_popup.bind("<Escape>", close_popup)

        


    def on_entry_click(self, event):
        if self.edit_box.get("1.0", "end-1c").strip() == "Write here.":
            self.edit_box.delete("1.0", "end")
            self.edit_box.configure(text_color=('black'), font=(self.default_font, 12))  # Change text color back to normal


    def on_entry_focusout(self, event):
        if not self.edit_box.get("1.0", "end-1c").strip():
            self.edit_box.insert("1.0", "Write here.")
            self.edit_box.configure(text_color='grey', font=(self.default_font, 25))  # Change text color back to normal



    def create_radio_button(self, x, y, value):
        # Assign a specific value to each radio button
        radio_button = tk.Radiobutton(self.canvas, variable=self.radio_value, value=value,
                                      bg="#2B3D2D", activebackground="#2B3D2D", borderwidth=0)
        self.canvas.create_window(x, y, window=radio_button, anchor="nw")


    def create_buttons(self):
        # Assuming you have a method relative_to_assets defined and it's working correctly
        button_images = ["button_3.png", "button_4.png"]
        commands = [self.process_and_update_tree_view, self.append_content_to_main_treeview]
        positions = [(931.0, 57.0), (1011.0, 57.0)]
        # positions = [(931.0, 57.0), (1011.0, 57.0), (931.0, 645.0), (1011.0, 645.0)]
        for i, img in enumerate(button_images):
            photo_image = PhotoImage(file=self.relative_to_assets(img))
            button = Button(
                self.window,
                image=photo_image,
                borderwidth=0,
                highlightthickness=0,
                command=commands[i],
                relief="flat"
            )
            button.image = photo_image
            button.place(x=positions[i][0], y=positions[i][1], width=66.0, height=28.0)
        


    def normalize_quotations(self, line):
        return line.replace('“', '"').replace('”', '"').replace('‘', "'").replace('’', "'").replace('`', "'")


    def get_allowed_prefixes(self):
        # Get prefixes from 삭제예외_entry
        삭제예외_prefixes = self.삭제예외_entry.get("1.0", "end-1c").strip().split('\n')
        삭제예외_prefixes = [prefix.strip() for prefix in 삭제예외_prefixes if prefix.strip()]
        
        # Get prefixes from 지문번호수동지정_entry
        지문번호수동지정_prefixes = self.지문번호수동지정_entry.get("1.0", "end-1c").strip().split('\n')
        지문번호수동지정_prefixes = [prefix.strip() for prefix in 지문번호수동지정_prefixes if prefix.strip()]
        
        if self.지문번호규칙radio_value.get() == 2: ## 수동인식 체크 되어있을 경우
            # Combine both lists of prefixes
            allowed_prefixes = 삭제예외_prefixes + 지문번호수동지정_prefixes
        else:
            allowed_prefixes = 삭제예외_prefixes

        return allowed_prefixes

    def is_korean_unless_allowed(self, line, allowed_prefixes):
        korean_start_pattern = re.compile(r'^[\uAC00-\uD7AF]')
        return korean_start_pattern.match(line) and not any(line.startswith(prefix) for prefix in allowed_prefixes)


    def process_and_update_tree_view(self):
        content = self.edit_box.get("1.0", "end-1c")
        content = (content
            .replace("\u2028", "\n")
            .replace("\u2029", "\n")
            .replace("\f", "")
            .replace("\u2013", "-")
            .replace("\u200B", "")   # zero-width space
            .replace("\u200C", "")   # zero-width non-joiner
            .replace("\u200D", "")   # zero-width joiner
            .replace("\uFEFF", "")   # byte order mark
        )

        #print(f"\ncontent:{content}")

        lines = content.split('\n')


        # Normalize quotation marks and remove lines with unwanted prefixes or that start with "*"
        #unwanted_prefixes = ["다음 글", "주어진 글", "밑줄", "다음 빈칸", "글의 흐름", "윗 글", "윗글", "→"]
        unwanted_prefixes = ["→"]
        custom_unwanted_prefixes = self.추가삭제옵션_entry.get("1.0", "end-1c").strip().split('\n')
        all_unwanted_prefixes = unwanted_prefixes + [prefix for prefix in custom_unwanted_prefixes if prefix.strip()]

        def is_line_unwanted(line):
            return (line.strip() and not any(line.strip().startswith(prefix) for prefix in all_unwanted_prefixes))

        lines = [self.normalize_quotations(line) for line in lines if is_line_unwanted(line)]

        # Optionally remove lines starting with a Korean character
        if self.한글삭제checkbox.get() == 1:
            allowed_prefixes = self.get_allowed_prefixes()
            lines = [line for line in lines if not self.is_korean_unless_allowed(line, allowed_prefixes)]



        # Check if the first radio button is selected
        if self.radio_value.get() == 1:  # Assuming 1 is the value for the first radio button
            titles_and_paragraphs = []
            current_title = None
            current_paragraphs = []

            # Fetch custom title prefixes if 지문번호수동지정 is enabled
            if self.지문번호규칙radio_value.get() == 2:
                custom_title_prefixes = self.지문번호수동지정_entry.get("1.0", "end-1c").strip().split('\n')
                custom_title_prefixes = [prefix.strip() for prefix in custom_title_prefixes if prefix.strip()]
            elif self.지문번호규칙radio_value.get() == 1:
                custom_title_prefixes = []

            for line in lines:
                # Determine if the line is a title based on the custom prefixes, if specified
                if custom_title_prefixes and any(line.startswith(prefix) for prefix in custom_title_prefixes):
                    is_title_line = True
                elif not custom_title_prefixes:
                    is_title_line = self.is_title(line)
                else:
                    is_title_line = False

                if is_title_line:
                    line = line.replace('\t', '')  # Remove tab characters from the title
                    if current_title is not None:
                        # Append the current title and paragraphs as a new entry
                        titles_and_paragraphs.append((current_title, '\n'.join(current_paragraphs)))
                        current_paragraphs = []
                    current_title = line
                else:
                    current_paragraphs.append(line)

            # Don't forget to add the last title and paragraphs
            if current_title and current_paragraphs:
                titles_and_paragraphs.append((current_title, '\n'.join(current_paragraphs)))

            ######## 트리뷰 미리보기에 입력
            # After attempting to populate titles_and_paragraphs
            if not titles_and_paragraphs:
                messagebox.showinfo("포맷 오류", "'지문 번호'가 인식되지 않았습니다. 데이터의 포맷과 지문 번호 규칙을 확인하세요.")
                return  # Optionally return early if you don't want to proceed further
            # Clear existing entries in the TreeView
            for item in self.tree_view.get_children():
                self.tree_view.delete(item)
            # Update tree_view
            for title, paragraph in titles_and_paragraphs:
                # Process each line to remove leading or trailing whitespaces, then join them back with line breaks
                cleaned_paragraph = "\n".join([line.strip() for line in paragraph.split("\n")])
                self.tree_view.insert('', 'end', values=(title, cleaned_paragraph))





        elif self.radio_value.get() == 2:
            title_count_map = {}  # This map keeps track of the count of paragraphs under each title
            titles_and_paragraphs = []
            current_title = ""
            current_paragraphs = []


            # Fetch custom title prefixes from 지문번호수동지정_entry
            custom_title_prefixes = self.지문번호수동지정_entry.get("1.0", "end-1c").strip().split('\n')
            custom_title_prefixes = [prefix.strip() for prefix in custom_title_prefixes if prefix.strip()]
                    
            # Define a new is_title function specific for this scenario
            def is_custom_title(line, prefixes):
                return any(line.startswith(prefix) for prefix in prefixes)
            

            if self.지문번호규칙radio_value.get() == 1: #자동인식

                # First pass to count paragraphs per title
                for line in lines:
                    if self.is_title(line):
                        current_title = line
                        if current_title not in title_count_map:
                            title_count_map[current_title] = 0
                    elif current_title:
                        title_count_map[current_title] += 1

                # Reset for the second pass
                current_title = ""
                paragraph_counter = 1

                # Second pass to append paragraphs with modified titles to titles_and_paragraphs
                for line in lines:
                    if self.is_title(line):
                        current_title = line
                        paragraph_counter = 1  # Reset counter for the new title
                    else:
                        # Append number only if there's more than one paragraph under the title
                        if not current_title:
                            messagebox.showinfo("포맷 오류", "첫 번째 글의 '지문 번호'가 인식되지 않았습니다. 데이터의 포맷과 지문 번호 규칙을 확인하세요.")
                            return  # Optionally return early if you don't want to proceed further
                        else:
                            title_with_count = f"{current_title} ({paragraph_counter})" if title_count_map[current_title] > 1 else current_title
                            titles_and_paragraphs.append((title_with_count, line.strip()))
                            paragraph_counter += 1

            elif self.지문번호규칙radio_value.get() == 2: #수동인식

                # First pass to count paragraphs per title
                for line in lines:

                    if is_custom_title(line, custom_title_prefixes):                    
                        current_title = line
                        if current_title not in title_count_map:
                            title_count_map[current_title] = 0
                    elif current_title:
                        title_count_map[current_title] += 1

                # Reset for the second pass
                current_title = ""
                paragraph_counter = 1

                # Second pass to append paragraphs with modified titles to titles_and_paragraphs
                for line in lines:
                    if is_custom_title(line, custom_title_prefixes):                    
                        current_title = line
                        paragraph_counter = 1  # Reset counter for the new title
                    else:
                        # Append number only if there's more than one paragraph under the title
                        title_with_count = f"{current_title} ({paragraph_counter})" if title_count_map[current_title] > 1 else current_title
                        titles_and_paragraphs.append((title_with_count, line.strip()))
                        paragraph_counter += 1


            ######## 트리뷰 미리보기에 입력
            # After attempting to populate titles_and_paragraphs
            if not titles_and_paragraphs:
                messagebox.showinfo("포맷 오류", "'지문 번호'가 인식되지 않았습니다. 데이터의 포맷과 지문 번호 규칙을 확인하세요.")
                return  # Optionally return early if you don't want to proceed further
            # Clear existing entries in the TreeView
            for item in self.tree_view.get_children():
                self.tree_view.delete(item)        
            # Update tree_view
            for title, paragraph in titles_and_paragraphs:
                # Process each line to remove leading or trailing whitespaces, then join them back with line breaks
                cleaned_paragraph = "\n".join([line.strip() for line in paragraph.split("\n")])
                self.tree_view.insert('', 'end', values=(title, cleaned_paragraph))



        elif self.radio_value.get() == 3:
            # Fetch custom title prefixes from 지문번호수동지정_entry
            custom_title_prefixes = self.지문번호수동지정_entry.get("1.0", "end-1c").strip().split('\n')
            custom_title_prefixes = [prefix.strip() for prefix in custom_title_prefixes if prefix.strip()]
                   
            # Define a new is_title function specific for this scenario
            def is_custom_title(line, prefixes):
                return any(line.startswith(prefix) for prefix in prefixes)


            found_content = False  # Initialize the flag to track whether content is found
            newData = []
            counterData = []  # To store paragraph counts for titles
            passageCounter = 0
            i = 0

            try:
                max_word_count = int(self.분류3_entry.get())
            except ValueError:
                max_word_count = 150


            while i < len(lines):
                currentText = lines[i]


                if self.지문번호규칙radio_value.get() == 1: #자동인식


                    if self.is_title(currentText):
                        found_content = True  # Valid title found
                        newData.append((currentText, ''))
                        counterData.append('')
                        passageCounter = 0  # Reset for the new title

                        peek = i + 1
                        passagesBeforeNextTitle = 0
                        while peek < len(lines) and not self.is_title(lines[peek]):
                            passagesBeforeNextTitle += 1
                            peek += 1

                        if passagesBeforeNextTitle == 1:
                            # If there is only one paragraph under this title, add it directly
                            newData.append((currentText, lines[i + 1]))
                            counterData.append('')
                            i += 2  # Skip the next line as it has been processed
                            continue  # Move to the next iteration

                        i += 1  # Move to processing paragraphs under the title
                        continue

                    startIndex = passageCounter + 1
                    currentChunk = [currentText]
                    passageCounter += 1

                    # Combine paragraphs until the word count exceeds 120 or a new title is encountered
                    while i < len(lines) - 1 and sum(len(chunk.split()) for chunk in currentChunk) < max_word_count and not self.is_title(lines[i + 1]):
                        i += 1
                        currentChunk.append(lines[i])
                        passageCounter += 1

                    endIndex = passageCounter
                    chunkText = "\n".join(currentChunk)

                    if startIndex == endIndex:
                        counterData.append(f'({startIndex})')
                    else:
                        counterData.append(f'({startIndex}-{endIndex})')

                    newData.append((None, chunkText))  # Use None for title to indicate continuation
                    i += 1


                if self.지문번호규칙radio_value.get() == 2: #수동인식

                    if is_custom_title(currentText, custom_title_prefixes):
                        found_content = True  # Valid title found
                        newData.append((currentText, ''))
                        counterData.append('')
                        passageCounter = 0  # Reset for the new title

                        peek = i + 1
                        passagesBeforeNextTitle = 0
                        while peek < len(lines) and not is_custom_title(lines[peek], custom_title_prefixes):
                            passagesBeforeNextTitle += 1
                            peek += 1

                        if passagesBeforeNextTitle == 1:
                            # If there is only one paragraph under this title, add it directly
                            newData.append((currentText, lines[i + 1]))
                            counterData.append('')
                            i += 2  # Skip the next line as it has been processed
                            continue  # Move to the next iteration

                        i += 1  # Move to processing paragraphs under the title
                        continue

                    startIndex = passageCounter + 1
                    currentChunk = [currentText]
                    passageCounter += 1

                    # Combine paragraphs until the word count exceeds 120 or a new title is encountered
                    while i < len(lines) - 1 and sum(len(chunk.split()) for chunk in currentChunk) < max_word_count and not is_custom_title(lines[i + 1], custom_title_prefixes):
                        i += 1
                        currentChunk.append(lines[i])
                        passageCounter += 1

                    endIndex = passageCounter
                    chunkText = "\n".join(currentChunk)

                    if startIndex == endIndex:
                        counterData.append(f'({startIndex})')
                    else:
                        counterData.append(f'({startIndex}-{endIndex})')

                    newData.append((None, chunkText))  # Use None for title to indicate continuation
                    i += 1




            # Merge titles with their paragraphs and apply numbering
            finalData = []
            currentTitle = None
            for (title, chunk), numberText in zip(newData, counterData):

                if title:
                    currentTitle = title
                if chunk:

                    if not currentTitle:
                        messagebox.showinfo("포맷 오류", "첫 번째 글의 '지문 번호'가 인식되지 않았습니다. 데이터의 포맷과 지문 번호 규칙을 확인하세요.")
                        return  # Optionally return early if you don't want to proceed further


                    numberedTitle = f"{currentTitle} {numberText}".strip()
                    finalData.append((numberedTitle, chunk))


            
            if not found_content:
                messagebox.showinfo("포맷 오류", "'지문 번호'가 인식되지 않았습니다. 데이터의 포맷과 지문 번호 규칙을 확인하세요.")
                return  # Exit the function early if no content was found


            # Clear existing entries in the TreeView
            self.tree_view.delete(*self.tree_view.get_children())

            # Update TreeView with the processed and numbered paragraphs
            for title, paragraph in finalData:
                cleaned_paragraph = "\n".join([line.strip() for line in paragraph.split("\n")])  # Keep line breaks, remove leading/trailing spaces
                self.tree_view.insert('', 'end', values=(title, cleaned_paragraph))


        self.on_treeview_change()

        # Validate all rows and highlight errors in red
        self.validate_and_highlight_errors()


    def validate_and_highlight_errors(self):
        """Validate tree_view rows and highlight errors in red"""
        # Configure error tag for red text
        self.tree_view.tag_configure('error', foreground='red')

        for child in self.tree_view.get_children():
            values = self.tree_view.item(child, "values")
            has_error = False

            # Get passage_id and passage_text
            passage_id = values[0].strip() if len(values) > 0 and values[0] else ""
            passage_text = values[1].strip() if len(values) > 1 and values[1] else ""

            # Check for mismatched data (one column filled, other empty)
            if passage_id and not passage_text:
                has_error = True
            elif passage_text and not passage_id:
                has_error = True

            # Validate word count in the second column (passage text)
            if passage_text:
                word_count = len(passage_text.split())

                if word_count < 50 or word_count > 700:
                    has_error = True

            # Mark row with error tag if it has validation errors
            if has_error:
                self.tree_view.item(child, tags=('error',))
            else:
                # Clear error tag if previously set
                self.tree_view.item(child, tags=())


    def is_title(self, line):

        disallowed_endings = []
        if self.checkbox_1.get() == 1:
            disallowed_endings.append('.')
        if self.checkbox_2.get() == 1:
            disallowed_endings.append('!')
        if self.checkbox_3.get() == 1:
            disallowed_endings.append('?')

        ends_with_disallowed = any(line.strip().endswith(symbol) for symbol in disallowed_endings)
        word_count = len(line.split())

        try:
            max_word_count = int(self.지문번호규칙_entry.get())
        except ValueError:
            max_word_count = 10
        #print(max_word_count)
        return not ends_with_disallowed and word_count < max_word_count


    #"대기열로 (저장)" 클릭 시
    def append_content_to_main_treeview(self, event=None):
        #print("콜")
        # Access globals through main_frame or module import
        import autoQM

        # Check if 'self.tree_view' is empty
        if not self.tree_view.get_children():
            messagebox.showinfo("Information", "'대기열 미리보기'가 비어 있습니다.")
            return  # Exit the function early

        # Use the excel_file passed during initialization
        excel_file = self.current_excel_file

        if not excel_file:
            # No file is open, so prompt to create/save a new file
            self.EditorApp_새파일저장묻기()  # Call the function
            # Re-fetch excel_file from autoQM globals after potentially creating a new file
            excel_file = self.get_excel_file()
            file_saved = self.get_file_saved()

            # Check if excel_file is still None after the function call
            if not excel_file or not file_saved:
                messagebox.showerror("Error", "파일이 선택되지 않았습니다. 파일을 먼저 생성하거나 열어주세요.")
                self.window.destroy()
                self.main_frame.editor_app = None  # Clear reference for guard check
                return

            # Update our instance variable
            self.current_excel_file = excel_file

        # At this point, excel_file should be valid (either already existed or just created)
        sheet_name = autoQM.sheet_name
        unsaved_changes = autoQM.unsaved_changes

        # Proceed with saving to the existing or newly created file
        self.update_sheet_selector(excel_file) ### 이게 시트셀렉터(콤보박스) 이름을 대기열로 바꿔줌. ####???? 이거 없어도 되나
        sheet_name = "대기열"
        self.load_excel_into_treeview(excel_file, sheet_name) ### 실제 대기열 내용을 불러옴.

        # First, remove rows with "" in both columns A and B
        for child in self.main_treeview.get_children():
            values = self.main_treeview.item(child, "values")
            # Check if both column A and B are empty strings
            if all(value == "" for value in values):
                self.main_treeview.delete(child)

        # Validate all rows first and collect errors
        validation_errors = []

        for row_idx, child in enumerate(self.tree_view.get_children(), start=1):
            values = self.tree_view.item(child, "values")

            # Get passage_id and passage_text
            passage_id = values[0].strip() if len(values) > 0 and values[0] else ""
            passage_text = values[1].strip() if len(values) > 1 and values[1] else ""

            # Check for mismatched data (one column filled, other empty)
            if passage_id and not passage_text:
                validation_errors.append(f"• 행 {row_idx}: 지문번호 '{passage_id}'에 지문 내용이 없습니다.")
            elif passage_text and not passage_id:
                validation_errors.append(f"• 행 {row_idx}: 지문 내용은 있지만 지문번호가 없습니다.")

            # Validate word count in the second column (passage text)
            if passage_text:
                word_count = len(passage_text.split())
                if self.username != 'emong111':
                    if word_count < 50:
                        validation_errors.append(f"• 행 {row_idx}: 지문번호 '{passage_id}'의 지문이 너무 짧습니다 (현재 {word_count}단어, 50단어 이상 필요)")
                    elif word_count > 700:
                        validation_errors.append(f"• 행 {row_idx}: 지문번호 '{passage_id}'의 지문이 너무 깁니다 (현재 {word_count}단어, 700단어 이하 필요)")

        # If there are validation errors, show them all and return
        if validation_errors:
            error_message = "다음 오류를 수정해주세요:\n\n" + "\n".join(validation_errors)
            messagebox.showerror("오류", error_message)
            return

        # Now, append new content from the popup's tree_view to the main window's treeview
        for child in self.tree_view.get_children():
            values = self.tree_view.item(child, "values")
            self.main_treeview.insert('', 'end', values=values)

        #unsaved_changes = True
        self.save_treeview_to_excel(excel_file, sheet_name)
        self.EditorApp에서트리뷰저장(excel_file, sheet_name)

        excel_file = unicodedata.normalize('NFC', excel_file)

        # Update the global excel_file in autoQM module
        import autoQM
        autoQM.excel_file = excel_file

        app_logger.info(f"★[알림] 저장 성공\n(파일 경로: {excel_file})\n삭제/덮어쓰기 주의!\n★Make Questions를 클릭하여 출제를 시작하세요.\n")
        #messagebox.showinfo("Success", f"파일을 '{excel_file}'에 성공적으로 저장하였습니다.")
        self.main_frame.filepath_label.configure(text=f"위에서 특정 행을 더블클릭하여 지문을 입력하거나 내용을 편집할 수 있습니다. 아래 엑셀 파일을 직접 열어보면 오류가 발생할 수 있습니다.\n현재 파일 경로 (클릭 시 이동): {excel_file}")

        # Add the file to recent files list
        self.main_frame.add_to_recent_files(excel_file)

        # Update button states (enable Make Questions and Export Questions buttons)
        self.main_frame.update_run_button_state()

        self.window.destroy()  # Close the popup
        self.main_frame.editor_app = None  # Clear reference for guard check



    def EditorApp에서트리뷰저장(self, excel_file, sheet_name):
        # Access global through module import
        import autoQM
        if not excel_file or not sheet_name:
            return
        else:
            wb = load_workbook(filename=excel_file)
            sheet = wb[sheet_name]

            # Clear existing data in the sheet
            for row in sheet['A1':f'{get_column_letter(sheet.max_column)}{sheet.max_row}']:
                for cell in row:
                    cell.value = None

            # Write data from treeview to sheet
            for i, item in enumerate(self.main_treeview.get_children(), start=1):  # Start from row 1
                row_values = self.main_treeview.item(item, "values")
                # Ensure you write all five columns back to the sheet
                for j, value in enumerate(row_values, start=1):  # Start from column 1 and include up to column 5
                    sheet.cell(row=i, column=j, value=value.strip())

            wb.save(filename=excel_file)

            for index, item in enumerate(self.main_treeview.get_children()):
                if index % 2 == 0:
                    self.main_treeview.item(item, tags=('evenrow',))
                else:
                    self.main_treeview.item(item, tags=('oddrow',))

            # After inserting rows into the Treeview, configure tag colors
            self.main_treeview.tag_configure('evenrow', background='#F5F5F5')
            self.main_treeview.tag_configure('oddrow', background='#FAFAFA')

            autoQM.unsaved_changes = False
            #load_excel_into_treeview(excel_file, sheet_name)
            self.main_frame.update_run_button_state()
            

        



    def relative_to_assets(self, path: str) -> Path:
        output_path = base_path
        assets_path = output_path / "assets" / "frame_new"
        return assets_path / Path(path)

    def button_clicked(self, img):
        pass

    def on_popup_close(self, event=None):
        self.main_frame.rebind_all_menubutton_tooltips()
        self.window.destroy()  # Close the popup
        self.main_frame.editor_app = None  # Clear reference for guard check










########################################## 로그인 창 GUI ##################################################







