#modules/excel_cache.py
"""
Excel file download cache for desktop app.
Prevents repeated downloads of the same file within a short time window.
"""

import time
import requests
from io import BytesIO
from threading import Lock

# Cache configuration
CACHE_TTL_SECONDS = 60  # Cache files for 60 seconds
MAX_CACHE_SIZE = 10  # Maximum number of files to cache

# Thread-safe cache storage
_cache = {}  # {cache_key: (BytesIO_data, timestamp)}
_cache_lock = Lock()


def _make_cache_key(url: str, headers: dict = None, params: dict = None) -> str:
    """Create a unique cache key from request parameters."""
    # Include URL and any params that affect the response
    key_parts = [url]
    if params:
        # Sort params for consistent key generation
        sorted_params = sorted(params.items())
        key_parts.append(str(sorted_params))
    return "|".join(key_parts)


def _cleanup_expired():
    """Remove expired entries from cache (called with lock held)."""
    now = time.time()
    expired_keys = [
        key for key, (_, timestamp) in _cache.items()
        if now - timestamp > CACHE_TTL_SECONDS
    ]
    for key in expired_keys:
        del _cache[key]


def _enforce_max_size():
    """Remove oldest entries if cache exceeds max size (called with lock held)."""
    if len(_cache) > MAX_CACHE_SIZE:
        # Sort by timestamp and remove oldest
        sorted_items = sorted(_cache.items(), key=lambda x: x[1][1])
        items_to_remove = len(_cache) - MAX_CACHE_SIZE
        for key, _ in sorted_items[:items_to_remove]:
            del _cache[key]


def get_excel_file_cached(url: str, headers: dict = None, params: dict = None, timeout: int = 30) -> tuple:
    """
    Fetch an Excel file with caching.

    Args:
        url: The URL to fetch
        headers: Optional request headers (e.g., Authorization)
        params: Optional query parameters
        timeout: Request timeout in seconds

    Returns:
        tuple: (BytesIO object with file content, status_code)
               Returns (None, status_code) on error
    """
    cache_key = _make_cache_key(url, headers, params)

    with _cache_lock:
        # Clean up expired entries
        _cleanup_expired()

        # Check cache
        if cache_key in _cache:
            cached_data, timestamp = _cache[cache_key]
            age = time.time() - timestamp
            print(f"[Cache HIT] {url} (age: {age:.1f}s)")
            # Return a new BytesIO with the same content (so it can be read multiple times)
            return BytesIO(cached_data), 200

    # Cache miss - fetch from server
    try:
        print(f"[Cache MISS] Fetching: {url}")
        response = requests.get(url, headers=headers, params=params, timeout=timeout)

        if response.status_code == 200:
            # Store in cache
            content = response.content
            with _cache_lock:
                _cache[cache_key] = (content, time.time())
                _enforce_max_size()
            print(f"[Cache STORE] {url} ({len(content)} bytes)")
            return BytesIO(content), 200
        else:
            print(f"[Cache ERROR] {url} returned status {response.status_code}")
            return None, response.status_code

    except requests.exceptions.RequestException as e:
        print(f"[Cache ERROR] Request failed for {url}: {e}")
        return None, 0


def clear_cache():
    """Clear all cached entries."""
    with _cache_lock:
        _cache.clear()
    print("[Cache] Cleared all entries")


def invalidate_file(filename: str):
    """
    Invalidate cache entries for a specific filename.
    Useful when you know a file has been updated.

    Args:
        filename: The filename to invalidate (will match any URL containing this filename)
    """
    with _cache_lock:
        keys_to_remove = [key for key in _cache.keys() if filename in key]
        for key in keys_to_remove:
            del _cache[key]
        if keys_to_remove:
            print(f"[Cache] Invalidated {len(keys_to_remove)} entries for '{filename}'")


def get_cache_stats() -> dict:
    """Get cache statistics for debugging."""
    with _cache_lock:
        now = time.time()
        entries = []
        for key, (data, timestamp) in _cache.items():
            entries.append({
                "key": key[:50] + "..." if len(key) > 50 else key,
                "size_bytes": len(data),
                "age_seconds": round(now - timestamp, 1)
            })
        return {
            "entry_count": len(_cache),
            "max_size": MAX_CACHE_SIZE,
            "ttl_seconds": CACHE_TTL_SECONDS,
            "entries": entries
        }
