#modules/export_questions.py
import tkinter as tk
from tkinter import filedialog, scrolledtext, messagebox, Toplevel, Text, Button, Scrollbar, END, ttk, BooleanVar, Tk, Canvas, Entry, PhotoImage, font
import customtkinter
from customtkinter import *
from CTkListbox import *
from modules.font_handler import font_handler
from modules.functions import ToolTip
from pathlib import Path
from pdf2image import convert_from_path
from PIL import Image, ImageDraw, ImageFont, ImageTk
import json
import math
import unicodedata
import shutil
import subprocess
import re
import os
import sys
from uuid import uuid4
from tkinter import colorchooser  # Import the colorchooser module
from pathlib import Path
import sys

# Determine project root (PyInstaller-compatible)
if getattr(sys, 'frozen', False):
    PROJECT_ROOT = Path(sys._MEIPASS)
else:
    PROJECT_ROOT = Path(__file__).resolve().parent.parent          # .../AutoQM

DEFAULT_BOOKCOVERS_DIR = PROJECT_ROOT / "assets" / "bookcovers"


class ScrollableImageFrame(customtkinter.CTkScrollableFrame):
    def __init__(self, master, image_paths, docx_folder, command=None, button_click_callback=None, 
                 first_button_callback=None, delete_callback=None, **kwargs):
        super().__init__(master, **kwargs)
        self.configure(fg_color="#B6C2B7")
        #num_columns = 5  # Assuming 5 columns for the grid
        #for col in range(num_columns):
        #    self.grid_columnconfigure(col, weight=1)  # Set weight=1 for all columns
        self.columns = 5
        self.cell_padx = 10
        self.cell_pady = 10

        for col in range(self.columns):
            self.grid_columnconfigure(col, minsize=150)
        font_handler.set_default_font()
        self.default_font = font_handler.default_font

        #self.grid_columnconfigure(0, weight=1)
        self.command = command
        self.image_paths = image_paths
        self.docx_folder = docx_folder        
        self.button_list = []                    
        self.button_click_callback = button_click_callback
        self.first_button_callback = first_button_callback

        self.load_images()
        self.delete_callback = delete_callback  # Add delete callback

        # Drag-and-drop state
        self.drag_enabled = False
        self.drag_data = {"button": None, "index": None, "offset": (10, 10), "original_fg": "transparent"}
        self.drag_hover_index = None
        self.drag_indicator = None
        self.drag_preview_window = None  # Floating preview window
        self.drag_bindings = {}



    def load_images(self):
        thumbnail_size = (140, 200)  # Thumbnail size
        # Sort the image paths by numeric prefix first, then alphabetically
        def sort_key(path):
            stem = Path(path).stem
            # Extract numeric prefix if exists (format: 001_, 002_, etc.)
            match = re.match(r'^(\d{3})_', stem)
            if match:
                return (int(match.group(1)), stem)
            else:
                return (999999, stem)  # Files without prefix go to the end

        sorted_image_paths = sorted(self.image_paths, key=sort_key)
        self.image_paths = sorted_image_paths

        for i, image_path in enumerate(sorted_image_paths):
            img_ctk = customtkinter.CTkImage(light_image=Image.open(image_path),
                                            dark_image=Image.open(image_path),
                                            size=thumbnail_size)
            docx_filename = Path(image_path).stem + ".docx"
            docx_path = self.docx_folder / docx_filename

            button = customtkinter.CTkButton(self, text="", image=img_ctk, width=thumbnail_size[0],
                                            height=thumbnail_size[1], corner_radius=10, fg_color="transparent", hover_color="#DDA15C")
            button.image = img_ctk  # Keep a reference
            button.grid(row=i // self.columns, column=i % self.columns, pady=(0, 10), padx=(10, 0))

            self.button_list.append(button)
            button.docx_path = docx_path  # Store the docx_path with the button for later reference

            # Initially set the first button's fg_color differently
            if i == 0:
                button.configure(fg_color="#BB6C25")
                # Assuming docx_path is stored in each button for reference
                #self.parent_window.last_clicked_button = str(docx_path)  # Set the default value for the first button
            else:
                button.configure(fg_color="transparent")

            # Create a separate method to configure the button and use it in the lambda
            def configure_button(path, btn=button):  # btn=button captures the current button
                return lambda: self.button_command(str(path), btn)

            button.configure(command=configure_button(docx_path))

            # Add right-click binding for delete
            button.bind("<Button-2>", lambda e, path=docx_path, img_path=image_path: 
                       self.on_right_click(e, path, img_path))
            button.bind("<Button-3>", lambda e, path=docx_path, img_path=image_path: 
                       self.on_right_click(e, path, img_path))



        if self.button_list:
            first_button = self.button_list[0]
            first_button_docx_path = first_button.docx_path
            # If you want to simulate a click on the first button
            if self.first_button_callback:
                
                self.first_button_callback(first_button_docx_path, first_button)




    def button_command(self, docx_path, clicked_button):
        if self.command:
            self.command(docx_path)
        if self.button_click_callback:
            self.button_click_callback(docx_path, clicked_button)  # Notify the parent window                    
        for button in self.button_list:
            button.configure(fg_color="transparent")  
        clicked_button.configure(fg_color="#BB6C25")  


    def on_right_click(self, event, docx_path, image_path):
        # Create context menu using Menu widget instead
        from tkinter import Menu
        
        context_menu = Menu(self, tearoff=0)
        context_menu.add_command(
            label="Delete Cover", 
            command=lambda: self.delete_book_cover_from_menu(docx_path, image_path)
        )
        
        # Display the menu
        try:
            context_menu.tk_popup(event.x_root, event.y_root)
        finally:
            context_menu.grab_release()

    def delete_book_cover_from_menu(self, docx_path, image_path):
        if self.delete_callback:
            self.delete_callback(docx_path, image_path)



    def delete_cover(self, docx_path, image_path, menu):
        menu.destroy()
        if self.delete_callback:
            self.delete_callback(docx_path, image_path)

    # ============ Drag-and-Drop Functionality ============

    def enable_drag_mode(self):
        """Enable drag-and-drop reordering for thumbnails"""
        if self.drag_enabled:
            return

        self.drag_enabled = True
        self.drag_bindings.clear()

        for button in self.button_list:
            button.configure(cursor="fleur")
            bindings = {}
            bindings["<Button-1>"] = button.bind(
                "<Button-1>",
                lambda event, btn=button: self._on_drag_start(event, btn),
                add="+"
            )
            bindings["<B1-Motion>"] = button.bind("<B1-Motion>", self._on_drag_motion, add="+")
            bindings["<ButtonRelease-1>"] = button.bind("<ButtonRelease-1>", self._on_drag_release, add="+")
            self.drag_bindings[button] = bindings

    def disable_drag_mode(self):
        """Disable drag-and-drop reordering"""
        if not self.drag_enabled:
            return

        self.drag_enabled = False

        for button, bindings in self.drag_bindings.items():
            button.configure(cursor="")
            for sequence, bind_id in bindings.items():
                if bind_id is not None:
                    button.unbind(sequence, bind_id)
        self.drag_bindings.clear()

        self._clear_drop_indicator()
        self._destroy_drag_preview()
        self.drag_data = {
            "button": None,
            "index": None,
            "offset": (10, 10),
            "original_fg": "transparent"
        }
        self.drag_hover_index = None

    def _on_drag_start(self, event, button):
        """Handle drag start"""
        if not self.drag_enabled:
            return "break"

        try:
            index = self.button_list.index(button)
        except ValueError:
            return "break"

        self.drag_data["button"] = button
        self.drag_data["index"] = index
        self.drag_data["offset"] = (
            event.x_root - button.winfo_rootx(),
            event.y_root - button.winfo_rooty()
        )
        try:
            self.drag_data["original_fg"] = button.cget("fg_color")
        except Exception:
            self.drag_data["original_fg"] = "transparent"

        # Add visual feedback - make button semi-transparent
        button.configure(fg_color="#8D9397")

        # Clear any existing indicator
        self._clear_drop_indicator()
        self.drag_hover_index = None

        # Create floating preview window
        self._create_drag_preview(event)

        return "break"

    def _on_drag_motion(self, event):
        """Handle drag motion - show drop target"""
        if not self.drag_enabled or self.drag_data["button"] is None:
            return "break"

        # Update floating preview position
        self._update_drag_preview(event)

        insert_index = self._calculate_drop_index(event.x_root, event.y_root)

        if insert_index is None:
            self._clear_drop_indicator()
            return "break"

        if insert_index != self.drag_hover_index:
            self.drag_hover_index = insert_index
            self._show_drop_indicator(insert_index)

        return "break"

    def _on_drag_release(self, event):
        """Handle drag release - perform reordering"""
        if not self.drag_enabled or self.drag_data["button"] is None:
            return "break"

        source_index = self.drag_data["index"]
        insert_index = self.drag_hover_index
        if insert_index is None:
            insert_index = self._calculate_drop_index(event.x_root, event.y_root)

        # Clear visual feedback
        original_fg = self.drag_data.get("original_fg", "transparent")
        try:
            self.drag_data["button"].configure(fg_color=original_fg)
        except Exception:
            self.drag_data["button"].configure(fg_color="transparent")
        self._clear_drop_indicator()
        self._destroy_drag_preview()

        if insert_index is not None:
            target_index = max(0, min(insert_index, len(self.button_list)))
            if source_index < target_index:
                target_index -= 1
            target_index = max(0, min(target_index, len(self.button_list) - 1))

            if target_index != source_index and 0 <= target_index < len(self.button_list):
                self._reorder_buttons(source_index, target_index)

        # Reset drag data
        self.drag_data = {
            "button": None,
            "index": None,
            "offset": (10, 10),
            "original_fg": "transparent"
        }
        self.drag_hover_index = None

        return "break"

    def _get_button_at_position(self, x_root, y_root):
        """Get the index of the button at the given screen position"""
        for i, button in enumerate(self.button_list):
            try:
                bx = button.winfo_rootx()
                by = button.winfo_rooty()
                bw = button.winfo_width()
                bh = button.winfo_height()

                if bx <= x_root <= bx + bw and by <= y_root <= by + bh:
                    return i
            except:
                continue
        return None

    def _calculate_drop_index(self, x_root, y_root):
        """Calculate the insertion slot index for the current cursor location."""
        if not self.button_list:
            return 0

        button_index = self._get_button_at_position(x_root, y_root)
        if button_index is not None:
            button = self.button_list[button_index]
            try:
                center_x = button.winfo_rootx() + button.winfo_width() / 2
            except Exception:
                center_x = x_root
            if x_root >= center_x:
                return button_index + 1
            return button_index

        inner = getattr(self, "_scrollable_frame", self)
        try:
            inner_x = x_root - inner.winfo_rootx()
            inner_y = y_root - inner.winfo_rooty()
        except Exception:
            return None

        sample_button = self.button_list[0]
        try:
            start_x = sample_button.winfo_x()
            start_y = sample_button.winfo_y()
            cell_width = sample_button.winfo_width() + self.cell_padx
            cell_height = sample_button.winfo_height() + self.cell_pady
        except Exception:
            return None

        if cell_width <= 0:
            cell_width = sample_button.winfo_width() + 10
        if cell_height <= 0:
            cell_height = sample_button.winfo_height() + 10

        relative_x = inner_x - start_x
        relative_y = inner_y - start_y

        row = math.floor(relative_y / cell_height) if cell_height else 0
        col = math.floor(relative_x / cell_width) if cell_width else 0

        if relative_y < 0:
            row = 0
        if relative_x < 0:
            col = 0

        max_rows = math.ceil((len(self.button_list) + 1) / self.columns)
        row = max(0, min(row, max_rows))
        col = max(0, min(col, self.columns - 1))

        slot = row * self.columns + col
        if slot < 0:
            slot = 0
        if slot > len(self.button_list):
            slot = len(self.button_list)

        return slot

    def _show_drop_indicator(self, slot_index):
        """Show a thin marker indicating where the item will be dropped."""
        if not self.button_list:
            return

        sample_button = self.button_list[0]
        try:
            start_x = sample_button.winfo_x()
            start_y = sample_button.winfo_y()
            cell_width = sample_button.winfo_width() + self.cell_padx
            cell_height = sample_button.winfo_height() + self.cell_pady
            indicator_height = sample_button.winfo_height()
        except Exception:
            return

        row = slot_index // self.columns
        col = slot_index % self.columns if slot_index < self.columns * (row + 1) else 0

        # When inserting at the very end and the row is complete, start a new row
        if slot_index == len(self.button_list) and len(self.button_list) % self.columns == 0:
            col = 0

        marker_x = start_x + col * cell_width
        marker_y = start_y + row * cell_height

        indicator_width = 6
        marker_x = max(0, marker_x - indicator_width // 2)

        if self.drag_indicator is None or not self.drag_indicator.winfo_exists():
            self.drag_indicator = customtkinter.CTkFrame(
                getattr(self, "_scrollable_frame", self),
                fg_color="#BB6C25",
                width=indicator_width,
                height=indicator_height,
                corner_radius=2
            )

        try:
            self.drag_indicator.configure(width=indicator_width, height=indicator_height)
        except Exception:
            pass

        self.drag_indicator.place(x=marker_x, y=marker_y)
        self.drag_indicator.lift()

    def _clear_drop_indicator(self):
        if self.drag_indicator is not None and self.drag_indicator.winfo_exists():
            self.drag_indicator.place_forget()
            self.drag_indicator.destroy()
        self.drag_indicator = None

    def _create_drag_preview(self, event):
        """Create a floating preview window that follows the cursor"""
        if self.drag_preview_window:
            try:
                self.drag_preview_window.destroy()
            except Exception:
                pass

        parent = self.winfo_toplevel()
        try:
            self.drag_preview_window = customtkinter.CTkToplevel(parent)
        except Exception:
            self.drag_preview_window = tk.Toplevel(parent)

        self.drag_preview_window.overrideredirect(True)
        try:
            self.drag_preview_window.attributes('-alpha', 0.85)
            self.drag_preview_window.attributes('-topmost', True)
        except Exception:
            pass

        # Match the dragged thumbnail image
        dragged_button = self.drag_data["button"]
        if hasattr(dragged_button, 'image') and dragged_button.image is not None:
            preview_label = customtkinter.CTkLabel(
                self.drag_preview_window,
                image=dragged_button.image,
                text=""
            )
            preview_label.pack()

        offset_x, offset_y = self.drag_data.get("offset", (10, 10))
        self.drag_preview_window.geometry(
            f"+{event.x_root - offset_x}+{event.y_root - offset_y}"
        )

    def _update_drag_preview(self, event):
        """Update the floating preview window position"""
        if self.drag_preview_window:
            offset_x, offset_y = self.drag_data.get("offset", (10, 10))
            self.drag_preview_window.geometry(
                f"+{event.x_root - offset_x}+{event.y_root - offset_y}"
            )

    def _destroy_drag_preview(self):
        """Destroy the floating preview window"""
        if self.drag_preview_window:
            try:
                self.drag_preview_window.destroy()
            except:
                pass
            self.drag_preview_window = None
        self.drag_data["offset"] = (10, 10)

    def _reorder_buttons(self, source_index, target_index):
        """Reorder buttons and save the new order"""
        # Reorder in button_list
        button = self.button_list.pop(source_index)
        self.button_list.insert(target_index, button)

        # Update grid positions
        for i, btn in enumerate(self.button_list):
            btn.grid(row=i // self.columns, column=i % self.columns, pady=(0, 10), padx=(10, 0))

        # Save the new order by renaming files
        self._save_order()

        # Update cached image path order to reflect current layout
        self.image_paths = [
            str(btn.docx_path.with_suffix('.png'))
            for btn in self.button_list
        ]
        parent = getattr(self, "master", None)
        if parent is not None:
            try:
                parent.last_clicked_button = str(button.docx_path)
                if hasattr(parent, "highlight_last_clicked_button"):
                    parent.highlight_last_clicked_button(parent.last_clicked_button)
                if hasattr(parent, "last_state") and isinstance(parent.last_state, dict):
                    parent.last_state["last_clicked_button"] = parent.last_clicked_button
                    if hasattr(parent, "save_preset_last_state_to_file"):
                        parent.save_preset_last_state_to_file(parent.last_state)
            except Exception as exc:
                pass

    def _save_order(self):
        """Save the current order by adding numeric prefixes to filenames"""
        rename_operations = []

        for i, button in enumerate(self.button_list):
            old_docx_path = button.docx_path
            old_png_path = Path(str(old_docx_path).replace('.docx', '.png'))

            old_stem = old_docx_path.stem
            base_name = re.sub(r'^\d{3}_', '', old_stem)
            new_stem = f"{i + 1:03d}_{base_name}"

            new_docx_path = old_docx_path.parent / f"{new_stem}.docx"
            new_png_path = old_png_path.parent / f"{new_stem}.png"

            rename_operations.append({
                "button": button,
                "old_docx": old_docx_path,
                "old_png": old_png_path,
                "new_docx": new_docx_path,
                "new_png": new_png_path
            })

        temp_entries = []

        for op in rename_operations:
            old_docx = op["old_docx"]
            new_docx = op["new_docx"]

            if old_docx == new_docx:
                continue

            temp_token = uuid4().hex
            temp_docx = old_docx.parent / f"__tmp__{temp_token}__{new_docx.name}"
            temp_png = None

            try:
                if old_docx.exists():
                    old_docx.rename(temp_docx)

                old_png = op["old_png"]
                if old_png.exists():
                    temp_png = old_png.parent / f"__tmp__{temp_token}__{op['new_png'].name}"
                    old_png.rename(temp_png)

                temp_entries.append({
                    "button": op["button"],
                    "temp_docx": temp_docx,
                    "temp_png": temp_png,
                    "new_docx": new_docx,
                    "new_png": op["new_png"]
                })
            except Exception as exc:
                print(f"Error staging rename for {old_docx.name}: {exc}")

        for entry in temp_entries:
            try:
                entry["temp_docx"].rename(entry["new_docx"])
                if entry["temp_png"] and entry["temp_png"].exists():
                    entry["temp_png"].rename(entry["new_png"])
                entry["button"].docx_path = entry["new_docx"]
            except Exception as exc:
                print(f"Error finalizing rename for {entry['new_docx'].name}: {exc}")

        # For entries that did not require renaming, ensure button references are correct
        for op in rename_operations:
            if op["old_docx"] == op["new_docx"]:
                op["button"].docx_path = op["old_docx"]

        self._sync_selection_state()

    def _normalized_cover_name(self, candidate):
        """Return filename with numeric prefix removed for comparison."""
        if not candidate or candidate == "No button clicked":
            return None
        try:
            name = Path(candidate).name
        except Exception:
            return None
        return re.sub(r'^\d{3}_', '', name)

    def _sync_selection_state(self):
        """Keep parent selection pointers aligned with any renamed files."""
        parent = getattr(self, "master", None)
        if parent is None:
            return

        def resolve_selection(attr_name):
            current_value = getattr(parent, attr_name, None)
            normalized = self._normalized_cover_name(current_value)
            return current_value, normalized

        _, last_normalized = resolve_selection("last_clicked_button")
        if last_normalized:
            for button in self.button_list:
                if self._normalized_cover_name(button.docx_path) == last_normalized:
                    parent.last_clicked_button = str(button.docx_path)
                    if hasattr(parent, "highlight_last_clicked_button"):
                        parent.highlight_last_clicked_button(parent.last_clicked_button)
                    break

        if hasattr(parent, "first_button") and self.button_list:
            parent.first_button = str(self.button_list[0].docx_path)

        if hasattr(parent, "last_state") and isinstance(parent.last_state, dict):
            parent.last_state["last_clicked_button"] = getattr(parent, "last_clicked_button", None)
            if hasattr(parent, "save_preset_last_state_to_file"):
                try:
                    parent.save_preset_last_state_to_file(parent.last_state)
                except Exception as exc:
                    pass


class GyojaeOptionsDialog(customtkinter.CTkToplevel):
    """Dialog for 교재제작 옵션 settings"""

    def __init__(self, master, parent_window, **kwargs):
        super().__init__(master, **kwargs)
        self.title("교재제작 옵션")
        self.geometry("1150x400")
        self.configure(fg_color="#EFF3F0")
        self.grab_set()
        self.focus_set()

        self.parent_window = parent_window

        # Clone current values for cancel handling
        self.original_values = {
            'jimun_analysis_font': parent_window.jimun_analysis_font_var.get(),
            'jimun_analysis_font_size': parent_window.jimun_analysis_font_size_var.get(),
            'jimun_analysis_font_color': parent_window.jimun_analysis_font_color_var.get(),
            'jimun_analysis_line_spacing': parent_window.jimun_analysis_line_spacing_var.get(),
            'jimun_analysis_sentence_split': parent_window.jimun_analysis_sentence_split_var.get(),
            'jimun_analysis_interpretation_right': parent_window.jimun_analysis_interpretation_right_var.get(),
            'hanjul_haesuk_font': parent_window.hanjul_haesuk_font_var.get(),
            'hanjul_haesuk_font_size': parent_window.hanjul_haesuk_font_size_var.get(),
            'hanjul_haesuk_font_color': parent_window.hanjul_haesuk_font_color_var.get(),
            'hanjul_haesuk_line_spacing': parent_window.hanjul_haesuk_line_spacing_var.get(),
            'hanjul_interpretation_include_answer': parent_window.hanjul_interpretation_include_answer_var.get(),
            'hanjul_writing_include_answer': parent_window.hanjul_writing_include_answer_var.get(),
            'hanjul_writing_word_scramble': parent_window.hanjul_writing_word_scramble_var.get(),
            'hanjul_writing_word_form_change': parent_window.hanjul_writing_word_form_change_var.get(),
            'hanjul_writing_korean_only': parent_window.hanjul_writing_korean_only_var.get()
        }

        font_handler.set_default_font()
        self.default_font = font_handler.default_font

        # Get font list from parent
        self.fonts = self.parent_window.fonts if hasattr(self.parent_window, 'fonts') else []

        self.setup_ui()

        # Bind ESC key to cancel button
        self.bind("<Escape>", lambda event: self.on_cancel())

    def setup_ui(self):
        # Main container with four sections side by side
        main_container = CTkFrame(self, fg_color="transparent")
        main_container.pack(fill="both", expand=True, padx=20, pady=20)
        main_container.grid_columnconfigure(0, weight=0)  # '지문분석' - fixed width
        main_container.grid_columnconfigure(1, weight=0)  # '한글해석' - fixed width
        main_container.grid_columnconfigure(2, weight=1)
        main_container.grid_columnconfigure(3, weight=1)

        # 1st section: '지문분석' 옵션
        self.create_jimun_analysis_section(main_container)

        # 2nd section: '한글해석' 옵션
        self.create_hanjul_haesuk_section(main_container)

        # 3rd section: '한줄해석연습' 옵션
        self.create_interpretation_practice_section(main_container)

        # 4th section: '한줄영작연습' 옵션
        self.create_writing_practice_section(main_container)

        # Bottom buttons
        self.create_action_buttons()

        # Initialize 한글해석 section state based on checkbox
        self.update_hanjul_haesuk_section_state()

    def create_jimun_analysis_section(self, parent):
        """Create left section for '지문분석' options"""
        section_frame = CTkFrame(parent, fg_color="#B6C2B7", corner_radius=10)
        section_frame.grid(row=0, column=0, padx=10, pady=10, sticky="ns")
        section_frame.grid_columnconfigure(1, weight=0)

        # Title with tooltip
        title_frame = CTkFrame(section_frame, fg_color="transparent")
        title_frame.grid(row=0, column=0, columnspan=2, pady=(10, 15), sticky="w", padx=10)

        title_label = CTkLabel(
            title_frame,
            text="'지문분석' 옵션",
            font=(self.default_font, 16, "bold"),
            text_color="black"
        )
        title_label.pack(side="left")

        # [?] tooltip label
        help_label = CTkLabel(
            title_frame,
            text=" [?]",
            font=(self.default_font, 14),
            text_color="#4B76B0",
            cursor="hand2"
        )
        help_label.pack(side="left", padx=(5, 0))

        tooltip_text = ("'지문분석'에 들어갈 지문 서식을 별도로 설정합니다.\n해당 부분을 수업용으로 사용하실 경우 학생들이 필기를 할 수 있도록\n글씨 크기를 키우고 줄 간격을 넓히면 좋습니다.")
        ToolTip(help_label).show_tip = lambda: self.show_custom_tooltip(help_label, tooltip_text)
        help_label.bind("<Enter>", lambda e: self.show_custom_tooltip(help_label, tooltip_text))
        help_label.bind("<Leave>", lambda e: self.hide_custom_tooltip())

        # Font selection
        row = 1
        CTkLabel(section_frame, text="폰트 선택:", font=(self.default_font, 12),
                text_color="black").grid(row=row, column=0, padx=(10, 5), pady=5, sticky="w")

        self.jimun_analysis_font_combobox = SearchableFontComboBox(
            section_frame,
            variable=self.parent_window.jimun_analysis_font_var,
            values=self.fonts,
            font_tuple=(self.default_font, 12),
            fg_color='white',
            text_color='black',
            border_color='#FEF9E0',
            button_color='#FEF9E0',
            button_hover_color='#CA7900',
            dropdown_hover_color='#CA7900',
            preview_text="AaBb가나",
            width=160
        )
        self.jimun_analysis_font_combobox.grid(row=row, column=1, padx=(0, 10), pady=5, sticky="w")

        # Font size
        row += 1
        CTkLabel(section_frame, text="폰트 크기:", font=(self.default_font, 12),
                text_color="black").grid(row=row, column=0, padx=(10, 5), pady=5, sticky="w")

        self.jimun_analysis_size_entry = CTkEntry(
            section_frame,
            textvariable=self.parent_window.jimun_analysis_font_size_var,
            width=80,
            justify=RIGHT
        )
        self.jimun_analysis_size_entry.grid(row=row, column=1, padx=(0, 10), pady=5, sticky="w")

        # Font color
        row += 1
        CTkLabel(section_frame, text="폰트 색상:", font=(self.default_font, 12),
                text_color="black").grid(row=row, column=0, padx=(10, 5), pady=5, sticky="w")

        self.color_button = CTkButton(
            section_frame,
            text="Select Color",
            command=self.select_color,
            text_color=self.parent_window.jimun_analysis_font_color_var.get(),
            fg_color="white",
            border_width=1,
            hover=False,
            width=120
        )
        self.color_button.grid(row=row, column=1, padx=(0, 10), pady=5, sticky="w")

        # Line spacing
        row += 1
        CTkLabel(section_frame, text="줄간격:", font=(self.default_font, 12),
                text_color="black").grid(row=row, column=0, padx=(10, 5), pady=5, sticky="w")

        self.jimun_analysis_line_entry = CTkEntry(
            section_frame,
            textvariable=self.parent_window.jimun_analysis_line_spacing_var,
            width=80,
            justify=RIGHT
        )
        self.jimun_analysis_line_entry.grid(row=row, column=1, padx=(0, 10), pady=5, sticky="w")

        # Checkbox: 지문을 문장별로 분리
        row += 1
        sentence_split_checkbox = CTkCheckBox(
            section_frame,
            text="지문을 문장별로 분리",
            variable=self.parent_window.jimun_analysis_sentence_split_var,
            font=(self.default_font, 13),
            border_color="grey",
            hover_color="#BB6C25",
            fg_color="#DDA15C",
            text_color="black"
        )
        sentence_split_checkbox.grid(row=row, column=0, columnspan=2, padx=10, pady=5, sticky="w")

        # Checkbox: 해석을 지문 우측에 두기
        row += 1
        interpretation_right_checkbox = CTkCheckBox(
            section_frame,
            text="좌지문 우해석 (해석을 지문 우측에)",
            variable=self.parent_window.jimun_analysis_interpretation_right_var,
            command=self.update_hanjul_haesuk_section_state,
            font=(self.default_font, 13),
            border_color="grey",
            hover_color="#BB6C25",
            fg_color="#DDA15C",
            text_color="black"
        )
        interpretation_right_checkbox.grid(row=row, column=0, columnspan=2, padx=10, pady=5, sticky="w")

    def create_hanjul_haesuk_section(self, parent):
        """Create section for '한글해석' options (font settings for Korean interpretation)"""
        self.hanjul_haesuk_section_frame = CTkFrame(parent, fg_color="#B6C2B7", corner_radius=10)
        self.hanjul_haesuk_section_frame.grid(row=0, column=1, padx=10, pady=10, sticky="ns")
        self.hanjul_haesuk_section_frame.grid_columnconfigure(1, weight=0)

        # Title with tooltip
        title_frame = CTkFrame(self.hanjul_haesuk_section_frame, fg_color="transparent")
        title_frame.grid(row=0, column=0, columnspan=2, pady=(10, 15), sticky="w", padx=10)

        self.hanjul_haesuk_title_label = CTkLabel(
            title_frame,
            text="'한글해석' 옵션 (좌지문 우해석)",
            font=(self.default_font, 16, "bold"),
            text_color="black"
        )
        self.hanjul_haesuk_title_label.pack(side="left")

        # [?] tooltip label
        help_label = CTkLabel(
            title_frame,
            text=" [?]",
            font=(self.default_font, 14),
            text_color="#4B76B0",
            cursor="hand2"
        )
        help_label.pack(side="left", padx=(5, 0))

        tooltip_text = ("'한글해석' 부분의 서식을 별도로 설정합니다.\n"
                       "지문 우측에 들어갈 한글 해석의 폰트, 크기, 색상, 줄간격을\n"
                       "별도로 설정할 수 있습니다.\n"
                       "※ '좌지문 우해석' 옵션이 켜져 있을 때만 사용됩니다.")
        help_label.bind("<Enter>", lambda e: self.show_custom_tooltip(help_label, tooltip_text))
        help_label.bind("<Leave>", lambda e: self.hide_custom_tooltip())

        # Store widgets for enabling/disabling
        self.hanjul_haesuk_widgets = []

        # Font selection
        row = 1
        self.hanjul_haesuk_font_label = CTkLabel(self.hanjul_haesuk_section_frame, text="폰트 선택:", font=(self.default_font, 12),
                text_color="black")
        self.hanjul_haesuk_font_label.grid(row=row, column=0, padx=(10, 5), pady=5, sticky="w")
        self.hanjul_haesuk_widgets.append(self.hanjul_haesuk_font_label)

        self.hanjul_haesuk_font_combobox = SearchableFontComboBox(
            self.hanjul_haesuk_section_frame,
            variable=self.parent_window.hanjul_haesuk_font_var,
            values=self.fonts,
            font_tuple=(self.default_font, 12),
            fg_color='white',
            text_color='black',
            border_color='#FEF9E0',
            button_color='#FEF9E0',
            button_hover_color='#CA7900',
            dropdown_hover_color='#CA7900',
            preview_text="AaBb가나",
            width=160
        )
        self.hanjul_haesuk_font_combobox.grid(row=row, column=1, padx=(0, 10), pady=5, sticky="w")
        self.hanjul_haesuk_widgets.append(self.hanjul_haesuk_font_combobox)

        # Font size
        row += 1
        self.hanjul_haesuk_size_label = CTkLabel(self.hanjul_haesuk_section_frame, text="폰트 크기:", font=(self.default_font, 12),
                text_color="black")
        self.hanjul_haesuk_size_label.grid(row=row, column=0, padx=(10, 5), pady=5, sticky="w")
        self.hanjul_haesuk_widgets.append(self.hanjul_haesuk_size_label)

        self.hanjul_haesuk_size_entry = CTkEntry(
            self.hanjul_haesuk_section_frame,
            textvariable=self.parent_window.hanjul_haesuk_font_size_var,
            width=80,
            justify=RIGHT
        )
        self.hanjul_haesuk_size_entry.grid(row=row, column=1, padx=(0, 10), pady=5, sticky="w")
        self.hanjul_haesuk_widgets.append(self.hanjul_haesuk_size_entry)

        # Font color
        row += 1
        self.hanjul_haesuk_color_label = CTkLabel(self.hanjul_haesuk_section_frame, text="폰트 색상:", font=(self.default_font, 12),
                text_color="black")
        self.hanjul_haesuk_color_label.grid(row=row, column=0, padx=(10, 5), pady=5, sticky="w")
        self.hanjul_haesuk_widgets.append(self.hanjul_haesuk_color_label)

        self.hanjul_haesuk_color_button = CTkButton(
            self.hanjul_haesuk_section_frame,
            text="Select Color",
            command=self.select_hanjul_haesuk_color,
            text_color=self.parent_window.hanjul_haesuk_font_color_var.get(),
            fg_color="white",
            border_width=1,
            hover=False,
            width=120
        )
        self.hanjul_haesuk_color_button.grid(row=row, column=1, padx=(0, 10), pady=5, sticky="w")
        self.hanjul_haesuk_widgets.append(self.hanjul_haesuk_color_button)

        # Line spacing
        row += 1
        self.hanjul_haesuk_line_label = CTkLabel(self.hanjul_haesuk_section_frame, text="줄간격:", font=(self.default_font, 12),
                text_color="black")
        self.hanjul_haesuk_line_label.grid(row=row, column=0, padx=(10, 5), pady=5, sticky="w")
        self.hanjul_haesuk_widgets.append(self.hanjul_haesuk_line_label)

        self.hanjul_haesuk_line_entry = CTkEntry(
            self.hanjul_haesuk_section_frame,
            textvariable=self.parent_window.hanjul_haesuk_line_spacing_var,
            width=80,
            justify=RIGHT
        )
        self.hanjul_haesuk_line_entry.grid(row=row, column=1, padx=(0, 10), pady=5, sticky="w")
        self.hanjul_haesuk_widgets.append(self.hanjul_haesuk_line_entry)

    def create_interpretation_practice_section(self, parent):
        """Create section for '한줄해석연습' options"""
        section_frame = CTkFrame(parent, fg_color="#B6C2B7", corner_radius=10)
        section_frame.grid(row=0, column=2, padx=10, pady=10, sticky="nsew")

        # Title with tooltip
        title_frame = CTkFrame(section_frame, fg_color="transparent")
        title_frame.grid(row=0, column=0, pady=(10, 15), sticky="w", padx=10)

        title_label = CTkLabel(
            title_frame,
            text="'한줄해석연습' 옵션",
            font=(self.default_font, 16, "bold"),
            text_color="black"
        )
        title_label.pack(side="left")

        # [?] tooltip label
        help_label = CTkLabel(
            title_frame,
            text=" [?]",
            font=(self.default_font, 14),
            text_color="#4B76B0",
            cursor="hand2"
        )
        help_label.pack(side="left", padx=(5, 0))

        tooltip_text = "'한줄해석연습'에서 정답지(해석)를 포함할지 선택할 수 있습니다."
        help_label.bind("<Enter>", lambda e: self.show_custom_tooltip(help_label, tooltip_text))
        help_label.bind("<Leave>", lambda e: self.hide_custom_tooltip())

        # Single checkbox
        row = 1
        checkbox = CTkCheckBox(
            section_frame,
            text="정답지(해석) 포함",
            variable=self.parent_window.hanjul_interpretation_include_answer_var,
            font=(self.default_font, 13),
            border_color="grey",
            hover_color="#BB6C25",
            fg_color="#DDA15C",
            text_color="black"
        )
        checkbox.grid(row=row, column=0, padx=10, pady=8, sticky="w")

    def create_writing_practice_section(self, parent):
        """Create section for '한줄영작연습' options"""
        section_frame = CTkFrame(parent, fg_color="#B6C2B7", corner_radius=10)
        section_frame.grid(row=0, column=3, padx=10, pady=10, sticky="nsew")

        # Title with tooltip
        title_frame = CTkFrame(section_frame, fg_color="transparent")
        title_frame.grid(row=0, column=0, pady=(10, 15), sticky="w", padx=10)

        title_label = CTkLabel(
            title_frame,
            text="'한줄영작연습' 옵션",
            font=(self.default_font, 16, "bold"),
            text_color="black"
        )
        title_label.pack(side="left")

        # [?] tooltip label
        help_label = CTkLabel(
            title_frame,
            text=" [?]",
            font=(self.default_font, 14),
            text_color="#4B76B0",
            cursor="hand2"
        )
        help_label.pack(side="left", padx=(5, 0))

        tooltip_text = ("'한줄영작연습'에서 정답지(해석) 포함 여부 및 힌트 제공 방식을 선택할 수 있습니다.\n"
                       "힌트 제공 방식 (복수 선택 가능):\n"
                       "1) '단어 스크램블' : 해당 문장의 모든 단어를 랜덤으로 섞어서 제시됩니다.\n"
                       "2) '단어 어형변화 가능' : 관사, 대명사가 제외되며, 동사, 준동사, 명사는 원형으로 제시됩니다.\n"
                       "3) '한글 해석만': 영어 단어 힌트 없이 한글 해석만 제시됩니다.")
        help_label.bind("<Enter>", lambda e: self.show_custom_tooltip(help_label, tooltip_text))
        help_label.bind("<Leave>", lambda e: self.hide_custom_tooltip())

        # First checkbox: 정답지(해석) 포함
        row = 1
        checkbox_answer = CTkCheckBox(
            section_frame,
            text="정답지(해석) 포함",
            variable=self.parent_window.hanjul_writing_include_answer_var,
            font=(self.default_font, 13),
            border_color="grey",
            hover_color="#BB6C25",
            fg_color="#DDA15C",
            text_color="black"
        )
        checkbox_answer.grid(row=row, column=0, padx=10, pady=8, sticky="w")

        # Label: 힌트 제공 방식
        row += 1
        hint_label = CTkLabel(
            section_frame,
            text="힌트 제공 방식 (중복시 각각 생성):",
            font=(self.default_font, 12, "bold"),
            text_color="black"
        )
        hint_label.grid(row=row, column=0, padx=10, pady=(10, 5), sticky="w")

        # Three hint checkboxes
        row += 1
        checkbox1 = CTkCheckBox(
            section_frame,
            text="단어 스크램블",
            variable=self.parent_window.hanjul_writing_word_scramble_var,
            font=(self.default_font, 13),
            border_color="grey",
            hover_color="#BB6C25",
            fg_color="#DDA15C",
            text_color="black"
        )
        checkbox1.grid(row=row, column=0, padx=10, pady=5, sticky="w")

        row += 1
        checkbox2 = CTkCheckBox(
            section_frame,
            text="단어 어형변화 가능",
            variable=self.parent_window.hanjul_writing_word_form_change_var,
            font=(self.default_font, 13),
            border_color="grey",
            hover_color="#BB6C25",
            fg_color="#DDA15C",
            text_color="black"
        )
        checkbox2.grid(row=row, column=0, padx=10, pady=5, sticky="w")

        row += 1
        checkbox3 = CTkCheckBox(
            section_frame,
            text="한글 해석만",
            variable=self.parent_window.hanjul_writing_korean_only_var,
            font=(self.default_font, 13),
            border_color="grey",
            hover_color="#BB6C25",
            fg_color="#DDA15C",
            text_color="black"
        )
        checkbox3.grid(row=row, column=0, padx=10, pady=5, sticky="w")

    def create_action_buttons(self):
        """Create Default/Cancel/OK buttons at bottom"""
        button_frame = CTkFrame(self, fg_color="transparent")
        button_frame.pack(side="bottom", pady=10)

        default_btn = CTkButton(
            button_frame,
            text="기본값",
            command=self.reset_to_defaults,
            font=(self.default_font, 13),
            fg_color="#5A7D9A",
            hover_color="#4A6D8A",
            width=100
        )
        default_btn.pack(side="left", padx=5)

        cancel_btn = CTkButton(
            button_frame,
            text="취소",
            command=self.on_cancel,
            font=(self.default_font, 13),
            fg_color="#8D9397",
            hover_color="#5F6C37",
            width=100
        )
        cancel_btn.pack(side="left", padx=5)

        ok_btn = CTkButton(
            button_frame,
            text="확인",
            command=self.on_ok,
            font=(self.default_font, 13),
            fg_color="#2B3D2D",
            hover_color="#5F6C37",
            width=100
        )
        ok_btn.pack(side="left", padx=5)

    def select_color(self):
        """Open color chooser dialog for '지문분석' font color"""
        self.master.focus_set()
        color_code, color_name = colorchooser.askcolor(
            title="Choose a color",
            initialcolor=self.parent_window.jimun_analysis_font_color_var.get()
        )
        if color_code:
            self.parent_window.jimun_analysis_font_color_var.set(color_name)
            self.color_button.configure(text_color=color_name)
        self.focus_set()

    def select_hanjul_haesuk_color(self):
        """Open color chooser dialog for '한글해석' font color"""
        self.master.focus_set()
        color_code, color_name = colorchooser.askcolor(
            title="Choose a color",
            initialcolor=self.parent_window.hanjul_haesuk_font_color_var.get()
        )
        if color_code:
            self.parent_window.hanjul_haesuk_font_color_var.set(color_name)
            self.hanjul_haesuk_color_button.configure(text_color=color_name)
        self.focus_set()

    def update_hanjul_haesuk_section_state(self):
        """Enable/disable 한글해석 section based on '해석을 지문 우측에 두기' checkbox"""
        is_enabled = self.parent_window.jimun_analysis_interpretation_right_var.get() == 1

        if is_enabled:
            # Enable all widgets
            for widget in self.hanjul_haesuk_widgets:
                widget.configure(state="normal")
            # Update title color to normal
            self.hanjul_haesuk_title_label.configure(text_color="black")
            # Update text colors for entry widgets to black
            self.hanjul_haesuk_size_entry.configure(text_color="black")
            self.hanjul_haesuk_line_entry.configure(text_color="black")
            # Update combobox text color to black
            self.hanjul_haesuk_font_combobox.configure(text_color="black")
        else:
            # Disable all widgets
            for widget in self.hanjul_haesuk_widgets:
                widget.configure(state="disabled")
            # Update title color to indicate disabled state
            self.hanjul_haesuk_title_label.configure(text_color="gray")
            # Update text colors for entry widgets to gray
            self.hanjul_haesuk_size_entry.configure(text_color="gray")
            self.hanjul_haesuk_line_entry.configure(text_color="gray")
            # Update combobox text color to gray
            self.hanjul_haesuk_font_combobox.configure(text_color="gray")

    def show_custom_tooltip(self, widget, text):
        """Show custom tooltip"""
        if hasattr(self, 'tooltip_window') and self.tooltip_window:
            return

        x = widget.winfo_rootx() + 20
        y = widget.winfo_rooty() + 25

        self.tooltip_window = tk.Toplevel(widget)
        self.tooltip_window.wm_overrideredirect(True)
        self.tooltip_window.wm_geometry(f"+{x}+{y}")

        label = tk.Label(
            self.tooltip_window,
            text=text,
            justify=tk.LEFT,
            background="#ffffe0",
            relief=tk.SOLID,
            borderwidth=1,
            font=(self.default_font, 13)
        )
        label.pack()

    def hide_custom_tooltip(self):
        """Hide custom tooltip"""
        if hasattr(self, 'tooltip_window') and self.tooltip_window:
            self.tooltip_window.destroy()
            self.tooltip_window = None

    def on_ok(self):
        """Apply changes and close dialog"""
        self.destroy()

    def reset_to_defaults(self):
        """Reset all dialog values to defaults"""
        # '지문분석' 옵션 defaults
        self.parent_window.jimun_analysis_font_var.set(self.default_font)
        self.parent_window.jimun_analysis_font_size_var.set("8")
        self.parent_window.jimun_analysis_font_color_var.set("#000000")
        self.parent_window.jimun_analysis_line_spacing_var.set("1.2")
        self.parent_window.jimun_analysis_sentence_split_var.set(0)
        self.parent_window.jimun_analysis_interpretation_right_var.set(0)

        # '한글해석' 옵션 defaults
        self.parent_window.hanjul_haesuk_font_var.set(self.default_font)
        self.parent_window.hanjul_haesuk_font_size_var.set("6")
        self.parent_window.hanjul_haesuk_font_color_var.set("#000000")
        self.parent_window.hanjul_haesuk_line_spacing_var.set("1")

        # '한줄해석연습' 옵션 defaults
        self.parent_window.hanjul_interpretation_include_answer_var.set(1)

        # '한줄영작연습' 옵션 defaults
        self.parent_window.hanjul_writing_include_answer_var.set(1)
        self.parent_window.hanjul_writing_word_scramble_var.set(1)
        self.parent_window.hanjul_writing_word_form_change_var.set(0)
        self.parent_window.hanjul_writing_korean_only_var.set(0)

        # Update color buttons to reflect default colors
        if hasattr(self, 'color_button'):
            self.color_button.configure(text_color="#000000")
        if hasattr(self, 'hanjul_haesuk_color_button'):
            self.hanjul_haesuk_color_button.configure(text_color="#000000")

        # Update font comboboxes
        if hasattr(self, 'jimun_analysis_font_combobox'):
            self.jimun_analysis_font_combobox.set(self.default_font)
        if hasattr(self, 'hanjul_haesuk_font_combobox'):
            self.hanjul_haesuk_font_combobox.set(self.default_font)

        # Update hanjul_haesuk section state based on checkbox
        self.update_hanjul_haesuk_section_state()

    def on_cancel(self):
        """Roll back changes and close dialog"""
        # Restore original values
        self.parent_window.jimun_analysis_font_var.set(self.original_values['jimun_analysis_font'])
        self.parent_window.jimun_analysis_font_size_var.set(self.original_values['jimun_analysis_font_size'])
        self.parent_window.jimun_analysis_font_color_var.set(self.original_values['jimun_analysis_font_color'])
        self.parent_window.jimun_analysis_line_spacing_var.set(self.original_values['jimun_analysis_line_spacing'])
        self.parent_window.jimun_analysis_sentence_split_var.set(self.original_values['jimun_analysis_sentence_split'])
        self.parent_window.jimun_analysis_interpretation_right_var.set(self.original_values['jimun_analysis_interpretation_right'])
        self.parent_window.hanjul_haesuk_font_var.set(self.original_values['hanjul_haesuk_font'])
        self.parent_window.hanjul_haesuk_font_size_var.set(self.original_values['hanjul_haesuk_font_size'])
        self.parent_window.hanjul_haesuk_font_color_var.set(self.original_values['hanjul_haesuk_font_color'])
        self.parent_window.hanjul_haesuk_line_spacing_var.set(self.original_values['hanjul_haesuk_line_spacing'])
        self.parent_window.hanjul_interpretation_include_answer_var.set(self.original_values['hanjul_interpretation_include_answer'])
        self.parent_window.hanjul_writing_include_answer_var.set(self.original_values['hanjul_writing_include_answer'])
        self.parent_window.hanjul_writing_word_scramble_var.set(self.original_values['hanjul_writing_word_scramble'])
        self.parent_window.hanjul_writing_word_form_change_var.set(self.original_values['hanjul_writing_word_form_change'])
        self.parent_window.hanjul_writing_korean_only_var.set(self.original_values['hanjul_writing_korean_only'])
        self.destroy()


class MunjeOptionsDialog(customtkinter.CTkToplevel):
    """Dialog for 문제 옵션 settings"""

    def __init__(self, master, parent_window, **kwargs):
        super().__init__(master, **kwargs)
        self.title("문제 옵션")
        self.geometry("400x300")
        self.configure(fg_color="#EFF3F0")
        self.grab_set()
        self.focus_set()

        self.parent_window = parent_window

        # Clone current value for cancel handling
        self.original_value = {
            'naeyongilchi_question_type': parent_window.naeyongilchi_question_type_var.get()
        }

        font_handler.set_default_font()
        self.default_font = font_handler.default_font

        self.setup_ui()

        # Bind ESC key to cancel button
        self.bind("<Escape>", lambda event: self.on_cancel())

    def setup_ui(self):
        # Main container
        main_container = CTkFrame(self, fg_color="transparent")
        main_container.pack(fill="both", expand=True, padx=20, pady=20)
        main_container.grid_columnconfigure(0, weight=1)

        # Section: '내용일치' 옵션
        self.create_naeyongilchi_section(main_container)

        # Bottom buttons
        self.create_action_buttons()

    def create_naeyongilchi_section(self, parent):
        """Create section for '내용일치' options"""
        section_frame = CTkFrame(parent, fg_color="#B6C2B7", corner_radius=10)
        section_frame.grid(row=0, column=0, padx=10, pady=10, sticky="nsew")
        section_frame.grid_columnconfigure(0, weight=1)

        # Title
        title_label = CTkLabel(
            section_frame,
            text="'내용일치' 옵션",
            font=(self.default_font, 16, "bold"),
            text_color="black"
        )
        title_label.grid(row=0, column=0, pady=(10, 15), sticky="w", padx=10)

        # Label: 문제 유형
        row = 1
        type_label = CTkLabel(
            section_frame,
            text="문제 유형:",
            font=(self.default_font, 13, "bold"),
            text_color="black"
        )
        type_label.grid(row=row, column=0, padx=10, pady=(5, 5), sticky="w")

        # Radio buttons frame
        row += 1
        radio_frame = CTkFrame(section_frame, fg_color="transparent")
        radio_frame.grid(row=row, column=0, padx=10, pady=(0, 10), sticky="w")

        # Radio button 1: T/F 형식 (default, value=0)
        radio1 = CTkRadioButton(
            radio_frame,
            text="T/F 형식",
            variable=self.parent_window.naeyongilchi_question_type_var,
            value=0,
            font=(self.default_font, 13),
            border_color="grey",
            hover_color="#BB6C25",
            fg_color="#DDA15C",
            text_color="black"
        )
        radio1.grid(row=0, column=0, padx=10, pady=5, sticky="w")

        # Radio button 2: 5지선다 (value=1)
        radio2 = CTkRadioButton(
            radio_frame,
            text="5지선다",
            variable=self.parent_window.naeyongilchi_question_type_var,
            value=1,
            font=(self.default_font, 13),
            border_color="grey",
            hover_color="#BB6C25",
            fg_color="#DDA15C",
            text_color="black"
        )
        radio2.grid(row=1, column=0, padx=10, pady=5, sticky="w")

    def create_action_buttons(self):
        """Create OK/Cancel buttons at bottom"""
        button_frame = CTkFrame(self, fg_color="transparent")
        button_frame.pack(side="bottom", pady=10)

        cancel_btn = CTkButton(
            button_frame,
            text="취소",
            command=self.on_cancel,
            font=(self.default_font, 13),
            fg_color="#8D9397",
            hover_color="#5F6C37",
            width=100
        )
        cancel_btn.pack(side="left", padx=5)

        ok_btn = CTkButton(
            button_frame,
            text="확인",
            command=self.on_ok,
            font=(self.default_font, 13),
            fg_color="#2B3D2D",
            hover_color="#5F6C37",
            width=100
        )
        ok_btn.pack(side="left", padx=5)

    def on_ok(self):
        """Apply changes and close dialog"""
        self.destroy()

    def on_cancel(self):
        """Roll back changes and close dialog"""
        # Restore original value
        self.parent_window.naeyongilchi_question_type_var.set(self.original_value['naeyongilchi_question_type'])
        self.destroy()


class NaejiSeosikOptionsDialog(customtkinter.CTkToplevel):
    """Dialog for 내지 서식 옵션 settings (다단 나누기, 지문 테두리, 지문번호 숫자로 사용, 정답지 분리, 유형마다 새 페이지, 페이지당 문제 수)"""

    def __init__(self, master, parent_window, **kwargs):
        super().__init__(master, **kwargs)
        self.title("내지 서식 옵션")
        self.geometry("350x430")
        self.configure(fg_color="#EFF3F0")
        self.grab_set()
        self.focus_set()

        self.parent_window = parent_window

        # Clone current values for cancel handling
        self.original_values = {
            'multicol': parent_window.multicol_var.get(),
            'border_flag': parent_window.border_flag_var.get(),
            'passage_number_numeric': parent_window.passage_number_numeric_var.get(),
            'separate_answer': parent_window.separate_answer_var.get(),
            'new_page_per_section': parent_window.new_page_per_section_var.get(),
            'page_break_after_passages_enabled': parent_window.page_break_after_passages_enabled_var.get(),
            'page_break_after_passages_count': parent_window.page_break_after_passages_count_var.get(),
        }

        font_handler.set_default_font()
        self.default_font = font_handler.default_font

        self.setup_ui()

        # Bind ESC key to cancel button
        self.bind("<Escape>", lambda event: self.on_cancel())

    def setup_ui(self):
        # Main container
        main_container = CTkFrame(self, fg_color="transparent")
        main_container.pack(fill="both", expand=True, padx=20, pady=20)
        main_container.grid_columnconfigure(0, weight=1)

        # Section frame
        section_frame = CTkFrame(main_container, fg_color="#B6C2B7", corner_radius=10)
        section_frame.grid(row=0, column=0, padx=10, pady=10, sticky="nsew")
        section_frame.grid_columnconfigure(0, weight=1)

        # Title
        title_label = CTkLabel(
            section_frame,
            text="내지 서식 옵션",
            font=(self.default_font, 16, "bold"),
            text_color="black"
        )
        title_label.grid(row=0, column=0, pady=(10, 15), sticky="w", padx=10)

        # Checkboxes frame
        checkbox_frame = CTkFrame(section_frame, fg_color="transparent")
        checkbox_frame.grid(row=1, column=0, padx=10, pady=(0, 10), sticky="w")

        # "다단 나누기" 체크박스
        self.multicol_checkbox = CTkCheckBox(
            checkbox_frame,
            text="다단 나누기",
            variable=self.parent_window.multicol_var,
            font=(self.default_font, 14),
            border_color="grey", hover_color="#BB6C25", fg_color="#DDA15C", text_color="black"
        )
        self.multicol_checkbox.grid(row=0, column=0, padx=10, pady=5, sticky="w")

        # "지문 테두리" 체크박스
        self.border_flag_checkbox = CTkCheckBox(
            checkbox_frame,
            text="지문 테두리",
            variable=self.parent_window.border_flag_var,
            font=(self.default_font, 14),
            border_color="grey", hover_color="#BB6C25", fg_color="#DDA15C", text_color="black"
        )
        self.border_flag_checkbox.grid(row=1, column=0, padx=10, pady=5, sticky="w")

        # "지문번호 숫자로 사용" 체크박스
        self.passage_number_numeric_checkbox = CTkCheckBox(
            checkbox_frame,
            text="지문번호 숫자로 사용",
            variable=self.parent_window.passage_number_numeric_var,
            font=(self.default_font, 14),
            border_color="grey", hover_color="#BB6C25", fg_color="#DDA15C", text_color="black"
        )
        self.passage_number_numeric_checkbox.grid(row=2, column=0, padx=10, pady=5, sticky="w")

        # "정답지 분리" 체크박스
        self.separate_answer_checkbox = CTkCheckBox(
            checkbox_frame,
            text="정답지 분리",
            variable=self.parent_window.separate_answer_var,
            font=(self.default_font, 14),
            border_color="grey", hover_color="#BB6C25", fg_color="#DDA15C", text_color="black"
        )
        self.separate_answer_checkbox.grid(row=3, column=0, padx=10, pady=5, sticky="w")

        # "유형마다 새 페이지" 체크박스
        self.new_page_per_section_checkbox = CTkCheckBox(
            checkbox_frame,
            text="유형마다 새 페이지",
            variable=self.parent_window.new_page_per_section_var,
            font=(self.default_font, 14),
            border_color="grey", hover_color="#BB6C25", fg_color="#DDA15C", text_color="black"
        )
        self.new_page_per_section_checkbox.grid(row=4, column=0, padx=10, pady=5, sticky="w")

        page_break_frame = CTkFrame(checkbox_frame, fg_color="transparent")
        page_break_frame.grid(row=5, column=0, padx=10, pady=5, sticky="w")

        self.page_break_after_passages_checkbox = CTkCheckBox(
            page_break_frame,
            text="페이지당 문제 수:",
            variable=self.parent_window.page_break_after_passages_enabled_var,
            font=(self.default_font, 14),
            border_color="grey", hover_color="#BB6C25", fg_color="#DDA15C", text_color="black",
            command=self._toggle_page_break_after_passages_entry
        )
        self.page_break_after_passages_checkbox.pack(side="left")

        self.page_break_after_passages_entry = CTkEntry(
            page_break_frame,
            textvariable=self.parent_window.page_break_after_passages_count_var,
            width=60,
            font=(self.default_font, 14),
            justify="center"
        )
        self.page_break_after_passages_entry.pack(side="left", padx=(6, 0))
        self._toggle_page_break_after_passages_entry()

        # Bottom buttons
        self.create_action_buttons()

    def _toggle_page_break_after_passages_entry(self):
        state = "normal" if self.parent_window.page_break_after_passages_enabled_var.get() else "disabled"
        self.page_break_after_passages_entry.configure(state=state)

    def create_action_buttons(self):
        """Create OK/Cancel buttons at bottom"""
        button_frame = CTkFrame(self, fg_color="transparent")
        button_frame.pack(side="bottom", pady=10)

        cancel_btn = CTkButton(
            button_frame,
            text="취소",
            command=self.on_cancel,
            font=(self.default_font, 13),
            fg_color="#8D9397",
            hover_color="#5F6C37",
            width=100
        )
        cancel_btn.pack(side="left", padx=5)

        ok_btn = CTkButton(
            button_frame,
            text="확인",
            command=self.on_ok,
            font=(self.default_font, 13),
            fg_color="#2B3D2D",
            hover_color="#5F6C37",
            width=100
        )
        ok_btn.pack(side="left", padx=5)

    def on_ok(self):
        """Apply changes and close dialog"""
        self.destroy()

    def on_cancel(self):
        """Roll back changes and close dialog"""
        # Restore original values
        self.parent_window.multicol_var.set(self.original_values['multicol'])
        self.parent_window.border_flag_var.set(self.original_values['border_flag'])
        self.parent_window.passage_number_numeric_var.set(self.original_values['passage_number_numeric'])
        self.parent_window.separate_answer_var.set(self.original_values['separate_answer'])
        self.parent_window.new_page_per_section_var.set(self.original_values['new_page_per_section'])
        self.parent_window.page_break_after_passages_enabled_var.set(self.original_values['page_break_after_passages_enabled'])
        self.parent_window.page_break_after_passages_count_var.set(self.original_values['page_break_after_passages_count'])
        self.destroy()


class MarginOptionsDialog(customtkinter.CTkToplevel):
    """Dialog for margin settings (여백설정) with visual paper preview"""

    # Paper preview dimensions
    PAPER_WIDTH = 160
    PAPER_HEIGHT = 220
    MARGIN_SCALE = 8  # 1cm = 8 pixels for visualization

    def __init__(self, master, parent_window, **kwargs):
        super().__init__(master, **kwargs)
        self.title("여백설정")
        self.geometry("550x580")
        self.resizable(False, False)  # Fixed size window
        self.configure(fg_color="#EFF3F0")
        self.grab_set()
        self.focus_set()

        self.parent_window = parent_window

        # Clone current values for cancel handling
        self.original_values = {
            'margin_top': parent_window.margin_top_var.get(),
            'margin_bottom': parent_window.margin_bottom_var.get(),
            'margin_left': parent_window.margin_left_var.get(),
            'margin_right': parent_window.margin_right_var.get(),
            'col_spacing': parent_window.col_spacing_var.get(),
            'mirror_margins': parent_window.mirror_margins_var.get(),
        }

        font_handler.set_default_font()
        self.default_font = font_handler.default_font

        # Track variable changes for live preview
        self._trace_ids = []

        self.setup_ui()
        self.bind("<Escape>", lambda event: self.on_cancel())
        self.protocol("WM_DELETE_WINDOW", self.on_cancel)  # Handle X button close

        # Initial draw
        self.after(50, self.update_paper_preview)

    def setup_ui(self):
        """Setup the visual paper-centric layout"""
        # Main container
        main_container = CTkFrame(self, fg_color="transparent")
        main_container.pack(fill="both", expand=True, padx=15, pady=15)

        # Title
        title_label = CTkLabel(
            main_container,
            text="여백설정 (cm)",
            font=(self.default_font, 16, "bold"),
            text_color="black"
        )
        title_label.pack(pady=(0, 10))

        # Preview section with paper
        self.create_preview_section(main_container)

        # Bottom buttons
        self.create_action_buttons()

    def create_preview_section(self, parent):
        """Create the visual paper preview with horizontal margin entries above it"""
        # Section frame with background
        section_frame = CTkFrame(parent, fg_color="#B6C2B7", corner_radius=10)
        section_frame.pack(fill="both", expand=True, padx=5, pady=5)

        # Store reference to section_frame
        self.section_frame = section_frame

        # Horizontal frame for all margin entries
        margin_entries_frame = CTkFrame(section_frame, fg_color="transparent")
        margin_entries_frame.pack(fill="x", pady=(15, 5), padx=15)

        # Create 5 columns for the margin entries
        for i in range(5):
            margin_entries_frame.grid_columnconfigure(i, weight=1)

        # Top margin (상)
        top_frame = CTkFrame(margin_entries_frame, fg_color="transparent")
        top_frame.grid(row=0, column=0, padx=5)
        CTkLabel(top_frame, text="상", font=(self.default_font, 11), text_color="black").pack()
        self.top_entry = CTkEntry(
            top_frame, textvariable=self.parent_window.margin_top_var,
            font=(self.default_font, 12), width=60, fg_color="white", text_color="black", justify="center"
        )
        self.top_entry.pack(pady=2)

        # Bottom margin (하)
        bottom_frame = CTkFrame(margin_entries_frame, fg_color="transparent")
        bottom_frame.grid(row=0, column=1, padx=5)
        CTkLabel(bottom_frame, text="하", font=(self.default_font, 11), text_color="black").pack()
        self.bottom_entry = CTkEntry(
            bottom_frame, textvariable=self.parent_window.margin_bottom_var,
            font=(self.default_font, 12), width=60, fg_color="white", text_color="black", justify="center"
        )
        self.bottom_entry.pack(pady=2)

        # Left margin (좌) - becomes "안쪽" in mirror mode
        self.left_frame = CTkFrame(margin_entries_frame, fg_color="transparent")
        self.left_frame.grid(row=0, column=2, padx=5)
        self.left_label = CTkLabel(self.left_frame, text="좌", font=(self.default_font, 11), text_color="black")
        self.left_label.pack()
        self.left_entry = CTkEntry(
            self.left_frame, textvariable=self.parent_window.margin_left_var,
            font=(self.default_font, 12), width=60, fg_color="white", text_color="black", justify="center"
        )
        self.left_entry.pack(pady=2)

        # Right margin (우) - becomes "바깥쪽" in mirror mode
        self.right_frame = CTkFrame(margin_entries_frame, fg_color="transparent")
        self.right_frame.grid(row=0, column=3, padx=5)
        self.right_label = CTkLabel(self.right_frame, text="우", font=(self.default_font, 11), text_color="black")
        self.right_label.pack()
        self.right_entry = CTkEntry(
            self.right_frame, textvariable=self.parent_window.margin_right_var,
            font=(self.default_font, 12), width=60, fg_color="white", text_color="black", justify="center"
        )
        self.right_entry.pack(pady=2)

        # Column spacing (2단 사이 간격)
        spacing_frame = CTkFrame(margin_entries_frame, fg_color="transparent")
        spacing_frame.grid(row=0, column=4, padx=5)
        CTkLabel(spacing_frame, text="2단 사이 간격", font=(self.default_font, 11), text_color="black").pack()
        self.spacing_entry = CTkEntry(
            spacing_frame, textvariable=self.parent_window.col_spacing_var,
            font=(self.default_font, 12), width=60, fg_color="white", text_color="black", justify="center"
        )
        self.spacing_entry.pack(pady=2)

        # Mirror margins checkbox (below entries, centered)
        self.mirror_checkbox = CTkCheckBox(
            section_frame,
            text="대칭 여백 (홀수/짝수 페이지 좌우 반전)",
            variable=self.parent_window.mirror_margins_var,
            command=self.on_mirror_toggle,
            font=(self.default_font, 12),
            text_color="black",
            checkbox_width=18,
            checkbox_height=18,
            fg_color="#2B3D2D",
            hover_color="#5F6C37",
            border_color="#999999"
        )
        self.mirror_checkbox.pack(pady=(10, 10))

        # Paper canvas (using tkinter Canvas for drawing)
        import tkinter as tk
        self.paper_canvas = tk.Canvas(
            section_frame,
            width=self.PAPER_WIDTH + 4,
            height=self.PAPER_HEIGHT + 4,
            bg="#B6C2B7",
            highlightthickness=0
        )
        self.paper_canvas.pack(pady=(5, 15))

        # Apply initial state for mirror mode
        self._apply_mirror_state()

        # Bind entry changes to update preview
        self._setup_trace_bindings()

        # Bind FocusOut for real-time validation
        self._setup_validation_bindings()

    def _setup_trace_bindings(self):
        """Setup trace callbacks for live preview updates"""
        vars_to_trace = [
            self.parent_window.margin_top_var,
            self.parent_window.margin_bottom_var,
            self.parent_window.margin_left_var,
            self.parent_window.margin_right_var,
            self.parent_window.col_spacing_var,
        ]
        for var in vars_to_trace:
            trace_id = var.trace_add("write", lambda *args: self.update_paper_preview())
            self._trace_ids.append((var, trace_id))

    def _setup_validation_bindings(self):
        """Setup FocusOut bindings for real-time validation"""
        # Margin entries with their ranges
        MARGIN_MIN, MARGIN_MAX = 0.5, 5.0
        COL_SPACING_MIN, COL_SPACING_MAX = 0.3, 3.0

        validation_items = [
            (self.top_entry, self.parent_window.margin_top_var, MARGIN_MIN, MARGIN_MAX, "1.5"),
            (self.bottom_entry, self.parent_window.margin_bottom_var, MARGIN_MIN, MARGIN_MAX, "1.6"),
            (self.left_entry, self.parent_window.margin_left_var, MARGIN_MIN, MARGIN_MAX, "1.9"),
            (self.right_entry, self.parent_window.margin_right_var, MARGIN_MIN, MARGIN_MAX, "1.9"),
            (self.spacing_entry, self.parent_window.col_spacing_var, COL_SPACING_MIN, COL_SPACING_MAX, "1.27"),
        ]

        for entry, var, min_val, max_val, default in validation_items:
            # Bind to FocusOut event
            entry.bind("<FocusOut>", lambda e, v=var, mn=min_val, mx=max_val, df=default:
                      self._validate_entry_on_focus_out(v, mn, mx, df))

    def _validate_entry_on_focus_out(self, var, min_val, max_val, default):
        """Validate and auto-correct entry value when focus leaves"""
        try:
            val = float(var.get())
            if val < min_val:
                var.set(str(min_val))
            elif val > max_val:
                var.set(str(max_val))
        except (ValueError, TypeError):
            # Invalid input - reset to default
            var.set(default)

    def _remove_trace_bindings(self):
        """Remove trace callbacks to prevent memory leaks"""
        for var, trace_id in self._trace_ids:
            try:
                var.trace_remove("write", trace_id)
            except Exception:
                pass
        self._trace_ids.clear()

    def _apply_mirror_state(self):
        """Apply mirror mode state to labels and canvas size"""
        is_mirror = self.parent_window.mirror_margins_var.get()
        if is_mirror:
            # Update labels for mirror mode
            self.left_label.configure(text="안쪽")
            self.right_label.configure(text="바깥쪽")

            # Resize canvas for dual preview (larger to fill space)
            self.paper_canvas.configure(width=420, height=300)
        else:
            # Update labels for normal mode
            self.left_label.configure(text="좌")
            self.right_label.configure(text="우")

            # Restore canvas to original size
            self.paper_canvas.configure(width=self.PAPER_WIDTH + 4, height=self.PAPER_HEIGHT + 4)

    def on_mirror_toggle(self):
        """Handle mirror margins checkbox toggle"""
        self._apply_mirror_state()
        self.update_paper_preview()

    def update_paper_preview(self):
        """Redraw preview - single or dual based on mirror mode"""
        if self.parent_window.mirror_margins_var.get():
            self.draw_dual_preview()
        else:
            self.draw_single_preview()

    def draw_single_preview(self):
        """Draw single page preview (normal mode)"""
        # Check if canvas still exists (prevents errors during widget destruction)
        try:
            if not hasattr(self, 'paper_canvas') or not self.paper_canvas.winfo_exists():
                return
            canvas = self.paper_canvas
            canvas.delete("all")
        except Exception:
            return

        # Get current values (with safe defaults)
        def safe_float(var, default):
            try:
                return max(0.1, float(var.get()))
            except (ValueError, TypeError):
                return default

        margin_top = safe_float(self.parent_window.margin_top_var, 1.5)
        margin_bottom = safe_float(self.parent_window.margin_bottom_var, 1.6)
        margin_left = safe_float(self.parent_window.margin_left_var, 1.9)
        margin_right = safe_float(self.parent_window.margin_right_var, 1.9)
        col_spacing = safe_float(self.parent_window.col_spacing_var, 1.27)

        # Scale margins to pixels (capped for visual clarity)
        def scale_margin(cm_val, max_px=25):
            return min(cm_val * self.MARGIN_SCALE, max_px)

        top_px = scale_margin(margin_top)
        bottom_px = scale_margin(margin_bottom)
        left_px = scale_margin(margin_left)
        right_px = scale_margin(margin_right)
        spacing_px = scale_margin(col_spacing, 15)

        # Paper position
        paper_x = 2
        paper_y = 2
        paper_w = self.PAPER_WIDTH
        paper_h = self.PAPER_HEIGHT

        # Draw paper shadow
        canvas.create_rectangle(
            paper_x + 3, paper_y + 3,
            paper_x + paper_w + 3, paper_y + paper_h + 3,
            fill="#999999", outline=""
        )

        # Draw paper background (white)
        canvas.create_rectangle(
            paper_x, paper_y,
            paper_x + paper_w, paper_y + paper_h,
            fill="#FFFFFF", outline="#AAAAAA", width=1
        )

        # Draw margin shaded areas (light gray)
        margin_color = "#E0E0E0"

        # Top margin
        canvas.create_rectangle(
            paper_x, paper_y,
            paper_x + paper_w, paper_y + top_px,
            fill=margin_color, outline=""
        )

        # Bottom margin
        canvas.create_rectangle(
            paper_x, paper_y + paper_h - bottom_px,
            paper_x + paper_w, paper_y + paper_h,
            fill=margin_color, outline=""
        )

        # Left margin
        canvas.create_rectangle(
            paper_x, paper_y + top_px,
            paper_x + left_px, paper_y + paper_h - bottom_px,
            fill=margin_color, outline=""
        )

        # Right margin
        canvas.create_rectangle(
            paper_x + paper_w - right_px, paper_y + top_px,
            paper_x + paper_w, paper_y + paper_h - bottom_px,
            fill=margin_color, outline=""
        )

        # Content area boundaries
        content_x1 = paper_x + left_px
        content_y1 = paper_y + top_px
        content_x2 = paper_x + paper_w - right_px
        content_y2 = paper_y + paper_h - bottom_px
        content_w = content_x2 - content_x1

        # Draw two-column layout if there's enough space
        if content_w > spacing_px + 20:
            col_w = (content_w - spacing_px) / 2

            # Column 1 (white rectangle)
            canvas.create_rectangle(
                content_x1, content_y1,
                content_x1 + col_w, content_y2,
                fill="#FFFFFF", outline="#CCCCCC", width=1
            )

            # Column 2 (white rectangle)
            canvas.create_rectangle(
                content_x2 - col_w, content_y1,
                content_x2, content_y2,
                fill="#FFFFFF", outline="#CCCCCC", width=1
            )

            # Column spacing indicator (dashed line in the middle)
            mid_x = (content_x1 + col_w + content_x2 - col_w) / 2
            canvas.create_line(
                mid_x, content_y1 + 5,
                mid_x, content_y2 - 5,
                fill="#999999", dash=(3, 3)
            )

            # Small spacing indicator arrow (centered, fixed size)
            arrow_y = content_y1 + (content_y2 - content_y1) / 2
            arrow_half_len = 8  # Small fixed size
            canvas.create_line(
                mid_x - arrow_half_len, arrow_y,
                mid_x + arrow_half_len, arrow_y,
                fill="#888888", arrow="both", arrowshape=(4, 5, 2)
            )
        else:
            # Single column fallback
            canvas.create_rectangle(
                content_x1, content_y1,
                content_x2, content_y2,
                fill="#FFFFFF", outline="#CCCCCC", width=1
            )

    def draw_dual_preview(self):
        """Draw dual page preview for mirror margins mode"""
        try:
            if not hasattr(self, 'paper_canvas') or not self.paper_canvas.winfo_exists():
                return
            canvas = self.paper_canvas
            canvas.delete("all")
        except Exception:
            return

        # Get current values (with safe defaults)
        def safe_float(var, default):
            try:
                return max(0.1, float(var.get()))
            except (ValueError, TypeError):
                return default

        margin_top = safe_float(self.parent_window.margin_top_var, 1.5)
        margin_bottom = safe_float(self.parent_window.margin_bottom_var, 1.6)
        inside_margin = safe_float(self.parent_window.margin_left_var, 1.9)  # Inside = Left in normal mode
        outside_margin = safe_float(self.parent_window.margin_right_var, 1.9)  # Outside = Right in normal mode
        col_spacing = safe_float(self.parent_window.col_spacing_var, 1.27)

        # Dual preview dimensions (larger pages side by side)
        page_w = 160  # Larger page width
        page_h = 220  # Larger page height
        gap = 20  # Gap between pages (where binding is)
        scale = 8  # Scale for margin visualization

        def scale_margin(cm_val, max_px=25):
            return min(cm_val * scale, max_px)

        top_px = scale_margin(margin_top)
        bottom_px = scale_margin(margin_bottom)
        inside_px = scale_margin(inside_margin)
        outside_px = scale_margin(outside_margin)
        spacing_px = scale_margin(col_spacing, 12)

        # Calculate starting positions to center two pages in the 420x300 canvas
        canvas_w = 420
        canvas_h = 300
        total_w = page_w * 2 + gap
        start_x = (canvas_w - total_w) / 2
        start_y = 10  # Leave room at top

        margin_color = "#E0E0E0"

        def draw_page(px, py, pw, ph, left_margin, right_margin, label):
            """Draw a single page with given margins"""
            # Shadow
            canvas.create_rectangle(px + 2, py + 2, px + pw + 2, py + ph + 2, fill="#999999", outline="")
            # Paper background
            canvas.create_rectangle(px, py, px + pw, py + ph, fill="#FFFFFF", outline="#AAAAAA", width=1)

            # Top margin
            canvas.create_rectangle(px, py, px + pw, py + top_px, fill=margin_color, outline="")
            # Bottom margin
            canvas.create_rectangle(px, py + ph - bottom_px, px + pw, py + ph, fill=margin_color, outline="")
            # Left margin
            canvas.create_rectangle(px, py + top_px, px + left_margin, py + ph - bottom_px, fill=margin_color, outline="")
            # Right margin
            canvas.create_rectangle(px + pw - right_margin, py + top_px, px + pw, py + ph - bottom_px, fill=margin_color, outline="")

            # Content area
            content_x1 = px + left_margin
            content_y1 = py + top_px
            content_x2 = px + pw - right_margin
            content_y2 = py + ph - bottom_px
            content_w = content_x2 - content_x1

            # Two-column layout
            if content_w > spacing_px + 10:
                col_w = (content_w - spacing_px) / 2
                canvas.create_rectangle(content_x1, content_y1, content_x1 + col_w, content_y2, fill="#FFFFFF", outline="#CCCCCC", width=1)
                canvas.create_rectangle(content_x2 - col_w, content_y1, content_x2, content_y2, fill="#FFFFFF", outline="#CCCCCC", width=1)
            else:
                canvas.create_rectangle(content_x1, content_y1, content_x2, content_y2, fill="#FFFFFF", outline="#CCCCCC", width=1)

            # Page label
            canvas.create_text(px + pw / 2, py + ph + 15, text=label, font=(self.default_font, 12), fill="#666666")

        # Draw even page (left) - Inside margin on RIGHT side (book binding in middle)
        draw_page(start_x, start_y, page_w, page_h, outside_px, inside_px, "짝수 페이지")

        # Draw odd page (right) - Inside margin on LEFT side (book binding in middle)
        draw_page(start_x + page_w + gap, start_y, page_w, page_h, inside_px, outside_px, "홀수 페이지")

        # Draw binding indicator between pages
        binding_x = start_x + page_w + gap / 2
        canvas.create_line(binding_x, start_y + 5, binding_x, start_y + page_h - 5, fill="#999999", dash=(2, 2))

    def create_action_buttons(self):
        """Create Cancel/Default/OK buttons at bottom"""
        button_frame = CTkFrame(self, fg_color="transparent")
        button_frame.pack(side="bottom", pady=10)

        # 취소 (Cancel) - leftmost
        cancel_btn = CTkButton(
            button_frame, text="취소", command=self.on_cancel,
            font=(self.default_font, 13), fg_color="#8D9397", hover_color="#5F6C37", width=100
        )
        cancel_btn.pack(side="left", padx=5)

        # 기본값 (Default) - center
        default_btn = CTkButton(
            button_frame, text="기본값", command=self.reset_to_defaults,
            font=(self.default_font, 13), fg_color="#8D9397", hover_color="#5F6C37", width=100
        )
        default_btn.pack(side="left", padx=5)

        # 확인 (OK) - rightmost
        ok_btn = CTkButton(
            button_frame, text="확인", command=self.on_ok,
            font=(self.default_font, 13), fg_color="#2B3D2D", hover_color="#5F6C37", width=100
        )
        ok_btn.pack(side="left", padx=5)

    def reset_to_defaults(self):
        """Reset all margin values to defaults"""
        self.parent_window.margin_top_var.set("1.5")
        self.parent_window.margin_bottom_var.set("1.6")
        self.parent_window.margin_left_var.set("1.9")
        self.parent_window.margin_right_var.set("1.9")
        self.parent_window.col_spacing_var.set("1.27")
        self.parent_window.mirror_margins_var.set(False)
        # Update labels and preview for normal mode
        self._apply_mirror_state()
        self.update_paper_preview()

    def validate_inputs(self):
        """Validate inputs with auto-correction for out-of-range values"""
        # Range constants
        MARGIN_MIN = 0.5
        MARGIN_MAX = 5.0
        COL_SPACING_MIN = 0.3
        COL_SPACING_MAX = 3.0

        # Margin fields with their entry widgets and ranges
        margin_items = [
            ('상', self.parent_window.margin_top_var, self.top_entry, MARGIN_MIN, MARGIN_MAX),
            ('하', self.parent_window.margin_bottom_var, self.bottom_entry, MARGIN_MIN, MARGIN_MAX),
            ('좌', self.parent_window.margin_left_var, self.left_entry, MARGIN_MIN, MARGIN_MAX),
            ('우', self.parent_window.margin_right_var, self.right_entry, MARGIN_MIN, MARGIN_MAX),
            ('2단 사이 간격', self.parent_window.col_spacing_var, self.spacing_entry, COL_SPACING_MIN, COL_SPACING_MAX),
        ]

        for name, var, entry, min_val, max_val in margin_items:
            try:
                val = float(var.get())
                if val < min_val:
                    messagebox.showwarning("범위 초과", f"{name}은(는) {min_val} ~ {max_val} cm 범위여야 합니다.\n{min_val}(으)로 설정합니다.")
                    var.set(str(min_val))
                    self._select_entry(entry)
                    return False
                elif val > max_val:
                    messagebox.showwarning("범위 초과", f"{name}은(는) {min_val} ~ {max_val} cm 범위여야 합니다.\n{max_val}(으)로 설정합니다.")
                    var.set(str(max_val))
                    self._select_entry(entry)
                    return False
            except ValueError:
                messagebox.showerror("입력 오류", f"{name}에 올바른 숫자를 입력하세요.")
                self._select_entry(entry)
                return False

        return True

    def _select_entry(self, entry):
        """Focus and select all text in the given entry widget"""
        try:
            entry.focus_set()
            # For CTkEntry, access the internal entry widget
            if hasattr(entry, '_entry'):
                entry._entry.select_range(0, 'end')
            else:
                entry.select_range(0, 'end')
        except Exception:
            pass

    def on_ok(self):
        """Validate, apply changes, and close dialog"""
        if self.validate_inputs():
            self._remove_trace_bindings()
            self.destroy()

    def on_cancel(self):
        """Restore original values and close dialog"""
        if hasattr(self, '_remove_trace_bindings'):
            self._remove_trace_bindings()
        # Restore original values before destroying
        self.parent_window.margin_top_var.set(self.original_values['margin_top'])
        self.parent_window.margin_bottom_var.set(self.original_values['margin_bottom'])
        self.parent_window.margin_left_var.set(self.original_values['margin_left'])
        self.parent_window.margin_right_var.set(self.original_values['margin_right'])
        self.parent_window.col_spacing_var.set(self.original_values['col_spacing'])
        self.parent_window.mirror_margins_var.set(self.original_values['mirror_margins'])
        self.destroy()


class SearchableFontComboBox(CTkFrame):
    """
    A searchable combobox widget for font selection with font preview.

    Features:
    - Prefix matching (search matches from START of font name)
    - Font preview (sample text rendered in each font)
    - Letter key jump (press letter while dropdown open to jump to fonts)
    - Persistent dropdown (stays open until explicit selection)
    """

    def __init__(
        self,
        master,
        variable: tk.StringVar = None,
        values: list = None,
        command=None,
        width: int = 180,
        height: int = 28,
        font_tuple: tuple = None,
        # Color configuration (matching existing style)
        fg_color: str = "white",
        text_color: str = "black",
        border_color: str = "#FEF9E0",
        button_color: str = "#FEF9E0",
        button_hover_color: str = "#CA7900",
        dropdown_fg_color: str = "white",
        dropdown_hover_color: str = "#CA7900",
        dropdown_text_color: str = "black",
        # Font preview options
        preview_text: str = "AaBb가나",
        **kwargs
    ):
        super().__init__(master, fg_color="transparent", **kwargs)

        # Store configuration
        self._variable = variable if variable else tk.StringVar()
        self._values = list(values) if values else []
        self._filtered_values = self._values.copy()
        self._command = command
        self._preview_text = preview_text

        # UI colors
        self._fg_color = fg_color
        self._text_color = text_color
        self._border_color = border_color
        self._button_color = button_color
        self._button_hover_color = button_hover_color
        self._dropdown_fg_color = dropdown_fg_color
        self._dropdown_hover_color = dropdown_hover_color
        self._dropdown_text_color = dropdown_text_color

        # Widget dimensions
        self._width = width
        self._height = height
        self._font_tuple = font_tuple if font_tuple else ("Arial", 12)

        # State tracking
        self._dropdown_window = None
        self._dropdown_container = None
        self._dropdown_items = {}  # {font_name: {"frame": frame, "name_label": label, "preview_label": label}}
        self._visible_items = []  # List of currently visible font names
        self._selected_index = -1
        self._is_dropdown_open = False
        self._updating = False  # Prevent re-entrance
        self._items_created = False  # Track if items have been created

        # Binding IDs for cleanup
        self._root_click_handler_id = None
        self._root_escape_handler_id = None

        self._previous_grab_owner = None

        # Build UI
        self._create_widgets()
        self._create_bindings()


    def _create_widgets(self):
        """Create the main entry widget with dropdown button."""

        # Main container frame with border
        self._main_frame = CTkFrame(
            self,
            fg_color=self._fg_color,
            border_color=self._border_color,
            border_width=1,
            corner_radius=6,
            width=self._width,
            height=self._height
        )
        self._main_frame.pack(fill="both", expand=True)
        self._main_frame.pack_propagate(False)

        # Configure grid: entry takes most space, button on right
        self._main_frame.grid_columnconfigure(0, weight=1)
        self._main_frame.grid_columnconfigure(1, weight=0)
        self._main_frame.grid_rowconfigure(0, weight=1)

        # Entry widget for search input (NOT using textvariable to avoid re-entrance)
        self._entry = CTkEntry(
            self._main_frame,
            font=self._font_tuple,
            fg_color=self._fg_color,
            text_color=self._text_color,
            border_width=0,
            corner_radius=0,
            height=self._height - 4
        )
        self._entry.grid(row=0, column=0, sticky="nsew", padx=(6, 0), pady=2)

        # Initialize entry with current variable value
        current_val = self._variable.get()
        if current_val:
            self._entry.insert(0, current_val)

        # Dropdown button with arrow
        self._dropdown_button = CTkButton(
            self._main_frame,
            text="▼",
            width=self._height,
            height=self._height - 4,
            font=(self._font_tuple[0], 10),
            fg_color=self._button_color,
            hover_color=self._button_hover_color,
            text_color=self._text_color,
            border_width=0,
            corner_radius=4,
            command=self._toggle_dropdown
        )
        self._dropdown_button.grid(row=0, column=1, sticky="nse", padx=2, pady=2)

    def _create_bindings(self):
        """Set up event bindings for the widget."""

        # Entry bindings
        self._entry.bind("<KeyRelease>", self._on_key_release)
        self._entry.bind("<Return>", self._on_enter_pressed)
        self._entry.bind("<Escape>", self._on_escape_pressed)
        self._entry.bind("<Down>", self._on_arrow_down)
        self._entry.bind("<Up>", self._on_arrow_up)
        self._entry.bind("<FocusIn>", self._on_entry_focus_in)

        # Entry click opens dropdown
        self._entry.bind("<Button-1>", self._on_entry_click)

        # Variable trace for external changes
        self._variable.trace_add("write", self._on_variable_change)

        # Cleanup on destroy
        self.bind("<Destroy>", self._on_destroy)

    def _on_entry_click(self, event):
        """Handle click on entry - open dropdown."""
        if not self._is_dropdown_open:
            self._open_dropdown()

    def _on_entry_focus_in(self, event):
        """Handle focus entering entry."""
        # Don't auto-open on focus, only on explicit click
        pass

    def _toggle_dropdown(self):
        """Toggle dropdown open/close."""
        if self._is_dropdown_open:
            self._close_dropdown()
        else:
            self._open_dropdown()

    def _open_dropdown(self):
        """Open the dropdown window with font list and previews."""

        if self._dropdown_window and self._dropdown_window.winfo_exists():
            return

        try:
            root = self.winfo_toplevel()
            grab_owner = root.grab_current()
            self._previous_grab_owner = grab_owner
        except Exception as exc:
            pass

        self._is_dropdown_open = True

        # Reset filter to show all fonts
        self._filtered_values = self._values.copy()
        self._visible_items = self._filtered_values.copy()

        # Create toplevel window
        self._dropdown_window = CTkToplevel(self)
        self._dropdown_window.withdraw()  # Hide until positioned
        self._dropdown_window.wm_overrideredirect(True)  # No window decorations
        self._dropdown_window.attributes("-topmost", True)
        self._dropdown_window.configure(fg_color=self._dropdown_fg_color)

        try:
            self._dropdown_window.transient(self.winfo_toplevel())
        except Exception:
            pass

        # Calculate dropdown dimensions
        max_visible_items = 10
        item_height = 32  # Height per font item (with preview)

        # Calculate appropriate height based on number of items
        if len(self._filtered_values) > max_visible_items:
            container_height = max_visible_items * item_height
        else:
            container_height = max(len(self._filtered_values), 1) * item_height + 4

        # Scrollable container for font items
        self._dropdown_container = CTkScrollableFrame(
            self._dropdown_window,
            fg_color=self._dropdown_fg_color,
            height=container_height,
            scrollbar_button_color="#D1D5DB",
            scrollbar_button_hover_color="#9CA3AF"
        )
        self._dropdown_container.pack(fill="both", expand=True, padx=1, pady=1)

        # Create all dropdown items (cached for performance)
        if not self._items_created:
            self._create_all_dropdown_items()
            self._items_created = True

        # Show all items initially
        self._show_filtered_items()

        # Find and select current value
        self._select_current_value()

        # Position and show dropdown
        self._position_dropdown()
        self._dropdown_window.deiconify()
        self._dropdown_window.lift()

        # Important: ensure dropdown gets pointer events even if parent has a grab (modal dialogs)
        try:
            self._dropdown_window.grab_set()
        except Exception as exc:
            pass

        # Set up event bindings for dropdown
        self._setup_dropdown_bindings()

        # Focus entry for keyboard input
        self._entry.focus_set()

    def _create_all_dropdown_items(self):
        """Create all dropdown item frames (cached for performance)."""
        self._dropdown_items.clear()

        for font_name in self._values:
            item_data = self._create_dropdown_item(font_name)
            self._dropdown_items[font_name] = item_data

    def _create_dropdown_item(self, font_name: str) -> dict:
        """Create a single dropdown item with font name and preview."""

        item_frame = CTkFrame(
            self._dropdown_container,
            fg_color=self._dropdown_fg_color,
            height=30,
            cursor="hand2"
        )
        # Don't pack yet - will be managed by _show_filtered_items

        # Configure internal layout
        item_frame.grid_columnconfigure(0, weight=1)
        item_frame.grid_columnconfigure(1, weight=0)
        item_frame.grid_rowconfigure(0, weight=1)
        item_frame.grid_propagate(False)

        # Font name label (left side, using UI font)
        name_label = CTkLabel(
            item_frame,
            text=font_name,
            font=self._font_tuple,
            text_color=self._dropdown_text_color,
            fg_color="transparent",
            anchor="w"
        )
        name_label.grid(row=0, column=0, sticky="nsw", padx=(8, 4))

        # Font preview label (right side, rendered in the actual font)
        preview_label = None
        try:
            preview_font = (font_name, 12)
            preview_label = CTkLabel(
                item_frame,
                text=self._preview_text,
                font=preview_font,
                text_color="#666666",
                fg_color="transparent",
                width=80,
                anchor="e"
            )
            preview_label.grid(row=0, column=1, sticky="nse", padx=(4, 8))
        except Exception:
            # If font rendering fails, show placeholder
            preview_label = CTkLabel(
                item_frame,
                text="(미리보기 불가)",
                font=self._font_tuple,
                text_color="#999999",
                fg_color="transparent",
                width=80,
                anchor="e"
            )
            preview_label.grid(row=0, column=1, sticky="nse", padx=(4, 8))

        # Store item data
        item_data = {
            "frame": item_frame,
            "name_label": name_label,
            "preview_label": preview_label,
            "font_name": font_name
        }

        # Bind events to all components
        for widget in [item_frame, name_label, preview_label]:
            if widget:
                widget.bind("<Enter>", lambda e, fn=font_name: self._on_item_hover(fn, True))
                widget.bind("<Leave>", lambda e, fn=font_name: self._on_item_hover(fn, False))
                widget.bind("<Button-1>", lambda e, fn=font_name: self._on_item_select(fn))
                
                # MouseWheel Binding
                widget.bind("<MouseWheel>", self._handle_mousewheel)
                widget.bind("<Button-4>", self._handle_mousewheel)
                widget.bind("<Button-5>", self._handle_mousewheel)

        return item_data

    def _show_filtered_items(self):
        """Show only filtered items in dropdown."""
        # Hide all items first
        for font_name, item_data in self._dropdown_items.items():
            item_data["frame"].pack_forget()

        # Show filtered items
        self._visible_items = self._filtered_values.copy()
        for font_name in self._filtered_values:
            if font_name in self._dropdown_items:
                self._dropdown_items[font_name]["frame"].pack(fill="x", padx=2, pady=1)

        # Update dropdown size
        if self._dropdown_window and self._dropdown_window.winfo_exists():
            self._dropdown_container.update_idletasks()
            self._position_dropdown()

    def _select_current_value(self):
        """Select the current value in dropdown."""
        current_value = self._variable.get()
        if current_value in self._filtered_values:
            self._selected_index = self._filtered_values.index(current_value)
            self._highlight_selected_item()
            # Scroll to selected item after a brief delay to ensure layout is complete
            self.after(50, self._scroll_to_selected_item)
        else:
            self._selected_index = 0 if self._filtered_values else -1
            self._highlight_selected_item()

    def _position_dropdown(self):
        """Position dropdown below the entry widget."""
        if not self._dropdown_window or not self._dropdown_window.winfo_exists():
            return

        self._dropdown_window.update_idletasks()

        # Get entry position
        entry_x = self.winfo_rootx()
        entry_y = self.winfo_rooty() + self.winfo_height()

        # Calculate dropdown dimensions
        dropdown_width = max(self.winfo_width(), 280)  # Minimum width for preview

        # Get actual dropdown height
        max_visible_items = 10
        item_height = 32
        num_items = min(len(self._filtered_values), max_visible_items)
        dropdown_height = max(num_items * item_height + 10, 50)

        # Screen bounds check
        screen_height = self.winfo_screenheight()
        screen_width = self.winfo_screenwidth()

        # Adjust X position if dropdown would go off screen
        if entry_x + dropdown_width > screen_width:
            entry_x = screen_width - dropdown_width - 10

        # Position above entry if not enough space below
        if entry_y + dropdown_height > screen_height - 50:
            entry_y = self.winfo_rooty() - dropdown_height

        self._dropdown_window.geometry(f"{dropdown_width}x{dropdown_height}+{entry_x}+{entry_y}")

    def _setup_dropdown_bindings(self):
        """Set up bindings specific to the dropdown window."""
        if not self._dropdown_window:
            return

        # Close on click outside
        try:
            root = self.winfo_toplevel()
            self._root_click_handler_id = root.bind("<Button-1>", self._check_click_outside, add="+")
        except Exception:
            pass

        # Keyboard navigation in dropdown
        self._dropdown_window.bind("<Escape>", self._on_escape_pressed)

        # Also bind ESC to root for reliability
        try:
            root = self.winfo_toplevel()
            self._root_escape_handler_id = root.bind("<Escape>", self._on_escape_pressed, add="+")
        except Exception:
            pass

        # Bind mouse wheel for scrolling (required for overrideredirect windows)
        self._dropdown_window.bind("<MouseWheel>", self._handle_mousewheel)
        self._dropdown_window.bind("<Button-4>", self._handle_mousewheel)
        self._dropdown_window.bind("<Button-5>", self._handle_mousewheel)
        
        self._dropdown_container.bind("<MouseWheel>", self._handle_mousewheel)
        self._dropdown_container.bind("<Button-4>", self._handle_mousewheel)
        self._dropdown_container.bind("<Button-5>", self._handle_mousewheel)
        
        try:
            canvas = self._dropdown_container._parent_canvas
            canvas.bind("<MouseWheel>", self._handle_mousewheel)
            canvas.bind("<Button-4>", self._handle_mousewheel)
            canvas.bind("<Button-5>", self._handle_mousewheel)
        except AttributeError:
            pass

    def _on_key_release(self, event):
        """Handle key release in entry - filter dropdown list."""

        # Ignore special keys
        if event.keysym in ('Return', 'Escape', 'Up', 'Down', 'Left', 'Right',
                           'Shift_L', 'Shift_R', 'Control_L', 'Control_R', 'Alt_L', 'Alt_R',
                           'Tab', 'BackSpace', 'Delete'):
            # Handle BackSpace and Delete specially - still filter
            if event.keysym not in ('BackSpace', 'Delete'):
                return

        # Open dropdown if not open
        if not self._is_dropdown_open:
            self._open_dropdown()

        # Get current search text
        search_text = self._entry.get().strip().lower()

        if search_text:
            # PREFIX MATCHING: Filter fonts that START with search text
            self._filtered_values = [
                font_name for font_name in self._values
                if font_name.lower().startswith(search_text)
            ]
        else:
            # Show all fonts if search is empty
            self._filtered_values = self._values.copy()

        # Refresh dropdown items
        self._show_filtered_items()

        # Reset selection to first item
        self._selected_index = 0 if self._filtered_values else -1
        self._highlight_selected_item()

        if self._filtered_values:
            self._scroll_to_selected_item()

    def _on_arrow_down(self, event):
        """Move selection down in dropdown."""
        if not self._is_dropdown_open:
            self._open_dropdown()
            return "break"

        if self._filtered_values:
            self._selected_index = min(self._selected_index + 1, len(self._filtered_values) - 1)
            self._highlight_selected_item()
            self._scroll_to_selected_item()
        return "break"

    def _on_arrow_up(self, event):
        """Move selection up in dropdown."""
        if not self._is_dropdown_open:
            return "break"

        if self._filtered_values:
            self._selected_index = max(self._selected_index - 1, 0)
            self._highlight_selected_item()
            self._scroll_to_selected_item()
        return "break"

    def _on_enter_pressed(self, event):
        """Confirm selection on Enter key."""
        if self._is_dropdown_open and self._selected_index >= 0:
            if self._selected_index < len(self._filtered_values):
                selected_font = self._filtered_values[self._selected_index]
                self._select_font(selected_font)
        return "break"

    def _on_escape_pressed(self, event):
        """Close dropdown on Escape key - preserve current entry text."""
        self._close_dropdown()
        return "break"

    def _on_item_hover(self, font_name: str, is_hovering: bool):
        """Handle mouse hover on dropdown item."""
        if font_name not in self._dropdown_items:
            return

        item_data = self._dropdown_items[font_name]

        if is_hovering:
            item_data["frame"].configure(fg_color=self._dropdown_hover_color)
            # Update selected index to this item
            if font_name in self._filtered_values:
                self._selected_index = self._filtered_values.index(font_name)
        else:
            # Reset color if not the currently selected item
            if font_name in self._filtered_values:
                idx = self._filtered_values.index(font_name)
                if idx != self._selected_index:
                    item_data["frame"].configure(fg_color=self._dropdown_fg_color)

    def _on_item_select(self, font_name: str):
        """Handle mouse click on dropdown item."""
        self._select_font(font_name)

    def _select_font(self, font_name: str):
        """Select a font and update the variable."""
        self._updating = True
        try:
            # Update entry text
            self._entry.delete(0, "end")
            self._entry.insert(0, font_name)

            # Update variable
            self._variable.set(font_name)

            # Close dropdown
            self._close_dropdown()

            # Call command callback if provided
            if self._command:
                self._command(font_name)
        finally:
            self._updating = False

    def _on_variable_change(self, *args):
        """Handle external changes to the variable."""
        if self._updating:
            return

        # Update entry if value changed externally
        current_var = self._variable.get()
        current_entry = self._entry.get()

        if current_entry != current_var:
            self._updating = True
            try:
                self._entry.delete(0, "end")
                self._entry.insert(0, current_var)
            finally:
                self._updating = False

    def _highlight_selected_item(self):
        """Update visual highlighting for selected item."""
        for i, font_name in enumerate(self._filtered_values):
            if font_name in self._dropdown_items:
                item_data = self._dropdown_items[font_name]
                if i == self._selected_index:
                    item_data["frame"].configure(fg_color=self._dropdown_hover_color)
                else:
                    item_data["frame"].configure(fg_color=self._dropdown_fg_color)

    def _scroll_to_selected_item(self):
        """Scroll dropdown to make selected item visible."""
        if self._selected_index < 0 or self._selected_index >= len(self._filtered_values):
            return

        if not self._dropdown_container:
            return

        try:
            font_name = self._filtered_values[self._selected_index]
            if font_name not in self._dropdown_items:
                return

            item_frame = self._dropdown_items[font_name]["frame"]

            # Get the internal canvas from CTkScrollableFrame
            canvas = self._dropdown_container._parent_canvas

            # Get item position relative to container
            item_y = item_frame.winfo_y()
            item_height = item_frame.winfo_height()
            container_height = self._dropdown_container.winfo_height()

            # Get scroll region
            scroll_region = canvas.bbox("all")
            if not scroll_region:
                return

            scroll_height = scroll_region[3] - scroll_region[2]
            if scroll_height <= 0:
                return

            # Calculate visible region
            current_scroll = canvas.yview()
            visible_top = current_scroll[0] * scroll_height
            visible_bottom = current_scroll[1] * scroll_height

            # Check if item is visible
            if item_y < visible_top:
                # Scroll up to show item
                new_pos = item_y / scroll_height
                canvas.yview_moveto(max(0, new_pos))
            elif item_y + item_height > visible_bottom:
                # Scroll down to show item
                new_pos = (item_y + item_height - container_height) / scroll_height
                canvas.yview_moveto(min(1, new_pos))
        except Exception:
            # Silently ignore scroll errors
            pass

    def _handle_mousewheel(self, event):
        """Cross-platform mouse wheel handler."""
        if not self._dropdown_container:
            return

        try:
            canvas = self._dropdown_container._parent_canvas
            if not canvas.winfo_exists(): return
        except AttributeError:
            return

        if sys.platform.startswith("darwin"):
            # macOS
            delta = event.delta
            if delta == 0: return "break"
            canvas.yview_scroll(int(-1 * delta), "units")
            
        elif sys.platform.startswith("win"):
            # Windows
            if event.delta:
                canvas.yview_scroll(int(-1 * (event.delta / 120)), "units")
                
        else:
            # Linux
            if event.num == 4:
                canvas.yview_scroll(-3, "units")
            elif event.num == 5:
                canvas.yview_scroll(3, "units")
            elif hasattr(event, "delta") and event.delta:
                canvas.yview_scroll(int(-1 * (event.delta / 120)), "units")
        
        return "break"

    def _check_click_outside(self, event):
        """Check if click is outside the widget and close dropdown."""
        if not self._dropdown_window or not self._dropdown_window.winfo_exists():
            return

        # Get widget bounds
        widget_x = self.winfo_rootx()
        widget_y = self.winfo_rooty()
        widget_width = self.winfo_width()
        widget_height = self.winfo_height()

        # Get dropdown bounds
        dropdown_x = self._dropdown_window.winfo_rootx()
        dropdown_y = self._dropdown_window.winfo_rooty()
        dropdown_width = self._dropdown_window.winfo_width()
        dropdown_height = self._dropdown_window.winfo_height()

        click_x = event.x_root
        click_y = event.y_root

        # Check if click is inside widget or dropdown
        in_widget = (widget_x <= click_x <= widget_x + widget_width and
                    widget_y <= click_y <= widget_y + widget_height)
        in_dropdown = (dropdown_x <= click_x <= dropdown_x + dropdown_width and
                      dropdown_y <= click_y <= dropdown_y + dropdown_height)

        if not in_widget and not in_dropdown:
            self._close_dropdown()

    def _close_dropdown(self):
        """Close the dropdown window."""
        # Unbind root click handler
        try:
            root = self.winfo_toplevel()
            if self._root_click_handler_id:
                root.unbind("<Button-1>", self._root_click_handler_id)
                self._root_click_handler_id = None
            if self._root_escape_handler_id:
                root.unbind("<Escape>", self._root_escape_handler_id)
                self._root_escape_handler_id = None
        except Exception:
            pass

        # Destroy dropdown window
        if self._dropdown_window and self._dropdown_window.winfo_exists():
            try:
                self._dropdown_window.grab_release()
            except Exception:
                pass
            self._dropdown_window.destroy()

        self._dropdown_window = None
        self._dropdown_container = None
        self._is_dropdown_open = False
        self._selected_index = -1

        # Clear cached items (will be recreated on next open)
        self._dropdown_items.clear()
        self._items_created = False

        # Restore previous grab owner (e.g., the modal 편집 옵션 window)
        if self._previous_grab_owner:
            try:
                if self._previous_grab_owner.winfo_exists():
                    self._previous_grab_owner.grab_set()
            except Exception as exc:
                pass
            self._previous_grab_owner = None

    def _on_destroy(self, event):
        """Clean up when widget is destroyed."""
        if event.widget == self:
            self._close_dropdown()

    # Public interface methods for compatibility
    def get(self) -> str:
        """Get the current selected font name."""
        return self._variable.get()

    def set(self, value: str):
        """Set the font selection."""
        self._updating = True
        try:
            self._variable.set(value)
            self._entry.delete(0, "end")
            self._entry.insert(0, value)
        finally:
            self._updating = False

    def configure(self, **kwargs):
        """Configure widget options."""
        if "variable" in kwargs:
            self._variable = kwargs.pop("variable")
            # Update entry
            val = self._variable.get()
            self._entry.delete(0, "end")
            self._entry.insert(0, val)

        if "values" in kwargs:
            self._values = list(kwargs.pop("values"))
            self._filtered_values = self._values.copy()
            self._items_created = False  # Force recreation

        if "command" in kwargs:
            self._command = kwargs.pop("command")

        if "state" in kwargs:
            state = kwargs.pop("state")
            self._state = state
            if state == "disabled":
                self._entry.configure(state="disabled")
                self._dropdown_button.configure(state="disabled")
            else:
                self._entry.configure(state="normal")
                self._dropdown_button.configure(state="normal")

        if "text_color" in kwargs:
            text_color = kwargs.pop("text_color")
            self._text_color = text_color
            self._entry.configure(text_color=text_color)

        super().configure(**kwargs)

    def cget(self, key):
        """Get widget configuration option."""
        if key == "variable":
            return self._variable
        elif key == "values":
            return self._values
        return super().cget(key)


class ThumbnailPopupWindow(customtkinter.CTkToplevel):
    def __init__(self, master, on_submit_callback, applyoption_var, docx_folder=None, **kwargs):
        # If caller passed docx_folder inside kwargs, pull it out so CTkToplevel doesn't see it
        if 'docx_folder' in kwargs:
            docx_folder = kwargs.pop('docx_folder')

        super().__init__(master, **kwargs)
        self.title("편집 옵션")
        self.geometry("1000x680")
        self.configure(fg_color="#EFF3F0")
        # self.attributes("-topmost", True)
        self.grab_set()
        self.focus_set()

        # Store the callback
        self.on_submit_callback = on_submit_callback
        self.applyoption_var = applyoption_var

        self.multicol_var = IntVar(value=0)  # 0=single, 1=two-column
        self.border_flag_var = IntVar(value=1)  # 1=borders enabled (checkbox checked), 0=no borders (checkbox unchecked)
        self.passage_number_numeric_var = IntVar(value=0)  # 0=use original passage_id, 1=use numeric numbering
        self.separate_answer_var = IntVar(value=0)  # 0=single file, 1=separate answer file
        self.new_page_per_section_var = IntVar(value=1)  # 1=new page between sections (default), 0=no page break
        self.page_break_after_passages_enabled_var = IntVar(value=0)  # 0=off, 1=on
        self.page_break_after_passages_count_var = StringVar(value="4")




        # ------- base/project path -------
        if getattr(sys, 'frozen', False):
            # Bundled app (e.g., PyInstaller): assets should be next to the executable
            self.project_root = Path(sys._MEIPASS)
        else:
            # Running from source: go up from modules/ to project root .../AutoQM
            self.project_root = Path(__file__).resolve().parent.parent

        app_data_dir = Path.home() / ".my_application_data"

        # Resolve the bookcovers folder
        self.docx_folder = Path(docx_folder) if docx_folder else (self.project_root / "assets" / "bookcovers")
        self.docx_folder = self.docx_folder.resolve()

        self.편집옵션_PRESETS_FILE_for_editor = app_data_dir / 'edit_presets.json'

        font_handler.set_default_font()
        self.default_font = font_handler.default_font

        # Initialize 교재제작 옵션 variables
        # '지문분석' 옵션 (same defaults as "지문" section)
        self.jimun_analysis_font_var = StringVar(value=self.default_font)
        self.jimun_analysis_font_size_var = StringVar(value="8")
        self.jimun_analysis_font_color_var = StringVar(value="#000000")
        self.jimun_analysis_line_spacing_var = StringVar(value="1.2")
        self.jimun_analysis_sentence_split_var = IntVar(value=0)  # 지문을 문장별로 분리
        self.jimun_analysis_interpretation_right_var = IntVar(value=0)  # 해석을 지문 우측에 두기

        # '한글해석' 옵션
        self.hanjul_haesuk_font_var = StringVar(value=self.default_font)
        self.hanjul_haesuk_font_size_var = StringVar(value="6")
        self.hanjul_haesuk_font_color_var = StringVar(value="#000000")
        self.hanjul_haesuk_line_spacing_var = StringVar(value="1")

        # '한줄해석연습' 옵션
        self.hanjul_interpretation_include_answer_var = IntVar(value=1)

        # '한줄영작연습' 옵션 (단어 스크램블 checked by default)
        self.hanjul_writing_include_answer_var = IntVar(value=1)
        self.hanjul_writing_word_scramble_var = IntVar(value=1)
        self.hanjul_writing_word_form_change_var = IntVar(value=0)
        self.hanjul_writing_korean_only_var = IntVar(value=0)

        # '문제' 옵션
        # 0 = T/F 형식 (default), 1 = 5지선다
        self.naeyongilchi_question_type_var = IntVar(value=0)

        # preset_last_state
        self.편집옵션_LAST_STATE_FILE_in_ThumbnailPopupWindow = app_data_dir / 'export_options_presets_last_state.json'

        # Ensure folder exists
        if not self.docx_folder.exists():
            raise FileNotFoundError(f"Bookcovers folder not found: {self.docx_folder}")

        # Generate thumbnails once
        image_paths = self.generate_thumbnails_for_docx(self.docx_folder)

        # UI setup
        self.setup_title_frame()
        self.setup_image_frame(image_paths, self.docx_folder)  # pass the resolved folder

        # **Add the new cover text frame here**
        self.setup_cover_text_frame()

        self.setup_font_selection_frame()
        self.setup_action_buttons()
        self.update_presets_combobox()

        # Load and apply the last state
        self.last_state = self.load_preset_last_state_from_file()
        self.apply_last_state_to_ui(self.last_state)

        # Print the last selected bookcover path after applying the state
        if hasattr(self, 'last_clicked_button'):
            if self.last_clicked_button and self.last_clicked_button != "No button clicked":
                if Path(self.last_clicked_button).exists():
                    print(f"Popup window opened. Last selected bookcover: {self.last_clicked_button}")
                else:
                    print(f"Popup window opened. Previous bookcover not found, using default: {self.last_clicked_button}")
            else:
                print("Popup window opened. No previous bookcover selection found.")
        else:
            print("Popup window opened. No previous bookcover selection found.")

        # Modifier key by OS
        if sys.platform.startswith('darwin'):
            modifier_key = 'Command'
        elif sys.platform.startswith('win'):
            modifier_key = 'Control'
        else:
            modifier_key = 'Control'

        self.bind(f'<{modifier_key}-Return>', self.submit_edit)
        self.bind("<Escape>", self.on_popup_close_edit)


    def _select_first_cover(self):
        """Programmatically select the first cover (with full side-effects)."""
        if getattr(self, "scrollable_image_frame", None) and self.scrollable_image_frame.button_list:
            first_btn = self.scrollable_image_frame.button_list[0]
            self.last_clicked_button = str(first_btn.docx_path)
            # Use the same routine as a real user click:
            self.scrollable_image_frame.button_command(self.last_clicked_button, first_btn)
            return True
        return False




    def load_presets_from_file(self):
        try:
            with open(self.편집옵션_PRESETS_FILE_for_editor, 'r', encoding='utf-8') as f:
                return json.load(f)
        except FileNotFoundError:
            return {}  # Return an empty dict if the file doesn't exist
        except json.JSONDecodeError:
            messagebox.showerror("Error", "Failed to decode presets file. Please check the file format.")
            return {}

    def save_presets_to_file(self, presets):
        try:
            with open(self.편집옵션_PRESETS_FILE_for_editor, 'w', encoding='utf-8') as f:
                json.dump(presets, f, ensure_ascii=False, indent=4)
        except Exception as e:
            messagebox.showerror("Error", f"Failed to save presets: {e}")


    def load_preset_last_state_from_file(self):
        #print("load_preset_last_state_from_file called")
        try:
            with open(self.편집옵션_LAST_STATE_FILE_in_ThumbnailPopupWindow, 'r', encoding='utf-8') as f:
                return json.load(f)
        except FileNotFoundError:
            return {}  # Return an empty dict if the file doesn't exist
        except json.JSONDecodeError:
            messagebox.showerror("Error", "Failed to decode presets file. Please check the file format.")
            return {}


    def save_preset_last_state_to_file(self, presets):
        #print("save_preset_last_state_to_file called")
        try:
            with open(self.편집옵션_LAST_STATE_FILE_in_ThumbnailPopupWindow, 'w', encoding='utf-8') as f:
                json.dump(presets, f, ensure_ascii=False, indent=4)
        except Exception as e:
            messagebox.showerror("Error", f"Failed to save presets: {e}")



    def apply_last_state_to_ui(self, last_state):
        if not last_state:
            # Prefer a real "click" on the first cover
            if self._select_first_cover():
                return
            return  # no buttons → nothing to do
                
        # Example of applying last state values
        self.font_for_title_var.set(last_state.get("selected_font_for_title", "Default Font"))

        # Apply the loaded preferences
        self.font_for_title_var.set(last_state.get("selected_font_for_title", "나눔고딕"))
        self.font_for_num_var.set(last_state.get("selected_font_for_num", "나눔고딕"))
        self.font_for_text_var.set(last_state.get("selected_font_for_text", "나눔고딕"))
        self.font_for_exp_var.set(last_state.get("selected_font_for_exp", "나눔고딕"))
        
        # Applying colors
        self.frame1_color_select_button.configure(text_color=last_state.get("color_heading", "#335581"))
        self.frame2_color_select_button.configure(text_color=last_state.get("color_num", "#4B76B0"))
        self.frame3_color_select_button.configure(text_color=last_state.get("color_text", "#000000"))
        self.frame4_color_select_button.configure(text_color=last_state.get("color_exp", "#000000"))
        
        # Applying sizes
        self.frame1_entry.delete(0, 'end')
        self.frame1_entry.insert(0, last_state.get("size_heading", 12))
        self.frame2_entry.delete(0, 'end')
        self.frame2_entry.insert(0, last_state.get("size_num", 10))
        self.frame3_entry.delete(0, 'end')
        self.frame3_entry.insert(0, last_state.get("size_text", 8))
        self.frame3_entry_line.delete(0, 'end')
        self.frame3_entry_line.insert(0, last_state.get("line_text", 1.2))
        self.frame4_entry.delete(0, 'end')
        self.frame4_entry.insert(0, last_state.get("size_exp", 7))
        self.frame4_entry_line.delete(0, 'end')
        self.frame4_entry_line.insert(0, last_state.get("line_exp", 1.1))
        
        # Applying the tag removal flag
        if last_state.get("tagremoval_flag") == 1:
            self.tagremoval_checkbox.select()
        elif last_state.get("tagremoval_flag") == 0:
            self.tagremoval_checkbox.deselect()

        # Restore 다단 나누기
        self.multicol_var.set(last_state.get("two_column", 0))
        #print(f"self.multicol_var: {self.multicol_var.get()}\n\n")

        # Restore 지문 테두리 (1=borders enabled/checked, 0=no borders/unchecked)
        border_flag_from_state = last_state.get("border_flag", 1)
        self.border_flag_var.set(border_flag_from_state)

        # Restore 지문번호 숫자로 사용 (0=use original passage_id, 1=use numeric numbering)
        passage_number_numeric_from_state = last_state.get("passage_number_numeric", 0)
        self.passage_number_numeric_var.set(passage_number_numeric_from_state)

        # Restore 정답지 분리 (0=single file, 1=separate answer file)
        separate_answer_key_from_state = last_state.get("separate_answer_key", 0)
        self.separate_answer_var.set(separate_answer_key_from_state)

        # Restore 유형마다 새 페이지 (1=new page between sections, 0=no page break)
        new_page_per_section_from_state = last_state.get("new_page_per_section", 1)
        self.new_page_per_section_var.set(new_page_per_section_from_state)

        # Restore 페이지당 문제 수 (enabled + count)
        page_break_after_passages_enabled = last_state.get("page_break_after_passages_enabled", 0)
        self.page_break_after_passages_enabled_var.set(page_break_after_passages_enabled)
        page_break_after_passages_count = last_state.get("page_break_after_passages_count", "4")
        self.page_break_after_passages_count_var.set(str(page_break_after_passages_count))

        # **Set the new text entries from last_state**
        self.school_grade_var.set(last_state.get("school_grade", "2025 한국학교 1학년 1학기"))
        self.exam_type_var.set(last_state.get("exam_type", "중간고사 내신대비"))
        self.week_var.set(last_state.get("week", "1주차 숙제"))

        # Restore 교재제작 옵션 from last_state
        self.jimun_analysis_font_var.set(last_state.get("jimun_analysis_font", self.default_font))
        self.jimun_analysis_font_size_var.set(last_state.get("jimun_analysis_font_size", "8"))
        self.jimun_analysis_font_color_var.set(last_state.get("jimun_analysis_font_color", "#000000"))
        self.jimun_analysis_line_spacing_var.set(last_state.get("jimun_analysis_line_spacing", "1.2"))
        self.jimun_analysis_sentence_split_var.set(last_state.get("jimun_analysis_sentence_split", 0))
        self.jimun_analysis_interpretation_right_var.set(last_state.get("jimun_analysis_interpretation_right", 0))
        self.hanjul_haesuk_font_var.set(last_state.get("hanjul_haesuk_font", self.default_font))
        self.hanjul_haesuk_font_size_var.set(last_state.get("hanjul_haesuk_font_size", "8"))
        self.hanjul_haesuk_font_color_var.set(last_state.get("hanjul_haesuk_font_color", "#000000"))
        self.hanjul_haesuk_line_spacing_var.set(last_state.get("hanjul_haesuk_line_spacing", "1.2"))
        self.hanjul_interpretation_include_answer_var.set(last_state.get("hanjul_interpretation_include_answer", 1))
        self.hanjul_writing_include_answer_var.set(last_state.get("hanjul_writing_include_answer", 1))
        self.hanjul_writing_word_scramble_var.set(last_state.get("hanjul_writing_word_scramble", 1))
        self.hanjul_writing_word_form_change_var.set(last_state.get("hanjul_writing_word_form_change", 0))
        self.hanjul_writing_korean_only_var.set(last_state.get("hanjul_writing_korean_only", 0))

        # Restore 문제 옵션 from last_state
        self.naeyongilchi_question_type_var.set(last_state.get("naeyongilchi_question_type", 0))

        # Restore 여백설정 from last_state (set on MainFrame)
        self.master.margin_top_var.set(last_state.get("margin_top", "1.5"))
        self.master.margin_bottom_var.set(last_state.get("margin_bottom", "1.6"))
        self.master.margin_left_var.set(last_state.get("margin_left", "1.9"))
        self.master.margin_right_var.set(last_state.get("margin_right", "1.9"))
        self.master.col_spacing_var.set(last_state.get("col_spacing", "1.27"))

        # Restore mirror margins with boolean coercion
        mirror_val = last_state.get("mirror_margins", False)
        if isinstance(mirror_val, str):
            mirror_val = mirror_val.lower() in ('true', '1', 'yes')
        self.master.mirror_margins_var.set(mirror_val)


        # Decide target path
        target = last_state.get("last_clicked_button")

        # If nothing saved or explicitly "No button clicked" → select first
        if not target or target == "No button clicked":
            if self._select_first_cover():
                return
            # If no buttons at all, just bail
            return

        # If saved path doesn’t exist → try basename recovery, otherwise select first
        if not Path(target).exists():
            saved_basename = Path(target).name
            if "_" in saved_basename:
                parts = saved_basename.split("_", 1)
                if parts[0].isdigit():
                    saved_basename = parts[1]
            recovered = self._find_book_cover_by_basename(saved_basename)
            if recovered:
                self.last_clicked_button = str(recovered)
                self.highlight_last_clicked_button(self.last_clicked_button)
                if hasattr(self, "last_state") and isinstance(self.last_state, dict):
                    self.last_state["last_clicked_button"] = self.last_clicked_button
                    if hasattr(self, "save_preset_last_state_to_file"):
                        self.save_preset_last_state_to_file(self.last_state)
                return
            if self._select_first_cover():
                return
            return

        # Otherwise highlight the saved one
        self.last_clicked_button = target
        self.highlight_last_clicked_button(self.last_clicked_button)
     
        """
        # Assuming last_state is already defined and contains your last state data
        self.last_clicked_button = last_state.get("last_clicked_button")

        # Print the last selected bookcover path
        #print(f"Last selected bookcover path: {self.last_clicked_button}")
        

        # Check if the file exists
        if self.last_clicked_button and self.last_clicked_button != "No button clicked":
            if not Path(self.last_clicked_button).exists():
                print(f"[WARNING] Previously saved cover not found: {self.last_clicked_button}")
                # Try to find the saved cover by its basename (handles numeric prefix renaming)
                saved_basename = Path(self.last_clicked_button).name
                found_path = self._find_book_cover_by_basename(saved_basename)
                if found_path:
                    self.book_cover_path_default = found_path
                    self.last_clicked_button = str(found_path)
                    print(f"[INFO] Found renamed cover: {found_path}")
                    # Update saved state with new path
                    self.save_preset_last_state_to_file()
                else:
                    # Fallback to default_bookcover01.docx
                    print(f"[WARNING] Could not find {saved_basename}, falling back to default")
                    found_path = self._find_book_cover_by_basename("default_bookcover01.docx")
                    if found_path:
                        self.book_cover_path_default = found_path
                        self.last_clicked_button = str(found_path)
                        print(f"[INFO] Using default cover: {found_path}")
                        # Update saved state with new path
                        self.save_preset_last_state_to_file()
                    else:
                        # Final fallback to hardcoded path
                        self.book_cover_path_default = self.project_root / "assets" / "bookcovers" / "default_bookcover01.docx"
                        self.last_clicked_button = str(self.book_cover_path_default)
                        print(f"[WARNING] Default cover not found, using hardcoded path: {self.book_cover_path_default}")

        # Check if last_clicked_button_value is the specific string indicating no button was clicked
        if self.last_clicked_button == "No button clicked":
            # If so, use self.last_clicked_button instead
            self.highlight_last_clicked_button(self.last_clicked_button)
        else:
            # Otherwise, use the value from last_state
            self.highlight_last_clicked_button(self.last_clicked_button)
        """

    def _find_book_cover_by_basename(self, target_basename):
        """
        Find a book cover file by its basename, accounting for optional numeric prefixes.
        Only matches exact filename with or without numeric prefix (e.g., ###_filename.docx).

        Args:
            target_basename: The basename to search for (e.g., "default_bookcover01.docx")

        Returns:
            Path object if found, None otherwise
        """
        bookcovers_dir = self.project_root / "assets" / "bookcovers"
        if not bookcovers_dir.exists():
            return None

        for file_path in bookcovers_dir.glob("*.docx"):
            filename = file_path.name

            # Check exact match (e.g., "default_bookcover01.docx")
            if filename == target_basename:
                return file_path

            # Check with numeric prefix (e.g., "001_default_bookcover01.docx")
            if "_" in filename:
                parts = filename.split("_", 1)
                if len(parts) == 2 and parts[0].isdigit() and parts[1] == target_basename:
                    return file_path

        return None

    def load_ui_from_preset(self, event=None):  # _=None to allow binding to events
        print("load_ui_from_preset called")

        selected_preset_name = self.edit_loadpreset_label_var.get()
        if not selected_preset_name:
            print("No preset selected.")
            return

        presets = self.load_presets_from_file()
        selected_preset = presets.get(selected_preset_name)

        if selected_preset is None:
            print(f"Preset {selected_preset_name} not found.")
            return


        # Apply the loaded preferences
        self.font_for_title_var.set(selected_preset.get("selected_font_for_title", "나눔고딕"))
        self.font_for_num_var.set(selected_preset.get("selected_font_for_num", "나눔고딕"))
        self.font_for_text_var.set(selected_preset.get("selected_font_for_text", "나눔고딕"))
        self.font_for_exp_var.set(selected_preset.get("selected_font_for_exp", "나눔고딕"))
        
        # Applying colors
        self.frame1_color_select_button.configure(text_color=selected_preset.get("color_heading", "#335581"))
        self.frame2_color_select_button.configure(text_color=selected_preset.get("color_num", "#4B76B0"))
        self.frame3_color_select_button.configure(text_color=selected_preset.get("color_text", "#000000"))
        self.frame4_color_select_button.configure(text_color=selected_preset.get("color_exp", "#000000"))
        
        # Applying sizes
        self.frame1_entry.delete(0, 'end')
        self.frame1_entry.insert(0, selected_preset.get("size_heading", 12))
        
        self.frame2_entry.delete(0, 'end')
        self.frame2_entry.insert(0, selected_preset.get("size_num", 10))
        
        self.frame3_entry.delete(0, 'end')
        self.frame3_entry.insert(0, selected_preset.get("size_text", 8))
        self.frame3_entry_line.delete(0, 'end')
        self.frame3_entry_line.insert(0, selected_preset.get("line_text", 1.2))
        
        self.frame4_entry.delete(0, 'end')
        self.frame4_entry.insert(0, selected_preset.get("size_exp", 7))
        self.frame4_entry_line.delete(0, 'end')
        self.frame4_entry_line.insert(0, selected_preset.get("line_exp", 1.1))
        
        # Applying the tag removal flag
        if selected_preset.get("tagremoval_flag") == 1:
            self.tagremoval_checkbox.select()
        elif selected_preset.get("tagremoval_flag") == 0:
            self.tagremoval_checkbox.deselect()

        # Restore 다단 나누기 in preset
        self.multicol_var.set(selected_preset.get("two_column", 0))
        print(f"self.multicol_var: {self.multicol_var.get()}\n\n")

        # Restore 지문 테두리 in preset (1=borders enabled/checked, 0=no borders/unchecked)
        self.border_flag_var.set(selected_preset.get("border_flag", 1))

        # Restore 지문번호 숫자로 사용 in preset (0=use original passage_id, 1=use numeric numbering)
        self.passage_number_numeric_var.set(selected_preset.get("passage_number_numeric", 0))

        # Restore 정답지 분리 in preset (0=single file, 1=separate answer file)
        self.separate_answer_var.set(selected_preset.get("separate_answer_key", 0))

        # Restore 유형마다 새 페이지 in preset (1=new page between sections, 0=no page break)
        self.new_page_per_section_var.set(selected_preset.get("new_page_per_section", 1))

        # Restore 페이지당 문제 수 in preset (enabled + count)
        self.page_break_after_passages_enabled_var.set(selected_preset.get("page_break_after_passages_enabled", 0))
        page_break_after_passages_count = selected_preset.get("page_break_after_passages_count", "4")
        self.page_break_after_passages_count_var.set(str(page_break_after_passages_count))

        # **Set the new text entries from preset**
        self.school_grade_var.set(selected_preset.get("school_grade", "2025 한국학교 1학년 1학기"))
        self.exam_type_var.set(selected_preset.get("exam_type", "중간고사 내신대비"))
        self.week_var.set(selected_preset.get("week", "1주차 숙제"))

        # Restore 교재제작 옵션 from preset
        self.jimun_analysis_font_var.set(selected_preset.get("jimun_analysis_font", self.default_font))
        self.jimun_analysis_font_size_var.set(selected_preset.get("jimun_analysis_font_size", "8"))
        self.jimun_analysis_font_color_var.set(selected_preset.get("jimun_analysis_font_color", "#000000"))
        self.jimun_analysis_line_spacing_var.set(selected_preset.get("jimun_analysis_line_spacing", "1.2"))
        self.jimun_analysis_sentence_split_var.set(selected_preset.get("jimun_analysis_sentence_split", 0))
        self.jimun_analysis_interpretation_right_var.set(selected_preset.get("jimun_analysis_interpretation_right", 0))
        self.hanjul_haesuk_font_var.set(selected_preset.get("hanjul_haesuk_font", self.default_font))
        self.hanjul_haesuk_font_size_var.set(selected_preset.get("hanjul_haesuk_font_size", "8"))
        self.hanjul_haesuk_font_color_var.set(selected_preset.get("hanjul_haesuk_font_color", "#000000"))
        self.hanjul_haesuk_line_spacing_var.set(selected_preset.get("hanjul_haesuk_line_spacing", "1.2"))
        self.hanjul_interpretation_include_answer_var.set(selected_preset.get("hanjul_interpretation_include_answer", 1))
        self.hanjul_writing_include_answer_var.set(selected_preset.get("hanjul_writing_include_answer", 1))
        self.hanjul_writing_word_scramble_var.set(selected_preset.get("hanjul_writing_word_scramble", 1))
        self.hanjul_writing_word_form_change_var.set(selected_preset.get("hanjul_writing_word_form_change", 0))
        self.hanjul_writing_korean_only_var.set(selected_preset.get("hanjul_writing_korean_only", 0))

        # Restore 문제 옵션 from preset
        self.naeyongilchi_question_type_var.set(selected_preset.get("naeyongilchi_question_type", 0))

        # Restore 여백설정 from preset (set on MainFrame)
        self.master.margin_top_var.set(selected_preset.get("margin_top", "1.5"))
        self.master.margin_bottom_var.set(selected_preset.get("margin_bottom", "1.6"))
        self.master.margin_left_var.set(selected_preset.get("margin_left", "1.9"))
        self.master.margin_right_var.set(selected_preset.get("margin_right", "1.9"))
        self.master.col_spacing_var.set(selected_preset.get("col_spacing", "1.27"))

        # Restore mirror margins with boolean coercion
        mirror_val = selected_preset.get("mirror_margins", False)
        if isinstance(mirror_val, str):
            mirror_val = mirror_val.lower() in ('true', '1', 'yes')
        self.master.mirror_margins_var.set(mirror_val)

        self.last_clicked_button = selected_preset.get("last_clicked_button")

        # Check if the saved book cover still exists, if not, try to find it with numeric prefix
        if self.last_clicked_button and self.last_clicked_button != "No button clicked":
            if not Path(self.last_clicked_button).exists():
                print(f"[WARNING] Preset cover not found: {self.last_clicked_button}")
                saved_basename = Path(self.last_clicked_button).name
                found_path = self._find_book_cover_by_basename(saved_basename)
                if found_path:
                    self.last_clicked_button = str(found_path)
                    print(f"[INFO] Found renamed preset cover: {found_path}")
                else:
                    print(f"[WARNING] Could not find {saved_basename} in preset")

        if self.last_clicked_button == "No button clicked":
            # If so, use self.last_clicked_button instead
            self.highlight_last_clicked_button(self.last_clicked_button)
        else:
            # Otherwise, use the value from last_state
            self.highlight_last_clicked_button(self.last_clicked_button)

        # Highlight the last clicked button if necessary
        #self.highlight_last_clicked_button(selected_preset.get("last_clicked_button"))
            



    def save_current_ui_as_preset(self):
        print("save_current_ui_as_preset called")
        preset_name = self.edit_savepreset_text.get().strip()
        if not preset_name.strip():
            messagebox.showerror("Error", "Please enter a name for the preset.")
            return  # Exit if no name is provided

        # Load existing presets
        existing_presets = self.load_presets_from_file()

        # Update or add the new preset
        existing_presets[preset_name] = {
            "selected_font_for_title": self.font_for_title_var.get(),
            "selected_font_for_num": self.font_for_num_var.get(),
            "selected_font_for_text": self.font_for_text_var.get(),
            "selected_font_for_exp": self.font_for_exp_var.get(),
            "color_heading": self.frame1_color_select_button.cget("text_color"),
            "size_heading": float(self.frame1_entry.get()),
            "color_num": self.frame2_color_select_button.cget("text_color"),
            "size_num": float(self.frame2_entry.get()),
            "color_text": self.frame3_color_select_button.cget("text_color"),
            "size_text": float(self.frame3_entry.get()),
            "line_text": float(self.frame3_entry_line.get()),
            "color_exp": self.frame4_color_select_button.cget("text_color"),
            "size_exp": float(self.frame4_entry.get()),
            "line_exp": float(self.frame4_entry_line.get()),
            "tagremoval_flag": self.tagremoval_checkbox.get(),
            "last_clicked_button": str(self.last_clicked_button) if self.last_clicked_button is not None else "No button clicked",
            "two_column": self.multicol_var.get(),
            "border_flag": self.border_flag_var.get(),
            "passage_number_numeric": self.passage_number_numeric_var.get(),
            "separate_answer_key": self.separate_answer_var.get(),
            "new_page_per_section": self.new_page_per_section_var.get(),
            "page_break_after_passages_enabled": self.page_break_after_passages_enabled_var.get(),
            "page_break_after_passages_count": self.page_break_after_passages_count_var.get(),
            "school_grade": self.school_grade_var.get(),
            "exam_type": self.exam_type_var.get(),
            "week": self.week_var.get(),
            # 교재제작 옵션
            "jimun_analysis_font": self.jimun_analysis_font_var.get(),
            "jimun_analysis_font_size": self.jimun_analysis_font_size_var.get(),
            "jimun_analysis_font_color": self.jimun_analysis_font_color_var.get(),
            "jimun_analysis_line_spacing": self.jimun_analysis_line_spacing_var.get(),
            "jimun_analysis_sentence_split": self.jimun_analysis_sentence_split_var.get(),
            "jimun_analysis_interpretation_right": self.jimun_analysis_interpretation_right_var.get(),
            "hanjul_haesuk_font": self.hanjul_haesuk_font_var.get(),
            "hanjul_haesuk_font_size": self.hanjul_haesuk_font_size_var.get(),
            "hanjul_haesuk_font_color": self.hanjul_haesuk_font_color_var.get(),
            "hanjul_haesuk_line_spacing": self.hanjul_haesuk_line_spacing_var.get(),
            "hanjul_interpretation_include_answer": self.hanjul_interpretation_include_answer_var.get(),
            "hanjul_writing_include_answer": self.hanjul_writing_include_answer_var.get(),
            "hanjul_writing_word_scramble": self.hanjul_writing_word_scramble_var.get(),
            "hanjul_writing_word_form_change": self.hanjul_writing_word_form_change_var.get(),
            "hanjul_writing_korean_only": self.hanjul_writing_korean_only_var.get(),
            # 문제 옵션
            "naeyongilchi_question_type": self.naeyongilchi_question_type_var.get(),
            # 여백설정 (from MainFrame)
            "margin_top": self.master.margin_top_var.get(),
            "margin_bottom": self.master.margin_bottom_var.get(),
            "margin_left": self.master.margin_left_var.get(),
            "margin_right": self.master.margin_right_var.get(),
            "col_spacing": self.master.col_spacing_var.get(),
            "mirror_margins": self.master.mirror_margins_var.get(),
        }
        # Save the updated presets back to the file
        self.save_presets_to_file(existing_presets)
        messagebox.showinfo("Success", f"Preset '{preset_name}' has been saved.")
        self.edit_savepreset_text.delete(0, 'end')  # Clear the preset name textbox
        self.submit_button.focus()
        self.update_presets_combobox()  # Update the combobox with new presets list
        self.edit_loadpreset_label_var.set(preset_name)  




    def delete_current_edit_preset(self):
        # Step 1: Retrieve the selected preset name
        selected_preset_name = self.edit_loadpreset_label_var.get()
        
        if not selected_preset_name or selected_preset_name == "저장된 서식 없음":
            messagebox.showerror("Error", "No preset selected to delete.")
            return
        
        # Step 2: Load existing presets
        existing_presets = self.load_presets_from_file()
        
        # Step 3: Delete the selected preset
        if selected_preset_name in existing_presets:
            del existing_presets[selected_preset_name]
            # Step 4: Save the updated presets
            self.save_presets_to_file(existing_presets)
            messagebox.showinfo("Success", f"Preset '{selected_preset_name}' has been deleted.")
        else:
            messagebox.showerror("Error", "Preset not found.")
        
        # Step 5: Update the combobox
        self.update_presets_combobox()
        











    def update_presets_combobox(self):
        presets = self.load_presets_from_file().keys()
        self.edit_loadpreset_label_combobox.configure(values=list(presets))
        if presets:
            self.edit_loadpreset_label_var.set("불러올 설정 선택") 
            #self.edit_loadpreset_label_var.set(next(iter(presets)))  # Set the first preset as the default
        else:
            self.edit_loadpreset_label_var.set("저장된 서식 없음")  # Set to empty if no presets are found




    def highlight_last_clicked_button(self, last_clicked_identifier):
        if not last_clicked_identifier or last_clicked_identifier == "No button clicked":
            return
        try:
            target = str(Path(last_clicked_identifier).resolve())
        except Exception:
            return

        for button in self.scrollable_image_frame.button_list:
            if hasattr(button, 'docx_path'):
                btn_path = str(Path(button.docx_path).resolve())
                button.configure(fg_color="#BB6C25" if btn_path == target else "transparent")




    def button_clicked_callback(self, docx_path, clicked_button):
        # Store the identifier for the last clicked button, such as its docx_path
        self.last_clicked_button = docx_path  # or another unique identifier



    def first_button_callbacked(self, docx_path, clicked_button):
        self.first_button = docx_path  # or another unique identifier
        


    def on_window_close(self):
        # Collect current state for persistence
        current_state = {
            "selected_font_for_title": self.font_for_title_var.get(),
            "selected_font_for_num": self.font_for_num_var.get(),
            "selected_font_for_text": self.font_for_text_var.get(),
            "selected_font_for_exp": self.font_for_exp_var.get(),
            "color_heading": self.frame1_color_select_button.cget("text_color"),
            "size_heading": float(self.frame1_entry.get()),
            "color_num": self.frame2_color_select_button.cget("text_color"),
            "size_num": float(self.frame2_entry.get()),
            "color_text": self.frame3_color_select_button.cget("text_color"),
            "size_text": float(self.frame3_entry.get()),
            "line_text": float(self.frame3_entry_line.get()),
            "color_exp": self.frame4_color_select_button.cget("text_color"),
            "size_exp": float(self.frame4_entry.get()),
            "line_exp": float(self.frame4_entry_line.get()),
            "tagremoval_flag": self.tagremoval_checkbox.get(),
            "last_clicked_button": str(self.last_clicked_button) if self.last_clicked_button is not None else "No button clicked",
            "two_column": self.multicol_var.get(),
            "border_flag": self.border_flag_var.get(),
            "passage_number_numeric": self.passage_number_numeric_var.get(),
            "separate_answer_key": self.separate_answer_var.get(),
            "new_page_per_section": self.new_page_per_section_var.get(),
            "page_break_after_passages_enabled": self.page_break_after_passages_enabled_var.get(),
            "page_break_after_passages_count": self.page_break_after_passages_count_var.get(),
            "school_grade": self.school_grade_var.get(),
            "exam_type": self.exam_type_var.get(),
            "week": self.week_var.get(),
            # 교재제작 옵션
            "jimun_analysis_font": self.jimun_analysis_font_var.get(),
            "jimun_analysis_font_size": self.jimun_analysis_font_size_var.get(),
            "jimun_analysis_font_color": self.jimun_analysis_font_color_var.get(),
            "jimun_analysis_line_spacing": self.jimun_analysis_line_spacing_var.get(),
            "jimun_analysis_sentence_split": self.jimun_analysis_sentence_split_var.get(),
            "jimun_analysis_interpretation_right": self.jimun_analysis_interpretation_right_var.get(),
            "hanjul_haesuk_font": self.hanjul_haesuk_font_var.get(),
            "hanjul_haesuk_font_size": self.hanjul_haesuk_font_size_var.get(),
            "hanjul_haesuk_font_color": self.hanjul_haesuk_font_color_var.get(),
            "hanjul_haesuk_line_spacing": self.hanjul_haesuk_line_spacing_var.get(),
            "hanjul_interpretation_include_answer": self.hanjul_interpretation_include_answer_var.get(),
            "hanjul_writing_include_answer": self.hanjul_writing_include_answer_var.get(),
            "hanjul_writing_word_scramble": self.hanjul_writing_word_scramble_var.get(),
            "hanjul_writing_word_form_change": self.hanjul_writing_word_form_change_var.get(),
            "hanjul_writing_korean_only": self.hanjul_writing_korean_only_var.get(),
            # 문제 옵션
            "naeyongilchi_question_type": self.naeyongilchi_question_type_var.get(),
            # 여백설정 (from MainFrame)
            "margin_top": self.master.margin_top_var.get(),
            "margin_bottom": self.master.margin_bottom_var.get(),
            "margin_left": self.master.margin_left_var.get(),
            "margin_right": self.master.margin_right_var.get(),
            "col_spacing": self.master.col_spacing_var.get(),
            "mirror_margins": self.master.mirror_margins_var.get(),

        }

        # Save the current state as the last state
        self.save_preset_last_state_to_file(current_state)

        # Unbind textvariables from entries that use parent_window StringVars
        # This prevents TclError when StringVar callbacks try to access destroyed widgets
        try:
            # jimun_analysis entries
            if hasattr(self, 'jimun_analysis_size_entry'):
                self.jimun_analysis_size_entry.configure(textvariable="")
            if hasattr(self, 'jimun_analysis_line_entry'):
                self.jimun_analysis_line_entry.configure(textvariable="")
            # hanjul_haesuk entries
            if hasattr(self, 'hanjul_haesuk_size_entry'):
                self.hanjul_haesuk_size_entry.configure(textvariable="")
            if hasattr(self, 'hanjul_haesuk_line_entry'):
                self.hanjul_haesuk_line_entry.configure(textvariable="")
        except Exception:
            pass  # Ignore errors during cleanup

        # Destroy the window after saving
        self.destroy()


    def generate_thumbnails_for_docx(self, docx_folder=None):
        folder = Path(docx_folder) if docx_folder else DEFAULT_BOOKCOVERS_DIR
        if not folder.exists():
            raise FileNotFoundError(f"Bookcovers folder not found: {folder}")

        thumbnails = []
        for entry in folder.iterdir():
            if entry.is_file() and entry.suffix.lower() == ".docx" and not entry.name.startswith("~$"):
                docx_path = str(entry)                           # 함수가 str 경로를 기대할 수 있으므로 str로
                thumbnail_path = str(entry.with_suffix(".png"))
                if not os.path.exists(thumbnail_path):
                    self.docx_first_page_to_image(docx_path, thumbnail_path)
                thumbnails.append(thumbnail_path)

        return thumbnails







    def create_placeholder_image(self, docx_path, thumbnail_path, size=(600, 800)):
        img = Image.new('RGB', size, 'white')
        draw = ImageDraw.Draw(img)

        # 1) Normalize the filename to NFC so Hangul renders as one syllable
        raw = Path(docx_path).stem
        filename = unicodedata.normalize('NFC', raw)

        # 2) Load your bundled TTF (PyInstaller-compatible)
        if getattr(sys, 'frozen', False):
            font_file = Path(sys._MEIPASS) / "assets" / "fonts" / "NanumGothic.ttf"
        else:
            font_file = Path(__file__).parent.parent / "assets" / "fonts" / "NanumGothic.ttf"

        try:
            font = ImageFont.truetype(str(font_file), 70)
        except Exception:
            font = ImageFont.load_default()

        # 3) Measure text
        try:
            bbox = draw.textbbox((0, 0), filename, font=font)
            w, h = bbox[2] - bbox[0], bbox[3] - bbox[1]
        except AttributeError:
            w, h = font.getsize(filename)

        # 4) Center & draw
        x = (size[0] - w) // 2
        y = (size[1] - h) // 2
        draw.text((x, y), filename, fill="black", font=font)

        # 5) Save out
        img.save(thumbnail_path)



    #def docx_first_page_to_image(self, docx_path, thumbnail_path, dpi=200):
    #    print("docx_first_page_to_image called")
    #    docx_path = Path(docx_path)
    #    thumbnail_path = Path(thumbnail_path)
    #    pdf_path = docx_path.with_suffix('.pdf')
    #    self.create_placeholder_image(docx_path, thumbnail_path)





    def docx_first_page_to_image(self, docx_path, thumbnail_path, dpi=200):
        docx_path = Path(docx_path)
        thumbnail_path = Path(thumbnail_path)
        pdf_path = docx_path.with_suffix('.pdf')

        # 1. Locate soffice
        soffice_exec = (shutil.which("soffice") or
                        shutil.which("libreoffice") or
                        "/Applications/LibreOffice.app/Contents/MacOS/soffice")
        if not Path(soffice_exec).exists():
            # No LibreOffice → placeholder
            return self.create_placeholder_image(docx_path, thumbnail_path)

        # 2. Convert DOCX → PDF
        original_cwd = Path.cwd()
        try:
            os.chdir(docx_path.parent)
            subprocess.run(
                [soffice_exec, "--convert-to", "pdf", docx_path.name, "--headless"],
                check=True, stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL
            )
        except Exception:
            # Conversion failed → placeholder
            return self.create_placeholder_image(docx_path, thumbnail_path)
        finally:
            os.chdir(original_cwd)

        # 3. If PDF wasn’t produced, fallback
        if not pdf_path.exists():
            return self.create_placeholder_image(docx_path, thumbnail_path)

        # 4. Convert first page of PDF → PNG
        try:
            images = convert_from_path(str(pdf_path), dpi=dpi,
                                    first_page=1, last_page=1)
            if images:
                images[0].save(thumbnail_path)
            else:
                # No pages? fallback
                self.create_placeholder_image(docx_path, thumbnail_path)
        except Exception:
            # Any other failure → placeholder
            self.create_placeholder_image(docx_path, thumbnail_path)




    def setup_title_frame(self):
        self.title_frame = CTkFrame(master=self, width=880, height=50, fg_color="transparent")
        self.title_frame.grid(row=0, column=0, padx=10, pady=10, sticky="nsew")
        self.title_frame.grid_columnconfigure(1, weight=1)




        # 1) Configure five columns: only the middle one (2) gets weight=1
        for col in range(5):
            self.title_frame.grid_columnconfigure(col, weight=1 if col == 2 else 0)

        # 2) “표지 추가” 버튼 in col 0
        self.import_button = CTkButton(
            self.title_frame, text="표지 추가",
            command=self.import_book_cover,
            font=(self.default_font, 12),
            fg_color="#5F6C37",
            hover_color="#2B3D2D",
            width=80
        )
        self.import_button.grid(row=0, column=0, padx=(0,10), sticky="w")

        # 3) “폴더 열기” 버튼 in col 1
        self.open_folder_button = CTkButton(
            self.title_frame, text="폴더 열기",
            command=self.open_covers_folder,
            font=(self.default_font, 12),
            fg_color="#5F6C37",
            hover_color="#2B3D2D",
            width=80
        )
        self.open_folder_button.grid(row=0, column=1, padx=(0,10), sticky="w")

        # 4) Title label in the *center* column 2, stretched horizontally
        self.title_label = CTkLabel(
            self.title_frame,
            font=(self.default_font, 18, "bold"),
            text="표지 선택",
            text_color="black",
            fg_color="transparent"
        )
        # sticky="ew" lets it expand to fill column 2
        self.title_label.grid(row=0, column=2, sticky="ew")

        # 5) “초기값” 버튼 in col 3
        self.reset_button = CTkButton(
            self.title_frame, text="초기값",
            command=self.reset_covers,
            font=(self.default_font, 12),
            fg_color="#5F6C37",
            hover_color="#2B3D2D",
            width=80
        )
        self.reset_button.grid(row=0, column=3, padx=(10,5), sticky="e")


        self.manage_mode_var = BooleanVar(value=False)

        # 6) “관리 모드” 버튼 in col 4
        self.manage_button = CTkButton(
            self.title_frame, text="관리 모드",
            command=self.toggle_manage_mode,
            font=(self.default_font, 12),
            fg_color="#5F6C37",
            hover_color="#2B3D2D",
            width=80
        )
        self.manage_button.grid(row=0, column=4, padx=(5,0), sticky="e")


    def open_covers_folder(self):
        """Cross-platform open of the bookcovers folder."""
        folder = self.project_root / "assets" / "bookcovers"
        folder_path = str(folder)
        try:
            if sys.platform.startswith("darwin"):
                subprocess.run(["open", folder_path])
            elif sys.platform.startswith("win"):
                os.startfile(folder_path)
            else:
                subprocess.run(["xdg-open", folder_path])
        except Exception as e:
            messagebox.showerror("오류", f"폴더를 열 수 없습니다:\n{e}")

    def reset_covers(self):
        """Reset all covers to defaults."""
        ok = messagebox.askyesno(
            "초기화 확인",
            "모든 표지를 초기값으로 되돌립니다 (사용자가 추가하거나 수정한 모든 표지가 삭제됩니다.)\n실행하시겠습니까?"
        )
        if not ok:
            return

        bookcovers = self.project_root / "assets" / "bookcovers"
        defaults   = self.project_root / "assets" / "bookcovers_default"

        # 1) delete everything under bookcovers
        for item in bookcovers.iterdir():
            try:
                if item.is_dir():
                    shutil.rmtree(item)
                else:
                    item.unlink()
            except Exception as e:
                print(f"Failed to delete {item}: {e}")

        # 2) copy defaults → bookcovers
        for item in defaults.iterdir():
            dest = bookcovers / item.name
            try:
                if item.is_dir():
                    shutil.copytree(item, dest)
                else:
                    shutil.copy2(item, dest)
            except Exception as e:
                print(f"Failed to copy {item} to {dest}: {e}")

        # 3) refresh UI
        self.refresh_book_covers()
        messagebox.showinfo("완료", "표지가 초기값으로 복원되었습니다.")


    def toggle_manage_mode(self):
        """Toggle between normal and management mode"""
        self.manage_mode_var.set(not self.manage_mode_var.get())

        if self.manage_mode_var.get():
            self.manage_button.configure(fg_color="#BB6C25", text="관리 완료")
            # Enable drag-and-drop reordering
            self.scrollable_image_frame.enable_drag_mode()
            # Show delete buttons on each cover
            for button in self.scrollable_image_frame.button_list:
                # Add a small delete button overlay
                delete_btn = CTkButton(
                    button.master,
                    text="X",
                    width=25,
                    height=25,
                    command=lambda p=button.docx_path: self.delete_book_cover(p, p.with_suffix('.png'))
                )
                # Position it at the top-right corner of each button
                delete_btn.place(in_=button, relx=0.9, rely=0.1, anchor="ne")
                button.delete_overlay = delete_btn
        else:
            self.manage_button.configure(fg_color="#8D9397", text="관리 모드")
            # Disable drag-and-drop reordering
            self.scrollable_image_frame.disable_drag_mode()
            # Hide delete buttons
            for button in self.scrollable_image_frame.button_list:
                if hasattr(button, 'delete_overlay'):
                    button.delete_overlay.destroy()

    def import_book_cover(self):
        """Allow user to import a new book cover"""
        from tkinter import filedialog
        import shutil
        
        # Create a custom file dialog that only shows .docx files
        file_path = filedialog.askopenfilename(
            title="Select Book Cover Document (.docx only)",
            filetypes=[("Word Documents", "*.docx")],  # Only one option
            defaultextension=".docx"
        )
        
        if file_path:
            # Double-check the extension (in case user typed a filename manually)
            if not file_path.lower().endswith('.docx'):
                messagebox.showerror(
                    "Invalid File Type", 
                    "Only .docx files are allowed. Please select a Word document."
                )
                return
            
            # Validate filename doesn't contain invalid characters
            filename = Path(file_path).name
            invalid_chars = '<>:"|?*'
            if any(char in filename for char in invalid_chars):
                messagebox.showerror(
                    "Invalid Filename", 
                    f"Filename contains invalid characters: {invalid_chars}"
                )
                return
            
            try:
                # Copy file to bookcovers directory
                dest_path = self.project_root / "assets" / "bookcovers" / filename
                
                # Check if file already exists
                if dest_path.exists():
                    response = messagebox.askyesno(
                        "File Exists", 
                        f"'{filename}' already exists. Do you want to replace it?"
                    )
                    if not response:
                        return
                
                # Copy the file
                shutil.copy2(file_path, dest_path)
                
                # Generate thumbnail
                thumbnail_path = dest_path.with_suffix('.png')
                result = self.docx_first_page_to_image(str(dest_path), str(thumbnail_path))
                
                # Check if thumbnail generation was successful
                if not thumbnail_path.exists():
                    # Delete the copied file if thumbnail generation failed
                    if dest_path.exists():
                        dest_path.unlink()
                    messagebox.showerror(
                        "Error", 
                        "Failed to generate thumbnail. The file may be corrupted or incompatible."
                    )
                    return
                
                # Refresh the display
                self.refresh_book_covers()
                
                messagebox.showinfo("Success", f"Book cover '{filename}' has been added.")
                
            except Exception as e:
                messagebox.showerror("Error", f"Failed to import book cover: {str(e)}")


    def delete_book_cover(self, docx_path, image_path):
        """Delete a book cover after confirmation"""
        filename = Path(docx_path).name
        
        # Confirmation dialog
        response = messagebox.askyesno(
            "Confirm Delete", 
            f"Are you sure you want to delete '{filename}'?"
        )
        
        if response:
            try:
                # Delete docx file
                if Path(docx_path).exists():
                    Path(docx_path).unlink()
                
                # Delete thumbnail
                if Path(image_path).exists():
                    Path(image_path).unlink()
                
                # Also try to delete the PDF if it exists
                pdf_path = Path(docx_path).with_suffix('.pdf')
                if pdf_path.exists():
                    pdf_path.unlink()
                
                # Refresh the display
                self.refresh_book_covers()
                
                messagebox.showinfo("Success", f"Book cover '{filename}' has been deleted.")
                
            except Exception as e:
                messagebox.showerror("Error", f"Failed to delete book cover: {e}")
    
    def refresh_book_covers(self):
        """Refresh the book covers display."""
        # Regenerate thumbnails
        docx_folder = self.project_root / "assets" / "bookcovers"
        image_paths = self.generate_thumbnails_for_docx(docx_folder)

        # Destroy old frame
        if hasattr(self, 'scrollable_image_frame'):
            self.scrollable_image_frame.destroy()

        # Create new frame
        self.setup_image_frame(image_paths, docx_folder)

        # If current selection vanished → first cover
        if hasattr(self, 'last_clicked_button') and not Path(str(self.last_clicked_button)).exists():
            self._select_first_cover()

        # **Re‑add delete overlays and drag mode if we're in 관리 모드**
        if self.manage_mode_var.get():
            # Re-enable drag mode
            self.scrollable_image_frame.enable_drag_mode()
            # Re-add delete overlays
            for button in self.scrollable_image_frame.button_list:
                # same overlay logic as in toggle_manage_mode
                delete_btn = CTkButton(
                    button.master,
                    text="X",
                    width=25,
                    height=25,
                    command=lambda p=button.docx_path: self.delete_book_cover(p, p.with_suffix('.png'))
                )
                delete_btn.place(in_=button, relx=0.9, rely=0.1, anchor="ne")
                button.delete_overlay = delete_btn

    def setup_image_frame(self, image_paths, docx_folder):
        self.scrollable_image_frame = ScrollableImageFrame(
            master=self, 
            width=880, 
            height=210,
            image_paths=image_paths, 
            docx_folder=docx_folder,
            command=self.image_frame_event, 
            button_click_callback=self.button_clicked_callback, 
            first_button_callback=self.first_button_callbacked,
            delete_callback=self.delete_book_cover,  # Add delete callback
            corner_radius=10
        )
        self.scrollable_image_frame.grid(row=1, column=0, padx=10, pady=(0, 10), sticky="nsew")



    def setup_cover_text_frame(self):
        # Create the cover text frame
        self.cover_text_frame = CTkFrame(master=self, width=880, height=50, fg_color="transparent")
        self.cover_text_frame.grid(row=2, column=0, padx=10, pady=(5), sticky="nsew")
        self.cover_text_frame.grid_columnconfigure(0, weight=1)


        # Title label
        self.cover_text_label = CTkLabel(self.cover_text_frame, font=(self.default_font, 18, "bold"), text="표지 문구", text_color="black", fg_color="transparent")
        self.cover_text_label.grid(row=0, column=0, pady=(0, 10), sticky="nsew")


        self.cover_text_frame_innerbox = CTkFrame(self.cover_text_frame, width=880, height=50, fg_color="#B6C2B7")
        self.cover_text_frame_innerbox.grid(row=1, column=0, pady=(0, 10), sticky="nsew")


        self.cover_text_frame_innerbox.grid_columnconfigure(1, weight=1)
        self.cover_text_frame_innerbox.grid_columnconfigure(3, weight=1)
        self.cover_text_frame_innerbox.grid_columnconfigure(5, weight=1)



        # Labels and Entries
        # 학교학년
        self.school_grade_label = CTkLabel(self.cover_text_frame_innerbox, text="학교학년:", font=(self.default_font, 12), text_color="black", fg_color="transparent")
        self.school_grade_label.grid(row=1, column=0, padx=(10, 5), pady=5, sticky="e")

        self.school_grade_var = StringVar(value="2025 한국학교 1학년 1학기")
        self.school_grade_entry = CTkEntry(self.cover_text_frame_innerbox, textvariable=self.school_grade_var, width=200, font=(self.default_font, 12))
        self.school_grade_entry.grid(row=1, column=1, padx=(0, 20), pady=5, sticky="w")

        # 시험종류
        self.exam_type_label = CTkLabel(self.cover_text_frame_innerbox, text="시험종류:", font=(self.default_font, 12), text_color="black", fg_color="transparent")
        self.exam_type_label.grid(row=1, column=2, padx=(10, 5), pady=5, sticky="e")

        self.exam_type_var = StringVar(value="중간고사 내신대비")
        self.exam_type_entry = CTkEntry(self.cover_text_frame_innerbox, textvariable=self.exam_type_var, width=200, font=(self.default_font, 12))
        self.exam_type_entry.grid(row=1, column=3, padx=(0, 20), pady=5, sticky="w")

        # 주차
        self.week_label = CTkLabel(self.cover_text_frame_innerbox, text="주차:", font=(self.default_font, 12), text_color="black", fg_color="transparent")
        self.week_label.grid(row=1, column=4, padx=(10, 5), pady=5, sticky="e")

        self.week_var = StringVar(value="1주차 숙제")
        self.week_entry = CTkEntry(self.cover_text_frame_innerbox, textvariable=self.week_var, width=200, font=(self.default_font, 12))
        self.week_entry.grid(row=1, column=5, padx=(0, 10), pady=5, sticky="w")



    def setup_font_selection_frame(self):

        # 교재 내지 선택 레이블
        self.title2_frame = CTkFrame(
            master=self, width=880, height=50, fg_color="transparent")
        self.title2_frame.grid(row=3, column=0, padx=10, pady=(5, 10), sticky="nsew")

        self.title2_frame.grid_columnconfigure(0, weight=1)   # left flexible space
        self.title2_frame.grid_columnconfigure(1, weight=0)   # label column
        self.title2_frame.grid_columnconfigure(2, weight=0)   # checkbox column
        self.title2_frame.grid_columnconfigure(3, weight=0)   # checkbox column
        self.title2_frame.grid_columnconfigure(4, weight=0)   # checkbox column
        self.title2_frame.grid_columnconfigure(5, weight=0)   # checkbox column
        self.title2_frame.grid_columnconfigure(6, weight=0)   # button column (교재)
        self.title2_frame.grid_columnconfigure(7, weight=0)   # button column (문제)
        self.title2_frame.grid_columnconfigure(8, weight=0)   # button column (여백설정)
        self.title2_frame.grid_columnconfigure(9, weight=1)   # right flexible space


        # "교재 내지 서식" 레이블
        self.title2_label = CTkLabel(
            self.title2_frame,
            font=(self.default_font, 18, "bold"),
            text="내지 서식",
            text_color="black",
            fg_color="transparent"
        )
        self.title2_label.grid(row=0, column=1, padx=(0,10))

        # "내지 서식 옵션" 버튼
        self.naeji_seosik_options_button = CTkButton(
            self.title2_frame,
            text="내지 서식 옵션",
            command=self.open_naeji_seosik_options_dialog,
            font=(self.default_font, 14),
            fg_color="#FEF9E0", hover_color="#DDA15C", text_color="black",
            width=110
        )
        self.naeji_seosik_options_button.grid(row=0, column=2, padx=5)

        # "교재제작 옵션" 버튼
        self.gyojae_options_button = CTkButton(
            self.title2_frame,
            text="'교재' 옵션",
            command=self.open_gyojae_options_dialog,
            font=(self.default_font, 14),
            fg_color="#FEF9E0", hover_color="#DDA15C", text_color="black",
            width=90
        )
        self.gyojae_options_button.grid(row=0, column=3, padx=5)

        # "'문제' 옵션" 버튼
        self.munje_options_button = CTkButton(
            self.title2_frame,
            text="'문제' 옵션",
            command=self.open_munje_options_dialog,
            font=(self.default_font, 14),
            fg_color="#FEF9E0", hover_color="#DDA15C", text_color="black",
            width=90
        )
        self.munje_options_button.grid(row=0, column=4, padx=5)

        # "여백설정" 버튼
        self.margin_options_button = CTkButton(
            self.title2_frame,
            text="여백설정",
            command=self.open_margin_options_dialog,
            font=(self.default_font, 14),
            fg_color="#FEF9E0", hover_color="#DDA15C", text_color="black",
            width=90
        )
        self.margin_options_button.grid(row=0, column=5, padx=5)


        ##################################


        # Container frame for all font selection frames
        self.font_selection_container = CTkFrame(master=self, width=880, fg_color="transparent")
        self.font_selection_container.grid(row=4, column=0, padx=10, pady=(0, 0), sticky="nsew")
        # Make sure the container frame expands horizontally with the window
        self.font_selection_container.grid_columnconfigure(0, weight=1)
        self.font_selection_container.grid_columnconfigure(1, weight=1)
        self.font_selection_container.grid_columnconfigure(2, weight=1)
        self.font_selection_container.grid_columnconfigure(3, weight=1) 


        # Subtitle above the first font selection frame
        self.subtitle1_label = CTkLabel(self.font_selection_container, text="유형 제목", font=(self.default_font, 14, "bold"), text_color="black", fg_color="transparent")
        self.subtitle1_label.grid(row=0, column=0, padx=5, pady=(0))  # Adjust pady to control space above the subtitle

        # Subtitle above the second font selection frame
        self.subtitle2_label = CTkLabel(self.font_selection_container, text="지문 번호", font=(self.default_font, 14, "bold"), text_color="black", fg_color="transparent")
        self.subtitle2_label.grid(row=0, column=1, padx=5, pady=(0))  # Adjust pady to control space above the subtitle

        # Subtitle above the third font selection frame
        self.subtitle3_label = CTkLabel(self.font_selection_container, text="지문", font=(self.default_font, 14, "bold"), text_color="black", fg_color="transparent")
        self.subtitle3_label.grid(row=0, column=2, padx=5, pady=(0))  # Adjust pady to control space above the subtitle

        # Subtitle above the third font selection frame
        self.subtitle3_label = CTkLabel(self.font_selection_container, text="해설지", font=(self.default_font, 14, "bold"), text_color="black", fg_color="transparent")
        self.subtitle3_label.grid(row=0, column=3, padx=5, pady=(0))  # Adjust pady to control space above the subtitle



        # First font selection frame
        self.font_selection_frame1 = CTkFrame(master=self.font_selection_container, fg_color="#B6C2B7", width=230, height=150)
        self.font_selection_frame1.grid(row=1, column=0, padx=5, pady=(0, 10), sticky="nsew")
        self.font_selection_frame1.grid_propagate(False)  # Prevents the frame from resizing based on its content
        self.font_selection_frame1.grid_columnconfigure(0, weight=0, minsize=70)  # Fixed width for labels
        self.font_selection_frame1.grid_columnconfigure(1, weight=1)


        # Second font selection frame
        self.font_selection_frame2 = CTkFrame(master=self.font_selection_container, fg_color="#B6C2B7", width=230, height=150)
        self.font_selection_frame2.grid(row=1, column=1, padx=5, pady=(0, 10), sticky="nsew")
        self.font_selection_frame2.grid_propagate(False)
        self.font_selection_frame2.grid_columnconfigure(0, weight=0, minsize=70)  # Fixed width for labels
        self.font_selection_frame2.grid_columnconfigure(1, weight=1)

        # Third font selection frame
        self.font_selection_frame3 = CTkFrame(master=self.font_selection_container, fg_color="#B6C2B7", width=230, height=150)
        self.font_selection_frame3.grid(row=1, column=2, padx=5, pady=(0, 10), sticky="nsew")
        self.font_selection_frame3.grid_propagate(False)
        self.font_selection_frame3.grid_columnconfigure(0, weight=0, minsize=70)  # Fixed width for labels
        self.font_selection_frame3.grid_columnconfigure(1, weight=1)

        # Define and configure the fourth font selection frame
        self.font_selection_frame4 = CTkFrame(master=self.font_selection_container, fg_color="#B6C2B7", width=230, height=150)
        self.font_selection_frame4.grid(row=1, column=3, padx=5, pady=(0, 10), sticky="nsew")
        self.font_selection_frame4.grid_propagate(False)  # Prevents the frame from resizing based on its content
        self.font_selection_frame4.grid_columnconfigure(0, weight=0, minsize=70)  # Fixed width for labels
        self.font_selection_frame4.grid_columnconfigure(1, weight=1)


        # Now, add your controls (labels, comboboxes, etc.) within each frame as needed, just like you did for the first frame.
        # For example, for the first frame:


        self.frame1_label1 = CTkLabel(self.font_selection_frame1, text="폰트 선택:", text_color="black", font=(self.default_font, 12))
        self.frame1_label1.grid(row=0, column=0, padx=(10, 5), pady=(10, 3))
        
        self.font_for_title_var = StringVar(value=self.default_font)  # Set default value
        self.fonts = self.list_system_fonts()  # Fetch the system fonts
        
                
        self.font_for_title_combobox = SearchableFontComboBox(
            self.font_selection_frame1,
            variable=self.font_for_title_var,
            values=self.fonts,
            font_tuple=(self.default_font, 12),
            fg_color='white',
            text_color='black',
            border_color='#FEF9E0',
            button_color='#FEF9E0',
            button_hover_color='#CA7900',
            dropdown_hover_color='#CA7900',
            preview_text="AaBb가나",
            width=180
        )
        self.font_for_title_combobox.grid(row=0, column=1, padx=(0, 20), pady=(10, 3), sticky="ew")
        


        self.frame1_label2 = CTkLabel(self.font_selection_frame1, text="폰트 크기:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame1_label2.grid(row=1, column=0, padx=(10, 5), pady=3, sticky="ew")

        self.frame1_entry = CTkEntry(self.font_selection_frame1, justify=RIGHT, width=40)
        self.frame1_entry.grid(row=1, column=1, padx=(0, 20), pady=3, sticky="w")
        self.frame1_entry.insert(0, "12")
        


        
        self.frame1_label3 = CTkLabel(self.font_selection_frame1, text="폰트 색상:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame1_label3.grid(row=2, column=0, padx=(10, 5), pady=3)
        # Add a button to open the color dialog
        self.frame1_color_select_button = CTkButton(self.font_selection_frame1, text="Select Color",
                                            command=self.open_color_dialog1, text_color="#335581", fg_color="white", border_width=1, hover=False, width=130)
        self.frame1_color_select_button.grid(row=2, column=1, padx=(0, 20), pady=3, sticky="w")


        self.frame_in_frame1 = CTkFrame(master=self.font_selection_frame1, fg_color="transparent")
        self.frame_in_frame1.grid(row=3, column=0, columnspan=2, padx=0, pady=0, sticky="nsew")
        self.frame_in_frame1.grid_propagate(False)  # Prevents the frame from resizing based on its contentㄹ
        self.frame_in_frame1.grid_columnconfigure(0, weight=2)
        self.frame_in_frame1.grid_columnconfigure(1, weight=1)

        
        self.frame1_label4 = CTkLabel(self.frame_in_frame1, text="난이도 태그 제거:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame1_label4.grid(row=0, column=0, padx=(10, 0), pady=3)
        self.tagremoval_checkbox = CTkCheckBox(self.frame_in_frame1, text="", border_color="#1D1E1E", hover_color="#DDA15C", fg_color="#BB6C25") 
        self.tagremoval_checkbox.grid(row=0, column=1, padx=(15, 5), pady=3)
        
        
        ########


        self.frame2_label1 = CTkLabel(self.font_selection_frame2, text="폰트 선택:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame2_label1.grid(row=0, column=0, padx=(10, 5), pady=(10, 3))

        self.font_for_num_var = StringVar(value=self.default_font)  # Set default value
        self.font_for_num_combobox = SearchableFontComboBox(
            self.font_selection_frame2,
            variable=self.font_for_num_var,
            values=self.fonts,
            font_tuple=(self.default_font, 12),
            fg_color='white',
            text_color='black',
            border_color='#FEF9E0',
            button_color='#FEF9E0',
            button_hover_color='#CA7900',
            dropdown_hover_color='#CA7900',
            preview_text="AaBb가나",
            width=180
        )
        self.font_for_num_combobox.grid(row=0, column=1, padx=(0, 20), pady=(10, 3), sticky="ew")


        self.frame2_label2 = CTkLabel(self.font_selection_frame2, text="폰트 크기:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame2_label2.grid(row=1, column=0, padx=(10, 5), pady=3)

        self.frame2_entry = CTkEntry(self.font_selection_frame2, width=50, justify=RIGHT)
        self.frame2_entry.grid(row=1, column=1, padx=(0, 20), pady=3, sticky="w")
        self.frame2_entry.insert(0, "10")

        self.frame2_label3 = CTkLabel(self.font_selection_frame2, text="폰트 색상:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame2_label3.grid(row=2, column=0, padx=(10, 5), pady=3)
        # Add a button to open the color dialog
        self.frame2_color_select_button = CTkButton(self.font_selection_frame2, text="Select Color",
                                            command=self.open_color_dialog2, text_color="#4B76B0", fg_color="white", border_width=1, hover=False)
        self.frame2_color_select_button.grid(row=2, column=1, padx=(0, 20), pady=3, sticky="w")


        ########



        self.frame3_label1 = CTkLabel(self.font_selection_frame3, text="폰트 선택:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame3_label1.grid(row=0, column=0, padx=(10, 5), pady=(10, 3))

        self.font_for_text_var = StringVar(value=self.default_font)  # Set default value
        self.font_for_text_combobox = SearchableFontComboBox(
            self.font_selection_frame3,
            variable=self.font_for_text_var,
            values=self.fonts,
            font_tuple=(self.default_font, 12),
            fg_color='white',
            text_color='black',
            border_color='#FEF9E0',
            button_color='#FEF9E0',
            button_hover_color='#CA7900',
            dropdown_hover_color='#CA7900',
            preview_text="AaBb가나",
            width=180
        )
        self.font_for_text_combobox.grid(row=0, column=1, padx=(0, 20), pady=(10, 3), sticky="ew")


        self.frame3_label2 = CTkLabel(self.font_selection_frame3, text="폰트 크기:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame3_label2.grid(row=1, column=0, padx=(10, 5), pady=3)

        self.frame3_entry = CTkEntry(self.font_selection_frame3, width=50, justify=RIGHT)
        self.frame3_entry.grid(row=1, column=1, padx=(0, 20), pady=3, sticky="w")
        self.frame3_entry.insert(0, "8")

        self.frame3_label3 = CTkLabel(self.font_selection_frame3, text="폰트 색상:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame3_label3.grid(row=2, column=0, padx=(10, 5), pady=3)
        # Add a button to open the color dialog
        self.frame3_color_select_button = CTkButton(self.font_selection_frame3, text="Select Color",
                                            command=self.open_color_dialog3, text_color="#000000", fg_color="white", border_width=1, hover=False)
        self.frame3_color_select_button.grid(row=2, column=1, padx=(0, 20), pady=3, sticky="w")

        self.frame3_label4 = CTkLabel(self.font_selection_frame3, text="줄간격:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame3_label4.grid(row=4, column=0, padx=(10, 5), pady=3, sticky="w")

        self.frame3_entry_line = CTkEntry(self.font_selection_frame3, width=50, justify=RIGHT)
        self.frame3_entry_line.grid(row=4, column=1, padx=(0, 20), pady=3, sticky="w")
        self.frame3_entry_line.insert(0, "1.2")


        ########


        self.frame4_label1 = CTkLabel(self.font_selection_frame4, text="폰트 선택:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame4_label1.grid(row=0, column=0, padx=(10, 5), pady=(10, 3))

        self.font_for_exp_var = StringVar(value=self.default_font)  # Set default value
        self.font_for_exp_combobox = SearchableFontComboBox(
            self.font_selection_frame4,
            variable=self.font_for_exp_var,
            values=self.fonts,
            font_tuple=(self.default_font, 12),
            fg_color='white',
            text_color='black',
            border_color='#FEF9E0',
            button_color='#FEF9E0',
            button_hover_color='#CA7900',
            dropdown_hover_color='#CA7900',
            preview_text="AaBb가나",
            width=180
        )
        self.font_for_exp_combobox.grid(row=0, column=1, padx=(0, 20), pady=(10, 3), sticky="ew")


        self.frame4_label2 = CTkLabel(self.font_selection_frame4, text="폰트 크기:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame4_label2.grid(row=1, column=0, padx=(10, 5), pady=3)

        self.frame4_entry = CTkEntry(self.font_selection_frame4, width=50, justify=RIGHT)
        self.frame4_entry.grid(row=1, column=1, padx=(0, 20), pady=3, sticky="w")
        self.frame4_entry.insert(0, "7")

        self.frame4_label3 = CTkLabel(self.font_selection_frame4, text="폰트 색상:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame4_label3.grid(row=2, column=0, padx=(10, 5), pady=3)
        # Add a button to open the color dialog
        self.frame4_color_select_button = CTkButton(self.font_selection_frame4, text="Select Color",
                                            command=self.open_color_dialog4, text_color="#000000", fg_color="white", border_width=1, hover=False)
        self.frame4_color_select_button.grid(row=2, column=1, padx=(0, 20), pady=3, sticky="w")

        self.frame4_label4 = CTkLabel(self.font_selection_frame4, text="줄간격:", text_color="black", fg_color="transparent", font=(self.default_font, 12))
        self.frame4_label4.grid(row=4, column=0, padx=(10, 5), pady=3, sticky="w")

        self.frame4_entry_line = CTkEntry(self.font_selection_frame4, width=50, justify=RIGHT)
        self.frame4_entry_line.grid(row=4, column=1, padx=(0, 20), pady=3, sticky="w")
        self.frame4_entry_line.insert(0, "1.1")


        





    def open_color_dialog1(self):
        self.master.focus_set()  # Try to set focus to the main window first

        # Open the color chooser dialog and get the selected color
        color_code, color_name = colorchooser.askcolor(title="Choose a color")
        if color_code:
            self.frame1_color_select_button.configure(text_color=color_name)
            self.focus_set()

            # Here, you can update your application with the selected color.
            # For example, update the font color or a label's background.
            # self.frame1_label3.config(fg=color_name)  # Example usage



    def open_color_dialog2(self):
        self.master.focus_set()  # Try to set focus to the main window first

        # Open the color chooser dialog and get the selected color
        color_code, color_name = colorchooser.askcolor(title="Choose a color")
        if color_code:
            self.frame2_color_select_button.configure(text_color=color_name)
            self.focus_set()
            # Here, you can update your application with the selected color.
            # For example, update the font color or a label's background.
            # self.frame1_label3.config(fg=color_name)  # Example usage



    def open_color_dialog3(self):
        self.master.focus_set()  # Try to set focus to the main window first

        # Open the color chooser dialog and get the selected color
        color_code, color_name = colorchooser.askcolor(title="Choose a color")
        if color_code:
            self.frame3_color_select_button.configure(text_color=color_name)
            self.focus_set()
            # Here, you can update your application with the selected color.
            # For example, update the font color or a label's background.
            # self.frame1_label3.config(fg=color_name)  # Example usage


    def open_color_dialog4(self):
        self.master.focus_set()  # Try to set focus to the main window first

        # Open the color chooser dialog and get the selected color
        color_code, color_name = colorchooser.askcolor(title="Choose a color")
        if color_code:
            self.frame4_color_select_button.configure(text_color=color_name)
            self.focus_set()
            # Here, you can update your application with the selected color.
            # For example, update the font color or a label's background.
            # self.frame1_label3.config(fg=color_name)  # Example usage

    def open_gyojae_options_dialog(self):
        """Open the 교재제작 옵션 dialog, preventing multiple instances"""
        # Check if dialog already exists
        if hasattr(self, 'gyojae_dialog') and self.gyojae_dialog and self.gyojae_dialog.winfo_exists():
            self.gyojae_dialog.focus_set()
            return

        # Create and show the dialog
        self.gyojae_dialog = GyojaeOptionsDialog(self, parent_window=self)

    def open_munje_options_dialog(self):
        """Open the 문제 옵션 dialog, preventing multiple instances"""
        # Check if dialog already exists
        if hasattr(self, 'munje_dialog') and self.munje_dialog and self.munje_dialog.winfo_exists():
            self.munje_dialog.focus_set()
            return

        # Create and show the dialog
        self.munje_dialog = MunjeOptionsDialog(self, parent_window=self)

    def open_naeji_seosik_options_dialog(self):
        """Open the 내지 서식 옵션 dialog, preventing multiple instances"""
        # Check if dialog already exists
        if hasattr(self, 'naeji_seosik_dialog') and self.naeji_seosik_dialog and self.naeji_seosik_dialog.winfo_exists():
            self.naeji_seosik_dialog.focus_set()
            return

        # Create and show the dialog
        self.naeji_seosik_dialog = NaejiSeosikOptionsDialog(self, parent_window=self)

    def open_margin_options_dialog(self):
        """Open the margin options dialog, preventing multiple instances"""
        if hasattr(self, 'margin_dialog') and self.margin_dialog and self.margin_dialog.winfo_exists():
            self.margin_dialog.focus_set()
            return
        # Pass parent_window=self.master (MainFrame) so dialog accesses MainFrame's StringVars
        self.margin_dialog = MarginOptionsDialog(self, parent_window=self.master)

    # $액션
    def setup_action_buttons(self):
        # Container frame for all font selection frames
        self.action_button_frame = CTkFrame(master=self, width=880, height=50, fg_color="transparent")
        self.action_button_frame.grid(row=5, column=0, padx=10, pady=(20, 10), sticky="nsew")
        # Make sure the container frame expands horizontally with the window
        self.action_button_frame.grid_columnconfigure(0, weight=1)
        self.action_button_frame.grid_columnconfigure(9, weight=1)


        
        CTkButton(self.action_button_frame, text="취소", command=self.on_popup_close_edit, font=(self.default_font, 12), fg_color="#2B3D2D", hover_color="#5F6C37", width=100).grid(row=0, column=0, sticky="w")

        self.edit_savepreset_label = CTkLabel(self.action_button_frame, text="저장할 서식:", font=(self.default_font, 12), fg_color="transparent", text_color="black")
        self.edit_savepreset_label.grid(row=0, column=1, padx=(5), pady=0)

        
        self.edit_savepreset_text = CTkEntry(self.action_button_frame, placeholder_text="저장할 서식 이름을 입력", font=(self.default_font, 12), fg_color="white")
        self.edit_savepreset_text.grid(row=0, column=2, padx=(0, 5), pady=0)
        
        CTkButton(self.action_button_frame, text="저장", command=self.save_current_ui_as_preset, font=(self.default_font, 12), fg_color="#8D9397", hover_color="#5F6C37", width=35).grid(row=0, column=3, padx=10)


        self.edit_loadpreset_label = CTkLabel(self.action_button_frame, text="불러올 서식:", font=(self.default_font, 12), fg_color="transparent", text_color="black")
        self.edit_loadpreset_label.grid(row=0, column=4, padx=(20, 5), pady=0)

        
        self.edit_loadpreset_label_var = StringVar(value="저장된 서식 없음")  # Set default value
        self.edit_loadpreset_label_combobox = CTkComboBox(self.action_button_frame, variable=self.edit_loadpreset_label_var, values="", font=(self.default_font, 12), state="readonly", command=self.load_ui_from_preset, border_color='#FEF9E0', button_color='#FEF9E0', fg_color='white', text_color='black', hover=True, button_hover_color='#CA7900', dropdown_hover_color='#CA7900')
        self.edit_loadpreset_label_combobox.grid(row=0, column=5, padx=(0, 5), pady=0)
        
        CTkButton(self.action_button_frame, text="삭제", command=self.delete_current_edit_preset, font=(self.default_font, 12), fg_color="#8D9397", hover_color="#5F6C37", width=35).grid(row=0, column=6, padx=(10, 10))
        CTkButton(self.action_button_frame, text="서식 초기화", command=self.default_edit, font=(self.default_font, 12), fg_color="#8D9397", hover_color="#5F6C37", width=70).grid(row=0, column=7, padx=(10, 10))


        self.submit_button = CTkButton(self.action_button_frame, text="선택", font=(self.default_font, 12), command=self.submit_edit, fg_color="#2B3D2D", hover_color="#5F6C37", width=100)
        self.submit_button.grid(row=0, column=9, sticky="e")



    def on_popup_close_edit(self, event=None):
        # Close popup event handling code
        self.destroy()  # Close the popup window after selection


    def default_edit(self, event=None):

        # Apply the loaded preferences
        self.font_for_title_var.set(self.default_font)
        self.font_for_num_var.set(self.default_font)
        self.font_for_text_var.set(self.default_font)
        self.font_for_exp_var.set(self.default_font)
        
        # Applying colors
        self.frame1_color_select_button.configure(text_color="#335581")
        self.frame2_color_select_button.configure(text_color="#4B76B0")
        self.frame3_color_select_button.configure(text_color="#000000")
        self.frame4_color_select_button.configure(text_color="#000000")
        
        # Applying sizes
        self.frame1_entry.delete(0, 'end')
        self.frame1_entry.insert(0, 12)
        
        self.frame2_entry.delete(0, 'end')
        self.frame2_entry.insert(0, 10)
        
        self.frame3_entry.delete(0, 'end')
        self.frame3_entry.insert(0, 8)
        self.frame3_entry_line.delete(0, 'end')
        self.frame3_entry_line.insert(0, 1.2)
        
        self.frame4_entry.delete(0, 'end')
        self.frame4_entry.insert(0, 7)
        self.frame4_entry_line.delete(0, 'end')
        self.frame4_entry_line.insert(0, 1.1)
        
        # Applying the tag removal flag
        self.tagremoval_checkbox.deselect()

        # **Reset the new text entries to default values**
        self.school_grade_var.set("2025 한국학교 1학년 1학기")
        self.exam_type_var.set("중간고사 내신대비")
        self.week_var.set("1주차 숙제")

        self.multicol_var.set(0)
        self.border_flag_var.set(1)  # Reset to default (borders enabled = checkbox checked)
        self.passage_number_numeric_var.set(0)  # Reset to default (use original passage_id = unchecked)
        self.separate_answer_var.set(0)  # Reset to default (single file = unchecked)
        self.new_page_per_section_var.set(1)  # Reset to default (new page between sections = checked)
        self.page_break_after_passages_enabled_var.set(0)  # Reset to default (disabled)
        self.page_break_after_passages_count_var.set("4")

        # Reset 교재제작 옵션 to defaults
        self.jimun_analysis_font_var.set(self.default_font)
        self.jimun_analysis_font_size_var.set("8")
        self.jimun_analysis_font_color_var.set("#000000")
        self.jimun_analysis_line_spacing_var.set("1.2")
        self.jimun_analysis_sentence_split_var.set(0)  # Default: unchecked
        self.jimun_analysis_interpretation_right_var.set(0)  # Default: unchecked
        self.hanjul_haesuk_font_var.set(self.default_font)
        self.hanjul_haesuk_font_size_var.set("8")
        self.hanjul_haesuk_font_color_var.set("#000000")
        self.hanjul_haesuk_line_spacing_var.set("1.2")
        self.hanjul_interpretation_include_answer_var.set(1)  # Default: checked
        self.hanjul_writing_include_answer_var.set(1)  # Default: checked
        self.hanjul_writing_word_scramble_var.set(1)  # Default: checked
        self.hanjul_writing_word_form_change_var.set(0)  # Default: unchecked
        self.hanjul_writing_korean_only_var.set(0)  # Default: unchecked

        # Reset 문제 옵션 to defaults
        self.naeyongilchi_question_type_var.set(0)  # Reset to default (T/F 형식)

        # Reset 여백설정 to defaults (set on MainFrame)
        self.master.margin_top_var.set("1.5")
        self.master.margin_bottom_var.set("1.6")
        self.master.margin_left_var.set("1.9")
        self.master.margin_right_var.set("1.9")
        self.master.col_spacing_var.set("1.27")
        self.master.mirror_margins_var.set(False)

        self.last_clicked_button = self.first_button
        self.highlight_last_clicked_button(self.last_clicked_button)






    def submit_edit(self, event=None):
        # Gather all necessary selections
        selections = {
            "selected_font_for_title": self.font_for_title_var.get(),
            "selected_font_for_num": self.font_for_num_var.get(),
            "selected_font_for_text": self.font_for_text_var.get(),
            "selected_font_for_exp": self.font_for_exp_var.get(),
            "color_heading": self.frame1_color_select_button.cget("text_color"),
            "size_heading": float(self.frame1_entry.get()),
            "color_num": self.frame2_color_select_button.cget("text_color"),
            "size_num": float(self.frame2_entry.get()),
            "color_text": self.frame3_color_select_button.cget("text_color"),
            "size_text": float(self.frame3_entry.get()),
            "line_text": float(self.frame3_entry_line.get()),
            "color_exp": self.frame4_color_select_button.cget("text_color"),
            "size_exp": float(self.frame4_entry.get()),
            "line_exp": float(self.frame4_entry_line.get()),
            "tagremoval_flag": self.tagremoval_checkbox.get(),
            "last_clicked_button": str(self.last_clicked_button) if self.last_clicked_button is not None else "No button clicked",
            "book_cover_path": str(self.last_clicked_button) if self.last_clicked_button is not None and self.last_clicked_button != "No button clicked" else None,
            "two_column": self.multicol_var.get(),
            "border_flag": self.border_flag_var.get(),  # Direct: checkbox checked (1) = borders enabled, unchecked (0) = no borders
            "passage_number_numeric": self.passage_number_numeric_var.get(),  # 0=use original passage_id, 1=use numeric numbering
            "separate_answer_key": self.separate_answer_var.get(),  # 0=single file, 1=separate answer file
            "new_page_per_section": self.new_page_per_section_var.get(),  # 1=new page between sections, 0=no page break
            "page_break_after_passages_enabled": self.page_break_after_passages_enabled_var.get(),
            "page_break_after_passages_count": self.page_break_after_passages_count_var.get(),
            "school_grade": self.school_grade_var.get(),
            "exam_type": self.exam_type_var.get(),
            "week": self.week_var.get(),
            # 교재제작 옵션
            "jimun_analysis_font": self.jimun_analysis_font_var.get(),
            "jimun_analysis_font_size": self.jimun_analysis_font_size_var.get(),
            "jimun_analysis_font_color": self.jimun_analysis_font_color_var.get(),
            "jimun_analysis_line_spacing": self.jimun_analysis_line_spacing_var.get(),
            "jimun_analysis_sentence_split": self.jimun_analysis_sentence_split_var.get(),
            "jimun_analysis_interpretation_right": self.jimun_analysis_interpretation_right_var.get(),
            "hanjul_haesuk_font": self.hanjul_haesuk_font_var.get(),
            "hanjul_haesuk_font_size": self.hanjul_haesuk_font_size_var.get(),
            "hanjul_haesuk_font_color": self.hanjul_haesuk_font_color_var.get(),
            "hanjul_haesuk_line_spacing": self.hanjul_haesuk_line_spacing_var.get(),
            "hanjul_interpretation_include_answer": self.hanjul_interpretation_include_answer_var.get(),
            "hanjul_writing_include_answer": self.hanjul_writing_include_answer_var.get(),
            "hanjul_writing_word_scramble": self.hanjul_writing_word_scramble_var.get(),
            "hanjul_writing_word_form_change": self.hanjul_writing_word_form_change_var.get(),
            "hanjul_writing_korean_only": self.hanjul_writing_korean_only_var.get(),
            # 문제 옵션
            "naeyongilchi_question_type": self.naeyongilchi_question_type_var.get(),
            # 여백설정 (from MainFrame)
            "margin_top": self.master.margin_top_var.get(),
            "margin_bottom": self.master.margin_bottom_var.get(),
            "margin_left": self.master.margin_left_var.get(),
            "margin_right": self.master.margin_right_var.get(),
            "col_spacing": self.master.col_spacing_var.get(),
            "mirror_margins": self.master.mirror_margins_var.get(),

        }

        #print("selections:")
        #print(selections)

        # Call the callback with the selections
        if self.on_submit_callback:
            self.on_submit_callback(selections)

        self.applyoption_var.set("1")  # applyoption_checkbox (옵선 적용) 체크


        # Save the current state and close the window
        self.on_window_close()
    
    
    
    def list_system_fonts(self):
        font_names = [font for font in sorted(list(font.families())) if not font.startswith('@')]

        # Combine English and Korean regex patterns
        english_pattern = re.compile(r'^[a-zA-Z\s]+')
        korean_pattern = re.compile(r'[\uAC00-\uD7AF]')

        filtered_font_names = {font_name for font_name in font_names if english_pattern.match(font_name) or korean_pattern.search(font_name)}

        # Custom sort function to prioritize Korean names
        def sort_fonts(font):
            if re.search("[가-힣]+", font):
                return (0, font)  # Prioritize Korean names
            else:
                return (1, font)  # English names come later
        
        # Apply custom sorting only once
        return sorted(filtered_font_names, key=sort_fonts)



    def image_frame_event(self, docx_path):
        pass
