"""
Excel → HWPX exporter aligned with the desktop DOCX workflow.

The desktop Word exporter groups questions by sheet type (Normal/Hard variants,
동형문제, 실전모의고사, 기출문제세트), then appends “정답 및 해설” and
“빠른 정답 찾기”.  This module mirrors that ordering while emitting a lightweight
HWPX document based on the bundled template at assets/hwp/questions.hwpx.

Key behaviors retained from the original exporter:
  * question heading paragraph
  * passage rendered inside a 1×1 table with borders
  * question text as plain paragraphs
  * summary sections for answers and quick lookup
"""

from __future__ import annotations

from collections import OrderedDict
from dataclasses import dataclass
from pathlib import Path
from typing import Dict, Iterable, List, Optional, Sequence, Tuple
import copy
import inspect
import logging
import zipfile
import xml.etree.ElementTree as ET

from openpyxl import load_workbook

from .editing_functions import FUNCTION_MAP


SECTION_NAMESPACES = {
    "ha": "http://www.hancom.co.kr/hwpml/2011/app",
    "hp": "http://www.hancom.co.kr/hwpml/2011/paragraph",
    "hp10": "http://www.hancom.co.kr/hwpml/2016/paragraph",
    "hs": "http://www.hancom.co.kr/hwpml/2011/section",
    "hc": "http://www.hancom.co.kr/hwpml/2011/core",
    "hh": "http://www.hancom.co.kr/hwpml/2011/head",
    "hhs": "http://www.hancom.co.kr/hwpml/2011/history",
    "hm": "http://www.hancom.co.kr/hwpml/2011/master-page",
    "hpf": "http://www.hancom.co.kr/schema/2011/hpf",
    "dc": "http://purl.org/dc/elements/1.1/",
    "opf": "http://www.idpf.org/2007/opf/",
    "ooxmlchart": "http://www.hancom.co.kr/hwpml/2016/ooxmlchart",
    "hwpunitchar": "http://www.hancom.co.kr/hwpml/2016/HwpUnitChar",
    "epub": "http://www.idpf.org/2007/ops",
    "config": "urn:oasis:names:tc:opendocument:xmlns:config:1.0",
}

for prefix, uri in SECTION_NAMESPACES.items():
    ET.register_namespace(prefix, uri)

HP_NS = SECTION_NAMESPACES["hp"]
XMLNS = "http://www.w3.org/2000/xmlns/"
XMLNS = "http://www.w3.org/2000/xmlns/"


@dataclass
class HwpxQuestion:
    section_title: str
    question_title: str
    passage_lines: List[str]
    question_lines: List[str]
    answer: str
    explanation_lines: List[str]


@dataclass
class AnswerEntry:
    question_title: str
    answer: str
    explanation_lines: List[str]


class ExcelToHwpxExporter:
    """
    Convert selected Excel questions into an HWPX file following the same
    ordering rules as ExcelToWordConverter.
    """

    def __init__(self, template_path: Optional[Path] = None, logger: Optional[logging.Logger] = None) -> None:
        # Determine base path (PyInstaller-compatible)
        if getattr(sys, 'frozen', False):
            base_path = Path(sys._MEIPASS)
        else:
            base_path = Path(__file__).resolve().parent.parent

        default_template = base_path / "assets" / "hwp" / "questions.hwpx"
        self.template_path = Path(template_path or default_template)
        if not self.template_path.exists():
            raise FileNotFoundError(
                f"HWPX template not found at {self.template_path}. "
                "Place questions.hwpx at assets/hwp/ or pass template_path."
            )

        # Initialize logger
        self.logger = logger or logging.getLogger(__name__)

    def export(
        self,
        excel_path: Path | str,
        output_path: Path | str,
        passage_ids: Iterable[str],
        rows_order: Sequence[str],
        selection_state: Dict[str, bool],
        *,
        copyquestion_rows: Sequence[str] = (),
        mock_test_rows: Sequence[str] = (),
        past_test_rows: Sequence[str] = (),
    ) -> Path:
        excel_path = Path(excel_path)
        if not excel_path.exists():
            raise FileNotFoundError(f"Excel file not found: {excel_path}")

        output_path = Path(output_path)
        output_path.parent.mkdir(parents=True, exist_ok=True)

        passage_order = [pid.strip() for pid in passage_ids if pid.strip()]
        passage_filter = set(passage_order)

        question_blocks: List[HwpxQuestion] = []
        answers_map: "OrderedDict[str, List[AnswerEntry]]" = OrderedDict()

        copy_set = set(copyquestion_rows)
        mock_set = set(mock_test_rows)
        past_set = set(past_test_rows)

        wb = load_workbook(filename=str(excel_path), read_only=True, data_only=True)
        try:
            section_counter = 1
            for row_key in rows_order:
                if row_key in copy_set:
                    if selection_state.get(row_key):
                        title = f"{section_counter}. {row_key}"
                        blocks, entries = self._collect_full_sheet(wb, row_key, title)
                        if blocks:
                            question_blocks.extend(blocks)
                            answers_map[title] = entries
                            section_counter += 1
                    continue

                if row_key in mock_set:
                    if selection_state.get(row_key):
                        title = f"{section_counter}. {row_key}"
                        blocks, entries = self._collect_full_sheet(
                            wb, row_key, title, number_within_section=True
                        )
                        if blocks:
                            question_blocks.extend(blocks)
                            answers_map[title] = entries
                            section_counter += 1
                    continue

                if row_key in past_set:
                    if selection_state.get(row_key):
                        title = f"{section_counter}. {row_key}"
                        blocks, entries = self._collect_full_sheet(
                            wb, row_key, title, number_within_section=True
                        )
                        if blocks:
                            question_blocks.extend(blocks)
                            answers_map[title] = entries
                            section_counter += 1
                    continue

                for suffix in ("Normal", "Hard"):
                    key = f"{row_key}_{suffix}"
                    if not selection_state.get(key):
                        continue
                    sheet_name = f"{row_key}_{suffix}"
                    if sheet_name not in wb.sheetnames:
                        continue
                    title = f"{section_counter}. {sheet_name}"
                    blocks, entries = self._collect_filtered_sheet(
                        wb, sheet_name, title, passage_order, passage_filter
                    )
                    if blocks:
                        question_blocks.extend(blocks)
                        answers_map[title] = entries
                        section_counter += 1
        finally:
            wb.close()

        if not question_blocks:
            raise ValueError("No questions matched the current selection for HWPX export.")

        names, contents = self._read_template()
        section_xml = self._build_section_xml(question_blocks, answers_map)
        contents["Contents/section0.xml"] = section_xml
        self._write_hwpx(output_path, names, contents)
        return output_path

    # ------------------------------------------------------------------
    # Excel ingestion helpers
    # ------------------------------------------------------------------

    def _process_question_editing(
        self,
        function_name: str,
        title: str,
        passage: str,
        result: str,
        answer_part: str,
        explanation_part: str,
    ) -> Tuple[str, str, str, str, str, str]:
        """
        Process question data using the appropriate editing function from FUNCTION_MAP.

        This mirrors the process_question_part() method from converter.py, handling:
        - Dynamic function lookup
        - Parameter assembly based on function signature
        - Return value parsing (5 or 6 element tuples)
        - Special handling for document-manipulating functions

        Args:
            function_name: Name of the editing function in FUNCTION_MAP
            title: Question title/identifier
            passage: Passage text
            result: Question text (may be processed)
            answer_part: Answer text
            explanation_part: Explanation text

        Returns:
            Tuple of (title, passage, question_line, question_part, answer_part, explanation_part)
            - question_line: Optional instruction line (e.g., "다음 글의 주제로 가장 적절한 것은?")
            - question_part: The actual question content
        """
        # Step 1: Check if this is a document-manipulating function
        # These functions call doc.add_paragraph() which doesn't work for HWPX
        if self._is_document_manipulating_function(function_name):
            self.logger.info(f"Using HWPX-specific handler for {function_name}")
            return self._process_document_manipulating_function(
                function_name, title, passage, result, answer_part, explanation_part
            )

        # Step 2: Function lookup
        func = FUNCTION_MAP.get(function_name)
        if not func:
            self.logger.error(f"Function {function_name} not found in FUNCTION_MAP.")
            # Return default values to prevent further processing
            return title, passage, "", "", answer_part, explanation_part

        # Step 3: Inspect function signature
        sig = inspect.signature(func)
        params = sig.parameters

        # Step 4: Assemble base arguments
        # Most functions expect: title, passage, result, answer_part, explanation_part
        base_args = [title, passage, result, answer_part, explanation_part]

        # Step 5: Add logger parameter (always required)
        base_args.append(self.logger)

        # Special case: 연결어 functions need two_column flag
        # For HWPX, we'll default to False for now
        if '연결어' in function_name:
            base_args.append(False)  # two_column flag

        # Step 6: Execute the function
        try:
            processed = func(*base_args)
        except Exception as e:
            self.logger.error(f"Error executing {function_name}: {e}")
            # Return default values in case of error
            return title, passage, "", "", answer_part, explanation_part

        # Step 7: Parse return values
        # Initialize default return values
        question_line = ""
        question_part = ""

        # Determine the structure of the returned tuple
        if processed and isinstance(processed, tuple):
            if len(processed) >= 6:
                # Most common format: title, passage, question_line, question_part, answer_part, explanation_part
                title, passage, question_line, question_part, answer_part, explanation_part = processed[:6]
            elif len(processed) >= 5:
                # Some functions return only 5 elements (no question_line)
                # Format: title, passage, question_part, answer_part, explanation_part
                title, passage, question_part, answer_part, explanation_part = processed[:5]
                question_line = ""
            else:
                self.logger.error(f"Unexpected return format from {function_name}: {len(processed)} elements")
                return title, passage, "", "", answer_part, explanation_part
        else:
            self.logger.error(f"Unexpected return type from {function_name}: {type(processed)}")
            return title, passage, "", "", answer_part, explanation_part

        return title, passage, question_line, question_part, answer_part, explanation_part

    def _determine_function_name(self, sheet_name: str) -> str:
        """
        Determine the editing function name from the sheet name.

        Mirrors the logic from converter.py lines 1772-1777.

        Args:
            sheet_name: Name of the Excel sheet (e.g., "어휘1단계_Normal", "실전모의고사01")

        Returns:
            Function name for FUNCTION_MAP lookup (e.g., "어휘1단계_편집", "실전모의고사01_편집")
        """
        function_name = sheet_name

        # Replace difficulty suffixes with _편집
        function_name = function_name.replace('_Normal', '_편집')
        function_name = function_name.replace('_Hard', '_편집')

        # Handle special character replacements
        function_name = function_name.replace("내용일치(한)", "내용일치한")
        function_name = function_name.replace("무관한 문장", "무관한문장")

        # If sheet name doesn't end with _편집, add it (for 동형문제, 실전모의고사, 기출문제세트)
        if not function_name.endswith('_편집'):
            function_name += '_편집'

        return function_name

    def _collect_filtered_sheet(
        self,
        wb,
        sheet_name: str,
        section_title: str,
        passage_order: Sequence[str],
        passage_filter: set[str],
    ) -> Tuple[List[HwpxQuestion], List[AnswerEntry]]:
        sheet = wb[sheet_name]
        rows: Dict[str, Tuple[str, str, str, str, str]] = {}
        for row in sheet.iter_rows(min_row=1, values_only=True):
            if row and row[0]:
                pid = str(row[0]).strip()
                # Extract up to 5 columns, padding with empty strings if needed
                row_data = ["" if cell is None else str(cell) for cell in row[:5]]
                # Pad to ensure exactly 5 elements
                while len(row_data) < 5:
                    row_data.append("")
                rows[pid] = tuple(row_data)

        # Determine the editing function name
        function_name = self._determine_function_name(sheet_name)

        blocks: List[HwpxQuestion] = []
        answers: List[AnswerEntry] = []
        for pid in passage_order:
            if pid not in passage_filter:
                continue
            data = rows.get(pid)
            if not data:
                continue
            title, passage, result, answer, explanation = data[:5]

            # Process the question data through editing function
            title, passage, question_line, question_part, answer, explanation = self._process_question_editing(
                function_name, title, passage, result, answer, explanation
            )

            # Combine question_line and question_part for HWPX
            # If question_line exists, it becomes the first line, followed by question_part
            if question_line and question_part:
                combined_question = f"{question_line}\n{question_part}"
            elif question_line:
                combined_question = question_line
            else:
                combined_question = question_part

            passage_lines = self._split_keep_blanks(passage)
            question_lines = self._split_keep_blanks(combined_question)
            explanation_lines = self._split_keep_blanks(explanation)

            blocks.append(
                HwpxQuestion(
                    section_title=section_title,
                    question_title=title or pid,
                    passage_lines=passage_lines,
                    question_lines=question_lines,
                    answer=answer,
                    explanation_lines=explanation_lines,
                )
            )
            answers.append(
                AnswerEntry(
                    question_title=title or pid,
                    answer=answer,
                    explanation_lines=explanation_lines,
                )
            )
        return blocks, answers

    def _collect_full_sheet(
        self,
        wb,
        sheet_name: str,
        section_title: str,
        *,
        number_within_section: bool = False,
    ) -> Tuple[List[HwpxQuestion], List[AnswerEntry]]:
        if sheet_name not in wb.sheetnames:
            return [], []
        sheet = wb[sheet_name]

        # Determine the editing function name
        function_name = self._determine_function_name(sheet_name)

        blocks: List[HwpxQuestion] = []
        answers: List[AnswerEntry] = []
        counter = 1
        for row in sheet.iter_rows(min_row=1, values_only=True):
            if not row or not row[0]:
                continue
            pid = str(row[0]).strip()
            # Handle variable column counts safely
            passage = "" if len(row) < 2 or row[1] is None else str(row[1])
            result = "" if len(row) < 3 or row[2] is None else str(row[2])
            answer = "" if len(row) < 4 or row[3] is None else str(row[3])
            explanation = "" if len(row) < 5 or row[4] is None else str(row[4])

            # Process the question data through editing function
            title, passage, question_line, question_part, answer, explanation = self._process_question_editing(
                function_name, pid, passage, result, answer, explanation
            )

            # Combine question_line and question_part for HWPX
            if question_line and question_part:
                combined_question = f"{question_line}\n{question_part}"
            elif question_line:
                combined_question = question_line
            else:
                combined_question = question_part

            heading = title or pid
            if number_within_section:
                heading = f"{counter}. {heading}"

            passage_lines = self._split_keep_blanks(passage)
            question_lines = self._split_keep_blanks(combined_question)
            explanation_lines = self._split_keep_blanks(explanation)

            blocks.append(
                HwpxQuestion(
                    section_title=section_title,
                    question_title=heading,
                    passage_lines=passage_lines,
                    question_lines=question_lines,
                    answer=answer,
                    explanation_lines=explanation_lines,
                )
            )
            answers.append(
                AnswerEntry(
                    question_title=heading,
                    answer=answer,
                    explanation_lines=explanation_lines,
                )
            )
            counter += 1
        return blocks, answers

    @staticmethod
    def _split_keep_blanks(text: str) -> List[str]:
        if not text:
            return []
        parts = text.replace("\r\n", "\n").split("\n")
        return [part.rstrip() for part in parts]

    # ------------------------------------------------------------------
    # Document-manipulating function handlers (Sprint 4)
    # ------------------------------------------------------------------

    def _is_document_manipulating_function(self, function_name: str) -> bool:
        """
        Check if a function is one of the 5 document-manipulating functions.

        These functions call doc.add_paragraph() in Word export, which doesn't
        work for HWPX. They need special handling.
        """
        doc_manip_functions = {
            '빈칸암기_편집',
            '빈칸암기_정답_편집',
            '연결어_편집',
            '동반의어문제1_편집',
            '동반의어문제2_편집',
        }
        return function_name in doc_manip_functions

    def _process_document_manipulating_function(
        self,
        function_name: str,
        title: str,
        passage: str,
        result: str,
        answer_part: str,
        explanation_part: str,
    ) -> Tuple[str, str, str, str, str, str]:
        """
        Process document-manipulating functions for HWPX export.

        These functions create formatted document structures in Word export.
        For HWPX, we extract the formatting information and return structured data.

        Args:
            function_name: Name of the editing function
            title: Question title/identifier
            passage: Passage text
            result: Question text (contains formatting markers)
            answer_part: Answer text
            explanation_part: Explanation text

        Returns:
            Tuple of (title, passage, question_line, question_part, answer_part, explanation_part)
        """
        if function_name == '빈칸암기_편집':
            return self._process_blank_memorization(
                title, passage, result, answer_part, explanation_part, answer_mode=False
            )
        elif function_name == '빈칸암기_정답_편집':
            return self._process_blank_memorization(
                title, passage, result, answer_part, explanation_part, answer_mode=True
            )
        elif function_name == '연결어_편집':
            return self._process_connective(
                title, passage, result, answer_part, explanation_part
            )
        elif function_name in ('동반의어문제1_편집', '동반의어문제2_편집'):
            return self._process_synonym_problem(
                title, passage, result, answer_part, explanation_part
            )
        else:
            # Fallback - shouldn't reach here
            self.logger.warning(f"Unknown document-manipulating function: {function_name}")
            return title, passage, "", result, answer_part, explanation_part

    def _process_blank_memorization(
        self,
        title: str,
        passage: str,
        result: str,
        answer_part: str,
        explanation_part: str,
        answer_mode: bool = False,
    ) -> Tuple[str, str, str, str, str, str]:
        """
        Process 빈칸암기_편집 and 빈칸암기_정답_편집 functions.

        These create bordered paragraphs with:
        - [word] → Gray (or black in answer mode) highlighting with first letter visible
        - {text} → Italic text
        - Explanation lines with {text} → italic

        For HWPX, we'll keep the markers in the text and handle formatting
        during XML generation.

        Args:
            answer_mode: If True, this is answer mode (black text), else gray text
        """
        if explanation_part is None or explanation_part.strip() == "":
            explanation_part = " "

        # For answer mode, we return different values
        if answer_mode:
            # 빈칸암기_정답_편집 clears passage, question, explanation
            # and puts result in answer position
            return title, "", "", result, "", ""
        else:
            # 빈칸암기_편집 keeps result as answer
            answer_part = result
            return title, passage, "", result, answer_part, explanation_part

    def _process_connective(
        self,
        title: str,
        passage: str,
        result: str,
        answer_part: str,
        explanation_part: str,
    ) -> Tuple[str, str, str, str, str, str]:
        """
        Process 연결어_편집 function.

        This function:
        1. Splits at "(1) " to separate passage and questions
        2. Creates bordered passage paragraph with:
           - <text> or [text] → Bold + underline
           - {text} → Italic
           - [text] replaced with ______ in passage
        3. Creates multiple question paragraphs with tab stops

        For HWPX, we keep the markers and process during XML generation.
        """
        if explanation_part is None:
            explanation_part = ""

        # Split at "(1) "
        split_point = result.find("(1) ")
        if split_point == -1:
            # No split point found, treat all as passage
            passage_text = result.strip()
            question_text = ""
        else:
            passage_text = result[:split_point].strip()
            question_text = result[split_point:].strip()

        # For HWPX, we keep the markers and formatting
        # The passage will be rendered in a bordered table
        # The questions will be rendered as multiple paragraphs

        return title, passage_text, "", question_text, answer_part, explanation_part

    def _process_synonym_problem(
        self,
        title: str,
        passage: str,
        result: str,
        answer_part: str,
        explanation_part: str,
    ) -> Tuple[str, str, str, str, str, str]:
        """
        Process 동반의어문제1_편집 and 동반의어문제2_편집 functions.

        These create paragraphs with:
        - '아닌' → Bold + underline
        - Tab stops at 1.5" intervals

        For HWPX, we keep the markers and process during XML generation.
        """
        # These functions just format the result text with bold/underline for '아닌'
        # and add tab stops - no major structure changes
        return title, passage, "", result, answer_part, explanation_part

    # ------------------------------------------------------------------
    # Template handling
    # ------------------------------------------------------------------

    def _read_template(self) -> Tuple[List[str], Dict[str, bytes]]:
        with zipfile.ZipFile(self.template_path, "r") as tpl:
            names = tpl.namelist()
            contents = {name: tpl.read(name) for name in names}
        if "Contents/section0.xml" not in contents:
            raise ValueError("Template missing Contents/section0.xml")
        if "mimetype" not in contents:
            raise ValueError("Template missing mimetype entry")
        return names, contents

    # ------------------------------------------------------------------
    # Section reconstruction
    # ------------------------------------------------------------------

    def _build_section_xml(
        self,
        question_blocks: Sequence[HwpxQuestion],
        answers_map: "OrderedDict[str, List[AnswerEntry]]",
    ) -> bytes:
        """
        Populate the new section XML using the format of the HWPX template.

        The template contains six prototypes in this order:
          0. Section heading paragraph (one per sheet/type)
          1. Passage identifier paragraph (e.g., "11-10")
          2. 1×1 passage table paragraph
          3. Question text paragraph
          4. 3-column 정답/해설 table paragraph
          5. 10-column 빠른 정답 찾기 table paragraph

        We clone each prototype, fill its text, and remove the template leftovers while
        keeping all original styling (fonts, alignment, cell margins, etc.).
        """

        section_root = ET.fromstring(self._read_section_template())
        prototype_children = list(section_root)
        if len(prototype_children) < 6:
            raise ValueError(
                "Template must include section title, passage ID, passage table, "
                "question text, answer table, and quick-answer table."
            )

        heading_proto = copy.deepcopy(prototype_children[0])
        passage_id_proto = copy.deepcopy(prototype_children[1])
        table_proto = copy.deepcopy(prototype_children[2])
        question_proto = copy.deepcopy(prototype_children[3])
        answer_table_proto = copy.deepcopy(prototype_children[4])
        quick_table_proto = copy.deepcopy(prototype_children[5])

        passage_cell_proto = table_proto.find(f".//{{{HP_NS}}}p")
        if passage_cell_proto is None:
            raise ValueError("Template passage table missing hp:p element.")

        answer_cell_proto = answer_table_proto.find(f".//{{{HP_NS}}}p")
        if answer_cell_proto is None:
            raise ValueError("Answer table prototype missing hp:p element.")

        quick_cell_proto = quick_table_proto.find(f".//{{{HP_NS}}}p")
        if quick_cell_proto is None:
            raise ValueError("Quick answer table prototype missing hp:p element.")

        namespace_attrs = {
            attr: value
            for attr, value in section_root.attrib.items()
            if attr.startswith(f"{{{XMLNS}}}")
        }

        for child in list(section_root):
            section_root.remove(child)

        current_section = None
        table_index = 1

        for idx, block in enumerate(question_blocks):
            # --- Question block: emit passage id, passage table, and all question text ---
            # Each iteration writes a single Excel row (one question). We mimic the
            # template structure: heading paragraph (only when the sheet changes),
            # passage-id paragraph, the passage table, then one or more question paragraphs.
            if block.section_title != current_section:
                # New sheet/type → emit the section heading and force a page break.
                heading_para = copy.deepcopy(heading_proto)
                self._set_text(heading_para, block.section_title, remove_secpr=current_section is not None)
                if current_section is not None:
                    heading_para.set("pageBreak", "1")
                section_root.append(heading_para)
                current_section = block.section_title

            # Paragraph for the passage identifier (e.g., "11-10").
            passage_para = copy.deepcopy(passage_id_proto)
            self._set_text(passage_para, block.question_title)
            section_root.append(passage_para)

            # 1×1 table that holds the passage text itself.
            table_para = copy.deepcopy(table_proto)
            self._populate_table(table_para, passage_cell_proto, block.passage_lines, table_index)
            table_index += 1
            section_root.append(table_para)

            question_lines = list(block.question_lines)
            if question_lines:
                # First question line becomes the "stem" paragraph (uses the template formatting).
                primary_question_text = question_lines.pop(0)
                question_intro_para = copy.deepcopy(question_proto)
                self._set_text(question_intro_para, primary_question_text)
                section_root.append(question_intro_para)

            for line in question_lines:
                # Remaining question lines (choices, follow-up instructions, etc.).
                question_line_para = copy.deepcopy(question_proto)
                self._set_text(question_line_para, line)
                section_root.append(question_line_para)

            # Blank paragraph to visually separate the next question block.
            separator_para = copy.deepcopy(question_proto)
            self._set_text(separator_para, "")
            section_root.append(separator_para)

        table_index = self._append_answers_section(
            section_root,
            heading_proto,
            passage_id_proto,
            answer_table_proto,
            answer_cell_proto,
            answers_map,
            table_index,
        )
        table_index = self._append_quick_answers(
            section_root,
            heading_proto,
            quick_table_proto,
            quick_cell_proto,
            answers_map,
            table_index,
        )

        section_root.attrib.update(namespace_attrs)
        return ET.tostring(section_root, encoding="utf-8", xml_declaration=True)

    def _read_section_template(self) -> bytes:
        with zipfile.ZipFile(self.template_path, "r") as tpl:
            return tpl.read("Contents/section0.xml")

    def _set_text(self, paragraph: ET.Element, text: str, remove_secpr: bool = False) -> None:
        run = paragraph.find(f"./{{{HP_NS}}}run")
        if run is None:
            run = ET.SubElement(paragraph, f"{{{HP_NS}}}run")
        if remove_secpr:
            sec_pr = run.find(f"./{{{HP_NS}}}secPr")
            if sec_pr is not None:
                run.remove(sec_pr)
            ctrl = run.find(f"./{{{HP_NS}}}ctrl")
            if ctrl is not None:
                run.remove(ctrl)
        runs = paragraph.findall(f"./{{{HP_NS}}}run")
        for extra_run in runs[1:]:
            paragraph.remove(extra_run)

        t_nodes = run.findall(f"./{{{HP_NS}}}t")
        if t_nodes:
            t_node = t_nodes[0]
            for extra in t_nodes[1:]:
                run.remove(extra)
        else:
            t_node = ET.SubElement(run, f"{{{HP_NS}}}t")
        t_node.text = text or ""

        for lineseg_array in paragraph.findall(f"./{{{HP_NS}}}linesegarray"):
            paragraph.remove(lineseg_array)

    def _populate_table(
        self,
        table_para: ET.Element,
        cell_para_proto: ET.Element,
        passage_lines: Sequence[str],
        table_index: int,
    ) -> None:
        table = table_para.find(f".//{{{HP_NS}}}tbl")
        if table is not None:
            table.set("id", f"tbl{table_index}")

        sub_list = table_para.find(f".//{{{HP_NS}}}subList")
        if sub_list is None:
            return

        for child in list(sub_list):
            sub_list.remove(child)

        if not passage_lines:
            passage_lines = [""]

        for line in passage_lines:
            para = copy.deepcopy(cell_para_proto)
            self._set_text(para, line)
            sub_list.append(para)

    def _append_answers_section(
        self,
        section_root: ET.Element,
        heading_proto: ET.Element,
        passage_proto: ET.Element,
        table_proto: ET.Element,
        cell_para_proto: ET.Element,
        answers_map: "OrderedDict[str, List[AnswerEntry]]",
        table_index: int,
    ) -> int:
        # --- Summary section: build 정답 및 해설 table using template styling ---
        # --- Quick answer section: alternating 지문번호 / 정답 rows in a 10-column table ---
        # --- Summary section: "정답 및 해설" ---
        #     We reuse the template's 3-column table (문항/정답/해설) so fonts and
        #     alignment stay consistent. Each sheet gets its own copy of this table.
        heading_para = copy.deepcopy(heading_proto)
        self._set_text(heading_para, "정답 및 해설", remove_secpr=True)
        heading_para.set("pageBreak", "1")
        section_root.append(heading_para)

        current_table_index = table_index

        for section_title, entries in answers_map.items():
            if not entries:
                continue

            # Heading paragraph for the sheet/type (e.g., "1. 어휘1단계_Normal").
            section_heading = copy.deepcopy(passage_proto)
            self._set_text(section_heading, section_title)
            section_root.append(section_heading)

            table_para = copy.deepcopy(table_proto)
            table = table_para.find(f".//{{{HP_NS}}}tbl")
            if table is None:
                raise ValueError("Answer table prototype missing hp:tbl element.")

            row_templates = table.findall(f"./{{{HP_NS}}}tr")
            if len(row_templates) < 2:
                raise ValueError("Answer table prototype must contain a header and data row.")

            header_template = row_templates[0]
            data_template = row_templates[1]
            column_widths = self._extract_column_widths(header_template)

            for tr in row_templates:
                table.remove(tr)

            table.set("id", f"tbl{current_table_index}")

            header_row = self._prepare_table_row(
                header_template,
                cell_para_proto,
                ["문항", "정답", "해설"],
                0,
                column_widths,
            )
            table.append(header_row)

            row_idx = 1
            for entry in entries:
                # One row per question: column order is passage id, answer choice, explanation text.
                values = [
                    entry.question_title,
                    entry.answer or "",
                    "\n".join(filter(None, entry.explanation_lines)) if entry.explanation_lines else "",
                ]
                data_row = self._prepare_table_row(
                    data_template,
                    cell_para_proto,
                    values,
                    row_idx,
                    column_widths,
                )
                table.append(data_row)
                row_idx += 1

            table.set("rowCnt", str(row_idx))
            table.set("colCnt", str(len(column_widths)))

            self._cleanup_table_paragraph(table_para)
            section_root.append(table_para)
            current_table_index += 1

        return current_table_index

    def _append_quick_answers(
        self,
        section_root: ET.Element,
        heading_proto: ET.Element,
        table_proto: ET.Element,
        cell_para_proto: ET.Element,
        answers_map: "OrderedDict[str, List[AnswerEntry]]",
        table_index: int,
    ) -> int:
        # --- Quick answer section: "빠른 정답 찾기" ---
        #     Alternating rows (지문번호 / 정답) in a fixed-width 10-column table. Each
        #     pair of rows can hold up to 10 passage IDs and answers; we append more
        #     row pairs when there are more questions.
        heading_para = copy.deepcopy(heading_proto)
        self._set_text(heading_para, "빠른 정답 찾기", remove_secpr=True)
        heading_para.set("pageBreak", "1")
        section_root.append(heading_para)

        flat_entries: List[AnswerEntry] = []
        for entry_list in answers_map.values():
            flat_entries.extend(entry_list)

        if not flat_entries:
            return table_index

        base_table = table_proto.find(f".//{{{HP_NS}}}tbl")
        if base_table is None:
            raise ValueError("Quick answer table prototype missing hp:tbl element.")

        row_templates = base_table.findall(f"./{{{HP_NS}}}tr")
        if len(row_templates) < 2:
            raise ValueError("Quick answer table prototype must contain two rows.")

        passage_row_template = row_templates[0]
        answer_row_template = row_templates[1]
        column_widths = self._extract_column_widths(passage_row_template)
        column_count = len(column_widths) if column_widths else len(passage_row_template.findall(f"./{{{HP_NS}}}tc"))
        if column_count == 0:
            column_count = 10

        table_para = copy.deepcopy(table_proto)
        table = table_para.find(f".//{{{HP_NS}}}tbl")
        if table is None:
            raise ValueError("Quick answer table prototype missing hp:tbl element.")

        for tr in table.findall(f"./{{{HP_NS}}}tr"):
            table.remove(tr)

        row_idx = 0
        # Loop through questions in groups of at most 10 so each table stays within the template layout.
        for start in range(0, len(flat_entries), column_count):
            chunk = flat_entries[start : start + column_count]
            passage_values = [entry.question_title for entry in chunk]
            answer_values = [entry.answer or "" for entry in chunk]
            passage_values.extend([""] * (column_count - len(passage_values)))
            answer_values.extend([""] * (column_count - len(answer_values)))

            # Odd row: passage identifiers (지문번호)
            passage_row = self._prepare_table_row(
                passage_row_template,
                cell_para_proto,
                passage_values,
                row_idx,
                column_widths,
            )
            table.append(passage_row)
            row_idx += 1

            # Even row: answers for the same column positions.
            answer_row = self._prepare_table_row(
                answer_row_template,
                cell_para_proto,
                answer_values,
                row_idx,
                column_widths,
            )
            table.append(answer_row)
            row_idx += 1

        table.set("id", f"tbl{table_index}")
        table.set("rowCnt", str(row_idx))
        table.set("colCnt", str(column_count))

        self._cleanup_table_paragraph(table_para)
        section_root.append(table_para)

        return table_index + 1

    def _extract_column_widths(self, row_template: ET.Element) -> List[int]:
        widths: List[int] = []
        for tc in row_template.findall(f"./{{{HP_NS}}}tc"):
            cell_sz = tc.find(f"./{{{HP_NS}}}cellSz")
            width = int(cell_sz.get("width", "0")) if cell_sz is not None else 0
            widths.append(width)
        return widths

    def _prepare_table_row(
        self,
        row_template: ET.Element,
        cell_para_proto: ET.Element,
        values: Sequence[str],
        row_index: int,
        column_widths: Sequence[int],
    ) -> ET.Element:
        row = copy.deepcopy(row_template)
        cells = row.findall(f"./{{{HP_NS}}}tc")
        padded_values = list(values) + [""] * (len(cells) - len(values))

        for col_idx, tc in enumerate(cells):
            width = column_widths[col_idx] if col_idx < len(column_widths) else 0
            value = padded_values[col_idx]
            # Reuse the template cell (keeping alignment/borders) but replace the text and metadata.
            self._set_cell_contents(tc, cell_para_proto, value, row_index, col_idx, width)

        return row

    def _set_cell_contents(
        self,
        tc: ET.Element,
        cell_para_proto: ET.Element,
        value: str,
        row_idx: int,
        col_idx: int,
        width: int,
    ) -> None:
        # Update cell coordinates so Hanword knows the row/column.
        cell_addr = tc.find(f"./{{{HP_NS}}}cellAddr")
        if cell_addr is None:
            cell_addr = ET.SubElement(tc, f"{{{HP_NS}}}cellAddr")
        cell_addr.set("colAddr", str(col_idx))
        cell_addr.set("rowAddr", str(row_idx))

        cell_span = tc.find(f"./{{{HP_NS}}}cellSpan")
        if cell_span is None:
            cell_span = ET.SubElement(tc, f"{{{HP_NS}}}cellSpan")
        cell_span.set("colSpan", "1")
        cell_span.set("rowSpan", "1")

        cell_sz = tc.find(f"./{{{HP_NS}}}cellSz")
        if cell_sz is None:
            cell_sz = ET.SubElement(tc, f"{{{HP_NS}}}cellSz")
        if width > 0:
            cell_sz.set("width", str(width))

        sub_list = tc.find(f"./{{{HP_NS}}}subList")
        if sub_list is None:
            sub_list = ET.SubElement(tc, f"{{{HP_NS}}}subList")

        paras = sub_list.findall(f"./{{{HP_NS}}}p")
        if paras:
            target_para = paras[0]
            for extra_para in paras[1:]:
                sub_list.remove(extra_para)
        else:
            if cell_para_proto is not None:
                target_para = copy.deepcopy(cell_para_proto)
            else:
                target_para = ET.Element(f"{{{HP_NS}}}p")
            sub_list.append(target_para)

        self._set_text(target_para, value)

    def _cleanup_table_paragraph(self, table_para: ET.Element) -> None:
        table_run = table_para.find(f".//{{{HP_NS}}}run")
        if table_run is not None:
            for t_node in list(table_run.findall(f"./{{{HP_NS}}}t")):
                table_run.remove(t_node)
        for lineseg in list(table_para.findall(f"./{{{HP_NS}}}linesegarray")):
            table_para.remove(lineseg)

    # ------------------------------------------------------------------
    # Archive writer
    # ------------------------------------------------------------------

    @staticmethod
    def _write_hwpx(
        output_path: Path,
        names: Sequence[str],
        contents: Dict[str, bytes],
    ) -> None:
        with zipfile.ZipFile(output_path, "w", compression=zipfile.ZIP_DEFLATED) as out_zip:
            mimetype_info = zipfile.ZipInfo(filename="mimetype")
            mimetype_info.compress_type = zipfile.ZIP_STORED
            out_zip.writestr(mimetype_info, contents["mimetype"])

            for name in names:
                if name == "mimetype":
                    continue
                out_zip.writestr(name, contents[name])
