"""
Excel → HWPX exporter aligned with the desktop DOCX workflow.

The desktop Word exporter groups questions by sheet type (Normal/Hard variants,
동형문제, 실전모의고사, 기출문제세트), then appends “정답 및 해설” and
“빠른 정답 찾기”.  This module mirrors that ordering while emitting a lightweight
HWPX document based on the bundled template at assets/hwp/questions.hwpx.

Key behaviors retained from the original exporter:
  * question heading paragraph
  * passage rendered inside a 1×1 table with borders
  * question text as plain paragraphs
  * summary sections for answers and quick lookup
"""

from __future__ import annotations

from collections import OrderedDict
from dataclasses import dataclass
from pathlib import Path
from typing import Dict, Iterable, List, Optional, Sequence, Tuple
import copy
import zipfile
import xml.etree.ElementTree as ET

from openpyxl import load_workbook


SECTION_NAMESPACES = {
    "ha": "http://www.hancom.co.kr/hwpml/2011/app",
    "hp": "http://www.hancom.co.kr/hwpml/2011/paragraph",
    "hp10": "http://www.hancom.co.kr/hwpml/2016/paragraph",
    "hs": "http://www.hancom.co.kr/hwpml/2011/section",
    "hc": "http://www.hancom.co.kr/hwpml/2011/core",
    "hh": "http://www.hancom.co.kr/hwpml/2011/head",
    "hhs": "http://www.hancom.co.kr/hwpml/2011/history",
    "hm": "http://www.hancom.co.kr/hwpml/2011/master-page",
    "hpf": "http://www.hancom.co.kr/schema/2011/hpf",
    "dc": "http://purl.org/dc/elements/1.1/",
    "opf": "http://www.idpf.org/2007/opf/",
    "ooxmlchart": "http://www.hancom.co.kr/hwpml/2016/ooxmlchart",
    "hwpunitchar": "http://www.hancom.co.kr/hwpml/2016/HwpUnitChar",
    "epub": "http://www.idpf.org/2007/ops",
    "config": "urn:oasis:names:tc:opendocument:xmlns:config:1.0",
}

for prefix, uri in SECTION_NAMESPACES.items():
    ET.register_namespace(prefix, uri)

HP_NS = SECTION_NAMESPACES["hp"]
XMLNS = "http://www.w3.org/2000/xmlns/"
XMLNS = "http://www.w3.org/2000/xmlns/"


@dataclass
class HwpxQuestion:
    section_title: str
    question_title: str
    passage_lines: List[str]
    question_lines: List[str]
    answer: str
    explanation_lines: List[str]


@dataclass
class AnswerEntry:
    question_title: str
    answer: str
    explanation_lines: List[str]


class ExcelToHwpxExporter:
    """
    Convert selected Excel questions into an HWPX file following the same
    ordering rules as ExcelToWordConverter.
    """

    def __init__(self, template_path: Optional[Path] = None) -> None:
        default_template = (
            Path(__file__).resolve().parent.parent / "assets" / "hwp" / "questions.hwpx"
        )
        self.template_path = Path(template_path or default_template)
        if not self.template_path.exists():
            raise FileNotFoundError(
                f"HWPX template not found at {self.template_path}. "
                "Place questions.hwpx at assets/hwp/ or pass template_path."
            )

    def export(
        self,
        excel_path: Path | str,
        output_path: Path | str,
        passage_ids: Iterable[str],
        rows_order: Sequence[str],
        selection_state: Dict[str, bool],
        *,
        copyquestion_rows: Sequence[str] = (),
        mock_test_rows: Sequence[str] = (),
        past_test_rows: Sequence[str] = (),
    ) -> Path:
        excel_path = Path(excel_path)
        if not excel_path.exists():
            raise FileNotFoundError(f"Excel file not found: {excel_path}")

        output_path = Path(output_path)
        output_path.parent.mkdir(parents=True, exist_ok=True)

        passage_order = [pid.strip() for pid in passage_ids if pid.strip()]
        passage_filter = set(passage_order)

        question_blocks: List[HwpxQuestion] = []
        answers_map: "OrderedDict[str, List[AnswerEntry]]" = OrderedDict()

        copy_set = set(copyquestion_rows)
        mock_set = set(mock_test_rows)
        past_set = set(past_test_rows)

        wb = load_workbook(filename=str(excel_path), read_only=True, data_only=True)
        try:
            section_counter = 1
            for row_key in rows_order:
                if row_key in copy_set:
                    if selection_state.get(row_key):
                        title = f"{section_counter}. {row_key}"
                        blocks, entries = self._collect_full_sheet(wb, row_key, title)
                        if blocks:
                            question_blocks.extend(blocks)
                            answers_map[title] = entries
                            section_counter += 1
                    continue

                if row_key in mock_set:
                    if selection_state.get(row_key):
                        title = f"{section_counter}. {row_key}"
                        blocks, entries = self._collect_full_sheet(
                            wb, row_key, title, number_within_section=True
                        )
                        if blocks:
                            question_blocks.extend(blocks)
                            answers_map[title] = entries
                            section_counter += 1
                    continue

                if row_key in past_set:
                    if selection_state.get(row_key):
                        title = f"{section_counter}. {row_key}"
                        blocks, entries = self._collect_full_sheet(
                            wb, row_key, title, number_within_section=True
                        )
                        if blocks:
                            question_blocks.extend(blocks)
                            answers_map[title] = entries
                            section_counter += 1
                    continue

                for suffix in ("Normal", "Hard"):
                    key = f"{row_key}_{suffix}"
                    if not selection_state.get(key):
                        continue
                    sheet_name = f"{row_key}_{suffix}"
                    if sheet_name not in wb.sheetnames:
                        continue
                    title = f"{section_counter}. {sheet_name}"
                    blocks, entries = self._collect_filtered_sheet(
                        wb, sheet_name, title, passage_order, passage_filter
                    )
                    if blocks:
                        question_blocks.extend(blocks)
                        answers_map[title] = entries
                        section_counter += 1
        finally:
            wb.close()

        if not question_blocks:
            raise ValueError("No questions matched the current selection for HWPX export.")

        names, contents = self._read_template()
        section_xml = self._build_section_xml(question_blocks, answers_map)
        contents["Contents/section0.xml"] = section_xml
        self._write_hwpx(output_path, names, contents)
        return output_path

    # ------------------------------------------------------------------
    # Excel ingestion helpers
    # ------------------------------------------------------------------

    def _collect_filtered_sheet(
        self,
        wb,
        sheet_name: str,
        section_title: str,
        passage_order: Sequence[str],
        passage_filter: set[str],
    ) -> Tuple[List[HwpxQuestion], List[AnswerEntry]]:
        sheet = wb[sheet_name]
        rows: Dict[str, Tuple[str, str, str, str, str]] = {}
        for row in sheet.iter_rows(min_row=1, values_only=True):
            if row and row[0]:
                pid = str(row[0]).strip()
                rows[pid] = tuple("" if cell is None else str(cell) for cell in row[:5])

        blocks: List[HwpxQuestion] = []
        answers: List[AnswerEntry] = []
        for pid in passage_order:
            if pid not in passage_filter:
                continue
            data = rows.get(pid)
            if not data:
                continue
            title, passage, question, answer, explanation = data[:5]
            passage_lines = self._split_keep_blanks(passage)
            question_lines = self._split_keep_blanks(question)
            explanation_lines = self._split_keep_blanks(explanation)
            blocks.append(
                HwpxQuestion(
                    section_title=section_title,
                    question_title=title or pid,
                    passage_lines=passage_lines,
                    question_lines=question_lines,
                    answer=answer,
                    explanation_lines=explanation_lines,
                )
            )
            answers.append(
                AnswerEntry(
                    question_title=title or pid,
                    answer=answer,
                    explanation_lines=explanation_lines,
                )
            )
        return blocks, answers

    def _collect_full_sheet(
        self,
        wb,
        sheet_name: str,
        section_title: str,
        *,
        number_within_section: bool = False,
    ) -> Tuple[List[HwpxQuestion], List[AnswerEntry]]:
        if sheet_name not in wb.sheetnames:
            return [], []
        sheet = wb[sheet_name]
        blocks: List[HwpxQuestion] = []
        answers: List[AnswerEntry] = []
        counter = 1
        for row in sheet.iter_rows(min_row=1, values_only=True):
            if not row or not row[0]:
                continue
            pid = str(row[0]).strip()
            passage = "" if row[1] is None else str(row[1])
            question = "" if row[2] is None else str(row[2])
            answer = "" if row[3] is None else str(row[3])
            explanation = "" if row[4] is None else str(row[4])

            heading = pid
            if number_within_section:
                heading = f"{counter}. {pid}"

            passage_lines = self._split_keep_blanks(passage)
            question_lines = self._split_keep_blanks(question)
            explanation_lines = self._split_keep_blanks(explanation)

            blocks.append(
                HwpxQuestion(
                    section_title=section_title,
                    question_title=heading,
                    passage_lines=passage_lines,
                    question_lines=question_lines,
                    answer=answer,
                    explanation_lines=explanation_lines,
                )
            )
            answers.append(
                AnswerEntry(
                    question_title=heading,
                    answer=answer,
                    explanation_lines=explanation_lines,
                )
            )
            counter += 1
        return blocks, answers

    @staticmethod
    def _split_keep_blanks(text: str) -> List[str]:
        if not text:
            return []
        parts = text.replace("\r\n", "\n").split("\n")
        return [part.rstrip() for part in parts]

    # ------------------------------------------------------------------
    # Template handling
    # ------------------------------------------------------------------

    def _read_template(self) -> Tuple[List[str], Dict[str, bytes]]:
        with zipfile.ZipFile(self.template_path, "r") as tpl:
            names = tpl.namelist()
            contents = {name: tpl.read(name) for name in names}
        if "Contents/section0.xml" not in contents:
            raise ValueError("Template missing Contents/section0.xml")
        if "mimetype" not in contents:
            raise ValueError("Template missing mimetype entry")
        return names, contents

    # ------------------------------------------------------------------
    # Section reconstruction
    # ------------------------------------------------------------------

    def _build_section_xml(
        self,
        question_blocks: Sequence[HwpxQuestion],
        answers_map: "OrderedDict[str, List[AnswerEntry]]",
    ) -> bytes:
        """
        Populate the new section XML using the format of the HWPX template.

        The template contains six prototypes in this order:
          0. Section heading paragraph (one per sheet/type)
          1. Passage identifier paragraph (e.g., "11-10")
          2. 1×1 passage table paragraph
          3. Question text paragraph
          4. 3-column 정답/해설 table paragraph
          5. 10-column 빠른 정답 찾기 table paragraph

        We clone each prototype, fill its text, and remove the template leftovers while
        keeping all original styling (fonts, alignment, cell margins, etc.).
        """

        section_root = ET.fromstring(self._read_section_template())
        prototype_children = list(section_root)
        if len(prototype_children) < 6:
            raise ValueError(
                "Template must include section title, passage ID, passage table, "
                "question text, answer table, and quick-answer table."
            )

        heading_proto = copy.deepcopy(prototype_children[0])
        passage_id_proto = copy.deepcopy(prototype_children[1])
        table_proto = copy.deepcopy(prototype_children[2])
        question_proto = copy.deepcopy(prototype_children[3])
        answer_table_proto = copy.deepcopy(prototype_children[4])
        quick_table_proto = copy.deepcopy(prototype_children[5])

        passage_cell_proto = table_proto.find(f".//{{{HP_NS}}}p")
        if passage_cell_proto is None:
            raise ValueError("Template passage table missing hp:p element.")

        answer_cell_proto = answer_table_proto.find(f".//{{{HP_NS}}}p")
        if answer_cell_proto is None:
            raise ValueError("Answer table prototype missing hp:p element.")

        quick_cell_proto = quick_table_proto.find(f".//{{{HP_NS}}}p")
        if quick_cell_proto is None:
            raise ValueError("Quick answer table prototype missing hp:p element.")

        namespace_attrs = {
            attr: value
            for attr, value in section_root.attrib.items()
            if attr.startswith(f"{{{XMLNS}}}")
        }

        for child in list(section_root):
            section_root.remove(child)

        current_section = None
        table_index = 1

        for idx, block in enumerate(question_blocks):
            # --- Question block: emit passage id, passage table, and all question text ---
            # Each iteration writes a single Excel row (one question). We mimic the
            # template structure: heading paragraph (only when the sheet changes),
            # passage-id paragraph, the passage table, then one or more question paragraphs.
            if block.section_title != current_section:
                # New sheet/type → emit the section heading and force a page break.
                heading_para = copy.deepcopy(heading_proto)
                self._set_text(heading_para, block.section_title, remove_secpr=current_section is not None)
                if current_section is not None:
                    heading_para.set("pageBreak", "1")
                section_root.append(heading_para)
                current_section = block.section_title

            # Paragraph for the passage identifier (e.g., "11-10").
            passage_para = copy.deepcopy(passage_id_proto)
            self._set_text(passage_para, block.question_title)
            section_root.append(passage_para)

            # 1×1 table that holds the passage text itself.
            table_para = copy.deepcopy(table_proto)
            self._populate_table(table_para, passage_cell_proto, block.passage_lines, table_index)
            table_index += 1
            section_root.append(table_para)

            question_lines = list(block.question_lines)
            if question_lines:
                # First question line becomes the "stem" paragraph (uses the template formatting).
                primary_question_text = question_lines.pop(0)
                question_intro_para = copy.deepcopy(question_proto)
                self._set_text(question_intro_para, primary_question_text)
                section_root.append(question_intro_para)

            for line in question_lines:
                # Remaining question lines (choices, follow-up instructions, etc.).
                question_line_para = copy.deepcopy(question_proto)
                self._set_text(question_line_para, line)
                section_root.append(question_line_para)

            # Blank paragraph to visually separate the next question block.
            separator_para = copy.deepcopy(question_proto)
            self._set_text(separator_para, "")
            section_root.append(separator_para)

        table_index = self._append_answers_section(
            section_root,
            heading_proto,
            passage_id_proto,
            answer_table_proto,
            answer_cell_proto,
            answers_map,
            table_index,
        )
        table_index = self._append_quick_answers(
            section_root,
            heading_proto,
            quick_table_proto,
            quick_cell_proto,
            answers_map,
            table_index,
        )

        section_root.attrib.update(namespace_attrs)
        return ET.tostring(section_root, encoding="utf-8", xml_declaration=True)

    def _read_section_template(self) -> bytes:
        with zipfile.ZipFile(self.template_path, "r") as tpl:
            return tpl.read("Contents/section0.xml")

    def _set_text(self, paragraph: ET.Element, text: str, remove_secpr: bool = False) -> None:
        run = paragraph.find(f"./{{{HP_NS}}}run")
        if run is None:
            run = ET.SubElement(paragraph, f"{{{HP_NS}}}run")
        if remove_secpr:
            sec_pr = run.find(f"./{{{HP_NS}}}secPr")
            if sec_pr is not None:
                run.remove(sec_pr)
            ctrl = run.find(f"./{{{HP_NS}}}ctrl")
            if ctrl is not None:
                run.remove(ctrl)
        runs = paragraph.findall(f"./{{{HP_NS}}}run")
        for extra_run in runs[1:]:
            paragraph.remove(extra_run)

        t_nodes = run.findall(f"./{{{HP_NS}}}t")
        if t_nodes:
            t_node = t_nodes[0]
            for extra in t_nodes[1:]:
                run.remove(extra)
        else:
            t_node = ET.SubElement(run, f"{{{HP_NS}}}t")
        t_node.text = text or ""

        for lineseg_array in paragraph.findall(f"./{{{HP_NS}}}linesegarray"):
            paragraph.remove(lineseg_array)

    def _populate_table(
        self,
        table_para: ET.Element,
        cell_para_proto: ET.Element,
        passage_lines: Sequence[str],
        table_index: int,
    ) -> None:
        table = table_para.find(f".//{{{HP_NS}}}tbl")
        if table is not None:
            table.set("id", f"tbl{table_index}")

        sub_list = table_para.find(f".//{{{HP_NS}}}subList")
        if sub_list is None:
            return

        for child in list(sub_list):
            sub_list.remove(child)

        if not passage_lines:
            passage_lines = [""]

        for line in passage_lines:
            para = copy.deepcopy(cell_para_proto)
            self._set_text(para, line)
            sub_list.append(para)

    def _append_answers_section(
        self,
        section_root: ET.Element,
        heading_proto: ET.Element,
        passage_proto: ET.Element,
        table_proto: ET.Element,
        cell_para_proto: ET.Element,
        answers_map: "OrderedDict[str, List[AnswerEntry]]",
        table_index: int,
    ) -> int:
        # --- Summary section: build 정답 및 해설 table using template styling ---
        # --- Quick answer section: alternating 지문번호 / 정답 rows in a 10-column table ---
        # --- Summary section: "정답 및 해설" ---
        #     We reuse the template's 3-column table (문항/정답/해설) so fonts and
        #     alignment stay consistent. Each sheet gets its own copy of this table.
        heading_para = copy.deepcopy(heading_proto)
        self._set_text(heading_para, "정답 및 해설", remove_secpr=True)
        heading_para.set("pageBreak", "1")
        section_root.append(heading_para)

        current_table_index = table_index

        for section_title, entries in answers_map.items():
            if not entries:
                continue

            # Heading paragraph for the sheet/type (e.g., "1. 어휘1단계_Normal").
            section_heading = copy.deepcopy(passage_proto)
            self._set_text(section_heading, section_title)
            section_root.append(section_heading)

            table_para = copy.deepcopy(table_proto)
            table = table_para.find(f".//{{{HP_NS}}}tbl")
            if table is None:
                raise ValueError("Answer table prototype missing hp:tbl element.")

            row_templates = table.findall(f"./{{{HP_NS}}}tr")
            if len(row_templates) < 2:
                raise ValueError("Answer table prototype must contain a header and data row.")

            header_template = row_templates[0]
            data_template = row_templates[1]
            column_widths = self._extract_column_widths(header_template)

            for tr in row_templates:
                table.remove(tr)

            table.set("id", f"tbl{current_table_index}")

            header_row = self._prepare_table_row(
                header_template,
                cell_para_proto,
                ["문항", "정답", "해설"],
                0,
                column_widths,
            )
            table.append(header_row)

            row_idx = 1
            for entry in entries:
                # One row per question: column order is passage id, answer choice, explanation text.
                values = [
                    entry.question_title,
                    entry.answer or "",
                    "\n".join(filter(None, entry.explanation_lines)) if entry.explanation_lines else "",
                ]
                data_row = self._prepare_table_row(
                    data_template,
                    cell_para_proto,
                    values,
                    row_idx,
                    column_widths,
                )
                table.append(data_row)
                row_idx += 1

            table.set("rowCnt", str(row_idx))
            table.set("colCnt", str(len(column_widths)))

            self._cleanup_table_paragraph(table_para)
            section_root.append(table_para)
            current_table_index += 1

        return current_table_index

    def _append_quick_answers(
        self,
        section_root: ET.Element,
        heading_proto: ET.Element,
        table_proto: ET.Element,
        cell_para_proto: ET.Element,
        answers_map: "OrderedDict[str, List[AnswerEntry]]",
        table_index: int,
    ) -> int:
        # --- Quick answer section: "빠른 정답 찾기" ---
        #     Alternating rows (지문번호 / 정답) in a fixed-width 10-column table. Each
        #     pair of rows can hold up to 10 passage IDs and answers; we append more
        #     row pairs when there are more questions.
        heading_para = copy.deepcopy(heading_proto)
        self._set_text(heading_para, "빠른 정답 찾기", remove_secpr=True)
        heading_para.set("pageBreak", "1")
        section_root.append(heading_para)

        flat_entries: List[AnswerEntry] = []
        for entry_list in answers_map.values():
            flat_entries.extend(entry_list)

        if not flat_entries:
            return table_index

        base_table = table_proto.find(f".//{{{HP_NS}}}tbl")
        if base_table is None:
            raise ValueError("Quick answer table prototype missing hp:tbl element.")

        row_templates = base_table.findall(f"./{{{HP_NS}}}tr")
        if len(row_templates) < 2:
            raise ValueError("Quick answer table prototype must contain two rows.")

        passage_row_template = row_templates[0]
        answer_row_template = row_templates[1]
        column_widths = self._extract_column_widths(passage_row_template)
        column_count = len(column_widths) if column_widths else len(passage_row_template.findall(f"./{{{HP_NS}}}tc"))
        if column_count == 0:
            column_count = 10

        table_para = copy.deepcopy(table_proto)
        table = table_para.find(f".//{{{HP_NS}}}tbl")
        if table is None:
            raise ValueError("Quick answer table prototype missing hp:tbl element.")

        for tr in table.findall(f"./{{{HP_NS}}}tr"):
            table.remove(tr)

        row_idx = 0
        # Loop through questions in groups of at most 10 so each table stays within the template layout.
        for start in range(0, len(flat_entries), column_count):
            chunk = flat_entries[start : start + column_count]
            passage_values = [entry.question_title for entry in chunk]
            answer_values = [entry.answer or "" for entry in chunk]
            passage_values.extend([""] * (column_count - len(passage_values)))
            answer_values.extend([""] * (column_count - len(answer_values)))

            # Odd row: passage identifiers (지문번호)
            passage_row = self._prepare_table_row(
                passage_row_template,
                cell_para_proto,
                passage_values,
                row_idx,
                column_widths,
            )
            table.append(passage_row)
            row_idx += 1

            # Even row: answers for the same column positions.
            answer_row = self._prepare_table_row(
                answer_row_template,
                cell_para_proto,
                answer_values,
                row_idx,
                column_widths,
            )
            table.append(answer_row)
            row_idx += 1

        table.set("id", f"tbl{table_index}")
        table.set("rowCnt", str(row_idx))
        table.set("colCnt", str(column_count))

        self._cleanup_table_paragraph(table_para)
        section_root.append(table_para)

        return table_index + 1

    def _extract_column_widths(self, row_template: ET.Element) -> List[int]:
        widths: List[int] = []
        for tc in row_template.findall(f"./{{{HP_NS}}}tc"):
            cell_sz = tc.find(f"./{{{HP_NS}}}cellSz")
            width = int(cell_sz.get("width", "0")) if cell_sz is not None else 0
            widths.append(width)
        return widths

    def _prepare_table_row(
        self,
        row_template: ET.Element,
        cell_para_proto: ET.Element,
        values: Sequence[str],
        row_index: int,
        column_widths: Sequence[int],
    ) -> ET.Element:
        row = copy.deepcopy(row_template)
        cells = row.findall(f"./{{{HP_NS}}}tc")
        padded_values = list(values) + [""] * (len(cells) - len(values))

        for col_idx, tc in enumerate(cells):
            width = column_widths[col_idx] if col_idx < len(column_widths) else 0
            value = padded_values[col_idx]
            # Reuse the template cell (keeping alignment/borders) but replace the text and metadata.
            self._set_cell_contents(tc, cell_para_proto, value, row_index, col_idx, width)

        return row

    def _set_cell_contents(
        self,
        tc: ET.Element,
        cell_para_proto: ET.Element,
        value: str,
        row_idx: int,
        col_idx: int,
        width: int,
    ) -> None:
        # Update cell coordinates so Hanword knows the row/column.
        cell_addr = tc.find(f"./{{{HP_NS}}}cellAddr")
        if cell_addr is None:
            cell_addr = ET.SubElement(tc, f"{{{HP_NS}}}cellAddr")
        cell_addr.set("colAddr", str(col_idx))
        cell_addr.set("rowAddr", str(row_idx))

        cell_span = tc.find(f"./{{{HP_NS}}}cellSpan")
        if cell_span is None:
            cell_span = ET.SubElement(tc, f"{{{HP_NS}}}cellSpan")
        cell_span.set("colSpan", "1")
        cell_span.set("rowSpan", "1")

        cell_sz = tc.find(f"./{{{HP_NS}}}cellSz")
        if cell_sz is None:
            cell_sz = ET.SubElement(tc, f"{{{HP_NS}}}cellSz")
        if width > 0:
            cell_sz.set("width", str(width))

        sub_list = tc.find(f"./{{{HP_NS}}}subList")
        if sub_list is None:
            sub_list = ET.SubElement(tc, f"{{{HP_NS}}}subList")

        paras = sub_list.findall(f"./{{{HP_NS}}}p")
        if paras:
            target_para = paras[0]
            for extra_para in paras[1:]:
                sub_list.remove(extra_para)
        else:
            if cell_para_proto is not None:
                target_para = copy.deepcopy(cell_para_proto)
            else:
                target_para = ET.Element(f"{{{HP_NS}}}p")
            sub_list.append(target_para)

        self._set_text(target_para, value)

    def _cleanup_table_paragraph(self, table_para: ET.Element) -> None:
        table_run = table_para.find(f".//{{{HP_NS}}}run")
        if table_run is not None:
            for t_node in list(table_run.findall(f"./{{{HP_NS}}}t")):
                table_run.remove(t_node)
        for lineseg in list(table_para.findall(f"./{{{HP_NS}}}linesegarray")):
            table_para.remove(lineseg)

    # ------------------------------------------------------------------
    # Archive writer
    # ------------------------------------------------------------------

    @staticmethod
    def _write_hwpx(
        output_path: Path,
        names: Sequence[str],
        contents: Dict[str, bytes],
    ) -> None:
        with zipfile.ZipFile(output_path, "w", compression=zipfile.ZIP_DEFLATED) as out_zip:
            mimetype_info = zipfile.ZipInfo(filename="mimetype")
            mimetype_info.compress_type = zipfile.ZIP_STORED
            out_zip.writestr(mimetype_info, contents["mimetype"])

            for name in names:
                if name == "mimetype":
                    continue
                out_zip.writestr(name, contents[name])
