# -*- coding: utf-8 -*-
"""
Macro Functions Module

This module provides keyboard macro functionality for the edit_cell_main popup in autoQM.
Extracted from autoQM.py to improve code organization and reduce main file size.

The macros support different question types with specialized text editing operations:
- 어법1단계 (Grammar Level 1) and 어휘1단계 (Vocabulary Level 1)
- 어휘2단계 (Vocabulary Level 2) with Original Words tracking
- 어법2단계 (Grammar Level 2) and 어휘3단계 (Vocabulary Level 3)
- 동반의어 (Synonyms/Antonyms)

Key bindings:
- F1: Process and select word (different behavior per question type)
- F2: Process and select two words (different behavior per question type)
- F3: Replace phrase or delete row (different behavior per question type)
- F4: Remove enclosing markers for 어법2단계
- F5: Insert correction symbol for 어법2단계
- Shift-F2: Vocabulary level 2 synonym change

Usage:
    from modules.macrofunctions import setup_macro_bindings

    # Inside edit_cell_main or similar method:
    # After creating the popup window and self.text_edits:
    setup_macro_bindings(self, popup)
"""

import tkinter as tk
from tkinter import messagebox
import re


def setup_macro_bindings(self, popup):
    """
    Set up all macro function key bindings for the editor popup.

    This function defines all macro functions in local scope and binds them to keyboard shortcuts.
    All functions are defined as nested local functions to have access to 'self' and 'popup'.

    The macro functions require access to:
    - self.text_edits (list of text widgets, particularly text_edits[0-4])
    - self.first_column_유형text (sheet name/type selector)
    - self.sheet_selector (sheet selection widget)
    - self.clipboard_A, self.clipboard_B (internal clipboards for 어법2단계)
    - self.circled_numbers (list of circled number characters)
    - self.macro (flag for macro state)
    - self.실모_유형_menu, self.실모_난이도_menu (mock test menus)
    - self.어휘2단계동의어변경flag (vocabulary synonym change flag)
    - self.fetch_circled_numbers, self.find_smallest_unused_number (helper methods)
    - self.get_char_index (helper method)
    - popup (the Toplevel window for focus management)

    Args:
        self: Reference to MainFrame instance with required attributes
        popup: Reference to the tk.Toplevel popup window
    """


    def select_next_word(event=None):
        # Clear any current selection
        self.text_edits[2].tag_remove(tk.SEL, "1.0", tk.END)

        # Find the current position of the cursor
        cursor_pos = self.text_edits[2].index(tk.INSERT)
        
        # Use a regex that includes apostrophes and hyphens as part of a word
        # Find the start of the next word from the cursor position
        next_word_start = self.text_edits[2].search(r"[\w\"'-]+", cursor_pos, regexp=True, stopindex=tk.END)
        if not next_word_start:
            return "break"  # No next word found, exit the function

        # Find the end of this next word by searching for the first non-word character
        next_word_end = self.text_edits[2].search(r"[^\w\"'-]", next_word_start, regexp=True, stopindex=tk.END)
        if not next_word_end:
            next_word_end = tk.END  # If no non-word character found, go to the end of the text

        # Select this next word
        self.text_edits[2].tag_add(tk.SEL, next_word_start, next_word_end)
        
        # Move cursor to the end of the selection
        self.text_edits[2].mark_set(tk.INSERT, next_word_end)
        self.text_edits[2].see(tk.INSERT)

        # Ensure the self.text_edits[2] widget retains focus
        self.text_edits[2].focus_set()

        return "break"




    def F1_process_and_select_word(event=None):
        mode = ""
        sheet_name = self.first_column_유형text.get()
        지문번호 = self.text_edits[0].get()
        self.macro = True

        실모유형 = ""
        실모난이도 = ""
        if "실전모의고사" in sheet_name:
            실모유형 = self.실모_유형_menu.get()
            실모난이도 = self.실모_난이도_menu.get()
            print(f"실모유형: {실모유형}")

        if sheet_name == "어법1단계_Normal":
            mode = '어법'
        elif "어휘1단계" in sheet_name:
            mode = '어휘'
        elif "어휘2단계" in sheet_name:  # Added this condition
            mode = '어휘2'
        elif "어법2단계" in sheet_name or "어법2단계" in 지문번호 or "어법2단계" in 실모유형:
            mode = '어법2단계'
        elif "어휘3단계" in sheet_name:  
            mode = '어휘3'

        if mode in ['어법', '어휘', '어휘2']:  # Added '어휘2' to this condition
            # Store the current cursor position for 어휘2 mode
            initial_cursor_pos = self.text_edits[2].index(tk.INSERT)
            
            # Check if there's already a selection
            try:
                selection_start = self.text_edits[2].index(tk.SEL_FIRST)
                selection_end = self.text_edits[2].index(tk.SEL_LAST)
                selected_text = self.text_edits[2].get(selection_start, selection_end)
                has_selection = True
            except tk.TclError:
                # No selection exists
                has_selection = False
                selected_text = ""
            
            if has_selection:
                # Use the pre-selected text
                cursor_pos = selection_start
                # Delete the selected text
                self.text_edits[2].delete(selection_start, selection_end)
                # Move cursor to the start position
                self.text_edits[2].mark_set(tk.INSERT, cursor_pos)
                
                # For 어휘2 mode, we need to add a number
                if mode == '어휘2':
                    # Don't add number yet, we'll determine position first
                    pass
                
                # Insert "❮ " (and "■ " if in 어휘 or 어휘2 mode)
                if mode in ['어휘', '어휘2']:
                    self.text_edits[2].insert(tk.INSERT, "❮ ■ ")
                else:
                    self.text_edits[2].insert(tk.INSERT, "❮ ")
                
                # Insert the selected text
                self.text_edits[2].insert(tk.INSERT, selected_text)
                
                # Get current position after inserting selected text
                current_pos = self.text_edits[2].index(tk.INSERT)
                
                # Insert " / " (and "□ " if in 어휘 or 어휘2 mode), then the selected text again
                if mode in ['어휘', '어휘2']:
                    self.text_edits[2].insert(tk.INSERT, " / □ ")
                else:
                    self.text_edits[2].insert(tk.INSERT, " / ")
                
                # Mark the position before inserting the duplicated text
                dup_start = self.text_edits[2].index(tk.INSERT)
                self.text_edits[2].insert(tk.INSERT, selected_text)
                dup_end = self.text_edits[2].index(tk.INSERT)
                
                # Insert " ❯"
                self.text_edits[2].insert(tk.INSERT, " ❯")
                
                # Select the duplicated text
                self.text_edits[2].tag_add(tk.SEL, dup_start, dup_end)
                self.text_edits[2].mark_set(tk.INSERT, dup_end)
                self.text_edits[2].see(tk.INSERT)
                self.text_edits[2].focus_set()
                
            else:
                # Get the current position of the cursor
                cursor_pos = self.text_edits[2].index(tk.INSERT)
                
                # First, check if we're in the right part of a hyphenated word
                # Look backwards for a hyphen
                current_pos = cursor_pos
                found_hyphen = False
                while True:
                    prev_pos = self.text_edits[2].index(f"{current_pos} -1c")
                    if prev_pos == current_pos:  # Reached the beginning of text
                        break
                    prev_char = self.text_edits[2].get(prev_pos, current_pos)
                    if prev_char.isspace():  # Reached a space
                        break
                    if prev_char == '-':
                        found_hyphen = True
                        # Continue looking backwards to find the start of the word
                        current_pos = prev_pos
                        while True:
                            prev_pos = self.text_edits[2].index(f"{current_pos} -1c")
                            if prev_pos == current_pos:  # Reached the beginning of text
                                break
                            prev_char = self.text_edits[2].get(prev_pos, current_pos)
                            if prev_char.isspace():  # Found the start of the word
                                cursor_pos = current_pos
                                break
                            current_pos = prev_pos
                        break
                    current_pos = prev_pos

                if not found_hyphen:
                    # Original logic for finding the start of the word
                    previous_char = self.text_edits[2].get(f"{cursor_pos} -1c", cursor_pos)
                    while previous_char and not previous_char.isspace() and previous_char not in ("'", '"', '-'):
                        cursor_pos = self.text_edits[2].index(f"{cursor_pos} -1c")
                        previous_char = self.text_edits[2].get(f"{cursor_pos} -1c", cursor_pos)
                
                # Move the cursor to the start of the word
                self.text_edits[2].mark_set(tk.INSERT, cursor_pos)

                # For 어휘2 mode, don't add number yet
                if mode == '어휘2':
                    pass

                # Insert "❮ " and "■ " if in 어휘 or 어휘2 mode
                if mode in ['어휘', '어휘2']:
                    self.text_edits[2].insert(tk.INSERT, "❮ ■ ")
                else:
                    self.text_edits[2].insert(tk.INSERT, "❮ ")

                # Get the current position of the cursor after insertion
                current_pos = self.text_edits[2].index(tk.INSERT)

                # Use regular expression to find the end of the next word, including apostrophes and hyphens
                text = self.text_edits[2].get(current_pos, tk.END)
                match = re.search(r"\b([\w\"'-]+)\b", text)
                if match:
                    selected_word = match.group(1)
                    word_length = len(selected_word)
                    # Calculate word_end using Tkinter's index method
                    word_end = self.text_edits[2].index(f"{current_pos} + {word_length}c")
                else:
                    # If no word is found, use the current position as word end
                    selected_word = ''
                    word_end = current_pos

                # Move cursor to the end of the selected word
                self.text_edits[2].mark_set(tk.INSERT, word_end)
                
                # Insert " / " (and "□ " if in 어휘 or 어휘2 mode), then the selected word
                if mode in ['어휘', '어휘2']:
                    self.text_edits[2].insert(tk.INSERT, " / □ ")
                else:
                    self.text_edits[2].insert(tk.INSERT, " / ")
                self.text_edits[2].insert(tk.INSERT, selected_word)

                # Insert " ❯"
                self.text_edits[2].insert(tk.INSERT, " ❯")
                self.text_edits[2].focus_set()

                # Position the cursor correctly to select the entire duplicated word
                # Calculate the position to set the cursor just after " / " or " / □ "
                if mode in ['어휘', '어휘2']:
                    adjusted_position_for_selection = f"{word_end} +4c"
                else:
                    adjusted_position_for_selection = f"{word_end} +2c"

                self.text_edits[2].mark_set(tk.INSERT, adjusted_position_for_selection)

                # Explicitly call select_next_word to select the entire next word
                select_next_word()

                # Adjust the selection to exclude the '□' symbol if in 어휘 or 어휘2 mode
                if mode in ['어휘', '어휘2']:
                    try:
                        selection_start = self.text_edits[2].index(tk.SEL_FIRST)
                        selection_end = self.text_edits[2].index(tk.SEL_LAST)
                        if self.text_edits[2].get(f"{selection_start}-1c") == "□":
                            self.text_edits[2].tag_remove(tk.SEL, f"{selection_start}-1c", selection_start)
                    except tk.TclError:
                        # No selection exists
                        pass

            # Special handling for 어휘2 mode - add to Original Words and renumber
            if mode == '어휘2':
                # Store the word we're working with for later selection
                word_to_select = selected_text if has_selection else selected_word
                
                # Get the entire text content
                text_content = self.text_edits[2].get("1.0", tk.END)
                
                # Find where we just inserted the new bracket
                # Count how many bracketed items come before our insertion point
                text_before_cursor = self.text_edits[2].get("1.0", cursor_pos)
                brackets_before = len(re.findall(r'❮[^❯]+❯', text_before_cursor))
                
                # The position for the new word is brackets_before + 1
                new_word_position = brackets_before + 1
                
                # Find the "Original Words:" section
                original_words_match = re.search(r'Original Words:\s*(.*?)(?:\n\n|\Z)', text_content, re.DOTALL)
                if original_words_match:
                    original_words_section = original_words_match.group(1).strip()
                    # Parse existing original words (handle multi-word entries)
                    original_words_list = re.findall(r'\((\d+)\)\s*([^(]+?)(?=\s*\(\d+\)|\s*$)', original_words_section)
                    
                    # Convert to a list of just words
                    words_list = [word.strip() for _, word in original_words_list]
                    
                    # Insert the new word at the correct position
                    words_list.insert(new_word_position - 1, word_to_select)  # -1 because list is 0-indexed
                    
                    # Renumber all items in the text
                    # First, remove all existing numbers
                    text_content = re.sub(r'\(\d+\)(❮[^❯]+❯)', r'\1', text_content)
                    
                    # Then add new numbers
                    def renumber_match(match):
                        nonlocal counter
                        result = f"({counter}){match.group(0)}"
                        counter += 1
                        return result
                    
                    counter = 1
                    text_content = re.sub(r'❮[^❯]+❯', renumber_match, text_content)
                    
                    # Create the new original words section with correct numbering
                    new_original_words = []
                    for i, word in enumerate(words_list, 1):
                        new_original_words.append(f"({i}) {word}")
                    
                    # Replace the original words section
                    new_original_words_text = "Original Words:\n" + " ".join(new_original_words)
                    text_content = re.sub(r'Original Words:\s*.*?(?:\n\n|\Z)', 
                                        new_original_words_text + "\n", 
                                        text_content, 
                                        flags=re.DOTALL)
                    
                    # Update the text widget
                    self.text_edits[2].delete("1.0", tk.END)
                    self.text_edits[2].insert("1.0", text_content.rstrip())
                    
                    # Now find and select the right-side word in the newly numbered bracket
                    # Updated pattern to handle both single words and multiple words
                    pattern = rf'\({new_word_position}\)❮[^/]+/\s*□\s*([^❯]+?)\s*❯'
                    match = re.search(pattern, text_content)
                    if match:
                        # Get the position of the right-side word(s)
                        start_pos = match.start(1)
                        end_pos = match.end(1)
                        
                        # Convert string position to tkinter index
                        text_up_to_word = text_content[:start_pos]
                        line_num = text_up_to_word.count('\n') + 1
                        col_num = len(text_up_to_word.split('\n')[-1])
                        
                        selection_start = f"{line_num}.{col_num}"
                        selection_end = f"{line_num}.{col_num + (end_pos - start_pos)}"
                        
                        # Select the right-side word(s)
                        self.text_edits[2].tag_add(tk.SEL, selection_start, selection_end)
                        self.text_edits[2].mark_set(tk.INSERT, selection_end)
                        self.text_edits[2].see(selection_start)
                        self.text_edits[2].focus_set()
                        
                else:
                    # If Original Words section doesn't exist, create it
                    self.text_edits[2].insert(tk.END, f"\nOriginal Words:\n(1) {word_to_select}")


                    
            return "break"

        

            
        elif mode in ['어법2단계', '어휘3']:

            """
            This function performs the following steps:
            1. Copies the selected phrase to clipboard_A.
            2. Encloses the selected phrase within [ ].
            3. Reselects the phrase inside the brackets.
            """
            text_widget = self.text_edits[2]  # CTkTextbox containing the main text

            try:
                # 1. Get the selected text
                selected_text = text_widget.get("sel.first", "sel.last")
            except tk.TclError:
                # No selection; attempt to select the word under the cursor
                current_index = text_widget.index("insert")
                # Get the start and end indices of the word under the cursor
                word_start = text_widget.index(f"{current_index} wordstart")
                word_end = text_widget.index(f"{current_index} wordend")
                selected_text = text_widget.get(word_start, word_end).strip()

                if selected_text:
                    # Select the word under the cursor
                    text_widget.tag_add("sel", word_start, word_end)
                else:
                    # If no word is found (e.g., cursor is on whitespace), show a warning and exit
                    messagebox.showwarning("Warning", "먼저 특정 어구를 선택하거나, 특정 단어에 커서를 위치시키세요.")  # "Please select a specific phrase first."
                    self.text_edits[2].focus_set()
                    return

            # 2. Copy selected text to clipboard_A
            self.clipboard_A = selected_text

            # 3. Enclose the selected text within [ ]
            text_widget.delete("sel.first", "sel.last")
            text_widget.insert("insert", f"[{selected_text}]")

            # 4. Reselect the text inside the brackets
            # Calculate the new selection indices
            current_index = text_widget.index("insert")
            # Move cursor back to the start of the inserted text
            start_index = f"{current_index} - {len(selected_text) + 1}c"  # +1 for the opening bracket
            end_index = f"{start_index} + {len(selected_text)}c"
            text_widget.tag_add("sel", start_index, end_index)
            text_widget.mark_set("insert", end_index)
            text_widget.focus_set()

        else:
            return "break"





    def F2_process_and_select_two_words(event=None):
        mode = ""
        sheet_name = self.first_column_유형text.get()
        지문번호 = self.text_edits[0].get()

        실모유형 = ""
        실모난이도 = ""
        if "실전모의고사" in sheet_name:
            실모유형 = self.실모_유형_menu.get()
            실모난이도 = self.실모_난이도_menu.get()
            print(f"실모유형: {실모유형}")

        if sheet_name == "어법1단계_Normal":
            mode = '어법1단계'
        elif "어휘1단계" in sheet_name:
            mode = '어휘1단계'
        elif "어휘2단계" in sheet_name:  # Added this condition
            mode = '어휘2'
        elif "어법2단계" in sheet_name or "어법2단계" in 지문번호 or "어법2단계" in 실모유형:
            mode = '어법2단계'
        elif "어휘3단계" in sheet_name:  
            mode = '어휘3'

        print(f"Current mode: {mode}")

        if mode in ['어법1단계', '어휘1단계', '어휘2']:  # Added '어휘2' to this condition
            self.macro = True
            cursor_pos = self.text_edits[2].index(tk.INSERT)

            # Move cursor to start of first word
            while True:
                previous_char = self.text_edits[2].get(f"{cursor_pos} -1c", cursor_pos)
                if not previous_char or previous_char.isspace():
                    break
                if previous_char == "'":
                    # Include apostrophe as part of the word
                    break
                cursor_pos = self.text_edits[2].index(f"{cursor_pos} -1c")

            # Move the cursor to the start of the word
            self.text_edits[2].mark_set(tk.INSERT, cursor_pos)

            # Insert "❮ " and "■ " if in '어휘' mode
            if mode in ['어휘1단계', '어휘2']:
                self.text_edits[2].insert(tk.INSERT, "❮ ■ ")
            else:
                self.text_edits[2].insert(tk.INSERT, "❮ ")

            # Get the current position after insertion
            current_pos = self.text_edits[2].index(tk.INSERT)

            # Use regular expression to find the next two words, including apostrophes
            text = self.text_edits[2].get(current_pos, tk.END)
            # Pattern explanation:
            # \b[\w']+\b matches a word (including apostrophes)
            # \s+ matches whitespace between words
            # \b[\w']+\b matches the second word
            match = re.match(r"\s*([\w']+)\s+([\w']+)", text)

            if match:
                selected_words = ' '.join(match.groups())
                # Calculate the end index of the second word
                word1 = match.group(1)
                word2 = match.group(2)
                # Find the position after the second word
                word_end_index = match.end(2)
                word_end = self.text_edits[2].index(f"{current_pos} + {word_end_index}c")
            else:
                # If not two words found, try to find one word
                match = re.match(r"\s*([\w']+)", text)
                if match:
                    selected_words = match.group(1)
                    word_end_index = match.end(1)
                    word_end = self.text_edits[2].index(f"{current_pos} + {word_end_index}c")
                else:
                    # If no words found, use current position
                    selected_words = ''
                    word_end = current_pos

            # Move cursor to the end of the selected words
            self.text_edits[2].mark_set(tk.INSERT, word_end)

            # Insert " / " (and "□ " if in '어휘' mode), then the selected words
            if mode in ['어휘1단계', '어휘2']:
                self.text_edits[2].insert(tk.INSERT, " / □ ")
            else:
                self.text_edits[2].insert(tk.INSERT, " / ")
            self.text_edits[2].insert(tk.INSERT, selected_words)

            # Insert " ❯"
            self.text_edits[2].insert(tk.INSERT, " ❯")
            self.text_edits[2].focus_set()

            # Position the cursor correctly to select the duplicated words
            # Calculate the position after " / " or " / □ "
            if mode in ['어휘1단계', '어휘2']:
                adjusted_position_for_selection = f"{word_end} +4c"
            else:
                adjusted_position_for_selection = f"{word_end} +2c"

            self.text_edits[2].mark_set(tk.INSERT, adjusted_position_for_selection)

            # Explicitly call select_next_two_words to select the two words
            select_next_two_words()

            # Adjust the selection to exclude the '□' symbol if in '어휘' mode
            if mode in ['어휘1단계', '어휘2']:
                selection_start = self.text_edits[2].index(tk.SEL_FIRST)
                selection_end = self.text_edits[2].index(tk.SEL_LAST)
                # Check if there's a '□' before the selection and exclude it
                preceding_char = self.text_edits[2].get(f"{selection_start}-1c", selection_start)
                if preceding_char == "□":
                    self.text_edits[2].tag_remove(tk.SEL, f"{selection_start}-1c", selection_start)

            # Special handling for 어휘2 mode - add to Original Words and renumber
            if mode == '어휘2':
                # Store the words we're working with for later selection
                words_to_select = selected_words
                
                # Get the entire text content
                text_content = self.text_edits[2].get("1.0", tk.END)
                
                # Find where we just inserted the new bracket
                # Count how many bracketed items come before our insertion point
                text_before_cursor = self.text_edits[2].get("1.0", cursor_pos)
                brackets_before = len(re.findall(r'❮[^❯]+❯', text_before_cursor))
                
                # The position for the new words is brackets_before + 1
                new_word_position = brackets_before + 1
                
                # Find the "Original Words:" section
                original_words_match = re.search(r'Original Words:\s*(.*?)(?:\n\n|\Z)', text_content, re.DOTALL)
                if original_words_match:
                    original_words_section = original_words_match.group(1).strip()
                    # Parse existing original words (note: some entries might be multi-word)
                    original_words_list = re.findall(r'\((\d+)\)\s*([^(]+?)(?=\s*\(\d+\)|\s*$)', original_words_section)
                    
                    # Convert to a list of just words/phrases
                    words_list = [word.strip() for _, word in original_words_list]
                    
                    # Insert the new words at the correct position
                    words_list.insert(new_word_position - 1, words_to_select)  # -1 because list is 0-indexed
                    
                    # Renumber all items in the text
                    # First, remove all existing numbers
                    text_content = re.sub(r'\(\d+\)(❮[^❯]+❯)', r'\1', text_content)
                    
                    # Then add new numbers
                    def renumber_match(match):
                        nonlocal counter
                        result = f"({counter}){match.group(0)}"
                        counter += 1
                        return result
                    
                    counter = 1
                    text_content = re.sub(r'❮[^❯]+❯', renumber_match, text_content)
                    
                    # Create the new original words section with correct numbering
                    new_original_words = []
                    for i, word in enumerate(words_list, 1):
                        new_original_words.append(f"({i}) {word}")
                    
                    # Replace the original words section
                    new_original_words_text = "Original Words:\n" + " ".join(new_original_words)
                    text_content = re.sub(r'Original Words:\s*.*?(?:\n\n|\Z)', 
                                        new_original_words_text + "\n", 
                                        text_content, 
                                        flags=re.DOTALL)
                    
                    # Update the text widget
                    self.text_edits[2].delete("1.0", tk.END)
                    self.text_edits[2].insert("1.0", text_content.rstrip())
                    
                    # Now find and select the right-side words in the newly numbered bracket
                    # Find all brackets with their positions
                    pattern = rf'\({new_word_position}\)❮[^/]+/\s*□\s*([^❯]+?)\s*❯'
                    match = re.search(pattern, text_content)
                    if match:
                        # Get the position of the right-side words
                        start_pos = match.start(1)
                        end_pos = match.end(1)
                        
                        # Convert string position to tkinter index
                        text_up_to_word = text_content[:start_pos]
                        line_num = text_up_to_word.count('\n') + 1
                        col_num = len(text_up_to_word.split('\n')[-1])
                        
                        selection_start = f"{line_num}.{col_num}"
                        selection_end = f"{line_num}.{col_num + (end_pos - start_pos)}"
                        
                        # Select the right-side words
                        self.text_edits[2].tag_add(tk.SEL, selection_start, selection_end)
                        self.text_edits[2].mark_set(tk.INSERT, selection_end)
                        self.text_edits[2].see(selection_start)
                        self.text_edits[2].focus_set()
                        
                else:
                    # If Original Words section doesn't exist, create it
                    self.text_edits[2].insert(tk.END, f"\nOriginal Words:\n(1) {words_to_select}")

            return "break"




        elif mode in ['어법2단계', '어휘3']:  # Change this line
            """
            This function performs the following steps:
            1. Checks if clipboard_A is empty; if so, shows a message.
            2. Finds and selects the text within [ ] based on cursor position and copies it to clipboard_B.
            3. Determines the smallest unused circled number.
            4. Prefixes the selected text with the chosen circled number.
            5. Updates self.text_edits[3] and self.text_edits[4] with the new circled number and correction.
            6. Sorts the entries in self.text_edits[3] and self.text_edits[4] in incremental order.
            7. Renumbers subsequent circled numbers if necessary.
            """
            if not self.clipboard_A:
                messagebox.showinfo("Info", "특정 어구를 선택한 뒤 F1을 먼저 누르세요.")  # "Please select a specific phrase and press F1 first."
                return

            text_widget = self.text_edits[2]       # CTkTextbox containing the main text
            numbers_widget = self.text_edits[3]    # CTkTextbox containing circled numbers
            corrections_widget = self.text_edits[4]  # CTkTextbox containing corrections

            # 1. Get current cursor position
            cursor_index = text_widget.index("insert")
            cursor_pos = self.get_char_index(text_widget, cursor_index)

            # Capture the current scroll position
            y_view = text_widget.yview()
            x_view = text_widget.xview()

            # 2. Find the '[' to the left and ']' to the right of the cursor
            text_content = text_widget.get("1.0", "end-1c")

            # Find the position of the nearest '[' to the left of the cursor
            left_bracket_pos = text_content.rfind('[', 0, cursor_pos)
            if left_bracket_pos == -1:
                messagebox.showinfo("Info", "커서 위치에 '[' 가 없습니다.")  # "No '[' found at the cursor position."
                return

            # Find the position of the nearest ']' to the right of the cursor
            right_bracket_pos = text_content.find(']', cursor_pos)
            if right_bracket_pos == -1:
                messagebox.showinfo("Info", "커서 위치에 ']' 가 없습니다.")  # "No ']' found at the cursor position."
                return

            # Extract the text within the brackets
            selected_phrase = text_content[left_bracket_pos + 1 : right_bracket_pos]
            self.clipboard_B = selected_phrase

            # 3. Determine the smallest unused circled number
            existing_numbers = self.fetch_circled_numbers(text_content)

            # Initialize selected_number
            selected_number = None

            # *** Modified Code Starts Here ***
            # Fetch the current position of the cursor
            # Find the circled number before and after the cursor
            before_num = None
            after_num = None

            # Search backwards for the circled number before the cursor
            for i in range(cursor_pos - 1, -1, -1):
                char = text_content[i]
                if char in self.circled_numbers:
                    before_num = char
                    break

            # Search forwards for the circled number after the cursor
            for i in range(cursor_pos, len(text_content)):
                char = text_content[i]
                if char in self.circled_numbers:
                    after_num = char
                    break

            # Define helper functions for circled number mapping
            def circled_num_to_int(circled_num_char):
                """
                Converts a circled number character to its integer equivalent.
                Returns None if the character is not a recognized circled number.
                """
                try:
                    return self.circled_numbers.index(circled_num_char) + 1
                except ValueError:
                    return None

            def int_to_circled_num(n):
                """
                Converts an integer to its corresponding circled number character.
                Returns None if the number is out of range.
                """
                if 1 <= n <= len(self.circled_numbers):
                    return self.circled_numbers[n - 1]
                return None

            def renumber_circled_numbers(text, threshold, shift=1):
                """
                Renumbers circled numbers in the provided text by incrementing
                numbers greater than or equal to the threshold by 'shift'.
                """
                
                pattern = r'[①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳㉑㉒㉓㉔㉕㉖㉗㉘㉙㉚㉛㉜㉝㉞㉟㊱㊲㊳㊴㊵]'

                def replace(match):
                    char = match.group()
                    num = circled_num_to_int(char)
                    if num and num >= threshold:
                        new_num = num + shift
                        new_char = int_to_circled_num(new_num)
                        return new_char if new_char else char
                    return char

                return re.sub(pattern, replace, text)

            # Case 1: Both before_num and after_num exist and are consecutive
            if before_num and after_num:
                before_index = self.circled_numbers.index(before_num) + 1  # Assuming list starts at '①' as 1
                after_index = self.circled_numbers.index(after_num) + 1

                # Check if they are consecutive
                if after_index == before_index + 1:
                    # Proceed to renumber: increment all numbers >= after_index by 1
                    threshold = after_index
                    updated_text = renumber_circled_numbers(text_content, threshold, shift=1)
                    text_widget.delete("1.0", "end")
                    text_widget.insert("1.0", updated_text)
                    print("Renumbered circled numbers in text_widget.")

                    # Renumber circled numbers in numbers_widget
                    numbers_text = numbers_widget.get("1.0", "end-1c")
                    updated_numbers = renumber_circled_numbers(numbers_text, threshold, shift=1)
                    numbers_widget.delete("1.0", "end")
                    numbers_widget.insert("1.0", updated_numbers)
                    print("Renumbered circled numbers in numbers_widget.")

                    # Renumber circled numbers in corrections_widget
                    corrections_text = corrections_widget.get("1.0", "end-1c")
                    updated_corrections = renumber_circled_numbers(corrections_text, threshold, shift=1)
                    corrections_widget.delete("1.0", "end")
                    corrections_widget.insert("1.0", updated_corrections)
                    print("Renumbered circled numbers in corrections_widget.")

                    # Set selected_number to after_index (now the new number is after_index)
                    selected_number = int_to_circled_num(after_index)
                    if not selected_number:
                        messagebox.showerror("Error", f"Cannot convert index {after_index} to circled number.")
                        return

            # Case 2: No before_num, but after_num is '①'
            elif not before_num and after_num == '①':
                # Insert '①' and shift existing numbers >=1 by 1
                threshold = 1
                updated_text = renumber_circled_numbers(text_content, threshold, shift=1)
                text_widget.delete("1.0", "end")
                text_widget.insert("1.0", updated_text)
                print("Renumbered circled numbers in text_widget (Case 2).")

                # Renumber circled numbers in numbers_widget
                numbers_text = numbers_widget.get("1.0", "end-1c")
                updated_numbers = renumber_circled_numbers(numbers_text, threshold, shift=1)
                numbers_widget.delete("1.0", "end")
                numbers_widget.insert("1.0", updated_numbers)
                print("Renumbered circled numbers in numbers_widget (Case 2).")

                # Renumber circled numbers in corrections_widget
                corrections_text = corrections_widget.get("1.0", "end-1c")
                updated_corrections = renumber_circled_numbers(corrections_text, threshold, shift=1)
                corrections_widget.delete("1.0", "end")
                corrections_widget.insert("1.0", updated_corrections)
                print("Renumbered circled numbers in corrections_widget (Case 2).")

                # Set selected_number to '①'
                selected_number = '①'






            # If selected_number is still None, determine the smallest unused circled number
            if not selected_number:
                selected_number = self.find_smallest_unused_number(existing_numbers)
                if not selected_number:
                    messagebox.showerror("Error", "더 이상 사용할 수 있는 번호가 없습니다.")  # "No more numbers available."
                    return


            # 4. Prefix the selected text with the chosen circled number
            # Replace [selected_phrase] with circled_number[selected_phrase]
            new_text = f"{selected_number}[{selected_phrase}]"
            text_widget.delete(f"1.0 + {left_bracket_pos}c", f"1.0 + {right_bracket_pos +1}c")  # +1 to remove ']'
            text_widget.insert(f"1.0 + {left_bracket_pos}c", new_text)



            # Now handle the mode-specific logic
            if mode == '어법2단계':
                if self.clipboard_A != self.clipboard_B:
                    # Update self.text_edits[3] with the new circled number and sort
                    numbers_text = numbers_widget.get("1.0", "end-1c")
                    updated_numbers = numbers_text + selected_number
                    sorted_numbers = ''.join(sorted(
                        updated_numbers,
                        key=lambda x: self.circled_numbers.index(x) if x in self.circled_numbers else len(self.circled_numbers)
                    ))
                    numbers_widget.delete("1.0", "end")
                    numbers_widget.insert("1.0", sorted_numbers)


                    if "어법2단계" in sheet_name or "어법2단계" in 지문번호 or "어법2단계" in 실모유형:

                        #이미 정답에 번호가 없는경우 번호 추가하는 로직 

                        corrections_text = corrections_widget.get("1.0", "end-1c")
                        
                        # Check if corrections_text doesn't start with a circled number
                        if corrections_text and not any(corrections_text.startswith(num) for num in self.circled_numbers):
                            # Extract the phrase before ' > '
                            parts = corrections_text.split(' > ')
                            if len(parts) >= 2:
                                phrase_to_find = parts[0].strip()
                                
                                # Search for this phrase in brackets in self.text_edits[2]
                                text_edits_2_content = self.text_edits[2].get("1.0", "end-1c")
                                
                                # Find the circled number before the bracket containing this phrase
                                
                                pattern = r'([①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳])\[([^\]]*)\]'
                                matches = re.findall(pattern, text_edits_2_content)
                                
                                found_number = None
                                for number, bracket_content in matches:
                                    if phrase_to_find in bracket_content:
                                        found_number = number
                                        break
                                
                                # If we found the number, prepend it to corrections_text
                                if found_number:
                                    corrections_text = f"{found_number} {corrections_text}"

                        new_correction_line = f"{selected_number} {self.clipboard_B} > {self.clipboard_A}"
                        corrections_lines = corrections_text.splitlines()
                        corrections_lines.append(new_correction_line)
                        sorted_corrections = sorted(
                            corrections_lines,
                            key=lambda line: self.circled_numbers.index(line[0]) if line and line[0] in self.circled_numbers else len(self.circled_numbers)
                        )
                        corrections_widget.delete("1.0", "end")
                        corrections_widget.insert("1.0", "\n".join(sorted_corrections))


                    #elif sheet_name == "어법2단계_Normal":
                    #    new_correction_line = f"{self.clipboard_B} > {self.clipboard_A}"
                    #    corrections_text = corrections_widget.get("1.0", "end-1c")
                    #    corrections_text = corrections_text + "\n" + new_correction_line
                    #    corrections_widget.delete("1.0", "end")
                    #    corrections_widget.insert("1.0", corrections_text.strip())



            elif mode == '어휘3':
                # Special handling for 어휘3 mode
                
                # Update self.text_edits[4] based on conditions
                corrections_text = corrections_widget.get("1.0", "end-1c")
                lines = corrections_text.split('\n')
                
                # Find where to insert the new content
                explanation_index = -1
                original_words_index = -1
                
                for i, line in enumerate(lines):
                    if line.strip() == "Explanation:":
                        explanation_index = i
                    elif line.strip() == "Original Words:":
                        original_words_index = i
                
                # Initialize collections for different sections
                correction_lines = []
                explanation_lines = []
                original_word_lines = []
                
                # Collect existing lines in each section
                for i in range(len(lines)):
                    if i < explanation_index and lines[i].strip():
                        correction_lines.append(lines[i])
                    elif explanation_index < i < original_words_index and lines[i].strip():
                        explanation_lines.append(lines[i])
                    elif i > original_words_index and lines[i].strip():
                        original_word_lines.append(lines[i])
                
                # Handle different cases
                if self.clipboard_A != self.clipboard_B:
                    if not self.어휘2단계동의어변경flag:
                        # Case 1: Different words, flag is False
                        # Update self.text_edits[3] with the new circled number and sort
                        numbers_text = numbers_widget.get("1.0", "end-1c")
                        updated_numbers = numbers_text + selected_number
                        sorted_numbers = ''.join(sorted(
                            updated_numbers,
                            key=lambda x: self.circled_numbers.index(x) if x in self.circled_numbers else len(self.circled_numbers)
                        ))
                        numbers_widget.delete("1.0", "end")
                        numbers_widget.insert("1.0", sorted_numbers)
                        
                        # Replace [ ] with < > in text_widget
                        # The text currently has {selected_number}[selected_phrase]
                        # We need to change it to {selected_number}<selected_phrase>
                        current_text = text_widget.get("1.0", "end-1c")
                        pattern_to_find = f"{selected_number}[{selected_phrase}]"
                        pattern_to_replace = f"{selected_number}<{selected_phrase}>"
                        
                        if pattern_to_find in current_text:
                            new_text_content = current_text.replace(pattern_to_find, pattern_to_replace, 1)  # Replace only first occurrence
                            text_widget.delete("1.0", "end")
                            text_widget.insert("1.0", new_text_content)
                        
                        # Add correction above Explanation:
                        new_correction_line = f"{selected_number} {self.clipboard_B} → {self.clipboard_A}"
                        correction_lines.append(new_correction_line)
                        
                        # Add explanation between Explanation: and Original Words:
                        new_explanation_line = f"{selected_number} 문맥상 {self.clipboard_B}를 {self.clipboard_A}로 바꾸어야 한다."
                        explanation_lines.append(new_explanation_line)
                        
                    else:
                        # Case 2: Different words, flag is True
                        # Do NOT update self.text_edits[3]
                        
                        # Add to Original Words section
                        new_original_line = f"{selected_number} {self.clipboard_B} = {self.clipboard_A}"
                        original_word_lines.append(new_original_line)
                        # Reset the flag after use
                        self.어휘2단계동의어변경flag = False
                        
                else:
                    # Case 3: Same words (regardless of flag)
                    # Do NOT update self.text_edits[3]
                    
                    # Add to Original Words section
                    new_line = f"{selected_number} 원문 단어 그대로"
                    original_word_lines.append(new_line)
                
                # Sort each section
                sorted_corrections = sorted(
                    correction_lines,
                    key=lambda line: self.circled_numbers.index(line[0]) if line and line[0] in self.circled_numbers else len(self.circled_numbers)
                )
                
                sorted_explanations = sorted(
                    explanation_lines,
                    key=lambda line: self.circled_numbers.index(line[0]) if line and line[0] in self.circled_numbers else len(self.circled_numbers)
                )
                
                sorted_original_words = sorted(
                    original_word_lines,
                    key=lambda line: self.circled_numbers.index(line[0]) if line and line[0] in self.circled_numbers else len(self.circled_numbers)
                )
                
                # Rebuild the complete text
                new_lines = []
                new_lines.extend(sorted_corrections)
                new_lines.append("Explanation:")
                new_lines.extend(sorted_explanations)
                new_lines.append("Original Words:")
                new_lines.extend(sorted_original_words)
                
                corrections_widget.delete("1.0", "end")
                corrections_widget.insert("1.0", '\n'.join(new_lines))



            # Restore the original cursor position and scroll view
            try:
                # Ensure that the cursor index is still valid after modifications
                text_widget.mark_set("insert", cursor_index)
                text_widget.see("insert")  # Scroll to make sure the cursor is visible

                # Restore the scroll position
                text_widget.yview_moveto(y_view[0])
                text_widget.xview_moveto(x_view[0])
            except Exception as e:
                # If the original index is no longer valid, set the cursor to the end
                print(f"Failed to set cursor to original position: {e}. Setting to end.")
                text_widget.mark_set("insert", "end-1c")
                text_widget.see("end-1c")


            return "break"  # Prevent further propagation of the event







    def select_next_two_words(event=None):
        mode = ""
        sheet_name = self.first_column_유형text.get()

        if sheet_name == "어법1단계_Normal":
            mode = '어법'
        elif "어휘1단계" in sheet_name:
            mode = '어휘'

        # Remove any existing selection
        self.text_edits[2].tag_remove(tk.SEL, "1.0", tk.END)
        cursor_pos = self.text_edits[2].index(tk.INSERT)

        # Use regex to find the next two words, including apostrophes
        # Pattern explanation:
        # [\w']+ matches a word (including apostrophes)
        # \s+ matches whitespace
        # [\w']+ matches the second word
        text_after_cursor = self.text_edits[2].get(cursor_pos, tk.END)
        match = re.search(r"[\w']+\s+[\w']+", text_after_cursor)

        if match:
            words = match.group()
            # Calculate the start and end indices relative to the cursor position
            start_offset = match.start()
            end_offset = match.end()

            # Split cursor_pos into line and column
            line, column = map(int, cursor_pos.split('.'))
            # Calculate absolute positions
            word_start = f"{line}.{column + start_offset}"
            word_end = f"{line}.{column + end_offset}"

            # Add selection from word_start to word_end
            self.text_edits[2].tag_add(tk.SEL, word_start, word_end)
            self.text_edits[2].mark_set(tk.INSERT, word_end)
            self.text_edits[2].see(tk.INSERT)
            self.text_edits[2].focus_set()

            # Handle '어휘' mode specifics
            if mode == '어휘':
                # Check if there's a '□' before the selection and remove it from selection
                preceding_char = self.text_edits[2].get(f"{word_start}-1c", word_start)
                if preceding_char == "□":
                    self.text_edits[2].tag_remove(tk.SEL, f"{word_start}-1c", word_start)

        else:
            # If not two words found, try to select one word
            match = re.search(r"[\w']+", text_after_cursor)
            if match:
                words = match.group()
                start_offset = match.start()
                end_offset = match.end()

                line, column = map(int, cursor_pos.split('.'))
                word_start = f"{line}.{column + start_offset}"
                word_end = f"{line}.{column + end_offset}"

                self.text_edits[2].tag_add(tk.SEL, word_start, word_end)
                self.text_edits[2].mark_set(tk.INSERT, word_end)
                self.text_edits[2].see(tk.INSERT)
                self.text_edits[2].focus_set()

                # Handle '어휘' mode specifics
                if mode == '어휘':
                    preceding_char = self.text_edits[2].get(f"{word_start}-1c", word_start)
                    if preceding_char == "□":
                        self.text_edits[2].tag_remove(tk.SEL, f"{word_start}-1c", word_start)
            else:
                print("No words found to select.")

        return "break"







    def F3_replace_phrase(event=None):
        print("replace_phrase called")
        mode = ""
        sheet_name = self.first_column_유형text.get()
        지문번호 = self.text_edits[0].get()
        print(f"지문번호: {지문번호}  sheet_name: {sheet_name} ")
        실모유형 = ""
        실모난이도 = ""
        실모유형난이도 = ""
        if "실전모의고사" in sheet_name:
            실모유형 = self.실모_유형_menu.get()
            실모난이도 = self.실모_난이도_menu.get()
            실모유형난이도 = 실모유형 + "_" + 실모난이도
            print(f"실모유형: {실모유형}")
            print(f"실모유형난이도: {실모유형난이도}")



        if sheet_name == "어법1단계_Normal":
            mode = '어법'
        elif "어휘1단계" in sheet_name:
            mode = '어휘'
        elif "어법2단계" in sheet_name or "어법2단계" in 지문번호 or "어법2단계" in 실모유형:
            mode = '어법2'
        elif "어휘2단계" in sheet_name:
            mode = '어휘2'
        elif "어휘3단계" in sheet_name or "어휘3단계" in 지문번호 or "어휘3단계" in 실모유형:
            mode = '어휘3'
        elif "동반의어_Normal" in sheet_name:
            mode = '동반의어'
        elif "한줄영작연습" in sheet_name or "한줄해석연습" in sheet_name:
            mode = '한줄영작연습'



        if mode == '한줄영작연습':
            # Get the current cursor position
            cursor_pos = self.text_edits[2].index("insert")

            # Get the entire content
            full_content = self.text_edits[2].get("1.0", "end-1c")

            # Get the current line
            line_start = self.text_edits[2].index(f"{cursor_pos} linestart")
            line_end = self.text_edits[2].index(f"{cursor_pos} lineend")
            current_line = self.text_edits[2].get(line_start, line_end).strip()

            # Extract line number (e.g., "1.", "2.", "3.")
            line_num_match = re.match(r'^(\d+)\.\s', current_line)

            if not line_num_match:
                messagebox.showwarning("경고", "현재 줄에서 번호를 찾을 수 없습니다")
                popup.focus_set()
                self.text_edits[2].focus_set()
                return "break"

            line_num = line_num_match.group(1)  # e.g., "1", "2", "3"

            # Split content into [해석] and [영어] sections
            sections = full_content.split('[영어]')
            if len(sections) != 2:
                messagebox.showwarning("경고", "[해석]과 [영어] 섹션을 찾을 수 없습니다")
                popup.focus_set()
                self.text_edits[2].focus_set()
                return "break"

            haesuk_section = sections[0]
            yeongeo_section = sections[1]

            # Remove line from [해석] section
            haesuk_lines = haesuk_section.split('\n')
            haesuk_filtered = []
            for line in haesuk_lines:
                if not re.match(f'^{line_num}\\.\\s', line.strip()):
                    haesuk_filtered.append(line)

            # Remove line from [영어] section
            yeongeo_lines = yeongeo_section.split('\n')
            yeongeo_filtered = []
            for line in yeongeo_lines:
                if not re.match(f'^{line_num}\\.\\s', line.strip()):
                    yeongeo_filtered.append(line)

            # Reconstruct the content
            new_content = '\n'.join(haesuk_filtered) + '[영어]' + '\n'.join(yeongeo_filtered)

            # Update the text widget
            self.text_edits[2].delete("1.0", "end")
            self.text_edits[2].insert("1.0", new_content)

            return "break"
        
        
        elif mode == '동반의어':
            # Get the current cursor position
            cursor_pos = self.text_edits[2].index("insert")
            
            # Get the start of the current line
            line_start = self.text_edits[2].index(f"{cursor_pos} linestart")
            
            # Get the start of the next line
            line_end = self.text_edits[2].index(f"{cursor_pos} lineend + 1c")
            
            # Get the current line content
            current_line = self.text_edits[2].get(line_start, line_end).strip()
            
            # Check if it's a header row using regex pattern
            header_pattern1 = r"\|\s*English Word\s*\|\s*Korean Translation\s*\|\s*Synonyms\s*\|\s*Antonyms\s*\|"
            header_pattern2 = r"\|\-+\|\-+\|\-+\|\-+\|"
            
            if re.match(header_pattern1, current_line) or re.match(header_pattern2, current_line):
                messagebox.showwarning("Warning", "헤더는 삭제할 수 없습니다")
                popup.focus_set()
                self.text_edits[2].focus_set()

                return "break"
            
            # Delete the current line if it's not a header
            self.text_edits[2].delete(line_start, line_end)
            
            return "break"

        elif mode == '어법2':
            answer_flag = True
            """
            This function performs the following steps:
            1. Detects the circled number where the cursor is placed within self.text_edits[2].
            2. Removes the detected circled number from self.text_edits[3] if it exists.
            3. Fetches the corresponding correction from self.text_edits[4] if available 
            and replaces the text within the brackets in self.text_edits[2].
            4. Removes the processed correction line from self.text_edits[4] if a correction was made.
            5. Renumbers subsequent circled numbers to fill any gaps created by the removal.
            """
            def get_char_index(text_widget, index):
                """
                Converts a Tkinter text index (e.g., '1.0') to a zero-based character index.
                """
                line, char = map(int, index.split('.'))
                lines = text_widget.get("1.0", f"{line}.0").split('\n')
                total_chars = sum(len(l) + 1 for l in lines[:-1])  # +1 for newline
                total_chars += char
                return total_chars

            # Helper functions for circled number mapping
            def circled_num_to_int(circled_num_char):
                """
                Converts a circled number character to its integer equivalent.
                Returns None if the character is not a recognized circled number.
                """
                circled_numbers = {
                    '①': 1, '②': 2, '③': 3, '④': 4, '⑤': 5,
                    '⑥': 6, '⑦': 7, '⑧': 8, '⑨': 9, '⑩': 10,
                    '⑪': 11, '⑫': 12, '⑬': 13, '⑭': 14, '⑮': 15,
                    '⑯': 16, '⑰': 17, '⑱': 18, '⑲': 19, '⑳': 20
                }
                return circled_numbers.get(circled_num_char, None)

            def int_to_circled_num(n):
                """
                Converts an integer to its corresponding circled number character.
                Returns None if the number is out of range.
                """
                circled_numbers = {
                    1: '①', 2: '②', 3: '③', 4: '④', 5: '⑤',
                    6: '⑥', 7: '⑦', 8: '⑧', 9: '⑨', 10: '⑩',
                    11: '⑪', 12: '⑫', 13: '⑬', 14: '⑭', 15: '⑮',
                    16: '⑯', 17: '⑰', 18: '⑱', 19: '⑲', 20: '⑳'
                }
                return circled_numbers.get(n, None)

            def renumber_circled_numbers(text, removed_num):
                """
                Renumbers circled numbers in the provided text by decrementing
                numbers greater than the removed_num by 1.
                """
                pattern = r'[①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳㉑㉒㉓㉔㉕㉖㉗㉘㉙㉚㉛㉜㉝㉞㉟㊱㊲㊳㊴㊵]'

                def replace(match):
                    char = match.group()
                    num = circled_num_to_int(char)
                    if num and num > removed_num:
                        new_num = num - 1
                        new_char = int_to_circled_num(new_num)
                        return new_char if new_char else char
                    return char

                return re.sub(pattern, replace, text)

            # Assign widgets
            text_widget = self.text_edits[2]         # CTkTextbox containing the main text
            numbers_widget = self.text_edits[3]      # CTkTextbox containing circled numbers
            corrections_widget = self.text_edits[4]  # CTkTextbox containing corrections

            # Step 1: Get the current cursor position in self.text_edits[2]
            cursor_index = text_widget.index("insert")
            cursor_pos = get_char_index(text_widget, cursor_index)

            # Capture the current scroll position
            y_view = text_widget.yview()
            x_view = text_widget.xview()

            # Get the entire text from self.text_edits[2]
            text_content = text_widget.get("1.0", "end-1c")
            print(f"text_content: {text_content}")

            # Define pattern to find circled numbers followed by [text]
            pattern = r'([①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳㉑㉒㉓㉔㉕㉖㉗㉘㉙㉚㉛㉜㉝㉞㉟㊱㊲㊳㊴㊵])\[(.*?)\]'
            matches = list(re.finditer(pattern, text_content))

            # Determine which match contains the cursor
            circled_number = None
            match_obj = None
            for match in matches:
                start, end = match.span()
                if start <= cursor_pos <= end:
                    circled_number = match.group(1)
                    match_obj = match
                    break

            if not circled_number:
                messagebox.showinfo("Info", "Cursor is not inside a bracket with a circled number.")
                popup.focus_set()
                self.text_edits[2].focus_set()
                return  # Cursor not in the target area

            # Step 2: Remove circled_number from self.text_edits[3] if it exists
            numbers_text = numbers_widget.get("1.0", "end-1c")
            if circled_number in numbers_text:
                updated_numbers = numbers_text.replace(circled_number, "")
                numbers_widget.delete("1.0", "end")
                numbers_widget.insert("1.0", updated_numbers)
            else:  # 정답 선지가 아님
                answer_flag = False
                phraseB = None



            # Step 3: Find the correction in self.text_edits[4] if available
            # 어법2단계_Normal and 어법2단계_Hard now use the same logic
            if sheet_name in ["어법2단계_Hard", "어법2단계_Normal"] or "어법2단계" in 지문번호 or "어법2단계" in 실모유형난이도:
                corrections_text = corrections_widget.get("1.0", "end-1c")
                lines = corrections_text.splitlines()
                phraseB = None
                new_corrections = []
                for line in lines:
                    if line.startswith(circled_number):
                        # Assuming format '① open > opens'
                        parts = line.split('>')
                        if len(parts) >= 2:
                            phraseB = parts[1].strip()
                    else:
                        new_corrections.append(line)






            if phraseB:
                # Replace the entire 'circled_number[text]' with 'phraseB' in self.text_edits[2]
                original_text = match_obj.group(0)  # e.g., '③[open]'
                replacement_text = phraseB  # Replace with the correction

                # Perform the replacement
                new_text_content = text_content[:match_obj.start()] + replacement_text + text_content[match_obj.end():]
                print(f"new_text_content: {new_text_content}")

                # Update the text in self.text_edits[2]
                text_widget.delete("1.0", "end")
                text_widget.insert("1.0", new_text_content)

                # Step 4: Update the corrections_widget by removing the processed line
                corrections_widget.delete("1.0", "end")
                corrections_widget.insert("1.0", "\n".join(new_corrections))

                print(f"Replaced {original_text} with {replacement_text}.")
            else:
                # No correction found; remove the circled_number and brackets, keeping the text inside
                original_text = match_obj.group(0)  # e.g., '③[open]'
                inner_text = match_obj.group(2)     # e.g., 'open'
                replacement_text = inner_text        # Replace with the text inside the brackets

                # Perform the replacement
                new_text_content = text_content[:match_obj.start()] + replacement_text + text_content[match_obj.end():]

                # Update the text in self.text_edits[2]
                text_widget.delete("1.0", "end")
                text_widget.insert("1.0", new_text_content)

                print(f"Removed {original_text} and kept inner text '{replacement_text}'.")

            # Renumber subsequent circled numbers in all widgets
            if circled_number:
                removed_num = circled_num_to_int(circled_number)
                if removed_num is not None:
                    # Renumber in text_widget
                    updated_text = renumber_circled_numbers(new_text_content, removed_num)
                    text_widget.delete("1.0", "end")
                    text_widget.insert("1.0", updated_text)
                    print("Renumbered circled numbers in text_widget.")

                    # Renumber in numbers_widget
                    numbers_text = numbers_widget.get("1.0", "end-1c")
                    updated_numbers = renumber_circled_numbers(numbers_text, removed_num)
                    numbers_widget.delete("1.0", "end")
                    numbers_widget.insert("1.0", updated_numbers)
                    print("Renumbered circled numbers in numbers_widget.")

                    # Renumber in corrections_widget
                    corrections_text = corrections_widget.get("1.0", "end-1c")
                    updated_corrections = renumber_circled_numbers(corrections_text, removed_num)
                    corrections_widget.delete("1.0", "end")
                    corrections_widget.insert("1.0", updated_corrections)
                    print("Renumbered circled numbers in corrections_widget.")

            # Restore the original cursor position and scroll view
            try:
                # Ensure that the cursor index is still valid after modifications
                text_widget.mark_set("insert", cursor_index)
                text_widget.see("insert")  # Scroll to make sure the cursor is visible

                # Restore the scroll position
                text_widget.yview_moveto(y_view[0])
                text_widget.xview_moveto(x_view[0])
            except Exception as e:
                # If the original index is no longer valid, set the cursor to the end
                print(f"Failed to set cursor to original position: {e}. Setting to end.")
                text_widget.mark_set("insert", "end-1c")
                text_widget.see("end-1c")

            # Optionally, show a success message
            # messagebox.showinfo("Success", f"Processed {circled_number} successfully.")

            return "break"  # Prevent further propagation of the event



                        

        elif mode == "어법" or mode == "어휘":
            cursor_pos = self.text_edits[2].index(tk.INSERT)
            
            # Find the start of the pattern (❮) by searching backwards from the cursor
            pattern_start = self.text_edits[2].search('❮', cursor_pos, backwards=True, stopindex='1.0')
            
            # Check if '❯' is found before '❮' when searching backwards
            closing_bracket = self.text_edits[2].search('❯', cursor_pos, backwards=True, stopindex='1.0')
            if closing_bracket and (not pattern_start or self.text_edits[2].compare(closing_bracket, '>', pattern_start)):
                #print("Cursor is not within the correct brackets.")
                return "break"

            # Find the end of the pattern (❯) by searching forwards from the cursor
            pattern_end = self.text_edits[2].search('❯', cursor_pos, stopindex=tk.END)
            
            if not pattern_start or not pattern_end:
                #print("Pattern start or end not found.")
                return "break"

            pattern_text = self.text_edits[2].get(pattern_start, f"{pattern_end}+1c")
            slash_pos = pattern_text.find('/')
            
            if slash_pos == -1:
                #print("Slash ('/') not found within the pattern.")
                return "break"

            if mode == '어휘':
                square_pos = pattern_text.find('■')
                if square_pos == -1:
                    #print("'■' not found within the pattern.")
                    return "break"
                phrase_to_keep = pattern_text[square_pos+1:slash_pos].strip()
            else:
                phrase_to_keep = pattern_text[1:slash_pos].strip()

            self.text_edits[2].delete(pattern_start, f"{pattern_end}+1c")
            self.text_edits[2].insert(pattern_start, phrase_to_keep)
            #print(f"Pattern replaced with '{phrase_to_keep}' at {pattern_start}")

            return "break"


        elif mode == '어휘2':
            cursor_pos = self.text_edits[2].index(tk.INSERT)
            
            # Get the entire text
            full_text = self.text_edits[2].get("1.0", "end-1c")
            
            # Find the cursor position as a character index
            cursor_char_index = len(self.text_edits[2].get("1.0", cursor_pos))
            
            # Pattern to find numbered patterns like (1)❮ ■ striking / □ unremarkable ❯
            pattern = r'\((\d+)\)❮[^❯]+❯'
            
            # Find all matches
            matches = list(re.finditer(pattern, full_text))
            
            # Find which pattern contains the cursor
            target_match = None
            target_number = None
            
            for match in matches:
                if match.start() <= cursor_char_index <= match.end():
                    target_match = match
                    target_number = int(match.group(1))
                    break
            
            if not target_match:
                return "break"
            
            # Split text to find Original Words section
            parts = full_text.split('Original Words:\n')
            if len(parts) != 2:
                return "break"
            
            main_text = parts[0]
            original_words_section = parts[1]
            
            # Parse original words
            word_pattern = r'\((\d+)\)\s*([^\(\)]+?)(?=\s*\(\d+\)|$)'
            word_matches = list(re.finditer(word_pattern, original_words_section))
            
            # Find the replacement word
            replacement_word = None
            for word_match in word_matches:
                if int(word_match.group(1)) == target_number:
                    replacement_word = word_match.group(2).strip()
                    break
            
            if not replacement_word:
                return "break"
            
            # Replace the pattern with the word
            new_main_text = main_text[:target_match.start()] + replacement_word + main_text[target_match.end():]
            
            # Remove the used word from original words
            new_word_matches = []
            for word_match in word_matches:
                if int(word_match.group(1)) != target_number:
                    new_word_matches.append(word_match.group(2).strip())
            
            # Renumber patterns in main text
            def renumber_patterns(text, removed_num):
                def replace_num(match):
                    num = int(match.group(1))
                    if num > removed_num:
                        return f"({num - 1}){match.group(2)}"
                    return match.group(0)
                
                return re.sub(r'\((\d+)\)(❮[^❯]+❯)', replace_num, text)
            
            # Apply renumbering
            new_main_text = renumber_patterns(new_main_text, target_number)
            
            # Rebuild original words section with new numbering
            new_original_words = "Original Words:\n"
            for i, word in enumerate(new_word_matches, 1):
                if i > 1:
                    new_original_words += " "
                new_original_words += f"({i}) {word}"
            
            # Update the text widget
            self.text_edits[2].delete("1.0", "end")
            self.text_edits[2].insert("1.0", new_main_text + new_original_words)
            
            # Try to maintain cursor position
            try:
                # Adjust cursor position if it was after the replaced pattern
                if cursor_char_index > target_match.start():
                    adjustment = len(replacement_word) - len(target_match.group(0))
                    new_cursor_index = cursor_char_index + adjustment
                    # Convert back to line.column format
                    lines = new_main_text[:new_cursor_index].split('\n')
                    line = len(lines)
                    column = len(lines[-1]) if lines else 0
                    self.text_edits[2].mark_set(tk.INSERT, f"{line}.{column}")
                else:
                    self.text_edits[2].mark_set(tk.INSERT, cursor_pos)
            except:
                pass
            
            return "break"



        elif mode == '어휘3':
            cursor_pos = self.text_edits[2].index(tk.INSERT)
            
            # Get the entire text from each widget
            main_text = self.text_edits[2].get("1.0", "end-1c")
            answers_text = self.text_edits[3].get("1.0", "end-1c").strip()
            explanations_text = self.text_edits[4].get("1.0", "end-1c")
            
            # Find the cursor position as a character index
            cursor_char_index = len(self.text_edits[2].get("1.0", cursor_pos))
            
            # Helper functions for circled number mapping
            def circled_num_to_int(circled_num_char):
                circled_numbers = {
                    '①': 1, '②': 2, '③': 3, '④': 4, '⑤': 5,
                    '⑥': 6, '⑦': 7, '⑧': 8, '⑨': 9, '⑩': 10,
                    '⑪': 11, '⑫': 12, '⑬': 13, '⑭': 14, '⑮': 15,
                    '⑯': 16, '⑰': 17, '⑱': 18, '⑲': 19, '⑳': 20
                }
                return circled_numbers.get(circled_num_char, None)
            
            def int_to_circled_num(n):
                circled_numbers = {
                    1: '①', 2: '②', 3: '③', 4: '④', 5: '⑤',
                    6: '⑥', 7: '⑦', 8: '⑧', 9: '⑨', 10: '⑩',
                    11: '⑪', 12: '⑫', 13: '⑬', 14: '⑭', 15: '⑮',
                    16: '⑯', 17: '⑰', 18: '⑱', 19: '⑲', 20: '⑳'
                }
                return circled_numbers.get(n, None)
            
            # Pattern to find circled numbers followed by angle brackets or square brackets
            angle_pattern = r'([①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳])<([^>]+)>'
            square_pattern = r'([①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳])\[([^\]]+)\]'
            
            # Find all matches
            angle_matches = list(re.finditer(angle_pattern, main_text))
            square_matches = list(re.finditer(square_pattern, main_text))
            
            # Find which pattern contains the cursor
            target_match = None
            target_number = None
            is_angle_bracket = False
            
            # Check angle brackets first
            for match in angle_matches:
                if match.start() <= cursor_char_index <= match.end():
                    target_match = match
                    target_number = match.group(1)
                    is_angle_bracket = True
                    break
            
            # If not in angle brackets, check square brackets
            if not target_match:
                for match in square_matches:
                    if match.start() <= cursor_char_index <= match.end():
                        target_match = match
                        target_number = match.group(1)
                        is_angle_bracket = False
                        break
            
            if not target_match:
                return "break"
            
            # Find the replacement word
            replacement_word = None
            
            if is_angle_bracket:
                # For angle brackets, look above "Explanation:"
                correction_pattern = rf'^{re.escape(target_number)}\s*[^→]+→\s*(.+?)$'
                correction_match = re.search(correction_pattern, explanations_text, re.MULTILINE)
                
                if correction_match:
                    replacement_word = correction_match.group(1).strip()
            else:
                # For square brackets, look below "Original Words:"
                original_words_match = re.search(r'Original Words:\n(.+)', explanations_text, re.DOTALL)
                
                if original_words_match:
                    original_words_section = original_words_match.group(1)
                    
                    # Look for the line starting with the target number
                    line_pattern = rf'^{re.escape(target_number)}\s*(.+?)$'
                    line_match = re.search(line_pattern, original_words_section, re.MULTILINE)
                    
                    if line_match:
                        line_content = line_match.group(1).strip()
                        
                        # Check if it contains '='
                        if '=' in line_content:
                            # Extract word after '='
                            parts = line_content.split('=')
                            if len(parts) >= 2:
                                replacement_word = parts[1].strip()
                        elif '원문 단어 그대로' in line_content:
                            # Use the original word (what's inside the brackets)
                            replacement_word = target_match.group(2)
            
            if not replacement_word:
                # If no replacement found, just remove brackets and number for square brackets
                if not is_angle_bracket:
                    replacement_word = target_match.group(2)
                else:
                    return "break"
            
            # Step 1: Replace the pattern in main text
            new_main_text = main_text[:target_match.start()] + replacement_word + main_text[target_match.end():]
            
            # Step 2: Remove the number from answers
            new_answers = answers_text.replace(target_number, '')
            
            # Step 3: Remove lines starting with this number from explanations
            explanation_lines = explanations_text.split('\n')
            new_explanation_lines = []
            skip_next = False
            in_original_words = False
            
            for i, line in enumerate(explanation_lines):
                # Always preserve "Explanation:" and "Original Words:" lines
                if "Explanation:" in line or "Original Words:" in line:
                    if "Original Words:" in line:
                        in_original_words = True
                    skip_next = False  # Reset skip flag
                    new_explanation_lines.append(line)
                    continue
                
                # Skip lines that start with the target number
                if line.strip().startswith(target_number):
                    # For angle brackets: remove from main explanation section
                    if is_angle_bracket and not in_original_words:
                        skip_next = True
                        continue
                    # For square brackets: remove from Original Words section
                    elif not is_angle_bracket and in_original_words:
                        continue
                    # For angle brackets: also remove from explanation section above Original Words
                    elif is_angle_bracket and in_original_words:
                        continue
                
                # For angle brackets, also skip the Korean explanation line that follows
                if skip_next and is_angle_bracket and line.strip() and not line.strip().startswith(('①', '②', '③', '④', '⑤', '⑥', '⑦', '⑧', '⑨', '⑩')):
                    skip_next = False
                    continue
                
                skip_next = False
                new_explanation_lines.append(line)
            
            new_explanations = '\n'.join(new_explanation_lines)
            
            # Step 4: Renumber everything
            removed_num = circled_num_to_int(target_number)
            
            if removed_num:
                # Renumber in main text (both [ ] and < > patterns)
                def renumber_in_main_text(text, removed_num):
                    def replace_num(match):
                        num_char = match.group(1)
                        num = circled_num_to_int(num_char)
                        if num and num > removed_num:
                            new_num = num - 1
                            new_char = int_to_circled_num(new_num)
                            return new_char + match.group(2)
                        return match.group(0)
                    
                    # Replace in both angle and square bracket patterns
                    text = re.sub(r'([①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳])(<[^>]+>)', replace_num, text)
                    text = re.sub(r'([①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳])(\[[^\]]+\])', replace_num, text)
                    return text
                
                new_main_text = renumber_in_main_text(new_main_text, removed_num)
                
                # Renumber in answers
                def renumber_answers(text, removed_num):
                    result = ""
                    for char in text:
                        num = circled_num_to_int(char)
                        if num and num > removed_num:
                            new_num = num - 1
                            new_char = int_to_circled_num(new_num)
                            result += new_char if new_char else char
                        else:
                            result += char
                    return result
                
                new_answers = renumber_answers(new_answers, removed_num)
                
                # Renumber in explanations
                def renumber_explanations(text, removed_num):
                    lines = text.split('\n')
                    new_lines = []
                    for line in lines:
                        # Check if line starts with a circled number
                        match = re.match(r'^([①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳])', line)
                        if match:
                            num_char = match.group(1)
                            num = circled_num_to_int(num_char)
                            if num and num > removed_num:
                                new_num = num - 1
                                new_char = int_to_circled_num(new_num)
                                line = new_char + line[1:]
                        new_lines.append(line)
                    return '\n'.join(new_lines)
                
                new_explanations = renumber_explanations(new_explanations, removed_num)
            
            # Update all three widgets
            self.text_edits[2].delete("1.0", "end")
            self.text_edits[2].insert("1.0", new_main_text)
            
            self.text_edits[3].delete("1.0", "end")
            self.text_edits[3].insert("1.0", new_answers)
            
            self.text_edits[4].delete("1.0", "end")
            self.text_edits[4].insert("1.0", new_explanations)
            
            # Try to maintain cursor position
            try:
                # Adjust cursor position based on the replacement
                if cursor_char_index > target_match.start():
                    adjustment = len(replacement_word) - len(target_match.group(0))
                    new_cursor_index = cursor_char_index + adjustment
                    # Convert back to line.column format
                    lines = new_main_text[:new_cursor_index].split('\n')
                    line = len(lines)
                    column = len(lines[-1]) if lines else 0
                    self.text_edits[2].mark_set(tk.INSERT, f"{line}.{column}")
                else:
                    self.text_edits[2].mark_set(tk.INSERT, cursor_pos)
                
                self.text_edits[2].see(tk.INSERT)
            except:
                pass
            
            return "break"




    def F4_어법2단계(event=None):
        sheet_name = self.sheet_selector.get()
        answer_flag = True


        if "실전모의고사" in sheet_name:
            실모유형 = self.실모_유형_menu.get()
            실모난이도 = self.실모_난이도_menu.get()
            실모유형난이도 = 실모유형 + "_" + 실모난이도
            print(f"실모유형난이도: {실모유형난이도}")
            if 실모유형난이도 == "어법2단계_Normal":
                sheet_name = "어법2단계_Normal"
            elif 실모유형난이도 == "어법2단계_Hard":
                sheet_name = "어법2단계_Hard"
            print(f"sheet_name: {sheet_name}")

        """
        This function performs the following steps:
        1. Detects the circled number where the cursor is placed within self.text_edits[2].
        2. Removes the detected circled number from self.text_edits[3].
        3. Fetches the corresponding correction from self.text_edits[4] and replaces the text within the brackets in self.text_edits[2].
        4. Removes the processed correction line from self.text_edits[4].
        """
        
        def get_char_index(text_widget, index):
            """
            Converts a Tkinter text index (e.g., '1.0') to a zero-based character index.
            """
            line, char = map(int, index.split('.'))
            lines = text_widget.get("1.0", f"{line}.0").split('\n')
            total_chars = sum(len(l) + 1 for l in lines[:-1])  # +1 for newline
            total_chars += char
            return total_chars

        # Assign widgets
        text_widget = self.text_edits[2]         # CTkTextbox containing the main text
        numbers_widget = self.text_edits[3]      # CTkTextbox containing circled numbers
        corrections_widget = self.text_edits[4]  # CTkTextbox containing corrections

        # Step 1: Get the current cursor position in self.text_edits[2]
        cursor_index = text_widget.index("insert")
        cursor_pos = get_char_index(text_widget, cursor_index)

        # Capture the current scroll position
        y_view = text_widget.yview()
        x_view = text_widget.xview()

        # Get the entire text from self.text_edits[2]
        text_content = text_widget.get("1.0", "end-1c")

        # Define pattern to find circled numbers followed by [text]
        pattern = r'([①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳㉑㉒㉓㉔㉕㉖㉗㉘㉙㉚㉛㉜㉝㉞㉟㊱㊲㊳㊴㊵])\[(.*?)\]'
        matches = list(re.finditer(pattern, text_content))

        # Determine which match contains the cursor
        circled_number = None
        match_obj = None
        for match in matches:
            start, end = match.span()
            if start <= cursor_pos <= end:
                circled_number = match.group(1)
                match_obj = match
                break

        if not circled_number:
            messagebox.showinfo("Info", "Cursor is not inside a bracket with a circled number.")
            popup.focus_set()
            self.text_edits[2].focus_set()
            return  # Cursor not in the target area

        # Step 2: Remove circled_number from self.text_edits[3]
        numbers_text = numbers_widget.get("1.0", "end-1c")
        if circled_number in numbers_text:
            updated_numbers = numbers_text.replace(circled_number, "")
            numbers_widget.delete("1.0", "end")
            numbers_widget.insert("1.0", updated_numbers)
        else:  # 정답 선지가 아님
            phraseB = None
            if not phraseB:
                messagebox.showerror("Error", f"{circled_number}는 정답 선지가 아니므로 원래 형태로 되돌릴 수 없습니다.")
                popup.focus_set()
                self.text_edits[2].focus_set()
                return "break" 

            answer_flag = False
            phraseB = None

        # Step 3: Find the correction in self.text_edits[4]
        # 어법2단계_Normal and 어법2단계_Hard now use the same logic
        if sheet_name in ["어법2단계_Hard", "어법2단계_Normal"]:
            corrections_text = corrections_widget.get("1.0", "end-1c")
            lines = corrections_text.splitlines()
            phraseB = None
            new_corrections = []
            for line in lines:
                if line.startswith(circled_number):
                    # Assuming format '① open > opens'
                    parts = line.split('>')
                    if len(parts) >= 2:
                        phraseB = parts[1].strip()
                else:
                    new_corrections.append(line)

        if not phraseB:
            messagebox.showerror("Error", f"{circled_number}는 정답 선지가 아니므로 원래 형태로 되돌릴 수 없습니다.")
            popup.focus_set()
            self.text_edits[2].focus_set()
            return "break" 

        # Replace the [text] with phraseB in self.text_edits[2]
        original_text = match_obj.group(0)  # e.g., '①[open]'
        replacement_text = f"{circled_number}[{phraseB}]"

        # To ensure only the specific occurrence is replaced, use string slicing
        new_text_content = text_content[:match_obj.start()] + replacement_text + text_content[match_obj.end():]

        # Update the text in self.text_edits[2]
        text_widget.delete("1.0", "end")
        text_widget.insert("1.0", new_text_content)

        # Step 4: Update the corrections_widget by removing the processed line
        corrections_widget.delete("1.0", "end")
        corrections_widget.insert("1.0", "\n".join(new_corrections))

        print(f"Replaced {original_text} with {replacement_text}.")
        # messagebox.showinfo("Success", f"Replaced {original_text} with {replacement_text}.")

        # Restore the original cursor position and scroll view
        try:
            # Ensure that the cursor index is still valid after modifications
            text_widget.mark_set("insert", cursor_index)
            text_widget.see("insert")  # Scroll to make sure the cursor is visible

            # Restore the scroll position
            text_widget.yview_moveto(y_view[0])
            text_widget.xview_moveto(x_view[0])
        except Exception as e:
            # If the original index is no longer valid, set the cursor to the end
            print(f"Failed to set cursor to original position: {e}. Setting to end.")
            text_widget.mark_set("insert", "end-1c")
            text_widget.see("end-1c")

        return "break"  # Prevent further propagation of the event
            




    def F5_어법2단계(event=None): #틀린 선지 수정 후 해설지에 반영
        sheet_name = self.sheet_selector.get()
        answer_flag = True


        if "실전모의고사" in sheet_name:
            실모유형 = self.실모_유형_menu.get()
            실모난이도 = self.실모_난이도_menu.get()
            실모유형난이도 = 실모유형 + "_" + 실모난이도
            print(f"실모유형난이도: {실모유형난이도}")
            if 실모유형난이도 == "어법2단계_Normal":
                sheet_name = "어법2단계_Normal"
            elif 실모유형난이도 == "어법2단계_Hard":
                sheet_name = "어법2단계_Hard"
            print(f"sheet_name: {sheet_name}")



        def get_char_index(text_widget, index):
            """
            Converts a Tkinter text index (e.g., '1.0') to a zero-based character index.
            """
            line, char = map(int, index.split('.'))
            lines = text_widget.get("1.0", f"{line}.0").split('\n')
            total_chars = sum(len(l) + 1 for l in lines[:-1])  # +1 for newline
            total_chars += char
            return total_chars

        # Assign widgets
        text_widget = self.text_edits[2]         # CTkTextbox containing the main text
        numbers_widget = self.text_edits[3]      # CTkTextbox containing circled numbers
        corrections_widget = self.text_edits[4]  # CTkTextbox containing corrections

        # Step 1: Get the current cursor position in self.text_edits[2]
        cursor_index = text_widget.index("insert")
        cursor_pos = get_char_index(text_widget, cursor_index)

        # Step 2: Find the phrase inside the brackets in self.text_edits[2]
        # Get the entire text from self.text_edits[2]
        text_content = text_widget.get("1.0", "end-1c")

        # Find the position of the nearest '[' to the left of the cursor
        left_bracket_pos = text_content.rfind('[', 0, cursor_pos)
        if left_bracket_pos == -1:
            messagebox.showinfo("Info", "커서 위치에 '[' 가 없습니다.")  # "No '[' found at the cursor position."
            return

        # Find the position of the nearest ']' to the right of the cursor
        right_bracket_pos = text_content.find(']', cursor_pos)
        if right_bracket_pos == -1:
            messagebox.showinfo("Info", "커서 위치에 ']' 가 없습니다.")  # "No ']' found at the cursor position."
            return

        # Extract the text within the brackets
        selected_phrase = text_content[left_bracket_pos + 1 : right_bracket_pos].strip()

        if not selected_phrase:
            messagebox.showwarning("Warning", "선택된 어구가 비어 있습니다.")  # "The selected phrase is empty."
            return

        # Step 3: Find the circled number in front of the brackets.
        # Define pattern to find circled numbers followed by [text]
        pattern = r'([①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳㉑㉒㉓㉔㉕㉖㉗㉘㉙㉚㉛㉜㉝㉞㉟㊱㊲㊳㊴㊵])\[(.*?)\]'
        matches = list(re.finditer(pattern, text_content))

        # Determine which match contains the cursor
        circled_number = None
        match_obj = None
        for match in matches:
            start, end = match.span()
            if start <= cursor_pos <= end:
                circled_number = match.group(1)
                match_obj = match
                break

        if not circled_number:
            popup.focus_set()
            self.text_edits[2].focus_set()

            messagebox.showinfo("Info", "Cursor is not inside a bracket with a circled number.")
            return  # Cursor not in the target area

        
        # Step 3.5 : 현재 선택된 것이 정답이 아니면 에러 반환
        numbers_text = numbers_widget.get("1.0", "end-1c")
        if circled_number not in numbers_text:
            messagebox.showerror("Error", f"{circled_number}는 정답 선지가 아니므로 해설지를 수정할 수 없습니다.")
            popup.focus_set()
            self.text_edits[2].focus_set()

            return "break"

        # Step 4: Find and Replace the Correction in self.text_edits[4]
        # 어법2단계_Normal and 어법2단계_Hard now use the same logic
        if sheet_name in ["어법2단계_Hard", "어법2단계_Normal"]:
            corrections_text = corrections_widget.get("1.0", "end-1c")
            lines = corrections_text.splitlines()
            phraseB = None
            new_corrections = []
            for line in lines:
                if line.startswith(circled_number):
                    # Assuming format '① open > opens'
                    # Use regex to accurately parse the line
                    match = re.match(rf'^{re.escape(circled_number)}\s*(.*?)\s*>\s*(.*)$', line)
                    if match:
                        original_phraseB = match.group(1).strip()
                        phraseC = match.group(2).strip()
                        phraseB = original_phraseB  # Store original phraseB for validation if needed

                        # Replace phraseB with selected_phrase
                        new_line = f"{circled_number} {selected_phrase} > {phraseC}"
                        new_corrections.append(new_line)
                    else:
                        # If the line doesn't match the expected format, keep it unchanged
                        new_corrections.append(line)
                else:
                    new_corrections.append(line)

        else:
            # If sheet_name doesn't match
            messagebox.showerror("Error", "해당 조건에 맞는 sheet_name이 아닙니다.")  # "Invalid sheet_name for the given condition."
            popup.focus_set()
            self.text_edits[2].focus_set()

            return "break"



        # Update the corrections_widget with the new corrections
        corrections_widget.delete("1.0", "end")
        corrections_widget.insert("1.0", "\n".join(new_corrections))
        print(f"Replaced phrase before '>' for {circled_number} with '{selected_phrase}'.")

        return "break"  # Prevent further propagation of the event

            






    def move_cursor_to_front_of_selected_word(event=None):
        try:
            # Get the start of the selection
            selection_start = self.text_edits[2].index(tk.SEL_FIRST)
            # Clear the current selection
            self.text_edits[2].tag_remove(tk.SEL, tk.SEL_FIRST, tk.SEL_LAST)
            # Move the cursor to the start of the selection
            self.text_edits[2].mark_set(tk.INSERT, selection_start)
            self.text_edits[2].see(tk.INSERT)
            self.text_edits[2].focus_set()
        except tk.TclError:
            pass  # No selection, so do nothing

        return "break"

    def unselect_word(event=None):
        try:
            # Clear the current selection
            self.text_edits[2].tag_remove(tk.SEL, tk.SEL_FIRST, tk.SEL_LAST)
            self.text_edits[2].focus_set()
        except tk.TclError:
            pass  # No selection, so do nothing

        return "break"


    
    """
    #def setup_bindings(self, sheet_name):
    if sheet_name == "어법1단계_Normal":
        mode = '어법'
    elif "어휘1단계" in sheet_name:
        mode = '어휘'
    else:
        # Default behavior or raise an exception
        print(f"Unsupported sheet name: {sheet_name}")
        #return
    """





    # Bind the macro functions to specific key combinations
    self.text_edits[2].bind("<F1>", lambda event: F1_process_and_select_word(event))
    self.text_edits[2].bind("<F3>", lambda event: F3_replace_phrase(event))
    self.text_edits[2].bind("<F4>", lambda event: F4_어법2단계(event))
    self.text_edits[2].bind("<F5>", lambda event: F5_어법2단계(event))
    self.text_edits[2].bind("<Shift-F2>", lambda event: F2_어휘2단계동의어변경(event))

    #self.text_edits[2].bind("<F2>", lambda event: process_and_select_two_words(event, mode=mode))


    def F2_어휘2단계동의어변경(event=None):
        print("F2_어휘2단계동의어변경 called")
        self.어휘2단계동의어변경flag = True
        F2_process_and_select_two_words()


    def general_key_handler(event):
        #print("general_key_handler called")
        if event.keysym == 'F2':
            F2_process_and_select_two_words(event)
            return "break"
        elif event.keysym in ('Left', 'Right') and self.macro == True:
            try:
                # Check if there is a selected text
                self.text_edits[2].index(tk.SEL_FIRST)
                if event.keysym == 'Left':
                    self.macro = False
                    return move_cursor_to_front_of_selected_word()
                    
                else:  # Right
                    self.macro = False  
                    return unselect_word()
                    
            except tk.TclError:
                pass  # No selection, proceed with default arrow key behavior
        else:
            self.macro = False
    self.text_edits[2].bind("<Key>", general_key_handler)
    







