#modules/manual_moctest.py

import tkinter as tk
from tkinter import messagebox
from tkinter import font as tkfont
from openpyxl import load_workbook
from pathlib import Path
import sys
import json
import random
import copy
import unicodedata
import customtkinter
from customtkinter import (CTkToplevel, CTkFrame, CTkLabel, CTkButton,
                           CTkEntry, CTkOptionMenu, CTkCheckBox, CTkScrollableFrame)
from modules.points_dashboard import PointsDashboard
from modules.functions import CopyManager





class ToolTip:
    """Create a tooltip for a given widget"""
    def __init__(self, widget, text='widget info', max_width=400):
        self.widget = widget
        self.text = text
        self.max_width = max_width
        self.widget.bind("<Enter>", self.on_enter)
        self.widget.bind("<Leave>", self.on_leave)
        self.widget.bind("<ButtonPress>", self.on_leave)
        self.tooltip = None

    def on_enter(self, event=None):
        self.show_tooltip()

    def on_leave(self, event=None):
        self.hide_tooltip()

    def show_tooltip(self):
        if self.tooltip or not self.text:
            return
        x, y, cx, cy = self.widget.bbox("insert") if hasattr(self.widget, 'bbox') else (0, 0, 0, 0)
        x += self.widget.winfo_rootx() + 25
        y += self.widget.winfo_rooty() + 25
        
        # Create tooltip window
        self.tooltip = tw = tk.Toplevel(self.widget)
        tw.wm_overrideredirect(True)
        
        # Create label with text wrapping for longer passages
        label = tk.Label(tw, text=self.text, justify='left',
                        background="lightyellow", relief='solid', borderwidth=1,
                        font=("Arial", "10", "normal"),
                        wraplength=self.max_width)  # Add word wrapping
        label.pack()
        
        # Update position after creating label to account for size
        tw.update_idletasks()
        
        # Adjust position if tooltip goes off-screen
        screen_width = tw.winfo_screenwidth()
        screen_height = tw.winfo_screenheight()
        tw_width = tw.winfo_width()
        tw_height = tw.winfo_height()
        
        if x + tw_width > screen_width:
            x = screen_width - tw_width - 10
        if y + tw_height > screen_height:
            y = y - tw_height - 50
            
        tw.wm_geometry(f"+{x}+{y}")

    def hide_tooltip(self):
        tw = self.tooltip
        self.tooltip = None
        if tw:
            tw.destroy()

    def update_text(self, new_text):
        self.text = new_text


#class CustomDropdown needed.
#Create a custom dropdown menu.
#When there is a duplicate in the items, the font should be in red.
#There should be a hover effect in the items in the dropdown menu.


class CustomDropdown(CTkFrame):
    """Custom dropdown widget with duplicate highlighting and hover support."""

    def __init__(
        self,
        master,
        values,
        variable=None,
        command=None,
        get_selected_items_func=None,
        placeholder_text="지문을 선택하세요",
        width=140,
        height=30,
        font=None,
        bg_color="transparent",
        fg_color="white",
        text_color=("black", "white"),
        button_color="#FEF9E0",
        button_hover_color="#CA7900",
        dropdown_bg_color="white",
        dropdown_hover_color="#CA7900",
        duplicate_text_color="#B22222",
        enable_edit_button=False,
        on_edit_callback=None,
        get_edited_status_func=None,
    ):
        super().__init__(master, fg_color=bg_color)

        self.values = list(values) if values else []
        self.variable = variable if variable is not None else tk.StringVar()
        self.command = command
        self.get_selected_items_func = get_selected_items_func
        self.placeholder_text = placeholder_text
        self.enable_edit_button = enable_edit_button
        self.on_edit_callback = on_edit_callback
        self.get_edited_status_func = get_edited_status_func

        self.width = width
        self.height = height
        self.font = font
        self.normal_text_color = text_color
        self.item_text_color = text_color
        self.button_color = button_color
        self.button_hover_color = button_hover_color
        self.dropdown_bg_color = dropdown_bg_color
        self.dropdown_hover_color = dropdown_hover_color
        self.duplicate_text_color = duplicate_text_color
        self.dropdown_item_bg_color = dropdown_bg_color

        self.dropdown_window = None
        self.dropdown_items = []
        self._variable_trace = self.variable.trace_add("write", self._on_variable_change)
        self._dropdown_click_handler_id = None  # Track dropdown click handler
        self._root_escape_handler_id = None  # Track root-level ESC handler
        self._root_click_handler_id = None  # Track root-level click handler

        # Create button frame to hold text and arrow separately
        self.button = CTkFrame(
            self,
            width=self.width,
            height=self.height,
            fg_color=self.button_color,
            border_width=1,
            border_color="#D1D5DB",
        )
        self.button.pack(fill="both", expand=True)
        self.button.pack_propagate(False)

        # Text label (left side)
        self.text_label = CTkLabel(
            self.button,
            text=self._get_display_text(),
            font=self.font,
            text_color=self.normal_text_color,
            fg_color="transparent",
            anchor="w",
            height=self.height - 4,  # Slightly smaller to fit within borders
        )
        self.text_label.pack(side="left", fill="x", expand=True, padx=(8, 0), pady=0)

        # Arrow label (right side)
        self.arrow_label = CTkLabel(
            self.button,
            text="▼",
            font=self.font,
            text_color=self.normal_text_color,
            fg_color="transparent",
            height=self.height - 4,  # Match text label height
        )
        self.arrow_label.pack(side="right", padx=(0, 8), pady=0)

        # Edit button (between text and arrow, visible only when passage selected)
        if self.enable_edit_button:
            self.edit_button_frame = CTkFrame(self.button, fg_color="transparent")
            self.edit_icon = CTkLabel(
                self.edit_button_frame,
                text="✏️",
                font=(self.font[0], self.font[1]-2) if self.font else ("Arial", 9),
                text_color=self.normal_text_color,
                fg_color="transparent",
                cursor="hand2"
            )
            self.edit_icon.pack()
            self.edit_button_frame.pack(side="right", padx=(0, 5))
            self.edit_icon.bind("<Button-1>", lambda e: self._on_edit_clicked())

            # Initially hidden (will be shown when passage is selected)
            self.edit_button_frame.pack_forget()

        # Bind click events to all components
        self.button.bind("<Button-1>", lambda e: self.toggle_dropdown())
        self.text_label.bind("<Button-1>", lambda e: self.toggle_dropdown())
        self.arrow_label.bind("<Button-1>", lambda e: self.toggle_dropdown())

        # Bind hover events for visual feedback
        for widget in [self.button, self.text_label, self.arrow_label]:
            widget.bind("<Enter>", lambda e: self.button.configure(fg_color=self.button_hover_color))
            widget.bind("<Leave>", lambda e: self.button.configure(fg_color=self.button_color))
            widget.configure(cursor="hand2")

        self.bind("<Destroy>", self._on_destroy, add="+")
        self.refresh_item_styles()

    def toggle_dropdown(self):
        if self.dropdown_window and self.dropdown_window.winfo_exists():
            self.close_dropdown()
            return

        if not self.values:
            return

        self._open_dropdown()

    def _open_dropdown(self):
        self.dropdown_window = CTkToplevel(self)
        self.dropdown_window.withdraw()
        self.dropdown_window.wm_overrideredirect(True)
        self.dropdown_window.attributes("-topmost", True)
        self.dropdown_window.configure(fg_color=self.dropdown_bg_color)
        try:
            self.dropdown_window.transient(self.winfo_toplevel())
        except Exception:
            pass

        # Calculate maximum dropdown height (show max 8 items before scrolling)
        max_visible_items = 15
        item_height = self.height   # Reduced spacing (was +4)
        max_dropdown_height = max_visible_items * item_height + 4

        # Always use scrollable frame to ensure all items are accessible
        # Calculate appropriate height based on number of items
        if len(self.values) > max_visible_items:
            container_height = max_dropdown_height - 4
        else:
            # For fewer items, size to fit content but still use scrollable frame for safety
            container_height = len(self.values) * item_height + 4

        container = CTkScrollableFrame(
            self.dropdown_window,
            fg_color=self.dropdown_bg_color,
            height=container_height,
            scrollbar_button_color="#D1D5DB",
            scrollbar_button_hover_color="#9CA3AF",
        )
        container.pack(fill="both", expand=True, padx=1, pady=1)

        # Store container reference for mousewheel binding
        self._dropdown_container = container

        self.dropdown_items.clear()
        for value in self.values:
            label = CTkLabel(
                container,
                text=value,
                font=self.font,
                fg_color=self.dropdown_item_bg_color,
                text_color=self.item_text_color,
                anchor="w",
                height=self.height,
                cursor="hand2",  # Show pointer cursor on hover
            )
            label.pack(fill="x", expand=True, padx=4, pady=0)  # Reduced pady from 1 to 0

            # Bind to both the CTkLabel and its internal widget for reliable hover detection
            label.bind("<Enter>", lambda event, lbl=label: self._on_item_hover(lbl, True))
            label.bind("<Leave>", lambda event, lbl=label: self._on_item_hover(lbl, False))
            label.bind("<ButtonRelease-1>", lambda event, val=value: self._on_item_select(val))

            # Also bind to the internal canvas widget for better event handling
            if hasattr(label, '_canvas'):
                label._canvas.bind("<Enter>", lambda event, lbl=label: self._on_item_hover(lbl, True), add="+")
                label._canvas.bind("<Leave>", lambda event, lbl=label: self._on_item_hover(lbl, False), add="+")

            self.dropdown_items.append({"value": value, "label": label})

        # Bind mousewheel to prevent parent scrolling (AFTER labels are created)
        self._bind_mousewheel_to_dropdown()

        self.dropdown_window.update_idletasks()

        button_root_x = self.button.winfo_rootx()
        button_root_y = self.button.winfo_rooty() + self.button.winfo_height()
        measured_width = self.dropdown_window.winfo_width()
        dropdown_width = max(self.button.winfo_width(), self.width, measured_width)

        # Calculate actual dropdown height (use container_height we calculated earlier)
        dropdown_height = self.dropdown_window.winfo_height()
        if dropdown_height <= 1:
            dropdown_height = container_height + 4  # Add small padding

        self.dropdown_window.geometry(f"{dropdown_width}x{dropdown_height}+{button_root_x}+{button_root_y}")
        self.dropdown_window.deiconify()
        self.dropdown_window.lift()

        # Bind ESC key to close dropdown (bind to both window and container for reliability)
        self.dropdown_window.bind("<Escape>", lambda event: self._on_escape_pressed(event, "dropdown_window"))
        container.bind("<Escape>", lambda event: self._on_escape_pressed(event, "container"))

        # Close dropdown when clicking outside
        self.dropdown_window.bind("<FocusOut>", lambda event: self._schedule_close_check())

        # Bind click detection to root window so outside clicks are captured
        root = self.winfo_toplevel()
        self._root_click_handler_id = root.bind("<Button-1>", self._check_click_outside, add="+")

        # Also watch for clicks delivered directly to the dropdown (grab redirects)
        self._dropdown_click_handler_id = self.dropdown_window.bind("<Button-1>", self._check_click_outside, add="+")

        # Force focus after window is mapped (critical for macOS overrideredirect windows)
        self.dropdown_window.after_idle(self._activate_dropdown_focus)

        self.refresh_item_styles()

    def _activate_dropdown_focus(self):
        """Force focus to dropdown after it's mapped (macOS overrideredirect fix)"""
        if not self.dropdown_window or not self.dropdown_window.winfo_exists():
            return

        try:
            # Release any lingering grabs on parent
            root = self.winfo_toplevel()
            try:
                root.grab_release()
            except:
                pass

            # Grab focus for dropdown
            dropdown_window = self.dropdown_window
            dropdown_window.grab_set()
            dropdown_window.focus_force()
            # Release grab after idle so root can detect outside clicks
            dropdown_window.after_idle(lambda win=dropdown_window: win.grab_release() if win.winfo_exists() else None)

            # Verify focus (diagnostic)
            dropdown_window.after(100, self._verify_focus)

            # Ensure ESC is captured even if focus drifts back to parent
            if self._root_escape_handler_id:
                try:
                    root.unbind("<Escape>", self._root_escape_handler_id)
                except Exception:
                    pass
            self._root_escape_handler_id = root.bind("<Escape>", lambda e: self._on_escape_pressed(e, "root"), add="+")

        except Exception as e:
            # If focus forcing fails we still allow dropdown to function without raising
            return

    def _verify_focus(self):
        """Diagnostic: Check where focus actually is"""
        if not self.dropdown_window or not self.dropdown_window.winfo_exists():
            return
        current_focus = self.dropdown_window.focus_displayof()
        expected_focus = self.dropdown_window
        # Optional diagnostics removed in production

    def _on_escape_pressed(self, event, source):
        """Handle ESC key press"""
        self.close_dropdown()
        return "break"  # Stop event propagation to parent

    def _check_click_outside(self, event):
        """Check if click is outside dropdown and close if so"""
        if not self.dropdown_window or not self.dropdown_window.winfo_exists():
            return

        # Get dropdown window geometry
        dropdown_x = self.dropdown_window.winfo_rootx()
        dropdown_y = self.dropdown_window.winfo_rooty()
        dropdown_width = self.dropdown_window.winfo_width()
        dropdown_height = self.dropdown_window.winfo_height()
        # Get button geometry
        button_x = self.button.winfo_rootx()
        button_y = self.button.winfo_rooty()
        button_width = self.button.winfo_width()
        button_height = self.button.winfo_height()

        # Check if click is outside both dropdown and button
        click_x = event.x_root
        click_y = event.y_root

        in_dropdown = (dropdown_x <= click_x <= dropdown_x + dropdown_width and
                       dropdown_y <= click_y <= dropdown_y + dropdown_height)
        in_button = (button_x <= click_x <= button_x + button_width and
                     button_y <= click_y <= button_y + button_height)

        if in_button and not in_dropdown:
            self.close_dropdown()
            return "break"
        if not in_dropdown and not in_button:
            self.close_dropdown()
            return "break"

    def _schedule_close_check(self):
        """Schedule a delayed check to close dropdown (for FocusOut events)"""
        if self.dropdown_window and self.dropdown_window.winfo_exists():
            self.after(100, self._check_if_should_close)

    def _check_if_should_close(self):
        """Check if dropdown should be closed after focus loss"""
        if not self.dropdown_window or not self.dropdown_window.winfo_exists():
            return

        # Check if focus is in dropdown or button
        try:
            focused = self.focus_get()
            if focused and (focused == self.dropdown_window or
                          self.dropdown_window in str(focused) or
                          focused in [self.button, self.text_label, self.arrow_label]):
                return  # Keep dropdown open
        except Exception:
            pass

        # Close if focus is elsewhere
        self.close_dropdown()

    def _on_item_hover(self, label, is_hover):
        if is_hover:
            label.configure(fg_color=self.dropdown_hover_color)
        else:
            label.configure(fg_color=self.dropdown_item_bg_color)

    def _on_item_select(self, value):
        if value != self.variable.get():
            self.variable.set(value)
            if self.command:
                self.command(value)
        self.close_dropdown()
        self.refresh_item_styles()

    def refresh_item_styles(self):
        duplicates = self._compute_duplicates()

        current_value = self.variable.get()
        if current_value and current_value in duplicates:
            self.text_label.configure(text_color=self.duplicate_text_color)
            self.arrow_label.configure(text_color=self.duplicate_text_color)
        else:
            self.text_label.configure(text_color=self.normal_text_color)
            self.arrow_label.configure(text_color=self.normal_text_color)

        for item in self.dropdown_items:
            label = item["label"]
            value = item["value"]
            if not hasattr(label, "winfo_exists") or not label.winfo_exists():
                continue
            if value and value in duplicates:
                label.configure(text_color=self.duplicate_text_color)
            else:
                label.configure(text_color=self.item_text_color)

    def _compute_duplicates(self):
        """Return items that are selected in OTHER rows (not the current row)"""
        duplicates = set()

        if callable(self.get_selected_items_func):
            try:
                # Get all selected items from other rows (excluding current row and placeholder)
                selected_items = [item for item in self.get_selected_items_func()
                                if item and item != self.placeholder_text]
            except Exception:
                selected_items = []

            # All items returned from get_selected_items_func are from other rows
            duplicates.update(selected_items)

        return duplicates

    def _on_variable_change(self, *args):
        self.text_label.configure(text=self._get_display_text())

        # Show/hide edit button based on selection
        if self.enable_edit_button:
            value = self.variable.get()
            if value and value != self.placeholder_text:
                self.edit_button_frame.pack(side="right", padx=(0, 5))
            else:
                self.edit_button_frame.pack_forget()

        self.refresh_item_styles()

    def _get_display_text(self):
        value = self.variable.get()
        text = value if value else self.placeholder_text

        # Add "(수정됨)" indicator if edited
        if value and self.get_edited_status_func and self.get_edited_status_func(value):
            text = f"{text} (수정됨)"

        return self._truncate_text(text)

    def _truncate_text(self, text):
        """Truncate text with ellipsis if it exceeds button width"""
        if not text:
            return text

        # Calculate available width (button width - arrow width - padding)
        # Arrow takes roughly 20px, padding on both sides ~16px, add some margin
        available_width = self.width - 40

        try:
            # Create a Font object to measure text width
            if self.font:
                if isinstance(self.font, tuple):
                    font_obj = tkfont.Font(family=self.font[0], size=self.font[1])
                else:
                    font_obj = tkfont.Font(font=self.font)
            else:
                font_obj = tkfont.Font()

            # Measure the text width
            text_width = font_obj.measure(text)

            # If text fits, return as is
            if text_width <= available_width:
                return text

            # Otherwise, truncate with ellipsis
            ellipsis = "..."
            ellipsis_width = font_obj.measure(ellipsis)

            # Binary search for the right length
            for i in range(len(text), 0, -1):
                truncated = text[:i] + ellipsis
                if font_obj.measure(truncated) <= available_width:
                    return truncated

            return ellipsis

        except Exception:
            # Fallback: simple character-based truncation
            max_chars = int(available_width / 8)  # Rough estimate: 8px per char
            if len(text) > max_chars:
                return text[:max_chars-3] + "..."
            return text

    def close_dropdown(self):
        dropdown_window = self.dropdown_window
        try:
            root = self.winfo_toplevel()
        except Exception:
            root = None

        # Unbind dropdown click handler before destroying window
        if dropdown_window and self._dropdown_click_handler_id:
            try:
                dropdown_window.unbind("<Button-1>", self._dropdown_click_handler_id)
            except Exception:
                pass
            self._dropdown_click_handler_id = None

        # Unbind root click handler
        if root and self._root_click_handler_id:
            try:
                root.unbind("<Button-1>", self._root_click_handler_id)
            except Exception:
                pass
            self._root_click_handler_id = None

        # Unbind root ESC handler
        if root and self._root_escape_handler_id:
            try:
                root.unbind("<Escape>", self._root_escape_handler_id)
            except Exception:
                pass
            self._root_escape_handler_id = None

        if dropdown_window and dropdown_window.winfo_exists():
            try:
                dropdown_window.grab_release()
            except Exception:
                pass
            dropdown_window.destroy()

        self.dropdown_window = None
        self.dropdown_items.clear()

    def _on_edit_clicked(self):
        """Called when edit button is clicked"""
        if not self.on_edit_callback:
            return

        current_value = self.variable.get()

        # Guard: Don't open editor for placeholder or empty values
        if not current_value or current_value == self.placeholder_text:
            return

        self.on_edit_callback(current_value)

    def refresh_display(self):
        """Public method to refresh display text and edit button visibility"""
        self._on_variable_change()

    def configure(self, **kwargs):
        if "command" in kwargs:
            self.command = kwargs.pop("command")
        if "values" in kwargs:
            self.set_values(kwargs.pop("values"))
        return super().configure(**kwargs)

    def cget(self, key):
        if key == "values":
            return list(self.values)
        return super().cget(key)

    def set(self, value):
        self.variable.set(value)
        self.refresh_item_styles()

    def get(self):
        return self.variable.get()

    def set_values(self, values):
        self.values = list(values) if values else []
        if self.variable.get() not in self.values:
            self.variable.set("")
        self.close_dropdown()
        self.refresh_item_styles()

    def _on_destroy(self, _event):
        if self._variable_trace:
            try:
                self.variable.trace_remove("write", self._variable_trace)
            except Exception as e:
                pass
            self._variable_trace = None

        # Clean up dropdown click handler if still present
        if self._dropdown_click_handler_id and self.dropdown_window:
            try:
                self.dropdown_window.unbind("<Button-1>", self._dropdown_click_handler_id)
            except Exception as e:
                pass
            self._dropdown_click_handler_id = None

        if self._root_click_handler_id:
            try:
                self.winfo_toplevel().unbind("<Button-1>", self._root_click_handler_id)
            except Exception as e:
                pass
            self._root_click_handler_id = None

        # Clean up root ESC handler if still bound
        if self._root_escape_handler_id:
            try:
                self.winfo_toplevel().unbind("<Escape>", self._root_escape_handler_id)
            except Exception as e:
                pass
            self._root_escape_handler_id = None

        self.close_dropdown()

    def _bind_mousewheel_to_dropdown(self):
        """Bind mouse wheel events to the dropdown to prevent parent scrolling."""
        if not hasattr(self, '_dropdown_container') or not self._dropdown_container:
            return

        container = self._dropdown_container

        # Get the internal canvas from CTkScrollableFrame
        try:
            canvas = container._parent_canvas
        except AttributeError:
            return

        def on_mousewheel(event):
            """Handle mouse wheel scrolling and stop propagation."""
            if not canvas.winfo_exists():
                return "break"

            # macOS uses event.delta directly, Windows/Linux use event.delta / 120
            if event.delta:
                if abs(event.delta) > 100:
                    # macOS
                    canvas.yview_scroll(int(-1 * (event.delta / 120)), "units")
                else:
                    # Windows
                    canvas.yview_scroll(int(-1 * event.delta), "units")
            elif hasattr(event, 'num'):
                # Linux uses Button-4 and Button-5
                if event.num == 4:
                    canvas.yview_scroll(-3, "units")
                elif event.num == 5:
                    canvas.yview_scroll(3, "units")

            # Stop event propagation to prevent parent scrollable frames from scrolling
            return "break"

        # Bind to dropdown window
        if self.dropdown_window:
            self.dropdown_window.bind("<MouseWheel>", on_mousewheel, add="+")
            self.dropdown_window.bind("<Button-4>", on_mousewheel, add="+")
            self.dropdown_window.bind("<Button-5>", on_mousewheel, add="+")

        # Bind to container
        container.bind("<MouseWheel>", on_mousewheel, add="+")
        container.bind("<Button-4>", on_mousewheel, add="+")
        container.bind("<Button-5>", on_mousewheel, add="+")

        # Bind to canvas
        canvas.bind("<MouseWheel>", on_mousewheel, add="+")
        canvas.bind("<Button-4>", on_mousewheel, add="+")
        canvas.bind("<Button-5>", on_mousewheel, add="+")

        # Bind to the inner frame of CTkScrollableFrame
        try:
            inner_frame = container._scrollable_frame
            inner_frame.bind("<MouseWheel>", on_mousewheel, add="+")
            inner_frame.bind("<Button-4>", on_mousewheel, add="+")
            inner_frame.bind("<Button-5>", on_mousewheel, add="+")
        except AttributeError:
            pass

        # Bind to all item labels
        for item_data in self.dropdown_items:
            label = item_data["label"]
            label.bind("<MouseWheel>", on_mousewheel, add="+")
            label.bind("<Button-4>", on_mousewheel, add="+")
            label.bind("<Button-5>", on_mousewheel, add="+")
            # Also bind to internal canvas if exists
            if hasattr(label, '_canvas'):
                label._canvas.bind("<MouseWheel>", on_mousewheel, add="+")
                label._canvas.bind("<Button-4>", on_mousewheel, add="+")
                label._canvas.bind("<Button-5>", on_mousewheel, add="+")

    def destroy(self):
        self.close_dropdown()
        super().destroy()


class PassageEditDialog(CTkToplevel):
    """Dialog for editing passage text with word count validation"""

    def __init__(self, parent, passage_id, current_passage_text, on_save_callback, default_font=None):
        super().__init__(parent)

        self.passage_id = passage_id
        self.on_save_callback = on_save_callback
        self.default_font = default_font or "Arial"
        self._word_count_update_scheduled = False  # Debounce flag

        self.title(f"지문 수정: {passage_id}")
        self.geometry("600x500")
        self.transient(parent)
        self.grab_set()

        # Main frame
        main_frame = CTkFrame(self, fg_color="transparent")
        main_frame.pack(fill="both", expand=True, padx=20, pady=20)

        # Textbox (undo=True enables undo/redo functionality)
        from customtkinter import CTkTextbox
        self.text_box = CTkTextbox(main_frame, wrap="word", font=(self.default_font, 13), undo=True)
        self.text_box.pack(fill="both", expand=True, pady=(0, 10))
        self.text_box.insert("1.0", current_passage_text)

        # Reset modified flag and clear undo stack after initial insert
        self.text_box.edit_modified(False)
        self.text_box.edit_reset()  # Clear undo stack so initial text isn't undoable

        # Bind text change to update word count
        self.text_box.bind("<<Modified>>", self._on_text_modified)

        # Word count frame
        count_frame = CTkFrame(main_frame, fg_color="transparent")
        count_frame.pack(fill="x", pady=(0, 10))

        self.word_count_label = CTkLabel(
            count_frame,
            text="단어 수: 0/2000 (최소 50)",
            font=(self.default_font, 11)
        )
        self.word_count_label.pack(side="left")

        self.validation_warning = CTkLabel(
            count_frame,
            text="",
            font=(self.default_font, 11),
            text_color="red"
        )
        self.validation_warning.pack(side="left", padx=(10, 0))

        # Button frame
        button_frame = CTkFrame(main_frame, fg_color="transparent")
        button_frame.pack(fill="x")

        self.save_button = CTkButton(
            button_frame,
            text="저장",
            width=80,
            command=self._on_save,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            font=(self.default_font, 12)
        )
        self.save_button.pack(side="right", padx=(5, 0))

        cancel_button = CTkButton(
            button_frame,
            text="취소",
            width=80,
            command=self.destroy,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            font=(self.default_font, 12)
        )
        cancel_button.pack(side="right")

        # Keyboard bindings
        self.bind("<Escape>", lambda e: self.destroy())
        self.bind("<Control-Return>", lambda e: self._on_save())  # Windows
        self.bind("<Command-Return>", lambda e: self._on_save())  # macOS

        # Initialize CopyManager for Korean keyboard support (copy/paste/undo/redo)
        self.copy_manager = CopyManager(self)

        # Initial word count update
        self._update_word_count()

    def _on_text_modified(self, event=None):
        """Called when text is modified - schedules deferred word count update"""
        # Reset modified flag immediately
        self.text_box.edit_modified(False)

        # Use after_idle for performance (debounce rapid typing)
        if not self._word_count_update_scheduled:
            self._word_count_update_scheduled = True
            self.after_idle(self._deferred_word_count_update)

    def _deferred_word_count_update(self):
        """Deferred word count update (after UI thread is idle)"""
        self._word_count_update_scheduled = False
        self._update_word_count()

    def _update_word_count(self):
        """Update word count and validation status"""
        text = self.text_box.get("1.0", "end-1c").strip()

        # Tokenization: simple whitespace split
        words = [w for w in text.split() if w]
        word_count = len(words)

        # Update label
        self.word_count_label.configure(text=f"단어 수: {word_count}/2000 (최소 50)")

        # Validation
        if word_count < 50:
            self.validation_warning.configure(text=" 최소 50 단어 필요")
            self.save_button.configure(state="disabled")
        elif word_count > 2000:
            self.validation_warning.configure(text=" 최대 2000 단어 초과")
            self.save_button.configure(state="disabled")
        else:
            self.validation_warning.configure(text="")
            self.save_button.configure(state="normal")

    def _on_save(self):
        """Save edited passage"""
        edited_text = self.text_box.get("1.0", "end-1c").strip()

        if self.on_save_callback:
            self.on_save_callback(self.passage_id, edited_text)

        self.destroy()


class ManualMocktestPopup(CTkToplevel): #Mac
    def __init__(self, parent, main_frame=None, excel_file=None, count=None, default_font=None,
                 실모rows=None, 실모rows_with_hard_level=None, 문제rows_with_options=None,
                 어휘3단계_option=None, 내용일치_option=None, 내용일치한_option=None, 밑줄의미_option=None, 함축의미_option=None, 무관한문장_option=None,
                 빈칸추론_option=None, 추론_option=None, 추론불가_option=None, 순서_option=None, 삽입_option=None, 주제_option=None, 제목_option=None,
                 요지_option=None, 요약_option=None, 영작1_option=None, 영작2_option=None, 주제영작_option=None, 동형문제_option=None, 어법2단계_option=None,
                 process_excel_callback=None, selected_passages=None, restore_state_data=None,
                 option_payload_registry=None):
        super().__init__(parent)

        self.process_excel_callback = process_excel_callback
        self.selected_passages = selected_passages  # List of pre-selected passage IDs from manual selection mode

        self.title("실전모의고사 수동출제")
        # Increased width to accommodate longer 동형문제-프리셋명 display
        self.geometry("820x500")
        self.minsize(820, 500)        
    
        self.copy_question_contents = {}  # Dictionary to store copy question content per row
        self.edited_passages = {}  # Dictionary to store edited passage text per row: {row_idx: {"passage_id": ..., "text": ...}}
        self.row_option_overrides = {}  # Row-specific option overrides
        self.row_matched_presets = {}  # {row_idx: ["프리셋1", "프리셋2"]} - matched 동형문제 preset names per row
        self._pending_option_button_refresh = False  # Debounce flag for async option button refresh

        # Define column widths as class constants
        self.COL_WIDTHS = {
            "드래그": 30,
            "번호": 40,
            "지문선택": 200,
            "문제유형": 200,  # Increased from 130 to accommodate 동형문제-프리셋명
            "난이도": 100,
            "출제방식": 60,
            "옵션": 50,
            "삭제": 50
        }


        self.default_font = default_font

        self.main_frame_ref = main_frame
        self.excel_file = excel_file
        self.count = count  # initial number of rows
        self.실모rows = 실모rows
        self.실모rows_with_hard_level = 실모rows_with_hard_level

        # Use the list passed from autoQM.py instead of hardcoding
        self.문제rows_with_options = 문제rows_with_options if 문제rows_with_options is not None else []
        self.option_payload_registry = option_payload_registry or {}


        # Assign option functions
        self.어휘3단계_option = 어휘3단계_option
        self.내용일치_option = 내용일치_option
        self.내용일치한_option = 내용일치한_option
        self.밑줄의미_option = 밑줄의미_option
        self.함축의미_option = 함축의미_option
        self.무관한문장_option = 무관한문장_option
        self.빈칸추론_option = 빈칸추론_option
        self.추론_option = 추론_option
        self.추론불가_option = 추론불가_option
        self.순서_option = 순서_option
        self.삽입_option = 삽입_option
        self.주제_option = 주제_option
        self.제목_option = 제목_option
        self.요지_option = 요지_option
        self.요약_option = 요약_option
        self.영작1_option = 영작1_option
        self.영작2_option = 영작2_option
        self.주제영작_option = 주제영작_option
        self.동형문제_option = 동형문제_option
        self.어법2단계_option = 어법2단계_option

        # Load items from "대기열" sheet column A
        self.waiting_list_items = self.load_waiting_list_items()

        # Lists to store references to the variables and widgets per row
        self.selected_waiting_items = []
        self.selected_problem_types = []
        self.hardness_menus = []
        self.option_buttons = []
        self.row_widgets = []

        # Initialize file paths (must be before add_row() calls)
        self.app_data_dir = Path.home() / ".my_application_data"
        self.app_data_dir.mkdir(exist_ok=True)
        # Options file for 실전모의고사 수동출제 (isolated from global make_questions_options.json)
        self.mocktest_options_file_path = str(self.app_data_dir / "manual_mocktest_options.json")
        self.presets_file = self.app_data_dir / "manual_mocktest_presets.json"
        self.last_state_file = self.app_data_dir / "manual_mocktest_last_state.json"
        self.copyquestion_presets_file = self.app_data_dir / "copyquestion_presets.json"

        main_frame = CTkFrame(self, fg_color="transparent")
        main_frame.pack(fill="both", expand=True, padx=10, pady=10)
        main_frame.grid_rowconfigure(1, weight=1)
        main_frame.grid_columnconfigure(0, weight=1)

        # Create header labels
        header_frame = CTkFrame(main_frame, fg_color="transparent")
        header_frame.grid(row=0, column=0, sticky="ew", padx=(5, 5))

        # Configure columns with exact widths (matching scrollable_frame)
        header_frame.grid_columnconfigure(0, weight=0, minsize=self.COL_WIDTHS["드래그"])
        header_frame.grid_columnconfigure(1, weight=0, minsize=self.COL_WIDTHS["번호"])
        header_frame.grid_columnconfigure(2, weight=0, minsize=self.COL_WIDTHS["지문선택"])
        header_frame.grid_columnconfigure(3, weight=0, minsize=self.COL_WIDTHS["문제유형"])
        header_frame.grid_columnconfigure(4, weight=0, minsize=self.COL_WIDTHS["난이도"])
        header_frame.grid_columnconfigure(5, weight=0, minsize=self.COL_WIDTHS["출제방식"])
        header_frame.grid_columnconfigure(6, weight=0, minsize=self.COL_WIDTHS["옵션"])
        header_frame.grid_columnconfigure(7, weight=0, minsize=self.COL_WIDTHS["삭제"])


        # Header Labels (complete section)
        # Empty header for drag handle column
        CTkLabel(header_frame, width=self.COL_WIDTHS["드래그"], height=18,
                text="", font=(self.default_font, 11),
                fg_color="transparent", text_color=("black", "white"),
                anchor='center'
                ).grid(row=0, column=0, padx=(0, 5), pady=(5, 0))

        CTkLabel(header_frame, width=self.COL_WIDTHS["번호"], height=18,
                text="번호", font=(self.default_font, 11),
                fg_color="transparent", text_color=("black", "white"),
                anchor='center'
                ).grid(row=0, column=1, padx=(0, 5), pady=(5, 0))

        # Create frame to hold "대기열에서 지문 선택" label and "랜덤" button
        passage_header_frame = CTkFrame(header_frame, fg_color="transparent")
        passage_header_frame.grid(row=0, column=2, padx=(0, 5), pady=(5, 0))

        CTkLabel(passage_header_frame,
                text="대기열에서 지문 선택", font=(self.default_font, 11),
                fg_color="transparent", text_color=("black", "white")
                ).pack(side='left', padx=(0, 5))

        CTkButton(passage_header_frame,
                text="랜덤",
                width=40, height=18,
                fg_color="#FEF9E0",
                hover_color="#DDA15C",
                text_color='black',
                font=(self.default_font, 9, 'bold'),
                command=self.randomize_waiting_items
                ).pack(side='left')

        # Create frame to hold "문제유형 선택" label and "랜덤" button
        problem_type_header_frame = CTkFrame(header_frame, fg_color="transparent")
        problem_type_header_frame.grid(row=0, column=3, padx=(0, 5), pady=(5, 0))

        CTkLabel(problem_type_header_frame,
                text="문제유형 선택", font=(self.default_font, 11),
                fg_color="transparent", text_color=("black", "white")
                ).pack(side='left', padx=(15, 5))

        CTkButton(problem_type_header_frame,
                text="랜덤",
                width=40, height=18,
                fg_color="#FEF9E0",
                hover_color="#DDA15C",
                text_color='black',
                font=(self.default_font, 9, 'bold'),
                command=self.randomize_problem_types
                ).pack(side='left')

        # Create frame to hold "난이도 선택" label and "All" button
        hardness_header_frame = CTkFrame(header_frame, fg_color="transparent")
        hardness_header_frame.grid(row=0, column=4, padx=(0, 5), pady=(5, 0))

        CTkLabel(hardness_header_frame,
                text="난이도 선택", font=(self.default_font, 11),
                fg_color="transparent", text_color=("black", "white")
                ).pack(side='left', padx=(20, 5))

        CTkButton(hardness_header_frame,
                text="All",
                width=35, height=18,
                fg_color="#FEF9E0",
                hover_color="#DDA15C",
                text_color='black',
                font=(self.default_font, 9, 'bold'),
                command=self.set_all_hard
                ).pack(side='left')

        CTkLabel(header_frame, width=self.COL_WIDTHS["출제방식"], height=18,
                text="출제방식", font=(self.default_font, 11),
                fg_color="transparent", text_color=("black", "white"),
                anchor='center'
                ).grid(row=0, column=5, padx=(0, 5), pady=(5, 0))

        CTkLabel(header_frame, width=self.COL_WIDTHS["옵션"], height=18,
                text="옵션", font=(self.default_font, 11),
                fg_color="transparent", text_color=("black", "white"),
                anchor='center'
                ).grid(row=0, column=6, padx=(0, 5), pady=(5, 0))

        CTkLabel(header_frame, width=self.COL_WIDTHS["삭제"], height=18,
                text="삭제", font=(self.default_font, 11),
                fg_color="transparent", text_color=("black", "white"),
                anchor='center'
                ).grid(row=0, column=7, padx=(0, 0), pady=(5, 0))

        # Create the scrollable frame
        # Create the scrollable frame with matching column configurations
        self.scrollable_frame = CTkScrollableFrame(main_frame, fg_color="transparent")
        self.scrollable_frame.grid(row=1, column=0, sticky='nsew', padx=(5, 5), pady=0)

        self.scrollable_frame.grid_columnconfigure(0, weight=0, minsize=self.COL_WIDTHS["드래그"])
        self.scrollable_frame.grid_columnconfigure(1, weight=0, minsize=self.COL_WIDTHS["번호"])
        self.scrollable_frame.grid_columnconfigure(2, weight=0, minsize=self.COL_WIDTHS["지문선택"])
        self.scrollable_frame.grid_columnconfigure(3, weight=0, minsize=self.COL_WIDTHS["문제유형"])
        self.scrollable_frame.grid_columnconfigure(4, weight=0, minsize=self.COL_WIDTHS["난이도"])
        self.scrollable_frame.grid_columnconfigure(5, weight=0, minsize=self.COL_WIDTHS["출제방식"])
        self.scrollable_frame.grid_columnconfigure(6, weight=0, minsize=self.COL_WIDTHS["옵션"])
        self.scrollable_frame.grid_columnconfigure(7, weight=0, minsize=self.COL_WIDTHS["삭제"])

        # Initialize drag manager
        self.init_drag_manager()

        # Define plus_button first, but don't place it yet
        self.plus_button = CTkButton(
            self.scrollable_frame,
            text="+",
            width=40,
            height=25,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.default_font, 13, 'bold'),
            command=self.add_row
        )
                
        # Now add the initial rows
        for i in range(self.count):
            self.add_row()

        # If restore_state_data is provided, populate with saved values
        if restore_state_data and "rows" in restore_state_data:
            saved_rows = restore_state_data["rows"]

            # First pass: Drop saved strings into self.copy_question_contents BEFORE triggering UI updates
            for i, row_state in enumerate(saved_rows):
                if i >= len(self.row_widgets):
                    break

                rd = self.row_widgets[i]
                row_idx = rd['row_idx']

                # Restore copy content into internal dict first (before UI updates so tooltips can pick them up)
                # Check only if key exists, not truthiness (to handle intentionally blank strings)
                if "copy_content" in row_state:
                    self.copy_question_contents[row_idx] = row_state["copy_content"]

            # Second pass: Set fields and trigger handlers so dependent widgets refresh
            for i, row_state in enumerate(saved_rows):
                if i >= len(self.row_widgets):
                    break

                rd = self.row_widgets[i]
                row_idx = rd['row_idx']

                # Set the variables
                waiting_selection = row_state.get("waiting_selection", "지문을 선택하세요")
                problem_type = row_state.get("problem_type", self.실모rows[0] if self.실모rows else "")
                hardness = row_state.get("hardness", "Normal")

                rd['waiting_var'].set(waiting_selection)
                rd['problem_var'].set(problem_type)
                rd['hardness_var'].set(hardness)

                # Trigger normal handlers so dependent widgets refresh
                self.on_waiting_type_change(rd, waiting_selection)
                self.on_problem_type_change(rd, problem_type)

            option_override_list = restore_state_data.get("option_overrides")
            self._apply_option_overrides_from_list(option_override_list)

            # Restore edited passages and refresh UI (convert string keys back to int)
            edited_passages_data = restore_state_data.get("edited_passages", {})
            if edited_passages_data:
                # JSON converts int keys to strings, convert back
                for k, v in edited_passages_data.items():
                    self.edited_passages[int(k)] = v
                # Queue UI updates to avoid touching destroyed widgets
                self.after(10, lambda: self._refresh_edited_passage_ui())

            # After all rows restored, update option button colors once
            self.update_all_option_button_colors()

            # Rebuild self.selected_passages and sync self.count
            self.selected_passages = [rd['waiting_var'].get() for rd in self.row_widgets]
            self.count = len(self.row_widgets)

            # Register drag handles for restored rows
            for idx, rd in enumerate(self.row_widgets):
                self.drag_manager.register_handle(rd['drag_handle'], idx)

        # After adding initial rows (or restoring), place the plus_button
        self.plus_button.grid(row=self.count+1, column=0, columnspan=8, pady=10)


        # Bottom frame
        button_frame = CTkFrame(self, fg_color="#2B3D2D")
        button_frame.pack(side='bottom', fill='x', pady=(5, 0))

        # Create two lines
        top_line_frame = CTkFrame(button_frame, fg_color="transparent")
        top_line_frame.pack(side='top', fill='x', pady=(5, 5))

        bottom_line_frame = CTkFrame(button_frame, fg_color="transparent")
        bottom_line_frame.pack(side='top', fill='x', pady=(0, 5))

        # === PRESET SAVE/LOAD UI on the top line ===
        preset_controls_frame = CTkFrame(top_line_frame, fg_color="transparent")
        preset_controls_frame.pack(side=tk.LEFT, fill='x', expand=True, padx=(10, 10), pady=0)

        CTkLabel(preset_controls_frame, text="문제유형 불러오기:", font=(self.default_font, 12), text_color=("white", "white")).pack(side=tk.LEFT, padx=(0, 5))

        self.preset_combobox = customtkinter.CTkComboBox(
            preset_controls_frame,
            font=(self.default_font, 12),
            command=self.on_preset_selected,
            state='readonly',
            width=180
        )
        self.preset_combobox.pack(side=tk.LEFT, padx=5)
        self.preset_combobox.set("불러올 프리셋 선택")

        delete_preset_button = CTkButton(
            preset_controls_frame,
            text="삭제",
            command=self.delete_preset,
            width=40,
            height=25,
            fg_color="white",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.default_font, 12, 'bold')
        )
        delete_preset_button.pack(side=tk.LEFT, padx=5)



        save_preset_button = CTkButton(
            preset_controls_frame,
            text="저장",
            command=self.save_preset,
            width=40,
            height=25,
            fg_color="white",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.default_font, 12, 'bold')
        )
        save_preset_button.pack(side=tk.RIGHT, padx=5)

        self.preset_name_entry = CTkEntry(
            preset_controls_frame,
            width=180,
            placeholder_text="저장할 문제유형 프리셋 이름",
            font=(self.default_font, 12),
            text_color="gray"
        )
        self.preset_name_entry.pack(side=tk.RIGHT, padx=5)


        # === Additional Buttons on the bottom line ===
        apply_first_type_button = CTkButton(
            bottom_line_frame, 
            text="1번 유형 전체 적용", 
            width=110, height=25, 
            fg_color="#FEF9E0", 
            hover_color="#DDA15C", 
            text_color='black', 
            font=(self.default_font, 11, 'bold'), 
            command=self.apply_first_type_to_all
        )
        apply_first_type_button.pack(side='left', padx=10, pady=5)

        start_button = CTkButton(
            bottom_line_frame, 
            text="출제시작", 
            width=70, 
            height=25, 
            fg_color="#FEF9E0", 
            hover_color="#DDA15C", 
            text_color='black', 
            font=(self.default_font, 13, 'bold'), 
            command=self.start
        )
        start_button.pack(side='right', padx=10, pady=5)

        cancel_button = CTkButton(
            bottom_line_frame,
            text="취소", 
            width=70, 
            height=25, 
            fg_color="#FEF9E0", 
            hover_color="#DDA15C", 
            text_color='black', 
            font=(self.default_font, 13, 'bold'), 
            command=self.cancel
        )
        cancel_button.pack(side='right', padx=10, pady=5)

        # Load presets (file paths already initialized earlier)
        self.load_presets()

        self.bind("<Escape>", self.cancel)
        self.protocol("WM_DELETE_WINDOW", self.cancel)



    def add_row(self):
        row_idx = len(self.row_widgets) + 1  # rows start at 1

        # Determine default values based on the previous row if exists
        if self.row_widgets:
            last_row_data = self.row_widgets[-1]
            default_problem_type = last_row_data['problem_var'].get()
            default_hardness = last_row_data['hardness_var'].get()
        else:
            default_problem_type = self.실모rows[0] if self.실모rows else ""
            default_hardness = "Normal"




        # Column 0: Drag Handle
        drag_handle = CTkLabel(
            self.scrollable_frame,
            text="≡",
            width=self.COL_WIDTHS["드래그"],
            height=19,
            font=(self.default_font, 14, 'bold'),
            text_color="#666666",
            fg_color="transparent",
            cursor="sb_v_double_arrow"
        )
        drag_handle.grid(row=row_idx, column=0, sticky='ew', padx=(0, 5), pady=3)

        # Column 1: Row number
        index_label = CTkLabel(self.scrollable_frame,
                            text=str(row_idx),
                            font=(self.default_font, 10, "bold"),
                            width=self.COL_WIDTHS["번호"],
                            height=19,
                            corner_radius=5,
                            text_color=("black", "white"),
                            fg_color="#B6C2B7")
        index_label.grid(row=row_idx, column=1, sticky='ew', padx=(0, 5), pady=3)

        # Column 2: 대기열에서 지문 선택
        waiting_var = tk.StringVar()

        # If selected_passages provided (manual selection mode), use those passages
        if self.selected_passages and row_idx-1 < len(self.selected_passages):
            waiting_var.set(self.selected_passages[row_idx-1])
        # Otherwise, use default sequential logic from waiting_list_items
        elif row_idx-1 < len(self.waiting_list_items):
            waiting_var.set(self.waiting_list_items[row_idx-1])
        else:
            waiting_var.set("지문을 선택하세요")

        all_values_for_waiting = self.waiting_list_items if self.waiting_list_items else [""]
        waiting_menu = CustomDropdown(
            self.scrollable_frame,
            values=all_values_for_waiting,
            variable=waiting_var,
            get_selected_items_func=lambda: [rd['waiting_var'].get() for rd in self.row_widgets
                                           if rd['waiting_var'] is not waiting_var],
            enable_edit_button=True,
            on_edit_callback=lambda passage_id, ridx=row_idx: self._open_passage_editor(ridx, passage_id),
            get_edited_status_func=lambda passage_id, ridx=row_idx: (
                ridx in self.edited_passages and
                self.edited_passages[ridx].get("passage_id") == passage_id
            ),
            width=self.COL_WIDTHS["지문선택"],
            height=30,
            bg_color='transparent',
            button_color='white',
            fg_color='white',
            text_color=("#1F2937", "#1F2937"),
            button_hover_color='#F3F4F6',
            dropdown_hover_color='#CA7900',
            font=(self.default_font, 11)
        )
        waiting_menu.grid(row=row_idx, column=2, sticky='ew', padx=(0, 5), pady=3)

        # Column 3: 문제유형 선택
        problem_var = tk.StringVar(value=default_problem_type)
        problem_menu = CTkOptionMenu(
            self.scrollable_frame,
            variable=problem_var,
            values=self.실모rows,
            width=self.COL_WIDTHS["문제유형"],
            height=30,
            bg_color='transparent',
            button_color='#FEF9E0',
            fg_color='white',
            text_color='black',
            hover=True,
            button_hover_color='#CA7900',
            dropdown_hover_color='#CA7900',
            font=(self.default_font, 11),
            dynamic_resizing=False,
        )
        problem_menu.grid(row=row_idx, column=3, sticky='ew', padx=(0, 5), pady=3)

        # Column 4: 난이도 선택
        hardness_var = tk.StringVar(value=default_hardness)
        hardness_values = ["Normal"]
        if problem_var.get() in self.실모rows_with_hard_level:
            hardness_values = ["Normal", "Hard"]
            if default_hardness not in hardness_values:
                hardness_var.set("Normal")
        else:
            hardness_var.set("Normal")

        hardness_menu = CTkOptionMenu(
            self.scrollable_frame,
            variable=hardness_var,
            values=hardness_values,
            width=self.COL_WIDTHS["난이도"],
            height=30,
            bg_color='transparent',
            button_color='#FEF9E0',
            fg_color='white',
            text_color='black',
            hover=True,
            button_hover_color='#CA7900',
            dropdown_hover_color='#CA7900',
            font=(self.default_font, 11),
            dynamic_resizing=False
        )
        hardness_menu.grid(row=row_idx, column=4, sticky='ew', padx=(0, 5), pady=0)

        # Column 5: 출제방식
        current_status_label = CTkLabel(self.scrollable_frame,
                                    width=self.COL_WIDTHS["출제방식"],
                                    font=(self.default_font, 11),
                                    fg_color="transparent")
        current_status_label.grid(row=row_idx, column=5, sticky='ew', padx=(0, 5), pady=3)

        # Column 6: 옵션 버튼
        option_button = CTkButton(
            self.scrollable_frame,
            text="옵션",
            width=self.COL_WIDTHS["옵션"],
            height=15,
            font=(self.default_font, 11),
            fg_color="white",
            hover_color="#DDA15E",
            text_color="black"
        )
        option_button.grid(row=row_idx, column=6, sticky='ew', padx=(0, 5), pady=0)

        # Column 7: 삭제 버튼
        delete_button = CTkButton(
            self.scrollable_frame,
            text="-",
            width=self.COL_WIDTHS["삭제"],
            height=25,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color='black',
            font=(self.default_font, 13, 'bold'),
            command=lambda: self.delete_row(row_data)
        )
        delete_button.grid(row=row_idx, column=7, sticky='ew', padx=(0, 0), pady=3)


        # Create tooltip with passage content (edited or original)
        initial_title = waiting_var.get()
        # Check if this specific row has an edit for the current passage
        if row_idx in self.edited_passages and self.edited_passages[row_idx].get("passage_id") == initial_title:
            initial_passage = self.edited_passages[row_idx]["text"]
        else:
            initial_passage = self.title_to_passage_map.get(initial_title, "지문을 선택해주세요")

        # Create tooltip with full passage
        tooltip = ToolTip(index_label, initial_passage, max_width=500)  # You can adjust max_width

        # Create tooltip for problem_menu to show 동형문제 content
        # Only shows when "동형문제" is selected and content exists
        problem_menu_tooltip = ToolTip(problem_menu, "", max_width=600)


        # Truncate passage if too long for tooltip (optional)
        #tooltip_text = initial_passage[:500] + "..." if len(initial_passage) > 500 else initial_passage
        #tooltip = ToolTip(index_label, tooltip_text)






        # Rest of the method remains the same
        row_data = {
            'row_idx': row_idx,
            'drag_handle': drag_handle,
            'index_label': index_label,
            'waiting_menu': waiting_menu,
            'problem_menu': problem_menu,
            'hardness_menu': hardness_menu,
            'option_button': option_button,
            'delete_button': delete_button,
            'waiting_var': waiting_var,
            'problem_var': problem_var,
            'hardness_var': hardness_var,
            'current_status_label': current_status_label,
            'tooltip': tooltip,  # Tooltip for passage on number label
            'problem_menu_tooltip': problem_menu_tooltip,  # Tooltip for problem type menu

        }

        waiting_menu.configure(command=lambda selected, rd=row_data: self.on_waiting_type_change(rd, selected))
        problem_menu.configure(command=lambda selected, rd=row_data: self.on_problem_type_selected(rd, selected))
        hardness_menu.configure(command=lambda value, rd=row_data: self.on_hardness_change(rd, value))

        # Update callbacks to use row_data['row_idx'] (dynamic) instead of captured row_idx (stale after reordering)
        waiting_menu.on_edit_callback = lambda passage_id, rd=row_data: self._open_passage_editor(rd['row_idx'], passage_id)
        waiting_menu.get_edited_status_func = lambda passage_id, rd=row_data: (
            rd['row_idx'] in self.edited_passages and
            self.edited_passages[rd['row_idx']].get("passage_id") == passage_id
        )


        self.row_widgets.append(row_data)

        # Register drag handle with drag manager
        self.drag_manager.register_handle(drag_handle, row_idx - 1)  # 0-based index

        self.configure_option_button_for_row(row_data, problem_var.get())

        if hardness_values == ["Normal"]:
            hardness_menu.set("Normal")
            hardness_menu.configure(state="disabled")

        self.on_problem_type_change(row_data, problem_var.get())

        passage_id유형난이도합본 = row_data['waiting_menu'].get() + ' // ' + row_data['problem_menu'].get() + '_' + row_data['hardness_menu'].get()
        self.update_current_status_label수동실모(row_idx, passage_id유형난이도합본)

        self.count += 1
        self.plus_button.grid_configure(row=self.count+1, columnspan=8)

        self.update_row_indices()
        self.refresh_waiting_dropdowns()

        # Trigger initial edit button visibility check for pre-selected passages
        waiting_menu.refresh_display()

        # Initialize option tooltip if content already exists for this row
        if row_idx in self.copy_question_contents:
            self.update_option_tooltip(row_idx, self.copy_question_contents[row_idx])

        # Update option button color based on saved options

        self.update_option_button_color_for_row(row_data)



    def _normalize_storage_key(self, question_type: str) -> str:
        if not question_type:
            return ""
        normalized = unicodedata.normalize("NFC", question_type).strip()
        return normalized.replace("(", "").replace(")", "")

    def _read_options_file(self) -> dict:
        """Read options from mocktest-specific JSON file"""
        mocktest_path = Path(self.mocktest_options_file_path)
        if mocktest_path.exists():
            try:
                with open(mocktest_path, 'r', encoding='utf-8') as f:
                    return json.load(f)
            except (json.JSONDecodeError, OSError):
                return {}
        return {}

    def _write_options_file(self, data: dict):
        """Write options to mocktest-specific JSON file"""
        try:
            with open(self.mocktest_options_file_path, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=4)
        except OSError as e:
            print(f" Failed to write manual_mocktest_options.json: {e}")

    def _get_keys_of_interest(self, storage_key: str) -> set:
        keys = set()
        for json_key, _ in self.option_payload_registry.get(storage_key, []):
            keys.add(json_key)
        keys.update({
            f"{storage_key}paraphrase_enabled",
            f"{storage_key}difficulty_level",
            f"{storage_key}length_level",
            f"{storage_key}content_change_mode"
        })
        return keys

    def _prepare_option_dialog(self, row_idx: int, problem_type: str):
        storage_key = self._normalize_storage_key(problem_type)
        if not storage_key:
            return None

        baseline_data = self._read_options_file()
        temp_data = copy.deepcopy(baseline_data)

        override_entry = self.row_option_overrides.get(row_idx)
        if override_entry and override_entry.get('storage_key') == storage_key:
            for key, value in override_entry.get('raw', {}).items():
                temp_data[key] = value

        self._write_options_file(temp_data)

        return {
            "row_idx": row_idx,
            "storage_key": storage_key,
            "baseline": baseline_data
        }

    def _restore_options_baseline(self, context: dict):
        if not context:
            return
        baseline = context.get("baseline")
        if baseline is not None:
            self._write_options_file(baseline)

    def _handle_option_popup_destroy(self, popup):
        context = getattr(popup, "_manual_option_context", None)
        if not context:
            return

        row_idx = context.get("row_idx")
        storage_key = context.get("storage_key")
        baseline = context.get("baseline", {})

        current_data = self._read_options_file()
        keys_of_interest = self._get_keys_of_interest(storage_key)

        raw_override = {}

        for key in keys_of_interest:
            normalized_key = self._normalize_storage_key(key)
            storage_variant = f"{storage_key}{key}" if not key.startswith(storage_key) else key
            current_value = current_data.get(storage_variant)
            if current_value is None and normalized_key != storage_key:
                current_value = current_data.get(key)
            baseline_value = baseline.get(storage_variant)
            if baseline_value is None and normalized_key != storage_key:
                baseline_value = baseline.get(key)

            if current_value != baseline_value:
                raw_override[storage_variant] = current_value

        if raw_override:
            self.row_option_overrides[row_idx] = {
                "storage_key": storage_key,
                "raw": raw_override
            }
        else:
            if row_idx in self.row_option_overrides:
                self.row_option_overrides.pop(row_idx, None)

        self._write_options_file(baseline)

        row_data = self._get_row_data_by_idx(row_idx)
        if row_data:
            self.update_option_button_color_for_row(row_data)
        else:
            self.update_all_option_button_colors()

        # Clean up context reference
        if hasattr(popup, "_manual_option_context"):
            delattr(popup, "_manual_option_context")

    def _get_row_data_by_idx(self, row_idx: int):
        for rd in self.row_widgets:
            if rd.get('row_idx') == row_idx:
                return rd
        return None

    def _remap_row_dependent_dicts(self, index_map: dict):
        if self.copy_question_contents:
            new_copy_contents = {}
            for old_idx, value in self.copy_question_contents.items():
                new_idx = index_map.get(old_idx)
                if new_idx is not None:
                    new_copy_contents[new_idx] = value
            self.copy_question_contents = new_copy_contents

        if self.row_option_overrides:
            new_overrides = {}
            for old_idx, entry in self.row_option_overrides.items():
                new_idx = index_map.get(old_idx)
                if new_idx is not None:
                    new_overrides[new_idx] = entry
            self.row_option_overrides = new_overrides

        if self.row_matched_presets:
            new_matched_presets = {}
            for old_idx, presets_list in self.row_matched_presets.items():
                new_idx = index_map.get(old_idx)
                if new_idx is not None:
                    new_matched_presets[new_idx] = presets_list
            self.row_matched_presets = new_matched_presets

        if self.edited_passages:
            new_edited_passages = {}
            for old_idx, edit_data in self.edited_passages.items():
                new_idx = index_map.get(old_idx)
                if new_idx is not None:
                    new_edited_passages[new_idx] = edit_data
            self.edited_passages = new_edited_passages

    def _build_option_override_payload(self):
        payload = []
        for rd in self.row_widgets:
            row_idx = rd['row_idx']
            entry = self.row_option_overrides.get(row_idx)
            if entry:
                payload.append(copy.deepcopy(entry))
            else:
                payload.append(None)
        return payload

    def _apply_option_overrides_from_list(self, overrides_list):
        self.row_option_overrides.clear()
        if not overrides_list:
            return
        for rd, entry in zip(self.row_widgets, overrides_list):
            if not entry:
                continue
            storage_key = self._normalize_storage_key(entry.get('storage_key', ''))
            raw = entry.get('raw', {})
            if storage_key and isinstance(raw, dict):
                self.row_option_overrides[rd['row_idx']] = {
                    "storage_key": storage_key,
                    "raw": copy.deepcopy(raw)
                }

    def _clear_override_if_type_changed(self, row_idx: int, storage_key: str):
        entry = self.row_option_overrides.get(row_idx)
        if entry and entry.get('storage_key') != storage_key:
            self.row_option_overrides.pop(row_idx, None)

    def update_current_status_label수동실모(self, row_idx=None, passage_id유형난이도합본=None):
        #print("\n\nupdate_current_status_label수동실모 called in ManualMockTest.py")
        #print(self.main_frame_ref.manual_edited_texts)
        #print(f"passage_id유형난이도합본: {passage_id유형난이도합본}\n\n")

        # If no specific row_idx is provided, update all rows
        if row_idx is None:
            for row_data in self.row_widgets:
                
                current_status_label = row_data['current_status_label']
                current_passage_id = row_data['waiting_menu'].get() + ' // ' + row_data['problem_menu'].get() + '_' + row_data['hardness_menu'].get()

                if current_passage_id in self.main_frame_ref.manual_edited_texts:
                    current_status_label.configure(text="수동", text_color="red")
                else:
                    current_status_label.configure(text="자동", text_color=("black", "white"))
        else:
            # Original code for specific row_idx
            for row_data in self.row_widgets:
                if row_data['row_idx'] == row_idx:
                    current_status_label = row_data['current_status_label']

                    if passage_id유형난이도합본 in self.main_frame_ref.manual_edited_texts:
                        current_status_label.configure(text="수동", text_color="red")
                    else:
                        current_status_label.configure(text="자동", text_color=("black", "white"))
                    break

        
    def configure_option_button_for_row(self, row_data, problem_type):
        """
        Enable/disable option button based on whether the problem type:
        1. Is in self.문제rows_with_options list (passed from autoQM.py)
        2. Has a valid option handler in get_option_command()

        If either condition fails, the button will be disabled.
        """
        option_button = row_data['option_button']
        if problem_type in self.문제rows_with_options:
            command = self.get_option_command(row_data)
            if command:
                option_button.configure(state="normal", command=command)
            else:
                option_button.configure(state="disabled", command=lambda: None)
        else:
            option_button.configure(state="disabled", command=lambda: None)
    

    """def configure_option_button_for_row(self, row_data, problem_type):
        option_button = row_data['option_button']
        # Just always enable, or enable/disable if necessary:
        option_button.configure(state="normal", command=self.get_option_command(row_data))
    """



    def update_row_indices(self):
        old_to_new = {}
        for i, rd in enumerate(self.row_widgets, start=1):
            old_idx = rd.get('row_idx', i)
            old_to_new[old_idx] = i
            rd['row_idx'] = i  # Update the stored row index
            rd['index_label'].configure(text=str(i))  # Update the displayed label
        if old_to_new:
            self._remap_row_dependent_dicts(old_to_new)

    def refresh_waiting_dropdowns(self):
        """Refresh duplicate highlighting across all custom dropdowns."""
        for rd in self.row_widgets:
            dropdown = rd.get('waiting_menu')
            if hasattr(dropdown, "refresh_item_styles"):
                dropdown.refresh_item_styles()


    def extract_question_from_sample(self, content):
        """Extract only the [Question] part from saved 동형문제 sample"""
        if not content:
            return ""

        # Find the [Question] section
        try:
            # Find where [Question] starts
            question_start = content.find("[Question]")
            if question_start == -1:
                return content  # Return full content if no [Question] tag found

            # Find where the next section starts (e.g., [Answer])
            question_end = content.find("[Answer]", question_start)
            if question_end == -1:
                # If no [Answer] found, take everything after [Question]
                question_part = content[question_start:]
            else:
                # Extract content between [Question] and [Answer]
                question_part = content[question_start:question_end]

            # Remove the [Question] tag and strip whitespace
            question_text = question_part.replace("[Question]", "").strip()
            return question_text
        except Exception:
            # If any error occurs, return the original content
            return content

    def _load_copyquestion_presets(self) -> dict:
        """Load presets from copyquestion_presets.json"""
        if self.copyquestion_presets_file.exists():
            try:
                with open(self.copyquestion_presets_file, 'r', encoding='utf-8') as f:
                    return json.load(f)
            except (json.JSONDecodeError, OSError):
                return {}
        return {}

    def _extract_fields_from_sample(self, content: str):
        """Extract (지문, 문제, 정답) from sample content string.

        Returns tuple of (passage, question, answer) or None if parsing fails.
        The content format is:
        [Passage]
        {passage text}
        [Question]
        {question text}
        [Answer]
        {answer text}
        [Explanation]
        {explanation text}
        [Transform]
        ...
        """
        if not content:
            return None

        try:
            # Extract [Passage] section
            passage_start = content.find("[Passage]")
            question_start = content.find("[Question]")
            answer_start = content.find("[Answer]")
            explanation_start = content.find("[Explanation]")

            if passage_start == -1 or question_start == -1 or answer_start == -1:
                return None

            # Extract passage (between [Passage] and [Question])
            passage = content[passage_start + len("[Passage]"):question_start].strip()

            # Extract question (between [Question] and [Answer])
            question = content[question_start + len("[Question]"):answer_start].strip()

            # Extract answer (between [Answer] and [Explanation] or end)
            if explanation_start != -1:
                answer = content[answer_start + len("[Answer]"):explanation_start].strip()
            else:
                # If no [Explanation], take until [Transform] or end
                transform_start = content.find("[Transform]")
                if transform_start != -1:
                    answer = content[answer_start + len("[Answer]"):transform_start].strip()
                else:
                    answer = content[answer_start + len("[Answer]"):].strip()

            return (passage, question, answer)
        except Exception:
            return None

    def _find_matching_presets(self, passage: str, question: str, answer: str) -> list:
        """Find all presets that match the given (passage, question, answer).

        Compares against presets in copyquestion_presets.json.
        Returns list of matching preset names.
        """
        presets_data = self._load_copyquestion_presets()
        matches = []

        categories = presets_data.get("categories", {})
        for cat_name, cat_data in categories.items():
            if not isinstance(cat_data, dict):
                continue
            for preset_name, preset_data in cat_data.items():
                # Skip metadata keys like "_order"
                if preset_name.startswith("_"):
                    continue
                if not isinstance(preset_data, dict):
                    continue

                # Get preset fields
                p_passage = preset_data.get("지문", "").strip()
                p_question = preset_data.get("문제", "").strip()
                p_answer = preset_data.get("정답", "").strip()

                # Compare (excluding 해설)
                if (passage == p_passage and
                    question == p_question and
                    answer == p_answer):
                    matches.append(preset_name)

        return matches

    def _update_problem_menu_display(self, row_data, matched_presets: list):
        """Update the problem_menu to show matched preset names.

        Displays '동형문제-프리셋1/프리셋2' if matches found,
        otherwise just '동형문제'.
        """
        problem_menu = row_data['problem_menu']
        row_idx = row_data['row_idx']

        # Only update display if current type is 동형문제
        if row_data['problem_var'].get() != "동형문제":
            self.row_matched_presets.pop(row_idx, None)
            return

        if matched_presets:
            # Store for later reference
            self.row_matched_presets[row_idx] = matched_presets

            # Format display text
            preset_str = "/".join(matched_presets)
            display_text = f"동형문제-{preset_str}"

            # Update the displayed text using internal label
            # CTkOptionMenu uses _text_label for display
            if hasattr(problem_menu, '_text_label'):
                problem_menu._text_label.configure(text=display_text)
        else:
            self.row_matched_presets.pop(row_idx, None)
            # Reset to just "동형문제"
            if hasattr(problem_menu, '_text_label'):
                problem_menu._text_label.configure(text="동형문제")

    def update_option_tooltip(self, row_idx, content):
        """Update the problem menu tooltip for a specific row when content is saved.

        Also triggers preset matching to display matched preset names in the dropdown.
        """
        for rd in self.row_widgets:
            if rd['row_idx'] == row_idx:
                # Update problem menu tooltip and text color (only if 동형문제 is selected)
                if 'problem_menu_tooltip' in rd and rd['problem_menu_tooltip']:
                    problem_type = rd['problem_var'].get()
                    if problem_type == "동형문제":
                        if content:
                            # Extract and show only the [Question] part
                            question_only = self.extract_question_from_sample(content)
                            rd['problem_menu_tooltip'].update_text(question_only)
                            # Set text color to normal (black)
                            rd['problem_menu'].configure(text_color="black")

                            # Check for matching presets and update display
                            fields = self._extract_fields_from_sample(content)
                            if fields:
                                passage, question, answer = fields
                                matched_presets = self._find_matching_presets(passage, question, answer)
                                self._update_problem_menu_display(rd, matched_presets)
                            else:
                                # No valid fields, reset display
                                self._update_problem_menu_display(rd, [])
                        else:
                            # Show instruction message when no content
                            rd['problem_menu_tooltip'].update_text("'옵션'을 클릭하여 동형문제 샘플을 입력하세요")
                            # Set text color to red to indicate missing content
                            rd['problem_menu'].configure(text_color="red")
                            # Clear preset display
                            self._update_problem_menu_display(rd, [])
                    else:
                        # Clear tooltip for non-동형문제
                        rd['problem_menu_tooltip'].update_text("")
                        # Reset text color to normal (black)
                        rd['problem_menu'].configure(text_color="black")
                        # Clear preset display
                        self._update_problem_menu_display(rd, [])
                break

    def _get_baseline_values_for_type(self, question_type):
        """Get hardcoded default values for a question type (mirrors dashboard_option.py logic)"""
        # Type categories
        TYPES_WITH_COUNTS = {"빈칸추론", "내용일치", "내용일치(한)", "추론", "추론불가"}
        TYPES_WITH_BLANK_MODE = {"영작1", "영작2"}
        COUNT_DEFAULTS = {
            "빈칸추론": {"normal_min": 1, "normal_max": 4, "hard_min": 5, "hard_max": 10},
            "내용일치": {"normal_min": 10, "normal_max": 20, "hard_min": 15, "hard_max": 25},
            "내용일치(한)": {"normal_min": 10, "normal_max": 20, "hard_min": 15, "hard_max": 25},
            "추론": {"normal_min": 10, "normal_max": 15, "hard_min": 17, "hard_max": 22},
            "추론불가": {"normal_min": 10, "normal_max": 15, "hard_min": 17, "hard_max": 22},
        }

        baseline = {
            "paraphrase_enabled": False,
            "difficulty_level": 2,
            "length_level": 2,
            "content_change_mode": 0,
        }

        CONTENT_MATCH_RANDOM_TYPES = {"내용일치", "내용일치(한)", "내용일치한"}

        if question_type in TYPES_WITH_COUNTS:
            defaults = COUNT_DEFAULTS.get(
                question_type,
                {"normal_min": 1, "normal_max": 5, "hard_min": 0, "hard_max": 3},
            )
            # 내용일치/내용일치(한)의 기본값은 "랜덤(1~4개)" = mode 0
            default_mode = 0 if question_type in CONTENT_MATCH_RANDOM_TYPES else 1
            baseline.update({
                "normal_min_val": defaults["normal_min"],
                "normal_max_val": defaults["normal_max"],
                "hard_min_val": defaults["hard_min"],
                "hard_max_val": defaults["hard_max"],
                "mode": default_mode,
            })
        elif question_type in {"요지"}:
            baseline["mode"] = 1

        if question_type == "주제영작":
            baseline["mode"] = 1
            baseline["number_of_words"] = 10

        if question_type in TYPES_WITH_BLANK_MODE:
            baseline["blank_mode"] = 0
            baseline["questions_per_passage"] = 1

        if question_type == "단어정리":
            baseline["vocab_level"] = 1

        if question_type == "어법2단계":
            baseline["mode"] = 1  # Default: 1개 (backward compatible)

        return baseline

    def _get_effective_values_for_row(self, row_idx, problem_type):
        """Get effective option values for this row (row override OR global from JSON)"""
        storage_key = self._normalize_storage_key(problem_type)

        # Start with global values from JSON
        global_data = self._read_options_file()
        effective_values = {}

        # Read common paraphrase options
        effective_values["paraphrase_enabled"] = bool(global_data.get(f"{storage_key}paraphrase_enabled", False))
        effective_values["difficulty_level"] = global_data.get(f"{storage_key}difficulty_level", 2)
        effective_values["length_level"] = global_data.get(f"{storage_key}length_level", 2)
        effective_values["content_change_mode"] = global_data.get(f"{storage_key}content_change_mode", 0)

        # Read type-specific options from registry
        if storage_key in self.option_payload_registry:
            for json_key, alias in self.option_payload_registry[storage_key]:
                full_key = f"{storage_key}{json_key}" if not json_key.startswith(storage_key) else json_key
                if full_key in global_data:
                    base_key = alias or json_key
                    if not base_key.startswith(storage_key):
                        effective_values[base_key] = global_data[full_key]
                    else:
                        # If alias is missing but json_key already includes storage key,
                        # strip it so comparisons use baseline-friendly keys.
                        effective_values[base_key.replace(storage_key, "", 1)] = global_data[full_key]

        # Apply row-specific overrides if they exist
        override_entry = self.row_option_overrides.get(row_idx)
        if override_entry and override_entry.get('storage_key') == storage_key:
            raw_overrides = override_entry.get('raw', {})
            for key, value in raw_overrides.items():
                # Strip storage_key prefix to get base key
                base_key = key.replace(storage_key, "", 1) if key.startswith(storage_key) else key
                effective_values[base_key] = value

        return effective_values

    def _compare_values_differ(self, effective_values, baseline_values):
        """Check if effective values differ from baseline (hardcoded defaults)"""
        # If paraphrase is disabled, transformation options don't affect highlighting
        paraphrase_enabled = effective_values.get("paraphrase_enabled", False)
        transformation_keys = {"difficulty_level", "length_level", "content_change_mode"}

        for key in baseline_values:
            # Skip transformation options if paraphrase is disabled
            if not paraphrase_enabled and key in transformation_keys:
                continue

            baseline_val = baseline_values[key]
            effective_val = effective_values.get(key)

            # Type normalization for comparison
            if isinstance(baseline_val, bool):
                effective_val = bool(effective_val)
            elif isinstance(baseline_val, int):
                try:
                    effective_val = int(effective_val) if effective_val is not None else baseline_val
                except (TypeError, ValueError):
                    effective_val = baseline_val

            if effective_val != baseline_val:
                return True

        return False

    def update_option_button_color_for_row(self, row_data):
        """Update option button color based on whether effective values differ from hardcoded defaults"""
        try:
            problem_type = row_data['problem_var'].get()
            option_button = row_data['option_button']
            row_idx = row_data['row_idx']


            # Check if this problem type has options
            if problem_type not in self.문제rows_with_options:
                print(f"     Type '{problem_type}' not in 문제rows_with_options, skipping")
                return

            # Check if the button widget still exists before updating
            if not option_button.winfo_exists():
                print(f"     Button widget destroyed, skipping update")
                return

            # Get effective values for this row (row override OR global)
            effective_values = self._get_effective_values_for_row(row_idx, problem_type)

            # Get hardcoded baseline defaults
            baseline_values = self._get_baseline_values_for_type(problem_type)

            # Compare effective vs baseline
            is_modified = self._compare_values_differ(effective_values, baseline_values)

            if is_modified:
                #print("   🟡 Effective values differ from defaults; highlighting button")
                option_button.configure(fg_color="#DDA15C", text_color="white", hover_color="#CA7900")
            else:
                #print(f"   ⚪ Effective values match defaults; setting button to default state")
                option_button.configure(fg_color="white", text_color="black", hover_color="#DDA15E")
        except Exception as e:
            # Silently skip widget errors (likely widget was destroyed)
            if "invalid command name" not in str(e):
                print(f" ERROR updating option button color: {e}")
                import traceback
                traceback.print_exc()

    def update_all_option_button_colors(self):
        """Schedule option button color updates for all rows (debounced)."""
        # Validate widget still exists before scheduling
        try:
            if not self.winfo_exists():
                print("  Popup window destroyed, skipping button color updates")
                return
        except Exception:
            return

        if self._pending_option_button_refresh:
            # An update is already queued; no need to schedule another
            return

        self._pending_option_button_refresh = True

        try:
            self.after(0, self._execute_option_button_color_update)
        except Exception:
            # Fallback to immediate execution if scheduling fails (e.g., widget already destroyed)
            self._execute_option_button_color_update()

    def _execute_option_button_color_update(self):
        """Perform the actual option button color refresh."""
        self._pending_option_button_refresh = False

        # Check window existence again before performing updates
        try:
            if not self.winfo_exists():
                print("  Popup window destroyed before refresh execution")
                return
        except Exception:
            return

        for row_data in self.row_widgets:
            self.update_option_button_color_for_row(row_data)


    def delete_row(self, row_data):
        # Clean up tooltips if they exist
        if 'tooltip' in row_data and row_data['tooltip']:
            row_data['tooltip'].hide_tooltip()
        if 'problem_menu_tooltip' in row_data and row_data['problem_menu_tooltip']:
            row_data['problem_menu_tooltip'].hide_tooltip()


        # Destroy widgets of this row
        for key, widget in row_data.items():
            if hasattr(widget, 'destroy') and widget is not None:
                widget.destroy()

        # Remove from row_widgets
        self.row_widgets.remove(row_data)
        self.count -= 1

        # Re-label the rows
        for i, rd in enumerate(self.row_widgets, start=1):
            rd['index_label'].configure(text=str(i))
            rd['drag_handle'].grid(row=i, column=0)
            rd['index_label'].grid(row=i, column=1)
            rd['waiting_menu'].grid(row=i, column=2)
            rd['problem_menu'].grid(row=i, column=3)
            rd['hardness_menu'].grid(row=i, column=4)
            rd['current_status_label'].grid(row=i, column=5)
            if rd['option_button']:
                rd['option_button'].grid(row=i, column=6)
            rd['delete_button'].grid(row=i, column=7)

        # Move plus button
        self.plus_button.grid_configure(row=len(self.row_widgets)+1, columnspan=8)

        # Update all row indices
        self.update_row_indices()
        self.refresh_waiting_dropdowns()

        # Refresh drag bindings
        self.drag_manager.reset_registry()
        for idx, rd in enumerate(self.row_widgets):
            self.drag_manager.register_handle(rd['drag_handle'], idx)

    def reorder_rows_and_regrid(self, start_index, target_index, insert_after=False):
        """Reorder rows via drag-and-drop and update all state.

        Args:
            start_index: 0-based index of dragged row in row_widgets
            target_index: 0-based index of drop target in row_widgets
        """
        if start_index == target_index and not insert_after:
            return  # No change

        if not (0 <= start_index < len(self.row_widgets)):
            return  # Invalid indices

        if target_index < 0:
            target_index = 0
        if target_index >= len(self.row_widgets):
            target_index = len(self.row_widgets) - 1

        # 1. Reorder row_widgets list
        item = self.row_widgets.pop(start_index)
        insert_pos = target_index
        if insert_pos > start_index:
            insert_pos -= 1
        if insert_after:
            insert_pos += 1
        insert_pos = max(0, min(insert_pos, len(self.row_widgets)))
        self.row_widgets.insert(insert_pos, item)

        # 2. Call update_row_indices() - it builds old→new map and remaps dictionaries
        self.update_row_indices()

        # 3. Update widget grid positions without tearing down the layout so there is
        #    no visible flicker while dragging.
        for i, rd in enumerate(self.row_widgets, start=1):
            rd['drag_handle'].grid_configure(row=i)
            rd['index_label'].grid_configure(row=i)
            rd['waiting_menu'].grid_configure(row=i)
            rd['problem_menu'].grid_configure(row=i)
            rd['hardness_menu'].grid_configure(row=i)
            rd['current_status_label'].grid_configure(row=i)
            if rd['option_button']:
                rd['option_button'].grid_configure(row=i)
            rd['delete_button'].grid_configure(row=i)

        # 4. Update plus button position
        self.plus_button.grid_configure(row=len(self.row_widgets)+1, columnspan=8)

        # 5. Refresh drag bindings
        self.drag_manager.reset_registry()
        for idx, rd in enumerate(self.row_widgets):
            self.drag_manager.register_handle(rd['drag_handle'], idx)

        # 6. Refresh waiting dropdowns (for duplicate highlighting)
        self.refresh_waiting_dropdowns()

    def load_waiting_list_items(self):
        """Load both titles (column A) and passages (column B) from 대기열 sheet"""
        self.title_to_passage_map = {}  # Dictionary to map titles to passages

        if not self.excel_file:
            return []
        try:
            wb = load_workbook(self.excel_file, read_only=True, data_only=True)
            if "대기열" not in wb.sheetnames:
                return []
            sheet = wb["대기열"]
            items = []

            # Load both column A (titles) and column B (passages)
            for row in sheet.iter_rows(min_col=1, max_col=2, values_only=True):
                title = row[0]
                passage = row[1] if len(row) > 1 else ""

                if title is not None and str(title).strip() != "지문을 선택하세요":
                    title_str = str(title).strip()
                    items.append(title_str)
                    # Store the mapping of title to passage
                    self.title_to_passage_map[title_str] = str(passage).strip() if passage else "내용이 없습니다."

            wb.close()
            return items
        except Exception as e:
            print(f"Error loading waiting list: {e}")
            return []

    def init_drag_manager(self):
        """Initialize drag and drop manager for row reordering"""

        class DragReorderManager:
            """Handles drag-and-drop reordering of question rows"""
            def __init__(self, container, owner):
                self.container = container  # scrollable_frame
                self.owner = owner  # ManualMocktestPopup instance
                self._inner = getattr(self.container, "_scrollable_frame", self.container)

                # Tracking dictionaries
                self.rowframe_by_gridrow = {}  # {grid_row: index label widget}
                self.rowdata_by_gridrow = {}   # {grid_row: row_data dict}
                self.orig_fg_by_gridrow = {}   # {grid_row: original fg color}

                # Drag state
                self.start_index = None        # 0-based index in row_widgets
                self.highlight_index = None    # 0-based index for drop target
                self.insert_after_flag = False

                # Autoscroll state
                self._autoscroll_job = None
                self._autoscroll_dir = 0  # -1(up), 0(stop), +1(down)

                # Find canvas for autoscroll
                self._canvas = getattr(self.container, "_parent_canvas", None)
                if self._canvas is None:
                    for child in self.container.winfo_children():
                        if child.winfo_class().lower().endswith('canvas'):
                            self._canvas = child
                            break

                # Drop target indicator (thin horizontal line)
                self.drop_indicator = CTkFrame(self._inner, width=10, height=4, fg_color="#E8DCC2")
                self.drop_indicator.place_forget()
                self._drop_indicator_visible = False

            def reset_registry(self):
                """Clear tracking dictionaries"""
                # Restore any highlighted row before we discard references
                self.clear_highlight()
                self.rowframe_by_gridrow.clear()
                self.rowdata_by_gridrow.clear()
                self.orig_fg_by_gridrow.clear()
                self.insert_after_flag = False

            def register_handle(self, widget, index: int):
                """Bind drag events to a drag handle widget

                Args:
                    widget: The drag handle widget
                    index: 0-based index in row_widgets list
                """
                grid_row = index + 1  # Convert to 1-based grid row

                # Track first widget in row (index_label) for highlighting
                row_data = self.owner.row_widgets[index]
                self.rowframe_by_gridrow[grid_row] = row_data['index_label']
                self.rowdata_by_gridrow[grid_row] = row_data
                try:
                    self.orig_fg_by_gridrow[grid_row] = row_data['index_label'].cget("fg_color")
                except:
                    self.orig_fg_by_gridrow[grid_row] = "#B6C2B7"

                # Bind drag events (NO add="+" - widgets persist!)
                widget.bind("<Button-1>", lambda e, idx=index: self._on_start(e, idx))
                widget.bind("<B1-Motion>", self._on_motion)
                widget.bind("<ButtonRelease-1>", self._on_drop)
                widget.configure(cursor="sb_v_double_arrow")

            def _on_start(self, event, index: int):
                """Handle drag start"""
                self.start_index = index
                grid_row, pointer_y = self._grid_row_from_pointer(event.x_root, event.y_root)
                if grid_row is None:
                    grid_row = index + 1
                self._set_highlight_by_grid_row(grid_row, pointer_y)
                self._update_autoscroll_from_pointer(event.y_root)

            def _on_motion(self, event):
                """Handle drag motion - update highlight"""
                self._update_autoscroll_from_pointer(event.y_root)
                result = self._grid_row_from_pointer(event.x_root, event.y_root)
                grid_row, pointer_y = (result if isinstance(result, tuple) else (result, None))
                if grid_row is None:
                    return
                self._set_highlight_by_grid_row(grid_row, pointer_y)

            def _on_drop(self, event):
                """Handle drag drop - complete reorder"""
                self._stop_autoscroll()
                if self.start_index is None:
                    self.clear_highlight()
                    return

                target_index = self.highlight_index if self.highlight_index is not None else self.start_index
                insert_after = self.insert_after_flag if self.highlight_index is not None else False

                # Perform reorder
                if 0 <= self.start_index < len(self.owner.row_widgets) and 0 <= target_index < len(self.owner.row_widgets):
                    if target_index != self.start_index or insert_after:
                        self.owner.reorder_rows_and_regrid(self.start_index, target_index, insert_after)

                # Clear state
                self.start_index = None
                self.clear_highlight()
                self.insert_after_flag = False

            def _set_highlight_by_grid_row(self, grid_row: int, pointer_y_rel: float = None):
                """Highlight target row"""
                if grid_row <= 0:
                    return
                if self.highlight_index is not None and (self.highlight_index + 1) != grid_row:
                    self.clear_highlight()
                row_data = self.rowdata_by_gridrow.get(grid_row)
                frame = row_data.get('index_label') if row_data else None
                total_rows = len(self.owner.row_widgets)
                if grid_row < 1:
                    grid_row = 1
                if grid_row > total_rows:
                    grid_row = total_rows
                row_data = self.rowdata_by_gridrow.get(grid_row)
                frame = row_data.get('index_label') if row_data else None
                if frame is not None:
                    try:
                        frame.configure(fg_color="#E8DCC2")
                        indicator_y, insert_after = self._indicator_y_for_row(row_data, grid_row, pointer_y_rel)
                        if indicator_y is not None:
                            self._show_drop_indicator(indicator_y)
                            self.insert_after_flag = insert_after
                        else:
                            self.insert_after_flag = False
                    except Exception:
                        self.insert_after_flag = False
                    self.highlight_index = grid_row - 1

            def clear_highlight(self):
                """Remove highlight from target row"""
                if self.highlight_index is None:
                    self._hide_drop_indicator()
                    self.insert_after_flag = False
                    return
                grid_row = self.highlight_index + 1
                row_data = self.rowdata_by_gridrow.get(grid_row)
                frame = row_data.get('index_label') if row_data else None
                if frame is not None:
                    try:
                        frame.configure(fg_color=self.orig_fg_by_gridrow.get(grid_row, "#B6C2B7"))
                    except:
                        pass
                self.highlight_index = None
                self._hide_drop_indicator()
                self.insert_after_flag = False

            def _show_drop_indicator(self, y_pos: int):
                try:
                    self._inner.update_idletasks()
                    self.drop_indicator.place(x=0, y=max(0, y_pos - 1), relwidth=1)
                    self.drop_indicator.lift()
                    self._drop_indicator_visible = True
                except Exception:
                    pass

            def _hide_drop_indicator(self):
                if self._drop_indicator_visible:
                    try:
                        self.drop_indicator.place_forget()
                    except Exception:
                        pass
                    self._drop_indicator_visible = False

            def _grid_row_from_pointer(self, x_root: int, y_root: int):
                widget = self._inner.winfo_containing(x_root, y_root)
                original = widget
                while widget is not None and widget not in (self._inner,) and widget.master is not self._inner:
                    widget = widget.master

                if widget is None:
                    return self._grid_row_from_y(y_root)

                if widget is self._inner:
                    return self._grid_row_from_y(y_root)

                gi = widget.grid_info() if widget is not None else None
                if gi and "row" in gi:
                    row = int(gi["row"])
                    _, y_rel = self._grid_row_from_y(y_root)
                    return row, y_rel

                return self._grid_row_from_y(y_root)

            def _grid_row_from_y(self, y_root: int):
                try:
                    self._inner.update_idletasks()
                except Exception:
                    return (None, None)
                y_rel = y_root - self._inner.winfo_rooty()
                closest_row = None
                closest_distance = None
                for idx, rd in enumerate(self.owner.row_widgets, start=1):
                    widget = rd.get('index_label')
                    if widget is None:
                        continue
                    y = widget.winfo_y()
                    h = widget.winfo_height() or 1
                    center = y + h / 2
                    distance = abs(y_rel - center)
                    if closest_distance is None or distance < closest_distance:
                        closest_distance = distance
                        closest_row = idx
                return closest_row, y_rel

            def _indicator_y_for_row(self, row_data, grid_row: int, pointer_y_rel: float = None):
                if not row_data:
                    return (None, False)
                anchor = row_data.get('drag_handle') or row_data.get('index_label')
                if anchor is None:
                    return (None, False)
                try:
                    anchor.update_idletasks()
                    top = anchor.winfo_y()
                    height = anchor.winfo_height() or 0
                except Exception:
                    return (None, False)

                insert_after = False
                if pointer_y_rel is not None:
                    insert_after = pointer_y_rel > (top + height / 2)
                elif self.start_index is not None:
                    insert_after = (grid_row - 1) > self.start_index

                if insert_after:
                    return (top + height + 2, True)
                return (max(0, top - 2), False)

            def _start_autoscroll(self, direction: int):
                """Start autoscroll in given direction"""
                if not self._can_autoscroll():
                    return
                if direction == self._autoscroll_dir and self._autoscroll_job is not None:
                    return
                self._autoscroll_dir = direction
                if self._autoscroll_job is None:
                    self._do_autoscroll()

            def _stop_autoscroll(self):
                """Stop autoscroll"""
                self._autoscroll_dir = 0
                if self._autoscroll_job:
                    try:
                        self.container.after_cancel(self._autoscroll_job)
                    except Exception:
                        pass
                    self._autoscroll_job = None

            def _do_autoscroll(self):
                """Perform autoscroll step"""
                if self._autoscroll_dir == 0 or not self._can_autoscroll():
                    self._autoscroll_job = None
                    return
                try:
                    steps = 2
                    for _ in range(steps):
                        if self._canvas is not None:
                            self._canvas.yview_scroll(self._autoscroll_dir, "units")
                except:
                    pass
                self._autoscroll_job = self.container.after(20, self._do_autoscroll)

            def _update_autoscroll_from_pointer(self, y_root: int):
                """Update autoscroll based on pointer position"""
                top = self.container.winfo_rooty()
                height = self.container.winfo_height()
                y_rel = y_root - top
                margin = 24
                if y_rel < margin:
                    self._start_autoscroll(-1)
                elif y_rel > height - margin:
                    self._start_autoscroll(1)
                else:
                    self._stop_autoscroll()

            def _can_autoscroll(self):
                if self._canvas is None:
                    return False
                try:
                    first, last = self._canvas.yview()
                except Exception:
                    return False
                return (last - first) < 0.999

        self.drag_manager = DragReorderManager(self.scrollable_frame, self)

    def _open_passage_editor(self, row_idx, passage_id):
        """Open passage editor dialog for the given row and passage_id"""
        # Guard: Validate passage_id
        if not passage_id or passage_id == "지문을 선택하세요":
            return

        # Get current passage text (edited or original) - check this specific row first
        if row_idx in self.edited_passages and self.edited_passages[row_idx].get("passage_id") == passage_id:
            current_text = self.edited_passages[row_idx]["text"]
        else:
            current_text = self.title_to_passage_map.get(passage_id, "")

        # Open edit dialog with row_idx in the callback
        PassageEditDialog(
            parent=self,
            passage_id=passage_id,
            current_passage_text=current_text,
            on_save_callback=lambda pid, txt: self._on_passage_edited(row_idx, pid, txt),
            default_font=self.default_font
        )

    def _on_passage_edited(self, row_idx, passage_id, edited_text):
        """Called when passage is saved in editor"""
        # Store edited passage for this specific row
        self.edited_passages[row_idx] = {"passage_id": passage_id, "text": edited_text}

        # Update ONLY the row that was edited
        for rd in self.row_widgets:
            if rd['row_idx'] == row_idx:
                # Use public refresh method to update "(수정됨)" indicator
                if hasattr(rd['waiting_menu'], 'refresh_display'):
                    rd['waiting_menu'].refresh_display()

                # Update tooltip to show edited passage (check existence first)
                if 'tooltip' in rd and rd['tooltip']:
                    rd['tooltip'].update_text(edited_text)
                break  # Found the row, no need to continue

    def _refresh_edited_passage_ui(self):
        """Refresh UI for edited passages (called after restore/preset load)"""
        for rd in self.row_widgets:
            row_idx = rd['row_idx']
            passage_id = rd['waiting_var'].get()
            # Check if this row has an edit for the current passage
            if row_idx in self.edited_passages and self.edited_passages[row_idx].get("passage_id") == passage_id:
                # Refresh dropdown display
                if hasattr(rd['waiting_menu'], 'refresh_display'):
                    rd['waiting_menu'].refresh_display()

                # Update tooltip (check existence)
                if 'tooltip' in rd and rd['tooltip']:
                    rd['tooltip'].update_text(self.edited_passages[row_idx]["text"])

    def get_option_command(self, row_data):
        """
        Return a function (closure) that fetches the most up-to-date values
        from the row_data dict each time the '옵션' button is clicked.
        """

        def _open_option():
            row_idx = row_data['row_idx']
            problem_type = row_data['problem_var'].get()
            waiting_text = row_data['waiting_var'].get()
            hardness = row_data['hardness_var'].get()

            자동실모flag = False
            실모flag = True

            context = self._prepare_option_dialog(row_idx, problem_type)
            popup = None

            try:
                if problem_type == "어휘3단계":
                    popup = self.어휘3단계_option(자동실모flag, 실모flag, waiting_text, row_idx, hardness, manual_mocktest_popup=self)

                elif problem_type == "내용일치":
                    popup = self.내용일치_option(manual_mocktest_popup=self, row_idx=row_idx)

                elif problem_type == "내용일치(한)":
                    popup = self.내용일치한_option(manual_mocktest_popup=self, row_idx=row_idx)

                elif problem_type == "밑줄의미":
                    popup = self.밑줄의미_option(manual_mocktest_popup=self, row_idx=row_idx)

                elif problem_type == "함축의미":
                    popup = self.함축의미_option(자동실모flag, 실모flag, waiting_text, row_idx, hardness, manual_mocktest_popup=self)

                elif problem_type == "무관한문장":
                    popup = self.무관한문장_option(manual_mocktest_popup=self, row_idx=row_idx)

                elif problem_type == "빈칸추론":
                    popup = self.빈칸추론_option(자동실모flag, 실모flag, waiting_text, row_idx, hardness, manual_mocktest_popup=self)

                elif problem_type == "추론":
                    popup = self.추론_option(manual_mocktest_popup=self, row_idx=row_idx)

                elif problem_type == "추론불가":
                    popup = self.추론불가_option(manual_mocktest_popup=self, row_idx=row_idx)

                elif problem_type == "순서":
                    popup = self.순서_option(자동실모flag, 실모flag, waiting_text, row_idx, hardness, manual_mocktest_popup=self)

                elif problem_type == "삽입":
                    popup = self.삽입_option(자동실모flag, 실모flag, waiting_text, row_idx, hardness, manual_mocktest_popup=self)

                elif problem_type == "주제":
                    popup = self.주제_option(manual_mocktest_popup=self, row_idx=row_idx)

                elif problem_type == "제목":
                    popup = self.제목_option(manual_mocktest_popup=self, row_idx=row_idx)

                elif problem_type == "요지":
                    popup = self.요지_option(manual_mocktest_popup=self, row_idx=row_idx)

                elif problem_type == "요약":
                    popup = self.요약_option(자동실모flag, 실모flag, waiting_text, row_idx, hardness, manual_mocktest_popup=self)

                elif problem_type == "영작1":
                    popup = self.영작1_option(자동실모flag, 실모flag, waiting_text, row_idx, hardness, manual_mocktest_popup=self)

                elif problem_type == "영작2":
                    popup = self.영작2_option(자동실모flag, 실모flag, waiting_text, row_idx, hardness, manual_mocktest_popup=self)

                elif problem_type == "주제영작":
                    popup = self.주제영작_option(자동실모flag, 실모flag, waiting_text, row_idx, hardness, manual_mocktest_popup=self)

                elif problem_type == "동형문제":
                    popup = self.동형문제_option(자동실모flag, 실모flag, waiting_text, row_idx, hardness, manual_mocktest_popup=self, is_manual_mocktest=True)

                elif problem_type == "어법2단계":
                    popup = self.어법2단계_option(자동실모flag, manual_mocktest_popup=self)

            finally:
                if context and popup is None:
                    self._restore_options_baseline(context)

            if popup and context:
                setattr(popup, "_manual_option_context", context)
                # Store reference to this manager so popup can call handler directly
                setattr(popup, "_option_override_manager", self)

        return _open_option



    def on_waiting_type_change(self, row_data, selected_type):
        row_idx = row_data['row_idx']

        # Clear the row's edit if the passage selection changed
        if row_idx in self.edited_passages:
            if self.edited_passages[row_idx].get("passage_id") != selected_type:
                del self.edited_passages[row_idx]

        # Update tooltip with edited or original passage (check existence first)
        if 'tooltip' in row_data and row_data['tooltip']:
            # Check if this specific row has an edit for the newly selected passage
            if row_idx in self.edited_passages and self.edited_passages[row_idx].get("passage_id") == selected_type:
                passage_text = self.edited_passages[row_idx]["text"]
            else:
                passage_text = self.title_to_passage_map.get(selected_type, "내용이 없습니다.")
            row_data['tooltip'].update_text(passage_text)

        # Update the current status label
        passage_id유형난이도합본 = row_data['waiting_menu'].get() + ' // ' + row_data['problem_menu'].get() + '_' + row_data['hardness_menu'].get()
        self.update_current_status_label수동실모(row_idx, passage_id유형난이도합본)
        self.refresh_waiting_dropdowns()

    def on_problem_type_selected(self, row_data, selected_type):
        """Called when user selects a problem type from the dropdown (user interaction only)"""
        # First, do the normal processing
        self.on_problem_type_change(row_data, selected_type)

        # If "동형문제" was selected, automatically open the dialog
        if selected_type == "동형문제":
            # Get the option command for this row
            option_command = self.get_option_command(row_data)
            if option_command:
                # Use after() to ensure the dialog opens after the current event completes
                self.after(100, option_command)

    def on_problem_type_change(self, row_data, selected_type):
        

        hardness_menu = row_data['hardness_menu']
        row_idx = row_data['row_idx']
        storage_key = self._normalize_storage_key(selected_type)
        self._clear_override_if_type_changed(row_idx, storage_key)

        # Determine the possible values for hardness based on selected_type
        if selected_type in self.실모rows_with_hard_level:
            hardness_menu.configure(values=["Normal", "Hard"])
            # If the current selection isn't in the new values, reset to "Normal"
            if hardness_menu.get() not in ["Normal", "Hard"]:
                hardness_menu.set("Normal")
            hardness_menu.configure(state="normal")
        else:
            hardness_menu.configure(values=["Normal"])
            hardness_menu.set("Normal")
            hardness_menu.configure(state="disabled")

        # Update option button
        self.configure_option_button_for_row(row_data, selected_type)

        # Update problem_menu tooltip and text color based on selected type and saved content
        if 'problem_menu_tooltip' in row_data and row_data['problem_menu_tooltip']:
            if selected_type == "동형문제":
                if row_idx in self.copy_question_contents:
                    # Extract and show only the [Question] part
                    content = self.copy_question_contents[row_idx]
                    question_only = self.extract_question_from_sample(content)
                    row_data['problem_menu_tooltip'].update_text(question_only)
                    # Set text color to normal (black)
                    row_data['problem_menu'].configure(text_color="black")

                    # Check for matching presets and update display
                    fields = self._extract_fields_from_sample(content)
                    if fields:
                        passage, question, answer = fields
                        matched_presets = self._find_matching_presets(passage, question, answer)
                        self._update_problem_menu_display(row_data, matched_presets)
                    else:
                        self._update_problem_menu_display(row_data, [])
                else:
                    # Show instruction message when no content is saved
                    row_data['problem_menu_tooltip'].update_text("'옵션'을 클릭하여 동형문제 샘플을 입력하세요")
                    # Set text color to red to indicate missing content
                    row_data['problem_menu'].configure(text_color="red")
                    # Clear preset display
                    self._update_problem_menu_display(row_data, [])
            else:
                # Clear tooltip for non-동형문제
                row_data['problem_menu_tooltip'].update_text("")
                # Reset text color to normal (black)
                row_data['problem_menu'].configure(text_color="black")
                # Clear preset display when type changes away from 동형문제
                self.row_matched_presets.pop(row_idx, None)

        # Update the current status label
        passage_id유형난이도합본 = row_data['waiting_menu'].get() + ' // ' + row_data['problem_menu'].get() + '_' + row_data['hardness_menu'].get()
        self.update_current_status_label수동실모(row_idx, passage_id유형난이도합본)

        # Update option button color based on saved options
        self.update_option_button_color_for_row(row_data)

    def on_hardness_change(self, row_data, new_value):
        # If you need to re-configure the option button here:
        self.configure_option_button_for_row(row_data, row_data['problem_var'].get())

        # Update the current status label
        row_idx = row_data['row_idx']
        passage_id유형난이도합본 = row_data['waiting_menu'].get() + ' // ' + row_data['problem_menu'].get() + '_' + row_data['hardness_menu'].get()
        self.update_current_status_label수동실모(row_idx, passage_id유형난이도합본)


    def cancel(self, event=None):
        self.save_last_state()
        self.destroy()


    def start(self):
        try:
            # --- 1) Check if any row with "동형문제" is missing saved content ---
            missing_content_rows = []
            for rd in self.row_widgets:
                row_idx = rd['row_idx']
                problem_type = rd['problem_var'].get()
                if problem_type == "동형문제":
                    # Check if this row has saved content
                    if row_idx not in self.copy_question_contents or not self.copy_question_contents[row_idx]:
                        missing_content_rows.append(row_idx)

            if missing_content_rows:
                rows_str = ", ".join(str(r) for r in missing_content_rows)
                messagebox.showwarning(
                    "경고",
                    f"{rows_str}번 문제의 동형문제 샘플이 입력되지 않았습니다.\n"
                    "'옵션'버튼을 클릭하여 동형으로 제작할 샘플을 입력해 주세요."
                )
                return   # ← abort, keep this popup open

            # --- 2) If any row is of type "동형문제", ensure the sample exists ---
            selected_types = [rd['problem_var'].get() for rd in self.row_widgets]
            if "동형문제" in selected_types:
                sample_path = self.main_frame_ref.저장된동형문제샘플
                if not sample_path.exists():
                    messagebox.showinfo(
                        "Info",
                        "동형문제 샘플이 제작되지 않았습니다. "
                        "'동형문제'를 선택한 문제의 '옵션'버튼을 클릭하여 동형으로 제작할 샘플을 입력해 주세요."
                    )
                    return   # ← abort, keep this popup open

            # --- 3) rest of your existing validation & process logic ---
            # Collect the user selections


            waiting_selections = [rd['waiting_var'].get() for rd in self.row_widgets]
            problem_selections = [rd['problem_var'].get() for rd in self.row_widgets]
            hardness_values = [rd['hardness_var'].get() for rd in self.row_widgets]

            # Check if waiting_selections is empty
            if not any(waiting_selections):
                messagebox.showwarning("경고", "지문이 없습니다. '+' 버튼을 클릭하여 지문을 1개 이상 추가하세요")
                return

            # Check if any waiting_selection is "지문을 선택하세요"
            invalid_rows = []
            for i, sel in enumerate(waiting_selections, start=1):
                if sel.strip() == "지문을 선택하세요":
                    invalid_rows.append(i)

            if invalid_rows:
                rows_str = ", ".join(str(r) for r in invalid_rows)
                messagebox.showwarning("경고", f"{rows_str}번 문제의 지문이 선택되지 않았습니다.")
                return

            # --- 4) Show PointsDashboard to preview credits ---
            # Build manual_mocktest_data from row_widgets with pre-computed paraphrase status
            manual_mocktest_data = []
            for rd in self.row_widgets:
                row_idx = rd['row_idx']
                problem_type = rd['problem_var'].get()
                # Get effective values for this row (includes global + per-row overrides)
                effective_values = self._get_effective_values_for_row(row_idx, problem_type)
                manual_mocktest_data.append({
                    "passage_id": rd['waiting_var'].get(),
                    "problem_type": problem_type,
                    "hardness": rd['hardness_var'].get(),
                    "paraphrase_enabled": effective_values.get("paraphrase_enabled", False),
                })

            # Get credit info from main_frame_ref
            current_credits = getattr(self.main_frame_ref, 'credits', 0) if self.main_frame_ref else 0
            base_credit_map = getattr(self.main_frame_ref, 'base_credit_map', {}) if self.main_frame_ref else {}
            sheet_type_mapping = getattr(self.main_frame_ref, 'sheet_type_mapping', {}) if self.main_frame_ref else {}

            # Build row_option_overrides list for paraphrase checking
            row_option_overrides = self._build_option_override_payload()

            # Show the PointsDashboard
            dashboard = PointsDashboard(
                parent=self,
                collected_data={},  # Not used in manual mocktest mode
                current_credits=current_credits,
                base_credit_map=base_credit_map,
                excel_file=self.excel_file,
                문제checkbox_vars={},  # Not used in manual mocktest mode
                자료checkbox_vars={},  # Not used in manual mocktest mode
                교재checkbox_vars={},  # Not used in manual mocktest mode
                sheet_type_mapping=sheet_type_mapping,
                enabled_categories={},  # Not used in manual mocktest mode
                default_font=self.default_font,
                manual_mocktest_data=manual_mocktest_data,
                manual_mocktest_row_overrides=row_option_overrides,
            )

            if not dashboard.show():
                # User cancelled - don't proceed
                return

            # Handle duplicate (지문, 유형, 난이도) combinations by adding numbering
            from collections import Counter

            # Create list of (지문, 유형, 난이도) combinations
            combination_list = [(waiting_selections[i], problem_selections[i], hardness_values[i])
                               for i in range(len(waiting_selections))]

            # Count occurrences of each combination
            combination_counter = Counter(combination_list)

            # Track seen combinations to append numbering
            combination_seen = {}
            modified_waiting_selections = []
            original_waiting_selections = waiting_selections.copy()  # Keep original names for 대기열 lookup

            for i in range(len(waiting_selections)):
                key = (waiting_selections[i], problem_selections[i], hardness_values[i])

                # If this combination appears more than once, add numbering (1), (2), (3), etc.
                if combination_counter[key] > 1:
                    if key not in combination_seen:
                        combination_seen[key] = 0
                    combination_seen[key] += 1
                    modified_waiting = f"{waiting_selections[i]} ({combination_seen[key]})"
                else:
                    modified_waiting = waiting_selections[i]

                modified_waiting_selections.append(modified_waiting)

            combined_types = [f"{ptype}_{hardness}" for ptype, hardness in zip(problem_selections, hardness_values)]
            self.user_selected_waiting = modified_waiting_selections
            self.user_selected_types = combined_types

            if self.process_excel_callback:
                # Pass copy_question_contents if it exists
                copy_question_contents = getattr(self, 'copy_question_contents', {})
                option_overrides_payload = self._build_option_override_payload()

                
                self.process_excel_callback(
                    selected_waiting=self.user_selected_waiting,
                    selected_types=self.user_selected_types,
                    original_waiting=original_waiting_selections,  # Pass original names for lookup
                    copy_question_contents=copy_question_contents,
                    row_option_overrides=option_overrides_payload,
                    edited_passages=dict(self.edited_passages)  # Pass edited passages
                )

            self.destroy()
        finally:
            # Always save state even if validation fails
            self.save_last_state()


    def load_presets(self):
        if self.presets_file.exists():
            try:
                with open(self.presets_file, 'r', encoding='utf-8') as file:
                    self.presets = json.load(file)
            except json.JSONDecodeError:
                messagebox.showerror("Error", "프리셋 파일을 불러오는 중 오류가 발생했습니다.")
                self.presets = {}
        else:
            self.presets = {}
        self.preset_combobox.configure(values=list(self.presets.keys()))
        if self.presets:
            self.preset_combobox.set("불러올 프리셋 선택")
        else:
            self.preset_combobox.set("")

    def save_presets_to_file(self):
        try:
            with open(self.presets_file, 'w', encoding='utf-8') as file:
                json.dump(self.presets, file, ensure_ascii=False, indent=4)
        except Exception as e:
            messagebox.showerror("Error", f"프리셋을 저장하는 중 오류가 발생했습니다: {e}")

    def save_preset(self):
        preset_name = self.preset_name_entry.get().strip()
        if not preset_name:
            messagebox.showwarning("Warning", "프리셋 이름을 입력해주세요.")
            return
        if preset_name in self.presets:
            overwrite = messagebox.askyesno("Overwrite", f"프리셋 '{preset_name}'을(를) 덮어쓰시겠습니까?")
            if not overwrite:
                return

        # Collect current selections
        problem_selections = [rd['problem_var'].get() for rd in self.row_widgets]
        hardness_values = [rd['hardness_var'].get() for rd in self.row_widgets]

        # Collect 동형문제 복사기 contents indexed by position (not row_idx)
        copy_contents_list = []
        for i, rd in enumerate(self.row_widgets):
            row_idx = rd['row_idx']
            # If this row has 동형문제 content saved, include it
            if row_idx in self.copy_question_contents:
                copy_contents_list.append(self.copy_question_contents[row_idx])
            else:
                copy_contents_list.append(None)

        option_overrides_list = self._build_option_override_payload()

        self.presets[preset_name] = {
            "types": problem_selections,
            "hardness": hardness_values,
            "copy_contents": copy_contents_list,  # Save 동형문제 contents
            "option_overrides": option_overrides_list,
            "edited_passages": dict(self.edited_passages)  # Save edited passages
        }

        self.save_presets_to_file()
        self.preset_combobox.configure(values=list(self.presets.keys()))
        self.preset_combobox.set(preset_name)
        self.preset_name_entry.delete(0, tk.END)
        messagebox.showinfo("Success", f"프리셋 '{preset_name}'이(가) 저장되었습니다.")

    def delete_preset(self):
        preset_name = self.preset_combobox.get()
        if not preset_name:
            messagebox.showwarning("Warning", "삭제할 프리셋을 선택해주세요.")
            return
        confirm = messagebox.askyesno("Confirm Delete", f"프리셋 '{preset_name}'을(를) 삭제하시겠습니까?")
        if confirm:
            del self.presets[preset_name]
            self.save_presets_to_file()
            self.preset_combobox.configure(values=list(self.presets.keys()))
            if self.presets:
                self.preset_combobox.set(list(self.presets.keys())[0])
            else:
                self.preset_combobox.set("")
            messagebox.showinfo("Deleted", f"프리셋 '{preset_name}'이(가) 삭제되었습니다.")

    def on_preset_selected(self, preset_name):
        if preset_name in self.presets:
            preset_data = self.presets[preset_name]

            problem_selections = preset_data.get("types", [])
            hardness_values = preset_data.get("hardness", [])
            copy_contents_list = preset_data.get("copy_contents", [])  # Load 동형문제 contents
            option_overrides_list = preset_data.get("option_overrides", [])

            # Determine how many rows the preset has
            preset_row_count = len(problem_selections)
            current_row_count = len(self.row_widgets)

            # Adjust row count to match preset
            if preset_row_count < current_row_count:
                # Remove extra rows
                while len(self.row_widgets) > preset_row_count:
                    self.delete_row(self.row_widgets[-1])
            elif preset_row_count > current_row_count:
                # Add missing rows
                for _ in range(preset_row_count - current_row_count):
                    self.add_row()

            # Now pad lists to match (should be equal now, but keep for safety)
            max_rows = len(self.row_widgets)
            problem_selections = (problem_selections[:max_rows] +
                                  [""]*(max_rows-len(problem_selections)))
            hardness_values = (hardness_values[:max_rows] +
                               ["Normal"]*(max_rows-len(hardness_values)))
            copy_contents_list = (copy_contents_list[:max_rows] +
                                  [None]*(max_rows-len(copy_contents_list)))
            option_overrides_list = (option_overrides_list[:max_rows] +
                                     [None]*(max_rows-len(option_overrides_list)))

            # Clear existing copy_question_contents and edited_passages and restore from preset
            self.copy_question_contents.clear()
            self.edited_passages.clear()

            for i, rd in enumerate(self.row_widgets):
                selected_type = problem_selections[i]
                hardness = hardness_values[i]

                # If the loaded problem type isn't valid in 실모rows, fall back to first type if available
                if selected_type not in self.실모rows and self.실모rows:
                    selected_type = self.실모rows[0]

                rd['problem_var'].set(selected_type)
                self.on_problem_type_change(rd, selected_type)

                hardness_menu = rd['hardness_menu']
                hardness_menu_values = hardness_menu.cget("values")
                if hardness in hardness_menu_values:
                    hardness_menu.set(hardness)
                else:
                    hardness_menu.set("Normal")

                # Restore 동형문제 복사기 content for this row if it exists
                if copy_contents_list[i] is not None:
                    row_idx = rd['row_idx']
                    self.copy_question_contents[row_idx] = copy_contents_list[i]
                    # Update the option tooltip to show the loaded content
                    self.update_option_tooltip(row_idx, copy_contents_list[i])
                else:
                    # Reset tooltip to default message if no content
                    row_idx = rd['row_idx']
                    self.update_option_tooltip(row_idx, None)

            self._apply_option_overrides_from_list(option_overrides_list)

            # Restore edited passages and refresh UI (convert string keys back to int)
            edited_passages_data = preset_data.get("edited_passages", {})
            if edited_passages_data:
                # JSON converts int keys to strings, convert back
                for k, v in edited_passages_data.items():
                    self.edited_passages[int(k)] = v
                # Queue UI updates to avoid touching destroyed widgets
                self.after(10, lambda: self._refresh_edited_passage_ui())

            # Update all option button colors after loading preset
            self.update_all_option_button_colors()

    def save_last_state(self):
        """Save current dialog state to JSON file (auto-save on any exit)"""
        try:
            state_data = {
                "row_count": len(self.row_widgets),
                "rows": []
            }

            for rd in self.row_widgets:
                row_idx = rd['row_idx']
                row_data = {
                    "waiting_selection": rd['waiting_var'].get(),
                    "problem_type": rd['problem_var'].get(),
                    "hardness": rd['hardness_var'].get()
                }

                # Always serialize copy_content, even if blank (to distinguish "no entry" vs "intentionally empty")
                if row_idx in self.copy_question_contents:
                    row_data["copy_content"] = self.copy_question_contents[row_idx]
                else:
                    row_data["copy_content"] = ""  # Explicitly mark as empty

                state_data["rows"].append(row_data)

            state_data["option_overrides"] = self._build_option_override_payload()
            state_data["edited_passages"] = dict(self.edited_passages)

            with open(self.last_state_file, 'w', encoding='utf-8') as f:
                json.dump(state_data, f, ensure_ascii=False, indent=2)
        except Exception as e:
            # Don't crash dialog if file write fails - just log it
            print(f"Warning: Failed to save mocktest state: {e}")

    def randomize_problem_types(self):
        if not self.실모rows:
            messagebox.showwarning("Warning", "실모 문제유형 목록이 비어있습니다.")
            return

        for rd in self.row_widgets:
            selected_type = random.choice(self.실모rows)
            rd['problem_var'].set(selected_type)
            self.on_problem_type_change(rd, selected_type)

        # Update all option button colors after randomizing
        self.update_all_option_button_colors()

    def apply_first_type_to_all(self):
        """Apply the problem type from the first row to all rows"""
        if not self.row_widgets:
            messagebox.showwarning("Warning", "추가된 문제가 없습니다.")
            return

        # Get the problem type from the first row
        first_row_type = self.row_widgets[0]['problem_var'].get()

        if not first_row_type:
            messagebox.showwarning("Warning", "1번 문제의 유형이 선택되지 않았습니다.")
            return

        # Apply to all rows
        for rd in self.row_widgets:
            rd['problem_var'].set(first_row_type)
            self.on_problem_type_change(rd, first_row_type)

        # Update all option button colors after applying type
        self.update_all_option_button_colors()
            

    def set_all_hard(self):
        """Toggle all difficulty levels between Hard and Normal"""
        # Check if all rows with Hard option are already set to Hard
        all_are_hard = True
        for rd in self.row_widgets:
            hardness_menu = rd['hardness_menu']
            values = hardness_menu.cget("values")
            if "Hard" in values and hardness_menu.get() != "Hard":
                all_are_hard = False
                break

        # Toggle: if all are Hard, set to Normal; otherwise set all to Hard
        target_level = "Normal" if all_are_hard else "Hard"

        for rd in self.row_widgets:
            hardness_menu = rd['hardness_menu']
            values = hardness_menu.cget("values")
            if target_level in values:
                hardness_menu.set(target_level)


    def randomize_waiting_items(self):
        """Randomly assign waiting items to rows without duplication"""
        if not self.waiting_list_items:
            messagebox.showwarning("Warning", "대기열에 지문이 없습니다.")
            return
        
        if not self.row_widgets:
            messagebox.showwarning("Warning", "추가된 문제가 없습니다.")
            return
        
        # Get available items and shuffle them
        available_items = self.waiting_list_items.copy()
        random.shuffle(available_items)
        
        # Assign shuffled items to rows (up to the minimum of available items or rows)
        num_to_assign = min(len(available_items), len(self.row_widgets))
        
        for i in range(num_to_assign):
            rd = self.row_widgets[i]
            selected_item = available_items[i]
            rd['waiting_var'].set(selected_item)
            
            # Update tooltip with the new passage content
            if 'tooltip' in rd:
                passage_text = self.title_to_passage_map.get(selected_item, "내용이 없습니다.")
                rd['tooltip'].update_text(passage_text)
            
            # Update the current status label
            row_idx = rd['row_idx']
            passage_id유형난이도합본 = (selected_item + ' // ' + 
                                rd['problem_menu'].get() + '_' + 
                                rd['hardness_menu'].get())
            self.update_current_status_label수동실모(row_idx, passage_id유형난이도합본)
        
        # If there are more rows than available items, leave the remaining rows unchanged
        if len(self.row_widgets) > len(available_items):
            messagebox.showinfo(
                "Info", 
                f"대기열에 {len(available_items)}개의 지문만 있어서 "
                f"처음 {len(available_items)}개 문제만 랜덤 배치되었습니다."
            )

        self.refresh_waiting_dropdowns()
