"""spaCy helpers shared across the AutoQM application."""

import os
import random
import sys

import spacy


def load_spacy_model():
    """Load the bundled spaCy model, supporting both frozen and normal runs."""
    try:
        if getattr(sys, 'frozen', False):
            base_path = sys._MEIPASS
            model_base = os.path.join(base_path, 'en_core_web_sm')
            found_version = None

            if os.path.exists(model_base):
                try:
                    subdirs = [
                        d
                        for d in os.listdir(model_base)
                        if os.path.isdir(os.path.join(model_base, d))
                    ]
                    version_dirs = [d for d in subdirs if d.startswith('en_core_web_sm-')]

                    if version_dirs:
                        if 'en_core_web_sm-3.7.0' in version_dirs:
                            found_version = 'en_core_web_sm-3.7.0'
                        else:
                            found_version = sorted(version_dirs, reverse=True)[0]

                        print(f"[INFO] Found spaCy model version: {found_version}")
                except Exception as exc:  # noqa: BLE001
                    print(f"[WARN] Could not scan for spaCy versions: {exc}")

            possible_paths = []
            if found_version:
                possible_paths.append(os.path.join(model_base, found_version))

            possible_paths.extend(
                [
                    os.path.join(base_path, 'en_core_web_sm', 'en_core_web_sm-3.7.0'),
                    os.path.join(base_path, 'en_core_web_sm', 'en_core_web_sm-3.7.1'),
                    os.path.join(base_path, 'en_core_web_sm', 'en_core_web_sm-3.8.0'),
                    os.path.join(base_path, 'en_core_web_sm'),
                ]
            )

            seen = set()
            possible_paths = [p for p in possible_paths if not (p in seen or seen.add(p))]

            print(f"[DEBUG] Checking for spaCy model in {len(possible_paths)} possible paths...")
            for model_path in possible_paths:
                exists = os.path.exists(model_path)
                print(f"[DEBUG] Path exists={exists}: {model_path}")
                if exists:
                    print(f"[INFO] Loading spaCy model from: {model_path}")
                    config_path = os.path.join(model_path, 'config.cfg')
                    if os.path.exists(config_path):
                        print(f"[INFO] config.cfg found at: {config_path}")
                    else:
                        print(f"[WARN] config.cfg NOT found at: {config_path}")
                        try:
                            contents = os.listdir(model_path)
                            print(f"[DEBUG] Directory contents: {contents}")
                        except Exception as exc:  # noqa: BLE001
                            print(f"[DEBUG] Cannot list directory: {exc}")
                    return spacy.load(model_path)

            print("[WARN] Model path not found in _MEIPASS, trying by name...")
            return spacy.load("en_core_web_sm")

        try:
            import en_core_web_sm  # type: ignore

            print(
                "[INFO] Loading spaCy model from installed package: "
                f"{en_core_web_sm.__version__}"
            )
            return en_core_web_sm.load()
        except ImportError:
            print("[WARN] en_core_web_sm package not found, loading by name...")
            return spacy.load("en_core_web_sm")

    except Exception as exc:  # noqa: BLE001
        print(f"[ERROR] SpaCy Load Error: {exc}")
        print("Please ensure en_core_web_sm is installed: python -m spacy download en_core_web_sm")
        raise RuntimeError(f"Failed to load spaCy model: {exc}") from exc


try:
    nlp = load_spacy_model()
    print("[INFO] spaCy model loaded successfully")
except Exception as exc:  # noqa: BLE001
    print(f"[ERROR] Failed to initialize spaCy model: {exc}")
    sys.exit(1)


def jumble_sentence(sentence: str) -> str:
    """Scramble the words in a sentence while respecting proper nouns."""
    cleaned_sentence = sentence.replace(',', '').replace('.', '')
    words = cleaned_sentence.split()

    if not words:
        return ''

    doc = nlp(cleaned_sentence)

    first_word = doc[0]
    if first_word.pos_ != "PROPN" and first_word.text and first_word.text[0].isupper():
        words[0] = first_word.text[0].lower() + first_word.text[1:]

    random.shuffle(words)
    return ' / '.join(words)


def scramble_sentences(text: str) -> str:
    """Scramble each paragraph within the provided text."""
    paragraphs = text.strip().split('\n')
    scrambled_paragraphs = [jumble_sentence(paragraph) for paragraph in paragraphs]
    return '\n'.join(scrambled_paragraphs)


__all__ = ["load_spacy_model", "nlp", "jumble_sentence", "scramble_sentences"]
