"""
Question Type Order Settings for Mock Test (실전모의고사)
Allows users to customize the order of question types when sorting mock test sheets.
"""

import sys
import json
from pathlib import Path
import tkinter as tk
from customtkinter import (
    CTkToplevel, CTkFrame, CTkLabel, CTkButton,
    CTkScrollableFrame, CTkRadioButton
)


class QuestionTypeOrderSettings:
    """Popup window for customizing question type sort order"""

    def __init__(self, parent, app_data_dir, default_font, 실모rows, callback=None):
        self.parent = parent
        self.app_data_dir = app_data_dir
        self.default_font = default_font
        self.실모rows = 실모rows
        self.callback = callback  # Function to call when order is confirmed

        # Settings file path
        self.settings_file = Path(app_data_dir) / 'mocktest_sort_settings.json'

        # Load saved settings or use defaults
        self.question_type_order = self.load_settings()

        # Create popup window
        self.create_popup()

    def load_settings(self):
        """Load saved question type order"""
        if self.settings_file.exists():
            try:
                with open(self.settings_file, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    question_type_order = data.get('question_type_order', list(self.실모rows))

                    # Ensure all question types from 실모rows are included
                    for qtype in self.실모rows:
                        if qtype not in question_type_order:
                            question_type_order.append(qtype)

                    return question_type_order
            except Exception as e:
                print(f"Error loading mocktest sort settings: {e}")

        # Return defaults if no saved settings
        return list(self.실모rows)

    def save_settings(self):
        """Save question type order"""
        try:
            self.settings_file.parent.mkdir(parents=True, exist_ok=True)
            data = {
                'question_type_order': self.question_type_order
            }
            with open(self.settings_file, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
            print(f"Saved mocktest sort settings: {self.settings_file}")
        except Exception as e:
            print(f"Error saving mocktest sort settings: {e}")

    def reset_to_default(self):
        """Reset order to 실모rows"""
        self.question_type_order = list(self.실모rows)
        self.redraw_list()

    def confirm_and_close(self):
        """Save settings and close popup"""
        # Save to file
        self.save_settings()

        # Call callback if provided
        if self.callback:
            self.callback(self.question_type_order)

        # Close popup
        self.popup.destroy()

    def create_popup(self):
        """Create the main popup window"""
        self.popup = CTkToplevel(self.parent)
        self.popup.title("유형별 정렬 세팅")

        if sys.platform.startswith('darwin'):
            self.popup.geometry("380x650")
            self.popup.minsize(width=380, height=700)
        else:
            self.popup.geometry("380x600")
            self.popup.minsize(width=380, height=650)

        self.popup.resizable(True, True)
        self.popup.configure(fg_color="#EFF3F0")
        self.popup.focus_set()

        # Main container
        main_frame = CTkFrame(self.popup, fg_color="transparent")
        main_frame.pack(padx=10, pady=10, fill='both', expand=True)

        # Title
        title_label = CTkLabel(
            main_frame,
            text="유형별 정렬 순서 설정",
            text_color='black',
            font=(self.default_font, 14, 'bold'),
            height=30
        )
        title_label.pack(pady=(0, 10))

        # Scrollable frame for question types
        self.scrollable_frame = CTkScrollableFrame(
            main_frame,
            fg_color="white",
            height=500
        )
        self.scrollable_frame.pack(fill='both', expand=True, pady=(0, 10))

        # Configure grid
        self.scrollable_frame.grid_columnconfigure(0, weight=1)

        # Initialize drag manager
        self.init_drag_manager()

        # Draw the list
        self.redraw_list()

        # Bottom button frame
        button_frame = CTkFrame(main_frame, fg_color="transparent")
        button_frame.pack(fill='x', pady=(5, 0))
        button_frame.grid_columnconfigure(0, weight=1)
        button_frame.grid_columnconfigure(1, weight=1)

        # Left: Reset button
        reset_btn = CTkButton(
            button_frame,
            text="초기값",
            font=(self.default_font, 11),
            fg_color="white",
            hover_color="#DDA15C",
            text_color="black",
            command=self.reset_to_default,
            height=30,
            width=80
        )
        reset_btn.grid(row=0, column=0, padx=5, pady=5, sticky='w')

        # Right: Confirm button
        confirm_btn = CTkButton(
            button_frame,
            text="확인",
            font=(self.default_font, 12),
            fg_color="white",
            hover_color="#DDA15C",
            text_color="black",
            command=self.confirm_and_close,
            height=30,
            width=80
        )
        confirm_btn.grid(row=0, column=1, padx=5, pady=5, sticky='e')

    def init_drag_manager(self):
        """Initialize drag and drop manager"""

        class DragReorderManager:
            """Drag and drop manager for reordering items"""
            def __init__(self, container, owner):
                self.container = container
                self.owner = owner

                self.rowframe_by_gridrow = {}
                self.orig_fg_by_gridrow = {}

                self.start_index = None
                self.highlight_index = None

                # Get inner scrollable frame
                self._inner = getattr(self.container, "_scrollable_frame", self.container)

                # Autoscroll
                self._autoscroll_job = None
                self._autoscroll_dir = 0  # -1(up), 0(stop), +1(down)

                # Find canvas for scrolling
                try:
                    self._canvas = getattr(self.container, "_parent_canvas", None)
                    if self._canvas is None:
                        for ch in self.container.winfo_children():
                            if ch.winfo_class().lower().endswith('canvas'):
                                self._canvas = ch
                                break
                except Exception:
                    self._canvas = None

            def reset_registry(self):
                self.rowframe_by_gridrow.clear()
                self.orig_fg_by_gridrow.clear()
                self.clear_highlight()

            def register_rowframe(self, grid_row: int, frame):
                self.rowframe_by_gridrow[grid_row] = frame
                try:
                    self.orig_fg_by_gridrow[grid_row] = frame.cget("fg_color")
                except Exception:
                    self.orig_fg_by_gridrow[grid_row] = "#B6C2B7"

            def clear_highlight(self):
                if self.highlight_index is None:
                    return
                grid_row = self.highlight_index + 1
                frame = self.rowframe_by_gridrow.get(grid_row)
                if frame is not None:
                    try:
                        frame.configure(fg_color=self.orig_fg_by_gridrow.get(grid_row, "#B6C2B7"))
                    except Exception:
                        pass
                self.highlight_index = None

            def _set_highlight_by_grid_row(self, grid_row: int):
                if grid_row <= 0:
                    return
                if self.highlight_index is not None and (self.highlight_index + 1) != grid_row:
                    self.clear_highlight()
                frame = self.rowframe_by_gridrow.get(grid_row)
                if frame is not None:
                    try:
                        frame.configure(fg_color="#E8DCC2")
                    except Exception:
                        pass
                    self.highlight_index = grid_row - 1

            def _container_child_under_pointer(self, x_root: int, y_root: int):
                w = self.container.winfo_containing(x_root, y_root)
                if w is None:
                    return None
                cur = w
                while getattr(cur, "master", None) is not None and cur.master is not self._inner:
                    cur = cur.master
                if getattr(cur, "master", None) is self._inner:
                    return cur
                return None

            def _start_autoscroll(self, direction: int):
                if direction == self._autoscroll_dir and self._autoscroll_job is not None:
                    return
                self._autoscroll_dir = direction
                if self._autoscroll_job is None:
                    self._do_autoscroll()

            def _stop_autoscroll(self):
                self._autoscroll_dir = 0
                if self._autoscroll_job:
                    try:
                        self.container.after_cancel(self._autoscroll_job)
                    except Exception:
                        pass
                    self._autoscroll_job = None

            def _do_autoscroll(self):
                if self._autoscroll_dir == 0:
                    self._autoscroll_job = None
                    return
                try:
                    steps = 2
                    for _ in range(steps):
                        if self._canvas is not None:
                            self._canvas.yview_scroll(self._autoscroll_dir, "units")
                except Exception:
                    pass
                self._autoscroll_job = self.container.after(20, self._do_autoscroll)

            def _update_autoscroll_from_pointer(self, y_root: int):
                top = self.container.winfo_rooty()
                height = self.container.winfo_height()
                y_rel = y_root - top
                margin = 24
                if y_rel < margin:
                    self._start_autoscroll(-1)
                elif y_rel > height - margin:
                    self._start_autoscroll(1)
                else:
                    self._stop_autoscroll()

            def bind_handle(self, widget, index: int):
                widget.bind("<Button-1>", lambda e, idx=index: self._on_start(e, idx), add="+")
                widget.bind("<B1-Motion>", self._on_motion, add="+")
                widget.bind("<ButtonRelease-1>", self._on_drop, add="+")
                try:
                    widget.configure(cursor="sb_v_double_arrow")
                except Exception:
                    pass

            def _on_start(self, event, index: int):
                self.start_index = index
                self._set_highlight_by_grid_row(index + 1)
                self._update_autoscroll_from_pointer(event.y_root)

            def _on_motion(self, event):
                self._update_autoscroll_from_pointer(event.y_root)
                child = self._container_child_under_pointer(event.x_root, event.y_root)
                if child is None:
                    return
                gi = child.grid_info()
                if gi and "row" in gi:
                    grid_row = int(gi["row"])
                    self._set_highlight_by_grid_row(grid_row)

            def _on_drop(self, event):
                self._stop_autoscroll()
                if self.start_index is None:
                    self.clear_highlight()
                    return

                target_index = self.highlight_index if self.highlight_index is not None else self.start_index

                # Reorder the list
                if 0 <= self.start_index < len(self.owner.question_type_order) and 0 <= target_index < len(self.owner.question_type_order):
                    if target_index != self.start_index:
                        item = self.owner.question_type_order.pop(self.start_index)
                        if target_index > self.start_index:
                            target_index -= 1
                        self.owner.question_type_order.insert(target_index, item)

                # Redraw
                self.owner.redraw_list()

                # Clear
                self.start_index = None
                self.clear_highlight()

        self.drag_manager = DragReorderManager(self.scrollable_frame, self)

    def redraw_list(self):
        """Redraw the question type list"""
        # Reset drag manager
        self.drag_manager.reset_registry()

        # Clear existing widgets (except grid row 0 which is header)
        for widget in self.scrollable_frame.winfo_children():
            gi = widget.grid_info()
            if gi and int(gi.get("row", 0)) > 0:
                widget.destroy()

        # Draw each question type
        for i, qtype in enumerate(self.question_type_order):
            row_num = i + 1

            # Determine background color (same logic as export popup)
            fg_color = "#B6C2B7"
            if qtype in ["커스텀", "동반의어", "영영정의", "요약문", "음성파일", "동반의어문제1", "동반의어문제2", "영영정의문제"]:
                fg_color = "#CBCBCB"
            elif qtype.startswith("실전모의고사"):
                fg_color = "#9AB09F"
            elif qtype.startswith("기출문제세트"):
                fg_color = "#A8C1AC"

            # Row frame
            row_frame = CTkFrame(
                self.scrollable_frame,
                width=350,
                height=30,
                corner_radius=5,
                fg_color=fg_color
            )
            row_frame.grid(row=row_num, column=0, sticky='ew', padx=10, pady=2)
            row_frame.grid_columnconfigure(1, weight=1)

            # Drag handle
            handle_btn = CTkButton(
                row_frame,
                text="≡",
                width=18,
                height=16,
                text_color="black",
                font=(self.default_font, 12, "bold"),
                fg_color="white",
                hover_color="#DDA15C",
                corner_radius=4
            )
            handle_btn.grid(row=0, column=0, padx=5, pady=2)
            self.drag_manager.bind_handle(handle_btn, i)

            # Question type label
            name_label = CTkLabel(
                row_frame,
                text=qtype,
                height=16,
                text_color="black",
                fg_color="transparent",
                font=(self.default_font, 10, "bold"),
                anchor='w'
            )
            name_label.grid(row=0, column=1, sticky='ew', padx=10, pady=2)

            # Register for drag and drop
            self.drag_manager.register_rowframe(row_num, row_frame)


def open_question_type_order_settings(parent, app_data_dir, default_font, 실모rows, callback=None):
    """Convenience function to open the settings popup"""
    QuestionTypeOrderSettings(parent, app_data_dir, default_font, 실모rows, callback)
