"""Custom checkbox list widget used across the AutoQM UI."""

import sys
import traceback
from pathlib import Path

import tkinter as tk
from customtkinter import CTkScrollableFrame, CTkFrame, CTkLabel, CTkCheckBox
from openpyxl import load_workbook


class PassageCheckboxList(CTkScrollableFrame):
    """Scrollable checkbox list that mirrors the previous CTkTextbox API."""

    def __init__(
        self,
        parent,
        excel_file=None,
        default_font=None,
        filter_mode='daegiyeol_only',
        on_checkbox_clicked=None,
        enable_drag=True,
        exclude_sheet_prefixes=None,
        **kwargs,
    ):
        super().__init__(parent, **kwargs)

        self.excel_file = excel_file
        self.default_font = default_font if default_font else ("Arial", 11)
        self.filter_mode = filter_mode  # 'daegiyeol_only' or 'exclude_daegiyeol'
        self.on_checkbox_clicked = on_checkbox_clicked  # Callback when checkbox is clicked
        self.enable_drag = enable_drag  # Whether to show drag handles and enable drag-drop
        self.exclude_sheet_prefixes = tuple(exclude_sheet_prefixes) if exclude_sheet_prefixes else ()
        self.checkboxes = {}  # {passage_id: CTkCheckBox}
        self.checkbox_vars = {}  # {passage_id: BooleanVar}
        self.checkbox_frames = {}  # {passage_id: CTkFrame}
        self.drag_handles = {}  # {passage_id: CTkLabel}
        self.passage_ids = []  # Ordered list of passage IDs
        self.last_clicked_index = None  # For shift-click selection
        self._modified = False  # Track modification state for backward compatibility
        self._checkbox_drag_candidate = None  # Pending drag info when starting from checkbox label
        self._checkbox_dragging = False  # Track active drag initiated from checkbox

        # Platform-specific modifier key
        self.is_mac = sys.platform == 'darwin'
        self.multi_select_modifier = 'Command' if self.is_mac else 'Control'

        # Drag-and-drop state
        self._inner = getattr(self, "_scrollable_frame", self)
        self._canvas = getattr(self, "_parent_canvas", None)
        self.drag_start_index = None
        self.drag_group_indices = []  # List of indices being dragged together
        self.drag_group_original_colors = {}  # Store original colors for restoration
        self.drag_current_index = None
        self.drop_indicator = None
        self._autoscroll_job = None
        self._autoscroll_dir = 0
        self._checkbox_press_state = None  # Track state prior to manual toggle
        self._checkbox_pending_updates = []  # States to enforce after release

        if excel_file:
            self.load_passages_from_excel(excel_file)

    def _should_skip_sheet(self, sheet_name):
        """Return True if the sheet should be skipped based on configured prefixes."""
        if not self.exclude_sheet_prefixes:
            return False
        return any(sheet_name.startswith(prefix) for prefix in self.exclude_sheet_prefixes)

    def load_passages_from_excel(self, excel_file):
        """Load passage IDs from all sheets and sort based on 대기열 order."""
        if not excel_file or not Path(excel_file).exists():
            return

        self.excel_file = excel_file

        try:
            wb = load_workbook(filename=excel_file, read_only=True)

            # Step 1: Collect all passage IDs from all sheets (optionally skipping 대기열)
            all_passage_ids = set()
            for sheet_name in wb.sheetnames:
                if self._should_skip_sheet(sheet_name):
                    continue
                # Skip 대기열 sheet when in exclude_daegiyeol mode
                if self.filter_mode == 'exclude_daegiyeol' and sheet_name == '대기열':
                    continue

                sheet = wb[sheet_name]
                for row in sheet.iter_rows(min_row=1, max_col=1, values_only=True):
                    passage_id = row[0]
                    if passage_id is not None and str(passage_id).strip():
                        # Extract only the part before ' // ' if it exists
                        passage_id_str = str(passage_id).strip()
                        if ' // ' in passage_id_str:
                            passage_id_str = passage_id_str.split(' // ')[0].strip()
                        all_passage_ids.add(passage_id_str)

            # Step 2: Get the order from 대기열 sheet if it exists
            daegiyeol_order = []
            if "대기열" in wb.sheetnames:
                sheet = wb["대기열"]
                for row in sheet.iter_rows(min_row=1, max_col=1, values_only=True):
                    passage_id = row[0]
                    if passage_id is not None and str(passage_id).strip():
                        passage_id_str = str(passage_id).strip()
                        if ' // ' in passage_id_str:
                            passage_id_str = passage_id_str.split(' // ')[0].strip()
                        daegiyeol_order.append(passage_id_str)

            wb.close()

            # Step 3: Filter based on mode
            if self.filter_mode == 'exclude_daegiyeol':
                # Follow 대기열 order first, then append new passages in sorted order
                in_daegiyeol = set(daegiyeol_order)

                # Passages that appear in 대기열 (maintain 대기열 order)
                passages_from_daegiyeol = [pid for pid in daegiyeol_order if pid in all_passage_ids]

                # Passages NOT in 대기열 (in sorted/incremental order)
                passages_not_in_daegiyeol = sorted([pid for pid in all_passage_ids if pid not in in_daegiyeol])

                # Combine: 대기열 order first, then sorted new passages
                final_passage_ids = passages_from_daegiyeol + passages_not_in_daegiyeol

            else:
                # Default: Show ONLY passages from 대기열 sheet
                final_passage_ids = [pid for pid in daegiyeol_order if pid in all_passage_ids]

            # Update the checkbox list
            self.set_passage_list(final_passage_ids)

        except Exception as exc:  # noqa: BLE001
            traceback.print_exc()

    def set_passage_list(self, passage_ids):
        """Set the list of passage IDs and create checkboxes with drag handles."""
        for frame in self.checkbox_frames.values():
            frame.destroy()

        self.checkboxes = {}
        self.checkbox_vars = {}
        self.checkbox_frames = {}
        self.drag_handles = {}
        self.passage_ids = passage_ids
        self.last_clicked_index = None
        self._checkbox_drag_candidate = None
        self._checkbox_dragging = False
        self._checkbox_press_state = None
        self._checkbox_pending_updates = []

        for idx, passage_id in enumerate(passage_ids):
            var = tk.BooleanVar(value=False)

            row_frame = CTkFrame(self, fg_color="transparent", height=25)
            row_frame.pack(fill='x', padx=2, pady=1)

            # Only create drag handle if drag is enabled
            if self.enable_drag:
                handle = CTkLabel(
                    row_frame,
                    text="≡",
                    font=(self.default_font[0], 14, 'bold'),
                    text_color="#666666",
                    width=20,
                    height=20,
                    cursor="sb_v_double_arrow",
                )
                handle.pack(side='left', padx=(2, 5))

                handle.bind('<Button-1>', lambda e, i=idx: self._on_drag_start(e, i))
                handle.bind('<B1-Motion>', self._on_drag_motion)
                handle.bind('<ButtonRelease-1>', self._on_drag_drop)

            checkbox = CTkCheckBox(
                row_frame,
                text=passage_id,
                variable=var,
                font=self.default_font,
                fg_color="#CA7900",
                hover_color="#DDA15C",
                text_color=("black", "white"),
                width=75,
                height=15,
                border_width=2,
                checkbox_width=17,
                checkbox_height=17,
            )
            checkbox.pack(side='left', fill='x', expand=True)

            # Always bind checkbox events for click/shift-click/drag-to-check functionality
            checkbox.bind('<ButtonPress-1>', lambda e, i=idx: self._on_checkbox_press(e, i))
            checkbox.bind('<B1-Motion>', self._on_checkbox_drag_motion, add='+')
            checkbox.bind('<ButtonRelease-1>', lambda e, i=idx: self._on_checkbox_release(e, i))

            self.checkboxes[passage_id] = checkbox
            self.checkbox_vars[passage_id] = var
            self.checkbox_frames[passage_id] = row_frame
            if self.enable_drag:
                self.drag_handles[passage_id] = handle

    def on_checkbox_click(self, event, clicked_index):
        """Handle checkbox click with Shift and Cmd/Ctrl support on release."""
        if clicked_index < 0 or clicked_index >= len(self.passage_ids):
            return "break"

        shift_pressed = (event.state & 0x0001) != 0

        passage_id = self.passage_ids[clicked_index]
        var = self.checkbox_vars.get(passage_id)
        if var is None:
            return "break"

        initial_state = self._checkbox_press_state
        self._checkbox_press_state = None
        if initial_state is None:
            initial_state = var.get()

        self._checkbox_pending_updates = []

        if shift_pressed and self.last_clicked_index is not None:
            start = min(self.last_clicked_index, clicked_index)
            end = max(self.last_clicked_index, clicked_index)

            last_passage_id = self.passage_ids[self.last_clicked_index]
            target_state = self.checkbox_vars[last_passage_id].get()

            for i in range(start, end + 1):
                pid = self.passage_ids[i]
                var_i = self.checkbox_vars.get(pid)
                if var_i is not None:
                    var_i.set(target_state)
                    self._checkbox_pending_updates.append((var_i, target_state))

        else:
            new_state = not initial_state
            var.set(new_state)
            self._checkbox_pending_updates.append((var, new_state))

        self._modified = True
        self.last_clicked_index = clicked_index

        return "break"

    def get_checked_passages(self):
        """Return ordered list of currently checked passage IDs."""
        checked = []
        for passage_id in self.passage_ids:
            if passage_id in self.checkbox_vars and self.checkbox_vars[passage_id].get():
                checked.append(passage_id)
        return checked

    def get(self, start="1.0", end="end-1c"):
        """Mimic CTkTextbox.get() by returning checked passages as newline-separated text."""
        return "\n".join(self.get_checked_passages())

    def set_checked_passages(self, passage_list):
        """Set the checked state from a sequence or newline-separated string."""
        if isinstance(passage_list, str):
            passage_list = [p.strip() for p in passage_list.split('\n') if p.strip()]

        for var in self.checkbox_vars.values():
            var.set(False)

        for passage_id in passage_list:
            if passage_id in self.checkbox_vars:
                self.checkbox_vars[passage_id].set(True)

        self._modified = True

    def select_all(self):
        """Check every checkbox."""
        for var in self.checkbox_vars.values():
            var.set(True)
        self._modified = True

    def deselect_all(self):
        """Uncheck every checkbox."""
        for var in self.checkbox_vars.values():
            var.set(False)
        self._modified = True

    def are_all_selected(self):
        """Return True if every checkbox is currently selected."""
        return bool(self.checkbox_vars) and all(var.get() for var in self.checkbox_vars.values())

    def toggle_all_checkboxes(self):
        """Toggle between all selected and all deselected states."""
        if self.are_all_selected():
            self.deselect_all()
        else:
            self.select_all()

    def set_dimmed(self, dimmed=True):
        """Visually dim or restore all checkboxes and drag handles."""
        if dimmed:
            text_color = ("#9CA3AF", "#6B7280")
            handle_color = "#D1D5DB"
            fg_color = "#D1D5DB"
            hover_color = "#E5E7EB"
        else:
            text_color = ("black", "white")
            handle_color = "#666666"
            fg_color = "#CA7900"
            hover_color = "#DDA15C"

        for checkbox in self.checkboxes.values():
            checkbox.configure(text_color=text_color, fg_color=fg_color, hover_color=hover_color)

        for handle in self.drag_handles.values():
            handle.configure(text_color=handle_color)

    def delete(self, start, end):
        """Compatibility shim for CTkTextbox.delete(); clears selections."""
        self.deselect_all()

    def insert(self, index, text):
        """Compatibility shim for CTkTextbox.insert(); selects passages from text."""
        if text.strip():
            self.set_checked_passages(text)

    def edit_modified(self, value=None):
        """CTkTextbox-compatible modified flag getter/setter."""
        if value is None:
            return self._modified
        if value is False:
            self._modified = False

    def index(self, mark):
        """CTkTextbox-compatible index() method."""
        return mark

    def _find_consecutive_checked_group(self, start_index):
        """Return indices for the consecutive checked group containing start_index."""
        if start_index < 0 or start_index >= len(self.passage_ids):
            return [start_index]

        start_pid = self.passage_ids[start_index]
        if not self.checkbox_vars[start_pid].get():
            return [start_index]

        group_indices = [start_index]

        i = start_index - 1
        while i >= 0:
            pid = self.passage_ids[i]
            if self.checkbox_vars[pid].get():
                group_indices.insert(0, i)
                i -= 1
            else:
                break

        i = start_index + 1
        while i < len(self.passage_ids):
            pid = self.passage_ids[i]
            if self.checkbox_vars[pid].get():
                group_indices.append(i)
                i += 1
            else:
                break

        return group_indices

    def _highlight_frames(self, indices):
        """Highlight frames for the given indices during drag operations."""
        self.drag_group_original_colors = {}
        highlight_color = "#DDA15C"

        for idx in indices:
            if idx < 0 or idx >= len(self.passage_ids):
                continue

            passage_id = self.passage_ids[idx]
            frame = self.checkbox_frames.get(passage_id)

            if frame is not None:
                try:
                    original_color = frame.cget("fg_color")
                    self.drag_group_original_colors[passage_id] = original_color
                    frame.configure(fg_color=highlight_color)
                except Exception:
                    pass

    def _unhighlight_frames(self):
        """Restore original colors to previously highlighted frames."""
        for passage_id, original_color in self.drag_group_original_colors.items():
            frame = self.checkbox_frames.get(passage_id)
            if frame is not None:
                try:
                    frame.configure(fg_color=original_color)
                except Exception:
                    pass

        self.drag_group_original_colors = {}

    def _clear_drop_indicator(self):
        """Remove the drop indicator widget if it exists."""
        if self.drop_indicator is not None:
            try:
                self.drop_indicator.place_forget()
                self.drop_indicator.destroy()
            except Exception:
                pass
            self.drop_indicator = None

    def _show_drop_indicator(self, frame, before=True):
        """Display the drop indicator around the given frame."""
        if frame is None:
            return

        try:
            frame.update_idletasks()
            x = frame.winfo_x()
            y = frame.winfo_y()
            width = frame.winfo_width()
            height = frame.winfo_height()
        except Exception:
            return

        if width <= 0:
            width = frame.winfo_reqwidth()
        if width <= 0:
            width = 200

        indicator_height = 3

        if before:
            y = max(0, y)
        else:
            y += height

        if self.drop_indicator is None or not self.drop_indicator.winfo_exists():
            try:
                self.drop_indicator = CTkFrame(
                    self._inner,
                    fg_color="#BB6C25",
                    height=indicator_height,
                    width=width,
                    corner_radius=2,
                )
            except Exception:
                return
        else:
            try:
                self.drop_indicator.configure(height=indicator_height, width=width)
            except Exception:
                pass

        try:
            self.drop_indicator.place(x=x, y=y)
            self.drop_indicator.lift()
        except Exception:
            pass

    def _get_frame_at_position(self, y_root):
        """Return (index, frame, local_y) for the frame at the given root y-position."""
        for idx, passage_id in enumerate(self.passage_ids):
            frame = self.checkbox_frames.get(passage_id)
            if frame is None:
                continue
            try:
                frame_top = frame.winfo_rooty()
                frame_bottom = frame_top + frame.winfo_height()
                if frame_top <= y_root <= frame_bottom:
                    return idx, frame, y_root - frame_top
            except Exception:
                continue
        return None, None, 0

    def _start_autoscroll(self, direction):
        """Begin auto-scrolling in the provided direction."""
        if direction == self._autoscroll_dir and self._autoscroll_job is not None:
            return
        self._autoscroll_dir = direction
        if self._autoscroll_job is None:
            self._do_autoscroll()

    def _stop_autoscroll(self):
        """Stop any ongoing auto-scroll operation."""
        self._autoscroll_dir = 0
        if self._autoscroll_job:
            try:
                self.after_cancel(self._autoscroll_job)
            except Exception:
                pass
            self._autoscroll_job = None

    def _do_autoscroll(self):
        """Perform a single auto-scroll step."""
        if self._autoscroll_dir == 0:
            self._autoscroll_job = None
            return
        try:
            if self._canvas is not None:
                self._canvas.yview_scroll(self._autoscroll_dir, "units")
        except Exception:
            pass
        self._autoscroll_job = self.after(20, self._do_autoscroll)

    def _update_autoscroll(self, y_root):
        """Update auto-scroll direction given the pointer position."""
        try:
            top = self.winfo_rooty()
            height = self.winfo_height()
            y_rel = y_root - top
            margin = 24

            can_scroll_up = False
            can_scroll_down = False

            if self._canvas is not None:
                try:
                    first, last = self._canvas.yview()
                    epsilon = 1e-6
                    can_scroll_up = first > epsilon
                    can_scroll_down = last < (1.0 - epsilon)
                except Exception:
                    pass

            if y_rel < margin and can_scroll_up:
                self._start_autoscroll(-1)
            elif y_rel > height - margin and can_scroll_down:
                self._start_autoscroll(1)
            else:
                self._stop_autoscroll()
        except Exception:
            self._stop_autoscroll()

    def _on_checkbox_press(self, event, index):
        """Record press state for drag detection and click handling."""
        passage_id = None
        initial_state = None
        if 0 <= index < len(self.passage_ids):
            passage_id = self.passage_ids[index]
            var = self.checkbox_vars.get(passage_id)
            if var is not None:
                toggled_state = var.get()
                initial_state = not toggled_state
                var.set(initial_state)

        self._checkbox_drag_candidate = {
            "index": index,
            "start_x": event.x_root,
            "start_y": event.y_root,
            "initial_state": initial_state,
            "passage_id": passage_id,
        }
        self._checkbox_dragging = False
        self._checkbox_press_state = initial_state

        return "break"

    def _on_checkbox_drag_motion(self, event):
        """Start and process drag operations initiated from checkbox labels."""
        if self._checkbox_drag_candidate is None:
            return

        if not self._checkbox_dragging:
            start = self._checkbox_drag_candidate
            dx = abs(event.x_root - start["start_x"])
            dy = abs(event.y_root - start["start_y"])
            if max(dx, dy) < 4:
                return

            # Only allow drag-to-reorder if enable_drag is True
            if self.enable_drag:
                index = start["index"]
                self._checkbox_dragging = True
                self._on_drag_start(event, index)

        # Only process drag motion for reordering if dragging is active
        if self._checkbox_dragging:
            self._on_drag_motion(event)
        return "break"

    def _on_checkbox_release(self, event, index):
        """Finalize drag started from checkbox or toggle selection on release."""
        if self._checkbox_drag_candidate is None:
            return "break"

        initial_state = self._checkbox_drag_candidate.get("initial_state")

        try:
            if self._checkbox_dragging:
                self._on_drag_drop(event)
                self._checkbox_pending_updates = []
            else:
                self._checkbox_press_state = initial_state
                self.on_checkbox_click(event, index)
                updates = list(self._checkbox_pending_updates)

                if updates:
                    widget = event.widget

                    def enforce_updates():
                        for var_obj, state in updates:
                            try:
                                var_obj.set(state)
                            except Exception:
                                pass

                    if hasattr(widget, 'after'):
                        widget.after(0, enforce_updates)
                    else:
                        enforce_updates()

                self._checkbox_pending_updates = []

            if self.on_checkbox_clicked:
                self.on_checkbox_clicked()

            return "break"
        finally:
            self._checkbox_drag_candidate = None
            self._checkbox_dragging = False
            self._checkbox_press_state = None

    def _on_drag_start(self, event, index):
        """Handle the beginning of a drag operation."""
        self.drag_start_index = index
        self.drag_current_index = index

        self.drag_group_indices = self._find_consecutive_checked_group(index)

        self._highlight_frames(self.drag_group_indices)

        self._update_autoscroll(event.y_root)
        self._clear_drop_indicator()

    def _on_drag_motion(self, event):
        """Track drag motion and update the drop indicator."""
        self._update_autoscroll(event.y_root)

        idx, frame, y_within_frame = self._get_frame_at_position(event.y_root)

        if idx is None:
            if len(self.passage_ids) > 0:
                first_frame = self.checkbox_frames.get(self.passage_ids[0])
                last_frame = self.checkbox_frames.get(self.passage_ids[-1])

                if first_frame is not None:
                    try:
                        if event.y_root < first_frame.winfo_rooty():
                            self.drag_current_index = 0
                            self._show_drop_indicator(first_frame, before=True)
                            return
                    except Exception:
                        pass

                if last_frame is not None:
                    try:
                        last_bottom = last_frame.winfo_rooty() + last_frame.winfo_height()
                        if event.y_root > last_bottom:
                            self.drag_current_index = len(self.passage_ids)
                            self._show_drop_indicator(last_frame, before=False)
                            return
                    except Exception:
                        pass

            self._clear_drop_indicator()
            return

        frame_height = frame.winfo_height() or 1
        before = y_within_frame < (frame_height / 2)

        if before:
            self.drag_current_index = idx
        else:
            self.drag_current_index = idx + 1

        self._show_drop_indicator(frame, before=before)

    def _on_drag_drop(self, event):
        """Complete the drag-and-drop operation."""
        try:
            self._stop_autoscroll()
            self._clear_drop_indicator()

            if self.drag_start_index is None or self.drag_current_index is None:
                return

            group_indices = self.drag_group_indices if self.drag_group_indices else [self.drag_start_index]
            target_idx = self.drag_current_index

            min_group_idx = min(group_indices)
            max_group_idx = max(group_indices)

            if min_group_idx <= target_idx <= max_group_idx + 1:
                return

            saved_states = {pid: self.checkbox_vars[pid].get() for pid in self.passage_ids}

            items_to_move = [self.passage_ids[idx] for idx in sorted(group_indices)]

            for idx in sorted(group_indices, reverse=True):
                self.passage_ids.pop(idx)

            items_removed_before_target = sum(1 for idx in group_indices if idx < target_idx)
            adjusted_target_idx = target_idx - items_removed_before_target
            adjusted_target_idx = max(0, min(adjusted_target_idx, len(self.passage_ids)))

            for i, item in enumerate(items_to_move):
                self.passage_ids.insert(adjusted_target_idx + i, item)

            self.set_passage_list(self.passage_ids)

            for pid, state in saved_states.items():
                if pid in self.checkbox_vars:
                    self.checkbox_vars[pid].set(state)

            self._modified = True

        finally:
            self._unhighlight_frames()
            self.drag_start_index = None
            self.drag_current_index = None
            self.drag_group_indices = []

    def move_selected_up(self):
        """Sort all checkboxes in ascending order."""
        saved_states = {pid: var.get() for pid, var in self.checkbox_vars.items()}

        try:
            sorted_ids = sorted(
                self.passage_ids,
                key=lambda x: float(x) if x.replace('.', '', 1).replace('-', '', 1).isdigit() else x,
            )
        except (ValueError, TypeError):
            sorted_ids = sorted(self.passage_ids)

        self.passage_ids = sorted_ids
        self.set_passage_list(self.passage_ids)

        for pid, state in saved_states.items():
            if pid in self.checkbox_vars:
                self.checkbox_vars[pid].set(state)

        self._modified = True

    def move_selected_down(self):
        """Sort all checkboxes in descending order."""
        saved_states = {pid: var.get() for pid, var in self.checkbox_vars.items()}

        try:
            sorted_ids = sorted(
                self.passage_ids,
                key=lambda x: float(x) if x.replace('.', '', 1).replace('-', '', 1).isdigit() else x,
                reverse=True,
            )
        except (ValueError, TypeError):
            sorted_ids = sorted(self.passage_ids, reverse=True)

        self.passage_ids = sorted_ids
        self.set_passage_list(self.passage_ids)

        for pid, state in saved_states.items():
            if pid in self.checkbox_vars:
                self.checkbox_vars[pid].set(state)

        self._modified = True

    def sort_by_daegiyeol_order(self):
        """Sort all checkboxes to match the order in 대기열 sheet. Items not in 대기열 are appended at the end."""
        if not self.excel_file or not Path(self.excel_file).exists():
            return

        saved_states = {pid: var.get() for pid, var in self.checkbox_vars.items()}

        try:
            wb = load_workbook(filename=self.excel_file, read_only=True)

            # Get the order from 대기열 sheet
            daegiyeol_order = []
            if "대기열" in wb.sheetnames:
                sheet = wb["대기열"]
                for row in sheet.iter_rows(min_row=1, max_col=1, values_only=True):
                    passage_id = row[0]
                    if passage_id is not None and str(passage_id).strip():
                        passage_id_str = str(passage_id).strip()
                        if ' // ' in passage_id_str:
                            passage_id_str = passage_id_str.split(' // ')[0].strip()
                        daegiyeol_order.append(passage_id_str)

            wb.close()

            # Reorder: 대기열 order first, then remaining items at the end
            current_passage_set = set(self.passage_ids)
            daegiyeol_set = set(daegiyeol_order)

            # Passages that appear in 대기열 (maintain 대기열 order)
            passages_from_daegiyeol = [pid for pid in daegiyeol_order if pid in current_passage_set]

            # Passages NOT in 대기열 (preserve their current relative order)
            passages_not_in_daegiyeol = [pid for pid in self.passage_ids if pid not in daegiyeol_set]

            # Combine: 대기열 order first, then remaining passages
            sorted_ids = passages_from_daegiyeol + passages_not_in_daegiyeol

            self.passage_ids = sorted_ids
            self.set_passage_list(self.passage_ids)

            for pid, state in saved_states.items():
                if pid in self.checkbox_vars:
                    self.checkbox_vars[pid].set(state)

            self._modified = True

        except Exception as exc:  # noqa: BLE001
            traceback.print_exc()


__all__ = ["PassageCheckboxList"]
