from __future__ import annotations

from pathlib import Path
from typing import Dict, Iterable, List, Tuple

from customtkinter import CTkButton, CTkFrame, CTkLabel, CTkTextbox, CTkToplevel
from openpyxl import load_workbook


class PointsDashboard(CTkToplevel):
    """Modal dashboard that previews expected question counts and credits."""

    def __init__(
        self,
        parent,
        collected_data: Dict,
        current_credits: int,
        base_credit_map: Dict,
        excel_file: str | None,
        문제checkbox_vars: Dict[str, object],
        자료checkbox_vars: Dict[str, object],
        교재checkbox_vars: Dict[str, object],
        sheet_type_mapping: Dict[str, str],
        enabled_categories: Dict[str, bool] | None,
        default_font,
        manual_mocktest_data: List[Dict] | None = None,
        manual_mocktest_row_overrides: List[Dict] | None = None,
    ) -> None:
        super().__init__(parent)

        self.parent = parent
        self.collected_data = collected_data or {}
        self.current_credits = current_credits or 0
        self.base_credit_map = base_credit_map or {}
        self.excel_file = excel_file
        self.문제checkbox_vars = 문제checkbox_vars or {}
        self.자료checkbox_vars = 자료checkbox_vars or {}
        self.교재checkbox_vars = 교재checkbox_vars or {}
        self.sheet_type_mapping = sheet_type_mapping or {}
        self.enabled_categories = enabled_categories or {
            "문제출제": True,
            "자료제작": True,
            "교재제작": True,
            "실전모의": True,
        }
        self.default_font = default_font
        self.manual_mocktest_data = manual_mocktest_data
        self.manual_mocktest_row_overrides = manual_mocktest_row_overrides or []

        self.word_count_cache: Dict[str, int] = {}
        self._word_counts_loaded = False
        self.user_confirmed = False
        self._types_container: CTkFrame | None = None
        self._types_container_default_height = 80
        self._types_container_min_height = 0
        self._base_height: int | None = None
        self._button_frame_height: int | None = None
        self._resize_bound = False

        self.withdraw()  # Build UI after data prep to avoid flicker
        self.transient(parent)
        self.resizable(True, True)
        self.protocol("WM_DELETE_WINDOW", self.on_cancel)

    # ------------------------------------------------------------------ #
    # Lifecycle
    # ------------------------------------------------------------------ #
    def show(self) -> bool:
        """Display the dashboard as a modal dialog."""
        self.load_word_counts_once()

        if self.manual_mocktest_data:
            total_credits, total_questions, breakdown = self.calculate_manual_mocktest_credits()
        else:
            total_credits, total_questions, breakdown = self.calculate_expected_credits()

        self.create_ui(total_credits, total_questions, breakdown)

        self.deiconify()
        self.grab_set()
        self.focus_set()
        self.wait_window()
        return self.user_confirmed

    # ------------------------------------------------------------------ #
    # UI helpers
    # ------------------------------------------------------------------ #
    def create_ui(self, total_credits: int, total_questions: int, breakdown: Dict[str, Dict[str, int]]) -> None:
        """Build dashboard UI."""
        for child in self.winfo_children():
            child.destroy()

        self.title("출제 예상 포인트")
        self.configure(fg_color="#2B3D2D")  # Match app's dark theme

        # Main content area with cream background - no expand to keep compact
        main_frame = CTkFrame(self, fg_color="#FEF9E0", corner_radius=8)
        main_frame.pack(side="top", fill="both", expand=True, padx=15, pady=(15, 10))

        content_frame = CTkFrame(main_frame, fg_color="#FFFFFF")
        content_frame.pack(fill="both", expand=True, padx=10, pady=10)

        title_label = CTkLabel(
            content_frame,
            text="출제할 유형 및 문제 수",
            font=(self.default_font, 16, "bold"),
            text_color="#2B3D2D",  # Dark green for contrast
        )
        title_label.pack(pady=(0, 10))

        # Simple table with neutral colors
        table_frame = CTkFrame(content_frame, fg_color="#F5F5F5", corner_radius=4)
        table_frame.pack(fill="x", pady=10)

        # Header with subtle gray - even column widths
        header = CTkFrame(table_frame, fg_color="#E0E0E0", corner_radius=0)
        header.pack(fill="x", padx=2, pady=2)

        COL_WIDTH = 100  # Even width for all columns
        CTkLabel(header, text="", width=COL_WIDTH, font=(self.default_font, 12, "bold"), text_color="#2B3D2D", anchor="w").grid(row=0, column=0, padx=5, pady=2, sticky="w")
        CTkLabel(header, text="지문 수", width=COL_WIDTH, font=(self.default_font, 12, "bold"), text_color="#2B3D2D", anchor="center").grid(row=0, column=1, padx=5, pady=2)
        CTkLabel(header, text="유형 수", width=COL_WIDTH, font=(self.default_font, 12, "bold"), text_color="#2B3D2D", anchor="center").grid(row=0, column=2, padx=5, pady=2)
        CTkLabel(header, text="전체 문제 수", width=COL_WIDTH, font=(self.default_font, 12, "bold"), text_color="#2B3D2D", anchor="center").grid(row=0, column=3, padx=5, pady=2)

        display_order = [
            ("문제출제", "문제출제"),
            ("자료제작", "자료제작"),
            ("교재제작", "교재제작"),
            ("실전모의", "실전모의"),
            ("수동실모", "실전모의 수동출제"),
        ]

        for key, label in display_order:
            if key not in breakdown:
                continue

            data = breakdown[key]
            row_frame = CTkFrame(table_frame, fg_color="white")
            row_frame.pack(fill="x", padx=2, pady=1)

            CTkLabel(row_frame, text=f"{label}:", width=COL_WIDTH, anchor="w", font=(self.default_font, 12), text_color="#2B3D2D").grid(row=0, column=0, padx=5, pady=2, sticky="w")

            passages_text = "" if key == "실전모의" else str(data.get("passages", 0))
            CTkLabel(row_frame, text=passages_text, width=COL_WIDTH, font=(self.default_font, 12), text_color="#2B3D2D", anchor="center").grid(row=0, column=1, pady=2)

            types_value = data.get("types")
            types_text = "" if types_value is None else str(types_value)
            CTkLabel(row_frame, text=types_text, width=COL_WIDTH, font=(self.default_font, 12), text_color="#2B3D2D", anchor="center").grid(row=0, column=2, pady=2)

            CTkLabel(row_frame, text=str(data.get("questions", 0)), width=COL_WIDTH, font=(self.default_font, 12), text_color="#2B3D2D", anchor="center").grid(row=0, column=3, pady=2)

        # Footer row - separator line
        footer_separator = CTkFrame(table_frame, fg_color="#B0B0B0", height=2)
        footer_separator.pack(fill="x", padx=2, pady=(3, 0))

        # Footer row - totals
        footer_frame = CTkFrame(table_frame, fg_color="#E8E8E8")
        footer_frame.pack(fill="x", padx=2, pady=2)

        CTkLabel(footer_frame, text="합계:", width=COL_WIDTH, anchor="w", font=(self.default_font, 12, "bold"), text_color="#2B3D2D").grid(row=0, column=0, padx=5, pady=2, sticky="w")
        CTkLabel(footer_frame, text="", width=COL_WIDTH, font=(self.default_font, 12, "bold"), text_color="#2B3D2D", anchor="center").grid(row=0, column=1, pady=2)
        CTkLabel(footer_frame, text="", width=COL_WIDTH, font=(self.default_font, 12, "bold"), text_color="#2B3D2D", anchor="center").grid(row=0, column=2, pady=2)
        CTkLabel(footer_frame, text=str(total_questions), width=COL_WIDTH, font=(self.default_font, 12, "bold"), text_color="#2B3D2D", anchor="center").grid(row=0, column=3, pady=2)

        separator1 = CTkFrame(content_frame, fg_color="#D0D0D0", height=1)
        separator1.pack(fill="x", pady=10)

        selected_label = CTkLabel(
            content_frame,
            text="*선택된 유형:",
            font=(self.default_font, 12, "bold"),
            text_color="#2B3D2D",
            anchor="w",
        )
        selected_label.pack(fill="x")

        selected_types = self.get_all_selected_types()

        # Import CTkScrollableFrame for scrollable badge container
        from customtkinter import CTkScrollableFrame

        # Create scrollable frame for badge-style type display (fixed height with scroll)
        types_container = CTkScrollableFrame(
            content_frame,
            fg_color="#FFFDF7",
            corner_radius=4,
            height=self._types_container_default_height,  # Fixed height - scrolls when content exceeds
        )
        types_container.pack(fill="both", expand=True, pady=(4, 10))
        self._types_container = types_container

        if selected_types:
            # Flow layout: place badges in rows
            self._create_badge_flow(types_container, selected_types)
        else:
            # No types selected
            no_types_label = CTkLabel(
                types_container,
                text="선택된 유형이 없습니다.",
                font=(self.default_font, 11),
                text_color="#2B3D2D",
            )
            no_types_label.pack(pady=10)

        separator2 = CTkFrame(content_frame, fg_color="#D0D0D0", height=1)
        separator2.pack(fill="x", pady=10)

        # Determine if insufficient credits
        is_insufficient = total_credits > self.current_credits
        current_bg_color = "#FFE6E6" if is_insufficient else "#E8F5E9"
        current_border_color = "#E57373" if is_insufficient else "#81C784"
        current_text_color = "#8B3A3A" if is_insufficient else "#2D5016"

        # Horizontal container for both point displays
        points_row_frame = CTkFrame(content_frame, fg_color="transparent")
        points_row_frame.pack(fill="x", pady=(0, 8), padx=2)

        # Left: 현재 남은 포인트
        current_points_frame = CTkFrame(points_row_frame, fg_color=current_bg_color, corner_radius=6, border_width=2, border_color=current_border_color)
        current_points_frame.pack(side="left", fill="both", expand=True, padx=(0, 4))

        current_label = CTkLabel(
            current_points_frame,
            text=f"현재 남은 포인트:\n{self.current_credits:,} 포인트",
            font=(self.default_font, 13, "bold"),
            text_color=current_text_color,
            anchor="center",
            justify="center",
        )
        current_label.pack(fill="both", expand=True, pady=10, padx=12)

        # Right: 소진 예상 포인트
        expected_points_frame = CTkFrame(points_row_frame, fg_color="#FFF4E6", corner_radius=6, border_width=2, border_color="#F2CC8F")
        expected_points_frame.pack(side="left", fill="both", expand=True, padx=(4, 0))

        points_label = CTkLabel(
            expected_points_frame,
            text=f"소진 예상 포인트:\n{total_credits:,} 포인트",
            font=(self.default_font, 13, "bold"),
            text_color="#8B5A00",
            anchor="center",
            justify="center",
        )
        points_label.pack(fill="both", expand=True, pady=10, padx=12)

        # Additional notes below the point displays (only render when needed)
        show_mocktest_note = "실전모의" in breakdown
        show_price_fallback = not self.base_credit_map
        if show_mocktest_note or show_price_fallback:
            notes_frame = CTkFrame(content_frame, fg_color="transparent")
            notes_frame.pack(fill="x", padx=2)

            if show_mocktest_note:
                note_label = CTkLabel(
                    notes_frame,
                    text="* 실전모의는 지문 배정 방식에 따라 실제 소진량이 다를 수 있습니다",
                    font=(self.default_font, 10),
                    text_color="#8B5A00",
                    anchor="w",
                )
                note_label.pack(fill="x")

            if show_price_fallback:
                warning_note = CTkLabel(
                    notes_frame,
                    text="* 가격 정보를 불러올 수 없어 기본값(3포인트)을 사용했습니다",
                    font=(self.default_font, 10),
                    text_color="#8B5A00",
                    anchor="w",
                )
                warning_note.pack(fill="x")

        if is_insufficient:
            warning_label = CTkLabel(
                content_frame,
                text="포인트가 부족합니다. 선택하신 유형을 모두 제작하려면 포인트를 충전해야 합니다.\n포인트가 모두 소진될 경우 출제가 중단됩니다.",
                font=(self.default_font, 11),
                text_color="#8B3A3A",
                anchor="w",
                justify="left",
            )
            warning_label.pack(fill="x", pady=(5, 0))

        # Bottom button frame matching "출제 문항 수 설정" dialog style
        button_frame = CTkFrame(self, fg_color="#2B3D2D")
        button_frame.pack(side="bottom", fill="x", padx=0, pady=0, before=main_frame)

        button_inner_frame = CTkFrame(button_frame, fg_color="transparent")
        button_inner_frame.pack(pady=(5, 12))

        # Left: 취소
        CTkButton(
            button_inner_frame,
            text="취소",
            command=self.on_cancel,
            width=100,
            height=30,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            font=(self.default_font, 13, "bold"),
        ).pack(side="left", padx=5)

        # Center: 포인트 구매
        CTkButton(
            button_inner_frame,
            text="포인트 구매",
            command=self.on_purchase,
            width=120,
            height=30,
            fg_color="#F2CC8F",
            hover_color="#CA7900",
            text_color="black",
            font=(self.default_font, 13, "bold"),
        ).pack(side="left", padx=5)

        # Right: 출제 시작
        CTkButton(
            button_inner_frame,
            text="출제 시작",
            command=self.on_start,
            width=120,
            height=30,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black",
            font=(self.default_font, 13, "bold"),
        ).pack(side="left", padx=5)

        button_frame.update_idletasks()
        button_frame.configure(height=button_frame.winfo_reqheight())
        button_frame.pack_propagate(False)
        self._button_frame_height = button_frame.winfo_reqheight()

        self.bind("<Escape>", lambda _event: self.on_cancel())
        self._base_height = None
        if not self._resize_bound:
            self.bind("<Configure>", self._on_configure)
            self._resize_bound = True
        self._apply_geometry()

    # ------------------------------------------------------------------ #
    # Data preparation
    # ------------------------------------------------------------------ #
    def load_word_counts_once(self) -> None:
        """Load passage word counts from Excel only once per dashboard."""
        if self._word_counts_loaded:
            return

        self.word_count_cache = {}
        self._word_counts_loaded = True

        if not self.excel_file:
            return

        workbook_path = Path(self.excel_file)
        if not workbook_path.exists():
            return

        try:
            wb = load_workbook(workbook_path, data_only=True, read_only=True)
            if "대기열" not in wb.sheetnames:
                wb.close()
                return

            sheet = wb["대기열"]
            for passage_id, passage_text in sheet.iter_rows(min_row=2, max_col=2, values_only=True):
                if not passage_id:
                    continue
                passage_id_str = str(passage_id).strip()
                text = (passage_text or "").strip()
                self.word_count_cache[passage_id_str] = len(text.split())
            wb.close()
        except Exception as exc:
            # Silently fail - word counts will default to 0
            self.word_count_cache = {}

    def get_word_count(self, passage_id: str) -> int:
        passage_id_str = str(passage_id).strip()
        return self.word_count_cache.get(passage_id_str, 0)

    def get_sanitized_passages(self, category: str) -> List[str]:
        if not self.enabled_categories.get(category, False):
            return []
        key_map = {
            "문제출제": "지문번호모음_문제출제",
            "자료제작": "지문번호모음_자료제작",
            "교재제작": "지문번호모음_교재제작",
            "실전모의": "entered_titles",
        }
        raw_list = self.collected_data.get(key_map.get(category, ""), []) or []
        return [item.strip() for item in raw_list if item and item.strip()]

    def count_unique_passages(self, passages: Iterable[str]) -> int:
        return len({p.strip() for p in passages if p and p.strip()})

    def get_selected_types(self, category: str) -> List[str]:
        if not self.enabled_categories.get(category, False):
            return []
        if category == "문제출제":
            source = self.문제checkbox_vars
        elif category == "자료제작":
            source = self.자료checkbox_vars
        elif category == "교재제작":
            source = self.교재checkbox_vars
        else:
            source = {}

        selected = []
        for key, var in source.items():
            try:
                is_checked = bool(var.get())
            except Exception:
                is_checked = False
            if is_checked:
                selected.append(key)
        return selected

    def get_all_selected_types(self) -> List[str]:
        ordered: List[str] = []
        seen = set()

        # Handle manual mocktest mode
        if self.manual_mocktest_data:
            for row in self.manual_mocktest_data:
                problem_type = row.get("problem_type", "")
                hardness = row.get("hardness", "Normal")
                sheet_type = f"{problem_type}_{hardness}"
                if sheet_type not in seen:
                    seen.add(sheet_type)
                    ordered.append(sheet_type)
            return ordered

        for category in ("문제출제", "자료제작", "교재제작"):
            if not self.enabled_categories.get(category, False):
                continue
            for sheet_type in self.get_selected_types(category):
                if sheet_type not in seen:
                    seen.add(sheet_type)
                    ordered.append(sheet_type)

        if self.enabled_categories.get("실전모의", False):
            entry_vars = self.collected_data.get("실모entry_vars", {}) or {}
            checkbox_states = self.collected_data.get("실모checkbox_vars", {}) or {}
            for base_type, count_str in entry_vars.items():
                try:
                    count = int(count_str)
                except (TypeError, ValueError):
                    continue
                if count <= 0:
                    continue
                difficulty = "_Hard" if checkbox_states.get(f"{base_type}_Hard") else "_Normal"
                sheet_type = f"{base_type}{difficulty}"
                if sheet_type not in seen:
                    seen.add(sheet_type)
                    ordered.append(sheet_type)

        return ordered

    # ------------------------------------------------------------------ #
    # Credit calculation
    # ------------------------------------------------------------------ #
    def calculate_expected_credits(self) -> Tuple[int, int, Dict[str, Dict[str, int]]]:
        total_credits = 0
        total_questions = 0
        breakdown: Dict[str, Dict[str, int]] = {}

        # Track passages that need preprocessing (across all categories)
        passages_needing_한글해석 = set()
        passages_needing_동반의어 = set()
        passages_needing_영영정의 = set()

        for category in ("문제출제", "자료제작", "교재제작"):
            if not self.enabled_categories.get(category, False):
                continue
            passages = self.get_sanitized_passages(category)
            selected_types = self.get_selected_types(category)

            # Check if base types are explicitly selected
            한글해석_selected = any(t for t in selected_types if t in ["한글해석", "한글해석_Normal", "한글해석_선택"])
            동반의어_selected = any(t for t in selected_types if t in ["동반의어", "동반의어_Normal"])
            영영정의_selected = any(t for t in selected_types if t in ["영영정의", "영영정의_Normal"])

            # Check if preprocessing types are selected
            한줄연습_types = [t for t in selected_types if "한줄해석연습" in t or "한줄영작연습" in t]
            동반의어문제_types = [t for t in selected_types if "동반의어문제1" in t or "동반의어문제2" in t]
            영영정의문제_types = [t for t in selected_types if "영영정의문제" in t]

            category_credits = 0
            category_questions = 0

            for passage_id in passages:
                word_count = self.get_word_count(passage_id)

                # Check if this passage needs 한글해석 processing for 한줄연습 types
                # Only add preprocessing credit if 한글해석 is NOT already selected
                if 한줄연습_types and not 한글해석_selected and not self._has_한글해석_data(passage_id):
                    if passage_id not in passages_needing_한글해석:
                        passages_needing_한글해석.add(passage_id)
                        # Get credit from BASE_CREDIT_MAP (한글해석_Normal = 1 point)
                        preprocessing_credit = self.get_base_credit_for_type("한글해석_Normal")
                        category_credits += preprocessing_credit
                        category_questions += 1

                # Check if this passage needs 동반의어 processing for 동반의어문제 types
                # Only add preprocessing credit if 동반의어_Normal is NOT already selected
                if 동반의어문제_types and not 동반의어_selected and not self._has_동반의어_data(passage_id):
                    if passage_id not in passages_needing_동반의어:
                        passages_needing_동반의어.add(passage_id)
                        # Get credit from BASE_CREDIT_MAP (동반의어_Normal = 2 points)
                        preprocessing_credit = self.get_base_credit_for_type("동반의어_Normal")
                        category_credits += preprocessing_credit
                        category_questions += 1

                # Check if this passage needs 영영정의 processing for 영영정의문제 types
                # Only add preprocessing credit if 영영정의_Normal is NOT already selected
                if 영영정의문제_types and not 영영정의_selected and not self._has_영영정의_data(passage_id):
                    if passage_id not in passages_needing_영영정의:
                        passages_needing_영영정의.add(passage_id)
                        # Get credit from BASE_CREDIT_MAP (영영정의_Normal = 2 points)
                        preprocessing_credit = self.get_base_credit_for_type("영영정의_Normal")
                        category_credits += preprocessing_credit
                        category_questions += 1

                for sheet_type_full in selected_types:
                    credits = self._calculate_credits_for_single_question(sheet_type_full, word_count, passage_id)
                    category_credits += credits
                    category_questions += 1

            if passages or selected_types:
                breakdown[category] = {
                    "passages": self.count_unique_passages(passages),
                    "types": len(selected_types),
                    "questions": category_questions,
                    "credits": category_credits,
                }
            else:
                breakdown[category] = {
                    "passages": 0,
                    "types": 0,
                    "questions": 0,
                    "credits": 0,
                }

            total_credits += category_credits
            total_questions += category_questions

        mock_credits, mock_questions = self.calculate_mocktest_credits()
        if mock_questions > 0:
            breakdown["실전모의"] = {
                "passages": len(self.get_sanitized_passages("실전모의")),
                "types": None,
                "questions": mock_questions,
                "credits": mock_credits,
            }
            total_credits += mock_credits
            total_questions += mock_questions
        elif self.enabled_categories.get("실전모의", False):
            breakdown["실전모의"] = {
                "passages": len(self.get_sanitized_passages("실전모의")),
                "types": None,
                "questions": 0,
                "credits": 0,
            }

        return total_credits, total_questions, breakdown

    def calculate_mocktest_credits(self) -> Tuple[int, int]:
        if not self.enabled_categories.get("실전모의", False):
            return 0, 0

        entry_vars = self.collected_data.get("실모entry_vars", {}) or {}
        checkbox_states = self.collected_data.get("실모checkbox_vars", {}) or {}
        passages = self.get_sanitized_passages("실전모의")

        if not entry_vars:
            return 0, 0

        if passages:
            total_words = sum(self.get_word_count(pid) for pid in passages)
            avg_word_count = total_words / len(passages) if passages else 0
        else:
            avg_word_count = 0

        # Word count adjustment (backend-accurate: 500/700 thresholds)
        if avg_word_count > 700:
            word_adjustment = 2
        elif avg_word_count > 500:
            word_adjustment = 1
        else:
            word_adjustment = 0

        total_credits = 0
        total_questions = 0

        for base_type, count_str in entry_vars.items():
            try:
                count = int(count_str)
            except (TypeError, ValueError):
                continue

            if count <= 0:
                continue

            difficulty_suffix = "_Hard" if checkbox_states.get(f"{base_type}_Hard") else "_Normal"
            sheet_type = f"{base_type}{difficulty_suffix}"

            credits_per_question = self.get_base_credit_for_type(sheet_type)
            if "동형문제" not in sheet_type:
                credits_per_question += word_adjustment

            # Cap at 7 BEFORE paraphrase
            credits_per_question = min(int(credits_per_question), 7)

            # Add paraphrase AFTER cap
            normalized_type = self.normalize_question_type(sheet_type)
            if self._has_paraphrase_enabled(normalized_type):
                credits_per_question += 1

            total_credits += credits_per_question * count
            total_questions += count

        return total_credits, total_questions

    def calculate_manual_mocktest_credits(self) -> Tuple[int, int, Dict[str, Dict[str, int]]]:
        """Calculate credits for manual mocktest mode (row-based structure).

        Each row in manual_mocktest_data contains:
        - passage_id: str
        - problem_type: str (e.g., "빈칸추론")
        - hardness: str ("Normal" or "Hard")
        """
        if not self.manual_mocktest_data:
            return 0, 0, {}

        total_credits = 0
        total_questions = len(self.manual_mocktest_data)

        # Collect unique passages and types for display
        unique_passages = set()
        type_count: Dict[str, int] = {}  # {sheet_type: count}

        for row_idx, row in enumerate(self.manual_mocktest_data):
            passage_id = row.get("passage_id", "")
            problem_type = row.get("problem_type", "")
            hardness = row.get("hardness", "Normal")

            if passage_id and passage_id != "지문을 선택하세요":
                unique_passages.add(passage_id)

            # Build sheet_type with difficulty suffix
            sheet_type = f"{problem_type}_{hardness}"

            # Count by type
            type_count[sheet_type] = type_count.get(sheet_type, 0) + 1

            # Get word count for this passage
            word_count = self.get_word_count(passage_id)

            # Calculate base credits for this question
            credits = self.get_base_credit_for_type(sheet_type)

            # Word count adjustment (backend-accurate: 500/700 thresholds)
            if "동형문제" not in sheet_type:
                if word_count > 700:
                    credits += 2
                elif word_count > 500:
                    credits += 1

            # Cap at 7 BEFORE paraphrase
            credits = min(int(credits), 7)

            # Add paraphrase AFTER cap (check row-specific setting)
            if row.get("paraphrase_enabled", False):
                credits += 1

            total_credits += credits

        # Build breakdown for UI display
        breakdown: Dict[str, Dict[str, int]] = {
            "수동실모": {
                "passages": len(unique_passages),
                "types": len(type_count),
                "questions": total_questions,
                "credits": total_credits,
            }
        }

        return total_credits, total_questions, breakdown

    def _calculate_credits_for_single_question(self, sheet_type_full: str, word_count: int, passage_id: str = "") -> int:
        credits = self.get_base_credit_for_type(sheet_type_full)

        # Word count adjustment (backend-accurate: 500/700 thresholds)
        if "동형문제" not in sheet_type_full:
            if word_count > 700:
                credits += 2
            elif word_count > 500:
                credits += 1

        # Cap at 7 BEFORE paraphrase
        credits = min(int(credits), 7)

        # Add paraphrase AFTER cap
        normalized_type = self.normalize_question_type(sheet_type_full)
        if self._has_paraphrase_enabled(normalized_type):
            credits += 1

        return credits

    def get_base_credit_for_type(self, sheet_type_full: str) -> int:
        if not self.base_credit_map:
            return 3

        # Normalize sheet type: replace "_선택" with "_Normal" for 교재제작 types
        normalized_type = sheet_type_full
        if sheet_type_full.endswith("_선택"):
            normalized_type = sheet_type_full.replace("_선택", "_Normal")

        candidates = [normalized_type]
        if "_" not in normalized_type:
            candidates.append(f"{normalized_type}_Normal")
        else:
            base = normalized_type.split("_", 1)[0]
            candidates.append(base)

        for candidate in candidates:
            credit = self._coerce_credit_value(self.base_credit_map.get(candidate))
            if credit is not None:
                return credit

        return 3

    @staticmethod
    def _coerce_credit_value(value) -> int | None:
        if value in (None, "-", "?", ""):
            return None
        try:
            return int(value)
        except (TypeError, ValueError):
            try:
                return int(float(value))
            except (TypeError, ValueError):
                return None

    def normalize_question_type(self, sheet_type_with_difficulty: str) -> str:
        base_type = sheet_type_with_difficulty.replace("_Normal", "").replace("_Hard", "").strip()
        return self.sheet_type_mapping.get(base_type, base_type)

    def _has_paraphrase_enabled(self, normalized_type: str) -> bool:
        transformation = self.collected_data.get("transformation_settings", {}) or {}
        settings = transformation.get(normalized_type)
        if not isinstance(settings, dict):
            return False
        return bool(settings.get("paraphrase_enabled"))

    def _has_한글해석_data(self, passage_id: str) -> bool:
        """Check if 한글해석 data exists for this passage in the Excel workbook."""
        if not self.excel_file:
            return False

        workbook_path = Path(self.excel_file)
        if not workbook_path.exists():
            return False

        try:
            from openpyxl import load_workbook
            wb = load_workbook(workbook_path, data_only=True, read_only=True)

            # Check if 한글해석 sheet exists
            if "한글해석" not in wb.sheetnames:
                wb.close()
                return False

            sheet = wb["한글해석"]

            # Search for passage_id in column A (passage IDs)
            for row in sheet.iter_rows(min_row=1, max_col=1, values_only=True):
                cell_value = row[0]
                if cell_value and str(cell_value).strip() == str(passage_id).strip():
                    wb.close()
                    return True

            wb.close()
            return False

        except Exception:
            # Silently fail - assume no data exists
            return False

    def _has_동반의어_data(self, passage_id: str) -> bool:
        """Check if 동반의어 data exists for this passage in the Excel workbook."""
        if not self.excel_file:
            return False

        workbook_path = Path(self.excel_file)
        if not workbook_path.exists():
            return False

        try:
            from openpyxl import load_workbook
            wb = load_workbook(workbook_path, data_only=True, read_only=True)

            # Check if 동반의어 sheet exists (with _Normal suffix)
            sheet_name = "동반의어_Normal"
            if sheet_name not in wb.sheetnames:
                wb.close()
                return False

            sheet = wb[sheet_name]

            # Search for passage_id in column A (passage IDs)
            for row in sheet.iter_rows(min_row=1, max_col=1, values_only=True):
                cell_value = row[0]
                if cell_value and str(cell_value).strip() == str(passage_id).strip():
                    wb.close()
                    return True

            wb.close()
            return False

        except Exception:
            # Silently fail - assume no data exists
            return False

    def _has_영영정의_data(self, passage_id: str) -> bool:
        """Check if 영영정의 data exists for this passage in the Excel workbook."""
        if not self.excel_file:
            return False

        workbook_path = Path(self.excel_file)
        if not workbook_path.exists():
            return False

        try:
            from openpyxl import load_workbook
            wb = load_workbook(workbook_path, data_only=True, read_only=True)

            # Check if 영영정의 sheet exists (with _Normal suffix)
            sheet_name = "영영정의_Normal"
            if sheet_name not in wb.sheetnames:
                wb.close()
                return False

            sheet = wb[sheet_name]

            # Search for passage_id in column A (passage IDs)
            for row in sheet.iter_rows(min_row=1, max_col=1, values_only=True):
                cell_value = row[0]
                if cell_value and str(cell_value).strip() == str(passage_id).strip():
                    wb.close()
                    return True

            wb.close()
            return False

        except Exception:
            # Silently fail - assume no data exists
            return False

    # ------------------------------------------------------------------ #
    # Button handlers
    # ------------------------------------------------------------------ #
    def on_cancel(self) -> None:
        self.user_confirmed = False
        self.destroy()

    def on_purchase(self) -> None:
        from modules.support_links import open_purchase_webpage

        open_purchase_webpage()

    def on_start(self) -> None:
        self.user_confirmed = True
        self.destroy()

    # ------------------------------------------------------------------ #
    # Layout helpers
    # ------------------------------------------------------------------ #
    def _apply_geometry(self) -> None:
        self.update_idletasks()
        width = max(620, min(960, self.winfo_reqwidth() + 16))
        # Use requested height + padding, fit to content
        height = self.winfo_reqheight() + 20
        if self._base_height is None:
            self._base_height = height

        if self.parent is not None:
            try:
                self.parent.update_idletasks()
                px = self.parent.winfo_rootx()
                py = self.parent.winfo_rooty()
                pw = self.parent.winfo_width()
                ph = self.parent.winfo_height()
                x = px + max((pw - width) // 2, 0)
                y = py + max((ph - height) // 2, 0)
                self.geometry(f"{width}x{height}+{x}+{y}")
            except Exception:
                self.geometry(f"{width}x{height}")
        else:
            self.geometry(f"{width}x{height}")

        min_height = max(1, self._button_frame_height or 0)
        self.minsize(620, min_height)
        self._adjust_types_container_height()

    def _on_configure(self, event) -> None:
        if event.widget is self:
            self._adjust_types_container_height(event.height)

    def _adjust_types_container_height(self, window_height: int | None = None) -> None:
        if self._types_container is None or self._base_height is None:
            return
        if window_height is None:
            window_height = self.winfo_height()
        if window_height <= 1:
            return

        delta = max(0, self._base_height - window_height)
        target_height = self._types_container_default_height - delta
        if target_height < self._types_container_min_height:
            target_height = self._types_container_min_height
        elif target_height > self._types_container_default_height:
            target_height = self._types_container_default_height

        current_height = int(float(self._types_container.cget("height")))
        target_height = int(target_height)
        if current_height != target_height:
            self._types_container.configure(height=target_height)

    def _create_badge_flow(self, parent_frame: CTkFrame, types: List[str]) -> None:
        """Create a flow layout of badge-style labels for selected types.

        Uses a simple row-based layout where badges wrap to new rows when
        the container width is exceeded.
        """
        # Badge styling constants
        BADGE_BG = "#E8F0E8"  # Light forest green
        BADGE_BORDER = "#B8D4B8"  # Slightly darker green border
        BADGE_TEXT = "#2B3D2D"  # Dark forest green text
        BADGE_PADX = 6
        BADGE_PADY = 1  # Reduced for smaller height
        BADGE_GAP = 4
        MAX_WIDTH = 540  # Maximum row width before wrapping (adjusted for scrollbar)

        current_row = CTkFrame(parent_frame, fg_color="transparent")
        current_row.pack(anchor="w", fill="x")
        current_width = 0

        for type_name in types:
            # Create badge frame
            badge = CTkFrame(
                current_row,
                fg_color=BADGE_BG,
                corner_radius=4,
                border_width=1,
                border_color=BADGE_BORDER,
            )

            # Create label inside badge
            label = CTkLabel(
                badge,
                text=type_name,
                font=(self.default_font, 10),
                text_color=BADGE_TEXT,
            )
            label.pack(padx=BADGE_PADX, pady=BADGE_PADY)

            # Estimate badge width (approximate)
            badge.update_idletasks()
            badge_width = badge.winfo_reqwidth() + BADGE_GAP

            # Check if we need to wrap to a new row
            if current_width + badge_width > MAX_WIDTH and current_width > 0:
                # Create new row
                current_row = CTkFrame(parent_frame, fg_color="transparent")
                current_row.pack(anchor="w", fill="x", pady=(2, 0))
                current_width = 0

                # Re-parent badge to new row
                badge.destroy()
                badge = CTkFrame(
                    current_row,
                    fg_color=BADGE_BG,
                    corner_radius=4,
                    border_width=1,
                    border_color=BADGE_BORDER,
                )
                label = CTkLabel(
                    badge,
                    text=type_name,
                    font=(self.default_font, 10),
                    text_color=BADGE_TEXT,
                )
                label.pack(padx=BADGE_PADX, pady=BADGE_PADY)
                badge.update_idletasks()
                badge_width = badge.winfo_reqwidth() + BADGE_GAP

            badge.pack(side="left", padx=(0, BADGE_GAP), pady=1)
            current_width += badge_width
