# -*- coding: utf-8 -*-
"""
Excel Sheet Sorting Module

This module provides the ExcelSheetSortingApp class for sorting Excel sheets
according to a predefined order and deleting empty sheets.

Extracted from autoQM.py for better code organization.
"""

import customtkinter
from tkinter import messagebox
from openpyxl import load_workbook
from openpyxl.styles import Alignment
from openpyxl.utils import get_column_letter
from modules.font_handler import font_handler


class ExcelSheetSortingApp(customtkinter.CTkToplevel):
    """
    A popup window for sorting Excel sheets and deleting empty sheets.

    This class provides a UI for reordering Excel sheets according to a predefined
    order (시트_DEFAULT_ORDER) and removing empty sheets (except for "대기열").

    Attributes:
        main_frame: Reference to the MainFrame instance
        sheet_selector: Reference to the sheet selection widget
        excel_file: Path to the Excel file to process
        sheet_name: Current sheet name
        시트_DEFAULT_ORDER: List defining the default order for sheets
        load_excel_into_treeview: Function to load Excel data into treeview
        update_sheet_selector: Function to update the sheet selector widget
    """

    def __init__(self, main_frame, sheet_selector, load_excel_func, update_sheet_func,
                 excel_file, sheet_name, 시트_DEFAULT_ORDER):
        """
        Initialize the ExcelSheetSortingApp.

        Args:
            main_frame: Reference to the MainFrame instance
            sheet_selector: Sheet selection widget reference
            load_excel_func: Function to load Excel data into treeview
            update_sheet_func: Function to update the sheet selector
            excel_file: Path to the Excel file
            sheet_name: Current sheet name
            시트_DEFAULT_ORDER: List defining the default order for sheets
        """
        super().__init__()

        self.main_frame = main_frame  # MainFrame의 참조 저장
        self.sheet_selector = sheet_selector  # 시트 선택 위젯의 참조
        self.excel_file = excel_file
        self.sheet_name = self.sheet_selector.get()
        self.시트_DEFAULT_ORDER = 시트_DEFAULT_ORDER
        self.load_excel_into_treeview = load_excel_func  # Excel을 treeview에 로드하는 함수
        self.update_sheet_selector = update_sheet_func  # 시트 선택기를 업데이트하는 함수

        self.title('유형순서 정렬 및 빈 시트 삭제 툴')
        self.geometry('400x250')  # 높이를 약간 늘림
        self.minsize(400, 250)

        self.default_font = font_handler.default_font

        # 메인 프레임 생성
        self.file_frame = customtkinter.CTkFrame(self, fg_color="transparent")
        self.file_frame.pack(pady=20, padx=20, fill='both', expand=True)

        order_text = ', '.join(self.시트_DEFAULT_ORDER)
        self.label = customtkinter.CTkLabel(
            self.file_frame,
            text=f'유형 순서를 초기값으로 재정렬합니다.\n\n초기값: {order_text}',
            font=(self.default_font, 12),
            width=350,
            height=100,
            wraplength=340  # 패딩을 고려하여 wraplength 설정
        )
        self.label.pack(side='top', padx=5, pady=(5, 10), fill='both', expand=True)

        # 실행 버튼
        execute_button = customtkinter.CTkButton(
            self.file_frame,
            text="실행",
            font=(self.default_font, 13, 'bold'),
            command=self.execute_sheet_sorting,
            width=120,
            height=40,
            fg_color="#FEF9E0",
            hover_color="#DDA15C",
            text_color="black"
        )
        execute_button.pack(side='top', padx=5, pady=(10, 5))

        # 팝업 닫기 바인딩
        self.bind("<Escape>", self.on_popup_close)

    def is_sheet_empty(self, sheet):
        """
        Check if an Excel sheet is empty.

        Args:
            sheet: openpyxl worksheet object

        Returns:
            bool: True if the sheet is empty, False otherwise
        """
        for row in sheet.iter_rows(values_only=True):
            for cell in row:
                if cell is not None:
                    return False
        return True

    def on_popup_close(self, event=None):
        """
        Handle popup close event.

        Args:
            event: Optional event object (default: None)
        """
        self.destroy()

    def execute_sheet_sorting(self):
        """
        Execute the sheet sorting and empty sheet deletion process.

        This method:
        1. Sorts sheets according to 시트_DEFAULT_ORDER
        2. Applies custom column widths and text wrapping
        3. Deletes empty sheets (except "대기열")
        4. Updates the UI to reflect changes
        """
        custom_column_widths = [10, 80, 80, 20, 60]  # 예시 너비

        # 시트 정렬 기준 함수
        def sheet_sort_key(sheet_name):
            base_name = sheet_name.replace('_Normal', '').replace('_Hard', '')
            order_index = self.시트_DEFAULT_ORDER.index(base_name) if base_name in self.시트_DEFAULT_ORDER else float('inf')
            tag_index = 0 if '_Normal' in sheet_name else 1 if '_Hard' in sheet_name else 2
            return (order_index, tag_index, sheet_name)

        try:
            workbook = load_workbook(self.excel_file)
            sorted_sheet_names = sorted(workbook.sheetnames, key=sheet_sort_key)

            # 사용자 정의 열 너비 및 텍스트 줄 바꿈 적용
            for sheet_name in sorted_sheet_names:
                sheet = workbook[sheet_name]
                for i, column_width in enumerate(custom_column_widths, start=1):
                    column_letter = get_column_letter(i)
                    sheet.column_dimensions[column_letter].width = column_width
                    for row in sheet.iter_rows(min_col=i, max_col=i):
                        for cell in row:
                            if cell.value is not None:
                                cell.alignment = Alignment(wrapText=True)

            # 시트 정렬 적용
            reordered_sheets = [workbook[s] for s in sorted_sheet_names]
            # 기존 시트 제거
            while workbook._sheets:
                workbook.remove(workbook._sheets[0])
            # 정렬된 시트 다시 추가
            for sheet in reordered_sheets:
                workbook._add_sheet(sheet)

            # 빈 시트 삭제 (단, "대기열" 시트는 삭제하지 않음)
            empty_sheets = [
                sheet
                for sheet in workbook.sheetnames
                if self.is_sheet_empty(workbook[sheet]) and sheet != "대기열"
            ]
            for sheet in empty_sheets:
                workbook.remove(workbook[sheet])

            workbook.save(self.excel_file)



            # 성공 메시지 표시
            if empty_sheets:
                deleted_sheets = ', '.join(empty_sheets)
                messagebox.showinfo("Success", f"실행이 완료되었습니다.\n빈 시트 삭제됨: {deleted_sheets}")
            else:
                messagebox.showinfo("Success", "실행이 완료되었습니다.")


            self.update_sheet_selector(self.excel_file)
            new_sheet_name = "대기열"
            self.load_excel_into_treeview(self.excel_file, new_sheet_name)


            self.destroy()

        except Exception as e:
            messagebox.showerror("Error", f"시트 정렬 또는 삭제 중 오류가 발생했습니다: {e}")
            self.destroy()
