# -*- coding: utf-8 -*-
"""
Thread-safe global state management for AutoQM application.
Provides locks and synchronized access to global variables.
"""

import threading
from typing import Any, Optional
from contextlib import contextmanager


class ThreadSafeState:
    """Thread-safe wrapper for global application state."""

    def __init__(self):
        self._lock = threading.RLock()  # Reentrant lock for nested access

        # Application state
        self._abort_flag = False
        self._processing_flag = False
        self._unsaved_changes = False
        self._credit_depleted_flag = False
        self._작업중단flag = False

        # User authentication state
        self._user_token: Optional[str] = None
        self._credits = 0
        self._is_logged_in = False

        # Excel file state
        self._excel_file: Optional[str] = None
        self._sheet_name: Optional[str] = None

        # Processing state
        self._credits_to_deduct = 0
        self._passage_id = ""

    @contextmanager
    def lock(self):
        """Context manager for acquiring the lock."""
        self._lock.acquire()
        try:
            yield
        finally:
            self._lock.release()

    # Abort flag
    def get_abort_flag(self) -> bool:
        with self._lock:
            return self._abort_flag

    def set_abort_flag(self, value: bool):
        with self._lock:
            self._abort_flag = value

    # Processing flag
    def get_processing_flag(self) -> bool:
        with self._lock:
            return self._processing_flag

    def set_processing_flag(self, value: bool):
        with self._lock:
            self._processing_flag = value

    # Unsaved changes
    def get_unsaved_changes(self) -> bool:
        with self._lock:
            return self._unsaved_changes

    def set_unsaved_changes(self, value: bool):
        with self._lock:
            self._unsaved_changes = value

    # Credit depleted flag
    def get_credit_depleted_flag(self) -> bool:
        with self._lock:
            return self._credit_depleted_flag

    def set_credit_depleted_flag(self, value: bool):
        with self._lock:
            self._credit_depleted_flag = value

    # 작업중단flag
    def get_작업중단flag(self) -> bool:
        with self._lock:
            return self._작업중단flag

    def set_작업중단flag(self, value: bool):
        with self._lock:
            self._작업중단flag = value

    # User token
    def get_user_token(self) -> Optional[str]:
        with self._lock:
            return self._user_token

    def set_user_token(self, value: Optional[str]):
        with self._lock:
            self._user_token = value

    # Credits
    def get_credits(self) -> int:
        with self._lock:
            return self._credits

    def set_credits(self, value: int):
        with self._lock:
            self._credits = value

    def deduct_credits(self, amount: int) -> int:
        """Thread-safe credit deduction. Returns new credit balance."""
        with self._lock:
            self._credits -= amount
            return self._credits

    def add_credits(self, amount: int) -> int:
        """Thread-safe credit addition. Returns new credit balance."""
        with self._lock:
            self._credits += amount
            return self._credits

    # Is logged in
    def get_is_logged_in(self) -> bool:
        with self._lock:
            return self._is_logged_in

    def set_is_logged_in(self, value: bool):
        with self._lock:
            self._is_logged_in = value

    # Excel file
    def get_excel_file(self) -> Optional[str]:
        with self._lock:
            return self._excel_file

    def set_excel_file(self, value: Optional[str]):
        with self._lock:
            self._excel_file = value

    # Sheet name
    def get_sheet_name(self) -> Optional[str]:
        with self._lock:
            return self._sheet_name

    def set_sheet_name(self, value: Optional[str]):
        with self._lock:
            self._sheet_name = value

    # Credits to deduct
    def get_credits_to_deduct(self) -> int:
        with self._lock:
            return self._credits_to_deduct

    def set_credits_to_deduct(self, value: int):
        with self._lock:
            self._credits_to_deduct = value

    # Passage ID
    def get_passage_id(self) -> str:
        with self._lock:
            return self._passage_id

    def set_passage_id(self, value: str):
        with self._lock:
            self._passage_id = value

    # Batch operations for atomic updates
    def update_processing_state(self, processing: bool, abort: bool = None):
        """Atomically update processing and abort flags together."""
        with self._lock:
            self._processing_flag = processing
            if abort is not None:
                self._abort_flag = abort

    def update_auth_state(self, token: Optional[str], credits: int, logged_in: bool):
        """Atomically update all authentication-related state."""
        with self._lock:
            self._user_token = token
            self._credits = credits
            self._is_logged_in = logged_in

    def reset_processing_state(self):
        """Reset all processing-related flags to default state."""
        with self._lock:
            self._abort_flag = False
            self._processing_flag = False
            self._credit_depleted_flag = False
            self._작업중단flag = False
            self._credits_to_deduct = 0
            self._passage_id = ""


# Global singleton instance
_global_state = ThreadSafeState()


def get_state() -> ThreadSafeState:
    """Get the global thread-safe state instance."""
    return _global_state
