# -*- coding: utf-8 -*-
"""
UndoManager Module

This module provides undo/redo functionality for the main application's treeview.
Manages state history for undo and redo operations, storing treeview snapshots
and coordinating with the MainFrame to save changes to Excel files.

Usage:
    undo_manager = UndoManager(main_frame)
    undo_manager.save_state(treeview)  # Save current state before changes
    undo_manager.undo(treeview)        # Undo last change
    undo_manager.redo(treeview)        # Redo undone change
"""


class UndoManager:
    """
    Manages undo/redo operations for the main application treeview.

    This class maintains two stacks (undo and redo) to track treeview state changes.
    When undo is called, it restores the previous state and saves the current state
    to the redo stack. It also synchronizes changes with the Excel file through
    the main_frame reference.

    Attributes:
        main_frame: Reference to the MainFrame instance for accessing Excel file and sheet name
        undo_stack: Stack of previous treeview states for undo operations
        redo_stack: Stack of undone states for redo operations
        기억할시트이름: Remembered sheet name for undo/redo operations
    """

    def __init__(self, main_frame):
        """
        Initialize the UndoManager.

        Args:
            main_frame: MainFrame instance that provides access to Excel file operations
        """
        self.main_frame = main_frame
        self.undo_stack = []
        self.redo_stack = []
        self.기억할시트이름 = ""

    def save_state(self, treeview):
        """
        Save the current state of the treeview.

        Captures a snapshot of all treeview items and their values, pushes it onto
        the undo stack, and clears the redo stack (since new actions invalidate
        previously undone states).

        Args:
            treeview: The treeview widget whose state should be saved
        """
        #print("\n----------------\n[save_state called]\n")
        # Save the current state of the treeview
        rows = [(item, treeview.item(item)['values']) for item in treeview.get_children()]
        self.undo_stack.append(rows)
        self.redo_stack.clear()  # Clear the redo stack on new action
        self.기억할시트이름 = self.main_frame.get_sheetname()

    # Methods remain the same but add calls to main_frame methods
    def undo(self, treeview):
        """
        Undo the last change to the treeview.

        Restores the previous state from the undo stack, saves the current state
        to the redo stack, and synchronizes the changes with the Excel file.

        Args:
            treeview: The treeview widget to restore previous state to
        """
        print("\n----------------\n[undo called]\n")
        if self.undo_stack:
            current_state = [(item, treeview.item(item)['values']) for item in treeview.get_children()]
            previous_state = self.undo_stack.pop()
            self.redo_stack.append(current_state)
            self._restore_state(treeview, previous_state)

            #sheet_name = self.main_frame.get_sheetname()
            print(f"sheet_name: {self.기억할시트이름}")
            self.main_frame.on_treeview_change()
            self.main_frame.save_treeview_to_excel(self.main_frame.get_current_excel_file(), self.기억할시트이름)
            self.main_frame.sheet_selector.set(self.기억할시트이름)

    def redo(self, treeview):
        """
        Redo a previously undone change to the treeview.

        Restores the next state from the redo stack, saves the current state
        to the undo stack, and synchronizes the changes with the Excel file.

        Args:
            treeview: The treeview widget to restore next state to
        """
        print("\n----------------\n[redo called]\n")
        if self.redo_stack:
            next_state = self.redo_stack.pop()
            current_state = [(item, treeview.item(item)['values']) for item in treeview.get_children()]
            self.undo_stack.append(current_state)
            self._restore_state(treeview, next_state)

            #sheet_name = self.main_frame.get_sheetname()
            print(f"sheet_name: {self.기억할시트이름}")
            self.main_frame.on_treeview_change()
            self.main_frame.save_treeview_to_excel(self.main_frame.get_current_excel_file(), self.기억할시트이름)
            self.main_frame.sheet_selector.set(self.기억할시트이름)

    def _restore_state(self, treeview, state):
        """
        Restore a saved state to the treeview.

        Clears all current items from the treeview and repopulates it with
        the items from the saved state.

        Args:
            treeview: The treeview widget to restore state to
            state: List of (item_id, values) tuples representing the saved state
        """
        # Clear the treeview
        for item in treeview.get_children():
            treeview.delete(item)
        # Restore the saved state
        for item_id, values in state:
            treeview.insert('', 'end', iid=item_id, values=values)
